import { CSSProperties, LegacyRef, ComponentType, ImgHTMLAttributes } from 'react';
import { StyleProp, ViewStyle } from 'react-native';

interface StandardProps<T = any, TouchEvent extends BaseTouchEvent<any> = ITouchEvent> extends EventProps<TouchEvent> {
  /** 组件的唯一标示, 保持整个页面唯一 */
  id?: string
  /** 同 `class`，在 React/Nerv 里一般使用 `className` 作为 `class` 的代称 */
  className?: string
  /** 组件的内联样式, 可以动态设置的内联样式 */
  style?: string | CSSProperties
  /** 如果列表中项目的位置会动态改变或者有新的项目添加到列表中，
   * 需要使用 `wx:key` 来指定列表中项目的唯一的标识符。
   */
  key?: string | number
  /** 组件是否显示, 所有组件默认显示 */
  hidden?: boolean
  /** 动画属性 */
  animation?: { actions: TaroGeneral.IAnyObject[] }
  /** 引用 */
  ref?: LegacyRef<T>
  /**
   * 渲染 HTML
   * @see https://taro-docs.jd.com/taro/docs/html
   */
  dangerouslySetInnerHTML?: {
    __html: string
  }
}

interface FormItemProps {
  /** 表单数据标识 */
  name?: string
}

interface EventProps<TouchEvent extends BaseTouchEvent<any> = ITouchEvent> {
  /** 手指触摸动作开始 */
  onTouchStart?: (event: TouchEvent) => void

  /** 手指触摸后移动 */
  onTouchMove?: (event: TouchEvent) => void

  /** 手指触摸动作被打断，如来电提醒，弹窗 */
  onTouchCancel?: (event: TouchEvent) => void

  /** 手指触摸动作结束 */
  onTouchEnd?: (event: TouchEvent) => void

  /** 手指触摸后马上离开 */
  onClick?: (event: ITouchEvent) => void

  /** 手指触摸后，超过350ms再离开，如果指定了事件回调函数并触发了这个事件，tap事件将不被触发 */
  onLongPress?: (event: CommonEvent) => void

  /** 手指触摸后，超过350ms再离开（推荐使用 longpress 事件代替） */
  onLongClick?: (event: CommonEvent) => void

  /** 会在 WXSS transition 或 Taro.createAnimation 动画结束后触发 */
  onTransitionEnd?: (event: CommonEvent) => void

  /** 会在一个 WXSS animation 动画开始时触发 */
  onAnimationStart?: (event: CommonEvent) => void

  /** 会在一个 WXSS animation 一次迭代结束时触发 */
  onAnimationIteration?: (event: CommonEvent) => void

  /** 会在一个 WXSS animation 动画完成时触发 */
  onAnimationEnd?: (event: CommonEvent) => void

  /** 在支持 3D Touch 的 iPhone 设备，重按时会触发 */
  onTouchForceChange?: (event: CommonEvent) => void
}

type BaseEventOrigFunction<T> = (event: BaseEventOrig<T>) => void

type TouchEventFunction = (event: ITouchEvent) => void

type CanvasTouchEventFunction = (event: CanvasTouchEvent) => void

type CommonEvent<T = any> = BaseEventOrig<T>

type CommonEventFunction<T = any> = BaseEventOrigFunction<T>

interface BaseEventOrig<T = any> {
  /** 事件类型 */
  type: string

  /** 事件生成时的时间戳 */
  timeStamp: number

  /** 触发事件的组件的一些属性值集合 */
  target: Target

  /** 当前组件的一些属性值集合 */
  currentTarget: currentTarget

  /** 额外的信息 */
  detail: T

  /** 阻止元素发生默认的行为 */
  preventDefault: () => void

  /** 阻止事件冒泡到父元素,阻止任何父事件处理程序被执行 */
  stopPropagation: () => void
}

interface BaseTouchEvent<TouchDetail, T = any> extends BaseEventOrig<T> {
  /** 触摸事件，当前停留在屏幕中的触摸点信息的数组 */
  touches: Array<TouchDetail>

  /** 触摸事件，当前变化的触摸点信息的数组 */
  changedTouches: Array<TouchDetail>
}

type CanvasTouchEvent = BaseTouchEvent<CanvasTouch, never>

type ITouchEvent = BaseTouchEvent<ITouch>

interface CanvasTouch {
  identifier: number
  x: number
  y: number
}

interface ITouch extends Touch {
  /** 触摸点的标识符 */
  identifier: number
  /** 距离文档左上角的距离，文档的左上角为原点 ，横向为X轴，纵向为Y轴 */
  pageX: number
  /** 距离文档左上角的距离，文档的左上角为原点 ，横向为X轴，纵向为Y轴 */
  pageY: number
  /** 距离页面可显示区域（屏幕除去导航条）左上角距离，横向为X轴，纵向为Y轴 */
  clientX: number
  /** 距离页面可显示区域（屏幕除去导航条）左上角距离，横向为X轴，纵向为Y轴 */
  clientY: number
}

interface Target {
  /** 事件源组件的id */
  id: string
  /** 当前组件的类型 */
  tagName: string
  /** 事件源组件上由data-开头的自定义属性组成的集合 */
  dataset: {
    [key: string]: any
  }
}

type currentTarget = Target

/** 网络状态数据 */
interface NetStatus {
  /* 当前视频编/码器输出的比特率，单位 kbps */
  videoBitrate?: number
  /* 当前音频编/码器输出的比特率，单位 kbps */
  audioBitrate?: number
  /* 当前视频帧率 */
  videoFPS?: number | string
  /* 当前视频 GOP,也就是每两个关键帧(I帧)间隔时长，单位 s */
  videoGOP?: number
  /* 当前的发送/接收速度 */
  netSpeed?: number
  /* 网络抖动情况，为 0 时表示没有任何抖动，值越大表明网络抖动越大，网络越不稳定 */
  netJitter?: number
  /* 视频画面的宽度 */
  videoWidth?: number | string
  /* 视频画面的高度 */
  videoHeight?: number | string
}

type EventHandler = (ev: Event) => void

interface TaroEvent<T extends EventTarget, D = any> extends Event {
  srcElement: T | null
  target: T
  detail: D
}

/**
 * @ignore
 */
declare const Block: ComponentType

interface CoverImageProps extends StandardProps {
  /** 图标路径，支持临时路径、网络地址、云文件ID。暂不支持base64格式。
   * @supported weapp, h5
   */
  src: string

  /** 图片加载成功时触发
   * @supported weapp, h5
   */
  onLoad?: CommonEventFunction

  /** 图片加载失败时触发
   * @supported weapp, h5
   */
  onError?: CommonEventFunction
}

/** 覆盖在原生组件之上的图片视图。可覆盖的原生组件同cover-view，支持嵌套在cover-view里。
 * @classification viewContainer
 * @supported weapp, swan, alipay, h5
 * @example_react
 * ```tsx
 * // js
 * class App extends Components {
 *   render () {
 *     return (
 *       <View className='container'>
 *       <Video id='myVideo' src='src'>
 *         <CoverView className='controls'>
 *           <CoverView className='play' onClick='play'>
 *             <CoverImage className='img' src='src' />
 *           </CoverView>
 *         </CoverView>
 *       </Video>
 *     )
 *   }
 * }
 * // css
 * .container {
 *   position: relative;
 * }
 * .controls {
 *   position: absolute;
 *   top: 50%;
 *   left: 50%;
 *   width: 300px;
 *   height: 225px;
 *   transform: translate(-50%, -50%);
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="container">
 *     <video id='myvideo' src='https://ugccsy.qq.com/uwMROfz2r5zBIaQXGdGnC2dfDma3J1MItM3912IN4IRQvkRM/o31507f7lcd.mp4?sdtfrom=v1010&guid=aa18cf106b7fdb7e40f2d20b206f2b4f&vkey=63B0FCCC7FC3ADC342C166D86571AE02772258CD9B515B065DC68DF3919D8C288AE831D570ED5E8FE0FF3E81E170D04FF11F874BFDDACF7AAA2C0CFF2ACB39FB1A94DAD1AB859BDA53E4DD6DBCDC1217CEF789A9AC079924E2BBC599EED7A1FFDD60A727F2EB7E7B6472CE63DD4B683C9199DFC78A6A6C4D9891E05467C4B64E'>
 *     </video>
 *     <cover-view class='controls'>
 *       <cover-view class='play' `@tap='play'>
 *         <cover-image class='img' src='https://img10.360buyimg.com/ling/s345x208_jfs/t1/133501/7/9865/382161/5f5ee31fEbdd6a418/0cdc0156ffff3c23.png' />
 *       </cover-view>
 *     </cover-view>
 *   </view>
 * </template>
 * 
 * <style>
 * .container {
 *   position: relative;
 * }
 * .controls {
 *   position: absolute;
 *   top: 50%;
 *   left: 50%;
 *   width: 300px;
 *   height: 225px;
 *   transform: translate(-50%, -50%);
 * }
 * </style>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/cover-image.html
 */
declare const CoverImage: ComponentType<CoverImageProps>

interface ViewProps extends StandardProps {
  /** 指定按下去的样式类。当 `hover-class="none"` 时，没有点击态效果
   * @default none
   * @supported weapp, swan, alipay, tt, h5
   * @rn 由于 RN 不支持 Class，故 RN 端的 View 组件实现了 `hoverStyle`属性，写法和 style 类似，只不过 `hoverStyle` 的样式是指定按下去的样式。
   */
  hoverClass?: string

  /** 由于 RN 不支持 Class，故 RN 端的 View 组件实现了 `hoverStyle`属性，写法和 style 类似，只不过 `hoverStyle` 的样式是指定按下去的样式。
   * @default none
   * @supported rn
   */
  hoverStyle?: StyleProp<ViewStyle>

  /** 指定是否阻止本节点的祖先节点出现点击态
   * @default false
   * @supported weapp, swan, alipay, tt
   */
  hoverStopPropagation?: boolean

  /** 按住后多久出现点击态，单位毫秒
   * @default 50
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  hoverStartTime?: number

  /** 手指松开后点击态保留时间，单位毫秒
   * @default 400
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  hoverStayTime?: number

  /** 是否以 catch 的形式绑定 touchmove 事件 */
  catchMove?: boolean
}

/** 视图容器
 * @classification viewContainer
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Text>flex-direction: row 横向布局</Text>
 *         <View className='flex-wrp' style='flex-direction:row;'>
 *           <View className='flex-item demo-text-1'/>
 *           <View className='flex-item demo-text-2'/>
 *           <View className='flex-item demo-text-3'/>
 *         </View>
 *         <Text>flex-direction: column 纵向布局</Text>
 *         <View className='flex-wrp' style='flex-direction:column;'>
 *           <View className='flex-item flex-item-V demo-text-1'/>
 *           <View className='flex-item flex-item-V demo-text-2'/>
 *           <View className='flex-item flex-item-V demo-text-3'/>
 *         </View>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <text>flex-direction: row 横向布局</text>
 *     <view class="flex-wrp flex-wrp-row" hover-class="hover" >
 *       <view class="flex-item demo-text-1" :hover-stop-propagation="true" />
 *       <view class="flex-item demo-text-2" hover-start-time="1000" hover-class="hover" />
 *       <view class="flex-item demo-text-3" hover-stayTime="1000" hover-class="hover" />
 *     </view>
 *     <text>flex-direction: column 纵向布局</text>
 *     <view class="flex-wrp flex-wrp-column">
 *       <view class="flex-item flex-item-V demo-text-1" />
 *       <view class="flex-item flex-item-V demo-text-2" />
 *       <view class="flex-item flex-item-V demo-text-3" />
 *     </view>
 *   </view>
 * </template>
 * 
 * <style>
 * .flex-wrp { display: flex; }
 * .flex-wrp-column{ flex-direction: column; }
 * .flex-wrp-row { flex-direction:row; padding: 20px; background: #f1f1f1; }
 * .flex-item { width: 180px; height: 90px; }
 * .demo-text-1 { background: #ccc; }
 * .demo-text-2 { background: #999; }
 * .demo-text-3 { background: #666; }
 * .hover {
 *   background: #000;
 * }
 * </style>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/view.html
 */
declare const View: ComponentType<ViewProps>

interface CoverViewProps extends ViewProps {
  /** 设置顶部滚动偏移量，仅在设置了 overflow-y: scroll 成为滚动元素后生效
   * @supported weapp
   */
  scrollTop?: number

  /**
   * 适用于地图组件 map 的自定义气泡 customCallout
   * @supported weapp
   */
  markerId?: string

  /**
   * @supported weapp
   */
  slot?: string
}

/** 覆盖在原生组件之上的文本视图。可覆盖的原生组件包括 map、video、canvas、camera、live-player、live-pusher 只支持嵌套 cover-view、cover-image，可在 cover-view 中使用 button。
 * @classification viewContainer
 * @supported weapp, swan, alipay, h5
 * @example_react
 * ```tsx
 * // js
 * class App extends Components {
 *   render () {
 *     return (
 *       <View className='container'>
 *         <Video id='myVideo' src='src'>
 *           <CoverView className='controls'>
 *             <CoverView className='play' onClick='play'>
 *               <CoverImage className='img' src='src' />
 *             </CoverView>
 *           </CoverView>
 *         </Video>
 *       </View>
 *     )
 *   }
 * }
 * // css
 * .container {
 *   position: relative;
 * }
 * .controls {
 *   position: absolute;
 *   top: 50%;
 *   left: 50%;
 *   width: 300px;
 *   height: 225px;
 *   transform: translate(-50%, -50%);
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="container">
 *     <video id='myvideo' src='https://ugccsy.qq.com/uwMROfz2r5zBIaQXGdGnC2dfDma3J1MItM3912IN4IRQvkRM/o31507f7lcd.mp4?sdtfrom=v1010&guid=aa18cf106b7fdb7e40f2d20b206f2b4f&vkey=63B0FCCC7FC3ADC342C166D86571AE02772258CD9B515B065DC68DF3919D8C288AE831D570ED5E8FE0FF3E81E170D04FF11F874BFDDACF7AAA2C0CFF2ACB39FB1A94DAD1AB859BDA53E4DD6DBCDC1217CEF789A9AC079924E2BBC599EED7A1FFDD60A727F2EB7E7B6472CE63DD4B683C9199DFC78A6A6C4D9891E05467C4B64E'>
 *     </video>
 *     <cover-view class='controls'>
 *       <cover-view class='play' `@tap='play'>
 *         <cover-image class='img' src='https://img10.360buyimg.com/ling/s345x208_jfs/t1/133501/7/9865/382161/5f5ee31fEbdd6a418/0cdc0156ffff3c23.png' />
 *       </cover-view>
 *     </cover-view>
 *   </view>
 * </template>
 * 
 * <style>
 * .container {
 *   position: relative;
 * }
 * .controls {
 *   position: absolute;
 *   top: 50%;
 *   left: 50%;
 *   width: 300px;
 *   height: 225px;
 *   transform: translate(-50%, -50%);
 * }
 * </style>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/cover-view.html
 */
declare const CoverView: ComponentType<CoverViewProps>

interface MatchMediaProps extends StandardProps {
  /** 页面最小宽度（ px 为单位）
   * @supported weapp
   */
  minWidth?:	number

  /** 页面最大宽度（ px 为单位）
   * @supported weapp
   */
  maxWidth?: number

  /** 页面宽度（ px 为单位）
   * @supported weapp
   */
  width?: number

  /** 页面最小高度（ px 为单位）
   * @supported weapp
   */
  minHeight?: number

  /** 页面最大高度（ px 为单位）
   * @supported weapp
   */
  maxHeight?: number

  /** 页面高度（ px 为单位）
   * @supported weapp
   */
  height?: number

  /** 屏幕方向（ landscape 或 portrait ）
   * @supported weapp
   */
  orientation?: string
}

/** media query 匹配检测节点。可以指定一组 media query 规则，满足时，这个节点才会被展示。
 * 通过这个节点可以实现“页面宽高在某个范围时才展示某个区域”这样的效果。
 * @supported weapp
 * @classification viewContainer
 * @example_react
 * ```tsx
 * class App extends Components {
 *   render () {
 *     return (
 *       <View>
 *         <MatchMedia minWidth="300" maxWidth="600">
 *           <view>当页面宽度在 300 ~ 500 px 之间时展示这里</view>
 *         </MatchMedia>
 *         <MatchMedia minHeight="400" orientation="landscape">
 *           <view>当页面高度不小于 400 px 且屏幕方向为纵向时展示这里</view>
 *         </MatchMedia>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <match-media min-width="300" max-width="500">
 *       <view>当页面宽度在 300 ~ 500 px 之间时展示这里</view>
 *     </match-media>
 *     <match-media min-height="400" orientation="landscape">
 *       <view>当页面高度不小于 400 px 且屏幕方向为纵向时展示这里</view>
 *     </match-media>
 *   </view>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/match-media.html
 */
declare const MatchMedia: ComponentType<MatchMediaProps>

interface MovableAreaProps extends StandardProps {
  /** 当里面的 movable-view 设置为支持双指缩放时，设置此值可将缩放手势生效区域修改为整个 movable-area
   * @default false
   * @supported weapp
   */
  scaleArea?: boolean
}

/** movable-view 的可移动区域
 * @classification viewContainer
 * @supported weapp, rn
 * @example_react
 * ```tsx
 * class App extends Components {
 *   render () {
 *     return (
 *       <MovableArea style='height: 200px; width: 200px; background: red;'>
 *         <MovableView style='height: 50px; width: 50px; background: blue;' direction='all'>旅行的意义</MovableView>
 *       </MovableArea>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 *   <movable-area style='height: 200px; width: 200px; background: red;'>
 *     <movable-view style='height: 50px; width: 50px; background: blue;' direction='all'>在路上</movable-view>
 *   </movable-area>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/movable-area.html
 */
declare const MovableArea: ComponentType<MovableAreaProps>

interface MovableViewProps extends Omit<StandardProps, 'animation'> {
  /** movable-view 的移动方向，属性值有`all`、`vertical`、`horizontal`、`none`
   * @default none
   * @supported weapp, rn
   */
  direction?: 'all' | 'vertical' | 'horizontal' | 'none'

  /** movable-view 是否带有惯性
   * @default false
   * @supported weapp
   */
  inertia?: boolean

  /** 超过可移动区域后，movable-view 是否还可以移动
   * @default false
   * @supported weapp
   */
  outOfBounds?: boolean

  /** 定义 x 轴方向的偏移，如果 x 的值不在可移动范围内，会自动移动到可移动范围；改变 x 的值会触发动画
   * @supported weapp, rn
   */
  x?: number | string

  /** 定义 y 轴方向的偏移，如果 y 的值不在可移动范围内，会自动移动到可移动范围；改变 y 的值会触发动画
   * @supported weapp, rn
   */
  y?: number | string

  /** 阻尼系数，用于控制x或y改变时的动画和过界回弹的动画，值越大移动越快
   * @default 20
   * @supported weapp
   */
  damping?: number

  /** 摩擦系数，用于控制惯性滑动的动画，值越大摩擦力越大，滑动越快停止；必须大于 0，否则会被设置成默认值
   * @default 2
   * @supported weapp
   */
  friction?: number

  /** 是否禁用
   * @default false
   * @supported weapp, rn
   */
  disabled?: boolean

  /** 是否支持双指缩放，默认缩放手势生效区域是在 movable-view 内
   * @default false
   * @supported weapp
   */
  scale?: boolean

  /** 定义缩放倍数最小值
   * @default 0.5
   * @supported weapp
   */
  scaleMin?: number

  /** 定义缩放倍数最大值
   * @default 10
   * @supported weapp
   */
  scaleMax?: number

  /** 定义缩放倍数，取值范围为 0.5 - 10
   * @default 1
   * @supported weapp
   */
  scaleValue?: number

  /** 是否使用动画
   * @default true
   * @supported weapp
   */
  animation?: boolean

  /** 拖动过程中触发的事件
   * @supported weapp
   */
  onChange?: CommonEventFunction<MovableViewProps.onChangeEventDetail>

  /** 缩放过程中触发的事件
   * @supported weapp
   */
  onScale?: CommonEventFunction<MovableViewProps.onScaleEventDetail>

  /** 初次手指触摸后移动为横向的移动，如果 catch 此事件，则意味着 touchmove 事件也被 catch
   * @supported weapp
   */
  onHTouchMove?: TouchEventFunction

  /** 初次手指触摸后移动为纵向的移动，如果 catch 此事件，则意味着 touchmove 事件也被 catch
   * @supported weapp
   */
  onVTouchMove?: TouchEventFunction
  /** 开始拖动时触发
   * @supported rn
   */
  onDragStart?: CommonEventFunction
  /** 拖动结束时触发
   * @supported rn
   */
  onDragEnd?: CommonEventFunction
}

declare namespace MovableViewProps {
  /** 拖动过程中触发的事件 */
  interface TChangeSource {
    /** 拖动 */
    touch
    /** 超出移动范围 */
    'touch-out-of-bounds'
    /** 超出移动范围后的回弹 */
    'out-of-bounds'
    /** 惯性 */
    'friction'
    /** setData */
    ''
  }

  interface onChangeEventDetail {
    /** X 坐标 */
    x: number
    /** Y 坐标 */
    y: number
    /** 触发事件 */
    source: keyof MovableViewProps.TChangeSource
  }

  interface onScaleEventDetail {
    /** X 坐标 */
    x: number
    /** Y 坐标 */
    y: number
    /** 缩放比例 */
    scale: number
  }
}

/** 可移动的视图容器，在页面中可以拖拽滑动。movable-view 必须在 movable-area 组件中，并且必须是直接子节点，否则不能移动。
 * @classification viewContainer
 * @supported weapp, swan, alipay, rn
 * @example_react
 * ```tsx
 * class App extends Components {
 *   render () {
 *     return (
 *       <MovableArea style='height: 200px; width: 200px; background: red;'>
 *         <MovableView style='height: 50px; width: 50px; background: blue;' direction='all'></MovableView>
 *       </MovableArea>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 *   <movable-area style='height: 200px; width: 200px; background: red;'>
 *     <movable-view style='height: 50px; width: 50px; background: blue;' direction='all'>带我走</movable-view>
 *   </movable-area>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/movable-view.html
 */
declare const MovableView: ComponentType<MovableViewProps>

interface PageContainerProps extends StandardProps {
  /** 是否显示容器组件
   * @default false
   * @supported weapp
   */
  show?: boolean

  /** 动画时长，单位毫秒
   * @default 300
   * @supported weapp
   */
  duration?: number

  /** z-index 层级
   * @default 100
   * @supported weapp
   */
  zIndex?: number

  /** 是否显示遮罩层
   * @default true
   * @supported weapp
   */
  overlay?: boolean

  /** 弹出位置，可选值为 top bottom right center
   * @default bottom
   * @supported weapp
   */
  position?: keyof PageContainerProps.Position

  /** 是否显示圆角
   * @default false
   * @supported weapp
   */
  round?: boolean

  /** 是否在下滑一段距离后关闭
   * @default false
   * @supported weapp
   */
  closeOnSlideDown?: boolean

  /** 自定义遮罩层样式
   * @supported weapp
   */
  overlayStyle?: string

  /** 自定义弹出层样式
   * @supported weapp
   */
  customStyle?: string

  /** 进入前触发
   * @supported weapp
   */
  onBeforeEnter?: CommonEventFunction

  /** 进入中触发
   * @supported weapp
   */
  onEnter?: CommonEventFunction

  /** 进入后触发
   * @supported weapp
   */
  onAfterEnter?: CommonEventFunction

  /** 离开前触发
   * @supported weapp
   */
  onBeforeLeave?: CommonEventFunction

  /** 离开中触发
   * @supported weapp
   */
  onLeave?: CommonEventFunction

  /** 离开后触发
   * @supported weapp
   */
  onAfterLeave?: CommonEventFunction

  /** 点击遮罩层时触发
   * @supported weapp
   */
  onClickOverlay?: CommonEventFunction
}

declare namespace PageContainerProps {
  /** 弹出位置 */
  interface Position {
    /** 上方弹出 */
    top
    /** 下方弹出 */
    bottom
    /** 右边弹出 */
    right
    /** 中央弹出 */
    center
  }
}

/** 页面容器
 *
 * 小程序如果在页面内进行复杂的界面设计（如在页面内弹出半屏的弹窗、在页面内加载一个全屏的子页面等），用户进行返回操作会直接离开当前页面，不符合用户预期，预期应为关闭当前弹出的组件。
 * 为此提供“假页”容器组件，效果类似于 `popup` 弹出层，页面内存在该容器时，当用户进行返回操作，关闭该容器不关闭页面。返回操作包括三种情形，右滑手势、安卓物理返回键和调用 `navigateBack` 接口。
 *
 * Bug & Tip
 *   1. tip: 当前页面最多只有 1 个容器，若已存在容器的情况下，无法增加新的容器
 *   2. tip: wx.navigateBack 无法在页面栈顶调用，此时没有上一级页面
 * @classification viewContainer
 * @supported weapp
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/page-container.html
 */
declare const PageContainer: ComponentType<PageContainerProps>

interface ScrollViewProps extends StandardProps {
  /** 允许横向滚动
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   * @rn 二选一
   */
  scrollX?: boolean

  /** 允许纵向滚动
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   * @rn 二选一
   */
  scrollY?: boolean

  /** 距顶部/左边多远时（单位px），触发 scrolltoupper 事件
   * @default 50
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  upperThreshold?: number

  /** 距底部/右边多远时（单位px），触发 scrolltolower 事件
   * @default 50
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  lowerThreshold?: number

  /** 设置竖向滚动条位置
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  scrollTop?: number

  /** 设置横向滚动条位置
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  scrollLeft?: number

  /** 值应为某子元素id（id不能以数字开头）。设置哪个方向可滚动，则在哪个方向滚动到该元素
   * @supported weapp, swan, alipay, tt, h5
   */
  scrollIntoView?: string

  /** 在设置滚动条位置时使用动画过渡
   * @supported weapp, swan, alipay, tt, h5, rn
   * @default false
   */
  scrollWithAnimation?: boolean

  /** iOS 点击顶部状态栏、安卓双击标题栏时，滚动条返回顶部，只支持竖向
   * @supported weapp, alipay, rn
   * @default false
   */
  enableBackToTop?: boolean

  /** 启用 flexbox 布局。开启后，当前节点声明了 `display: flex` 就会成为 flex container，并作用于其孩子节点。
   * @supported weapp
   * @default false
   */
  enableFlex?: boolean

  /** 开启 scroll anchoring 特性，即控制滚动位置不随内容变化而抖动，仅在 iOS 下生效，安卓下可参考 CSS `overflow-anchor` 属性。
   * @supported weapp
   * @default false
   */
  scrollAnchoring?: boolean

  /** 开启自定义下拉刷新
   * @supported weapp
   * @default false
   */
  refresherEnabled?: boolean

  /** 设置自定义下拉刷新阈值
   * @supported weapp
   * @default 45
   */
  refresherThreshold?: number

  /** 设置自定义下拉刷新默认样式，支持设置 `black | white | none`， none 表示不使用默认样式
   * @supported weapp
   * @default 'black'
   */
  refresherDefaultStyle?: string

  /** 设置自定义下拉刷新区域背景颜色
   * @supported weapp
   * @default '#FFF'
   */
  refresherBackground?: string

  /** 设置当前下拉刷新状态，true 表示下拉刷新已经被触发，false 表示下拉刷新未被触发
   * @supported weapp
   * @default false
   */
  refresherTriggered?: boolean

  /** 启用 scroll-view 增强特性
   * @supported weapp
   * @default false
   */
  enhanced?: boolean

  /** iOS 下 scroll-view 边界弹性控制 (同时开启 enhanced 属性后生效)
   * @supported weapp
   * @default true
   */
  bounces?: boolean

  /** 滚动条显隐控制 (同时开启 enhanced 属性后生效)
   * @supported weapp
   * @default true
   */
  showScrollbar?: boolean

  /** 分页滑动效果 (同时开启 enhanced 属性后生效)
   * @supported weapp
   * @default false
   */
  pagingEnabled?: boolean

  /** boolean	false	滑动减速速率控制 (同时开启 enhanced 属性后生效)
   * @supported weapp
   * @default false
   */
  fastDeceleration?: boolean

  /** 滚动到顶部/左边，会触发 scrolltoupper 事件
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  onScrollToUpper?: CommonEventFunction

  /** 滚动到底部/右边，会触发 scrolltolower 事件
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  onScrollToLower?: CommonEventFunction

  /** 滚动时触发
   * `event.detail = {scrollLeft, scrollTop, scrollHeight, scrollWidth, deltaX, deltaY}`
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  onScroll?: BaseEventOrigFunction<ScrollViewProps.onScrollDetail>

  /** 自定义下拉刷新控件被下拉
   * @supported weapp
   */
  onRefresherPulling?: CommonEventFunction

  /** 自定义下拉刷新被触发
   * @supported weapp
   */
  onRefresherRefresh?: CommonEventFunction

  /** 自定义下拉刷新被复位
   * @supported weapp
   */
  onRefresherRestore?: CommonEventFunction

  /** 自定义下拉刷新被中止
   * @supported weapp
   */
  onRefresherAbort?: CommonEventFunction

  /** 滑动开始事件 (同时开启 enhanced 属性后生效)
   * @supported weapp
   */
  onDragStart?: CommonEventFunction

  /** 滑动事件 (同时开启 enhanced 属性后生效)
   * @supported weapp
   */
  onDragging?: CommonEventFunction

  /** 滑动结束事件 (同时开启 enhanced 属性后生效)
   * @supported weapp
   */
  onDragEnd?: CommonEventFunction
}

declare namespace ScrollViewProps {
  interface onScrollDetail {
    /** 横向滚动条位置 */
    scrollLeft: number
    /** 竖向滚动条位置 */
    scrollTop: number
    /** 滚动条高度 */
    scrollHeight: number
    /** 滚动条宽度 */
    scrollWidth: number
    deltaX: number
    deltaY: number
  }
}

/** 可滚动视图区域。使用竖向滚动时，需要给scroll-view一个固定高度，通过 WXSS 设置 height。组件属性的长度单位默认为 px
 *
 * Tips:
 * H5 中 ScrollView 组件是通过一个高度（或宽度）固定的容器内部滚动来实现的，因此务必正确的设置容器的高度。例如: 如果 ScrollView 的高度将 body 撑开，就会同时存在两个滚动条（body 下的滚动条，以及 ScrollView 的滚动条）。
 * 微信小程序 中 ScrollView 组件如果设置 scrollX 横向滚动时，并且子元素为多个时（单个子元素时设置固定宽度则可以正常横向滚动），需要通过 WXSS 设置 `white-space: nowrap` 来保证元素不换行，并对 ScrollView 内部元素设置 `display: inline-block` 来使其能够横向滚动。
 * @classification viewContainer
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   onScrollToUpper() {}
 *
 *   // or 使用箭头函数
 *   // onScrollToUpper = () => {}
 *
 *   onScroll(e){
 *     console.log(e.detail)
 *   }
 *
 *   render() {
 *     const scrollStyle = {
 *       height: '150px'
 *     }
 *     const scrollTop = 0
 *     const Threshold = 20
 *     const vStyleA = {
 *       height: '150px',
 *       'background-color': 'rgb(26, 173, 25)'
 *     }
 *     const vStyleB = {
 *        height: '150px',
 *       'background-color': 'rgb(39,130,215)'
 *     }
 *     const vStyleC = {
 *       height: '150px',
 *       'background-color': 'rgb(241,241,241)',
 *       color: '#333'
 *     }
 *     return (
 *       <ScrollView
 *         className='scrollview'
 *         scrollY
 *         scrollWithAnimation
 *         scrollTop={scrollTop}
 *         style={scrollStyle}
 *         lowerThreshold={Threshold}
 *         upperThreshold={Threshold}
 *         onScrollToUpper={this.onScrollToUpper.bind(this)} // 使用箭头函数的时候 可以这样写 `onScrollToUpper={this.onScrollToUpper}`
 *         onScroll={this.onScroll}
 *       >
 *         <View style={vStyleA}>A</View>
 *         <View style={vStyleB}>B</View>
 *         <View style={vStyleC}>C</View>
 *       </ScrollView>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="container">
 *     <view class="page-body">
 *       <view class="page-section">
 *         <view class="page-section-title">
 *           <text>Vertical Scroll - 纵向滚动</text>
 *         </view>
 *         <view class="page-section-spacing">
 *           <scroll-view :scroll-y="true" style="height: 300rpx;" `@scrolltoupper="upper" `@scrolltolower="lower" `@scroll="scroll" :scroll-into-view="toView" :scroll-top="scrollTop">
 *             <view id="demo1" class="scroll-view-item demo-text-1">1</view>
 *             <view id="demo2"  class="scroll-view-item demo-text-2">2</view>
 *             <view id="demo3" class="scroll-view-item demo-text-3">3</view>
 *           </scroll-view>
 *         </view>
 *       </view>
 *       <view class="page-section">
 *         <view class="page-section-title">
 *           <text>Horizontal Scroll - 横向滚动</text>
 *         </view>
 *         <view class="page-section-spacing">
 *           <scroll-view class="scroll-view_H" :scroll-x="true" `@scroll="scroll" style="width: 100%">
 *             <view id="demo21" class="scroll-view-item_H demo-text-1">a</view>
 *             <view id="demo22"  class="scroll-view-item_H demo-text-2">b</view>
 *             <view id="demo23" class="scroll-view-item_H demo-text-3">c</view>
 *           </scroll-view>
 *         </view>
 *       </view>
 *     </view>
 *   </view>
 * </template>
 * 
 * <script>
 * const order = ['demo1', 'demo2', 'demo3']
 * export default {
 *   name: 'Index',
 *   data() {
 *     return {
 *       scrollTop: 0,
 *       toView: 'demo2'
 *     }
 *   },
 * 
 *   methods: {
 *     upper(e) {
 *       console.log('upper:', e)
 *     },
 * 
 *     lower(e) {
 *       console.log('lower:', e)
 *     },
 * 
 *     scroll(e) {
 *       console.log('scroll:', e)
 *     }
 *   }
 * }
 * </script>
 * 
 * <style>
 * .page-section-spacing{
 *   margin-top: 60rpx;
 * }
 * .scroll-view_H{
 *   white-space: nowrap;
 * }
 * .scroll-view-item{
 *   height: 300rpx;
 * }
 * .scroll-view-item_H{
 *   display: inline-block;
 *   width: 100%;
 *   height: 300rpx;
 * }
 * 
 * .demo-text-1 { background: #ccc; }
 * .demo-text-2 { background: #999; }
 * .demo-text-3 { background: #666; }
 * </style>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/scroll-view.html
 */
declare const ScrollView: ComponentType<ScrollViewProps>

interface ShareElementProps extends StandardProps {
  /** 映射标记
   * @supported weapp
   */
  mapkey: string

  /** 是否进行动画
   * @default false
   * @supported weapp
   */
  transform?: boolean

  /** 动画时长，单位毫秒
   * @default 300
   * @supported weapp
   */
  duration?: number

  /** css缓动函数
   * @default ease-out
   * @supported weapp
   */
  easingFunction?: number
}

/** 共享元素
 *
 * 共享元素是一种动画形式，类似于 [`flutter Hero`](https://flutterchina.club/animations/hero-animations/) 动画，表现为元素像是在页面间穿越一样。该组件需与 [`PageContainer`](/docs/components/viewContainer/page-container) 组件结合使用。
 * 使用时需在当前页放置 `ShareElement` 组件，同时在 `PageContainer` 容器中放置对应的 `ShareElement` 组件，对应关系通过属性值 key 映射。当设置 `PageContainer` `显示时，transform` 属性为 `true` 的共享元素会产生动画。当前页面容器退出时，会产生返回动画。
 * @classification viewContainer
 * @supported weapp
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/share-element.html
 */
declare const ShareElement: ComponentType<ShareElementProps>

interface SwiperProps extends StandardProps {
  /** 是否显示面板指示点
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  indicatorDots?: boolean

  /** 指示点颜色
   * @default "rgba(0, 0, 0, .3)"
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  indicatorColor?: string

  /** 当前选中的指示点颜色
   * @default "#000000"
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  indicatorActiveColor?: string

  /** 是否自动切换
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  autoplay?: boolean

  /** 当前所在滑块的 index
   * @default 0
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  current?: number

  /** 当前所在滑块的 item-id ，不能与 current 被同时指定
   * @default ""
   */
  currentItemId?: string

  /** 自动切换时间间隔
   * @default 5000
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  interval?: number

  /** 滑动动画时长
   * @default 500
   * @supported weapp, swan, alipay, tt, h5
   */
  duration?: number

  /** 是否采用衔接滑动
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  circular?: boolean

  /** 滑动方向是否为纵向
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  vertical?: boolean

  /** 前边距，可用于露出前一项的一小部分，接受 px 和 rpx 值
   * @default "0px"
   * @supported weapp, h5
   */
  previousMargin?: string

  /** 后边距，可用于露出后一项的一小部分，接受 px 和 rpx 值
   * @default "0px"
   * @supported weapp, h5
   */
  nextMargin?: string

  /**
   * 当 swiper-item 的个数大于等于 2，关闭 circular 并且开启 previous-margin 或 next-margin 的时候，可以指定这个边距是否应用到第一个、最后一个元素
   * @default false
   * @supported weapp
   */
  snapToEdge?: boolean

  /** 同时显示的滑块数量
   * @default 1
   * @supported weapp, swan, tt, h5
   */
  displayMultipleItems?: number

  /** 是否跳过未显示的滑块布局，设为 true 可优化复杂情况下的滑动性能，但会丢失隐藏状态滑块的布局信息
   * @default false
   * @supported weapp, swan
   */
  skipHiddenItemLayout?: boolean

  /** 指定 swiper 切换缓动动画类型
   * @default "default"
   * @supported weapp
   */
  easingFunction?: keyof SwiperProps.TEasingFunction

  /** current 改变时会触发 change 事件
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  onChange?: CommonEventFunction<SwiperProps.onChangeEventDetail>

  /** swiper-item 的位置发生改变时会触发 transition 事件
   * @supported weapp
   */
  onTransition?: CommonEventFunction<SwiperProps.onTransitionEventDetail>

  /** 动画结束时会触发 animationfinish 事件
   * @supported weapp, swan, h5, rn
   */
  onAnimationFinish?: SwiperProps['onChange']
  
  /** 是否禁止用户 touch 操作
   * @default false
   * @supported alipay
   */
  disableTouch?: boolean
}

declare namespace SwiperProps {
  /** 导致变更的原因 */
  interface TChangeSource {
    /** 自动播放 */
    autoplay
    /** 用户划动 */
    touch
    /** 其它原因 */
    ''
  }

  /** 指定 swiper 切换缓动动画类型 */
  interface TEasingFunction {
    /** 默认缓动函数 */
    default
    /** 线性动画 */
    linear
    /** 缓入动画 */
    easeInCubic
    /** 缓出动画 */
    easeOutCubic
    /** 缓入缓出动画 */
    easeInOutCubic
  }

  interface onChangeEventDetail {
    /** 当前所在滑块的索引 */
    current: number
    /** 导致变更的原因 */
    source: keyof SwiperProps.TChangeSource
    /** SwiperItem的itemId参数值 */
    currentItemId?: string 
  }

  interface onTransitionEventDetail {
    /** X 坐标 */
    dx: number
    /** Y 坐标 */
    dy: number
  }
}

/** 滑块视图容器。其中只可放置 swiper-item 组件，否则会导致未定义的行为。
 * > 不要为 `SwiperItem` 设置 **style** 属性，可以通过 class 设置样式。[7147](https://github.com/NervJS/taro/issues/7147)
 * @classification viewContainer
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * class App extends Component {
 *   render () {
 *     return (
 *       <Swiper
 *         className='test-h'
 *         indicatorColor='#999'
 *         indicatorActiveColor='#333'
 *         vertical
 *         circular
 *         indicatorDots
 *         autoplay>
 *         <SwiperItem>
 *           <View className='demo-text-1'>1</View>
 *         </SwiperItem>
 *         <SwiperItem>
 *           <View className='demo-text-2'>2</View>
 *         </SwiperItem>
 *         <SwiperItem>
 *           <View className='demo-text-3'>3</View>
 *         </SwiperItem>
 *       </Swiper>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <swiper
 *     class='test-h'
 *     indicator-color='#999'
 *     indicator-active-color='#333'
 *     :vertical="true"
 *     :circular="true"
 *     :indicator-dots="true"
 *     :autoplay="true"
 *   >
 *     <swiper-item>
 *       <view class='demo-text-1'>1</view>
 *     </swiper-item>
 *     <swiper-item>
 *       <view class='demo-text-2'>2</view>
 *     </swiper-item>
 *     <swiper-item>
 *       <view class='demo-text-3'>3</view>
 *     </swiper-item>
 *   </swiper>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/swiper.html
 */
declare const Swiper: ComponentType<SwiperProps>

interface SwiperItemProps extends StandardProps {
  /** 该 swiper-item 的标识符
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  itemId?: string
}

/** 仅可放置在 swiper 组件中，宽高自动设置为100%
 * > 不要为 `SwiperItem` 设置 **style** 属性，可以通过 class 设置样式。[7147](https://github.com/NervJS/taro/issues/7147)
 * @classification viewContainer
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * class App extends Component {
 *   render () {
 *     return (
 *       <Swiper
 *         className='test-h'
 *         indicatorColor='#999'
 *         indicatorActiveColor='#333'
 *         vertical
 *         circular
 *         indicatorDots
 *         autoplay>
 *         <SwiperItem>
 *           <View className='demo-text-1'>1</View>
 *         </SwiperItem>
 *         <SwiperItem>
 *           <View className='demo-text-2'>2</View>
 *         </SwiperItem>
 *         <SwiperItem>
 *           <View className='demo-text-3'>3</View>
 *         </SwiperItem>
 *       </Swiper>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <swiper
 *     class='test-h'
 *     indicator-color='#999'
 *     indicator-active-color='#333'
 *     :vertical="true"
 *     :circular="true"
 *     :indicator-dots="true"
 *     :autoplay="true"
 *   >
 *     <swiper-item>
 *       <view class='demo-text-1'>1</view>
 *     </swiper-item>
 *     <swiper-item>
 *       <view class='demo-text-2'>2</view>
 *     </swiper-item>
 *     <swiper-item>
 *       <view class='demo-text-3'>3</view>
 *     </swiper-item>
 *   </swiper>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/swiper-item.html
 */
declare const SwiperItem: ComponentType<SwiperItemProps>

interface IconProps extends StandardProps {
  /** icon 的类型
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  type: keyof IconProps.TIconType

  /** icon 的大小，单位px
   * @default 23
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  size?: string,

  /** icon 的颜色，同 css 的 color
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  color?: string
}

declare namespace IconProps {
  /** icon 的类型 */
  interface TIconType {
    /** 成功图标 */
    success
    /** 成功图标（不带外圈） */
    success_no_circle
    /** 信息图标 */
    info
    /** 警告图标 */
    warn
    /** 等待图标 */
    waiting
    /** 取消图标 */
    cancel
    /** 下载图标 */
    download
    /** 搜索图标 */
    search
    /** 清除图标 */
    clear
    /** 圆环图标(多选控件图标未选择)「微信文档未标注属性」 */
    circle
    /** 带圆环的信息图标「微信文档未标注属性」 */
    info_circle
  }
}

/** 图标。组件属性的长度单位默认为 px
 * @classification base
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Icon size='60' type='success' />
 *         <Icon size='60' type='info' />
 *         <Icon size='60' type='warn' color='#ccc' />
 *         <Icon size='60' type='warn' />
 *         <Icon size='60' type='waiting' />
 *         <Icon size='20' type='success_no_circle' />
 *         <Icon size='20' type='warn' />
 *         <Icon size='20' type='success' />
 *         <Icon size='20' type='download' />
 *         <Icon size='20' type='clear' color='red' />
 *         <Icon size='20' type='search' />
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * <template>
 *   <view class="components-page">
 *     <icon size="60" type="success" />
 *     <icon size="60" type="info" />
 *     <icon size="60" type="warn" color="#ccc" />
 *     <icon size="60" type="warn" />
 *     <icon size="60" type="waiting" />
 *     <icon size="20" type="success_no_circle" />
 *     <icon size="20" type="warn" />
 *     <icon size="20" type="success" />
 *     <icon size="20" type="download" />
 *     <icon size="20" type="clear" color="red" />
 *     <icon size="20" type="search" />
 *   </view>
 * </template>
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/icon.html
 */
declare const Icon: ComponentType<IconProps>

interface ProgressProps extends StandardProps {
  /** 百分比 0~100
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  percent?: number

  /** 在进度条右侧显示百分比
   * @default false
   * @supported weapp, swan, alipay, h5, rn
   */
  showInfo?: boolean

  /** 圆角大小
   * @default 0
   * @supported weapp, h5
   */
  borderRadius?: number | string

  /** 右侧百分比字体大小
   * @default 16
   * @supported weapp, h5
   */
  fontSize?: number | string

  /** 进度条线的宽度
   * @default 6
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  strokeWidth?: number | string

  /** 进度条颜色 (请使用 activeColor)
   * @default "#09BB07"
   * @supported weapp, swan, alipay, h5, tt
   */
  color?: string

  /** 已选择的进度条的颜色
   * @default "#09BB07"
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  activeColor?: string

  /** 未选择的进度条的颜色
   * @default "#EBEBEB"
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  backgroundColor?: string

  /** 进度条从左往右的动画
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  active?: boolean

  /** backwards: 动画从头播
   *
   * forwards: 动画从上次结束点接着播
   * @default backwards
   * @supported weapp, swan, tt, rn, h5
   */
  activeMode?: 'backwards' | 'forwards'

  /** 进度增加 1% 所需毫秒数
   * @default 30
   * @supported weapp, h5
   */
  duration?: number

  /** 动画完成事件
   * @supported weapp, h5
   */
  onActiveEnd?: CommonEventFunction
}

/** 进度条。组件属性的长度单位默认为 px
 * @classification base
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 * 
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Progress percent={20} showInfo strokeWidth={2} />
 *         <Progress percent={40} strokeWidth={2} active />
 *         <Progress percent={60}  strokeWidth={2} active />
 *         <Progress percent={80}  strokeWidth={2} active activeColor='blue' />
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <progress percent="20" stroke-width="2" :show-info="true" />
 *     <progress percent="40" stroke-width="2" :active="true" />
 *     <progress percent="60" stroke-width="2" :active="true" />
 *     <progress percent="80" stroke-width="2" :active="true" active-color="blue" />
 *   </view>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/progress.html
 */
declare const Progress: ComponentType<ProgressProps>

/** 节点类型
 * > 现支持两种节点，通过type来区分，分别是元素节点和文本节点，默认是元素节点，在富文本区域里显示的HTML节点 元素节点：type = node*
 */
type Nodes = Array<RichTextProps.Text | RichTextProps.HTMLElement> | string

interface RichTextProps extends StandardProps {
  /** 节点列表/ HTML String */
  nodes?: Nodes
  /** 显示连续空格
   * @supported weapp
   */
  space?: keyof RichTextProps.TSpace
}

declare namespace RichTextProps {
  /** space 的合法值 */
  interface TSpace {
    /** 中文字符空格一半大小 */
    ensp
    /** 中文字符空格大小 */
    emsp
    /** 根据字体设置的空格大小 */
    nbsp
  }

  /** 文本节点 */
  interface Text {
    /** 文本类型 */
    type: 'text'

    /** 文本字符串
     * @remarks 支持 entities
     * @default ""
     */
    text: string
  }

  /** 元素节点，默认为元素节点
   * 全局支持class和style属性，不支持 id 属性。
   */
  interface HTMLElement {
    /** HTML 类型 */
    type?: 'node'

    /** 标签名
     * @remarks 支持部分受信任的 HTML 节点
     */
    name: string

    /** 属性
     * @remarks 支持部分受信任的属性，遵循 Pascal 命名法
     */
    attrs?: Object

    /** 子节点列表
     * @remarks 结构和 nodes 一致
     */
    children?: Nodes
  }
}

/** 富文本
 * @classification base
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * class App extends Components {
 *   state = {
 *     nodes: [{
 *       name: 'div',
 *       attrs: {
 *         class: 'div_class',
 *         style: 'line-height: 60px; color: red;'
 *       },
 *       children: [{
 *         type: 'text',
 *         text: 'Hello World!'
 *       }]
 *     }]
 *   }
 *   render () {
 *     return (
 *       <RichText nodes={this.state.nodes} />
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <rich-text :nodes="nodes"></rich-text>
 *   </view>
 * </template>
 * 
 * <script>
 * export default {
 *   name: 'Index',
 *   data() {
 *     return {
 *       nodes: [{
 *         name: 'div',
 *         attrs: {
 *           class: 'div_class',
 *           style: 'line-height: 60px; color: red;'
 *         },
 *         children: [{
 *           type: 'text',
 *           text: 'Hello World!'
 *         }]
 *       }]
 *     }
 *   },
 *   onReady () {
 *     console.log('onReady')
 *   }
 * }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/rich-text.html
 */
declare const RichText: ComponentType<RichTextProps>

interface TextProps extends StandardProps {
  /** 文本是否可选
   * @default false
   * @supported weapp, swan, alipay, tt, h5, rn
   */
  selectable?: boolean

  /** 文本是否可选，该属性会使文本节点显示为 inline-block
   * @default false
   * @supported weapp
   */
  userSelect?: boolean

  /** 显示连续空格
   * @supported weapp, swan, tt
   */
  space?: keyof TextProps.TSpace

  /** 是否解码
   * @default false
   * @supported weapp, tt
   */
  decode?: boolean
}

declare namespace TextProps {
  /** space 的合法值 */
  interface TSpace {
    /** 中文字符空格一半大小 */
    ensp
    /** 中文字符空格大小 */
    emsp
    /** 根据字体设置的空格大小 */
    nbsp
  }
}

/** 文本
 * @classification base
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   state = {
 *     contents: [],
 *     contentsLen: 0
 *   }
 *
 *   add = () => {
 *     this.setState(prev => {
 *       const cot = prev.contents.slice()
 *       cot.push({ text: 'hello world' })
 *       return {
 *         contents: cot,
 *         contentsLen: cot.length
 *       }
 *     })
 *   }
 *
 *   remove = () => {
 *     this.setState(prev => {
 *       const cot = prev.contents.slice()
 *       cot.pop()
 *       return {
 *         contents: cot,
 *         contentsLen: cot.length
 *       }
 *     })
 *   }
 *
 *   render () {
 *     return (
 *       <View className='container'>
 *         {this.state.contents.map((item, index) => (
 *           <Text key={index}>{item.text}</Text>
 *         ))}
 *         <Button className='btn-max-w button_style' plain type='default' onClick={this.add}>add line</Button>
 *         <Button className='btn-max-w button_style' plain type='default' disabled={this.state.contentsLen ? false : true} onClick={this.remove}>remove line</Button>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ``` html
 * <template>
 *   <view class="container">
 *     <view v-for="(item, index) in contents">
 *       <text>{{item.text}} line {{index + 1}}</text>
 *     </view>
 *     <button class="btn-max-w button_style" :plain="true" type="default" `@tap="add">add line</button>
 *     <button class="btn-max-w button_style" :plain="true" type="default" :disabled="contentsLen ? false : true" `@tap="remove">remove line</button>
 * </template>
 * 
 * <script>
 * export default {
 *   data() {
 *     return {
 *       contents: [],
 *       contentsLen: 0
 *     }
 *   },
 *   methods: {
 *     add () {
 *       const cot = this.contents.slice()
 *       cot.push({ text: 'hello world' })
 *       this.contents = cot
 *       this.contentsLen = cot.length
 *     },
 * 
 *     remove () {
 *       const cot = this.contents.slice()
 *       cot.pop()
 *       this.contents = cot
 *       this.contentsLen = cot.length
 *     }
 *   }
 * }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/text.html
 */
declare const Text: ComponentType<TextProps>

interface ButtonProps extends StandardProps {
  /** 按钮的大小
   * @supported weapp, h5, rn, alipay
   * @default default
   */
  size?: keyof ButtonProps.Size

  /** 按钮的样式类型
   * @supported weapp, h5, rn, alipay
   * @default default
   */
  type?: keyof ButtonProps.Type

  /** 按钮是否镂空，背景色透明
   * @supported weapp, h5, rn, alipay
   * @default false
   */
  plain?: boolean

  /** 是否禁用
   * @supported weapp, h5, rn, alipay
   * @default false
   */
  disabled?: boolean

  /** 名称前是否带 loading 图标
   * @supported weapp, h5, rn, alipay
   * @default false
   */
  loading?: boolean

  /** 用于 `<form/>` 组件，点击分别会触发 `<form/>` 组件的 submit/reset 事件
   * @supported weapp, alipay
   */
  formType?: keyof ButtonProps.FormType

  /** 微信开放能力
   * @supported weapp, alipay, qq
   */
  openType?: ButtonProps.OpenType

  /** 指定按下去的样式类。当 `hover-class="none"` 时，没有点击态效果
   * @default button-hover
   * @supported weapp, alipay, h5
   * @rn 支持 hoverStyle 属性，但框架未支持 hoverClass
   */
  hoverClass?: string

  /** 由于 RN 不支持 Class，故 RN 端的 Button 组件实现了 `hoverStyle`属性，写法和 style 类似，只不过 `hoverStyle` 的样式是指定按下去的样式。
   * @default none
   * @supported rn
   */
  hoverStyle?: StyleProp<ViewStyle>

  /** 指定是否阻止本节点的祖先节点出现点击态
   * @default false
   * @supported weapp, alipay
   */
  hoverStopPropagation?: boolean

  /** 按住后多久出现点击态，单位毫秒
   * @default 20
   * @supported weapp, alipay, h5, rn
   */
  hoverStartTime?: number

  /** 手指松开后点击态保留时间，单位毫秒
   * @default 70
   * @supported weapp, alipay, h5, rn
   */
  hoverStayTime?: number

  /** 指定返回用户信息的语言，zh_CN 简体中文，zh_TW 繁体中文，en 英文。
   *
   * 生效时机: `open-type="getUserInfo"`
   * @supported weapp
   */
  lang?: keyof ButtonProps.Lang

  /** 会话来源
   *
   * 生效时机：`open-type="contact"`
   * @supported weapp
   */
  sessionFrom?: string

  /** 会话内消息卡片标题
   *
   * 生效时机：`open-type="contact"`
   * @default 当前标题
   * @supported weapp
   */
  sendMessageTitle?: string

  /** 会话内消息卡片点击跳转小程序路径
   *
   * 生效时机：`open-type="contact"`
   * @default 当前标题
   * @supported weapp
   */
  sendMessagePath?: string

  /** 会话内消息卡片图片
   *
   * 生效时机：`open-type="contact"`
   * @default 截图
   * @supported weapp
   */
  sendMessageImg?: string

  /** 打开 APP 时，向 APP 传递的参数
   *
   * 生效时机：`open-type="launchApp"`
   * @supported weapp, qq
   */
  appParameter?: string

  /** 微信小程序子商户能力中，添加 business-id 字段指向子商户
   *
   * 生效时机：`open-type="contact"`
   * @supported weapp
   */
  businessId?: string

  /** 支付宝小程序 scope
   *
   * 生效时机：`open-type="getAuthorize"`
   * @supported weapp
   */
  scope?: 'userInfo' | 'phoneNumber'

  /** 显示会话内消息卡片
   *
   * 生效时机：`open-type="contact"`
   * @default false
   */
  showMessageCard?: boolean

  /** 用户点击该按钮时，会返回获取到的用户信息，回调的detail数据与 Taro.getUserInfo 返回的一致
   *
   * 生效时机: `open-type="getUserInfo"`
   * @supported weapp
   */
  onGetUserInfo?: CommonEventFunction<ButtonProps.onGetUserInfoEventDetail>

  /** 支付宝获取会员基础信息授权回调
   *
   * 生效时机：`open-type="getAuthorize"`
   * @supported alipay
   */
  onGetAuthorize?: CommonEventFunction

  /** 客服消息回调
   *
   * 生效时机：`open-type="contact"`
   * @supported weapp
   */
  onContact?: CommonEventFunction<ButtonProps.onContactEventDetail>

  /** 获取用户手机号回调
   *
   * 生效时机：`open-type="getphonenumber"`
   * @supported weapp
   */
  onGetPhoneNumber?: CommonEventFunction<ButtonProps.onGetPhoneNumberEventDetail>

  /** 获取头像信息
   *
   * 生效时机：`open-type="chooseavatar"`
   * @supported weapp
   */
  onChooseAvatar?: CommonEventFunction<ButtonProps.onChooseAvatarEventDetail>

  /** 获取用户实名
   *
   * 生效时机：`open-type="getRealnameAuthInfo"`
   * @supported weapp
   */
  onGetRealnameAuthInfo?: CommonEventFunction

  /** 当使用开放能力时，发生错误的回调
   *
   * 生效时机：`open-type="launchApp"`
   * @supported weapp
   */
  onError?: CommonEventFunction

  /** 在打开授权设置页后回调
   *
   * 生效时机：`open-type="openSetting"`
   * @supported weapp
   */
  onOpenSetting?: CommonEventFunction<ButtonProps.onOpenSettingEventDetail>

  /** 打开 APP 成功的回调
   *
   * 生效时机：`open-type="launchApp"`
   * @supported weapp
   */
  onLaunchapp?: CommonEventFunction
}

declare namespace ButtonProps {
  /** size 的合法值 */
  interface Size {
    /** 默认大小 */
    default
    /** 小尺寸 */
    mini
  }
  /** type 的合法值 */
  interface Type {
    /** 绿色 */
    primary
    /** 白色 */
    default
    /** 红色 */
    warn
  }
  /** form-type 的合法值 */
  interface FormType {
    /** 提交表单 */
    submit
    /** 重置表单 */
    reset
  }
  /** open-type 的合法值 */
  type OpenType = keyof openTypeKeys["weapp"] | keyof openTypeKeys["alipay"] | keyof openTypeKeys["qq"]
  /** open-type 的合法值 */
  interface openTypeKeys {
    weapp: {
      /** 打开客服会话，如果用户在会话中点击消息卡片后返回小程序，可以从 bindcontact 回调中获得具体信息
       * @see https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/customer-message/customer-message.html
       */
      contact
      /** 触发用户转发，使用前建议先阅读使用指引
       * @see https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/share.html#%E4%BD%BF%E7%94%A8%E6%8C%87%E5%BC%95
       */
      share
      /** 获取用户手机号，可以从 bindgetphonenumber 回调中获取到用户信息
       * @see https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/getPhoneNumber.html
       */
      getPhoneNumber
      /** 获取用户头像，可以从 bindchooseavatar 回调中获取到头像信息
       * @see https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/userProfile.html
       */
      chooseAvatar
      /** 获取用户信息，可以从 bindgetuserinfo 回调中获取到用户信息 */
      getUserInfo
      /** 用户实名信息授权，已经弃用 */
      getRealnameAuthInfo
      /** 打开APP，可以通过app-parameter属性设定向APP传的参数
       * @see https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/launchApp.html
       */
      launchApp
      /** 打开授权设置页 */
      openSetting
      /** 打开“意见反馈”页面，用户可提交反馈内容并上传日志，开发者可以登录小程序管理后台后进入左侧菜单“客服反馈”页面获取到反馈内容 */
      feedback
    }
    /** 支付宝小程序专属的 open-type 合法值
     * @see https://opendocs.alipay.com/mini/component/button
     */
    alipay: {
      /** 触发 自定义分享 */
      share
      /** 支持小程序授权 */
      getAuthorize
      /** 分享到通讯录好友 */
      contactShare
      /** 关注生活号 */
      lifestyle
    }
    /** QQ 小程序专属的 open-type 合法值
     * @see https://q.qq.com/wiki/develop/miniprogram/component/form/button.html
     */
    qq: {
      /** 触发用户转发，使用前建议先阅读使用指引
       * @see https://q.qq.com/wiki/develop/miniprogram/frame/open_ability/open_share.html#%E8%BD%AC%E5%8F%91-2
       */
      share
      /** 获取用户信息，可以从 onGetUserInfo 回调中获取到用户信息 */
      getUserInfo
      /** 打开APP，可以通过 app-parameter 属性设定向APP传的参数
       * @see https://q.qq.com/wiki/develop/miniprogram/frame/open_ability/open_app.html
       */
      launchApp
      /** 打开授权设置页 */
      openSetting
      /** 呼起吐个槽反馈页面，开发者可以到官网查看反馈 */
      feedback
      /** 呼起群资料卡页面，可以通过 group-id 属性设定需要打开的群资料卡的群号，同时 app.json 中必须配置 groupIdList（数量不超过 10 个），表明可以打开群资料卡的群号 */
      openGroupProfile
      /** 添加好友，对方需要通过该小程序进行授权，允许被加好友后才能调用成功[用户授权](https://q.qq.com/wiki/develop/miniprogram/frame/open_ability/open_userinfo.html#%E6%8E%88%E6%9D%83) */
      addFriend
      /** 添加彩签，点击后添加状态有用户提示，无回调 */
      addColorSign
      /** 打开公众号资料卡，可以通过 public-id 属性设定需要打开的公众号资料卡的号码，同时 app.json 中必须配置 publicIdList（目前只能配置 1 个），表明可以打开的公众号资料卡的号码 */
      openPublicProfile
      /** 添加群应用（只有管理员或群主有权操作，建议先调用 qq.getGroupInfo 获取当前用户是否为管理员，如果是管理员才显示该按钮），添加后给button绑定 onAddGroupApp 事件接收回调数据。 */
      addGroupApp
      /** 在自定义开放数据域组件中,向指定好友发起分享据 */
      shareMessageToFriend
    }
  }
  /** lang 的合法值 */
  interface Lang {
    /** 英文 */
    en
    /** 简体中文 */
    zh_CN
    /** 繁体中文 */
    zh_TW
  }

  interface onGetUserInfoEventDetail {
    userInfo: {
      /** 昵称 */
      nickName: string
      /** 头像 */
      avatarUrl: string
      /** 性别 */
      gender: keyof Gender
      /** 省份，如：`Yunnan` */
      province: string
      /** 城市，如：`Dalian` */
      city: string
      /** 国家，如：`China` */
      country: string
    }
    /** 不包括敏感信息的原始数据 `JSON` 字符串，用于计算签名 */
    rawData: string
    /** 使用 `sha1(rawData + sessionkey)` 得到字符串，用于校验用户信息 */
    signature: string
    /** 包括敏感数据在内的完整用户信息的加密数据 */
    encryptedData: string
    /** 加密算法的初始向量 */
    iv: string
    /* 用户信息的调用状态 */
    errMsg: string
    /** 敏感数据对应的云 ID，开通[云开发](https://developers.weixin.qq.com/miniprogram/dev/wxcloud/basis/getting-started.html)的小程序才会返回，可通过云调用直接获取开放数据，详细见[云调用直接获取开放数据](https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/signature.html#method-cloud) */
    cloudID?: string
  }

  /** 性别的合法值 */
  interface Gender {
    /** 未知 */
    0
    /** 男 */
    1
    /** 女 */
    2
  }

  interface onContactEventDetail {
    /* 小程序消息的调用状态 */
    errMsg: string
    /** 小程序消息指定的路径 */
    path: string
    /** 小程序消息指定的查询参数 */
    query: Record<string, any>
  }

  interface onGetPhoneNumberEventDetail {
    /* 获取用户手机号的调用状态 */
    errMsg: string
    /** 包括敏感数据在内的完整用户信息的加密数据 */
    encryptedData: string
    /** 加密算法的初始向量 */
    iv: string
    /** 动态令牌。可通过动态令牌换取用户手机号。使用方法详情 [phonenumber.getPhoneNumber](https://developers.weixin.qq.com/miniprogram/dev/api-backend/open-api/phonenumber/phonenumber.getPhoneNumber.html) 接口 */
    code:string
  }

  interface onChooseAvatarEventDetail {
    /* 获取用户头像的临时链接 */
    avatarUrl: string
  }

  interface onOpenSettingEventDetail {
    /* 打开授权设置页的调用状态 */
    errMsg: string
    /* 用户授权结果 */
    authSetting: Record<string, boolean>
  }
}

/** 按钮
 * @classification forms
 * @supported weapp, h5, rn
 * @example_react
 * ```tsx
 * export default class PageButton extends Component {
 *   state = {
 *     btn: [
 *       {
 *         text: '页面主操作 Normal',
 *         size: 'default',
 *         type: 'primary'
 *       },
 *       {
 *         text: '页面主操作 Loading',
 *         size: 'default',
 *         type: 'primary',
 *         loading: true,
 *       },
 *       {
 *         text: '页面主操作 Disabled',
 *         size: 'default',
 *         type: 'primary',
 *         disabled: true,
 *       },
 *       {
 *         text: '页面次要操作 Normal',
 *         size: 'default',
 *         type: 'default'
 *       },
 *       {
 *         text: '页面次要操作 Disabled',
 *         size: 'default',
 *         type: 'default',
 *         disabled: true,
 *       },
 *       {
 *         text: '警告类操作 Normal',
 *         size: 'default',
 *         type: 'warn'
 *       },
 *       {
 *         text: '警告类操作 Disabled',
 *         size: 'default',
 *         type: 'warn',
 *         disabled: true,
 *       }
 *     ]
 *   }
 *   render () {
 *     return (
 *       <View className='container'>
 *         {this.state.btn.map(item => {
 *           return (
 *             <Button
 *               size={item.size ? item.size : ''}
 *               type={item.type ? item.type : ''}
 *               loading={item.loading ? item.loading : false}
 *               disabled={item.disabled ? item.disabled : false}
 *             >
 *               {item.text}
 *             </Button>
 *           )
 *         })}
 *         <Button className='btn-max-w' plain type='primary'>按钮</Button>
 *         <Button className='btn-max-w' plain type='primary' disabled>不可点击的按钮</Button>
 *         <Button className='btn-max-w' plain >按钮</Button>
 *         <Button className='btn-max-w' plain disabled >按钮</Button>
 *         <Button size='mini' type='primary'>按钮</Button>
 *         <Button size='mini' >按钮</Button>
 *         <Button size='mini' type='warn'>按钮</Button>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="container">
 *     <button
 *       v-for="item in btn"
 *       :size="item.size ? item.size : ''"
 *       :type="item.type ? item.type : ''"
 *       :loading="item.loading ? item.loading : false"
 *       :disabled="item.disabled ? item.disabled : false"
 *     >
 *       {{ item.text }}
 *     </button>
 *     <button class="btn-max-w" :plain="true" type="primary">按钮</button>
 *     <button class="btn-max-w" :plain="true" type="primary" :disabled="true">不可点击的按钮</button>
 *     <button class="btn-max-w" :plain="true">按钮</button>
 *     <button class="btn-max-w" :plain="true" :disabled="true">按钮</button>
 *     <button size="mini" type="primary">按钮</button>
 *     <button size="mini" >按钮</button>
 *     <button size="mini" type="warn">按钮</button>
 *   </view>
 * </template>
 *
 * <script>
 * export default {
 *   data() {
 *     return {
 *       btn: [
 *         {
 *           text: '页面主操作 Normal',
 *           size: 'default',
 *           type: 'primary'
 *         },
 *         {
 *           text: '页面主操作 Loading',
 *           size: 'default',
 *           type: 'primary',
 *           loading: true,
 *         },
 *         {
 *           text: '页面主操作 Disabled',
 *           size: 'default',
 *           type: 'primary',
 *           disabled: true,
 *         },
 *         {
 *           text: '页面次要操作 Normal',
 *           size: 'default',
 *           type: 'default'
 *         },
 *         {
 *           text: '页面次要操作 Disabled',
 *           size: 'default',
 *           type: 'default',
 *           disabled: true,
 *         },
 *         {
 *           text: '警告类操作 Normal',
 *           size: 'default',
 *           type: 'warn'
 *         },
 *         {
 *           text: '警告类操作 Disabled',
 *           size: 'default',
 *           type: 'warn',
 *           disabled: true,
 *         }
 *       ]
 *     }
 *   }
 * }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/button.html
 */
declare const Button: ComponentType<ButtonProps>

interface CheckboxProps extends StandardProps {

  /** `<Checkbox/>`标识，选中时触发`<CheckboxGroup/>`的 change 事件，并携带 `<Checkbox/>` 的 value
   * @supported weapp, rn
   */
  value: string

  /** 是否禁用
   * @supported weapp, h5, rn
   * @default false
   */
  disabled?: boolean

  /** 当前是否选中，可用来设置默认选中
   * @supported weapp, h5, rn
   * @default false
   */
  checked?: boolean

  /** checkbox的颜色，同 css 的 color
   * @supported weapp, h5, rn
   */
  color?: string

  /** 选中项发生变化时触发 change 事件，小程序无此 API
   * @supported h5, rn
   */
  onChange?: CommonEventFunction<{ value: string[] }>

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

/** 多选项目
 * @classification forms
 * @supported weapp, h5, rn
 * @example_react
 * ```tsx
 * export default class PageCheckbox extends Component {
 *   state = {
 *     list: [
 *       {
 *         value: '美国',
 *         text: '美国',
 *         checked: false
 *       },
 *       {
 *         value: '中国',
 *         text: '中国',
 *         checked: true
 *       },
 *       {
 *         value: '巴西',
 *         text: '巴西',
 *         checked: false
 *       },
 *       {
 *         value: '日本',
 *         text: '日本',
 *         checked: false
 *       },
 *       {
 *         value: '英国',
 *         text: '英国',
 *         checked: false
 *       },
 *       {
 *         value: '法国',
 *         text: '法国',
 *         checked: false
 *       }
 *     ]
 *   }
 *   render () {
 *     return (
 *       <View className='page-body'>
 *         <View className='page-section'>
 *           <Text>默认样式</Text>
 *           <Checkbox value='选中' checked>选中</Checkbox>
 *           <Checkbox style='margin-left: 20rpx' value='未选中'>未选中</Checkbox>
 *         </View>
 *         <View className='page-section'>
 *           <Text>推荐展示样式</Text>
 *           {this.state.list.map((item, i) => {
 *             return (
 *               <Label className='checkbox-list__label' for={i} key={i}>
 *                 <Checkbox className='checkbox-list__checkbox' value={item.value} checked={item.checked}>{item.text}</Checkbox>
 *               </Label>
 *             )
 *           })}
 *         </View>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="container">
 *     <view class="page-section">
 *       <text>默认样式</text>
 *       <checkbox value="选中" :checked="true">选中</checkbox>
 *       <checkbox style="margin-left: 20rpx;" value="未选中">未选中</checkbox>
 *     </view>
 *     <view class="page-section">
 *       <text>推荐展示样式(Taro 团队成员):</text>
 *       <label v-for="item in list" class="checkbox-list__label">
 *         <checkbox class="checkbox-list__checkbox" :value="item.value" :checked="item.checked">{{ item.text }}</checkbox>
 *       </label>
 *     </view>
 *   </view>
 * </template>
 * 
 * <script>
 * export default {
 *   data() {
 *     return {
 *       list: [
 *         {
 *           value: '美国',
 *           text: '美国',
 *           checked: false
 *         },
 *         {
 *           value: '中国',
 *           text: '中国',
 *           checked: true
 *         },
 *         {
 *           value: '巴西',
 *           text: '巴西',
 *           checked: false
 *         },
 *         {
 *           value: '日本',
 *           text: '日本',
 *           checked: false
 *         },
 *         {
 *           value: '英国',
 *           text: '英国',
 *           checked: false
 *         },
 *         {
 *           value: '法国',
 *           text: '法国',
 *           checked: false
 *         }
 *       ]
 *     }
 *   }
 * }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/checkbox.html
 */
declare const Checkbox: ComponentType<CheckboxProps>

interface CheckboxGroupProps extends StandardProps, FormItemProps {
  /** 表单组件中加上 name 来作为 key
   * @supported h5
   */
  name?: string

  /** `<CheckboxGroup/>` 中选中项发生改变是触发 change 事件
   * 
   * event.detail = { value: [选中的checkbox的 value 的数组] }
   * @supported weapp, h5, rn
   */
  onChange?: CommonEventFunction<{ value: string[] }>
}

/** 多项选择器，内部由多个checkbox组成
 * @classification forms
 * @supported weapp, h5, rn
 * @example
 * ```tsx
 * export default class PageCheckbox extends Component {
 *   state = {
 *     list: [
 *       {
 *         value: '美国',
 *         text: '美国',
 *         checked: false
 *       },
 *       {
 *         value: '中国',
 *         text: '中国',
 *         checked: true
 *       },
 *       {
 *         value: '巴西',
 *         text: '巴西',
 *         checked: false
 *       },
 *       {
 *         value: '日本',
 *         text: '日本',
 *         checked: false
 *       },
 *       {
 *         value: '英国',
 *         text: '英国',
 *         checked: false
 *       },
 *       {
 *         value: '法国',
 *         text: '法国',
 *         checked: false
 *       }
 *     ]
 *   }
 *   render () {
 *     return (
 *       <View className='page-body'>
 *         <View className='page-section'>
 *           <Text>默认样式</Text>
 *           <Checkbox value='选中' checked>选中</Checkbox>
 *           <Checkbox style='margin-left: 20rpx' value='未选中'>未选中</Checkbox>
 *         </View>
 *         <View className='page-section'>
 *           <Text>推荐展示样式</Text>
 *           {this.state.list.map((item, i) => {
 *             return (
 *               <Label className='checkbox-list__label' for={i} key={i}>
 *                 <Checkbox className='checkbox-list__checkbox' value={item.value} checked={item.checked}>{item.text}</Checkbox>
 *               </Label>
 *             )
 *           })}
 *         </View>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/checkbox-group.html
 */
declare const CheckboxGroup: ComponentType<CheckboxGroupProps>

interface EditorProps extends StandardProps {
  /** 设置编辑器为只读
   * @default false
   * @supported weapp
   */
  readOnly?: boolean

  /** 提示信息
   * @supported weapp
   */
  placeholder?: string

  /** 点击图片时显示图片大小控件
   * @default false
   * @supported weapp
   */
  showImgSize?: boolean

  /** 点击图片时显示工具栏控件
   * @default false
   * @supported weapp
   */
  showImgToolbar?: boolean

  /** 点击图片时显示修改尺寸控件
   * @default false
   * @supported weapp
   */	
  showImgResize?: boolean

  /** 编辑器初始化完成时触发
   * @supported weapp
   */
  onReady?: CommonEventFunction

  /** 编辑器聚焦时触发
   * event.detail = { html, text, delta }
   * @supported weapp
   */
  onFocus?: CommonEventFunction<EditorProps.editorEventDetail>

  /** 编辑器失去焦点时触发
   * detail = { html, text, delta }
   * @supported weapp
   */
  onBlur?: CommonEventFunction<EditorProps.editorEventDetail>

  /** 编辑器内容改变时触发
   * detail = { html, text, delta }
   * @supported weapp
   */
  onInput?: CommonEventFunction<EditorProps.editorEventDetail>

  /** 通过 Context 方法改变编辑器内样式时触发，返回选区已设置的样式
   * @supported weapp
   */
  onStatuschange?: CommonEventFunction
}

declare namespace EditorProps {
  interface editorEventDetail {
    html, text, delta
  }
}

/** 富文本编辑器，可以对图片、文字进行编辑。
 * 
 * 编辑器导出内容支持带标签的 html和纯文本的 text，编辑器内部采用 delta 格式进行存储。
 * 
 * 通过 setContents 接口设置内容时，解析插入的 html 可能会由于一些非法标签导致解析错误，建议开发者在小程序内使用时通过 delta 进行插入。
 * 
 * 富文本组件内部引入了一些基本的样式使得内容可以正确的展示，开发时可以进行覆盖。需要注意的是，在其它组件或环境中使用富文本组件导出的 html 时，需要额外引入 这段样式，并维护 `<ql-container><ql-editor></ql-editor></ql-container>` 的结构。
 * 
 * 图片控件仅初始化时设置有效。
 * 
 * *编辑器内支持部分 HTML 标签和内联样式，不支持 **class** 和 **id***
 * @classification forms
 * @supported weapp
 * @example_react
 * ```tsx
 * class App extends Components {
 *   state = {
 *     placeholder: '来，输入隔壁的名字试试...'
 *   }
 * 
 *   editorReady = e => {
 *     Taro.createSelectorQuery().select('#editor').context((res) => {
 *       this.editorCtx = res.context
 *     }).exec()
 *   }
 * 
 *   undo = e => {
 *     this.editorCtx.undo()
 *   }
 * 
 *   render () {
 *     return (
 *       <View>
 *         <Editor id='editor' className='editor' placeholder={this.state.placeholder} onReady={this.editorReady}></Editor>
 *         <Button type='warn' onClick={this.undo}>撤销</Button>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="container">
 *     <editor id="editor" class="editor" :placeholder="placeholder" `@ready="editorReady"></editor>
 *     <button type="warn" `@tap="undo">撤销</button>
 *   </view>
 * </template>
 * 
 * <script>
 *   import Taro from '@tarojs/taro'
 *   export default {
 *     data() {
 *       return {
 *         placeholder: '来，输入隔壁的名字试试...'
 *       }
 *     },
 *     methods: {
 *       editorReady() {
 *         Taro.createSelectorQuery().select('#editor').context((res) => {
 *           this.editorCtx = res.context
 *         }).exec()
 *       },
 *       undo() {
 *         this.editorCtx.undo()
 *       }
 *     }
 *   }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/editor.html
 */
declare const Editor: ComponentType<EditorProps>

interface FormProps extends StandardProps {
  /** 是否返回 `formId` 用于发送模板消息。
   * @default false
   * @supported weapp, h5
   */
  reportSubmit?: boolean

  /** 等待一段时间（毫秒数）以确认 `formId` 是否生效。
   * 如果未指定这个参数，`formId` 有很小的概率是无效的（如遇到网络失败的情况）。
   * 指定这个参数将可以检测 `formId` 是否有效，
   * 以这个参数的时间作为这项检测的超时时间。
   * 如果失败，将返回 `requestFormId:fail` 开头的 `formId`。
   * @default 0
   * @supported weapp
   */
  reportSubmitTimeout?: number

  /** 携带 form 中的数据触发 submit 事件
   * event.detail = { value : {'name': 'value'} , formId: '' }
   * @supported weapp, rn
   */
  onSubmit?: CommonEventFunction<FormProps.onSubmitEventDetail>

  /** 表单重置时会触发 reset 事件
   * @supported weapp, h5, rn
   */
  onReset?: CommonEventFunction
}

declare namespace FormProps {
  interface onSubmitEventDetail {
    /** 当点击 `<form>` 表单中 `form-type` 为 `submit` 的 `<button>` 组件时，
     * 会将表单组件中的 `value` 值进行提交，
     * 需要在表单组件中加上 `name` 来作为 `key`。
     */
    value?: {
      [formItemName: string]: any
    }
    /** 当 `reportSubmit` 为 `true` 时，返回 `formId` 用于发送模板消息。
     */
    formId?: string
  }
}

/** 表单。将组件内的用户输入的 switch input checkbox slider radio picker 提交。
 * 
 * 当点击 form 表单中 form-type 为 submit 的 button 组件时，会将表单组件中的 value 值进行提交，需要在表单组件中加上 name 来作为 key。
 * @classification forms
 * @supported weapp, h5, rn
 * @example_react
 * ```tsx
 * class App extends Component {
 * 
 *   formSubmit = e => {
 *     console.log(e)
 *   }
 * 
 *   formReset = e => {
 *     console.log(e)
 *   }
 * 
 *   render () {
 *     return (
 *       <Form onSubmit={this.formSubmit} onReset={this.formReset} >
 *         <View className='example-body'>
 *           <Switch name='switch' className='form-switch'></Switch>
 *         </View>
 *       </Form>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <form `@submit="formSubmit" `@reset="formReset" >
 *       <view class="taro-example-body">
 *         <switch name="switch" class="form-switch"></Switch>
 *       </view>
 *       <view class="taro-example-btns">
 *         <button form-type="submit">Submit</button>
 *         <button type="default" form-type="reset">Reset</button>
 *     </view>
 *   </form>
 * </template>
 * 
 * <script>
 * export default {
 *   data() {
 *     return {}
 *   },
 *   methods: {
 *     formSubmit (e) {
 *       console.log(e)
 *     },
 * 
 *     formReset (e) {
 *       console.log(e)
 *     }
 *   }
 * }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/form.html
 */
declare const Form: ComponentType<FormProps>

interface InputProps extends StandardProps, FormItemProps {
  /** 输入框的初始内容
   * @supported weapp, h5, rn
   */
  value?: string

  /** input 的类型
   * @default "text"
   * @supported weapp, h5, rn
   * @rn 部分支持
   */
  type?: 'text' | 'number' | 'idcard' | 'digit' | 'safe-password' | 'nickname'

  /** 是否是密码类型
   * @supported weapp, h5, rn
   */
  password?: boolean

  /** 输入框为空时占位符
   * @supported weapp, h5, rn
   */
  placeholder?: string

  /** 指定 placeholder 的样式
   * @supported weapp
   */
  placeholderStyle?: string

  /** 指定 placeholder 的样式类
   * @default "input-placeholder"
   * @supported weapp
   */
  placeholderClass?: string

  /** 是否禁用
   * @supported weapp, h5, rn
   */
  disabled?: boolean

  /** 最大输入长度，设置为 -1 的时候不限制最大长度
   * @default 140
   * @supported weapp, h5, rn
   */
  maxlength?: number

  /** 指定光标与键盘的距离，单位 px 。取 input 距离底部的距离和 cursor-spacing 指定的距离的最小值作为光标与键盘的距离
   * @default 0
   * @supported weapp
   */
  cursorSpacing?: number

  /** (即将废弃，请直接使用 focus )自动聚焦，拉起键盘
   * @default false
   * @supported weapp
   */
  autoFocus?: boolean

  /** 获取焦点
   * @supported weapp, h5, rn
   */
  focus?: boolean

  /** 设置键盘右下角按钮的文字
   * @default done
   * @supported weapp, rn
   */
  confirmType?: 'send' | 'search' | 'next' | 'go' | 'done'

  /** 点击键盘右下角按钮时是否保持键盘不收起
   * @default false
   * @supported weapp
   */
  confirmHold?: boolean

  /** 指定focus时的光标位置
   * @supported weapp, rn
   */
  cursor?: number

  /** 光标起始位置，自动聚集时有效，需与selection-end搭配使用
   * @default -1
   * @supported weapp, rn
   */
  selectionStart?: number

  /** 光标结束位置，自动聚集时有效，需与selection-start搭配使用
   * @default -1
   * @supported weapp, rn
   */
  selectionEnd?: number

  /** 键盘弹起时，是否自动上推页面
   * @default false
   * @supported weapp
   */
  adjustPosition?: boolean

  /** focus 时，点击页面的时候不收起键盘
   * @default false
   * @supported weapp
   */
  holdKeyboard?: boolean

  /**
   * 强制 input 处于同层状态，默认 focus 时 input 会切到非同层状态 (仅在 iOS 下生效)
   * @default false
   * @supported weapp
   */
  alwaysEmbed?: boolean

  /**
   * 安全键盘加密公钥的路径，只支持包内路径
   * @supported weapp
   */
  safePasswordCertPath?: string

  /**
   * 安全键盘输入密码长度
   * @supported weapp
   */
  safePasswordLength?: number

  /**
   * 安全键盘加密时间戳
   * @supported weapp
   */
  safePasswordTimeStamp?: number

  /**
   * 安全键盘加密盐值
   * @supported weapp
   */
  safePasswordNonce?: string
  /**
   * 安全键盘计算hash盐值，若指定custom-hash 则无效
   * @supported weapp
   */
  safePasswordSalt?: string

  /**
   * 安全键盘计算hash的算法表达式，如 `md5(sha1('foo' + sha256(sm3(password + 'bar'))))`
   * @supported weapp
   */
  safePasswordCustomHash?: string

  /**
   * 当 type 为 number, digit, idcard 数字键盘是否随机排列
   * @default false
   * @supported alipay
   */
  randomNumber?: boolean

  /**
   * 是否为受控组件
   * @default false
   * @supported alipay
   */
  controlled?: boolean

  /** 当键盘输入时，触发input事件，event.detail = {value, cursor, keyCode}，处理函数可以直接 return 一个字符串，将替换输入框的内容。
   * @supported weapp, h5, rn
   */
  onInput?: CommonEventFunction<InputProps.inputEventDetail>

  /** 输入框聚焦时触发，event.detail = { value, height }，height 为键盘高度
   * @supported weapp, h5, rn
   */
  onFocus?: CommonEventFunction<InputProps.inputForceEventDetail>

  /** 输入框失去焦点时触发
   *
   * event.detail = {value: value}
   * @supported weapp, h5, rn
   */
  onBlur?: CommonEventFunction<InputProps.inputValueEventDetail>

  /** 点击完成按钮时触发
   *
   * event.detail = {value: value}
   * @supported weapp, rn
   * @h5 借用[Form 组件](./form)的`onSubmit`事件来替代
   */
  onConfirm?: CommonEventFunction<InputProps.inputValueEventDetail>

  /** 键盘高度发生变化的时候触发此事件
   *
   * event.detail = {height: height, duration: duration}
   * @supported weapp
   */
  onKeyboardHeightChange?: CommonEventFunction<InputProps.onKeyboardHeightChangeEventDetail>

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

declare namespace InputProps {
  /** > 注意：React-Native 端 `inputEventDetail` 仅实现参数 `value`，若需实时获取光标位置则可通过 [`onSelectionChange`](https://reactnative.dev/docs/textinput#onselectionchange) 实现。 */
  interface inputEventDetail {
    /** 输入值 */
    value: string
    /** 光标位置 */
    cursor: number
    /** 键值 */
    keyCode: number
  }
  interface inputForceEventDetail {
    /** 输入值 */
    value: string
    /** 键盘高度 */
    height: number
  }
  interface inputValueEventDetail {
    /** 输入值 */
    value: string
  }

  interface onKeyboardHeightChangeEventDetail {
    /** 键盘高度 */
    height: number
    /** 持续时间 */
    duration: number
  }
}

/** 输入框。该组件是原生组件，使用时请注意相关限制
 * @classification forms
 * @supported weapp, h5, rn
 * @example_react
 * ```tsx
 * class App extends Component {
 *
 *   render () {
 *     return (
 *         <View className='example-body'>
 *           <Text>可以自动聚焦的 input</Text>
 *             <Input type='text' placeholder='将会获取焦点' focus/>
 *             <Text>控制最大输入长度的 input</Text>
 *             <Input type='text' placeholder='最大输入长度为 10' maxlength='10'/>
 *             <Text>数字输入的 input</Text>
 *             <Input type='number' placeholder='这是一个数字输入框'/>
 *             <Text>密码输入的 input</Text>
 *             <Input type='password' password placeholder='这是一个密码输入框'/>
 *             <Text>带小数点的 input</Text>
 *             <Input type='digit' placeholder='带小数点的数字键盘'/>
 *             <Text>身份证输入的 input</Text>
 *             <Input type='idcard' placeholder='身份证输入键盘'/>
 *             <Text>控制占位符颜色的 input</Text>
 *             <Input type='text' placeholder='占位符字体是红色的' placeholderStyle='color:red'/>
 *         </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="example-body">
 *     <text>可以自动聚焦的 input</text>
 *     <input type="text" placeholder="将会获取焦点" :focus="true" />
 *     <text>控制最大输入长度的 input</text>
 *     <input type="text" placeholder="最大输入长度为 10" maxlength="10"/>
 *     <text>数字输入的 input</text>
 *     <input type="number" placeholder="这是一个数字输入框"/>
 *     <text>密码输入的 input</text>
 *     <input type="password" :password="true" placeholder="这是一个密码输入框"/>
 *     <text>带小数点的 input</text>
 *     <input type="digit" placeholder="带小数点的数字键盘"/>
 *     <text>身份证输入的 input</text>
 *     <input type="idcard" placeholder="身份证输入键盘"/>
 *     <text>控制占位符颜色的 input</text>
 *     <input type="text" placeholder="占位符字体是红色的" placeholder-style="color:red;"/>
 *   </view>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/input.html
 */
declare const Input: ComponentType<InputProps>

/** @ignore */
type KeyboardAccessoryProps = StandardProps

/** 设置 input / textarea 聚焦时键盘上方 cover-view / cover-image 工具栏视图
 *
 * @classification forms
 * @supported weapp
 * @example
 * ```tsx
 * class App extends Component {
 *   render () {
 *     return (
 *         <Textarea holdKeyboard="{{true}}">
 *           <KeyboardAccessory className="container" style={{ height: 50 }} >
 *             <CoverView onClick={() => { TODO }} style={{ flex: 1, background: 'green' }}>1</CoverView>
 *             <CoverView onClick={() => { TODO }} style={{ flex: 1, background: 'red' }}>2</CoverView>
 *           </KeyboardAccessory>
 *         </Textarea>
 *     )
 *   }
 * }
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/keyboard-accessory.html
 */
declare const KeyboardAccessory: ComponentType<KeyboardAccessoryProps>

interface LabelProps extends StandardProps {

  /** 绑定控件的 id */
  for?: string
}

/** 用来改进表单组件的可用性。
 * 
 * 使用for属性找到对应的id，或者将控件放在该标签下，当点击时，就会触发对应的控件。 for优先级高于内部控件，内部有多个控件的时候默认触发第一个控件。 目前可以绑定的控件有：button, checkbox, radio, switch。
 * @classification forms
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * class App extends Components {
 * 
 *   render () {
 *     return (
 *       <RadioGroup>
 *         <Label className='example-body__label' for='1' key='1'>
 *           <Radio value='USA'>USA</Radio>
 *         </Label>
 *         <Label className='example-body__label' for='2' key='2'>
 *           <Radio value='CHN' checked>
 *           CHN
 *           </Radio>
 *         </Label>
 *       </RadioGroup>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <radio-group>
 *     <label class="example-body__label" for="1" key="1">
 *       <radio id="1" value="USA" />
 *       USA
 *     </label>
 *     <label class="example-body__label" for="2" key="2">
 *       <radio id="2" value="CHN" :checked="true" />
 *       CHN
 *     </label>
 *   </radio-group>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/label.html
 */
declare const Label: ComponentType<LabelProps>

/** 选择器通用参数 */
interface PickerStandardProps extends StandardProps, FormItemProps {
  /**
   * 选择器类型，默认是普通选择器
   * @default "selector"
   * @supported weapp, h5, rn
   */
  mode?: keyof PickerStandardProps.Mode
  /**
   * 是否禁用
   * @default false
   * @supported weapp, h5, rn
   */
  disabled?: boolean
  /**
   * 取消选择或点遮罩层收起 picker 时触发
   * @supported weapp, h5, rn
   */
  onCancel?: CommonEventFunction
}

declare namespace PickerStandardProps {
  /** 选择器类型 */
  interface Mode {
    /** 普通选择器 */
    selector
    /** 多列选择器 */
    multiSelector
    /** 时间选择器 */
    time
    /** 日期选择器 */
    date
    /** 省市区选择器 */
    region
  }
}

/** 普通选择器：mode = selector */
interface PickerSelectorProps extends PickerStandardProps {
  /** 选择器类型 */
  mode?: 'selector'
  /**
   * mode为 selector 或 multiSelector 时，range 有效
   * @supported weapp, h5, rn
   * @default []
   */
  range: string[] | number[] | Record<string, any>[]
  /**
   * 当 range 是一个 Object Array 时，通过 rangeKey 来指定 Object 中 key 的值作为选择器显示内容
   * @supported weapp, h5, rn
   */
  rangeKey?: string
  /**
   * 表示选择了 range 中的第几个（下标从 0 开始）
   * @supported weapp, h5, rn
   * @default 0
   */
  value?: number
  /**
   * value 改变时触发 change 事件，event.detail = {value}
   * @supported weapp, h5, rn
   */
  onChange: CommonEventFunction<PickerSelectorProps.ChangeEventDetail>
}

declare namespace PickerSelectorProps {
  interface ChangeEventDetail {
    /** 表示变更值的下标 */
    value: string | number
  }
}

/** 多列选择器：mode = multiSelector */
interface PickerMultiSelectorProps extends PickerStandardProps {
  /** 选择器类型 */
  mode: 'multiSelector'
  /**
   * mode为 selector 或 multiSelector 时，range 有效
   * @supported weapp, h5, rn
   * @default []
   */
  range: Array<string[]> | Array<number[]> | Array<Record<string, any>[]>
  /**
   * 当 range 是一个 Object Array 时，通过 rangeKey 来指定 Object 中 key 的值作为选择器显示内容
   * @supported weapp, h5, rn
   */
  rangeKey?: string
  /**
   * 表示选择了 range 中的第几个（下标从 0 开始）
   * @supported weapp, h5, rn
   * @default []
   */
  value: number[] | string[] | Record<string, any>[]
  /**
   * 当 value 改变时触发 change 事件，event.detail = {value}
   * @supported weapp, h5, rn
   */
  onChange: CommonEventFunction<PickerMultiSelectorProps.ChangeEventDetail>
  /**
   * 列改变时触发
   * @supported weapp, h5, rn
   */
  onColumnChange?: CommonEventFunction<PickerMultiSelectorProps.ColumnChangeEventDetail>
}

declare namespace PickerMultiSelectorProps {
  interface ChangeEventDetail {
    /** 表示变更值的下标 */
    value: number[]
  }
  interface ColumnChangeEventDetail {
    /** 表示改变了第几列（下标从0开始） */
    column: number
    /** 表示变更值的下标 */
    value: number
  }
}

/** 时间选择器：mode = time */
interface PickerTimeProps extends PickerStandardProps {
  /** 选择器类型 */
  mode: 'time'
  /**
   * value 的值表示选择了 range 中的第几个（下标从 0 开始）
   * @supported weapp, h5, rn
   */
  value: string
  /**
   * 仅当 mode = time|date 时有效，表示有效时间范围的开始，字符串格式为"hh:mm"
   * @supported weapp, h5, rn
   */
  start?: string
  /**
   * 仅当 mode = time|date 时有效，表示有效时间范围的结束，字符串格式为"hh:mm"
   * @supported weapp, h5, rn
   */
  end?: string
  /**
   * value 改变时触发 change 事件，event.detail = {value}
   * @supported weapp, h5, rn
   */
  onChange: CommonEventFunction<PickerTimeProps.ChangeEventDetail>
}

declare namespace PickerTimeProps {
  interface ChangeEventDetail {
    /** 表示选中的时间 */
    value: string
  }
}

/** 日期选择器：mode = date */
interface PickerDateProps extends PickerStandardProps {
  /** 选择器类型 */
  mode: 'date'
  /**
   * 表示选中的日期，格式为"YYYY-MM-DD"
   * @supported weapp, h5, rn
   * @default 0
   */
  value: string
  /**
   * 仅当 mode = time|date 时有效，表示有效时间范围的开始，字符串格式为"hh:mm"
   * @supported weapp, h5, rn
   */
  start?: string
  /**
   * 仅当 mode = time|date 时有效，表示有效时间范围的结束，字符串格式为"hh:mm"
   * @supported weapp, h5, rn
   */
  end?: string
  /**
   * 有效值 year, month, day，表示选择器的粒度
   * @supported weapp, h5, rn
   * @default "day"
   */
  fields?: keyof PickerDateProps.Fields
  /**
   * value 改变时触发 change 事件，event.detail = {value}
   * @supported weapp, h5, rn
   */
  onChange: CommonEventFunction<PickerDateProps.ChangeEventDetail>
}

declare namespace PickerDateProps {
  interface Fields {
    /** 选择器粒度为年 */
    year
    /** 选择器粒度为月份 */
    month
    /** 选择器粒度为天 */
    day
  }
  interface ChangeEventDetail {
    /** 表示选中的日期 */
    value: string
  }
}

/** 省市区选择器：mode = region */
interface PickerRegionProps extends PickerStandardProps {
  /** 选择器类型 */
  mode: 'region'
  /**
   * 表示选中的省市区，默认选中每一列的第一个值
   * @supported weapp, h5, rn
   * @default []
   */
  value: string[]
  /**
   * 可为每一列的顶部添加一个自定义的项
   * @supported weapp, h5, rn
   */
  customItem?: string
  /**
   * 自定义省市区数据
   * @supported rn
   */
  regionData?: PickerRegionProps.RegionData[]
  /**
   * value 改变时触发 change 事件，event.detail = {value, code, postcode}，其中字段 code 是统计用区划代码，postcode 是邮政编码
   * @supported weapp, h5, rn
   */
  onChange: CommonEventFunction<PickerRegionProps.ChangeEventDetail>
}

declare namespace PickerRegionProps {
  interface ChangeEventDetail {
    /** 表示选中的省市区 */
    value: string[]
    /** 统计用区划代码 */
    code: string[]
    /** 邮政编码 */
    postcode?: string
  }
  interface RegionData {
    value: string
    code: string
    postcode?: string
  }
}

/**
 * 从底部弹起的滚动选择器
 * @classification forms
 * @supported weapp, h5, rn, swan, alipay, tt
 * @example_react
 * ```tsx
 * export default class PagePicker extends Component {
 *   state = {
 *     selector: ['美国', '中国', '巴西', '日本'],
 *     selectorChecked: '美国',
 *     timeSel: '12:01',
 *     dateSel: '2018-04-22'
 *   }
 *
 *   onChange = e => {
 *     this.setState({
 *       selectorChecked: this.state.selector[e.detail.value]
 *     })
 *   }
 *
 *   onTimeChange = e => {
 *     this.setState({
 *       timeSel: e.detail.value
 *     })
 *   }
 *   onDateChange = e => {
 *     this.setState({
 *       dateSel: e.detail.value
 *     })
 *   }
 *
 *   render () {
 *     return (
 *       <View className='container'>
 *         <View className='page-body'>
 *           <View className='page-section'>
 *             <Text>地区选择器</Text>
 *             <View>
 *               <Picker mode='selector' range={this.state.selector} onChange={this.onChange}>
 *                 <View className='picker'>
 *                   当前选择：{this.state.selectorChecked}
 *                 </View>
 *               </Picker>
 *             </View>
 *           </View>
 *           <View className='page-section'>
 *             <Text>时间选择器</Text>
 *             <View>
 *               <Picker mode='time' onChange={this.onTimeChange}>
 *                 <View className='picker'>
 *                   当前选择：{this.state.timeSel}
 *                 </View>
 *               </Picker>
 *             </View>
 *           </View>
 *           <View className='page-section'>
 *             <Text>日期选择器</Text>
 *             <View>
 *               <Picker mode='date' onChange={this.onDateChange}>
 *                 <View className='picker'>
 *                   当前选择：{this.state.dateSel}
 *                 </View>
 *               </Picker>
 *             </View>
 *           </View>
 *         </View>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="page-body">
 *     <view class="page-section">
 *       <text>地区选择器</text>
 *       <view>
 *         <picker mode="selector" :range="selector" `@change="onChange">
 *           <view class="picker">
 *             当前选择：{{selectorChecked}}
 *           </view>
 *         </picker>
 *       </view>
 *     </view>
 *     <view class="page-section">
 *       <text>时间选择器</text>
 *       <view>
 *         <picker mode="time" `@change="onTimeChange">
 *           <view class="picker">
 *             当前选择：{{timeSel}}
 *           </view>
 *         </picker>
 *       </view>
 *     </view>
 *     <view class="page-section">
 *       <text>日期选择器</text>
 *       <view>
 *         <picker mode="date" `@change="onDateChange">
 *           <view class="picker">
 *             当前选择：{{dateSel}}
 *           </view>
 *         </picker>
 *       </view>
 *     </view>
 *   </view>
 * </template>
 * 
 * <script>
 *   export default {
 *     data() {
 *       return {
 *         selector: ['美国', '中国', '巴西', '日本'],
 *         selectorChecked: '美国',
 *         timeSel: '12:01',
 *         dateSel: '2018-04-22'
 *       }
 *     },
 *     methods: {
 *       onChange: function(e) {
 *         this.selectorChecked = this.selector[e.detail.value]
 *       },
 * 
 *       onTimeChange: function(e) {
 *         this.timeSel = e.detail.value
 *       },
 * 
 *       onDateChange: function(e) {
 *         this.dateSel = e.detail.value
 *       }
 *     }
 *   }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/picker.html
 */
declare const Picker: ComponentType<PickerMultiSelectorProps | PickerTimeProps | PickerDateProps | PickerRegionProps | PickerSelectorProps>

interface PickerViewProps extends StandardProps {

  /** 数组中的数字依次表示 picker-view 内的 picker-view-column 选择的第几项（下标从 0 开始），数字大于 picker-view-column 可选项长度时，选择最后一项。
   * @supported weapp, swan, alipay, tt, rn
   */
  value?: number[]

  /** 设置选择器中间选中框的样式
   * @supported weapp, swan, alipay, tt, rn
   */
  indicatorStyle?: string

  /** 设置选择器中间选中框的类名
   * @supported weapp, swan, alipay
   */
  indicatorClass?: string

  /** 设置蒙层的样式
   * @supported weapp, swan, alipay, tt
   */
  maskStyle?: string

  /** 设置蒙层的类名
   * @supported weapp, swan, alipay
   */
  maskClass?: string

  /** 是否在手指松开时立即触发 change 事件。若不开启则会在滚动动画结束后触发 change 事件。
   * @supported weapp
   * @default false
   */
  immediateChange?: boolean

  /** 当滚动选择，value 改变时触发 change 事件，event.detail = {value: value}；value为数组，表示 picker-view 内的 picker-view-column 当前选择的是第几项（下标从 0 开始）
   * @supported weapp, swan, alipay, tt, rn
   */
  onChange?: CommonEventFunction<PickerViewProps.onChangeEventDetail>

  /** 当滚动选择开始时候触发事件
   * @supported weapp
   */
  onPickStart?: CommonEventFunction

  /** 当滚动选择结束时候触发事件
   * @supported weapp
   */
  onPickEnd?: CommonEventFunction
}

declare namespace PickerViewProps {
  interface onChangeEventDetail {
    value: number[]
  }
}

/** 嵌入页面的滚动选择器
 * 其中只可放置 picker-view-column 组件，其它节点不会显示
 * @classification forms
 * @supported weapp, swan, alipay, tt, h5, rn
 * @example_react
 * ```tsx
 * export default class Picks extends Component {
 *
 *   constructor () {
 *     super(...arguments)
 *     const date = new Date()
 *     const years = []
 *     const months = []
 *     const days = []
 *     for (let i = 1990; i <= date.getFullYear(); i++) {
 *       years.push(i)
 *     }
 *     for (let i = 1; i <= 12; i++) {
 *       months.push(i)
 *     }
 *     for (let i = 1; i <= 31; i++) {
 *       days.push(i)
 *     }
 *     this.state = {
 *       years: years,
 *       year: date.getFullYear(),
 *       months: months,
 *       month: 2,
 *       days: days,
 *       day: 2,
 *       value: [9999, 1, 1]
 *     }
 *   }
 *
 *   onChange = e => {
 *     const val = e.detail.value
 *     this.setState({
 *       year: this.state.years[val[0]],
 *       month: this.state.months[val[1]],
 *       day: this.state.days[val[2]],
 *       value: val
 *     })
 *   }
 *
 *   render() {
 *     return (
 *       <View>
 *         <View>{this.state.year}年{this.state.month}月{this.state.day}日</View>
 *         <PickerView indicatorStyle='height: 50px;' style='width: 100%; height: 300px;' value={this.state.value} onChange={this.onChange}>
 *           <PickerViewColumn>
 *             {this.state.years.map(item => {
 *               return (
 *                 <View>{item}年</View>
 *               );
 *             })}
 *           </PickerViewColumn>
 *           <PickerViewColumn>
 *             {this.state.months.map(item => {
 *               return (
 *                 <View>{item}月</View>
 *               )
 *             })}
 *           </PickerViewColumn>
 *           <PickerViewColumn>
 *             {this.state.days.map(item => {
 *               return (
 *                 <View>{item}日</View>
 *               )
 *             })}
 *           </PickerViewColumn>
 *         </PickerView>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="taro-example">
 *   <view>{{year}}年{{month}}月{{day}}日</view>
 *   <picker-view indicator-style="height: 30px;" style="width: 100%; height: 300px;" :value="value" `@change="onChange">
 *     <picker-view-column>
 *       <view v-for="(item, index) in years" :key="index">{{item}}年</view>
 *     </picker-view-column>
 *     <picker-view-column>
 *       <view v-for="(item, index) in months" :key="index">{{item}}月</view>
 *     </picker-view-column>
 *     <picker-view-column>
 *       <view v-for="(item, index) in days" :key="index">{{item}}日</view>
 *     </picker-view-column>
 *   </picker-view>
 * </view>
 * </template>
 *
 * <script>
 *   export default {
 *     name: "Index",
 *     data() {
 *       const date = new Date()
 *       const years = []
 *       const months = []
 *       const days = []
 *       for (let i = 1990; i <= date.getFullYear(); i++) {
 *         years.push(i)
 *       }
 *       for (let i = 1; i <= 12; i++) {
 *         months.push(i)
 *       }
 *       for (let i = 1; i <= 31; i++) {
 *         days.push(i)
 *       }
 *       return {
 *         years: years,
 *         year: date.getFullYear(),
 *         months: months,
 *         month: 2,
 *         days: days,
 *         day: 2,
 *         value: [3, 1, 1]
 *       }
 *     },
 *
 *     methods: {
 *       onChange: function(e) {
 *         const val = e.detail.value
 *         console.log(val)
 *         this.year = this.years[val[0]]
 *         this.month = this.months[val[1]]
 *         this.day = this.days[val[2]]
 *       }
 *     }
 *   }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/picker-view.html
 */
declare const PickerView: ComponentType<PickerViewProps>

/** 滚动选择器子项
 * 仅可放置于 `<PickerView />` 中，其孩子节点的高度会自动设置成与 picker-view 的选中框的高度一致
 * @classification forms
 * @supported weapp, swan, alipay, tt, rn
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/picker-view-column.html
 */
declare const PickerViewColumn: ComponentType<StandardProps>

interface RadioProps extends StandardProps {
  /** `<Radio/>` 标识。当该`<Radio/>` 选中时，`<RadioGroup/>`的 change 事件会携带`<Radio/>`的 value
   * @supported weapp, rn
   */
  value?: string

  /** 当前是否选中
   * @default false
   * @supported weapp, h5, rn
   */
  checked?: boolean

  /** 是否禁用
   * @default false
   * @supported weapp, h5, rn
   */
  disabled?: boolean

  /** Radio 的颜色，同 css 的 color
   * @default "#09BB07"
   * @supported weapp, rn
   */
  color?: string

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

/** 单选项目
 * @classification forms
 * @supported weapp, h5, rn
 * @example_react
 * ```tsx
 * export default class PageRadio extends Component {
 *   state = {
 *     list: [
 *       {
 *         value: '美国',
 *         text: '美国',
 *         checked: false
 *       },
 *       {
 *         value: '中国',
 *         text: '中国',
 *         checked: true
 *       },
 *       {
 *         value: '巴西',
 *         text: '巴西',
 *         checked: false
 *       },
 *       {
 *         value: '日本',
 *         text: '日本',
 *         checked: false
 *       },
 *       {
 *         value: '英国',
 *         text: '英国',
 *         checked: false
 *       },
 *       {
 *         value: '法国',
 *         text: '法国',
 *         checked: false
 *       }
 *     ]
 *   }
 *   render () {
 *     return (
 *       <View className='container'>
 *         <Head title='Radio' />
 *         <View className='page-body'>
 *           <View className='page-section'>
 *             <Text>默认样式</Text>
 *             <Radio value='选中' checked>选中</Radio>
 *             <Radio style='margin-left: 20rpx' value='未选中'>未选中</Radio>
 *           </View>
 *           <View className='page-section'>
 *             <Text>推荐展示样式</Text>
 *             <View className='radio-list'>
 *               <RadioGroup>
 *                 {this.state.list.map((item, i) => {
 *                   return (
 *                     <Label className='radio-list__label' for={i} key={i}>
 *                       <Radio className='radio-list__radio' value={item.value} checked={item.checked}>{item.text}</Radio>
 *                     </Label>
 *                   )
 *                 })}
 *               </RadioGroup>
 *             </View>
 *           </View>
 *         </View>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="container">
 *     <view class="page-section">
 *       <text>默认样式</text>
 *       <radio value="选中" :checked="true">选中</radio>
 *       <radio style="margin-left: 20rpx;" value="未选中">未选中</radio>
 *     </view>
 *     <view class="page-section">
 *       <text>推荐展示样式(Taro 团队成员):</text>
 *         <radio-group `@change="onChange">
 *           <label v-for="item in list" class="checkbox-list__label">
 *             <radio class="checkbox-list__checkbox" :value="item.value" :checked="item.checked">{{ item.text }}</radio>
 *           </label>
 *         </radio-group>
 *     </view>
 *   </view>
 * </template>
 * 
 * <script>
 * export default {
 *   data() {
 *     return {
 *       list: [
 *         {
 *           value: 'yuche',
 *           text: 'yuche',
 *           checked: false
 *         },
 *         {
 *           value: 'cjj',
 *           text: 'cjj',
 *           checked: false
 *         },
 *         {
 *           value: 'xiexiaoli',
 *           text: 'xiexiaoli',
 *           checked: false
 *         },
 *         {
 *           value: 'honly',
 *           text: 'honly',
 *           checked: false
 *         },
 *         {
 *           value: 'cs',
 *           text: 'cs',
 *           checked: false
 *         },
 *         {
 *           value: 'zhutianjian',
 *           text: 'zhutianjian',
 *           checked: false
 *         },
 *         {
 *           value: '隔壁老李',
 *           text: '隔壁老李',
 *           checked: true
 *         }
 *       ]
 *     }
 *   },
 *   methods: {
 *     onChange: function(e) {
 *       console.log(e.detail.value)
 *     }
 *   }
 * }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/radio.html
 */
declare const Radio: ComponentType<RadioProps>

interface RadioGroupProps extends StandardProps, FormItemProps {
  /** RadioGroup 中选中项发生改变时触发 change 事件，detail = {value:[选中的radio的value的数组]}
   * @supported weapp, h5
   */
  onChange?: CommonEventFunction
}

declare namespace RadioGroupProps {
  interface onChangeEventDetail {
    value: string[]
  }
}

/** 单项选择器，内部由多个 Radio 组成。
 * @classification forms
 * @supported weapp, h5, rn
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/radio-group.html
 */
declare const RadioGroup: ComponentType<RadioGroupProps>

interface SliderProps extends StandardProps, FormItemProps {
  /** 最小值
   * @default 0
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  min?: number

  /** 最大值
   * @default 100
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  max?: number

  /** 步长，取值必须大于 0，并且可被(max - min)整除
   * @default 1
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  step?: number

  /** 是否禁用
   * @default false
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  disabled?: boolean

  /** 当前取值
   * @default 0
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  value?: number

  /** 背景条的颜色（请使用 backgroundColor）
   * @default "#e9e9e9"
   * @supported weapp, tt
   */
  color?: string

  /** 已选择的颜色（请使用 activeColor）
   * @default "#1aad19"
   * @supported weapp, tt
   */
  selectedColor?: string

  /** 已选择的颜色
   * @default "#1aad19"
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  activeColor?: string

  /** 背景条的颜色
   * @default "#e9e9e9"
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  backgroundColor?: string

  /** 滑块的大小，取值范围为 12 - 28
   * @default 28
   *  @supported weapp, h5, swan, alipay, tt
   */
  blockSize?: number

  /** 滑块的颜色
   * @default "#ffffff"
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  blockColor?: string

  /** 是否显示当前 value
   * @default false
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  showValue?: boolean

  /** 完成一次拖动后触发的事件
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  onChange?: CommonEventFunction<SliderProps.onChangeEventDetail>

  /** 拖动过程中触发的事件
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  onChanging?: CommonEventFunction<SliderProps.onChangeEventDetail>
}

declare namespace SliderProps {
  interface onChangeEventDetail {
    value
  }
}

/** 滑动选择器
 * @classification forms
 * @supported weapp, h5, rn, swan, alipay, tt
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 * 
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Text>设置 step</Text>
 *         <Slider step={1} value={50}/>
 *         <Text>显示当前 value</Text>
 *         <Slider step={1} value={50} showValue/>
 *         <Text>设置最小/最大值</Text>
 *         <Slider step={1} value={100} showValue min={50} max={200}/>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <text>设置 step</text>
 *     <slider step="1" value="50"/>
 *     <text>显示当前 value</text>
 *     <slider step="1" value="50" :show-value="true" />
 *     <text>设置最小/最大值</text>
 *     <slider step="1" value="100" :show-value="true" min="50" max="200"/>
 *   </view>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/slider.html
 */
declare const Slider: ComponentType<SliderProps>

interface SwitchProps extends StandardProps, FormItemProps {
  /** 是否选中
   * @default false
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  checked?: boolean

  /** 是否禁用
   * @default false
   * @supported weapp, rn
   */
  disabled?: boolean

  /** 样式，有效值：switch, checkbox
   * @default "switch"
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  type?: 'switch' | 'checkbox'

  /** switch 的颜色，同 css 的 color
   * @default "#04BE02"
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  color?: string

  /** checked 改变时触发 change 事件
   * @supported weapp, rn
   */
  onChange?: CommonEventFunction<SwitchProps.onChangeEventDetail>

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

declare namespace SwitchProps {
  interface onChangeEventDetail {
    value: boolean
  }
}

/** 开关选择器
 * @classification forms
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 * 
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Text>默认样式</Text>
 *         <Switch checked/>
 *         <Switch/>
 *         <Text>推荐展示样式</Text>
 *         <Switch checked/>
 *         <Switch/>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class='components-page'>
 *     <text>默认样式</text>
 *     <switch :checked="true" />
 *     <switch />
 *     <text>推荐展示样式</text>
 *     <switch :checked="true" />
 *     <switch />
 *   </view>
 * </template>
 * ```
 * @supported weapp, h5, rn, swan, alipay, tt
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/switch.html
 */
declare const Switch: ComponentType<SwitchProps>

interface TextareaProps extends StandardProps, FormItemProps {
  /** 输入框的内容
   * @supported weapp, h5, rn
   */
  value?: string

  /** 输入框为空时占位符
   * @supported weapp, h5, rn
   */
  placeholder?: string

  /** 指定 placeholder 的样式
   * @supported weapp
   */
  placeholderStyle?: string

  /** 指定 placeholder 的样式类
   * @default "textarea-placeholder"
   * @supported weapp
   */
  placeholderClass?: string

  /** 是否禁用
   * @default false
   * @supported weapp, h5, rn
   */
  disabled?: boolean

  /** 最大输入长度，设置为 -1 的时候不限制最大长度
   * @default 140
   * @supported weapp, h5, rn
   */
  maxlength?: number

  /** 自动聚焦，拉起键盘
   * @default false
   * @supported weapp, h5
   */
  autoFocus?: boolean

  /** 获取焦点
   * @default false
   * @supported weapp, rn
   */
  focus?: boolean

  /** 是否自动增高，设置 autoHeight 时，style.height不生效
   * @default false
   * @supported weapp, rn
   */
  autoHeight?: boolean

  /** 如果 Textarea 是在一个 `position:fixed` 的区域，需要显示指定属性 fixed 为 true
   * @default false
   * @supported weapp
   */
  fixed?: boolean

  /** 指定光标与键盘的距离，单位 px 。取 Textarea 距离底部的距离和 cursorSpacing 指定的距离的最小值作为光标与键盘的距离
   * @default 0
   * @supported weapp
   */
  cursorSpacing?: number

  /** 指定 focus 时的光标位置
   * @default -1
   * @supported weapp
   */
  cursor?: number

  /** 是否显示键盘上方带有”完成“按钮那一栏
   * @default true
   * @supported weapp
   */
  showConfirmBar?: boolean

  /** 光标起始位置，自动聚集时有效，需与 selectionEnd 搭配使用
   * @default -1
   * @supported weapp, rn
   */
  selectionStart?: number

  /** 光标结束位置，自动聚集时有效，需与 selectionStart 搭配使用
   * @default -1
   * @supported weapp, rn
   */
  selectionEnd?: number

  /** 键盘弹起时，是否自动上推页面
   * @default true
   * @supported weapp
   */
  adjustPosition?: boolean

  /** focus 时，点击页面的时候不收起键盘
   * @default false
   * @supported weapp
   */
  holdKeyboard?: boolean

  /** 是否去掉 iOS 下的默认内边距
   * @default false
   * @supported weapp
   */
  disableDefaultPadding?: boolean

  /** 输入框聚焦时触发
   * @supported weapp, h5, rn
   */
  onFocus?: CommonEventFunction<TextareaProps.onFocusEventDetail>

  /** 输入框失去焦点时触发
   * @supported weapp, h5, rn
   */
  onBlur?: CommonEventFunction<TextareaProps.onBlurEventDetail>

  /** 输入框行数变化时调用，event.detail = {height: 0, heightRpx: 0, lineCount: 0}
   * @supported weapp, rn
   */
  onLineChange?: CommonEventFunction<TextareaProps.onLineChangeEventDetail>

  /** 当键盘输入时，触发 input 事件，event.detail = {value, cursor, keyCode}
   *
   * **onInput 处理函数的返回值并不会反映到 textarea 上**
   * @supported weapp, h5, rn
   */
  onInput?: CommonEventFunction<TextareaProps.onInputEventDetail>

  /** 点击完成时， 触发 confirm 事件，event.detail = {value: value}
   * @supported weapp, rn
   */
  onConfirm?: CommonEventFunction<TextareaProps.onConfirmEventDetail>

  /** 键盘高度发生变化的时候触发此事件
   * @supported weapp
   */
  onKeyboardHeightChange?: CommonEventFunction<TextareaProps.onKeyboardHeightChangeEventDetail>

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

declare namespace TextareaProps {
  interface onFocusEventDetail {
    /** 输入值 */
    value: string
    /** 键盘高度 */
    height: number
  }
  interface onBlurEventDetail {
    /** 输入值 */
    value: string
    /** 光标位置 */
    cursor: number
  }
  interface onLineChangeEventDetail {
    height: number
    heightRpx: number
    lineCount: number
  }
  interface onInputEventDetail {
    /** 输入值 */
    value: string
    /** 光标位置 */
    cursor: number
    /** 键值 */
    keyCode: number
  }
  interface onConfirmEventDetail {
    /** 输入值 */
    value: string
  }
  interface onKeyboardHeightChangeEventDetail {
    /** 键盘高度 */
    height: number
    /** 持续时间 */
    duration: number
  }
}

/** 多行输入框。该组件是原生组件，使用时请注意相关限制
 * @classification forms
 * @supported weapp, h5, rn
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Text>输入区域高度自适应，不会出现滚动条</Text>
 *         <Textarea style='background:#fff;width:100%;min-height:80px;padding:0 30rpx;' autoHeight/>
 *         <Text>这是一个可以自动聚焦的 textarea</Text>
 *         <Textarea style='background:#fff;width:100%;height:80px;padding:0 30rpx;' autoFocus/>
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <text>输入区域高度自适应，不会出现滚动条</text>
 *     <textarea style="background:#efefef;width:100%;min-height:80px;padding:0 30rpx;" :auto-height="true" />
 *     <text>这是一个可以自动聚焦的 textarea</text>
 *     <textarea style="background:#efefef;width:100%;height:80px;padding:0 30rpx;" :auto-focusd="true" />
 *   </view>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/textarea.html
 */
declare const Textarea: ComponentType<TextareaProps>

interface FunctionalPageNavigatorProps extends StandardProps {
  /** 跳转到的小程序版本，有效值 develop（开发版），trial（体验版），release（正式版）；线上版本必须设置为 release
   * @default "release"
   * @supported weapp
   */
  version?: keyof FunctionalPageNavigatorProps.Version

  /** 要跳转到的功能页
   * @supported weapp
   */
  name?: keyof FunctionalPageNavigatorProps.Name

  /** 功能页参数，参数格式与具体功能页相关
   * @supported weapp
   */
  args?: object

  /** 功能页返回，且操作成功时触发， detail 格式与具体功能页相关
   * @supported weapp
   */
  onSuccess?: CommonEventFunction

  /** 功能页返回，且操作失败时触发， detail 格式与具体功能页相关
   * @supported weapp
   */
  onFail?: CommonEventFunction

  /** 因用户操作从功能页返回时触发
   * @supported weapp
   */
  onCancel?: CommonEventFunction
}

declare namespace FunctionalPageNavigatorProps {
  /** version 的合法值 */
  interface Version {
    /** 开发版 */
    develop
    /** 体验版 */
    trial
    /** 正式版 */
    release
  }
  /** name 的合法值 */
  interface Name {
    /** [用户信息功能页](https://developers.weixin.qq.com/miniprogram/dev/framework/plugin/functional-pages/user-info.html) */
    loginAndGetUserInfo
    /** [支付功能页](https://developers.weixin.qq.com/miniprogram/dev/framework/plugin/functional-pages/request-payment.html) */
    requestPayment
    /** [收货地址功能页](https://developers.weixin.qq.com/miniprogram/dev/framework/plugin/functional-pages/choose-address.html) */
    chooseAddress
  }
}

/** 仅在插件中有效，用于跳转到插件功能页
 * @classification navig
 * @supported weapp
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/functional-page-navigator.html
 */
declare const FunctionalPageNavigator: ComponentType<FunctionalPageNavigatorProps>

interface NavigatorProps extends StandardProps {
  /** 在哪个目标上发生跳转，默认当前小程序
   * @default "self"
   * @supported weapp, swan
   */
  target?: keyof NavigatorProps.Target

  /** 当前小程序内的跳转链接
   * @supported weapp, swan, alipay, tt
   */
  url?: string

  /** 跳转方式
   * @default "navigate"
   * @supported weapp, swan, alipay, tt
   */
  openType?: keyof NavigatorProps.OpenType

  /** 当 open-type 为 'navigateBack' 时有效，表示回退的层数
   * @supported weapp, swan, tt
   */
  delta?: number

  /** 当 `target="miniProgram"` 时有效，要打开的小程序 appId
   * @supported weapp, swan
   */
  appId?: string

  /** 当 `target="miniProgram"` 时有效，打开的页面路径，如果为空则打开首页
   * @supported weapp, swan
   */
  path?: string

  /** 当 `target="miniProgram"` 时有效，需要传递给目标小程序的数据，目标小程序可在 `App.onLaunch()`，`App.onShow()` 中获取到这份数据.
   * @supported weapp, swan
   */
  extraData?: object

  /** 当 `target="miniProgram"` 时有效，要打开的小程序版本
   * @supported weapp, swan
   */
  version?: keyof NavigatorProps.Version

  /** 指定按下去的样式类。当 `hover-class="none"` 时，没有点击态效果
   * @default "navigator-hover"
   * @supported weapp, swan, alipay, tt
   */
  hoverClass?: string

  /** 指定是否阻止本节点的祖先节点出现点击态
   * @default false
   * @supported weapp, swan, tt
   */
  hoverStopPropagation?: boolean

  /** 按住后多久出现点击态，单位毫秒
   * @default 50
   * @supported weapp, swan, alipay, tt
   */
  hoverStartTime?: number

  /** 手指松开后点击态保留时间，单位毫秒
   * @default 600
   * @supported weapp, swan, alipay, tt
   */
  hoverStayTime?: number

  /** 当 `target="miniProgram"` 时有效，跳转小程序成功
   * @supported weapp, swan
   */
  onSuccess?: CommonEventFunction

  /** 当 `target="miniProgram"` 时有效，跳转小程序失败
   * @supported weapp, swan
   */
  onFail?: CommonEventFunction

  /** 当 `target="miniProgram"` 时有效，跳转小程序完成
   * @supported weapp, swan
   */
  onComplete?: CommonEventFunction
}

declare namespace NavigatorProps {
  /** target 的合法值 */
  interface Target {
    /** 当前小程序 */
    self
    /** 其它小程序 */
    miniProgram
  }
  /** open-type 的合法值 */
  interface OpenType {
    /** 对应 Taro.navigateTo 或 Taro.navigateToMiniProgram 的功能 */
    navigate
    /** 对应 Taro.redirectTo 的功能 */
    redirect
    /** 对应 Taro.switchTab 的功能 */
    switchTab
    /** 对应 Taro.reLaunch 的功能 */
    reLaunch
    /** 对应 Taro.navigateBack 的功能 */
    navigateBack
    /** 退出小程序，`target="miniProgram"` 时生效 */
    exit
  }
  /** version 的合法值 */
  interface Version {
    /** 开发版 */
    develop
    /** 体验版 */
    trial
    /** 正式版，仅在当前小程序为开发版或体验版时此参数有效；如果当前小程序是正式版，则打开的小程序必定是正式版。 */
    release
  }
}

/** 页面链接
 * @classification navig
 * @supported weapp, swan, alipay, tt
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/navigator.html
 */
declare const Navigator: ComponentType<NavigatorProps>

interface NavigationBarProps extends StandardProps {
  /** 导航条标题
   * @supported weapp
   */
  title?: string

  /** 是否在导航条显示 loading 加载提示
   * @supported weapp
   */
  loading?: boolean

  /** 导航条前景颜色值，包括按钮、标题、状态栏的颜色，仅支持 #ffffff 和 #000000
   * @supported weapp
   */
  frontColor?: string

  /** 导航条背景颜色值，有效值为十六进制颜色
   * @supported weapp
   */
  backgroundColor?: string

  /** 改变导航栏颜色时的动画时长，默认为 0 （即没有动画效果）
   * @default 0
   * @supported weapp
   */
  colorAnimationDuration?: string

  /** 改变导航栏颜色时的动画方式，支持 linear 、 easeIn 、 easeOut 和 easeInOut
   * @default "linear"
   * @supported weapp
   */
  colorAnimationTimingFunc?: 'linear' | 'easeIn' | 'easeOut' | 'easeInOut'
}

/** 页面导航条配置节点，用于指定导航栏的一些属性。只能是 PageMeta 组件内的第一个节点，需要配合它一同使用。
 * 通过这个节点可以获得类似于调用 Taro.setNavigationBarTitle Taro.setNavigationBarColor 等接口调用的效果。
 * @supported weapp
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/open-data.html
 */
declare const NavigationBar: ComponentType<NavigationBarProps>

interface AudioProps extends StandardProps {
  /** audio 组件的唯一标识符
   * @supported weapp
   */
  id?: string

  /** 要播放音频的资源地址
   * @supported weapp, h5, swan
   */
  src?: string

  /** 是否循环播放
   * @default false
   * @supported weapp, h5, swan
   */
  loop?: boolean

  /** 是否静音播放
   * @default false
   * @supported h5
   * @deprecated
   */
  muted?: boolean

  /** 是否显示默认控件
   * @default false
   * @supported weapp, h5, swan
   */
  controls?: boolean

  /** 默认控件上的音频封面的图片资源地址，如果 controls 属性值为 false 则设置 poster 无效
   * @supported weapp, swan
   */
  poster?: string

  /** 默认控件上的音频名字，如果 controls 属性值为 false 则设置 name 无效
   * @default "未知音频"
   * @supported weapp
   */
  name?: string

  /** 默认控件上的作者名字，如果 controls 属性值为 false 则设置 author 无效
   * @default "未知作者"
   * @supported weapp
   */
  author?: string

  /** 当发生错误时触发 error 事件，detail = {errMsg: MediaError.code}
   * @supported weapp, h5, swan
   */
  onError?: CommonEventFunction<AudioProps.onErrorEventDetail>

  /** 当开始/继续播放时触发play事件
   * @supported weapp, h5, swan
   */
  onPlay?: CommonEventFunction

  /** 当暂停播放时触发 pause 事件
   * @supported weapp, h5, swan
   */
  onPause?: CommonEventFunction

  /** 当播放进度改变时触发 timeupdate 事件，detail = {currentTime, duration}
   * @supported weapp, h5, swan
   */
  onTimeUpdate?: CommonEventFunction<AudioProps.onTimeUpdateEventDetail>

  /** 当播放到末尾时触发 ended 事件
   * @supported weapp, h5, swan
   */
  onEnded?: CommonEventFunction

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

declare namespace AudioProps {
  interface onErrorEventDetail {
    errMsg: keyof MediaError.Code
  }
  interface onTimeUpdateEventDetail {
    /** 当前时间 */
    currentTime: number
    /** 持续时间 */
    duration: number
  }
  namespace MediaError {
    interface Code {
      /** 获取资源被用户禁止 */
      1
      /** 网络错误 */
      2
      /** 解码错误 */
      3
      /** 不合适资源 */
      4
    }
  }
}

/** 音频。1.6.0版本开始，该组件不再维护。建议使用能力更强的 Taro.createInnerAudioContext 接口
 * @classification media
 * @deprecated
 * @supported weapp, h5, swan
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Audio
 *           src='https://ws.stream.qqmusic.qq.com/M500001VfvsJ21xFqb.mp3?guid=ffffffff82def4af4b12b3cd9337d5e7&uin=346897220&vkey=6292F51E1E384E06DCBDC9AB7C49FD713D632D313AC4858BACB8DDD29067D3C601481D36E62053BF8DFEAF74C0A5CCFADD6471160CAF3E6A&fromtag=46'
 *           controls={true}
 *           autoplay={false}
 *           loop={false}
 *           muted={true}
 *           initialTime='30'
 *           id='video'
 *         />
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <audio
 *       id="video"
 *       src="https://ws.stream.qqmusic.qq.com/M500001VfvsJ21xFqb.mp3?guid=ffffffff82def4af4b12b3cd9337d5e7&uin=346897220&vkey=6292F51E1E384E06DCBDC9AB7C49FD713D632D313AC4858BACB8DDD29067D3C601481D36E62053BF8DFEAF74C0A5CCFADD6471160CAF3E6A&fromtag=46"
 *       :controls="true"
 *       :autoplay="false"
 *       :loop="false"
 *       :muted="true"
 *     />
 *   </view>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/audio.html
 */
declare const Audio: ComponentType<AudioProps>

interface CameraProps extends StandardProps {
  /** 模式，有效值为normal, scanCode
   * @default "normal"
   * @supported weapp, rn
   */
  mode?: keyof CameraProps.Mode

  /** 分辨率，不支持动态修改
   * @default "medium"
   * @supported weapp
   */
  resolution?: keyof CameraProps.Resolution

  /** 摄像头朝向
   * @default "back"
   * @supported weapp, rn
   */
  devicePosition?: keyof CameraProps.DevicePosition

  /** 闪光灯
   * @default "auto"
   * @supported weapp, rn
   */
  flash?: keyof CameraProps.Flash

  /** 指定期望的相机帧数据尺寸
   * @default "medium"
   * @supported weapp
   */
  frameSize?: keyof CameraProps.FrameSize

  /** 扫码识别区域，格式为[x, y, w, h]，
   * x,y是相对于camera显示区域的左上角，
   * w,h为区域宽度，单位px，仅在 mode="scanCode" 时生效
   * @supported weapp
   */
  scanArea?: number[]

  /** 摄像头在非正常终止时触发，
   * 如退出后台等情况
   * @supported weapp, rn
   */
  onStop?: CommonEventFunction

  /** 用户不允许使用摄像头时触发
   * @supported weapp, rn
   */
  onError?: CommonEventFunction

  /** 相机初始化完成时触发
   * @supported weapp, rn
   */
  onInitDone?: CommonEventFunction<CameraProps.onInitDoneEventDetail>

  /** 在成功识别到一维码时触发，
   * 仅在 mode="scanCode" 时生效
   * @supported weapp, rn
   */
  onScanCode?: CommonEventFunction
}

declare namespace CameraProps {
  /** mode 的合法值 */
  interface Mode {
    /** 相机模式 */
    normal
    /** 扫码模式 */
    scanCode
  }
  /** resolution 的合法值 */
  interface Resolution {
    /** 低 */
    low
    /** 中 */
    medium
    /** 高 */
    high
  }
  /** device-position 的合法值 */
  interface DevicePosition {
    /** 前置 */
    front
    /** 后置 */
    back
  }
  /** flash 的合法值 */
  interface Flash {
    /** 自动 */
    auto
    /** 打开 */
    on
    /** 关闭 */
    off
    /** 常亮 */
    torch
  }
  /** frame-size 的合法值 */
  interface FrameSize {
    /** 小尺寸帧数据 */
    small
    /** 中尺寸帧数据 */
    medium
    /** 大尺寸帧数据 */
    large
  }

  interface onInitDoneEventDetail {
    /** 最大变焦 */
    maxZoom: number
  }
}

/** 系统相机
 * @classification media
 * @supported weapp, rn
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/camera.html
 */
declare const Camera: ComponentType<CameraProps>

interface ImageProps extends StandardProps {
  /** 图片资源地址
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  src: string

  /** 图片裁剪、缩放的模式
   * @default "scaleToFill"
   * @supported weapp, h5, rn, swan, alipay, tt
   * @rn 部分支持 scaleToFill, aspectFit, aspectFill, widthFix
   */
  mode?: keyof ImageProps.Mode

	/** 默认不解析 webP 格式，只支持网络资源
   * @default false
   * @supported weapp
	 */
	webp?: boolean

  /** 默认不解析 svg 格式，svg 图片只支持 aspectFit
   * @default false
   * @supported rn
	 */
   svg?: boolean

  /** 图片懒加载。只针对 page 与 scroll-view 下的 image 有效
   * @default false
   * @supported weapp, swan, alipay, tt, h5
   */
  lazyLoad?: boolean

  /** 开启长按图片显示识别小程序码菜单
   * @default false
   * @supported weapp
   */
  showMenuByLongpress?: boolean

  /** 当错误发生时，发布到 AppService 的事件名，事件对象
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  onError?: CommonEventFunction<ImageProps.onErrorEventDetail>

  /** 当图片载入完毕时，发布到 AppService 的事件名，事件对象
   * @supported weapp, h5, rn, swan, alipay, tt
   */
  onLoad?: CommonEventFunction<ImageProps.onLoadEventDetail>

  /**
   * 为 img 标签额外增加的属性
   * @supported h5
   */
  imgProps?: ImgHTMLAttributes<HTMLImageElement>

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

declare namespace ImageProps {
  /** mode 的合法值 */
  interface Mode {
    /** 缩放模式，不保持纵横比缩放图片，使图片的宽高完全拉伸至填满 image 元素 */
    scaleToFill
    /** 缩放模式，保持纵横比缩放图片，使图片的长边能完全显示出来。也就是说，可以完整地将图片显示出来。 */
    aspectFit
    /** 缩放模式，保持纵横比缩放图片，只保证图片的短边能完全显示出来。也就是说，图片通常只在水平或垂直方向是完整的，另一个方向将会发生截取。 */
    aspectFill
    /** 缩放模式，宽度不变，高度自动变化，保持原图宽高比不变 */
    widthFix
    /** 缩放模式，高度不变，宽度自动变化，保持原图宽高比不变 */
    heightFix
    /** 裁剪模式，不缩放图片，只显示图片的顶部区域 */
    top
    /** 裁剪模式，不缩放图片，只显示图片的底部区域 */
    bottom
    /** 裁剪模式，不缩放图片，只显示图片的中间区域 */
    center
    /** 裁剪模式，不缩放图片，只显示图片的左边区域 */
    left
    /** 裁剪模式，不缩放图片，只显示图片的右边区域 */
    right
    /** 裁剪模式，不缩放图片，只显示图片的左上边区域 */
    'top left'
    /** 裁剪模式，不缩放图片，只显示图片的右上边区域 */
    'top right'
    /** 裁剪模式，不缩放图片，只显示图片的左下边区域 */
    'bottom left'
    /** 裁剪模式，不缩放图片，只显示图片的右下边区域 */
    'bottom right'
  }
  interface onErrorEventDetail {
    /** 错误信息 */
    errMsg: string
  }
  interface onLoadEventDetail {
    /** 图片高度 */
    height: number | string
    /** 图片宽度 */
    width: number | string
  }
}

/** 图片。支持 JPG、PNG、SVG、WEBP、GIF 等格式以及云文件ID。
 *
 * **Note:** 为实现小程序的 `mode` 特性，在 H5 组件中使用一个 `div` 容器来对内部的 `img` 进行展示区域的裁剪，因此请勿使用元素选择器来重置 `img` 的样式！
 * @classification media
 * @supported weapp, h5, rn, swan, alipay, tt
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Image
 *           style='width: 300px;height: 100px;background: #fff;'
 *           src='nerv_logo.png'
 *         />
 *         <Image
 *           style='width: 300px;height: 100px;background: #fff;'
 *           src='https://camo.githubusercontent.com/3e1b76e514b895760055987f164ce6c95935a3aa/687474703a2f2f73746f726167652e333630627579696d672e636f6d2f6d74642f686f6d652f6c6f676f2d3278313531333833373932363730372e706e67'
 *         />
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <view class="components-page">
 *     <image
 *       style="width: 300px;height: 100px;background: #fff;"
 *       src="nerv_logo.png"
 *     />
 *     <image
 *       style="width: 300px;height: 100px;background: #fff;"
 *       src="https://camo.githubusercontent.com/3e1b76e514b895760055987f164ce6c95935a3aa/687474703a2f2f73746f726167652e333630627579696d672e636f6d2f6d74642f686f6d652f6c6f676f2d3278313531333833373932363730372e706e67"
 *     />
 *   </view>
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/image.html
 */
declare const Image: ComponentType<ImageProps>

interface LivePlayerProps extends StandardProps {
  /** 音视频地址。目前仅支持 flv, rtmp 格式
   * @supported weapp
   */
  src?: string

  /** 模式
   * @default "live"
   * @supported weapp
   */
  mode?: keyof LivePlayerProps.Mode

  /** 自动播放
   * @default false
   * @supported weapp
   */
  autoplay?: boolean

  /** 是否静音
   * @default false
   * @supported weapp
   */
  muted?: boolean

  /** 画面方向
   * @default "vertical"
   * @supported weapp
   */
  orientation?: keyof LivePlayerProps.Orientation

  /** 填充模式
   * @default "contain"
   * @supported weapp
   */
  objectFit?: keyof LivePlayerProps.objectFit

  /** 进入后台时是否静音（已废弃，默认退台静音）
   * @default false
   * @supported weapp
   * @deprecated
   */
  backgroundMute?: boolean

  /** 最小缓冲区，单位s
   * @default 1
   * @supported weapp
   */
  minCache?: number

  /** 最大缓冲区，单位s
   * @default 3
   * @supported weapp
   */
  maxCache?: number

  /** 声音输出方式
   * @default "speaker"
   * @supported weapp
   */
  soundMode?: keyof LivePlayerProps.soundMode

  /** 当跳转到本小程序的其他页面时，是否自动暂停本页面的实时音视频播放
   * @default true
   * @supported weapp
   */
  autoPauseIfNavigate?: boolean

  /** 当跳转到其它微信原生页面时，是否自动暂停本页面的实时音视频播放
   * @default true
   * @supported weapp
   */
  autoPauseIfOpenNavigate?: boolean

  /** 设置小窗模式： push, pop，空字符串或通过数组形式设置多种模式（如： ["push", "pop"]）
   * @supported weapp
   */
  pictureInPictureMode?: ('push' | 'pop')[] | 'push' | 'pop' | ''

  /** 播放状态变化事件，detail = {code}
   * @supported weapp
   */
  onStateChange?: CommonEventFunction<LivePlayerProps.onStateChangeEventDetail>

  /** 全屏变化事件，detail = {direction, fullScreen}
   * @supported weapp
   */
  onFullScreenChange?: CommonEventFunction<LivePlayerProps.onFullScreenChangeEventDetail>

  /** 网络状态通知，detail = {info}
   * @supported weapp
   */
  onNetStatus?: CommonEventFunction<LivePlayerProps.onNetStatusEventDetail>

  /** 播放音量大小通知，detail = {}
   * @supported weapp
   */
  onAudioVolumeNotify?: CommonEventFunction<{}>

  /** 播放器进入小窗
   * @supported weapp
   */
  onEnterPictureInPicture?: CommonEventFunction

  /** 播放器退出小窗
   * @supported weapp
   */
  onLeavePictureInPicture?: CommonEventFunction
}

declare namespace LivePlayerProps {
  /** mode 的合法值 */
  interface Mode {
    /** 直播 */
    live
    /** 实时通话，该模式时延更低 */
    RTC
  }
  /** orientation 的合法值 */
  interface Orientation {
    /** 竖直 */
    vertical
    /** 水平 */
    horizontal
  }
  /** objectFit 的合法值 */
  interface objectFit {
    /** 图像长边填满屏幕，短边区域会被填充⿊⾊ */
    contain
    /** 图像铺满屏幕，超出显示区域的部分将被截掉 */
    fillCrop
  }
  /** soundMode 的合法值 */
  interface soundMode {
    /** 扬声器 */
    speaker
    /** 听筒 */
    ear
  }

  interface onStateChangeEventDetail {
    /** 状态码 */
    code: number
  }
  interface onFullScreenChangeEventDetail {
    /** 方向 */
    direction: number
    /** 全屏 */
    fullScreen: number | boolean
  }
  interface onNetStatusEventDetail {
    info: NetStatus
  }
  /** 状态码 */
  interface Status {
    /** 已经连接服务器 */
    2001
    /** 已经连接 RTMP 服务器,开始拉流 */
    2002
    /** 网络接收到首个视频数据包(IDR) */
    2003
    /** 视频播放开始 */
    2004
    /** 视频播放进度 */
    2005
    /** 视频播放结束 */
    2006
    /** 视频播放Loading */
    2007
    /** 解码器启动 */
    2008
    /** 视频分辨率改变 */
    2009
    /** 网络断连，且经多次重连抢救无效，更多重试请自行重启播放 */
    '-2301'
    /** 获取加速拉流地址失败 */
    '-2302'
    /** 当前视频帧解码失败 */
    2101
    /** 当前音频帧解码失败 */
    2102
    /** 网络断连, 已启动自动重连 */
    2103
    /** 网络来包不稳：可能是下行带宽不足，或由于主播端出流不均匀 */
    2104
    /** 当前视频播放出现卡顿 */
    2105
    /** 硬解启动失败，采用软解 */
    2106
    /** 当前视频帧不连续，可能丢帧 */
    2107
    /** 当前流硬解第一个I帧失败，SDK自动切软解 */
    2108
    /** RTMP -DNS解析失败 */
    3001
    /** RTMP服务器连接失败 */
    3002
    /** RTMP服务器握手失败 */
    3003
    /** RTMP 读/写失败 */
    3005
  }
}

/** 实时音视频播放。相关api：Taro.createLivePlayerContext
 *
 * 需要先通过类目审核，再在小程序管理后台，“设置”-“接口设置”中自助开通该组件权限。
 * @classification media
 * @supported weapp
 * @example_react
 * ```tsx
 * class App extends Components {
 *   render () {
 *     return (
 *       <LivePlayer src='url' mode='live' autoplay  />
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <live-player src="url" mode="live" :autoplay="true"  />
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/live-player.html
 */
declare const LivePlayer: ComponentType<LivePlayerProps>

/** 实时音视频录制。
 * 需要用户授权 scope.camera、scope.record
 * 暂只针对国内主体如下类目的小程序开放，需要先通过类目审核，再在小程序管理后台，“设置”-“接口设置”中自助开通该组件权限。
 */
interface LivePusherProps extends StandardProps {
  /** 推流地址。目前仅支持 rtmp 格式
   * @supported weapp
   */
  url?: string

  /** SD（标清）, HD（高清）, FHD（超清）, RTC（实时通话）
   * @default "RTC"
   * @supported weapp
   */
  mode?: 'SD' | 'HD' | 'FHD' | 'RTC'

  /** 自动推流
   * @default false
   * @supported weapp
   */
  autopush?: boolean

  /** 是否静音。即将废弃，可用 enable-mic 替代
   * @default false
   * @deprecated
   * @supported weapp
   */
  muted?: boolean

  /** 开启摄像头
   * @default true
   * @supported weapp
   */
  enableCamera?: boolean

  /** 自动聚集
   * @default true
   * @supported weapp
   */
  autoFocus?: boolean

  /** 画面方向
   * @default "vertical"
   * @supported weapp
   */
  orientation?: keyof LivePusherProps.Orientation

  /** 美颜，取值范围 0-9 ，0 表示关闭
   * @default 0
   * @supported weapp
   */
  beauty?: number

  /** 美白，取值范围 0-9 ，0 表示关闭
   * @default 0
   * @supported weapp
   */
  whiteness?: number

  /** 宽高比，可选值有 3:4, 9:16
   * @default "9:16"
   * @supported weapp
   */
  aspect?: '9:16' | '3:4'

  /** 最小码率
   * @default 200
   * @supported weapp
   */
  minBitrate?: number

  /** 最大码率
   * @default 1000
   * @supported weapp
   */
  maxBitrate?: number

  /** 高音质(48KHz)或低音质(16KHz)，值为high, low
   * @default "high"
   * @supported weapp
   */
  audioQuality?: string

  /** 进入后台时推流的等待画面
   * @supported weapp
   */
  waitingImage?: string

  /** 等待画面资源的MD5值
   * @supported weapp
   */
  waitingImageHash?: string

  /** 调整焦距
   * @default false
   * @supported weapp
   */
  zoom?: boolean

  /** 前置或后置，值为front, back
   * @default "front"
   * @supported weapp
   */
  devicePosition?: string

  /** 进入后台时是否静音
   * @default false
   * @supported weapp
   */
  backgroundMute?: boolean

  /** 设置推流画面是否镜像，产生的效果在 LivePlayer 反应到
   * @default false
   * @supported weapp
   * @deprecated
   */
  mirror?: boolean

  /** 设置推流画面是否镜像，产生的效果在 LivePlayer 反应到
   *
   * **Note:** 同 mirror 属性，后续 mirror 将废弃
   * @default false
   * @supported weapp
   */
  remoteMirror?: boolean

  /** 控制本地预览画面是否镜像
   * @default "auto"
   * @supported weapp
   */
  localMirror?: keyof LivePusherProps.LocalMirror

  /** 音频混响类型
   * @default 0
   * @supported weapp
   */
  audioReverbType?: keyof LivePusherProps.AudioReverbType

  /** 开启或关闭麦克风
   * @default true
   * @supported weapp
   */
  enableMic?: boolean

  /** 是否开启音频自动增益
   * @default false
   * @supported weapp
   */
  enableAgc?: boolean

  /** 是否开启音频噪声抑制
   * @default false
   * @supported weapp
   */
  enableAns?: boolean

  /** 音量类型
   * @default "voicecall"
   * @supported weapp
   */
  audioVolumeType?: keyof LivePusherProps.AudioVolumeType

  /** 上推的视频流的分辨率宽度
   * @default 360
   * @supported weapp
   */
  videoWidth?: number

  /** 上推的视频流的分辨率高度
   * @default 640
   * @supported weapp
   */
  videoHeight?: number

  /** 设置美颜类型
   * @default smooth
   * @supported weapp
   */
  beautyStyle?: keyof LivePusherProps.BeautyStyleType

  /** 设置色彩滤镜
   * @default standard
   * @supported weapp
   */
  filter?: keyof LivePusherProps.FilterType

  /** 状态变化事件，detail = {code}
   * @supported weapp
   */
  onStateChange?: CommonEventFunction<LivePusherProps.onStateChangeEventDetail>

  /** 网络状态通知，detail = {info}
   * @supported weapp
   */
  onNetstatus?: CommonEventFunction<LivePusherProps.onNetstatusEventDetail>

  /** 渲染错误事件，detail = {errMsg, errCode}
   * @supported weapp
   */
  onError?: CommonEventFunction<LivePusherProps.onErrorEventDetail>

  /** 背景音开始播放时触发
   * @supported weapp
   */
  onBgmStart?: CommonEventFunction

  /** 背景音进度变化时触发，detail = {progress, duration}
   * @supported weapp
   */
  onBgmProgress?: CommonEventFunction<LivePusherProps.onBgmProgressEventDetail>

  /** 背景音播放完成时触发
   * @supported weapp
   */
  onBgmComplete?: CommonEventFunction

  /** 返回麦克风采集的音量大小
   * @supported weapp
   */
  onAudioVolumeNotify?: CommonEventFunction
}

declare namespace LivePusherProps {
  /** orientation 的合法值 */
  interface Orientation {
    /** 竖直 */
    vertical
    /** 水平 */
    horizontal
  }
  /** localMirror 的合法值 */
  interface LocalMirror {
    /** 前置摄像头镜像，后置摄像头不镜像 */
    auto
    /** 前后置摄像头均镜像 */
    enable
    /** 前后置摄像头均不镜像 */
    disable
  }
  /** audioReverbType 的合法值 */
  interface AudioReverbType {
    /** 关闭 */
    0
    /** KTV */
    1
    /** 小房间 */
    2
    /** 大会堂 */
    3
    /** 低沉 */
    4
    /** 洪亮 */
    5
    /** 金属声 */
    6
    /** 磁性 */
    7
  }
  /** audioVolumeType 的合法值 */
  interface AudioVolumeType {
    /** 媒体音量 */
    media
    /** 通话音量 */
    voicecall
  }
  /** beautyStyleType 的合法值 */
  interface BeautyStyleType {
    /** 光滑美颜 */
    smooth
    /** 自然美颜 */
    nature
  }
  /** filterType 的合法值 */
  interface FilterType {
    /** 标准 */
    standard
    /** 粉嫩 */
    pink
    /** 怀旧 */
    nostalgia
    /** 蓝调 */
    blues
    /** 浪漫 */
    romantic
    /** 清凉 */
    cool
    /** 清新 */
    fresher
    /** 日系 */
    solor
    /** 唯美 */
    aestheticism
    /** 美白 */
    whitening
    /** 樱红 */
    cerisered
  }
  interface onStateChangeEventDetail {
    /** 状态码 */
    code: number
  }
  interface onNetstatusEventDetail {
    /** 网络状态 */
    info: NetStatus
  }
  interface onErrorEventDetail {
    /** 错误信息 */
    errMsg: string
    /** 错误码 */
    errCode: string | number
  }
  interface onBgmProgressEventDetail {
    /** 进展 */
    progress
    /** 持续时间 */
    duration: number
  }
}

/** 实时音视频录制。需要用户授权 scope.camera、scope.record
 * 需要先通过类目审核，再在小程序管理后台，「开发」-「接口设置」中自助开通该组件权限。
 * @classification media
 * @supported weapp
 * @example_react
 * ```tsx
 * class App extends Components {
 *   render () {
 *     return (
 *       <LivePusher url='url' mode='RTC' autopush  />
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <live-pusher url="url" mode="RTC" :autopush="true"  />
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/live-pusher.html
 */
declare const LivePusher: ComponentType<LivePusherProps>

interface VideoProps extends StandardProps {
  /** 要播放视频的资源地址
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  src: string

  /** 指定视频时长
   * @supported weapp, swan, alipay, rn
   * @h5 待定
   */
  duration?: number

  /** 是否显示默认播放控件（播放/暂停按钮、播放进度、时间）
   * @default true
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  controls?: boolean

  /** 弹幕列表
   * @supported weapp, swan
   * @h5 待定
   */
  danmuList?: any[]

  /** 是否显示弹幕按钮，只在初始化时有效，不能动态变更
   * @default false
   * @supported weapp, swan
   * @h5 待定
   */
  danmuBtn?: boolean

  /** 是否展示弹幕，只在初始化时有效，不能动态变更
   * @default false
   * @supported weapp, swan
   * @h5 待定
   */
  enableDanmu?: boolean

  /** 是否自动播放
   * @default false
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  autoplay?: boolean

  /** 是否循环播放
   * @default false
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  loop?: boolean

  /** 是否静音播放
   * @default false
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  muted?: boolean

  /** 指定视频初始播放位置
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  initialTime?: number

  /** 在非全屏模式下，是否开启亮度与音量调节手势
   * @default false
   * @h5 待定
   */
  pageGesture?: boolean

  /** 设置全屏时视频的方向，不指定则根据宽高比自动判断。有效值为 0（正常竖向）, 90（屏幕逆时针90度）, -90（屏幕顺时针90度）
   * @supported weapp, swan, alipay
   * @h5 待定
   */
  direction?: number

  /** 若不设置，宽度大于240时才会显示
   * @default true
   * @supported weapp, swan
   * @h5 待定
   */
  showProgress?: boolean

  /** 是否显示全屏按钮
   * @default true
   * @supported weapp, swan, alipay
   * @h5 待定
   */
  showFullscreenBtn?: boolean

  /** 是否显示视频底部控制栏的播放按钮
   * @default true
   * @supported weapp, swan, alipay
   * @h5 待定
   */
  showPlayBtn?: boolean

  /** 是否显示视频中间的播放按钮
   * @default true
   * @supported weapp, swan, alipay, rn
   * @h5 待定
   */
  showCenterPlayBtn?: boolean

  /** 是否开启控制进度的手势
   * @default true
   * @supported weapp, swan, alipay
   * @h5 待定
   */
  enableProgressGesture?: boolean

  /** 当视频大小与 video 容器大小不一致时，视频的表现形式
   * @default "contain"
   * @supported weapp, swan, alipay
   * @h5 待定
   */
  objectFit?: keyof VideoProps.ObjectFit

  /** 视频封面的图片网络资源地址，如果 controls 属性值为 false 则设置 poster 无效
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  poster?: string

  /** 是否显示静音按钮
   * @default false
   * @supported weapp
   * @h5 待定
   */
  showMuteBtn?: boolean

  /** 视频的标题，全屏时在顶部展示
   * @supported weapp
   * @h5 待定
   */
  title?: string

  /** 播放按钮的位置
   * - `bottom`: controls bar 上
   * - `center`: 视频中间
   *
   * @default 'bottom'
   * @supported weapp
   * @h5 待定
   */
  playBtnPosition?: keyof VideoProps.PlayBtnPosition

  /** 是否开启播放手势，即双击切换播放/暂停
   * @default false
   * @supported weapp
   * @h5 待定
   */
  enablePlayGesture?: boolean

  /** 当跳转到其它小程序页面时，是否自动暂停本页面的视频
   * @default true
   * @supported weapp
   * @h5 待定
   */
  autoPauseIfNavigate?: boolean

  /** 当跳转到其它微信原生页面时，是否自动暂停本页面的视频
   * @default true
   * @supported weapp
   * @h5 待定
   */
  autoPauseIfOpenNative?: boolean

  /** 在非全屏模式下，是否开启亮度与音量调节手势（同 `page-gesture`）
   * @default false
   * @supported weapp, swan
   * @h5 待定
   */
  vslideGesture?: boolean

  /** 在全屏模式下，是否开启亮度与音量调节手势
   * @default true
   * @supported weapp
   * @h5 待定
   */
  vslideGestureInFullscreen?: boolean

  /** 视频前贴广告单元ID，更多详情可参考开放能力[视频前贴广告](https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/ad/video-patch-ad.html)
   * @supported weapp
   */
  adUnitId?: string

  /** 用于给搜索等场景作为视频封面展示，建议使用无播放 icon 的视频封面图，只支持网络地址
   * @supported weapp
   */
  posterForCrawler?: string

  /** 显示投屏按钮。只安卓且同层渲染下生效，支持 DLNA 协议
   * @supported weapp
   */
  showCastingButton?: boolean

  /**
   * 设置小窗模式： push, pop，空字符串或通过数组形式设置多种模式（如： ["push", "pop"]）
   * @supported weapp
   */
  pictureInPictureMode?: ('push' | 'pop')[] | 'push' | 'pop' | ''

  /**
   * 是否在小窗模式下显示播放进度（目前有bug，先注释掉）
   * @supported weapp
   *
   * 先注释掉，原因如下：
   * 该属性超过了 wxml 属性的长度限制，实际无法使用且导致编译报错。可等微信官方修复后再放开。
   * 参考1：https://developers.weixin.qq.com/community/develop/doc/000a429beb87f0eac07acc0fc5b400
   * 参考2: https://developers.weixin.qq.com/community/develop/doc/0006883619c48054286a4308258c00?_at=vyxqpllafi
   *
   */
  // pictureInPictureShowProgress?: boolean

  /**
   * 是否开启手机横屏时自动全屏，当系统设置开启自动旋转时生效
   * @supported weapp
   */
  enableAutoRotation?: boolean

  /**
   * 是否显示锁屏按钮，仅在全屏时显示，锁屏后控制栏的操作
   * @supported weapp
   */
  showScreenLockButton?: boolean

  /** 当开始/继续播放时触发 play 事件
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  onPlay?: CommonEventFunction

  /** 当暂停播放时触发 pause 事件
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  onPause?: CommonEventFunction

  /** 当播放到末尾时触发 ended 事件
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  onEnded?: CommonEventFunction

  /** 播放进度变化时触发, 触发频率 250ms 一次
   *
   * event.detail = {currentTime, duration}
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  onTimeUpdate?: CommonEventFunction<VideoProps.onTimeUpdateEventDetail>

  /** 当视频进入和退出全屏时触发
   *
   * event.detail = {fullScreen, direction}，direction取为 vertical 或 horizontal
   * @supported weapp, swan, alipay, rn
   * @h5 待定
   */
  onFullscreenChange?: CommonEventFunction<VideoProps.onFullscreenChangeEventDetail>

  /** 当视频进入和退出全屏时触发
   *
   * event.detail = {fullScreen, direction}，direction 取为 vertical 或 horizontal
   * @supported weapp, swan
   * @h5 待定
   */
  onWaiting?: CommonEventFunction<VideoProps.onWaitingEventDetail>

  /** 视频播放出错时触发
   * @supported weapp, h5, swan, alipay, tt, rn
   */
  onError?: CommonEventFunction
  /** 加载进度变化时触发，只支持一段加载
   * @supported weapp, alipay
   * @h5 待定
   */
  onProgress?: CommonEventFunction<VideoProps.onProgressEventDetail>

  /** 视频元数据加载完成时触发。event.detail = {width, height, duration}
   * @supported weapp, rn
   */
  onLoadedMetaData?: CommonEventFunction

  /**
   * 切换 controls 显示隐藏时触发。event.detail = {show}
   * @supported weapp
   */
  onControlsToggle?: CommonEventFunction<VideoProps.onControlsToggleEventDetail>

  /**
   * 播放器进入小窗
   * @supported weapp
   */
  onEnterPictureInPicture?: CommonEventFunction

  /**
   * 播放器退出小窗
   * @supported weapp
   */
  onLeavePictureInPicture?: CommonEventFunction

  /**
   * seek 完成时触发
   * @supported weapp
   */
  onSeekComplete?: CommonEventFunction

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

declare namespace VideoProps {
  /** direction 的合法值 */
  interface direction {
    /** 正常竖向 */
    0
    /** 屏幕逆时针90度 */
    90
    /** 屏幕顺时针90度 */
    '-90'
  }
  /** objectFit 的合法值 */
  interface ObjectFit {
    /** 包含 */
    contain
    /** 填充 */
    fill
    /** 覆盖 */
    cover
  }
  /** playBtnPosition 的合法值 */
  interface PlayBtnPosition {
    /** controls bar上 */
    bottom
    /** 视频中间 */
    center
  }

  interface onTimeUpdateEventDetail {
    /** 当前时间 */
    currentTime: number
    /** 持续时间 */
    duration: number
  }
  interface onFullscreenChangeEventDetail {
    /** 方向 */
    direction: number
    /** 全屏 */
    fullScreen: number | boolean
  }
  interface onWaitingEventDetail {
    /** 方向 */
    direction: number
    /** 全屏 */
    fullScreen: number | boolean
  }
  interface onProgressEventDetail {
    /** 百分比 */
    buffered: number
  }

  interface onControlsToggleEventDetail {
    /** 是否显示 */
    show: boolean
  }
}

/** 视频。相关api：Taro.createVideoContext
 *
 * 备注：h5上因为没有测试，所以暂时写了“待定”，需要`Video`来确认。
 * @classification media
 * @supported weapp, h5, swan, alipay, tt
 * @example_react
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   render() {
 *     return (
 *       <View className='components-page'>
 *         <Video
 *           id='video'
 *           src='https://wxsnsdy.tc.qq.com/105/20210/snsdyvideodownload?filekey=30280201010421301f0201690402534804102ca905ce620b1241b726bc41dcff44e00204012882540400&bizid=1023&hy=SH&fileparam=302c020101042530230204136ffd93020457e3c4ff02024ef202031e8d7f02030f42400204045a320a0201000400'
 *           poster='https://misc.aotu.io/booxood/mobile-video/cover_900x500.jpg'
 *           initialTime='0'
 *           controls={true}
 *           autoplay={false}
 *           loop={false}
 *           muted={false}
 *         />
 *       </View>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <video
 *     id="video"
 *     src="https://wxsnsdy.tc.qq.com/105/20210/snsdyvideodownload?filekey=30280201010421301f0201690402534804102ca905ce620b1241b726bc41dcff44e00204012882540400&bizid=1023&hy=SH&fileparam=302c020101042530230204136ffd93020457e3c4ff02024ef202031e8d7f02030f42400204045a320a0201000400"
 *     poster="https://misc.aotu.io/booxood/mobile-video/cover_900x500.jpg"
 *     initial-time="0"
 *     :controls="true"
 *     :autoplay="false"
 *     :loop="false"
 *     :muted="false"
 *   />
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/video.html
 */
declare const Video: ComponentType<VideoProps>

interface VoipRoomProps extends StandardProps {
  /** 进入房间用户的 openid
   * @default none
   * @supported weapp
   */
  openId?: string

  /** 对话窗口类型，自身传入 camera，其它用户传入 video
   * @default camera
   * @supported weapp
   */
  mode?: keyof VoipRoomProps.Mode

  /** 仅在 mode 为 camera 时有效，前置或后置，值为front, back
   * @default front
   * @supported weapp
   */
  devicePosition?: keyof VoipRoomProps.DevicePosition

  /** 创建对话窗口失败时触发
   * @supported weapp
   */
  onError?: CommonEventFunction
}

declare namespace VoipRoomProps {
  /** 对话窗口类型 */
  interface Mode {
    camera
    video
  }

  /** 摄像头类型 */
  interface DevicePosition {
    front
    back
  }
}

/** 多人音视频对话
 *
 * 需用户授权 `scope.camera`、`scope.record`。相关接口： [Taro.joinVoIPChat](/docs/apis/media/voip/joinVoIPChat)
 * 开通该组件权限后，开发者可在 joinVoIPChat 成功后，获取房间成员的 openid，传递给 voip-room 组件，以显示成员画面。
 * @classification media
 * @supported weapp
 * @example
 * ```tsx
 * export default class PageView extends Component {
 *   constructor() {
 *     super(...arguments)
 *   }
 *
 *   render() {
 *     return (
 *       <VoipRoom
 *         openId="{{item}}"
 *         mode="{{selfOpenId === item ? 'camera' : 'video'}}">
 *       </VoipRoom>
 *     )
 *   }
 * }
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/voip-room.html
 */
declare const VoipRoom: ComponentType<VoipRoomProps>

interface MapProps extends StandardProps {
  /** 中心经度
   * @supported weapp, swan, alipay
   */
  longitude: number

  /** 中心纬度
   * @supported weapp, swan, alipay
   */
  latitude: number

  /** 缩放级别，取值范围为3-20
   * @default 16
   * @supported weapp, swan, alipay
   * @swan 取值范围为4-21
   * @alipay 取值范围为5-18
   */
  scale?: number
  /** 最小缩放级别3-20
   * @default 3
   * @supported weapp
   */
  minScale?: number
  /** 最大缩放级别3-20
   * @default 20
   * @supported weapp
   */
  maxScale?: number

  /** 标记点
   * @supported weapp, swan, alipay
   */
  markers?: MapProps.marker[]

  /** 标记点
   * @deprecated 即将移除，请使用 markers
   * @supported weapp
   */
  covers?: any[]

  /** 路线
   * @supported weapp, swan, alipay
   */
  polyline?: MapProps.polyline[]

  /** 圆
   * @supported weapp, swan, alipay
   */
  circles?: MapProps.circle[]

  /** 控件（即将废弃，建议使用 cover-view 代替）
   * @deprecated
   * @supported weapp, swan, alipay
   */
  controls?: MapProps.control[]

  /** 缩放视野以包含所有给定的坐标点
   * @supported weapp, swan, alipay
   */
  includePoints?: MapProps.point[]

  /** 显示带有方向的当前定位点
   * @default false
   * @supported weapp, swan, alipay
   */
  showLocation?: boolean

  /** 多边形
   * @supported weapp, swan, alipay
   */
  polygons?: MapProps.polygon[]

  /** 个性化地图使用的 key
   * @supported weapp
   */
  subkey?: string

  /** 个性化地图配置的 style，不支持动态修改
   * @default 1
   * @supported weapp
   */
  layerStyle?: number

  /** 旋转角度，范围 0 ~ 360, 地图正北和设备 y 轴角度的夹角
   * @default 0
   * @supported weapp
   */
  rotate?: number

  /** 倾斜角度，范围 0 ~ 40 , 关于 z 轴的倾角
   * @default 0
   * @supported weapp
   */
  skew?: number

  /** 展示 3D 楼块
   * @default false
   * @supported weapp, swan
   */
  enable3D?: boolean

  /** 显示指南针
   * @default false
   * @supported weapp, swan
   */
  showCompass?: boolean

  /** 显示比例尺
   * @default false
   * @supported weapp
   */
  showScale?: boolean

  /** 开启俯视
   * @default false
   * @supported weapp, swan
   */
  enableOverlooking?: boolean

  /** 是否支持缩放
   * @default true
   * @supported weapp, swan
   */
  enableZoom?: boolean

  /** 是否支持拖动
   * @default true
   * @supported weapp, swan
   */
  enableScroll?: boolean

  /** 是否支持旋转
   * @default false
   * @supported weapp, swan
   */
  enableRotate?: boolean

  /** 是否开启卫星图
   * @default false
   * @supported weapp
   */
  enableSatellite?: boolean

  /** 是否开启实时路况
   * @default false
   * @supported weapp
   */
  enableTraffic?: boolean

  /** 配置项
   *
   * 提供 setting 对象统一设置地图配置。同时对于一些动画属性如 rotate 和 skew，通过 setData 分开设置时无法同时生效，需通过 settting 统一修改。
   * @supported weapp, alipay
   */
  setting?: MapProps | { [key: string]: number | string | any }

  /** 点击地图时触发
   * @supported weapp, swan, alipay
   */
  onTap?: CommonEventFunction

  /** 点击标记点时触发，e.detail = {markerId}
   * @supported weapp, swan, alipay
   */
  onMarkerTap?: CommonEventFunction<MapProps.onMarkerTapEventDetail>

  /** 点击label时触发，e.detail = {markerId}
   * @supported weapp
   */
  onLabelTap?: CommonEventFunction<MapProps.onLabelTapEventDetail>

  /** 点击控件时触发，e.detail = {controlId}
   * @supported weapp, swan, alipay
   */
  onControlTap?: CommonEventFunction<MapProps.onControlTapEventDetail>

  /** 点击标记点对应的气泡时触发，e.detail = {markerId}
   * @supported weapp, swan, alipay
   */
  onCalloutTap?: CommonEventFunction<MapProps.onCalloutTapEventDetail>

  /** 在地图渲染更新完成时触发
   * @supported weapp, swan
   */
  onUpdated?: CommonEventFunction

  /** 视野发生变化时触发
   * @supported weapp, swan, alipay
   */
  onRegionChange?: CommonEventFunction<MapProps.onRegionEventDetail<'begin'> | MapProps.onRegionEventDetail<'end'>>

  /** 点击地图poi点时触发，e.detail = {name, longitude, latitude}
   * @supported weapp, swan
   */
  onPoiTap?: CommonEventFunction<MapProps.onPoiTapEventDetail>

  /** 视野在地图 padding 范围内展示
   * @supported alipay
   */
  includePadding?: { [key in ('left' | 'right' | 'top' | 'bottom')]: number | string }

  /** 覆盖物，自定义贴图
   * @supported alipay
   */
  groundOverlays?: any[]

  /** 覆盖物，网格贴图
   * @supported alipay
   */
  tileOverlay?: any[]

  /** 开启 optimize 模式后，无需再监听 onRegionChange 来获取并设置新的 scale 值以保证地图不会再回到原来的缩放比例。
   * @supported alipay
   */
  optimize?: boolean
}

declare namespace MapProps {
  /** 标记点用于在地图上显示标记的位置 */
  interface marker {
    /** 标记点id
     * @remarks marker 点击事件回调会返回此id。建议为每个 marker 设置上 Number 类型 id，保证更新 marker 时有更好的性能。
     */
    id?: number
    /** 纬度
     * @remarks 浮点数，范围 -90 ~ 90
     */
    latitude: number
    /** 经度
     * @remarks 浮点数，范围 -180 ~ 180
     */
    longitude: number
    /** 标注点名
     * @remarks 点击时显示，callout 存在时将被忽略
     */
    title?: string
    /** 显示层级
     */
    zIndex?: number
    /** 显示的图标
     * @remarks 项目目录下的图片路径，支持相对路径写法，以'/'开头则表示相对小程序根目录；也支持临时路径和网络图片
     */
    iconPath: string
    /** 旋转角度
     * @remarks 顺时针旋转的角度，范围 0 ~ 360，默认为 0
     */
    rotate?: number
    /** 标注的透明度
     * @remarks 默认1，无透明，范围 0 ~ 1
     */
    alpha?: number
    /** 标注图标宽度
     * @remarks 默认为图片实际宽度
     */
    width?: number | string
    /** 标注图标高度
     * @remarks 默认为图片实际高度
     */
    height?: number | string
    /** 标记点上方的气泡窗口
     * @remarks 支持的属性见下表，可识别换行符。
     */
    callout?: callout
    /** 自定义气泡窗口
     * @remarks 支持的属性见下表，可识别换行符。
     */
    customCallout?: customCallout
    /** 为标记点旁边增加标签
     * @remarks 支持的属性见下表
     */
    label?: label
    /** 经纬度在标注图标的锚点，默认底边中点
     * @remarks {x, y}，x表示横向(0-1)，y表示竖向(0-1)。{x: .5, y: 1} 表示底边中点
     */
    anchor?: {
      x: number
      y: number
    }
    /** 无障碍访问，（属性）元素的额外描述 */
    ariaLabel?: string
  }

  /** marker 上的气泡 callout */
  interface callout {
    /** 文本 */
    content: string
    /** 文本颜色 */
    color: string
    /** 文字大小 */
    fontSize: number
    /** 横向偏移量，向右为正数 */
    anchorX: number
    /** 纵向偏移量，向下为正数 */
    anchorY: number
    /** 边框圆角 */
    borderRadius: number
    /** 边框宽度 */
    borderWidth: number
    /** 边框颜色 */
    borderColor: string
    /** 背景色 */
    bgColor: string
    /** 文本边缘留白 */
    padding: number
    /** 'BYCLICK':点击显示; 'ALWAYS':常显 */
    display: 'BYCLICK' | 'ALWAYS'
    /** 文本对齐方式。有效值: left, right, center */
    textAlign: 'left' | 'right' | 'center'
  }

  /** marker 上的自定义气泡 customCallout
   *
   * `customCallout` 存在时将忽略 `callout` 与 `title` 属性。自定义气泡采用采用 `cover-view` 定制，灵活度更高。
   */
  interface customCallout {
    /** 'BYCLICK':点击显示; 'ALWAYS':常显 */
    display: 'BYCLICK' | 'ALWAYS' | string
    /** 横向偏移量，向右为正数 */
    anchorX: number
    /** 纵向偏移量，向下为正数 */
    anchorY: number
  }

  /** marker 上的气泡 label */
  interface label {
    /** 文本 */
    content: string
    /** 文本颜色 */
    color: string
    /** 文字大小 */
    fontSize: number
    /** label的坐标（废弃）
     * @deprecated
     */
    x: number
    /** label的坐标（废弃）
     * @deprecated
     */
    y: number
    /** label的坐标，原点是 marker 对应的经纬度 */
    anchorX: number
    /** label的坐标，原点是 marker 对应的经纬度 */
    anchorY: number
    /** 边框宽度 */
    borderWidth: number
    /** 边框颜色 */
    borderColor: string
    /** 边框圆角 */
    borderRadius: number
    /** 背景色 */
    bgColor: string
    /** 文本边缘留白 */
    padding: number
    /** 文本对齐方式。有效值: left, right, center */
    textAlign: 'left' | 'right' | 'center'
  }

  /** 指定一系列坐标点，从数组第一项连线至最后一项 */
  interface polyline {
    /** 经纬度数组
     * @remarks [{latitude: 0, longitude: 0}]
     */
    points: point[]
    /** 线的颜色
     * @remarks 十六进制
     */
    color?: string
    /** 线的宽度 */
    width?: number
    /** 是否虚线
     * @remarks 默认 false
     */
    dottedLine?: boolean
    /** 带箭头的线
     * @remarks 默认 false，开发者工具暂不支持该属性
     */
    arrowLine?: boolean
    /** 更换箭头图标
     * @remarks 在 arrowLine 为 true 时生效
     */
    arrowIconPath?: string
    /** 线的边框颜色 */
    borderColor?: string
    /** 线的厚度 */
    borderWidth?: number
  }

  /** 指定一系列坐标点，根据 points 坐标数据生成闭合多边形 */
  interface polygon {
    /** 经纬度数组
     * @remarks [{latitude: 0, longitude: 0}]
     */
    points: point[]
    /** 描边的宽度 */
    strokeWidth?: number
    /** 描边的颜色
     * @remarks 十六进制
     */
    strokeColor?: string
    /** 填充颜色
     * @remarks 十六进制
     */
    fillColor?: string
    /** 设置多边形Z轴数值 */
    zIndex?: number
  }

  /** 在地图上显示圆 */
  interface circle {
    /** 纬度
     * @remarks 浮点数，范围 -90 ~ 90
     */
    latitude: number
    /** 经度
     * @remarks 浮点数，范围 -180 ~ 180
     */
    longitude?: number
    /** 描边的颜色
     * @remarks 十六进制
     */
    color?: string
    /** 填充颜色
     * @remarks 十六进制
     */
    fillColor?: string
    /** 半径 */
    radius: number
    /** 描边的宽度 */
    strokeWidth?: number
  }

  /** 在地图上显示控件，控件不随着地图移动。即将废弃，请使用 cover-view
   * @deprecated
   */
  interface control {
    /** 控件id
     * @remarks 在控件点击事件回调会返回此id
     */
    id?: number
    /** 控件在地图的位置
     * @remarks 控件相对地图位置
     */
    position: position
    /** 显示的图标
     * @remarks 项目目录下的图片路径，支持本地路径、代码包路径
     */
    iconPath: string
    /** 是否可点击
     * @remarks 默认不可点击
     */
    clickable?: boolean
  }

  interface point {
    /** 经度 */
    longitude: number
    /** 纬度 */
    latitude: number
  }

  interface position {
    /** 距离地图的左边界多远
     * @default 0
     */
    left: number
    /** 距离地图的上边界多远
     * @default 0
     */
    top: number
    /** 控件宽度
     * @default 图片宽度
     */
    width: number
    /** 控件高度
     * @default 图片宽度
     */
    height: number
  }

  interface onMarkerTapEventDetail {
    markerId: number | string
  }
  interface onLabelTapEventDetail {
    markerId: number | string
  }
  interface onControlTapEventDetail {
    controlId: number | string
  }
  interface onCalloutTapEventDetail {
    markerId: number | string
  }

  namespace RegionChangeDetail {
    interface type {
      begin
      end
    }

    interface CausedByBegin {
      /** 手势触发 */
      gesture
      /** 接口触发 */
      update
    }

    interface CausedByEnd {
      /** 拖动导致 */
      drag
      /** 缩放导致 */
      scale
      /** 调用更新接口导致 */
      update
    }
  }
  interface onRegionEventDetail<T = keyof RegionChangeDetail.type> {
    /** 视野变化开始、结束时触发
     * @remarks 视野变化开始为begin，结束为end
     */
    type: T
    /** 导致视野变化的原因
     * @remarks 有效值为 gesture（手势触发）、update（接口触发或调用更新接口导致）、drag（拖动导致）、scale（缩放导致）
     */
    causedBy: keyof (T extends 'begin' ? RegionChangeDetail.CausedByBegin : RegionChangeDetail.CausedByEnd)
    /** 视野改变详情 */
    detail: regionChangeDetail<RegionChangeDetail.type>
  }
  interface regionChangeDetail<T = keyof RegionChangeDetail.type> {
    /** 旋转角度 */
    rotate: number
    /** 倾斜角度 */
    skew: number
    causedBy: keyof (T extends 'begin' ? RegionChangeDetail.CausedByBegin : RegionChangeDetail.CausedByEnd)
    type: T | string
    scale: number
    centerLocation: point
    region: {
      northeast: point
      southeast: point
    }
  }
  interface onPoiTapEventDetail {
    name: string
    longitude: number
    latitude: number
  }
}

/** 地图。相关api Taro.createMapContext。
 * @classification maps
 * @supported weapp, alipay, swan
 * @example_react
 * ```tsx
 * class App extends Component {
 *   onTap () {}
 *   render () {
 *     return (
 *       <Map onClick={this.onTap} />
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <map
 *     id="map"
 *     style="width: 100%; height: 300px;"
 *     longitude="113.324520"
 *     latitude="23.099994"
 *     scale="14"
 *     :markers="markers"
 *     :polyline="polyline"
 *     :show-location="true"
 *     `@regionchange="regionchange"
 *     `@markertap="markertap"
 *   />
 * </template>
 *
 * <script>
 * export default {
 *   data() {
 *     return {
 *       markers: [{
 *         iconPath: "https://avatars2.githubusercontent.com/u/1782542?s=460&u=d20514a52100ed1f82282bcfca6f49052793c889&v=4",
 *         id: 0,
 *         latitude: 23.099994,
 *         longitude: 113.324520,
 *         width: 50,
 *         height: 50
 *       }],
 *       polyline: [{
 *         points: [{
 *           longitude: 113.3245211,
 *           latitude: 23.10229
 *         }, {
 *           longitude: 113.324520,
 *           latitude: 23.21229
 *         }],
 *         color:"#FF0000DD",
 *         width: 2,
 *         dottedLine: true
 *       }]
 *     }
 *   },
 *   methods: {
 *     regionchange(e) {
 *       console.log(e.type)
 *     },
 *     markertap(e) {
 *       console.log("markertap:", e.detail.markerId)
 *     }
 *   }
 * }
 * </script>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/map.html#map
 */
declare const Map: ComponentType<MapProps>

interface CanvasProps extends StandardProps<any, CanvasTouchEvent> {
  /** 指定 canvas 类型，支持 2d 和 webgl
   * @supported weapp
   */
  type?: string

  /** canvas 组件的唯一标识符，若指定了 type 则无需再指定该属性
   * @supported weapp
   */
  canvasId?: string

  /** 当在 canvas 中移动时且有绑定手势事件时，禁止屏幕滚动以及下拉刷新
   * @default false
   * @supported weapp
   */
  disableScroll?: boolean

  /** 手指触摸动作开始
   * @supported weapp
   */
  onTouchStart?: CanvasTouchEventFunction

  /** 手指触摸后移动
   * @supported weapp
   */
  onTouchMove?: CanvasTouchEventFunction

  /** 手指触摸动作结束
   * @supported weapp
   */
  onTouchEnd?: CanvasTouchEventFunction

  /** 手指触摸动作被打断，如来电提醒，弹窗
   * @supported weapp
   */
  onTouchCancel?: CanvasTouchEventFunction

  /** 手指长按 500ms 之后触发，触发了长按事件后进行移动不会触发屏幕的滚动
   * @supported weapp
   */
  onLongTap?: CommonEventFunction

  /** 当发生错误时触发 error 事件，detail = {errMsg: 'something wrong'}
   * @supported weapp
   */
  onError?: CommonEventFunction<CanvasProps.onErrorEventDetail>

  /** 用于透传 `WebComponents` 上的属性到内部 H5 标签上
   * @supported h5
   */
  nativeProps?: Record<string, unknown>
}

declare namespace CanvasProps {
  interface onErrorEventDetail {
    errMsg: string
  }
}

/** 画布
 *
 * `<Canvas />` 组件的 RN 版本尚未实现。
 * @classification canvas
 * @supported weapp
 * @example_react
 * ```tsx
 * class App extends Components {
 *   render () {
 *     // 如果是支付宝小程序，则要加上 id 属性，值和canvasId一致
 *     return (
 *       <Canvas style='width: 300px; height: 200px;' canvasId='canvas' />
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <!-- 如果是支付宝小程序，则要加上 id 属性，值和canvasId一致 -->
 *   <canvas style="width: 300px; height: 200px;" canvas-id="canvas" />
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/canvas.html
 */
declare const Canvas: ComponentType<CanvasProps>

interface AdProps extends StandardProps {
  /** 广告单元id，可在[小程序管理后台](https://mp.weixin.qq.com/)的流量主模块新建
   * @supported weapp
   */
  unitId: string

  /** 广告自动刷新的间隔时间，单位为秒，参数值必须大于等于30（该参数不传入时 Banner 广告不会自动刷新）
   * @supported weapp
   */
  adIntervals?: number

  /** 广告类型，默认为展示`banner`，可通过设置该属性为`video`展示视频广告, `grid`为格子广告
   * @supported weapp
   */
  adType?: 'banner' | 'video' | 'grid'

  /** 广告主题样式设置
   * @supported weapp
   */
  adTheme?: 'white' | 'black'

  /** 广告加载成功的回调
   * @supported weapp
   */
  onLoad?: CommonEventFunction

  /** 当广告发生错误时，触发的事件，可以通过该事件获取错误码及原因，事件对象event.detail = {errCode: 1002}
   * @supported weapp
   */
  onError?: CommonEventFunction<AdProps.onErrorEventDetail>

  /** 广告关闭的回调
   * @supported weapp
   */
  onClose?: CommonEventFunction
}

declare namespace AdProps {
  interface onErrorEventDetail {
    errCode: keyof AdErrCode
  }
  /**
   * 广告错误码
   *
   * 错误码是通过onError获取到的错误信息。调试期间，可以通过异常返回来捕获信息。
   * 在小程序发布上线之后，如果遇到异常问题，可以在[“运维中心“](https://mp.weixin.qq.com/)里面搜寻错误日志，还可以针对异常返回加上适当的监控信息。
   */
  interface AdErrCode {
    /**
     * @illustrate 后端错误调用失败
     * @reason 该项错误不是开发者的异常情况
     * @solution 一般情况下忽略一段时间即可恢复。
     */
    1000
    /**
     * @illustrate 参数错误
     * @reason 使用方法错误
     * @solution 可以前往 developers.weixin.qq.com 确认具体教程（小程序和小游戏分别有各自的教程，可以在顶部选项中，“设计”一栏的右侧进行切换。
     */
    1001
    /**
     * @illustrate 广告单元无效
     * @reason 可能是拼写错误、或者误用了其他APP的广告ID
     * @solution 请重新前往 mp.weixin.qq.com 确认广告位ID。
     */
    1002
    /**
     * @illustrate 内部错误
     * @reason 该项错误不是开发者的异常情况
     * @solution 一般情况下忽略一段时间即可恢复。
     */
    1003
    /**
     * @illustrate 无合适的广告
     * @reason 广告不是每一次都会出现，这次没有出现可能是由于该用户不适合浏览广告
     * @solution 属于正常情况，且开发者需要针对这种情况做形态上的兼容。
     */
    1004
    /**
     * @illustrate 广告组件审核中
     * @reason 你的广告正在被审核，无法展现广告
     * @solution 请前往 mp.weixin.qq.com 确认审核状态，且开发者需要针对这种情况做形态上的兼容。
     */
    1005
    /**
     * @illustrate 广告组件被驳回
     * @reason 你的广告审核失败，无法展现广告
     * @solution 请前往 mp.weixin.qq.com 确认审核状态，且开发者需要针对这种情况做形态上的兼容。
     */
    1006
    /**
     * @illustrate 广告组件被封禁
     * @reason 你的广告能力已经被封禁，封禁期间无法展现广告
     * @solution 请前往 mp.weixin.qq.com 确认小程序广告封禁状态。
     */
    1007
    /**
     * @illustrate 广告单元已关闭
     * @reason 该广告位的广告能力已经被关闭
     * @solution 请前往 mp.weixin.qq.com 重新打开对应广告位的展现。
     */
    1008
    // [key: number]: string
  }
}

/** Banner 广告
 * @classification open
 * @supported weapp
 * @example_react
 * ```tsx
 * class App extends Component {
 *   render () {
 *     return (
 *       <Ad
 *         unitId=''
 *         adIntervals={60}
 *         onLoad={() => console.log('ad onLoad')}
 *         onError={() => console.log('ad onError')}
 *         onClose={() => console.log('ad onClose')}
 *       />
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <ad
 *     unit-id=""
 *     ad-intervals="60"
 *     `@load="onLoad"
 *     `@error="onError"
 *     `@close="onClose"
 *   />
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/ad.html
 */
declare const Ad: ComponentType<AdProps>

interface AdCustomProps extends StandardProps {
  /** 广告单元id，可在[小程序管理后台](https://mp.weixin.qq.com/)的流量主模块新建
   * @supported weapp
   */
  unitId: string

  /** 广告自动刷新的间隔时间，单位为秒，参数值必须大于等于30（该参数不传入时 Banner 广告不会自动刷新）
   * @supported weapp
   */
  adIntervals?: number

  /** 广告加载成功的回调
   * @supported weapp
   */
  onLoad?: CommonEventFunction

  /** 当广告发生错误时，触发的事件，可以通过该事件获取错误码及原因，事件对象event.detail = {errCode: 1002}
   * @supported weapp
   */
  onError?: CommonEventFunction<AdProps.onErrorEventDetail>
}

/** Banner 广告
 * @classification open
 * @supported weapp
 * @example
 * ```tsx
 * class App extends Component {
 *   render () {
 *     return (
 *       <AdCustom
 *         unitId=''
 *         adIntervals={60}
 *         onLoad={() => console.log('ad onLoad')}
 *         onError={() => console.log('ad onError')}
 *         onClose={() => console.log('ad onClose')}
 *       />
 *     )
 *   }
 * }
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/ad.html
 */
declare const AdCustom: ComponentType<AdCustomProps>

interface OfficialAccountProps extends StandardProps {
  /** 组件加载成功时触发
   * @supported weapp
   */
  onLoad?: CommonEventFunction<OfficialAccountProps.Detail>

  /** 组件加载失败时触发
   * @supported weapp
   */
  onError?: CommonEventFunction<OfficialAccountProps.Detail>
}

declare namespace OfficialAccountProps {
  /** detail 对象 */
  interface Detail {
    /** 状态码 */
    status: number
    /** 错误信息 */
    errMsg: string
  }
  /** status 有效值 */
  interface Status {
    /** 网络错误 */
    '-2'
    /** 数据解析错误 */
    '-1'
    /** 加载成功 */
    0
    /** 小程序关注公众号功能被封禁 */
    1
    /** 关联公众号被封禁 */
    2
    /** 关联关系解除或未选中关联公众号 */
    3
    /** 未开启关注公众号功能 */
    4
    /** 场景值错误 */
    5
    /** 重复创建 */
    6
  }
}

/** 公众号关注组件。当用户扫小程序码打开小程序时，开发者可在小程序内配置公众号关注组件，方便用户快捷关注公众号，可嵌套在原生组件内。
 * 
 * Tips
 * 使用组件前，需前往小程序后台，在“设置”->“关注公众号”中设置要展示的公众号。注：设置的公众号需与小程序主体一致。
 * 
 * 在一个小程序的生命周期内，只有从以下场景进入小程序，才具有展示引导关注公众号组件的能力:
 * 
 * 当小程序从扫小程序码场景（场景值1047，场景值1124）打开时
 * 当小程序从聊天顶部场景（场景值1089）中的「最近使用」内打开时，若小程序之前未被销毁，则该组件保持上一次打开小程序时的状态
 * 当从其他小程序返回小程序（场景值1038）时，若小程序之前未被销毁，则该组件保持上一次打开小程序时的状态
 * @classification open
 * @supported weapp
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/official-account.html
 */
declare const OfficialAccount: ComponentType<OfficialAccountProps>

interface OpenDataProps extends StandardProps {
  /** 开放数据类型
   * @supported weapp
   */
  type: keyof OpenDataProps.Type
  /** 当 type="groupName" 时生效, 群id
   * @supported weapp
   */
  openGid?: string
  /** 当 type="user*" 时生效，以哪种语言展示 userInfo
   * @default "en"
   * @supported weapp
   */
  lang?: keyof OpenDataProps.Lang

  /** 数据为空时的默认文案
   * @supported weapp
   */
  defaultText?: string

  /** 用户头像为空时的默认图片，支持相对路径和网络图片路径
   * @supported weapp
   */
  defaultAvatar?: string

  /** 群名称或用户信息为空时触发
   * @supported weapp
   */
  onError?: CommonEventFunction
}

declare namespace OpenDataProps {
  /** type 的合法值 */
  interface Type {
    /** 拉取群名称 */
    groupName
    /** 用户昵称 */
    userNickName
    /** 用户头像 */
    userAvatarUrl
    /** 用户性别 */
    userGender
    /** 用户所在城市 */
    userCity
    /** 用户所在省份 */
    userProvince
    /** 用户所在国家 */
    userCountry
    /** 用户的语言 */
    userLanguage
  }
  /** lang 的合法值 */
  interface Lang {
    /** 英文 */
    en
    /** 简体中文 */
    zh_CN
    /** 繁体中文 */
    zh_TW
  }
}

/** 用于展示微信开放的数据
 * @classification open
 * @supported weapp
 * @example_react
 * ```tsx
 * class App extends Component {
 *   render () {
 *     return (
 *       <OpenData type='userAvatarUrl'/>
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <open-data type="userAvatarUrl" />
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/open-data.html
 */
declare const OpenData: ComponentType<OpenDataProps>

interface WebViewProps extends StandardProps {
  /** webview 指向网页的链接。可打开关联的公众号的文章，其它网页需登录小程序管理后台配置业务域名。
   * @supported weapp, h5
   */
  src: string

  /** 网页向小程序 postMessage 时，会在特定时机（小程序后退、组件销毁、分享）触发并收到消息。e.detail = { data }
   * @supported weapp
   */
  onMessage?: CommonEventFunction<WebViewProps.onMessageEventDetail>

  /** 网页加载成功时候触发此事件。e.detail = { src }
   * @supported weapp, h5
   */
  onLoad?: CommonEventFunction<WebViewProps.onLoadEventDetail>

  /** 网页加载失败的时候触发此事件。e.detail = { src }
   * @supported weapp, h5
   */
  onError?: CommonEventFunction<WebViewProps.onErrorEventDetail>
}

declare namespace WebViewProps {
  interface onMessageEventDetail {
    /** 消息数据，是多次 postMessage 的参数组成的数组 */
    data: any[]
  }
  interface onLoadEventDetail {
    /** 网页链接 */
    src: string
  }
  interface onErrorEventDetail {
    /** 网页链接 */
    src: string
  }
}

/** web-view 组件是一个可以用来承载网页的容器，会自动铺满整个小程序页面。个人类型与海外类型的小程序暂不支持使用。
 * @classification open
 * @supported weapp, h5, rn
 * @example_react
 * ```tsx
 * class App extends Component {
 *   handleMessage () {}
 *   
 *   render () {
 *     return (
 *       <WebView src='https://mp.weixin.qq.com/' onMessage={this.handleMessage} />
 *     )
 *   }
 * }
 * ```
 * @example_vue
 * ```html
 * <template>
 *   <web-view src='https://mp.weixin.qq.com/' `@message="handleMessage" />
 * </template>
 * ```
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/web-view.html
 */
declare const WebView: ComponentType<WebViewProps>

interface PageMetaProps extends StandardProps {
  /** 下拉背景字体、loading 图的样式，仅支持 dark 和 light
   * @supported weapp
   */
  backgroundTextStyle?: string

  /** 窗口的背景色，必须为十六进制颜色值
   * @supported weapp
   */
  backgroundColor?: string

  /** 顶部窗口的背景色，必须为十六进制颜色值，仅 iOS 支持
   * @supported weapp
   */
  backgroundColorTop?: string

  /** 底部窗口的背景色，必须为十六进制颜色值，仅 iOS 支持
   * @supported weapp
   */
  backgroundColorBottom?: string

  /** 滚动位置，可以使用 px 或者 rpx 为单位，在被设置时，页面会滚动到对应位置
   * @default ""
   * @supported weapp
   */
  scrollTop?: string

  /** 滚动动画时长
   * @default 300
   * @supported weapp
   */
  scrollDuration?: number

  /** 页面根节点样式，页面根节点是所有页面节点的祖先节点，相当于 HTML 中的 body 节点
   * @default ""
   * @supported weapp
   */
  pageStyle?: string

  /** 页面的根字体大小，页面中的所有 rem 单位，将使用这个字体大小作为参考值，即 1rem 等于这个字体大小
   * @default ""
   * @supported weapp
   */
  rootFontSize?: string

  /** 页面尺寸变化时会触发 resize 事件，event.detail = { size: { windowWidth, windowHeight } }
   * @supported weapp
   */
  onResize?: CommonEventFunction<PageMetaProps.onResizeEventDetail>

  /** 页面滚动时会触发 scroll 事件，event.detail = { scrollTop }
   * @supported weapp
   */
  onScroll?: CommonEventFunction<PageMetaProps.onScrollEventDetail>

  /** 如果通过改变 scroll-top 属性来使页面滚动，页面滚动结束后会触发 scrolldone 事件
   * @supported weapp
   */
  onScrollDone?: CommonEventFunction
}

declare namespace PageMetaProps {
  interface onResizeEventDetail {
    /** 窗口尺寸 */
    size: resizeType
  }
  /** 窗口尺寸类型 */
  interface resizeType {
    /** 窗口宽度 */
    windowWidth: number
    /** 窗口高度 */
    windowHeight: number
  }
  interface onScrollEventDetail {
    scrollTop: number
  }
}

/** 页面属性配置节点，用于指定页面的一些属性、监听页面事件。只能是页面内的第一个节点。可以配合 navigation-bar 组件一同使用。
 * 通过这个节点可以获得类似于调用 Taro.setBackgroundTextStyle Taro.setBackgroundColor 等接口调用的效果。
 * @supported weapp
 * @see https://developers.weixin.qq.com/miniprogram/dev/component/page-meta.html
 */
declare const PageMeta: ComponentType<PageMetaProps>

interface CustomWrapperProps extends StandardProps {
}

/** custom-wrapper 自定义组件包裹器
 * 当数据更新层级较深时，可用此组件将需要更新的区域包裹起来，这样更新层级将大大减少
 * @supported weapp, swan, alipay, tt, jd, qq
 * @example
 * ```tsx
 * import { Component } from 'react'
 * import { CustomWrapper, View, Text } from '@tarojs/components'
 * 
 * export default class C extends Component {
 *   render () {
 *     return (
 *       <View>
 *         <CustomWrapper>
 *            <Text>Hello, world!</Text>
 *         </CustomWrapper>
 *       </View>
 *     )
 *   }
 * }
 * ```
 */
declare const CustomWrapper: ComponentType<CustomWrapperProps>

interface SlotProps {
  /** 指定插入的 slot 位置
   * @default none
   * @supported weapp, swan, alipay, tt, jd, qq
   */
  name?: string,
  /** scoped slot 传入数据源
   * @default none
   * @supported swan
   */
  varName?: string
}

/** slot 插槽
 * @supported weapp, swan, alipay, tt, jd, qq
 * @example
 * ```tsx
 * import { Slot, View, Text } from '@tarojs/components'
 * 
 * export default class SlotView extends Component {
 *   render () {
 *     return (
 *       <View>
 *         <custom-component>
 *           <Slot name='title'>
 *            <Text>Hello, world!</Text>
 *           </Slot>
 *         </custom-component>
 *       </View>
 *     )
 *   }
 * }
 * ```
 */
declare const Slot: ComponentType<SlotProps>

export { Ad, AdCustom, AdCustomProps, AdProps, Audio, AudioProps, BaseEventOrig, BaseEventOrigFunction, Block, Button, ButtonProps, Camera, CameraProps, Canvas, CanvasProps, CanvasTouch, CanvasTouchEvent, CanvasTouchEventFunction, Checkbox, CheckboxGroup, CheckboxGroupProps, CheckboxProps, CommonEvent, CommonEventFunction, CoverImage, CoverImageProps, CoverView, CoverViewProps, CustomWrapper, CustomWrapperProps, Editor, EditorProps, EventHandler, EventProps, Form, FormItemProps, FormProps, FunctionalPageNavigator, FunctionalPageNavigatorProps, ITouch, ITouchEvent, Icon, IconProps, Image, ImageProps, Input, InputProps, KeyboardAccessory, KeyboardAccessoryProps, Label, LabelProps, LivePlayer, LivePlayerProps, LivePusher, LivePusherProps, Map, MapProps, MatchMedia, MatchMediaProps, MovableArea, MovableAreaProps, MovableView, MovableViewProps, NavigationBar, NavigationBarProps, Navigator, NavigatorProps, NetStatus, OfficialAccount, OfficialAccountProps, OpenData, OpenDataProps, PageContainer, PageContainerProps, PageMeta, PageMetaProps, Picker, PickerDateProps, PickerMultiSelectorProps, PickerRegionProps, PickerSelectorProps, PickerStandardProps, PickerTimeProps, PickerView, PickerViewColumn, PickerViewProps, Progress, ProgressProps, Radio, RadioGroup, RadioGroupProps, RadioProps, RichText, RichTextProps, ScrollView, ScrollViewProps, ShareElement, ShareElementProps, Slider, SliderProps, Slot, SlotProps, StandardProps, Swiper, SwiperItem, SwiperItemProps, SwiperProps, Switch, SwitchProps, Target, TaroEvent, Text, TextProps, Textarea, TextareaProps, TouchEventFunction, Video, VideoProps, View, ViewProps, VoipRoom, VoipRoomProps, WebView, WebViewProps, currentTarget };
