"use strict";
var data_set_1 = require('./data-set/data-set');
var helpers_1 = require('./helpers');
var Rx_1 = require('rxjs/Rx');
var _ = require('underscore');
var Grid = (function () {
    function Grid(source, settings) {
        this.createFormShown = false;
        this.onSelectRowSource = new Rx_1.Subject();
        this.setSettings(settings);
        this.setSource(source);
    }
    Grid.prototype.showActionColumn = function () {
        return this.getSetting('actions.add') || this.getSetting('actions.edit') || this.getSetting('actions.delete');
    };
    Grid.prototype.getNewRow = function () {
        return this.dataSet.newRow;
    };
    Grid.prototype.setSettings = function (settings) {
        this.settings = settings;
        this.dataSet = new data_set_1.DataSet([], this.getSetting('columns'));
    };
    Grid.prototype.updateSettings = function (settings) {
        this.settings = settings;
        this.dataSet.createColumns(settings);
    };
    Grid.prototype.getDataSet = function () {
        return this.dataSet;
    };
    Grid.prototype.getDataSource = function () {
        return this.source;
    };
    Grid.prototype.setSource = function (source) {
        var _this = this;
        this.source = this.prepareSource(source);
        this.source.onChanged().subscribe(function (changes) { return _this.processDataChange(changes); });
        this.source.onUpdated().subscribe(function (data) {
            var changedRow = _this.dataSet.findRowByData(data);
            changedRow.setData(data);
        });
    };
    Grid.prototype.getSetting = function (name, defaultValue) {
        var keys = name.split('.');
        var level = this.settings;
        keys.forEach(function (k) {
            if (level && typeof level[k] !== 'undefined') {
                level = level[k];
            }
        });
        return typeof level === 'undefined' ? defaultValue : level;
    };
    Grid.prototype.getColumns = function () {
        var columns = _.filter(this.dataSet.getColumns(), function (column) {
            return (column.isVisible == null || column.isVisible == undefined || column.isVisible === true);
        });
        return columns;
    };
    Grid.prototype.getAllColumns = function () {
        var columns = _.filter(this.dataSet.getColumns(), function (column) {
            return (column.title != null && column.title != undefined);
        });
        return columns;
    };
    Grid.prototype.getRows = function () {
        return this.dataSet.getRows();
    };
    Grid.prototype.getSelectedRows = function () {
        return this.dataSet.getSelectedRows();
    };
    Grid.prototype.selectRow = function (row) {
        this.dataSet.selectRow(row);
    };
    Grid.prototype.onSelectRow = function () {
        return this.onSelectRowSource.asObservable();
    };
    Grid.prototype.create = function (row, confirmEmitter) {
        var _this = this;
        var deferred = new helpers_1.Deferred();
        deferred.promise.then(function (newData) {
            newData = newData ? newData : row.getNewData();
            _this.source.prepend(newData).then(function () {
                _this.createFormShown = false;
                _this.dataSet.createNewRow();
            });
        }).catch(function (err) {
            // doing nothing
        });
        if (this.getSetting('add.confirmCreate')) {
            confirmEmitter.emit({
                newData: row.getNewData(),
                source: this.source,
                confirm: deferred
            });
        }
        else {
            deferred.resolve();
        }
    };
    Grid.prototype.processDataChange = function (changes) {
        if (this.shouldProcessChange(changes)) {
            this.dataSet.setData(changes['elements']);
            this.dataSet.updateRows();
        }
    };
    Grid.prototype.shouldProcessChange = function (changes) {
        if (['filter', 'sort', 'page', 'paging', 'remove', 'refresh', 'load'].indexOf(changes['action']) !== -1) {
            return true;
        }
        else if (['prepend', 'append'].indexOf(changes['action']) !== -1 && !this.getSetting('pager.display')) {
            return true;
        }
        return false;
    };
    // TODO: move to selectable? Separate directive
    Grid.prototype.determineRowToSelect = function (changes) {
        if (['load', 'page', 'filter', 'sort', 'refresh'].indexOf(changes['action']) !== -1) {
            return this.dataSet.select();
        }
        if (changes['action'] === 'add') {
            return this.dataSet.selectFirstRow();
        }
        if (changes['action'] === 'update') {
            return this.dataSet.selectFirstRow();
        }
        return null;
    };
    Grid.prototype.prepareSource = function (source) {
        var initialSource = this.getInitialSort();
        if (initialSource && initialSource['field'] && initialSource['direction']) {
            source.setSort([initialSource], false);
        }
        if (this.getSetting('pager.display') === true) {
            source.setPaging(1, this.getSetting('pager.perPage'), false);
        }
        source.refresh();
        return source;
    };
    Grid.prototype.getInitialSort = function () {
        var sortConf = {};
        this.getColumns().forEach(function (column) {
            if (column.isSortable && column.defaultSortDirection) {
                sortConf['field'] = column.id;
                sortConf['direction'] = column.defaultSortDirection;
                sortConf['compare'] = column.getCompareFunction();
            }
        });
        return sortConf;
    };
    Grid.prototype.getPreviousRow = function (currentRow) {
        var currentRowIndex = this.getRowIndex(currentRow);
        var previousRowIndex = currentRowIndex - 1;
        var previousRow;
        if (previousRowIndex > -1) {
            previousRow = this.getRows()[previousRowIndex];
        }
        return previousRow;
    };
    Grid.prototype.getRowIndex = function (row) {
        var rowIndex = -1;
        this.getRows().forEach(function (innerRow, index, rows) {
            if (innerRow.id === row.id) {
                rowIndex = index;
                return true;
            }
        });
        return rowIndex;
    };
    Grid.prototype.getNextRow = function (currentRow) {
        var rows = this.getRows();
        var nextRow;
        var currentRowIndex = this.getRowIndex(currentRow);
        var nextRowIndex = currentRowIndex + 1;
        if (nextRowIndex < rows.length - 1) {
            nextRow = rows[nextRowIndex];
        }
        return nextRow;
    };
    return Grid;
}());
exports.Grid = Grid;
//# sourceMappingURL=grid.js.map