"use strict";
/**
 * This function returns true if input is String, Date, or RegExp
 *
 * @param val
 * @returns {boolean}
 */
function isStringDateOrRegExp(val) {
    return (val instanceof String
        || val instanceof Date
        || val instanceof RegExp);
}
/**
 * This function returns clones input of String, Date, or RegExp
 *
 * @param val
 * @returns {boolean}
 */
function cloneStringDateOrRegExp(val) {
    if (val instanceof String) {
        return String(val.length);
    }
    else if (val instanceof Date) {
        return new Date(val.getTime());
    }
    else if (val instanceof RegExp) {
        return new RegExp(val);
    }
    else {
        throw new Error('Unexpected situation');
    }
}
/**
 * This will recursively clone an array.
 *
 * @param array
 * @returns {Array}
 */
function deepCloneArray(array) {
    var clone = [];
    array.forEach(function (item, index) {
        if (typeof item === 'object' && item !== null) {
            if (Array.isArray(item)) {
                clone[index] = deepCloneArray(item);
            }
            else if (isStringDateOrRegExp(item)) {
                clone[index] = cloneStringDateOrRegExp(item);
            }
            else {
                clone[index] = exports.deepExtend({}, item);
            }
        }
        else {
            clone[index] = item;
        }
    });
    return clone;
}
/**
 * Extending object that entered in first argument.
 *
 * Returns extended object or false if have no target object or incorrect type.
 *
 *     deepExtend(objectYouWantToExtend, objectToAdd_1[, objectToAdd_N]);
 *
 * If you wish to clone source object (without modifying it), just use empty new
 * object as first argument, like this:
 *
 *     deepExtend({}, yourObj_1, [yourObj_N]);
 *
 * @param objects
 * @returns {any}
 */
exports.deepExtend = function () {
    var objects = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        objects[_i - 0] = arguments[_i];
    }
    if (arguments == null || arguments == undefined || arguments.length < 1 || typeof arguments[0] !== 'object') {
        return false;
    }
    if (arguments.length < 2) {
        return arguments[0];
    }
    var targetObject = arguments[0];
    var otherObjects = Array.prototype.slice.call(arguments, 1);
    otherObjects.forEach(function (obj) {
        addOtherObjectToTarget(obj, targetObject);
    });
    return targetObject;
};
function addOtherObjectToTarget(obj, targetObject) {
    // skip argument if it is array or isn't object
    if (obj == null || obj == undefined || typeof obj !== 'object' || Array.isArray(obj)) {
        return;
    }
    Object.keys(obj).forEach(function (key) {
        var source = targetObject[key]; // source value
        var newVal = obj[key]; // new value
        // recursion prevention
        if (newVal === targetObject)
            return;
        // if new value isn't object then just overwrite with new value instead of extending.
        if (typeof newVal !== 'object' || newVal === null) {
            targetObject[key] = newVal;
            return;
        }
        // Clone arrays (and recursively clone objects inside)
        if (Array.isArray(newVal)) {
            targetObject[key] = deepCloneArray(newVal);
            return;
        }
        // Custom cloning and overwrite for String, Date, RegExp objects
        if (isStringDateOrRegExp(newVal)) {
            targetObject[key] = cloneStringDateOrRegExp(newVal);
            return;
        }
        // if source isn't object or array overwrite with new value
        if (typeof source !== 'object' || source === null || Array.isArray(source)) {
            targetObject[key] = exports.deepExtend({}, newVal); // TODO: Unexpected side effects of deleting?
            return;
        }
        // source value and new value are both objects, extending...
        targetObject[key] = exports.deepExtend(source, newVal);
        return;
    });
}
var Deferred = (function () {
    function Deferred() {
        var _this = this;
        this.promise = new Promise(function (resolve, reject) {
            _this.resolve = resolve;
            _this.reject = reject;
        });
    }
    return Deferred;
}());
exports.Deferred = Deferred;
//# sourceMappingURL=helpers.js.map