/**
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import * as api from '../protos/firestore_proto_api';
import { BatchId } from '../core/types';
import { TargetId } from '../core/types';
import { ResourcePath } from '../model/path';
import { EncodedResourcePath } from './encoded_resource_path';
export declare const SCHEMA_VERSION = 1;
/** Performs database creation and (in the future) upgrades between versions. */
export declare function createOrUpgradeDb(db: IDBDatabase, oldVersion: number): void;
/**
 * Wrapper class to store timestamps (seconds and nanos) in IndexedDb objects.
 */
export declare class DbTimestamp {
    seconds: number;
    nanos: number;
    constructor(seconds: number, nanos: number);
}
export declare type DbOwnerKey = 'owner';
/**
 * A singleton object to be stored in the 'owner' store in IndexedDb.
 *
 * A given database can be owned by a single tab at a given time. That tab
 * must validate that it is still the owner before every write operation and
 * should regularly write an updated timestamp to prevent other tabs from
 * "stealing" ownership of the db.
 */
export declare class DbOwner {
    ownerId: string;
    leaseTimestampMs: number;
    /** Name of the IndexedDb object store. */
    static store: string;
    constructor(ownerId: string, leaseTimestampMs: number);
}
/** Object keys in the 'mutationQueues' store are userId strings. */
export declare type DbMutationQueueKey = string;
/**
 * An object to be stored in the 'mutationQueues' store in IndexedDb.
 *
 * Each user gets a single queue of MutationBatches to apply to the server.
 * DbMutationQueue tracks the metadata about the queue.
 */
export declare class DbMutationQueue {
    /**
       * The normalized user ID to which this queue belongs.
       */
    userId: string;
    /**
       * An identifier for the highest numbered batch that has been acknowledged
       * by the server. All MutationBatches in this queue with batchIds less
       * than or equal to this value are considered to have been acknowledged by
       * the server.
       */
    lastAcknowledgedBatchId: number;
    /**
       * A stream token that was previously sent by the server.
       *
       * See StreamingWriteRequest in datastore.proto for more details about
       * usage.
       *
       * After sending this token, earlier tokens may not be used anymore so
       * only a single stream token is retained.
       */
    lastStreamToken: string;
    /** Name of the IndexedDb object store.  */
    static store: string;
    /** Keys are automatically assigned via the userId property. */
    static keyPath: string;
    constructor(
        /**
           * The normalized user ID to which this queue belongs.
           */
        userId: string, 
        /**
           * An identifier for the highest numbered batch that has been acknowledged
           * by the server. All MutationBatches in this queue with batchIds less
           * than or equal to this value are considered to have been acknowledged by
           * the server.
           */
        lastAcknowledgedBatchId: number, 
        /**
           * A stream token that was previously sent by the server.
           *
           * See StreamingWriteRequest in datastore.proto for more details about
           * usage.
           *
           * After sending this token, earlier tokens may not be used anymore so
           * only a single stream token is retained.
           */
        lastStreamToken: string);
}
/** keys in the 'mutations' object store are [userId, batchId] pairs. */
export declare type DbMutationBatchKey = [string, BatchId];
/**
 * An object to be stored in the 'mutations' store in IndexedDb.
 *
 * Represents a batch of user-level mutations intended to be sent to the server
 * in a single write. Each user-level batch gets a separate DbMutationBatch
 * with a new batchId.
 */
export declare class DbMutationBatch {
    /**
       * The normalized user ID to which this batch belongs.
       */
    userId: string;
    /**
       * An identifier for this batch, allocated by the mutation queue in a
       * monotonically increasing manner.
       */
    batchId: BatchId;
    /**
       * The local write time of the batch, stored as milliseconds since the
       * epoch.
       */
    localWriteTimeMs: number;
    /**
       * A list of mutations to apply. All mutations will be applied atomically.
       *
       * Mutations are serialized via JsonProtoSerializer.toMutation().
       */
    mutations: api.Write[];
    /** Name of the IndexedDb object store.  */
    static store: string;
    /** Keys are automatically assigned via the userId, batchId properties. */
    static keyPath: string[];
    constructor(
        /**
           * The normalized user ID to which this batch belongs.
           */
        userId: string, 
        /**
           * An identifier for this batch, allocated by the mutation queue in a
           * monotonically increasing manner.
           */
        batchId: BatchId, 
        /**
           * The local write time of the batch, stored as milliseconds since the
           * epoch.
           */
        localWriteTimeMs: number, 
        /**
           * A list of mutations to apply. All mutations will be applied atomically.
           *
           * Mutations are serialized via JsonProtoSerializer.toMutation().
           */
        mutations: api.Write[]);
}
/**
 * The key for a db document mutation, which is made up of a userID, path, and
 * batchId. Note that the path must be serialized into a form that indexedDB can
 * sort.
 */
export declare type DbDocumentMutationKey = [string, EncodedResourcePath, BatchId];
/**
 * An object to be stored in the 'documentMutations' store in IndexedDb.
 *
 * A manually maintained index of all the mutation batches that affect a given
 * document key. The rows in this table are references based on the contents of
 * DbMutationBatch.mutations.
 */
export declare class DbDocumentMutation {
    static store: string;
    /**
     * Creates a [userId] key for use in the DbDocumentMutations index to iterate
     * over all of a user's document mutations.
     */
    static prefixForUser(userId: string): [string];
    /**
     * Creates a [userId, encodedPath] key for use in the DbDocumentMutations
     * index to iterate over all at document mutations for a given path or lower.
     */
    static prefixForPath(userId: string, path: ResourcePath): [string, EncodedResourcePath];
    /**
     * Creates a full index key of [userId, encodedPath, batchId] for inserting
     * and deleting into the DbDocumentMutations index.
     */
    static key(userId: string, path: ResourcePath, batchId: BatchId): DbDocumentMutationKey;
    /**
     * Because we store all the useful information for this store in the key,
     * there is no useful information to store as the value. The raw (unencoded)
     * path cannot be stored because IndexedDb doesn't store prototype
     * information.
     */
    static PLACEHOLDER: DbDocumentMutation;
    private constructor();
}
/**
 * A key in the 'remoteDocuments' object store is a string array containing the
 * segments that make up the path.
 */
export declare type DbRemoteDocumentKey = string[];
/**
 * Represents the known absence of a document at a particular version.
 * Stored in IndexedDb as part of a DbRemoteDocument object.
 */
export declare class DbNoDocument {
    path: string[];
    readTime: DbTimestamp;
    constructor(path: string[], readTime: DbTimestamp);
}
/**
 * An object to be stored in the 'remoteDocuments' store in IndexedDb. It
 * represents either a cached document (if it exists) or a cached "no-document"
 * (if it is known to not exist).
 *
 * Note: This is the persisted equivalent of a MaybeDocument and could perhaps
 * be made more general if necessary.
 */
export declare class DbRemoteDocument {
    /**
       * Set to an instance of a DbNoDocument if it is known that no document
       * exists.
       */
    noDocument: DbNoDocument | null;
    /**
       * Set to an instance of a Document if there's a cached version of the
       * document.
       */
    document: api.Document | null;
    static store: string;
    constructor(
        /**
           * Set to an instance of a DbNoDocument if it is known that no document
           * exists.
           */
        noDocument: DbNoDocument | null, 
        /**
           * Set to an instance of a Document if there's a cached version of the
           * document.
           */
        document: api.Document | null);
}
/**
 * A key in the 'targets' object store is a targetId of the query.
 */
export declare type DbTargetKey = TargetId;
/**
 * The persisted type for a query nested with in the 'targets' store in
 * IndexedDb. We use the proto definitions for these two kinds of queries in
 * order to avoid writing extra serialization logic.
 */
export declare type DbQuery = api.QueryTarget | api.DocumentsTarget;
/**
 * An object to be stored in the 'targets' store in IndexedDb.
 *
 * This is based on and should be kept in sync with the proto used in the iOS
 * client.
 *
 * Each query the client listens to against the server is tracked on disk so
 * that the query can be efficiently resumed on restart.
 */
export declare class DbTarget {
    /**
       * An auto-generated sequential numeric identifier for the query.
       *
       * Queries are stored using their canonicalId as the key, but these
       * canonicalIds can be quite long so we additionally assign a unique
       * queryId which can be used by referenced data structures (e.g.
       * indexes) to minimize the on-disk cost.
       */
    targetId: TargetId;
    /**
       * The canonical string representing this query. This is not unique.
       */
    canonicalId: string;
    /**
       * The last readTime received from the Watch Service for this query.
       *
       * This is the same value as TargetChange.read_time in the protos.
       */
    readTime: DbTimestamp;
    /**
       * An opaque, server-assigned token that allows watching a query to be
       * resumed after disconnecting without retransmitting all the data
       * that matches the query. The resume token essentially identifies a
       * point in time from which the server should resume sending results.
       *
       * This is related to the snapshotVersion in that the resumeToken
       * effectively also encodes that value, but the resumeToken is opaque
       * and sometimes encodes additional information.
       *
       * A consequence of this is that the resumeToken should be used when
       * asking the server to reason about where this client is in the watch
       * stream, but the client should use the snapshotVersion for its own
       * purposes.
       *
       * This is the same value as TargetChange.resume_token in the protos.
       */
    resumeToken: string;
    /**
       * A sequence number representing the last time this query was
       * listened to, used for garbage collection purposes.
       *
       * Conventionally this would be a timestamp value, but device-local
       * clocks are unreliable and they must be able to create new listens
       * even while disconnected. Instead this should be a monotonically
       * increasing number that's incremented on each listen call.
       *
       * This is different from the queryId since the queryId is an
       * immutable identifier assigned to the Query on first use while
       * lastListenSequenceNumber is updated every time the query is
       * listened to.
       */
    lastListenSequenceNumber: number;
    /**
       * The query for this target.
       *
       * Because canonical ids are not unique we must store the actual query. We
       * use the proto to have an object we can persist without having to
       * duplicate translation logic to and from a `Query` object.
       */
    query: DbQuery;
    static store: string;
    /** Keys are automatically assigned via the targetId property. */
    static keyPath: string;
    /** The name of the queryTargets index. */
    static queryTargetsIndexName: string;
    /**
     * The index of all canonicalIds to the targets that they match. This is not
     * a unique mapping because canonicalId does not promise a unique name for all
     * possible queries, so we append the targetId to make the mapping unique.
     */
    static queryTargetsKeyPath: string[];
    constructor(
        /**
           * An auto-generated sequential numeric identifier for the query.
           *
           * Queries are stored using their canonicalId as the key, but these
           * canonicalIds can be quite long so we additionally assign a unique
           * queryId which can be used by referenced data structures (e.g.
           * indexes) to minimize the on-disk cost.
           */
        targetId: TargetId, 
        /**
           * The canonical string representing this query. This is not unique.
           */
        canonicalId: string, 
        /**
           * The last readTime received from the Watch Service for this query.
           *
           * This is the same value as TargetChange.read_time in the protos.
           */
        readTime: DbTimestamp, 
        /**
           * An opaque, server-assigned token that allows watching a query to be
           * resumed after disconnecting without retransmitting all the data
           * that matches the query. The resume token essentially identifies a
           * point in time from which the server should resume sending results.
           *
           * This is related to the snapshotVersion in that the resumeToken
           * effectively also encodes that value, but the resumeToken is opaque
           * and sometimes encodes additional information.
           *
           * A consequence of this is that the resumeToken should be used when
           * asking the server to reason about where this client is in the watch
           * stream, but the client should use the snapshotVersion for its own
           * purposes.
           *
           * This is the same value as TargetChange.resume_token in the protos.
           */
        resumeToken: string, 
        /**
           * A sequence number representing the last time this query was
           * listened to, used for garbage collection purposes.
           *
           * Conventionally this would be a timestamp value, but device-local
           * clocks are unreliable and they must be able to create new listens
           * even while disconnected. Instead this should be a monotonically
           * increasing number that's incremented on each listen call.
           *
           * This is different from the queryId since the queryId is an
           * immutable identifier assigned to the Query on first use while
           * lastListenSequenceNumber is updated every time the query is
           * listened to.
           */
        lastListenSequenceNumber: number, 
        /**
           * The query for this target.
           *
           * Because canonical ids are not unique we must store the actual query. We
           * use the proto to have an object we can persist without having to
           * duplicate translation logic to and from a `Query` object.
           */
        query: DbQuery);
}
/**
 * The key for a DbTargetDocument, containing a targetId and an encoded resource
 * path.
 */
export declare type DbTargetDocumentKey = [TargetId, EncodedResourcePath];
/**
 * An object representing an association between a target and a document.
 * Stored in the targetDocument object store to store the documents tracked by a
 * particular target.
 */
export declare class DbTargetDocument {
    /**
       * The targetId identifying a target.
       */
    targetId: TargetId;
    /**
       * The path to the document, as encoded in the key.
       */
    path: EncodedResourcePath;
    /** Name of the IndexedDb object store.  */
    static store: string;
    /** Keys are automatically assigned via the targetId, path properties. */
    static keyPath: string[];
    /** The index name for the reverse index. */
    static documentTargetsIndex: string;
    /** We also need to create the reverse index for these properties. */
    static documentTargetsKeyPath: string[];
    constructor(
        /**
           * The targetId identifying a target.
           */
        targetId: TargetId, 
        /**
           * The path to the document, as encoded in the key.
           */
        path: EncodedResourcePath);
}
/**
 * The type to represent the single allowed key for the DbTargetGlobal store.
 */
export declare type DbTargetGlobalKey = typeof DbTargetGlobal.key;
/**
 * A record of global state tracked across all Targets, tracked separately
 * to avoid the need for extra indexes.
 *
 * This should be kept in-sync with the proto used in the iOS client.
 */
export declare class DbTargetGlobal {
    /**
       * The highest numbered target id across all targets.
       *
       * See DbTarget.targetId.
       */
    highestTargetId: TargetId;
    /**
       * The highest numbered lastListenSequenceNumber across all targets.
       *
       * See DbTarget.lastListenSequenceNumber.
       */
    highestListenSequenceNumber: number;
    /**
       * A global snapshot version representing the last consistent snapshot we
       * received from the backend. This is monotonically increasing and any
       * snapshots received from the backend prior to this version (e.g. for
       * targets resumed with a resumeToken) should be suppressed (buffered)
       * until the backend has caught up to this snapshot version again. This
       * prevents our cache from ever going backwards in time.
       */
    lastRemoteSnapshotVersion: DbTimestamp;
    /**
     * The key string used for the single object that exists in the
     * DbTargetGlobal store.
     */
    static key: string;
    static store: string;
    constructor(
        /**
           * The highest numbered target id across all targets.
           *
           * See DbTarget.targetId.
           */
        highestTargetId: TargetId, 
        /**
           * The highest numbered lastListenSequenceNumber across all targets.
           *
           * See DbTarget.lastListenSequenceNumber.
           */
        highestListenSequenceNumber: number, 
        /**
           * A global snapshot version representing the last consistent snapshot we
           * received from the backend. This is monotonically increasing and any
           * snapshots received from the backend prior to this version (e.g. for
           * targets resumed with a resumeToken) should be suppressed (buffered)
           * until the backend has caught up to this snapshot version again. This
           * prevents our cache from ever going backwards in time.
           */
        lastRemoteSnapshotVersion: DbTimestamp);
}
/**
 * The list of all IndexedDB stored used by the SDK. This is used when creating
 * transactions so that access across all stores is done atomically.
 */
export declare const ALL_STORES: string[];
