/**
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { User } from '../auth/user';
import { DatabaseInfo } from '../core/database_info';
import { JsonProtoSerializer } from '../remote/serializer';
import { MutationQueue } from './mutation_queue';
import { Persistence } from './persistence';
import { PersistencePromise } from './persistence_promise';
import { QueryCache } from './query_cache';
import { RemoteDocumentCache } from './remote_document_cache';
import { SimpleDbTransaction } from './simple_db';
/**
 * An IndexedDB-backed instance of Persistence. Data is stored persistently
 * across sessions.
 *
 * Currently the Firestore SDK only supports a single consumer of the database,
 * but browsers obviously support multiple tabs. IndexedDbPersistence ensures a
 * single consumer of the database via an "owner lease" stored in the database.
 *
 * On startup, IndexedDbPersistence assigns itself a random "ownerId" and writes
 * it to a special "owner" object in the database (if no entry exists already or
 * the current entry is expired). This owner lease is then verified inside every
 * transaction to ensure the lease has not been lost.
 *
 * If a tab opts not to acquire the owner lease (because there's an existing
 * non-expired owner) or loses the owner lease, IndexedDbPersistence enters a
 * failed state and all subsequent operations will automatically fail.
 *
 * The current owner regularly refreshes the owner lease with new timestamps to
 * prevent newly-opened tabs from taking over ownership.
 *
 * Additionally there is an optimization so that when a tab is closed, the owner
 * lease is released immediately (this is especially important to make sure that
 * a refreshed tab is able to immediately re-acquire the owner lease).
 * Unfortunately, IndexedDB cannot be reliably used in window.unload since it is
 * an asynchronous API. So in addition to attempting to give up the lease,
 * the owner writes its ownerId to a "zombiedOwnerId" entry in LocalStorage
 * which acts as an indicator that another tab should go ahead and take the
 * owner lease immediately regardless of the current lease timestamp.
 */
export declare class IndexedDbPersistence implements Persistence {
    /**
     * The name of the main (and currently only) IndexedDB database. this name is
     * appended to the prefix provided to the IndexedDbPersistence constructor.
     */
    static MAIN_DATABASE: string;
    private simpleDb;
    private started;
    private dbName;
    private localStoragePrefix;
    private ownerId;
    /**
     * Set to an Error object if we encounter an unrecoverable error. All further
     * transactions will be failed with this error.
     */
    private persistenceError;
    /** The setInterval() handle tied to refreshing the owner lease. */
    private ownerLeaseRefreshHandle;
    /** Our window.unload handler, if registered. */
    private windowUnloadHandler;
    private serializer;
    constructor(prefix: string, serializer: JsonProtoSerializer);
    start(): Promise<void>;
    shutdown(): Promise<void>;
    getMutationQueue(user: User): MutationQueue;
    getQueryCache(): QueryCache;
    getRemoteDocumentCache(): RemoteDocumentCache;
    runTransaction<T>(action: string, operation: (transaction: SimpleDbTransaction) => PersistencePromise<T>): Promise<T>;
    static isAvailable(): boolean;
    /**
     * Generates a string used as a prefix when storing data in IndexedDB and
     * LocalStorage.
     */
    static buildStoragePrefix(databaseInfo: DatabaseInfo): string;
    /**
     * Acquires the owner lease if there's no valid owner. Else returns a rejected
     * promise.
     */
    private tryAcquireOwnerLease();
    /** Checks the owner lease and deletes it if we are the current owner. */
    private releaseOwnerLease();
    /**
     * Checks the owner lease and returns a rejected promise if we are not the
     * current owner. This should be included in every transaction to guard
     * against losing the owner lease.
     */
    private ensureOwnerLease(txn);
    /**
     * Returns true if the provided owner exists, has a recent timestamp, and
     * isn't zombied.
     *
     * NOTE: To determine if the owner is zombied, this method reads from
     * LocalStorage which could be mildly expensive.
     */
    private validOwner(dbOwner);
    /**
     * Schedules a recurring timer to update the owner lease timestamp to prevent
     * other tabs from taking the lease.
     */
    private scheduleOwnerLeaseRefreshes();
    private stopOwnerLeaseRefreshes();
    /**
     * Attaches a window.unload handler that will synchronously write our
     * ownerId to a "zombie owner id" location in localstorage. This can be used
     * by tabs trying to acquire the lease to determine that the lease should be
     * acquired immediately even if the timestamp is recent. This is particularly
     * important for the refresh case (so the tab correctly re-acquires the owner
     * lease). LocalStorage is used for this rather than IndexedDb because it is
     * a synchronous API and so can be used reliably from an unload handler.
     */
    private attachWindowUnloadHook();
    private detachWindowUnloadHook();
    /**
     * Returns any recorded "zombied owner" (i.e. a previous owner that became
     * zombied due to their tab closing) from LocalStorage, or null if no such
     * record exists.
     */
    private getZombiedOwnerId();
    /**
     * Records a zombied owner (an owner that had its tab closed) in LocalStorage
     * or, if passed null, deletes any recorded zombied owner.
     */
    private setZombiedOwnerId(zombieOwnerId);
    private zombiedOwnerLocalStorageKey();
    private generateOwnerId();
}
