"use strict";
// The MIT License (MIT)
//
// Copyright (c) 2017 Firebase
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
Object.defineProperty(exports, "__esModule", { value: true });
const _ = require("lodash");
const firebase = require("firebase-admin");
const index_1 = require("./index");
const sha1 = require("sha1");
/** @internal */
function apps() {
    if (typeof apps.singleton === 'undefined') {
        apps.init(index_1.config());
    }
    return apps.singleton;
}
exports.apps = apps;
/** @internal */
(function (apps) {
    /** @internal */
    apps.garbageCollectionInterval = 2 * 60 * 1000;
    /** @internal */
    function delay(delay) {
        return new Promise(resolve => {
            setTimeout(resolve, delay);
        });
    }
    apps.delay = delay;
    apps.init = (config) => apps.singleton = new Apps(config);
    /** @internal */
    class Apps {
        constructor(config) {
            this._config = config;
            this._refCounter = {};
        }
        _appAlive(appName) {
            try {
                let app = firebase.app(appName);
                return !_.get(app, 'isDeleted_');
            }
            catch (e) {
                return false;
            }
        }
        _appName(auth) {
            if (!auth || typeof auth !== 'object') {
                return '__noauth__';
            }
            else if (auth.admin) {
                return '__admin__';
            }
            else if (!auth.variable) {
                return '__noauth__';
            }
            else {
                // Use hash of auth variable as name of user-authenticated app
                return sha1(JSON.stringify(auth.variable));
            }
        }
        _destroyApp(appName) {
            if (!this._appAlive(appName)) {
                return;
            }
            firebase.app(appName).delete().catch(_.noop);
        }
        retain(payload) {
            let auth = _.get(payload, 'auth', null);
            let increment = n => {
                return (n || 0) + 1;
            };
            // Increment counter for admin because function might use event.data.adminRef
            _.update(this._refCounter, '__admin__', increment);
            // Increment counter according to auth type because function might use event.data.ref
            _.update(this._refCounter, this._appName(auth), increment);
        }
        release(payload) {
            let auth = _.get(payload, 'auth', null);
            let decrement = n => {
                return n - 1;
            };
            return delay(apps.garbageCollectionInterval).then(() => {
                _.update(this._refCounter, '__admin__', decrement);
                _.update(this._refCounter, this._appName(auth), decrement);
                _.forEach(this._refCounter, (count, key) => {
                    if (count <= 0) {
                        this._destroyApp(key);
                    }
                });
            });
        }
        get admin() {
            if (this._appAlive('__admin__')) {
                return firebase.app('__admin__');
            }
            return firebase.initializeApp(this.firebaseArgs, '__admin__');
        }
        get noauth() {
            if (this._appAlive('__noauth__')) {
                return firebase.app('__noauth__');
            }
            const param = _.extend({}, this.firebaseArgs, {
                databaseAuthVariableOverride: null,
            });
            return firebase.initializeApp(param, '__noauth__');
        }
        forMode(auth) {
            if (typeof auth !== 'object') {
                return this.noauth;
            }
            if (auth.admin) {
                return this.admin;
            }
            if (!auth.variable) {
                return this.noauth;
            }
            const appName = this._appName(auth);
            if (this._appAlive(appName)) {
                return firebase.app(appName);
            }
            const param = _.extend({}, this.firebaseArgs, {
                databaseAuthVariableOverride: auth.variable,
            });
            return firebase.initializeApp(param, appName);
        }
        get firebaseArgs() {
            return _.get(this._config, 'firebase', {});
        }
    }
    apps.Apps = Apps;
})(apps = exports.apps || (exports.apps = {}));
