import { Event, CloudFunction } from '../cloud-functions';
import * as firebase from 'firebase-admin';
/**
 * Pick the Realtime Database instance to use. If omitted, will pick the default database for your project.
 */
export declare function instance(instance: string): InstanceBuilder;
export declare class InstanceBuilder {
    private instance;
    ref(path: string): RefBuilder;
}
/**
 * Handle events at a Firebase Realtime Database Reference.
 *
 * This method behaves very similarly to the method of the same name in the
 * client and Admin Firebase SDKs. Any change to the Database that affects the
 * data at or below the provided `path` will fire an event in Cloud Functions.
 *
 * There are three important differences between listening to a Realtime
 * Database event in Cloud Functions and using the Realtime Database in the
 * client and Admin SDKs:
 * 1. Cloud Functions allows wildcards in the `path` name. Any `path` component
 *    in curly brackets (`{}`) is a wildcard that matches all strings. The value
 *    that matched a certain invocation of a Cloud Function is returned as part
 *    of the `event.params` object. For example, `ref("messages/{messageId}")`
 *    matches changes at `/messages/message1` or `/messages/message2`, resulting
 *    in  `event.params.messageId` being set to `"message1"` or `"message2"`,
 *    respectively.
 * 2. Cloud Functions do not fire an event for data that already existed before
 *    the Cloud Function was deployed.
 * 3. Cloud Function events have access to more information, including a
 *    snapshot of the previous event data and information about the user who
 *    triggered the Cloud Function.
 */
export declare function ref(path: string): RefBuilder;
/** Builder used to create Cloud Functions for Firebase Realtime Database References. */
export declare class RefBuilder {
    private apps;
    private resource;
    /** Respond to any write that affects a ref. */
    onWrite(handler: (event: Event<DeltaSnapshot>) => PromiseLike<any> | any): CloudFunction<DeltaSnapshot>;
    /** Respond to new data on a ref. */
    onCreate(handler: (event: Event<DeltaSnapshot>) => PromiseLike<any> | any): CloudFunction<DeltaSnapshot>;
    /** Respond to update on a ref. */
    onUpdate(handler: (event: Event<DeltaSnapshot>) => PromiseLike<any> | any): CloudFunction<DeltaSnapshot>;
    /** Respond to all data being deleted from a ref. */
    onDelete(handler: (event: Event<DeltaSnapshot>) => PromiseLike<any> | any): CloudFunction<DeltaSnapshot>;
    private onOperation(handler, eventType);
}
export declare class DeltaSnapshot {
    private app;
    private adminApp;
    private instance;
    private _adminRef;
    private _ref;
    private _path;
    private _data;
    private _delta;
    private _newData;
    private _childPath;
    private _isPrevious;
    constructor(app: firebase.app.App, adminApp: firebase.app.App, instance: string, data: any, delta: any, path?: string);
    readonly ref: firebase.database.Reference;
    readonly adminRef: firebase.database.Reference;
    readonly key: string;
    val(): any;
    exportVal(): any;
    getPriority(): string | number | null;
    exists(): boolean;
    child(childPath: string): DeltaSnapshot;
    readonly previous: DeltaSnapshot;
    readonly current: DeltaSnapshot;
    changed(): boolean;
    forEach(action: (a: DeltaSnapshot) => boolean): boolean;
    hasChild(childPath: string): boolean;
    hasChildren(): boolean;
    numChildren(): number;
    /**
     * Prints the value of the snapshot; use '.previous.toJSON()' and '.current.toJSON()' to explicitly see
     * the previous and current values of the snapshot.
     */
    toJSON(): Object;
    private _checkAndConvertToArray(node);
    private _dup(previous, childPath?);
    private _fullPath();
}
