"use strict";
/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* jshint maxlen: false */
const apirequest_1 = require("../../lib/apirequest");
/**
 * Google Mirror API
 *
 * Interacts with Glass users via the timeline.
 *
 * @example
 * const google = require('googleapis');
 * const mirror = google.mirror('v1');
 *
 * @namespace mirror
 * @type {Function}
 * @version v1
 * @variation v1
 * @param {object=} options Options for Mirror
 */
function Mirror(options) {
    const self = this;
    self._options = options || {};
    self.accounts = {
        /**
         * mirror.accounts.insert
         *
         * @desc Inserts a new account for a user
         *
         * @alias mirror.accounts.insert
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.accountName The name of the account to be passed to the Android Account Manager.
         * @param {string} params.accountType Account type to be passed to Android Account Manager.
         * @param {string} params.userToken The ID for the user.
         * @param {mirror(v1).Account} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        insert: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/accounts/{userToken}/{accountType}/{accountName}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                requiredParams: ['userToken', 'accountType', 'accountName'],
                pathParams: ['accountName', 'accountType', 'userToken'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.contacts = {
        /**
         * mirror.contacts.delete
         *
         * @desc Deletes a contact.
         *
         * @alias mirror.contacts.delete
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the contact.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        delete: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/contacts/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'DELETE'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.contacts.get
         *
         * @desc Gets a single contact by ID.
         *
         * @alias mirror.contacts.get
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the contact.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/contacts/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.contacts.insert
         *
         * @desc Inserts a new contact.
         *
         * @alias mirror.contacts.insert
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {mirror(v1).Contact} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        insert: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/contacts').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.contacts.list
         *
         * @desc Retrieves a list of contacts for the authenticated user.
         *
         * @alias mirror.contacts.list
         * @memberOf! mirror(v1)
         *
         * @param {object=} params Parameters for request
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/contacts').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.contacts.patch
         *
         * @desc Updates a contact in place. This method supports patch semantics.
         *
         * @alias mirror.contacts.patch
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the contact.
         * @param {mirror(v1).Contact} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        patch: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/contacts/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PATCH'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.contacts.update
         *
         * @desc Updates a contact in place.
         *
         * @alias mirror.contacts.update
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the contact.
         * @param {mirror(v1).Contact} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        update: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/contacts/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PUT'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.locations = {
        /**
         * mirror.locations.get
         *
         * @desc Gets a single location by ID.
         *
         * @alias mirror.locations.get
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the location or latest for the last known location.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/locations/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.locations.list
         *
         * @desc Retrieves a list of locations for the user.
         *
         * @alias mirror.locations.list
         * @memberOf! mirror(v1)
         *
         * @param {object=} params Parameters for request
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/locations').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.settings = {
        /**
         * mirror.settings.get
         *
         * @desc Gets a single setting by ID.
         *
         * @alias mirror.settings.get
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the setting. The following IDs are valid:  - locale - The key to the user’s language/locale (BCP 47 identifier) that Glassware should use to render localized content.  - timezone - The key to the user’s current time zone region as defined in the tz database. Example: America/Los_Angeles.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/settings/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.subscriptions = {
        /**
         * mirror.subscriptions.delete
         *
         * @desc Deletes a subscription.
         *
         * @alias mirror.subscriptions.delete
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the subscription.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        delete: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/subscriptions/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'DELETE'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.subscriptions.insert
         *
         * @desc Creates a new subscription.
         *
         * @alias mirror.subscriptions.insert
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {mirror(v1).Subscription} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        insert: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/subscriptions').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.subscriptions.list
         *
         * @desc Retrieves a list of subscriptions for the authenticated user and service.
         *
         * @alias mirror.subscriptions.list
         * @memberOf! mirror(v1)
         *
         * @param {object=} params Parameters for request
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/subscriptions').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.subscriptions.update
         *
         * @desc Updates an existing subscription in place.
         *
         * @alias mirror.subscriptions.update
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the subscription.
         * @param {mirror(v1).Subscription} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        update: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/subscriptions/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PUT'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.timeline = {
        /**
         * mirror.timeline.delete
         *
         * @desc Deletes a timeline item.
         *
         * @alias mirror.timeline.delete
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the timeline item.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        delete: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/timeline/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'DELETE'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.timeline.get
         *
         * @desc Gets a single timeline item by ID.
         *
         * @alias mirror.timeline.get
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the timeline item.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/timeline/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.timeline.insert
         *
         * @desc Inserts a new item into the timeline.
         *
         * @alias mirror.timeline.insert
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param  {object} params.resource Media resource metadata
         * @param {object} params.media Media object
         * @param {string} params.media.mimeType Media mime-type
         * @param {string|object} params.media.body Media body contents
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        insert: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/timeline').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                mediaUrl: (rootUrl + '/upload/mirror/v1/timeline').replace(/([^:]\/)\/+/g, '$1'),
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.timeline.list
         *
         * @desc Retrieves a list of timeline items for the authenticated user.
         *
         * @alias mirror.timeline.list
         * @memberOf! mirror(v1)
         *
         * @param {object=} params Parameters for request
         * @param {string=} params.bundleId If provided, only items with the given bundleId will be returned.
         * @param {boolean=} params.includeDeleted If true, tombstone records for deleted items will be returned.
         * @param {integer=} params.maxResults The maximum number of items to include in the response, used for paging.
         * @param {string=} params.orderBy Controls the order in which timeline items are returned.
         * @param {string=} params.pageToken Token for the page of results to return.
         * @param {boolean=} params.pinnedOnly If true, only pinned items will be returned.
         * @param {string=} params.sourceItemId If provided, only items with the given sourceItemId will be returned.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/timeline').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.timeline.patch
         *
         * @desc Updates a timeline item in place. This method supports patch semantics.
         *
         * @alias mirror.timeline.patch
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the timeline item.
         * @param {mirror(v1).TimelineItem} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        patch: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/timeline/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PATCH'
                }, options),
                params: params,
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * mirror.timeline.update
         *
         * @desc Updates a timeline item in place.
         *
         * @alias mirror.timeline.update
         * @memberOf! mirror(v1)
         *
         * @param {object} params Parameters for request
         * @param {string} params.id The ID of the timeline item.
         * @param  {object} params.resource Media resource metadata
         * @param {object} params.media Media object
         * @param {string} params.media.mimeType Media mime-type
         * @param {string|object} params.media.body Media body contents
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        update: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/mirror/v1/timeline/{id}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PUT'
                }, options),
                params: params,
                mediaUrl: (rootUrl + '/upload/mirror/v1/timeline/{id}').replace(/([^:]\/)\/+/g, '$1'),
                requiredParams: ['id'],
                pathParams: ['id'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        attachments: {
            /**
             * mirror.timeline.attachments.delete
             *
             * @desc Deletes an attachment from a timeline item.
             *
             * @alias mirror.timeline.attachments.delete
             * @memberOf! mirror(v1)
             *
             * @param {object} params Parameters for request
             * @param {string} params.attachmentId The ID of the attachment.
             * @param {string} params.itemId The ID of the timeline item the attachment belongs to.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            delete: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/mirror/v1/timeline/{itemId}/attachments/{attachmentId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'DELETE'
                    }, options),
                    params: params,
                    requiredParams: ['itemId', 'attachmentId'],
                    pathParams: ['attachmentId', 'itemId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * mirror.timeline.attachments.get
             *
             * @desc Retrieves an attachment on a timeline item by item ID and attachment ID.
             *
             * @alias mirror.timeline.attachments.get
             * @memberOf! mirror(v1)
             *
             * @param {object} params Parameters for request
             * @param {string} params.attachmentId The ID of the attachment.
             * @param {string} params.itemId The ID of the timeline item the attachment belongs to.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            get: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/mirror/v1/timeline/{itemId}/attachments/{attachmentId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['itemId', 'attachmentId'],
                    pathParams: ['attachmentId', 'itemId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * mirror.timeline.attachments.insert
             *
             * @desc Adds a new attachment to a timeline item.
             *
             * @alias mirror.timeline.attachments.insert
             * @memberOf! mirror(v1)
             *
             * @param {object} params Parameters for request
             * @param {string} params.itemId The ID of the timeline item the attachment belongs to.
             * @param {object} params.media Media object
             * @param {string} params.media.mimeType Media mime-type
             * @param {string|object} params.media.body Media body contents
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            insert: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/mirror/v1/timeline/{itemId}/attachments').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'POST'
                    }, options),
                    params: params,
                    mediaUrl: (rootUrl + '/upload/mirror/v1/timeline/{itemId}/attachments').replace(/([^:]\/)\/+/g, '$1'),
                    requiredParams: ['itemId'],
                    pathParams: ['itemId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * mirror.timeline.attachments.list
             *
             * @desc Returns a list of attachments for a timeline item.
             *
             * @alias mirror.timeline.attachments.list
             * @memberOf! mirror(v1)
             *
             * @param {object} params Parameters for request
             * @param {string} params.itemId The ID of the timeline item whose attachments should be listed.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/mirror/v1/timeline/{itemId}/attachments').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['itemId'],
                    pathParams: ['itemId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        }
    };
}
module.exports = Mirror;
//# sourceMappingURL=v1.js.map