"use strict";
/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* jshint maxlen: false */
const apirequest_1 = require("../../lib/apirequest");
/**
 * PageSpeed Insights API
 *
 * Analyzes the performance of a web page and provides tailored suggestions to make that page faster.
 *
 * @example
 * const google = require('googleapis');
 * const pagespeedonline = google.pagespeedonline('v1');
 *
 * @namespace pagespeedonline
 * @type {Function}
 * @version v1
 * @variation v1
 * @param {object=} options Options for Pagespeedonline
 */
function Pagespeedonline(options) {
    const self = this;
    self._options = options || {};
    self.pagespeedapi = {
        /**
         * pagespeedonline.pagespeedapi.runpagespeed
         *
         * @desc Runs PageSpeed analysis on the page at the specified URL, and returns a PageSpeed score, a list of suggestions to make that page faster, and other information.
         *
         * @alias pagespeedonline.pagespeedapi.runpagespeed
         * @memberOf! pagespeedonline(v1)
         *
         * @param {object} params Parameters for request
         * @param {boolean=} params.filter_third_party_resources Indicates if third party resources should be filtered out before PageSpeed analysis.
         * @param {string=} params.locale The locale used to localize formatted results
         * @param {string=} params.rule A PageSpeed rule to run; if none are given, all rules are run
         * @param {boolean=} params.screenshot Indicates if binary data containing a screenshot should be included
         * @param {string=} params.strategy The analysis strategy to use
         * @param {string} params.url The URL to fetch and analyze
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        runpagespeed: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/pagespeedonline/v1/runPagespeed').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['url'],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
}
module.exports = Pagespeedonline;
//# sourceMappingURL=v1.js.map