import { PersistencePromise } from './persistence_promise';
/**
 * Provides a wrapper around IndexedDb with a simplified interface that uses
 * Promise-like return values to chain operations. Real promises cannot be used
 * since .then() continuations are executed asynchronously (e.g. via
 * .setImmediate), which would cause IndexedDB to end the transaction.
 * See PersistencePromise for more details.
 */
export declare class SimpleDb {
    private db;
    /** Opens the specified database, creating or upgrading it if necessary. */
    static openOrCreate(name: string, version: number, runUpgrade: (db: IDBDatabase, oldVersion: number) => void): Promise<SimpleDb>;
    /** Deletes the specified database. */
    static delete(name: string): Promise<void>;
    /** Returns true if IndexedDB is available in the current environment. */
    static isAvailable(): boolean;
    constructor(db: IDBDatabase);
    runTransaction<T>(mode: 'readonly' | 'readwrite', objectStores: string[], transactionFn: (transaction: SimpleDbTransaction) => PersistencePromise<T>): Promise<T>;
    close(): void;
}
/**
 * A controller for iterating over a key range or index. It allows an iterate
 * callback to delete the currently-referenced object, or jump to a new key
 * within the key range or index.
 */
export declare class IterationController {
    private dbCursor;
    private shouldStop;
    private nextKey;
    constructor(dbCursor: IDBCursorWithValue);
    readonly isDone: boolean;
    readonly skipToKey: IDBValidKey | IDBKeyRange | null;
    cursor: IDBCursorWithValue;
    /**
     * This function can be called to stop iteration at any point.
     */
    done(): void;
    /**
     * This function can be called to skip to that next key, which could be
     * an index or a primary key.
     */
    skip(key: IDBValidKey | IDBKeyRange): void;
    /**
     * Delete the current cursor value from the object store.
     *
     * NOTE: You CANNOT do this with a keysOnly query.
     */
    delete(): PersistencePromise<void>;
}
/**
 * Callback used with iterate() method.
 */
export declare type IterateCallback<KeyType, ValueType> = (key: KeyType, value: ValueType, control: IterationController) => void | PersistencePromise<void>;
/** Options available to the iterate() method. */
export interface IterateOptions {
    /** Index to iterate over (else primary keys will be iterated) */
    index?: string;
    /** IndxedDB Range to iterate over (else entire store will be iterated) */
    range?: IDBKeyRange;
    /** If true, values aren't read while iterating. */
    keysOnly?: boolean;
    /** If true, iterate over the store in reverse. */
    reverse?: boolean;
}
/**
 * Wraps an IDBTransaction and exposes a store() method to get a handle to a
 * specific object store.
 */
export declare class SimpleDbTransaction {
    private transaction;
    private aborted;
    /**
     * A promise that resolves with the result of the IndexedDb transaction.
     *
     * Note: A transaction explicitly aborted via abort() is considered successful
     * and this promise will resolve as successful.
     */
    readonly completionPromise: Promise<void>;
    constructor(db: IDBDatabase, mode: string, objectStoresNames: string[]);
    abort(): void;
    /**
     * Returns a SimpleDbStore<KeyType, ValueType> for the specified store. All
     * operations performed on the SimpleDbStore happen within the context of this
     * transaction and it cannot be used anymore once the transaction is
     * completed.
     *
     * Note that we can't actually enforce that the KeyType and ValueType are
     * correct, but they allow type safety through the rest of the consuming code.
     */
    store<KeyType extends IDBValidKey, ValueType>(storeName: string): SimpleDbStore<KeyType, ValueType>;
}
/**
 * A wrapper around an IDBObjectStore providing an API that:
 *
 * 1) Has generic KeyType / ValueType parameters to provide strongly-typed
 * methods for acting against the object store.
 * 2) Deals with IndexedDB's onsuccess / onerror event callbacks, making every
 * method return a PersistencePromise instead.
 * 3) Provides a higher-level API to avoid needing to do excessive wrapping of
 * intermediate IndexedDB types (IDBCursorWithValue, etc.)
 */
export declare class SimpleDbStore<KeyType extends IDBValidKey, ValueType> {
    private store;
    constructor(store: IDBObjectStore);
    /**
     * Writes a value into the Object Store.
     *
     * @param key Optional explicit key to use when writing the object, else the
     * key will be auto-assigned (e.g. via the defined keyPath for the store).
     * @param value The object to write.
     */
    put(value: ValueType): PersistencePromise<void>;
    put(key: KeyType, value: ValueType): PersistencePromise<void>;
    /**
     * Gets the object with the specified key from the specified store, or null
     * if no object exists with the specified key.
     *
     * @key The key of the object to get.
     * @return The object with the specified key or null if no object exists.
     */
    get(key: KeyType): PersistencePromise<ValueType | null>;
    delete(key: KeyType | IDBKeyRange): PersistencePromise<void>;
    loadAll(): PersistencePromise<ValueType[]>;
    loadAll(range: IDBKeyRange): PersistencePromise<ValueType[]>;
    loadAll(index: string, range: IDBKeyRange): PersistencePromise<ValueType[]>;
    deleteAll(): PersistencePromise<void>;
    deleteAll(range: IDBKeyRange): PersistencePromise<void>;
    deleteAll(index: string, range: IDBKeyRange): PersistencePromise<void>;
    /**
     * Iterates over keys and values in an object store.
     *
     * @param options Options specifying how to iterate the objects in the store.
     * @param callback will be called for each iterated object. Iteration can be
     * canceled at any point by calling the doneFn passed to the callback.
     * The callback can return a PersistencePromise if it performs async
     * operations but note that iteration will continue without waiting for them
     * to complete.
     * @returns A PersistencePromise that resolves once all PersistencePromises
     * returned by callbacks resolve.
     */
    iterate(callback: IterateCallback<KeyType, ValueType>): PersistencePromise<void>;
    iterate(options: IterateOptions, callback: IterateCallback<KeyType, ValueType>): PersistencePromise<void>;
    private iterateCursor(cursorRequest, fn);
    private options(indexOrRange?, range?);
    private cursor(options);
}
