export declare const DOCUMENT_KEY_NAME = "__name__";
/**
 * Path represents an ordered sequence of string segments.
 */
export declare abstract class Path {
    private segments;
    private offset;
    private len;
    constructor(segments: string[], offset?: number, length?: number);
    /**
     * An initialization method that can be called from outside the constructor.
     * We need this so that we can have a non-static construct method that returns
     * the polymorphic `this` type.
     */
    private init(segments, offset?, length?);
    /**
     * Constructs a new instance of Path using the same concrete type as `this`.
     * We need this instead of using the normal constructor, because polymorphic
     * `this` doesn't work on static methods.
     */
    private construct(segments, offset?, length?);
    readonly length: number;
    equals(other: Path): boolean;
    child(nameOrPath: string | this): this;
    /** The index of one past the last segment of the path. */
    private limit();
    popFirst(size?: number): this;
    popLast(): this;
    firstSegment(): string;
    lastSegment(): string;
    get(index: number): string;
    isEmpty(): boolean;
    isPrefixOf(other: this): boolean;
    forEach(fn: (segment: string) => void): void;
    toArray(): string[];
    static comparator(p1: Path, p2: Path): number;
}
/**
 * A slash-separated path for navigating resources (documents and collections)
 * within Firestore.
 */
export declare class ResourcePath extends Path {
    canonicalString(): string;
    toString(): string;
    /**
     * Creates a resource path from the given slash-delimited string.
     */
    static fromString(path: string): ResourcePath;
    static EMPTY_PATH: ResourcePath;
}
/** A dot-separated path for navigating sub-objects within a document. */
export declare class FieldPath extends Path {
    /**
     * Returns true if the string could be used as a segment in a field path
     * without escaping.
     */
    private static isValidIdentifier(segment);
    canonicalString(): string;
    toString(): string;
    /**
     * Returns true if this field references the key of a document.
     */
    isKeyField(): boolean;
    /**
     * The field designating the key of a document.
     */
    static keyField(): FieldPath;
    /**
     * Parses a field string from the given server-formatted string.
     *
     * - Splitting the empty string is not allowed (for now at least).
     * - Empty segments within the string (e.g. if there are two consecutive
     *   separators) are not allowed.
     *
     * TODO(b/37244157): we should make this more strict. Right now, it allows
     * non-identifier path components, even if they aren't escaped.
     */
    static fromServerFormat(path: string): FieldPath;
    static EMPTY_PATH: FieldPath;
}
