/**
 * A helper for running delayed tasks following an exponential backoff curve
 * between attempts.
 *
 * Each delay is made up of a "base" delay which follows the exponential
 * backoff curve, and a +/- 50% "jitter" that is calculated and added to the
 * base delay. This prevents clients from accidentally synchronizing their
 * delays causing spikes of load to the backend.
 */
export declare class ExponentialBackoff {
    /**
       * The initial delay (used as the base delay on the first retry attempt).
       * Note that jitter will still be applied, so the actual delay could be as
       * little as 0.5*initialDelayMs.
       */
    private readonly initialDelayMs;
    /**
       * The multiplier to use to determine the extended base delay after each
       * attempt.
       */
    private readonly backoffFactor;
    /**
       * The maximum base delay after which no further backoff is performed.
       * Note that jitter will still be applied, so the actual delay could be as
       * much as 1.5*maxDelayMs.
       */
    private readonly maxDelayMs;
    private currentBaseMs;
    constructor(
        /**
           * The initial delay (used as the base delay on the first retry attempt).
           * Note that jitter will still be applied, so the actual delay could be as
           * little as 0.5*initialDelayMs.
           */
        initialDelayMs: number, 
        /**
           * The multiplier to use to determine the extended base delay after each
           * attempt.
           */
        backoffFactor: number, 
        /**
           * The maximum base delay after which no further backoff is performed.
           * Note that jitter will still be applied, so the actual delay could be as
           * much as 1.5*maxDelayMs.
           */
        maxDelayMs: number);
    /**
     * Resets the backoff delay.
     *
     * The very next backoffAndWait() will have no delay. If it is called again
     * (i.e. due to an error), initialDelayMs (plus jitter) will be used, and
     * subsequent ones will increase according to the backoffFactor.
     */
    reset(): void;
    /**
     * Resets the backoff delay to the maximum delay (e.g. for use after a
     * RESOURCE_EXHAUSTED error).
     */
    resetToMax(): void;
    /**
     * Returns a promise that resolves after currentDelayMs, and increases the
     * delay for any subsequent attempts.
     */
    backoffAndWait(): Promise<void>;
    /** Returns a random value in the range [-currentBaseMs/2, currentBaseMs/2] */
    private jitterDelayMs();
}
