import { CloudFunction, Event } from '../cloud-functions';
/**
 * Handle events related to Crashlytics issues. An issue in Crashlytics is an
 * aggregation of crashes which have a shared root cause.
 */
export declare function issue(): IssueBuilder;
/** Builder used to create Cloud Functions for Crashlytics issue events. */
export declare class IssueBuilder {
    private resource;
    /** Handle Crashlytics New Issue events. */
    onNewDetected(handler: (event: Event<Issue>) => PromiseLike<any> | any): CloudFunction<Issue>;
    /** Handle Crashlytics Regressed Issue events. */
    onRegressed(handler: (event: Event<Issue>) => PromiseLike<any> | any): CloudFunction<Issue>;
    /** Handle Crashlytics Velocity Alert events. */
    onVelocityAlert(handler: (event: Event<Issue>) => PromiseLike<any> | any): CloudFunction<Issue>;
}
/**
 * Interface representing a Crashlytics issue event that was logged for a specific issue.
 */
export declare class Issue {
    /** Fabric Issue ID. */
    issueId: string;
    /** Issue title. */
    issueTitle: string;
    /** App information. */
    appInfo: AppInfo;
    /** When the issue was created (ISO8601 time stamp). */
    createTime: string;
    /** When the issue was resolved, if the issue has been resolved (ISO8601 time stamp). */
    resolvedTime?: string;
    /** Contains details about the velocity alert, if this event was triggered by a velocity alert. */
    velocityAlert?: VelocityAlert;
}
export declare class VelocityAlert {
    /** The percentage of sessions which have been impacted by this issue. Example: .04 */
    crashPercentage: number;
    /** The number of crashes that this issue has caused. */
    crashes: number;
}
/**
 * Interface representing the application where this issue occurred.
 */
export interface AppInfo {
    /** The app's name. Example: "My Awesome App". */
    appName: string;
    /** The app's platform. Examples: "android", "ios". */
    appPlatform: string;
    /** Unique application identifier within an app store, either the Android package name or the iOS bundle id. */
    appId: string;
    /**
     *  The latest app version which is affected by the issue.
     *  Examples: "1.0", "4.3.1.1.213361", "2.3 (1824253)", "v1.8b22p6".
     */
    latestAppVersion: string;
}
