"use strict";
// The MIT License (MIT)
//
// Copyright (c) 2017 Firebase
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const _ = require("lodash");
const firebase = require("firebase-admin");
const apps_1 = require("../apps");
const cloud_functions_1 = require("../cloud-functions");
const encoder_1 = require("../encoder");
/** @internal */
exports.provider = 'cloud.firestore';
/** @internal */
exports.defaultDatabase = '(default)';
let firestoreInstance;
function database(database = exports.defaultDatabase) {
    return new DatabaseBuilder(path_1.posix.join('projects', process.env.GCLOUD_PROJECT, 'databases', database));
}
exports.database = database;
function namespace(namespace) {
    return database().namespace(namespace);
}
exports.namespace = namespace;
function document(path) {
    return database().document(path);
}
exports.document = document;
class DatabaseBuilder {
    /** @internal */
    constructor(resource) {
        this.resource = resource;
    }
    namespace(namespace) {
        return new NamespaceBuilder(`${path_1.posix.join(this.resource, 'documents')}@${namespace}`);
    }
    document(path) {
        return (new NamespaceBuilder(path_1.posix.join(this.resource, 'documents'))).document(path);
    }
}
exports.DatabaseBuilder = DatabaseBuilder;
class NamespaceBuilder {
    /** @internal */
    constructor(resource) {
        this.resource = resource;
    }
    document(path) {
        return new DocumentBuilder(path_1.posix.join(this.resource, path));
    }
}
exports.NamespaceBuilder = NamespaceBuilder;
;
function isDeltaDocumentSnapshot(data) {
    return 'exists' in data;
}
;
function getValueProto(event) {
    let data = event.data;
    if (_.isEmpty(_.get(data, 'value'))) {
        // Firestore#snapshot_ takes resource string instead of proto for a non-existent snapshot
        return event.resource;
    }
    let proto = {
        fields: convertToFieldsProto(_.get(data, 'value.fields', {})),
        createTime: encoder_1.dateToTimestampProto(_.get(data, 'value.createTime')),
        updateTime: encoder_1.dateToTimestampProto(_.get(data, 'value.updateTime')),
        name: _.get(data, 'value.name', event.resource),
    };
    return proto;
}
;
function getOldValueProto(event) {
    let data = event.data;
    let proto = {
        fields: convertToFieldsProto(_.get(data, 'oldValue.fields', {})),
        createTime: encoder_1.dateToTimestampProto(_.get(data, 'oldValue.createTime')),
        updateTime: encoder_1.dateToTimestampProto(_.get(data, 'oldValue.updateTime')),
        name: _.get(data, 'oldValue.name', event.resource),
    };
    return proto;
}
;
function convertToFieldsProto(fields) {
    if (!fields) {
        return {};
    }
    function convertHelper(data) {
        let result;
        _.forEach(data, (value, valueType) => {
            let dataPart;
            if (valueType === 'arrayValue') {
                let array = _.get(value, 'values', []);
                dataPart = {
                    arrayValue: {
                        values: _.map(array, (elem) => {
                            return convertHelper(elem);
                        }),
                    },
                };
            }
            else if (valueType === 'mapValue') {
                let map = _.get(value, 'fields', {});
                dataPart = {
                    mapValue: {
                        fields: _.mapValues(map, (val) => {
                            return convertHelper(val);
                        }),
                    },
                };
            }
            else if (valueType === 'timestampValue') {
                dataPart = { timestampValue: encoder_1.dateToTimestampProto(value) };
            }
            else {
                dataPart = data;
            }
            result = _.merge({}, dataPart, { valueType: valueType });
        });
        return result;
    }
    return _.mapValues(fields, (data) => {
        return convertHelper(data);
    });
}
;
/** @internal */
function dataConstructor(raw) {
    if (isDeltaDocumentSnapshot(raw.data)) {
        return raw.data;
    }
    if (!firestoreInstance) {
        firestoreInstance = firebase.firestore(apps_1.apps().admin);
    }
    let valueProto = getValueProto(raw);
    let readTime = encoder_1.dateToTimestampProto(_.get(raw.data, 'value.readTime'));
    let snapshot = firestoreInstance.snapshot_(valueProto, readTime);
    Object.defineProperty(snapshot, 'previous', {
        get: () => {
            if (_.isEmpty(_.get(raw, 'data.oldValue', {}))) {
                return null;
            }
            let oldValueProto = getOldValueProto(raw);
            let oldReadTime = encoder_1.dateToTimestampProto(_.get(raw.data, 'value.oldValue.readTime'));
            return firestoreInstance.snapshot_(oldValueProto, oldReadTime);
        },
    });
    return snapshot;
}
exports.dataConstructor = dataConstructor;
;
class DocumentBuilder {
    /** @internal */
    constructor(resource) {
        this.resource = resource;
        // TODO what validation do we want to do here?
    }
    /** Respond to all document writes (creates, updates, or deletes). */
    onWrite(handler) {
        return this.onOperation(handler, 'document.write');
    }
    /** Respond only to document creations. */
    onCreate(handler) {
        return this.onOperation(handler, 'document.create');
    }
    /** Respond only to document updates. */
    onUpdate(handler) {
        return this.onOperation(handler, 'document.update');
    }
    /** Respond only to document deletions. */
    onDelete(handler) {
        return this.onOperation(handler, 'document.delete');
    }
    onOperation(handler, eventType) {
        return cloud_functions_1.makeCloudFunction({
            provider: exports.provider, handler,
            resource: this.resource,
            eventType: eventType,
            dataConstructor,
        });
    }
}
exports.DocumentBuilder = DocumentBuilder;
