"use strict";
/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* jshint maxlen: false */
const apirequest_1 = require("../../lib/apirequest");
/**
 * AdSense Management API
 *
 * Accesses AdSense publishers&#39; inventory and generates performance reports.
 *
 * @example
 * const google = require('googleapis');
 * const adsense = google.adsense('v1.4');
 *
 * @namespace adsense
 * @type {Function}
 * @version v1.4
 * @variation v1.4
 * @param {object=} options Options for Adsense
 */
function Adsense(options) {
    const self = this;
    self._options = options || {};
    self.accounts = {
        /**
         * adsense.accounts.get
         *
         * @desc Get information about the selected AdSense account.
         *
         * @alias adsense.accounts.get
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.accountId Account to get information about.
         * @param {boolean=} params.tree Whether the tree of sub accounts should be returned.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/accounts/{accountId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['accountId'],
                pathParams: ['accountId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * adsense.accounts.list
         *
         * @desc List all accounts available to this AdSense account.
         *
         * @alias adsense.accounts.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object=} params Parameters for request
         * @param {integer=} params.maxResults The maximum number of accounts to include in the response, used for paging.
         * @param {string=} params.pageToken A continuation token, used to page through accounts. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/accounts').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        adclients: {
            /**
             * adsense.accounts.adclients.list
             *
             * @desc List all ad clients in the specified account.
             *
             * @alias adsense.accounts.adclients.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account for which to list ad clients.
             * @param {integer=} params.maxResults The maximum number of ad clients to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through ad clients. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId'],
                    pathParams: ['accountId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        },
        adunits: {
            /**
             * adsense.accounts.adunits.get
             *
             * @desc Gets the specified ad unit in the specified ad client for the specified account.
             *
             * @alias adsense.accounts.adunits.get
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account to which the ad client belongs.
             * @param {string} params.adClientId Ad client for which to get the ad unit.
             * @param {string} params.adUnitId Ad unit to retrieve.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            get: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/adunits/{adUnitId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'adClientId', 'adUnitId'],
                    pathParams: ['accountId', 'adClientId', 'adUnitId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * adsense.accounts.adunits.getAdCode
             *
             * @desc Get ad code for the specified ad unit.
             *
             * @alias adsense.accounts.adunits.getAdCode
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account which contains the ad client.
             * @param {string} params.adClientId Ad client with contains the ad unit.
             * @param {string} params.adUnitId Ad unit to get the code for.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            getAdCode: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/adunits/{adUnitId}/adcode').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'adClientId', 'adUnitId'],
                    pathParams: ['accountId', 'adClientId', 'adUnitId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * adsense.accounts.adunits.list
             *
             * @desc List all ad units in the specified ad client for the specified account.
             *
             * @alias adsense.accounts.adunits.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account to which the ad client belongs.
             * @param {string} params.adClientId Ad client for which to list ad units.
             * @param {boolean=} params.includeInactive Whether to include inactive ad units. Default: true.
             * @param {integer=} params.maxResults The maximum number of ad units to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through ad units. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/adunits').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'adClientId'],
                    pathParams: ['accountId', 'adClientId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            customchannels: {
                /**
                 * adsense.accounts.adunits.customchannels.list
                 *
                 * @desc List all custom channels which the specified ad unit belongs to.
                 *
                 * @alias adsense.accounts.adunits.customchannels.list
                 * @memberOf! adsense(v1.4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.accountId Account to which the ad client belongs.
                 * @param {string} params.adClientId Ad client which contains the ad unit.
                 * @param {string} params.adUnitId Ad unit for which to list custom channels.
                 * @param {integer=} params.maxResults The maximum number of custom channels to include in the response, used for paging.
                 * @param {string=} params.pageToken A continuation token, used to page through custom channels. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                list: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/adunits/{adUnitId}/customchannels').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'GET'
                        }, options),
                        params: params,
                        requiredParams: ['accountId', 'adClientId', 'adUnitId'],
                        pathParams: ['accountId', 'adClientId', 'adUnitId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                }
            }
        },
        alerts: {
            /**
             * adsense.accounts.alerts.delete
             *
             * @desc Dismiss (delete) the specified alert from the specified publisher AdSense account.
             *
             * @alias adsense.accounts.alerts.delete
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account which contains the ad unit.
             * @param {string} params.alertId Alert to delete.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            delete: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/alerts/{alertId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'DELETE'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'alertId'],
                    pathParams: ['accountId', 'alertId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * adsense.accounts.alerts.list
             *
             * @desc List the alerts for the specified AdSense account.
             *
             * @alias adsense.accounts.alerts.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account for which to retrieve the alerts.
             * @param {string=} params.locale The locale to use for translating alert messages. The account locale will be used if this is not supplied. The AdSense default (English) will be used if the supplied locale is invalid or unsupported.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/alerts').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId'],
                    pathParams: ['accountId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        },
        customchannels: {
            /**
             * adsense.accounts.customchannels.get
             *
             * @desc Get the specified custom channel from the specified ad client for the specified account.
             *
             * @alias adsense.accounts.customchannels.get
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account to which the ad client belongs.
             * @param {string} params.adClientId Ad client which contains the custom channel.
             * @param {string} params.customChannelId Custom channel to retrieve.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            get: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/customchannels/{customChannelId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'adClientId', 'customChannelId'],
                    pathParams: ['accountId', 'adClientId', 'customChannelId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * adsense.accounts.customchannels.list
             *
             * @desc List all custom channels in the specified ad client for the specified account.
             *
             * @alias adsense.accounts.customchannels.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account to which the ad client belongs.
             * @param {string} params.adClientId Ad client for which to list custom channels.
             * @param {integer=} params.maxResults The maximum number of custom channels to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through custom channels. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/customchannels').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'adClientId'],
                    pathParams: ['accountId', 'adClientId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            adunits: {
                /**
                 * adsense.accounts.customchannels.adunits.list
                 *
                 * @desc List all ad units in the specified custom channel.
                 *
                 * @alias adsense.accounts.customchannels.adunits.list
                 * @memberOf! adsense(v1.4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.accountId Account to which the ad client belongs.
                 * @param {string} params.adClientId Ad client which contains the custom channel.
                 * @param {string} params.customChannelId Custom channel for which to list ad units.
                 * @param {boolean=} params.includeInactive Whether to include inactive ad units. Default: true.
                 * @param {integer=} params.maxResults The maximum number of ad units to include in the response, used for paging.
                 * @param {string=} params.pageToken A continuation token, used to page through ad units. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                list: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/customchannels/{customChannelId}/adunits').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'GET'
                        }, options),
                        params: params,
                        requiredParams: ['accountId', 'adClientId', 'customChannelId'],
                        pathParams: ['accountId', 'adClientId', 'customChannelId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                }
            }
        },
        payments: {
            /**
             * adsense.accounts.payments.list
             *
             * @desc List the payments for the specified AdSense account.
             *
             * @alias adsense.accounts.payments.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account for which to retrieve the payments.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/payments').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId'],
                    pathParams: ['accountId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        },
        reports: {
            /**
             * adsense.accounts.reports.generate
             *
             * @desc Generate an AdSense report based on the report request sent in the query parameters. Returns the result as JSON; to retrieve output in CSV format specify "alt=csv" as a query parameter.
             *
             * @alias adsense.accounts.reports.generate
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account upon which to report.
             * @param {string=} params.currency Optional currency to use when reporting on monetary metrics. Defaults to the account's currency if not set.
             * @param {string=} params.dimension Dimensions to base the report on.
             * @param {string} params.endDate End of the date range to report on in "YYYY-MM-DD" format, inclusive.
             * @param {string=} params.filter Filters to be run on the report.
             * @param {string=} params.locale Optional locale to use for translating report output to a local language. Defaults to "en_US" if not specified.
             * @param {integer=} params.maxResults The maximum number of rows of report data to return.
             * @param {string=} params.metric Numeric columns to include in the report.
             * @param {string=} params.sort The name of a dimension or metric to sort the resulting report on, optionally prefixed with "+" to sort ascending or "-" to sort descending. If no prefix is specified, the column is sorted ascending.
             * @param {string} params.startDate Start of the date range to report on in "YYYY-MM-DD" format, inclusive.
             * @param {integer=} params.startIndex Index of the first row of report data to return.
             * @param {boolean=} params.useTimezoneReporting Whether the report should be generated in the AdSense account's local timezone. If false default PST/PDT timezone will be used.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            generate: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/reports').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'startDate', 'endDate'],
                    pathParams: ['accountId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            saved: {
                /**
                 * adsense.accounts.reports.saved.generate
                 *
                 * @desc Generate an AdSense report based on the saved report ID sent in the query parameters.
                 *
                 * @alias adsense.accounts.reports.saved.generate
                 * @memberOf! adsense(v1.4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.accountId Account to which the saved reports belong.
                 * @param {string=} params.locale Optional locale to use for translating report output to a local language. Defaults to "en_US" if not specified.
                 * @param {integer=} params.maxResults The maximum number of rows of report data to return.
                 * @param {string} params.savedReportId The saved report to retrieve.
                 * @param {integer=} params.startIndex Index of the first row of report data to return.
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                generate: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/reports/{savedReportId}').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'GET'
                        }, options),
                        params: params,
                        requiredParams: ['accountId', 'savedReportId'],
                        pathParams: ['accountId', 'savedReportId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                },
                /**
                 * adsense.accounts.reports.saved.list
                 *
                 * @desc List all saved reports in the specified AdSense account.
                 *
                 * @alias adsense.accounts.reports.saved.list
                 * @memberOf! adsense(v1.4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.accountId Account to which the saved reports belong.
                 * @param {integer=} params.maxResults The maximum number of saved reports to include in the response, used for paging.
                 * @param {string=} params.pageToken A continuation token, used to page through saved reports. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                list: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/reports/saved').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'GET'
                        }, options),
                        params: params,
                        requiredParams: ['accountId'],
                        pathParams: ['accountId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                }
            }
        },
        savedadstyles: {
            /**
             * adsense.accounts.savedadstyles.get
             *
             * @desc List a specific saved ad style for the specified account.
             *
             * @alias adsense.accounts.savedadstyles.get
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account for which to get the saved ad style.
             * @param {string} params.savedAdStyleId Saved ad style to retrieve.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            get: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/savedadstyles/{savedAdStyleId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'savedAdStyleId'],
                    pathParams: ['accountId', 'savedAdStyleId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * adsense.accounts.savedadstyles.list
             *
             * @desc List all saved ad styles in the specified account.
             *
             * @alias adsense.accounts.savedadstyles.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account for which to list saved ad styles.
             * @param {integer=} params.maxResults The maximum number of saved ad styles to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through saved ad styles. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/savedadstyles').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId'],
                    pathParams: ['accountId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        },
        urlchannels: {
            /**
             * adsense.accounts.urlchannels.list
             *
             * @desc List all URL channels in the specified ad client for the specified account.
             *
             * @alias adsense.accounts.urlchannels.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.accountId Account to which the ad client belongs.
             * @param {string} params.adClientId Ad client for which to list URL channels.
             * @param {integer=} params.maxResults The maximum number of URL channels to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through URL channels. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/accounts/{accountId}/adclients/{adClientId}/urlchannels').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['accountId', 'adClientId'],
                    pathParams: ['accountId', 'adClientId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        }
    };
    self.adclients = {
        /**
         * adsense.adclients.list
         *
         * @desc List all ad clients in this AdSense account.
         *
         * @alias adsense.adclients.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object=} params Parameters for request
         * @param {integer=} params.maxResults The maximum number of ad clients to include in the response, used for paging.
         * @param {string=} params.pageToken A continuation token, used to page through ad clients. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/adclients').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.adunits = {
        /**
         * adsense.adunits.get
         *
         * @desc Gets the specified ad unit in the specified ad client.
         *
         * @alias adsense.adunits.get
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.adClientId Ad client for which to get the ad unit.
         * @param {string} params.adUnitId Ad unit to retrieve.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/adunits/{adUnitId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['adClientId', 'adUnitId'],
                pathParams: ['adClientId', 'adUnitId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * adsense.adunits.getAdCode
         *
         * @desc Get ad code for the specified ad unit.
         *
         * @alias adsense.adunits.getAdCode
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.adClientId Ad client with contains the ad unit.
         * @param {string} params.adUnitId Ad unit to get the code for.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        getAdCode: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/adunits/{adUnitId}/adcode').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['adClientId', 'adUnitId'],
                pathParams: ['adClientId', 'adUnitId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * adsense.adunits.list
         *
         * @desc List all ad units in the specified ad client for this AdSense account.
         *
         * @alias adsense.adunits.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.adClientId Ad client for which to list ad units.
         * @param {boolean=} params.includeInactive Whether to include inactive ad units. Default: true.
         * @param {integer=} params.maxResults The maximum number of ad units to include in the response, used for paging.
         * @param {string=} params.pageToken A continuation token, used to page through ad units. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/adunits').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['adClientId'],
                pathParams: ['adClientId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        customchannels: {
            /**
             * adsense.adunits.customchannels.list
             *
             * @desc List all custom channels which the specified ad unit belongs to.
             *
             * @alias adsense.adunits.customchannels.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.adClientId Ad client which contains the ad unit.
             * @param {string} params.adUnitId Ad unit for which to list custom channels.
             * @param {integer=} params.maxResults The maximum number of custom channels to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through custom channels. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/adunits/{adUnitId}/customchannels').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['adClientId', 'adUnitId'],
                    pathParams: ['adClientId', 'adUnitId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        }
    };
    self.alerts = {
        /**
         * adsense.alerts.delete
         *
         * @desc Dismiss (delete) the specified alert from the publisher's AdSense account.
         *
         * @alias adsense.alerts.delete
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.alertId Alert to delete.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        delete: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/alerts/{alertId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'DELETE'
                }, options),
                params: params,
                requiredParams: ['alertId'],
                pathParams: ['alertId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * adsense.alerts.list
         *
         * @desc List the alerts for this AdSense account.
         *
         * @alias adsense.alerts.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object=} params Parameters for request
         * @param {string=} params.locale The locale to use for translating alert messages. The account locale will be used if this is not supplied. The AdSense default (English) will be used if the supplied locale is invalid or unsupported.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/alerts').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.customchannels = {
        /**
         * adsense.customchannels.get
         *
         * @desc Get the specified custom channel from the specified ad client.
         *
         * @alias adsense.customchannels.get
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.adClientId Ad client which contains the custom channel.
         * @param {string} params.customChannelId Custom channel to retrieve.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/customchannels/{customChannelId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['adClientId', 'customChannelId'],
                pathParams: ['adClientId', 'customChannelId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * adsense.customchannels.list
         *
         * @desc List all custom channels in the specified ad client for this AdSense account.
         *
         * @alias adsense.customchannels.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.adClientId Ad client for which to list custom channels.
         * @param {integer=} params.maxResults The maximum number of custom channels to include in the response, used for paging.
         * @param {string=} params.pageToken A continuation token, used to page through custom channels. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/customchannels').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['adClientId'],
                pathParams: ['adClientId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        adunits: {
            /**
             * adsense.customchannels.adunits.list
             *
             * @desc List all ad units in the specified custom channel.
             *
             * @alias adsense.customchannels.adunits.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.adClientId Ad client which contains the custom channel.
             * @param {string} params.customChannelId Custom channel for which to list ad units.
             * @param {boolean=} params.includeInactive Whether to include inactive ad units. Default: true.
             * @param {integer=} params.maxResults The maximum number of ad units to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through ad units. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/customchannels/{customChannelId}/adunits').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['adClientId', 'customChannelId'],
                    pathParams: ['adClientId', 'customChannelId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        }
    };
    self.metadata = {
        dimensions: {
            /**
             * adsense.metadata.dimensions.list
             *
             * @desc List the metadata for the dimensions available to this AdSense account.
             *
             * @alias adsense.metadata.dimensions.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object=} params Parameters for request
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/metadata/dimensions').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: [],
                    pathParams: [],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        },
        metrics: {
            /**
             * adsense.metadata.metrics.list
             *
             * @desc List the metadata for the metrics available to this AdSense account.
             *
             * @alias adsense.metadata.metrics.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object=} params Parameters for request
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/metadata/metrics').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: [],
                    pathParams: [],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        }
    };
    self.payments = {
        /**
         * adsense.payments.list
         *
         * @desc List the payments for this AdSense account.
         *
         * @alias adsense.payments.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object=} params Parameters for request
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/payments').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.reports = {
        /**
         * adsense.reports.generate
         *
         * @desc Generate an AdSense report based on the report request sent in the query parameters. Returns the result as JSON; to retrieve output in CSV format specify "alt=csv" as a query parameter.
         *
         * @alias adsense.reports.generate
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string=} params.accountId Accounts upon which to report.
         * @param {string=} params.currency Optional currency to use when reporting on monetary metrics. Defaults to the account's currency if not set.
         * @param {string=} params.dimension Dimensions to base the report on.
         * @param {string} params.endDate End of the date range to report on in "YYYY-MM-DD" format, inclusive.
         * @param {string=} params.filter Filters to be run on the report.
         * @param {string=} params.locale Optional locale to use for translating report output to a local language. Defaults to "en_US" if not specified.
         * @param {integer=} params.maxResults The maximum number of rows of report data to return.
         * @param {string=} params.metric Numeric columns to include in the report.
         * @param {string=} params.sort The name of a dimension or metric to sort the resulting report on, optionally prefixed with "+" to sort ascending or "-" to sort descending. If no prefix is specified, the column is sorted ascending.
         * @param {string} params.startDate Start of the date range to report on in "YYYY-MM-DD" format, inclusive.
         * @param {integer=} params.startIndex Index of the first row of report data to return.
         * @param {boolean=} params.useTimezoneReporting Whether the report should be generated in the AdSense account's local timezone. If false default PST/PDT timezone will be used.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        generate: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/reports').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['startDate', 'endDate'],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        saved: {
            /**
             * adsense.reports.saved.generate
             *
             * @desc Generate an AdSense report based on the saved report ID sent in the query parameters.
             *
             * @alias adsense.reports.saved.generate
             * @memberOf! adsense(v1.4)
             *
             * @param {object} params Parameters for request
             * @param {string=} params.locale Optional locale to use for translating report output to a local language. Defaults to "en_US" if not specified.
             * @param {integer=} params.maxResults The maximum number of rows of report data to return.
             * @param {string} params.savedReportId The saved report to retrieve.
             * @param {integer=} params.startIndex Index of the first row of report data to return.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            generate: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/reports/{savedReportId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['savedReportId'],
                    pathParams: ['savedReportId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * adsense.reports.saved.list
             *
             * @desc List all saved reports in this AdSense account.
             *
             * @alias adsense.reports.saved.list
             * @memberOf! adsense(v1.4)
             *
             * @param {object=} params Parameters for request
             * @param {integer=} params.maxResults The maximum number of saved reports to include in the response, used for paging.
             * @param {string=} params.pageToken A continuation token, used to page through saved reports. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/adsense/v1.4/reports/saved').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: [],
                    pathParams: [],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        }
    };
    self.savedadstyles = {
        /**
         * adsense.savedadstyles.get
         *
         * @desc Get a specific saved ad style from the user's account.
         *
         * @alias adsense.savedadstyles.get
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.savedAdStyleId Saved ad style to retrieve.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/savedadstyles/{savedAdStyleId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['savedAdStyleId'],
                pathParams: ['savedAdStyleId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * adsense.savedadstyles.list
         *
         * @desc List all saved ad styles in the user's account.
         *
         * @alias adsense.savedadstyles.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object=} params Parameters for request
         * @param {integer=} params.maxResults The maximum number of saved ad styles to include in the response, used for paging.
         * @param {string=} params.pageToken A continuation token, used to page through saved ad styles. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/savedadstyles').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.urlchannels = {
        /**
         * adsense.urlchannels.list
         *
         * @desc List all URL channels in the specified ad client for this AdSense account.
         *
         * @alias adsense.urlchannels.list
         * @memberOf! adsense(v1.4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.adClientId Ad client for which to list URL channels.
         * @param {integer=} params.maxResults The maximum number of URL channels to include in the response, used for paging.
         * @param {string=} params.pageToken A continuation token, used to page through URL channels. To retrieve the next page, set this parameter to the value of "nextPageToken" from the previous response.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/adsense/v1.4/adclients/{adClientId}/urlchannels').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['adClientId'],
                pathParams: ['adClientId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
}
module.exports = Adsense;
//# sourceMappingURL=v1.4.js.map