"use strict";
/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* jshint maxlen: false */
const apirequest_1 = require("../../lib/apirequest");
/**
 * URL Shortener API
 *
 * Lets you create, inspect, and manage goo.gl short URLs
 *
 * @example
 * const google = require('googleapis');
 * const urlshortener = google.urlshortener('v1');
 *
 * @namespace urlshortener
 * @type {Function}
 * @version v1
 * @variation v1
 * @param {object=} options Options for Urlshortener
 */
function Urlshortener(options) {
    const self = this;
    self._options = options || {};
    self.url = {
        /**
         * urlshortener.url.get
         *
         * @desc Expands a short URL or gets creation time and analytics.
         *
         * @alias urlshortener.url.get
         * @memberOf! urlshortener(v1)
         *
         * @param {object} params Parameters for request
         * @param {string=} params.projection Additional information to return.
         * @param {string} params.shortUrl The short URL, including the protocol.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/urlshortener/v1/url').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['shortUrl'],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * urlshortener.url.insert
         *
         * @desc Creates a new short URL.
         *
         * @alias urlshortener.url.insert
         * @memberOf! urlshortener(v1)
         *
         * @param {object} params Parameters for request
         * @param {urlshortener(v1).Url} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        insert: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/urlshortener/v1/url').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * urlshortener.url.list
         *
         * @desc Retrieves a list of URLs shortened by a user.
         *
         * @alias urlshortener.url.list
         * @memberOf! urlshortener(v1)
         *
         * @param {object=} params Parameters for request
         * @param {string=} params.projection Additional information to return.
         * @param {string=} params.start-token Token for requesting successive pages of results.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/urlshortener/v1/url/history').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
}
module.exports = Urlshortener;
//# sourceMappingURL=v1.js.map