"use strict";
/**
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @constructor
 * @struct
 * @param {!string} type The event type
 * @param {!Node} snapshotNode The data
 * @param {string=} childName The name for this child, if it's a child event
 * @param {Node=} oldSnap Used for intermediate processing of child changed events
 * @param {string=} prevName The name for the previous child, if applicable
 */
var Change = /** @class */ (function () {
    function Change(type, snapshotNode, childName, oldSnap, prevName) {
        this.type = type;
        this.snapshotNode = snapshotNode;
        this.childName = childName;
        this.oldSnap = oldSnap;
        this.prevName = prevName;
    }
    /**
     * @param {!Node} snapshot
     * @return {!Change}
     */
    Change.valueChange = function (snapshot) {
        return new Change(Change.VALUE, snapshot);
    };
    /**
     * @param {string} childKey
     * @param {!Node} snapshot
     * @return {!Change}
     */
    Change.childAddedChange = function (childKey, snapshot) {
        return new Change(Change.CHILD_ADDED, snapshot, childKey);
    };
    /**
     * @param {string} childKey
     * @param {!Node} snapshot
     * @return {!Change}
     */
    Change.childRemovedChange = function (childKey, snapshot) {
        return new Change(Change.CHILD_REMOVED, snapshot, childKey);
    };
    /**
     * @param {string} childKey
     * @param {!Node} newSnapshot
     * @param {!Node} oldSnapshot
     * @return {!Change}
     */
    Change.childChangedChange = function (childKey, newSnapshot, oldSnapshot) {
        return new Change(Change.CHILD_CHANGED, newSnapshot, childKey, oldSnapshot);
    };
    /**
     * @param {string} childKey
     * @param {!Node} snapshot
     * @return {!Change}
     */
    Change.childMovedChange = function (childKey, snapshot) {
        return new Change(Change.CHILD_MOVED, snapshot, childKey);
    };
    //event types
    /** Event type for a child added */
    Change.CHILD_ADDED = 'child_added';
    /** Event type for a child removed */
    Change.CHILD_REMOVED = 'child_removed';
    /** Event type for a child changed */
    Change.CHILD_CHANGED = 'child_changed';
    /** Event type for a child moved */
    Change.CHILD_MOVED = 'child_moved';
    /** Event type for a value change */
    Change.VALUE = 'value';
    return Change;
}());
exports.Change = Change;

//# sourceMappingURL=Change.js.map
