/**
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { User } from '../auth/user';
import { FirebaseApp } from '@firebase/app';
export interface FirstPartyCredentialsSettings {
    type: 'gapi';
    client: Gapi;
    sessionIndex: string;
}
export interface GoogleAuthCredentialsSettings {
    type: 'google-auth';
    client: GoogleAuthClient;
}
export interface ProviderCredentialsSettings {
    type: 'provider';
    client: CredentialsProvider;
}
/** Settings for private credentials */
export declare type CredentialsSettings = FirstPartyCredentialsSettings | GoogleAuthCredentialsSettings | ProviderCredentialsSettings;
export declare type TokenType = 'OAuth' | 'FirstParty';
export interface Token {
    /** Type of token. */
    type: TokenType;
    /**
     * The user with which the token is associated (used for persisting user
     * state on disk, etc.).
     */
    user: User;
    /** Extra header values to be passed along with a request */
    authHeaders: {
        [header: string]: string;
    };
}
export declare class OAuthToken implements Token {
    user: User;
    type: TokenType;
    authHeaders: {
        [header: string]: string;
    };
    constructor(value: string, user: User);
}
/**
 * A Listener for user change events.
 */
export declare type UserListener = (user: User) => void;
/**
 * Provides methods for getting the uid and token for the current user and
 * listening for changes.
 */
export interface CredentialsProvider {
    /**
     * Requests a token for the current user, optionally forcing a refreshed
     * token to be fetched.
     */
    getToken(forceRefresh: boolean): Promise<Token | null>;
    /**
     * Specifies a listener to be notified of user changes (sign-in / sign-out).
     * It immediately called once with the initial user.
     */
    setUserChangeListener(listener: UserListener): void;
    /** Removes the previously-set user change listener. */
    removeUserChangeListener(): void;
}
/** A CredentialsProvider that always yields an empty token. */
export declare class EmptyCredentialsProvider implements CredentialsProvider {
    /**
     * Stores the User listener registered with setUserChangeListener()
     * This isn't actually necessary since the UID never changes, but we use this
     * to verify the listen contract is adhered to in tests.
     */
    private userListener;
    constructor();
    getToken(forceRefresh: boolean): Promise<Token | null>;
    setUserChangeListener(listener: UserListener): void;
    removeUserChangeListener(): void;
}
export declare class FirebaseCredentialsProvider implements CredentialsProvider {
    private readonly app;
    /**
     * The auth token listener registered with FirebaseApp, retained here so we
     * can unregister it.
     */
    private tokenListener;
    /** Tracks the current User. */
    private currentUser;
    /**
     * Counter used to detect if the user changed while a getToken request was
     * outstanding.
     */
    private userCounter;
    /** The User listener registered with setUserChangeListener(). */
    private userListener;
    constructor(app: FirebaseApp);
    getToken(forceRefresh: boolean): Promise<Token | null>;
    setUserChangeListener(listener: UserListener): void;
    removeUserChangeListener(): void;
    private getUser();
}
export declare class GoogleCredentialsProvider implements CredentialsProvider {
    private authClient;
    constructor(authClient: GoogleAuthClient);
    getToken(forceRefresh: boolean): Promise<Token | null>;
    setUserChangeListener(listener: UserListener): void;
    removeUserChangeListener(): void;
}
/**
 * Very incomplete typing for an auth client from
 * https://github.com/google/google-auth-library-nodejs/
 */
export interface GoogleAuthClient {
    getAccessToken(callback: (error?: Error, token?: string) => void): void;
}
export declare type Gapi = any;
export declare class FirstPartyToken implements Token {
    private gapi;
    private sessionIndex;
    type: TokenType;
    user: User;
    constructor(gapi: Gapi, sessionIndex: string);
    readonly authHeaders: {
        [header: string]: string;
    };
}
export declare class FirstPartyCredentialsProvider implements CredentialsProvider {
    private gapi;
    private sessionIndex;
    constructor(gapi: Gapi, sessionIndex: string);
    getToken(forceRefresh: boolean): Promise<Token | null>;
    setUserChangeListener(listener: UserListener): void;
    removeUserChangeListener(): void;
}
/**
 * Builds a CredentialsProvider depending on the type of
 * the credentials passed in.
 */
export declare function makeCredentialsProvider(credentials?: CredentialsSettings): CredentialsProvider;
