/**
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { Query } from './query';
import { SyncEngine } from './sync_engine';
import { OnlineState, TargetId } from './types';
import { ViewSnapshot } from './view_snapshot';
import { EventHandler } from '../util/misc';
/**
 * Interface for handling events from the EventManager.
 */
export interface Observer<T> {
    next: EventHandler<T>;
    error: EventHandler<Error>;
}
/**
 * EventManager is responsible for mapping queries to query event emitters.
 * It handles "fan-out". -- Identical queries will re-use the same watch on the
 * backend.
 */
export declare class EventManager {
    private syncEngine;
    private queries;
    private onlineState;
    constructor(syncEngine: SyncEngine);
    listen(listener: QueryListener): Promise<TargetId>;
    unlisten(listener: QueryListener): Promise<void>;
    onChange(viewSnaps: ViewSnapshot[]): void;
    onError(query: Query, error: Error): void;
    onOnlineStateChanged(onlineState: OnlineState): void;
}
export interface ListenOptions {
    /** Raise events when only metadata of documents changes */
    readonly includeDocumentMetadataChanges?: boolean;
    /** Raise events when only metadata of the query changes */
    readonly includeQueryMetadataChanges?: boolean;
    /**
     * Wait for a sync with the server when online, but still raise events while
     * offline.
     */
    readonly waitForSyncWhenOnline?: boolean;
}
/**
 * QueryListener takes a series of internal view snapshots and determines
 * when to raise the event.
 *
 * It uses an Observer to dispatch events.
 */
export declare class QueryListener {
    readonly query: Query;
    private queryObserver;
    /**
     * Initial snapshots (e.g. from cache) may not be propagated to the wrapped
     * observer. This flag is set to true once we've actually raised an event.
     */
    private raisedInitialEvent;
    private options;
    private snap;
    private onlineState;
    constructor(query: Query, queryObserver: Observer<ViewSnapshot>, options?: ListenOptions);
    onViewSnapshot(snap: ViewSnapshot): void;
    onError(error: Error): void;
    onOnlineStateChanged(onlineState: OnlineState): void;
    private shouldRaiseInitialEvent(snap, onlineState);
    private shouldRaiseEvent(snap);
    private raiseInitialEvent(snap);
    /** Returns changes as if all documents in the snap were added. */
    private static getInitialViewChanges(snap);
}
