/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
"use strict";
var __extends = (this && this.__extends) || (function () {
    var extendStatics = Object.setPrototypeOf ||
        ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
var oauth2client_1 = require("./oauth2client");
var UserRefreshClient = (function (_super) {
    __extends(UserRefreshClient, _super);
    /**
     * User Refresh Token credentials.
     *
     * @param {string} clientId The authentication client ID.
     * @param {string} clientSecret The authentication client secret.
     * @param {string} refreshToken The authentication refresh token.
     * @constructor
     */
    function UserRefreshClient(clientId, clientSecret, refreshToken) {
        var _this = _super.call(this, clientId, clientSecret) || this;
        // Named to avoid collision with the method refreshToken_
        _this._refreshToken = refreshToken;
        return _this;
    }
    // Executes the given callback if it is not null.
    UserRefreshClient.prototype.callback = function (c, err, res) {
        if (c) {
            c(err, res);
        }
    };
    /**
     * Refreshes the access token.
     * @param {object=} ignored_
     * @param {function=} callback Optional callback.
     * @private
     */
    UserRefreshClient.prototype.refreshToken = function (ignored_, callback) {
        return _super.prototype.refreshToken.call(this, this._refreshToken, callback);
    };
    /**
     * Create a UserRefreshClient credentials instance using the given input
     * options.
     * @param {object=} json The input object.
     * @param {function=} callback Optional callback.
     */
    UserRefreshClient.prototype.fromJSON = function (json, callback) {
        if (!json) {
            this.callback(callback, new Error('Must pass in a JSON object containing the user refresh token'));
            return;
        }
        if (json.type !== 'authorized_user') {
            this.callback(callback, new Error('The incoming JSON object does not have the "authorized_user" type'));
            return;
        }
        if (!json.client_id) {
            this.callback(callback, new Error('The incoming JSON object does not contain a client_id field'));
            return;
        }
        if (!json.client_secret) {
            this.callback(callback, new Error('The incoming JSON object does not contain a client_secret field'));
            return;
        }
        if (!json.refresh_token) {
            this.callback(callback, new Error('The incoming JSON object does not contain a refresh_token field'));
            return;
        }
        this._clientId = json.client_id;
        this._clientSecret = json.client_secret;
        this._refreshToken = json.refresh_token;
        this.credentials.refresh_token = json.refresh_token;
        this.callback(callback);
    };
    /**
     * Create a UserRefreshClient credentials instance using the given input
     * stream.
     * @param {object=} stream The input stream.
     * @param {function=} callback Optional callback.
     */
    UserRefreshClient.prototype.fromStream = function (stream, callback) {
        var _this = this;
        if (!stream) {
            setImmediate(function () {
                _this.callback(callback, new Error('Must pass in a stream containing the user refresh token.'));
            });
            return;
        }
        var s = '';
        stream.setEncoding('utf8');
        stream.on('data', function (chunk) {
            s += chunk;
        });
        stream.on('end', function () {
            try {
                var data = JSON.parse(s);
                _this.fromJSON(data, callback);
            }
            catch (err) {
                _this.callback(callback, err);
            }
        });
    };
    return UserRefreshClient;
}(oauth2client_1.OAuth2Client));
exports.UserRefreshClient = UserRefreshClient;
//# sourceMappingURL=refreshclient.js.map