"use strict";
/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* jshint maxlen: false */
const apirequest_1 = require("../../lib/apirequest");
/**
 * Google App Engine Admin API
 *
 * The App Engine Admin API enables developers to provision and manage their App Engine applications.
 *
 * @example
 * const google = require('googleapis');
 * const appengine = google.appengine('v1beta4');
 *
 * @namespace appengine
 * @type {Function}
 * @version v1beta4
 * @variation v1beta4
 * @param {object=} options Options for Appengine
 */
function Appengine(options) {
    const self = this;
    self._options = options || {};
    self.apps = {
        /**
         * appengine.apps.create
         *
         * @desc Creates an App Engine application for a Google Cloud Platform project. Required fields: id - The ID of the target Cloud Platform project. location - The region (https://cloud.google.com/appengine/docs/locations) where you want the App Engine application located.For more information about App Engine applications, see Managing Projects, Applications, and Billing (https://cloud.google.com/appengine/docs/python/console/).
         *
         * @alias appengine.apps.create
         * @memberOf! appengine(v1beta4)
         *
         * @param {object} params Parameters for request
         * @param {appengine(v1beta4).Application} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        create: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/v1beta4/apps').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                requiredParams: [],
                pathParams: [],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * appengine.apps.get
         *
         * @desc Gets information about an application.
         *
         * @alias appengine.apps.get
         * @memberOf! appengine(v1beta4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.appsId Part of `name`. Name of the application to get. Example: apps/myapp.
         * @param {boolean=} params.ensureResourcesExist Certain resources associated with an application are created on-demand. Controls whether these resources should be created when performing the GET operation. If specified and any resources could not be created, the request will fail with an error code. Additionally, this parameter can cause the request to take longer to complete.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/v1beta4/apps/{appsId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['appsId'],
                pathParams: ['appsId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * appengine.apps.patch
         *
         * @desc Updates the specified Application resource. You can update the following fields: auth_domain (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps#Application.FIELDS.auth_domain) default_cookie_expiration (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps#Application.FIELDS.default_cookie_expiration)
         *
         * @alias appengine.apps.patch
         * @memberOf! appengine(v1beta4)
         *
         * @param {object} params Parameters for request
         * @param {string} params.appsId Part of `name`. Name of the Application resource to update. Example: apps/myapp.
         * @param {string=} params.mask Standard field mask for the set of fields to be updated.
         * @param {appengine(v1beta4).Application} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        patch: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/v1beta4/apps/{appsId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PATCH'
                }, options),
                params: params,
                requiredParams: ['appsId'],
                pathParams: ['appsId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        locations: {
            /**
             * appengine.apps.locations.get
             *
             * @desc Get information about a location.
             *
             * @alias appengine.apps.locations.get
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. Resource name for the location.
             * @param {string} params.locationsId Part of `name`. See documentation of `appsId`.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            get: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/locations/{locationsId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['appsId', 'locationsId'],
                    pathParams: ['appsId', 'locationsId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * appengine.apps.locations.list
             *
             * @desc Lists information about the supported locations for this service.
             *
             * @alias appengine.apps.locations.list
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. The resource that owns the locations collection, if applicable.
             * @param {string=} params.filter The standard list filter.
             * @param {integer=} params.pageSize The standard list page size.
             * @param {string=} params.pageToken The standard list page token.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/locations').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['appsId'],
                    pathParams: ['appsId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        },
        modules: {
            /**
             * appengine.apps.modules.delete
             *
             * @desc Deletes the specified module and all enclosed versions.
             *
             * @alias appengine.apps.modules.delete
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default.
             * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            delete: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'DELETE'
                    }, options),
                    params: params,
                    requiredParams: ['appsId', 'modulesId'],
                    pathParams: ['appsId', 'modulesId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * appengine.apps.modules.get
             *
             * @desc Gets the current configuration of the specified module.
             *
             * @alias appengine.apps.modules.get
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default.
             * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            get: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['appsId', 'modulesId'],
                    pathParams: ['appsId', 'modulesId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * appengine.apps.modules.list
             *
             * @desc Lists all the modules in the application.
             *
             * @alias appengine.apps.modules.list
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp.
             * @param {integer=} params.pageSize Maximum results to return per page.
             * @param {string=} params.pageToken Continuation token for fetching the next page of results.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/modules').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['appsId'],
                    pathParams: ['appsId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * appengine.apps.modules.patch
             *
             * @desc Updates the configuration of the specified module.
             *
             * @alias appengine.apps.modules.patch
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. Name of the resource to update. Example: apps/myapp/modules/default.
             * @param {string=} params.mask Standard field mask for the set of fields to be updated.
             * @param {boolean=} params.migrateTraffic Set to true to gradually shift traffic to one or more versions that you specify. By default, traffic is shifted immediately. For gradual traffic migration, the target versions must be located within instances that are configured for both warmup requests (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps.modules.versions#inboundservicetype) and automatic scaling (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps.modules.versions#automaticscaling). You must specify the shardBy (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps.modules#shardby) field in the Module resource. Gradual traffic migration is not supported in the App Engine flexible environment. For examples, see Migrating and Splitting Traffic (https://cloud.google.com/appengine/docs/admin-api/migrating-splitting-traffic).
             * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
             * @param {appengine(v1beta4).Module} params.resource Request body data
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            patch: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'PATCH'
                    }, options),
                    params: params,
                    requiredParams: ['appsId', 'modulesId'],
                    pathParams: ['appsId', 'modulesId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            versions: {
                /**
                 * appengine.apps.modules.versions.create
                 *
                 * @desc Deploys code and resource files to a new version.
                 *
                 * @alias appengine.apps.modules.versions.create
                 * @memberOf! appengine(v1beta4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.appsId Part of `name`. Name of the resource to update. Example: apps/myapp/modules/default.
                 * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                 * @param {appengine(v1beta4).Version} params.resource Request body data
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                create: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'POST'
                        }, options),
                        params: params,
                        requiredParams: ['appsId', 'modulesId'],
                        pathParams: ['appsId', 'modulesId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                },
                /**
                 * appengine.apps.modules.versions.delete
                 *
                 * @desc Deletes an existing version.
                 *
                 * @alias appengine.apps.modules.versions.delete
                 * @memberOf! appengine(v1beta4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default/versions/v1.
                 * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                 * @param {string} params.versionsId Part of `name`. See documentation of `appsId`.
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                delete: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions/{versionsId}').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'DELETE'
                        }, options),
                        params: params,
                        requiredParams: ['appsId', 'modulesId', 'versionsId'],
                        pathParams: ['appsId', 'modulesId', 'versionsId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                },
                /**
                 * appengine.apps.modules.versions.get
                 *
                 * @desc Gets the specified Version resource. By default, only a BASIC_VIEW will be returned. Specify the FULL_VIEW parameter to get the full resource.
                 *
                 * @alias appengine.apps.modules.versions.get
                 * @memberOf! appengine(v1beta4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default/versions/v1.
                 * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                 * @param {string} params.versionsId Part of `name`. See documentation of `appsId`.
                 * @param {string=} params.view Controls the set of fields returned in the Get response.
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                get: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions/{versionsId}').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'GET'
                        }, options),
                        params: params,
                        requiredParams: ['appsId', 'modulesId', 'versionsId'],
                        pathParams: ['appsId', 'modulesId', 'versionsId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                },
                /**
                 * appengine.apps.modules.versions.list
                 *
                 * @desc Lists the versions of a module.
                 *
                 * @alias appengine.apps.modules.versions.list
                 * @memberOf! appengine(v1beta4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default.
                 * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                 * @param {integer=} params.pageSize Maximum results to return per page.
                 * @param {string=} params.pageToken Continuation token for fetching the next page of results.
                 * @param {string=} params.view Controls the set of fields returned in the List response.
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                list: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'GET'
                        }, options),
                        params: params,
                        requiredParams: ['appsId', 'modulesId'],
                        pathParams: ['appsId', 'modulesId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                },
                /**
                 * appengine.apps.modules.versions.patch
                 *
                 * @desc Updates the specified Version resource. You can specify the following fields depending on the App Engine environment and type of scaling that the version resource uses: serving_status (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps.modules.versions#Version.FIELDS.serving_status):  For Version resources that use basic scaling, manual scaling, or run in  the App Engine flexible environment. instance_class (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps.modules.versions#Version.FIELDS.instance_class):  For Version resources that run in the App Engine standard environment. automatic_scaling.min_idle_instances (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps.modules.versions#Version.FIELDS.automatic_scaling):  For Version resources that use automatic scaling and run in the App  Engine standard environment. automatic_scaling.max_idle_instances (https://cloud.google.com/appengine/docs/admin-api/reference/rest/v1beta4/apps.modules.versions#Version.FIELDS.automatic_scaling):  For Version resources that use automatic scaling and run in the App  Engine standard environment.
                 *
                 * @alias appengine.apps.modules.versions.patch
                 * @memberOf! appengine(v1beta4)
                 *
                 * @param {object} params Parameters for request
                 * @param {string} params.appsId Part of `name`. Name of the resource to update. Example: apps/myapp/modules/default/versions/1.
                 * @param {string=} params.mask Standard field mask for the set of fields to be updated.
                 * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                 * @param {string} params.versionsId Part of `name`. See documentation of `appsId`.
                 * @param {appengine(v1beta4).Version} params.resource Request body data
                 * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                 * @param {callback} callback The callback that handles the response.
                 * @return {object} Request object
                 */
                patch: function (params, options, callback) {
                    if (typeof options === 'function') {
                        callback = options;
                        options = {};
                    }
                    options || (options = {});
                    const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                    const parameters = {
                        options: Object.assign({
                            url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions/{versionsId}').replace(/([^:]\/)\/+/g, '$1'),
                            method: 'PATCH'
                        }, options),
                        params: params,
                        requiredParams: ['appsId', 'modulesId', 'versionsId'],
                        pathParams: ['appsId', 'modulesId', 'versionsId'],
                        context: self
                    };
                    return apirequest_1.default(parameters, callback);
                },
                instances: {
                    /**
                     * appengine.apps.modules.versions.instances.debug
                     *
                     * @desc Enables debugging on a VM instance. This allows you to use the SSH command to connect to the virtual machine where the instance lives. While in "debug mode", the instance continues to serve live traffic. You should delete the instance when you are done debugging and then allow the system to take over and determine if another instance should be started.Only applicable for instances in App Engine flexible environment.
                     *
                     * @alias appengine.apps.modules.versions.instances.debug
                     * @memberOf! appengine(v1beta4)
                     *
                     * @param {object} params Parameters for request
                     * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default/versions/v1/instances/instance-1.
                     * @param {string} params.instancesId Part of `name`. See documentation of `appsId`.
                     * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                     * @param {string} params.versionsId Part of `name`. See documentation of `appsId`.
                     * @param {appengine(v1beta4).DebugInstanceRequest} params.resource Request body data
                     * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                     * @param {callback} callback The callback that handles the response.
                     * @return {object} Request object
                     */
                    debug: function (params, options, callback) {
                        if (typeof options === 'function') {
                            callback = options;
                            options = {};
                        }
                        options || (options = {});
                        const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                        const parameters = {
                            options: Object.assign({
                                url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions/{versionsId}/instances/{instancesId}:debug').replace(/([^:]\/)\/+/g, '$1'),
                                method: 'POST'
                            }, options),
                            params: params,
                            requiredParams: ['appsId', 'modulesId', 'versionsId', 'instancesId'],
                            pathParams: ['appsId', 'instancesId', 'modulesId', 'versionsId'],
                            context: self
                        };
                        return apirequest_1.default(parameters, callback);
                    },
                    /**
                     * appengine.apps.modules.versions.instances.delete
                     *
                     * @desc Stops a running instance.
                     *
                     * @alias appengine.apps.modules.versions.instances.delete
                     * @memberOf! appengine(v1beta4)
                     *
                     * @param {object} params Parameters for request
                     * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default/versions/v1/instances/instance-1.
                     * @param {string} params.instancesId Part of `name`. See documentation of `appsId`.
                     * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                     * @param {string} params.versionsId Part of `name`. See documentation of `appsId`.
                     * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                     * @param {callback} callback The callback that handles the response.
                     * @return {object} Request object
                     */
                    delete: function (params, options, callback) {
                        if (typeof options === 'function') {
                            callback = options;
                            options = {};
                        }
                        options || (options = {});
                        const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                        const parameters = {
                            options: Object.assign({
                                url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions/{versionsId}/instances/{instancesId}').replace(/([^:]\/)\/+/g, '$1'),
                                method: 'DELETE'
                            }, options),
                            params: params,
                            requiredParams: ['appsId', 'modulesId', 'versionsId', 'instancesId'],
                            pathParams: ['appsId', 'instancesId', 'modulesId', 'versionsId'],
                            context: self
                        };
                        return apirequest_1.default(parameters, callback);
                    },
                    /**
                     * appengine.apps.modules.versions.instances.get
                     *
                     * @desc Gets instance information.
                     *
                     * @alias appengine.apps.modules.versions.instances.get
                     * @memberOf! appengine(v1beta4)
                     *
                     * @param {object} params Parameters for request
                     * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default/versions/v1/instances/instance-1.
                     * @param {string} params.instancesId Part of `name`. See documentation of `appsId`.
                     * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                     * @param {string} params.versionsId Part of `name`. See documentation of `appsId`.
                     * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                     * @param {callback} callback The callback that handles the response.
                     * @return {object} Request object
                     */
                    get: function (params, options, callback) {
                        if (typeof options === 'function') {
                            callback = options;
                            options = {};
                        }
                        options || (options = {});
                        const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                        const parameters = {
                            options: Object.assign({
                                url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions/{versionsId}/instances/{instancesId}').replace(/([^:]\/)\/+/g, '$1'),
                                method: 'GET'
                            }, options),
                            params: params,
                            requiredParams: ['appsId', 'modulesId', 'versionsId', 'instancesId'],
                            pathParams: ['appsId', 'instancesId', 'modulesId', 'versionsId'],
                            context: self
                        };
                        return apirequest_1.default(parameters, callback);
                    },
                    /**
                     * appengine.apps.modules.versions.instances.list
                     *
                     * @desc Lists the instances of a version.Tip: To aggregate details about instances over time, see the Stackdriver Monitoring API (https://cloud.google.com/monitoring/api/ref_v3/rest/v3/projects.timeSeries/list).
                     *
                     * @alias appengine.apps.modules.versions.instances.list
                     * @memberOf! appengine(v1beta4)
                     *
                     * @param {object} params Parameters for request
                     * @param {string} params.appsId Part of `name`. Name of the resource requested. Example: apps/myapp/modules/default/versions/v1.
                     * @param {string} params.modulesId Part of `name`. See documentation of `appsId`.
                     * @param {integer=} params.pageSize Maximum results to return per page.
                     * @param {string=} params.pageToken Continuation token for fetching the next page of results.
                     * @param {string} params.versionsId Part of `name`. See documentation of `appsId`.
                     * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
                     * @param {callback} callback The callback that handles the response.
                     * @return {object} Request object
                     */
                    list: function (params, options, callback) {
                        if (typeof options === 'function') {
                            callback = options;
                            options = {};
                        }
                        options || (options = {});
                        const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                        const parameters = {
                            options: Object.assign({
                                url: (rootUrl + '/v1beta4/apps/{appsId}/modules/{modulesId}/versions/{versionsId}/instances').replace(/([^:]\/)\/+/g, '$1'),
                                method: 'GET'
                            }, options),
                            params: params,
                            requiredParams: ['appsId', 'modulesId', 'versionsId'],
                            pathParams: ['appsId', 'modulesId', 'versionsId'],
                            context: self
                        };
                        return apirequest_1.default(parameters, callback);
                    }
                }
            }
        },
        operations: {
            /**
             * appengine.apps.operations.get
             *
             * @desc Gets the latest state of a long-running operation. Clients can use this method to poll the operation result at intervals as recommended by the API service.
             *
             * @alias appengine.apps.operations.get
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. The name of the operation resource.
             * @param {string} params.operationsId Part of `name`. See documentation of `appsId`.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            get: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/operations/{operationsId}').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['appsId', 'operationsId'],
                    pathParams: ['appsId', 'operationsId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            },
            /**
             * appengine.apps.operations.list
             *
             * @desc Lists operations that match the specified filter in the request. If the server doesn't support this method, it returns UNIMPLEMENTED.NOTE: the name binding allows API services to override the binding to use different resource name schemes, such as users/x/operations. To override the binding, API services can add a binding such as "/v1/{name=users/x}/operations" to their service configuration. For backwards compatibility, the default name includes the operations collection id, however overriding users must ensure the name binding is the parent resource, without the operations collection id.
             *
             * @alias appengine.apps.operations.list
             * @memberOf! appengine(v1beta4)
             *
             * @param {object} params Parameters for request
             * @param {string} params.appsId Part of `name`. The name of the operation's parent resource.
             * @param {string=} params.filter The standard list filter.
             * @param {integer=} params.pageSize The standard list page size.
             * @param {string=} params.pageToken The standard list page token.
             * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
             * @param {callback} callback The callback that handles the response.
             * @return {object} Request object
             */
            list: function (params, options, callback) {
                if (typeof options === 'function') {
                    callback = options;
                    options = {};
                }
                options || (options = {});
                const rootUrl = options.rootUrl || 'https://appengine.googleapis.com/';
                const parameters = {
                    options: Object.assign({
                        url: (rootUrl + '/v1beta4/apps/{appsId}/operations').replace(/([^:]\/)\/+/g, '$1'),
                        method: 'GET'
                    }, options),
                    params: params,
                    requiredParams: ['appsId'],
                    pathParams: ['appsId'],
                    context: self
                };
                return apirequest_1.default(parameters, callback);
            }
        }
    };
}
module.exports = Appengine;
//# sourceMappingURL=v1beta4.js.map