"use strict";
/**
 * Copyright 2015 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* jshint maxlen: false */
const apirequest_1 = require("../../lib/apirequest");
/**
 * Google Play Game Services Publishing API
 *
 * The Publishing API for Google Play Game Services.
 *
 * @example
 * const google = require('googleapis');
 * const gamesConfiguration = google.gamesConfiguration('v1configuration');
 *
 * @namespace gamesConfiguration
 * @type {Function}
 * @version v1configuration
 * @variation v1configuration
 * @param {object=} options Options for Gamesconfiguration
 */
function Gamesconfiguration(options) {
    const self = this;
    self._options = options || {};
    self.achievementConfigurations = {
        /**
         * gamesConfiguration.achievementConfigurations.delete
         *
         * @desc Delete the achievement configuration with the given ID.
         *
         * @alias gamesConfiguration.achievementConfigurations.delete
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.achievementId The ID of the achievement used by this method.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        delete: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/achievements/{achievementId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'DELETE'
                }, options),
                params: params,
                requiredParams: ['achievementId'],
                pathParams: ['achievementId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.achievementConfigurations.get
         *
         * @desc Retrieves the metadata of the achievement configuration with the given ID.
         *
         * @alias gamesConfiguration.achievementConfigurations.get
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.achievementId The ID of the achievement used by this method.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/achievements/{achievementId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['achievementId'],
                pathParams: ['achievementId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.achievementConfigurations.insert
         *
         * @desc Insert a new achievement configuration in this application.
         *
         * @alias gamesConfiguration.achievementConfigurations.insert
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.applicationId The application ID from the Google Play developer console.
         * @param {gamesConfiguration(v1configuration).AchievementConfiguration} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        insert: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/applications/{applicationId}/achievements').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                requiredParams: ['applicationId'],
                pathParams: ['applicationId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.achievementConfigurations.list
         *
         * @desc Returns a list of the achievement configurations in this application.
         *
         * @alias gamesConfiguration.achievementConfigurations.list
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.applicationId The application ID from the Google Play developer console.
         * @param {integer=} params.maxResults The maximum number of resource configurations to return in the response, used for paging. For any response, the actual number of resources returned may be less than the specified maxResults.
         * @param {string=} params.pageToken The token returned by the previous request.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/applications/{applicationId}/achievements').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['applicationId'],
                pathParams: ['applicationId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.achievementConfigurations.patch
         *
         * @desc Update the metadata of the achievement configuration with the given ID. This method supports patch semantics.
         *
         * @alias gamesConfiguration.achievementConfigurations.patch
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.achievementId The ID of the achievement used by this method.
         * @param {gamesConfiguration(v1configuration).AchievementConfiguration} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        patch: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/achievements/{achievementId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PATCH'
                }, options),
                params: params,
                requiredParams: ['achievementId'],
                pathParams: ['achievementId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.achievementConfigurations.update
         *
         * @desc Update the metadata of the achievement configuration with the given ID.
         *
         * @alias gamesConfiguration.achievementConfigurations.update
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.achievementId The ID of the achievement used by this method.
         * @param {gamesConfiguration(v1configuration).AchievementConfiguration} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        update: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/achievements/{achievementId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PUT'
                }, options),
                params: params,
                requiredParams: ['achievementId'],
                pathParams: ['achievementId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.imageConfigurations = {
        /**
         * gamesConfiguration.imageConfigurations.upload
         *
         * @desc Uploads an image for a resource with the given ID and image type.
         *
         * @alias gamesConfiguration.imageConfigurations.upload
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.imageType Selects which image in a resource for this method.
         * @param {string} params.resourceId The ID of the resource used by this method.
         * @param {object} params.media Media object
         * @param {string} params.media.mimeType Media mime-type
         * @param {string|object} params.media.body Media body contents
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        upload: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/images/{resourceId}/imageType/{imageType}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                mediaUrl: (rootUrl + '/upload/games/v1configuration/images/{resourceId}/imageType/{imageType}').replace(/([^:]\/)\/+/g, '$1'),
                requiredParams: ['resourceId', 'imageType'],
                pathParams: ['imageType', 'resourceId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
    self.leaderboardConfigurations = {
        /**
         * gamesConfiguration.leaderboardConfigurations.delete
         *
         * @desc Delete the leaderboard configuration with the given ID.
         *
         * @alias gamesConfiguration.leaderboardConfigurations.delete
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.leaderboardId The ID of the leaderboard.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        delete: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/leaderboards/{leaderboardId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'DELETE'
                }, options),
                params: params,
                requiredParams: ['leaderboardId'],
                pathParams: ['leaderboardId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.leaderboardConfigurations.get
         *
         * @desc Retrieves the metadata of the leaderboard configuration with the given ID.
         *
         * @alias gamesConfiguration.leaderboardConfigurations.get
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.leaderboardId The ID of the leaderboard.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        get: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/leaderboards/{leaderboardId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['leaderboardId'],
                pathParams: ['leaderboardId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.leaderboardConfigurations.insert
         *
         * @desc Insert a new leaderboard configuration in this application.
         *
         * @alias gamesConfiguration.leaderboardConfigurations.insert
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.applicationId The application ID from the Google Play developer console.
         * @param {gamesConfiguration(v1configuration).LeaderboardConfiguration} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        insert: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/applications/{applicationId}/leaderboards').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'POST'
                }, options),
                params: params,
                requiredParams: ['applicationId'],
                pathParams: ['applicationId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.leaderboardConfigurations.list
         *
         * @desc Returns a list of the leaderboard configurations in this application.
         *
         * @alias gamesConfiguration.leaderboardConfigurations.list
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.applicationId The application ID from the Google Play developer console.
         * @param {integer=} params.maxResults The maximum number of resource configurations to return in the response, used for paging. For any response, the actual number of resources returned may be less than the specified maxResults.
         * @param {string=} params.pageToken The token returned by the previous request.
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        list: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/applications/{applicationId}/leaderboards').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'GET'
                }, options),
                params: params,
                requiredParams: ['applicationId'],
                pathParams: ['applicationId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.leaderboardConfigurations.patch
         *
         * @desc Update the metadata of the leaderboard configuration with the given ID. This method supports patch semantics.
         *
         * @alias gamesConfiguration.leaderboardConfigurations.patch
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.leaderboardId The ID of the leaderboard.
         * @param {gamesConfiguration(v1configuration).LeaderboardConfiguration} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        patch: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/leaderboards/{leaderboardId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PATCH'
                }, options),
                params: params,
                requiredParams: ['leaderboardId'],
                pathParams: ['leaderboardId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        },
        /**
         * gamesConfiguration.leaderboardConfigurations.update
         *
         * @desc Update the metadata of the leaderboard configuration with the given ID.
         *
         * @alias gamesConfiguration.leaderboardConfigurations.update
         * @memberOf! gamesConfiguration(v1configuration)
         *
         * @param {object} params Parameters for request
         * @param {string} params.leaderboardId The ID of the leaderboard.
         * @param {gamesConfiguration(v1configuration).LeaderboardConfiguration} params.resource Request body data
         * @param {object} [options] Optionally override request options, such as `url`, `method`, and `encoding`.
         * @param {callback} callback The callback that handles the response.
         * @return {object} Request object
         */
        update: function (params, options, callback) {
            if (typeof options === 'function') {
                callback = options;
                options = {};
            }
            options || (options = {});
            const rootUrl = options.rootUrl || 'https://www.googleapis.com/';
            const parameters = {
                options: Object.assign({
                    url: (rootUrl + '/games/v1configuration/leaderboards/{leaderboardId}').replace(/([^:]\/)\/+/g, '$1'),
                    method: 'PUT'
                }, options),
                params: params,
                requiredParams: ['leaderboardId'],
                pathParams: ['leaderboardId'],
                context: self
            };
            return apirequest_1.default(parameters, callback);
        }
    };
}
module.exports = Gamesconfiguration;
//# sourceMappingURL=v1configuration.js.map