export default class Container {
    constructor(registry, resolver = null) {
        this._registry = registry;
        this._resolver = resolver;
        this._lookups = {};
        this._factoryDefinitionLookups = {};
    }
    factoryFor(specifier) {
        let factoryDefinition = this._factoryDefinitionLookups[specifier];
        if (!factoryDefinition) {
            if (this._resolver) {
                factoryDefinition = this._resolver.retrieve(specifier);
            }
            if (!factoryDefinition) {
                factoryDefinition = this._registry.registration(specifier);
            }
            if (factoryDefinition) {
                this._factoryDefinitionLookups[specifier] = factoryDefinition;
            }
        }
        if (!factoryDefinition) {
            return;
        }
        return this.buildFactory(specifier, factoryDefinition);
    }
    lookup(specifier) {
        let singleton = (this._registry.registeredOption(specifier, 'singleton') !== false);
        if (singleton) {
            let lookup = this._lookups[specifier];
            if (lookup) {
                return lookup.instance;
            }
        }
        let factory = this.factoryFor(specifier);
        if (!factory) {
            return;
        }
        if (this._registry.registeredOption(specifier, 'instantiate') === false) {
            return factory.class;
        }
        let instance = factory.create();
        if (singleton && instance) {
            this._lookups[specifier] = { factory, instance };
        }
        return instance;
    }
    defaultInjections(specifier) {
        return {};
    }
    teardown() {
        let specifiers = Object.keys(this._lookups);
        for (let i = 0; i < specifiers.length; i++) {
            let specifier = specifiers[i];
            let { factory, instance } = this._lookups[specifier];
            factory.teardown(instance);
        }
    }
    defaultTeardown(instance) {
    }
    buildInjections(specifier) {
        let hash = this.defaultInjections(specifier);
        let injections = this._registry.registeredInjections(specifier);
        let injection;
        for (let i = 0; i < injections.length; i++) {
            injection = injections[i];
            hash[injection.property] = this.lookup(injection.source);
        }
        return hash;
    }
    buildFactory(specifier, factoryDefinition) {
        let injections = this.buildInjections(specifier);
        return {
            class: factoryDefinition,
            teardown: (instance) => {
                if (factoryDefinition.teardown) {
                    factoryDefinition.teardown(instance);
                }
                else {
                    this.defaultTeardown(instance);
                }
            },
            create(options) {
                let mergedOptions = Object.assign({}, injections, options);
                return factoryDefinition.create(mergedOptions);
            }
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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