'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _di = require('@glimmer/di');

var _debug = require('./utils/debug');

var _specifiers = require('./utils/specifiers');

class Resolver {
    constructor(config, registry) {
        this.config = config;
        this.registry = registry;
    }
    identify(specifier, referrer) {
        if ((0, _di.isSpecifierStringAbsolute)(specifier)) {
            return specifier;
        }
        let s = (0, _di.deserializeSpecifier)(specifier);
        let result;
        if (referrer) {
            let r = (0, _di.deserializeSpecifier)(referrer);
            if ((0, _di.isSpecifierObjectAbsolute)(r)) {
                (0, _debug.assert)('Specifier must not include a rootName, collection, or namespace when combined with an absolute referrer', s.rootName === undefined && s.collection === undefined && s.namespace === undefined);
                s.rootName = r.rootName;
                s.collection = r.collection;
                let definitiveCollection = this._definitiveCollection(s.type);
                if (!s.name) {
                    /*
                     * For specifiers without a name use the referrer's name and
                     * do not fallback to any other resolution rules.
                     */
                    s.namespace = r.namespace;
                    s.name = r.name;
                    return this._serializeAndVerify(s);
                }
                s.namespace = r.namespace ? r.namespace + '/' + r.name : r.name;
                if ((0, _specifiers.detectLocalResolutionCollection)(s) === definitiveCollection) {
                    /*
                     * For specifiers with a name, try local resolution. Based on
                     * the referrer.
                     */
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Look for a private collection in the referrer's namespace
                if (definitiveCollection) {
                    s.namespace += '/-' + definitiveCollection;
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Because local and private resolution has failed, clear all but `name` and `type`
                // to proceed with top-level resolution
                s.rootName = s.collection = s.namespace = undefined;
            } else {
                (0, _debug.assert)('Referrer must either be "absolute" or include a `type` to determine the associated type', r.type);
                // Look in the definitive collection for the associated type
                s.collection = this._definitiveCollection(r.type);
                if (!s.namespace) {
                    s.namespace = r.rootName;
                }
                (0, _debug.assert)(`'${r.type}' does not have a definitive collection`, s.collection);
            }
        }
        // If the collection is unspecified, use the definitive collection for the `type`
        if (!s.collection) {
            s.collection = this._definitiveCollection(s.type);
            (0, _debug.assert)(`'${s.type}' does not have a definitive collection`, s.collection);
        }
        if (!s.rootName) {
            // If the root name is unspecified, try the app's `rootName` first
            s.rootName = this.config.app.rootName || 'app';
            if (result = this._serializeAndVerify(s)) {
                return result;
            }
            // Then look for an addon with a matching `rootName`
            if (s.namespace) {
                s.rootName = s.namespace;
                s.namespace = undefined;
            } else {
                s.rootName = s.name;
                s.name = 'main';
            }
        }
        if (result = this._serializeAndVerify(s)) {
            return result;
        }
    }
    retrieve(specifier) {
        return this.registry.get(specifier);
    }
    resolve(specifier, referrer) {
        let id = this.identify(specifier, referrer);
        if (id) {
            return this.retrieve(id);
        }
    }
    _definitiveCollection(type) {
        let typeDef = this.config.types[type];
        (0, _debug.assert)(`'${type}' is not a recognized type`, typeDef);
        return typeDef.definitiveCollection;
    }
    _serializeAndVerify(specifier) {
        let serialized = (0, _di.serializeSpecifier)(specifier);
        if (this.registry.has(serialized)) {
            return serialized;
        }
    }
}
exports.default = Resolver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb2x2ZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzcmMvcmVzb2x2ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsQUFBTyxBQUdMLEFBQXlCLEFBQ3pCLEFBQXlCLEFBQ3pCLEFBQW9CLEFBQ3BCLEFBQWtCLEFBQ25CLEFBQU0sQUFBYSxBQUFDOztBQUNyQixBQUFPLEFBQUUsQUFBTSxBQUFFLEFBQU0sQUFBZSxBQUFDOztBQUN2QyxBQUFPLEFBQUUsQUFBK0IsQUFBRSxBQUFNLEFBQW9CLEFBQUMsQUFJckUsQUFBTSxBQUFDLEFBQU87OztBQUlaLGdCQUFZLEFBQTZCLFFBQUUsQUFBd0I7QUFDakUsQUFBSSxhQUFDLEFBQU0sU0FBRyxBQUFNLEFBQUM7QUFDckIsQUFBSSxhQUFDLEFBQVEsV0FBRyxBQUFRLEFBQUMsQUFDM0I7QUFBQztBQUVELEFBQVEsYUFBQyxBQUFpQixXQUFFLEFBQWlCO0FBQzNDLEFBQUUsQUFBQyxZQUFDLEFBQXlCLG1DQUFDLEFBQVMsQUFBQyxBQUFDLFlBQUMsQUFBQztBQUN6QyxBQUFNLG1CQUFDLEFBQVMsQUFBQyxBQUNuQjtBQUFDO0FBRUQsWUFBSSxBQUFDLElBQUcsQUFBb0IsOEJBQUMsQUFBUyxBQUFDLEFBQUM7QUFDeEMsWUFBSSxBQUFjLEFBQUM7QUFFbkIsQUFBRSxBQUFDLFlBQUMsQUFBUSxBQUFDLFVBQUMsQUFBQztBQUNiLGdCQUFJLEFBQUMsSUFBRyxBQUFvQiw4QkFBQyxBQUFRLEFBQUMsQUFBQztBQUV2QyxBQUFFLEFBQUMsZ0JBQUMsQUFBeUIsbUNBQUMsQUFBQyxBQUFDLEFBQUMsSUFBQyxBQUFDO0FBQ2pDLEFBQU0sbUNBQUMsQUFBeUcsMkdBQUUsQUFBQyxFQUFDLEFBQVEsYUFBSyxBQUFTLGFBQUksQUFBQyxFQUFDLEFBQVUsZUFBSyxBQUFTLGFBQUksQUFBQyxFQUFDLEFBQVMsY0FBSyxBQUFTLEFBQUMsQUFBQztBQUV2TSxBQUFDLGtCQUFDLEFBQVEsV0FBRyxBQUFDLEVBQUMsQUFBUSxBQUFDO0FBQ3hCLEFBQUMsa0JBQUMsQUFBVSxhQUFHLEFBQUMsRUFBQyxBQUFVLEFBQUM7QUFDNUIsb0JBQUksQUFBb0IsdUJBQUcsQUFBSSxLQUFDLEFBQXFCLHNCQUFDLEFBQUMsRUFBQyxBQUFJLEFBQUMsQUFBQztBQUU5RCxBQUFFLEFBQUMsb0JBQUMsQ0FBQyxBQUFDLEVBQUMsQUFBSSxBQUFDLE1BQUMsQUFBQztBQUNaLEFBR0c7Ozs7QUFDSCxBQUFDLHNCQUFDLEFBQVMsWUFBRyxBQUFDLEVBQUMsQUFBUyxBQUFDO0FBQzFCLEFBQUMsc0JBQUMsQUFBSSxPQUFHLEFBQUMsRUFBQyxBQUFJLEFBQUM7QUFDaEIsQUFBTSwyQkFBQyxBQUFJLEtBQUMsQUFBbUIsb0JBQUMsQUFBQyxBQUFDLEFBQUMsQUFDckM7QUFBQztBQUVELEFBQUMsa0JBQUMsQUFBUyxZQUFHLEFBQUMsRUFBQyxBQUFTLFlBQUcsQUFBQyxFQUFDLEFBQVMsWUFBRyxBQUFHLE1BQUcsQUFBQyxFQUFDLEFBQUksT0FBRyxBQUFDLEVBQUMsQUFBSSxBQUFDO0FBQ2hFLEFBQUUsQUFBQyxvQkFBQyxBQUErQixpREFBQyxBQUFDLEFBQUMsT0FBSyxBQUFvQixBQUFDLHNCQUFDLEFBQUM7QUFDaEUsQUFHRzs7OztBQUNILEFBQUUsQUFBQyx3QkFBQyxBQUFNLFNBQUcsQUFBSSxLQUFDLEFBQW1CLG9CQUFDLEFBQUMsQUFBQyxBQUFDLElBQUMsQUFBQztBQUFDLEFBQU0sK0JBQUMsQUFBTSxBQUFDLEFBQUM7QUFBQyxBQUM5RDtBQUFDO0FBRUQsQUFBNEQ7QUFDNUQsQUFBRSxBQUFDLG9CQUFDLEFBQW9CLEFBQUMsc0JBQUMsQUFBQztBQUN6QixBQUFDLHNCQUFDLEFBQVMsYUFBSSxBQUFJLE9BQUcsQUFBb0IsQUFBQztBQUMzQyxBQUFFLEFBQUMsd0JBQUMsQUFBTSxTQUFHLEFBQUksS0FBQyxBQUFtQixvQkFBQyxBQUFDLEFBQUMsQUFBQyxJQUFDLEFBQUM7QUFBQyxBQUFNLCtCQUFDLEFBQU0sQUFBQyxBQUFDO0FBQUMsQUFDOUQ7QUFBQztBQUVELEFBQW1GO0FBQ25GLEFBQXVDO0FBQ3ZDLEFBQUMsa0JBQUMsQUFBUSxXQUFHLEFBQUMsRUFBQyxBQUFVLGFBQUcsQUFBQyxFQUFDLEFBQVMsWUFBRyxBQUFTLEFBQUMsQUFDdEQ7QUFBQyxBQUFDLEFBQUksbUJBQUMsQUFBQztBQUNOLEFBQU0sbUNBQUMsQUFBeUYsMkZBQUUsQUFBQyxFQUFDLEFBQUksQUFBQyxBQUFDO0FBRTFHLEFBQTREO0FBQzVELEFBQUMsa0JBQUMsQUFBVSxhQUFHLEFBQUksS0FBQyxBQUFxQixzQkFBQyxBQUFDLEVBQUMsQUFBSSxBQUFDLEFBQUM7QUFDbEQsQUFBRSxBQUFDLG9CQUFDLENBQUMsQUFBQyxFQUFDLEFBQVMsQUFBQyxXQUFDLEFBQUM7QUFDakIsQUFBQyxzQkFBQyxBQUFTLFlBQUcsQUFBQyxFQUFDLEFBQVEsQUFBQyxBQUMzQjtBQUFDO0FBQ0QsQUFBTSxBQUFDLHVDQUFJLEFBQUMsRUFBQyxBQUFJLElBQXlDLDJDQUFFLEFBQUMsRUFBQyxBQUFVLEFBQUMsQUFBQyxBQUM1RTtBQUFDLEFBQ0g7QUFBQztBQUVELEFBQWlGO0FBQ2pGLEFBQUUsQUFBQyxZQUFDLENBQUMsQUFBQyxFQUFDLEFBQVUsQUFBQyxZQUFDLEFBQUM7QUFDbEIsQUFBQyxjQUFDLEFBQVUsYUFBRyxBQUFJLEtBQUMsQUFBcUIsc0JBQUMsQUFBQyxFQUFDLEFBQUksQUFBQyxBQUFDO0FBQ2xELEFBQU0sQUFBQyxtQ0FBSSxBQUFDLEVBQUMsQUFBSSxJQUF5QywyQ0FBRSxBQUFDLEVBQUMsQUFBVSxBQUFDLEFBQUMsQUFDNUU7QUFBQztBQUVELEFBQUUsQUFBQyxZQUFDLENBQUMsQUFBQyxFQUFDLEFBQVEsQUFBQyxVQUFDLEFBQUM7QUFDaEIsQUFBa0U7QUFDbEUsQUFBQyxjQUFDLEFBQVEsV0FBRyxBQUFJLEtBQUMsQUFBTSxPQUFDLEFBQUcsSUFBQyxBQUFRLFlBQUksQUFBSyxBQUFDO0FBQy9DLEFBQUUsQUFBQyxnQkFBQyxBQUFNLFNBQUcsQUFBSSxLQUFDLEFBQW1CLG9CQUFDLEFBQUMsQUFBQyxBQUFDLElBQUMsQUFBQztBQUFDLEFBQU0sdUJBQUMsQUFBTSxBQUFDLEFBQUM7QUFBQztBQUU1RCxBQUFvRDtBQUNwRCxBQUFFLEFBQUMsZ0JBQUMsQUFBQyxFQUFDLEFBQVMsQUFBQyxXQUFDLEFBQUM7QUFDaEIsQUFBQyxrQkFBQyxBQUFRLFdBQUcsQUFBQyxFQUFDLEFBQVMsQUFBQztBQUN6QixBQUFDLGtCQUFDLEFBQVMsWUFBRyxBQUFTLEFBQUMsQUFFMUI7QUFBQyxBQUFDLEFBQUksbUJBQUMsQUFBQztBQUNOLEFBQUMsa0JBQUMsQUFBUSxXQUFHLEFBQUMsRUFBQyxBQUFJLEFBQUM7QUFDcEIsQUFBQyxrQkFBQyxBQUFJLE9BQUcsQUFBTSxBQUFDLEFBQ2xCO0FBQUMsQUFDSDtBQUFDO0FBRUQsQUFBRSxBQUFDLFlBQUMsQUFBTSxTQUFHLEFBQUksS0FBQyxBQUFtQixvQkFBQyxBQUFDLEFBQUMsQUFBQyxJQUFDLEFBQUM7QUFBQyxBQUFNLG1CQUFDLEFBQU0sQUFBQyxBQUFDO0FBQUMsQUFDOUQ7QUFBQztBQUVELEFBQVEsYUFBQyxBQUFpQjtBQUN4QixBQUFNLGVBQUMsQUFBSSxLQUFDLEFBQVEsU0FBQyxBQUFHLElBQUMsQUFBUyxBQUFDLEFBQUMsQUFDdEM7QUFBQztBQUVELEFBQU8sWUFBQyxBQUFpQixXQUFFLEFBQWlCO0FBQzFDLFlBQUksQUFBRSxLQUFHLEFBQUksS0FBQyxBQUFRLFNBQUMsQUFBUyxXQUFFLEFBQVEsQUFBQyxBQUFDO0FBQzVDLEFBQUUsQUFBQyxZQUFDLEFBQUUsQUFBQyxJQUFDLEFBQUM7QUFDUCxBQUFNLG1CQUFDLEFBQUksS0FBQyxBQUFRLFNBQUMsQUFBRSxBQUFDLEFBQUMsQUFDM0I7QUFBQyxBQUNIO0FBQUM7QUFFTyxBQUFxQiwwQkFBQyxBQUFZO0FBQ3hDLFlBQUksQUFBTyxVQUFHLEFBQUksS0FBQyxBQUFNLE9BQUMsQUFBSyxNQUFDLEFBQUksQUFBQyxBQUFDO0FBQ3RDLEFBQU0sQUFBQywrQkFBSSxBQUFJLElBQTRCLDhCQUFFLEFBQU8sQUFBQyxBQUFDO0FBQ3RELEFBQU0sZUFBQyxBQUFPLFFBQUMsQUFBb0IsQUFBQyxBQUN0QztBQUFDO0FBRU8sQUFBbUIsd0JBQUMsQUFBb0I7QUFDOUMsWUFBSSxBQUFVLGFBQUcsQUFBa0IsNEJBQUMsQUFBUyxBQUFDLEFBQUM7QUFDL0MsQUFBRSxBQUFDLFlBQUMsQUFBSSxLQUFDLEFBQVEsU0FBQyxBQUFHLElBQUMsQUFBVSxBQUFDLEFBQUMsYUFBQyxBQUFDO0FBQ2xDLEFBQU0sbUJBQUMsQUFBVSxBQUFDLEFBQ3BCO0FBQUMsQUFDSDtBQUFDLEFBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1xuICBSZXNvbHZlciBhcyBJUmVzb2x2ZXIsXG4gIFNwZWNpZmllcixcbiAgaXNTcGVjaWZpZXJTdHJpbmdBYnNvbHV0ZSxcbiAgaXNTcGVjaWZpZXJPYmplY3RBYnNvbHV0ZSxcbiAgZGVzZXJpYWxpemVTcGVjaWZpZXIsXG4gIHNlcmlhbGl6ZVNwZWNpZmllclxufSBmcm9tICdAZ2xpbW1lci9kaSc7XG5pbXBvcnQgeyBhc3NlcnQgfSBmcm9tICcuL3V0aWxzL2RlYnVnJztcbmltcG9ydCB7IGRldGVjdExvY2FsUmVzb2x1dGlvbkNvbGxlY3Rpb24gfSBmcm9tICcuL3V0aWxzL3NwZWNpZmllcnMnO1xuaW1wb3J0IHsgTW9kdWxlUmVnaXN0cnkgfSBmcm9tICcuL21vZHVsZS1yZWdpc3RyeSc7XG5pbXBvcnQgeyBSZXNvbHZlckNvbmZpZ3VyYXRpb24gfSBmcm9tICcuL3Jlc29sdmVyLWNvbmZpZ3VyYXRpb24nO1xuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBSZXNvbHZlciBpbXBsZW1lbnRzIElSZXNvbHZlciB7XG4gIHB1YmxpYyBjb25maWc6IFJlc29sdmVyQ29uZmlndXJhdGlvbjtcbiAgcHVibGljIHJlZ2lzdHJ5OiBNb2R1bGVSZWdpc3RyeTtcblxuICBjb25zdHJ1Y3Rvcihjb25maWc6IFJlc29sdmVyQ29uZmlndXJhdGlvbiwgcmVnaXN0cnk6IE1vZHVsZVJlZ2lzdHJ5KSB7XG4gICAgdGhpcy5jb25maWcgPSBjb25maWc7XG4gICAgdGhpcy5yZWdpc3RyeSA9IHJlZ2lzdHJ5O1xuICB9XG5cbiAgaWRlbnRpZnkoc3BlY2lmaWVyOiBzdHJpbmcsIHJlZmVycmVyPzogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBpZiAoaXNTcGVjaWZpZXJTdHJpbmdBYnNvbHV0ZShzcGVjaWZpZXIpKSB7XG4gICAgICByZXR1cm4gc3BlY2lmaWVyO1xuICAgIH1cblxuICAgIGxldCBzID0gZGVzZXJpYWxpemVTcGVjaWZpZXIoc3BlY2lmaWVyKTtcbiAgICBsZXQgcmVzdWx0OiBzdHJpbmc7XG5cbiAgICBpZiAocmVmZXJyZXIpIHtcbiAgICAgIGxldCByID0gZGVzZXJpYWxpemVTcGVjaWZpZXIocmVmZXJyZXIpO1xuXG4gICAgICBpZiAoaXNTcGVjaWZpZXJPYmplY3RBYnNvbHV0ZShyKSkge1xuICAgICAgICBhc3NlcnQoJ1NwZWNpZmllciBtdXN0IG5vdCBpbmNsdWRlIGEgcm9vdE5hbWUsIGNvbGxlY3Rpb24sIG9yIG5hbWVzcGFjZSB3aGVuIGNvbWJpbmVkIHdpdGggYW4gYWJzb2x1dGUgcmVmZXJyZXInLCBzLnJvb3ROYW1lID09PSB1bmRlZmluZWQgJiYgcy5jb2xsZWN0aW9uID09PSB1bmRlZmluZWQgJiYgcy5uYW1lc3BhY2UgPT09IHVuZGVmaW5lZCk7XG5cbiAgICAgICAgcy5yb290TmFtZSA9IHIucm9vdE5hbWU7XG4gICAgICAgIHMuY29sbGVjdGlvbiA9IHIuY29sbGVjdGlvbjtcbiAgICAgICAgbGV0IGRlZmluaXRpdmVDb2xsZWN0aW9uID0gdGhpcy5fZGVmaW5pdGl2ZUNvbGxlY3Rpb24ocy50eXBlKTtcblxuICAgICAgICBpZiAoIXMubmFtZSkge1xuICAgICAgICAgIC8qXG4gICAgICAgICAgICogRm9yIHNwZWNpZmllcnMgd2l0aG91dCBhIG5hbWUgdXNlIHRoZSByZWZlcnJlcidzIG5hbWUgYW5kXG4gICAgICAgICAgICogZG8gbm90IGZhbGxiYWNrIHRvIGFueSBvdGhlciByZXNvbHV0aW9uIHJ1bGVzLlxuICAgICAgICAgICAqL1xuICAgICAgICAgIHMubmFtZXNwYWNlID0gci5uYW1lc3BhY2U7XG4gICAgICAgICAgcy5uYW1lID0gci5uYW1lO1xuICAgICAgICAgIHJldHVybiB0aGlzLl9zZXJpYWxpemVBbmRWZXJpZnkocyk7XG4gICAgICAgIH1cblxuICAgICAgICBzLm5hbWVzcGFjZSA9IHIubmFtZXNwYWNlID8gci5uYW1lc3BhY2UgKyAnLycgKyByLm5hbWUgOiByLm5hbWU7XG4gICAgICAgIGlmIChkZXRlY3RMb2NhbFJlc29sdXRpb25Db2xsZWN0aW9uKHMpID09PSBkZWZpbml0aXZlQ29sbGVjdGlvbikge1xuICAgICAgICAgIC8qXG4gICAgICAgICAgICogRm9yIHNwZWNpZmllcnMgd2l0aCBhIG5hbWUsIHRyeSBsb2NhbCByZXNvbHV0aW9uLiBCYXNlZCBvblxuICAgICAgICAgICAqIHRoZSByZWZlcnJlci5cbiAgICAgICAgICAgKi9cbiAgICAgICAgICBpZiAocmVzdWx0ID0gdGhpcy5fc2VyaWFsaXplQW5kVmVyaWZ5KHMpKSB7IHJldHVybiByZXN1bHQ7IH1cbiAgICAgICAgfVxuXG4gICAgICAgIC8vIExvb2sgZm9yIGEgcHJpdmF0ZSBjb2xsZWN0aW9uIGluIHRoZSByZWZlcnJlcidzIG5hbWVzcGFjZVxuICAgICAgICBpZiAoZGVmaW5pdGl2ZUNvbGxlY3Rpb24pIHtcbiAgICAgICAgICBzLm5hbWVzcGFjZSArPSAnLy0nICsgZGVmaW5pdGl2ZUNvbGxlY3Rpb247XG4gICAgICAgICAgaWYgKHJlc3VsdCA9IHRoaXMuX3NlcmlhbGl6ZUFuZFZlcmlmeShzKSkgeyByZXR1cm4gcmVzdWx0OyB9XG4gICAgICAgIH1cblxuICAgICAgICAvLyBCZWNhdXNlIGxvY2FsIGFuZCBwcml2YXRlIHJlc29sdXRpb24gaGFzIGZhaWxlZCwgY2xlYXIgYWxsIGJ1dCBgbmFtZWAgYW5kIGB0eXBlYFxuICAgICAgICAvLyB0byBwcm9jZWVkIHdpdGggdG9wLWxldmVsIHJlc29sdXRpb25cbiAgICAgICAgcy5yb290TmFtZSA9IHMuY29sbGVjdGlvbiA9IHMubmFtZXNwYWNlID0gdW5kZWZpbmVkO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgYXNzZXJ0KCdSZWZlcnJlciBtdXN0IGVpdGhlciBiZSBcImFic29sdXRlXCIgb3IgaW5jbHVkZSBhIGB0eXBlYCB0byBkZXRlcm1pbmUgdGhlIGFzc29jaWF0ZWQgdHlwZScsIHIudHlwZSk7XG5cbiAgICAgICAgLy8gTG9vayBpbiB0aGUgZGVmaW5pdGl2ZSBjb2xsZWN0aW9uIGZvciB0aGUgYXNzb2NpYXRlZCB0eXBlXG4gICAgICAgIHMuY29sbGVjdGlvbiA9IHRoaXMuX2RlZmluaXRpdmVDb2xsZWN0aW9uKHIudHlwZSk7XG4gICAgICAgIGlmICghcy5uYW1lc3BhY2UpIHtcbiAgICAgICAgICBzLm5hbWVzcGFjZSA9IHIucm9vdE5hbWU7XG4gICAgICAgIH1cbiAgICAgICAgYXNzZXJ0KGAnJHtyLnR5cGV9JyBkb2VzIG5vdCBoYXZlIGEgZGVmaW5pdGl2ZSBjb2xsZWN0aW9uYCwgcy5jb2xsZWN0aW9uKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBJZiB0aGUgY29sbGVjdGlvbiBpcyB1bnNwZWNpZmllZCwgdXNlIHRoZSBkZWZpbml0aXZlIGNvbGxlY3Rpb24gZm9yIHRoZSBgdHlwZWBcbiAgICBpZiAoIXMuY29sbGVjdGlvbikge1xuICAgICAgcy5jb2xsZWN0aW9uID0gdGhpcy5fZGVmaW5pdGl2ZUNvbGxlY3Rpb24ocy50eXBlKTtcbiAgICAgIGFzc2VydChgJyR7cy50eXBlfScgZG9lcyBub3QgaGF2ZSBhIGRlZmluaXRpdmUgY29sbGVjdGlvbmAsIHMuY29sbGVjdGlvbik7XG4gICAgfVxuXG4gICAgaWYgKCFzLnJvb3ROYW1lKSB7XG4gICAgICAvLyBJZiB0aGUgcm9vdCBuYW1lIGlzIHVuc3BlY2lmaWVkLCB0cnkgdGhlIGFwcCdzIGByb290TmFtZWAgZmlyc3RcbiAgICAgIHMucm9vdE5hbWUgPSB0aGlzLmNvbmZpZy5hcHAucm9vdE5hbWUgfHwgJ2FwcCc7XG4gICAgICBpZiAocmVzdWx0ID0gdGhpcy5fc2VyaWFsaXplQW5kVmVyaWZ5KHMpKSB7IHJldHVybiByZXN1bHQ7IH1cblxuICAgICAgLy8gVGhlbiBsb29rIGZvciBhbiBhZGRvbiB3aXRoIGEgbWF0Y2hpbmcgYHJvb3ROYW1lYFxuICAgICAgaWYgKHMubmFtZXNwYWNlKSB7XG4gICAgICAgIHMucm9vdE5hbWUgPSBzLm5hbWVzcGFjZTtcbiAgICAgICAgcy5uYW1lc3BhY2UgPSB1bmRlZmluZWQ7XG5cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHMucm9vdE5hbWUgPSBzLm5hbWU7XG4gICAgICAgIHMubmFtZSA9ICdtYWluJztcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAocmVzdWx0ID0gdGhpcy5fc2VyaWFsaXplQW5kVmVyaWZ5KHMpKSB7IHJldHVybiByZXN1bHQ7IH1cbiAgfVxuXG4gIHJldHJpZXZlKHNwZWNpZmllcjogc3RyaW5nKTogYW55IHtcbiAgICByZXR1cm4gdGhpcy5yZWdpc3RyeS5nZXQoc3BlY2lmaWVyKTtcbiAgfVxuXG4gIHJlc29sdmUoc3BlY2lmaWVyOiBzdHJpbmcsIHJlZmVycmVyPzogc3RyaW5nKTogYW55IHtcbiAgICBsZXQgaWQgPSB0aGlzLmlkZW50aWZ5KHNwZWNpZmllciwgcmVmZXJyZXIpO1xuICAgIGlmIChpZCkge1xuICAgICAgcmV0dXJuIHRoaXMucmV0cmlldmUoaWQpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgX2RlZmluaXRpdmVDb2xsZWN0aW9uKHR5cGU6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgbGV0IHR5cGVEZWYgPSB0aGlzLmNvbmZpZy50eXBlc1t0eXBlXTtcbiAgICBhc3NlcnQoYCcke3R5cGV9JyBpcyBub3QgYSByZWNvZ25pemVkIHR5cGVgLCB0eXBlRGVmKTtcbiAgICByZXR1cm4gdHlwZURlZi5kZWZpbml0aXZlQ29sbGVjdGlvbjtcbiAgfVxuXG4gIHByaXZhdGUgX3NlcmlhbGl6ZUFuZFZlcmlmeShzcGVjaWZpZXI6IFNwZWNpZmllcik6IHN0cmluZyB7XG4gICAgbGV0IHNlcmlhbGl6ZWQgPSBzZXJpYWxpemVTcGVjaWZpZXIoc3BlY2lmaWVyKTtcbiAgICBpZiAodGhpcy5yZWdpc3RyeS5oYXMoc2VyaWFsaXplZCkpIHtcbiAgICAgIHJldHVybiBzZXJpYWxpemVkO1xuICAgIH1cbiAgfVxufVxuIl19