import { isSpecifierStringAbsolute, isSpecifierObjectAbsolute, deserializeSpecifier, serializeSpecifier } from '@glimmer/di';
import { assert } from './utils/debug';
import { detectLocalResolutionCollection } from './utils/specifiers';
export default class Resolver {
    constructor(config, registry) {
        this.config = config;
        this.registry = registry;
    }
    identify(specifier, referrer) {
        if (isSpecifierStringAbsolute(specifier)) {
            return specifier;
        }
        let s = deserializeSpecifier(specifier);
        let result;
        if (referrer) {
            let r = deserializeSpecifier(referrer);
            if (isSpecifierObjectAbsolute(r)) {
                assert('Specifier must not include a rootName, collection, or namespace when combined with an absolute referrer', s.rootName === undefined && s.collection === undefined && s.namespace === undefined);
                s.rootName = r.rootName;
                s.collection = r.collection;
                let definitiveCollection = this._definitiveCollection(s.type);
                if (!s.name) {
                    /*
                     * For specifiers without a name use the referrer's name and
                     * do not fallback to any other resolution rules.
                     */
                    s.namespace = r.namespace;
                    s.name = r.name;
                    return this._serializeAndVerify(s);
                }
                s.namespace = r.namespace ? r.namespace + '/' + r.name : r.name;
                if (detectLocalResolutionCollection(s) === definitiveCollection) {
                    /*
                     * For specifiers with a name, try local resolution. Based on
                     * the referrer.
                     */
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Look for a private collection in the referrer's namespace
                if (definitiveCollection) {
                    s.namespace += '/-' + definitiveCollection;
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Because local and private resolution has failed, clear all but `name` and `type`
                // to proceed with top-level resolution
                s.rootName = s.collection = s.namespace = undefined;
            }
            else {
                assert('Referrer must either be "absolute" or include a `type` to determine the associated type', r.type);
                // Look in the definitive collection for the associated type
                s.collection = this._definitiveCollection(r.type);
                if (!s.namespace) {
                    s.namespace = r.rootName;
                }
                assert(`'${r.type}' does not have a definitive collection`, s.collection);
            }
        }
        // If the collection is unspecified, use the definitive collection for the `type`
        if (!s.collection) {
            s.collection = this._definitiveCollection(s.type);
            assert(`'${s.type}' does not have a definitive collection`, s.collection);
        }
        if (!s.rootName) {
            // If the root name is unspecified, try the app's `rootName` first
            s.rootName = this.config.app.rootName || 'app';
            if (result = this._serializeAndVerify(s)) {
                return result;
            }
            // Then look for an addon with a matching `rootName`
            if (s.namespace) {
                s.rootName = s.namespace;
                s.namespace = undefined;
            }
            else {
                s.rootName = s.name;
                s.name = 'main';
            }
        }
        if (result = this._serializeAndVerify(s)) {
            return result;
        }
    }
    retrieve(specifier) {
        return this.registry.get(specifier);
    }
    resolve(specifier, referrer) {
        let id = this.identify(specifier, referrer);
        if (id) {
            return this.retrieve(id);
        }
    }
    _definitiveCollection(type) {
        let typeDef = this.config.types[type];
        assert(`'${type}' is not a recognized type`, typeDef);
        return typeDef.definitiveCollection;
    }
    _serializeAndVerify(specifier) {
        let serialized = serializeSpecifier(specifier);
        if (this.registry.has(serialized)) {
            return serialized;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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