function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

import { isSpecifierStringAbsolute, isSpecifierObjectAbsolute, deserializeSpecifier, serializeSpecifier } from '@glimmer/di';
import { assert } from './utils/debug';
import { detectLocalResolutionCollection } from './utils/specifiers';

var Resolver = function () {
    function Resolver(config, registry) {
        _classCallCheck(this, Resolver);

        this.config = config;
        this.registry = registry;
    }

    Resolver.prototype.identify = function identify(specifier, referrer) {
        if (isSpecifierStringAbsolute(specifier)) {
            return specifier;
        }
        var s = deserializeSpecifier(specifier);
        var result = void 0;
        if (referrer) {
            var r = deserializeSpecifier(referrer);
            if (isSpecifierObjectAbsolute(r)) {
                assert('Specifier must not include a rootName, collection, or namespace when combined with an absolute referrer', s.rootName === undefined && s.collection === undefined && s.namespace === undefined);
                s.rootName = r.rootName;
                s.collection = r.collection;
                var definitiveCollection = this._definitiveCollection(s.type);
                if (!s.name) {
                    /*
                     * For specifiers without a name use the referrer's name and
                     * do not fallback to any other resolution rules.
                     */
                    s.namespace = r.namespace;
                    s.name = r.name;
                    return this._serializeAndVerify(s);
                }
                s.namespace = r.namespace ? r.namespace + '/' + r.name : r.name;
                if (detectLocalResolutionCollection(s) === definitiveCollection) {
                    /*
                     * For specifiers with a name, try local resolution. Based on
                     * the referrer.
                     */
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Look for a private collection in the referrer's namespace
                if (definitiveCollection) {
                    s.namespace += '/-' + definitiveCollection;
                    if (result = this._serializeAndVerify(s)) {
                        return result;
                    }
                }
                // Because local and private resolution has failed, clear all but `name` and `type`
                // to proceed with top-level resolution
                s.rootName = s.collection = s.namespace = undefined;
            } else {
                assert('Referrer must either be "absolute" or include a `type` to determine the associated type', r.type);
                // Look in the definitive collection for the associated type
                s.collection = this._definitiveCollection(r.type);
                if (!s.namespace) {
                    s.namespace = r.rootName;
                }
                assert('\'' + r.type + '\' does not have a definitive collection', s.collection);
            }
        }
        // If the collection is unspecified, use the definitive collection for the `type`
        if (!s.collection) {
            s.collection = this._definitiveCollection(s.type);
            assert('\'' + s.type + '\' does not have a definitive collection', s.collection);
        }
        if (!s.rootName) {
            // If the root name is unspecified, try the app's `rootName` first
            s.rootName = this.config.app.rootName || 'app';
            if (result = this._serializeAndVerify(s)) {
                return result;
            }
            // Then look for an addon with a matching `rootName`
            if (s.namespace) {
                s.rootName = s.namespace;
                s.namespace = undefined;
            } else {
                s.rootName = s.name;
                s.name = 'main';
            }
        }
        if (result = this._serializeAndVerify(s)) {
            return result;
        }
    };

    Resolver.prototype.retrieve = function retrieve(specifier) {
        return this.registry.get(specifier);
    };

    Resolver.prototype.resolve = function resolve(specifier, referrer) {
        var id = this.identify(specifier, referrer);
        if (id) {
            return this.retrieve(id);
        }
    };

    Resolver.prototype._definitiveCollection = function _definitiveCollection(type) {
        var typeDef = this.config.types[type];
        assert('\'' + type + '\' is not a recognized type', typeDef);
        return typeDef.definitiveCollection;
    };

    Resolver.prototype._serializeAndVerify = function _serializeAndVerify(specifier) {
        var serialized = serializeSpecifier(specifier);
        if (this.registry.has(serialized)) {
            return serialized;
        }
    };

    return Resolver;
}();

export default Resolver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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