"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
class DebuggingProtocolFactory {
    create() {
        return new DebuggingProtocol();
    }
}
exports.default = DebuggingProtocolFactory;
class DebuggingProtocol extends events_1.EventEmitter {
    constructor() {
        super();
        this.seq = 0;
        this.pendingRequests = new Map();
    }
    onMessage(data) {
        try {
            let msg = JSON.parse(data);
            if (msg.id !== undefined) {
                let req = this.pendingRequests.get(msg.id);
                this.pendingRequests.delete(msg.id);
                req.resolve(msg);
            }
            else {
                this.emit(msg.method, msg.params);
            }
        }
        catch (err) {
            this.onError(err);
            this.socket.close();
        }
    }
    onClose() {
        this.clearPending(new Error("socket disconnect"));
        this.emit("close");
    }
    onError(err) {
        this.clearPending(err);
        this.socket.close();
        this.emit("error", err);
    }
    clearPending(err) {
        if (this.pendingRequests.size) {
            this.pendingRequests.forEach((req) => {
                req.reject(err);
            });
            this.pendingRequests.clear();
        }
    }
    send(method, params) {
        return new Promise((resolve, reject) => {
            let id = this.seq++;
            this.socket.send(JSON.stringify({ id, method, params }));
            this.pendingRequests.set(id, { id, method, params, resolve, reject });
        }).then((res) => {
            if (res.error) {
                throw new ProtocolError(res.error);
            }
            return res.result;
        });
    }
}
class ProtocolError extends Error {
    constructor(err) {
        super(err.message);
        this.code = err.code;
    }
}
//# sourceMappingURL=debugging-protocol-client-factory.js.map