;(function() {
/*!
 * @overview  Ember - JavaScript Application Framework
 * @copyright Copyright 2011-2017 Tilde Inc. and contributors
 *            Portions Copyright 2006-2011 Strobe Inc.
 *            Portions Copyright 2008-2011 Apple Inc. All rights reserved.
 * @license   Licensed under MIT license
 *            See https://raw.github.com/emberjs/ember.js/master/LICENSE
 * @version   2.11.3
 */

var enifed, requireModule, Ember;
var mainContext = this; // Used in ember-environment/lib/global.js

(function() {
  var isNode = typeof window === 'undefined' &&
    typeof process !== 'undefined' && {}.toString.call(process) === '[object process]';

  if (!isNode) {
    Ember = this.Ember = this.Ember || {};
  }

  if (typeof Ember === 'undefined') { Ember = {}; }

  if (typeof Ember.__loader === 'undefined') {
    var registry = {};
    var seen = {};

    enifed = function(name, deps, callback) {
      var value = { };

      if (!callback) {
        value.deps = [];
        value.callback = deps;
      } else {
        value.deps = deps;
        value.callback = callback;
      }

      registry[name] = value;
    };

    requireModule = function(name) {
      return internalRequire(name, null);
    };

    // setup `require` module
    requireModule['default'] = requireModule;

    requireModule.has = function registryHas(moduleName) {
      return !!registry[moduleName] || !!registry[moduleName + '/index'];
    };

    function missingModule(name, referrerName) {
      if (referrerName) {
        throw new Error('Could not find module ' + name + ' required by: ' + referrerName);
      } else {
        throw new Error('Could not find module ' + name);
      }
    }

    function internalRequire(_name, referrerName) {
      var name = _name;
      var mod = registry[name];

      if (!mod) {
        name = name + '/index';
        mod = registry[name];
      }

      var exports = seen[name];

      if (exports !== undefined) {
        return exports;
      }

      exports = seen[name] = {};

      if (!mod) {
        missingModule(_name, referrerName);
      }

      var deps = mod.deps;
      var callback = mod.callback;
      var reified = new Array(deps.length);

      for (var i = 0; i < deps.length; i++) {
        if (deps[i] === 'exports') {
          reified[i] = exports;
        } else if (deps[i] === 'require') {
          reified[i] = requireModule;
        } else {
          reified[i] = internalRequire(deps[i], name);
        }
      }

      callback.apply(this, reified);

      return exports;
    }

    requireModule._eak_seen = registry;

    Ember.__loader = {
      define: enifed,
      require: requireModule,
      registry: registry
    };
  } else {
    enifed = Ember.__loader.define;
    requireModule = Ember.__loader.require;
  }
})();

function inherits(subClass, superClass) {
  subClass.prototype = Object.create(superClass && superClass.prototype, {
    constructor: {
      value: subClass,
      enumerable: false,
      writable: true,
      configurable: true
    }
  });

  if (superClass) Object.setPrototypeOf ? Object.setPrototypeOf(subClass, superClass) : defaults(subClass, superClass);
}

function taggedTemplateLiteralLoose(strings, raw) {
  strings.raw = raw;
  return strings;
}

function defineProperties(target, props) {
  for (var i = 0; i < props.length; i++) {
    var descriptor = props[i];
    descriptor.enumerable = descriptor.enumerable || false;
    descriptor.configurable = true;
    if ('value' in descriptor) descriptor.writable = true;
    Object.defineProperty(target, descriptor.key, descriptor);
  }
}

function createClass(Constructor, protoProps, staticProps) {
  if (protoProps) defineProperties(Constructor.prototype, protoProps);
  if (staticProps) defineProperties(Constructor, staticProps);
  return Constructor;
}

function interopExportWildcard(obj, defaults) {
  var newObj = defaults({}, obj);
  delete newObj['default'];
  return newObj;
}

function defaults(obj, defaults) {
  var keys = Object.getOwnPropertyNames(defaults);
  for (var i = 0; i < keys.length; i++) {
    var key = keys[i];
    var value = Object.getOwnPropertyDescriptor(defaults, key);
    if (value && value.configurable && obj[key] === undefined) {
      Object.defineProperty(obj, key, value);
    }
  }
  return obj;
}

var babelHelpers = {
  inherits: inherits,
  taggedTemplateLiteralLoose: taggedTemplateLiteralLoose,
  slice: Array.prototype.slice,
  createClass: createClass,
  interopExportWildcard: interopExportWildcard,
  defaults: defaults
};

enifed('backburner', ['exports'], function (exports) { 'use strict';

var NUMBER = /\d+/;

function each(collection, callback) {
  for (var i = 0; i < collection.length; i++) {
    callback(collection[i]);
  }
}

function isString(suspect) {
  return typeof suspect === 'string';
}

function isFunction(suspect) {
  return typeof suspect === 'function';
}

function isNumber(suspect) {
  return typeof suspect === 'number';
}

function isCoercableNumber(number) {
  return isNumber(number) || NUMBER.test(number);
}

function binarySearch(time, timers) {
  var start = 0;
  var end = timers.length - 2;
  var middle, l;

  while (start < end) {
    // since timers is an array of pairs 'l' will always
    // be an integer
    l = (end - start) / 2;

    // compensate for the index in case even number
    // of pairs inside timers
    middle = start + l - (l % 2);

    if (time >= timers[middle]) {
      start = middle + 2;
    } else {
      end = middle;
    }
  }

  return (time >= timers[start]) ? start + 2 : start;
}

function Queue(name, options, globalOptions) {
  this.name = name;
  this.globalOptions = globalOptions || {};
  this.options = options;
  this._queue = [];
  this.targetQueues = {};
  this._queueBeingFlushed = undefined;
}

Queue.prototype = {
  push: function(target, method, args, stack) {
    var queue = this._queue;
    queue.push(target, method, args, stack);

    return {
      queue: this,
      target: target,
      method: method
    };
  },

  pushUniqueWithoutGuid: function(target, method, args, stack) {
    var queue = this._queue;

    for (var i = 0, l = queue.length; i < l; i += 4) {
      var currentTarget = queue[i];
      var currentMethod = queue[i+1];

      if (currentTarget === target && currentMethod === method) {
        queue[i+2] = args;  // replace args
        queue[i+3] = stack; // replace stack
        return;
      }
    }

    queue.push(target, method, args, stack);
  },

  targetQueue: function(targetQueue, target, method, args, stack) {
    var queue = this._queue;

    for (var i = 0, l = targetQueue.length; i < l; i += 2) {
      var currentMethod = targetQueue[i];
      var currentIndex  = targetQueue[i + 1];

      if (currentMethod === method) {
        queue[currentIndex + 2] = args;  // replace args
        queue[currentIndex + 3] = stack; // replace stack
        return;
      }
    }

    targetQueue.push(
      method,
      queue.push(target, method, args, stack) - 4
    );
  },

  pushUniqueWithGuid: function(guid, target, method, args, stack) {
    var hasLocalQueue = this.targetQueues[guid];

    if (hasLocalQueue) {
      this.targetQueue(hasLocalQueue, target, method, args, stack);
    } else {
      this.targetQueues[guid] = [
        method,
        this._queue.push(target, method, args, stack) - 4
      ];
    }

    return {
      queue: this,
      target: target,
      method: method
    };
  },

  pushUnique: function(target, method, args, stack) {
    var KEY = this.globalOptions.GUID_KEY;

    if (target && KEY) {
      var guid = target[KEY];
      if (guid) {
        return this.pushUniqueWithGuid(guid, target, method, args, stack);
      }
    }

    this.pushUniqueWithoutGuid(target, method, args, stack);

    return {
      queue: this,
      target: target,
      method: method
    };
  },

  invoke: function(target, method, args /*, onError, errorRecordedForStack */) {
    if (args && args.length > 0) {
      method.apply(target, args);
    } else {
      method.call(target);
    }
  },

  invokeWithOnError: function(target, method, args, onError, errorRecordedForStack) {
    try {
      if (args && args.length > 0) {
        method.apply(target, args);
      } else {
        method.call(target);
      }
    } catch(error) {
      onError(error, errorRecordedForStack);
    }
  },

  flush: function(sync) {
    var queue = this._queue;
    var length = queue.length;

    if (length === 0) {
      return;
    }

    var globalOptions = this.globalOptions;
    var options = this.options;
    var before = options && options.before;
    var after = options && options.after;
    var onError = globalOptions.onError || (globalOptions.onErrorTarget &&
                                            globalOptions.onErrorTarget[globalOptions.onErrorMethod]);
    var target, method, args, errorRecordedForStack;
    var invoke = onError ? this.invokeWithOnError : this.invoke;

    this.targetQueues = Object.create(null);
    var queueItems = this._queueBeingFlushed = this._queue.slice();
    this._queue = [];

    if (before) {
      before();
    }

    for (var i = 0; i < length; i += 4) {
      target                = queueItems[i];
      method                = queueItems[i+1];
      args                  = queueItems[i+2];
      errorRecordedForStack = queueItems[i+3]; // Debugging assistance

      if (isString(method)) {
        method = target[method];
      }

      // method could have been nullified / canceled during flush
      if (method) {
        //
        //    ** Attention intrepid developer **
        //
        //    To find out the stack of this task when it was scheduled onto
        //    the run loop, add the following to your app.js:
        //
        //    Ember.run.backburner.DEBUG = true; // NOTE: This slows your app, don't leave it on in production.
        //
        //    Once that is in place, when you are at a breakpoint and navigate
        //    here in the stack explorer, you can look at `errorRecordedForStack.stack`,
        //    which will be the captured stack when this job was scheduled.
        //
        //    One possible long-term solution is the following Chrome issue:
        //       https://bugs.chromium.org/p/chromium/issues/detail?id=332624
        //
        invoke(target, method, args, onError, errorRecordedForStack);
      }
    }

    if (after) {
      after();
    }

    this._queueBeingFlushed = undefined;

    if (sync !== false &&
        this._queue.length > 0) {
      // check if new items have been added
      this.flush(true);
    }
  },

  cancel: function(actionToCancel) {
    var queue = this._queue, currentTarget, currentMethod, i, l;
    var target = actionToCancel.target;
    var method = actionToCancel.method;
    var GUID_KEY = this.globalOptions.GUID_KEY;

    if (GUID_KEY && this.targetQueues && target) {
      var targetQueue = this.targetQueues[target[GUID_KEY]];

      if (targetQueue) {
        for (i = 0, l = targetQueue.length; i < l; i++) {
          if (targetQueue[i] === method) {
            targetQueue.splice(i, 1);
          }
        }
      }
    }

    for (i = 0, l = queue.length; i < l; i += 4) {
      currentTarget = queue[i];
      currentMethod = queue[i+1];

      if (currentTarget === target &&
          currentMethod === method) {
        queue.splice(i, 4);
        return true;
      }
    }

    // if not found in current queue
    // could be in the queue that is being flushed
    queue = this._queueBeingFlushed;

    if (!queue) {
      return;
    }

    for (i = 0, l = queue.length; i < l; i += 4) {
      currentTarget = queue[i];
      currentMethod = queue[i+1];

      if (currentTarget === target &&
          currentMethod === method) {
        // don't mess with array during flush
        // just nullify the method
        queue[i+1] = null;
        return true;
      }
    }
  }
};

function DeferredActionQueues(queueNames, options) {
  var queues = this.queues = {};
  this.queueNames = queueNames = queueNames || [];

  this.options = options;

  each(queueNames, function(queueName) {
    queues[queueName] = new Queue(queueName, options[queueName], options);
  });
}

function noSuchQueue(name) {
  throw new Error('You attempted to schedule an action in a queue (' + name + ') that doesn\'t exist');
}

function noSuchMethod(name) {
  throw new Error('You attempted to schedule an action in a queue (' + name + ') for a method that doesn\'t exist');
}

DeferredActionQueues.prototype = {
  schedule: function(name, target, method, args, onceFlag, stack) {
    var queues = this.queues;
    var queue = queues[name];

    if (!queue) {
      noSuchQueue(name);
    }

    if (!method) {
      noSuchMethod(name);
    }

    if (onceFlag) {
      return queue.pushUnique(target, method, args, stack);
    } else {
      return queue.push(target, method, args, stack);
    }
  },

  flush: function() {
    var queues = this.queues;
    var queueNames = this.queueNames;
    var queueName, queue;
    var queueNameIndex = 0;
    var numberOfQueues = queueNames.length;

    while (queueNameIndex < numberOfQueues) {
      queueName = queueNames[queueNameIndex];
      queue = queues[queueName];

      var numberOfQueueItems = queue._queue.length;

      if (numberOfQueueItems === 0) {
        queueNameIndex++;
      } else {
        queue.flush(false /* async */);
        queueNameIndex = 0;
      }
    }
  }
};

function Backburner(queueNames, options) {
  this.queueNames = queueNames;
  this.options = options || {};
  if (!this.options.defaultQueue) {
    this.options.defaultQueue = queueNames[0];
  }
  this.instanceStack = [];
  this._debouncees = [];
  this._throttlers = [];
  this._eventCallbacks = {
    end: [],
    begin: []
  };

  var _this = this;
  this._boundClearItems = function() {
    clearItems();
  };

  this._timerTimeoutId = undefined;
  this._timers = [];

  this._platform = this.options._platform || {
    setTimeout: function (fn, ms) {
      return setTimeout(fn, ms);
    },
    clearTimeout: function (id) {
      clearTimeout(id);
    }
  };

  this._boundRunExpiredTimers = function () {
    _this._runExpiredTimers();
  };
}

Backburner.prototype = {
  begin: function() {
    var options = this.options;
    var onBegin = options && options.onBegin;
    var previousInstance = this.currentInstance;

    if (previousInstance) {
      this.instanceStack.push(previousInstance);
    }

    this.currentInstance = new DeferredActionQueues(this.queueNames, options);
    this._trigger('begin', this.currentInstance, previousInstance);
    if (onBegin) {
      onBegin(this.currentInstance, previousInstance);
    }
  },

  end: function() {
    var options = this.options;
    var onEnd = options && options.onEnd;
    var currentInstance = this.currentInstance;
    var nextInstance = null;

    // Prevent double-finally bug in Safari 6.0.2 and iOS 6
    // This bug appears to be resolved in Safari 6.0.5 and iOS 7
    var finallyAlreadyCalled = false;
    try {
      currentInstance.flush();
    } finally {
      if (!finallyAlreadyCalled) {
        finallyAlreadyCalled = true;

        this.currentInstance = null;

        if (this.instanceStack.length) {
          nextInstance = this.instanceStack.pop();
          this.currentInstance = nextInstance;
        }
        this._trigger('end', currentInstance, nextInstance);
        if (onEnd) {
          onEnd(currentInstance, nextInstance);
        }
      }
    }
  },

  /**
   Trigger an event. Supports up to two arguments. Designed around
   triggering transition events from one run loop instance to the
   next, which requires an argument for the first instance and then
   an argument for the next instance.

   @private
   @method _trigger
   @param {String} eventName
   @param {any} arg1
   @param {any} arg2
   */
  _trigger: function(eventName, arg1, arg2) {
    var callbacks = this._eventCallbacks[eventName];
    if (callbacks) {
      for (var i = 0; i < callbacks.length; i++) {
        callbacks[i](arg1, arg2);
      }
    }
  },

  on: function(eventName, callback) {
    if (typeof callback !== 'function') {
      throw new TypeError('Callback must be a function');
    }
    var callbacks = this._eventCallbacks[eventName];
    if (callbacks) {
      callbacks.push(callback);
    } else {
      throw new TypeError('Cannot on() event "' + eventName + '" because it does not exist');
    }
  },

  off: function(eventName, callback) {
    if (eventName) {
      var callbacks = this._eventCallbacks[eventName];
      var callbackFound = false;
      if (!callbacks) return;
      if (callback) {
        for (var i = 0; i < callbacks.length; i++) {
          if (callbacks[i] === callback) {
            callbackFound = true;
            callbacks.splice(i, 1);
            i--;
          }
        }
      }
      if (!callbackFound) {
        throw new TypeError('Cannot off() callback that does not exist');
      }
    } else {
      throw new TypeError('Cannot off() event "' + eventName + '" because it does not exist');
    }
  },

  run: function(/* target, method, args */) {
    var length = arguments.length;
    var method, target, args;

    if (length === 1) {
      method = arguments[0];
      target = null;
    } else {
      target = arguments[0];
      method = arguments[1];
    }

    if (isString(method)) {
      method = target[method];
    }

    if (length > 2) {
      args = new Array(length - 2);
      for (var i = 0, l = length - 2; i < l; i++) {
        args[i] = arguments[i + 2];
      }
    } else {
      args = [];
    }

    var onError = getOnError(this.options);

    this.begin();

    // guard against Safari 6's double-finally bug
    var didFinally = false;

    if (onError) {
      try {
        return method.apply(target, args);
      } catch(error) {
        onError(error);
      } finally {
        if (!didFinally) {
          didFinally = true;
          this.end();
        }
      }
    } else {
      try {
        return method.apply(target, args);
      } finally {
        if (!didFinally) {
          didFinally = true;
          this.end();
        }
      }
    }
  },

  /*
    Join the passed method with an existing queue and execute immediately,
    if there isn't one use `Backburner#run`.

    The join method is like the run method except that it will schedule into
    an existing queue if one already exists. In either case, the join method will
    immediately execute the passed in function and return its result.

    @method join
    @param {Object} target
    @param {Function} method The method to be executed
    @param {any} args The method arguments
    @return method result
  */
  join: function(/* target, method, args */) {
    if (!this.currentInstance) {
      return this.run.apply(this, arguments);
    }

    var length = arguments.length;
    var method, target;

    if (length === 1) {
      method = arguments[0];
      target = null;
    } else {
      target = arguments[0];
      method = arguments[1];
    }

    if (isString(method)) {
      method = target[method];
    }

    if (length === 1) {
      return method();
    } else if (length === 2) {
      return method.call(target);
    } else {
      var args = new Array(length - 2);
      for (var i = 0, l = length - 2; i < l; i++) {
        args[i] = arguments[i + 2];
      }
      return method.apply(target, args);
    }
  },


  /*
    Defer the passed function to run inside the specified queue.

    @method defer
    @param {String} queueName
    @param {Object} target
    @param {Function|String} method The method or method name to be executed
    @param {any} args The method arguments
    @return method result
  */
  defer: function(queueName /* , target, method, args */) {
    var length = arguments.length;
    var method, target, args;

    if (length === 2) {
      method = arguments[1];
      target = null;
    } else {
      target = arguments[1];
      method = arguments[2];
    }

    if (isString(method)) {
      method = target[method];
    }

    var stack = this.DEBUG ? new Error() : undefined;

    if (length > 3) {
      args = new Array(length - 3);
      for (var i = 3; i < length; i++) {
        args[i-3] = arguments[i];
      }
    } else {
      args = undefined;
    }

    if (!this.currentInstance) { createAutorun(this); }
    return this.currentInstance.schedule(queueName, target, method, args, false, stack);
  },

  deferOnce: function(queueName /* , target, method, args */) {
    var length = arguments.length;
    var method, target, args;

    if (length === 2) {
      method = arguments[1];
      target = null;
    } else {
      target = arguments[1];
      method = arguments[2];
    }

    if (isString(method)) {
      method = target[method];
    }

    var stack = this.DEBUG ? new Error() : undefined;

    if (length > 3) {
      args = new Array(length - 3);
      for (var i = 3; i < length; i++) {
        args[i-3] = arguments[i];
      }
    } else {
      args = undefined;
    }

    if (!this.currentInstance) {
      createAutorun(this);
    }
    return this.currentInstance.schedule(queueName, target, method, args, true, stack);
  },

  setTimeout: function() {
    var l = arguments.length;
    var args = new Array(l);

    for (var x = 0; x < l; x++) {
      args[x] = arguments[x];
    }

    var length = args.length,
        method, wait, target,
        methodOrTarget, methodOrWait, methodOrArgs;

    if (length === 0) {
      return;
    } else if (length === 1) {
      method = args.shift();
      wait = 0;
    } else if (length === 2) {
      methodOrTarget = args[0];
      methodOrWait = args[1];

      if (isFunction(methodOrWait) || isFunction(methodOrTarget[methodOrWait])) {
        target = args.shift();
        method = args.shift();
        wait = 0;
      } else if (isCoercableNumber(methodOrWait)) {
        method = args.shift();
        wait = args.shift();
      } else {
        method = args.shift();
        wait =  0;
      }
    } else {
      var last = args[args.length - 1];

      if (isCoercableNumber(last)) {
        wait = args.pop();
      } else {
        wait = 0;
      }

      methodOrTarget = args[0];
      methodOrArgs = args[1];

      if (isFunction(methodOrArgs) || (isString(methodOrArgs) &&
                                      methodOrTarget !== null &&
                                      methodOrArgs in methodOrTarget)) {
        target = args.shift();
        method = args.shift();
      } else {
        method = args.shift();
      }
    }

    var executeAt = Date.now() + parseInt(wait !== wait ? 0 : wait, 10);

    if (isString(method)) {
      method = target[method];
    }

    var onError = getOnError(this.options);

    function fn() {
      if (onError) {
        try {
          method.apply(target, args);
        } catch (e) {
          onError(e);
        }
      } else {
        method.apply(target, args);
      }
    }

    return this._setTimeout(fn, executeAt);
  },

  _setTimeout: function (fn, executeAt) {
    if (this._timers.length === 0) {
      this._timers.push(executeAt, fn);
      this._installTimerTimeout();
      return fn;
    }

    // find position to insert
    var i = binarySearch(executeAt, this._timers);

    this._timers.splice(i, 0, executeAt, fn);

    // we should be the new earliest timer if i == 0
    if (i === 0) {
      this._reinstallTimerTimeout();
    }

    return fn;
  },

  throttle: function(target, method /* , args, wait, [immediate] */) {
    var backburner = this;
    var args = new Array(arguments.length);
    for (var i = 0; i < arguments.length; i++) {
      args[i] = arguments[i];
    }
    var immediate = args.pop();
    var wait, throttler, index, timer;

    if (isNumber(immediate) || isString(immediate)) {
      wait = immediate;
      immediate = true;
    } else {
      wait = args.pop();
    }

    wait = parseInt(wait, 10);

    index = findThrottler(target, method, this._throttlers);
    if (index > -1) { return this._throttlers[index]; } // throttled

    timer = this._platform.setTimeout(function() {
      if (!immediate) {
        backburner.run.apply(backburner, args);
      }
      var index = findThrottler(target, method, backburner._throttlers);
      if (index > -1) {
        backburner._throttlers.splice(index, 1);
      }
    }, wait);

    if (immediate) {
      this.run.apply(this, args);
    }

    throttler = [target, method, timer];

    this._throttlers.push(throttler);

    return throttler;
  },

  debounce: function(target, method /* , args, wait, [immediate] */) {
    var backburner = this;
    var args = new Array(arguments.length);
    for (var i = 0; i < arguments.length; i++) {
      args[i] = arguments[i];
    }

    var immediate = args.pop();
    var wait, index, debouncee, timer;

    if (isNumber(immediate) || isString(immediate)) {
      wait = immediate;
      immediate = false;
    } else {
      wait = args.pop();
    }

    wait = parseInt(wait, 10);
    // Remove debouncee
    index = findDebouncee(target, method, this._debouncees);

    if (index > -1) {
      debouncee = this._debouncees[index];
      this._debouncees.splice(index, 1);
      this._platform.clearTimeout(debouncee[2]);
    }

    timer = this._platform.setTimeout(function() {
      if (!immediate) {
        backburner.run.apply(backburner, args);
      }
      var index = findDebouncee(target, method, backburner._debouncees);
      if (index > -1) {
        backburner._debouncees.splice(index, 1);
      }
    }, wait);

    if (immediate && index === -1) {
      backburner.run.apply(backburner, args);
    }

    debouncee = [
      target,
      method,
      timer
    ];

    backburner._debouncees.push(debouncee);

    return debouncee;
  },

  cancelTimers: function() {
    each(this._throttlers, this._boundClearItems);
    this._throttlers = [];

    each(this._debouncees, this._boundClearItems);
    this._debouncees = [];

    this._clearTimerTimeout();
    this._timers = [];

    if (this._autorun) {
      this._platform.clearTimeout(this._autorun);
      this._autorun = null;
    }
  },

  hasTimers: function() {
    return !!this._timers.length || !!this._debouncees.length || !!this._throttlers.length || this._autorun;
  },

  cancel: function (timer) {
    var timerType = typeof timer;

    if (timer && timerType === 'object' && timer.queue && timer.method) { // we're cancelling a deferOnce
      return timer.queue.cancel(timer);
    } else if (timerType === 'function') { // we're cancelling a setTimeout
      for (var i = 0, l = this._timers.length; i < l; i += 2) {
        if (this._timers[i + 1] === timer) {
          this._timers.splice(i, 2); // remove the two elements
          if (i === 0) {
            this._reinstallTimerTimeout();
          }
          return true;
        }
      }
    } else if (Object.prototype.toString.call(timer) === '[object Array]'){ // we're cancelling a throttle or debounce
      return this._cancelItem(findThrottler, this._throttlers, timer) ||
               this._cancelItem(findDebouncee, this._debouncees, timer);
    } else {
      return; // timer was null or not a timer
    }
  },

  _cancelItem: function(findMethod, array, timer){
    var item, index;

    if (timer.length < 3) { return false; }

    index = findMethod(timer[0], timer[1], array);

    if (index > -1) {

      item = array[index];

      if (item[2] === timer[2]) {
        array.splice(index, 1);
        this._platform.clearTimeout(timer[2]);
        return true;
      }
    }

    return false;
  },

  _runExpiredTimers: function () {
    this._timerTimeoutId = undefined;
    this.run(this, this._scheduleExpiredTimers);
  },

  _scheduleExpiredTimers: function () {
    var n = Date.now();
    var timers = this._timers;
    var i = 0;
    var l = timers.length;
    for (; i < l; i += 2) {
      var executeAt = timers[i];
      var fn = timers[i+1];
      if (executeAt <= n) {
        this.schedule(this.options.defaultQueue, null, fn);
      } else {
        break;
      }
    }
    timers.splice(0, i);
    this._installTimerTimeout();
  },

  _reinstallTimerTimeout: function () {
    this._clearTimerTimeout();
    this._installTimerTimeout();
  },

  _clearTimerTimeout: function () {
    if (!this._timerTimeoutId) {
      return;
    }
    this._platform.clearTimeout(this._timerTimeoutId);
    this._timerTimeoutId = undefined;
  },

  _installTimerTimeout: function () {
    if (!this._timers.length) {
      return;
    }
    var minExpiresAt = this._timers[0];
    var n = Date.now();
    var wait = Math.max(0, minExpiresAt - n);
    this._timerTimeoutId = this._platform.setTimeout(this._boundRunExpiredTimers, wait);
  }
};

Backburner.prototype.schedule = Backburner.prototype.defer;
Backburner.prototype.scheduleOnce = Backburner.prototype.deferOnce;
Backburner.prototype.later = Backburner.prototype.setTimeout;

function getOnError(options) {
  return options.onError || (options.onErrorTarget && options.onErrorTarget[options.onErrorMethod]);
}

function createAutorun(backburner) {
  var setTimeout = backburner._platform.setTimeout;
  backburner.begin();
  backburner._autorun = setTimeout(function() {
    backburner._autorun = null;
    backburner.end();
  }, 0);
}

function findDebouncee(target, method, debouncees) {
  return findItem(target, method, debouncees);
}

function findThrottler(target, method, throttlers) {
  return findItem(target, method, throttlers);
}

function findItem(target, method, collection) {
  var item;
  var index = -1;

  for (var i = 0, l = collection.length; i < l; i++) {
    item = collection[i];
    if (item[0] === target && item[1] === method) {
      index = i;
      break;
    }
  }

  return index;
}

function clearItems(item) {
  this._platform.clearTimeout(item[2]);
}

exports['default'] = Backburner;

Object.defineProperty(exports, '__esModule', { value: true });

});
enifed('container/container', ['exports', 'ember-utils', 'ember-environment', 'ember-metal'], function (exports, _emberUtils, _emberEnvironment, _emberMetal) {
  'use strict';

  exports.default = Container;
  exports.buildFakeContainerWithDeprecations = buildFakeContainerWithDeprecations;

  var CONTAINER_OVERRIDE = _emberUtils.symbol('CONTAINER_OVERRIDE');

  /**
   A container used to instantiate and cache objects.
  
   Every `Container` must be associated with a `Registry`, which is referenced
   to determine the factory and options that should be used to instantiate
   objects.
  
   The public API for `Container` is still in flux and should not be considered
   stable.
  
   @private
   @class Container
   */

  function Container(registry, options) {
    this.registry = registry;
    this.owner = options && options.owner ? options.owner : null;
    this.cache = _emberUtils.dictionary(options && options.cache ? options.cache : null);
    this.factoryCache = _emberUtils.dictionary(options && options.factoryCache ? options.factoryCache : null);
    this.validationCache = _emberUtils.dictionary(options && options.validationCache ? options.validationCache : null);
    this._fakeContainerToInject = buildFakeContainerWithDeprecations(this);
    this[CONTAINER_OVERRIDE] = undefined;
    this.isDestroyed = false;
  }

  Container.prototype = {
    /**
     @private
     @property owner
     @type Object
     */
    owner: null,

    /**
     @private
     @property registry
     @type Registry
     @since 1.11.0
     */
    registry: null,

    /**
     @private
     @property cache
     @type InheritingDict
     */
    cache: null,

    /**
     @private
     @property factoryCache
     @type InheritingDict
     */
    factoryCache: null,

    /**
     @private
     @property validationCache
     @type InheritingDict
     */
    validationCache: null,

    /**
     Given a fullName return a corresponding instance.
      The default behaviour is for lookup to return a singleton instance.
     The singleton is scoped to the container, allowing multiple containers
     to all have their own locally scoped singletons.
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      registry.register('api:twitter', Twitter);
      let twitter = container.lookup('api:twitter');
      twitter instanceof Twitter; // => true
      // by default the container will return singletons
     let twitter2 = container.lookup('api:twitter');
     twitter2 instanceof Twitter; // => true
      twitter === twitter2; //=> true
     ```
      If singletons are not wanted, an optional flag can be provided at lookup.
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      registry.register('api:twitter', Twitter);
      let twitter = container.lookup('api:twitter', { singleton: false });
     let twitter2 = container.lookup('api:twitter', { singleton: false });
      twitter === twitter2; //=> false
     ```
      @private
     @method lookup
     @param {String} fullName
     @param {Object} [options]
     @param {String} [options.source] The fullname of the request source (used for local lookup)
     @return {any}
     */
    lookup: function (fullName, options) {
      return lookup(this, this.registry.normalize(fullName), options);
    },

    /**
     Given a fullName, return the corresponding factory.
      @private
     @method lookupFactory
     @param {String} fullName
     @param {Object} [options]
     @param {String} [options.source] The fullname of the request source (used for local lookup)
     @return {any}
     */
    lookupFactory: function (fullName, options) {
      return factoryFor(this, this.registry.normalize(fullName), options);
    },

    /**
     A depth first traversal, destroying the container, its descendant containers and all
     their managed objects.
      @private
     @method destroy
     */
    destroy: function () {
      eachDestroyable(this, function (item) {
        if (item.destroy) {
          item.destroy();
        }
      });

      this.isDestroyed = true;
    },

    /**
     Clear either the entire cache or just the cache for a particular key.
      @private
     @method reset
     @param {String} fullName optional key to reset; if missing, resets everything
     */
    reset: function (fullName) {
      if (arguments.length > 0) {
        resetMember(this, this.registry.normalize(fullName));
      } else {
        resetCache(this);
      }
    },

    /**
     Returns an object that can be used to provide an owner to a
     manually created instance.
      @private
     @method ownerInjection
     @returns { Object }
    */
    ownerInjection: function () {
      var _ref;

      return _ref = {}, _ref[_emberUtils.OWNER] = this.owner, _ref;
    }
  };

  function isSingleton(container, fullName) {
    return container.registry.getOption(fullName, 'singleton') !== false;
  }

  function lookup(container, fullName) {
    var options = arguments.length <= 2 || arguments[2] === undefined ? {} : arguments[2];

    if (options.source) {
      fullName = container.registry.expandLocalLookup(fullName, options);

      // if expandLocalLookup returns falsey, we do not support local lookup
      if (!fullName) {
        return;
      }
    }

    if (container.cache[fullName] !== undefined && options.singleton !== false) {
      return container.cache[fullName];
    }

    var value = instantiate(container, fullName);

    if (value === undefined) {
      return;
    }

    if (isSingleton(container, fullName) && options.singleton !== false) {
      container.cache[fullName] = value;
    }

    return value;
  }

  function markInjectionsAsDynamic(injections) {
    injections._dynamic = true;
  }

  function areInjectionsDynamic(injections) {
    return !!injections._dynamic;
  }

  function buildInjections() /* container, ...injections */{
    var hash = {};

    if (arguments.length > 1) {
      var container = arguments[0];
      var injections = [];
      var injection = undefined;

      for (var i = 1; i < arguments.length; i++) {
        if (arguments[i]) {
          injections = injections.concat(arguments[i]);
        }
      }

      container.registry.validateInjections(injections);

      for (var i = 0; i < injections.length; i++) {
        injection = injections[i];
        hash[injection.property] = lookup(container, injection.fullName);
        if (!isSingleton(container, injection.fullName)) {
          markInjectionsAsDynamic(hash);
        }
      }
    }

    return hash;
  }

  function factoryFor(container, fullName) {
    var options = arguments.length <= 2 || arguments[2] === undefined ? {} : arguments[2];

    var registry = container.registry;

    if (options.source) {
      fullName = registry.expandLocalLookup(fullName, options);

      // if expandLocalLookup returns falsey, we do not support local lookup
      if (!fullName) {
        return;
      }
    }

    var cache = container.factoryCache;
    if (cache[fullName]) {
      return cache[fullName];
    }
    var factory = registry.resolve(fullName);
    if (factory === undefined) {
      return;
    }

    var type = fullName.split(':')[0];
    if (!factory || typeof factory.extend !== 'function' || !_emberEnvironment.ENV.MODEL_FACTORY_INJECTIONS && type === 'model') {
      if (factory && typeof factory._onLookup === 'function') {
        factory._onLookup(fullName);
      }

      // TODO: think about a 'safe' merge style extension
      // for now just fallback to create time injection
      cache[fullName] = factory;
      return factory;
    } else {
      var injections = injectionsFor(container, fullName);
      var factoryInjections = factoryInjectionsFor(container, fullName);
      var cacheable = !areInjectionsDynamic(injections) && !areInjectionsDynamic(factoryInjections);

      factoryInjections[_emberUtils.NAME_KEY] = registry.makeToString(factory, fullName);

      var injectedFactory = factory.extend(injections);

      // TODO - remove all `container` injections when Ember reaches v3.0.0
      injectDeprecatedContainer(injectedFactory.prototype, container);
      injectedFactory.reopenClass(factoryInjections);

      if (factory && typeof factory._onLookup === 'function') {
        factory._onLookup(fullName);
      }

      if (cacheable) {
        cache[fullName] = injectedFactory;
      }

      return injectedFactory;
    }
  }

  function injectionsFor(container, fullName) {
    var registry = container.registry;
    var splitName = fullName.split(':');
    var type = splitName[0];

    var injections = buildInjections(container, registry.getTypeInjections(type), registry.getInjections(fullName));
    injections._debugContainerKey = fullName;

    _emberUtils.setOwner(injections, container.owner);

    return injections;
  }

  function factoryInjectionsFor(container, fullName) {
    var registry = container.registry;
    var splitName = fullName.split(':');
    var type = splitName[0];

    var factoryInjections = buildInjections(container, registry.getFactoryTypeInjections(type), registry.getFactoryInjections(fullName));
    factoryInjections._debugContainerKey = fullName;

    return factoryInjections;
  }

  function instantiate(container, fullName) {
    var factory = factoryFor(container, fullName);
    var lazyInjections = undefined,
        validationCache = undefined;

    if (container.registry.getOption(fullName, 'instantiate') === false) {
      return factory;
    }

    if (factory) {
      if (typeof factory.create !== 'function') {
        throw new Error('Failed to create an instance of \'' + fullName + '\'. Most likely an improperly defined class or' + ' an invalid module export.');
      }

      validationCache = container.validationCache;

      validationCache[fullName] = true;

      var obj = undefined;

      if (typeof factory.extend === 'function') {
        // assume the factory was extendable and is already injected
        obj = factory.create();
      } else {
        // assume the factory was extendable
        // to create time injections
        // TODO: support new'ing for instantiation and merge injections for pure JS Functions
        var injections = injectionsFor(container, fullName);

        // Ensure that a container is available to an object during instantiation.
        // TODO - remove when Ember reaches v3.0.0
        // This "fake" container will be replaced after instantiation with a
        // property that raises deprecations every time it is accessed.
        injections.container = container._fakeContainerToInject;
        obj = factory.create(injections);

        // TODO - remove when Ember reaches v3.0.0
        if (!Object.isFrozen(obj) && 'container' in obj) {
          injectDeprecatedContainer(obj, container);
        }
      }

      return obj;
    }
  }

  // TODO - remove when Ember reaches v3.0.0
  function injectDeprecatedContainer(object, container) {
    Object.defineProperty(object, 'container', {
      configurable: true,
      enumerable: false,
      get: function () {
        return this[CONTAINER_OVERRIDE] || container;
      },

      set: function (value) {

        this[CONTAINER_OVERRIDE] = value;

        return value;
      }
    });
  }

  function eachDestroyable(container, callback) {
    var cache = container.cache;
    var keys = Object.keys(cache);

    for (var i = 0; i < keys.length; i++) {
      var key = keys[i];
      var value = cache[key];

      if (container.registry.getOption(key, 'instantiate') !== false) {
        callback(value);
      }
    }
  }

  function resetCache(container) {
    eachDestroyable(container, function (value) {
      if (value.destroy) {
        value.destroy();
      }
    });

    container.cache.dict = _emberUtils.dictionary(null);
  }

  function resetMember(container, fullName) {
    var member = container.cache[fullName];

    delete container.factoryCache[fullName];

    if (member) {
      delete container.cache[fullName];

      if (member.destroy) {
        member.destroy();
      }
    }
  }

  function buildFakeContainerWithDeprecations(container) {
    var fakeContainer = {};
    var propertyMappings = {
      lookup: 'lookup',
      lookupFactory: '_lookupFactory'
    };

    for (var containerProperty in propertyMappings) {
      fakeContainer[containerProperty] = buildFakeContainerFunction(container, containerProperty, propertyMappings[containerProperty]);
    }

    return fakeContainer;
  }

  function buildFakeContainerFunction(container, containerProperty, ownerProperty) {
    return function () {
      return container[containerProperty].apply(container, arguments);
    };
  }
});

// Ensure that all lazy injections are valid at instantiation time
enifed('container/index', ['exports', 'container/registry', 'container/container'], function (exports, _containerRegistry, _containerContainer) {
  /*
  Public API for the container is still in flux.
  The public API, specified on the application namespace should be considered the stable API.
  // @module container
    @private
  */

  'use strict';

  exports.Registry = _containerRegistry.default;
  exports.privatize = _containerRegistry.privatize;
  exports.Container = _containerContainer.default;
  exports.buildFakeContainerWithDeprecations = _containerContainer.buildFakeContainerWithDeprecations;
});
enifed('container/registry', ['exports', 'ember-utils', 'ember-metal', 'container/container'], function (exports, _emberUtils, _emberMetal, _containerContainer) {
  'use strict';

  exports.default = Registry;
  exports.privatize = privatize;

  var VALID_FULL_NAME_REGEXP = /^[^:]+:[^:]+$/;

  /**
   A registry used to store factory and option information keyed
   by type.
  
   A `Registry` stores the factory and option information needed by a
   `Container` to instantiate and cache objects.
  
   The API for `Registry` is still in flux and should not be considered stable.
  
   @private
   @class Registry
   @since 1.11.0
  */

  function Registry(options) {
    this.fallback = options && options.fallback ? options.fallback : null;

    if (options && options.resolver) {
      this.resolver = options.resolver;

      if (typeof this.resolver === 'function') {
        deprecateResolverFunction(this);
      }
    }

    this.registrations = _emberUtils.dictionary(options && options.registrations ? options.registrations : null);

    this._typeInjections = _emberUtils.dictionary(null);
    this._injections = _emberUtils.dictionary(null);
    this._factoryTypeInjections = _emberUtils.dictionary(null);
    this._factoryInjections = _emberUtils.dictionary(null);

    this._localLookupCache = new _emberUtils.EmptyObject();
    this._normalizeCache = _emberUtils.dictionary(null);
    this._resolveCache = _emberUtils.dictionary(null);
    this._failCache = _emberUtils.dictionary(null);

    this._options = _emberUtils.dictionary(null);
    this._typeOptions = _emberUtils.dictionary(null);
  }

  Registry.prototype = {
    /**
     A backup registry for resolving registrations when no matches can be found.
      @private
     @property fallback
     @type Registry
     */
    fallback: null,

    /**
     An object that has a `resolve` method that resolves a name.
      @private
     @property resolver
     @type Resolver
     */
    resolver: null,

    /**
     @private
     @property registrations
     @type InheritingDict
     */
    registrations: null,

    /**
     @private
      @property _typeInjections
     @type InheritingDict
     */
    _typeInjections: null,

    /**
     @private
      @property _injections
     @type InheritingDict
     */
    _injections: null,

    /**
     @private
      @property _factoryTypeInjections
     @type InheritingDict
     */
    _factoryTypeInjections: null,

    /**
     @private
      @property _factoryInjections
     @type InheritingDict
     */
    _factoryInjections: null,

    /**
     @private
      @property _normalizeCache
     @type InheritingDict
     */
    _normalizeCache: null,

    /**
     @private
      @property _resolveCache
     @type InheritingDict
     */
    _resolveCache: null,

    /**
     @private
      @property _options
     @type InheritingDict
     */
    _options: null,

    /**
     @private
      @property _typeOptions
     @type InheritingDict
     */
    _typeOptions: null,

    /**
     Creates a container based on this registry.
      @private
     @method container
     @param {Object} options
     @return {Container} created container
     */
    container: function (options) {
      return new _containerContainer.default(this, options);
    },

    /**
     Registers a factory for later injection.
      Example:
      ```javascript
     let registry = new Registry();
      registry.register('model:user', Person, {singleton: false });
     registry.register('fruit:favorite', Orange);
     registry.register('communication:main', Email, {singleton: false});
     ```
      @private
     @method register
     @param {String} fullName
     @param {Function} factory
     @param {Object} options
     */
    register: function (fullName, factory) {
      var options = arguments.length <= 2 || arguments[2] === undefined ? {} : arguments[2];

      if (factory === undefined) {
        throw new TypeError('Attempting to register an unknown factory: \'' + fullName + '\'');
      }

      var normalizedName = this.normalize(fullName);

      if (this._resolveCache[normalizedName]) {
        throw new Error('Cannot re-register: \'' + fullName + '\', as it has already been resolved.');
      }

      delete this._failCache[normalizedName];
      this.registrations[normalizedName] = factory;
      this._options[normalizedName] = options;
    },

    /**
     Unregister a fullName
      ```javascript
     let registry = new Registry();
     registry.register('model:user', User);
      registry.resolve('model:user').create() instanceof User //=> true
      registry.unregister('model:user')
     registry.resolve('model:user') === undefined //=> true
     ```
      @private
     @method unregister
     @param {String} fullName
     */
    unregister: function (fullName) {

      var normalizedName = this.normalize(fullName);

      this._localLookupCache = new _emberUtils.EmptyObject();

      delete this.registrations[normalizedName];
      delete this._resolveCache[normalizedName];
      delete this._failCache[normalizedName];
      delete this._options[normalizedName];
    },

    /**
     Given a fullName return the corresponding factory.
      By default `resolve` will retrieve the factory from
     the registry.
      ```javascript
     let registry = new Registry();
     registry.register('api:twitter', Twitter);
      registry.resolve('api:twitter') // => Twitter
     ```
      Optionally the registry can be provided with a custom resolver.
     If provided, `resolve` will first provide the custom resolver
     the opportunity to resolve the fullName, otherwise it will fallback
     to the registry.
      ```javascript
     let registry = new Registry();
     registry.resolver = function(fullName) {
        // lookup via the module system of choice
      };
      // the twitter factory is added to the module system
     registry.resolve('api:twitter') // => Twitter
     ```
      @private
     @method resolve
     @param {String} fullName
     @param {Object} [options]
     @param {String} [options.source] the fullname of the request source (used for local lookups)
     @return {Function} fullName's factory
     */
    resolve: function (fullName, options) {
      var factory = resolve(this, this.normalize(fullName), options);
      if (factory === undefined && this.fallback) {
        var _fallback;

        factory = (_fallback = this.fallback).resolve.apply(_fallback, arguments);
      }
      return factory;
    },

    /**
     A hook that can be used to describe how the resolver will
     attempt to find the factory.
      For example, the default Ember `.describe` returns the full
     class name (including namespace) where Ember's resolver expects
     to find the `fullName`.
      @private
     @method describe
     @param {String} fullName
     @return {string} described fullName
     */
    describe: function (fullName) {
      if (this.resolver && this.resolver.lookupDescription) {
        return this.resolver.lookupDescription(fullName);
      } else if (this.fallback) {
        return this.fallback.describe(fullName);
      } else {
        return fullName;
      }
    },

    /**
     A hook to enable custom fullName normalization behaviour
      @private
     @method normalizeFullName
     @param {String} fullName
     @return {string} normalized fullName
     */
    normalizeFullName: function (fullName) {
      if (this.resolver && this.resolver.normalize) {
        return this.resolver.normalize(fullName);
      } else if (this.fallback) {
        return this.fallback.normalizeFullName(fullName);
      } else {
        return fullName;
      }
    },

    /**
     Normalize a fullName based on the application's conventions
      @private
     @method normalize
     @param {String} fullName
     @return {string} normalized fullName
     */
    normalize: function (fullName) {
      return this._normalizeCache[fullName] || (this._normalizeCache[fullName] = this.normalizeFullName(fullName));
    },

    /**
     @method makeToString
      @private
     @param {any} factory
     @param {string} fullName
     @return {function} toString function
     */
    makeToString: function (factory, fullName) {
      if (this.resolver && this.resolver.makeToString) {
        return this.resolver.makeToString(factory, fullName);
      } else if (this.fallback) {
        return this.fallback.makeToString(factory, fullName);
      } else {
        return factory.toString();
      }
    },

    /**
     Given a fullName check if the container is aware of its factory
     or singleton instance.
      @private
     @method has
     @param {String} fullName
     @param {Object} [options]
     @param {String} [options.source] the fullname of the request source (used for local lookups)
     @return {Boolean}
     */
    has: function (fullName, options) {
      if (!this.isValidFullName(fullName)) {
        return false;
      }

      var source = options && options.source && this.normalize(options.source);

      return has(this, this.normalize(fullName), source);
    },

    /**
     Allow registering options for all factories of a type.
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      // if all of type `connection` must not be singletons
     registry.optionsForType('connection', { singleton: false });
      registry.register('connection:twitter', TwitterConnection);
     registry.register('connection:facebook', FacebookConnection);
      let twitter = container.lookup('connection:twitter');
     let twitter2 = container.lookup('connection:twitter');
      twitter === twitter2; // => false
      let facebook = container.lookup('connection:facebook');
     let facebook2 = container.lookup('connection:facebook');
      facebook === facebook2; // => false
     ```
      @private
     @method optionsForType
     @param {String} type
     @param {Object} options
     */
    optionsForType: function (type, options) {
      this._typeOptions[type] = options;
    },

    getOptionsForType: function (type) {
      var optionsForType = this._typeOptions[type];
      if (optionsForType === undefined && this.fallback) {
        optionsForType = this.fallback.getOptionsForType(type);
      }
      return optionsForType;
    },

    /**
     @private
     @method options
     @param {String} fullName
     @param {Object} options
     */
    options: function (fullName) {
      var options = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

      var normalizedName = this.normalize(fullName);
      this._options[normalizedName] = options;
    },

    getOptions: function (fullName) {
      var normalizedName = this.normalize(fullName);
      var options = this._options[normalizedName];

      if (options === undefined && this.fallback) {
        options = this.fallback.getOptions(fullName);
      }
      return options;
    },

    getOption: function (fullName, optionName) {
      var options = this._options[fullName];

      if (options && options[optionName] !== undefined) {
        return options[optionName];
      }

      var type = fullName.split(':')[0];
      options = this._typeOptions[type];

      if (options && options[optionName] !== undefined) {
        return options[optionName];
      } else if (this.fallback) {
        return this.fallback.getOption(fullName, optionName);
      }
    },

    /**
     Used only via `injection`.
      Provides a specialized form of injection, specifically enabling
     all objects of one type to be injected with a reference to another
     object.
      For example, provided each object of type `controller` needed a `router`.
     one would do the following:
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      registry.register('router:main', Router);
     registry.register('controller:user', UserController);
     registry.register('controller:post', PostController);
      registry.typeInjection('controller', 'router', 'router:main');
      let user = container.lookup('controller:user');
     let post = container.lookup('controller:post');
      user.router instanceof Router; //=> true
     post.router instanceof Router; //=> true
      // both controllers share the same router
     user.router === post.router; //=> true
     ```
      @private
     @method typeInjection
     @param {String} type
     @param {String} property
     @param {String} fullName
     */
    typeInjection: function (type, property, fullName) {

      var fullNameType = fullName.split(':')[0];
      if (fullNameType === type) {
        throw new Error('Cannot inject a \'' + fullName + '\' on other ' + type + '(s).');
      }

      var injections = this._typeInjections[type] || (this._typeInjections[type] = []);

      injections.push({
        property: property,
        fullName: fullName
      });
    },

    /**
     Defines injection rules.
      These rules are used to inject dependencies onto objects when they
     are instantiated.
      Two forms of injections are possible:
      * Injecting one fullName on another fullName
     * Injecting one fullName on a type
      Example:
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      registry.register('source:main', Source);
     registry.register('model:user', User);
     registry.register('model:post', Post);
      // injecting one fullName on another fullName
     // eg. each user model gets a post model
     registry.injection('model:user', 'post', 'model:post');
      // injecting one fullName on another type
     registry.injection('model', 'source', 'source:main');
      let user = container.lookup('model:user');
     let post = container.lookup('model:post');
      user.source instanceof Source; //=> true
     post.source instanceof Source; //=> true
      user.post instanceof Post; //=> true
      // and both models share the same source
     user.source === post.source; //=> true
     ```
      @private
     @method injection
     @param {String} factoryName
     @param {String} property
     @param {String} injectionName
     */
    injection: function (fullName, property, injectionName) {
      this.validateFullName(injectionName);
      var normalizedInjectionName = this.normalize(injectionName);

      if (fullName.indexOf(':') === -1) {
        return this.typeInjection(fullName, property, normalizedInjectionName);
      }

      var normalizedName = this.normalize(fullName);

      var injections = this._injections[normalizedName] || (this._injections[normalizedName] = []);

      injections.push({
        property: property,
        fullName: normalizedInjectionName
      });
    },

    /**
     Used only via `factoryInjection`.
      Provides a specialized form of injection, specifically enabling
     all factory of one type to be injected with a reference to another
     object.
      For example, provided each factory of type `model` needed a `store`.
     one would do the following:
      ```javascript
     let registry = new Registry();
      registry.register('store:main', SomeStore);
      registry.factoryTypeInjection('model', 'store', 'store:main');
      let store = registry.lookup('store:main');
     let UserFactory = registry.lookupFactory('model:user');
      UserFactory.store instanceof SomeStore; //=> true
     ```
      @private
     @method factoryTypeInjection
     @param {String} type
     @param {String} property
     @param {String} fullName
     */
    factoryTypeInjection: function (type, property, fullName) {
      var injections = this._factoryTypeInjections[type] || (this._factoryTypeInjections[type] = []);

      injections.push({
        property: property,
        fullName: this.normalize(fullName)
      });
    },

    /**
     Defines factory injection rules.
      Similar to regular injection rules, but are run against factories, via
     `Registry#lookupFactory`.
      These rules are used to inject objects onto factories when they
     are looked up.
      Two forms of injections are possible:
      * Injecting one fullName on another fullName
     * Injecting one fullName on a type
      Example:
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      registry.register('store:main', Store);
     registry.register('store:secondary', OtherStore);
     registry.register('model:user', User);
     registry.register('model:post', Post);
      // injecting one fullName on another type
     registry.factoryInjection('model', 'store', 'store:main');
      // injecting one fullName on another fullName
     registry.factoryInjection('model:post', 'secondaryStore', 'store:secondary');
      let UserFactory = container.lookupFactory('model:user');
     let PostFactory = container.lookupFactory('model:post');
     let store = container.lookup('store:main');
      UserFactory.store instanceof Store; //=> true
     UserFactory.secondaryStore instanceof OtherStore; //=> false
      PostFactory.store instanceof Store; //=> true
     PostFactory.secondaryStore instanceof OtherStore; //=> true
      // and both models share the same source instance
     UserFactory.store === PostFactory.store; //=> true
     ```
      @private
     @method factoryInjection
     @param {String} factoryName
     @param {String} property
     @param {String} injectionName
     */
    factoryInjection: function (fullName, property, injectionName) {
      var normalizedName = this.normalize(fullName);
      var normalizedInjectionName = this.normalize(injectionName);

      this.validateFullName(injectionName);

      if (fullName.indexOf(':') === -1) {
        return this.factoryTypeInjection(normalizedName, property, normalizedInjectionName);
      }

      var injections = this._factoryInjections[normalizedName] || (this._factoryInjections[normalizedName] = []);

      injections.push({
        property: property,
        fullName: normalizedInjectionName
      });
    },

    /**
     @private
     @method knownForType
     @param {String} type the type to iterate over
    */
    knownForType: function (type) {
      var fallbackKnown = undefined,
          resolverKnown = undefined;

      var localKnown = _emberUtils.dictionary(null);
      var registeredNames = Object.keys(this.registrations);
      for (var index = 0; index < registeredNames.length; index++) {
        var fullName = registeredNames[index];
        var itemType = fullName.split(':')[0];

        if (itemType === type) {
          localKnown[fullName] = true;
        }
      }

      if (this.fallback) {
        fallbackKnown = this.fallback.knownForType(type);
      }

      if (this.resolver && this.resolver.knownForType) {
        resolverKnown = this.resolver.knownForType(type);
      }

      return _emberUtils.assign({}, fallbackKnown, localKnown, resolverKnown);
    },

    validateFullName: function (fullName) {
      if (!this.isValidFullName(fullName)) {
        throw new TypeError('Invalid Fullname, expected: \'type:name\' got: ' + fullName);
      }

      return true;
    },

    isValidFullName: function (fullName) {
      return !!VALID_FULL_NAME_REGEXP.test(fullName);
    },

    validateInjections: function (injections) {
      if (!injections) {
        return;
      }

      var fullName = undefined;

      for (var i = 0; i < injections.length; i++) {
        fullName = injections[i].fullName;

        if (!this.has(fullName)) {
          throw new Error('Attempting to inject an unknown injection: \'' + fullName + '\'');
        }
      }
    },

    normalizeInjectionsHash: function (hash) {
      var injections = [];

      for (var key in hash) {
        if (hash.hasOwnProperty(key)) {

          injections.push({
            property: key,
            fullName: hash[key]
          });
        }
      }

      return injections;
    },

    getInjections: function (fullName) {
      var injections = this._injections[fullName] || [];
      if (this.fallback) {
        injections = injections.concat(this.fallback.getInjections(fullName));
      }
      return injections;
    },

    getTypeInjections: function (type) {
      var injections = this._typeInjections[type] || [];
      if (this.fallback) {
        injections = injections.concat(this.fallback.getTypeInjections(type));
      }
      return injections;
    },

    getFactoryInjections: function (fullName) {
      var injections = this._factoryInjections[fullName] || [];
      if (this.fallback) {
        injections = injections.concat(this.fallback.getFactoryInjections(fullName));
      }
      return injections;
    },

    getFactoryTypeInjections: function (type) {
      var injections = this._factoryTypeInjections[type] || [];
      if (this.fallback) {
        injections = injections.concat(this.fallback.getFactoryTypeInjections(type));
      }
      return injections;
    }
  };

  function deprecateResolverFunction(registry) {
    registry.resolver = {
      resolve: registry.resolver
    };
  }

  /**
   Given a fullName and a source fullName returns the fully resolved
   fullName. Used to allow for local lookup.
  
   ```javascript
   let registry = new Registry();
  
   // the twitter factory is added to the module system
   registry.expandLocalLookup('component:post-title', { source: 'template:post' }) // => component:post/post-title
   ```
  
   @private
   @method expandLocalLookup
   @param {String} fullName
   @param {Object} [options]
   @param {String} [options.source] the fullname of the request source (used for local lookups)
   @return {String} fullName
   */
  Registry.prototype.expandLocalLookup = function Registry_expandLocalLookup(fullName, options) {
    if (this.resolver && this.resolver.expandLocalLookup) {

      var normalizedFullName = this.normalize(fullName);
      var normalizedSource = this.normalize(options.source);

      return expandLocalLookup(this, normalizedFullName, normalizedSource);
    } else if (this.fallback) {
      return this.fallback.expandLocalLookup(fullName, options);
    } else {
      return null;
    }
  };

  function expandLocalLookup(registry, normalizedName, normalizedSource) {
    var cache = registry._localLookupCache;
    var normalizedNameCache = cache[normalizedName];

    if (!normalizedNameCache) {
      normalizedNameCache = cache[normalizedName] = new _emberUtils.EmptyObject();
    }

    var cached = normalizedNameCache[normalizedSource];

    if (cached !== undefined) {
      return cached;
    }

    var expanded = registry.resolver.expandLocalLookup(normalizedName, normalizedSource);

    return normalizedNameCache[normalizedSource] = expanded;
  }

  function resolve(registry, normalizedName, options) {
    if (options && options.source) {
      // when `source` is provided expand normalizedName
      // and source into the full normalizedName
      normalizedName = registry.expandLocalLookup(normalizedName, options);

      // if expandLocalLookup returns falsey, we do not support local lookup
      if (!normalizedName) {
        return;
      }
    }

    var cached = registry._resolveCache[normalizedName];
    if (cached !== undefined) {
      return cached;
    }
    if (registry._failCache[normalizedName]) {
      return;
    }

    var resolved = undefined;

    if (registry.resolver) {
      resolved = registry.resolver.resolve(normalizedName);
    }

    if (resolved === undefined) {
      resolved = registry.registrations[normalizedName];
    }

    if (resolved === undefined) {
      registry._failCache[normalizedName] = true;
    } else {
      registry._resolveCache[normalizedName] = resolved;
    }

    return resolved;
  }

  function has(registry, fullName, source) {
    return registry.resolve(fullName, { source: source }) !== undefined;
  }

  var privateNames = _emberUtils.dictionary(null);
  var privateSuffix = '' + Math.random() + Date.now();

  function privatize(_ref) {
    var fullName = _ref[0];

    var name = privateNames[fullName];
    if (name) {
      return name;
    }

    var _fullName$split = fullName.split(':');

    var type = _fullName$split[0];
    var rawName = _fullName$split[1];

    return privateNames[fullName] = _emberUtils.intern(type + ':' + rawName + '-' + privateSuffix);
  }
});
enifed('dag-map', ['exports'], function (exports) { 'use strict';

/**
 * A map of key/value pairs with dependencies contraints that can be traversed
 * in topological order and is checked for cycles.
 *
 * @class DAG
 * @constructor
 */
var DAG = (function () {
    function DAG() {
        this._vertices = new Vertices();
    }
    /**
     * Adds a key/value pair with dependencies on other key/value pairs.
     *
     * @public
     * @method addEdges
     * @param {string[]}   key The key of the vertex to be added.
     * @param {any}      value The value of that vertex.
     * @param {string[]|string|undefined}  before A key or array of keys of the vertices that must
     *                                            be visited before this vertex.
     * @param {string[]|string|undefined}   after An string or array of strings with the keys of the
     *                                            vertices that must be after this vertex is visited.
     */
    DAG.prototype.add = function (key, value, before, after) {
        var vertices = this._vertices;
        var v = vertices.add(key);
        v.val = value;
        if (before) {
            if (typeof before === "string") {
                vertices.addEdge(v, vertices.add(before));
            }
            else {
                for (var i = 0; i < before.length; i++) {
                    vertices.addEdge(v, vertices.add(before[i]));
                }
            }
        }
        if (after) {
            if (typeof after === "string") {
                vertices.addEdge(vertices.add(after), v);
            }
            else {
                for (var i = 0; i < after.length; i++) {
                    vertices.addEdge(vertices.add(after[i]), v);
                }
            }
        }
    };
    /**
     * Visits key/value pairs in topological order.
     *
     * @public
     * @method  topsort
     * @param {Function} fn The function to be invoked with each key/value.
     */
    DAG.prototype.topsort = function (callback) {
        this._vertices.topsort(callback);
    };
    return DAG;
}());
var Vertices = (function () {
    function Vertices() {
        this.stack = new IntStack();
        this.result = new IntStack();
        this.vertices = [];
    }
    Vertices.prototype.add = function (key) {
        if (!key)
            throw new Error("missing key");
        var vertices = this.vertices;
        var i = 0;
        var vertex;
        for (; i < vertices.length; i++) {
            vertex = vertices[i];
            if (vertex.key === key)
                return vertex;
        }
        return vertices[i] = {
            id: i,
            key: key,
            val: null,
            inc: null,
            out: false,
            mark: false
        };
    };
    Vertices.prototype.addEdge = function (v, w) {
        this.check(v, w.key);
        var inc = w.inc;
        if (!inc) {
            w.inc = [v.id];
        }
        else {
            var i = 0;
            for (; i < inc.length; i++) {
                if (inc[i] === v.id)
                    return;
            }
            inc[i] = v.id;
        }
        v.out = true;
    };
    Vertices.prototype.topsort = function (cb) {
        this.reset();
        var vertices = this.vertices;
        for (var i = 0; i < vertices.length; i++) {
            var vertex = vertices[i];
            if (vertex.out)
                continue;
            this.visit(vertex, undefined);
        }
        this.each(cb);
    };
    Vertices.prototype.check = function (v, w) {
        if (v.key === w) {
            throw new Error("cycle detected: " + w + " <- " + w);
        }
        var inc = v.inc;
        // quick check
        if (!inc || inc.length === 0)
            return;
        var vertices = this.vertices;
        // shallow check
        for (var i = 0; i < inc.length; i++) {
            var key = vertices[inc[i]].key;
            if (key === w) {
                throw new Error("cycle detected: " + w + " <- " + v.key + " <- " + w);
            }
        }
        // deep check
        this.reset();
        this.visit(v, w);
        if (this.result.len > 0) {
            var msg_1 = "cycle detected: " + w;
            this.each(function (key) {
                msg_1 += " <- " + key;
            });
            throw new Error(msg_1);
        }
    };
    Vertices.prototype.each = function (cb) {
        var _a = this, result = _a.result, vertices = _a.vertices;
        for (var i = 0; i < result.len; i++) {
            var vertex = vertices[result.stack[i]];
            cb(vertex.key, vertex.val);
        }
    };
    // reuse between cycle check and topsort
    Vertices.prototype.reset = function () {
        this.stack.len = 0;
        this.result.len = 0;
        var vertices = this.vertices;
        for (var i = 0; i < vertices.length; i++) {
            vertices[i].mark = false;
        }
    };
    Vertices.prototype.visit = function (start, search) {
        var _a = this, stack = _a.stack, result = _a.result, vertices = _a.vertices;
        stack.push(start.id);
        while (stack.len) {
            var index = stack.pop();
            if (index < 0) {
                index = ~index;
                if (search) {
                    result.pop();
                }
                else {
                    result.push(index);
                }
            }
            else {
                var vertex = vertices[index];
                if (vertex.mark) {
                    continue;
                }
                if (search) {
                    result.push(index);
                    if (search === vertex.key) {
                        return;
                    }
                }
                vertex.mark = true;
                stack.push(~index);
                var incoming = vertex.inc;
                if (incoming) {
                    var i = incoming.length;
                    while (i--) {
                        index = incoming[i];
                        if (!vertices[index].mark) {
                            stack.push(index);
                        }
                    }
                }
            }
        }
    };
    return Vertices;
}());
var IntStack = (function () {
    function IntStack() {
        this.stack = [0, 0, 0, 0, 0, 0];
        this.len = 0;
    }
    IntStack.prototype.push = function (n) {
        this.stack[this.len++] = n;
    };
    IntStack.prototype.pop = function () {
        return this.stack[--this.len];
    };
    return IntStack;
}());

exports['default'] = DAG;

Object.defineProperty(exports, '__esModule', { value: true });

});
enifed('ember-application/index', ['exports', 'ember-application/initializers/dom-templates', 'ember-application/system/application', 'ember-application/system/application-instance', 'ember-application/system/resolver', 'ember-application/system/engine', 'ember-application/system/engine-instance', 'ember-application/system/engine-parent'], function (exports, _emberApplicationInitializersDomTemplates, _emberApplicationSystemApplication, _emberApplicationSystemApplicationInstance, _emberApplicationSystemResolver, _emberApplicationSystemEngine, _emberApplicationSystemEngineInstance, _emberApplicationSystemEngineParent) {
  /**
  @module ember
  @submodule ember-application
  */

  'use strict';

  exports.Application = _emberApplicationSystemApplication.default;
  exports.ApplicationInstance = _emberApplicationSystemApplicationInstance.default;
  exports.Resolver = _emberApplicationSystemResolver.default;
  exports.Engine = _emberApplicationSystemEngine.default;
  exports.EngineInstance = _emberApplicationSystemEngineInstance.default;
  exports.getEngineParent = _emberApplicationSystemEngineParent.getEngineParent;
  exports.setEngineParent = _emberApplicationSystemEngineParent.setEngineParent;

  // add domTemplates initializer (only does something if `ember-template-compiler`
  // is loaded already)
});
enifed('ember-application/initializers/dom-templates', ['exports', 'require', 'ember-glimmer', 'ember-environment', 'ember-application/system/application'], function (exports, _require, _emberGlimmer, _emberEnvironment, _emberApplicationSystemApplication) {
  'use strict';

  var bootstrap = function () {};

  _emberApplicationSystemApplication.default.initializer({
    name: 'domTemplates',
    initialize: function () {
      var bootstrapModuleId = 'ember-template-compiler/system/bootstrap';
      var context = undefined;
      if (_emberEnvironment.environment.hasDOM && _require.has(bootstrapModuleId)) {
        bootstrap = _require.default(bootstrapModuleId).default;
        context = document;
      }

      bootstrap({ context: context, hasTemplate: _emberGlimmer.hasTemplate, setTemplate: _emberGlimmer.setTemplate });
    }
  });
});
enifed('ember-application/system/application-instance', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime', 'ember-environment', 'ember-views', 'ember-application/system/engine-instance'], function (exports, _emberUtils, _emberMetal, _emberRuntime, _emberEnvironment, _emberViews, _emberApplicationSystemEngineInstance) {
  /**
  @module ember
  @submodule ember-application
  */

  'use strict';

  var BootOptions = undefined;

  /**
    The `ApplicationInstance` encapsulates all of the stateful aspects of a
    running `Application`.
  
    At a high-level, we break application boot into two distinct phases:
  
    * Definition time, where all of the classes, templates, and other
      dependencies are loaded (typically in the browser).
    * Run time, where we begin executing the application once everything
      has loaded.
  
    Definition time can be expensive and only needs to happen once since it is
    an idempotent operation. For example, between test runs and FastBoot
    requests, the application stays the same. It is only the state that we want
    to reset.
  
    That state is what the `ApplicationInstance` manages: it is responsible for
    creating the container that contains all application state, and disposing of
    it once the particular test run or FastBoot request has finished.
  
    @public
    @class Ember.ApplicationInstance
    @extends Ember.EngineInstance
  */

  var ApplicationInstance = _emberApplicationSystemEngineInstance.default.extend({
    /**
      The `Application` for which this is an instance.
       @property {Ember.Application} application
      @private
    */
    application: null,

    /**
      The DOM events for which the event dispatcher should listen.
       By default, the application's `Ember.EventDispatcher` listens
      for a set of standard DOM events, such as `mousedown` and
      `keyup`, and delegates them to your application's `Ember.View`
      instances.
       @private
      @property {Object} customEvents
    */
    customEvents: null,

    /**
      The root DOM element of the Application as an element or a
      [jQuery-compatible selector
      string](http://api.jquery.com/category/selectors/).
       @private
      @property {String|DOMElement} rootElement
    */
    rootElement: null,

    init: function () {
      this._super.apply(this, arguments);

      // Register this instance in the per-instance registry.
      //
      // Why do we need to register the instance in the first place?
      // Because we need a good way for the root route (a.k.a ApplicationRoute)
      // to notify us when it has created the root-most view. That view is then
      // appended to the rootElement, in the case of apps, to the fixture harness
      // in tests, or rendered to a string in the case of FastBoot.
      this.register('-application-instance:main', this, { instantiate: false });
    },

    /**
      Overrides the base `EngineInstance._bootSync` method with concerns relevant
      to booting application (instead of engine) instances.
       This method should only contain synchronous boot concerns. Asynchronous
      boot concerns should eventually be moved to the `boot` method, which
      returns a promise.
       Until all boot code has been made asynchronous, we need to continue to
      expose this method for use *internally* in places where we need to boot an
      instance synchronously.
       @private
    */
    _bootSync: function (options) {
      if (this._booted) {
        return this;
      }

      options = new BootOptions(options);

      this.setupRegistry(options);

      if (options.rootElement) {
        this.rootElement = options.rootElement;
      } else {
        this.rootElement = this.application.rootElement;
      }

      if (options.location) {
        var router = _emberMetal.get(this, 'router');
        _emberMetal.set(router, 'location', options.location);
      }

      this.application.runInstanceInitializers(this);

      if (options.isInteractive) {
        this.setupEventDispatcher();
      }

      this._booted = true;

      return this;
    },

    setupRegistry: function (options) {
      this.constructor.setupRegistry(this.__registry__, options);
    },

    router: _emberMetal.computed(function () {
      return this.lookup('router:main');
    }).readOnly(),

    /**
      This hook is called by the root-most Route (a.k.a. the ApplicationRoute)
      when it has finished creating the root View. By default, we simply take the
      view and append it to the `rootElement` specified on the Application.
       In cases like FastBoot and testing, we can override this hook and implement
      custom behavior, such as serializing to a string and sending over an HTTP
      socket rather than appending to DOM.
       @param view {Ember.View} the root-most view
      @private
    */
    didCreateRootView: function (view) {
      view.appendTo(this.rootElement);
    },

    /**
      Tells the router to start routing. The router will ask the location for the
      current URL of the page to determine the initial URL to start routing to.
      To start the app at a specific URL, call `handleURL` instead.
       @private
    */
    startRouting: function () {
      var router = _emberMetal.get(this, 'router');
      router.startRouting();
      this._didSetupRouter = true;
    },

    /**
      @private
       Sets up the router, initializing the child router and configuring the
      location before routing begins.
       Because setup should only occur once, multiple calls to `setupRouter`
      beyond the first call have no effect.
    */
    setupRouter: function () {
      if (this._didSetupRouter) {
        return;
      }
      this._didSetupRouter = true;

      var router = _emberMetal.get(this, 'router');
      router.setupRouter();
    },

    /**
      Directs the router to route to a particular URL. This is useful in tests,
      for example, to tell the app to start at a particular URL.
       @param url {String} the URL the router should route to
      @private
    */
    handleURL: function (url) {
      var router = _emberMetal.get(this, 'router');

      this.setupRouter();
      return router.handleURL(url);
    },

    /**
      @private
    */
    setupEventDispatcher: function () {
      var dispatcher = this.lookup('event_dispatcher:main');
      var applicationCustomEvents = _emberMetal.get(this.application, 'customEvents');
      var instanceCustomEvents = _emberMetal.get(this, 'customEvents');

      var customEvents = _emberUtils.assign({}, applicationCustomEvents, instanceCustomEvents);
      dispatcher.setup(customEvents, this.rootElement);

      return dispatcher;
    },

    /**
      Returns the current URL of the app instance. This is useful when your
      app does not update the browsers URL bar (i.e. it uses the `'none'`
      location adapter).
       @public
      @return {String} the current URL
    */
    getURL: function () {
      var router = _emberMetal.get(this, 'router');
      return _emberMetal.get(router, 'url');
    },

    // `instance.visit(url)` should eventually replace `instance.handleURL()`;
    // the test helpers can probably be switched to use this implementation too

    /**
      Navigate the instance to a particular URL. This is useful in tests, for
      example, or to tell the app to start at a particular URL. This method
      returns a promise that resolves with the app instance when the transition
      is complete, or rejects if the transion was aborted due to an error.
       @public
      @param url {String} the destination URL
      @return {Promise}
    */
    visit: function (url) {
      var _this = this;

      this.setupRouter();

      var bootOptions = this.__container__.lookup('-environment:main');

      var router = _emberMetal.get(this, 'router');

      var handleTransitionResolve = function () {
        if (!bootOptions.options.shouldRender) {
          // No rendering is needed, and routing has completed, simply return.
          return _this;
        } else {
          return new _emberRuntime.RSVP.Promise(function (resolve) {
            // Resolve once rendering is completed. `router.handleURL` returns the transition (as a thennable)
            // which resolves once the transition is completed, but the transition completion only queues up
            // a scheduled revalidation (into the `render` queue) in the Renderer.
            //
            // This uses `run.schedule('afterRender', ....)` to resolve after that rendering has completed.
            _emberMetal.run.schedule('afterRender', null, resolve, _this);
          });
        }
      };

      var handleTransitionReject = function (error) {
        if (error.error) {
          throw error.error;
        } else if (error.name === 'TransitionAborted' && router.router.activeTransition) {
          return router.router.activeTransition.then(handleTransitionResolve, handleTransitionReject);
        } else if (error.name === 'TransitionAborted') {
          throw new Error(error.message);
        } else {
          throw error;
        }
      };

      var location = _emberMetal.get(router, 'location');

      // Keeps the location adapter's internal URL in-sync
      location.setURL(url);

      // getURL returns the set url with the rootURL stripped off
      return router.handleURL(location.getURL()).then(handleTransitionResolve, handleTransitionReject);
    }
  });

  ApplicationInstance.reopenClass({
    /**
     @private
     @method setupRegistry
     @param {Registry} registry
     @param {BootOptions} options
    */
    setupRegistry: function (registry) {
      var options = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

      if (!options.toEnvironment) {
        options = new BootOptions(options);
      }

      registry.register('-environment:main', options.toEnvironment(), { instantiate: false });
      registry.register('service:-document', options.document, { instantiate: false });

      this._super(registry, options);
    }
  });

  /**
    A list of boot-time configuration options for customizing the behavior of
    an `Ember.ApplicationInstance`.
  
    This is an interface class that exists purely to document the available
    options; you do not need to construct it manually. Simply pass a regular
    JavaScript object containing the desired options into methods that require
    one of these options object:
  
    ```javascript
    MyApp.visit("/", { location: "none", rootElement: "#container" });
    ```
  
    Not all combinations of the supported options are valid. See the documentation
    on `Ember.Application#visit` for the supported configurations.
  
    Internal, experimental or otherwise unstable flags are marked as private.
  
    @class BootOptions
    @namespace Ember.ApplicationInstance
    @public
  */
  BootOptions = function BootOptions() {
    var options = arguments.length <= 0 || arguments[0] === undefined ? {} : arguments[0];

    /**
      Provide a specific instance of jQuery. This is useful in conjunction with
      the `document` option, as it allows you to use a copy of `jQuery` that is
      appropriately bound to the foreign `document` (e.g. a jsdom).
       This is highly experimental and support very incomplete at the moment.
       @property jQuery
      @type Object
      @default auto-detected
      @private
    */
    this.jQuery = _emberViews.jQuery; // This default is overridable below

    /**
      Interactive mode: whether we need to set up event delegation and invoke
      lifecycle callbacks on Components.
       @property isInteractive
      @type boolean
      @default auto-detected
      @private
    */
    this.isInteractive = _emberEnvironment.environment.hasDOM; // This default is overridable below

    /**
      Run in a full browser environment.
       When this flag is set to `false`, it will disable most browser-specific
      and interactive features. Specifically:
       * It does not use `jQuery` to append the root view; the `rootElement`
        (either specified as a subsequent option or on the application itself)
        must already be an `Element` in the given `document` (as opposed to a
        string selector).
       * It does not set up an `EventDispatcher`.
       * It does not run any `Component` lifecycle hooks (such as `didInsertElement`).
       * It sets the `location` option to `"none"`. (If you would like to use
        the location adapter specified in the app's router instead, you can also
        specify `{ location: null }` to specifically opt-out.)
       @property isBrowser
      @type boolean
      @default auto-detected
      @public
    */
    if (options.isBrowser !== undefined) {
      this.isBrowser = !!options.isBrowser;
    } else {
      this.isBrowser = _emberEnvironment.environment.hasDOM;
    }

    if (!this.isBrowser) {
      this.jQuery = null;
      this.isInteractive = false;
      this.location = 'none';
    }

    /**
      Disable rendering completely.
       When this flag is set to `true`, it will disable the entire rendering
      pipeline. Essentially, this puts the app into "routing-only" mode. No
      templates will be rendered, and no Components will be created.
       @property shouldRender
      @type boolean
      @default true
      @public
    */
    if (options.shouldRender !== undefined) {
      this.shouldRender = !!options.shouldRender;
    } else {
      this.shouldRender = true;
    }

    if (!this.shouldRender) {
      this.jQuery = null;
      this.isInteractive = false;
    }

    /**
      If present, render into the given `Document` object instead of the
      global `window.document` object.
       In practice, this is only useful in non-browser environment or in
      non-interactive mode, because Ember's `jQuery` dependency is
      implicitly bound to the current document, causing event delegation
      to not work properly when the app is rendered into a foreign
      document object (such as an iframe's `contentDocument`).
       In non-browser mode, this could be a "`Document`-like" object as
      Ember only interact with a small subset of the DOM API in non-
      interactive mode. While the exact requirements have not yet been
      formalized, the `SimpleDOM` library's implementation is known to
      work.
       @property document
      @type Document
      @default the global `document` object
      @public
    */
    if (options.document) {
      this.document = options.document;
    } else {
      this.document = typeof document !== 'undefined' ? document : null;
    }

    /**
      If present, overrides the application's `rootElement` property on
      the instance. This is useful for testing environment, where you
      might want to append the root view to a fixture area.
       In non-browser mode, because Ember does not have access to jQuery,
      this options must be specified as a DOM `Element` object instead of
      a selector string.
       See the documentation on `Ember.Applications`'s `rootElement` for
      details.
       @property rootElement
      @type String|Element
      @default null
      @public
     */
    if (options.rootElement) {
      this.rootElement = options.rootElement;
    }

    // Set these options last to give the user a chance to override the
    // defaults from the "combo" options like `isBrowser` (although in
    // practice, the resulting combination is probably invalid)

    /**
      If present, overrides the router's `location` property with this
      value. This is useful for environments where trying to modify the
      URL would be inappropriate.
       @property location
      @type string
      @default null
      @public
    */
    if (options.location !== undefined) {
      this.location = options.location;
    }

    if (options.jQuery !== undefined) {
      this.jQuery = options.jQuery;
    }

    if (options.isInteractive !== undefined) {
      this.isInteractive = !!options.isInteractive;
    }
  };

  BootOptions.prototype.toEnvironment = function () {
    var env = _emberUtils.assign({}, _emberEnvironment.environment);
    // For compatibility with existing code
    env.hasDOM = this.isBrowser;
    env.isInteractive = this.isInteractive;
    env.options = this;
    return env;
  };

  Object.defineProperty(ApplicationInstance.prototype, 'container', {
    configurable: true,
    enumerable: false,
    get: function () {
      var instance = this;
      return {
        lookup: function () {
          return instance.lookup.apply(instance, arguments);
        }
      };
    }
  });

  Object.defineProperty(ApplicationInstance.prototype, 'registry', {
    configurable: true,
    enumerable: false,
    get: function () {
      return _emberRuntime.buildFakeRegistryWithDeprecations(this, 'ApplicationInstance');
    }
  });

  exports.default = ApplicationInstance;
});
enifed('ember-application/system/application', ['exports', 'ember-utils', 'ember-environment', 'ember-metal', 'ember-runtime', 'ember-views', 'ember-routing', 'ember-application/system/application-instance', 'container', 'ember-application/system/engine', 'ember-glimmer'], function (exports, _emberUtils, _emberEnvironment, _emberMetal, _emberRuntime, _emberViews, _emberRouting, _emberApplicationSystemApplicationInstance, _container, _emberApplicationSystemEngine, _emberGlimmer) {
  /**
  @module ember
  @submodule ember-application
  */
  'use strict';

  var _templateObject = babelHelpers.taggedTemplateLiteralLoose(['-bucket-cache:main'], ['-bucket-cache:main']);

  var librariesRegistered = false;

  /**
    An instance of `Ember.Application` is the starting point for every Ember
    application. It helps to instantiate, initialize and coordinate the many
    objects that make up your app.
  
    Each Ember app has one and only one `Ember.Application` object. In fact, the
    very first thing you should do in your application is create the instance:
  
    ```javascript
    window.App = Ember.Application.create();
    ```
  
    Typically, the application object is the only global variable. All other
    classes in your app should be properties on the `Ember.Application` instance,
    which highlights its first role: a global namespace.
  
    For example, if you define a view class, it might look like this:
  
    ```javascript
    App.MyView = Ember.View.extend();
    ```
  
    By default, calling `Ember.Application.create()` will automatically initialize
    your application by calling the `Ember.Application.initialize()` method. If
    you need to delay initialization, you can call your app's `deferReadiness()`
    method. When you are ready for your app to be initialized, call its
    `advanceReadiness()` method.
  
    You can define a `ready` method on the `Ember.Application` instance, which
    will be run by Ember when the application is initialized.
  
    Because `Ember.Application` inherits from `Ember.Namespace`, any classes
    you create will have useful string representations when calling `toString()`.
    See the `Ember.Namespace` documentation for more information.
  
    While you can think of your `Ember.Application` as a container that holds the
    other classes in your application, there are several other responsibilities
    going on under-the-hood that you may want to understand.
  
    ### Event Delegation
  
    Ember uses a technique called _event delegation_. This allows the framework
    to set up a global, shared event listener instead of requiring each view to
    do it manually. For example, instead of each view registering its own
    `mousedown` listener on its associated element, Ember sets up a `mousedown`
    listener on the `body`.
  
    If a `mousedown` event occurs, Ember will look at the target of the event and
    start walking up the DOM node tree, finding corresponding views and invoking
    their `mouseDown` method as it goes.
  
    `Ember.Application` has a number of default events that it listens for, as
    well as a mapping from lowercase events to camel-cased view method names. For
    example, the `keypress` event causes the `keyPress` method on the view to be
    called, the `dblclick` event causes `doubleClick` to be called, and so on.
  
    If there is a bubbling browser event that Ember does not listen for by
    default, you can specify custom events and their corresponding view method
    names by setting the application's `customEvents` property:
  
    ```javascript
    let App = Ember.Application.create({
      customEvents: {
        // add support for the paste event
        paste: 'paste'
      }
    });
    ```
  
    To prevent Ember from setting up a listener for a default event,
    specify the event name with a `null` value in the `customEvents`
    property:
  
    ```javascript
    let App = Ember.Application.create({
      customEvents: {
        // prevent listeners for mouseenter/mouseleave events
        mouseenter: null,
        mouseleave: null
      }
    });
    ```
  
    By default, the application sets up these event listeners on the document
    body. However, in cases where you are embedding an Ember application inside
    an existing page, you may want it to set up the listeners on an element
    inside the body.
  
    For example, if only events inside a DOM element with the ID of `ember-app`
    should be delegated, set your application's `rootElement` property:
  
    ```javascript
    let App = Ember.Application.create({
      rootElement: '#ember-app'
    });
    ```
  
    The `rootElement` can be either a DOM element or a jQuery-compatible selector
    string. Note that *views appended to the DOM outside the root element will
    not receive events.* If you specify a custom root element, make sure you only
    append views inside it!
  
    To learn more about the events Ember components use, see
    [components/handling-events](https://guides.emberjs.com/v2.6.0/components/handling-events/#toc_event-names).
  
    ### Initializers
  
    Libraries on top of Ember can add initializers, like so:
  
    ```javascript
    Ember.Application.initializer({
      name: 'api-adapter',
  
      initialize: function(application) {
        application.register('api-adapter:main', ApiAdapter);
      }
    });
    ```
  
    Initializers provide an opportunity to access the internal registry, which
    organizes the different components of an Ember application. Additionally
    they provide a chance to access the instantiated application. Beyond
    being used for libraries, initializers are also a great way to organize
    dependency injection or setup in your own application.
  
    ### Routing
  
    In addition to creating your application's router, `Ember.Application` is
    also responsible for telling the router when to start routing. Transitions
    between routes can be logged with the `LOG_TRANSITIONS` flag, and more
    detailed intra-transition logging can be logged with
    the `LOG_TRANSITIONS_INTERNAL` flag:
  
    ```javascript
    let App = Ember.Application.create({
      LOG_TRANSITIONS: true, // basic logging of successful transitions
      LOG_TRANSITIONS_INTERNAL: true // detailed logging of all routing steps
    });
    ```
  
    By default, the router will begin trying to translate the current URL into
    application state once the browser emits the `DOMContentReady` event. If you
    need to defer routing, you can call the application's `deferReadiness()`
    method. Once routing can begin, call the `advanceReadiness()` method.
  
    If there is any setup required before routing begins, you can implement a
    `ready()` method on your app that will be invoked immediately before routing
    begins.
  
    @class Application
    @namespace Ember
    @extends Ember.Engine
    @uses RegistryProxyMixin
    @public
  */

  var Application = _emberApplicationSystemEngine.default.extend({
    _suppressDeferredDeprecation: true,

    /**
      The root DOM element of the Application. This can be specified as an
      element or a
      [jQuery-compatible selector string](http://api.jquery.com/category/selectors/).
       This is the element that will be passed to the Application's,
      `eventDispatcher`, which sets up the listeners for event delegation. Every
      view in your application should be a child of the element you specify here.
       @property rootElement
      @type DOMElement
      @default 'body'
      @public
    */
    rootElement: 'body',

    /**
      The `Ember.EventDispatcher` responsible for delegating events to this
      application's views.
       The event dispatcher is created by the application at initialization time
      and sets up event listeners on the DOM element described by the
      application's `rootElement` property.
       See the documentation for `Ember.EventDispatcher` for more information.
       @property eventDispatcher
      @type Ember.EventDispatcher
      @default null
      @public
    */
    eventDispatcher: null,

    /**
      The DOM events for which the event dispatcher should listen.
       By default, the application's `Ember.EventDispatcher` listens
      for a set of standard DOM events, such as `mousedown` and
      `keyup`, and delegates them to your application's `Ember.View`
      instances.
       If you would like additional bubbling events to be delegated to your
      views, set your `Ember.Application`'s `customEvents` property
      to a hash containing the DOM event name as the key and the
      corresponding view method name as the value. Setting an event to
      a value of `null` will prevent a default event listener from being
      added for that event.
       To add new events to be listened to:
       ```javascript
      let App = Ember.Application.create({
        customEvents: {
          // add support for the paste event
          paste: 'paste'
        }
      });
      ```
       To prevent default events from being listened to:
       ```javascript
      let App = Ember.Application.create({
        customEvents: {
          // remove support for mouseenter / mouseleave events
          mouseenter: null,
          mouseleave: null
        }
      });
      ```
      @property customEvents
      @type Object
      @default null
      @public
    */
    customEvents: null,

    /**
      Whether the application should automatically start routing and render
      templates to the `rootElement` on DOM ready. While default by true,
      other environments such as FastBoot or a testing harness can set this
      property to `false` and control the precise timing and behavior of the boot
      process.
       @property autoboot
      @type Boolean
      @default true
      @private
    */
    autoboot: true,

    /**
      Whether the application should be configured for the legacy "globals mode".
      Under this mode, the Application object serves as a global namespace for all
      classes.
       ```javascript
      let App = Ember.Application.create({
        ...
      });
       App.Router.reopen({
        location: 'none'
      });
       App.Router.map({
        ...
      });
       App.MyComponent = Ember.Component.extend({
        ...
      });
      ```
       This flag also exposes other internal APIs that assumes the existence of
      a special "default instance", like `App.__container__.lookup(...)`.
       This option is currently not configurable, its value is derived from
      the `autoboot` flag – disabling `autoboot` also implies opting-out of
      globals mode support, although they are ultimately orthogonal concerns.
       Some of the global modes features are already deprecated in 1.x. The
      existence of this flag is to untangle the globals mode code paths from
      the autoboot code paths, so that these legacy features can be reviewed
      for deprecation/removal separately.
       Forcing the (autoboot=true, _globalsMode=false) here and running the tests
      would reveal all the places where we are still relying on these legacy
      behavior internally (mostly just tests).
       @property _globalsMode
      @type Boolean
      @default true
      @private
    */
    _globalsMode: true,

    init: function (options) {
      this._super.apply(this, arguments);

      if (!this.$) {
        this.$ = _emberViews.jQuery;
      }

      registerLibraries();
      logLibraryVersions();

      // Start off the number of deferrals at 1. This will be decremented by
      // the Application's own `boot` method.
      this._readinessDeferrals = 1;
      this._booted = false;

      this.autoboot = this._globalsMode = !!this.autoboot;

      if (this._globalsMode) {
        this._prepareForGlobalsMode();
      }

      if (this.autoboot) {
        this.waitForDOMReady();
      }
    },

    /**
      Create an ApplicationInstance for this application.
       @private
      @method buildInstance
      @return {Ember.ApplicationInstance} the application instance
    */
    buildInstance: function () {
      var options = arguments.length <= 0 || arguments[0] === undefined ? {} : arguments[0];

      options.base = this;
      options.application = this;
      return _emberApplicationSystemApplicationInstance.default.create(options);
    },

    /**
      Enable the legacy globals mode by allowing this application to act
      as a global namespace. See the docs on the `_globalsMode` property
      for details.
       Most of these features are already deprecated in 1.x, so we can
      stop using them internally and try to remove them.
       @private
      @method _prepareForGlobalsMode
    */
    _prepareForGlobalsMode: function () {
      // Create subclass of Ember.Router for this Application instance.
      // This is to ensure that someone reopening `App.Router` does not
      // tamper with the default `Ember.Router`.
      this.Router = (this.Router || _emberRouting.Router).extend();

      this._buildDeprecatedInstance();
    },

    /*
      Build the deprecated instance for legacy globals mode support.
      Called when creating and resetting the application.
       This is orthogonal to autoboot: the deprecated instance needs to
      be created at Application construction (not boot) time to expose
      App.__container__. If autoboot sees that this instance exists,
      it will continue booting it to avoid doing unncessary work (as
      opposed to building a new instance at boot time), but they are
      otherwise unrelated.
       @private
      @method _buildDeprecatedInstance
    */
    _buildDeprecatedInstance: function () {
      // Build a default instance
      var instance = this.buildInstance();

      // Legacy support for App.__container__ and other global methods
      // on App that rely on a single, default instance.
      this.__deprecatedInstance__ = instance;
      this.__container__ = instance.__container__;
    },

    /**
      Automatically kick-off the boot process for the application once the
      DOM has become ready.
       The initialization itself is scheduled on the actions queue which
      ensures that code-loading finishes before booting.
       If you are asynchronously loading code, you should call `deferReadiness()`
      to defer booting, and then call `advanceReadiness()` once all of your code
      has finished loading.
       @private
      @method waitForDOMReady
    */
    waitForDOMReady: function () {
      if (!this.$ || this.$.isReady) {
        _emberMetal.run.schedule('actions', this, 'domReady');
      } else {
        this.$().ready(_emberMetal.run.bind(this, 'domReady'));
      }
    },

    /**
      This is the autoboot flow:
       1. Boot the app by calling `this.boot()`
      2. Create an instance (or use the `__deprecatedInstance__` in globals mode)
      3. Boot the instance by calling `instance.boot()`
      4. Invoke the `App.ready()` callback
      5. Kick-off routing on the instance
       Ideally, this is all we would need to do:
       ```javascript
      _autoBoot() {
        this.boot().then(() => {
          let instance = (this._globalsMode) ? this.__deprecatedInstance__ : this.buildInstance();
          return instance.boot();
        }).then((instance) => {
          App.ready();
          instance.startRouting();
        });
      }
      ```
       Unfortunately, we cannot actually write this because we need to participate
      in the "synchronous" boot process. While the code above would work fine on
      the initial boot (i.e. DOM ready), when `App.reset()` is called, we need to
      boot a new instance synchronously (see the documentation on `_bootSync()`
      for details).
       Because of this restriction, the actual logic of this method is located
      inside `didBecomeReady()`.
       @private
      @method domReady
    */
    domReady: function () {
      if (this.isDestroyed) {
        return;
      }

      this._bootSync();

      // Continues to `didBecomeReady`
    },

    /**
      Use this to defer readiness until some condition is true.
       Example:
       ```javascript
      let App = Ember.Application.create();
       App.deferReadiness();
       // Ember.$ is a reference to the jQuery object/function
      Ember.$.getJSON('/auth-token', function(token) {
        App.token = token;
        App.advanceReadiness();
      });
      ```
       This allows you to perform asynchronous setup logic and defer
      booting your application until the setup has finished.
       However, if the setup requires a loading UI, it might be better
      to use the router for this purpose.
       @method deferReadiness
      @public
    */
    deferReadiness: function () {
      this._readinessDeferrals++;
    },

    /**
      Call `advanceReadiness` after any asynchronous setup logic has completed.
      Each call to `deferReadiness` must be matched by a call to `advanceReadiness`
      or the application will never become ready and routing will not begin.
       @method advanceReadiness
      @see {Ember.Application#deferReadiness}
      @public
    */
    advanceReadiness: function () {
      this._readinessDeferrals--;

      if (this._readinessDeferrals === 0) {
        _emberMetal.run.once(this, this.didBecomeReady);
      }
    },

    /**
      Initialize the application and return a promise that resolves with the `Ember.Application`
      object when the boot process is complete.
       Run any application initializers and run the application load hook. These hooks may
      choose to defer readiness. For example, an authentication hook might want to defer
      readiness until the auth token has been retrieved.
       By default, this method is called automatically on "DOM ready"; however, if autoboot
      is disabled, this is automatically called when the first application instance is
      created via `visit`.
       @private
      @method boot
      @return {Promise<Ember.Application,Error>}
    */
    boot: function () {
      if (this._bootPromise) {
        return this._bootPromise;
      }

      try {
        this._bootSync();
      } catch (_) {
        // Ignore th error: in the asynchronous boot path, the error is already reflected
        // in the promise rejection
      }

      return this._bootPromise;
    },

    /**
      Unfortunately, a lot of existing code assumes the booting process is
      "synchronous". Specifically, a lot of tests assumes the last call to
      `app.advanceReadiness()` or `app.reset()` will result in the app being
      fully-booted when the current runloop completes.
       We would like new code (like the `visit` API) to stop making this assumption,
      so we created the asynchronous version above that returns a promise. But until
      we have migrated all the code, we would have to expose this method for use
      *internally* in places where we need to boot an app "synchronously".
       @private
    */
    _bootSync: function () {
      if (this._booted) {
        return;
      }

      // Even though this returns synchronously, we still need to make sure the
      // boot promise exists for book-keeping purposes: if anything went wrong in
      // the boot process, we need to store the error as a rejection on the boot
      // promise so that a future caller of `boot()` can tell what failed.
      var defer = this._bootResolver = new _emberRuntime.RSVP.defer();
      this._bootPromise = defer.promise;

      try {
        this.runInitializers();
        _emberRuntime.runLoadHooks('application', this);
        this.advanceReadiness();
        // Continues to `didBecomeReady`
      } catch (error) {
        // For the asynchronous boot path
        defer.reject(error);

        // For the synchronous boot path
        throw error;
      }
    },

    /**
      Reset the application. This is typically used only in tests. It cleans up
      the application in the following order:
       1. Deactivate existing routes
      2. Destroy all objects in the container
      3. Create a new application container
      4. Re-route to the existing url
       Typical Example:
       ```javascript
      let App;
       run(function() {
        App = Ember.Application.create();
      });
       module('acceptance test', {
        setup: function() {
          App.reset();
        }
      });
       test('first test', function() {
        // App is freshly reset
      });
       test('second test', function() {
        // App is again freshly reset
      });
      ```
       Advanced Example:
       Occasionally you may want to prevent the app from initializing during
      setup. This could enable extra configuration, or enable asserting prior
      to the app becoming ready.
       ```javascript
      let App;
       run(function() {
        App = Ember.Application.create();
      });
       module('acceptance test', {
        setup: function() {
          run(function() {
            App.reset();
            App.deferReadiness();
          });
        }
      });
       test('first test', function() {
        ok(true, 'something before app is initialized');
         run(function() {
          App.advanceReadiness();
        });
         ok(true, 'something after app is initialized');
      });
      ```
       @method reset
      @public
    */
    reset: function () {

      var instance = this.__deprecatedInstance__;

      this._readinessDeferrals = 1;
      this._bootPromise = null;
      this._bootResolver = null;
      this._booted = false;

      function handleReset() {
        _emberMetal.run(instance, 'destroy');
        this._buildDeprecatedInstance();
        _emberMetal.run.schedule('actions', this, '_bootSync');
      }

      _emberMetal.run.join(this, handleReset);
    },

    /**
      @private
      @method didBecomeReady
    */
    didBecomeReady: function () {
      try {
        // TODO: Is this still needed for _globalsMode = false?
        if (!_emberMetal.isTesting()) {
          // Eagerly name all classes that are already loaded
          _emberRuntime.Namespace.processAll();
          _emberRuntime.setNamespaceSearchDisabled(true);
        }

        // See documentation on `_autoboot()` for details
        if (this.autoboot) {
          var instance = undefined;

          if (this._globalsMode) {
            // If we already have the __deprecatedInstance__ lying around, boot it to
            // avoid unnecessary work
            instance = this.__deprecatedInstance__;
          } else {
            // Otherwise, build an instance and boot it. This is currently unreachable,
            // because we forced _globalsMode to === autoboot; but having this branch
            // allows us to locally toggle that flag for weeding out legacy globals mode
            // dependencies independently
            instance = this.buildInstance();
          }

          instance._bootSync();

          // TODO: App.ready() is not called when autoboot is disabled, is this correct?
          this.ready();

          instance.startRouting();
        }

        // For the asynchronous boot path
        this._bootResolver.resolve(this);

        // For the synchronous boot path
        this._booted = true;
      } catch (error) {
        // For the asynchronous boot path
        this._bootResolver.reject(error);

        // For the synchronous boot path
        throw error;
      }
    },

    /**
      Called when the Application has become ready, immediately before routing
      begins. The call will be delayed until the DOM has become ready.
       @event ready
      @public
    */
    ready: function () {
      return this;
    },

    // This method must be moved to the application instance object
    willDestroy: function () {
      this._super.apply(this, arguments);
      _emberRuntime.setNamespaceSearchDisabled(false);
      this._booted = false;
      this._bootPromise = null;
      this._bootResolver = null;

      if (_emberRuntime._loaded.application === this) {
        _emberRuntime._loaded.application = undefined;
      }

      if (this._globalsMode && this.__deprecatedInstance__) {
        this.__deprecatedInstance__.destroy();
      }
    },

    /**
      Boot a new instance of `Ember.ApplicationInstance` for the current
      application and navigate it to the given `url`. Returns a `Promise` that
      resolves with the instance when the initial routing and rendering is
      complete, or rejects with any error that occured during the boot process.
       When `autoboot` is disabled, calling `visit` would first cause the
      application to boot, which runs the application initializers.
       This method also takes a hash of boot-time configuration options for
      customizing the instance's behavior. See the documentation on
      `Ember.ApplicationInstance.BootOptions` for details.
       `Ember.ApplicationInstance.BootOptions` is an interface class that exists
      purely to document the available options; you do not need to construct it
      manually. Simply pass a regular JavaScript object containing of the
      desired options:
       ```javascript
      MyApp.visit("/", { location: "none", rootElement: "#container" });
      ```
       ### Supported Scenarios
       While the `BootOptions` class exposes a large number of knobs, not all
      combinations of them are valid; certain incompatible combinations might
      result in unexpected behavior.
       For example, booting the instance in the full browser environment
      while specifying a foriegn `document` object (e.g. `{ isBrowser: true,
      document: iframe.contentDocument }`) does not work correctly today,
      largely due to Ember's jQuery dependency.
       Currently, there are three officially supported scenarios/configurations.
      Usages outside of these scenarios are not guaranteed to work, but please
      feel free to file bug reports documenting your experience and any issues
      you encountered to help expand support.
       #### Browser Applications (Manual Boot)
       The setup is largely similar to how Ember works out-of-the-box. Normally,
      Ember will boot a default instance for your Application on "DOM ready".
      However, you can customize this behavior by disabling `autoboot`.
       For example, this allows you to render a miniture demo of your application
      into a specific area on your marketing website:
       ```javascript
      import MyApp from 'my-app';
       $(function() {
        let App = MyApp.create({ autoboot: false });
         let options = {
          // Override the router's location adapter to prevent it from updating
          // the URL in the address bar
          location: 'none',
           // Override the default `rootElement` on the app to render into a
          // specific `div` on the page
          rootElement: '#demo'
        };
         // Start the app at the special demo URL
        App.visit('/demo', options);
      });
      ````
       Or perhaps you might want to boot two instances of your app on the same
      page for a split-screen multiplayer experience:
       ```javascript
      import MyApp from 'my-app';
       $(function() {
        let App = MyApp.create({ autoboot: false });
         let sessionId = MyApp.generateSessionID();
         let player1 = App.visit(`/matches/join?name=Player+1&session=${sessionId}`, { rootElement: '#left', location: 'none' });
        let player2 = App.visit(`/matches/join?name=Player+2&session=${sessionId}`, { rootElement: '#right', location: 'none' });
         Promise.all([player1, player2]).then(() => {
          // Both apps have completed the initial render
          $('#loading').fadeOut();
        });
      });
      ```
       Do note that each app instance maintains their own registry/container, so
      they will run in complete isolation by default.
       #### Server-Side Rendering (also known as FastBoot)
       This setup allows you to run your Ember app in a server environment using
      Node.js and render its content into static HTML for SEO purposes.
       ```javascript
      const HTMLSerializer = new SimpleDOM.HTMLSerializer(SimpleDOM.voidMap);
       function renderURL(url) {
        let dom = new SimpleDOM.Document();
        let rootElement = dom.body;
        let options = { isBrowser: false, document: dom, rootElement: rootElement };
         return MyApp.visit(options).then(instance => {
          try {
            return HTMLSerializer.serialize(rootElement.firstChild);
          } finally {
            instance.destroy();
          }
        });
      }
      ```
       In this scenario, because Ember does not have access to a global `document`
      object in the Node.js environment, you must provide one explicitly. In practice,
      in the non-browser environment, the stand-in `document` object only need to
      implement a limited subset of the full DOM API. The `SimpleDOM` library is known
      to work.
       Since there is no access to jQuery in the non-browser environment, you must also
      specify a DOM `Element` object in the same `document` for the `rootElement` option
      (as opposed to a selector string like `"body"`).
       See the documentation on the `isBrowser`, `document` and `rootElement` properties
      on `Ember.ApplicationInstance.BootOptions` for details.
       #### Server-Side Resource Discovery
       This setup allows you to run the routing layer of your Ember app in a server
      environment using Node.js and completely disable rendering. This allows you
      to simulate and discover the resources (i.e. AJAX requests) needed to fufill
      a given request and eagerly "push" these resources to the client.
       ```app/initializers/network-service.js
      import BrowserNetworkService from 'app/services/network/browser';
      import NodeNetworkService from 'app/services/network/node';
       // Inject a (hypothetical) service for abstracting all AJAX calls and use
      // the appropiate implementaion on the client/server. This also allows the
      // server to log all the AJAX calls made during a particular request and use
      // that for resource-discovery purpose.
       export function initialize(application) {
        if (window) { // browser
          application.register('service:network', BrowserNetworkService);
        } else { // node
          application.register('service:network', NodeNetworkService);
        }
         application.inject('route', 'network', 'service:network');
      };
       export default {
        name: 'network-service',
        initialize: initialize
      };
      ```
       ```app/routes/post.js
      import Ember from 'ember';
       // An example of how the (hypothetical) service is used in routes.
       export default Ember.Route.extend({
        model(params) {
          return this.network.fetch(`/api/posts/${params.post_id}.json`);
        },
         afterModel(post) {
          if (post.isExternalContent) {
            return this.network.fetch(`/api/external/?url=${post.externalURL}`);
          } else {
            return post;
          }
        }
      });
      ```
       ```javascript
      // Finally, put all the pieces together
       function discoverResourcesFor(url) {
        return MyApp.visit(url, { isBrowser: false, shouldRender: false }).then(instance => {
          let networkService = instance.lookup('service:network');
          return networkService.requests; // => { "/api/posts/123.json": "..." }
        });
      }
      ```
       @public
      @method visit
      @param url {String} The initial URL to navigate to
      @param options {Ember.ApplicationInstance.BootOptions}
      @return {Promise<Ember.ApplicationInstance, Error>}
    */
    visit: function (url, options) {
      var _this = this;

      return this.boot().then(function () {
        var instance = _this.buildInstance();

        return instance.boot(options).then(function () {
          return instance.visit(url);
        }).catch(function (error) {
          _emberMetal.run(instance, 'destroy');
          throw error;
        });
      });
    }
  });

  Object.defineProperty(Application.prototype, 'registry', {
    configurable: true,
    enumerable: false,
    get: function () {
      return _emberRuntime.buildFakeRegistryWithDeprecations(this, 'Application');
    }
  });

  Application.reopenClass({
    /**
      This creates a registry with the default Ember naming conventions.
       It also configures the registry:
       * registered views are created every time they are looked up (they are
        not singletons)
      * registered templates are not factories; the registered value is
        returned directly.
      * the router receives the application as its `namespace` property
      * all controllers receive the router as their `target` and `controllers`
        properties
      * all controllers receive the application as their `namespace` property
      * the application view receives the application controller as its
        `controller` property
      * the application view receives the application template as its
        `defaultTemplate` property
       @method buildRegistry
      @static
      @param {Ember.Application} namespace the application for which to
        build the registry
      @return {Ember.Registry} the built registry
      @private
    */
    buildRegistry: function (application) {
      var options = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

      var registry = this._super.apply(this, arguments);

      commonSetupRegistry(registry);

      _emberGlimmer.setupApplicationRegistry(registry);

      return registry;
    }
  });

  function commonSetupRegistry(registry) {
    registry.register('-view-registry:main', { create: function () {
        return _emberUtils.dictionary(null);
      } });

    registry.register('route:basic', _emberRouting.Route);
    registry.register('event_dispatcher:main', _emberViews.EventDispatcher);

    registry.injection('router:main', 'namespace', 'application:main');

    registry.register('location:auto', _emberRouting.AutoLocation);
    registry.register('location:hash', _emberRouting.HashLocation);
    registry.register('location:history', _emberRouting.HistoryLocation);
    registry.register('location:none', _emberRouting.NoneLocation);

    registry.register(_container.privatize(_templateObject), _emberRouting.BucketCache);
  }

  function registerLibraries() {
    if (!librariesRegistered) {
      librariesRegistered = true;

      if (_emberEnvironment.environment.hasDOM && typeof _emberViews.jQuery === 'function') {
        _emberMetal.libraries.registerCoreLibrary('jQuery', _emberViews.jQuery().jquery);
      }
    }
  }

  function logLibraryVersions() {
    if (_emberEnvironment.ENV.LOG_VERSION) {
      // we only need to see this once per Application#init
      _emberEnvironment.ENV.LOG_VERSION = false;
      var libs = _emberMetal.libraries._registry;

      var nameLengths = libs.map(function (item) {
        return _emberMetal.get(item, 'name.length');
      });

      var maxNameLength = Math.max.apply(this, nameLengths);

      for (var i = 0; i < libs.length; i++) {
        var lib = libs[i];
        var spaces = new Array(maxNameLength - lib.name.length + 1).join(' ');
      }
    }
  }

  exports.default = Application;
});
enifed('ember-application/system/engine-instance', ['exports', 'ember-utils', 'ember-runtime', 'ember-metal', 'container', 'ember-application/system/engine-parent'], function (exports, _emberUtils, _emberRuntime, _emberMetal, _container, _emberApplicationSystemEngineParent) {
  /**
  @module ember
  @submodule ember-application
  */

  'use strict';

  var _templateObject = babelHelpers.taggedTemplateLiteralLoose(['-bucket-cache:main'], ['-bucket-cache:main']);

  /**
    The `EngineInstance` encapsulates all of the stateful aspects of a
    running `Engine`.
  
    @public
    @class Ember.EngineInstance
    @extends Ember.Object
    @uses RegistryProxyMixin
    @uses ContainerProxyMixin
  */

  var EngineInstance = _emberRuntime.Object.extend(_emberRuntime.RegistryProxyMixin, _emberRuntime.ContainerProxyMixin, {
    /**
      The base `Engine` for which this is an instance.
       @property {Ember.Engine} engine
      @private
    */
    base: null,

    init: function () {
      this._super.apply(this, arguments);

      _emberUtils.guidFor(this);

      var base = this.base;

      if (!base) {
        base = this.application;
        this.base = base;
      }

      // Create a per-instance registry that will use the application's registry
      // as a fallback for resolving registrations.
      var registry = this.__registry__ = new _container.Registry({
        fallback: base.__registry__
      });

      // Create a per-instance container from the instance's registry
      this.__container__ = registry.container({ owner: this });

      this._booted = false;
    },

    /**
      Initialize the `Ember.EngineInstance` and return a promise that resolves
      with the instance itself when the boot process is complete.
       The primary task here is to run any registered instance initializers.
       See the documentation on `BootOptions` for the options it takes.
       @private
      @method boot
      @param options {Object}
      @return {Promise<Ember.EngineInstance,Error>}
    */
    boot: function (options) {
      var _this = this;

      if (this._bootPromise) {
        return this._bootPromise;
      }

      this._bootPromise = new _emberRuntime.RSVP.Promise(function (resolve) {
        return resolve(_this._bootSync(options));
      });

      return this._bootPromise;
    },

    /**
      Unfortunately, a lot of existing code assumes booting an instance is
      synchronous – specifically, a lot of tests assume the last call to
      `app.advanceReadiness()` or `app.reset()` will result in a new instance
      being fully-booted when the current runloop completes.
       We would like new code (like the `visit` API) to stop making this
      assumption, so we created the asynchronous version above that returns a
      promise. But until we have migrated all the code, we would have to expose
      this method for use *internally* in places where we need to boot an instance
      synchronously.
       @private
    */
    _bootSync: function (options) {
      if (this._booted) {
        return this;
      }

      this.cloneParentDependencies();

      this.setupRegistry(options);

      this.base.runInstanceInitializers(this);

      this._booted = true;

      return this;
    },

    setupRegistry: function () {
      var options = arguments.length <= 0 || arguments[0] === undefined ? this.__container__.lookup('-environment:main') : arguments[0];

      this.constructor.setupRegistry(this.__registry__, options);
    },

    /**
     Unregister a factory.
      Overrides `RegistryProxy#unregister` in order to clear any cached instances
     of the unregistered factory.
      @public
     @method unregister
     @param {String} fullName
     */
    unregister: function (fullName) {
      this.__container__.reset(fullName);
      this._super.apply(this, arguments);
    },

    /**
      @private
    */
    willDestroy: function () {
      this._super.apply(this, arguments);
      _emberMetal.run(this.__container__, 'destroy');
    },

    /**
      Build a new `Ember.EngineInstance` that's a child of this instance.
       Engines must be registered by name with their parent engine
      (or application).
       @private
      @method buildChildEngineInstance
      @param name {String} the registered name of the engine.
      @param options {Object} options provided to the engine instance.
      @return {Ember.EngineInstance,Error}
    */
    buildChildEngineInstance: function (name) {
      var options = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

      var Engine = this.lookup('engine:' + name);

      if (!Engine) {
        throw new _emberMetal.Error('You attempted to mount the engine \'' + name + '\', but it is not registered with its parent.');
      }

      var engineInstance = Engine.buildInstance(options);

      _emberApplicationSystemEngineParent.setEngineParent(engineInstance, this);

      return engineInstance;
    },

    /**
      Clone dependencies shared between an engine instance and its parent.
       @private
      @method cloneParentDependencies
    */
    cloneParentDependencies: function () {
      var _this2 = this;

      var parent = _emberApplicationSystemEngineParent.getEngineParent(this);

      var registrations = ['route:basic', 'event_dispatcher:main', 'service:-routing', 'service:-glimmer-environment'];

      registrations.forEach(function (key) {
        return _this2.register(key, parent.resolveRegistration(key));
      });

      var env = parent.lookup('-environment:main');
      this.register('-environment:main', env, { instantiate: false });

      var singletons = ['router:main', _container.privatize(_templateObject), '-view-registry:main', 'renderer:-' + (env.isInteractive ? 'dom' : 'inert')];

      singletons.forEach(function (key) {
        return _this2.register(key, parent.lookup(key), { instantiate: false });
      });

      this.inject('view', '_environment', '-environment:main');
      this.inject('route', '_environment', '-environment:main');
    }
  });

  EngineInstance.reopenClass({
    /**
     @private
     @method setupRegistry
     @param {Registry} registry
     @param {BootOptions} options
     */
    setupRegistry: function (registry, options) {
      // when no options/environment is present, do nothing
      if (!options) {
        return;
      }

      registry.injection('view', '_environment', '-environment:main');
      registry.injection('route', '_environment', '-environment:main');

      if (options.isInteractive) {
        registry.injection('view', 'renderer', 'renderer:-dom');
        registry.injection('component', 'renderer', 'renderer:-dom');
      } else {
        registry.injection('view', 'renderer', 'renderer:-inert');
        registry.injection('component', 'renderer', 'renderer:-inert');
      }
    }
  });

  exports.default = EngineInstance;
});
enifed('ember-application/system/engine-parent', ['exports', 'ember-utils'], function (exports, _emberUtils) {
  'use strict';

  exports.getEngineParent = getEngineParent;
  exports.setEngineParent = setEngineParent;
  var ENGINE_PARENT = _emberUtils.symbol('ENGINE_PARENT');

  exports.ENGINE_PARENT = ENGINE_PARENT;
  /**
    `getEngineParent` retrieves an engine instance's parent instance.
  
    @method getEngineParent
    @param {EngineInstance} engine An engine instance.
    @return {EngineInstance} The parent engine instance.
    @for Ember
    @public
  */

  function getEngineParent(engine) {
    return engine[ENGINE_PARENT];
  }

  /**
    `setEngineParent` sets an engine instance's parent instance.
  
    @method setEngineParent
    @param {EngineInstance} engine An engine instance.
    @param {EngineInstance} parent The parent engine instance.
    @private
  */

  function setEngineParent(engine, parent) {
    engine[ENGINE_PARENT] = parent;
  }
});
enifed('ember-application/system/engine', ['exports', 'ember-utils', 'ember-runtime', 'container', 'dag-map', 'ember-metal', 'ember-application/system/resolver', 'ember-application/system/engine-instance', 'ember-routing', 'ember-extension-support', 'ember-views', 'ember-glimmer'], function (exports, _emberUtils, _emberRuntime, _container, _dagMap, _emberMetal, _emberApplicationSystemResolver, _emberApplicationSystemEngineInstance, _emberRouting, _emberExtensionSupport, _emberViews, _emberGlimmer) {
  /**
  @module ember
  @submodule ember-application
  */
  'use strict';

  var _templateObject = babelHelpers.taggedTemplateLiteralLoose(['-bucket-cache:main'], ['-bucket-cache:main']);

  function props(obj) {
    var properties = [];

    for (var key in obj) {
      properties.push(key);
    }

    return properties;
  }

  /**
    The `Engine` class contains core functionality for both applications and
    engines.
  
    Each engine manages a registry that's used for dependency injection and
    exposed through `RegistryProxy`.
  
    Engines also manage initializers and instance initializers.
  
    Engines can spawn `EngineInstance` instances via `buildInstance()`.
  
    @class Engine
    @namespace Ember
    @extends Ember.Namespace
    @uses RegistryProxy
    @public
  */
  var Engine = _emberRuntime.Namespace.extend(_emberRuntime.RegistryProxyMixin, {
    init: function () {
      this._super.apply(this, arguments);

      this.buildRegistry();
    },

    /**
      A private flag indicating whether an engine's initializers have run yet.
       @private
      @property _initializersRan
    */
    _initializersRan: false,

    /**
      Ensure that initializers are run once, and only once, per engine.
       @private
      @method ensureInitializers
    */
    ensureInitializers: function () {
      if (!this._initializersRan) {
        this.runInitializers();
        this._initializersRan = true;
      }
    },

    /**
      Create an EngineInstance for this engine.
       @private
      @method buildInstance
      @return {Ember.EngineInstance} the engine instance
    */
    buildInstance: function () {
      var options = arguments.length <= 0 || arguments[0] === undefined ? {} : arguments[0];

      this.ensureInitializers();
      options.base = this;
      return _emberApplicationSystemEngineInstance.default.create(options);
    },

    /**
      Build and configure the registry for the current engine.
       @private
      @method buildRegistry
      @return {Ember.Registry} the configured registry
    */
    buildRegistry: function () {
      var registry = this.__registry__ = this.constructor.buildRegistry(this);

      return registry;
    },

    /**
      @private
      @method initializer
    */
    initializer: function (options) {
      this.constructor.initializer(options);
    },

    /**
      @private
      @method instanceInitializer
    */
    instanceInitializer: function (options) {
      this.constructor.instanceInitializer(options);
    },

    /**
      @private
      @method runInitializers
    */
    runInitializers: function () {
      var _this = this;

      this._runInitializer('initializers', function (name, initializer) {
        if (initializer.initialize.length === 2) {

          initializer.initialize(_this.__registry__, _this);
        } else {
          initializer.initialize(_this);
        }
      });
    },

    /**
      @private
      @since 1.12.0
      @method runInstanceInitializers
    */
    runInstanceInitializers: function (instance) {
      this._runInitializer('instanceInitializers', function (name, initializer) {
        initializer.initialize(instance);
      });
    },

    _runInitializer: function (bucketName, cb) {
      var initializersByName = _emberMetal.get(this.constructor, bucketName);
      var initializers = props(initializersByName);
      var graph = new _dagMap.default();
      var initializer = undefined;

      for (var i = 0; i < initializers.length; i++) {
        initializer = initializersByName[initializers[i]];
        graph.add(initializer.name, initializer, initializer.before, initializer.after);
      }

      graph.topsort(cb);
    }
  });

  Engine.reopenClass({
    initializers: new _emberUtils.EmptyObject(),
    instanceInitializers: new _emberUtils.EmptyObject(),

    /**
      The goal of initializers should be to register dependencies and injections.
      This phase runs once. Because these initializers may load code, they are
      allowed to defer application readiness and advance it. If you need to access
      the container or store you should use an InstanceInitializer that will be run
      after all initializers and therefore after all code is loaded and the app is
      ready.
       Initializer receives an object which has the following attributes:
      `name`, `before`, `after`, `initialize`. The only required attribute is
      `initialize`, all others are optional.
       * `name` allows you to specify under which name the initializer is registered.
      This must be a unique name, as trying to register two initializers with the
      same name will result in an error.
       ```javascript
      Ember.Application.initializer({
        name: 'namedInitializer',
         initialize: function(application) {
          Ember.debug('Running namedInitializer!');
        }
      });
      ```
       * `before` and `after` are used to ensure that this initializer is ran prior
      or after the one identified by the value. This value can be a single string
      or an array of strings, referencing the `name` of other initializers.
       An example of ordering initializers, we create an initializer named `first`:
       ```javascript
      Ember.Application.initializer({
        name: 'first',
         initialize: function(application) {
          Ember.debug('First initializer!');
        }
      });
       // DEBUG: First initializer!
      ```
       We add another initializer named `second`, specifying that it should run
      after the initializer named `first`:
       ```javascript
      Ember.Application.initializer({
        name: 'second',
        after: 'first',
         initialize: function(application) {
          Ember.debug('Second initializer!');
        }
      });
       // DEBUG: First initializer!
      // DEBUG: Second initializer!
      ```
       Afterwards we add a further initializer named `pre`, this time specifying
      that it should run before the initializer named `first`:
       ```javascript
      Ember.Application.initializer({
        name: 'pre',
        before: 'first',
         initialize: function(application) {
          Ember.debug('Pre initializer!');
        }
      });
       // DEBUG: Pre initializer!
      // DEBUG: First initializer!
      // DEBUG: Second initializer!
      ```
       Finally we add an initializer named `post`, specifying it should run after
      both the `first` and the `second` initializers:
       ```javascript
      Ember.Application.initializer({
        name: 'post',
        after: ['first', 'second'],
         initialize: function(application) {
          Ember.debug('Post initializer!');
        }
      });
       // DEBUG: Pre initializer!
      // DEBUG: First initializer!
      // DEBUG: Second initializer!
      // DEBUG: Post initializer!
      ```
       * `initialize` is a callback function that receives one argument,
        `application`, on which you can operate.
       Example of using `application` to register an adapter:
       ```javascript
      Ember.Application.initializer({
        name: 'api-adapter',
         initialize: function(application) {
          application.register('api-adapter:main', ApiAdapter);
        }
      });
      ```
       @method initializer
      @param initializer {Object}
      @public
    */

    initializer: buildInitializerMethod('initializers', 'initializer'),

    /**
      Instance initializers run after all initializers have run. Because
      instance initializers run after the app is fully set up. We have access
      to the store, container, and other items. However, these initializers run
      after code has loaded and are not allowed to defer readiness.
       Instance initializer receives an object which has the following attributes:
      `name`, `before`, `after`, `initialize`. The only required attribute is
      `initialize`, all others are optional.
       * `name` allows you to specify under which name the instanceInitializer is
      registered. This must be a unique name, as trying to register two
      instanceInitializer with the same name will result in an error.
       ```javascript
      Ember.Application.instanceInitializer({
        name: 'namedinstanceInitializer',
         initialize: function(application) {
          Ember.debug('Running namedInitializer!');
        }
      });
      ```
       * `before` and `after` are used to ensure that this initializer is ran prior
      or after the one identified by the value. This value can be a single string
      or an array of strings, referencing the `name` of other initializers.
       * See Ember.Application.initializer for discussion on the usage of before
      and after.
       Example instanceInitializer to preload data into the store.
       ```javascript
      Ember.Application.initializer({
        name: 'preload-data',
         initialize: function(application) {
          var userConfig, userConfigEncoded, store;
          // We have a HTML escaped JSON representation of the user's basic
          // configuration generated server side and stored in the DOM of the main
          // index.html file. This allows the app to have access to a set of data
          // without making any additional remote calls. Good for basic data that is
          // needed for immediate rendering of the page. Keep in mind, this data,
          // like all local models and data can be manipulated by the user, so it
          // should not be relied upon for security or authorization.
          //
          // Grab the encoded data from the meta tag
          userConfigEncoded = Ember.$('head meta[name=app-user-config]').attr('content');
          // Unescape the text, then parse the resulting JSON into a real object
          userConfig = JSON.parse(unescape(userConfigEncoded));
          // Lookup the store
          store = application.lookup('service:store');
          // Push the encoded JSON into the store
          store.pushPayload(userConfig);
        }
      });
      ```
       @method instanceInitializer
      @param instanceInitializer
      @public
    */
    instanceInitializer: buildInitializerMethod('instanceInitializers', 'instance initializer'),

    /**
      This creates a registry with the default Ember naming conventions.
       It also configures the registry:
       * registered views are created every time they are looked up (they are
        not singletons)
      * registered templates are not factories; the registered value is
        returned directly.
      * the router receives the application as its `namespace` property
      * all controllers receive the router as their `target` and `controllers`
        properties
      * all controllers receive the application as their `namespace` property
      * the application view receives the application controller as its
        `controller` property
      * the application view receives the application template as its
        `defaultTemplate` property
       @method buildRegistry
      @static
      @param {Ember.Application} namespace the application for which to
        build the registry
      @return {Ember.Registry} the built registry
      @private
    */
    buildRegistry: function (namespace) {
      var options = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

      var registry = new _container.Registry({
        resolver: resolverFor(namespace)
      });

      registry.set = _emberMetal.set;

      registry.register('application:main', namespace, { instantiate: false });

      commonSetupRegistry(registry);
      _emberGlimmer.setupEngineRegistry(registry);

      return registry;
    },

    /**
      Set this to provide an alternate class to `Ember.DefaultResolver`
        @deprecated Use 'Resolver' instead
      @property resolver
      @public
    */
    resolver: null,

    /**
      Set this to provide an alternate class to `Ember.DefaultResolver`
       @property resolver
      @public
    */
    Resolver: null
  });

  /**
    This function defines the default lookup rules for container lookups:
  
    * templates are looked up on `Ember.TEMPLATES`
    * other names are looked up on the application after classifying the name.
      For example, `controller:post` looks up `App.PostController` by default.
    * if the default lookup fails, look for registered classes on the container
  
    This allows the application to register default injections in the container
    that could be overridden by the normal naming convention.
  
    @private
    @method resolverFor
    @param {Ember.Namespace} namespace the namespace to look for classes
    @return {*} the resolved value for a given lookup
  */
  function resolverFor(namespace) {
    var ResolverClass = namespace.get('Resolver') || _emberApplicationSystemResolver.default;

    return ResolverClass.create({
      namespace: namespace
    });
  }

  function buildInitializerMethod(bucketName, humanName) {
    return function (initializer) {
      // If this is the first initializer being added to a subclass, we are going to reopen the class
      // to make sure we have a new `initializers` object, which extends from the parent class' using
      // prototypal inheritance. Without this, attempting to add initializers to the subclass would
      // pollute the parent class as well as other subclasses.
      if (this.superclass[bucketName] !== undefined && this.superclass[bucketName] === this[bucketName]) {
        var attrs = {};
        attrs[bucketName] = Object.create(this[bucketName]);
        this.reopenClass(attrs);
      }

      this[bucketName][initializer.name] = initializer;
    };
  }

  function commonSetupRegistry(registry) {
    registry.optionsForType('component', { singleton: false });
    registry.optionsForType('view', { singleton: false });

    registry.register('controller:basic', _emberRuntime.Controller, { instantiate: false });

    registry.injection('view', '_viewRegistry', '-view-registry:main');
    registry.injection('renderer', '_viewRegistry', '-view-registry:main');
    registry.injection('event_dispatcher:main', '_viewRegistry', '-view-registry:main');

    registry.injection('route', '_topLevelViewTemplate', 'template:-outlet');

    registry.injection('view:-outlet', 'namespace', 'application:main');

    registry.injection('controller', 'target', 'router:main');
    registry.injection('controller', 'namespace', 'application:main');

    registry.injection('router', '_bucketCache', _container.privatize(_templateObject));
    registry.injection('route', '_bucketCache', _container.privatize(_templateObject));

    registry.injection('route', 'router', 'router:main');

    // Register the routing service...
    registry.register('service:-routing', _emberRouting.RoutingService);
    // Then inject the app router into it
    registry.injection('service:-routing', 'router', 'router:main');

    // DEBUGGING
    registry.register('resolver-for-debugging:main', registry.resolver, { instantiate: false });
    registry.injection('container-debug-adapter:main', 'resolver', 'resolver-for-debugging:main');
    registry.injection('data-adapter:main', 'containerDebugAdapter', 'container-debug-adapter:main');
    // Custom resolver authors may want to register their own ContainerDebugAdapter with this key

    registry.register('container-debug-adapter:main', _emberExtensionSupport.ContainerDebugAdapter);

    registry.register('component-lookup:main', _emberViews.ComponentLookup);
  }

  exports.default = Engine;
});
enifed('ember-application/system/resolver', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime', 'ember-application/utils/validate-type', 'ember-glimmer'], function (exports, _emberUtils, _emberMetal, _emberRuntime, _emberApplicationUtilsValidateType, _emberGlimmer) {
  /**
  @module ember
  @submodule ember-application
  */

  'use strict';

  var Resolver = _emberRuntime.Object.extend({
    /*
      This will be set to the Application instance when it is
      created.
       @property namespace
    */
    namespace: null,
    normalize: null, // required
    resolve: null, // required
    parseName: null, // required
    lookupDescription: null, // required
    makeToString: null, // required
    resolveOther: null, // required
    _logLookup: null // required
  });

  exports.Resolver = Resolver;
  /**
    The DefaultResolver defines the default lookup rules to resolve
    container lookups before consulting the container for registered
    items:
  
    * templates are looked up on `Ember.TEMPLATES`
    * other names are looked up on the application after converting
      the name. For example, `controller:post` looks up
      `App.PostController` by default.
    * there are some nuances (see examples below)
  
    ### How Resolving Works
  
    The container calls this object's `resolve` method with the
    `fullName` argument.
  
    It first parses the fullName into an object using `parseName`.
  
    Then it checks for the presence of a type-specific instance
    method of the form `resolve[Type]` and calls it if it exists.
    For example if it was resolving 'template:post', it would call
    the `resolveTemplate` method.
  
    Its last resort is to call the `resolveOther` method.
  
    The methods of this object are designed to be easy to override
    in a subclass. For example, you could enhance how a template
    is resolved like so:
  
    ```javascript
    App = Ember.Application.create({
      Resolver: Ember.DefaultResolver.extend({
        resolveTemplate: function(parsedName) {
          let resolvedTemplate = this._super(parsedName);
          if (resolvedTemplate) { return resolvedTemplate; }
          return Ember.TEMPLATES['not_found'];
        }
      })
    });
    ```
  
    Some examples of how names are resolved:
  
    ```
    'template:post'           //=> Ember.TEMPLATES['post']
    'template:posts/byline'   //=> Ember.TEMPLATES['posts/byline']
    'template:posts.byline'   //=> Ember.TEMPLATES['posts/byline']
    'template:blogPost'       //=> Ember.TEMPLATES['blogPost']
                              //   OR
                              //   Ember.TEMPLATES['blog_post']
    'controller:post'         //=> App.PostController
    'controller:posts.index'  //=> App.PostsIndexController
    'controller:blog/post'    //=> Blog.PostController
    'controller:basic'        //=> Ember.Controller
    'route:post'              //=> App.PostRoute
    'route:posts.index'       //=> App.PostsIndexRoute
    'route:blog/post'         //=> Blog.PostRoute
    'route:basic'             //=> Ember.Route
    'view:post'               //=> App.PostView
    'view:posts.index'        //=> App.PostsIndexView
    'view:blog/post'          //=> Blog.PostView
    'view:basic'              //=> Ember.View
    'foo:post'                //=> App.PostFoo
    'model:post'              //=> App.Post
    ```
  
    @class DefaultResolver
    @namespace Ember
    @extends Ember.Object
    @public
  */

  exports.default = _emberRuntime.Object.extend({
    /**
      This will be set to the Application instance when it is
      created.
       @property namespace
      @public
    */
    namespace: null,

    init: function () {
      this._parseNameCache = _emberUtils.dictionary(null);
    },
    normalize: function (fullName) {
      var _fullName$split = fullName.split(':', 2);

      var type = _fullName$split[0];
      var name = _fullName$split[1];

      if (type !== 'template') {
        var result = name;

        if (result.indexOf('.') > -1) {
          result = result.replace(/\.(.)/g, function (m) {
            return m.charAt(1).toUpperCase();
          });
        }

        if (name.indexOf('_') > -1) {
          result = result.replace(/_(.)/g, function (m) {
            return m.charAt(1).toUpperCase();
          });
        }

        if (name.indexOf('-') > -1) {
          result = result.replace(/-(.)/g, function (m) {
            return m.charAt(1).toUpperCase();
          });
        }

        return type + ':' + result;
      } else {
        return fullName;
      }
    },

    /**
      This method is called via the container's resolver method.
      It parses the provided `fullName` and then looks up and
      returns the appropriate template or class.
       @method resolve
      @param {String} fullName the lookup string
      @return {Object} the resolved factory
      @public
    */
    resolve: function (fullName) {
      var parsedName = this.parseName(fullName);
      var resolveMethodName = parsedName.resolveMethodName;
      var resolved;

      if (this[resolveMethodName]) {
        resolved = this[resolveMethodName](parsedName);
      }

      resolved = resolved || this.resolveOther(parsedName);

      if (parsedName.root && parsedName.root.LOG_RESOLVER) {
        this._logLookup(resolved, parsedName);
      }

      if (resolved) {
        _emberApplicationUtilsValidateType.default(resolved, parsedName);
      }

      return resolved;
    },

    /**
      Convert the string name of the form 'type:name' to
      a Javascript object with the parsed aspects of the name
      broken out.
       @param {String} fullName the lookup string
      @method parseName
      @protected
    */

    parseName: function (fullName) {
      return this._parseNameCache[fullName] || (this._parseNameCache[fullName] = this._parseName(fullName));
    },

    _parseName: function (fullName) {
      var _fullName$split2 = fullName.split(':');

      var type = _fullName$split2[0];
      var fullNameWithoutType = _fullName$split2[1];

      var name = fullNameWithoutType;
      var namespace = _emberMetal.get(this, 'namespace');
      var root = namespace;
      var lastSlashIndex = name.lastIndexOf('/');
      var dirname = lastSlashIndex !== -1 ? name.slice(0, lastSlashIndex) : null;

      if (type !== 'template' && lastSlashIndex !== -1) {
        var parts = name.split('/');
        name = parts[parts.length - 1];
        var namespaceName = _emberRuntime.String.capitalize(parts.slice(0, -1).join('.'));
        root = _emberRuntime.Namespace.byName(namespaceName);
      }

      var resolveMethodName = fullNameWithoutType === 'main' ? 'Main' : _emberRuntime.String.classify(type);

      if (!(name && type)) {
        throw new TypeError('Invalid fullName: `' + fullName + '`, must be of the form `type:name` ');
      }

      return {
        fullName: fullName,
        type: type,
        fullNameWithoutType: fullNameWithoutType,
        dirname: dirname,
        name: name,
        root: root,
        resolveMethodName: 'resolve' + resolveMethodName
      };
    },

    /**
      Returns a human-readable description for a fullName. Used by the
      Application namespace in assertions to describe the
      precise name of the class that Ember is looking for, rather than
      container keys.
       @param {String} fullName the lookup string
      @method lookupDescription
      @protected
    */
    lookupDescription: function (fullName) {
      var parsedName = this.parseName(fullName);
      var description = undefined;

      if (parsedName.type === 'template') {
        return 'template at ' + parsedName.fullNameWithoutType.replace(/\./g, '/');
      }

      description = parsedName.root + '.' + _emberRuntime.String.classify(parsedName.name).replace(/\./g, '');

      if (parsedName.type !== 'model') {
        description += _emberRuntime.String.classify(parsedName.type);
      }

      return description;
    },

    makeToString: function (factory, fullName) {
      return factory.toString();
    },

    /**
      Given a parseName object (output from `parseName`), apply
      the conventions expected by `Ember.Router`
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method useRouterNaming
      @protected
    */
    useRouterNaming: function (parsedName) {
      parsedName.name = parsedName.name.replace(/\./g, '_');
      if (parsedName.name === 'basic') {
        parsedName.name = '';
      }
    },
    /**
      Look up the template in Ember.TEMPLATES
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method resolveTemplate
      @protected
    */
    resolveTemplate: function (parsedName) {
      var templateName = parsedName.fullNameWithoutType.replace(/\./g, '/');

      return _emberGlimmer.getTemplate(templateName) || _emberGlimmer.getTemplate(_emberRuntime.String.decamelize(templateName));
    },

    /**
      Lookup the view using `resolveOther`
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method resolveView
      @protected
    */
    resolveView: function (parsedName) {
      this.useRouterNaming(parsedName);
      return this.resolveOther(parsedName);
    },

    /**
      Lookup the controller using `resolveOther`
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method resolveController
      @protected
    */
    resolveController: function (parsedName) {
      this.useRouterNaming(parsedName);
      return this.resolveOther(parsedName);
    },
    /**
      Lookup the route using `resolveOther`
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method resolveRoute
      @protected
    */
    resolveRoute: function (parsedName) {
      this.useRouterNaming(parsedName);
      return this.resolveOther(parsedName);
    },

    /**
      Lookup the model on the Application namespace
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method resolveModel
      @protected
    */
    resolveModel: function (parsedName) {
      var className = _emberRuntime.String.classify(parsedName.name);
      var factory = _emberMetal.get(parsedName.root, className);

      return factory;
    },
    /**
      Look up the specified object (from parsedName) on the appropriate
      namespace (usually on the Application)
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method resolveHelper
      @protected
    */
    resolveHelper: function (parsedName) {
      return this.resolveOther(parsedName);
    },
    /**
      Look up the specified object (from parsedName) on the appropriate
      namespace (usually on the Application)
       @param {Object} parsedName a parseName object with the parsed
        fullName lookup string
      @method resolveOther
      @protected
    */
    resolveOther: function (parsedName) {
      var className = _emberRuntime.String.classify(parsedName.name) + _emberRuntime.String.classify(parsedName.type);
      var factory = _emberMetal.get(parsedName.root, className);
      return factory;
    },

    resolveMain: function (parsedName) {
      var className = _emberRuntime.String.classify(parsedName.type);
      return _emberMetal.get(parsedName.root, className);
    },

    /**
     @method _logLookup
     @param {Boolean} found
     @param {Object} parsedName
     @private
    */
    _logLookup: function (found, parsedName) {
      var symbol = undefined,
          padding = undefined;

      if (found) {
        symbol = '[✓]';
      } else {
        symbol = '[ ]';
      }

      if (parsedName.fullName.length > 60) {
        padding = '.';
      } else {
        padding = new Array(60 - parsedName.fullName.length).join('.');
      }
    },

    /**
     Used to iterate all items of a given type.
      @method knownForType
     @param {String} type the type to search for
     @private
     */
    knownForType: function (type) {
      var namespace = _emberMetal.get(this, 'namespace');
      var suffix = _emberRuntime.String.classify(type);
      var typeRegexp = new RegExp(suffix + '$');

      var known = _emberUtils.dictionary(null);
      var knownKeys = Object.keys(namespace);
      for (var index = 0; index < knownKeys.length; index++) {
        var _name = knownKeys[index];

        if (typeRegexp.test(_name)) {
          var containerName = this.translateToContainerFullname(type, _name);

          known[containerName] = true;
        }
      }

      return known;
    },

    /**
     Converts provided name from the backing namespace into a container lookup name.
      Examples:
      App.FooBarHelper -> helper:foo-bar
     App.THelper -> helper:t
      @method translateToContainerFullname
     @param {String} type
     @param {String} name
     @private
     */

    translateToContainerFullname: function (type, name) {
      var suffix = _emberRuntime.String.classify(type);
      var namePrefix = name.slice(0, suffix.length * -1);
      var dasherizedName = _emberRuntime.String.dasherize(namePrefix);

      return type + ':' + dasherizedName;
    }
  });
});
enifed('ember-application/utils/validate-type', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  /**
  @module ember
  @submodule ember-application
  */

  'use strict';

  exports.default = validateType;

  var VALIDATED_TYPES = {
    route: ['assert', 'isRouteFactory', 'Ember.Route'],
    component: ['deprecate', 'isComponentFactory', 'Ember.Component'],
    view: ['deprecate', 'isViewFactory', 'Ember.View'],
    service: ['deprecate', 'isServiceFactory', 'Ember.Service']
  };

  function validateType(resolvedType, parsedName) {
    var validationAttributes = VALIDATED_TYPES[parsedName.type];

    if (!validationAttributes) {
      return;
    }

    var action = validationAttributes[0];
    var factoryFlag = validationAttributes[1];
    var expectedType = validationAttributes[2];

    if (action === 'deprecate') {} else {}
  }
});
enifed('ember-console/index', ['exports', 'ember-environment'], function (exports, _emberEnvironment) {
  'use strict';

  function K() {}

  function consoleMethod(name) {
    var consoleObj = undefined;
    if (_emberEnvironment.context.imports.console) {
      consoleObj = _emberEnvironment.context.imports.console;
    } else if (typeof console !== 'undefined') {
      consoleObj = console;
    }

    var method = typeof consoleObj === 'object' ? consoleObj[name] : null;

    if (typeof method !== 'function') {
      return;
    }

    if (typeof method.bind === 'function') {
      return method.bind(consoleObj);
    }

    return function () {
      method.apply(consoleObj, arguments);
    };
  }

  function assertPolyfill(test, message) {
    if (!test) {
      try {
        // attempt to preserve the stack
        throw new Error('assertion failed: ' + message);
      } catch (error) {
        setTimeout(function () {
          throw error;
        }, 0);
      }
    }
  }

  /**
    Inside Ember-Metal, simply uses the methods from `imports.console`.
    Override this to provide more robust logging functionality.
  
    @class Logger
    @namespace Ember
    @public
  */
  exports.default = {
    /**
     Logs the arguments to the console.
     You can pass as many arguments as you want and they will be joined together with a space.
       ```javascript
      var foo = 1;
      Ember.Logger.log('log value of foo:', foo);
      // "log value of foo: 1" will be printed to the console
      ```
      @method log
     @for Ember.Logger
     @param {*} arguments
     @public
    */
    log: consoleMethod('log') || K,

    /**
     Prints the arguments to the console with a warning icon.
     You can pass as many arguments as you want and they will be joined together with a space.
       ```javascript
      Ember.Logger.warn('Something happened!');
      // "Something happened!" will be printed to the console with a warning icon.
      ```
      @method warn
     @for Ember.Logger
     @param {*} arguments
     @public
    */
    warn: consoleMethod('warn') || K,

    /**
     Prints the arguments to the console with an error icon, red text and a stack trace.
     You can pass as many arguments as you want and they will be joined together with a space.
       ```javascript
      Ember.Logger.error('Danger! Danger!');
      // "Danger! Danger!" will be printed to the console in red text.
      ```
      @method error
     @for Ember.Logger
     @param {*} arguments
     @public
    */
    error: consoleMethod('error') || K,

    /**
     Logs the arguments to the console.
     You can pass as many arguments as you want and they will be joined together with a space.
       ```javascript
      var foo = 1;
      Ember.Logger.info('log value of foo:', foo);
      // "log value of foo: 1" will be printed to the console
      ```
      @method info
     @for Ember.Logger
     @param {*} arguments
     @public
    */
    info: consoleMethod('info') || K,

    /**
     Logs the arguments to the console in blue text.
     You can pass as many arguments as you want and they will be joined together with a space.
       ```javascript
      var foo = 1;
      Ember.Logger.debug('log value of foo:', foo);
      // "log value of foo: 1" will be printed to the console
      ```
      @method debug
     @for Ember.Logger
     @param {*} arguments
     @public
    */
    debug: consoleMethod('debug') || consoleMethod('info') || K,

    /**
     If the value passed into `Ember.Logger.assert` is not truthy it will throw an error with a stack trace.
       ```javascript
      Ember.Logger.assert(true); // undefined
      Ember.Logger.assert(true === false); // Throws an Assertion failed error.
      Ember.Logger.assert(true === false, 'Something invalid'); // Throws an Assertion failed error with message.
      ```
      @method assert
     @for Ember.Logger
     @param {Boolean} bool Value to test
     @param {String} message Assertion message on failed
     @public
    */
    assert: consoleMethod('assert') || assertPolyfill
  };
});
enifed('ember-environment/global', ['exports'], function (exports) {
  /* globals global, window, self, mainContext */

  // from lodash to catch fake globals
  'use strict';

  function checkGlobal(value) {
    return value && value.Object === Object ? value : undefined;
  }

  // element ids can ruin global miss checks
  function checkElementIdShadowing(value) {
    return value && value.nodeType === undefined ? value : undefined;
  }

  // export real global
  exports.default = checkGlobal(checkElementIdShadowing(typeof global === 'object' && global)) || checkGlobal(typeof self === 'object' && self) || checkGlobal(typeof window === 'object' && window) || mainContext || // set before strict mode in Ember loader/wrapper
  new Function('return this')();
  // eval outside of strict mode
});
enifed('ember-environment/index', ['exports', 'ember-environment/global', 'ember-environment/utils'], function (exports, _emberEnvironmentGlobal, _emberEnvironmentUtils) {
  /* globals module */
  'use strict';

  /**
    The hash of environment variables used to control various configuration
    settings. To specify your own or override default settings, add the
    desired properties to a global hash named `EmberENV` (or `ENV` for
    backwards compatibility with earlier versions of Ember). The `EmberENV`
    hash must be created before loading Ember.
  
    @class EmberENV
    @type Object
    @public
  */
  var ENV = typeof _emberEnvironmentGlobal.default.EmberENV === 'object' && _emberEnvironmentGlobal.default.EmberENV || typeof _emberEnvironmentGlobal.default.ENV === 'object' && _emberEnvironmentGlobal.default.ENV || {};

  exports.ENV = ENV;
  // ENABLE_ALL_FEATURES was documented, but you can't actually enable non optional features.
  if (ENV.ENABLE_ALL_FEATURES) {
    ENV.ENABLE_OPTIONAL_FEATURES = true;
  }

  /**
    Determines whether Ember should add to `Array`, `Function`, and `String`
    native object prototypes, a few extra methods in order to provide a more
    friendly API.
  
    We generally recommend leaving this option set to true however, if you need
    to turn it off, you can add the configuration property
    `EXTEND_PROTOTYPES` to `EmberENV` and set it to `false`.
  
    Note, when disabled (the default configuration for Ember Addons), you will
    instead have to access all methods and functions from the Ember
    namespace.
  
    @property EXTEND_PROTOTYPES
    @type Boolean
    @default true
    @for EmberENV
    @public
  */
  ENV.EXTEND_PROTOTYPES = _emberEnvironmentUtils.normalizeExtendPrototypes(ENV.EXTEND_PROTOTYPES);

  /**
    The `LOG_STACKTRACE_ON_DEPRECATION` property, when true, tells Ember to log
    a full stack trace during deprecation warnings.
  
    @property LOG_STACKTRACE_ON_DEPRECATION
    @type Boolean
    @default true
    @for EmberENV
    @public
  */
  ENV.LOG_STACKTRACE_ON_DEPRECATION = _emberEnvironmentUtils.defaultTrue(ENV.LOG_STACKTRACE_ON_DEPRECATION);

  /**
    The `LOG_VERSION` property, when true, tells Ember to log versions of all
    dependent libraries in use.
  
    @property LOG_VERSION
    @type Boolean
    @default true
    @for EmberENV
    @public
  */
  ENV.LOG_VERSION = _emberEnvironmentUtils.defaultTrue(ENV.LOG_VERSION);

  // default false
  ENV.MODEL_FACTORY_INJECTIONS = _emberEnvironmentUtils.defaultFalse(ENV.MODEL_FACTORY_INJECTIONS);

  /**
    Debug parameter you can turn on. This will log all bindings that fire to
    the console. This should be disabled in production code. Note that you
    can also enable this from the console or temporarily.
  
    @property LOG_BINDINGS
    @for EmberENV
    @type Boolean
    @default false
    @public
  */
  ENV.LOG_BINDINGS = _emberEnvironmentUtils.defaultFalse(ENV.LOG_BINDINGS);

  ENV.RAISE_ON_DEPRECATION = _emberEnvironmentUtils.defaultFalse(ENV.RAISE_ON_DEPRECATION);

  // check if window exists and actually is the global
  var hasDOM = typeof window !== 'undefined' && window === _emberEnvironmentGlobal.default && window.document && window.document.createElement && !ENV.disableBrowserEnvironment; // is this a public thing?

  // legacy imports/exports/lookup stuff (should we keep this??)
  var originalContext = _emberEnvironmentGlobal.default.Ember || {};

  var context = {
    // import jQuery
    imports: originalContext.imports || _emberEnvironmentGlobal.default,
    // export Ember
    exports: originalContext.exports || _emberEnvironmentGlobal.default,
    // search for Namespaces
    lookup: originalContext.lookup || _emberEnvironmentGlobal.default
  };

  exports.context = context;
  // TODO: cleanup single source of truth issues with this stuff
  var environment = hasDOM ? {
    hasDOM: true,
    isChrome: !!window.chrome && !window.opera,
    isFirefox: typeof InstallTrigger !== 'undefined',
    isPhantom: !!window.callPhantom,
    location: window.location,
    history: window.history,
    userAgent: window.navigator.userAgent,
    window: window
  } : {
    hasDOM: false,
    isChrome: false,
    isFirefox: false,
    isPhantom: false,
    location: null,
    history: null,
    userAgent: 'Lynx (textmode)',
    window: null
  };
  exports.environment = environment;
});
enifed("ember-environment/utils", ["exports"], function (exports) {
  "use strict";

  exports.defaultTrue = defaultTrue;
  exports.defaultFalse = defaultFalse;
  exports.normalizeExtendPrototypes = normalizeExtendPrototypes;

  function defaultTrue(v) {
    return v === false ? false : true;
  }

  function defaultFalse(v) {
    return v === true ? true : false;
  }

  function normalizeExtendPrototypes(obj) {
    if (obj === false) {
      return { String: false, Array: false, Function: false };
    } else if (!obj || obj === true) {
      return { String: true, Array: true, Function: true };
    } else {
      return {
        String: defaultTrue(obj.String),
        Array: defaultTrue(obj.Array),
        Function: defaultTrue(obj.Function)
      };
    }
  }
});
enifed('ember-extension-support/container_debug_adapter', ['exports', 'ember-metal', 'ember-runtime'], function (exports, _emberMetal, _emberRuntime) {
  'use strict';

  /**
  @module ember
  @submodule ember-extension-support
  */

  /**
    The `ContainerDebugAdapter` helps the container and resolver interface
    with tools that debug Ember such as the
    [Ember Extension](https://github.com/tildeio/ember-extension)
    for Chrome and Firefox.
  
    This class can be extended by a custom resolver implementer
    to override some of the methods with library-specific code.
  
    The methods likely to be overridden are:
  
    * `canCatalogEntriesByType`
    * `catalogEntriesByType`
  
    The adapter will need to be registered
    in the application's container as `container-debug-adapter:main`.
  
    Example:
  
    ```javascript
    Application.initializer({
      name: "containerDebugAdapter",
  
      initialize(application) {
        application.register('container-debug-adapter:main', require('app/container-debug-adapter'));
      }
    });
    ```
  
    @class ContainerDebugAdapter
    @namespace Ember
    @extends Ember.Object
    @since 1.5.0
    @public
  */
  exports.default = _emberRuntime.Object.extend({
    /**
      The resolver instance of the application
      being debugged. This property will be injected
      on creation.
       @property resolver
      @default null
      @public
    */
    resolver: null,

    /**
      Returns true if it is possible to catalog a list of available
      classes in the resolver for a given type.
       @method canCatalogEntriesByType
      @param {String} type The type. e.g. "model", "controller", "route".
      @return {boolean} whether a list is available for this type.
      @public
    */
    canCatalogEntriesByType: function (type) {
      if (type === 'model' || type === 'template') {
        return false;
      }

      return true;
    },

    /**
      Returns the available classes a given type.
       @method catalogEntriesByType
      @param {String} type The type. e.g. "model", "controller", "route".
      @return {Array} An array of strings.
      @public
    */
    catalogEntriesByType: function (type) {
      var namespaces = _emberRuntime.A(_emberRuntime.Namespace.NAMESPACES);
      var types = _emberRuntime.A();
      var typeSuffixRegex = new RegExp(_emberRuntime.String.classify(type) + '$');

      namespaces.forEach(function (namespace) {
        if (namespace !== _emberMetal.default) {
          for (var key in namespace) {
            if (!namespace.hasOwnProperty(key)) {
              continue;
            }
            if (typeSuffixRegex.test(key)) {
              var klass = namespace[key];
              if (_emberRuntime.typeOf(klass) === 'class') {
                types.push(_emberRuntime.String.dasherize(key.replace(typeSuffixRegex, '')));
              }
            }
          }
        }
      });
      return types;
    }
  });
});
// Ember as namespace
enifed('ember-extension-support/data_adapter', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime', 'ember-application'], function (exports, _emberUtils, _emberMetal, _emberRuntime, _emberApplication) {
  'use strict';

  /**
  @module ember
  @submodule ember-extension-support
  */

  /**
    The `DataAdapter` helps a data persistence library
    interface with tools that debug Ember such
    as the [Ember Extension](https://github.com/tildeio/ember-extension)
    for Chrome and Firefox.
  
    This class will be extended by a persistence library
    which will override some of the methods with
    library-specific code.
  
    The methods likely to be overridden are:
  
    * `getFilters`
    * `detect`
    * `columnsForType`
    * `getRecords`
    * `getRecordColumnValues`
    * `getRecordKeywords`
    * `getRecordFilterValues`
    * `getRecordColor`
    * `observeRecord`
  
    The adapter will need to be registered
    in the application's container as `dataAdapter:main`.
  
    Example:
  
    ```javascript
    Application.initializer({
      name: "data-adapter",
  
      initialize: function(application) {
        application.register('data-adapter:main', DS.DataAdapter);
      }
    });
    ```
  
    @class DataAdapter
    @namespace Ember
    @extends EmberObject
    @public
  */
  exports.default = _emberRuntime.Object.extend({
    init: function () {
      this._super.apply(this, arguments);
      this.releaseMethods = _emberRuntime.A();
    },

    /**
      The container-debug-adapter which is used
      to list all models.
       @property containerDebugAdapter
      @default undefined
      @since 1.5.0
      @public
    **/
    containerDebugAdapter: undefined,

    /**
      The number of attributes to send
      as columns. (Enough to make the record
      identifiable).
       @private
      @property attributeLimit
      @default 3
      @since 1.3.0
    */
    attributeLimit: 3,

    /**
       Ember Data > v1.0.0-beta.18
       requires string model names to be passed
       around instead of the actual factories.
        This is a stamp for the Ember Inspector
       to differentiate between the versions
       to be able to support older versions too.
        @public
       @property acceptsModelName
     */
    acceptsModelName: true,

    /**
      Stores all methods that clear observers.
      These methods will be called on destruction.
       @private
      @property releaseMethods
      @since 1.3.0
    */
    releaseMethods: _emberRuntime.A(),

    /**
      Specifies how records can be filtered.
      Records returned will need to have a `filterValues`
      property with a key for every name in the returned array.
       @public
      @method getFilters
      @return {Array} List of objects defining filters.
       The object should have a `name` and `desc` property.
    */
    getFilters: function () {
      return _emberRuntime.A();
    },

    /**
      Fetch the model types and observe them for changes.
       @public
      @method watchModelTypes
       @param {Function} typesAdded Callback to call to add types.
      Takes an array of objects containing wrapped types (returned from `wrapModelType`).
       @param {Function} typesUpdated Callback to call when a type has changed.
      Takes an array of objects containing wrapped types.
       @return {Function} Method to call to remove all observers
    */
    watchModelTypes: function (typesAdded, typesUpdated) {
      var _this = this;

      var modelTypes = this.getModelTypes();
      var releaseMethods = _emberRuntime.A();
      var typesToSend = undefined;

      typesToSend = modelTypes.map(function (type) {
        var klass = type.klass;
        var wrapped = _this.wrapModelType(klass, type.name);
        releaseMethods.push(_this.observeModelType(type.name, typesUpdated));
        return wrapped;
      });

      typesAdded(typesToSend);

      var release = function () {
        releaseMethods.forEach(function (fn) {
          return fn();
        });
        _this.releaseMethods.removeObject(release);
      };
      this.releaseMethods.pushObject(release);
      return release;
    },

    _nameToClass: function (type) {
      if (typeof type === 'string') {
        type = _emberUtils.getOwner(this)._lookupFactory('model:' + type);
      }
      return type;
    },

    /**
      Fetch the records of a given type and observe them for changes.
       @public
      @method watchRecords
       @param {String} modelName The model name.
       @param {Function} recordsAdded Callback to call to add records.
      Takes an array of objects containing wrapped records.
      The object should have the following properties:
        columnValues: {Object} The key and value of a table cell.
        object: {Object} The actual record object.
       @param {Function} recordsUpdated Callback to call when a record has changed.
      Takes an array of objects containing wrapped records.
       @param {Function} recordsRemoved Callback to call when a record has removed.
      Takes the following parameters:
        index: The array index where the records were removed.
        count: The number of records removed.
       @return {Function} Method to call to remove all observers.
    */
    watchRecords: function (modelName, recordsAdded, recordsUpdated, recordsRemoved) {
      var _this2 = this;

      var releaseMethods = _emberRuntime.A();
      var klass = this._nameToClass(modelName);
      var records = this.getRecords(klass, modelName);
      var release = undefined;

      function recordUpdated(updatedRecord) {
        recordsUpdated([updatedRecord]);
      }

      var recordsToSend = records.map(function (record) {
        releaseMethods.push(_this2.observeRecord(record, recordUpdated));
        return _this2.wrapRecord(record);
      });

      var contentDidChange = function (array, idx, removedCount, addedCount) {
        for (var i = idx; i < idx + addedCount; i++) {
          var record = _emberRuntime.objectAt(array, i);
          var wrapped = _this2.wrapRecord(record);
          releaseMethods.push(_this2.observeRecord(record, recordUpdated));
          recordsAdded([wrapped]);
        }

        if (removedCount) {
          recordsRemoved(idx, removedCount);
        }
      };

      var observer = { didChange: contentDidChange, willChange: function () {
          return this;
        } };
      _emberRuntime.addArrayObserver(records, this, observer);

      release = function () {
        releaseMethods.forEach(function (fn) {
          return fn();
        });
        _emberRuntime.removeArrayObserver(records, _this2, observer);
        _this2.releaseMethods.removeObject(release);
      };

      recordsAdded(recordsToSend);

      this.releaseMethods.pushObject(release);
      return release;
    },

    /**
      Clear all observers before destruction
      @private
      @method willDestroy
    */
    willDestroy: function () {
      this._super.apply(this, arguments);
      this.releaseMethods.forEach(function (fn) {
        return fn();
      });
    },

    /**
      Detect whether a class is a model.
       Test that against the model class
      of your persistence library.
       @private
      @method detect
      @param {Class} klass The class to test.
      @return boolean Whether the class is a model class or not.
    */
    detect: function (klass) {
      return false;
    },

    /**
      Get the columns for a given model type.
       @private
      @method columnsForType
      @param {Class} type The model type.
      @return {Array} An array of columns of the following format:
       name: {String} The name of the column.
       desc: {String} Humanized description (what would show in a table column name).
    */
    columnsForType: function (type) {
      return _emberRuntime.A();
    },

    /**
      Adds observers to a model type class.
       @private
      @method observeModelType
      @param {String} modelName The model type name.
      @param {Function} typesUpdated Called when a type is modified.
      @return {Function} The function to call to remove observers.
    */

    observeModelType: function (modelName, typesUpdated) {
      var _this3 = this;

      var klass = this._nameToClass(modelName);
      var records = this.getRecords(klass, modelName);

      function onChange() {
        typesUpdated([this.wrapModelType(klass, modelName)]);
      }

      var observer = {
        didChange: function () {
          _emberMetal.run.scheduleOnce('actions', this, onChange);
        },
        willChange: function () {
          return this;
        }
      };

      _emberRuntime.addArrayObserver(records, this, observer);

      var release = function () {
        return _emberRuntime.removeArrayObserver(records, _this3, observer);
      };

      return release;
    },

    /**
      Wraps a given model type and observes changes to it.
       @private
      @method wrapModelType
      @param {Class} klass A model class.
      @param {String} modelName Name of the class.
      @return {Object} Contains the wrapped type and the function to remove observers
      Format:
        type: {Object} The wrapped type.
          The wrapped type has the following format:
            name: {String} The name of the type.
            count: {Integer} The number of records available.
            columns: {Columns} An array of columns to describe the record.
            object: {Class} The actual Model type class.
        release: {Function} The function to remove observers.
    */
    wrapModelType: function (klass, name) {
      var records = this.getRecords(klass, name);
      var typeToSend = undefined;

      typeToSend = {
        name: name,
        count: _emberMetal.get(records, 'length'),
        columns: this.columnsForType(klass),
        object: klass
      };

      return typeToSend;
    },

    /**
      Fetches all models defined in the application.
       @private
      @method getModelTypes
      @return {Array} Array of model types.
    */
    getModelTypes: function () {
      var _this4 = this;

      var containerDebugAdapter = this.get('containerDebugAdapter');
      var types = undefined;

      if (containerDebugAdapter.canCatalogEntriesByType('model')) {
        types = containerDebugAdapter.catalogEntriesByType('model');
      } else {
        types = this._getObjectsOnNamespaces();
      }

      // New adapters return strings instead of classes.
      types = _emberRuntime.A(types).map(function (name) {
        return {
          klass: _this4._nameToClass(name),
          name: name
        };
      });
      types = _emberRuntime.A(types).filter(function (type) {
        return _this4.detect(type.klass);
      });

      return _emberRuntime.A(types);
    },

    /**
      Loops over all namespaces and all objects
      attached to them.
       @private
      @method _getObjectsOnNamespaces
      @return {Array} Array of model type strings.
    */
    _getObjectsOnNamespaces: function () {
      var _this5 = this;

      var namespaces = _emberRuntime.A(_emberRuntime.Namespace.NAMESPACES);
      var types = _emberRuntime.A();

      namespaces.forEach(function (namespace) {
        for (var key in namespace) {
          if (!namespace.hasOwnProperty(key)) {
            continue;
          }
          // Even though we will filter again in `getModelTypes`,
          // we should not call `lookupFactory` on non-models
          // (especially when `EmberENV.MODEL_FACTORY_INJECTIONS` is `true`)
          if (!_this5.detect(namespace[key])) {
            continue;
          }
          var _name = _emberRuntime.String.dasherize(key);
          if (!(namespace instanceof _emberApplication.Application) && namespace.toString()) {
            _name = namespace + '/' + _name;
          }
          types.push(_name);
        }
      });
      return types;
    },

    /**
      Fetches all loaded records for a given type.
       @private
      @method getRecords
      @return {Array} An array of records.
       This array will be observed for changes,
       so it should update when new records are added/removed.
    */
    getRecords: function (type) {
      return _emberRuntime.A();
    },

    /**
      Wraps a record and observers changes to it.
       @private
      @method wrapRecord
      @param {Object} record The record instance.
      @return {Object} The wrapped record. Format:
      columnValues: {Array}
      searchKeywords: {Array}
    */
    wrapRecord: function (record) {
      var recordToSend = { object: record };

      recordToSend.columnValues = this.getRecordColumnValues(record);
      recordToSend.searchKeywords = this.getRecordKeywords(record);
      recordToSend.filterValues = this.getRecordFilterValues(record);
      recordToSend.color = this.getRecordColor(record);

      return recordToSend;
    },

    /**
      Gets the values for each column.
       @private
      @method getRecordColumnValues
      @return {Object} Keys should match column names defined
      by the model type.
    */
    getRecordColumnValues: function (record) {
      return {};
    },

    /**
      Returns keywords to match when searching records.
       @private
      @method getRecordKeywords
      @return {Array} Relevant keywords for search.
    */
    getRecordKeywords: function (record) {
      return _emberRuntime.A();
    },

    /**
      Returns the values of filters defined by `getFilters`.
       @private
      @method getRecordFilterValues
      @param {Object} record The record instance.
      @return {Object} The filter values.
    */
    getRecordFilterValues: function (record) {
      return {};
    },

    /**
      Each record can have a color that represents its state.
       @private
      @method getRecordColor
      @param {Object} record The record instance
      @return {String} The records color.
        Possible options: black, red, blue, green.
    */
    getRecordColor: function (record) {
      return null;
    },

    /**
      Observes all relevant properties and re-sends the wrapped record
      when a change occurs.
       @private
      @method observerRecord
      @param {Object} record The record instance.
      @param {Function} recordUpdated The callback to call when a record is updated.
      @return {Function} The function to call to remove all observers.
    */
    observeRecord: function (record, recordUpdated) {
      return function () {};
    }
  });
});
enifed('ember-extension-support/index', ['exports', 'ember-extension-support/data_adapter', 'ember-extension-support/container_debug_adapter'], function (exports, _emberExtensionSupportData_adapter, _emberExtensionSupportContainer_debug_adapter) {
  /**
  @module ember
  @submodule ember-extension-support
  */

  'use strict';

  exports.DataAdapter = _emberExtensionSupportData_adapter.default;
  exports.ContainerDebugAdapter = _emberExtensionSupportContainer_debug_adapter.default;
});
enifed('ember-glimmer/component', ['exports', 'ember-utils', 'ember-views', 'ember-runtime', 'ember-metal', 'ember-glimmer/utils/references', 'glimmer-reference', 'glimmer-runtime'], function (exports, _emberUtils, _emberViews, _emberRuntime, _emberMetal, _emberGlimmerUtilsReferences, _glimmerReference, _glimmerRuntime) {
  'use strict';

  var _CoreView$extend;

  var DIRTY_TAG = _emberUtils.symbol('DIRTY_TAG');
  exports.DIRTY_TAG = DIRTY_TAG;
  var ARGS = _emberUtils.symbol('ARGS');
  exports.ARGS = ARGS;
  var ROOT_REF = _emberUtils.symbol('ROOT_REF');
  exports.ROOT_REF = ROOT_REF;
  var IS_DISPATCHING_ATTRS = _emberUtils.symbol('IS_DISPATCHING_ATTRS');
  exports.IS_DISPATCHING_ATTRS = IS_DISPATCHING_ATTRS;
  var HAS_BLOCK = _emberUtils.symbol('HAS_BLOCK');
  exports.HAS_BLOCK = HAS_BLOCK;
  var BOUNDS = _emberUtils.symbol('BOUNDS');

  exports.BOUNDS = BOUNDS;
  /**
  @module ember
  @submodule ember-glimmer
  */

  /**
    An `Ember.Component` is a view that is completely
    isolated. Properties accessed in its templates go
    to the view object and actions are targeted at
    the view object. There is no access to the
    surrounding context or outer controller; all
    contextual information must be passed in.
  
    The easiest way to create an `Ember.Component` is via
    a template. If you name a template
    `app/components/my-foo.hbs`, you will be able to use
    `{{my-foo}}` in other templates, which will make
    an instance of the isolated component.
  
    ```app/components/my-foo.hbs
    {{person-profile person=currentUser}}
    ```
  
    ```app/components/person-profile.hbs
    <h1>{{person.title}}</h1>
    <img src={{person.avatar}}>
    <p class='signature'>{{person.signature}}</p>
    ```
  
    You can use `yield` inside a template to
    include the **contents** of any block attached to
    the component. The block will be executed in the
    context of the surrounding context or outer controller:
  
    ```handlebars
    {{#person-profile person=currentUser}}
      <p>Admin mode</p>
      {{! Executed in the controller's context. }}
    {{/person-profile}}
    ```
  
    ```app/components/person-profile.hbs
    <h1>{{person.title}}</h1>
    {{! Executed in the component's context. }}
    {{yield}} {{! block contents }}
    ```
  
    If you want to customize the component, in order to
    handle events or actions, you implement a subclass
    of `Ember.Component` named after the name of the
    component.
  
    For example, you could implement the action
    `hello` for the `person-profile` component:
  
    ```app/components/person-profile.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      actions: {
        hello(name) {
          console.log("Hello", name);
        }
      }
    });
    ```
  
    And then use it in the component's template:
  
    ```app/templates/components/person-profile.hbs
    <h1>{{person.title}}</h1>
    {{yield}} <!-- block contents -->
    <button {{action 'hello' person.name}}>
      Say Hello to {{person.name}}
    </button>
    ```
  
    Components must have a `-` in their name to avoid
    conflicts with built-in controls that wrap HTML
    elements. This is consistent with the same
    requirement in web components.
  
  
    ## HTML Tag
  
    The default HTML tag name used for a component's DOM representation is `div`.
    This can be customized by setting the `tagName` property.
    The following component class:
  
    ```app/components/emphasized-paragraph.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      tagName: 'em'
    });
    ```
  
    Would result in instances with the following HTML:
  
    ```html
    <em id="ember1" class="ember-view"></em>
    ```
  
  
    ## HTML `class` Attribute
  
    The HTML `class` attribute of a component's tag can be set by providing a
    `classNames` property that is set to an array of strings:
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      classNames: ['my-class', 'my-other-class']
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view my-class my-other-class"></div>
    ```
  
    `class` attribute values can also be set by providing a `classNameBindings`
    property set to an array of properties names for the component. The return value
    of these properties will be added as part of the value for the components's `class`
    attribute. These properties can be computed properties:
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      classNameBindings: ['propertyA', 'propertyB'],
      propertyA: 'from-a',
      propertyB: Ember.computed(function() {
        if (someLogic) { return 'from-b'; }
      })
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view from-a from-b"></div>
    ```
  
    If the value of a class name binding returns a boolean the property name
    itself will be used as the class name if the property is true.
    The class name will not be added if the value is `false` or `undefined`.
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      classNameBindings: ['hovered'],
      hovered: true
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view hovered"></div>
    ```
  
    When using boolean class name bindings you can supply a string value other
    than the property name for use as the `class` HTML attribute by appending the
    preferred value after a ":" character when defining the binding:
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      classNameBindings: ['awesome:so-very-cool'],
      awesome: true
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view so-very-cool"></div>
    ```
  
    Boolean value class name bindings whose property names are in a
    camelCase-style format will be converted to a dasherized format:
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      classNameBindings: ['isUrgent'],
      isUrgent: true
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view is-urgent"></div>
    ```
  
    Class name bindings can also refer to object values that are found by
    traversing a path relative to the component itself:
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      classNameBindings: ['messages.empty'],
      messages: Ember.Object.create({
        empty: true
      })
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view empty"></div>
    ```
  
    If you want to add a class name for a property which evaluates to true and
    and a different class name if it evaluates to false, you can pass a binding
    like this:
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      classNameBindings: ['isEnabled:enabled:disabled'],
      isEnabled: true
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view enabled"></div>
    ```
  
    When isEnabled is `false`, the resulting HTML representation looks like
    this:
  
    ```html
    <div id="ember1" class="ember-view disabled"></div>
    ```
  
    This syntax offers the convenience to add a class if a property is `false`:
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    // Applies no class when isEnabled is true and class 'disabled' when isEnabled is false
    export default Ember.Component.extend({
      classNameBindings: ['isEnabled::disabled'],
      isEnabled: true
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view"></div>
    ```
  
    When the `isEnabled` property on the component is set to `false`, it will result
    in component instances with an HTML representation of:
  
    ```html
    <div id="ember1" class="ember-view disabled"></div>
    ```
  
    Updates to the value of a class name binding will result in automatic
    update of the  HTML `class` attribute in the component's rendered HTML
    representation. If the value becomes `false` or `undefined` the class name
    will be removed.
    Both `classNames` and `classNameBindings` are concatenated properties. See
    [Ember.Object](/api/classes/Ember.Object.html) documentation for more
    information about concatenated properties.
  
  
    ## HTML Attributes
  
    The HTML attribute section of a component's tag can be set by providing an
    `attributeBindings` property set to an array of property names on the component.
    The return value of these properties will be used as the value of the component's
    HTML associated attribute:
  
    ```app/components/my-anchor.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      tagName: 'a',
      attributeBindings: ['href'],
      href: 'http://google.com'
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <a id="ember1" class="ember-view" href="http://google.com"></a>
    ```
  
    One property can be mapped on to another by placing a ":" between
    the source property and the destination property:
  
    ```app/components/my-anchor.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      tagName: 'a',
      attributeBindings: ['url:href'],
      url: 'http://google.com'
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <a id="ember1" class="ember-view" href="http://google.com"></a>
    ```
  
    Namespaced attributes (e.g. `xlink:href`) are supported, but have to be
    mapped, since `:` is not a valid character for properties in Javascript:
  
    ```app/components/my-use.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      tagName: 'use',
      attributeBindings: ['xlinkHref:xlink:href'],
      xlinkHref: '#triangle'
    });
    ```
  
    Will result in component instances with an HTML representation of:
  
    ```html
    <use xlink:href="#triangle"></use>
    ```
  
    If the return value of an `attributeBindings` monitored property is a boolean
    the attribute will be present or absent depending on the value:
  
    ```app/components/my-text-input.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      tagName: 'input',
      attributeBindings: ['disabled'],
      disabled: false
    });
    ```
  
    Will result in a component instance with an HTML representation of:
  
    ```html
    <input id="ember1" class="ember-view" />
    ```
  
    `attributeBindings` can refer to computed properties:
  
    ```app/components/my-text-input.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      tagName: 'input',
      attributeBindings: ['disabled'],
      disabled: Ember.computed(function() {
        if (someLogic) {
          return true;
        } else {
          return false;
        }
      })
    });
    ```
  
    To prevent setting an attribute altogether, use `null` or `undefined` as the
    return value of the `attributeBindings` monitored property:
  
    ```app/components/my-text-input.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      tagName: 'form',
      attributeBindings: ['novalidate'],
      novalidate: null
    });
    ```
  
    Updates to the property of an attribute binding will result in automatic
    update of the  HTML attribute in the component's rendered HTML representation.
    `attributeBindings` is a concatenated property. See [Ember.Object](/api/classes/Ember.Object.html)
    documentation for more information about concatenated properties.
  
  
    ## Layouts
  
    See [Ember.Templates.helpers.yield](/api/classes/Ember.Templates.helpers.html#method_yield)
    for more information.
  
  
    ## Responding to Browser Events
  
    Components can respond to user-initiated events in one of three ways: method
    implementation, through an event manager, and through `{{action}}` helper use
    in their template or layout.
  
  
    ### Method Implementation
  
    Components can respond to user-initiated events by implementing a method that
    matches the event name. A `jQuery.Event` object will be passed as the
    argument to this method.
  
    ```app/components/my-widget.js
    import Ember from 'ember';
  
    export default Ember.Component.extend({
      click(event) {
        // will be called when an instance's
        // rendered element is clicked
      }
    });
    ```
  
  
    ### `{{action}}` Helper
  
    See [Ember.Templates.helpers.action](/api/classes/Ember.Templates.helpers.html#method_action).
  
  
    ### Event Names
  
    All of the event handling approaches described above respond to the same set
    of events. The names of the built-in events are listed below. (The hash of
    built-in events exists in `Ember.EventDispatcher`.) Additional, custom events
    can be registered by using `Ember.Application.customEvents`.
  
    Touch events:
  
    * `touchStart`
    * `touchMove`
    * `touchEnd`
    * `touchCancel`
  
    Keyboard events:
  
    * `keyDown`
    * `keyUp`
    * `keyPress`
  
    Mouse events:
  
    * `mouseDown`
    * `mouseUp`
    * `contextMenu`
    * `click`
    * `doubleClick`
    * `mouseMove`
    * `focusIn`
    * `focusOut`
    * `mouseEnter`
    * `mouseLeave`
  
    Form events:
  
    * `submit`
    * `change`
    * `focusIn`
    * `focusOut`
    * `input`
  
    HTML5 drag and drop events:
  
    * `dragStart`
    * `drag`
    * `dragEnter`
    * `dragLeave`
    * `dragOver`
    * `dragEnd`
    * `drop`
  
    @class Component
    @namespace Ember
    @extends Ember.CoreView
    @uses Ember.TargetActionSupport
    @uses Ember.ClassNamesSupport
    @uses Ember.ActionSupport
    @uses Ember.ViewMixin
    @uses Ember.ViewStateSupport
    @public
  */
  var Component = _emberViews.CoreView.extend(_emberViews.ChildViewsSupport, _emberViews.ViewStateSupport, _emberViews.ClassNamesSupport, _emberRuntime.TargetActionSupport, _emberViews.ActionSupport, _emberViews.ViewMixin, (_CoreView$extend = {
    isComponent: true,

    init: function () {
      this._super.apply(this, arguments);
      this[IS_DISPATCHING_ATTRS] = false;
      this[DIRTY_TAG] = new _glimmerReference.DirtyableTag();
      this[ROOT_REF] = new _emberGlimmerUtilsReferences.RootReference(this);
      this[BOUNDS] = null;

      // If a `defaultLayout` was specified move it to the `layout` prop.
      // `layout` is no longer a CP, so this just ensures that the `defaultLayout`
      // logic is supported with a deprecation
      if (this.defaultLayout && !this.layout) {

        this.layout = this.defaultLayout;
      }

      // If in a tagless component, assert that no event handlers are defined
      // indicate that the assertion should be triggered
    },

    rerender: function () {
      this[DIRTY_TAG].dirty();
      this._super();
    },

    __defineNonEnumerable: function (property) {
      this[property.name] = property.descriptor.value;
    }

  }, _CoreView$extend[_emberMetal.PROPERTY_DID_CHANGE] = function (key) {
    if (this[IS_DISPATCHING_ATTRS]) {
      return;
    }

    var args = undefined,
        reference = undefined;

    if ((args = this[ARGS]) && (reference = args[key])) {
      if (reference[_emberGlimmerUtilsReferences.UPDATE]) {
        reference[_emberGlimmerUtilsReferences.UPDATE](_emberMetal.get(this, key));
      }
    }
  }, _CoreView$extend.getAttr = function (key) {
    // TODO Intimate API should be deprecated
    return this.get(key);
  }, _CoreView$extend.readDOMAttr = function (name) {
    var element = _emberViews.getViewElement(this);
    return _glimmerRuntime.readDOMAttr(element, name);
  }, _CoreView$extend));

  /**
   The WAI-ARIA role of the control represented by this view. For example, a
   button may have a role of type 'button', or a pane may have a role of
   type 'alertdialog'. This property is used by assistive software to help
   visually challenged users navigate rich web applications.
    The full list of valid WAI-ARIA roles is available at:
   [http://www.w3.org/TR/wai-aria/roles#roles_categorization](http://www.w3.org/TR/wai-aria/roles#roles_categorization)
    @property ariaRole
   @type String
   @default null
   @public
   */

  /**
   Enables components to take a list of parameters as arguments.
   For example, a component that takes two parameters with the names
   `name` and `age`:
    ```javascript
   let MyComponent = Ember.Component.extend;
    MyComponent.reopenClass({
     positionalParams: ['name', 'age']
   });
   ```
    It can then be invoked like this:
    ```hbs
   {{my-component "John" 38}}
   ```
    The parameters can be referred to just like named parameters:
    ```hbs
   Name: {{name}}, Age: {{age}}.
   ```
    Using a string instead of an array allows for an arbitrary number of
   parameters:
    ```javascript
   let MyComponent = Ember.Component.extend;
    MyComponent.reopenClass({
     positionalParams: 'names'
   });
   ```
    It can then be invoked like this:
    ```hbs
   {{my-component "John" "Michael" "Scott"}}
   ```
   The parameters can then be referred to by enumerating over the list:
    ```hbs
   {{#each names as |name|}}{{name}}{{/each}}
   ```
    @static
   @public
   @property positionalParams
   @since 1.13.0
   */

  /**
   Called when the attributes passed into the component have been updated.
   Called both during the initial render of a container and during a rerender.
   Can be used in place of an observer; code placed here will be executed
   every time any attribute updates.
   @method didReceiveAttrs
   @public
   @since 1.13.0
   */

  /**
   Called when the attributes passed into the component have been updated.
   Called both during the initial render of a container and during a rerender.
   Can be used in place of an observer; code placed here will be executed
   every time any attribute updates.
   @event didReceiveAttrs
   @public
   @since 1.13.0
   */

  /**
   Called after a component has been rendered, both on initial render and
   in subsequent rerenders.
   @method didRender
   @public
   @since 1.13.0
   */

  /**
   Called after a component has been rendered, both on initial render and
   in subsequent rerenders.
   @event didRender
   @public
   @since 1.13.0
   */

  /**
   Called before a component has been rendered, both on initial render and
   in subsequent rerenders.
   @method willRender
   @public
   @since 1.13.0
   */

  /**
   Called before a component has been rendered, both on initial render and
   in subsequent rerenders.
   @event willRender
   @public
   @since 1.13.0
   */

  /**
   Called when the attributes passed into the component have been changed.
   Called only during a rerender, not during an initial render.
   @method didUpdateAttrs
   @public
   @since 1.13.0
   */

  /**
   Called when the attributes passed into the component have been changed.
   Called only during a rerender, not during an initial render.
   @event didUpdateAttrs
   @public
   @since 1.13.0
   */

  /**
   Called when the component is about to update and rerender itself.
   Called only during a rerender, not during an initial render.
   @method willUpdate
   @public
   @since 1.13.0
   */

  /**
   Called when the component is about to update and rerender itself.
   Called only during a rerender, not during an initial render.
   @event willUpdate
   @public
   @since 1.13.0
   */

  /**
   Called when the component has updated and rerendered itself.
   Called only during a rerender, not during an initial render.
   @method didUpdate
   @public
   @since 1.13.0
   */

  /**
   Called when the component has updated and rerendered itself.
   Called only during a rerender, not during an initial render.
   @event didUpdate
   @public
   @since 1.13.0
   */

  /**
    A component may contain a layout. A layout is a regular template but
    supersedes the `template` property during rendering. It is the
    responsibility of the layout template to retrieve the `template`
    property from the component (or alternatively, call `Handlebars.helpers.yield`,
    `{{yield}}`) to render it in the correct location.
    This is useful for a component that has a shared wrapper, but which delegates
    the rendering of the contents of the wrapper to the `template` property
    on a subclass.
    @property layout
    @type Function
    @public
  */

  /**
    The name of the layout to lookup if no layout is provided.
    By default `Ember.Component` will lookup a template with this name in
    `Ember.TEMPLATES` (a shared global object).
    @property layoutName
    @type String
    @default null
    @private
  */

  /**
    Returns a jQuery object for this component's element. If you pass in a selector
    string, this method will return a jQuery object, using the current element
    as its buffer.
    For example, calling `component.$('li')` will return a jQuery object containing
    all of the `li` elements inside the DOM element of this component.
    @method $
    @param {String} [selector] a jQuery-compatible selector string
    @return {jQuery} the jQuery object for the DOM node
    @public
  */

  /**
    The HTML `id` of the component's element in the DOM. You can provide this
    value yourself but it must be unique (just as in HTML):
     ```handlebars
    {{my-component elementId="a-really-cool-id"}}
    ```
    If not manually set a default value will be provided by the framework.
    Once rendered an element's `elementId` is considered immutable and you
    should never change it. If you need to compute a dynamic value for the
    `elementId`, you should do this when the component or element is being
    instantiated:
     ```javascript
    export default Ember.Component.extend({
      init() {
        this._super(...arguments);
         var index = this.get('index');
        this.set('elementId', `component-id${index}`);
      }
    });
    ```
     @property elementId
    @type String
    @public
  */

  /**
   If `false`, the view will appear hidden in DOM.
    @property isVisible
   @type Boolean
   @default null
   @public
   */
  Component[_emberUtils.NAME_KEY] = 'Ember.Component';

  Component.reopenClass({
    isComponentFactory: true,
    positionalParams: []
  });

  exports.default = Component;
});

/**
  Normally, Ember's component model is "write-only". The component takes a
  bunch of attributes that it got passed in, and uses them to render its
  template.
   One nice thing about this model is that if you try to set a value to the
  same thing as last time, Ember (through HTMLBars) will avoid doing any
  work on the DOM.
   This is not just a performance optimization. If an attribute has not
  changed, it is important not to clobber the element's "hidden state".
  For example, if you set an input's `value` to the same value as before,
  it will clobber selection state and cursor position. In other words,
  setting an attribute is not **always** idempotent.
   This method provides a way to read an element's attribute and also
  update the last value Ember knows about at the same time. This makes
  setting an attribute idempotent.
   In particular, what this means is that if you get an `<input>` element's
  `value` attribute and then re-render the template with the same value,
  it will avoid clobbering the cursor and selection position.
  Since most attribute sets are idempotent in the browser, you typically
  can get away with reading attributes using jQuery, but the most reliable
  way to do so is through this method.
  @method readDOMAttr
   @param {String} name the name of the attribute
  @return String
  @public
 */
enifed('ember-glimmer/components/checkbox', ['exports', 'ember-metal', 'ember-glimmer/component', 'ember-glimmer/templates/empty'], function (exports, _emberMetal, _emberGlimmerComponent, _emberGlimmerTemplatesEmpty) {
  'use strict';

  /**
  @module ember
  @submodule ember-views
  */

  /**
    The internal class used to create text inputs when the `{{input}}`
    helper is used with `type` of `checkbox`.
  
    See [Ember.Templates.helpers.input](/api/classes/Ember.Templates.helpers.html#method_input)  for usage details.
  
    ## Direct manipulation of `checked`
  
    The `checked` attribute of an `Ember.Checkbox` object should always be set
    through the Ember object or by interacting with its rendered element
    representation via the mouse, keyboard, or touch. Updating the value of the
    checkbox via jQuery will result in the checked value of the object and its
    element losing synchronization.
  
    ## Layout and LayoutName properties
  
    Because HTML `input` elements are self closing `layout` and `layoutName`
    properties will not be applied. See [Ember.View](/api/classes/Ember.View.html)'s
    layout section for more information.
  
    @class Checkbox
    @namespace Ember
    @extends Ember.Component
    @public
  */
  exports.default = _emberGlimmerComponent.default.extend({
    layout: _emberGlimmerTemplatesEmpty.default,
    classNames: ['ember-checkbox'],

    tagName: 'input',

    attributeBindings: ['type', 'checked', 'indeterminate', 'disabled', 'tabindex', 'name', 'autofocus', 'required', 'form'],

    type: 'checkbox',
    checked: false,
    disabled: false,
    indeterminate: false,

    didInsertElement: function () {
      this._super.apply(this, arguments);
      _emberMetal.get(this, 'element').indeterminate = !!_emberMetal.get(this, 'indeterminate');
    },

    change: function () {
      _emberMetal.set(this, 'checked', this.$().prop('checked'));
    }
  });
});
enifed('ember-glimmer/components/link-to', ['exports', 'ember-console', 'ember-metal', 'ember-runtime', 'ember-views', 'ember-glimmer/templates/link-to', 'ember-glimmer/component'], function (exports, _emberConsole, _emberMetal, _emberRuntime, _emberViews, _emberGlimmerTemplatesLinkTo, _emberGlimmerComponent) {
  /**
  @module ember
  @submodule ember-glimmer
  */

  /**
    The `{{link-to}}` component renders a link to the supplied
    `routeName` passing an optionally supplied model to the
    route as its `model` context of the route. The block
    for `{{link-to}}` becomes the innerHTML of the rendered
    element:
  
    ```handlebars
    {{#link-to 'photoGallery'}}
      Great Hamster Photos
    {{/link-to}}
    ```
  
    You can also use an inline form of `{{link-to}}` component by
    passing the link text as the first argument
    to the component:
  
    ```handlebars
    {{link-to 'Great Hamster Photos' 'photoGallery'}}
    ```
  
    Both will result in:
  
    ```html
    <a href="/hamster-photos">
      Great Hamster Photos
    </a>
    ```
  
    ### Supplying a tagName
    By default `{{link-to}}` renders an `<a>` element. This can
    be overridden for a single use of `{{link-to}}` by supplying
    a `tagName` option:
  
    ```handlebars
    {{#link-to 'photoGallery' tagName="li"}}
      Great Hamster Photos
    {{/link-to}}
    ```
  
    ```html
    <li>
      Great Hamster Photos
    </li>
    ```
  
    To override this option for your entire application, see
    "Overriding Application-wide Defaults".
  
    ### Disabling the `link-to` component
    By default `{{link-to}}` is enabled.
    any passed value to the `disabled` component property will disable
    the `link-to` component.
  
    static use: the `disabled` option:
  
    ```handlebars
    {{#link-to 'photoGallery' disabled=true}}
      Great Hamster Photos
    {{/link-to}}
    ```
  
    dynamic use: the `disabledWhen` option:
  
    ```handlebars
    {{#link-to 'photoGallery' disabledWhen=controller.someProperty}}
      Great Hamster Photos
    {{/link-to}}
    ```
  
    any passed value to `disabled` will disable it except `undefined`.
    to ensure that only `true` disable the `link-to` component you can
    override the global behaviour of `Ember.LinkComponent`.
  
    ```javascript
    Ember.LinkComponent.reopen({
      disabled: Ember.computed(function(key, value) {
        if (value !== undefined) {
          this.set('_isDisabled', value === true);
        }
        return value === true ? get(this, 'disabledClass') : false;
      })
    });
    ```
  
    see "Overriding Application-wide Defaults" for more.
  
    ### Handling `href`
    `{{link-to}}` will use your application's Router to
    fill the element's `href` property with a url that
    matches the path to the supplied `routeName` for your
    router's configured `Location` scheme, which defaults
    to Ember.HashLocation.
  
    ### Handling current route
    `{{link-to}}` will apply a CSS class name of 'active'
    when the application's current route matches
    the supplied routeName. For example, if the application's
    current route is 'photoGallery.recent' the following
    use of `{{link-to}}`:
  
    ```handlebars
    {{#link-to 'photoGallery.recent'}}
      Great Hamster Photos
    {{/link-to}}
    ```
  
    will result in
  
    ```html
    <a href="/hamster-photos/this-week" class="active">
      Great Hamster Photos
    </a>
    ```
  
    The CSS class name used for active classes can be customized
    for a single use of `{{link-to}}` by passing an `activeClass`
    option:
  
    ```handlebars
    {{#link-to 'photoGallery.recent' activeClass="current-url"}}
      Great Hamster Photos
    {{/link-to}}
    ```
  
    ```html
    <a href="/hamster-photos/this-week" class="current-url">
      Great Hamster Photos
    </a>
    ```
  
    To override this option for your entire application, see
    "Overriding Application-wide Defaults".
  
    ### Keeping a link active for other routes
  
    If you need a link to be 'active' even when it doesn't match
    the current route, you can use the `current-when` argument.
  
    ```handlebars
    {{#link-to 'photoGallery' current-when='photos'}}
      Photo Gallery
    {{/link-to}}
    ```
  
    This may be helpful for keeping links active for:
  
    * non-nested routes that are logically related
    * some secondary menu approaches
    * 'top navigation' with 'sub navigation' scenarios
  
    A link will be active if `current-when` is `true` or the current
    route is the route this link would transition to.
  
    To match multiple routes 'space-separate' the routes:
  
    ```handlebars
    {{#link-to 'gallery' current-when='photos drawings paintings'}}
      Art Gallery
    {{/link-to}}
    ```
  
    ### Supplying a model
    An optional model argument can be used for routes whose
    paths contain dynamic segments. This argument will become
    the model context of the linked route:
  
    ```javascript
    Router.map(function() {
      this.route("photoGallery", {path: "hamster-photos/:photo_id"});
    });
    ```
  
    ```handlebars
    {{#link-to 'photoGallery' aPhoto}}
      {{aPhoto.title}}
    {{/link-to}}
    ```
  
    ```html
    <a href="/hamster-photos/42">
      Tomster
    </a>
    ```
  
    ### Supplying multiple models
    For deep-linking to route paths that contain multiple
    dynamic segments, multiple model arguments can be used.
    As the router transitions through the route path, each
    supplied model argument will become the context for the
    route with the dynamic segments:
  
    ```javascript
    Router.map(function() {
      this.route("photoGallery", { path: "hamster-photos/:photo_id" }, function() {
        this.route("comment", {path: "comments/:comment_id"});
      });
    });
    ```
    This argument will become the model context of the linked route:
  
    ```handlebars
    {{#link-to 'photoGallery.comment' aPhoto comment}}
      {{comment.body}}
    {{/link-to}}
    ```
  
    ```html
    <a href="/hamster-photos/42/comments/718">
      A+++ would snuggle again.
    </a>
    ```
  
    ### Supplying an explicit dynamic segment value
    If you don't have a model object available to pass to `{{link-to}}`,
    an optional string or integer argument can be passed for routes whose
    paths contain dynamic segments. This argument will become the value
    of the dynamic segment:
  
    ```javascript
    Router.map(function() {
      this.route("photoGallery", { path: "hamster-photos/:photo_id" });
    });
    ```
  
    ```handlebars
    {{#link-to 'photoGallery' aPhotoId}}
      {{aPhoto.title}}
    {{/link-to}}
    ```
  
    ```html
    <a href="/hamster-photos/42">
      Tomster
    </a>
    ```
  
    When transitioning into the linked route, the `model` hook will
    be triggered with parameters including this passed identifier.
  
    ### Allowing Default Action
  
   By default the `{{link-to}}` component prevents the default browser action
   by calling `preventDefault()` as this sort of action bubbling is normally
   handled internally and we do not want to take the browser to a new URL (for
   example).
  
   If you need to override this behavior specify `preventDefault=false` in
   your template:
  
    ```handlebars
    {{#link-to 'photoGallery' aPhotoId preventDefault=false}}
      {{aPhotoId.title}}
    {{/link-to}}
    ```
  
    ### Overriding attributes
    You can override any given property of the `Ember.LinkComponent`
    that is generated by the `{{link-to}}` component by passing
    key/value pairs, like so:
  
    ```handlebars
    {{#link-to  aPhoto tagName='li' title='Following this link will change your life' classNames='pic sweet'}}
      Uh-mazing!
    {{/link-to}}
    ```
  
    See [Ember.LinkComponent](/api/classes/Ember.LinkComponent.html) for a
    complete list of overrideable properties. Be sure to also
    check out inherited properties of `LinkComponent`.
  
    ### Overriding Application-wide Defaults
    ``{{link-to}}`` creates an instance of `Ember.LinkComponent`
    for rendering. To override options for your entire
    application, reopen `Ember.LinkComponent` and supply the
    desired values:
  
    ``` javascript
    Ember.LinkComponent.reopen({
      activeClass: "is-active",
      tagName: 'li'
    })
    ```
  
    It is also possible to override the default event in
    this manner:
  
    ``` javascript
    Ember.LinkComponent.reopen({
      eventName: 'customEventName'
    });
    ```
  
    @method link-to
    @for Ember.Templates.helpers
    @param {String} routeName
    @param {Object} [context]*
    @param [options] {Object} Handlebars key/value pairs of options, you can override any property of Ember.LinkComponent
    @return {String} HTML string
    @see {Ember.LinkComponent}
    @public
  */

  'use strict';

  /**
    `Ember.LinkComponent` renders an element whose `click` event triggers a
    transition of the application's instance of `Ember.Router` to
    a supplied route by name.
  
    `Ember.LinkComponent` components are invoked with {{#link-to}}. Properties
    of this class can be overridden with `reopen` to customize application-wide
    behavior.
  
    @class LinkComponent
    @namespace Ember
    @extends Ember.Component
    @see {Ember.Templates.helpers.link-to}
    @public
  **/
  var LinkComponent = _emberGlimmerComponent.default.extend({
    layout: _emberGlimmerTemplatesLinkTo.default,

    tagName: 'a',

    /**
      @deprecated Use current-when instead.
      @property currentWhen
      @private
    */
    currentWhen: _emberRuntime.deprecatingAlias('current-when', { id: 'ember-routing-view.deprecated-current-when', until: '3.0.0' }),

    /**
      Used to determine when this `LinkComponent` is active.
       @property currentWhen
      @public
    */
    'current-when': null,

    /**
      Sets the `title` attribute of the `LinkComponent`'s HTML element.
       @property title
      @default null
      @public
    **/
    title: null,

    /**
      Sets the `rel` attribute of the `LinkComponent`'s HTML element.
       @property rel
      @default null
      @public
    **/
    rel: null,

    /**
      Sets the `tabindex` attribute of the `LinkComponent`'s HTML element.
       @property tabindex
      @default null
      @public
    **/
    tabindex: null,

    /**
      Sets the `target` attribute of the `LinkComponent`'s HTML element.
       @since 1.8.0
      @property target
      @default null
      @public
    **/
    target: null,

    /**
      The CSS class to apply to `LinkComponent`'s element when its `active`
      property is `true`.
       @property activeClass
      @type String
      @default active
      @public
    **/
    activeClass: 'active',

    /**
      The CSS class to apply to `LinkComponent`'s element when its `loading`
      property is `true`.
       @property loadingClass
      @type String
      @default loading
      @private
    **/
    loadingClass: 'loading',

    /**
      The CSS class to apply to a `LinkComponent`'s element when its `disabled`
      property is `true`.
       @property disabledClass
      @type String
      @default disabled
      @private
    **/
    disabledClass: 'disabled',
    _isDisabled: false,

    /**
      Determines whether the `LinkComponent` will trigger routing via
      the `replaceWith` routing strategy.
       @property replace
      @type Boolean
      @default false
      @public
    **/
    replace: false,

    /**
      By default the `{{link-to}}` component will bind to the `href` and
      `title` attributes. It's discouraged that you override these defaults,
      however you can push onto the array if needed.
       @property attributeBindings
      @type Array | String
      @default ['title', 'rel', 'tabindex', 'target']
      @public
    */
    attributeBindings: ['href', 'title', 'rel', 'tabindex', 'target'],

    /**
      By default the `{{link-to}}` component will bind to the `active`, `loading`,
      and `disabled` classes. It is discouraged to override these directly.
       @property classNameBindings
      @type Array
      @default ['active', 'loading', 'disabled', 'ember-transitioning-in', 'ember-transitioning-out']
      @public
    */
    classNameBindings: ['active', 'loading', 'disabled', 'transitioningIn', 'transitioningOut'],

    /**
      By default the `{{link-to}}` component responds to the `click` event. You
      can override this globally by setting this property to your custom
      event name.
       This is particularly useful on mobile when one wants to avoid the 300ms
      click delay using some sort of custom `tap` event.
       @property eventName
      @type String
      @default click
      @private
    */
    eventName: 'click',

    // this is doc'ed here so it shows up in the events
    // section of the API documentation, which is where
    // people will likely go looking for it.
    /**
      Triggers the `LinkComponent`'s routing behavior. If
      `eventName` is changed to a value other than `click`
      the routing behavior will trigger on that custom event
      instead.
       @event click
      @private
    */

    /**
      An overridable method called when `LinkComponent` objects are instantiated.
       Example:
       ```javascript
      App.MyLinkComponent = Ember.LinkComponent.extend({
        init: function() {
          this._super(...arguments);
          Ember.Logger.log('Event is ' + this.get('eventName'));
        }
      });
      ```
       NOTE: If you do override `init` for a framework class like `Ember.View`,
      be sure to call `this._super(...arguments)` in your
      `init` declaration! If you don't, Ember may not have an opportunity to
      do important setup work, and you'll see strange behavior in your
      application.
       @method init
      @private
    */
    init: function () {
      this._super.apply(this, arguments);

      // Map desired event name to invoke function
      var eventName = _emberMetal.get(this, 'eventName');
      this.on(eventName, this, this._invoke);
    },

    _routing: _emberRuntime.inject.service('-routing'),

    /**
      Accessed as a classname binding to apply the `LinkComponent`'s `disabledClass`
      CSS `class` to the element when the link is disabled.
       When `true` interactions with the element will not trigger route changes.
      @property disabled
      @private
    */
    disabled: _emberMetal.computed({
      get: function (key, value) {
        return false;
      },
      set: function (key, value) {
        if (value !== undefined) {
          this.set('_isDisabled', value);
        }

        return value ? _emberMetal.get(this, 'disabledClass') : false;
      }
    }),

    _computeActive: function (routerState) {
      if (_emberMetal.get(this, 'loading')) {
        return false;
      }

      var routing = _emberMetal.get(this, '_routing');
      var models = _emberMetal.get(this, 'models');
      var resolvedQueryParams = _emberMetal.get(this, 'resolvedQueryParams');

      var currentWhen = _emberMetal.get(this, 'current-when');
      var isCurrentWhenSpecified = !!currentWhen;
      currentWhen = currentWhen || _emberMetal.get(this, 'qualifiedRouteName');
      currentWhen = currentWhen.split(' ');

      for (var i = 0; i < currentWhen.length; i++) {
        if (routing.isActiveForRoute(models, resolvedQueryParams, currentWhen[i], routerState, isCurrentWhenSpecified)) {
          return _emberMetal.get(this, 'activeClass');
        }
      }

      return false;
    },

    /**
      Accessed as a classname binding to apply the `LinkComponent`'s `activeClass`
      CSS `class` to the element when the link is active.
       A `LinkComponent` is considered active when its `currentWhen` property is `true`
      or the application's current route is the route the `LinkComponent` would trigger
      transitions into.
       The `currentWhen` property can match against multiple routes by separating
      route names using the ` ` (space) character.
       @property active
      @private
    */
    active: _emberMetal.computed('attrs.params', '_routing.currentState', function computeLinkToComponentActive() {
      var currentState = _emberMetal.get(this, '_routing.currentState');
      if (!currentState) {
        return false;
      }

      return this._computeActive(currentState);
    }),

    willBeActive: _emberMetal.computed('_routing.targetState', function computeLinkToComponentWillBeActive() {
      var routing = _emberMetal.get(this, '_routing');
      var targetState = _emberMetal.get(routing, 'targetState');
      if (_emberMetal.get(routing, 'currentState') === targetState) {
        return;
      }

      return !!this._computeActive(targetState);
    }),

    transitioningIn: _emberMetal.computed('active', 'willBeActive', function computeLinkToComponentTransitioningIn() {
      var willBeActive = _emberMetal.get(this, 'willBeActive');
      if (typeof willBeActive === 'undefined') {
        return false;
      }

      return !_emberMetal.get(this, 'active') && willBeActive && 'ember-transitioning-in';
    }),

    transitioningOut: _emberMetal.computed('active', 'willBeActive', function computeLinkToComponentTransitioningOut() {
      var willBeActive = _emberMetal.get(this, 'willBeActive');
      if (typeof willBeActive === 'undefined') {
        return false;
      }

      return _emberMetal.get(this, 'active') && !willBeActive && 'ember-transitioning-out';
    }),

    /**
      Event handler that invokes the link, activating the associated route.
       @method _invoke
      @param {Event} event
      @private
    */
    _invoke: function (event) {
      if (!_emberViews.isSimpleClick(event)) {
        return true;
      }

      var preventDefault = _emberMetal.get(this, 'preventDefault');
      var targetAttribute = _emberMetal.get(this, 'target');

      if (preventDefault !== false) {
        if (!targetAttribute || targetAttribute === '_self') {
          event.preventDefault();
        }
      }

      if (_emberMetal.get(this, 'bubbles') === false) {
        event.stopPropagation();
      }

      if (_emberMetal.get(this, '_isDisabled')) {
        return false;
      }

      if (_emberMetal.get(this, 'loading')) {
        _emberConsole.default.warn('This link-to is in an inactive loading state because at least one of its parameters presently has a null/undefined value, or the provided route name is invalid.');
        return false;
      }

      if (targetAttribute && targetAttribute !== '_self') {
        return false;
      }

      var qualifiedRouteName = _emberMetal.get(this, 'qualifiedRouteName');
      var models = _emberMetal.get(this, 'models');
      var queryParams = _emberMetal.get(this, 'queryParams.values');
      var shouldReplace = _emberMetal.get(this, 'replace');

      var payload = {
        queryParams: queryParams,
        routeName: qualifiedRouteName
      };

      _emberMetal.flaggedInstrument('interaction.link-to', payload, this._generateTransition(payload, qualifiedRouteName, models, queryParams, shouldReplace));
    },

    _generateTransition: function (payload, qualifiedRouteName, models, queryParams, shouldReplace) {
      var routing = _emberMetal.get(this, '_routing');
      return function () {
        payload.transition = routing.transitionTo(qualifiedRouteName, models, queryParams, shouldReplace);
      };
    },

    queryParams: null,

    qualifiedRouteName: _emberMetal.computed('targetRouteName', '_routing.currentState', function computeLinkToComponentQualifiedRouteName() {
      var params = _emberMetal.get(this, 'params').slice();
      var lastParam = params[params.length - 1];
      if (lastParam && lastParam.isQueryParams) {
        params.pop();
      }
      var onlyQueryParamsSupplied = this[_emberGlimmerComponent.HAS_BLOCK] ? params.length === 0 : params.length === 1;
      if (onlyQueryParamsSupplied) {
        return _emberMetal.get(this, '_routing.currentRouteName');
      }
      return _emberMetal.get(this, 'targetRouteName');
    }),

    resolvedQueryParams: _emberMetal.computed('queryParams', function computeLinkToComponentResolvedQueryParams() {
      var resolvedQueryParams = {};
      var queryParams = _emberMetal.get(this, 'queryParams');

      if (!queryParams) {
        return resolvedQueryParams;
      }

      var values = queryParams.values;
      for (var key in values) {
        if (!values.hasOwnProperty(key)) {
          continue;
        }
        resolvedQueryParams[key] = values[key];
      }

      return resolvedQueryParams;
    }),

    /**
      Sets the element's `href` attribute to the url for
      the `LinkComponent`'s targeted route.
       If the `LinkComponent`'s `tagName` is changed to a value other
      than `a`, this property will be ignored.
       @property href
      @private
    */
    href: _emberMetal.computed('models', 'qualifiedRouteName', function computeLinkToComponentHref() {
      if (_emberMetal.get(this, 'tagName') !== 'a') {
        return;
      }

      var qualifiedRouteName = _emberMetal.get(this, 'qualifiedRouteName');
      var models = _emberMetal.get(this, 'models');

      if (_emberMetal.get(this, 'loading')) {
        return _emberMetal.get(this, 'loadingHref');
      }

      var routing = _emberMetal.get(this, '_routing');
      var queryParams = _emberMetal.get(this, 'queryParams.values');

      return routing.generateURL(qualifiedRouteName, models, queryParams);
    }),

    loading: _emberMetal.computed('_modelsAreLoaded', 'qualifiedRouteName', function computeLinkToComponentLoading() {
      var qualifiedRouteName = _emberMetal.get(this, 'qualifiedRouteName');
      var modelsAreLoaded = _emberMetal.get(this, '_modelsAreLoaded');

      if (!modelsAreLoaded || qualifiedRouteName == null) {
        return _emberMetal.get(this, 'loadingClass');
      }
    }),

    _modelsAreLoaded: _emberMetal.computed('models', function computeLinkToComponentModelsAreLoaded() {
      var models = _emberMetal.get(this, 'models');
      for (var i = 0; i < models.length; i++) {
        if (models[i] == null) {
          return false;
        }
      }

      return true;
    }),

    _getModels: function (params) {
      var modelCount = params.length - 1;
      var models = new Array(modelCount);

      for (var i = 0; i < modelCount; i++) {
        var value = params[i + 1];

        while (_emberRuntime.ControllerMixin.detect(value)) {
          value = value.get('model');
        }

        models[i] = value;
      }

      return models;
    },

    /**
      The default href value to use while a link-to is loading.
      Only applies when tagName is 'a'
       @property loadingHref
      @type String
      @default #
      @private
    */
    loadingHref: '#',

    didReceiveAttrs: function () {
      var queryParams = undefined;

      var params = _emberMetal.get(this, 'params');

      if (params) {
        // Do not mutate params in place
        params = params.slice();
      }

      var disabledWhen = _emberMetal.get(this, 'disabledWhen');
      if (disabledWhen !== undefined) {
        this.set('disabled', disabledWhen);
      }

      // Process the positional arguments, in order.
      // 1. Inline link title comes first, if present.
      if (!this[_emberGlimmerComponent.HAS_BLOCK]) {
        this.set('linkTitle', params.shift());
      }

      // 2. `targetRouteName` is now always at index 0.
      this.set('targetRouteName', params[0]);

      // 3. The last argument (if still remaining) is the `queryParams` object.
      var lastParam = params[params.length - 1];

      if (lastParam && lastParam.isQueryParams) {
        queryParams = params.pop();
      } else {
        queryParams = { values: {} };
      }
      this.set('queryParams', queryParams);

      // 4. Any remaining indices (excepting `targetRouteName` at 0) are `models`.
      if (params.length > 1) {
        this.set('models', this._getModels(params));
      } else {
        this.set('models', []);
      }
    }
  });

  LinkComponent.toString = function () {
    return 'LinkComponent';
  };

  LinkComponent.reopenClass({
    positionalParams: 'params'
  });

  exports.default = LinkComponent;
});

/*
 * Unfortunately, to get decent error messages, we need to do this.
 * In some future state we should be able to use a "feature flag"
 * which allows us to strip this without needing to call it twice.
 *
 * if (isDebugBuild()) {
 *   // Do the useful debug thing, probably including try/catch.
 * } else {
 *   // Do the performant thing.
 * }
 */
enifed('ember-glimmer/components/text_area', ['exports', 'ember-glimmer/component', 'ember-views', 'ember-glimmer/templates/empty'], function (exports, _emberGlimmerComponent, _emberViews, _emberGlimmerTemplatesEmpty) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  /**
    `{{textarea}}` inserts a new instance of `<textarea>` tag into the template.
    The attributes of `{{textarea}}` match those of the native HTML tags as
    closely as possible.
  
    The following HTML attributes can be set:
  
      * `value`
      * `name`
      * `rows`
      * `cols`
      * `placeholder`
      * `disabled`
      * `maxlength`
      * `tabindex`
      * `selectionEnd`
      * `selectionStart`
      * `selectionDirection`
      * `wrap`
      * `readonly`
      * `autofocus`
      * `form`
      * `spellcheck`
      * `required`
  
    When set to a quoted string, these value will be directly applied to the HTML
    element. When left unquoted, these values will be bound to a property on the
    template's current rendering context (most typically a controller instance).
  
    Unbound:
  
    ```handlebars
    {{textarea value="Lots of static text that ISN'T bound"}}
    ```
  
    Would result in the following HTML:
  
    ```html
    <textarea class="ember-text-area">
      Lots of static text that ISN'T bound
    </textarea>
    ```
  
    Bound:
  
    In the following example, the `writtenWords` property on `App.ApplicationController`
    will be updated live as the user types 'Lots of text that IS bound' into
    the text area of their browser's window.
  
    ```javascript
    App.ApplicationController = Ember.Controller.extend({
      writtenWords: "Lots of text that IS bound"
    });
    ```
  
    ```handlebars
    {{textarea value=writtenWords}}
    ```
  
     Would result in the following HTML:
  
    ```html
    <textarea class="ember-text-area">
      Lots of text that IS bound
    </textarea>
    ```
  
    If you wanted a one way binding between the text area and a div tag
    somewhere else on your screen, you could use `Ember.computed.oneWay`:
  
    ```javascript
    App.ApplicationController = Ember.Controller.extend({
      writtenWords: "Lots of text that IS bound",
      outputWrittenWords: Ember.computed.oneWay("writtenWords")
    });
    ```
  
    ```handlebars
    {{textarea value=writtenWords}}
    <div>
      {{outputWrittenWords}}
    </div>
    ```
  
    Would result in the following HTML:
  
    ```html
    <textarea class="ember-text-area">
      Lots of text that IS bound
    </textarea>
    <-- the following div will be updated in real time as you type -->
    <div>
      Lots of text that IS bound
    </div>
    ```
  
    Finally, this example really shows the power and ease of Ember when two
    properties are bound to eachother via `Ember.computed.alias`. Type into
    either text area box and they'll both stay in sync. Note that
    `Ember.computed.alias` costs more in terms of performance, so only use it when
    your really binding in both directions:
  
    ```javascript
    App.ApplicationController = Ember.Controller.extend({
      writtenWords: "Lots of text that IS bound",
      twoWayWrittenWords: Ember.computed.alias("writtenWords")
    });
    ```
  
    ```handlebars
    {{textarea value=writtenWords}}
    {{textarea value=twoWayWrittenWords}}
    ```
  
    ```html
    <textarea id="ember1" class="ember-text-area">
      Lots of text that IS bound
    </textarea>
    <-- both updated in real time -->
    <textarea id="ember2" class="ember-text-area">
      Lots of text that IS bound
    </textarea>
    ```
  
    ### Actions
  
    The helper can send multiple actions based on user events.
    The action property defines the action which is send when
    the user presses the return key.
  
    ```handlebars
    {{input action="submit"}}
    ```
  
    The helper allows some user events to send actions.
  
    * `enter`
    * `insert-newline`
    * `escape-press`
    * `focus-in`
    * `focus-out`
    * `key-press`
  
    For example, if you desire an action to be sent when the input is blurred,
    you only need to setup the action name to the event name property.
  
    ```handlebars
    {{textarea focus-in="alertMessage"}}
    ```
  
    See more about [Text Support Actions](/api/classes/Ember.TextArea.html)
  
    ### Extension
  
    Internally, `{{textarea}}` creates an instance of `Ember.TextArea`, passing
    arguments from the helper to `Ember.TextArea`'s `create` method. You can
    extend the capabilities of text areas in your application by reopening this
    class. For example, if you are building a Bootstrap project where `data-*`
    attributes are used, you can globally add support for a `data-*` attribute
    on all `{{textarea}}`s' in your app by reopening `Ember.TextArea` or
    `Ember.TextSupport` and adding it to the `attributeBindings` concatenated
    property:
  
    ```javascript
    Ember.TextArea.reopen({
      attributeBindings: ['data-error']
    });
    ```
  
    Keep in mind when writing `Ember.TextArea` subclasses that `Ember.TextArea`
    itself extends `Ember.Component`. Expect isolated component semantics, not
    legacy 1.x view semantics (like `controller` being present).
  
    See more about [Ember components](/api/classes/Ember.Component.html)
  
    @method textarea
    @for Ember.Templates.helpers
    @param {Hash} options
    @public
  */

  /**
    The internal class used to create textarea element when the `{{textarea}}`
    helper is used.
  
    See [Ember.Templates.helpers.textarea](/api/classes/Ember.Templates.helpers.html#method_textarea)  for usage details.
  
    ## Layout and LayoutName properties
  
    Because HTML `textarea` elements do not contain inner HTML the `layout` and
    `layoutName` properties will not be applied. See [Ember.View](/api/classes/Ember.View.html)'s
    layout section for more information.
  
    @class TextArea
    @namespace Ember
    @extends Ember.Component
    @uses Ember.TextSupport
    @public
  */
  exports.default = _emberGlimmerComponent.default.extend(_emberViews.TextSupport, {
    classNames: ['ember-text-area'],

    layout: _emberGlimmerTemplatesEmpty.default,

    tagName: 'textarea',
    attributeBindings: ['rows', 'cols', 'name', 'selectionEnd', 'selectionStart', 'wrap', 'lang', 'dir', 'value'],
    rows: null,
    cols: null
  });
});
enifed('ember-glimmer/components/text_field', ['exports', 'ember-utils', 'ember-metal', 'ember-environment', 'ember-glimmer/component', 'ember-glimmer/templates/empty', 'ember-views'], function (exports, _emberUtils, _emberMetal, _emberEnvironment, _emberGlimmerComponent, _emberGlimmerTemplatesEmpty, _emberViews) {
  /**
  @module ember
  @submodule ember-views
  */
  'use strict';

  var inputTypeTestElement = undefined;
  var inputTypes = new _emberUtils.EmptyObject();
  function canSetTypeOfInput(type) {
    if (type in inputTypes) {
      return inputTypes[type];
    }

    // if running in outside of a browser always return the
    // original type
    if (!_emberEnvironment.environment.hasDOM) {
      inputTypes[type] = type;

      return type;
    }

    if (!inputTypeTestElement) {
      inputTypeTestElement = document.createElement('input');
    }

    try {
      inputTypeTestElement.type = type;
    } catch (e) {}

    return inputTypes[type] = inputTypeTestElement.type === type;
  }

  /**
  
    The internal class used to create text inputs when the `{{input}}`
    helper is used with `type` of `text`.
  
    See [Ember.Templates.helpers.input](/api/classes/Ember.Templates.helpers.html#method_input)  for usage details.
  
    ## Layout and LayoutName properties
  
    Because HTML `input` elements are self closing `layout` and `layoutName`
    properties will not be applied. See [Ember.View](/api/classes/Ember.View.html)'s
    layout section for more information.
  
    @class TextField
    @namespace Ember
    @extends Ember.Component
    @uses Ember.TextSupport
    @public
  */
  exports.default = _emberGlimmerComponent.default.extend(_emberViews.TextSupport, {
    layout: _emberGlimmerTemplatesEmpty.default,
    classNames: ['ember-text-field'],
    tagName: 'input',
    attributeBindings: ['accept', 'autocomplete', 'autosave', 'dir', 'formaction', 'formenctype', 'formmethod', 'formnovalidate', 'formtarget', 'height', 'inputmode', 'lang', 'list', 'max', 'min', 'multiple', 'name', 'pattern', 'size', 'step', 'type', 'value', 'width'],

    /**
      The `value` attribute of the input element. As the user inputs text, this
      property is updated live.
       @property value
      @type String
      @default ""
      @public
    */
    value: '',

    /**
      The `type` attribute of the input element.
       @property type
      @type String
      @default "text"
      @public
    */
    type: _emberMetal.computed({
      get: function () {
        return 'text';
      },

      set: function (key, value) {
        var type = 'text';

        if (canSetTypeOfInput(value)) {
          type = value;
        }

        return type;
      }
    }),

    /**
      The `size` of the text field in characters.
       @property size
      @type String
      @default null
      @public
    */
    size: null,

    /**
      The `pattern` attribute of input element.
       @property pattern
      @type String
      @default null
      @public
    */
    pattern: null,

    /**
      The `min` attribute of input element used with `type="number"` or `type="range"`.
       @property min
      @type String
      @default null
      @since 1.4.0
      @public
    */
    min: null,

    /**
      The `max` attribute of input element used with `type="number"` or `type="range"`.
       @property max
      @type String
      @default null
      @since 1.4.0
      @public
    */
    max: null
  });
});
enifed('ember-glimmer/dom', ['exports', 'glimmer-runtime', 'glimmer-node'], function (exports, _glimmerRuntime, _glimmerNode) {
  'use strict';

  exports.DOMChanges = _glimmerRuntime.DOMChanges;
  exports.DOMTreeConstruction = _glimmerRuntime.DOMTreeConstruction;
  exports.NodeDOMTreeConstruction = _glimmerNode.NodeDOMTreeConstruction;
});
enifed('ember-glimmer/environment', ['exports', 'ember-utils', 'ember-metal', 'ember-views', 'glimmer-runtime', 'ember-glimmer/syntax/curly-component', 'ember-glimmer/syntax', 'ember-glimmer/syntax/dynamic-component', 'ember-glimmer/utils/iterable', 'ember-glimmer/utils/references', 'ember-glimmer/utils/debug-stack', 'ember-glimmer/helpers/if-unless', 'ember-glimmer/utils/bindings', 'ember-glimmer/helpers/action', 'ember-glimmer/helpers/component', 'ember-glimmer/helpers/concat', 'ember-glimmer/helpers/debugger', 'ember-glimmer/helpers/get', 'ember-glimmer/helpers/hash', 'ember-glimmer/helpers/loc', 'ember-glimmer/helpers/log', 'ember-glimmer/helpers/mut', 'ember-glimmer/helpers/readonly', 'ember-glimmer/helpers/unbound', 'ember-glimmer/helpers/-class', 'ember-glimmer/helpers/-input-type', 'ember-glimmer/helpers/query-param', 'ember-glimmer/helpers/each-in', 'ember-glimmer/helpers/-normalize-class', 'ember-glimmer/helpers/-html-safe', 'ember-glimmer/protocol-for-url', 'ember-glimmer/modifiers/action'], function (exports, _emberUtils, _emberMetal, _emberViews, _glimmerRuntime, _emberGlimmerSyntaxCurlyComponent, _emberGlimmerSyntax, _emberGlimmerSyntaxDynamicComponent, _emberGlimmerUtilsIterable, _emberGlimmerUtilsReferences, _emberGlimmerUtilsDebugStack, _emberGlimmerHelpersIfUnless, _emberGlimmerUtilsBindings, _emberGlimmerHelpersAction, _emberGlimmerHelpersComponent, _emberGlimmerHelpersConcat, _emberGlimmerHelpersDebugger, _emberGlimmerHelpersGet, _emberGlimmerHelpersHash, _emberGlimmerHelpersLoc, _emberGlimmerHelpersLog, _emberGlimmerHelpersMut, _emberGlimmerHelpersReadonly, _emberGlimmerHelpersUnbound, _emberGlimmerHelpersClass, _emberGlimmerHelpersInputType, _emberGlimmerHelpersQueryParam, _emberGlimmerHelpersEachIn, _emberGlimmerHelpersNormalizeClass, _emberGlimmerHelpersHtmlSafe, _emberGlimmerProtocolForUrl, _emberGlimmerModifiersAction) {
  'use strict';

  var builtInComponents = {
    textarea: '-text-area'
  };

  var Environment = (function (_GlimmerEnvironment) {
    babelHelpers.inherits(Environment, _GlimmerEnvironment);

    Environment.create = function create(options) {
      return new Environment(options);
    };

    function Environment(_ref) {
      var _this = this;

      var owner = _ref[_emberUtils.OWNER];

      _GlimmerEnvironment.apply(this, arguments);
      this.owner = owner;
      this.isInteractive = owner.lookup('-environment:main').isInteractive;

      // can be removed once https://github.com/tildeio/glimmer/pull/305 lands
      this.destroyedComponents = [];

      _emberGlimmerProtocolForUrl.default(this);

      this._definitionCache = new _emberMetal.Cache(2000, function (_ref2) {
        var name = _ref2.name;
        var source = _ref2.source;
        var owner = _ref2.owner;

        var _lookupComponent = _emberViews.lookupComponent(owner, name, { source: source });

        var ComponentClass = _lookupComponent.component;
        var layout = _lookupComponent.layout;

        if (ComponentClass || layout) {
          return new _emberGlimmerSyntaxCurlyComponent.CurlyComponentDefinition(name, ComponentClass, layout);
        }
      }, function (_ref3) {
        var name = _ref3.name;
        var source = _ref3.source;
        var owner = _ref3.owner;

        var expandedName = source && owner._resolveLocalLookupName(name, source) || name;
        var ownerGuid = _emberUtils.guidFor(owner);

        return ownerGuid + '|' + expandedName;
      });

      this._templateCache = new _emberMetal.Cache(1000, function (_ref4) {
        var Template = _ref4.Template;
        var owner = _ref4.owner;

        if (Template.create) {
          var _Template$create;

          // we received a factory
          return Template.create((_Template$create = { env: _this }, _Template$create[_emberUtils.OWNER] = owner, _Template$create));
        } else {
          // we were provided an instance already
          return Template;
        }
      }, function (_ref5) {
        var Template = _ref5.Template;
        var owner = _ref5.owner;
        return _emberUtils.guidFor(owner) + '|' + Template.id;
      });

      this._compilerCache = new _emberMetal.Cache(10, function (Compiler) {
        return new _emberMetal.Cache(2000, function (template) {
          var compilable = new Compiler(template);
          return _glimmerRuntime.compileLayout(compilable, _this);
        }, function (template) {
          var owner = template.meta.owner;
          return _emberUtils.guidFor(owner) + '|' + template.id;
        });
      }, function (Compiler) {
        return Compiler.id;
      });

      this.builtInModifiers = {
        action: new _emberGlimmerModifiersAction.default()
      };

      this.builtInHelpers = {
        if: _emberGlimmerHelpersIfUnless.inlineIf,
        action: _emberGlimmerHelpersAction.default,
        component: _emberGlimmerHelpersComponent.default,
        concat: _emberGlimmerHelpersConcat.default,
        debugger: _emberGlimmerHelpersDebugger.default,
        get: _emberGlimmerHelpersGet.default,
        hash: _emberGlimmerHelpersHash.default,
        loc: _emberGlimmerHelpersLoc.default,
        log: _emberGlimmerHelpersLog.default,
        mut: _emberGlimmerHelpersMut.default,
        'query-params': _emberGlimmerHelpersQueryParam.default,
        readonly: _emberGlimmerHelpersReadonly.default,
        unbound: _emberGlimmerHelpersUnbound.default,
        unless: _emberGlimmerHelpersIfUnless.inlineUnless,
        '-class': _emberGlimmerHelpersClass.default,
        '-each-in': _emberGlimmerHelpersEachIn.default,
        '-input-type': _emberGlimmerHelpersInputType.default,
        '-normalize-class': _emberGlimmerHelpersNormalizeClass.default,
        '-html-safe': _emberGlimmerHelpersHtmlSafe.default,
        '-get-dynamic-var': _glimmerRuntime.getDynamicVar
      };
    }

    // Hello future traveler, welcome to the world of syntax refinement.
    // The method below is called by Glimmer's runtime compiler to allow
    // us to take generic statement syntax and refine it to more meaniful
    // syntax for Ember's use case. This on the fly switch-a-roo sounds fine
    // and dandy, however Ember has precedence on statement refinement that you
    // need to be aware of. The presendence for language constructs is as follows:
    //
    // ------------------------
    // Native & Built-in Syntax
    // ------------------------
    //   User-land components
    // ------------------------
    //     User-land helpers
    // ------------------------
    //
    // The one caveat here is that Ember also allows for dashed references that are
    // not a component or helper:
    //
    // export default Component.extend({
    //   'foo-bar': 'LAME'
    // });
    //
    // {{foo-bar}}
    //
    // The heuristic for the above situation is a dashed "key" in inline form
    // that does not resolve to a defintion. In this case refine statement simply
    // isn't going to return any syntax and the Glimmer engine knows how to handle
    // this case.

    Environment.prototype.refineStatement = function refineStatement(statement, symbolTable) {
      // 1. resolve any native syntax – if, unless, with, each, and partial
      var nativeSyntax = _GlimmerEnvironment.prototype.refineStatement.call(this, statement, symbolTable);

      if (nativeSyntax) {
        return nativeSyntax;
      }

      var appendType = statement.appendType;
      var isSimple = statement.isSimple;
      var isInline = statement.isInline;
      var isBlock = statement.isBlock;
      var isModifier = statement.isModifier;
      var key = statement.key;
      var path = statement.path;
      var args = statement.args;

      if (isSimple && (isInline || isBlock) && appendType !== 'get') {
        // 2. built-in syntax

        var RefinedSyntax = _emberGlimmerSyntax.findSyntaxBuilder(key);
        if (RefinedSyntax) {
          return RefinedSyntax.create(this, args, symbolTable);
        }

        var internalKey = builtInComponents[key];
        var definition = null;

        if (internalKey) {
          definition = this.getComponentDefinition([internalKey], symbolTable);
        } else if (key.indexOf('-') >= 0) {
          definition = this.getComponentDefinition(path, symbolTable);
        }

        if (definition) {
          _emberGlimmerUtilsBindings.wrapComponentClassAttribute(args);

          return new _emberGlimmerSyntaxCurlyComponent.CurlyComponentSyntax(args, definition, symbolTable);
        }
      }

      if (isInline && !isSimple && appendType !== 'helper') {
        return statement.original.deopt();
      }

      if (!isSimple && path) {
        return _emberGlimmerSyntaxDynamicComponent.DynamicComponentSyntax.fromPath(this, path, args, symbolTable);
      }
    };

    Environment.prototype.hasComponentDefinition = function hasComponentDefinition() {
      return false;
    };

    Environment.prototype.getComponentDefinition = function getComponentDefinition(path, symbolTable) {
      var name = path[0];
      var blockMeta = symbolTable.getMeta();
      var owner = blockMeta.owner;
      var source = blockMeta.moduleName && 'template:' + blockMeta.moduleName;

      return this._definitionCache.get({ name: name, source: source, owner: owner });
    };

    // normally templates should be exported at the proper module name
    // and cached in the container, but this cache supports templates
    // that have been set directly on the component's layout property

    Environment.prototype.getTemplate = function getTemplate(Template, owner) {
      return this._templateCache.get({ Template: Template, owner: owner });
    };

    // a Compiler can wrap the template so it needs its own cache

    Environment.prototype.getCompiledBlock = function getCompiledBlock(Compiler, template) {
      var compilerCache = this._compilerCache.get(Compiler);
      return compilerCache.get(template);
    };

    Environment.prototype.hasPartial = function hasPartial(name, symbolTable) {
      var _symbolTable$getMeta = symbolTable.getMeta();

      var owner = _symbolTable$getMeta.owner;

      return _emberViews.hasPartial(name, owner);
    };

    Environment.prototype.lookupPartial = function lookupPartial(name, symbolTable) {
      var _symbolTable$getMeta2 = symbolTable.getMeta();

      var owner = _symbolTable$getMeta2.owner;

      var partial = {
        template: _emberViews.lookupPartial(name, owner)
      };

      if (partial.template) {
        return partial;
      } else {
        throw new Error(name + ' is not a partial');
      }
    };

    Environment.prototype.hasHelper = function hasHelper(nameParts, symbolTable) {

      // helpers are not allowed to include a dot in their invocation
      if (nameParts.length > 1) {
        return false;
      }

      var name = nameParts[0];

      if (this.builtInHelpers[name]) {
        return true;
      }

      var blockMeta = symbolTable.getMeta();
      var owner = blockMeta.owner;
      var options = { source: 'template:' + blockMeta.moduleName };

      return owner.hasRegistration('helper:' + name, options) || owner.hasRegistration('helper:' + name);
    };

    Environment.prototype.lookupHelper = function lookupHelper(nameParts, symbolTable) {

      var name = nameParts[0];
      var helper = this.builtInHelpers[name];

      if (helper) {
        return helper;
      }

      var blockMeta = symbolTable.getMeta();
      var owner = blockMeta.owner;
      var options = blockMeta.moduleName && { source: 'template:' + blockMeta.moduleName } || {};

      helper = owner.lookup('helper:' + name, options) || owner.lookup('helper:' + name);

      // TODO: try to unify this into a consistent protocol to avoid wasteful closure allocations
      if (helper.isHelperInstance) {
        return function (vm, args) {
          return _emberGlimmerUtilsReferences.SimpleHelperReference.create(helper.compute, args);
        };
      } else if (helper.isHelperFactory) {
        return function (vm, args) {
          return _emberGlimmerUtilsReferences.ClassBasedHelperReference.create(helper, vm, args);
        };
      } else {
        throw new Error(nameParts + ' is not a helper');
      }
    };

    Environment.prototype.hasModifier = function hasModifier(nameParts) {

      // modifiers are not allowed to include a dot in their invocation
      if (nameParts.length > 1) {
        return false;
      }

      return !!this.builtInModifiers[nameParts[0]];
    };

    Environment.prototype.lookupModifier = function lookupModifier(nameParts) {

      var modifier = this.builtInModifiers[nameParts[0]];

      if (modifier) {
        return modifier;
      } else {
        throw new Error(nameParts + ' is not a modifier');
      }
    };

    Environment.prototype.toConditionalReference = function toConditionalReference(reference) {
      return _emberGlimmerUtilsReferences.ConditionalReference.create(reference);
    };

    Environment.prototype.iterableFor = function iterableFor(ref, args) {
      var keyPath = args.named.get('key').value();
      return _emberGlimmerUtilsIterable.default(ref, keyPath);
    };

    Environment.prototype.scheduleInstallModifier = function scheduleInstallModifier() {
      if (this.isInteractive) {
        var _GlimmerEnvironment$prototype$scheduleInstallModifier;

        (_GlimmerEnvironment$prototype$scheduleInstallModifier = _GlimmerEnvironment.prototype.scheduleInstallModifier).call.apply(_GlimmerEnvironment$prototype$scheduleInstallModifier, [this].concat(babelHelpers.slice.call(arguments)));
      }
    };

    Environment.prototype.scheduleUpdateModifier = function scheduleUpdateModifier() {
      if (this.isInteractive) {
        var _GlimmerEnvironment$prototype$scheduleUpdateModifier;

        (_GlimmerEnvironment$prototype$scheduleUpdateModifier = _GlimmerEnvironment.prototype.scheduleUpdateModifier).call.apply(_GlimmerEnvironment$prototype$scheduleUpdateModifier, [this].concat(babelHelpers.slice.call(arguments)));
      }
    };

    Environment.prototype.didDestroy = function didDestroy(destroyable) {
      destroyable.destroy();
    };

    Environment.prototype.begin = function begin() {
      this.inTransaction = true;

      _GlimmerEnvironment.prototype.begin.call(this);
    };

    Environment.prototype.commit = function commit() {
      var destroyedComponents = this.destroyedComponents;
      this.destroyedComponents = [];
      // components queued for destruction must be destroyed before firing
      // `didCreate` to prevent errors when removing and adding a component
      // with the same name (would throw an error when added to view registry)
      for (var i = 0; i < destroyedComponents.length; i++) {
        destroyedComponents[i].destroy();
      }

      _GlimmerEnvironment.prototype.commit.call(this);

      this.inTransaction = false;
    };

    return Environment;
  })(_glimmerRuntime.Environment);

  exports.default = Environment;
});
enifed('ember-glimmer/helper', ['exports', 'ember-utils', 'ember-runtime', 'glimmer-reference'], function (exports, _emberUtils, _emberRuntime, _glimmerReference) {
  /**
  @module ember
  @submodule ember-glimmer
  */

  'use strict';

  exports.helper = helper;
  var RECOMPUTE_TAG = _emberUtils.symbol('RECOMPUTE_TAG');

  exports.RECOMPUTE_TAG = RECOMPUTE_TAG;
  /**
    Ember Helpers are functions that can compute values, and are used in templates.
    For example, this code calls a helper named `format-currency`:
  
    ```handlebars
    <div>{{format-currency cents currency="$"}}</div>
    ```
  
    Additionally a helper can be called as a nested helper (sometimes called a
    subexpression). In this example, the computed value of a helper is passed
    to a component named `show-money`:
  
    ```handlebars
    {{show-money amount=(format-currency cents currency="$")}}
    ```
  
    Helpers defined using a class must provide a `compute` function. For example:
  
    ```js
    export default Ember.Helper.extend({
      compute(params, hash) {
        let cents = params[0];
        let currency = hash.currency;
        return `${currency}${cents * 0.01}`;
      }
    });
    ```
  
    Each time the input to a helper changes, the `compute` function will be
    called again.
  
    As instances, these helpers also have access to the container an will accept
    injected dependencies.
  
    Additionally, class helpers can call `recompute` to force a new computation.
  
    @class Ember.Helper
    @public
    @since 1.13.0
  */
  var Helper = _emberRuntime.FrameworkObject.extend({
    isHelperInstance: true,

    init: function () {
      this._super.apply(this, arguments);
      this[RECOMPUTE_TAG] = new _glimmerReference.DirtyableTag();
    },

    /**
      On a class-based helper, it may be useful to force a recomputation of that
      helpers value. This is akin to `rerender` on a component.
       For example, this component will rerender when the `currentUser` on a
      session service changes:
       ```js
      // app/helpers/current-user-email.js
      export default Ember.Helper.extend({
        session: Ember.inject.service(),
        onNewUser: Ember.observer('session.currentUser', function() {
          this.recompute();
        }),
        compute() {
          return this.get('session.currentUser.email');
        }
      });
      ```
       @method recompute
      @public
      @since 1.13.0
    */
    recompute: function () {
      this[RECOMPUTE_TAG].dirty();
    }

    /**
      Override this function when writing a class-based helper.
       @method compute
      @param {Array} params The positional arguments to the helper
      @param {Object} hash The named arguments to the helper
      @public
      @since 1.13.0
    */
  });

  Helper.reopenClass({
    isHelperFactory: true
  });

  /**
    In many cases, the ceremony of a full `Ember.Helper` class is not required.
    The `helper` method create pure-function helpers without instances. For
    example:
  
    ```js
    // app/helpers/format-currency.js
    export default Ember.Helper.helper(function(params, hash) {
      let cents = params[0];
      let currency = hash.currency;
      return `${currency}${cents * 0.01}`;
    });
    ```
  
    @static
    @param {Function} helper The helper function
    @method helper
    @public
    @since 1.13.0
  */

  function helper(helperFn) {
    return {
      isHelperInstance: true,
      compute: helperFn
    };
  }

  exports.default = Helper;
});
enifed('ember-glimmer/helpers/-class', ['exports', 'ember-glimmer/utils/references', 'ember-runtime'], function (exports, _emberGlimmerUtilsReferences, _emberRuntime) {
  'use strict';

  function classHelper(_ref) {
    var positional = _ref.positional;

    var path = positional.at(0);
    var args = positional.length;
    var value = path.value();

    if (value === true) {
      if (args > 1) {
        return _emberRuntime.String.dasherize(positional.at(1).value());
      }
      return null;
    }

    if (value === false) {
      if (args > 2) {
        return _emberRuntime.String.dasherize(positional.at(2).value());
      }
      return null;
    }

    return value;
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(classHelper, args);
  };
});
enifed('ember-glimmer/helpers/-html-safe', ['exports', 'ember-glimmer/utils/references', 'ember-glimmer/utils/string'], function (exports, _emberGlimmerUtilsReferences, _emberGlimmerUtilsString) {
  'use strict';

  function htmlSafe(_ref) {
    var positional = _ref.positional;

    var path = positional.at(0);
    return new _emberGlimmerUtilsString.SafeString(path.value());
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(htmlSafe, args);
  };
});
enifed('ember-glimmer/helpers/-input-type', ['exports', 'ember-glimmer/utils/references'], function (exports, _emberGlimmerUtilsReferences) {
  'use strict';

  function inputTypeHelper(_ref) {
    var positional = _ref.positional;
    var named = _ref.named;

    var type = positional.at(0).value();
    if (type === 'checkbox') {
      return '-checkbox';
    }
    return '-text-field';
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(inputTypeHelper, args);
  };
});
enifed('ember-glimmer/helpers/-normalize-class', ['exports', 'ember-glimmer/utils/references', 'ember-runtime'], function (exports, _emberGlimmerUtilsReferences, _emberRuntime) {
  'use strict';

  function normalizeClass(_ref) {
    var positional = _ref.positional;
    var named = _ref.named;

    var classNameParts = positional.at(0).value().split('.');
    var className = classNameParts[classNameParts.length - 1];
    var value = positional.at(1).value();

    if (value === true) {
      return _emberRuntime.String.dasherize(className);
    } else if (!value && value !== 0) {
      return '';
    } else {
      return String(value);
    }
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(normalizeClass, args);
  };
});
enifed('ember-glimmer/helpers/action', ['exports', 'ember-utils', 'ember-metal', 'ember-glimmer/utils/references', 'glimmer-runtime', 'glimmer-reference'], function (exports, _emberUtils, _emberMetal, _emberGlimmerUtilsReferences, _glimmerRuntime, _glimmerReference) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  var INVOKE = _emberUtils.symbol('INVOKE');
  exports.INVOKE = INVOKE;
  var ACTION = _emberUtils.symbol('ACTION');

  exports.ACTION = ACTION;
  /**
    The `{{action}}` helper provides a way to pass triggers for behavior (usually
    just a function) between components, and into components from controllers.
  
    ### Passing functions with the action helper
  
    There are three contexts an action helper can be used in. The first two
    contexts to discuss are attribute context, and Handlebars value context.
  
    ```handlebars
    {{! An example of attribute context }}
    <div onclick={{action "save"}}></div>
    {{! Examples of Handlebars value context }}
    {{input on-input=(action "save")}}
    {{yield (action "refreshData") andAnotherParam}}
    ```
  
    In these contexts,
    the helper is called a "closure action" helper. Its behavior is simple:
    If passed a function name, read that function off the `actions` property
    of the current context. Once that function is read (or if a function was
    passed), create a closure over that function and any arguments.
    The resulting value of an action helper used this way is simply a function.
  
    For example, in the attribute context:
  
    ```handlebars
    {{! An example of attribute context }}
    <div onclick={{action "save"}}></div>
    ```
  
    The resulting template render logic would be:
  
    ```js
    var div = document.createElement('div');
    var actionFunction = (function(context){
      return function() {
        return context.actions.save.apply(context, arguments);
      };
    })(context);
    div.onclick = actionFunction;
    ```
  
    Thus when the div is clicked, the action on that context is called.
    Because the `actionFunction` is just a function, closure actions can be
    passed between components and still execute in the correct context.
  
    Here is an example action handler on a component:
  
    ```js
    export default Ember.Component.extend({
      actions: {
        save() {
          this.get('model').save();
        }
      }
    });
    ```
  
    Actions are always looked up on the `actions` property of the current context.
    This avoids collisions in the naming of common actions, such as `destroy`.
    Two options can be passed to the `action` helper when it is used in this way.
  
    * `target=someProperty` will look to `someProperty` instead of the current
      context for the `actions` hash. This can be useful when targetting a
      service for actions.
    * `value="target.value"` will read the path `target.value` off the first
      argument to the action when it is called and rewrite the first argument
      to be that value. This is useful when attaching actions to event listeners.
  
    ### Invoking an action
  
    Closure actions curry both their scope and any arguments. When invoked, any
    additional arguments are added to the already curried list.
    Actions should be invoked using the [sendAction](/api/classes/Ember.Component.html#method_sendAction)
    method. The first argument to `sendAction` is the action to be called, and
    additional arguments are passed to the action function. This has interesting
    properties combined with currying of arguments. For example:
  
    ```js
    export default Ember.Component.extend({
      actions: {
        // Usage {{input on-input=(action (action 'setName' model) value="target.value")}}
        setName(model, name) {
          model.set('name', name);
        }
      }
    });
    ```
  
    The first argument (`model`) was curried over, and the run-time argument (`event`)
    becomes a second argument. Action calls can be nested this way because each simply
    returns a function. Any function can be passed to the `{{action}}` helper, including
    other actions.
  
    Actions invoked with `sendAction` have the same currying behavior as demonstrated
    with `on-input` above. For example:
  
    ```js
    export default Ember.Component.extend({
      actions: {
        setName(model, name) {
          model.set('name', name);
        }
      }
    });
    ```
  
    ```handlebars
    {{my-input submit=(action 'setName' model)}}
    ```
  
    ```js
    // app/components/my-component.js
    export default Ember.Component.extend({
      click() {
        // Note that model is not passed, it was curried in the template
        this.sendAction('submit', 'bob');
      }
    });
    ```
  
    ### Attaching actions to DOM elements
  
    The third context of the `{{action}}` helper can be called "element space".
    For example:
  
    ```handlebars
    {{! An example of element space }}
    <div {{action "save"}}></div>
    ```
  
    Used this way, the `{{action}}` helper provides a useful shortcut for
    registering an HTML element in a template for a single DOM event and
    forwarding that interaction to the template's context (controller or component).
    If the context of a template is a controller, actions used this way will
    bubble to routes when the controller does not implement the specified action.
    Once an action hits a route, it will bubble through the route hierarchy.
  
    ### Event Propagation
  
    `{{action}}` helpers called in element space can control event bubbling. Note
    that the closure style actions cannot.
  
    Events triggered through the action helper will automatically have
    `.preventDefault()` called on them. You do not need to do so in your event
    handlers. If you need to allow event propagation (to handle file inputs for
    example) you can supply the `preventDefault=false` option to the `{{action}}` helper:
  
    ```handlebars
    <div {{action "sayHello" preventDefault=false}}>
      <input type="file" />
      <input type="checkbox" />
    </div>
    ```
  
    To disable bubbling, pass `bubbles=false` to the helper:
  
    ```handlebars
    <button {{action 'edit' post bubbles=false}}>Edit</button>
    ```
  
    To disable bubbling with closure style actions you must create your own
    wrapper helper that makes use of `event.stopPropagation()`:
  
    ```handlebars
    <div onclick={{disable-bubbling (action "sayHello")}}>Hello</div>
    ```
  
    ```js
    // app/helpers/disable-bubbling.js
    import Ember from 'ember';
    export function disableBubbling([action]) {
      return function(event) {
        event.stopPropagation();
        return action(event);
      };
    }
    export default Ember.Helper.helper(disableBubbling);
    ```
  
    If you need the default handler to trigger you should either register your
    own event handler, or use event methods on your view class. See
    ["Responding to Browser Events"](/api/classes/Ember.View.html#toc_responding-to-browser-events)
    in the documentation for Ember.View for more information.
  
    ### Specifying DOM event type
  
    `{{action}}` helpers called in element space can specify an event type.
    By default the `{{action}}` helper registers for DOM `click` events. You can
    supply an `on` option to the helper to specify a different DOM event name:
  
    ```handlebars
    <div {{action "anActionName" on="doubleClick"}}>
      click me
    </div>
    ```
  
    See ["Event Names"](/api/classes/Ember.View.html#toc_event-names) for a list of
    acceptable DOM event names.
  
    ### Specifying whitelisted modifier keys
  
    `{{action}}` helpers called in element space can specify modifier keys.
    By default the `{{action}}` helper will ignore click events with pressed modifier
    keys. You can supply an `allowedKeys` option to specify which keys should not be ignored.
  
    ```handlebars
    <div {{action "anActionName" allowedKeys="alt"}}>
      click me
    </div>
    ```
  
    This way the action will fire when clicking with the alt key pressed down.
    Alternatively, supply "any" to the `allowedKeys` option to accept any combination of modifier keys.
  
    ```handlebars
    <div {{action "anActionName" allowedKeys="any"}}>
      click me with any key pressed
    </div>
    ```
  
    ### Specifying a Target
  
    A `target` option can be provided to the helper to change
    which object will receive the method call. This option must be a path
    to an object, accessible in the current context:
  
    ```handlebars
    {{! app/templates/application.hbs }}
    <div {{action "anActionName" target=someService}}>
      click me
    </div>
    ```
  
    ```javascript
    // app/controllers/application.js
    export default Ember.Controller.extend({
      someService: Ember.inject.service()
    });
    ```
  
    @method action
    @for Ember.Templates.helpers
    @public
  */

  exports.default = function (vm, args) {
    var named = args.named;
    var positional = args.positional;

    // The first two argument slots are reserved.
    // pos[0] is the context (or `this`)
    // pos[1] is the action name or function
    // Anything else is an action argument.
    var context = positional.at(0);
    var action = positional.at(1);

    // TODO: Is there a better way of doing this?
    var debugKey = action._propertyKey;

    var restArgs = undefined;

    if (positional.length === 2) {
      restArgs = _glimmerRuntime.EvaluatedPositionalArgs.empty();
    } else {
      restArgs = _glimmerRuntime.EvaluatedPositionalArgs.create(positional.values.slice(2));
    }

    var target = named.has('target') ? named.get('target') : context;
    var processArgs = makeArgsProcessor(named.has('value') && named.get('value'), restArgs);

    var fn = undefined;

    if (typeof action[INVOKE] === 'function') {
      fn = makeClosureAction(action, action, action[INVOKE], processArgs, debugKey);
    } else if (_glimmerReference.isConst(target) && _glimmerReference.isConst(action)) {
      fn = makeClosureAction(context.value(), target.value(), action.value(), processArgs, debugKey);
    } else {
      fn = makeDynamicClosureAction(context.value(), target, action, processArgs, debugKey);
    }

    fn[ACTION] = true;

    return new _emberGlimmerUtilsReferences.UnboundReference(fn);
  };

  function NOOP(args) {
    return args;
  }

  function makeArgsProcessor(valuePathRef, actionArgsRef) {
    var mergeArgs = null;

    if (actionArgsRef.length > 0) {
      mergeArgs = function (args) {
        return actionArgsRef.value().concat(args);
      };
    }

    var readValue = null;

    if (valuePathRef) {
      readValue = function (args) {
        var valuePath = valuePathRef.value();

        if (valuePath && args.length > 0) {
          args[0] = _emberMetal.get(args[0], valuePath);
        }

        return args;
      };
    }

    if (mergeArgs && readValue) {
      return function (args) {
        return readValue(mergeArgs(args));
      };
    } else {
      return mergeArgs || readValue || NOOP;
    }
  }

  function makeDynamicClosureAction(context, targetRef, actionRef, processArgs, debugKey) {

    return function () {
      return makeClosureAction(context, targetRef.value(), actionRef.value(), processArgs, debugKey).apply(undefined, arguments);
    };

    // We don't allow undefined/null values, so this creates a throw-away action to trigger the assertions
  }

  function makeClosureAction(context, target, action, processArgs, debugKey) {
    var self = undefined,
        fn = undefined;

    if (typeof action[INVOKE] === 'function') {
      self = action;
      fn = action[INVOKE];
    } else {
      var typeofAction = typeof action;

      if (typeofAction === 'string') {
        self = target;
        fn = target.actions && target.actions[action];
      } else if (typeofAction === 'function') {
        self = context;
        fn = action;
      } else {}
    }

    return function () {
      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      var payload = { target: self, args: args, label: 'glimmer-closure-action' };
      return _emberMetal.flaggedInstrument('interaction.ember-action', payload, function () {
        return _emberMetal.run.join.apply(_emberMetal.run, [self, fn].concat(processArgs(args)));
      });
    };
  }
});
enifed('ember-glimmer/helpers/component', ['exports', 'ember-utils', 'ember-glimmer/utils/references', 'ember-glimmer/syntax/curly-component', 'glimmer-runtime', 'ember-metal'], function (exports, _emberUtils, _emberGlimmerUtilsReferences, _emberGlimmerSyntaxCurlyComponent, _glimmerRuntime, _emberMetal) {
  /**
    @module ember
    @submodule ember-glimmer
  */
  'use strict';

  /**
    The `{{component}}` helper lets you add instances of `Ember.Component` to a
    template. See [Ember.Component](/api/classes/Ember.Component.html) for
    additional information on how a `Component` functions.
    `{{component}}`'s primary use is for cases where you want to dynamically
    change which type of component is rendered as the state of your application
    changes. This helper has three modes: inline, block, and nested.
  
    ### Inline Form
  
    Given the following template:
  
    ```app/application.hbs
    {{component infographicComponentName}}
    ```
  
    And the following application code:
  
    ```app/controllers/application.js
    export default Ember.Controller.extend({
      infographicComponentName: computed('isMarketOpen', {
        get() {
          if (this.get('isMarketOpen')) {
            return 'live-updating-chart';
          } else {
            return 'market-close-summary';
          }
        }
      })
    });
    ```
  
    The `live-updating-chart` component will be appended when `isMarketOpen` is
    `true`, and the `market-close-summary` component will be appended when
    `isMarketOpen` is `false`. If the value changes while the app is running,
    the component will be automatically swapped out accordingly.
    Note: You should not use this helper when you are consistently rendering the same
    component. In that case, use standard component syntax, for example:
  
    ```app/templates/application.hbs
    {{live-updating-chart}}
    ```
  
    ### Block Form
  
    Using the block form of this helper is similar to using the block form
    of a component. Given the following application template:
  
    ```app/templates/application.hbs
    {{#component infographicComponentName}}
      Last update: {{lastUpdateTimestamp}}
    {{/component}}
    ```
  
    The following controller code:
  
    ```app/controllers/application.js
    export default Ember.Controller.extend({
      lastUpdateTimestamp: computed(function() {
        return new Date();
      }),
  
      infographicComponentName: computed('isMarketOpen', {
        get() {
          if (this.get('isMarketOpen')) {
            return 'live-updating-chart';
          } else {
            return 'market-close-summary';
          }
        }
      })
    });
    ```
  
    And the following component template:
  
    ```app/templates/components/live-updating-chart.hbs
    {{! chart }}
    {{yield}}
    ```
  
    The `Last Update: {{lastUpdateTimestamp}}` will be rendered in place of the `{{yield}}`.
  
    ### Nested Usage
  
    The `component` helper can be used to package a component path with initial attrs.
    The included attrs can then be merged during the final invocation.
    For example, given a `person-form` component with the following template:
  
    ```app/templates/components/person-form.hbs
    {{yield (hash
      nameInput=(component "my-input-component" value=model.name placeholder="First Name")
    )}}
    ```
  
    When yielding the component via the `hash` helper, the component is invoked directly.
    See the following snippet:
  
    ```
    {{#person-form as |form|}}
      {{form.nameInput placeholder="Username"}}
    {{/person-form}}
    ```
  
    Which outputs an input whose value is already bound to `model.name` and `placeholder`
    is "Username".
  
    When yielding the component without the hash helper use the `component` helper.
    For example, below is a `full-name` component template:
  
    ```handlebars
    {{yield (component "my-input-component" value=model.name placeholder="Name")}}
    ```
  
    ```
    {{#full-name as |field|}}
      {{component field placeholder="Full name"}}
    {{/full-name}}
    ```
  
    @method component
    @since 1.11.0
    @for Ember.Templates.helpers
    @public
  */

  var ClosureComponentReference = (function (_CachedReference) {
    babelHelpers.inherits(ClosureComponentReference, _CachedReference);

    ClosureComponentReference.create = function create(args, symbolTable, env) {
      return new ClosureComponentReference(args, symbolTable, env);
    };

    function ClosureComponentReference(args, symbolTable, env) {
      _CachedReference.call(this);
      this.defRef = args.positional.at(0);
      this.env = env;
      this.tag = args.positional.at(0).tag;
      this.symbolTable = symbolTable;
      this.args = args;
      this.lastDefinition = undefined;
      this.lastName = undefined;
    }

    ClosureComponentReference.prototype.compute = function compute() {
      // TODO: Figure out how to extract this because it's nearly identical to
      // DynamicComponentReference::compute(). The only differences besides
      // currying are in the assertion messages.
      var args = this.args;
      var defRef = this.defRef;
      var env = this.env;
      var symbolTable = this.symbolTable;
      var lastDefinition = this.lastDefinition;
      var lastName = this.lastName;

      var nameOrDef = defRef.value();
      var definition = null;

      if (nameOrDef && nameOrDef === lastName) {
        return lastDefinition;
      }

      this.lastName = nameOrDef;

      if (typeof nameOrDef === 'string') {
        definition = env.getComponentDefinition([nameOrDef], symbolTable);
      } else if (_glimmerRuntime.isComponentDefinition(nameOrDef)) {
        definition = nameOrDef;
      } else {
        return null;
      }

      var newDef = createCurriedDefinition(definition, args);

      this.lastDefinition = newDef;

      return newDef;
    };

    return ClosureComponentReference;
  })(_emberGlimmerUtilsReferences.CachedReference);

  exports.ClosureComponentReference = ClosureComponentReference;

  function createCurriedDefinition(definition, args) {
    var curriedArgs = curryArgs(definition, args);

    return new _emberGlimmerSyntaxCurlyComponent.CurlyComponentDefinition(definition.name, definition.ComponentClass, definition.template, curriedArgs);
  }

  function curryArgs(definition, newArgs) {
    var args = definition.args;
    var ComponentClass = definition.ComponentClass;
    var positionalParams = ComponentClass.positionalParams;

    // The args being passed in are from the (component ...) invocation,
    // so the first positional argument is actually the name or component
    // definition. It needs to be dropped in order for any actual positional
    // args to coincide with the ComponentClass's positionParams.

    // For "normal" curly components this slicing is done at the syntax layer,
    // but we don't have that luxury here.

    var _newArgs$positional$values = newArgs.positional.values;

    var slicedPositionalArgs = _newArgs$positional$values.slice(1);

    if (positionalParams && slicedPositionalArgs.length) {
      _emberGlimmerSyntaxCurlyComponent.validatePositionalParameters(newArgs.named, slicedPositionalArgs, positionalParams);
    }

    var isRest = typeof positionalParams === 'string';

    // For non-rest position params, we need to perform the position -> name mapping
    // at each layer to avoid a collision later when the args are used to construct
    // the component instance (inside of processArgs(), inside of create()).
    var positionalToNamedParams = {};

    if (!isRest && positionalParams && positionalParams.length > 0) {
      var limit = Math.min(positionalParams.length, slicedPositionalArgs.length);

      for (var i = 0; i < limit; i++) {
        var _name = positionalParams[i];
        positionalToNamedParams[_name] = slicedPositionalArgs[i];
      }

      slicedPositionalArgs.length = 0; // Throw them away since you're merged in.
    }

    // args (aka 'oldArgs') may be undefined or simply be empty args, so
    // we need to fall back to an empty array or object.
    var oldNamed = args && args.named && args.named.map || {};
    var oldPositional = args && args.positional && args.positional.values || [];

    // Merge positional arrays
    var mergedPositional = new Array(Math.max(oldPositional.length, slicedPositionalArgs.length));
    mergedPositional.splice.apply(mergedPositional, [0, oldPositional.length].concat(oldPositional));
    mergedPositional.splice.apply(mergedPositional, [0, slicedPositionalArgs.length].concat(slicedPositionalArgs));

    // Merge named maps
    var mergedNamed = _emberUtils.assign({}, oldNamed, positionalToNamedParams, newArgs.named.map);

    var mergedArgs = _glimmerRuntime.EvaluatedArgs.create(_glimmerRuntime.EvaluatedPositionalArgs.create(mergedPositional), _glimmerRuntime.EvaluatedNamedArgs.create(mergedNamed), _glimmerRuntime.Blocks.empty());

    return mergedArgs;
  }

  exports.default = function (vm, args, symbolTable) {
    return ClosureComponentReference.create(args, symbolTable, vm.env);
  };
});
enifed('ember-glimmer/helpers/concat', ['exports', 'ember-glimmer/utils/references', 'glimmer-runtime'], function (exports, _emberGlimmerUtilsReferences, _glimmerRuntime) {
  'use strict';

  /**
  @module ember
  @submodule ember-glimmer
  */

  /**
    Concatenates the given arguments into a string.
  
    Example:
  
    ```handlebars
    {{some-component name=(concat firstName " " lastName)}}
  
    {{! would pass name="<first name value> <last name value>" to the component}}
    ```
  
    @public
    @method concat
    @for Ember.Templates.helpers
    @since 1.13.0
  */
  function concat(_ref) {
    var positional = _ref.positional;

    return positional.value().map(_glimmerRuntime.normalizeTextValue).join('');
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(concat, args);
  };
});
enifed('ember-glimmer/helpers/debugger', ['exports', 'ember-metal/debug', 'glimmer-runtime'], function (exports, _emberMetalDebug, _glimmerRuntime) {
  /*jshint debug:true*/

  /**
  @module ember
  @submodule ember-htmlbars
  */

  'use strict';

  exports.default = debuggerHelper;
  exports.setDebuggerCallback = setDebuggerCallback;
  exports.resetDebuggerCallback = resetDebuggerCallback;

  /**
    Execute the `debugger` statement in the current template's context.
  
    ```handlebars
    {{debugger}}
    ```
  
    When using the debugger helper you will have access to a `get` function. This
    function retrieves values available in the context of the template.
    For example, if you're wondering why a value `{{foo}}` isn't rendering as
    expected within a template, you could place a `{{debugger}}` statement and,
    when the `debugger;` breakpoint is hit, you can attempt to retrieve this value:
  
    ```
    > get('foo')
    ```
  
    `get` is also aware of block variables. So in this situation
  
    ```handlebars
    {{#each items as |item|}}
      {{debugger}}
    {{/each}}
    ```
  
    You'll be able to get values from the current item:
  
    ```
    > get('item.name')
    ```
  
    You can also access the context of the view to make sure it is the object that
    you expect:
  
    ```
    > context
    ```
  
    @method debugger
    @for Ember.Templates.helpers
    @public
  */
  function defaultCallback(context, get) {

    debugger;

    /* jshint debug: true */
  }

  var callback = defaultCallback;

  function debuggerHelper(vm, args, symbolTable) {
    var context = vm.getSelf().value();

    // Note: this is totally an overkill since we are only compiling
    // expressions, but this is the only kind of SymbolLookup we can
    // construct. The symbol table itself should really be sufficient
    // here – we should refactor the Glimmer code to make that possible.
    var symbolLookup = new _glimmerRuntime.CompileIntoList(vm.env, symbolTable);

    function get(path) {
      // Problem: technically, we are getting a `PublicVM` here, but to
      // evaluate an expression it requires the full VM. We happen to know
      // that they are the same thing, so this would work for now. However
      // this might break in the future.
      return _glimmerRuntime.GetSyntax.build(path).compile(symbolLookup).evaluate(vm).value();
    }

    callback(context, get);

    return _glimmerRuntime.UNDEFINED_REFERENCE;
  }

  // These are exported for testing

  function setDebuggerCallback(newCallback) {
    callback = newCallback;
  }

  function resetDebuggerCallback() {
    callback = defaultCallback;
  }
});
enifed('ember-glimmer/helpers/each-in', ['exports', 'ember-utils'], function (exports, _emberUtils) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  exports.isEachIn = isEachIn;

  /**
    The `{{#each}}` helper loops over elements in a collection. It is an extension
    of the base Handlebars `{{#each}}` helper.
    The default behavior of `{{#each}}` is to yield its inner block once for every
    item in an array passing the item as the first block parameter.
  
    ```javascript
    var developers = [{ name: 'Yehuda' },{ name: 'Tom' }, { name: 'Paul' }];
    ```
  
    ```handlebars
    {{#each developers key="name" as |person|}}
      {{person.name}}
      {{! `this` is whatever it was outside the #each }}
    {{/each}}
    ```
  
    The same rules apply to arrays of primitives.
  
    ```javascript
    var developerNames = ['Yehuda', 'Tom', 'Paul']
    ```
  
    ```handlebars
    {{#each developerNames key="@index" as |name|}}
      {{name}}
    {{/each}}
    ```
  
    During iteration, the index of each item in the array is provided as a second block parameter.
  
    ```handlebars
    <ul>
      {{#each people as |person index|}}
        <li>Hello, {{person.name}}! You're number {{index}} in line</li>
      {{/each}}
    </ul>
    ```
  
    ### Specifying Keys
  
    The `key` option is used to tell Ember how to determine if the array being
    iterated over with `{{#each}}` has changed between renders. By helping Ember
    detect that some elements in the array are the same, DOM elements can be
    re-used, significantly improving rendering speed.
  
    For example, here's the `{{#each}}` helper with its `key` set to `id`:
  
    ```handlebars
    {{#each model key="id" as |item|}}
    {{/each}}
    ```
  
    When this `{{#each}}` re-renders, Ember will match up the previously rendered
    items (and reorder the generated DOM elements) based on each item's `id`
    property.
    By default the item's own reference is used.
  
    ### {{else}} condition
  
    `{{#each}}` can have a matching `{{else}}`. The contents of this block will render
    if the collection is empty.
  
    ```handlebars
    {{#each developers as |person|}}
      {{person.name}}
    {{else}}
      <p>Sorry, nobody is available for this task.</p>
    {{/each}}
    ```
  
    @method each
    @for Ember.Templates.helpers
    @public
   */

  /**
    The `{{each-in}}` helper loops over properties on an object.
  
    For example, given a `user` object that looks like:
  
    ```javascript
    {
      "name": "Shelly Sails",
      "age": 42
    }
    ```
  
    This template would display all properties on the `user`
    object in a list:
  
    ```handlebars
    <ul>
    {{#each-in user as |key value|}}
      <li>{{key}}: {{value}}</li>
    {{/each-in}}
    </ul>
    ```
  
    Outputting their name and age.
  
    @method each-in
    @for Ember.Templates.helpers
    @public
    @since 2.1.0
  */
  var EACH_IN_REFERENCE = _emberUtils.symbol('EACH_IN');

  function isEachIn(ref) {
    return ref && ref[EACH_IN_REFERENCE];
  }

  exports.default = function (vm, args) {
    var ref = Object.create(args.positional.at(0));
    ref[EACH_IN_REFERENCE] = true;
    return ref;
  };
});
enifed('ember-glimmer/helpers/get', ['exports', 'ember-metal', 'ember-glimmer/utils/references', 'glimmer-reference'], function (exports, _emberMetal, _emberGlimmerUtilsReferences, _glimmerReference) {
  'use strict';

  /**
  @module ember
  @submodule ember-glimmer
  */

  /**
    Dynamically look up a property on an object. The second argument to `{{get}}`
    should have a string value, although it can be bound.
  
    For example, these two usages are equivilent:
  
    ```handlebars
    {{person.height}}
    {{get person "height"}}
    ```
  
    If there were several facts about a person, the `{{get}}` helper can dynamically
    pick one:
  
    ```handlebars
    {{get person factName}}
    ```
  
    For a more complex example, this template would allow the user to switch
    between showing the user's height and weight with a click:
  
    ```handlebars
    {{get person factName}}
    <button {{action (action (mut factName)) "height"}}>Show height</button>
    <button {{action (action (mut factName)) "weight"}}>Show weight</button>
    ```
  
    The `{{get}}` helper can also respect mutable values itself. For example:
  
    ```handlebars
    {{input value=(mut (get person factName)) type="text"}}
    <button {{action (action (mut factName)) "height"}}>Show height</button>
    <button {{action (action (mut factName)) "weight"}}>Show weight</button>
    ```
  
    Would allow the user to swap what fact is being displayed, and also edit
    that fact via a two-way mutable binding.
  
    @public
    @method get
    @for Ember.Templates.helpers
    @since 2.1.0
   */

  exports.default = function (vm, args) {
    return GetHelperReference.create(args.positional.at(0), args.positional.at(1));
  };

  var GetHelperReference = (function (_CachedReference) {
    babelHelpers.inherits(GetHelperReference, _CachedReference);

    GetHelperReference.create = function create(sourceReference, pathReference) {
      if (_glimmerReference.isConst(pathReference)) {
        var parts = pathReference.value().split('.');
        return _glimmerReference.referenceFromParts(sourceReference, parts);
      } else {
        return new GetHelperReference(sourceReference, pathReference);
      }
    };

    function GetHelperReference(sourceReference, pathReference) {
      _CachedReference.call(this);
      this.sourceReference = sourceReference;
      this.pathReference = pathReference;

      this.lastPath = null;
      this.innerReference = null;

      var innerTag = this.innerTag = new _glimmerReference.UpdatableTag(_glimmerReference.CONSTANT_TAG);

      this.tag = _glimmerReference.combine([sourceReference.tag, pathReference.tag, innerTag]);
    }

    GetHelperReference.prototype.compute = function compute() {
      var lastPath = this.lastPath;
      var innerReference = this.innerReference;
      var innerTag = this.innerTag;

      var path = this.lastPath = this.pathReference.value();

      if (path !== lastPath) {
        if (path) {
          var pathType = typeof path;

          if (pathType === 'string') {
            innerReference = this.innerReference = _glimmerReference.referenceFromParts(this.sourceReference, path.split('.'));
          } else if (pathType === 'number') {
            innerReference = this.innerReference = this.sourceReference.get(path);
          }

          innerTag.update(innerReference.tag);
        } else {
          innerReference = this.innerReference = null;
          innerTag.update(_glimmerReference.CONSTANT_TAG);
        }
      }

      return innerReference ? innerReference.value() : null;
    };

    GetHelperReference.prototype[_emberGlimmerUtilsReferences.UPDATE] = function (value) {
      _emberMetal.set(this.sourceReference.value(), this.pathReference.value(), value);
    };

    return GetHelperReference;
  })(_emberGlimmerUtilsReferences.CachedReference);
});
enifed("ember-glimmer/helpers/hash", ["exports"], function (exports) {
   /**
   @module ember
   @submodule ember-glimmer
   */

   /**
      Use the `{{hash}}` helper to create a hash to pass as an option to your
      components. This is specially useful for contextual components where you can
      just yield a hash:
   
      ```handlebars
      {{yield (hash
         name='Sarah'
         title=office
      )}}
      ```
   
      Would result in an object such as:
   
      ```js
      { name: 'Sarah', title: this.get('office') }
      ```
   
      Where the `title` is bound to updates of the `office` property.
   
      @method hash
      @for Ember.Templates.helpers
      @param {Object} options
      @return {Object} Hash
      @public
    */

   "use strict";

   exports.default = function (vm, args) {
      return args.named;
   };
});
enifed('ember-glimmer/helpers/if-unless', ['exports', 'ember-metal', 'ember-glimmer/utils/references', 'glimmer-reference'], function (exports, _emberMetal, _emberGlimmerUtilsReferences, _glimmerReference) {
  /**
  @module ember
  @submodule ember-glimmer
  */

  'use strict';

  exports.inlineIf = inlineIf;
  exports.inlineUnless = inlineUnless;

  /**
    Use the `if` block helper to conditionally render a block depending on a
    property. If the property is "falsey", for example: `false`, `undefined`,
    `null`, `""`, `0`, `NaN` or an empty array, the block will not be rendered.
  
    ```handlebars
    {{! will not render if foo is falsey}}
    {{#if foo}}
      Welcome to the {{foo.bar}}
    {{/if}}
    ```
  
    You can also specify a template to show if the property is falsey by using
    the `else` helper.
  
    ```handlebars
    {{! is it raining outside?}}
    {{#if isRaining}}
      Yes, grab an umbrella!
    {{else}}
      No, it's lovely outside!
    {{/if}}
    ```
  
    You are also able to combine `else` and `if` helpers to create more complex
    conditional logic.
  
    ```handlebars
    {{#if isMorning}}
      Good morning
    {{else if isAfternoon}}
      Good afternoon
    {{else}}
      Good night
    {{/if}}
    ```
  
    You can use `if` inline to conditionally render a single property or string.
    This helper acts like a ternary operator. If the first property is truthy,
    the second argument will be displayed, if not, the third argument will be
    displayed
  
    ```handlebars
    {{if useLongGreeting "Hello" "Hi"}} Dave
    ```
  
    Finally, you can use the `if` helper inside another helper as a subexpression.
  
    ```handlebars
    {{some-component height=(if isBig "100" "10")}}
    ```
  
    @method if
    @for Ember.Templates.helpers
    @public
  */

  var ConditionalHelperReference = (function (_CachedReference) {
    babelHelpers.inherits(ConditionalHelperReference, _CachedReference);

    ConditionalHelperReference.create = function create(_condRef, _truthyRef, _falsyRef) {
      var condRef = _emberGlimmerUtilsReferences.ConditionalReference.create(_condRef);
      var truthyRef = _truthyRef || _emberGlimmerUtilsReferences.UNDEFINED_REFERENCE;
      var falsyRef = _falsyRef || _emberGlimmerUtilsReferences.UNDEFINED_REFERENCE;

      if (_glimmerReference.isConst(condRef)) {
        return condRef.value() ? truthyRef : falsyRef;
      } else {
        return new ConditionalHelperReference(condRef, truthyRef, falsyRef);
      }
    };

    function ConditionalHelperReference(cond, truthy, falsy) {
      _CachedReference.call(this);

      this.branchTag = new _glimmerReference.UpdatableTag(_glimmerReference.CONSTANT_TAG);
      this.tag = _glimmerReference.combine([cond.tag, this.branchTag]);

      this.cond = cond;
      this.truthy = truthy;
      this.falsy = falsy;
    }

    /**
      The inline `if` helper conditionally renders a single property or string.
      This helper acts like a ternary operator. If the first property is truthy,
      the second argument will be displayed, otherwise, the third argument will be
      displayed
    
      ```handlebars
      {{if useLongGreeting "Hello" "Hi"}} Alex
      ```
    
      You can use the `if` helper inside another helper as a subexpression.
    
      ```handlebars
      {{some-component height=(if isBig "100" "10")}}
      ```
    
      @method if
      @for Ember.Templates.helpers
      @public
    */

    ConditionalHelperReference.prototype.compute = function compute() {
      var cond = this.cond;
      var truthy = this.truthy;
      var falsy = this.falsy;

      var branch = cond.value() ? truthy : falsy;

      this.branchTag.update(branch.tag);

      return branch.value();
    };

    return ConditionalHelperReference;
  })(_emberGlimmerUtilsReferences.CachedReference);

  function inlineIf(vm, _ref) {
    var positional = _ref.positional;

    switch (positional.length) {
      case 2:
        return ConditionalHelperReference.create(positional.at(0), positional.at(1), null);
      case 3:
        return ConditionalHelperReference.create(positional.at(0), positional.at(1), positional.at(2));
      default:
    }
  }

  /**
    The inline `unless` helper conditionally renders a single property or string.
    This helper acts like a ternary operator. If the first property is falsy,
    the second argument will be displayed, otherwise, the third argument will be
    displayed
  
    ```handlebars
    {{unless useLongGreeting "Hi" "Hello"}} Ben
    ```
  
    You can use the `unless` helper inside another helper as a subexpression.
  
    ```handlebars
    {{some-component height=(unless isBig "10" "100")}}
    ```
  
    @method unless
    @for Ember.Templates.helpers
    @public
  */

  function inlineUnless(vm, _ref2) {
    var positional = _ref2.positional;

    switch (positional.length) {
      case 2:
        return ConditionalHelperReference.create(positional.at(0), null, positional.at(1));
      case 3:
        return ConditionalHelperReference.create(positional.at(0), positional.at(2), positional.at(1));
      default:
    }
  }
});
enifed('ember-glimmer/helpers/loc', ['exports', 'ember-glimmer/utils/references', 'ember-runtime'], function (exports, _emberGlimmerUtilsReferences, _emberRuntime) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  /**
    Calls [Ember.String.loc](/api/classes/Ember.String.html#method_loc) with the
    provided string. This is a convenient way to localize text within a template.
    For example:
  
    ```javascript
    Ember.STRINGS = {
      '_welcome_': 'Bonjour'
    };
    ```
  
    ```handlebars
    <div class='message'>
      {{loc '_welcome_'}}
    </div>
    ```
  
    ```html
    <div class='message'>
      Bonjour
    </div>
    ```
  
    See [Ember.String.loc](/api/classes/Ember.String.html#method_loc) for how to
    set up localized string references.
  
    @method loc
    @for Ember.Templates.helpers
    @param {String} str The string to format.
    @see {Ember.String#loc}
    @public
  */
  function locHelper(_ref) {
    var positional = _ref.positional;

    return _emberRuntime.String.loc.apply(null, positional.value());
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(locHelper, args);
  };
});
enifed('ember-glimmer/helpers/log', ['exports', 'ember-glimmer/utils/references', 'ember-console'], function (exports, _emberGlimmerUtilsReferences, _emberConsole) {
  'use strict';

  /**
    `log` allows you to output the value of variables in the current rendering
    context. `log` also accepts primitive types such as strings or numbers.
  
    ```handlebars
    {{log "myVariable:" myVariable }}
    ```
  
    @method log
    @for Ember.Templates.helpers
    @param {Array} params
    @public
  */
  function log(_ref) {
    var positional = _ref.positional;

    _emberConsole.default.log.apply(null, positional.value());
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(log, args);
  };
});

/**
@module ember
@submodule ember-glimmer
*/
enifed('ember-glimmer/helpers/mut', ['exports', 'ember-utils', 'ember-metal', 'ember-glimmer/utils/references', 'ember-glimmer/helpers/action'], function (exports, _emberUtils, _emberMetal, _emberGlimmerUtilsReferences, _emberGlimmerHelpersAction) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  exports.isMut = isMut;
  exports.unMut = unMut;

  /**
    The `mut` helper lets you __clearly specify__ that a child `Component` can update the
    (mutable) value passed to it, which will __change the value of the parent component__.
  
    To specify that a parameter is mutable, when invoking the child `Component`:
  
    ```handlebars
    {{my-child childClickCount=(mut totalClicks)}}
    ```
  
    The child `Component` can then modify the parent's value just by modifying its own
    property:
  
    ```javascript
    // my-child.js
    export default Component.extend({
      click() {
        this.incrementProperty('childClickCount');
      }
    });
    ```
  
    Note that for curly components (`{{my-component}}`) the bindings are already mutable,
    making the `mut` unnecessary.
  
    Additionally, the `mut` helper can be combined with the `action` helper to
    mutate a value. For example:
  
    ```handlebars
    {{my-child childClickCount=totalClicks click-count-change=(action (mut totalClicks))}}
    ```
  
    The child `Component` would invoke the action with the new click value:
  
    ```javascript
    // my-child.js
    export default Component.extend({
      click() {
        this.get('click-count-change')(this.get('childClickCount') + 1);
      }
    });
    ```
  
    The `mut` helper changes the `totalClicks` value to what was provided as the action argument.
  
    The `mut` helper, when used with `action`, will return a function that
    sets the value passed to `mut` to its first argument. This works like any other
    closure action and interacts with the other features `action` provides.
    As an example, we can create a button that increments a value passing the value
    directly to the `action`:
  
    ```handlebars
    {{! inc helper is not provided by Ember }}
    <button onclick={{action (mut count) (inc count)}}>
      Increment count
    </button>
    ```
  
    You can also use the `value` option:
  
    ```handlebars
    <input value={{name}} oninput={{action (mut name) value="target.value"}}>
    ```
  
    @method mut
    @param {Object} [attr] the "two-way" attribute that can be modified.
    @for Ember.Templates.helpers
    @public
  */
  var MUT_REFERENCE = _emberUtils.symbol('MUT');
  var SOURCE = _emberUtils.symbol('SOURCE');

  function isMut(ref) {
    return ref && ref[MUT_REFERENCE];
  }

  function unMut(ref) {
    return ref[SOURCE] || ref;
  }

  exports.default = function (vm, args) {
    var rawRef = args.positional.at(0);

    if (isMut(rawRef)) {
      return rawRef;
    }

    // TODO: Improve this error message. This covers at least two distinct
    // cases:
    //
    // 1. (mut "not a path") – passing a literal, result from a helper
    //    invocation, etc
    //
    // 2. (mut receivedValue) – passing a value received from the caller
    //    that was originally derived from a literal, result from a helper
    //    invocation, etc
    //
    // This message is alright for the first case, but could be quite
    // confusing for the second case.

    var wrappedRef = Object.create(rawRef);

    wrappedRef[SOURCE] = rawRef;
    wrappedRef[_emberGlimmerHelpersAction.INVOKE] = rawRef[_emberGlimmerUtilsReferences.UPDATE];
    wrappedRef[MUT_REFERENCE] = true;

    return wrappedRef;
  };
});
enifed('ember-glimmer/helpers/query-param', ['exports', 'ember-utils', 'ember-glimmer/utils/references', 'ember-metal', 'ember-routing'], function (exports, _emberUtils, _emberGlimmerUtilsReferences, _emberMetal, _emberRouting) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  /**
    This is a helper to be used in conjunction with the link-to helper.
    It will supply url query parameters to the target route.
  
    Example
  
    ```handlebars
    {{#link-to 'posts' (query-params direction="asc")}}Sort{{/link-to}}
    ```
  
    @method query-params
    @for Ember.Templates.helpers
    @param {Object} hash takes a hash of query parameters
    @return {Object} A `QueryParams` object for `{{link-to}}`
    @public
  */
  function queryParams(_ref) {
    var positional = _ref.positional;
    var named = _ref.named;

    return _emberRouting.QueryParams.create({
      values: _emberUtils.assign({}, named.value())
    });
  }

  exports.default = function (vm, args) {
    return new _emberGlimmerUtilsReferences.InternalHelperReference(queryParams, args);
  };
});
enifed('ember-glimmer/helpers/readonly', ['exports', 'ember-glimmer/utils/references', 'ember-glimmer/helpers/mut'], function (exports, _emberGlimmerUtilsReferences, _emberGlimmerHelpersMut) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  /**
    The `readonly` helper let's you specify that a binding is one-way only,
    instead of two-way.
    When you pass a `readonly` binding from an outer context (e.g. parent component),
    to to an inner context (e.g. child component), you are saying that changing that
    property in the inner context does not change the value in the outer context.
  
    To specify that a binding is read-only, when invoking the child `Component`:
  
    ```app/components/my-parent.js
    export default Component.extend({
      totalClicks: 3
    });
    ```
  
    ```app/templates/components/my-parent.hbs
    {{log totalClicks}} // -> 3
    {{my-child childClickCount=(readonly totalClicks)}}
    ```
  
    Now, when you update `childClickCount`:
  
    ```app/components/my-child.js
    export default Component.extend({
      click() {
        this.incrementProperty('childClickCount');
      }
    });
    ```
  
    The value updates in the child component, but not the parent component:
  
    ```app/templates/components/my-child.hbs
    {{log childClickCount}} //-> 4
    ```
  
    ```app/templates/components/my-parent.hbs
    {{log totalClicks}} //-> 3
    {{my-child childClickCount=(readonly totalClicks)}}
    ```
  
    ### Objects and Arrays
  
    When passing a property that is a complex object (e.g. object, array) instead of a primitive object (e.g. number, string),
    only the reference to the object is protected using the readonly helper.
    This means that you can change properties of the object both on the parent component, as well as the child component.
    The `readonly` binding behaves similar to the `const` keyword in JavaScript.
  
    Let's look at an example:
  
    First let's set up the parent component:
  
    ```app/components/my-parent.js
    export default Ember.Component.extend({
      clicks: null,
  
      init() {
        this._super(...arguments);
        this.set('clicks', { total: 3 });
      }
    });
    ```
  
    ```app/templates/components/my-parent.hbs
    {{log clicks.total}} //-> 3
    {{my-child childClicks=(readonly clicks)}}
    ```
  
    Now, if you update the `total` property of `childClicks`:
  
    ```app/components/my-child.js
    export default Ember.Component.extend({
      click() {
        this.get('clicks').incrementProperty('total');
      }
    });
    ```
  
    You will see the following happen:
  
    ```app/templates/components/my-parent.hbs
    {{log clicks.total}} //-> 4
    {{my-child childClicks=(readonly clicks)}}
    ```
  
    ```app/templates/components/my-child.hbs
    {{log childClicks.total}} //-> 4
    ```
  
    @method readonly
    @param {Object} [attr] the read-only attribute.
    @for Ember.Templates.helpers
    @private
  */

  exports.default = function (vm, args) {
    var ref = _emberGlimmerHelpersMut.unMut(args.positional.at(0));

    var wrapped = Object.create(ref);

    wrapped[_emberGlimmerUtilsReferences.UPDATE] = undefined;

    return wrapped;
  };
});
enifed('ember-glimmer/helpers/unbound', ['exports', 'ember-metal', 'ember-glimmer/utils/references'], function (exports, _emberMetal, _emberGlimmerUtilsReferences) {
  /**
  @module ember
  @submodule ember-glimmer
  */

  'use strict';

  /**
    The `{{unbound}}` helper disconnects the one-way binding of a property,
    essentially freezing its value at the moment of rendering. For example,
    in this example the display of the variable `name` will not change even
    if it is set with a new value:
  
    ```handlebars
    {{unbound name}}
    ```
  
    Like any helper, the `unbound` helper can accept a nested helper expression.
    This allows for custom helpers to be rendered unbound:
  
    ```handlebars
    {{unbound (some-custom-helper)}}
    {{unbound (capitalize name)}}
    {{! You can use any helper, including unbound, in a nested expression }}
    {{capitalize (unbound name)}}
    ```
  
    The `unbound` helper only accepts a single argument, and it return an
    unbound value.
  
    @method unbound
    @for Ember.Templates.helpers
    @public
  */

  exports.default = function (vm, args) {

    return _emberGlimmerUtilsReferences.UnboundReference.create(args.positional.at(0).value());
  };
});
enifed('ember-glimmer/index', ['exports', 'ember-glimmer/helpers/action', 'ember-glimmer/templates/root', 'ember-glimmer/syntax', 'ember-glimmer/template', 'ember-glimmer/components/checkbox', 'ember-glimmer/components/text_field', 'ember-glimmer/components/text_area', 'ember-glimmer/components/link-to', 'ember-glimmer/component', 'ember-glimmer/helper', 'ember-glimmer/environment', 'ember-glimmer/make-bound-helper', 'ember-glimmer/utils/string', 'ember-glimmer/renderer', 'ember-glimmer/template_registry', 'ember-glimmer/setup-registry', 'ember-glimmer/dom'], function (exports, _emberGlimmerHelpersAction, _emberGlimmerTemplatesRoot, _emberGlimmerSyntax, _emberGlimmerTemplate, _emberGlimmerComponentsCheckbox, _emberGlimmerComponentsText_field, _emberGlimmerComponentsText_area, _emberGlimmerComponentsLinkTo, _emberGlimmerComponent, _emberGlimmerHelper, _emberGlimmerEnvironment, _emberGlimmerMakeBoundHelper, _emberGlimmerUtilsString, _emberGlimmerRenderer, _emberGlimmerTemplate_registry, _emberGlimmerSetupRegistry, _emberGlimmerDom) {
  /**
    [Glimmer](https://github.com/tildeio/glimmer) is a templating engine used by Ember.js that is compatible with a subset of the [Handlebars](http://handlebarsjs.com/) syntax.
  
    ### Showing a property
  
    Templates manage the flow of an application's UI, and display state (through
    the DOM) to a user. For example, given a component with the property "name",
    that component's template can use the name in several ways:
  
    ```app/components/person.js
      export default Ember.Component.extend({
        name: 'Jill'
      });
    ```
  
    ```app/components/person.hbs
    {{name}}
    <div>{{name}}</div>
    <span data-name={{name}}></span>
    ```
  
    Any time the "name" property on the component changes, the DOM will be
    updated.
  
    Properties can be chained as well:
  
    ```handlebars
    {{aUserModel.name}}
    <div>{{listOfUsers.firstObject.name}}</div>
    ```
  
    ### Using Ember helpers
  
    When content is passed in mustaches `{{}}`, Ember will first try to find a helper
    or component with that name. For example, the `if` helper:
  
    ```handlebars
    {{if name "I have a name" "I have no name"}}
    <span data-has-name={{if name true}}></span>
    ```
  
    The returned value is placed where the `{{}}` is called. The above style is
    called "inline". A second style of helper usage is called "block". For example:
  
    ```handlebars
    {{#if name}}
    I have a name
    {{else}}
    I have no name
    {{/if}}
    ```
  
    The block form of helpers allows you to control how the UI is created based
    on the values of properties.
    A third form of helper is called "nested". For example here the concat
    helper will add " Doe" to a displayed name if the person has no last name:
  
    ```handlebars
    <span data-name={{concat firstName (
    if lastName (concat " " lastName) "Doe"
    )}}></span>
    ```
  
    Ember's built-in helpers are described under the [Ember.Templates.helpers](/api/classes/Ember.Templates.helpers.html)
    namespace. Documentation on creating custom helpers can be found under
    [Ember.Helper](/api/classes/Ember.Helper.html).
  
    ### Invoking a Component
  
    Ember components represent state to the UI of an application. Further
    reading on components can be found under [Ember.Component](/api/classes/Ember.Component.html).
  
    @module ember
    @submodule ember-glimmer
    @main ember-glimmer
    @public
   */

  /**
    Use the `{{with}}` helper when you want to alias a property to a new name. This is helpful
    for semantic clarity as it allows you to retain default scope or to reference a property from another
    `{{with}}` block.
  
    If the aliased property is "falsey", for example: `false`, `undefined` `null`, `""`, `0`, NaN or
    an empty array, the block will not be rendered.
  
    ```handlebars
    {{! Will only render if user.posts contains items}}
    {{#with user.posts as |blogPosts|}}
      <div class="notice">
        There are {{blogPosts.length}} blog posts written by {{user.name}}.
      </div>
      {{#each blogPosts as |post|}}
        <li>{{post.title}}</li>
      {{/each}}
    {{/with}}
    ```
  
    Without the `as` operator, it would be impossible to reference `user.name` in the example above.
  
    NOTE: The alias should not reuse a name from the bound property path.
  
    For example: `{{#with foo.bar as |foo|}}` is not supported because it attempts to alias using
    the first part of the property path, `foo`. Instead, use `{{#with foo.bar as |baz|}}`.
  
    @method with
    @for Ember.Templates.helpers
    @param {Object} options
    @return {String} HTML string
    @public
   */

  /**
    Execute the `debugger` statement in the current template's context.
  
    ```handlebars
    {{debugger}}
    ```
  
    When using the debugger helper you will have access to a `get` function. This
    function retrieves values available in the context of the template.
    For example, if you're wondering why a value `{{foo}}` isn't rendering as
    expected within a template, you could place a `{{debugger}}` statement and,
    when the `debugger;` breakpoint is hit, you can attempt to retrieve this value:
  
    ```
    > get('foo')
    ```
  
    `get` is also aware of keywords. So in this situation
  
    ```handlebars
    {{#each items as |item|}}
      {{debugger}}
    {{/each}}
    ```
  
    You'll be able to get values from the current item:
  
    ```
    > get('item.name')
    ```
  
    You can also access the context of the view to make sure it is the object that
    you expect:
  
    ```
    > context
    ```
  
    @method debugger
    @for Ember.Templates.helpers
    @public
   */

  /**
    The `partial` helper renders another template without
    changing the template context:
  
    ```handlebars
    {{foo}}
    {{partial "nav"}}
    ```
  
    The above example template will render a template named
    "-nav", which has the same context as the parent template
    it's rendered into, so if the "-nav" template also referenced
    `{{foo}}`, it would print the same thing as the `{{foo}}`
    in the above example.
  
    If a "-nav" template isn't found, the `partial` helper will
    fall back to a template named "nav".
  
    ### Bound template names
  
    The parameter supplied to `partial` can also be a path
    to a property containing a template name, e.g.:
  
    ```handlebars
    {{partial someTemplateName}}
    ```
  
    The above example will look up the value of `someTemplateName`
    on the template context (e.g. a controller) and use that
    value as the name of the template to render. If the resolved
    value is falsy, nothing will be rendered. If `someTemplateName`
    changes, the partial will be re-rendered using the new template
    name.
  
    @method partial
    @for Ember.Templates.helpers
    @param {String} partialName The name of the template to render minus the leading underscore.
    @public
  */

  'use strict';

  exports.INVOKE = _emberGlimmerHelpersAction.INVOKE;
  exports.RootTemplate = _emberGlimmerTemplatesRoot.default;
  exports.registerSyntax = _emberGlimmerSyntax.registerSyntax;
  exports.template = _emberGlimmerTemplate.default;
  exports.Checkbox = _emberGlimmerComponentsCheckbox.default;
  exports.TextField = _emberGlimmerComponentsText_field.default;
  exports.TextArea = _emberGlimmerComponentsText_area.default;
  exports.LinkComponent = _emberGlimmerComponentsLinkTo.default;
  exports.Component = _emberGlimmerComponent.default;
  exports.Helper = _emberGlimmerHelper.default;
  exports.helper = _emberGlimmerHelper.helper;
  exports.Environment = _emberGlimmerEnvironment.default;
  exports.makeBoundHelper = _emberGlimmerMakeBoundHelper.default;
  exports.SafeString = _emberGlimmerUtilsString.SafeString;
  exports.escapeExpression = _emberGlimmerUtilsString.escapeExpression;
  exports.htmlSafe = _emberGlimmerUtilsString.htmlSafe;
  exports.isHTMLSafe = _emberGlimmerUtilsString.isHTMLSafe;
  exports._getSafeString = _emberGlimmerUtilsString.getSafeString;
  exports.Renderer = _emberGlimmerRenderer.Renderer;
  exports.InertRenderer = _emberGlimmerRenderer.InertRenderer;
  exports.InteractiveRenderer = _emberGlimmerRenderer.InteractiveRenderer;
  exports.getTemplate = _emberGlimmerTemplate_registry.getTemplate;
  exports.setTemplate = _emberGlimmerTemplate_registry.setTemplate;
  exports.hasTemplate = _emberGlimmerTemplate_registry.hasTemplate;
  exports.getTemplates = _emberGlimmerTemplate_registry.getTemplates;
  exports.setTemplates = _emberGlimmerTemplate_registry.setTemplates;
  exports.setupEngineRegistry = _emberGlimmerSetupRegistry.setupEngineRegistry;
  exports.setupApplicationRegistry = _emberGlimmerSetupRegistry.setupApplicationRegistry;
  exports.DOMChanges = _emberGlimmerDom.DOMChanges;
  exports.NodeDOMTreeConstruction = _emberGlimmerDom.NodeDOMTreeConstruction;
  exports.DOMTreeConstruction = _emberGlimmerDom.DOMTreeConstruction;
});
enifed('ember-glimmer/make-bound-helper', ['exports', 'ember-metal', 'ember-glimmer/helper'], function (exports, _emberMetal, _emberGlimmerHelper) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  exports.default = makeBoundHelper;

  /**
    Create a bound helper. Accepts a function that receives the ordered and hash parameters
    from the template. If a bound property was provided in the template, it will be resolved to its
    value and any changes to the bound property cause the helper function to be re-run with the updated
    values.
  
    * `params` - An array of resolved ordered parameters.
    * `hash` - An object containing the hash parameters.
  
    For example:
  
    * With an unquoted ordered parameter:
  
      ```javascript
      {{x-capitalize foo}}
      ```
  
      Assuming `foo` was set to `"bar"`, the bound helper would receive `["bar"]` as its first argument, and
      an empty hash as its second.
  
    * With a quoted ordered parameter:
  
      ```javascript
      {{x-capitalize "foo"}}
      ```
  
      The bound helper would receive `["foo"]` as its first argument, and an empty hash as its second.
  
    * With an unquoted hash parameter:
  
      ```javascript
      {{x-repeat "foo" count=repeatCount}}
      ```
  
      Assuming that `repeatCount` resolved to 2, the bound helper would receive `["foo"]` as its first argument,
      and { count: 2 } as its second.
  
    @private
    @method makeBoundHelper
    @for Ember.HTMLBars
    @param {Function} fn
    @since 1.10.0
  */

  function makeBoundHelper(fn) {
    return _emberGlimmerHelper.helper(fn);
  }
});
enifed('ember-glimmer/modifiers/action', ['exports', 'ember-utils', 'ember-metal', 'ember-views', 'ember-glimmer/helpers/action'], function (exports, _emberUtils, _emberMetal, _emberViews, _emberGlimmerHelpersAction) {
  'use strict';

  var MODIFIERS = ['alt', 'shift', 'meta', 'ctrl'];
  var POINTER_EVENT_TYPE_REGEX = /^click|mouse|touch/;

  function isAllowedEvent(event, allowedKeys) {
    if (allowedKeys === null || typeof allowedKeys === 'undefined') {
      if (POINTER_EVENT_TYPE_REGEX.test(event.type)) {
        return _emberViews.isSimpleClick(event);
      } else {
        allowedKeys = '';
      }
    }

    if (allowedKeys.indexOf('any') >= 0) {
      return true;
    }

    for (var i = 0; i < MODIFIERS.length; i++) {
      if (event[MODIFIERS[i] + 'Key'] && allowedKeys.indexOf(MODIFIERS[i]) === -1) {
        return false;
      }
    }

    return true;
  }

  var ActionHelper = {
    // registeredActions is re-exported for compatibility with older plugins
    // that were using this undocumented API.
    registeredActions: _emberViews.ActionManager.registeredActions,

    registerAction: function (actionState) {
      var actionId = actionState.actionId;

      var actions = _emberViews.ActionManager.registeredActions[actionId];

      if (!actions) {
        actions = _emberViews.ActionManager.registeredActions[actionId] = [];
      }

      actions.push(actionState);

      return actionId;
    },

    unregisterAction: function (actionState) {
      var actionId = actionState.actionId;

      var actions = _emberViews.ActionManager.registeredActions[actionId];

      if (!actions) {
        return;
      }

      var index = actions.indexOf(actionState);

      if (index !== -1) {
        actions.splice(index, 1);
      }

      if (actions.length === 0) {
        delete _emberViews.ActionManager.registeredActions[actionId];
      }
    }
  };

  exports.ActionHelper = ActionHelper;

  var ActionState = (function () {
    function ActionState(element, actionId, actionName, actionArgs, namedArgs, positionalArgs, implicitTarget, dom) {
      this.element = element;
      this.actionId = actionId;
      this.actionName = actionName;
      this.actionArgs = actionArgs;
      this.namedArgs = namedArgs;
      this.positional = positionalArgs;
      this.implicitTarget = implicitTarget;
      this.dom = dom;
      this.eventName = this.getEventName();
    }

    // implements ModifierManager<Action>

    ActionState.prototype.getEventName = function getEventName() {
      return this.namedArgs.get('on').value() || 'click';
    };

    ActionState.prototype.getActionArgs = function getActionArgs() {
      var result = new Array(this.actionArgs.length);

      for (var i = 0; i < this.actionArgs.length; i++) {
        result[i] = this.actionArgs[i].value();
      }

      return result;
    };

    ActionState.prototype.getTarget = function getTarget() {
      var implicitTarget = this.implicitTarget;
      var namedArgs = this.namedArgs;

      var target = undefined;

      if (namedArgs.has('target')) {
        target = namedArgs.get('target').value();
      } else {
        target = implicitTarget.value();
      }

      return target;
    };

    ActionState.prototype.handler = function handler(event) {
      var _this = this;

      var actionName = this.actionName;
      var namedArgs = this.namedArgs;

      var bubbles = namedArgs.get('bubbles');
      var preventDefault = namedArgs.get('preventDefault');
      var allowedKeys = namedArgs.get('allowedKeys');
      var target = this.getTarget();

      if (!isAllowedEvent(event, allowedKeys.value())) {
        return true;
      }

      if (preventDefault.value() !== false) {
        event.preventDefault();
      }

      if (bubbles.value() === false) {
        event.stopPropagation();
      }

      _emberMetal.run(function () {
        var args = _this.getActionArgs();
        var payload = {
          args: args,
          target: target
        };
        if (typeof actionName[_emberGlimmerHelpersAction.INVOKE] === 'function') {
          _emberMetal.flaggedInstrument('interaction.ember-action', payload, function () {
            actionName[_emberGlimmerHelpersAction.INVOKE].apply(actionName, args);
          });
          return;
        }
        if (typeof actionName === 'function') {
          _emberMetal.flaggedInstrument('interaction.ember-action', payload, function () {
            actionName.apply(target, args);
          });
          return;
        }
        payload.name = actionName;
        if (target.send) {
          _emberMetal.flaggedInstrument('interaction.ember-action', payload, function () {
            target.send.apply(target, [actionName].concat(args));
          });
        } else {
          _emberMetal.flaggedInstrument('interaction.ember-action', payload, function () {
            target[actionName].apply(target, args);
          });
        }
      });
    };

    ActionState.prototype.destroy = function destroy() {
      ActionHelper.unregisterAction(this);
    };

    return ActionState;
  })();

  exports.ActionState = ActionState;

  var ActionModifierManager = (function () {
    function ActionModifierManager() {}

    ActionModifierManager.prototype.create = function create(element, args, dynamicScope, dom) {
      var named = args.named;
      var positional = args.positional;

      var implicitTarget = undefined;
      var actionName = undefined;
      var actionNameRef = undefined;
      if (positional.length > 1) {
        implicitTarget = positional.at(0);
        actionNameRef = positional.at(1);

        if (actionNameRef[_emberGlimmerHelpersAction.INVOKE]) {
          actionName = actionNameRef;
        } else {
          var actionLabel = actionNameRef._propertyKey;
          actionName = actionNameRef.value();
        }
      }

      var actionArgs = [];
      // The first two arguments are (1) `this` and (2) the action name.
      // Everything else is a param.
      for (var i = 2; i < positional.length; i++) {
        actionArgs.push(positional.at(i));
      }

      var actionId = _emberUtils.uuid();
      return new ActionState(element, actionId, actionName, actionArgs, named, positional, implicitTarget, dom);
    };

    ActionModifierManager.prototype.install = function install(actionState) {
      var dom = actionState.dom;
      var element = actionState.element;
      var actionId = actionState.actionId;

      ActionHelper.registerAction(actionState);

      dom.setAttribute(element, 'data-ember-action', '');
      dom.setAttribute(element, 'data-ember-action-' + actionId, actionId);
    };

    ActionModifierManager.prototype.update = function update(actionState) {
      var positional = actionState.positional;

      var actionNameRef = positional.at(1);

      if (!actionNameRef[_emberGlimmerHelpersAction.INVOKE]) {
        actionState.actionName = actionNameRef.value();
      }
      actionState.eventName = actionState.getEventName();

      // Not sure if this is needed? If we mutate the actionState is that good enough?
      ActionHelper.unregisterAction(actionState);
      ActionHelper.registerAction(actionState);
    };

    ActionModifierManager.prototype.getDestructor = function getDestructor(modifier) {
      return modifier;
    };

    return ActionModifierManager;
  })();

  exports.default = ActionModifierManager;
});
enifed('ember-glimmer/protocol-for-url', ['exports', 'ember-environment'], function (exports, _emberEnvironment) {
  /* globals module, URL */

  'use strict';

  exports.default = installProtocolForURL;

  var nodeURL = undefined;
  var parsingNode = undefined;

  function installProtocolForURL(environment) {
    var protocol = undefined;

    if (_emberEnvironment.environment.hasDOM) {
      protocol = browserProtocolForURL.call(environment, 'foobar:baz');
    }

    // Test to see if our DOM implementation parses
    // and normalizes URLs.
    if (protocol === 'foobar:') {
      // Swap in the method that doesn't do this test now that
      // we know it works.
      environment.protocolForURL = browserProtocolForURL;
    } else if (typeof URL === 'object') {
      // URL globally provided, likely from FastBoot's sandbox
      nodeURL = URL;
      environment.protocolForURL = nodeProtocolForURL;
    } else if (typeof module === 'object' && typeof module.require === 'function') {
      // Otherwise, we need to fall back to our own URL parsing.
      // Global `require` is shadowed by Ember's loader so we have to use the fully
      // qualified `module.require`.
      nodeURL = module.require('url');
      environment.protocolForURL = nodeProtocolForURL;
    } else {
      throw new Error('Could not find valid URL parsing mechanism for URL Sanitization');
    }
  }

  function browserProtocolForURL(url) {
    if (!parsingNode) {
      parsingNode = document.createElement('a');
    }

    parsingNode.href = url;
    return parsingNode.protocol;
  }

  function nodeProtocolForURL(url) {
    var protocol = null;
    if (typeof url === 'string') {
      protocol = nodeURL.parse(url).protocol;
    }
    return protocol === null ? ':' : protocol;
  }
});
enifed('ember-glimmer/renderer', ['exports', 'ember-glimmer/utils/references', 'ember-metal', 'glimmer-reference', 'ember-views', 'ember-glimmer/component', 'ember-glimmer/syntax/curly-component', 'ember-glimmer/syntax/outlet'], function (exports, _emberGlimmerUtilsReferences, _emberMetal, _glimmerReference, _emberViews, _emberGlimmerComponent, _emberGlimmerSyntaxCurlyComponent, _emberGlimmerSyntaxOutlet) {
  'use strict';

  var runInTransaction = undefined;

  if (false || false) {
    runInTransaction = _emberMetal.runInTransaction;
  } else {
    runInTransaction = function (context, methodName) {
      context[methodName]();
      return false;
    };
  }

  var backburner = _emberMetal.run.backburner;

  var DynamicScope = (function () {
    function DynamicScope(view, outletState, rootOutletState, targetObject) {
      this.view = view;
      this.outletState = outletState;
      this.rootOutletState = rootOutletState;
    }

    DynamicScope.prototype.child = function child() {
      return new DynamicScope(this.view, this.outletState, this.rootOutletState);
    };

    DynamicScope.prototype.get = function get(key) {
      return this.outletState;
    };

    DynamicScope.prototype.set = function set(key, value) {
      this.outletState = value;
      return value;
    };

    return DynamicScope;
  })();

  var RootState = (function () {
    function RootState(root, env, template, self, parentElement, dynamicScope) {
      var _this = this;

      this.id = _emberViews.getViewId(root);
      this.env = env;
      this.root = root;
      this.result = undefined;
      this.shouldReflush = false;
      this.destroyed = false;
      this._removing = false;

      var options = this.options = {
        alwaysRevalidate: false
      };

      this.render = function () {
        var result = _this.result = template.render(self, parentElement, dynamicScope);

        // override .render function after initial render
        _this.render = function () {
          result.rerender(options);
        };
      };
    }

    RootState.prototype.isFor = function isFor(possibleRoot) {
      return this.root === possibleRoot;
    };

    RootState.prototype.destroy = function destroy() {
      var result = this.result;
      var env = this.env;

      this.destroyed = true;

      this.env = null;
      this.root = null;
      this.result = null;
      this.render = null;

      if (result) {
        /*
         Handles these scenarios:
          * When roots are removed during standard rendering process, a transaction exists already
           `.begin()` / `.commit()` are not needed.
         * When roots are being destroyed manually (`component.append(); component.destroy() case), no
           transaction exists already.
         * When roots are being destroyed during `Renderer#destroy`, no transaction exists
          */
        var needsTransaction = !env.inTransaction;

        if (needsTransaction) {
          env.begin();
        }

        result.destroy();

        if (needsTransaction) {
          env.commit();
        }
      }
    };

    return RootState;
  })();

  var renderers = [];

  _emberMetal.setHasViews(function () {
    return renderers.length > 0;
  });

  function register(renderer) {
    renderers.push(renderer);
  }

  function deregister(renderer) {
    var index = renderers.indexOf(renderer);

    renderers.splice(index, 1);
  }

  function loopBegin() {
    for (var i = 0; i < renderers.length; i++) {
      renderers[i]._scheduleRevalidate();
    }
  }

  function K() {}

  var loops = 0;
  function loopEnd(current, next) {
    for (var i = 0; i < renderers.length; i++) {
      if (!renderers[i]._isValid()) {
        if (loops > 10) {
          loops = 0;
          // TODO: do something better
          renderers[i].destroy();
          throw new Error('infinite rendering invalidation detected');
        }
        loops++;
        return backburner.join(null, K);
      }
    }
    loops = 0;
  }

  backburner.on('begin', loopBegin);
  backburner.on('end', loopEnd);

  var Renderer = (function () {
    function Renderer(env, rootTemplate) {
      var _viewRegistry = arguments.length <= 2 || arguments[2] === undefined ? _emberViews.fallbackViewRegistry : arguments[2];

      var destinedForDOM = arguments.length <= 3 || arguments[3] === undefined ? false : arguments[3];

      this._env = env;
      this._rootTemplate = rootTemplate;
      this._viewRegistry = _viewRegistry;
      this._destinedForDOM = destinedForDOM;
      this._destroyed = false;
      this._roots = [];
      this._lastRevision = null;
      this._isRenderingRoots = false;
      this._removedRoots = [];
    }

    // renderer HOOKS

    Renderer.prototype.appendOutletView = function appendOutletView(view, target) {
      var definition = new _emberGlimmerSyntaxOutlet.TopLevelOutletComponentDefinition(view);
      var outletStateReference = view.toReference();
      var targetObject = view.outletState.render.controller;

      this._appendDefinition(view, definition, target, outletStateReference, targetObject);
    };

    Renderer.prototype.appendTo = function appendTo(view, target) {
      var rootDef = new _emberGlimmerSyntaxCurlyComponent.RootComponentDefinition(view);

      this._appendDefinition(view, rootDef, target);
    };

    Renderer.prototype._appendDefinition = function _appendDefinition(root, definition, target) {
      var outletStateReference = arguments.length <= 3 || arguments[3] === undefined ? _glimmerReference.UNDEFINED_REFERENCE : arguments[3];
      var targetObject = arguments.length <= 4 || arguments[4] === undefined ? null : arguments[4];

      var self = new _emberGlimmerUtilsReferences.RootReference(definition);
      var dynamicScope = new DynamicScope(null, outletStateReference, outletStateReference, true, targetObject);
      var rootState = new RootState(root, this._env, this._rootTemplate, self, target, dynamicScope);

      this._renderRoot(rootState);
    };

    Renderer.prototype.rerender = function rerender(view) {
      this._scheduleRevalidate();
    };

    Renderer.prototype.register = function register(view) {
      var id = _emberViews.getViewId(view);

      this._viewRegistry[id] = view;
    };

    Renderer.prototype.unregister = function unregister(view) {
      delete this._viewRegistry[_emberViews.getViewId(view)];
    };

    Renderer.prototype.remove = function remove(view) {
      view._transitionTo('destroying');

      this.cleanupRootFor(view);

      _emberViews.setViewElement(view, null);

      if (this._destinedForDOM) {
        view.trigger('didDestroyElement');
      }

      if (!view.isDestroying) {
        view.destroy();
      }
    };

    Renderer.prototype.cleanupRootFor = function cleanupRootFor(view) {
      // no need to cleanup roots if we have already been destroyed
      if (this._destroyed) {
        return;
      }

      var roots = this._roots;

      // traverse in reverse so we can remove items
      // without mucking up the index
      var i = this._roots.length;
      while (i--) {
        var root = roots[i];
        if (root.isFor(view)) {
          root.destroy();
        }
      }
    };

    Renderer.prototype.destroy = function destroy() {
      if (this._destroyed) {
        return;
      }
      this._destroyed = true;
      this._clearAllRoots();
    };

    Renderer.prototype.getElement = function getElement(view) {
      // overriden in the subclasses
    };

    Renderer.prototype.getBounds = function getBounds(view) {
      var bounds = view[_emberGlimmerComponent.BOUNDS];

      var parentElement = bounds.parentElement();
      var firstNode = bounds.firstNode();
      var lastNode = bounds.lastNode();

      return { parentElement: parentElement, firstNode: firstNode, lastNode: lastNode };
    };

    Renderer.prototype.createElement = function createElement(tagName) {
      return this._env.getAppendOperations().createElement(tagName);
    };

    Renderer.prototype._renderRoot = function _renderRoot(root) {
      var roots = this._roots;

      roots.push(root);

      if (roots.length === 1) {
        register(this);
      }

      this._renderRootsTransaction();
    };

    Renderer.prototype._renderRoots = function _renderRoots() {
      var roots = this._roots;
      var env = this._env;
      var removedRoots = this._removedRoots;

      var globalShouldReflush = undefined,
          initialRootsLength = undefined;

      do {
        env.begin();

        // ensure that for the first iteration of the loop
        // each root is processed
        initialRootsLength = roots.length;
        globalShouldReflush = false;

        for (var i = 0; i < roots.length; i++) {
          var root = roots[i];

          if (root.destroyed) {
            // add to the list of roots to be removed
            // they will be removed from `this._roots` later
            removedRoots.push(root);

            // skip over roots that have been marked as destroyed
            continue;
          }

          var shouldReflush = root.shouldReflush;

          // when processing non-initial reflush loops,
          // do not process more roots than needed
          if (i >= initialRootsLength && !shouldReflush) {
            continue;
          }

          root.options.alwaysRevalidate = shouldReflush;
          // track shouldReflush based on this roots render result
          shouldReflush = root.shouldReflush = runInTransaction(root, 'render');

          // globalShouldReflush should be `true` if *any* of
          // the roots need to reflush
          globalShouldReflush = globalShouldReflush || shouldReflush;
        }

        this._lastRevision = _glimmerReference.CURRENT_TAG.value();

        env.commit();
      } while (globalShouldReflush || roots.length > initialRootsLength);

      // remove any roots that were destroyed during this transaction
      while (removedRoots.length) {
        var root = removedRoots.pop();

        var rootIndex = roots.indexOf(root);
        roots.splice(rootIndex, 1);
      }

      if (this._roots.length === 0) {
        deregister(this);
      }
    };

    Renderer.prototype._renderRootsTransaction = function _renderRootsTransaction() {
      if (this._isRenderingRoots) {
        // currently rendering roots, a new root was added and will
        // be processed by the existing _renderRoots invocation
        return;
      }

      // used to prevent calling _renderRoots again (see above)
      // while we are actively rendering roots
      this._isRenderingRoots = true;

      var completedWithoutError = false;
      try {
        this._renderRoots();
        completedWithoutError = true;
      } finally {
        if (!completedWithoutError) {
          this._lastRevision = _glimmerReference.CURRENT_TAG.value();
        }
        this._isRenderingRoots = false;
      }
    };

    Renderer.prototype._clearAllRoots = function _clearAllRoots() {
      var roots = this._roots;
      for (var i = 0; i < roots.length; i++) {
        var root = roots[i];
        root.destroy();
      }

      this._removedRoots.length = 0;
      this._roots = null;

      // if roots were present before destroying
      // deregister this renderer instance
      if (roots.length) {
        deregister(this);
      }
    };

    Renderer.prototype._scheduleRevalidate = function _scheduleRevalidate() {
      backburner.scheduleOnce('render', this, this._revalidate);
    };

    Renderer.prototype._isValid = function _isValid() {
      return this._destroyed || this._roots.length === 0 || _glimmerReference.CURRENT_TAG.validate(this._lastRevision);
    };

    Renderer.prototype._revalidate = function _revalidate() {
      if (this._isValid()) {
        return;
      }
      this._renderRootsTransaction();
    };

    return Renderer;
  })();

  var InertRenderer = (function (_Renderer) {
    babelHelpers.inherits(InertRenderer, _Renderer);

    function InertRenderer() {
      _Renderer.apply(this, arguments);
    }

    InertRenderer.create = function create(_ref) {
      var env = _ref.env;
      var rootTemplate = _ref.rootTemplate;
      var _viewRegistry = _ref._viewRegistry;

      return new this(env, rootTemplate, _viewRegistry, false);
    };

    InertRenderer.prototype.getElement = function getElement(view) {
      throw new Error('Accessing `this.element` is not allowed in non-interactive environments (such as FastBoot).');
    };

    return InertRenderer;
  })(Renderer);

  exports.InertRenderer = InertRenderer;

  var InteractiveRenderer = (function (_Renderer2) {
    babelHelpers.inherits(InteractiveRenderer, _Renderer2);

    function InteractiveRenderer() {
      _Renderer2.apply(this, arguments);
    }

    InteractiveRenderer.create = function create(_ref2) {
      var env = _ref2.env;
      var rootTemplate = _ref2.rootTemplate;
      var _viewRegistry = _ref2._viewRegistry;

      return new this(env, rootTemplate, _viewRegistry, true);
    };

    InteractiveRenderer.prototype.getElement = function getElement(view) {
      return _emberViews.getViewElement(view);
    };

    return InteractiveRenderer;
  })(Renderer);

  exports.InteractiveRenderer = InteractiveRenderer;
});
enifed('ember-glimmer/setup-registry', ['exports', 'ember-environment', 'container', 'ember-glimmer/renderer', 'ember-glimmer/dom', 'ember-glimmer/views/outlet', 'ember-glimmer/components/text_field', 'ember-glimmer/components/text_area', 'ember-glimmer/components/checkbox', 'ember-glimmer/components/link-to', 'ember-glimmer/component', 'ember-glimmer/templates/component', 'ember-glimmer/templates/root', 'ember-glimmer/templates/outlet', 'ember-glimmer/environment'], function (exports, _emberEnvironment, _container, _emberGlimmerRenderer, _emberGlimmerDom, _emberGlimmerViewsOutlet, _emberGlimmerComponentsText_field, _emberGlimmerComponentsText_area, _emberGlimmerComponentsCheckbox, _emberGlimmerComponentsLinkTo, _emberGlimmerComponent, _emberGlimmerTemplatesComponent, _emberGlimmerTemplatesRoot, _emberGlimmerTemplatesOutlet, _emberGlimmerEnvironment) {
  'use strict';

  exports.setupApplicationRegistry = setupApplicationRegistry;
  exports.setupEngineRegistry = setupEngineRegistry;

  var _templateObject = babelHelpers.taggedTemplateLiteralLoose(['template:-root'], ['template:-root']),
      _templateObject2 = babelHelpers.taggedTemplateLiteralLoose(['template:components/-default'], ['template:components/-default']),
      _templateObject3 = babelHelpers.taggedTemplateLiteralLoose(['component:-default'], ['component:-default']);

  function setupApplicationRegistry(registry) {
    registry.injection('service:-glimmer-environment', 'appendOperations', 'service:-dom-tree-construction');
    registry.injection('renderer', 'env', 'service:-glimmer-environment');

    registry.register(_container.privatize(_templateObject), _emberGlimmerTemplatesRoot.default);
    registry.injection('renderer', 'rootTemplate', _container.privatize(_templateObject));

    registry.register('renderer:-dom', _emberGlimmerRenderer.InteractiveRenderer);
    registry.register('renderer:-inert', _emberGlimmerRenderer.InertRenderer);

    if (_emberEnvironment.environment.hasDOM) {
      registry.injection('service:-glimmer-environment', 'updateOperations', 'service:-dom-changes');
    }

    registry.register('service:-dom-changes', {
      create: function (_ref) {
        var document = _ref.document;
        return new _emberGlimmerDom.DOMChanges(document);
      }
    });

    registry.register('service:-dom-tree-construction', {
      create: function (_ref2) {
        var document = _ref2.document;

        var Implementation = _emberEnvironment.environment.hasDOM ? _emberGlimmerDom.DOMTreeConstruction : _emberGlimmerDom.NodeDOMTreeConstruction;
        return new Implementation(document);
      }
    });
  }

  function setupEngineRegistry(registry) {
    registry.register('view:-outlet', _emberGlimmerViewsOutlet.default);
    registry.register('template:-outlet', _emberGlimmerTemplatesOutlet.default);
    registry.injection('view:-outlet', 'template', 'template:-outlet');

    registry.injection('service:-dom-changes', 'document', 'service:-document');
    registry.injection('service:-dom-tree-construction', 'document', 'service:-document');

    registry.register(_container.privatize(_templateObject2), _emberGlimmerTemplatesComponent.default);

    registry.register('service:-glimmer-environment', _emberGlimmerEnvironment.default);
    registry.injection('template', 'env', 'service:-glimmer-environment');

    registry.optionsForType('helper', { instantiate: false });

    registry.register('component:-text-field', _emberGlimmerComponentsText_field.default);
    registry.register('component:-text-area', _emberGlimmerComponentsText_area.default);
    registry.register('component:-checkbox', _emberGlimmerComponentsCheckbox.default);
    registry.register('component:link-to', _emberGlimmerComponentsLinkTo.default);
    registry.register(_container.privatize(_templateObject3), _emberGlimmerComponent.default);
  }
});
enifed('ember-glimmer/syntax', ['exports', 'ember-glimmer/syntax/render', 'ember-glimmer/syntax/outlet', 'ember-glimmer/syntax/mount', 'ember-glimmer/syntax/dynamic-component', 'ember-glimmer/syntax/input', 'glimmer-runtime'], function (exports, _emberGlimmerSyntaxRender, _emberGlimmerSyntaxOutlet, _emberGlimmerSyntaxMount, _emberGlimmerSyntaxDynamicComponent, _emberGlimmerSyntaxInput, _glimmerRuntime) {
  'use strict';

  exports.registerSyntax = registerSyntax;
  exports.findSyntaxBuilder = findSyntaxBuilder;

  var syntaxKeys = [];
  var syntaxes = [];

  function registerSyntax(key, syntax) {
    syntaxKeys.push(key);
    syntaxes.push(syntax);
  }

  function findSyntaxBuilder(key) {
    var index = syntaxKeys.indexOf(key);

    if (index > -1) {
      return syntaxes[index];
    }
  }

  registerSyntax('render', _emberGlimmerSyntaxRender.RenderSyntax);
  registerSyntax('outlet', _emberGlimmerSyntaxOutlet.OutletSyntax);
  registerSyntax('mount', _emberGlimmerSyntaxMount.MountSyntax);
  registerSyntax('component', _emberGlimmerSyntaxDynamicComponent.DynamicComponentSyntax);
  registerSyntax('input', _emberGlimmerSyntaxInput.InputSyntax);

  registerSyntax('-with-dynamic-vars', (function () {
    function _class() {}

    _class.create = function create(environment, args, symbolTable) {
      return new _glimmerRuntime.WithDynamicVarsSyntax(args);
    };

    return _class;
  })());

  registerSyntax('-in-element', (function () {
    function _class2() {}

    _class2.create = function create(environment, args, symbolTable) {
      return new _glimmerRuntime.InElementSyntax(args);
    };

    return _class2;
  })());
});
enifed('ember-glimmer/syntax/abstract-manager', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  var AbstractManager = function AbstractManager() {};

  exports.default = AbstractManager;
});
enifed('ember-glimmer/syntax/curly-component', ['exports', 'ember-utils', 'glimmer-runtime', 'ember-glimmer/utils/bindings', 'ember-glimmer/component', 'ember-metal', 'ember-views', 'ember-glimmer/utils/process-args', 'container', 'ember-glimmer/syntax/abstract-manager'], function (exports, _emberUtils, _glimmerRuntime, _emberGlimmerUtilsBindings, _emberGlimmerComponent, _emberMetal, _emberViews, _emberGlimmerUtilsProcessArgs, _container, _emberGlimmerSyntaxAbstractManager) {
  'use strict';

  exports.validatePositionalParameters = validatePositionalParameters;

  var _templateObject = babelHelpers.taggedTemplateLiteralLoose(['template:components/-default'], ['template:components/-default']);

  var DEFAULT_LAYOUT = _container.privatize(_templateObject);

  function processComponentInitializationAssertions(component, props) {}

  function validatePositionalParameters(named, positional, positionalParamsDefinition) {}

  function aliasIdToElementId(args, props) {
    if (args.named.has('id')) {
      props.elementId = props.id;
    }
  }

  // We must traverse the attributeBindings in reverse keeping track of
  // what has already been applied. This is essentially refining the concated
  // properties applying right to left.
  function applyAttributeBindings(element, attributeBindings, component, operations) {
    var seen = [];
    var i = attributeBindings.length - 1;

    while (i !== -1) {
      var binding = attributeBindings[i];
      var parsed = _emberGlimmerUtilsBindings.AttributeBinding.parse(binding);
      var attribute = parsed[1];

      if (seen.indexOf(attribute) === -1) {
        seen.push(attribute);
        _emberGlimmerUtilsBindings.AttributeBinding.install(element, component, parsed, operations);
      }

      i--;
    }

    if (seen.indexOf('id') === -1) {
      operations.addStaticAttribute(element, 'id', component.elementId);
    }

    if (seen.indexOf('style') === -1) {
      _emberGlimmerUtilsBindings.IsVisibleBinding.install(element, component, operations);
    }
  }

  var CurlyComponentSyntax = (function (_StatementSyntax) {
babelHelpers.inherits(CurlyComponentSyntax, _StatementSyntax);

    function CurlyComponentSyntax(args, definition, symbolTable) {
      _StatementSyntax.call(this);
      this.args = args;
      this.definition = definition;
      this.symbolTable = symbolTable;
      this.shadow = null;
    }

    CurlyComponentSyntax.prototype.compile = function compile(builder) {
      builder.component.static(this.definition, this.args, this.symbolTable, this.shadow);
    };

    return CurlyComponentSyntax;
  })(_glimmerRuntime.StatementSyntax);

  exports.CurlyComponentSyntax = CurlyComponentSyntax;

  function NOOP() {}

  var ComponentStateBucket = (function () {
    function ComponentStateBucket(environment, component, args, finalizer) {
      this.environment = environment;
      this.component = component;
      this.classRef = null;
      this.args = args;
      this.argsRevision = args.tag.value();
      this.finalizer = finalizer;
    }

    ComponentStateBucket.prototype.destroy = function destroy() {
      var component = this.component;
      var environment = this.environment;

      if (environment.isInteractive) {
        component.trigger('willDestroyElement');
        component.trigger('willClearRender');
      }

      environment.destroyedComponents.push(component);
    };

    ComponentStateBucket.prototype.finalize = function finalize() {
      var finalizer = this.finalizer;

      finalizer();
      this.finalizer = NOOP;
    };

    return ComponentStateBucket;
  })();

  function initialRenderInstrumentDetails(component) {
    return component.instrumentDetails({ initialRender: true });
  }

  function rerenderInstrumentDetails(component) {
    return component.instrumentDetails({ initialRender: false });
  }

  var CurlyComponentManager = (function (_AbstractManager) {
babelHelpers.inherits(CurlyComponentManager, _AbstractManager);

    function CurlyComponentManager() {
      _AbstractManager.apply(this, arguments);
    }

    CurlyComponentManager.prototype.prepareArgs = function prepareArgs(definition, args) {
      validatePositionalParameters(args.named, args.positional.values, definition.ComponentClass.positionalParams);

      return _emberGlimmerUtilsProcessArgs.gatherArgs(args, definition);
    };

    CurlyComponentManager.prototype.create = function create(environment, definition, args, dynamicScope, callerSelfRef, hasBlock) {

      var parentView = dynamicScope.view;

      var klass = definition.ComponentClass;
      var processedArgs = _emberGlimmerUtilsProcessArgs.ComponentArgs.create(args);

      var _processedArgs$value = processedArgs.value();

      var props = _processedArgs$value.props;

      aliasIdToElementId(args, props);

      props.parentView = parentView;
      props[_emberGlimmerComponent.HAS_BLOCK] = hasBlock;

      props._targetObject = callerSelfRef.value();

      var component = klass.create(props);

      var finalizer = _emberMetal._instrumentStart('render.component', initialRenderInstrumentDetails, component);

      dynamicScope.view = component;

      if (parentView !== null) {
        parentView.appendChild(component);
      }

      // We usually do this in the `didCreateElement`, but that hook doesn't fire for tagless components
      if (component.tagName === '') {
        if (environment.isInteractive) {
          component.trigger('willRender');
        }

        component._transitionTo('hasElement');

        if (environment.isInteractive) {
          component.trigger('willInsertElement');
        }
      }

      var bucket = new ComponentStateBucket(environment, component, processedArgs, finalizer);

      if (args.named.has('class')) {
        bucket.classRef = args.named.get('class');
      }

      processComponentInitializationAssertions(component, props);

      if (environment.isInteractive && component.tagName !== '') {
        component.trigger('willRender');
      }

      return bucket;
    };

    CurlyComponentManager.prototype.layoutFor = function layoutFor(definition, bucket, env) {
      var template = definition.template;
      if (!template) {
        var component = bucket.component;

        template = this.templateFor(component, env);
      }
      return env.getCompiledBlock(CurlyComponentLayoutCompiler, template);
    };

    CurlyComponentManager.prototype.templateFor = function templateFor(component, env) {
      var Template = _emberMetal.get(component, 'layout');
      var owner = component[_emberUtils.OWNER];
      if (Template) {
        return env.getTemplate(Template, owner);
      }
      var layoutName = _emberMetal.get(component, 'layoutName');
      if (layoutName) {
        var template = owner.lookup('template:' + layoutName);
        if (template) {
          return template;
        }
      }
      return owner.lookup(DEFAULT_LAYOUT);
    };

    CurlyComponentManager.prototype.getSelf = function getSelf(_ref) {
      var component = _ref.component;

      return component[_emberGlimmerComponent.ROOT_REF];
    };

    CurlyComponentManager.prototype.didCreateElement = function didCreateElement(_ref2, element, operations) {
      var component = _ref2.component;
      var classRef = _ref2.classRef;
      var environment = _ref2.environment;

      _emberViews.setViewElement(component, element);

      var attributeBindings = component.attributeBindings;
      var classNames = component.classNames;
      var classNameBindings = component.classNameBindings;

      if (attributeBindings && attributeBindings.length) {
        applyAttributeBindings(element, attributeBindings, component, operations);
      } else {
        operations.addStaticAttribute(element, 'id', component.elementId);
        _emberGlimmerUtilsBindings.IsVisibleBinding.install(element, component, operations);
      }

      if (classRef) {
        operations.addDynamicAttribute(element, 'class', classRef);
      }

      if (classNames && classNames.length) {
        classNames.forEach(function (name) {
          operations.addStaticAttribute(element, 'class', name);
        });
      }

      if (classNameBindings && classNameBindings.length) {
        classNameBindings.forEach(function (binding) {
          _emberGlimmerUtilsBindings.ClassNameBinding.install(element, component, binding, operations);
        });
      }

      component._transitionTo('hasElement');

      if (environment.isInteractive) {
        component.trigger('willInsertElement');
      }
    };

    CurlyComponentManager.prototype.didRenderLayout = function didRenderLayout(bucket, bounds) {
      bucket.component[_emberGlimmerComponent.BOUNDS] = bounds;
      bucket.finalize();
    };

    CurlyComponentManager.prototype.getTag = function getTag(_ref3) {
      var component = _ref3.component;

      return component[_emberGlimmerComponent.DIRTY_TAG];
    };

    CurlyComponentManager.prototype.didCreate = function didCreate(_ref4) {
      var component = _ref4.component;
      var environment = _ref4.environment;

      if (environment.isInteractive) {
        component._transitionTo('inDOM');
        component.trigger('didInsertElement');
        component.trigger('didRender');
      }
    };

    CurlyComponentManager.prototype.update = function update(bucket, _, dynamicScope) {
      var component = bucket.component;
      var args = bucket.args;
      var argsRevision = bucket.argsRevision;
      var environment = bucket.environment;

      bucket.finalizer = _emberMetal._instrumentStart('render.component', rerenderInstrumentDetails, component);

      if (!args.tag.validate(argsRevision)) {
        var _args$value = args.value();

        var attrs = _args$value.attrs;
        var props = _args$value.props;

        bucket.argsRevision = args.tag.value();

        var oldAttrs = component.attrs;
        var newAttrs = attrs;

        component[_emberGlimmerComponent.IS_DISPATCHING_ATTRS] = true;
        component.setProperties(props);
        component[_emberGlimmerComponent.IS_DISPATCHING_ATTRS] = false;

        component.trigger('didUpdateAttrs', { oldAttrs: oldAttrs, newAttrs: newAttrs });
        component.trigger('didReceiveAttrs', { oldAttrs: oldAttrs, newAttrs: newAttrs });
      }

      if (environment.isInteractive) {
        component.trigger('willUpdate');
        component.trigger('willRender');
      }
    };

    CurlyComponentManager.prototype.didUpdateLayout = function didUpdateLayout(bucket) {
      bucket.finalize();
    };

    CurlyComponentManager.prototype.didUpdate = function didUpdate(_ref5) {
      var component = _ref5.component;
      var environment = _ref5.environment;

      if (environment.isInteractive) {
        component.trigger('didUpdate');
        component.trigger('didRender');
      }
    };

    CurlyComponentManager.prototype.getDestructor = function getDestructor(stateBucket) {
      return stateBucket;
    };

    return CurlyComponentManager;
  })(_emberGlimmerSyntaxAbstractManager.default);

  var MANAGER = new CurlyComponentManager();

  var TopComponentManager = (function (_CurlyComponentManager) {
babelHelpers.inherits(TopComponentManager, _CurlyComponentManager);

    function TopComponentManager() {
      _CurlyComponentManager.apply(this, arguments);
    }

    TopComponentManager.prototype.create = function create(environment, definition, args, dynamicScope, currentScope, hasBlock) {
      var component = definition.ComponentClass;

      var finalizer = _emberMetal._instrumentStart('render.component', initialRenderInstrumentDetails, component);

      dynamicScope.view = component;

      // We usually do this in the `didCreateElement`, but that hook doesn't fire for tagless components
      if (component.tagName === '') {
        if (environment.isInteractive) {
          component.trigger('willRender');
        }

        component._transitionTo('hasElement');

        if (environment.isInteractive) {
          component.trigger('willInsertElement');
        }
      }

      processComponentInitializationAssertions(component, {});

      return new ComponentStateBucket(environment, component, args, finalizer);
    };

    return TopComponentManager;
  })(CurlyComponentManager);

  var ROOT_MANAGER = new TopComponentManager();

  function tagName(vm) {
    var tagName = vm.dynamicScope().view.tagName;

    return _glimmerRuntime.PrimitiveReference.create(tagName === '' ? null : tagName || 'div');
  }

  function ariaRole(vm) {
    return vm.getSelf().get('ariaRole');
  }

  var CurlyComponentDefinition = (function (_ComponentDefinition) {
babelHelpers.inherits(CurlyComponentDefinition, _ComponentDefinition);

    function CurlyComponentDefinition(name, ComponentClass, template, args) {
      _ComponentDefinition.call(this, name, MANAGER, ComponentClass);
      this.template = template;
      this.args = args;
    }

    return CurlyComponentDefinition;
  })(_glimmerRuntime.ComponentDefinition);

  exports.CurlyComponentDefinition = CurlyComponentDefinition;

  var RootComponentDefinition = (function (_ComponentDefinition2) {
babelHelpers.inherits(RootComponentDefinition, _ComponentDefinition2);

    function RootComponentDefinition(instance) {
      _ComponentDefinition2.call(this, '-root', ROOT_MANAGER, instance);
      this.template = undefined;
      this.args = undefined;
    }

    return RootComponentDefinition;
  })(_glimmerRuntime.ComponentDefinition);

  exports.RootComponentDefinition = RootComponentDefinition;

  var CurlyComponentLayoutCompiler = (function () {
    function CurlyComponentLayoutCompiler(template) {
      this.template = template;
    }

    CurlyComponentLayoutCompiler.prototype.compile = function compile(builder) {
      builder.wrapLayout(this.template.asLayout());
      builder.tag.dynamic(tagName);
      builder.attrs.dynamic('role', ariaRole);
      builder.attrs.static('class', 'ember-view');
    };

    return CurlyComponentLayoutCompiler;
  })();

  CurlyComponentLayoutCompiler.id = 'curly';
});
enifed('ember-glimmer/syntax/dynamic-component', ['exports', 'glimmer-runtime', 'glimmer-reference', 'ember-metal'], function (exports, _glimmerRuntime, _glimmerReference, _emberMetal) {
  'use strict';

  function dynamicComponentFor(vm, symbolTable) {
    var env = vm.env;
    var args = vm.getArgs();
    var nameRef = args.positional.at(0);

    return new DynamicComponentReference({ nameRef: nameRef, env: env, symbolTable: symbolTable });
  }

  var DynamicComponentSyntax = (function (_StatementSyntax) {
    babelHelpers.inherits(DynamicComponentSyntax, _StatementSyntax);

    // for {{component componentName}}

    DynamicComponentSyntax.create = function create(environment, args, symbolTable) {
      var definitionArgs = _glimmerRuntime.ArgsSyntax.fromPositionalArgs(args.positional.slice(0, 1));
      var invocationArgs = _glimmerRuntime.ArgsSyntax.build(args.positional.slice(1), args.named, args.blocks);

      return new this(definitionArgs, invocationArgs, symbolTable);
    };

    // Transforms {{foo.bar with=args}} or {{#foo.bar with=args}}{{/foo.bar}}
    // into {{component foo.bar with=args}} or
    // {{#component foo.bar with=args}}{{/component}}
    // with all of it's arguments

    DynamicComponentSyntax.fromPath = function fromPath(environment, path, args, symbolTable) {
      var positional = _glimmerRuntime.ArgsSyntax.fromPositionalArgs(_glimmerRuntime.PositionalArgsSyntax.build([_glimmerRuntime.GetSyntax.build(path.join('.'))]));

      return new this(positional, args, symbolTable);
    };

    function DynamicComponentSyntax(definitionArgs, args, symbolTable) {
      _StatementSyntax.call(this);
      this.definition = dynamicComponentFor;
      this.definitionArgs = definitionArgs;
      this.args = args;
      this.symbolTable = symbolTable;
      this.shadow = null;
    }

    DynamicComponentSyntax.prototype.compile = function compile(builder) {
      builder.component.dynamic(this.definitionArgs, this.definition, this.args, this.symbolTable, this.shadow);
    };

    return DynamicComponentSyntax;
  })(_glimmerRuntime.StatementSyntax);

  exports.DynamicComponentSyntax = DynamicComponentSyntax;

  var DynamicComponentReference = (function () {
    function DynamicComponentReference(_ref) {
      var nameRef = _ref.nameRef;
      var env = _ref.env;
      var symbolTable = _ref.symbolTable;
      var args = _ref.args;

      this.tag = nameRef.tag;
      this.nameRef = nameRef;
      this.env = env;
      this.symbolTable = symbolTable;
      this.args = args;
    }

    DynamicComponentReference.prototype.value = function value() {
      var env = this.env;
      var nameRef = this.nameRef;
      var symbolTable = this.symbolTable;

      var nameOrDef = nameRef.value();

      if (typeof nameOrDef === 'string') {
        var definition = env.getComponentDefinition([nameOrDef], symbolTable);

        return definition;
      } else if (_glimmerRuntime.isComponentDefinition(nameOrDef)) {
        return nameOrDef;
      } else {
        return null;
      }
    };

    DynamicComponentReference.prototype.get = function get() {
      return _glimmerReference.UNDEFINED_REFERENCE;
    };

    return DynamicComponentReference;
  })();
});
enifed('ember-glimmer/syntax/input', ['exports', 'ember-metal', 'ember-glimmer/syntax/curly-component', 'ember-glimmer/syntax/dynamic-component', 'ember-glimmer/utils/bindings'], function (exports, _emberMetal, _emberGlimmerSyntaxCurlyComponent, _emberGlimmerSyntaxDynamicComponent, _emberGlimmerUtilsBindings) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  function buildTextFieldSyntax(args, getDefinition, symbolTable) {
    var definition = getDefinition('-text-field');
    _emberGlimmerUtilsBindings.wrapComponentClassAttribute(args);
    return new _emberGlimmerSyntaxCurlyComponent.CurlyComponentSyntax(args, definition, symbolTable);
  }

  /**
    The `{{input}}` helper lets you create an HTML `<input />` component.
    It causes an `Ember.TextField` component to be rendered.  For more info,
    see the [Ember.TextField](/api/classes/Ember.TextField.html) docs and
    the [templates guide](http://emberjs.com/guides/templates/input-helpers/).
  
    ```handlebars
    {{input value="987"}}
    ```
  
    renders as:
  
    ```HTML
    <input type="text" value="987" />
    ```
  
    ### Text field
  
    If no `type` option is specified, a default of type 'text' is used.
    Many of the standard HTML attributes may be passed to this helper.
    <table>
      <tr><td>`readonly`</td><td>`required`</td><td>`autofocus`</td></tr>
      <tr><td>`value`</td><td>`placeholder`</td><td>`disabled`</td></tr>
      <tr><td>`size`</td><td>`tabindex`</td><td>`maxlength`</td></tr>
      <tr><td>`name`</td><td>`min`</td><td>`max`</td></tr>
      <tr><td>`pattern`</td><td>`accept`</td><td>`autocomplete`</td></tr>
      <tr><td>`autosave`</td><td>`formaction`</td><td>`formenctype`</td></tr>
      <tr><td>`formmethod`</td><td>`formnovalidate`</td><td>`formtarget`</td></tr>
      <tr><td>`height`</td><td>`inputmode`</td><td>`multiple`</td></tr>
      <tr><td>`step`</td><td>`width`</td><td>`form`</td></tr>
      <tr><td>`selectionDirection`</td><td>`spellcheck`</td><td>&nbsp;</td></tr>
    </table>
    When set to a quoted string, these values will be directly applied to the HTML
    element. When left unquoted, these values will be bound to a property on the
    template's current rendering context (most typically a controller instance).
    A very common use of this helper is to bind the `value` of an input to an Object's attribute:
  
    ```handlebars
    Search:
    {{input value=searchWord}}
    ```
  
    In this example, the inital value in the `<input />` will be set to the value of `searchWord`.
    If the user changes the text, the value of `searchWord` will also be updated.
  
    ### Actions
  
    The helper can send multiple actions based on user events.
    The action property defines the action which is sent when
    the user presses the return key.
  
    ```handlebars
    {{input action="submit"}}
    ```
  
    The helper allows some user events to send actions.
  
    * `enter`
    * `insert-newline`
    * `escape-press`
    * `focus-in`
    * `focus-out`
    * `key-press`
    * `key-up`
  
    For example, if you desire an action to be sent when the input is blurred,
    you only need to setup the action name to the event name property.
  
    ```handlebars
    {{input focus-out="alertMessage"}}
    ```
    See more about [Text Support Actions](/api/classes/Ember.TextField.html)
  
    ### Extending `Ember.TextField`
  
    Internally, `{{input type="text"}}` creates an instance of `Ember.TextField`, passing
    arguments from the helper to `Ember.TextField`'s `create` method. You can extend the
    capabilities of text inputs in your applications by reopening this class. For example,
    if you are building a Bootstrap project where `data-*` attributes are used, you
    can add one to the `TextField`'s `attributeBindings` property:
  
    ```javascript
    Ember.TextField.reopen({
      attributeBindings: ['data-error']
    });
    ```
  
    Keep in mind when writing `Ember.TextField` subclasses that `Ember.TextField`
    itself extends `Ember.Component`. Expect isolated component semantics, not
    legacy 1.x view semantics (like `controller` being present).
    See more about [Ember components](/api/classes/Ember.Component.html)
  
    ### Checkbox
  
    Checkboxes are special forms of the `{{input}}` helper.  To create a `<checkbox />`:
  
    ```handlebars
    Emberize Everything:
    {{input type="checkbox" name="isEmberized" checked=isEmberized}}
    ```
  
    This will bind checked state of this checkbox to the value of `isEmberized`  -- if either one changes,
    it will be reflected in the other.
  
    The following HTML attributes can be set via the helper:
  
    * `checked`
    * `disabled`
    * `tabindex`
    * `indeterminate`
    * `name`
    * `autofocus`
    * `form`
  
    ### Extending `Ember.Checkbox`
  
    Internally, `{{input type="checkbox"}}` creates an instance of `Ember.Checkbox`, passing
    arguments from the helper to `Ember.Checkbox`'s `create` method. You can extend the
    capablilties of checkbox inputs in your applications by reopening this class. For example,
    if you wanted to add a css class to all checkboxes in your application:
  
    ```javascript
    Ember.Checkbox.reopen({
      classNames: ['my-app-checkbox']
    });
    ```
  
    @method input
    @for Ember.Templates.helpers
    @param {Hash} options
    @public
  */
  var InputSyntax = {
    create: function (environment, args, symbolTable) {
      var getDefinition = function (path) {
        return environment.getComponentDefinition([path], symbolTable);
      };

      if (args.named.has('type')) {
        var typeArg = args.named.at('type');
        if (typeArg.type === 'value') {
          if (typeArg.value === 'checkbox') {

            _emberGlimmerUtilsBindings.wrapComponentClassAttribute(args);
            var definition = getDefinition('-checkbox');
            return new _emberGlimmerSyntaxCurlyComponent.CurlyComponentSyntax(args, definition, symbolTable);
          } else {
            return buildTextFieldSyntax(args, getDefinition, symbolTable);
          }
        }
      } else {
        return buildTextFieldSyntax(args, getDefinition, symbolTable);
      }

      return _emberGlimmerSyntaxDynamicComponent.DynamicComponentSyntax.create(environment, args, symbolTable);
    }
  };
  exports.InputSyntax = InputSyntax;
});
enifed('ember-glimmer/syntax/mount', ['exports', 'glimmer-runtime', 'glimmer-reference', 'ember-metal', 'ember-glimmer/utils/references', 'ember-routing', 'ember-glimmer/syntax/outlet', 'ember-glimmer/syntax/abstract-manager'], function (exports, _glimmerRuntime, _glimmerReference, _emberMetal, _emberGlimmerUtilsReferences, _emberRouting, _emberGlimmerSyntaxOutlet, _emberGlimmerSyntaxAbstractManager) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  /**
    The `{{mount}}` helper lets you embed a routeless engine in a template.
    Mounting an engine will cause an instance to be booted and its `application`
    template to be rendered.
  
    For example, the following template mounts the `ember-chat` engine:
  
    ```handlebars
    {{! application.hbs }}
    {{mount "ember-chat"}}
    ```
  
    Currently, the engine name is the only argument that can be passed to
    `{{mount}}`.
  
    @method mount
    @for Ember.Templates.helpers
    @category ember-application-engines
    @public
  */

  var MountSyntax = (function (_StatementSyntax) {
    babelHelpers.inherits(MountSyntax, _StatementSyntax);

    MountSyntax.create = function create(env, args, symbolTable) {

      var name = args.positional.at(0).inner();

      var definition = new MountDefinition(name, env);

      return new MountSyntax(definition, symbolTable);
    };

    function MountSyntax(definition, symbolTable) {
      _StatementSyntax.call(this);
      this.definition = definition;
      this.symbolTable = symbolTable;
    }

    MountSyntax.prototype.compile = function compile(builder) {
      builder.component.static(this.definition, _glimmerRuntime.ArgsSyntax.empty(), null, this.symbolTable, null);
    };

    return MountSyntax;
  })(_glimmerRuntime.StatementSyntax);

  exports.MountSyntax = MountSyntax;

  var MountManager = (function (_AbstractManager) {
    babelHelpers.inherits(MountManager, _AbstractManager);

    function MountManager() {
      _AbstractManager.apply(this, arguments);
    }

    MountManager.prototype.prepareArgs = function prepareArgs(definition, args) {
      return args;
    };

    MountManager.prototype.create = function create(environment, _ref, args, dynamicScope) {
      var name = _ref.name;
      var env = _ref.env;

      dynamicScope.outletState = _glimmerReference.UNDEFINED_REFERENCE;

      var engine = env.owner.buildChildEngineInstance(name);

      engine.boot();

      return { engine: engine };
    };

    MountManager.prototype.layoutFor = function layoutFor(definition, _ref2, env) {
      var engine = _ref2.engine;

      var template = engine.lookup('template:application');
      return env.getCompiledBlock(_emberGlimmerSyntaxOutlet.OutletLayoutCompiler, template);
    };

    MountManager.prototype.getSelf = function getSelf(_ref3) {
      var engine = _ref3.engine;

      var factory = engine._lookupFactory('controller:application') || _emberRouting.generateControllerFactory(engine, 'application');
      return new _emberGlimmerUtilsReferences.RootReference(factory.create());
    };

    MountManager.prototype.getTag = function getTag() {
      return null;
    };

    MountManager.prototype.getDestructor = function getDestructor(_ref4) {
      var engine = _ref4.engine;

      return engine;
    };

    MountManager.prototype.didCreateElement = function didCreateElement() {};

    MountManager.prototype.didRenderLayout = function didRenderLayout() {};

    MountManager.prototype.didCreate = function didCreate(state) {};

    MountManager.prototype.update = function update(state, args, dynamicScope) {};

    MountManager.prototype.didUpdateLayout = function didUpdateLayout() {};

    MountManager.prototype.didUpdate = function didUpdate(state) {};

    return MountManager;
  })(_emberGlimmerSyntaxAbstractManager.default);

  var MOUNT_MANAGER = new MountManager();

  var MountDefinition = (function (_ComponentDefinition) {
    babelHelpers.inherits(MountDefinition, _ComponentDefinition);

    function MountDefinition(name, env) {
      _ComponentDefinition.call(this, name, MOUNT_MANAGER, null);
      this.env = env;
    }

    return MountDefinition;
  })(_glimmerRuntime.ComponentDefinition);
});
enifed('ember-glimmer/syntax/outlet', ['exports', 'ember-utils', 'glimmer-runtime', 'ember-metal', 'ember-glimmer/utils/references', 'glimmer-reference', 'ember-glimmer/syntax/abstract-manager'], function (exports, _emberUtils, _glimmerRuntime, _emberMetal, _emberGlimmerUtilsReferences, _glimmerReference, _emberGlimmerSyntaxAbstractManager) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  function outletComponentFor(vm) {
    var _vm$dynamicScope = vm.dynamicScope();

    var outletState = _vm$dynamicScope.outletState;

    var args = vm.getArgs();
    var outletNameRef = undefined;
    if (args.positional.length === 0) {
      outletNameRef = new _glimmerReference.ConstReference('main');
    } else {
      outletNameRef = args.positional.at(0);
    }

    return new OutletComponentReference(outletNameRef, outletState);
  }

  /**
    The `{{outlet}}` helper lets you specify where a child route will render in
    your template. An important use of the `{{outlet}}` helper is in your
    application's `application.hbs` file:
  
    ```handlebars
    {{! app/templates/application.hbs }}
    <!-- header content goes here, and will always display -->
    {{my-header}}
    <div class="my-dynamic-content">
      <!-- this content will change based on the current route, which depends on the current URL -->
      {{outlet}}
    </div>
    <!-- footer content goes here, and will always display -->
    {{my-footer}}
    ```
  
    See [templates guide](http://emberjs.com/guides/templates/the-application-template/) for
    additional information on using `{{outlet}}` in `application.hbs`.
    You may also specify a name for the `{{outlet}}`, which is useful when using more than one
    `{{outlet}}` in a template:
  
    ```handlebars
    {{outlet "menu"}}
    {{outlet "sidebar"}}
    {{outlet "main"}}
    ```
  
    Your routes can then render into a specific one of these `outlet`s by specifying the `outlet`
    attribute in your `renderTemplate` function:
  
    ```javascript
    // app/routes/menu.js
    export default Ember.Route.extend({
      renderTemplate() {
        this.render({ outlet: 'menu' });
      }
    });
    ```
  
    See the [routing guide](http://emberjs.com/guides/routing/rendering-a-template/) for more
    information on how your `route` interacts with the `{{outlet}}` helper.
    Note: Your content __will not render__ if there isn't an `{{outlet}}` for it.
  
    @method outlet
    @param {String} [name]
    @for Ember.Templates.helpers
    @public
  */

  var OutletSyntax = (function (_StatementSyntax) {
    babelHelpers.inherits(OutletSyntax, _StatementSyntax);

    OutletSyntax.create = function create(environment, args, symbolTable) {
      var definitionArgs = _glimmerRuntime.ArgsSyntax.fromPositionalArgs(args.positional.slice(0, 1));
      return new this(environment, definitionArgs, symbolTable);
    };

    function OutletSyntax(environment, args, symbolTable) {
      _StatementSyntax.call(this);
      this.definitionArgs = args;
      this.definition = outletComponentFor;
      this.args = _glimmerRuntime.ArgsSyntax.empty();
      this.symbolTable = symbolTable;
      this.shadow = null;
    }

    OutletSyntax.prototype.compile = function compile(builder) {
      builder.component.dynamic(this.definitionArgs, this.definition, this.args, this.symbolTable, this.shadow);
    };

    return OutletSyntax;
  })(_glimmerRuntime.StatementSyntax);

  exports.OutletSyntax = OutletSyntax;

  var OutletComponentReference = (function () {
    function OutletComponentReference(outletNameRef, parentOutletStateRef) {
      this.outletNameRef = outletNameRef;
      this.parentOutletStateRef = parentOutletStateRef;
      this.definition = null;
      this.lastState = null;
      var outletStateTag = this.outletStateTag = new _glimmerReference.UpdatableTag(parentOutletStateRef.tag);
      this.tag = _glimmerReference.combine([outletStateTag.tag, outletNameRef.tag]);
    }

    OutletComponentReference.prototype.value = function value() {
      var outletNameRef = this.outletNameRef;
      var parentOutletStateRef = this.parentOutletStateRef;
      var definition = this.definition;
      var lastState = this.lastState;

      var outletName = outletNameRef.value();
      var outletStateRef = parentOutletStateRef.get('outlets').get(outletName);
      var newState = this.lastState = outletStateRef.value();

      this.outletStateTag.update(outletStateRef.tag);

      definition = revalidate(definition, lastState, newState);

      var hasTemplate = newState && newState.render.template;

      if (definition) {
        return definition;
      } else if (hasTemplate) {
        return this.definition = new OutletComponentDefinition(outletName, newState.render.template);
      } else {
        return this.definition = null;
      }
    };

    return OutletComponentReference;
  })();

  function revalidate(definition, lastState, newState) {
    if (!lastState && !newState) {
      return definition;
    }

    if (!lastState && newState || lastState && !newState) {
      return null;
    }

    if (newState.render.template === lastState.render.template && newState.render.controller === lastState.render.controller) {
      return definition;
    }

    return null;
  }

  function instrumentationPayload(_ref) {
    var _ref$render = _ref.render;
    var name = _ref$render.name;
    var outlet = _ref$render.outlet;

    return { object: name + ':' + outlet };
  }

  function NOOP() {}

  var StateBucket = (function () {
    function StateBucket(outletState) {
      this.outletState = outletState;
      this.instrument();
    }

    StateBucket.prototype.instrument = function instrument() {
      this.finalizer = _emberMetal._instrumentStart('render.outlet', instrumentationPayload, this.outletState);
    };

    StateBucket.prototype.finalize = function finalize() {
      var finalizer = this.finalizer;

      finalizer();
      this.finalizer = NOOP;
    };

    return StateBucket;
  })();

  var OutletComponentManager = (function (_AbstractManager) {
    babelHelpers.inherits(OutletComponentManager, _AbstractManager);

    function OutletComponentManager() {
      _AbstractManager.apply(this, arguments);
    }

    OutletComponentManager.prototype.prepareArgs = function prepareArgs(definition, args) {
      return args;
    };

    OutletComponentManager.prototype.create = function create(environment, definition, args, dynamicScope) {

      var outletStateReference = dynamicScope.outletState = dynamicScope.outletState.get('outlets').get(definition.outletName);
      var outletState = outletStateReference.value();
      return new StateBucket(outletState);
    };

    OutletComponentManager.prototype.layoutFor = function layoutFor(definition, bucket, env) {
      return env.getCompiledBlock(OutletLayoutCompiler, definition.template);
    };

    OutletComponentManager.prototype.getSelf = function getSelf(_ref2) {
      var outletState = _ref2.outletState;

      return new _emberGlimmerUtilsReferences.RootReference(outletState.render.controller);
    };

    OutletComponentManager.prototype.getTag = function getTag() {
      return null;
    };

    OutletComponentManager.prototype.getDestructor = function getDestructor() {
      return null;
    };

    OutletComponentManager.prototype.didRenderLayout = function didRenderLayout(bucket) {
      bucket.finalize();
    };

    OutletComponentManager.prototype.didCreateElement = function didCreateElement() {};

    OutletComponentManager.prototype.didCreate = function didCreate(state) {};

    OutletComponentManager.prototype.update = function update(bucket) {};

    OutletComponentManager.prototype.didUpdateLayout = function didUpdateLayout(bucket) {};

    OutletComponentManager.prototype.didUpdate = function didUpdate(state) {};

    return OutletComponentManager;
  })(_emberGlimmerSyntaxAbstractManager.default);

  var MANAGER = new OutletComponentManager();

  var TopLevelOutletComponentManager = (function (_OutletComponentManager) {
    babelHelpers.inherits(TopLevelOutletComponentManager, _OutletComponentManager);

    function TopLevelOutletComponentManager() {
      _OutletComponentManager.apply(this, arguments);
    }

    TopLevelOutletComponentManager.prototype.create = function create(environment, definition, args, dynamicScope) {

      return new StateBucket(dynamicScope.outletState.value());
    };

    TopLevelOutletComponentManager.prototype.layoutFor = function layoutFor(definition, bucket, env) {
      return env.getCompiledBlock(TopLevelOutletLayoutCompiler, definition.template);
    };

    return TopLevelOutletComponentManager;
  })(OutletComponentManager);

  var TOP_LEVEL_MANAGER = new TopLevelOutletComponentManager();

  var TopLevelOutletComponentDefinition = (function (_ComponentDefinition) {
    babelHelpers.inherits(TopLevelOutletComponentDefinition, _ComponentDefinition);

    function TopLevelOutletComponentDefinition(instance) {
      _ComponentDefinition.call(this, 'outlet', TOP_LEVEL_MANAGER, instance);
      this.template = instance.template;
      _emberUtils.generateGuid(this);
    }

    return TopLevelOutletComponentDefinition;
  })(_glimmerRuntime.ComponentDefinition);

  exports.TopLevelOutletComponentDefinition = TopLevelOutletComponentDefinition;

  var TopLevelOutletLayoutCompiler = (function () {
    function TopLevelOutletLayoutCompiler(template) {
      this.template = template;
    }

    TopLevelOutletLayoutCompiler.prototype.compile = function compile(builder) {
      builder.wrapLayout(this.template.asLayout());
      builder.tag.static('div');
      builder.attrs.static('id', _emberUtils.guidFor(this));
      builder.attrs.static('class', 'ember-view');
    };

    return TopLevelOutletLayoutCompiler;
  })();

  TopLevelOutletLayoutCompiler.id = 'top-level-outlet';

  var OutletComponentDefinition = (function (_ComponentDefinition2) {
    babelHelpers.inherits(OutletComponentDefinition, _ComponentDefinition2);

    function OutletComponentDefinition(outletName, template) {
      _ComponentDefinition2.call(this, 'outlet', MANAGER, null);
      this.outletName = outletName;
      this.template = template;
      _emberUtils.generateGuid(this);
    }

    return OutletComponentDefinition;
  })(_glimmerRuntime.ComponentDefinition);

  var OutletLayoutCompiler = (function () {
    function OutletLayoutCompiler(template) {
      this.template = template;
    }

    OutletLayoutCompiler.prototype.compile = function compile(builder) {
      builder.wrapLayout(this.template.asLayout());
    };

    return OutletLayoutCompiler;
  })();

  exports.OutletLayoutCompiler = OutletLayoutCompiler;

  OutletLayoutCompiler.id = 'outlet';
});
enifed('ember-glimmer/syntax/render', ['exports', 'glimmer-runtime', 'glimmer-reference', 'ember-metal', 'ember-glimmer/utils/references', 'ember-routing', 'ember-glimmer/syntax/outlet', 'ember-glimmer/syntax/abstract-manager'], function (exports, _glimmerRuntime, _glimmerReference, _emberMetal, _emberGlimmerUtilsReferences, _emberRouting, _emberGlimmerSyntaxOutlet, _emberGlimmerSyntaxAbstractManager) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  function makeComponentDefinition(vm) {
    var env = vm.env;
    var args = vm.getArgs();
    var nameRef = args.positional.at(0);

    var templateName = nameRef.value();

    var template = env.owner.lookup('template:' + templateName);

    var controllerName = undefined;

    if (args.named.has('controller')) {
      var controllerNameRef = args.named.get('controller');

      controllerName = controllerNameRef.value();
    } else {
      controllerName = templateName;
    }

    if (args.positional.length === 1) {
      return new _glimmerReference.ConstReference(new RenderDefinition(controllerName, template, env, SINGLETON_RENDER_MANAGER));
    } else {
      return new _glimmerReference.ConstReference(new RenderDefinition(controllerName, template, env, NON_SINGLETON_RENDER_MANAGER));
    }
  }

  /**
    Calling ``{{render}}`` from within a template will insert another
    template that matches the provided name. The inserted template will
    access its properties on its own controller (rather than the controller
    of the parent template).
  
    If a view class with the same name exists, the view class also will be used.
    Note: A given controller may only be used *once* in your app in this manner.
    A singleton instance of the controller will be created for you.
  
    Example:
  
    ```javascript
    App.NavigationController = Ember.Controller.extend({
      who: "world"
    });
    ```
  
    ```handlebars
    <!-- navigation.hbs -->
    Hello, {{who}}.
    ```
  
    ```handlebars
    <!-- application.hbs -->
    <h1>My great app</h1>
    {{render "navigation"}}
    ```
  
    ```html
    <h1>My great app</h1>
    <div class='ember-view'>
      Hello, world.
    </div>
    ```
  
    Optionally you may provide a second argument: a property path
    that will be bound to the `model` property of the controller.
    If a `model` property path is specified, then a new instance of the
    controller will be created and `{{render}}` can be used multiple times
    with the same name.
  
    For example if you had this `author` template.
  
    ```handlebars
    <div class="author">
      Written by {{firstName}} {{lastName}}.
      Total Posts: {{postCount}}
    </div>
    ```
  
    You could render it inside the `post` template using the `render` helper.
  
    ```handlebars
    <div class="post">
      <h1>{{title}}</h1>
      <div>{{body}}</div>
      {{render "author" author}}
    </div>
    ```
  
    @method render
    @for Ember.Templates.helpers
    @param {String} name
    @param {Object?} context
    @param {Hash} options
    @return {String} HTML string
    @public
  */

  var RenderSyntax = (function (_StatementSyntax) {
    babelHelpers.inherits(RenderSyntax, _StatementSyntax);

    RenderSyntax.create = function create(environment, args, symbolTable) {
      return new this(environment, args, symbolTable);
    };

    function RenderSyntax(environment, args, symbolTable) {
      _StatementSyntax.call(this);
      this.definitionArgs = args;
      this.definition = makeComponentDefinition;
      this.args = _glimmerRuntime.ArgsSyntax.fromPositionalArgs(args.positional.slice(1, 2));
      this.symbolTable = symbolTable;
      this.shadow = null;
    }

    RenderSyntax.prototype.compile = function compile(builder) {
      builder.component.dynamic(this.definitionArgs, this.definition, this.args, this.symbolTable, this.shadow);
    };

    return RenderSyntax;
  })(_glimmerRuntime.StatementSyntax);

  exports.RenderSyntax = RenderSyntax;

  var AbstractRenderManager = (function (_AbstractManager) {
    babelHelpers.inherits(AbstractRenderManager, _AbstractManager);

    function AbstractRenderManager() {
      _AbstractManager.apply(this, arguments);
    }

    AbstractRenderManager.prototype.prepareArgs = function prepareArgs(definition, args) {
      return args;
    };

    /* abstract create(environment, definition, args, dynamicScope); */

    AbstractRenderManager.prototype.layoutFor = function layoutFor(definition, bucket, env) {
      return env.getCompiledBlock(_emberGlimmerSyntaxOutlet.OutletLayoutCompiler, definition.template);
    };

    AbstractRenderManager.prototype.getSelf = function getSelf(_ref) {
      var controller = _ref.controller;

      return new _emberGlimmerUtilsReferences.RootReference(controller);
    };

    AbstractRenderManager.prototype.getTag = function getTag() {
      return null;
    };

    AbstractRenderManager.prototype.getDestructor = function getDestructor() {
      return null;
    };

    AbstractRenderManager.prototype.didCreateElement = function didCreateElement() {};

    AbstractRenderManager.prototype.didRenderLayout = function didRenderLayout() {};

    AbstractRenderManager.prototype.didCreate = function didCreate() {};

    AbstractRenderManager.prototype.update = function update() {};

    AbstractRenderManager.prototype.didUpdateLayout = function didUpdateLayout() {};

    AbstractRenderManager.prototype.didUpdate = function didUpdate() {};

    return AbstractRenderManager;
  })(_emberGlimmerSyntaxAbstractManager.default);

  var SingletonRenderManager = (function (_AbstractRenderManager) {
    babelHelpers.inherits(SingletonRenderManager, _AbstractRenderManager);

    function SingletonRenderManager() {
      _AbstractRenderManager.apply(this, arguments);
    }

    SingletonRenderManager.prototype.create = function create(environment, definition, args, dynamicScope) {
      var name = definition.name;
      var env = definition.env;

      var controller = env.owner.lookup('controller:' + name) || _emberRouting.generateController(env.owner, name);

      if (dynamicScope.rootOutletState) {
        dynamicScope.outletState = dynamicScope.rootOutletState.getOrphan(name);
      }

      return { controller: controller };
    };

    return SingletonRenderManager;
  })(AbstractRenderManager);

  var SINGLETON_RENDER_MANAGER = new SingletonRenderManager();

  var NonSingletonRenderManager = (function (_AbstractRenderManager2) {
    babelHelpers.inherits(NonSingletonRenderManager, _AbstractRenderManager2);

    function NonSingletonRenderManager() {
      _AbstractRenderManager2.apply(this, arguments);
    }

    NonSingletonRenderManager.prototype.create = function create(environment, definition, args, dynamicScope) {
      var name = definition.name;
      var env = definition.env;

      var modelRef = args.positional.at(0);

      var factory = env.owner._lookupFactory('controller:' + name) || _emberRouting.generateControllerFactory(env.owner, name);
      var controller = factory.create({ model: modelRef.value() });

      if (dynamicScope.rootOutletState) {
        dynamicScope.outletState = dynamicScope.rootOutletState.getOrphan(name);
      }

      return { controller: controller };
    };

    NonSingletonRenderManager.prototype.update = function update(_ref2, args, dynamicScope) {
      var controller = _ref2.controller;

      controller.set('model', args.positional.at(0).value());
    };

    NonSingletonRenderManager.prototype.getDestructor = function getDestructor(_ref3) {
      var controller = _ref3.controller;

      return controller;
    };

    return NonSingletonRenderManager;
  })(AbstractRenderManager);

  var NON_SINGLETON_RENDER_MANAGER = new NonSingletonRenderManager();

  var RenderDefinition = (function (_ComponentDefinition) {
    babelHelpers.inherits(RenderDefinition, _ComponentDefinition);

    function RenderDefinition(name, template, env, manager) {
      _ComponentDefinition.call(this, 'render', manager, null);

      this.name = name;
      this.template = template;
      this.env = env;
    }

    return RenderDefinition;
  })(_glimmerRuntime.ComponentDefinition);
});
enifed('ember-glimmer/template', ['exports', 'ember-utils', 'glimmer-runtime'], function (exports, _emberUtils, _glimmerRuntime) {
  'use strict';

  exports.default = template;

  function template(json) {
    var factory = _glimmerRuntime.templateFactory(json);

    return {
      id: factory.id,
      meta: factory.meta,
      create: function (props) {
        return factory.create(props.env, { owner: props[_emberUtils.OWNER] });
      }
    };
  }
});
enifed("ember-glimmer/template_registry", ["exports"], function (exports) {
  // STATE within a module is frowned apon, this exists
  // to support Ember.TEMPLATES but shield ember internals from this legacy
  // global API.
  "use strict";

  exports.setTemplates = setTemplates;
  exports.getTemplates = getTemplates;
  exports.getTemplate = getTemplate;
  exports.hasTemplate = hasTemplate;
  exports.setTemplate = setTemplate;
  var TEMPLATES = {};

  function setTemplates(templates) {
    TEMPLATES = templates;
  }

  function getTemplates() {
    return TEMPLATES;
  }

  function getTemplate(name) {
    if (TEMPLATES.hasOwnProperty(name)) {
      return TEMPLATES[name];
    }
  }

  function hasTemplate(name) {
    return TEMPLATES.hasOwnProperty(name);
  }

  function setTemplate(name, template) {
    return TEMPLATES[name] = template;
  }
});
enifed("ember-glimmer/templates/component", ["exports", "ember-glimmer/template"], function (exports, _emberGlimmerTemplate) {
  "use strict";

  exports.default = _emberGlimmerTemplate.default({ "id": "ZoGfVsSJ", "block": "{\"statements\":[[\"yield\",\"default\"]],\"locals\":[],\"named\":[],\"yields\":[\"default\"],\"blocks\":[],\"hasPartials\":false}", "meta": { "moduleName": "ember-glimmer/templates/component.hbs" } });
});
enifed("ember-glimmer/templates/empty", ["exports", "ember-glimmer/template"], function (exports, _emberGlimmerTemplate) {
  "use strict";

  exports.default = _emberGlimmerTemplate.default({ "id": "qEHL4OLi", "block": "{\"statements\":[],\"locals\":[],\"named\":[],\"yields\":[],\"blocks\":[],\"hasPartials\":false}", "meta": { "moduleName": "ember-glimmer/templates/empty.hbs" } });
});
enifed("ember-glimmer/templates/link-to", ["exports", "ember-glimmer/template"], function (exports, _emberGlimmerTemplate) {
  "use strict";

  exports.default = _emberGlimmerTemplate.default({ "id": "Ca7iQMR7", "block": "{\"statements\":[[\"block\",[\"if\"],[[\"get\",[\"linkTitle\"]]],null,1,0]],\"locals\":[],\"named\":[],\"yields\":[\"default\"],\"blocks\":[{\"statements\":[[\"yield\",\"default\"]],\"locals\":[]},{\"statements\":[[\"append\",[\"unknown\",[\"linkTitle\"]],false]],\"locals\":[]}],\"hasPartials\":false}", "meta": { "moduleName": "ember-glimmer/templates/link-to.hbs" } });
});
enifed("ember-glimmer/templates/outlet", ["exports", "ember-glimmer/template"], function (exports, _emberGlimmerTemplate) {
  "use strict";

  exports.default = _emberGlimmerTemplate.default({ "id": "sYQo9vi/", "block": "{\"statements\":[[\"append\",[\"unknown\",[\"outlet\"]],false]],\"locals\":[],\"named\":[],\"yields\":[],\"blocks\":[],\"hasPartials\":false}", "meta": { "moduleName": "ember-glimmer/templates/outlet.hbs" } });
});
enifed("ember-glimmer/templates/root", ["exports", "ember-glimmer/template"], function (exports, _emberGlimmerTemplate) {
  "use strict";

  exports.default = _emberGlimmerTemplate.default({ "id": "Eaf3RPY3", "block": "{\"statements\":[[\"append\",[\"helper\",[\"component\"],[[\"get\",[null]]],null],false]],\"locals\":[],\"named\":[],\"yields\":[],\"blocks\":[],\"hasPartials\":false}", "meta": { "moduleName": "ember-glimmer/templates/root.hbs" } });
});
enifed('ember-glimmer/utils/bindings', ['exports', 'glimmer-reference', 'glimmer-runtime', 'ember-metal', 'ember-runtime', 'ember-glimmer/component', 'ember-glimmer/utils/string'], function (exports, _glimmerReference, _glimmerRuntime, _emberMetal, _emberRuntime, _emberGlimmerComponent, _emberGlimmerUtilsString) {
  'use strict';

  exports.wrapComponentClassAttribute = wrapComponentClassAttribute;

  function referenceForKey(component, key) {
    return component[_emberGlimmerComponent.ROOT_REF].get(key);
  }

  function referenceForParts(component, parts) {
    var isAttrs = parts[0] === 'attrs';

    // TODO deprecate this
    if (isAttrs) {
      parts.shift();

      if (parts.length === 1) {
        return referenceForKey(component, parts[0]);
      }
    }

    return _glimmerReference.referenceFromParts(component[_emberGlimmerComponent.ROOT_REF], parts);
  }

  // TODO we should probably do this transform at build time

  function wrapComponentClassAttribute(args) {
    var named = args.named;

    var index = named.keys.indexOf('class');

    if (index !== -1) {
      var _named$values$index = named.values[index];
      var ref = _named$values$index.ref;
      var type = _named$values$index.type;

      if (type === 'get') {
        var propName = ref.parts[ref.parts.length - 1];
        named.values[index] = _glimmerRuntime.HelperSyntax.fromSpec(['helper', ['-class'], [['get', ref.parts], propName], null]);
      }
    }

    return args;
  }

  var AttributeBinding = {
    parse: function (microsyntax) {
      var colonIndex = microsyntax.indexOf(':');

      if (colonIndex === -1) {
        return [microsyntax, microsyntax, true];
      } else {
        var prop = microsyntax.substring(0, colonIndex);
        var attribute = microsyntax.substring(colonIndex + 1);

        return [prop, attribute, false];
      }
    },

    install: function (element, component, parsed, operations) {
      var prop = parsed[0];
      var attribute = parsed[1];
      var isSimple = parsed[2];

      if (attribute === 'id') {
        var elementId = _emberMetal.get(component, prop);
        if (elementId === undefined || elementId === null) {
          elementId = component.elementId;
        }
        operations.addStaticAttribute(element, 'id', elementId);
        return;
      }

      var isPath = prop.indexOf('.') > -1;
      var reference = isPath ? referenceForParts(component, prop.split('.')) : referenceForKey(component, prop);

      if (attribute === 'style') {
        reference = new StyleBindingReference(reference, referenceForKey(component, 'isVisible'));
      }

      operations.addDynamicAttribute(element, attribute, reference);
    }
  };

  exports.AttributeBinding = AttributeBinding;
  var DISPLAY_NONE = 'display: none;';
  var SAFE_DISPLAY_NONE = _emberGlimmerUtilsString.htmlSafe(DISPLAY_NONE);

  var StyleBindingReference = (function (_CachedReference) {
    babelHelpers.inherits(StyleBindingReference, _CachedReference);

    function StyleBindingReference(inner, isVisible) {
      _CachedReference.call(this);

      this.tag = _glimmerReference.combine([inner.tag, isVisible.tag]);
      this.inner = inner;
      this.isVisible = isVisible;
    }

    StyleBindingReference.prototype.compute = function compute() {
      var value = this.inner.value();
      var isVisible = this.isVisible.value();

      if (isVisible !== false) {
        return value;
      } else if (!value && value !== 0) {
        return SAFE_DISPLAY_NONE;
      } else {
        var style = value + ' ' + DISPLAY_NONE;
        return _emberGlimmerUtilsString.isHTMLSafe(value) ? _emberGlimmerUtilsString.htmlSafe(style) : style;
      }
    };

    return StyleBindingReference;
  })(_glimmerReference.CachedReference);

  var IsVisibleBinding = {
    install: function (element, component, operations) {
      operations.addDynamicAttribute(element, 'style', _glimmerReference.map(referenceForKey(component, 'isVisible'), this.mapStyleValue));
    },

    mapStyleValue: function (isVisible) {
      return isVisible === false ? SAFE_DISPLAY_NONE : null;
    }
  };

  exports.IsVisibleBinding = IsVisibleBinding;
  var ClassNameBinding = {
    install: function (element, component, microsyntax, operations) {
      var _microsyntax$split = microsyntax.split(':');

      var prop = _microsyntax$split[0];
      var truthy = _microsyntax$split[1];
      var falsy = _microsyntax$split[2];

      var isStatic = prop === '';

      if (isStatic) {
        operations.addStaticAttribute(element, 'class', truthy);
      } else {
        var isPath = prop.indexOf('.') > -1;
        var parts = isPath && prop.split('.');
        var value = isPath ? referenceForParts(component, parts) : referenceForKey(component, prop);
        var ref = undefined;

        if (truthy === undefined) {
          ref = new SimpleClassNameBindingReference(value, isPath ? parts[parts.length - 1] : prop);
        } else {
          ref = new ColonClassNameBindingReference(value, truthy, falsy);
        }

        operations.addDynamicAttribute(element, 'class', ref);
      }
    }
  };

  exports.ClassNameBinding = ClassNameBinding;

  var SimpleClassNameBindingReference = (function (_CachedReference2) {
    babelHelpers.inherits(SimpleClassNameBindingReference, _CachedReference2);

    function SimpleClassNameBindingReference(inner, path) {
      _CachedReference2.call(this);

      this.tag = inner.tag;
      this.inner = inner;
      this.path = path;
      this.dasherizedPath = null;
    }

    SimpleClassNameBindingReference.prototype.compute = function compute() {
      var value = this.inner.value();

      if (value === true) {
        var path = this.path;
        var dasherizedPath = this.dasherizedPath;

        return dasherizedPath || (this.dasherizedPath = _emberRuntime.String.dasherize(path));
      } else if (value || value === 0) {
        return value;
      } else {
        return null;
      }
    };

    return SimpleClassNameBindingReference;
  })(_glimmerReference.CachedReference);

  var ColonClassNameBindingReference = (function (_CachedReference3) {
    babelHelpers.inherits(ColonClassNameBindingReference, _CachedReference3);

    function ColonClassNameBindingReference(inner, truthy, falsy) {
      _CachedReference3.call(this);

      this.tag = inner.tag;
      this.inner = inner;
      this.truthy = truthy || null;
      this.falsy = falsy || null;
    }

    ColonClassNameBindingReference.prototype.compute = function compute() {
      var inner = this.inner;
      var truthy = this.truthy;
      var falsy = this.falsy;

      return inner.value() ? truthy : falsy;
    };

    return ColonClassNameBindingReference;
  })(_glimmerReference.CachedReference);
});
enifed('ember-glimmer/utils/debug-stack', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  var DebugStack = undefined;

  exports.default = DebugStack;
});
enifed('ember-glimmer/utils/iterable', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime', 'ember-glimmer/utils/references', 'ember-glimmer/helpers/each-in', 'glimmer-reference'], function (exports, _emberUtils, _emberMetal, _emberRuntime, _emberGlimmerUtilsReferences, _emberGlimmerHelpersEachIn, _glimmerReference) {
  'use strict';

  exports.default = iterableFor;

  var ITERATOR_KEY_GUID = 'be277757-bbbe-4620-9fcb-213ef433cca2';

  function iterableFor(ref, keyPath) {
    if (_emberGlimmerHelpersEachIn.isEachIn(ref)) {
      return new EachInIterable(ref, keyForEachIn(keyPath));
    } else {
      return new ArrayIterable(ref, keyForArray(keyPath));
    }
  }

  function keyForEachIn(keyPath) {
    switch (keyPath) {
      case '@index':
      case undefined:
      case null:
        return index;
      case '@identity':
        return identity;
      default:
        return function (item) {
          return _emberMetal.get(item, keyPath);
        };
    }
  }

  function keyForArray(keyPath) {
    switch (keyPath) {
      case '@index':
        return index;
      case '@identity':
      case undefined:
      case null:
        return identity;
      default:
        return function (item) {
          return _emberMetal.get(item, keyPath);
        };
    }
  }

  function index(item, index) {
    return String(index);
  }

  function identity(item) {
    switch (typeof item) {
      case 'string':
      case 'number':
        return String(item);
      default:
        return _emberUtils.guidFor(item);
    }
  }

  function ensureUniqueKey(seen, key) {
    var seenCount = seen[key];

    if (seenCount) {
      seen[key]++;
      return '' + key + ITERATOR_KEY_GUID + seenCount;
    } else {
      seen[key] = 1;
    }

    return key;
  }

  var ArrayIterator = (function () {
    function ArrayIterator(array, keyFor) {
      this.array = array;
      this.length = array.length;
      this.keyFor = keyFor;
      this.position = 0;
      this.seen = new _emberUtils.EmptyObject();
    }

    ArrayIterator.prototype.isEmpty = function isEmpty() {
      return false;
    };

    ArrayIterator.prototype.next = function next() {
      var array = this.array;
      var length = this.length;
      var keyFor = this.keyFor;
      var position = this.position;
      var seen = this.seen;

      if (position >= length) {
        return null;
      }

      var value = array[position];
      var memo = position;
      var key = ensureUniqueKey(seen, keyFor(value, memo));

      this.position++;

      return { key: key, value: value, memo: memo };
    };

    return ArrayIterator;
  })();

  var EmberArrayIterator = (function () {
    function EmberArrayIterator(array, keyFor) {
      this.array = array;
      this.length = _emberMetal.get(array, 'length');
      this.keyFor = keyFor;
      this.position = 0;
      this.seen = new _emberUtils.EmptyObject();
    }

    EmberArrayIterator.prototype.isEmpty = function isEmpty() {
      return this.length === 0;
    };

    EmberArrayIterator.prototype.next = function next() {
      var array = this.array;
      var length = this.length;
      var keyFor = this.keyFor;
      var position = this.position;
      var seen = this.seen;

      if (position >= length) {
        return null;
      }

      var value = _emberRuntime.objectAt(array, position);
      var memo = position;
      var key = ensureUniqueKey(seen, keyFor(value, memo));

      this.position++;

      return { key: key, value: value, memo: memo };
    };

    return EmberArrayIterator;
  })();

  var ObjectKeysIterator = (function () {
    function ObjectKeysIterator(keys, values, keyFor) {
      this.keys = keys;
      this.values = values;
      this.keyFor = keyFor;
      this.position = 0;
      this.seen = new _emberUtils.EmptyObject();
    }

    ObjectKeysIterator.prototype.isEmpty = function isEmpty() {
      return this.keys.length === 0;
    };

    ObjectKeysIterator.prototype.next = function next() {
      var keys = this.keys;
      var values = this.values;
      var keyFor = this.keyFor;
      var position = this.position;
      var seen = this.seen;

      if (position >= keys.length) {
        return null;
      }

      var value = values[position];
      var memo = keys[position];
      var key = ensureUniqueKey(seen, keyFor(value, memo));

      this.position++;

      return { key: key, value: value, memo: memo };
    };

    return ObjectKeysIterator;
  })();

  var EmptyIterator = (function () {
    function EmptyIterator() {}

    EmptyIterator.prototype.isEmpty = function isEmpty() {
      return true;
    };

    EmptyIterator.prototype.next = function next() {
      throw new Error('Cannot call next() on an empty iterator');
    };

    return EmptyIterator;
  })();

  var EMPTY_ITERATOR = new EmptyIterator();

  var EachInIterable = (function () {
    function EachInIterable(ref, keyFor) {
      this.ref = ref;
      this.keyFor = keyFor;

      var valueTag = this.valueTag = new _glimmerReference.UpdatableTag(_glimmerReference.CONSTANT_TAG);

      this.tag = _glimmerReference.combine([ref.tag, valueTag]);
    }

    EachInIterable.prototype.iterate = function iterate() {
      var ref = this.ref;
      var keyFor = this.keyFor;
      var valueTag = this.valueTag;

      var iterable = ref.value();

      valueTag.update(_emberMetal.tagFor(iterable));

      if (_emberMetal.isProxy(iterable)) {
        iterable = _emberMetal.get(iterable, 'content');
      }

      var typeofIterable = typeof iterable;

      if (iterable && (typeofIterable === 'object' || typeofIterable === 'function')) {
        var keys = Object.keys(iterable);
        var values = keys.map(function (key) {
          return iterable[key];
        });
        return keys.length > 0 ? new ObjectKeysIterator(keys, values, keyFor) : EMPTY_ITERATOR;
      } else {
        return EMPTY_ITERATOR;
      }
    };

    // {{each-in}} yields |key value| instead of |value key|, so the memo and
    // value are flipped

    EachInIterable.prototype.valueReferenceFor = function valueReferenceFor(item) {
      return new _emberGlimmerUtilsReferences.UpdatablePrimitiveReference(item.memo);
    };

    EachInIterable.prototype.updateValueReference = function updateValueReference(reference, item) {
      reference.update(item.memo);
    };

    EachInIterable.prototype.memoReferenceFor = function memoReferenceFor(item) {
      return new _emberGlimmerUtilsReferences.UpdatableReference(item.value);
    };

    EachInIterable.prototype.updateMemoReference = function updateMemoReference(reference, item) {
      reference.update(item.value);
    };

    return EachInIterable;
  })();

  var ArrayIterable = (function () {
    function ArrayIterable(ref, keyFor) {
      this.ref = ref;
      this.keyFor = keyFor;

      var valueTag = this.valueTag = new _glimmerReference.UpdatableTag(_glimmerReference.CONSTANT_TAG);

      this.tag = _glimmerReference.combine([ref.tag, valueTag]);
    }

    ArrayIterable.prototype.iterate = function iterate() {
      var ref = this.ref;
      var keyFor = this.keyFor;
      var valueTag = this.valueTag;

      var iterable = ref.value();

      valueTag.update(_emberMetal.tagForProperty(iterable, '[]'));

      if (!iterable || typeof iterable !== 'object') {
        return EMPTY_ITERATOR;
      }

      if (Array.isArray(iterable)) {
        return iterable.length > 0 ? new ArrayIterator(iterable, keyFor) : EMPTY_ITERATOR;
      } else if (_emberRuntime.isEmberArray(iterable)) {
        return _emberMetal.get(iterable, 'length') > 0 ? new EmberArrayIterator(iterable, keyFor) : EMPTY_ITERATOR;
      } else if (typeof iterable.forEach === 'function') {
        var _ret = (function () {
          var array = [];
          iterable.forEach(function (item) {
            array.push(item);
          });
          return {
            v: array.length > 0 ? new ArrayIterator(array, keyFor) : EMPTY_ITERATOR
          };
        })();

        if (typeof _ret === 'object') return _ret.v;
      } else {
        return EMPTY_ITERATOR;
      }
    };

    ArrayIterable.prototype.valueReferenceFor = function valueReferenceFor(item) {
      return new _emberGlimmerUtilsReferences.UpdatableReference(item.value);
    };

    ArrayIterable.prototype.updateValueReference = function updateValueReference(reference, item) {
      reference.update(item.value);
    };

    ArrayIterable.prototype.memoReferenceFor = function memoReferenceFor(item) {
      return new _emberGlimmerUtilsReferences.UpdatablePrimitiveReference(item.memo);
    };

    ArrayIterable.prototype.updateMemoReference = function updateMemoReference(reference, item) {
      reference.update(item.memo);
    };

    return ArrayIterable;
  })();
});
enifed('ember-glimmer/utils/process-args', ['exports', 'ember-utils', 'glimmer-reference', 'ember-glimmer/component', 'ember-glimmer/utils/references', 'ember-views', 'ember-glimmer/helpers/action', 'glimmer-runtime'], function (exports, _emberUtils, _glimmerReference, _emberGlimmerComponent, _emberGlimmerUtilsReferences, _emberViews, _emberGlimmerHelpersAction, _glimmerRuntime) {
  'use strict';

  exports.gatherArgs = gatherArgs;

  // Maps all variants of positional and dynamically scoped arguments
  // into the named arguments. Input `args` and return value are both
  // `EvaluatedArgs`.

  function gatherArgs(args, definition) {
    var namedMap = gatherNamedMap(args, definition);
    var positionalValues = gatherPositionalValues(args, definition);
    return mergeArgs(namedMap, positionalValues, args.blocks, definition.ComponentClass);
  }

  function gatherNamedMap(args, definition) {
    var namedMap = args.named.map;
    if (definition.args) {
      return _emberUtils.assign({}, definition.args.named.map, namedMap);
    } else {
      return namedMap;
    }
  }

  function gatherPositionalValues(args, definition) {
    var positionalValues = args.positional.values;
    if (definition.args) {
      var oldPositional = definition.args.positional.values;
      var newPositional = [];
      newPositional.push.apply(newPositional, oldPositional);
      newPositional.splice.apply(newPositional, [0, positionalValues.length].concat(positionalValues));
      return newPositional;
    } else {
      return positionalValues;
    }
  }

  function mergeArgs(namedMap, positionalValues, blocks, componentClass) {
    var positionalParamsDefinition = componentClass.positionalParams;

    if (positionalParamsDefinition && positionalParamsDefinition.length > 0 && positionalValues.length > 0) {
      if (typeof positionalParamsDefinition === 'string') {
        namedMap = mergeRestArg(namedMap, positionalValues, positionalParamsDefinition);
      } else {
        namedMap = mergePositionalParams(namedMap, positionalValues, positionalParamsDefinition);
      }
    }
    return _glimmerRuntime.EvaluatedArgs.named(namedMap, blocks);
  }

  var EMPTY_ARGS = {
    tag: _glimmerReference.CONSTANT_TAG,
    value: function () {
      var _props;

      return { attrs: {}, props: (_props = { attrs: {} }, _props[_emberGlimmerComponent.ARGS] = {}, _props) };
    }
  };

  // ComponentArgs takes EvaluatedNamedArgs and converts them into the
  // inputs needed by CurlyComponents (attrs and props, with mutable
  // cells, etc).

  var ComponentArgs = (function () {
    ComponentArgs.create = function create(args) {
      if (args.named.keys.length === 0) {
        return EMPTY_ARGS;
      } else {
        return new ComponentArgs(args.named);
      }
    };

    function ComponentArgs(namedArgs) {
      this.tag = namedArgs.tag;
      this.namedArgs = namedArgs;
    }

    ComponentArgs.prototype.value = function value() {
      var namedArgs = this.namedArgs;

      var keys = namedArgs.keys;
      var attrs = namedArgs.value();
      var props = new _emberUtils.EmptyObject();
      var args = new _emberUtils.EmptyObject();

      props[_emberGlimmerComponent.ARGS] = args;

      for (var i = 0, l = keys.length; i < l; i++) {
        var _name = keys[i];
        var ref = namedArgs.get(_name);
        var value = attrs[_name];

        if (typeof value === 'function' && value[_emberGlimmerHelpersAction.ACTION]) {
          attrs[_name] = value;
        } else if (ref[_emberGlimmerUtilsReferences.UPDATE]) {
          attrs[_name] = new MutableCell(ref, value);
        }

        args[_name] = ref;
        props[_name] = value;
      }

      props.attrs = attrs;

      return { attrs: attrs, props: props };
    };

    return ComponentArgs;
  })();

  exports.ComponentArgs = ComponentArgs;

  function mergeRestArg(namedMap, positionalValues, restArgName) {
    var mergedNamed = _emberUtils.assign({}, namedMap);
    mergedNamed[restArgName] = _glimmerRuntime.EvaluatedPositionalArgs.create(positionalValues);
    return mergedNamed;
  }

  function mergePositionalParams(namedMap, values, positionalParamNames) {
    var mergedNamed = _emberUtils.assign({}, namedMap);
    var length = Math.min(values.length, positionalParamNames.length);
    for (var i = 0; i < length; i++) {
      var _name2 = positionalParamNames[i];
      mergedNamed[_name2] = values[i];
    }
    return mergedNamed;
  }

  var REF = _emberUtils.symbol('REF');

  var MutableCell = (function () {
    function MutableCell(ref, value) {
      this[_emberViews.MUTABLE_CELL] = true;
      this[REF] = ref;
      this.value = value;
    }

    MutableCell.prototype.update = function update(val) {
      this[REF][_emberGlimmerUtilsReferences.UPDATE](val);
    };

    return MutableCell;
  })();
});
enifed('ember-glimmer/utils/references', ['exports', 'ember-utils', 'ember-metal', 'glimmer-reference', 'glimmer-runtime', 'ember-glimmer/utils/to-bool', 'ember-glimmer/helper'], function (exports, _emberUtils, _emberMetal, _glimmerReference, _glimmerRuntime, _emberGlimmerUtilsToBool, _emberGlimmerHelper) {
  'use strict';

  var UPDATE = _emberUtils.symbol('UPDATE');

  exports.UPDATE = UPDATE;
  exports.NULL_REFERENCE = _glimmerRuntime.NULL_REFERENCE;
  exports.UNDEFINED_REFERENCE = _glimmerRuntime.UNDEFINED_REFERENCE;

  // @abstract
  // @implements PathReference

  var EmberPathReference = (function () {
    function EmberPathReference() {}

    // @abstract

    // @abstract get tag()
    // @abstract value()

    EmberPathReference.prototype.get = function get(key) {
      return PropertyReference.create(this, key);
    };

    return EmberPathReference;
  })();

  var CachedReference = (function (_EmberPathReference) {
    babelHelpers.inherits(CachedReference, _EmberPathReference);

    function CachedReference() {
      _EmberPathReference.call(this);
      this._lastRevision = null;
      this._lastValue = null;
    }

    // @implements PathReference

    CachedReference.prototype.value = function value() {
      var tag = this.tag;
      var _lastRevision = this._lastRevision;
      var _lastValue = this._lastValue;

      if (!_lastRevision || !tag.validate(_lastRevision)) {
        _lastValue = this._lastValue = this.compute();
        this._lastRevision = tag.value();
      }

      return _lastValue;
    };

    // @abstract compute()
    return CachedReference;
  })(EmberPathReference);

  exports.CachedReference = CachedReference;

  var RootReference = (function (_ConstReference) {
    babelHelpers.inherits(RootReference, _ConstReference);

    function RootReference(value) {
      _ConstReference.call(this, value);
      this.children = new _emberUtils.EmptyObject();
    }

    RootReference.prototype.get = function get(propertyKey) {
      var ref = this.children[propertyKey];

      if (!ref) {
        ref = this.children[propertyKey] = new RootPropertyReference(this.inner, propertyKey);
      }

      return ref;
    };

    return RootReference;
  })(_glimmerReference.ConstReference);

  exports.RootReference = RootReference;

  var TwoWayFlushDetectionTag = undefined;

  if (false || false) {
    TwoWayFlushDetectionTag = (function () {
      function _class(tag, key, ref) {
        this.tag = tag;
        this.parent = null;
        this.key = key;
        this.ref = ref;
      }

      _class.prototype.value = function value() {
        return this.tag.value();
      };

      _class.prototype.validate = function validate(ticket) {
        var parent = this.parent;
        var key = this.key;

        var isValid = this.tag.validate(ticket);

        if (isValid && parent) {
          _emberMetal.didRender(parent, key, this.ref);
        }

        return isValid;
      };

      _class.prototype.didCompute = function didCompute(parent) {
        this.parent = parent;
        _emberMetal.didRender(parent, this.key, this.ref);
      };

      return _class;
    })();
  }

  var PropertyReference = (function (_CachedReference) {
    babelHelpers.inherits(PropertyReference, _CachedReference);

    function PropertyReference() {
      _CachedReference.apply(this, arguments);
    }

    PropertyReference.create = function create(parentReference, propertyKey) {
      if (_glimmerReference.isConst(parentReference)) {
        return new RootPropertyReference(parentReference.value(), propertyKey);
      } else {
        return new NestedPropertyReference(parentReference, propertyKey);
      }
    };

    PropertyReference.prototype.get = function get(key) {
      return new NestedPropertyReference(this, key);
    };

    return PropertyReference;
  })(CachedReference);

  exports.PropertyReference = PropertyReference;

  var RootPropertyReference = (function (_PropertyReference) {
    babelHelpers.inherits(RootPropertyReference, _PropertyReference);

    function RootPropertyReference(parentValue, propertyKey) {
      _PropertyReference.call(this);

      this._parentValue = parentValue;
      this._propertyKey = propertyKey;

      if (false || false) {
        this.tag = new TwoWayFlushDetectionTag(_emberMetal.tagForProperty(parentValue, propertyKey), propertyKey, this);
      } else {
        this.tag = _emberMetal.tagForProperty(parentValue, propertyKey);
      }

      if (false) {
        _emberMetal.watchKey(parentValue, propertyKey);
      }
    }

    RootPropertyReference.prototype.compute = function compute() {
      var _parentValue = this._parentValue;
      var _propertyKey = this._propertyKey;

      if (false || false) {
        this.tag.didCompute(_parentValue);
      }

      return _emberMetal.get(_parentValue, _propertyKey);
    };

    RootPropertyReference.prototype[UPDATE] = function (value) {
      _emberMetal.set(this._parentValue, this._propertyKey, value);
    };

    return RootPropertyReference;
  })(PropertyReference);

  exports.RootPropertyReference = RootPropertyReference;

  var NestedPropertyReference = (function (_PropertyReference2) {
    babelHelpers.inherits(NestedPropertyReference, _PropertyReference2);

    function NestedPropertyReference(parentReference, propertyKey) {
      _PropertyReference2.call(this);

      var parentReferenceTag = parentReference.tag;
      var parentObjectTag = new _glimmerReference.UpdatableTag(_glimmerReference.CONSTANT_TAG);

      this._parentReference = parentReference;
      this._parentObjectTag = parentObjectTag;
      this._propertyKey = propertyKey;

      if (false || false) {
        var tag = _glimmerReference.combine([parentReferenceTag, parentObjectTag]);
        this.tag = new TwoWayFlushDetectionTag(tag, propertyKey, this);
      } else {
        this.tag = _glimmerReference.combine([parentReferenceTag, parentObjectTag]);
      }
    }

    NestedPropertyReference.prototype.compute = function compute() {
      var _parentReference = this._parentReference;
      var _parentObjectTag = this._parentObjectTag;
      var _propertyKey = this._propertyKey;

      var parentValue = _parentReference.value();

      _parentObjectTag.update(_emberMetal.tagForProperty(parentValue, _propertyKey));

      if (typeof parentValue === 'string' && _propertyKey === 'length') {
        return parentValue.length;
      }

      if (typeof parentValue === 'object' && parentValue) {
        if (false) {
          _emberMetal.watchKey(parentValue, _propertyKey);
        }

        if (false || false) {
          this.tag.didCompute(parentValue);
        }

        return _emberMetal.get(parentValue, _propertyKey);
      } else {
        return undefined;
      }
    };

    NestedPropertyReference.prototype[UPDATE] = function (value) {
      var parent = this._parentReference.value();
      _emberMetal.set(parent, this._propertyKey, value);
    };

    return NestedPropertyReference;
  })(PropertyReference);

  exports.NestedPropertyReference = NestedPropertyReference;

  var UpdatableReference = (function (_EmberPathReference2) {
    babelHelpers.inherits(UpdatableReference, _EmberPathReference2);

    function UpdatableReference(value) {
      _EmberPathReference2.call(this);

      this.tag = new _glimmerReference.DirtyableTag();
      this._value = value;
    }

    UpdatableReference.prototype.value = function value() {
      return this._value;
    };

    UpdatableReference.prototype.update = function update(value) {
      var _value = this._value;

      if (value !== _value) {
        this.tag.dirty();
        this._value = value;
      }
    };

    return UpdatableReference;
  })(EmberPathReference);

  exports.UpdatableReference = UpdatableReference;

  var UpdatablePrimitiveReference = (function (_UpdatableReference) {
    babelHelpers.inherits(UpdatablePrimitiveReference, _UpdatableReference);

    function UpdatablePrimitiveReference() {
      _UpdatableReference.apply(this, arguments);
    }

    UpdatablePrimitiveReference.prototype.get = function get() {
      return _glimmerRuntime.UNDEFINED_REFERENCE;
    };

    return UpdatablePrimitiveReference;
  })(UpdatableReference);

  exports.UpdatablePrimitiveReference = UpdatablePrimitiveReference;

  var ConditionalReference = (function (_GlimmerConditionalReference) {
    babelHelpers.inherits(ConditionalReference, _GlimmerConditionalReference);

    ConditionalReference.create = function create(reference) {
      if (_glimmerReference.isConst(reference)) {
        var value = reference.value();

        if (_emberMetal.isProxy(value)) {
          return new RootPropertyReference(value, 'isTruthy');
        } else {
          return _glimmerRuntime.PrimitiveReference.create(_emberGlimmerUtilsToBool.default(value));
        }
      }

      return new ConditionalReference(reference);
    };

    function ConditionalReference(reference) {
      _GlimmerConditionalReference.call(this, reference);

      this.objectTag = new _glimmerReference.UpdatableTag(_glimmerReference.CONSTANT_TAG);
      this.tag = _glimmerReference.combine([reference.tag, this.objectTag]);
    }

    ConditionalReference.prototype.toBool = function toBool(predicate) {
      if (_emberMetal.isProxy(predicate)) {
        this.objectTag.update(_emberMetal.tagForProperty(predicate, 'isTruthy'));
        return _emberMetal.get(predicate, 'isTruthy');
      } else {
        this.objectTag.update(_emberMetal.tagFor(predicate));
        return _emberGlimmerUtilsToBool.default(predicate);
      }
    };

    return ConditionalReference;
  })(_glimmerRuntime.ConditionalReference);

  exports.ConditionalReference = ConditionalReference;

  var SimpleHelperReference = (function (_CachedReference2) {
    babelHelpers.inherits(SimpleHelperReference, _CachedReference2);

    SimpleHelperReference.create = function create(helper, args) {
      if (_glimmerReference.isConst(args)) {
        var positional = args.positional;
        var named = args.named;

        var positionalValue = positional.value();
        var namedValue = named.value();

        var result = helper(positionalValue, namedValue);

        if (result === null) {
          return _glimmerRuntime.NULL_REFERENCE;
        } else if (result === undefined) {
          return _glimmerRuntime.UNDEFINED_REFERENCE;
        } else if (typeof result === 'object') {
          return new RootReference(result);
        } else {
          return _glimmerRuntime.PrimitiveReference.create(result);
        }
      } else {
        return new SimpleHelperReference(helper, args);
      }
    };

    function SimpleHelperReference(helper, args) {
      _CachedReference2.call(this);

      this.tag = args.tag;
      this.helper = helper;
      this.args = args;
    }

    SimpleHelperReference.prototype.compute = function compute() {
      var helper = this.helper;
      var _args = this.args;
      var positional = _args.positional;
      var named = _args.named;

      var positionalValue = positional.value();
      var namedValue = named.value();

      return helper(positionalValue, namedValue);
    };

    return SimpleHelperReference;
  })(CachedReference);

  exports.SimpleHelperReference = SimpleHelperReference;

  var ClassBasedHelperReference = (function (_CachedReference3) {
    babelHelpers.inherits(ClassBasedHelperReference, _CachedReference3);

    ClassBasedHelperReference.create = function create(helperClass, vm, args) {
      var instance = helperClass.create();
      vm.newDestroyable(instance);
      return new ClassBasedHelperReference(instance, args);
    };

    function ClassBasedHelperReference(instance, args) {
      _CachedReference3.call(this);

      this.tag = _glimmerReference.combine([instance[_emberGlimmerHelper.RECOMPUTE_TAG], args.tag]);
      this.instance = instance;
      this.args = args;
    }

    ClassBasedHelperReference.prototype.compute = function compute() {
      var instance = this.instance;
      var _args2 = this.args;
      var positional = _args2.positional;
      var named = _args2.named;

      var positionalValue = positional.value();
      var namedValue = named.value();

      return instance.compute(positionalValue, namedValue);
    };

    return ClassBasedHelperReference;
  })(CachedReference);

  exports.ClassBasedHelperReference = ClassBasedHelperReference;

  var InternalHelperReference = (function (_CachedReference4) {
    babelHelpers.inherits(InternalHelperReference, _CachedReference4);

    function InternalHelperReference(helper, args) {
      _CachedReference4.call(this);

      this.tag = args.tag;
      this.helper = helper;
      this.args = args;
    }

    // @implements PathReference

    InternalHelperReference.prototype.compute = function compute() {
      var helper = this.helper;
      var args = this.args;

      return helper(args);
    };

    return InternalHelperReference;
  })(CachedReference);

  exports.InternalHelperReference = InternalHelperReference;

  var UnboundReference = (function (_ConstReference2) {
    babelHelpers.inherits(UnboundReference, _ConstReference2);

    function UnboundReference() {
      _ConstReference2.apply(this, arguments);
    }

    UnboundReference.create = function create(value) {
      if (value === null) {
        return _glimmerRuntime.NULL_REFERENCE;
      } else if (value === undefined) {
        return _glimmerRuntime.UNDEFINED_REFERENCE;
      } else if (typeof value === 'object') {
        return new UnboundReference(value);
      } else {
        return _glimmerRuntime.PrimitiveReference.create(value);
      }
    };

    UnboundReference.prototype.get = function get(key) {
      return new UnboundReference(_emberMetal.get(this.inner, key));
    };

    return UnboundReference;
  })(_glimmerReference.ConstReference);

  exports.UnboundReference = UnboundReference;
});
enifed('ember-glimmer/utils/string', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  /**
  @module ember
  @submodule ember-glimmer
  */

  'use strict';

  exports.getSafeString = getSafeString;
  exports.escapeExpression = escapeExpression;
  exports.htmlSafe = htmlSafe;
  exports.isHTMLSafe = isHTMLSafe;

  var SafeString = (function () {
    function SafeString(string) {
      this.string = string;
    }

    SafeString.prototype.toString = function toString() {
      return '' + this.string;
    };

    SafeString.prototype.toHTML = function toHTML() {
      return this.toString();
    };

    return SafeString;
  })();

  exports.SafeString = SafeString;

  function getSafeString() {

    return SafeString;
  }

  var escape = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    // jscs:disable
    "'": '&#x27;',
    // jscs:enable
    '`': '&#x60;',
    '=': '&#x3D;'
  };

  var possible = /[&<>"'`=]/;
  var badChars = /[&<>"'`=]/g;

  function escapeChar(chr) {
    return escape[chr];
  }

  function escapeExpression(string) {
    if (typeof string !== 'string') {
      // don't escape SafeStrings, since they're already safe
      if (string && string.toHTML) {
        return string.toHTML();
      } else if (string == null) {
        return '';
      } else if (!string) {
        return string + '';
      }

      // Force a string conversion as this will be done by the append regardless and
      // the regex test will do this transparently behind the scenes, causing issues if
      // an object's to string has escaped characters in it.
      string = '' + string;
    }

    if (!possible.test(string)) {
      return string;
    }
    return string.replace(badChars, escapeChar);
  }

  /**
    Mark a string as safe for unescaped output with Ember templates. If you
    return HTML from a helper, use this function to
    ensure Ember's rendering layer does not escape the HTML.
  
    ```javascript
    Ember.String.htmlSafe('<div>someString</div>')
    ```
  
    @method htmlSafe
    @for Ember.String
    @static
    @return {Handlebars.SafeString} A string that will not be HTML escaped by Handlebars.
    @public
  */

  function htmlSafe(str) {
    if (str === null || str === undefined) {
      str = '';
    } else if (typeof str !== 'string') {
      str = '' + str;
    }
    return new SafeString(str);
  }

  /**
    Detects if a string was decorated using `Ember.String.htmlSafe`.
  
    ```javascript
    var plainString = 'plain string',
        safeString = Ember.String.htmlSafe('<div>someValue</div>');
  
    Ember.String.isHTMLSafe(plainString); // false
    Ember.String.isHTMLSafe(safeString);  // true
    ```
  
    @method isHTMLSafe
    @for Ember.String
    @static
    @return {Boolean} `true` if the string was decorated with `htmlSafe`, `false` otherwise.
    @public
  */

  function isHTMLSafe(str) {
    return str && typeof str.toHTML === 'function';
  }
});
enifed('ember-glimmer/utils/to-bool', ['exports', 'ember-runtime', 'ember-metal'], function (exports, _emberRuntime, _emberMetal) {
  'use strict';

  exports.default = toBool;

  function toBool(predicate) {
    if (!predicate) {
      return false;
    }

    if (predicate === true) {
      return true;
    }

    if (_emberRuntime.isArray(predicate)) {
      return _emberMetal.get(predicate, 'length') !== 0;
    }

    return true;
  }
});
enifed('ember-glimmer/views/outlet', ['exports', 'ember-utils', 'glimmer-reference', 'ember-environment', 'ember-metal'], function (exports, _emberUtils, _glimmerReference, _emberEnvironment, _emberMetal) {
  /**
  @module ember
  @submodule ember-glimmer
  */
  'use strict';

  var OutletStateReference = (function () {
    function OutletStateReference(outletView) {
      this.outletView = outletView;
      this.tag = outletView._tag;
    }

    // So this is a relic of the past that SHOULD go away
    // in 3.0. Preferably it is deprecated in the release that
    // follows the Glimmer release.

    OutletStateReference.prototype.get = function get(key) {
      return new ChildOutletStateReference(this, key);
    };

    OutletStateReference.prototype.value = function value() {
      return this.outletView.outletState;
    };

    OutletStateReference.prototype.getOrphan = function getOrphan(name) {
      return new OrphanedOutletStateReference(this, name);
    };

    OutletStateReference.prototype.update = function update(state) {
      this.outletView.setOutletState(state);
    };

    return OutletStateReference;
  })();

  var OrphanedOutletStateReference = (function (_OutletStateReference) {
    babelHelpers.inherits(OrphanedOutletStateReference, _OutletStateReference);

    function OrphanedOutletStateReference(root, name) {
      _OutletStateReference.call(this, root.outletView);
      this.root = root;
      this.name = name;
    }

    OrphanedOutletStateReference.prototype.value = function value() {
      var rootState = this.root.value();

      var orphans = rootState.outlets.main.outlets.__ember_orphans__;

      if (!orphans) {
        return null;
      }

      var matched = orphans.outlets[this.name];

      if (!matched) {
        return null;
      }

      var state = new _emberUtils.EmptyObject();
      state[matched.render.outlet] = matched;
      matched.wasUsed = true;
      return { outlets: state };
    };

    return OrphanedOutletStateReference;
  })(OutletStateReference);

  var ChildOutletStateReference = (function () {
    function ChildOutletStateReference(parent, key) {
      this.parent = parent;
      this.key = key;
      this.tag = parent.tag;
    }

    ChildOutletStateReference.prototype.get = function get(key) {
      return new ChildOutletStateReference(this, key);
    };

    ChildOutletStateReference.prototype.value = function value() {
      return this.parent.value()[this.key];
    };

    return ChildOutletStateReference;
  })();

  var OutletView = (function () {
    OutletView.extend = function extend(injections) {
      return (function (_OutletView) {
        babelHelpers.inherits(_class, _OutletView);

        function _class() {
          _OutletView.apply(this, arguments);
        }

        _class.create = function create(options) {
          if (options) {
            return _OutletView.create.call(this, _emberUtils.assign({}, injections, options));
          } else {
            return _OutletView.create.call(this, injections);
          }
        };

        return _class;
      })(OutletView);
    };

    OutletView.reopenClass = function reopenClass(injections) {
      _emberUtils.assign(this, injections);
    };

    OutletView.create = function create(options) {
      var _environment = options._environment;
      var renderer = options.renderer;
      var template = options.template;

      var owner = options[_emberUtils.OWNER];
      return new OutletView(_environment, renderer, owner, template);
    };

    function OutletView(_environment, renderer, owner, template) {
      this._environment = _environment;
      this.renderer = renderer;
      this.owner = owner;
      this.template = template;
      this.outletState = null;
      this._tag = new _glimmerReference.DirtyableTag();
    }

    OutletView.prototype.appendTo = function appendTo(selector) {
      var env = this._environment || _emberEnvironment.environment;
      var target = undefined;

      if (env.hasDOM) {
        target = typeof selector === 'string' ? document.querySelector(selector) : selector;
      } else {
        target = selector;
      }

      _emberMetal.run.schedule('render', this.renderer, 'appendOutletView', this, target);
    };

    OutletView.prototype.rerender = function rerender() {};

    OutletView.prototype.setOutletState = function setOutletState(state) {
      this.outletState = {
        outlets: {
          main: state
        },
        render: {
          owner: undefined,
          into: undefined,
          outlet: 'main',
          name: '-top-level',
          controller: undefined,
          ViewClass: undefined,
          template: undefined
        }
      };
      this._tag.dirty();
    };

    OutletView.prototype.toReference = function toReference() {
      return new OutletStateReference(this);
    };

    OutletView.prototype.destroy = function destroy() {};

    return OutletView;
  })();

  exports.default = OutletView;
});
enifed('ember-metal/alias', ['exports', 'ember-utils', 'ember-metal/debug', 'ember-metal/property_get', 'ember-metal/property_set', 'ember-metal/error', 'ember-metal/properties', 'ember-metal/computed', 'ember-metal/meta', 'ember-metal/dependent_keys'], function (exports, _emberUtils, _emberMetalDebug, _emberMetalProperty_get, _emberMetalProperty_set, _emberMetalError, _emberMetalProperties, _emberMetalComputed, _emberMetalMeta, _emberMetalDependent_keys) {
  'use strict';

  exports.default = alias;
  exports.AliasedProperty = AliasedProperty;

  var CONSUMED = {};

  function alias(altKey) {
    return new AliasedProperty(altKey);
  }

  function AliasedProperty(altKey) {
    this.isDescriptor = true;
    this.altKey = altKey;
    this._dependentKeys = [altKey];
  }

  AliasedProperty.prototype = Object.create(_emberMetalProperties.Descriptor.prototype);

  AliasedProperty.prototype.setup = function (obj, keyName) {
    var meta = _emberMetalMeta.meta(obj);
    if (meta.peekWatching(keyName)) {
      _emberMetalDependent_keys.addDependentKeys(this, obj, keyName, meta);
    }
  };

  AliasedProperty.prototype.teardown = function (obj, keyName) {
    var meta = _emberMetalMeta.meta(obj);
    if (meta.peekWatching(keyName)) {
      _emberMetalDependent_keys.removeDependentKeys(this, obj, keyName, meta);
    }
  };

  AliasedProperty.prototype.willWatch = function (obj, keyName) {
    _emberMetalDependent_keys.addDependentKeys(this, obj, keyName, _emberMetalMeta.meta(obj));
  };

  AliasedProperty.prototype.didUnwatch = function (obj, keyName) {
    _emberMetalDependent_keys.removeDependentKeys(this, obj, keyName, _emberMetalMeta.meta(obj));
  };

  AliasedProperty.prototype.get = function AliasedProperty_get(obj, keyName) {
    var ret = _emberMetalProperty_get.get(obj, this.altKey);
    var meta = _emberMetalMeta.meta(obj);
    var cache = meta.writableCache();
    if (cache[keyName] !== CONSUMED) {
      cache[keyName] = CONSUMED;
      _emberMetalDependent_keys.addDependentKeys(this, obj, keyName, meta);
    }
    return ret;
  };

  AliasedProperty.prototype.set = function AliasedProperty_set(obj, keyName, value) {
    return _emberMetalProperty_set.set(obj, this.altKey, value);
  };

  AliasedProperty.prototype.readOnly = function () {
    this.set = AliasedProperty_readOnlySet;
    return this;
  };

  function AliasedProperty_readOnlySet(obj, keyName, value) {
    throw new _emberMetalError.default('Cannot set read-only property \'' + keyName + '\' on object: ' + _emberUtils.inspect(obj));
  }

  AliasedProperty.prototype.oneWay = function () {
    this.set = AliasedProperty_oneWaySet;
    return this;
  };

  function AliasedProperty_oneWaySet(obj, keyName, value) {
    _emberMetalProperties.defineProperty(obj, keyName, null);
    return _emberMetalProperty_set.set(obj, keyName, value);
  }

  // Backwards compatibility with Ember Data.
  AliasedProperty.prototype._meta = undefined;
  AliasedProperty.prototype.meta = _emberMetalComputed.ComputedProperty.prototype.meta;
});
enifed('ember-metal/binding', ['exports', 'ember-utils', 'ember-console', 'ember-environment', 'ember-metal/run_loop', 'ember-metal/debug', 'ember-metal/property_get', 'ember-metal/property_set', 'ember-metal/events', 'ember-metal/observer', 'ember-metal/path_cache'], function (exports, _emberUtils, _emberConsole, _emberEnvironment, _emberMetalRun_loop, _emberMetalDebug, _emberMetalProperty_get, _emberMetalProperty_set, _emberMetalEvents, _emberMetalObserver, _emberMetalPath_cache) {
  'use strict';

  exports.bind = bind;

  /**
  @module ember
  @submodule ember-metal
  */

  // ..........................................................
  // BINDING
  //

  function Binding(toPath, fromPath) {
    // Configuration
    this._from = fromPath;
    this._to = toPath;
    this._oneWay = undefined;

    // State
    this._direction = undefined;
    this._readyToSync = undefined;
    this._fromObj = undefined;
    this._fromPath = undefined;
    this._toObj = undefined;
  }

  /**
    @class Binding
    @namespace Ember
    @deprecated See http://emberjs.com/deprecations/v2.x#toc_ember-binding
    @public
  */

  Binding.prototype = {
    /**
      This copies the Binding so it can be connected to another object.
       @method copy
      @return {Ember.Binding} `this`
      @public
    */
    copy: function () {
      var copy = new Binding(this._to, this._from);
      if (this._oneWay) {
        copy._oneWay = true;
      }
      return copy;
    },

    // ..........................................................
    // CONFIG
    //

    /**
      This will set `from` property path to the specified value. It will not
      attempt to resolve this property path to an actual object until you
      connect the binding.
       The binding will search for the property path starting at the root object
      you pass when you `connect()` the binding. It follows the same rules as
      `get()` - see that method for more information.
       @method from
      @param {String} path The property path to connect to.
      @return {Ember.Binding} `this`
      @public
    */
    from: function (path) {
      this._from = path;
      return this;
    },

    /**
      This will set the `to` property path to the specified value. It will not
      attempt to resolve this property path to an actual object until you
      connect the binding.
       The binding will search for the property path starting at the root object
      you pass when you `connect()` the binding. It follows the same rules as
      `get()` - see that method for more information.
       @method to
      @param {String|Tuple} path A property path or tuple.
      @return {Ember.Binding} `this`
      @public
    */
    to: function (path) {
      this._to = path;
      return this;
    },

    /**
      Configures the binding as one way. A one-way binding will relay changes
      on the `from` side to the `to` side, but not the other way around. This
      means that if you change the `to` side directly, the `from` side may have
      a different value.
       @method oneWay
      @return {Ember.Binding} `this`
      @public
    */
    oneWay: function () {
      this._oneWay = true;
      return this;
    },

    /**
      @method toString
      @return {String} string representation of binding
      @public
    */
    toString: function () {
      var oneWay = this._oneWay ? '[oneWay]' : '';
      return 'Ember.Binding<' + _emberUtils.guidFor(this) + '>(' + this._from + ' -> ' + this._to + ')' + oneWay;
    },

    // ..........................................................
    // CONNECT AND SYNC
    //

    /**
      Attempts to connect this binding instance so that it can receive and relay
      changes. This method will raise an exception if you have not set the
      from/to properties yet.
       @method connect
      @param {Object} obj The root object for this binding.
      @return {Ember.Binding} `this`
      @public
    */
    connect: function (obj) {

      var fromObj = undefined,
          fromPath = undefined,
          possibleGlobal = undefined;

      // If the binding's "from" path could be interpreted as a global, verify
      // whether the path refers to a global or not by consulting `Ember.lookup`.
      if (_emberMetalPath_cache.isGlobalPath(this._from)) {
        var _name = _emberMetalPath_cache.getFirstKey(this._from);
        possibleGlobal = _emberEnvironment.context.lookup[_name];

        if (possibleGlobal) {
          fromObj = possibleGlobal;
          fromPath = _emberMetalPath_cache.getTailPath(this._from);
        }
      }

      if (fromObj === undefined) {
        fromObj = obj;
        fromPath = this._from;
      }

      _emberMetalProperty_set.trySet(obj, this._to, _emberMetalProperty_get.get(fromObj, fromPath));

      // Add an observer on the object to be notified when the binding should be updated.
      _emberMetalObserver.addObserver(fromObj, fromPath, this, 'fromDidChange');

      // If the binding is a two-way binding, also set up an observer on the target.
      if (!this._oneWay) {
        _emberMetalObserver.addObserver(obj, this._to, this, 'toDidChange');
      }

      _emberMetalEvents.addListener(obj, 'willDestroy', this, 'disconnect');

      fireDeprecations(obj, this._to, this._from, possibleGlobal, this._oneWay, !possibleGlobal && !this._oneWay);

      this._readyToSync = true;
      this._fromObj = fromObj;
      this._fromPath = fromPath;
      this._toObj = obj;

      return this;
    },

    /**
      Disconnects the binding instance. Changes will no longer be relayed. You
      will not usually need to call this method.
       @method disconnect
      @return {Ember.Binding} `this`
      @public
    */
    disconnect: function () {

      // Remove an observer on the object so we're no longer notified of
      // changes that should update bindings.
      _emberMetalObserver.removeObserver(this._fromObj, this._fromPath, this, 'fromDidChange');

      // If the binding is two-way, remove the observer from the target as well.
      if (!this._oneWay) {
        _emberMetalObserver.removeObserver(this._toObj, this._to, this, 'toDidChange');
      }

      this._readyToSync = false; // Disable scheduled syncs...
      return this;
    },

    // ..........................................................
    // PRIVATE
    //

    /* Called when the from side changes. */
    fromDidChange: function (target) {
      this._scheduleSync('fwd');
    },

    /* Called when the to side changes. */
    toDidChange: function (target) {
      this._scheduleSync('back');
    },

    _scheduleSync: function (dir) {
      var existingDir = this._direction;

      // If we haven't scheduled the binding yet, schedule it.
      if (existingDir === undefined) {
        _emberMetalRun_loop.default.schedule('sync', this, '_sync');
        this._direction = dir;
      }

      // If both a 'back' and 'fwd' sync have been scheduled on the same object,
      // default to a 'fwd' sync so that it remains deterministic.
      if (existingDir === 'back' && dir === 'fwd') {
        this._direction = 'fwd';
      }
    },

    _sync: function () {
      var log = _emberEnvironment.ENV.LOG_BINDINGS;

      var toObj = this._toObj;

      // Don't synchronize destroyed objects or disconnected bindings.
      if (toObj.isDestroyed || !this._readyToSync) {
        return;
      }

      // Get the direction of the binding for the object we are
      // synchronizing from.
      var direction = this._direction;

      var fromObj = this._fromObj;
      var fromPath = this._fromPath;

      this._direction = undefined;

      // If we're synchronizing from the remote object...
      if (direction === 'fwd') {
        var fromValue = _emberMetalProperty_get.get(fromObj, fromPath);
        if (log) {
          _emberConsole.default.log(' ', this.toString(), '->', fromValue, fromObj);
        }
        if (this._oneWay) {
          _emberMetalProperty_set.trySet(toObj, this._to, fromValue);
        } else {
          _emberMetalObserver._suspendObserver(toObj, this._to, this, 'toDidChange', function () {
            _emberMetalProperty_set.trySet(toObj, this._to, fromValue);
          });
        }
        // If we're synchronizing *to* the remote object.
      } else if (direction === 'back') {
          var toValue = _emberMetalProperty_get.get(toObj, this._to);
          if (log) {
            _emberConsole.default.log(' ', this.toString(), '<-', toValue, toObj);
          }
          _emberMetalObserver._suspendObserver(fromObj, fromPath, this, 'fromDidChange', function () {
            _emberMetalProperty_set.trySet(fromObj, fromPath, toValue);
          });
        }
    }

  };

  function fireDeprecations(obj, toPath, fromPath, deprecateGlobal, deprecateOneWay, deprecateAlias) {
    var deprecateGlobalMessage = '`Ember.Binding` is deprecated. Since you' + ' are binding to a global consider using a service instead.';
    var deprecateOneWayMessage = '`Ember.Binding` is deprecated. Since you' + ' are using a `oneWay` binding consider using a `readOnly` computed' + ' property instead.';
    var deprecateAliasMessage = '`Ember.Binding` is deprecated. Consider' + ' using an `alias` computed property instead.';

    var objectInfo = 'The `' + toPath + '` property of `' + obj + '` is an `Ember.Binding` connected to `' + fromPath + '`, but ';
  }

  function mixinProperties(to, from) {
    for (var key in from) {
      if (from.hasOwnProperty(key)) {
        to[key] = from[key];
      }
    }
  }

  mixinProperties(Binding, {

    /*
      See `Ember.Binding.from`.
       @method from
      @static
    */
    from: function (from) {
      var C = this;
      return new C(undefined, from);
    },

    /*
      See `Ember.Binding.to`.
       @method to
      @static
    */
    to: function (to) {
      var C = this;
      return new C(to, undefined);
    }
  });
  /**
    An `Ember.Binding` connects the properties of two objects so that whenever
    the value of one property changes, the other property will be changed also.
  
    ## Automatic Creation of Bindings with `/^*Binding/`-named Properties.
  
    You do not usually create Binding objects directly but instead describe
    bindings in your class or object definition using automatic binding
    detection.
  
    Properties ending in a `Binding` suffix will be converted to `Ember.Binding`
    instances. The value of this property should be a string representing a path
    to another object or a custom binding instance created using Binding helpers
    (see "One Way Bindings"):
  
    ```
    valueBinding: "MyApp.someController.title"
    ```
  
    This will create a binding from `MyApp.someController.title` to the `value`
    property of your object instance automatically. Now the two values will be
    kept in sync.
  
    ## One Way Bindings
  
    One especially useful binding customization you can use is the `oneWay()`
    helper. This helper tells Ember that you are only interested in
    receiving changes on the object you are binding from. For example, if you
    are binding to a preference and you want to be notified if the preference
    has changed, but your object will not be changing the preference itself, you
    could do:
  
    ```
    bigTitlesBinding: Ember.Binding.oneWay("MyApp.preferencesController.bigTitles")
    ```
  
    This way if the value of `MyApp.preferencesController.bigTitles` changes the
    `bigTitles` property of your object will change also. However, if you
    change the value of your `bigTitles` property, it will not update the
    `preferencesController`.
  
    One way bindings are almost twice as fast to setup and twice as fast to
    execute because the binding only has to worry about changes to one side.
  
    You should consider using one way bindings anytime you have an object that
    may be created frequently and you do not intend to change a property; only
    to monitor it for changes (such as in the example above).
  
    ## Adding Bindings Manually
  
    All of the examples above show you how to configure a custom binding, but the
    result of these customizations will be a binding template, not a fully active
    Binding instance. The binding will actually become active only when you
    instantiate the object the binding belongs to. It is useful, however, to
    understand what actually happens when the binding is activated.
  
    For a binding to function it must have at least a `from` property and a `to`
    property. The `from` property path points to the object/key that you want to
    bind from while the `to` path points to the object/key you want to bind to.
  
    When you define a custom binding, you are usually describing the property
    you want to bind from (such as `MyApp.someController.value` in the examples
    above). When your object is created, it will automatically assign the value
    you want to bind `to` based on the name of your binding key. In the
    examples above, during init, Ember objects will effectively call
    something like this on your binding:
  
    ```javascript
    binding = Ember.Binding.from("valueBinding").to("value");
    ```
  
    This creates a new binding instance based on the template you provide, and
    sets the to path to the `value` property of the new object. Now that the
    binding is fully configured with a `from` and a `to`, it simply needs to be
    connected to become active. This is done through the `connect()` method:
  
    ```javascript
    binding.connect(this);
    ```
  
    Note that when you connect a binding you pass the object you want it to be
    connected to. This object will be used as the root for both the from and
    to side of the binding when inspecting relative paths. This allows the
    binding to be automatically inherited by subclassed objects as well.
  
    This also allows you to bind between objects using the paths you declare in
    `from` and `to`:
  
    ```javascript
    // Example 1
    binding = Ember.Binding.from("App.someObject.value").to("value");
    binding.connect(this);
  
    // Example 2
    binding = Ember.Binding.from("parentView.value").to("App.someObject.value");
    binding.connect(this);
    ```
  
    Now that the binding is connected, it will observe both the from and to side
    and relay changes.
  
    If you ever needed to do so (you almost never will, but it is useful to
    understand this anyway), you could manually create an active binding by
    using the `Ember.bind()` helper method. (This is the same method used by
    to setup your bindings on objects):
  
    ```javascript
    Ember.bind(MyApp.anotherObject, "value", "MyApp.someController.value");
    ```
  
    Both of these code fragments have the same effect as doing the most friendly
    form of binding creation like so:
  
    ```javascript
    MyApp.anotherObject = Ember.Object.create({
      valueBinding: "MyApp.someController.value",
  
      // OTHER CODE FOR THIS OBJECT...
    });
    ```
  
    Ember's built in binding creation method makes it easy to automatically
    create bindings for you. You should always use the highest-level APIs
    available, even if you understand how it works underneath.
  
    @class Binding
    @namespace Ember
    @since Ember 0.9
    @public
  */
  // Ember.Binding = Binding; ES6TODO: where to put this?

  /**
    Global helper method to create a new binding. Just pass the root object
    along with a `to` and `from` path to create and connect the binding.
  
    @method bind
    @for Ember
    @param {Object} obj The root object of the transform.
    @param {String} to The path to the 'to' side of the binding.
      Must be relative to obj.
    @param {String} from The path to the 'from' side of the binding.
      Must be relative to obj or a global path.
    @return {Ember.Binding} binding instance
    @public
  */

  function bind(obj, to, from) {
    return new Binding(to, from).connect(obj);
  }

  exports.Binding = Binding;
});
enifed('ember-metal/cache', ['exports', 'ember-utils', 'ember-metal/meta'], function (exports, _emberUtils, _emberMetalMeta) {
  'use strict';

  var Cache = (function () {
    function Cache(limit, func, key, store) {
      this.size = 0;
      this.misses = 0;
      this.hits = 0;
      this.limit = limit;
      this.func = func;
      this.key = key;
      this.store = store || new DefaultStore();
    }

    Cache.prototype.get = function get(obj) {
      var key = this.key === undefined ? obj : this.key(obj);
      var value = this.store.get(key);
      if (value === undefined) {
        this.misses++;
        value = this._set(key, this.func(obj));
      } else if (value === _emberMetalMeta.UNDEFINED) {
        this.hits++;
        value = undefined;
      } else {
        this.hits++;
        // nothing to translate
      }

      return value;
    };

    Cache.prototype.set = function set(obj, value) {
      var key = this.key === undefined ? obj : this.key(obj);
      return this._set(key, value);
    };

    Cache.prototype._set = function _set(key, value) {
      if (this.limit > this.size) {
        this.size++;
        if (value === undefined) {
          this.store.set(key, _emberMetalMeta.UNDEFINED);
        } else {
          this.store.set(key, value);
        }
      }

      return value;
    };

    Cache.prototype.purge = function purge() {
      this.store.clear();
      this.size = 0;
      this.hits = 0;
      this.misses = 0;
    };

    return Cache;
  })();

  exports.default = Cache;

  var DefaultStore = (function () {
    function DefaultStore() {
      this.data = new _emberUtils.EmptyObject();
    }

    DefaultStore.prototype.get = function get(key) {
      return this.data[key];
    };

    DefaultStore.prototype.set = function set(key, value) {
      this.data[key] = value;
    };

    DefaultStore.prototype.clear = function clear() {
      this.data = new _emberUtils.EmptyObject();
    };

    return DefaultStore;
  })();
});
enifed('ember-metal/chains', ['exports', 'ember-utils', 'ember-metal/property_get', 'ember-metal/meta', 'ember-metal/watch_key', 'ember-metal/computed', 'ember-metal/watch_path'], function (exports, _emberUtils, _emberMetalProperty_get, _emberMetalMeta, _emberMetalWatch_key, _emberMetalComputed, _emberMetalWatch_path) {
  'use strict';

  exports.finishChains = finishChains;

  var FIRST_KEY = /^([^\.]+)/;

  function firstKey(path) {
    return path.match(FIRST_KEY)[0];
  }

  function isObject(obj) {
    return typeof obj === 'object' && obj;
  }

  function isVolatile(obj) {
    return !(isObject(obj) && obj.isDescriptor && obj._volatile === false);
  }

  function ChainWatchers() {
    // chain nodes that reference a key in this obj by key
    // we only create ChainWatchers when we are going to add them
    // so create this upfront
    this.chains = new _emberUtils.EmptyObject();
  }

  ChainWatchers.prototype = {
    add: function (key, node) {
      var nodes = this.chains[key];
      if (nodes === undefined) {
        this.chains[key] = [node];
      } else {
        nodes.push(node);
      }
    },

    remove: function (key, node) {
      var nodes = this.chains[key];
      if (nodes) {
        for (var i = 0; i < nodes.length; i++) {
          if (nodes[i] === node) {
            nodes.splice(i, 1);
            break;
          }
        }
      }
    },

    has: function (key, node) {
      var nodes = this.chains[key];
      if (nodes) {
        for (var i = 0; i < nodes.length; i++) {
          if (nodes[i] === node) {
            return true;
          }
        }
      }
      return false;
    },

    revalidateAll: function () {
      for (var key in this.chains) {
        this.notify(key, true, undefined);
      }
    },

    revalidate: function (key) {
      this.notify(key, true, undefined);
    },

    // key: the string key that is part of a path changed
    // revalidate: boolean; the chains that are watching this value should revalidate
    // callback: function that will be called with the object and path that
    //           will be/are invalidated by this key change, depending on
    //           whether the revalidate flag is passed
    notify: function (key, revalidate, callback) {
      var nodes = this.chains[key];
      if (nodes === undefined || nodes.length === 0) {
        return;
      }

      var affected = undefined;

      if (callback) {
        affected = [];
      }

      for (var i = 0; i < nodes.length; i++) {
        nodes[i].notify(revalidate, affected);
      }

      if (callback === undefined) {
        return;
      }

      // we gather callbacks so we don't notify them during revalidation
      for (var i = 0; i < affected.length; i += 2) {
        var obj = affected[i];
        var path = affected[i + 1];
        callback(obj, path);
      }
    }
  };

  function makeChainWatcher() {
    return new ChainWatchers();
  }

  function addChainWatcher(obj, keyName, node) {
    var m = _emberMetalMeta.meta(obj);
    m.writableChainWatchers(makeChainWatcher).add(keyName, node);
    _emberMetalWatch_key.watchKey(obj, keyName, m);
  }

  function removeChainWatcher(obj, keyName, node, _meta) {
    if (!isObject(obj)) {
      return;
    }

    var meta = _meta || _emberMetalMeta.peekMeta(obj);

    if (!meta || !meta.readableChainWatchers()) {
      return;
    }

    // make meta writable
    meta = _emberMetalMeta.meta(obj);

    meta.readableChainWatchers().remove(keyName, node);

    _emberMetalWatch_key.unwatchKey(obj, keyName, meta);
  }

  // A ChainNode watches a single key on an object. If you provide a starting
  // value for the key then the node won't actually watch it. For a root node
  // pass null for parent and key and object for value.
  function ChainNode(parent, key, value) {
    this._parent = parent;
    this._key = key;

    // _watching is true when calling get(this._parent, this._key) will
    // return the value of this node.
    //
    // It is false for the root of a chain (because we have no parent)
    // and for global paths (because the parent node is the object with
    // the observer on it)
    this._watching = value === undefined;

    this._chains = undefined;
    this._object = undefined;
    this.count = 0;

    this._value = value;
    this._paths = {};
    if (this._watching) {
      var obj = parent.value();

      if (!isObject(obj)) {
        return;
      }

      this._object = obj;

      addChainWatcher(this._object, this._key, this);
    }
  }

  function lazyGet(obj, key) {
    if (!isObject(obj)) {
      return;
    }

    var meta = _emberMetalMeta.peekMeta(obj);

    // check if object meant only to be a prototype
    if (meta && meta.proto === obj) {
      return;
    }

    // Use `get` if the return value is an EachProxy or an uncacheable value.
    if (isVolatile(obj[key])) {
      return _emberMetalProperty_get.get(obj, key);
      // Otherwise attempt to get the cached value of the computed property
    } else {
        var cache = meta.readableCache();
        if (cache) {
          return _emberMetalComputed.cacheFor.get(cache, key);
        }
      }
  }

  ChainNode.prototype = {
    value: function () {
      if (this._value === undefined && this._watching) {
        var obj = this._parent.value();
        this._value = lazyGet(obj, this._key);
      }
      return this._value;
    },

    destroy: function () {
      if (this._watching) {
        var obj = this._object;
        if (obj) {
          removeChainWatcher(obj, this._key, this);
        }
        this._watching = false; // so future calls do nothing
      }
    },

    // copies a top level object only
    copy: function (obj) {
      var ret = new ChainNode(null, null, obj);
      var paths = this._paths;
      var path = undefined;

      for (path in paths) {
        // this check will also catch non-number vals.
        if (paths[path] <= 0) {
          continue;
        }
        ret.add(path);
      }
      return ret;
    },

    // called on the root node of a chain to setup watchers on the specified
    // path.
    add: function (path) {
      var paths = this._paths;
      paths[path] = (paths[path] || 0) + 1;

      var key = firstKey(path);
      var tail = path.slice(key.length + 1);

      this.chain(key, tail);
    },

    // called on the root node of a chain to teardown watcher on the specified
    // path
    remove: function (path) {
      var paths = this._paths;
      if (paths[path] > 0) {
        paths[path]--;
      }

      var key = firstKey(path);
      var tail = path.slice(key.length + 1);

      this.unchain(key, tail);
    },

    chain: function (key, path) {
      var chains = this._chains;
      var node = undefined;
      if (chains === undefined) {
        chains = this._chains = new _emberUtils.EmptyObject();
      } else {
        node = chains[key];
      }

      if (node === undefined) {
        node = chains[key] = new ChainNode(this, key, undefined);
      }

      node.count++; // count chains...

      // chain rest of path if there is one
      if (path) {
        key = firstKey(path);
        path = path.slice(key.length + 1);
        node.chain(key, path);
      }
    },

    unchain: function (key, path) {
      var chains = this._chains;
      var node = chains[key];

      // unchain rest of path first...
      if (path && path.length > 1) {
        var nextKey = firstKey(path);
        var nextPath = path.slice(nextKey.length + 1);
        node.unchain(nextKey, nextPath);
      }

      // delete node if needed.
      node.count--;
      if (node.count <= 0) {
        chains[node._key] = undefined;
        node.destroy();
      }
    },

    notify: function (revalidate, affected) {
      if (revalidate && this._watching) {
        var parentValue = this._parent.value();

        if (parentValue !== this._object) {
          if (this._object) {
            removeChainWatcher(this._object, this._key, this);
          }

          if (isObject(parentValue)) {
            this._object = parentValue;
            addChainWatcher(parentValue, this._key, this);
          } else {
            this._object = undefined;
          }
        }
        this._value = undefined;
      }

      // then notify chains...
      var chains = this._chains;
      var node = undefined;
      if (chains) {
        for (var key in chains) {
          node = chains[key];
          if (node !== undefined) {
            node.notify(revalidate, affected);
          }
        }
      }

      if (affected && this._parent) {
        this._parent.populateAffected(this._key, 1, affected);
      }
    },

    populateAffected: function (path, depth, affected) {
      if (this._key) {
        path = this._key + '.' + path;
      }

      if (this._parent) {
        this._parent.populateAffected(path, depth + 1, affected);
      } else {
        if (depth > 1) {
          affected.push(this.value(), path);
        }
      }
    }
  };

  function finishChains(obj) {
    // We only create meta if we really have to
    var m = _emberMetalMeta.peekMeta(obj);
    if (m) {
      m = _emberMetalMeta.meta(obj);

      // finish any current chains node watchers that reference obj
      var chainWatchers = m.readableChainWatchers();
      if (chainWatchers) {
        chainWatchers.revalidateAll();
      }
      // ensure that if we have inherited any chains they have been
      // copied onto our own meta.
      if (m.readableChains()) {
        m.writableChains(_emberMetalWatch_path.makeChainNode);
      }
    }
  }

  exports.removeChainWatcher = removeChainWatcher;
  exports.ChainNode = ChainNode;
});
enifed('ember-metal/computed', ['exports', 'ember-utils', 'ember-metal/debug', 'ember-metal/property_set', 'ember-metal/meta', 'ember-metal/expand_properties', 'ember-metal/error', 'ember-metal/properties', 'ember-metal/property_events', 'ember-metal/dependent_keys'], function (exports, _emberUtils, _emberMetalDebug, _emberMetalProperty_set, _emberMetalMeta, _emberMetalExpand_properties, _emberMetalError, _emberMetalProperties, _emberMetalProperty_events, _emberMetalDependent_keys) {
  'use strict';

  exports.default = computed;

  /**
  @module ember
  @submodule ember-metal
  */

  var DEEP_EACH_REGEX = /\.@each\.[^.]+\./;

  /**
    A computed property transforms an object literal with object's accessor function(s) into a property.
  
    By default the function backing the computed property will only be called
    once and the result will be cached. You can specify various properties
    that your computed property depends on. This will force the cached
    result to be recomputed if the dependencies are modified.
  
    In the following example we declare a computed property - `fullName` - by calling
    `.Ember.computed()` with property dependencies (`firstName` and `lastName`) as leading arguments and getter accessor function. The `fullName` getter function
    will be called once (regardless of how many times it is accessed) as long
    as its dependencies have not changed. Once `firstName` or `lastName` are updated
    any future calls (or anything bound) to `fullName` will incorporate the new
    values.
  
    ```javascript
    let Person = Ember.Object.extend({
      // these will be supplied by `create`
      firstName: null,
      lastName: null,
  
      fullName: Ember.computed('firstName', 'lastName', function() {
        let firstName = this.get('firstName'),
            lastName  = this.get('lastName');
  
        return firstName + ' ' + lastName;
      })
    });
  
    let tom = Person.create({
      firstName: 'Tom',
      lastName: 'Dale'
    });
  
    tom.get('fullName') // 'Tom Dale'
    ```
  
    You can also define what Ember should do when setting a computed property by providing additional function (`set`) in hash argument.
    If you try to set a computed property, it will try to invoke setter accessor function with the key and
    value you want to set it to as arguments.
  
    ```javascript
    let Person = Ember.Object.extend({
      // these will be supplied by `create`
      firstName: null,
      lastName: null,
  
      fullName: Ember.computed('firstName', 'lastName', {
        get(key) {
          let firstName = this.get('firstName'),
              lastName  = this.get('lastName');
  
          return firstName + ' ' + lastName;
        },
        set(key, value) {
          let [firstName, lastName] = value.split(' ');
  
          this.set('firstName', firstName);
          this.set('lastName', lastName);
  
          return value;
        }
      })
    });
  
    let person = Person.create();
  
    person.set('fullName', 'Peter Wagenet');
    person.get('firstName'); // 'Peter'
    person.get('lastName');  // 'Wagenet'
    ```
  
    You can overwrite computed property with normal property (no longer computed), that won't change if dependencies change, if you set computed property and it won't have setter accessor function defined.
  
    You can also mark computed property as `.readOnly()` and block all attempts to set it.
  
    ```javascript
    let Person = Ember.Object.extend({
      // these will be supplied by `create`
      firstName: null,
      lastName: null,
  
      fullName: Ember.computed('firstName', 'lastName', {
        get(key) {
          let firstName = this.get('firstName');
          let lastName  = this.get('lastName');
  
          return firstName + ' ' + lastName;
        }
      }).readOnly()
    });
  
    let person = Person.create();
    person.set('fullName', 'Peter Wagenet'); // Uncaught Error: Cannot set read-only property "fullName" on object: <(...):emberXXX>
    ```
  
    Additional resources:
    - [New CP syntax RFC](https://github.com/emberjs/rfcs/blob/master/text/0011-improved-cp-syntax.md)
    - [New computed syntax explained in "Ember 1.12 released" ](http://emberjs.com/blog/2015/05/13/ember-1-12-released.html#toc_new-computed-syntax)
  
    @class ComputedProperty
    @namespace Ember
    @public
  */
  function ComputedProperty(config, opts) {
    this.isDescriptor = true;
    if (typeof config === 'function') {
      this._getter = config;
    } else {
      this._getter = config.get;
      this._setter = config.set;
    }

    this._dependentKeys = undefined;
    this._suspended = undefined;
    this._meta = undefined;
    this._volatile = false;
    this._dependentKeys = opts && opts.dependentKeys;
    this._readOnly = false;
  }

  ComputedProperty.prototype = new _emberMetalProperties.Descriptor();
  ComputedProperty.prototype.constructor = ComputedProperty;

  var ComputedPropertyPrototype = ComputedProperty.prototype;

  /**
    Call on a computed property to set it into non-cached mode. When in this
    mode the computed property will not automatically cache the return value.
  
    It also does not automatically fire any change events. You must manually notify
    any changes if you want to observe this property.
  
    Dependency keys have no effect on volatile properties as they are for cache
    invalidation and notification when cached value is invalidated.
  
    ```javascript
    let outsideService = Ember.Object.extend({
      value: Ember.computed(function() {
        return OutsideService.getValue();
      }).volatile()
    }).create();
    ```
  
    @method volatile
    @return {Ember.ComputedProperty} this
    @chainable
    @public
  */
  ComputedPropertyPrototype.volatile = function () {
    this._volatile = true;
    return this;
  };

  /**
    Call on a computed property to set it into read-only mode. When in this
    mode the computed property will throw an error when set.
  
    ```javascript
    let Person = Ember.Object.extend({
      guid: Ember.computed(function() {
        return 'guid-guid-guid';
      }).readOnly()
    });
  
    let person = Person.create();
  
    person.set('guid', 'new-guid'); // will throw an exception
    ```
  
    @method readOnly
    @return {Ember.ComputedProperty} this
    @chainable
    @public
  */
  ComputedPropertyPrototype.readOnly = function () {
    this._readOnly = true;

    return this;
  };

  /**
    Sets the dependent keys on this computed property. Pass any number of
    arguments containing key paths that this computed property depends on.
  
    ```javascript
    let President = Ember.Object.extend({
      fullName: Ember.computed(function() {
        return this.get('firstName') + ' ' + this.get('lastName');
  
        // Tell Ember that this computed property depends on firstName
        // and lastName
      }).property('firstName', 'lastName')
    });
  
    let president = President.create({
      firstName: 'Barack',
      lastName: 'Obama'
    });
  
    president.get('fullName'); // 'Barack Obama'
    ```
  
    @method property
    @param {String} path* zero or more property paths
    @return {Ember.ComputedProperty} this
    @chainable
    @public
  */
  ComputedPropertyPrototype.property = function () {
    var args = [];

    function addArg(property) {
      args.push(property);
    }

    for (var i = 0; i < arguments.length; i++) {
      _emberMetalExpand_properties.default(arguments[i], addArg);
    }

    this._dependentKeys = args;
    return this;
  };

  /**
    In some cases, you may want to annotate computed properties with additional
    metadata about how they function or what values they operate on. For example,
    computed property functions may close over variables that are then no longer
    available for introspection.
  
    You can pass a hash of these values to a computed property like this:
  
    ```
    person: Ember.computed(function() {
      let personId = this.get('personId');
      return App.Person.create({ id: personId });
    }).meta({ type: App.Person })
    ```
  
    The hash that you pass to the `meta()` function will be saved on the
    computed property descriptor under the `_meta` key. Ember runtime
    exposes a public API for retrieving these values from classes,
    via the `metaForProperty()` function.
  
    @method meta
    @param {Object} meta
    @chainable
    @public
  */
  ComputedPropertyPrototype.meta = function (meta) {
    if (arguments.length === 0) {
      return this._meta || {};
    } else {
      this._meta = meta;
      return this;
    }
  };

  // invalidate cache when CP key changes
  ComputedPropertyPrototype.didChange = function (obj, keyName) {
    // _suspended is set via a CP.set to ensure we don't clear
    // the cached value set by the setter
    if (this._volatile || this._suspended === obj) {
      return;
    }

    // don't create objects just to invalidate
    var meta = _emberMetalMeta.peekMeta(obj);
    if (!meta || meta.source !== obj) {
      return;
    }

    var cache = meta.readableCache();
    if (cache && cache[keyName] !== undefined) {
      cache[keyName] = undefined;
      _emberMetalDependent_keys.removeDependentKeys(this, obj, keyName, meta);
    }
  };

  ComputedPropertyPrototype.get = function (obj, keyName) {
    if (this._volatile) {
      return this._getter.call(obj, keyName);
    }

    var meta = _emberMetalMeta.meta(obj);
    var cache = meta.writableCache();

    var result = cache[keyName];
    if (result === _emberMetalMeta.UNDEFINED) {
      return undefined;
    } else if (result !== undefined) {
      return result;
    }

    var ret = this._getter.call(obj, keyName);
    if (ret === undefined) {
      cache[keyName] = _emberMetalMeta.UNDEFINED;
    } else {
      cache[keyName] = ret;
    }

    var chainWatchers = meta.readableChainWatchers();
    if (chainWatchers) {
      chainWatchers.revalidate(keyName);
    }
    _emberMetalDependent_keys.addDependentKeys(this, obj, keyName, meta);

    return ret;
  };

  ComputedPropertyPrototype.set = function computedPropertySetEntry(obj, keyName, value) {
    if (this._readOnly) {
      this._throwReadOnlyError(obj, keyName);
    }

    if (!this._setter) {
      return this.clobberSet(obj, keyName, value);
    }

    if (this._volatile) {
      return this.volatileSet(obj, keyName, value);
    }

    return this.setWithSuspend(obj, keyName, value);
  };

  ComputedPropertyPrototype._throwReadOnlyError = function computedPropertyThrowReadOnlyError(obj, keyName) {
    throw new _emberMetalError.default('Cannot set read-only property "' + keyName + '" on object: ' + _emberUtils.inspect(obj));
  };

  ComputedPropertyPrototype.clobberSet = function computedPropertyClobberSet(obj, keyName, value) {
    var cachedValue = cacheFor(obj, keyName);
    _emberMetalProperties.defineProperty(obj, keyName, null, cachedValue);
    _emberMetalProperty_set.set(obj, keyName, value);
    return value;
  };

  ComputedPropertyPrototype.volatileSet = function computedPropertyVolatileSet(obj, keyName, value) {
    return this._setter.call(obj, keyName, value);
  };

  ComputedPropertyPrototype.setWithSuspend = function computedPropertySetWithSuspend(obj, keyName, value) {
    var oldSuspended = this._suspended;
    this._suspended = obj;
    try {
      return this._set(obj, keyName, value);
    } finally {
      this._suspended = oldSuspended;
    }
  };

  ComputedPropertyPrototype._set = function computedPropertySet(obj, keyName, value) {
    // cache requires own meta
    var meta = _emberMetalMeta.meta(obj);
    // either there is a writable cache or we need one to update
    var cache = meta.writableCache();
    var hadCachedValue = false;
    var cachedValue = undefined;
    if (cache[keyName] !== undefined) {
      if (cache[keyName] !== _emberMetalMeta.UNDEFINED) {
        cachedValue = cache[keyName];
      }
      hadCachedValue = true;
    }

    var ret = this._setter.call(obj, keyName, value, cachedValue);

    // allows setter to return the same value that is cached already
    if (hadCachedValue && cachedValue === ret) {
      return ret;
    }

    _emberMetalProperty_events.propertyWillChange(obj, keyName);

    if (hadCachedValue) {
      cache[keyName] = undefined;
    }

    if (!hadCachedValue) {
      _emberMetalDependent_keys.addDependentKeys(this, obj, keyName, meta);
    }

    if (ret === undefined) {
      cache[keyName] = _emberMetalMeta.UNDEFINED;
    } else {
      cache[keyName] = ret;
    }

    _emberMetalProperty_events.propertyDidChange(obj, keyName);

    return ret;
  };

  /* called before property is overridden */
  ComputedPropertyPrototype.teardown = function (obj, keyName) {
    if (this._volatile) {
      return;
    }
    var meta = _emberMetalMeta.meta(obj);
    var cache = meta.readableCache();
    if (cache && cache[keyName] !== undefined) {
      _emberMetalDependent_keys.removeDependentKeys(this, obj, keyName, meta);
      cache[keyName] = undefined;
    }
  };

  /**
    This helper returns a new property descriptor that wraps the passed
    computed property function. You can use this helper to define properties
    with mixins or via `Ember.defineProperty()`.
  
    If you pass a function as an argument, it will be used as a getter. A computed
    property defined in this way might look like this:
  
    ```js
    let Person = Ember.Object.extend({
      init() {
        this._super(...arguments);
  
        this.firstName = 'Betty';
        this.lastName = 'Jones';
      },
  
      fullName: Ember.computed('firstName', 'lastName', function() {
        return `${this.get('firstName')} ${this.get('lastName')}`;
      })
    });
  
    let client = Person.create();
  
    client.get('fullName'); // 'Betty Jones'
  
    client.set('lastName', 'Fuller');
    client.get('fullName'); // 'Betty Fuller'
    ```
  
    You can pass a hash with two functions, `get` and `set`, as an
    argument to provide both a getter and setter:
  
    ```js
    let Person = Ember.Object.extend({
      init() {
        this._super(...arguments);
  
        this.firstName = 'Betty';
        this.lastName = 'Jones';
      },
  
      fullName: Ember.computed('firstName', 'lastName', {
        get(key) {
          return `${this.get('firstName')} ${this.get('lastName')}`;
        },
        set(key, value) {
          let [firstName, lastName] = value.split(/\s+/);
          this.setProperties({ firstName, lastName });
          return value;
        }
      });
    })
  
    let client = Person.create();
    client.get('firstName'); // 'Betty'
  
    client.set('fullName', 'Carroll Fuller');
    client.get('firstName'); // 'Carroll'
    ```
  
    The `set` function should accept two parameters, `key` and `value`. The value
    returned from `set` will be the new value of the property.
  
    _Note: This is the preferred way to define computed properties when writing third-party
    libraries that depend on or use Ember, since there is no guarantee that the user
    will have [prototype Extensions](http://emberjs.com/guides/configuring-ember/disabling-prototype-extensions/) enabled._
  
    The alternative syntax, with prototype extensions, might look like:
  
    ```js
    fullName: function() {
      return this.get('firstName') + ' ' + this.get('lastName');
    }.property('firstName', 'lastName')
    ```
  
    @class computed
    @namespace Ember
    @constructor
    @static
    @param {String} [dependentKeys*] Optional dependent keys that trigger this computed property.
    @param {Function} func The computed property function.
    @return {Ember.ComputedProperty} property descriptor instance
    @public
  */

  function computed(func) {
    var args = undefined;

    if (arguments.length > 1) {
      args = [].slice.call(arguments);
      func = args.pop();
    }

    var cp = new ComputedProperty(func);

    if (args) {
      cp.property.apply(cp, args);
    }

    return cp;
  }

  /**
    Returns the cached value for a property, if one exists.
    This can be useful for peeking at the value of a computed
    property that is generated lazily, without accidentally causing
    it to be created.
  
    @method cacheFor
    @for Ember
    @param {Object} obj the object whose property you want to check
    @param {String} key the name of the property whose cached value you want
      to return
    @return {Object} the cached value
    @public
  */
  function cacheFor(obj, key) {
    var meta = _emberMetalMeta.peekMeta(obj);
    var cache = meta && meta.source === obj && meta.readableCache();
    var ret = cache && cache[key];

    if (ret === _emberMetalMeta.UNDEFINED) {
      return undefined;
    }
    return ret;
  }

  cacheFor.set = function (cache, key, value) {
    if (value === undefined) {
      cache[key] = _emberMetalMeta.UNDEFINED;
    } else {
      cache[key] = value;
    }
  };

  cacheFor.get = function (cache, key) {
    var ret = cache[key];
    if (ret === _emberMetalMeta.UNDEFINED) {
      return undefined;
    }
    return ret;
  };

  cacheFor.remove = function (cache, key) {
    cache[key] = undefined;
  };

  exports.ComputedProperty = ComputedProperty;
  exports.computed = computed;
  exports.cacheFor = cacheFor;
});
enifed('ember-metal/core', ['exports', 'ember-environment'], function (exports, _emberEnvironment) {
  'use strict';

  /**
  @module ember
  @submodule ember-metal
  */

  /**
    This namespace contains all Ember methods and functions. Future versions of
    Ember may overwrite this namespace and therefore, you should avoid adding any
    new properties.
  
    At the heart of Ember is Ember-Runtime, a set of core functions that provide
    cross-platform compatibility and object property observing.  Ember-Runtime is
    small and performance-focused so you can use it alongside other
    cross-platform libraries such as jQuery. For more details, see
    [Ember-Runtime](http://emberjs.com/api/modules/ember-runtime.html).
  
    @class Ember
    @static
    @public
  */
  var Ember = typeof _emberEnvironment.context.imports.Ember === 'object' && _emberEnvironment.context.imports.Ember || {};

  // Make sure these are set whether Ember was already defined or not
  Ember.isNamespace = true;
  Ember.toString = function () {
    return 'Ember';
  };

  // ..........................................................
  // BOOTSTRAP
  //

  exports.default = Ember;
});
enifed("ember-metal/debug", ["exports"], function (exports) {
  "use strict";

  exports.getDebugFunction = getDebugFunction;
  exports.setDebugFunction = setDebugFunction;
  exports.assert = assert;
  exports.info = info;
  exports.warn = warn;
  exports.debug = debug;
  exports.deprecate = deprecate;
  exports.deprecateFunc = deprecateFunc;
  exports.runInDebug = runInDebug;
  exports.debugSeal = debugSeal;
  exports.debugFreeze = debugFreeze;
  var debugFunctions = {
    assert: function () {},
    info: function () {},
    warn: function () {},
    debug: function () {},
    deprecate: function () {},
    deprecateFunc: function () {
      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      return args[args.length - 1];
    },
    runInDebug: function () {},
    debugSeal: function () {},
    debugFreeze: function () {}
  };

  exports.debugFunctions = debugFunctions;

  function getDebugFunction(name) {
    return debugFunctions[name];
  }

  function setDebugFunction(name, fn) {
    debugFunctions[name] = fn;
  }

  function assert() {
    return debugFunctions.assert.apply(undefined, arguments);
  }

  function info() {
    return debugFunctions.info.apply(undefined, arguments);
  }

  function warn() {
    return debugFunctions.warn.apply(undefined, arguments);
  }

  function debug() {
    return debugFunctions.debug.apply(undefined, arguments);
  }

  function deprecate() {
    return debugFunctions.deprecate.apply(undefined, arguments);
  }

  function deprecateFunc() {
    return debugFunctions.deprecateFunc.apply(undefined, arguments);
  }

  function runInDebug() {
    return debugFunctions.runInDebug.apply(undefined, arguments);
  }

  function debugSeal() {
    return debugFunctions.debugSeal.apply(undefined, arguments);
  }

  function debugFreeze() {
    return debugFunctions.debugFreeze.apply(undefined, arguments);
  }
});
enifed('ember-metal/dependent_keys', ['exports', 'ember-metal/watching'], function (exports, _emberMetalWatching) {
  'no use strict';
  // Remove "use strict"; from transpiled module until
  // https://bugs.webkit.org/show_bug.cgi?id=138038 is fixed

  exports.addDependentKeys = addDependentKeys;
  exports.removeDependentKeys = removeDependentKeys;

  /**
  @module ember
  @submodule ember-metal
  */

  // ..........................................................
  // DEPENDENT KEYS
  //

  function addDependentKeys(desc, obj, keyName, meta) {
    // the descriptor has a list of dependent keys, so
    // add all of its dependent keys.
    var idx = undefined,
        depKey = undefined;
    var depKeys = desc._dependentKeys;
    if (!depKeys) {
      return;
    }

    for (idx = 0; idx < depKeys.length; idx++) {
      depKey = depKeys[idx];
      // Increment the number of times depKey depends on keyName.
      meta.writeDeps(depKey, keyName, (meta.peekDeps(depKey, keyName) || 0) + 1);
      // Watch the depKey
      _emberMetalWatching.watch(obj, depKey, meta);
    }
  }

  function removeDependentKeys(desc, obj, keyName, meta) {
    // the descriptor has a list of dependent keys, so
    // remove all of its dependent keys.
    var depKeys = desc._dependentKeys;
    if (!depKeys) {
      return;
    }

    for (var idx = 0; idx < depKeys.length; idx++) {
      var depKey = depKeys[idx];
      // Decrement the number of times depKey depends on keyName.
      meta.writeDeps(depKey, keyName, (meta.peekDeps(depKey, keyName) || 0) - 1);
      // Unwatch the depKey
      _emberMetalWatching.unwatch(obj, depKey, meta);
    }
  }
});
enifed('ember-metal/deprecate_property', ['exports', 'ember-metal/debug', 'ember-metal/property_get', 'ember-metal/property_set'], function (exports, _emberMetalDebug, _emberMetalProperty_get, _emberMetalProperty_set) {
  /**
  @module ember
  @submodule ember-metal
  */

  'use strict';

  exports.deprecateProperty = deprecateProperty;

  /**
    Used internally to allow changing properties in a backwards compatible way, and print a helpful
    deprecation warning.
  
    @method deprecateProperty
    @param {Object} object The object to add the deprecated property to.
    @param {String} deprecatedKey The property to add (and print deprecation warnings upon accessing).
    @param {String} newKey The property that will be aliased.
    @private
    @since 1.7.0
  */

  function deprecateProperty(object, deprecatedKey, newKey, options) {
    function _deprecate() {}

    Object.defineProperty(object, deprecatedKey, {
      configurable: true,
      enumerable: false,
      set: function (value) {
        _deprecate();
        _emberMetalProperty_set.set(this, newKey, value);
      },
      get: function () {
        _deprecate();
        return _emberMetalProperty_get.get(this, newKey);
      }
    });
  }
});
enifed('ember-metal/descriptor', ['exports', 'ember-metal/properties'], function (exports, _emberMetalProperties) {
  'use strict';

  exports.default = descriptor;

  function descriptor(desc) {
    return new Descriptor(desc);
  }

  /**
    A wrapper for a native ES5 descriptor. In an ideal world, we wouldn't need
    this at all, however, the way we currently flatten/merge our mixins require
    a special value to denote a descriptor.
  
    @class Descriptor
    @private
  */

  var Descriptor = (function (_EmberDescriptor) {
    babelHelpers.inherits(Descriptor, _EmberDescriptor);

    function Descriptor(desc) {
      _EmberDescriptor.call(this);
      this.desc = desc;
    }

    Descriptor.prototype.setup = function setup(obj, key) {
      Object.defineProperty(obj, key, this.desc);
    };

    Descriptor.prototype.teardown = function teardown(obj, key) {};

    return Descriptor;
  })(_emberMetalProperties.Descriptor);
});
enifed("ember-metal/error", ["exports"], function (exports) {

  /**
    A subclass of the JavaScript Error object for use in Ember.
  
    @class Error
    @namespace Ember
    @extends Error
    @constructor
    @public
  */
  "use strict";

  exports.default = EmberError;

  function EmberError(message) {
    if (!(this instanceof EmberError)) {
      return new EmberError(message);
    }

    var error = Error.call(this, message);

    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, EmberError);
    } else {
      this.stack = error.stack;
    }

    this.description = error.description;
    this.fileName = error.fileName;
    this.lineNumber = error.lineNumber;
    this.message = error.message;
    this.name = error.name;
    this.number = error.number;
    this.code = error.code;
  }

  EmberError.prototype = Object.create(Error.prototype);
});
enifed('ember-metal/error_handler', ['exports', 'ember-console', 'ember-metal/testing'], function (exports, _emberConsole, _emberMetalTesting) {
  'use strict';

  exports.getOnerror = getOnerror;
  exports.setOnerror = setOnerror;
  exports.dispatchError = dispatchError;
  exports.getDispatchOverride = getDispatchOverride;
  exports.setDispatchOverride = setDispatchOverride;

  // To maintain stacktrace consistency across browsers
  var getStack = function (error) {
    var stack = error.stack;
    var message = error.message;

    if (stack && stack.indexOf(message) === -1) {
      stack = message + '\n' + stack;
    }

    return stack;
  };

  var onerror = undefined;
  // Ember.onerror getter

  function getOnerror() {
    return onerror;
  }

  // Ember.onerror setter

  function setOnerror(handler) {
    onerror = handler;
  }

  var dispatchOverride = undefined;
  // dispatch error

  function dispatchError(error) {
    if (dispatchOverride) {
      dispatchOverride(error);
    } else {
      defaultDispatch(error);
    }
  }

  // allows testing adapter to override dispatch

  function getDispatchOverride() {
    return dispatchOverride;
  }

  function setDispatchOverride(handler) {
    dispatchOverride = handler;
  }

  function defaultDispatch(error) {
    if (_emberMetalTesting.isTesting()) {
      throw error;
    }
    if (onerror) {
      onerror(error);
    } else {
      _emberConsole.default.error(getStack(error));
    }
  }
});
enifed('ember-metal/events', ['exports', 'ember-utils', 'ember-metal/debug', 'ember-metal/meta', 'ember-metal/meta_listeners'], function (exports, _emberUtils, _emberMetalDebug, _emberMetalMeta, _emberMetalMeta_listeners) {
  'no use strict';
  // Remove "use strict"; from transpiled module until
  // https://bugs.webkit.org/show_bug.cgi?id=138038 is fixed

  /**
  @module ember
  @submodule ember-metal
  */
  exports.accumulateListeners = accumulateListeners;
  exports.addListener = addListener;
  exports.removeListener = removeListener;
  exports.suspendListener = suspendListener;
  exports.suspendListeners = suspendListeners;
  exports.watchedEvents = watchedEvents;
  exports.sendEvent = sendEvent;
  exports.hasListeners = hasListeners;
  exports.listenersFor = listenersFor;
  exports.on = on;

  /*
    The event system uses a series of nested hashes to store listeners on an
    object. When a listener is registered, or when an event arrives, these
    hashes are consulted to determine which target and action pair to invoke.
  
    The hashes are stored in the object's meta hash, and look like this:
  
        // Object's meta hash
        {
          listeners: {       // variable name: `listenerSet`
            "foo:changed": [ // variable name: `actions`
              target, method, flags
            ]
          }
        }
  
  */

  function indexOf(array, target, method) {
    var index = -1;
    // hashes are added to the end of the event array
    // so it makes sense to start searching at the end
    // of the array and search in reverse
    for (var i = array.length - 3; i >= 0; i -= 3) {
      if (target === array[i] && method === array[i + 1]) {
        index = i;
        break;
      }
    }
    return index;
  }

  function accumulateListeners(obj, eventName, otherActions) {
    var meta = _emberMetalMeta.peekMeta(obj);
    if (!meta) {
      return;
    }
    var actions = meta.matchingListeners(eventName);
    var newActions = [];

    for (var i = actions.length - 3; i >= 0; i -= 3) {
      var target = actions[i];
      var method = actions[i + 1];
      var flags = actions[i + 2];
      var actionIndex = indexOf(otherActions, target, method);

      if (actionIndex === -1) {
        otherActions.push(target, method, flags);
        newActions.push(target, method, flags);
      }
    }

    return newActions;
  }

  /**
    Add an event listener
  
    @method addListener
    @for Ember
    @param obj
    @param {String} eventName
    @param {Object|Function} target A target object or a function
    @param {Function|String} method A function or the name of a function to be called on `target`
    @param {Boolean} once A flag whether a function should only be called once
    @public
  */

  function addListener(obj, eventName, target, method, once) {

    if (!method && 'function' === typeof target) {
      method = target;
      target = null;
    }

    var flags = 0;
    if (once) {
      flags |= _emberMetalMeta_listeners.ONCE;
    }

    _emberMetalMeta.meta(obj).addToListeners(eventName, target, method, flags);

    if ('function' === typeof obj.didAddListener) {
      obj.didAddListener(eventName, target, method);
    }
  }

  /**
    Remove an event listener
  
    Arguments should match those passed to `Ember.addListener`.
  
    @method removeListener
    @for Ember
    @param obj
    @param {String} eventName
    @param {Object|Function} target A target object or a function
    @param {Function|String} method A function or the name of a function to be called on `target`
    @public
  */

  function removeListener(obj, eventName, target, method) {

    if (!method && 'function' === typeof target) {
      method = target;
      target = null;
    }

    _emberMetalMeta.meta(obj).removeFromListeners(eventName, target, method, function () {
      if ('function' === typeof obj.didRemoveListener) {
        obj.didRemoveListener.apply(obj, arguments);
      }
    });
  }

  /**
    Suspend listener during callback.
  
    This should only be used by the target of the event listener
    when it is taking an action that would cause the event, e.g.
    an object might suspend its property change listener while it is
    setting that property.
  
    @method suspendListener
    @for Ember
  
    @private
    @param obj
    @param {String} eventName
    @param {Object|Function} target A target object or a function
    @param {Function|String} method A function or the name of a function to be called on `target`
    @param {Function} callback
  */

  function suspendListener(obj, eventName, target, method, callback) {
    return suspendListeners(obj, [eventName], target, method, callback);
  }

  /**
    Suspends multiple listeners during a callback.
  
    @method suspendListeners
    @for Ember
  
    @private
    @param obj
    @param {Array} eventNames Array of event names
    @param {Object|Function} target A target object or a function
    @param {Function|String} method A function or the name of a function to be called on `target`
    @param {Function} callback
  */

  function suspendListeners(obj, eventNames, target, method, callback) {
    if (!method && 'function' === typeof target) {
      method = target;
      target = null;
    }
    return _emberMetalMeta.meta(obj).suspendListeners(eventNames, target, method, callback);
  }

  /**
    Return a list of currently watched events
  
    @private
    @method watchedEvents
    @for Ember
    @param obj
  */

  function watchedEvents(obj) {
    return _emberMetalMeta.meta(obj).watchedEvents();
  }

  /**
    Send an event. The execution of suspended listeners
    is skipped, and once listeners are removed. A listener without
    a target is executed on the passed object. If an array of actions
    is not passed, the actions stored on the passed object are invoked.
  
    @method sendEvent
    @for Ember
    @param obj
    @param {String} eventName
    @param {Array} params Optional parameters for each listener.
    @param {Array} actions Optional array of actions (listeners).
    @return true
    @public
  */

  function sendEvent(obj, eventName, params, actions) {
    if (!actions) {
      var meta = _emberMetalMeta.peekMeta(obj);
      actions = meta && meta.matchingListeners(eventName);
    }

    if (!actions || actions.length === 0) {
      return;
    }

    for (var i = actions.length - 3; i >= 0; i -= 3) {
      // looping in reverse for once listeners
      var target = actions[i];
      var method = actions[i + 1];
      var flags = actions[i + 2];

      if (!method) {
        continue;
      }
      if (flags & _emberMetalMeta_listeners.SUSPENDED) {
        continue;
      }
      if (flags & _emberMetalMeta_listeners.ONCE) {
        removeListener(obj, eventName, target, method);
      }
      if (!target) {
        target = obj;
      }
      if ('string' === typeof method) {
        if (params) {
          _emberUtils.applyStr(target, method, params);
        } else {
          target[method]();
        }
      } else {
        if (params) {
          method.apply(target, params);
        } else {
          method.call(target);
        }
      }
    }
    return true;
  }

  /**
    @private
    @method hasListeners
    @for Ember
    @param obj
    @param {String} eventName
  */

  function hasListeners(obj, eventName) {
    var meta = _emberMetalMeta.peekMeta(obj);
    if (!meta) {
      return false;
    }
    return meta.matchingListeners(eventName).length > 0;
  }

  /**
    @private
    @method listenersFor
    @for Ember
    @param obj
    @param {String} eventName
  */

  function listenersFor(obj, eventName) {
    var ret = [];
    var meta = _emberMetalMeta.peekMeta(obj);
    var actions = meta && meta.matchingListeners(eventName);

    if (!actions) {
      return ret;
    }

    for (var i = 0; i < actions.length; i += 3) {
      var target = actions[i];
      var method = actions[i + 1];
      ret.push([target, method]);
    }

    return ret;
  }

  /**
    Define a property as a function that should be executed when
    a specified event or events are triggered.
  
  
    ``` javascript
    let Job = Ember.Object.extend({
      logCompleted: Ember.on('completed', function() {
        console.log('Job completed!');
      })
    });
  
    let job = Job.create();
  
    Ember.sendEvent(job, 'completed'); // Logs 'Job completed!'
   ```
  
    @method on
    @for Ember
    @param {String} eventNames*
    @param {Function} func
    @return func
    @public
  */

  function on() {
    for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    var func = args.pop();
    var events = args;
    func.__ember_listens__ = events;
    return func;
  }
});
enifed('ember-metal/expand_properties', ['exports', 'ember-metal/debug'], function (exports, _emberMetalDebug) {
  'use strict';

  exports.default = expandProperties;

  /**
  @module ember
  @submodule ember-metal
  */

  var SPLIT_REGEX = /\{|\}/;
  var END_WITH_EACH_REGEX = /\.@each$/;

  /**
    Expands `pattern`, invoking `callback` for each expansion.
  
    The only pattern supported is brace-expansion, anything else will be passed
    once to `callback` directly.
  
    Example
  
    ```js
    function echo(arg){ console.log(arg); }
  
    Ember.expandProperties('foo.bar', echo);              //=> 'foo.bar'
    Ember.expandProperties('{foo,bar}', echo);            //=> 'foo', 'bar'
    Ember.expandProperties('foo.{bar,baz}', echo);        //=> 'foo.bar', 'foo.baz'
    Ember.expandProperties('{foo,bar}.baz', echo);        //=> 'foo.baz', 'bar.baz'
    Ember.expandProperties('foo.{bar,baz}.[]', echo)      //=> 'foo.bar.[]', 'foo.baz.[]'
    Ember.expandProperties('{foo,bar}.{spam,eggs}', echo) //=> 'foo.spam', 'foo.eggs', 'bar.spam', 'bar.eggs'
    Ember.expandProperties('{foo}.bar.{baz}')             //=> 'foo.bar.baz'
    ```
  
    @method expandProperties
    @for Ember
    @private
    @param {String} pattern The property pattern to expand.
    @param {Function} callback The callback to invoke.  It is invoked once per
    expansion, and is passed the expansion.
  */

  function expandProperties(pattern, callback) {

    var parts = pattern.split(SPLIT_REGEX);
    var properties = [parts];

    for (var i = 0; i < parts.length; i++) {
      var part = parts[i];
      if (part.indexOf(',') >= 0) {
        properties = duplicateAndReplace(properties, part.split(','), i);
      }
    }

    for (var i = 0; i < properties.length; i++) {
      callback(properties[i].join('').replace(END_WITH_EACH_REGEX, '.[]'));
    }
  }

  function duplicateAndReplace(properties, currentParts, index) {
    var all = [];

    properties.forEach(function (property) {
      currentParts.forEach(function (part) {
        var current = property.slice(0);
        current[index] = part;
        all.push(current);
      });
    });

    return all;
  }
});
enifed('ember-metal/features', ['exports', 'ember-utils', 'ember-environment', 'ember/features'], function (exports, _emberUtils, _emberEnvironment, _emberFeatures) {
  'use strict';

  exports.default = isEnabled;

  /**
    The hash of enabled Canary features. Add to this, any canary features
    before creating your application.
  
    Alternatively (and recommended), you can also define `EmberENV.FEATURES`
    if you need to enable features flagged at runtime.
  
    @class FEATURES
    @namespace Ember
    @static
    @since 1.1.0
    @public
  */
  var FEATURES = _emberUtils.assign(_emberFeatures.default, _emberEnvironment.ENV.FEATURES);

  exports.FEATURES = FEATURES;
  /**
    Determine whether the specified `feature` is enabled. Used by Ember's
    build tools to exclude experimental features from beta/stable builds.
  
    You can define the following configuration options:
  
    * `EmberENV.ENABLE_OPTIONAL_FEATURES` - enable any features that have not been explicitly
      enabled/disabled.
  
    @method isEnabled
    @param {String} feature The feature to check
    @return {Boolean}
    @for Ember.FEATURES
    @since 1.1.0
    @public
  */

  function isEnabled(feature) {
    var featureValue = FEATURES[feature];

    if (featureValue === true || featureValue === false || featureValue === undefined) {
      return featureValue;
    } else if (_emberEnvironment.ENV.ENABLE_OPTIONAL_FEATURES) {
      return true;
    } else {
      return false;
    }
  }

  exports.DEFAULT_FEATURES = _emberFeatures.default;
});
enifed('ember-metal/get_properties', ['exports', 'ember-metal/property_get'], function (exports, _emberMetalProperty_get) {
  'use strict';

  exports.default = getProperties;

  /**
    To get multiple properties at once, call `Ember.getProperties`
    with an object followed by a list of strings or an array:
  
    ```javascript
    Ember.getProperties(record, 'firstName', 'lastName', 'zipCode');
    // { firstName: 'John', lastName: 'Doe', zipCode: '10011' }
    ```
  
    is equivalent to:
  
    ```javascript
    Ember.getProperties(record, ['firstName', 'lastName', 'zipCode']);
    // { firstName: 'John', lastName: 'Doe', zipCode: '10011' }
    ```
  
    @method getProperties
    @for Ember
    @param {Object} obj
    @param {String...|Array} list of keys to get
    @return {Object}
    @public
  */

  function getProperties(obj) {
    var ret = {};
    var propertyNames = arguments;
    var i = 1;

    if (arguments.length === 2 && Array.isArray(arguments[1])) {
      i = 0;
      propertyNames = arguments[1];
    }
    for (; i < propertyNames.length; i++) {
      ret[propertyNames[i]] = _emberMetalProperty_get.get(obj, propertyNames[i]);
    }
    return ret;
  }
});
enifed('ember-metal/index', ['exports', 'require', 'ember-metal/core', 'ember-metal/computed', 'ember-metal/alias', 'ember-metal/merge', 'ember-metal/debug', 'ember-metal/instrumentation', 'ember-metal/testing', 'ember-metal/error_handler', 'ember-metal/meta', 'ember-metal/error', 'ember-metal/cache', 'ember-metal/features', 'ember-metal/property_get', 'ember-metal/property_set', 'ember-metal/weak_map', 'ember-metal/events', 'ember-metal/is_none', 'ember-metal/is_empty', 'ember-metal/is_blank', 'ember-metal/is_present', 'ember-metal/run_loop', 'ember-metal/observer_set', 'ember-metal/property_events', 'ember-metal/properties', 'ember-metal/watch_key', 'ember-metal/chains', 'ember-metal/watch_path', 'ember-metal/watching', 'ember-metal/libraries', 'ember-metal/map', 'ember-metal/get_properties', 'ember-metal/set_properties', 'ember-metal/expand_properties', 'ember-metal/observer', 'ember-metal/mixin', 'ember-metal/binding', 'ember-metal/path_cache', 'ember-metal/injected_property', 'ember-metal/tags', 'ember-metal/replace', 'ember-metal/transaction', 'ember-metal/is_proxy', 'ember-metal/descriptor'], function (exports, _require, _emberMetalCore, _emberMetalComputed, _emberMetalAlias, _emberMetalMerge, _emberMetalDebug, _emberMetalInstrumentation, _emberMetalTesting, _emberMetalError_handler, _emberMetalMeta, _emberMetalError, _emberMetalCache, _emberMetalFeatures, _emberMetalProperty_get, _emberMetalProperty_set, _emberMetalWeak_map, _emberMetalEvents, _emberMetalIs_none, _emberMetalIs_empty, _emberMetalIs_blank, _emberMetalIs_present, _emberMetalRun_loop, _emberMetalObserver_set, _emberMetalProperty_events, _emberMetalProperties, _emberMetalWatch_key, _emberMetalChains, _emberMetalWatch_path, _emberMetalWatching, _emberMetalLibraries, _emberMetalMap, _emberMetalGet_properties, _emberMetalSet_properties, _emberMetalExpand_properties, _emberMetalObserver, _emberMetalMixin, _emberMetalBinding, _emberMetalPath_cache, _emberMetalInjected_property, _emberMetalTags, _emberMetalReplace, _emberMetalTransaction, _emberMetalIs_proxy, _emberMetalDescriptor) {
  /**
  @module ember
  @submodule ember-metal
  */

  'use strict';

  exports.default = _emberMetalCore.default;
  // reexports
  exports.computed = _emberMetalComputed.default;
  exports.cacheFor = _emberMetalComputed.cacheFor;
  exports.ComputedProperty = _emberMetalComputed.ComputedProperty;
  exports.alias = _emberMetalAlias.default;
  exports.merge = _emberMetalMerge.default;
  exports.assert = _emberMetalDebug.assert;
  exports.info = _emberMetalDebug.info;
  exports.warn = _emberMetalDebug.warn;
  exports.debug = _emberMetalDebug.debug;
  exports.deprecate = _emberMetalDebug.deprecate;
  exports.deprecateFunc = _emberMetalDebug.deprecateFunc;
  exports.runInDebug = _emberMetalDebug.runInDebug;
  exports.setDebugFunction = _emberMetalDebug.setDebugFunction;
  exports.getDebugFunction = _emberMetalDebug.getDebugFunction;
  exports.debugSeal = _emberMetalDebug.debugSeal;
  exports.debugFreeze = _emberMetalDebug.debugFreeze;
  exports.instrument = _emberMetalInstrumentation.instrument;
  exports.flaggedInstrument = _emberMetalInstrumentation.flaggedInstrument;
  exports._instrumentStart = _emberMetalInstrumentation._instrumentStart;
  exports.instrumentationReset = _emberMetalInstrumentation.reset;
  exports.instrumentationSubscribe = _emberMetalInstrumentation.subscribe;
  exports.instrumentationUnsubscribe = _emberMetalInstrumentation.unsubscribe;
  exports.isTesting = _emberMetalTesting.isTesting;
  exports.setTesting = _emberMetalTesting.setTesting;
  exports.getOnerror = _emberMetalError_handler.getOnerror;
  exports.setOnerror = _emberMetalError_handler.setOnerror;
  exports.dispatchError = _emberMetalError_handler.dispatchError;
  exports.setDispatchOverride = _emberMetalError_handler.setDispatchOverride;
  exports.META_DESC = _emberMetalMeta.META_DESC;
  exports.meta = _emberMetalMeta.meta;
  exports.peekMeta = _emberMetalMeta.peekMeta;
  exports.Error = _emberMetalError.default;
  exports.Cache = _emberMetalCache.default;
  exports.isFeatureEnabled = _emberMetalFeatures.default;
  exports.FEATURES = _emberMetalFeatures.FEATURES;
  exports.DEFAULT_FEATURES = _emberMetalFeatures.DEFAULT_FEATURES;
  exports._getPath = _emberMetalProperty_get._getPath;
  exports.get = _emberMetalProperty_get.get;
  exports.getWithDefault = _emberMetalProperty_get.getWithDefault;
  exports.set = _emberMetalProperty_set.set;
  exports.trySet = _emberMetalProperty_set.trySet;
  exports.WeakMap = _emberMetalWeak_map.default;
  exports.accumulateListeners = _emberMetalEvents.accumulateListeners;
  exports.addListener = _emberMetalEvents.addListener;
  exports.hasListeners = _emberMetalEvents.hasListeners;
  exports.listenersFor = _emberMetalEvents.listenersFor;
  exports.on = _emberMetalEvents.on;
  exports.removeListener = _emberMetalEvents.removeListener;
  exports.sendEvent = _emberMetalEvents.sendEvent;
  exports.suspendListener = _emberMetalEvents.suspendListener;
  exports.suspendListeners = _emberMetalEvents.suspendListeners;
  exports.watchedEvents = _emberMetalEvents.watchedEvents;
  exports.isNone = _emberMetalIs_none.default;
  exports.isEmpty = _emberMetalIs_empty.default;
  exports.isBlank = _emberMetalIs_blank.default;
  exports.isPresent = _emberMetalIs_present.default;
  exports.run = _emberMetalRun_loop.default;
  exports.ObserverSet = _emberMetalObserver_set.default;
  exports.beginPropertyChanges = _emberMetalProperty_events.beginPropertyChanges;
  exports.changeProperties = _emberMetalProperty_events.changeProperties;
  exports.endPropertyChanges = _emberMetalProperty_events.endPropertyChanges;
  exports.overrideChains = _emberMetalProperty_events.overrideChains;
  exports.propertyDidChange = _emberMetalProperty_events.propertyDidChange;
  exports.propertyWillChange = _emberMetalProperty_events.propertyWillChange;
  exports.PROPERTY_DID_CHANGE = _emberMetalProperty_events.PROPERTY_DID_CHANGE;
  exports.defineProperty = _emberMetalProperties.defineProperty;
  exports.Descriptor = _emberMetalProperties.Descriptor;
  exports.watchKey = _emberMetalWatch_key.watchKey;
  exports.unwatchKey = _emberMetalWatch_key.unwatchKey;
  exports.ChainNode = _emberMetalChains.ChainNode;
  exports.finishChains = _emberMetalChains.finishChains;
  exports.removeChainWatcher = _emberMetalChains.removeChainWatcher;
  exports.watchPath = _emberMetalWatch_path.watchPath;
  exports.unwatchPath = _emberMetalWatch_path.unwatchPath;
  exports.destroy = _emberMetalWatching.destroy;
  exports.isWatching = _emberMetalWatching.isWatching;
  exports.unwatch = _emberMetalWatching.unwatch;
  exports.watch = _emberMetalWatching.watch;
  exports.watcherCount = _emberMetalWatching.watcherCount;
  exports.libraries = _emberMetalLibraries.default;
  exports.Map = _emberMetalMap.Map;
  exports.MapWithDefault = _emberMetalMap.MapWithDefault;
  exports.OrderedSet = _emberMetalMap.OrderedSet;
  exports.getProperties = _emberMetalGet_properties.default;
  exports.setProperties = _emberMetalSet_properties.default;
  exports.expandProperties = _emberMetalExpand_properties.default;
  exports._suspendObserver = _emberMetalObserver._suspendObserver;
  exports._suspendObservers = _emberMetalObserver._suspendObservers;
  exports.addObserver = _emberMetalObserver.addObserver;
  exports.observersFor = _emberMetalObserver.observersFor;
  exports.removeObserver = _emberMetalObserver.removeObserver;
  exports._addBeforeObserver = _emberMetalObserver._addBeforeObserver;
  exports._removeBeforeObserver = _emberMetalObserver._removeBeforeObserver;
  exports.Mixin = _emberMetalMixin.Mixin;
  exports.aliasMethod = _emberMetalMixin.aliasMethod;
  exports._immediateObserver = _emberMetalMixin._immediateObserver;
  exports._beforeObserver = _emberMetalMixin._beforeObserver;
  exports.mixin = _emberMetalMixin.mixin;
  exports.observer = _emberMetalMixin.observer;
  exports.required = _emberMetalMixin.required;
  exports.REQUIRED = _emberMetalMixin.REQUIRED;
  exports.hasUnprocessedMixins = _emberMetalMixin.hasUnprocessedMixins;
  exports.clearUnprocessedMixins = _emberMetalMixin.clearUnprocessedMixins;
  exports.detectBinding = _emberMetalMixin.detectBinding;
  exports.Binding = _emberMetalBinding.Binding;
  exports.bind = _emberMetalBinding.bind;
  exports.isGlobalPath = _emberMetalPath_cache.isGlobalPath;
  exports.InjectedProperty = _emberMetalInjected_property.default;
  exports.setHasViews = _emberMetalTags.setHasViews;
  exports.tagForProperty = _emberMetalTags.tagForProperty;
  exports.tagFor = _emberMetalTags.tagFor;
  exports.markObjectAsDirty = _emberMetalTags.markObjectAsDirty;
  exports.replace = _emberMetalReplace.default;
  exports.runInTransaction = _emberMetalTransaction.default;
  exports.didRender = _emberMetalTransaction.didRender;
  exports.assertNotRendered = _emberMetalTransaction.assertNotRendered;
  exports.isProxy = _emberMetalIs_proxy.isProxy;
  exports.descriptor = _emberMetalDescriptor.default;

  // TODO: this needs to be deleted once we refactor the build tooling
  // do this for side-effects of updating Ember.assert, warn, etc when
  // ember-debug is present
  // This needs to be called before any deprecateFunc

  if (_require.has('ember-debug')) {
    _require.default('ember-debug');
  }
});
enifed('ember-metal/injected_property', ['exports', 'ember-utils', 'ember-metal/debug', 'ember-metal/computed', 'ember-metal/alias', 'ember-metal/properties'], function (exports, _emberUtils, _emberMetalDebug, _emberMetalComputed, _emberMetalAlias, _emberMetalProperties) {
  'use strict';

  exports.default = InjectedProperty;

  /**
    Read-only property that returns the result of a container lookup.
  
    @class InjectedProperty
    @namespace Ember
    @constructor
    @param {String} type The container type the property will lookup
    @param {String} name (optional) The name the property will lookup, defaults
           to the property's name
    @private
  */

  function InjectedProperty(type, name) {
    this.type = type;
    this.name = name;

    this._super$Constructor(injectedPropertyGet);
    AliasedPropertyPrototype.oneWay.call(this);
  }

  function injectedPropertyGet(keyName) {
    var desc = this[keyName];
    var owner = _emberUtils.getOwner(this) || this.container; // fallback to `container` for backwards compat

    return owner.lookup(desc.type + ':' + (desc.name || keyName));
  }

  InjectedProperty.prototype = Object.create(_emberMetalProperties.Descriptor.prototype);

  var InjectedPropertyPrototype = InjectedProperty.prototype;
  var ComputedPropertyPrototype = _emberMetalComputed.ComputedProperty.prototype;
  var AliasedPropertyPrototype = _emberMetalAlias.AliasedProperty.prototype;

  InjectedPropertyPrototype._super$Constructor = _emberMetalComputed.ComputedProperty;

  InjectedPropertyPrototype.get = ComputedPropertyPrototype.get;
  InjectedPropertyPrototype.readOnly = ComputedPropertyPrototype.readOnly;
  InjectedPropertyPrototype.teardown = ComputedPropertyPrototype.teardown;
});
enifed('ember-metal/instrumentation', ['exports', 'ember-environment', 'ember-metal/features'], function (exports, _emberEnvironment, _emberMetalFeatures) {
  'use strict';

  exports.instrument = instrument;
  exports._instrumentStart = _instrumentStart;
  exports.subscribe = subscribe;
  exports.unsubscribe = unsubscribe;
  exports.reset = reset;

  /**
    The purpose of the Ember Instrumentation module is
    to provide efficient, general-purpose instrumentation
    for Ember.
  
    Subscribe to a listener by using `Ember.subscribe`:
  
    ```javascript
    Ember.subscribe("render", {
      before(name, timestamp, payload) {
  
      },
  
      after(name, timestamp, payload) {
  
      }
    });
    ```
  
    If you return a value from the `before` callback, that same
    value will be passed as a fourth parameter to the `after`
    callback.
  
    Instrument a block of code by using `Ember.instrument`:
  
    ```javascript
    Ember.instrument("render.handlebars", payload, function() {
      // rendering logic
    }, binding);
    ```
  
    Event names passed to `Ember.instrument` are namespaced
    by periods, from more general to more specific. Subscribers
    can listen for events by whatever level of granularity they
    are interested in.
  
    In the above example, the event is `render.handlebars`,
    and the subscriber listened for all events beginning with
    `render`. It would receive callbacks for events named
    `render`, `render.handlebars`, `render.container`, or
    even `render.handlebars.layout`.
  
    @class Instrumentation
    @namespace Ember
    @static
    @private
  */
  var subscribers = [];
  exports.subscribers = subscribers;
  var cache = {};

  function populateListeners(name) {
    var listeners = [];
    var subscriber = undefined;

    for (var i = 0; i < subscribers.length; i++) {
      subscriber = subscribers[i];
      if (subscriber.regex.test(name)) {
        listeners.push(subscriber.object);
      }
    }

    cache[name] = listeners;
    return listeners;
  }

  var time = (function () {
    var perf = 'undefined' !== typeof window ? window.performance || {} : {};
    var fn = perf.now || perf.mozNow || perf.webkitNow || perf.msNow || perf.oNow;
    // fn.bind will be available in all the browsers that support the advanced window.performance... ;-)
    return fn ? fn.bind(perf) : function () {
      return +new Date();
    };
  })();

  /**
    Notifies event's subscribers, calls `before` and `after` hooks.
  
    @method instrument
    @namespace Ember.Instrumentation
  
    @param {String} [name] Namespaced event name.
    @param {Object} _payload
    @param {Function} callback Function that you're instrumenting.
    @param {Object} binding Context that instrument function is called with.
    @private
  */

  function instrument(name, _payload, callback, binding) {
    if (arguments.length <= 3 && typeof _payload === 'function') {
      binding = callback;
      callback = _payload;
      _payload = undefined;
    }
    if (subscribers.length === 0) {
      return callback.call(binding);
    }
    var payload = _payload || {};
    var finalizer = _instrumentStart(name, function () {
      return payload;
    });

    if (finalizer) {
      return withFinalizer(callback, finalizer, payload, binding);
    } else {
      return callback.call(binding);
    }
  }

  var flaggedInstrument = undefined;
  if (false) {
    exports.flaggedInstrument = flaggedInstrument = instrument;
  } else {
    exports.flaggedInstrument = flaggedInstrument = function (name, payload, callback) {
      return callback();
    };
  }
  exports.flaggedInstrument = flaggedInstrument;

  function withFinalizer(callback, finalizer, payload, binding) {
    var result = undefined;
    try {
      result = callback.call(binding);
    } catch (e) {
      payload.exception = e;
      result = payload;
    } finally {
      finalizer();
      return result;
    }
  }

  function NOOP() {}

  // private for now

  function _instrumentStart(name, _payload, _payloadParam) {
    if (subscribers.length === 0) {
      return NOOP;
    }

    var listeners = cache[name];

    if (!listeners) {
      listeners = populateListeners(name);
    }

    if (listeners.length === 0) {
      return NOOP;
    }

    var payload = _payload(_payloadParam);

    var STRUCTURED_PROFILE = _emberEnvironment.ENV.STRUCTURED_PROFILE;
    var timeName = undefined;
    if (STRUCTURED_PROFILE) {
      timeName = name + ': ' + payload.object;
      console.time(timeName);
    }

    var beforeValues = new Array(listeners.length);
    var i = undefined,
        listener = undefined;
    var timestamp = time();
    for (i = 0; i < listeners.length; i++) {
      listener = listeners[i];
      beforeValues[i] = listener.before(name, timestamp, payload);
    }

    return function _instrumentEnd() {
      var i = undefined,
          listener = undefined;
      var timestamp = time();
      for (i = 0; i < listeners.length; i++) {
        listener = listeners[i];
        if (typeof listener.after === 'function') {
          listener.after(name, timestamp, payload, beforeValues[i]);
        }
      }

      if (STRUCTURED_PROFILE) {
        console.timeEnd(timeName);
      }
    };
  }

  /**
    Subscribes to a particular event or instrumented block of code.
  
    @method subscribe
    @namespace Ember.Instrumentation
  
    @param {String} [pattern] Namespaced event name.
    @param {Object} [object] Before and After hooks.
  
    @return {Subscriber}
    @private
  */

  function subscribe(pattern, object) {
    var paths = pattern.split('.');
    var path = undefined;
    var regex = [];

    for (var i = 0; i < paths.length; i++) {
      path = paths[i];
      if (path === '*') {
        regex.push('[^\\.]*');
      } else {
        regex.push(path);
      }
    }

    regex = regex.join('\\.');
    regex = regex + '(\\..*)?';

    var subscriber = {
      pattern: pattern,
      regex: new RegExp('^' + regex + '$'),
      object: object
    };

    subscribers.push(subscriber);
    cache = {};

    return subscriber;
  }

  /**
    Unsubscribes from a particular event or instrumented block of code.
  
    @method unsubscribe
    @namespace Ember.Instrumentation
  
    @param {Object} [subscriber]
    @private
  */

  function unsubscribe(subscriber) {
    var index = undefined;

    for (var i = 0; i < subscribers.length; i++) {
      if (subscribers[i] === subscriber) {
        index = i;
      }
    }

    subscribers.splice(index, 1);
    cache = {};
  }

  /**
    Resets `Ember.Instrumentation` by flushing list of subscribers.
  
    @method reset
    @namespace Ember.Instrumentation
    @private
  */

  function reset() {
    subscribers.length = 0;
    cache = {};
  }
});
enifed('ember-metal/is_blank', ['exports', 'ember-metal/is_empty'], function (exports, _emberMetalIs_empty) {
  'use strict';

  exports.default = isBlank;

  /**
    A value is blank if it is empty or a whitespace string.
  
    ```javascript
    Ember.isBlank();                // true
    Ember.isBlank(null);            // true
    Ember.isBlank(undefined);       // true
    Ember.isBlank('');              // true
    Ember.isBlank([]);              // true
    Ember.isBlank('\n\t');          // true
    Ember.isBlank('  ');            // true
    Ember.isBlank({});              // false
    Ember.isBlank('\n\t Hello');    // false
    Ember.isBlank('Hello world');   // false
    Ember.isBlank([1,2,3]);         // false
    ```
  
    @method isBlank
    @for Ember
    @param {Object} obj Value to test
    @return {Boolean}
    @since 1.5.0
    @public
  */

  function isBlank(obj) {
    return _emberMetalIs_empty.default(obj) || typeof obj === 'string' && obj.match(/\S/) === null;
  }
});
enifed('ember-metal/is_empty', ['exports', 'ember-metal/property_get', 'ember-metal/is_none'], function (exports, _emberMetalProperty_get, _emberMetalIs_none) {
  'use strict';

  exports.default = isEmpty;

  /**
    Verifies that a value is `null` or an empty string, empty array,
    or empty function.
  
    Constrains the rules on `Ember.isNone` by returning true for empty
    string and empty arrays.
  
    ```javascript
    Ember.isEmpty();                // true
    Ember.isEmpty(null);            // true
    Ember.isEmpty(undefined);       // true
    Ember.isEmpty('');              // true
    Ember.isEmpty([]);              // true
    Ember.isEmpty({});              // false
    Ember.isEmpty('Adam Hawkins');  // false
    Ember.isEmpty([0,1,2]);         // false
    Ember.isEmpty('\n\t');          // false
    Ember.isEmpty('  ');            // false
    ```
  
    @method isEmpty
    @for Ember
    @param {Object} obj Value to test
    @return {Boolean}
    @public
  */

  function isEmpty(obj) {
    var none = _emberMetalIs_none.default(obj);
    if (none) {
      return none;
    }

    if (typeof obj.size === 'number') {
      return !obj.size;
    }

    var objectType = typeof obj;

    if (objectType === 'object') {
      var size = _emberMetalProperty_get.get(obj, 'size');
      if (typeof size === 'number') {
        return !size;
      }
    }

    if (typeof obj.length === 'number' && objectType !== 'function') {
      return !obj.length;
    }

    if (objectType === 'object') {
      var _length = _emberMetalProperty_get.get(obj, 'length');
      if (typeof _length === 'number') {
        return !_length;
      }
    }

    return false;
  }
});
enifed("ember-metal/is_none", ["exports"], function (exports) {
  /**
    Returns true if the passed value is null or undefined. This avoids errors
    from JSLint complaining about use of ==, which can be technically
    confusing.
  
    ```javascript
    Ember.isNone();              // true
    Ember.isNone(null);          // true
    Ember.isNone(undefined);     // true
    Ember.isNone('');            // false
    Ember.isNone([]);            // false
    Ember.isNone(function() {}); // false
    ```
  
    @method isNone
    @for Ember
    @param {Object} obj Value to test
    @return {Boolean}
    @public
  */
  "use strict";

  exports.default = isNone;

  function isNone(obj) {
    return obj === null || obj === undefined;
  }
});
enifed('ember-metal/is_present', ['exports', 'ember-metal/is_blank'], function (exports, _emberMetalIs_blank) {
  'use strict';

  exports.default = isPresent;

  /**
    A value is present if it not `isBlank`.
  
    ```javascript
    Ember.isPresent();                // false
    Ember.isPresent(null);            // false
    Ember.isPresent(undefined);       // false
    Ember.isPresent('');              // false
    Ember.isPresent('  ');            // false
    Ember.isPresent('\n\t');          // false
    Ember.isPresent([]);              // false
    Ember.isPresent({ length: 0 })    // false
    Ember.isPresent(false);           // true
    Ember.isPresent(true);            // true
    Ember.isPresent('string');        // true
    Ember.isPresent(0);               // true
    Ember.isPresent(function() {})    // true
    Ember.isPresent({});              // true
    Ember.isPresent(false);           // true
    Ember.isPresent('\n\t Hello');    // true
    Ember.isPresent([1,2,3]);         // true
    ```
  
    @method isPresent
    @for Ember
    @param {Object} obj Value to test
    @return {Boolean}
    @since 1.8.0
    @public
  */

  function isPresent(obj) {
    return !_emberMetalIs_blank.default(obj);
  }
});
enifed('ember-metal/is_proxy', ['exports', 'ember-metal/meta'], function (exports, _emberMetalMeta) {
  'use strict';

  exports.isProxy = isProxy;

  function isProxy(value) {
    if (typeof value === 'object' && value) {
      var meta = _emberMetalMeta.peekMeta(value);
      return meta && meta.isProxy();
    }

    return false;
  }
});
enifed('ember-metal/libraries', ['exports', 'ember-metal/debug', 'ember-metal/features'], function (exports, _emberMetalDebug, _emberMetalFeatures) {
  'use strict';

  exports.Libraries = Libraries;

  /**
    Helper class that allows you to register your library with Ember.
  
    Singleton created at `Ember.libraries`.
  
    @class Libraries
    @constructor
    @private
  */

  function Libraries() {
    this._registry = [];
    this._coreLibIndex = 0;
  }

  Libraries.prototype = {
    constructor: Libraries,

    _getLibraryByName: function (name) {
      var libs = this._registry;
      var count = libs.length;

      for (var i = 0; i < count; i++) {
        if (libs[i].name === name) {
          return libs[i];
        }
      }
    },

    register: function (name, version, isCoreLibrary) {
      var index = this._registry.length;

      if (!this._getLibraryByName(name)) {
        if (isCoreLibrary) {
          index = this._coreLibIndex++;
        }
        this._registry.splice(index, 0, { name: name, version: version });
      } else {}
    },

    registerCoreLibrary: function (name, version) {
      this.register(name, version, true);
    },

    deRegister: function (name) {
      var lib = this._getLibraryByName(name);
      var index = undefined;

      if (lib) {
        index = this._registry.indexOf(lib);
        this._registry.splice(index, 1);
      }
    }
  };

  if (false) {
    Libraries.prototype.isRegistered = function (name) {
      return !!this._getLibraryByName(name);
    };
  }

  exports.default = new Libraries();
});
enifed('ember-metal/map', ['exports', 'ember-utils'], function (exports, _emberUtils) {
  /**
  @module ember
  @submodule ember-metal
  */

  /*
    JavaScript (before ES6) does not have a Map implementation. Objects,
    which are often used as dictionaries, may only have Strings as keys.
  
    Because Ember has a way to get a unique identifier for every object
    via `Ember.guidFor`, we can implement a performant Map with arbitrary
    keys. Because it is commonly used in low-level bookkeeping, Map is
    implemented as a pure JavaScript object for performance.
  
    This implementation follows the current iteration of the ES6 proposal for
    maps (http://wiki.ecmascript.org/doku.php?id=harmony:simple_maps_and_sets),
    with one exception:  as we do not have the luxury of in-VM iteration, we implement a
    forEach method for iteration.
  
    Map is mocked out to look like an Ember object, so you can do
    `Ember.Map.create()` for symmetry with other Ember classes.
  */
  'use strict';

  function missingFunction(fn) {
    throw new TypeError(Object.prototype.toString.call(fn) + ' is not a function');
  }

  function missingNew(name) {
    throw new TypeError('Constructor ' + name + ' requires \'new\'');
  }

  function copyNull(obj) {
    var output = new _emberUtils.EmptyObject();

    for (var prop in obj) {
      // hasOwnPropery is not needed because obj is new EmptyObject();
      output[prop] = obj[prop];
    }

    return output;
  }

  function copyMap(original, newObject) {
    var keys = original._keys.copy();
    var values = copyNull(original._values);

    newObject._keys = keys;
    newObject._values = values;
    newObject.size = original.size;

    return newObject;
  }

  /**
    This class is used internally by Ember and Ember Data.
    Please do not use it at this time. We plan to clean it up
    and add many tests soon.
  
    @class OrderedSet
    @namespace Ember
    @constructor
    @private
  */
  function OrderedSet() {
    if (this instanceof OrderedSet) {
      this.clear();
      this._silenceRemoveDeprecation = false;
    } else {
      missingNew('OrderedSet');
    }
  }

  /**
    @method create
    @static
    @return {Ember.OrderedSet}
    @private
  */
  OrderedSet.create = function () {
    var Constructor = this;

    return new Constructor();
  };

  OrderedSet.prototype = {
    constructor: OrderedSet,
    /**
      @method clear
      @private
    */
    clear: function () {
      this.presenceSet = new _emberUtils.EmptyObject();
      this.list = [];
      this.size = 0;
    },

    /**
      @method add
      @param obj
      @param guid (optional, and for internal use)
      @return {Ember.OrderedSet}
      @private
    */
    add: function (obj, _guid) {
      var guid = _guid || _emberUtils.guidFor(obj);
      var presenceSet = this.presenceSet;
      var list = this.list;

      if (presenceSet[guid] !== true) {
        presenceSet[guid] = true;
        this.size = list.push(obj);
      }

      return this;
    },

    /**
      @since 1.8.0
      @method delete
      @param obj
      @param _guid (optional and for internal use only)
      @return {Boolean}
      @private
    */
    delete: function (obj, _guid) {
      var guid = _guid || _emberUtils.guidFor(obj);
      var presenceSet = this.presenceSet;
      var list = this.list;

      if (presenceSet[guid] === true) {
        delete presenceSet[guid];
        var index = list.indexOf(obj);
        if (index > -1) {
          list.splice(index, 1);
        }
        this.size = list.length;
        return true;
      } else {
        return false;
      }
    },

    /**
      @method isEmpty
      @return {Boolean}
      @private
    */
    isEmpty: function () {
      return this.size === 0;
    },

    /**
      @method has
      @param obj
      @return {Boolean}
      @private
    */
    has: function (obj) {
      if (this.size === 0) {
        return false;
      }

      var guid = _emberUtils.guidFor(obj);
      var presenceSet = this.presenceSet;

      return presenceSet[guid] === true;
    },

    /**
      @method forEach
      @param {Function} fn
      @param self
      @private
    */
    forEach: function (fn /*, ...thisArg*/) {
      if (typeof fn !== 'function') {
        missingFunction(fn);
      }

      if (this.size === 0) {
        return;
      }

      var list = this.list;

      if (arguments.length === 2) {
        for (var i = 0; i < list.length; i++) {
          fn.call(arguments[1], list[i]);
        }
      } else {
        for (var i = 0; i < list.length; i++) {
          fn(list[i]);
        }
      }
    },

    /**
      @method toArray
      @return {Array}
      @private
    */
    toArray: function () {
      return this.list.slice();
    },

    /**
      @method copy
      @return {Ember.OrderedSet}
      @private
    */
    copy: function () {
      var Constructor = this.constructor;
      var set = new Constructor();

      set._silenceRemoveDeprecation = this._silenceRemoveDeprecation;
      set.presenceSet = copyNull(this.presenceSet);
      set.list = this.toArray();
      set.size = this.size;

      return set;
    }
  };

  /**
    A Map stores values indexed by keys. Unlike JavaScript's
    default Objects, the keys of a Map can be any JavaScript
    object.
  
    Internally, a Map has two data structures:
  
    1. `keys`: an OrderedSet of all of the existing keys
    2. `values`: a JavaScript Object indexed by the `Ember.guidFor(key)`
  
    When a key/value pair is added for the first time, we
    add the key to the `keys` OrderedSet, and create or
    replace an entry in `values`. When an entry is deleted,
    we delete its entry in `keys` and `values`.
  
    @class Map
    @namespace Ember
    @private
    @constructor
  */
  function Map() {
    if (this instanceof Map) {
      this._keys = OrderedSet.create();
      this._keys._silenceRemoveDeprecation = true;
      this._values = new _emberUtils.EmptyObject();
      this.size = 0;
    } else {
      missingNew('Map');
    }
  }

  /**
    @method create
    @static
    @private
  */
  Map.create = function () {
    var Constructor = this;
    return new Constructor();
  };

  Map.prototype = {
    constructor: Map,

    /**
      This property will change as the number of objects in the map changes.
       @since 1.8.0
      @property size
      @type number
      @default 0
      @private
    */
    size: 0,

    /**
      Retrieve the value associated with a given key.
       @method get
      @param {*} key
      @return {*} the value associated with the key, or `undefined`
      @private
    */
    get: function (key) {
      if (this.size === 0) {
        return;
      }

      var values = this._values;
      var guid = _emberUtils.guidFor(key);

      return values[guid];
    },

    /**
      Adds a value to the map. If a value for the given key has already been
      provided, the new value will replace the old value.
       @method set
      @param {*} key
      @param {*} value
      @return {Ember.Map}
      @private
    */
    set: function (key, value) {
      var keys = this._keys;
      var values = this._values;
      var guid = _emberUtils.guidFor(key);

      // ensure we don't store -0
      var k = key === -0 ? 0 : key;

      keys.add(k, guid);

      values[guid] = value;

      this.size = keys.size;

      return this;
    },

    /**
      Removes a value from the map for an associated key.
       @since 1.8.0
      @method delete
      @param {*} key
      @return {Boolean} true if an item was removed, false otherwise
      @private
    */
    delete: function (key) {
      if (this.size === 0) {
        return false;
      }
      // don't use ES6 "delete" because it will be annoying
      // to use in browsers that are not ES6 friendly;
      var keys = this._keys;
      var values = this._values;
      var guid = _emberUtils.guidFor(key);

      if (keys.delete(key, guid)) {
        delete values[guid];
        this.size = keys.size;
        return true;
      } else {
        return false;
      }
    },

    /**
      Check whether a key is present.
       @method has
      @param {*} key
      @return {Boolean} true if the item was present, false otherwise
      @private
    */
    has: function (key) {
      return this._keys.has(key);
    },

    /**
      Iterate over all the keys and values. Calls the function once
      for each key, passing in value, key, and the map being iterated over,
      in that order.
       The keys are guaranteed to be iterated over in insertion order.
       @method forEach
      @param {Function} callback
      @param {*} self if passed, the `this` value inside the
        callback. By default, `this` is the map.
      @private
    */
    forEach: function (callback /*, ...thisArg*/) {
      if (typeof callback !== 'function') {
        missingFunction(callback);
      }

      if (this.size === 0) {
        return;
      }

      var map = this;
      var cb = undefined,
          thisArg = undefined;

      if (arguments.length === 2) {
        thisArg = arguments[1];
        cb = function (key) {
          return callback.call(thisArg, map.get(key), key, map);
        };
      } else {
        cb = function (key) {
          return callback(map.get(key), key, map);
        };
      }

      this._keys.forEach(cb);
    },

    /**
      @method clear
      @private
    */
    clear: function () {
      this._keys.clear();
      this._values = new _emberUtils.EmptyObject();
      this.size = 0;
    },

    /**
      @method copy
      @return {Ember.Map}
      @private
    */
    copy: function () {
      return copyMap(this, new Map());
    }
  };

  /**
    @class MapWithDefault
    @namespace Ember
    @extends Ember.Map
    @private
    @constructor
    @param [options]
      @param {*} [options.defaultValue]
  */
  function MapWithDefault(options) {
    this._super$constructor();
    this.defaultValue = options.defaultValue;
  }

  /**
    @method create
    @static
    @param [options]
      @param {*} [options.defaultValue]
    @return {Ember.MapWithDefault|Ember.Map} If options are passed, returns
      `Ember.MapWithDefault` otherwise returns `Ember.Map`
    @private
  */
  MapWithDefault.create = function (options) {
    if (options) {
      return new MapWithDefault(options);
    } else {
      return new Map();
    }
  };

  MapWithDefault.prototype = Object.create(Map.prototype);
  MapWithDefault.prototype.constructor = MapWithDefault;
  MapWithDefault.prototype._super$constructor = Map;
  MapWithDefault.prototype._super$get = Map.prototype.get;

  /**
    Retrieve the value associated with a given key.
  
    @method get
    @param {*} key
    @return {*} the value associated with the key, or the default value
    @private
  */
  MapWithDefault.prototype.get = function (key) {
    var hasValue = this.has(key);

    if (hasValue) {
      return this._super$get(key);
    } else {
      var defaultValue = this.defaultValue(key);
      this.set(key, defaultValue);
      return defaultValue;
    }
  };

  /**
    @method copy
    @return {Ember.MapWithDefault}
    @private
  */
  MapWithDefault.prototype.copy = function () {
    var Constructor = this.constructor;
    return copyMap(this, new Constructor({
      defaultValue: this.defaultValue
    }));
  };

  exports.default = Map;
  exports.OrderedSet = OrderedSet;
  exports.Map = Map;
  exports.MapWithDefault = MapWithDefault;
});
enifed('ember-metal/merge', ['exports'], function (exports) {
  /**
    Merge the contents of two objects together into the first object.
  
    ```javascript
    Ember.merge({ first: 'Tom' }, { last: 'Dale' }); // { first: 'Tom', last: 'Dale' }
    var a = { first: 'Yehuda' };
    var b = { last: 'Katz' };
    Ember.merge(a, b); // a == { first: 'Yehuda', last: 'Katz' }, b == { last: 'Katz' }
    ```
  
    @method merge
    @for Ember
    @param {Object} original The object to merge into
    @param {Object} updates The object to copy properties from
    @return {Object}
    @public
  */
  'use strict';

  exports.default = merge;

  function merge(original, updates) {
    if (!updates || typeof updates !== 'object') {
      return original;
    }

    var props = Object.keys(updates);
    var prop = undefined;

    for (var i = 0; i < props.length; i++) {
      prop = props[i];
      original[prop] = updates[prop];
    }

    return original;
  }
});
enifed('ember-metal/meta', ['exports', 'ember-utils', 'ember-metal/features', 'ember-metal/meta_listeners', 'ember-metal/debug', 'ember-metal/chains', 'require'], function (exports, _emberUtils, _emberMetalFeatures, _emberMetalMeta_listeners, _emberMetalDebug, _emberMetalChains, _require) {
  'no use strict';
  // Remove "use strict"; from transpiled module until
  // https://bugs.webkit.org/show_bug.cgi?id=138038 is fixed

  exports.Meta = Meta;
  exports.deleteMeta = deleteMeta;
  exports.meta = meta;

  var counters = {
    peekCalls: 0,
    peekParentCalls: 0,
    peekPrototypeWalks: 0,
    setCalls: 0,
    deleteCalls: 0,
    metaCalls: 0,
    metaInstantiated: 0
  };

  /**
  @module ember-metal
  */

  /*
   This declares several meta-programmed members on the Meta class. Such
   meta!
  
   In general, the `readable` variants will give you an object (if it
   already exists) that you can read but should not modify. The
   `writable` variants will give you a mutable object, and they will
   create it if it didn't already exist.
  
   The following methods will get generated metaprogrammatically, and
   I'm including them here for greppability:
  
   writableCache, readableCache, writeWatching,
   peekWatching, clearWatching, writeMixins,
   peekMixins, clearMixins, writeBindings,
   peekBindings, clearBindings, writeValues,
   peekValues, clearValues, writeDeps, forEachInDeps
   writableChainWatchers, readableChainWatchers, writableChains,
   readableChains, writableTag, readableTag, writableTags,
   readableTags
  */
  var members = {
    cache: ownMap,
    weak: ownMap,
    watching: inheritedMap,
    mixins: inheritedMap,
    bindings: inheritedMap,
    values: inheritedMap,
    chainWatchers: ownCustomObject,
    chains: inheritedCustomObject,
    tag: ownCustomObject,
    tags: ownMap
  };

  // FLAGS
  var SOURCE_DESTROYING = 1 << 1;
  var SOURCE_DESTROYED = 1 << 2;
  var META_DESTROYED = 1 << 3;
  var IS_PROXY = 1 << 4;

  if (false || false) {
    members.lastRendered = ownMap;
    if (_require.has('ember-debug')) {
      //https://github.com/emberjs/ember.js/issues/14732
      members.lastRenderedReferenceMap = ownMap;
      members.lastRenderedTemplateMap = ownMap;
    }
  }

  var memberNames = Object.keys(members);
  var META_FIELD = '__ember_meta__';

  function Meta(obj, parentMeta) {

    this._cache = undefined;
    this._weak = undefined;
    this._watching = undefined;
    this._mixins = undefined;
    this._bindings = undefined;
    this._values = undefined;
    this._deps = undefined;
    this._chainWatchers = undefined;
    this._chains = undefined;
    this._tag = undefined;
    this._tags = undefined;

    // initial value for all flags right now is false
    // see FLAGS const for detailed list of flags used
    this._flags = 0;

    // used only internally
    this.source = obj;

    // when meta(obj).proto === obj, the object is intended to be only a
    // prototype and doesn't need to actually be observable itself
    this.proto = undefined;

    // The next meta in our inheritance chain. We (will) track this
    // explicitly instead of using prototypical inheritance because we
    // have detailed knowledge of how each property should really be
    // inherited, and we can optimize it much better than JS runtimes.
    this.parent = parentMeta;

    if (false || false) {
      this._lastRendered = undefined;
    }

    this._initializeListeners();
  }

  Meta.prototype.isInitialized = function (obj) {
    return this.proto !== obj;
  };

  var NODE_STACK = [];

  Meta.prototype.destroy = function () {
    if (this.isMetaDestroyed()) {
      return;
    }

    // remove chainWatchers to remove circular references that would prevent GC
    var node = undefined,
        nodes = undefined,
        key = undefined,
        nodeObject = undefined;
    node = this.readableChains();
    if (node) {
      NODE_STACK.push(node);
      // process tree
      while (NODE_STACK.length > 0) {
        node = NODE_STACK.pop();
        // push children
        nodes = node._chains;
        if (nodes) {
          for (key in nodes) {
            if (nodes[key] !== undefined) {
              NODE_STACK.push(nodes[key]);
            }
          }
        }

        // remove chainWatcher in node object
        if (node._watching) {
          nodeObject = node._object;
          if (nodeObject) {
            var foreignMeta = peekMeta(nodeObject);
            // avoid cleaning up chain watchers when both current and
            // foreign objects are being destroyed
            // if both are being destroyed manual cleanup is not needed
            // as they will be GC'ed and no non-destroyed references will
            // be remaining
            if (foreignMeta && !foreignMeta.isSourceDestroying()) {
              _emberMetalChains.removeChainWatcher(nodeObject, node._key, node, foreignMeta);
            }
          }
        }
      }
    }

    this.setMetaDestroyed();
  };

  for (var _name in _emberMetalMeta_listeners.protoMethods) {
    Meta.prototype[_name] = _emberMetalMeta_listeners.protoMethods[_name];
  }
  memberNames.forEach(function (name) {
    return members[name](name, Meta);
  });

  Meta.prototype.isSourceDestroying = function isSourceDestroying() {
    return (this._flags & SOURCE_DESTROYING) !== 0;
  };

  Meta.prototype.setSourceDestroying = function setSourceDestroying() {
    this._flags |= SOURCE_DESTROYING;
  };

  Meta.prototype.isSourceDestroyed = function isSourceDestroyed() {
    return (this._flags & SOURCE_DESTROYED) !== 0;
  };

  Meta.prototype.setSourceDestroyed = function setSourceDestroyed() {
    this._flags |= SOURCE_DESTROYED;
  };

  Meta.prototype.isMetaDestroyed = function isMetaDestroyed() {
    return (this._flags & META_DESTROYED) !== 0;
  };

  Meta.prototype.setMetaDestroyed = function setMetaDestroyed() {
    this._flags |= META_DESTROYED;
  };

  Meta.prototype.isProxy = function isProxy() {
    return (this._flags & IS_PROXY) !== 0;
  };

  Meta.prototype.setProxy = function setProxy() {
    this._flags |= IS_PROXY;
  };

  // Implements a member that is a lazily created, non-inheritable
  // POJO.
  function ownMap(name, Meta) {
    var key = memberProperty(name);
    var capitalized = capitalize(name);
    Meta.prototype['writable' + capitalized] = function () {
      return this._getOrCreateOwnMap(key);
    };
    Meta.prototype['readable' + capitalized] = function () {
      return this[key];
    };
  }

  Meta.prototype._getOrCreateOwnMap = function (key) {
    var ret = this[key];
    if (!ret) {
      ret = this[key] = new _emberUtils.EmptyObject();
    }
    return ret;
  };

  // Implements a member that is a lazily created POJO with inheritable
  // values.
  function inheritedMap(name, Meta) {
    var key = memberProperty(name);
    var capitalized = capitalize(name);

    Meta.prototype['write' + capitalized] = function (subkey, value) {

      var map = this._getOrCreateOwnMap(key);
      map[subkey] = value;
    };

    Meta.prototype['peek' + capitalized] = function (subkey) {
      return this._findInherited(key, subkey);
    };

    Meta.prototype['forEach' + capitalized] = function (fn) {
      var pointer = this;
      var seen = new _emberUtils.EmptyObject();
      while (pointer !== undefined) {
        var map = pointer[key];
        if (map) {
          for (var _key in map) {
            if (!seen[_key]) {
              seen[_key] = true;
              fn(_key, map[_key]);
            }
          }
        }
        pointer = pointer.parent;
      }
    };

    Meta.prototype['clear' + capitalized] = function () {

      this[key] = undefined;
    };

    Meta.prototype['deleteFrom' + capitalized] = function (subkey) {
      delete this._getOrCreateOwnMap(key)[subkey];
    };

    Meta.prototype['hasIn' + capitalized] = function (subkey) {
      return this._findInherited(key, subkey) !== undefined;
    };
  }

  Meta.prototype._getInherited = function (key) {
    var pointer = this;
    while (pointer !== undefined) {
      if (pointer[key]) {
        return pointer[key];
      }
      pointer = pointer.parent;
    }
  };

  Meta.prototype._findInherited = function (key, subkey) {
    var pointer = this;
    while (pointer !== undefined) {
      var map = pointer[key];
      if (map) {
        var value = map[subkey];
        if (value !== undefined) {
          return value;
        }
      }
      pointer = pointer.parent;
    }
  };

  var UNDEFINED = _emberUtils.symbol('undefined');

  exports.UNDEFINED = UNDEFINED;
  // Implements a member that provides a lazily created map of maps,
  // with inheritance at both levels.
  Meta.prototype.writeDeps = function writeDeps(subkey, itemkey, value) {

    var outerMap = this._getOrCreateOwnMap('_deps');
    var innerMap = outerMap[subkey];
    if (!innerMap) {
      innerMap = outerMap[subkey] = new _emberUtils.EmptyObject();
    }
    innerMap[itemkey] = value;
  };

  Meta.prototype.peekDeps = function peekDeps(subkey, itemkey) {
    var pointer = this;
    while (pointer !== undefined) {
      var map = pointer._deps;
      if (map) {
        var value = map[subkey];
        if (value) {
          if (value[itemkey] !== undefined) {
            return value[itemkey];
          }
        }
      }
      pointer = pointer.parent;
    }
  };

  Meta.prototype.hasDeps = function hasDeps(subkey) {
    var pointer = this;
    while (pointer !== undefined) {
      if (pointer._deps && pointer._deps[subkey]) {
        return true;
      }
      pointer = pointer.parent;
    }
    return false;
  };

  Meta.prototype.forEachInDeps = function forEachInDeps(subkey, fn) {
    return this._forEachIn('_deps', subkey, fn);
  };

  Meta.prototype._forEachIn = function (key, subkey, fn) {
    var pointer = this;
    var seen = new _emberUtils.EmptyObject();
    var calls = [];
    while (pointer !== undefined) {
      var map = pointer[key];
      if (map) {
        var innerMap = map[subkey];
        if (innerMap) {
          for (var innerKey in innerMap) {
            if (!seen[innerKey]) {
              seen[innerKey] = true;
              calls.push([innerKey, innerMap[innerKey]]);
            }
          }
        }
      }
      pointer = pointer.parent;
    }
    for (var i = 0; i < calls.length; i++) {
      var _calls$i = calls[i];
      var innerKey = _calls$i[0];
      var value = _calls$i[1];

      fn(innerKey, value);
    }
  };

  // Implements a member that provides a non-heritable, lazily-created
  // object using the method you provide.
  function ownCustomObject(name, Meta) {
    var key = memberProperty(name);
    var capitalized = capitalize(name);
    Meta.prototype['writable' + capitalized] = function (create) {

      var ret = this[key];
      if (!ret) {
        ret = this[key] = create(this.source);
      }
      return ret;
    };
    Meta.prototype['readable' + capitalized] = function () {
      return this[key];
    };
  }

  // Implements a member that provides an inheritable, lazily-created
  // object using the method you provide. We will derived children from
  // their parents by calling your object's `copy()` method.
  function inheritedCustomObject(name, Meta) {
    var key = memberProperty(name);
    var capitalized = capitalize(name);
    Meta.prototype['writable' + capitalized] = function (create) {

      var ret = this[key];
      if (!ret) {
        if (this.parent) {
          ret = this[key] = this.parent['writable' + capitalized](create).copy(this.source);
        } else {
          ret = this[key] = create(this.source);
        }
      }
      return ret;
    };
    Meta.prototype['readable' + capitalized] = function () {
      return this._getInherited(key);
    };
  }

  function memberProperty(name) {
    return '_' + name;
  }

  // there's a more general-purpose capitalize in ember-runtime, but we
  // don't want to make ember-metal depend on ember-runtime.
  function capitalize(name) {
    return name.replace(/^\w/, function (m) {
      return m.toUpperCase();
    });
  }

  var META_DESC = {
    writable: true,
    configurable: true,
    enumerable: false,
    value: null
  };

  exports.META_DESC = META_DESC;
  var EMBER_META_PROPERTY = {
    name: META_FIELD,
    descriptor: META_DESC
  };

  if (false) {
    Meta.prototype.readInheritedValue = function (key, subkey) {
      var internalKey = '_' + key;

      var pointer = this;

      while (pointer !== undefined) {
        var map = pointer[internalKey];
        if (map) {
          var value = map[subkey];
          if (value !== undefined || subkey in map) {
            return map[subkey];
          }
        }
        pointer = pointer.parent;
      }

      return UNDEFINED;
    };

    Meta.prototype.writeValue = function (obj, key, value) {
      var descriptor = _emberUtils.lookupDescriptor(obj, key);
      var isMandatorySetter = descriptor && descriptor.set && descriptor.set.isMandatorySetter;

      if (isMandatorySetter) {
        this.writeValues(key, value);
      } else {
        obj[key] = value;
      }
    };
  }

  var setMeta = undefined,
      peekMeta = undefined;

  // choose the one appropriate for given platform
  if (_emberUtils.HAS_NATIVE_WEAKMAP) {
    (function () {
      var getPrototypeOf = Object.getPrototypeOf;
      var metaStore = new WeakMap();

      exports.setMeta = setMeta = function WeakMap_setMeta(obj, meta) {
        metaStore.set(obj, meta);
      };

      exports.peekMeta = peekMeta = function WeakMap_peekMeta(obj) {

        return metaStore.get(obj);
      };

      exports.peekMeta = peekMeta = function WeakMap_peekParentMeta(obj) {
        var pointer = obj;
        var meta = undefined;
        while (pointer) {
          meta = metaStore.get(pointer);
          // jshint loopfunc:true

          // stop if we find a `null` value, since
          // that means the meta was deleted
          // any other truthy value is a "real" meta
          if (meta === null || meta) {
            return meta;
          }

          pointer = getPrototypeOf(pointer);
        }
      };
    })();
  } else {
    exports.setMeta = setMeta = function Fallback_setMeta(obj, meta) {
      // if `null` already, just set it to the new value
      // otherwise define property first
      if (obj[META_FIELD] !== null) {
        if (obj.__defineNonEnumerable) {
          obj.__defineNonEnumerable(EMBER_META_PROPERTY);
        } else {
          Object.defineProperty(obj, META_FIELD, META_DESC);
        }
      }

      obj[META_FIELD] = meta;
    };

    exports.peekMeta = peekMeta = function Fallback_peekMeta(obj) {
      return obj[META_FIELD];
    };
  }

  function deleteMeta(obj) {

    var meta = peekMeta(obj);
    if (meta) {
      meta.destroy();
    }
  }

  /**
    Retrieves the meta hash for an object. If `writable` is true ensures the
    hash is writable for this object as well.
  
    The meta object contains information about computed property descriptors as
    well as any watched properties and other information. You generally will
    not access this information directly but instead work with higher level
    methods that manipulate this hash indirectly.
  
    @method meta
    @for Ember
    @private
  
    @param {Object} obj The object to retrieve meta for
    @param {Boolean} [writable=true] Pass `false` if you do not intend to modify
      the meta hash, allowing the method to avoid making an unnecessary copy.
    @return {Object} the meta hash for an object
  */

  function meta(obj) {

    var maybeMeta = peekMeta(obj);
    var parent = undefined;

    // remove this code, in-favor of explicit parent
    if (maybeMeta) {
      if (maybeMeta.source === obj) {
        return maybeMeta;
      }
      parent = maybeMeta;
    }

    var newMeta = new Meta(obj, parent);
    setMeta(obj, newMeta);
    return newMeta;
  }

  exports.peekMeta = peekMeta;
  exports.setMeta = setMeta;
  exports.counters = counters;
});
enifed('ember-metal/meta_listeners', ['exports'], function (exports) {
  /*
   When we render a rich template hierarchy, the set of events that
   *might* happen tends to be much larger than the set of events that
   actually happen. This implies that we should make listener creation &
   destruction cheap, even at the cost of making event dispatch more
   expensive.
  
   Thus we store a new listener with a single push and no new
   allocations, without even bothering to do deduplication -- we can
   save that for dispatch time, if an event actually happens.
   */

  /* listener flags */
  'use strict';

  var ONCE = 1;
  exports.ONCE = ONCE;
  var SUSPENDED = 2;

  exports.SUSPENDED = SUSPENDED;
  var protoMethods = {

    addToListeners: function (eventName, target, method, flags) {
      if (!this._listeners) {
        this._listeners = [];
      }
      this._listeners.push(eventName, target, method, flags);
    },

    _finalizeListeners: function () {
      if (this._listenersFinalized) {
        return;
      }
      if (!this._listeners) {
        this._listeners = [];
      }
      var pointer = this.parent;
      while (pointer) {
        var listeners = pointer._listeners;
        if (listeners) {
          this._listeners = this._listeners.concat(listeners);
        }
        if (pointer._listenersFinalized) {
          break;
        }
        pointer = pointer.parent;
      }
      this._listenersFinalized = true;
    },

    removeFromListeners: function (eventName, target, method, didRemove) {
      var pointer = this;
      while (pointer) {
        var listeners = pointer._listeners;
        if (listeners) {
          for (var index = listeners.length - 4; index >= 0; index -= 4) {
            if (listeners[index] === eventName && (!method || listeners[index + 1] === target && listeners[index + 2] === method)) {
              if (pointer === this) {
                // we are modifying our own list, so we edit directly
                if (typeof didRemove === 'function') {
                  didRemove(eventName, target, listeners[index + 2]);
                }
                listeners.splice(index, 4);
              } else {
                // we are trying to remove an inherited listener, so we do
                // just-in-time copying to detach our own listeners from
                // our inheritance chain.
                this._finalizeListeners();
                return this.removeFromListeners(eventName, target, method);
              }
            }
          }
        }
        if (pointer._listenersFinalized) {
          break;
        }
        pointer = pointer.parent;
      }
    },

    matchingListeners: function (eventName) {
      var pointer = this;
      var result = [];
      while (pointer) {
        var listeners = pointer._listeners;
        if (listeners) {
          for (var index = 0; index < listeners.length - 3; index += 4) {
            if (listeners[index] === eventName) {
              pushUniqueListener(result, listeners, index);
            }
          }
        }
        if (pointer._listenersFinalized) {
          break;
        }
        pointer = pointer.parent;
      }
      var sus = this._suspendedListeners;
      if (sus) {
        for (var susIndex = 0; susIndex < sus.length - 2; susIndex += 3) {
          if (eventName === sus[susIndex]) {
            for (var resultIndex = 0; resultIndex < result.length - 2; resultIndex += 3) {
              if (result[resultIndex] === sus[susIndex + 1] && result[resultIndex + 1] === sus[susIndex + 2]) {
                result[resultIndex + 2] |= SUSPENDED;
              }
            }
          }
        }
      }
      return result;
    },

    suspendListeners: function (eventNames, target, method, callback) {
      var sus = this._suspendedListeners;
      if (!sus) {
        sus = this._suspendedListeners = [];
      }
      for (var i = 0; i < eventNames.length; i++) {
        sus.push(eventNames[i], target, method);
      }
      try {
        return callback.call(target);
      } finally {
        if (sus.length === eventNames.length) {
          this._suspendedListeners = undefined;
        } else {
          for (var i = sus.length - 3; i >= 0; i -= 3) {
            if (sus[i + 1] === target && sus[i + 2] === method && eventNames.indexOf(sus[i]) !== -1) {
              sus.splice(i, 3);
            }
          }
        }
      }
    },

    watchedEvents: function () {
      var pointer = this;
      var names = {};
      while (pointer) {
        var listeners = pointer._listeners;
        if (listeners) {
          for (var index = 0; index < listeners.length - 3; index += 4) {
            names[listeners[index]] = true;
          }
        }
        if (pointer._listenersFinalized) {
          break;
        }
        pointer = pointer.parent;
      }
      return Object.keys(names);
    },

    _initializeListeners: function () {
      this._listeners = undefined;
      this._listenersFinalized = undefined;
      this._suspendedListeners = undefined;
    }
  };

  exports.protoMethods = protoMethods;
  function pushUniqueListener(destination, source, index) {
    var target = source[index + 1];
    var method = source[index + 2];
    for (var destinationIndex = 0; destinationIndex < destination.length - 2; destinationIndex += 3) {
      if (destination[destinationIndex] === target && destination[destinationIndex + 1] === method) {
        return;
      }
    }
    destination.push(target, method, source[index + 3]);
  }
});
enifed('ember-metal/mixin', ['exports', 'ember-utils', 'ember-metal/error', 'ember-metal/debug', 'ember-metal/meta', 'ember-metal/expand_properties', 'ember-metal/properties', 'ember-metal/computed', 'ember-metal/binding', 'ember-metal/observer', 'ember-metal/events'], function (exports, _emberUtils, _emberMetalError, _emberMetalDebug, _emberMetalMeta, _emberMetalExpand_properties, _emberMetalProperties, _emberMetalComputed, _emberMetalBinding, _emberMetalObserver, _emberMetalEvents) {
  'no use strict';
  // Remove "use strict"; from transpiled module until
  // https://bugs.webkit.org/show_bug.cgi?id=138038 is fixed

  /**
  @module ember
  @submodule ember-metal
  */
  exports.detectBinding = detectBinding;
  exports.mixin = mixin;
  exports.default = Mixin;
  exports.hasUnprocessedMixins = hasUnprocessedMixins;
  exports.clearUnprocessedMixins = clearUnprocessedMixins;
  exports.required = required;
  exports.aliasMethod = aliasMethod;
  exports.observer = observer;
  exports._immediateObserver = _immediateObserver;
  exports._beforeObserver = _beforeObserver;

  function ROOT() {}
  ROOT.__hasSuper = false;

  var a_slice = [].slice;

  function isMethod(obj) {
    return 'function' === typeof obj && obj.isMethod !== false && obj !== Boolean && obj !== Object && obj !== Number && obj !== Array && obj !== Date && obj !== String;
  }

  var CONTINUE = {};

  function mixinProperties(mixinsMeta, mixin) {
    var guid = undefined;

    if (mixin instanceof Mixin) {
      guid = _emberUtils.guidFor(mixin);
      if (mixinsMeta.peekMixins(guid)) {
        return CONTINUE;
      }
      mixinsMeta.writeMixins(guid, mixin);
      return mixin.properties;
    } else {
      return mixin; // apply anonymous mixin properties
    }
  }

  function concatenatedMixinProperties(concatProp, props, values, base) {
    var concats = undefined;

    // reset before adding each new mixin to pickup concats from previous
    concats = values[concatProp] || base[concatProp];
    if (props[concatProp]) {
      concats = concats ? concats.concat(props[concatProp]) : props[concatProp];
    }

    return concats;
  }

  function giveDescriptorSuper(meta, key, property, values, descs, base) {
    var superProperty = undefined;

    // Computed properties override methods, and do not call super to them
    if (values[key] === undefined) {
      // Find the original descriptor in a parent mixin
      superProperty = descs[key];
    }

    // If we didn't find the original descriptor in a parent mixin, find
    // it on the original object.
    if (!superProperty) {
      var possibleDesc = base[key];
      var superDesc = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor ? possibleDesc : undefined;

      superProperty = superDesc;
    }

    if (superProperty === undefined || !(superProperty instanceof _emberMetalComputed.ComputedProperty)) {
      return property;
    }

    // Since multiple mixins may inherit from the same parent, we need
    // to clone the computed property so that other mixins do not receive
    // the wrapped version.
    property = Object.create(property);
    property._getter = _emberUtils.wrap(property._getter, superProperty._getter);
    if (superProperty._setter) {
      if (property._setter) {
        property._setter = _emberUtils.wrap(property._setter, superProperty._setter);
      } else {
        property._setter = superProperty._setter;
      }
    }

    return property;
  }

  function giveMethodSuper(obj, key, method, values, descs) {
    var superMethod = undefined;

    // Methods overwrite computed properties, and do not call super to them.
    if (descs[key] === undefined) {
      // Find the original method in a parent mixin
      superMethod = values[key];
    }

    // If we didn't find the original value in a parent mixin, find it in
    // the original object
    superMethod = superMethod || obj[key];

    // Only wrap the new method if the original method was a function
    if (superMethod === undefined || 'function' !== typeof superMethod) {
      return method;
    }

    return _emberUtils.wrap(method, superMethod);
  }

  function applyConcatenatedProperties(obj, key, value, values) {
    var baseValue = values[key] || obj[key];
    var ret = undefined;

    if (baseValue) {
      if ('function' === typeof baseValue.concat) {
        if (value === null || value === undefined) {
          ret = baseValue;
        } else {
          ret = baseValue.concat(value);
        }
      } else {
        ret = _emberUtils.makeArray(baseValue).concat(value);
      }
    } else {
      ret = _emberUtils.makeArray(value);
    }

    return ret;
  }

  function applyMergedProperties(obj, key, value, values) {
    var baseValue = values[key] || obj[key];

    if (!baseValue) {
      return value;
    }

    var newBase = _emberUtils.assign({}, baseValue);
    var hasFunction = false;

    for (var prop in value) {
      if (!value.hasOwnProperty(prop)) {
        continue;
      }

      var propValue = value[prop];
      if (isMethod(propValue)) {
        // TODO: support for Computed Properties, etc?
        hasFunction = true;
        newBase[prop] = giveMethodSuper(obj, prop, propValue, baseValue, {});
      } else {
        newBase[prop] = propValue;
      }
    }

    if (hasFunction) {
      newBase._super = ROOT;
    }

    return newBase;
  }

  function addNormalizedProperty(base, key, value, meta, descs, values, concats, mergings) {
    if (value instanceof _emberMetalProperties.Descriptor) {
      if (value === REQUIRED && descs[key]) {
        return CONTINUE;
      }

      // Wrap descriptor function to implement
      // _super() if needed
      if (value._getter) {
        value = giveDescriptorSuper(meta, key, value, values, descs, base);
      }

      descs[key] = value;
      values[key] = undefined;
    } else {
      if (concats && concats.indexOf(key) >= 0 || key === 'concatenatedProperties' || key === 'mergedProperties') {
        value = applyConcatenatedProperties(base, key, value, values);
      } else if (mergings && mergings.indexOf(key) >= 0) {
        value = applyMergedProperties(base, key, value, values);
      } else if (isMethod(value)) {
        value = giveMethodSuper(base, key, value, values, descs);
      }

      descs[key] = undefined;
      values[key] = value;
    }
  }

  function mergeMixins(mixins, m, descs, values, base, keys) {
    var currentMixin = undefined,
        props = undefined,
        key = undefined,
        concats = undefined,
        mergings = undefined;

    function removeKeys(keyName) {
      delete descs[keyName];
      delete values[keyName];
    }

    for (var i = 0; i < mixins.length; i++) {
      currentMixin = mixins[i];

      props = mixinProperties(m, currentMixin);
      if (props === CONTINUE) {
        continue;
      }

      if (props) {
        if (base.willMergeMixin) {
          base.willMergeMixin(props);
        }
        concats = concatenatedMixinProperties('concatenatedProperties', props, values, base);
        mergings = concatenatedMixinProperties('mergedProperties', props, values, base);

        for (key in props) {
          if (!props.hasOwnProperty(key)) {
            continue;
          }
          keys.push(key);
          addNormalizedProperty(base, key, props[key], m, descs, values, concats, mergings);
        }

        // manually copy toString() because some JS engines do not enumerate it
        if (props.hasOwnProperty('toString')) {
          base.toString = props.toString;
        }
      } else if (currentMixin.mixins) {
        mergeMixins(currentMixin.mixins, m, descs, values, base, keys);
        if (currentMixin._without) {
          currentMixin._without.forEach(removeKeys);
        }
      }
    }
  }

  function detectBinding(key) {
    var length = key.length;

    return length > 7 && key.charCodeAt(length - 7) === 66 && key.indexOf('inding', length - 6) !== -1;
  }

  // warm both paths of above function
  detectBinding('notbound');
  detectBinding('fooBinding');

  function connectBindings(obj, m) {
    // TODO Mixin.apply(instance) should disconnect binding if exists
    m.forEachBindings(function (key, binding) {
      if (binding) {
        var to = key.slice(0, -7); // strip Binding off end
        if (binding instanceof _emberMetalBinding.Binding) {
          binding = binding.copy(); // copy prototypes' instance
          binding.to(to);
        } else {
          // binding is string path
          binding = new _emberMetalBinding.Binding(to, binding);
        }
        binding.connect(obj);
        obj[key] = binding;
      }
    });
    // mark as applied
    m.clearBindings();
  }

  function finishPartial(obj, m) {
    connectBindings(obj, m || _emberMetalMeta.meta(obj));
    return obj;
  }

  function followAlias(obj, desc, m, descs, values) {
    var altKey = desc.methodName;
    var value = undefined;
    var possibleDesc = undefined;
    if (descs[altKey] || values[altKey]) {
      value = values[altKey];
      desc = descs[altKey];
    } else if ((possibleDesc = obj[altKey]) && possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor) {
      desc = possibleDesc;
      value = undefined;
    } else {
      desc = undefined;
      value = obj[altKey];
    }

    return { desc: desc, value: value };
  }

  function updateObserversAndListeners(obj, key, observerOrListener, pathsKey, updateMethod) {
    var paths = observerOrListener[pathsKey];

    if (paths) {
      for (var i = 0; i < paths.length; i++) {
        updateMethod(obj, paths[i], null, key);
      }
    }
  }

  function replaceObserversAndListeners(obj, key, observerOrListener) {
    var prev = obj[key];

    if ('function' === typeof prev) {
      updateObserversAndListeners(obj, key, prev, '__ember_observesBefore__', _emberMetalObserver._removeBeforeObserver);
      updateObserversAndListeners(obj, key, prev, '__ember_observes__', _emberMetalObserver.removeObserver);
      updateObserversAndListeners(obj, key, prev, '__ember_listens__', _emberMetalEvents.removeListener);
    }

    if ('function' === typeof observerOrListener) {
      updateObserversAndListeners(obj, key, observerOrListener, '__ember_observesBefore__', _emberMetalObserver._addBeforeObserver);
      updateObserversAndListeners(obj, key, observerOrListener, '__ember_observes__', _emberMetalObserver.addObserver);
      updateObserversAndListeners(obj, key, observerOrListener, '__ember_listens__', _emberMetalEvents.addListener);
    }
  }

  function applyMixin(obj, mixins, partial) {
    var descs = {};
    var values = {};
    var m = _emberMetalMeta.meta(obj);
    var keys = [];
    var key = undefined,
        value = undefined,
        desc = undefined;

    obj._super = ROOT;

    // Go through all mixins and hashes passed in, and:
    //
    // * Handle concatenated properties
    // * Handle merged properties
    // * Set up _super wrapping if necessary
    // * Set up computed property descriptors
    // * Copying `toString` in broken browsers
    mergeMixins(mixins, m, descs, values, obj, keys);

    for (var i = 0; i < keys.length; i++) {
      key = keys[i];
      if (key === 'constructor' || !values.hasOwnProperty(key)) {
        continue;
      }

      desc = descs[key];
      value = values[key];

      if (desc === REQUIRED) {
        continue;
      }

      while (desc && desc instanceof Alias) {
        var followed = followAlias(obj, desc, m, descs, values);
        desc = followed.desc;
        value = followed.value;
      }

      if (desc === undefined && value === undefined) {
        continue;
      }

      replaceObserversAndListeners(obj, key, value);

      if (detectBinding(key)) {
        m.writeBindings(key, value);
      }

      _emberMetalProperties.defineProperty(obj, key, desc, value, m);
    }

    if (!partial) {
      // don't apply to prototype
      finishPartial(obj, m);
    }

    return obj;
  }

  /**
    @method mixin
    @for Ember
    @param obj
    @param mixins*
    @return obj
    @private
  */

  function mixin(obj) {
    for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      args[_key - 1] = arguments[_key];
    }

    applyMixin(obj, args, false);
    return obj;
  }

  /**
    The `Ember.Mixin` class allows you to create mixins, whose properties can be
    added to other classes. For instance,
  
    ```javascript
    const EditableMixin = Ember.Mixin.create({
      edit() {
        console.log('starting to edit');
        this.set('isEditing', true);
      },
      isEditing: false
    });
  
    // Mix mixins into classes by passing them as the first arguments to
    // `.extend.`
    const Comment = Ember.Object.extend(EditableMixin, {
      post: null
    });
  
    let comment = Comment.create(post: somePost);
    comment.edit(); // outputs 'starting to edit'
    ```
  
    Note that Mixins are created with `Ember.Mixin.create`, not
    `Ember.Mixin.extend`.
  
    Note that mixins extend a constructor's prototype so arrays and object literals
    defined as properties will be shared amongst objects that implement the mixin.
    If you want to define a property in a mixin that is not shared, you can define
    it either as a computed property or have it be created on initialization of the object.
  
    ```javascript
    // filters array will be shared amongst any object implementing mixin
    const FilterableMixin = Ember.Mixin.create({
      filters: Ember.A()
    });
  
    // filters will be a separate array for every object implementing the mixin
    const FilterableMixin = Ember.Mixin.create({
      filters: Ember.computed(function() {
        return Ember.A();
      })
    });
  
    // filters will be created as a separate array during the object's initialization
    const Filterable = Ember.Mixin.create({
      init() {
        this._super(...arguments);
        this.set("filters", Ember.A());
      }
    });
    ```
  
    @class Mixin
    @namespace Ember
    @public
  */

  function Mixin(args, properties) {
    this.properties = properties;

    var length = args && args.length;

    if (length > 0) {
      var m = new Array(length);

      for (var i = 0; i < length; i++) {
        var x = args[i];
        if (x instanceof Mixin) {
          m[i] = x;
        } else {
          m[i] = new Mixin(undefined, x);
        }
      }

      this.mixins = m;
    } else {
      this.mixins = undefined;
    }
    this.ownerConstructor = undefined;
    this._without = undefined;
    this[_emberUtils.GUID_KEY] = null;
    this[_emberUtils.NAME_KEY] = null;
  }

  Mixin._apply = applyMixin;

  Mixin.applyPartial = function (obj) {
    var args = a_slice.call(arguments, 1);
    return applyMixin(obj, args, true);
  };

  Mixin.finishPartial = finishPartial;

  var unprocessedFlag = false;

  function hasUnprocessedMixins() {
    return unprocessedFlag;
  }

  function clearUnprocessedMixins() {
    unprocessedFlag = false;
  }

  /**
    @method create
    @static
    @param arguments*
    @public
  */
  Mixin.create = function () {
    // ES6TODO: this relies on a global state?
    unprocessedFlag = true;
    var M = this;

    for (var _len2 = arguments.length, args = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
      args[_key2] = arguments[_key2];
    }

    return new M(args, undefined);
  };

  var MixinPrototype = Mixin.prototype;

  /**
    @method reopen
    @param arguments*
    @private
  */
  MixinPrototype.reopen = function () {
    var currentMixin = undefined;

    if (this.properties) {
      currentMixin = new Mixin(undefined, this.properties);
      this.properties = undefined;
      this.mixins = [currentMixin];
    } else if (!this.mixins) {
      this.mixins = [];
    }

    var mixins = this.mixins;
    var idx = undefined;

    for (idx = 0; idx < arguments.length; idx++) {
      currentMixin = arguments[idx];

      if (currentMixin instanceof Mixin) {
        mixins.push(currentMixin);
      } else {
        mixins.push(new Mixin(undefined, currentMixin));
      }
    }

    return this;
  };

  /**
    @method apply
    @param obj
    @return applied object
    @private
  */
  MixinPrototype.apply = function (obj) {
    return applyMixin(obj, [this], false);
  };

  MixinPrototype.applyPartial = function (obj) {
    return applyMixin(obj, [this], true);
  };

  MixinPrototype.toString = Object.toString;

  function _detect(curMixin, targetMixin, seen) {
    var guid = _emberUtils.guidFor(curMixin);

    if (seen[guid]) {
      return false;
    }
    seen[guid] = true;

    if (curMixin === targetMixin) {
      return true;
    }
    var mixins = curMixin.mixins;
    var loc = mixins ? mixins.length : 0;
    while (--loc >= 0) {
      if (_detect(mixins[loc], targetMixin, seen)) {
        return true;
      }
    }
    return false;
  }

  /**
    @method detect
    @param obj
    @return {Boolean}
    @private
  */
  MixinPrototype.detect = function (obj) {
    if (typeof obj !== 'object' || obj === null) {
      return false;
    }
    if (obj instanceof Mixin) {
      return _detect(obj, this, {});
    }
    var m = _emberMetalMeta.peekMeta(obj);
    if (!m) {
      return false;
    }
    return !!m.peekMixins(_emberUtils.guidFor(this));
  };

  MixinPrototype.without = function () {
    var ret = new Mixin([this]);

    for (var _len3 = arguments.length, args = Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
      args[_key3] = arguments[_key3];
    }

    ret._without = args;
    return ret;
  };

  function _keys(ret, mixin, seen) {
    if (seen[_emberUtils.guidFor(mixin)]) {
      return;
    }
    seen[_emberUtils.guidFor(mixin)] = true;

    if (mixin.properties) {
      var props = Object.keys(mixin.properties);
      for (var i = 0; i < props.length; i++) {
        var key = props[i];
        ret[key] = true;
      }
    } else if (mixin.mixins) {
      mixin.mixins.forEach(function (x) {
        return _keys(ret, x, seen);
      });
    }
  }

  MixinPrototype.keys = function () {
    var keys = {};
    var seen = {};

    _keys(keys, this, seen);
    var ret = Object.keys(keys);
    return ret;
  };

  // returns the mixins currently applied to the specified object
  // TODO: Make Ember.mixin
  Mixin.mixins = function (obj) {
    var m = _emberMetalMeta.peekMeta(obj);
    var ret = [];
    if (!m) {
      return ret;
    }

    m.forEachMixins(function (key, currentMixin) {
      // skip primitive mixins since these are always anonymous
      if (!currentMixin.properties) {
        ret.push(currentMixin);
      }
    });

    return ret;
  };

  var REQUIRED = new _emberMetalProperties.Descriptor();
  REQUIRED.toString = function () {
    return '(Required Property)';
  };

  /**
    Denotes a required property for a mixin
  
    @method required
    @for Ember
    @private
  */

  function required() {
    return REQUIRED;
  }

  function Alias(methodName) {
    this.isDescriptor = true;
    this.methodName = methodName;
  }

  Alias.prototype = new _emberMetalProperties.Descriptor();

  /**
    Makes a method available via an additional name.
  
    ```javascript
    App.Person = Ember.Object.extend({
      name: function() {
        return 'Tomhuda Katzdale';
      },
      moniker: Ember.aliasMethod('name')
    });
  
    let goodGuy = App.Person.create();
  
    goodGuy.name();    // 'Tomhuda Katzdale'
    goodGuy.moniker(); // 'Tomhuda Katzdale'
    ```
  
    @method aliasMethod
    @for Ember
    @param {String} methodName name of the method to alias
    @public
  */

  function aliasMethod(methodName) {
    return new Alias(methodName);
  }

  // ..........................................................
  // OBSERVER HELPER
  //

  /**
    Specify a method that observes property changes.
  
    ```javascript
    Ember.Object.extend({
      valueObserver: Ember.observer('value', function() {
        // Executes whenever the "value" property changes
      })
    });
    ```
  
    Also available as `Function.prototype.observes` if prototype extensions are
    enabled.
  
    @method observer
    @for Ember
    @param {String} propertyNames*
    @param {Function} func
    @return func
    @public
  */

  function observer() {
    for (var _len4 = arguments.length, args = Array(_len4), _key4 = 0; _key4 < _len4; _key4++) {
      args[_key4] = arguments[_key4];
    }

    var func = args.slice(-1)[0];
    var paths = undefined;

    var addWatchedProperty = function (path) {
      paths.push(path);
    };
    var _paths = args.slice(0, -1);

    if (typeof func !== 'function') {

      func = args[0];

      // revert to old, soft-deprecated argument ordering
      _paths = args.slice(1);
    }

    paths = [];

    for (var i = 0; i < _paths.length; ++i) {
      _emberMetalExpand_properties.default(_paths[i], addWatchedProperty);
    }

    if (typeof func !== 'function') {
      throw new _emberMetalError.default('Ember.observer called without a function');
    }

    func.__ember_observes__ = paths;
    return func;
  }

  /**
    Specify a method that observes property changes.
  
    ```javascript
    Ember.Object.extend({
      valueObserver: Ember.immediateObserver('value', function() {
        // Executes whenever the "value" property changes
      })
    });
    ```
  
    In the future, `Ember.observer` may become asynchronous. In this event,
    `Ember.immediateObserver` will maintain the synchronous behavior.
  
    Also available as `Function.prototype.observesImmediately` if prototype extensions are
    enabled.
  
    @method _immediateObserver
    @for Ember
    @param {String} propertyNames*
    @param {Function} func
    @deprecated Use `Ember.observer` instead.
    @return func
    @private
  */

  function _immediateObserver() {

    for (var i = 0; i < arguments.length; i++) {
      var arg = arguments[i];
    }

    return observer.apply(this, arguments);
  }

  /**
    When observers fire, they are called with the arguments `obj`, `keyName`.
  
    Note, `@each.property` observer is called per each add or replace of an element
    and it's not called with a specific enumeration item.
  
    A `_beforeObserver` fires before a property changes.
  
    @method beforeObserver
    @for Ember
    @param {String} propertyNames*
    @param {Function} func
    @return func
    @deprecated
    @private
  */

  function _beforeObserver() {
    for (var _len5 = arguments.length, args = Array(_len5), _key5 = 0; _key5 < _len5; _key5++) {
      args[_key5] = arguments[_key5];
    }

    var func = args.slice(-1)[0];
    var paths = undefined;

    var addWatchedProperty = function (path) {
      paths.push(path);
    };

    var _paths = args.slice(0, -1);

    if (typeof func !== 'function') {
      // revert to old, soft-deprecated argument ordering

      func = args[0];
      _paths = args.slice(1);
    }

    paths = [];

    for (var i = 0; i < _paths.length; ++i) {
      _emberMetalExpand_properties.default(_paths[i], addWatchedProperty);
    }

    if (typeof func !== 'function') {
      throw new _emberMetalError.default('_beforeObserver called without a function');
    }

    func.__ember_observesBefore__ = paths;
    return func;
  }

  exports.Mixin = Mixin;
  exports.required = required;
  exports.REQUIRED = REQUIRED;
});

// it is possible to use concatenatedProperties with strings (which cannot be frozen)
// only freeze objects...

// prevent mutating `concatenatedProperties` array after it is applied
// use conditional to avoid stringifying every time
enifed('ember-metal/observer', ['exports', 'ember-metal/watching', 'ember-metal/events'], function (exports, _emberMetalWatching, _emberMetalEvents) {
  'use strict';

  exports.addObserver = addObserver;
  exports.observersFor = observersFor;
  exports.removeObserver = removeObserver;
  exports._addBeforeObserver = _addBeforeObserver;
  exports._suspendObserver = _suspendObserver;
  exports._suspendObservers = _suspendObservers;
  exports._removeBeforeObserver = _removeBeforeObserver;

  /**
  @module ember-metal
  */

  var AFTER_OBSERVERS = ':change';
  var BEFORE_OBSERVERS = ':before';

  function changeEvent(keyName) {
    return keyName + AFTER_OBSERVERS;
  }

  function beforeEvent(keyName) {
    return keyName + BEFORE_OBSERVERS;
  }

  /**
    @method addObserver
    @for Ember
    @param obj
    @param {String} _path
    @param {Object|Function} target
    @param {Function|String} [method]
    @public
  */

  function addObserver(obj, _path, target, method) {
    _emberMetalEvents.addListener(obj, changeEvent(_path), target, method);
    _emberMetalWatching.watch(obj, _path);

    return this;
  }

  function observersFor(obj, path) {
    return _emberMetalEvents.listenersFor(obj, changeEvent(path));
  }

  /**
    @method removeObserver
    @for Ember
    @param obj
    @param {String} path
    @param {Object|Function} target
    @param {Function|String} [method]
    @public
  */

  function removeObserver(obj, path, target, method) {
    _emberMetalWatching.unwatch(obj, path);
    _emberMetalEvents.removeListener(obj, changeEvent(path), target, method);

    return this;
  }

  /**
    @method _addBeforeObserver
    @for Ember
    @param obj
    @param {String} path
    @param {Object|Function} target
    @param {Function|String} [method]
    @deprecated
    @private
  */

  function _addBeforeObserver(obj, path, target, method) {
    _emberMetalEvents.addListener(obj, beforeEvent(path), target, method);
    _emberMetalWatching.watch(obj, path);

    return this;
  }

  // Suspend observer during callback.
  //
  // This should only be used by the target of the observer
  // while it is setting the observed path.

  function _suspendObserver(obj, path, target, method, callback) {
    return _emberMetalEvents.suspendListener(obj, changeEvent(path), target, method, callback);
  }

  function _suspendObservers(obj, paths, target, method, callback) {
    var events = paths.map(changeEvent);
    return _emberMetalEvents.suspendListeners(obj, events, target, method, callback);
  }

  /**
    @method removeBeforeObserver
    @for Ember
    @param obj
    @param {String} path
    @param {Object|Function} target
    @param {Function|String} [method]
    @deprecated
    @private
  */

  function _removeBeforeObserver(obj, path, target, method) {
    _emberMetalWatching.unwatch(obj, path);
    _emberMetalEvents.removeListener(obj, beforeEvent(path), target, method);

    return this;
  }
});
enifed('ember-metal/observer_set', ['exports', 'ember-utils', 'ember-metal/events'], function (exports, _emberUtils, _emberMetalEvents) {
  'use strict';

  exports.default = ObserverSet;

  /*
    this.observerSet = {
      [senderGuid]: { // variable name: `keySet`
        [keyName]: listIndex
      }
    },
    this.observers = [
      {
        sender: obj,
        keyName: keyName,
        eventName: eventName,
        listeners: [
          [target, method, flags]
        ]
      },
      ...
    ]
  */

  function ObserverSet() {
    this.clear();
  }

  ObserverSet.prototype.add = function (sender, keyName, eventName) {
    var observerSet = this.observerSet;
    var observers = this.observers;
    var senderGuid = _emberUtils.guidFor(sender);
    var keySet = observerSet[senderGuid];
    var index = undefined;

    if (!keySet) {
      observerSet[senderGuid] = keySet = {};
    }
    index = keySet[keyName];
    if (index === undefined) {
      index = observers.push({
        sender: sender,
        keyName: keyName,
        eventName: eventName,
        listeners: []
      }) - 1;
      keySet[keyName] = index;
    }
    return observers[index].listeners;
  };

  ObserverSet.prototype.flush = function () {
    var observers = this.observers;
    var i = undefined,
        observer = undefined,
        sender = undefined;
    this.clear();
    for (i = 0; i < observers.length; ++i) {
      observer = observers[i];
      sender = observer.sender;
      if (sender.isDestroying || sender.isDestroyed) {
        continue;
      }
      _emberMetalEvents.sendEvent(sender, observer.eventName, [sender, observer.keyName], observer.listeners);
    }
  };

  ObserverSet.prototype.clear = function () {
    this.observerSet = {};
    this.observers = [];
  };
});
enifed('ember-metal/path_cache', ['exports', 'ember-metal/cache'], function (exports, _emberMetalCache) {
  'use strict';

  exports.isGlobal = isGlobal;
  exports.isGlobalPath = isGlobalPath;
  exports.hasThis = hasThis;
  exports.isPath = isPath;
  exports.getFirstKey = getFirstKey;
  exports.getTailPath = getTailPath;

  var IS_GLOBAL = /^[A-Z$]/;
  var IS_GLOBAL_PATH = /^[A-Z$].*[\.]/;
  var HAS_THIS = 'this.';

  var isGlobalCache = new _emberMetalCache.default(1000, function (key) {
    return IS_GLOBAL.test(key);
  });
  var isGlobalPathCache = new _emberMetalCache.default(1000, function (key) {
    return IS_GLOBAL_PATH.test(key);
  });
  var hasThisCache = new _emberMetalCache.default(1000, function (key) {
    return key.lastIndexOf(HAS_THIS, 0) === 0;
  });
  var firstDotIndexCache = new _emberMetalCache.default(1000, function (key) {
    return key.indexOf('.');
  });

  var firstKeyCache = new _emberMetalCache.default(1000, function (path) {
    var index = firstDotIndexCache.get(path);
    if (index === -1) {
      return path;
    } else {
      return path.slice(0, index);
    }
  });

  var tailPathCache = new _emberMetalCache.default(1000, function (path) {
    var index = firstDotIndexCache.get(path);
    if (index !== -1) {
      return path.slice(index + 1);
    }
  });

  var caches = {
    isGlobalCache: isGlobalCache,
    isGlobalPathCache: isGlobalPathCache,
    hasThisCache: hasThisCache,
    firstDotIndexCache: firstDotIndexCache,
    firstKeyCache: firstKeyCache,
    tailPathCache: tailPathCache
  };

  exports.caches = caches;

  function isGlobal(path) {
    return isGlobalCache.get(path);
  }

  function isGlobalPath(path) {
    return isGlobalPathCache.get(path);
  }

  function hasThis(path) {
    return hasThisCache.get(path);
  }

  function isPath(path) {
    return firstDotIndexCache.get(path) !== -1;
  }

  function getFirstKey(path) {
    return firstKeyCache.get(path);
  }

  function getTailPath(path) {
    return tailPathCache.get(path);
  }
});
enifed('ember-metal/properties', ['exports', 'ember-metal/debug', 'ember-metal/features', 'ember-metal/meta', 'ember-metal/property_events'], function (exports, _emberMetalDebug, _emberMetalFeatures, _emberMetalMeta, _emberMetalProperty_events) {
  /**
  @module ember-metal
  */

  'use strict';

  exports.Descriptor = Descriptor;
  exports.MANDATORY_SETTER_FUNCTION = MANDATORY_SETTER_FUNCTION;
  exports.DEFAULT_GETTER_FUNCTION = DEFAULT_GETTER_FUNCTION;
  exports.INHERITING_GETTER_FUNCTION = INHERITING_GETTER_FUNCTION;
  exports.defineProperty = defineProperty;

  // ..........................................................
  // DESCRIPTOR
  //

  /**
    Objects of this type can implement an interface to respond to requests to
    get and set. The default implementation handles simple properties.
  
    @class Descriptor
    @private
  */

  function Descriptor() {
    this.isDescriptor = true;
  }

  var REDEFINE_SUPPORTED = (function () {
    // https://github.com/spalger/kibana/commit/b7e35e6737df585585332857a4c397dc206e7ff9
    var a = Object.create(Object.prototype, {
      prop: {
        configurable: true,
        value: 1
      }
    });

    Object.defineProperty(a, 'prop', {
      configurable: true,
      value: 2
    });

    return a.prop === 2;
  })();
  // ..........................................................
  // DEFINING PROPERTIES API
  //

  function MANDATORY_SETTER_FUNCTION(name) {
    function SETTER_FUNCTION(value) {
      var m = _emberMetalMeta.peekMeta(this);
      if (!m.isInitialized(this)) {
        m.writeValues(name, value);
      } else {}
    }

    SETTER_FUNCTION.isMandatorySetter = true;
    return SETTER_FUNCTION;
  }

  function DEFAULT_GETTER_FUNCTION(name) {
    return function GETTER_FUNCTION() {
      var meta = _emberMetalMeta.peekMeta(this);
      return meta && meta.peekValues(name);
    };
  }

  function INHERITING_GETTER_FUNCTION(name) {
    function IGETTER_FUNCTION() {
      var meta = _emberMetalMeta.peekMeta(this);
      var val = meta && meta.readInheritedValue('values', name);

      if (val === _emberMetalMeta.UNDEFINED) {
        var proto = Object.getPrototypeOf(this);
        return proto && proto[name];
      } else {
        return val;
      }
    }

    IGETTER_FUNCTION.isInheritingGetter = true;
    return IGETTER_FUNCTION;
  }

  /**
    NOTE: This is a low-level method used by other parts of the API. You almost
    never want to call this method directly. Instead you should use
    `Ember.mixin()` to define new properties.
  
    Defines a property on an object. This method works much like the ES5
    `Object.defineProperty()` method except that it can also accept computed
    properties and other special descriptors.
  
    Normally this method takes only three parameters. However if you pass an
    instance of `Descriptor` as the third param then you can pass an
    optional value as the fourth parameter. This is often more efficient than
    creating new descriptor hashes for each property.
  
    ## Examples
  
    ```javascript
    // ES5 compatible mode
    Ember.defineProperty(contact, 'firstName', {
      writable: true,
      configurable: false,
      enumerable: true,
      value: 'Charles'
    });
  
    // define a simple property
    Ember.defineProperty(contact, 'lastName', undefined, 'Jolley');
  
    // define a computed property
    Ember.defineProperty(contact, 'fullName', Ember.computed('firstName', 'lastName', function() {
      return this.firstName+' '+this.lastName;
    }));
    ```
  
    @private
    @method defineProperty
    @for Ember
    @param {Object} obj the object to define this property on. This may be a prototype.
    @param {String} keyName the name of the property
    @param {Descriptor} [desc] an instance of `Descriptor` (typically a
      computed property) or an ES5 descriptor.
      You must provide this or `data` but not both.
    @param {*} [data] something other than a descriptor, that will
      become the explicit value of this property.
  */

  function defineProperty(obj, keyName, desc, data, meta) {
    var possibleDesc = undefined,
        existingDesc = undefined,
        watching = undefined,
        value = undefined;

    if (!meta) {
      meta = _emberMetalMeta.meta(obj);
    }
    var watchEntry = meta.peekWatching(keyName);
    possibleDesc = obj[keyName];
    existingDesc = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor ? possibleDesc : undefined;

    watching = watchEntry !== undefined && watchEntry > 0;

    if (existingDesc) {
      existingDesc.teardown(obj, keyName);
    }

    if (desc instanceof Descriptor) {
      value = desc;
      if (false) {
        if (watching) {
          Object.defineProperty(obj, keyName, {
            configurable: true,
            enumerable: true,
            writable: true,
            value: value
          });
        } else {
          obj[keyName] = value;
        }
      } else {
        obj[keyName] = value;
      }
      if (desc.setup) {
        desc.setup(obj, keyName);
      }
    } else {
      if (desc == null) {
        value = data;

        if (false) {
          if (watching) {
            meta.writeValues(keyName, data);

            var defaultDescriptor = {
              configurable: true,
              enumerable: true,
              set: MANDATORY_SETTER_FUNCTION(keyName),
              get: DEFAULT_GETTER_FUNCTION(keyName)
            };

            if (REDEFINE_SUPPORTED) {
              Object.defineProperty(obj, keyName, defaultDescriptor);
            } else {
              handleBrokenPhantomDefineProperty(obj, keyName, defaultDescriptor);
            }
          } else {
            obj[keyName] = data;
          }
        } else {
          obj[keyName] = data;
        }
      } else {
        value = desc;

        // fallback to ES5
        Object.defineProperty(obj, keyName, desc);
      }
    }

    // if key is being watched, override chains that
    // were initialized with the prototype
    if (watching) {
      _emberMetalProperty_events.overrideChains(obj, keyName, meta);
    }

    // The `value` passed to the `didDefineProperty` hook is
    // either the descriptor or data, whichever was passed.
    if (obj.didDefineProperty) {
      obj.didDefineProperty(obj, keyName, value);
    }

    return this;
  }

  function handleBrokenPhantomDefineProperty(obj, keyName, desc) {
    // https://github.com/ariya/phantomjs/issues/11856
    Object.defineProperty(obj, keyName, { configurable: true, writable: true, value: 'iCry' });
    Object.defineProperty(obj, keyName, desc);
  }
});
enifed('ember-metal/property_events', ['exports', 'ember-utils', 'ember-metal/meta', 'ember-metal/events', 'ember-metal/tags', 'ember-metal/observer_set', 'ember-metal/features', 'ember-metal/transaction'], function (exports, _emberUtils, _emberMetalMeta, _emberMetalEvents, _emberMetalTags, _emberMetalObserver_set, _emberMetalFeatures, _emberMetalTransaction) {
  'use strict';

  var PROPERTY_DID_CHANGE = _emberUtils.symbol('PROPERTY_DID_CHANGE');

  exports.PROPERTY_DID_CHANGE = PROPERTY_DID_CHANGE;
  var beforeObserverSet = new _emberMetalObserver_set.default();
  var observerSet = new _emberMetalObserver_set.default();
  var deferred = 0;

  // ..........................................................
  // PROPERTY CHANGES
  //

  /**
    This function is called just before an object property is about to change.
    It will notify any before observers and prepare caches among other things.
  
    Normally you will not need to call this method directly but if for some
    reason you can't directly watch a property you can invoke this method
    manually along with `Ember.propertyDidChange()` which you should call just
    after the property value changes.
  
    @method propertyWillChange
    @for Ember
    @param {Object} obj The object with the property that will change
    @param {String} keyName The property key (or path) that will change.
    @return {void}
    @private
  */
  function propertyWillChange(obj, keyName, _meta) {
    var meta = _meta || _emberMetalMeta.peekMeta(obj);

    if (meta && !meta.isInitialized(obj)) {
      return;
    }

    var watching = meta && meta.peekWatching(keyName) > 0;
    var possibleDesc = obj[keyName];
    var desc = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor ? possibleDesc : undefined;

    if (desc && desc.willChange) {
      desc.willChange(obj, keyName);
    }

    if (watching) {
      dependentKeysWillChange(obj, keyName, meta);
      chainsWillChange(obj, keyName, meta);
      notifyBeforeObservers(obj, keyName, meta);
    }
  }

  /**
    This function is called just after an object property has changed.
    It will notify any observers and clear caches among other things.
  
    Normally you will not need to call this method directly but if for some
    reason you can't directly watch a property you can invoke this method
    manually along with `Ember.propertyWillChange()` which you should call just
    before the property value changes.
  
    @method propertyDidChange
    @for Ember
    @param {Object} obj The object with the property that will change
    @param {String} keyName The property key (or path) that will change.
    @param {Meta} meta The objects meta.
    @return {void}
    @private
  */
  function propertyDidChange(obj, keyName, _meta) {
    var meta = _meta || _emberMetalMeta.peekMeta(obj);

    if (meta && !meta.isInitialized(obj)) {
      return;
    }

    var watching = meta && meta.peekWatching(keyName) > 0;
    var possibleDesc = obj[keyName];
    var desc = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor ? possibleDesc : undefined;

    // shouldn't this mean that we're watching this key?
    if (desc && desc.didChange) {
      desc.didChange(obj, keyName);
    }

    if (watching) {
      if (meta.hasDeps(keyName)) {
        dependentKeysDidChange(obj, keyName, meta);
      }

      chainsDidChange(obj, keyName, meta, false);
      notifyObservers(obj, keyName, meta);
    }

    if (obj[PROPERTY_DID_CHANGE]) {
      obj[PROPERTY_DID_CHANGE](keyName);
    }

    if (meta && meta.isSourceDestroying()) {
      return;
    }

    _emberMetalTags.markObjectAsDirty(meta, keyName);

    if (false || false) {
      _emberMetalTransaction.assertNotRendered(obj, keyName, meta);
    }
  }

  var WILL_SEEN = undefined,
      DID_SEEN = undefined;
  // called whenever a property is about to change to clear the cache of any dependent keys (and notify those properties of changes, etc...)
  function dependentKeysWillChange(obj, depKey, meta) {
    if (meta && meta.isSourceDestroying()) {
      return;
    }

    if (meta && meta.hasDeps(depKey)) {
      var seen = WILL_SEEN;
      var _top = !seen;

      if (_top) {
        seen = WILL_SEEN = {};
      }

      iterDeps(propertyWillChange, obj, depKey, seen, meta);

      if (_top) {
        WILL_SEEN = null;
      }
    }
  }

  // called whenever a property has just changed to update dependent keys
  function dependentKeysDidChange(obj, depKey, meta) {
    if (meta && meta.isSourceDestroying()) {
      return;
    }

    if (meta && meta.hasDeps(depKey)) {
      var seen = DID_SEEN;
      var _top2 = !seen;

      if (_top2) {
        seen = DID_SEEN = {};
      }

      iterDeps(propertyDidChange, obj, depKey, seen, meta);

      if (_top2) {
        DID_SEEN = null;
      }
    }
  }

  function iterDeps(method, obj, depKey, seen, meta) {
    var possibleDesc = undefined,
        desc = undefined;
    var guid = _emberUtils.guidFor(obj);
    var current = seen[guid];

    if (!current) {
      current = seen[guid] = {};
    }

    if (current[depKey]) {
      return;
    }

    current[depKey] = true;

    meta.forEachInDeps(depKey, function (key, value) {
      if (!value) {
        return;
      }

      possibleDesc = obj[key];
      desc = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor ? possibleDesc : undefined;

      if (desc && desc._suspended === obj) {
        return;
      }

      method(obj, key, meta);
    });
  }

  function chainsWillChange(obj, keyName, meta) {
    var chainWatchers = meta.readableChainWatchers();
    if (chainWatchers) {
      chainWatchers.notify(keyName, false, propertyWillChange);
    }
  }

  function chainsDidChange(obj, keyName, meta) {
    var chainWatchers = meta.readableChainWatchers();
    if (chainWatchers) {
      chainWatchers.notify(keyName, true, propertyDidChange);
    }
  }

  function overrideChains(obj, keyName, meta) {
    var chainWatchers = meta.readableChainWatchers();
    if (chainWatchers) {
      chainWatchers.revalidate(keyName);
    }
  }

  /**
    @method beginPropertyChanges
    @chainable
    @private
  */
  function beginPropertyChanges() {
    deferred++;
  }

  /**
    @method endPropertyChanges
    @private
  */
  function endPropertyChanges() {
    deferred--;
    if (deferred <= 0) {
      beforeObserverSet.clear();
      observerSet.flush();
    }
  }

  /**
    Make a series of property changes together in an
    exception-safe way.
  
    ```javascript
    Ember.changeProperties(function() {
      obj1.set('foo', mayBlowUpWhenSet);
      obj2.set('bar', baz);
    });
    ```
  
    @method changeProperties
    @param {Function} callback
    @param [binding]
    @private
  */
  function changeProperties(callback, binding) {
    beginPropertyChanges();
    try {
      callback.call(binding);
    } finally {
      endPropertyChanges.call(binding);
    }
  }

  function notifyBeforeObservers(obj, keyName, meta) {
    if (meta && meta.isSourceDestroying()) {
      return;
    }

    var eventName = keyName + ':before';
    var listeners = undefined,
        added = undefined;
    if (deferred) {
      listeners = beforeObserverSet.add(obj, keyName, eventName);
      added = _emberMetalEvents.accumulateListeners(obj, eventName, listeners);
      _emberMetalEvents.sendEvent(obj, eventName, [obj, keyName], added);
    } else {
      _emberMetalEvents.sendEvent(obj, eventName, [obj, keyName]);
    }
  }

  function notifyObservers(obj, keyName, meta) {
    if (meta && meta.isSourceDestroying()) {
      return;
    }

    var eventName = keyName + ':change';
    var listeners = undefined;
    if (deferred) {
      listeners = observerSet.add(obj, keyName, eventName);
      _emberMetalEvents.accumulateListeners(obj, eventName, listeners);
    } else {
      _emberMetalEvents.sendEvent(obj, eventName, [obj, keyName]);
    }
  }

  exports.propertyWillChange = propertyWillChange;
  exports.propertyDidChange = propertyDidChange;
  exports.overrideChains = overrideChains;
  exports.beginPropertyChanges = beginPropertyChanges;
  exports.endPropertyChanges = endPropertyChanges;
  exports.changeProperties = changeProperties;
});
enifed('ember-metal/property_get', ['exports', 'ember-metal/debug', 'ember-metal/path_cache'], function (exports, _emberMetalDebug, _emberMetalPath_cache) {
  /**
  @module ember-metal
  */

  'use strict';

  exports.get = get;
  exports._getPath = _getPath;
  exports.getWithDefault = getWithDefault;

  var ALLOWABLE_TYPES = {
    object: true,
    function: true,
    string: true
  };

  // ..........................................................
  // GET AND SET
  //
  // If we are on a platform that supports accessors we can use those.
  // Otherwise simulate accessors by looking up the property directly on the
  // object.

  /**
    Gets the value of a property on an object. If the property is computed,
    the function will be invoked. If the property is not defined but the
    object implements the `unknownProperty` method then that will be invoked.
  
    ```javascript
    Ember.get(obj, "name");
    ```
  
    If you plan to run on IE8 and older browsers then you should use this
    method anytime you want to retrieve a property on an object that you don't
    know for sure is private. (Properties beginning with an underscore '_'
    are considered private.)
  
    On all newer browsers, you only need to use this method to retrieve
    properties if the property might not be defined on the object and you want
    to respect the `unknownProperty` handler. Otherwise you can ignore this
    method.
  
    Note that if the object itself is `undefined`, this method will throw
    an error.
  
    @method get
    @for Ember
    @param {Object} obj The object to retrieve from.
    @param {String} keyName The property key to retrieve
    @return {Object} the property value or `null`.
    @public
  */

  function get(obj, keyName) {

    var value = obj[keyName];
    var desc = value !== null && typeof value === 'object' && value.isDescriptor ? value : undefined;
    var ret = undefined;

    if (desc === undefined && _emberMetalPath_cache.isPath(keyName)) {
      return _getPath(obj, keyName);
    }

    if (desc) {
      return desc.get(obj, keyName);
    } else {
      ret = value;

      if (ret === undefined && 'object' === typeof obj && !(keyName in obj) && 'function' === typeof obj.unknownProperty) {
        return obj.unknownProperty(keyName);
      }

      return ret;
    }
  }

  function _getPath(root, path) {
    var obj = root;
    var parts = path.split('.');

    for (var i = 0; i < parts.length; i++) {
      if (!isGettable(obj)) {
        return undefined;
      }

      obj = get(obj, parts[i]);

      if (obj && obj.isDestroyed) {
        return undefined;
      }
    }

    return obj;
  }

  function isGettable(obj) {
    if (obj == null) {
      return false;
    }

    return ALLOWABLE_TYPES[typeof obj];
  }

  /**
    Retrieves the value of a property from an Object, or a default value in the
    case that the property returns `undefined`.
  
    ```javascript
    Ember.getWithDefault(person, 'lastName', 'Doe');
    ```
  
    @method getWithDefault
    @for Ember
    @param {Object} obj The object to retrieve from.
    @param {String} keyName The name of the property to retrieve
    @param {Object} defaultValue The value to return if the property value is undefined
    @return {Object} The property value or the defaultValue.
    @public
  */

  function getWithDefault(root, key, defaultValue) {
    var value = get(root, key);

    if (value === undefined) {
      return defaultValue;
    }
    return value;
  }

  exports.default = get;
});
enifed('ember-metal/property_set', ['exports', 'ember-utils', 'ember-metal/debug', 'ember-metal/features', 'ember-metal/property_get', 'ember-metal/property_events', 'ember-metal/error', 'ember-metal/path_cache', 'ember-metal/meta'], function (exports, _emberUtils, _emberMetalDebug, _emberMetalFeatures, _emberMetalProperty_get, _emberMetalProperty_events, _emberMetalError, _emberMetalPath_cache, _emberMetalMeta) {
  'use strict';

  exports.set = set;
  exports.trySet = trySet;

  /**
    Sets the value of a property on an object, respecting computed properties
    and notifying observers and other listeners of the change. If the
    property is not defined but the object implements the `setUnknownProperty`
    method then that will be invoked as well.
  
    ```javascript
    Ember.set(obj, "name", value);
    ```
  
    @method set
    @for Ember
    @param {Object} obj The object to modify.
    @param {String} keyName The property key to set
    @param {Object} value The value to set
    @return {Object} the passed value.
    @public
  */

  function set(obj, keyName, value, tolerant) {

    if (_emberMetalPath_cache.isPath(keyName)) {
      return setPath(obj, keyName, value, tolerant);
    }

    var meta = _emberMetalMeta.peekMeta(obj);
    var possibleDesc = obj[keyName];

    var desc = undefined,
        currentValue = undefined;
    if (possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor) {
      desc = possibleDesc;
    } else {
      currentValue = possibleDesc;
    }

    if (desc) {
      /* computed property */
      desc.set(obj, keyName, value);
    } else if (obj.setUnknownProperty && currentValue === undefined && !(keyName in obj)) {
      obj.setUnknownProperty(keyName, value);
      /* unknown property */
    } else if (currentValue === value) {
      /* no change */
      return value;
    } else {
      _emberMetalProperty_events.propertyWillChange(obj, keyName);

      if (false) {
        setWithMandatorySetter(meta, obj, keyName, value);
      } else {
        obj[keyName] = value;
      }

      _emberMetalProperty_events.propertyDidChange(obj, keyName);
    }

    return value;
  }

  if (false) {
    var setWithMandatorySetter = function (meta, obj, keyName, value) {
      if (meta && meta.peekWatching(keyName) > 0) {
        makeEnumerable(obj, keyName);
        meta.writeValue(obj, keyName, value);
      } else {
        obj[keyName] = value;
      }
    };

    var makeEnumerable = function (obj, key) {
      var desc = Object.getOwnPropertyDescriptor(obj, key);

      if (desc && desc.set && desc.set.isMandatorySetter) {
        desc.enumerable = true;
        Object.defineProperty(obj, key, desc);
      }
    };
  }

  function setPath(root, path, value, tolerant) {
    // get the last part of the path
    var keyName = path.slice(path.lastIndexOf('.') + 1);

    // get the first part of the part
    path = path === keyName ? keyName : path.slice(0, path.length - (keyName.length + 1));

    // unless the path is this, look up the first part to
    // get the root
    if (path !== 'this') {
      root = _emberMetalProperty_get._getPath(root, path);
    }

    if (!keyName || keyName.length === 0) {
      throw new _emberMetalError.default('Property set failed: You passed an empty path');
    }

    if (!root) {
      if (tolerant) {
        return;
      } else {
        throw new _emberMetalError.default('Property set failed: object in path "' + path + '" could not be found or was destroyed.');
      }
    }

    return set(root, keyName, value);
  }

  /**
    Error-tolerant form of `Ember.set`. Will not blow up if any part of the
    chain is `undefined`, `null`, or destroyed.
  
    This is primarily used when syncing bindings, which may try to update after
    an object has been destroyed.
  
    @method trySet
    @for Ember
    @param {Object} root The object to modify.
    @param {String} path The property path to set
    @param {Object} value The value to set
    @public
  */

  function trySet(root, path, value) {
    return set(root, path, value, true);
  }
});
enifed("ember-metal/replace", ["exports"], function (exports) {
  "use strict";

  exports.default = replace;
  var splice = Array.prototype.splice;

  function replace(array, idx, amt, objects) {
    var args = [].concat(objects);
    var ret = [];
    // https://code.google.com/p/chromium/issues/detail?id=56588
    var size = 60000;
    var start = idx;
    var ends = amt;
    var count = undefined,
        chunk = undefined;

    while (args.length) {
      count = ends > size ? size : ends;
      if (count <= 0) {
        count = 0;
      }

      chunk = args.splice(0, size);
      chunk = [start, count].concat(chunk);

      start += size;
      ends -= count;

      ret = ret.concat(splice.apply(array, chunk));
    }
    return ret;
  }
});
enifed('ember-metal/run_loop', ['exports', 'ember-utils', 'ember-metal/debug', 'ember-metal/testing', 'ember-metal/error_handler', 'ember-metal/property_events', 'backburner'], function (exports, _emberUtils, _emberMetalDebug, _emberMetalTesting, _emberMetalError_handler, _emberMetalProperty_events, _backburner) {
  'use strict';

  exports.default = run;

  function onBegin(current) {
    run.currentRunLoop = current;
  }

  function onEnd(current, next) {
    run.currentRunLoop = next;
  }

  var onErrorTarget = {
    get onerror() {
      return _emberMetalError_handler.dispatchError;
    },
    set onerror(handler) {
      return _emberMetalError_handler.setOnerror(handler);
    }
  };

  var backburner = new _backburner.default(['sync', 'actions', 'destroy'], {
    GUID_KEY: _emberUtils.GUID_KEY,
    sync: {
      before: _emberMetalProperty_events.beginPropertyChanges,
      after: _emberMetalProperty_events.endPropertyChanges
    },
    defaultQueue: 'actions',
    onBegin: onBegin,
    onEnd: onEnd,
    onErrorTarget: onErrorTarget,
    onErrorMethod: 'onerror'
  });

  // ..........................................................
  // run - this is ideally the only public API the dev sees
  //

  /**
    Runs the passed target and method inside of a RunLoop, ensuring any
    deferred actions including bindings and views updates are flushed at the
    end.
  
    Normally you should not need to invoke this method yourself. However if
    you are implementing raw event handlers when interfacing with other
    libraries or plugins, you should probably wrap all of your code inside this
    call.
  
    ```javascript
    run(function() {
      // code to be executed within a RunLoop
    });
    ```
  
    @class run
    @namespace Ember
    @static
    @constructor
    @param {Object} [target] target of method to call
    @param {Function|String} method Method to invoke.
      May be a function or a string. If you pass a string
      then it will be looked up on the passed target.
    @param {Object} [args*] Any additional arguments you wish to pass to the method.
    @return {Object} return value from invoking the passed function.
    @public
  */

  function run() {
    return backburner.run.apply(backburner, arguments);
  }

  /**
    If no run-loop is present, it creates a new one. If a run loop is
    present it will queue itself to run on the existing run-loops action
    queue.
  
    Please note: This is not for normal usage, and should be used sparingly.
  
    If invoked when not within a run loop:
  
    ```javascript
    run.join(function() {
      // creates a new run-loop
    });
    ```
  
    Alternatively, if called within an existing run loop:
  
    ```javascript
    run(function() {
      // creates a new run-loop
      run.join(function() {
        // joins with the existing run-loop, and queues for invocation on
        // the existing run-loops action queue.
      });
    });
    ```
  
    @method join
    @namespace Ember
    @param {Object} [target] target of method to call
    @param {Function|String} method Method to invoke.
      May be a function or a string. If you pass a string
      then it will be looked up on the passed target.
    @param {Object} [args*] Any additional arguments you wish to pass to the method.
    @return {Object} Return value from invoking the passed function. Please note,
    when called within an existing loop, no return value is possible.
    @public
  */
  run.join = function () {
    return backburner.join.apply(backburner, arguments);
  };

  /**
    Allows you to specify which context to call the specified function in while
    adding the execution of that function to the Ember run loop. This ability
    makes this method a great way to asynchronously integrate third-party libraries
    into your Ember application.
  
    `run.bind` takes two main arguments, the desired context and the function to
    invoke in that context. Any additional arguments will be supplied as arguments
    to the function that is passed in.
  
    Let's use the creation of a TinyMCE component as an example. Currently,
    TinyMCE provides a setup configuration option we can use to do some processing
    after the TinyMCE instance is initialized but before it is actually rendered.
    We can use that setup option to do some additional setup for our component.
    The component itself could look something like the following:
  
    ```javascript
    App.RichTextEditorComponent = Ember.Component.extend({
      initializeTinyMCE: Ember.on('didInsertElement', function() {
        tinymce.init({
          selector: '#' + this.$().prop('id'),
          setup: Ember.run.bind(this, this.setupEditor)
        });
      }),
  
      setupEditor: function(editor) {
        this.set('editor', editor);
  
        editor.on('change', function() {
          console.log('content changed!');
        });
      }
    });
    ```
  
    In this example, we use Ember.run.bind to bind the setupEditor method to the
    context of the App.RichTextEditorComponent and to have the invocation of that
    method be safely handled and executed by the Ember run loop.
  
    @method bind
    @namespace Ember
    @param {Object} [target] target of method to call
    @param {Function|String} method Method to invoke.
      May be a function or a string. If you pass a string
      then it will be looked up on the passed target.
    @param {Object} [args*] Any additional arguments you wish to pass to the method.
    @return {Function} returns a new function that will always have a particular context
    @since 1.4.0
    @public
  */
  run.bind = function () {
    for (var _len = arguments.length, curried = Array(_len), _key = 0; _key < _len; _key++) {
      curried[_key] = arguments[_key];
    }

    return function () {
      for (var _len2 = arguments.length, args = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        args[_key2] = arguments[_key2];
      }

      return run.join.apply(run, curried.concat(args));
    };
  };

  run.backburner = backburner;
  run.currentRunLoop = null;
  run.queues = backburner.queueNames;

  /**
    Begins a new RunLoop. Any deferred actions invoked after the begin will
    be buffered until you invoke a matching call to `run.end()`. This is
    a lower-level way to use a RunLoop instead of using `run()`.
  
    ```javascript
    run.begin();
    // code to be executed within a RunLoop
    run.end();
    ```
  
    @method begin
    @return {void}
    @public
  */
  run.begin = function () {
    backburner.begin();
  };

  /**
    Ends a RunLoop. This must be called sometime after you call
    `run.begin()` to flush any deferred actions. This is a lower-level way
    to use a RunLoop instead of using `run()`.
  
    ```javascript
    run.begin();
    // code to be executed within a RunLoop
    run.end();
    ```
  
    @method end
    @return {void}
    @public
  */
  run.end = function () {
    backburner.end();
  };

  /**
    Array of named queues. This array determines the order in which queues
    are flushed at the end of the RunLoop. You can define your own queues by
    simply adding the queue name to this array. Normally you should not need
    to inspect or modify this property.
  
    @property queues
    @type Array
    @default ['sync', 'actions', 'destroy']
    @private
  */

  /**
    Adds the passed target/method and any optional arguments to the named
    queue to be executed at the end of the RunLoop. If you have not already
    started a RunLoop when calling this method one will be started for you
    automatically.
  
    At the end of a RunLoop, any methods scheduled in this way will be invoked.
    Methods will be invoked in an order matching the named queues defined in
    the `run.queues` property.
  
    ```javascript
    run.schedule('sync', this, function() {
      // this will be executed in the first RunLoop queue, when bindings are synced
      console.log('scheduled on sync queue');
    });
  
    run.schedule('actions', this, function() {
      // this will be executed in the 'actions' queue, after bindings have synced.
      console.log('scheduled on actions queue');
    });
  
    // Note the functions will be run in order based on the run queues order.
    // Output would be:
    //   scheduled on sync queue
    //   scheduled on actions queue
    ```
  
    @method schedule
    @param {String} queue The name of the queue to schedule against.
      Default queues are 'sync' and 'actions'
    @param {Object} [target] target object to use as the context when invoking a method.
    @param {String|Function} method The method to invoke. If you pass a string it
      will be resolved on the target object at the time the scheduled item is
      invoked allowing you to change the target function.
    @param {Object} [arguments*] Optional arguments to be passed to the queued method.
    @return {*} Timer information for use in cancelling, see `run.cancel`.
    @public
  */
  run.schedule = function () /* queue, target, method */{

    return backburner.schedule.apply(backburner, arguments);
  };

  // Used by global test teardown
  run.hasScheduledTimers = function () {
    return backburner.hasTimers();
  };

  // Used by global test teardown
  run.cancelTimers = function () {
    backburner.cancelTimers();
  };

  /**
    Immediately flushes any events scheduled in the 'sync' queue. Bindings
    use this queue so this method is a useful way to immediately force all
    bindings in the application to sync.
  
    You should call this method anytime you need any changed state to propagate
    throughout the app immediately without repainting the UI (which happens
    in the later 'render' queue added by the `ember-views` package).
  
    ```javascript
    run.sync();
    ```
  
    @method sync
    @return {void}
    @private
  */
  run.sync = function () {
    if (backburner.currentInstance) {
      backburner.currentInstance.queues.sync.flush();
    }
  };

  /**
    Invokes the passed target/method and optional arguments after a specified
    period of time. The last parameter of this method must always be a number
    of milliseconds.
  
    You should use this method whenever you need to run some action after a
    period of time instead of using `setTimeout()`. This method will ensure that
    items that expire during the same script execution cycle all execute
    together, which is often more efficient than using a real setTimeout.
  
    ```javascript
    run.later(myContext, function() {
      // code here will execute within a RunLoop in about 500ms with this == myContext
    }, 500);
    ```
  
    @method later
    @param {Object} [target] target of method to invoke
    @param {Function|String} method The method to invoke.
      If you pass a string it will be resolved on the
      target at the time the method is invoked.
    @param {Object} [args*] Optional arguments to pass to the timeout.
    @param {Number} wait Number of milliseconds to wait.
    @return {*} Timer information for use in cancelling, see `run.cancel`.
    @public
  */
  run.later = function () /*target, method*/{
    return backburner.later.apply(backburner, arguments);
  };

  /**
    Schedule a function to run one time during the current RunLoop. This is equivalent
    to calling `scheduleOnce` with the "actions" queue.
  
    @method once
    @param {Object} [target] The target of the method to invoke.
    @param {Function|String} method The method to invoke.
      If you pass a string it will be resolved on the
      target at the time the method is invoked.
    @param {Object} [args*] Optional arguments to pass to the timeout.
    @return {Object} Timer information for use in cancelling, see `run.cancel`.
    @public
  */
  run.once = function () {
    for (var _len3 = arguments.length, args = Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
      args[_key3] = arguments[_key3];
    }

    args.unshift('actions');
    return backburner.scheduleOnce.apply(backburner, args);
  };

  /**
    Schedules a function to run one time in a given queue of the current RunLoop.
    Calling this method with the same queue/target/method combination will have
    no effect (past the initial call).
  
    Note that although you can pass optional arguments these will not be
    considered when looking for duplicates. New arguments will replace previous
    calls.
  
    ```javascript
    function sayHi() {
      console.log('hi');
    }
  
    run(function() {
      run.scheduleOnce('afterRender', myContext, sayHi);
      run.scheduleOnce('afterRender', myContext, sayHi);
      // sayHi will only be executed once, in the afterRender queue of the RunLoop
    });
    ```
  
    Also note that passing an anonymous function to `run.scheduleOnce` will
    not prevent additional calls with an identical anonymous function from
    scheduling the items multiple times, e.g.:
  
    ```javascript
    function scheduleIt() {
      run.scheduleOnce('actions', myContext, function() {
        console.log('Closure');
      });
    }
  
    scheduleIt();
    scheduleIt();
  
    // "Closure" will print twice, even though we're using `run.scheduleOnce`,
    // because the function we pass to it is anonymous and won't match the
    // previously scheduled operation.
    ```
  
    Available queues, and their order, can be found at `run.queues`
  
    @method scheduleOnce
    @param {String} [queue] The name of the queue to schedule against. Default queues are 'sync' and 'actions'.
    @param {Object} [target] The target of the method to invoke.
    @param {Function|String} method The method to invoke.
      If you pass a string it will be resolved on the
      target at the time the method is invoked.
    @param {Object} [args*] Optional arguments to pass to the timeout.
    @return {Object} Timer information for use in cancelling, see `run.cancel`.
    @public
  */
  run.scheduleOnce = function () /*queue, target, method*/{
    return backburner.scheduleOnce.apply(backburner, arguments);
  };

  /**
    Schedules an item to run from within a separate run loop, after
    control has been returned to the system. This is equivalent to calling
    `run.later` with a wait time of 1ms.
  
    ```javascript
    run.next(myContext, function() {
      // code to be executed in the next run loop,
      // which will be scheduled after the current one
    });
    ```
  
    Multiple operations scheduled with `run.next` will coalesce
    into the same later run loop, along with any other operations
    scheduled by `run.later` that expire right around the same
    time that `run.next` operations will fire.
  
    Note that there are often alternatives to using `run.next`.
    For instance, if you'd like to schedule an operation to happen
    after all DOM element operations have completed within the current
    run loop, you can make use of the `afterRender` run loop queue (added
    by the `ember-views` package, along with the preceding `render` queue
    where all the DOM element operations happen).
  
    Example:
  
    ```javascript
    export default Ember.Component.extend({
      didInsertElement() {
        this._super(...arguments);
        run.scheduleOnce('afterRender', this, 'processChildElements');
      },
  
      processChildElements() {
        // ... do something with component's child component
        // elements after they've finished rendering, which
        // can't be done within this component's
        // `didInsertElement` hook because that gets run
        // before the child elements have been added to the DOM.
      }
    });
    ```
  
    One benefit of the above approach compared to using `run.next` is
    that you will be able to perform DOM/CSS operations before unprocessed
    elements are rendered to the screen, which may prevent flickering or
    other artifacts caused by delaying processing until after rendering.
  
    The other major benefit to the above approach is that `run.next`
    introduces an element of non-determinism, which can make things much
    harder to test, due to its reliance on `setTimeout`; it's much harder
    to guarantee the order of scheduled operations when they are scheduled
    outside of the current run loop, i.e. with `run.next`.
  
    @method next
    @param {Object} [target] target of method to invoke
    @param {Function|String} method The method to invoke.
      If you pass a string it will be resolved on the
      target at the time the method is invoked.
    @param {Object} [args*] Optional arguments to pass to the timeout.
    @return {Object} Timer information for use in cancelling, see `run.cancel`.
    @public
  */
  run.next = function () {
    for (var _len4 = arguments.length, args = Array(_len4), _key4 = 0; _key4 < _len4; _key4++) {
      args[_key4] = arguments[_key4];
    }

    args.push(1);
    return backburner.later.apply(backburner, args);
  };

  /**
    Cancels a scheduled item. Must be a value returned by `run.later()`,
    `run.once()`, `run.scheduleOnce()`, `run.next()`, `run.debounce()`, or
    `run.throttle()`.
  
    ```javascript
    let runNext = run.next(myContext, function() {
      // will not be executed
    });
  
    run.cancel(runNext);
  
    let runLater = run.later(myContext, function() {
      // will not be executed
    }, 500);
  
    run.cancel(runLater);
  
    let runScheduleOnce = run.scheduleOnce('afterRender', myContext, function() {
      // will not be executed
    });
  
    run.cancel(runScheduleOnce);
  
    let runOnce = run.once(myContext, function() {
      // will not be executed
    });
  
    run.cancel(runOnce);
  
    let throttle = run.throttle(myContext, function() {
      // will not be executed
    }, 1, false);
  
    run.cancel(throttle);
  
    let debounce = run.debounce(myContext, function() {
      // will not be executed
    }, 1);
  
    run.cancel(debounce);
  
    let debounceImmediate = run.debounce(myContext, function() {
      // will be executed since we passed in true (immediate)
    }, 100, true);
  
    // the 100ms delay until this method can be called again will be cancelled
    run.cancel(debounceImmediate);
    ```
  
    @method cancel
    @param {Object} timer Timer object to cancel
    @return {Boolean} true if cancelled or false/undefined if it wasn't found
    @public
  */
  run.cancel = function (timer) {
    return backburner.cancel(timer);
  };

  /**
    Delay calling the target method until the debounce period has elapsed
    with no additional debounce calls. If `debounce` is called again before
    the specified time has elapsed, the timer is reset and the entire period
    must pass again before the target method is called.
  
    This method should be used when an event may be called multiple times
    but the action should only be called once when the event is done firing.
    A common example is for scroll events where you only want updates to
    happen once scrolling has ceased.
  
    ```javascript
    function whoRan() {
      console.log(this.name + ' ran.');
    }
  
    let myContext = { name: 'debounce' };
  
    run.debounce(myContext, whoRan, 150);
  
    // less than 150ms passes
    run.debounce(myContext, whoRan, 150);
  
    // 150ms passes
    // whoRan is invoked with context myContext
    // console logs 'debounce ran.' one time.
    ```
  
    Immediate allows you to run the function immediately, but debounce
    other calls for this function until the wait time has elapsed. If
    `debounce` is called again before the specified time has elapsed,
    the timer is reset and the entire period must pass again before
    the method can be called again.
  
    ```javascript
    function whoRan() {
      console.log(this.name + ' ran.');
    }
  
    let myContext = { name: 'debounce' };
  
    run.debounce(myContext, whoRan, 150, true);
  
    // console logs 'debounce ran.' one time immediately.
    // 100ms passes
    run.debounce(myContext, whoRan, 150, true);
  
    // 150ms passes and nothing else is logged to the console and
    // the debouncee is no longer being watched
    run.debounce(myContext, whoRan, 150, true);
  
    // console logs 'debounce ran.' one time immediately.
    // 150ms passes and nothing else is logged to the console and
    // the debouncee is no longer being watched
  
    ```
  
    @method debounce
    @param {Object} [target] target of method to invoke
    @param {Function|String} method The method to invoke.
      May be a function or a string. If you pass a string
      then it will be looked up on the passed target.
    @param {Object} [args*] Optional arguments to pass to the timeout.
    @param {Number} wait Number of milliseconds to wait.
    @param {Boolean} immediate Trigger the function on the leading instead
      of the trailing edge of the wait interval. Defaults to false.
    @return {Array} Timer information for use in cancelling, see `run.cancel`.
    @public
  */
  run.debounce = function () {
    return backburner.debounce.apply(backburner, arguments);
  };

  /**
    Ensure that the target method is never called more frequently than
    the specified spacing period. The target method is called immediately.
  
    ```javascript
    function whoRan() {
      console.log(this.name + ' ran.');
    }
  
    let myContext = { name: 'throttle' };
  
    run.throttle(myContext, whoRan, 150);
    // whoRan is invoked with context myContext
    // console logs 'throttle ran.'
  
    // 50ms passes
    run.throttle(myContext, whoRan, 150);
  
    // 50ms passes
    run.throttle(myContext, whoRan, 150);
  
    // 150ms passes
    run.throttle(myContext, whoRan, 150);
    // whoRan is invoked with context myContext
    // console logs 'throttle ran.'
    ```
  
    @method throttle
    @param {Object} [target] target of method to invoke
    @param {Function|String} method The method to invoke.
      May be a function or a string. If you pass a string
      then it will be looked up on the passed target.
    @param {Object} [args*] Optional arguments to pass to the timeout.
    @param {Number} spacing Number of milliseconds to space out requests.
    @param {Boolean} immediate Trigger the function on the leading instead
      of the trailing edge of the wait interval. Defaults to true.
    @return {Array} Timer information for use in cancelling, see `run.cancel`.
    @public
  */
  run.throttle = function () {
    return backburner.throttle.apply(backburner, arguments);
  };

  /**
    Add a new named queue after the specified queue.
  
    The queue to add will only be added once.
  
    @method _addQueue
    @param {String} name the name of the queue to add.
    @param {String} after the name of the queue to add after.
    @private
  */
  run._addQueue = function (name, after) {
    if (run.queues.indexOf(name) === -1) {
      run.queues.splice(run.queues.indexOf(after) + 1, 0, name);
    }
  };
});
enifed('ember-metal/set_properties', ['exports', 'ember-metal/property_events', 'ember-metal/property_set'], function (exports, _emberMetalProperty_events, _emberMetalProperty_set) {
  'use strict';

  exports.default = setProperties;

  /**
    Set a list of properties on an object. These properties are set inside
    a single `beginPropertyChanges` and `endPropertyChanges` batch, so
    observers will be buffered.
  
    ```javascript
    let anObject = Ember.Object.create();
  
    anObject.setProperties({
      firstName: 'Stanley',
      lastName: 'Stuart',
      age: 21
    });
    ```
  
    @method setProperties
    @param obj
    @param {Object} properties
    @return properties
    @public
  */

  function setProperties(obj, properties) {
    if (!properties || typeof properties !== 'object') {
      return properties;
    }
    _emberMetalProperty_events.changeProperties(function () {
      var props = Object.keys(properties);
      var propertyName = undefined;

      for (var i = 0; i < props.length; i++) {
        propertyName = props[i];

        _emberMetalProperty_set.set(obj, propertyName, properties[propertyName]);
      }
    });
    return properties;
  }
});
enifed('ember-metal/tags', ['exports', 'glimmer-reference', 'ember-metal/meta', 'require', 'ember-metal/is_proxy'], function (exports, _glimmerReference, _emberMetalMeta, _require, _emberMetalIs_proxy) {
  'use strict';

  exports.setHasViews = setHasViews;
  exports.tagForProperty = tagForProperty;
  exports.tagFor = tagFor;
  exports.markObjectAsDirty = markObjectAsDirty;

  var hasViews = function () {
    return false;
  };

  function setHasViews(fn) {
    hasViews = fn;
  }

  function makeTag() {
    return new _glimmerReference.DirtyableTag();
  }

  function tagForProperty(object, propertyKey, _meta) {
    if (_emberMetalIs_proxy.isProxy(object)) {
      return tagFor(object, _meta);
    }

    if (typeof object === 'object' && object) {
      var meta = _meta || _emberMetalMeta.meta(object);
      var tags = meta.writableTags();
      var tag = tags[propertyKey];
      if (tag) {
        return tag;
      }

      return tags[propertyKey] = makeTag();
    } else {
      return _glimmerReference.CONSTANT_TAG;
    }
  }

  function tagFor(object, _meta) {
    if (typeof object === 'object' && object) {
      var meta = _meta || _emberMetalMeta.meta(object);
      return meta.writableTag(makeTag);
    } else {
      return _glimmerReference.CONSTANT_TAG;
    }
  }

  function markObjectAsDirty(meta, propertyKey) {
    var objectTag = meta && meta.readableTag();

    if (objectTag) {
      objectTag.dirty();
    }

    var tags = meta && meta.readableTags();
    var propertyTag = tags && tags[propertyKey];

    if (propertyTag) {
      propertyTag.dirty();
    }

    if (objectTag || propertyTag) {
      ensureRunloop();
    }
  }

  var run = undefined;

  function K() {}

  function ensureRunloop() {
    if (!run) {
      run = _require.default('ember-metal/run_loop').default;
    }

    if (hasViews() && !run.backburner.currentInstance) {
      run.schedule('actions', K);
    }
  }
});
enifed("ember-metal/testing", ["exports"], function (exports) {
  "use strict";

  exports.isTesting = isTesting;
  exports.setTesting = setTesting;
  var testing = false;

  function isTesting() {
    return testing;
  }

  function setTesting(value) {
    testing = !!value;
  }
});
enifed('ember-metal/transaction', ['exports', 'ember-metal/meta', 'ember-metal/debug', 'ember-metal/features'], function (exports, _emberMetalMeta, _emberMetalDebug, _emberMetalFeatures) {
  'use strict';

  var runInTransaction = undefined,
      didRender = undefined,
      assertNotRendered = undefined;

  var raise = _emberMetalDebug.assert;
  if (false) {
    raise = function (message, test) {};
  }

  var implication = undefined;
  if (false) {
    implication = 'will be removed in Ember 3.0.';
  } else if (false) {
    implication = 'is no longer supported. See https://github.com/emberjs/ember.js/issues/13948 for more details.';
  }

  if (false || false) {
    (function () {
      var counter = 0;
      var inTransaction = false;
      var shouldReflush = undefined;
      var debugStack = undefined;

      exports.default = runInTransaction = function (context, methodName) {
        shouldReflush = false;
        inTransaction = true;

        context[methodName]();
        inTransaction = false;
        counter++;
        return shouldReflush;
      };

      exports.didRender = didRender = function (object, key, reference) {
        if (!inTransaction) {
          return;
        }
        var meta = _emberMetalMeta.meta(object);
        var lastRendered = meta.writableLastRendered();
        lastRendered[key] = counter;
      };

      exports.assertNotRendered = assertNotRendered = function (object, key, _meta) {
        var meta = _meta || _emberMetalMeta.meta(object);
        var lastRendered = meta.readableLastRendered();

        if (lastRendered && lastRendered[key] === counter) {
          raise((function () {
            var templateMap = meta.readableLastRenderedTemplateMap();
            var lastRenderedIn = templateMap[key];
            var currentlyIn = debugStack.peek();

            var referenceMap = meta.readableLastRenderedReferenceMap();
            var lastRef = referenceMap[key];
            var parts = [];
            var label = undefined;

            if (lastRef) {
              while (lastRef && lastRef._propertyKey) {
                parts.unshift(lastRef._propertyKey);
                lastRef = lastRef._parentReference;
              }

              label = parts.join('.');
            } else {
              label = 'the same value';
            }

            return 'You modified "' + label + '" twice on ' + object + ' in a single render. It was rendered in ' + lastRenderedIn + ' and modified in ' + currentlyIn + '. This was unreliable and slow in Ember 1.x and ' + implication;
          })(), false);

          shouldReflush = true;
        }
      };
    })();
  } else {
    exports.default = runInTransaction = function () {
      throw new Error('Cannot call runInTransaction without Glimmer');
    };

    exports.didRender = didRender = function () {
      throw new Error('Cannot call didRender without Glimmer');
    };

    exports.assertNotRendered = assertNotRendered = function () {
      throw new Error('Cannot call assertNotRendered without Glimmer');
    };
  }

  exports.default = runInTransaction;
  exports.didRender = didRender;
  exports.assertNotRendered = assertNotRendered;
});
enifed('ember-metal/watch_key', ['exports', 'ember-utils', 'ember-metal/features', 'ember-metal/meta', 'ember-metal/properties'], function (exports, _emberUtils, _emberMetalFeatures, _emberMetalMeta, _emberMetalProperties) {
  'use strict';

  exports.watchKey = watchKey;
  exports.unwatchKey = unwatchKey;

  var handleMandatorySetter = undefined;

  function watchKey(obj, keyName, meta) {
    if (typeof obj !== 'object' || obj === null) {
      return;
    }
    var m = meta || _emberMetalMeta.meta(obj);

    // activate watching first time
    if (!m.peekWatching(keyName)) {
      m.writeWatching(keyName, 1);

      var possibleDesc = obj[keyName];
      var desc = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor ? possibleDesc : undefined;
      if (desc && desc.willWatch) {
        desc.willWatch(obj, keyName);
      }

      if ('function' === typeof obj.willWatchProperty) {
        obj.willWatchProperty(keyName);
      }

      if (false) {
        // NOTE: this is dropped for prod + minified builds
        handleMandatorySetter(m, obj, keyName);
      }
    } else {
      m.writeWatching(keyName, (m.peekWatching(keyName) || 0) + 1);
    }
  }

  if (false) {
    (function () {
      var hasOwnProperty = function (obj, key) {
        return Object.prototype.hasOwnProperty.call(obj, key);
      };

      var propertyIsEnumerable = function (obj, key) {
        return Object.prototype.propertyIsEnumerable.call(obj, key);
      };

      // Future traveler, although this code looks scary. It merely exists in
      // development to aid in development asertions. Production builds of
      // ember strip this entire block out
      handleMandatorySetter = function handleMandatorySetter(m, obj, keyName) {
        var descriptor = _emberUtils.lookupDescriptor(obj, keyName);
        var configurable = descriptor ? descriptor.configurable : true;
        var isWritable = descriptor ? descriptor.writable : true;
        var hasValue = descriptor ? 'value' in descriptor : true;
        var possibleDesc = descriptor && descriptor.value;
        var isDescriptor = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor;

        if (isDescriptor) {
          return;
        }

        // this x in Y deopts, so keeping it in this function is better;
        if (configurable && isWritable && hasValue && keyName in obj) {
          var desc = {
            configurable: true,
            set: _emberMetalProperties.MANDATORY_SETTER_FUNCTION(keyName),
            enumerable: propertyIsEnumerable(obj, keyName),
            get: undefined
          };

          if (hasOwnProperty(obj, keyName)) {
            m.writeValues(keyName, obj[keyName]);
            desc.get = _emberMetalProperties.DEFAULT_GETTER_FUNCTION(keyName);
          } else {
            desc.get = _emberMetalProperties.INHERITING_GETTER_FUNCTION(keyName);
          }

          Object.defineProperty(obj, keyName, desc);
        }
      };
    })();
  }

  function unwatchKey(obj, keyName, _meta) {
    if (typeof obj !== 'object' || obj === null) {
      return;
    }
    var meta = _meta || _emberMetalMeta.meta(obj);

    // do nothing of this object has already been destroyed
    if (meta.isSourceDestroyed()) {
      return;
    }

    var count = meta.peekWatching(keyName);
    if (count === 1) {
      meta.writeWatching(keyName, 0);

      var possibleDesc = obj[keyName];
      var desc = possibleDesc !== null && typeof possibleDesc === 'object' && possibleDesc.isDescriptor ? possibleDesc : undefined;

      if (desc && desc.didUnwatch) {
        desc.didUnwatch(obj, keyName);
      }

      if ('function' === typeof obj.didUnwatchProperty) {
        obj.didUnwatchProperty(keyName);
      }

      if (false) {
        // It is true, the following code looks quite WAT. But have no fear, It
        // exists purely to improve development ergonomics and is removed from
        // ember.min.js and ember.prod.js builds.
        //
        // Some further context: Once a property is watched by ember, bypassing `set`
        // for mutation, will bypass observation. This code exists to assert when
        // that occurs, and attempt to provide more helpful feedback. The alternative
        // is tricky to debug partially observable properties.
        if (!desc && keyName in obj) {
          var maybeMandatoryDescriptor = _emberUtils.lookupDescriptor(obj, keyName);

          if (maybeMandatoryDescriptor.set && maybeMandatoryDescriptor.set.isMandatorySetter) {
            if (maybeMandatoryDescriptor.get && maybeMandatoryDescriptor.get.isInheritingGetter) {
              var possibleValue = meta.readInheritedValue('values', keyName);
              if (possibleValue === _emberMetalMeta.UNDEFINED) {
                delete obj[keyName];
                return;
              }
            }

            Object.defineProperty(obj, keyName, {
              configurable: true,
              enumerable: Object.prototype.propertyIsEnumerable.call(obj, keyName),
              writable: true,
              value: meta.peekValues(keyName)
            });
            meta.deleteFromValues(keyName);
          }
        }
      }
    } else if (count > 1) {
      meta.writeWatching(keyName, count - 1);
    }
  }
});
enifed('ember-metal/watch_path', ['exports', 'ember-metal/meta', 'ember-metal/chains'], function (exports, _emberMetalMeta, _emberMetalChains) {
  'use strict';

  exports.makeChainNode = makeChainNode;
  exports.watchPath = watchPath;
  exports.unwatchPath = unwatchPath;

  // get the chains for the current object. If the current object has
  // chains inherited from the proto they will be cloned and reconfigured for
  // the current object.
  function chainsFor(obj, meta) {
    return (meta || _emberMetalMeta.meta(obj)).writableChains(makeChainNode);
  }

  function makeChainNode(obj) {
    return new _emberMetalChains.ChainNode(null, null, obj);
  }

  function watchPath(obj, keyPath, meta) {
    if (typeof obj !== 'object' || obj === null) {
      return;
    }
    var m = meta || _emberMetalMeta.meta(obj);
    var counter = m.peekWatching(keyPath) || 0;
    if (!counter) {
      // activate watching first time
      m.writeWatching(keyPath, 1);
      chainsFor(obj, m).add(keyPath);
    } else {
      m.writeWatching(keyPath, counter + 1);
    }
  }

  function unwatchPath(obj, keyPath, meta) {
    if (typeof obj !== 'object' || obj === null) {
      return;
    }
    var m = meta || _emberMetalMeta.meta(obj);
    var counter = m.peekWatching(keyPath) || 0;

    if (counter === 1) {
      m.writeWatching(keyPath, 0);
      chainsFor(obj, m).remove(keyPath);
    } else if (counter > 1) {
      m.writeWatching(keyPath, counter - 1);
    }
  }
});
enifed('ember-metal/watching', ['exports', 'ember-metal/watch_key', 'ember-metal/watch_path', 'ember-metal/path_cache', 'ember-metal/meta'], function (exports, _emberMetalWatch_key, _emberMetalWatch_path, _emberMetalPath_cache, _emberMetalMeta) {
  /**
  @module ember-metal
  */

  'use strict';

  exports.isWatching = isWatching;
  exports.watcherCount = watcherCount;
  exports.unwatch = unwatch;
  exports.destroy = destroy;

  /**
    Starts watching a property on an object. Whenever the property changes,
    invokes `Ember.propertyWillChange` and `Ember.propertyDidChange`. This is the
    primitive used by observers and dependent keys; usually you will never call
    this method directly but instead use higher level methods like
    `Ember.addObserver()`
  
    @private
    @method watch
    @for Ember
    @param obj
    @param {String} _keyPath
  */
  function watch(obj, _keyPath, m) {
    if (!_emberMetalPath_cache.isPath(_keyPath)) {
      _emberMetalWatch_key.watchKey(obj, _keyPath, m);
    } else {
      _emberMetalWatch_path.watchPath(obj, _keyPath, m);
    }
  }

  exports.watch = watch;

  function isWatching(obj, key) {
    if (typeof obj !== 'object' || obj === null) {
      return false;
    }
    var meta = _emberMetalMeta.peekMeta(obj);
    return (meta && meta.peekWatching(key)) > 0;
  }

  function watcherCount(obj, key) {
    var meta = _emberMetalMeta.peekMeta(obj);
    return meta && meta.peekWatching(key) || 0;
  }

  function unwatch(obj, _keyPath, m) {
    if (!_emberMetalPath_cache.isPath(_keyPath)) {
      _emberMetalWatch_key.unwatchKey(obj, _keyPath, m);
    } else {
      _emberMetalWatch_path.unwatchPath(obj, _keyPath, m);
    }
  }

  /**
    Tears down the meta on an object so that it can be garbage collected.
    Multiple calls will have no effect.
  
    @method destroy
    @for Ember
    @param {Object} obj  the object to destroy
    @return {void}
    @private
  */

  function destroy(obj) {
    _emberMetalMeta.deleteMeta(obj);
  }
});
enifed('ember-metal/weak_map', ['exports', 'ember-utils', 'ember-metal/meta'], function (exports, _emberUtils, _emberMetalMeta) {
  'use strict';

  exports.default = WeakMap;

  var id = 0;

  // Returns whether Type(value) is Object according to the terminology in the spec
  function isObject(value) {
    return typeof value === 'object' && value !== null || typeof value === 'function';
  }

  /*
   * @class Ember.WeakMap
   * @public
   * @category ember-metal-weakmap
   *
   * A partial polyfill for [WeakMap](http://www.ecma-international.org/ecma-262/6.0/#sec-weakmap-objects).
   *
   * There is a small but important caveat. This implementation assumes that the
   * weak map will live longer (in the sense of garbage collection) than all of its
   * keys, otherwise it is possible to leak the values stored in the weak map. In
   * practice, most use cases satisfy this limitation which is why it is included
   * in ember-metal.
   */

  function WeakMap(iterable) {
    if (!(this instanceof WeakMap)) {
      throw new TypeError('Constructor WeakMap requires \'new\'');
    }

    this._id = _emberUtils.GUID_KEY + id++;

    if (iterable === null || iterable === undefined) {
      return;
    } else if (Array.isArray(iterable)) {
      for (var i = 0; i < iterable.length; i++) {
        var _iterable$i = iterable[i];
        var key = _iterable$i[0];
        var value = _iterable$i[1];

        this.set(key, value);
      }
    } else {
      throw new TypeError('The weak map constructor polyfill only supports an array argument');
    }
  }

  /*
   * @method get
   * @param key {Object | Function}
   * @return {Any} stored value
   */
  WeakMap.prototype.get = function (obj) {
    if (!isObject(obj)) {
      return undefined;
    }

    var meta = _emberMetalMeta.peekMeta(obj);
    if (meta) {
      var map = meta.readableWeak();
      if (map) {
        if (map[this._id] === _emberMetalMeta.UNDEFINED) {
          return undefined;
        }

        return map[this._id];
      }
    }
  };

  /*
   * @method set
   * @param key {Object | Function}
   * @param value {Any}
   * @return {WeakMap} the weak map
   */
  WeakMap.prototype.set = function (obj, value) {
    if (!isObject(obj)) {
      throw new TypeError('Invalid value used as weak map key');
    }

    if (value === undefined) {
      value = _emberMetalMeta.UNDEFINED;
    }

    _emberMetalMeta.meta(obj).writableWeak()[this._id] = value;

    return this;
  };

  /*
   * @method has
   * @param key {Object | Function}
   * @return {boolean} if the key exists
   */
  WeakMap.prototype.has = function (obj) {
    if (!isObject(obj)) {
      return false;
    }

    var meta = _emberMetalMeta.peekMeta(obj);
    if (meta) {
      var map = meta.readableWeak();
      if (map) {
        return map[this._id] !== undefined;
      }
    }

    return false;
  };

  /*
   * @method delete
   * @param key {Object | Function}
   * @return {boolean} if the key was deleted
   */
  WeakMap.prototype.delete = function (obj) {
    if (this.has(obj)) {
      delete _emberMetalMeta.meta(obj).writableWeak()[this._id];
      return true;
    } else {
      return false;
    }
  };

  /*
   * @method toString
   * @return {String}
   */
  WeakMap.prototype.toString = function () {
    return '[object WeakMap]';
  };
});
enifed('ember-routing/ext/controller', ['exports', 'ember-metal', 'ember-runtime', 'ember-routing/utils'], function (exports, _emberMetal, _emberRuntime, _emberRoutingUtils) {
  'use strict';

  /**
  @module ember
  @submodule ember-routing
  */

  _emberRuntime.ControllerMixin.reopen({
    concatenatedProperties: ['queryParams'],

    /**
      Defines which query parameters the controller accepts.
      If you give the names `['category','page']` it will bind
      the values of these query parameters to the variables
      `this.category` and `this.page`
       @property queryParams
      @public
    */
    queryParams: null,

    /**
     This property is updated to various different callback functions depending on
     the current "state" of the backing route. It is used by
     `Ember.Controller.prototype._qpChanged`.
      The methods backing each state can be found in the `Ember.Route.prototype._qp` computed
     property return value (the `.states` property). The current values are listed here for
     the sanity of future travelers:
      * `inactive` - This state is used when this controller instance is not part of the active
       route heirarchy. Set in `Ember.Route.prototype._reset` (a `router.js` microlib hook) and
       `Ember.Route.prototype.actions.finalizeQueryParamChange`.
     * `active` - This state is used when this controller instance is part of the active
       route heirarchy. Set in `Ember.Route.prototype.actions.finalizeQueryParamChange`.
     * `allowOverrides` - This state is used in `Ember.Route.prototype.setup` (`route.js` microlib hook).
       @method _qpDelegate
      @private
    */
    _qpDelegate: null, // set by route

    /**
     During `Ember.Route#setup` observers are created to invoke this method
     when any of the query params declared in `Ember.Controller#queryParams` property
     are changed.
       When invoked this method uses the currently active query param update delegate
     (see `Ember.Controller.prototype._qpDelegate` for details) and invokes it with
     the QP key/value being changed.
       @method _qpChanged
      @private
    */
    _qpChanged: function (controller, _prop) {
      var prop = _prop.substr(0, _prop.length - 3);

      var delegate = controller._qpDelegate;
      var value = _emberMetal.get(controller, prop);
      delegate(prop, value);
    },

    /**
      Transition the application into another route. The route may
      be either a single route or route path:
       ```javascript
      aController.transitionToRoute('blogPosts');
      aController.transitionToRoute('blogPosts.recentEntries');
      ```
       Optionally supply a model for the route in question. The model
      will be serialized into the URL using the `serialize` hook of
      the route:
       ```javascript
      aController.transitionToRoute('blogPost', aPost);
      ```
       If a literal is passed (such as a number or a string), it will
      be treated as an identifier instead. In this case, the `model`
      hook of the route will be triggered:
       ```javascript
      aController.transitionToRoute('blogPost', 1);
      ```
       Multiple models will be applied last to first recursively up the
      route tree.
       ```javascript
      App.Router.map(function() {
        this.route('blogPost', { path: ':blogPostId' }, function() {
          this.route('blogComment', { path: ':blogCommentId', resetNamespace: true });
        });
      });
       aController.transitionToRoute('blogComment', aPost, aComment);
      aController.transitionToRoute('blogComment', 1, 13);
      ```
       It is also possible to pass a URL (a string that starts with a
      `/`). This is intended for testing and debugging purposes and
      should rarely be used in production code.
       ```javascript
      aController.transitionToRoute('/');
      aController.transitionToRoute('/blog/post/1/comment/13');
      aController.transitionToRoute('/blog/posts?sort=title');
      ```
       An options hash with a `queryParams` property may be provided as
      the final argument to add query parameters to the destination URL.
       ```javascript
      aController.transitionToRoute('blogPost', 1, {
        queryParams: { showComments: 'true' }
      });
       // if you just want to transition the query parameters without changing the route
      aController.transitionToRoute({ queryParams: { sort: 'date' } });
      ```
       See also [replaceRoute](/api/classes/Ember.ControllerMixin.html#method_replaceRoute).
       @param {String} name the name of the route or a URL
      @param {...Object} models the model(s) or identifier(s) to be used
        while transitioning to the route.
      @param {Object} [options] optional hash with a queryParams property
        containing a mapping of query parameters
      @for Ember.ControllerMixin
      @method transitionToRoute
      @public
    */
    transitionToRoute: function () {
      // target may be either another controller or a router
      var target = _emberMetal.get(this, 'target');
      var method = target.transitionToRoute || target.transitionTo;

      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      return method.apply(target, _emberRoutingUtils.prefixRouteNameArg(this, args));
    },

    /**
      Transition into another route while replacing the current URL, if possible.
      This will replace the current history entry instead of adding a new one.
      Beside that, it is identical to `transitionToRoute` in all other respects.
       ```javascript
      aController.replaceRoute('blogPosts');
      aController.replaceRoute('blogPosts.recentEntries');
      ```
       Optionally supply a model for the route in question. The model
      will be serialized into the URL using the `serialize` hook of
      the route:
       ```javascript
      aController.replaceRoute('blogPost', aPost);
      ```
       If a literal is passed (such as a number or a string), it will
      be treated as an identifier instead. In this case, the `model`
      hook of the route will be triggered:
       ```javascript
      aController.replaceRoute('blogPost', 1);
      ```
       Multiple models will be applied last to first recursively up the
      route tree.
       ```javascript
      App.Router.map(function() {
        this.route('blogPost', { path: ':blogPostId' }, function() {
          this.route('blogComment', { path: ':blogCommentId', resetNamespace: true });
        });
      });
       aController.replaceRoute('blogComment', aPost, aComment);
      aController.replaceRoute('blogComment', 1, 13);
      ```
       It is also possible to pass a URL (a string that starts with a
      `/`). This is intended for testing and debugging purposes and
      should rarely be used in production code.
       ```javascript
      aController.replaceRoute('/');
      aController.replaceRoute('/blog/post/1/comment/13');
      ```
       @param {String} name the name of the route or a URL
      @param {...Object} models the model(s) or identifier(s) to be used
      while transitioning to the route.
      @for Ember.ControllerMixin
      @method replaceRoute
      @public
    */
    replaceRoute: function () {
      // target may be either another controller or a router
      var target = _emberMetal.get(this, 'target');
      var method = target.replaceRoute || target.replaceWith;

      for (var _len2 = arguments.length, args = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        args[_key2] = arguments[_key2];
      }

      return method.apply(target, _emberRoutingUtils.prefixRouteNameArg(target, args));
    }
  });

  exports.default = _emberRuntime.ControllerMixin;
});
enifed('ember-routing/ext/run_loop', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  /**
  @module ember
  @submodule ember-views
  */

  // Add a new named queue after the 'actions' queue (where RSVP promises
  // resolve), which is used in router transitions to prevent unnecessary
  // loading state entry if all context promises resolve on the
  // 'actions' queue first.
  _emberMetal.run._addQueue('routerTransitions', 'actions');
});
enifed('ember-routing/index', ['exports', 'ember-routing/ext/run_loop', 'ember-routing/ext/controller', 'ember-routing/location/api', 'ember-routing/location/none_location', 'ember-routing/location/hash_location', 'ember-routing/location/history_location', 'ember-routing/location/auto_location', 'ember-routing/system/generate_controller', 'ember-routing/system/controller_for', 'ember-routing/system/dsl', 'ember-routing/system/router', 'ember-routing/system/route', 'ember-routing/system/query_params', 'ember-routing/services/routing', 'ember-routing/system/cache'], function (exports, _emberRoutingExtRun_loop, _emberRoutingExtController, _emberRoutingLocationApi, _emberRoutingLocationNone_location, _emberRoutingLocationHash_location, _emberRoutingLocationHistory_location, _emberRoutingLocationAuto_location, _emberRoutingSystemGenerate_controller, _emberRoutingSystemController_for, _emberRoutingSystemDsl, _emberRoutingSystemRouter, _emberRoutingSystemRoute, _emberRoutingSystemQuery_params, _emberRoutingServicesRouting, _emberRoutingSystemCache) {
  /**
  @module ember
  @submodule ember-routing
  */

  // ES6TODO: Cleanup modules with side-effects below
  'use strict';

  exports.Location = _emberRoutingLocationApi.default;
  exports.NoneLocation = _emberRoutingLocationNone_location.default;
  exports.HashLocation = _emberRoutingLocationHash_location.default;
  exports.HistoryLocation = _emberRoutingLocationHistory_location.default;
  exports.AutoLocation = _emberRoutingLocationAuto_location.default;
  exports.generateController = _emberRoutingSystemGenerate_controller.default;
  exports.generateControllerFactory = _emberRoutingSystemGenerate_controller.generateControllerFactory;
  exports.controllerFor = _emberRoutingSystemController_for.default;
  exports.RouterDSL = _emberRoutingSystemDsl.default;
  exports.Router = _emberRoutingSystemRouter.default;
  exports.Route = _emberRoutingSystemRoute.default;
  exports.QueryParams = _emberRoutingSystemQuery_params.default;
  exports.RoutingService = _emberRoutingServicesRouting.default;
  exports.BucketCache = _emberRoutingSystemCache.default;
});
enifed('ember-routing/location/api', ['exports', 'ember-metal', 'ember-environment', 'ember-routing/location/util'], function (exports, _emberMetal, _emberEnvironment, _emberRoutingLocationUtil) {
  'use strict';

  /**
  @module ember
  @submodule ember-routing
  */

  /**
    Ember.Location returns an instance of the correct implementation of
    the `location` API.
  
    ## Implementations
  
    You can pass an implementation name (`hash`, `history`, `none`) to force a
    particular implementation to be used in your application.
  
    ### HashLocation
  
    Using `HashLocation` results in URLs with a `#` (hash sign) separating the
    server side URL portion of the URL from the portion that is used by Ember.
    This relies upon the `hashchange` event existing in the browser.
  
    Example:
  
    ```javascript
    App.Router.map(function() {
      this.route('posts', function() {
        this.route('new');
      });
    });
  
    App.Router.reopen({
      location: 'hash'
    });
    ```
  
    This will result in a posts.new url of `/#/posts/new`.
  
    ### HistoryLocation
  
    Using `HistoryLocation` results in URLs that are indistinguishable from a
    standard URL. This relies upon the browser's `history` API.
  
    Example:
  
    ```javascript
    App.Router.map(function() {
      this.route('posts', function() {
        this.route('new');
      });
    });
  
    App.Router.reopen({
      location: 'history'
    });
    ```
  
    This will result in a posts.new url of `/posts/new`.
  
    Keep in mind that your server must serve the Ember app at all the routes you
    define.
  
    ### AutoLocation
  
    Using `AutoLocation`, the router will use the best Location class supported by
    the browser it is running in.
  
    Browsers that support the `history` API will use `HistoryLocation`, those that
    do not, but still support the `hashchange` event will use `HashLocation`, and
    in the rare case neither is supported will use `NoneLocation`.
  
    Example:
  
    ```javascript
    App.Router.map(function() {
      this.route('posts', function() {
        this.route('new');
      });
    });
  
    App.Router.reopen({
      location: 'auto'
    });
    ```
  
    This will result in a posts.new url of `/posts/new` for modern browsers that
    support the `history` api or `/#/posts/new` for older ones, like Internet
    Explorer 9 and below.
  
    When a user visits a link to your application, they will be automatically
    upgraded or downgraded to the appropriate `Location` class, with the URL
    transformed accordingly, if needed.
  
    Keep in mind that since some of your users will use `HistoryLocation`, your
    server must serve the Ember app at all the routes you define.
  
    ### NoneLocation
  
    Using `NoneLocation` causes Ember to not store the applications URL state
    in the actual URL. This is generally used for testing purposes, and is one
    of the changes made when calling `App.setupForTesting()`.
  
    ## Location API
  
    Each location implementation must provide the following methods:
  
    * implementation: returns the string name used to reference the implementation.
    * getURL: returns the current URL.
    * setURL(path): sets the current URL.
    * replaceURL(path): replace the current URL (optional).
    * onUpdateURL(callback): triggers the callback when the URL changes.
    * formatURL(url): formats `url` to be placed into `href` attribute.
    * detect() (optional): instructs the location to do any feature detection
        necessary. If the location needs to redirect to a different URL, it
        can cancel routing by setting the `cancelRouterSetup` property on itself
        to `false`.
  
    Calling setURL or replaceURL will not trigger onUpdateURL callbacks.
  
    ## Custom implementation
  
    Ember scans `app/locations/*` for extending the Location API.
  
    Example:
  
    ```javascript
    import Ember from 'ember';
  
    export default Ember.HistoryLocation.extend({
      implementation: 'history-url-logging',
  
      pushState: function (path) {
        console.log(path);
        this._super.apply(this, arguments);
      }
    });
    ```
  
    @class Location
    @namespace Ember
    @static
    @private
  */
  exports.default = {
    /**
     This is deprecated in favor of using the container to lookup the location
     implementation as desired.
      For example:
      ```javascript
     // Given a location registered as follows:
     container.register('location:history-test', HistoryTestLocation);
      // You could create a new instance via:
     container.lookup('location:history-test');
     ```
       @method create
      @param {Object} options
      @return {Object} an instance of an implementation of the `location` API
      @deprecated Use the container to lookup the location implementation that you
      need.
      @private
    */
    create: function (options) {
      var implementation = options && options.implementation;

      var implementationClass = this.implementations[implementation];

      return implementationClass.create.apply(implementationClass, arguments);
    },

    implementations: {},
    _location: _emberEnvironment.environment.location,

    /**
      Returns the current `location.hash` by parsing location.href since browsers
      inconsistently URL-decode `location.hash`.
       https://bugzilla.mozilla.org/show_bug.cgi?id=483304
       @private
      @method getHash
      @since 1.4.0
    */
    _getHash: function () {
      return _emberRoutingLocationUtil.getHash(this.location);
    }
  };
});
enifed('ember-routing/location/auto_location', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime', 'ember-environment', 'ember-routing/location/util'], function (exports, _emberUtils, _emberMetal, _emberRuntime, _emberEnvironment, _emberRoutingLocationUtil) {
  'use strict';

  exports.getHistoryPath = getHistoryPath;
  exports.getHashPath = getHashPath;

  /**
  @module ember
  @submodule ember-routing
  */

  /**
    Ember.AutoLocation will select the best location option based off browser
    support with the priority order: history, hash, none.
  
    Clean pushState paths accessed by hashchange-only browsers will be redirected
    to the hash-equivalent and vice versa so future transitions are consistent.
  
    Keep in mind that since some of your users will use `HistoryLocation`, your
    server must serve the Ember app at all the routes you define.
  
    @class AutoLocation
    @namespace Ember
    @static
    @private
  */
  exports.default = _emberRuntime.Object.extend({
    /**
      @private
       The browser's `location` object. This is typically equivalent to
      `window.location`, but may be overridden for testing.
       @property location
      @default environment.location
    */
    location: _emberEnvironment.environment.location,

    /**
      @private
       The browser's `history` object. This is typically equivalent to
      `window.history`, but may be overridden for testing.
       @since 1.5.1
      @property history
      @default environment.history
    */
    history: _emberEnvironment.environment.history,

    /**
     @private
      The user agent's global variable. In browsers, this will be `window`.
      @since 1.11
     @property global
     @default window
    */
    global: _emberEnvironment.environment.window,

    /**
      @private
       The browser's `userAgent`. This is typically equivalent to
      `navigator.userAgent`, but may be overridden for testing.
       @since 1.5.1
      @property userAgent
      @default environment.history
    */
    userAgent: _emberEnvironment.environment.userAgent,

    /**
      @private
       This property is used by the router to know whether to cancel the routing
      setup process, which is needed while we redirect the browser.
       @since 1.5.1
      @property cancelRouterSetup
      @default false
    */
    cancelRouterSetup: false,

    /**
      @private
       Will be pre-pended to path upon state change.
       @since 1.5.1
      @property rootURL
      @default '/'
    */
    rootURL: '/',

    /**
     Called by the router to instruct the location to do any feature detection
     necessary. In the case of AutoLocation, we detect whether to use history
     or hash concrete implementations.
      @private
    */
    detect: function () {
      var rootURL = this.rootURL;

      var implementation = detectImplementation({
        location: this.location,
        history: this.history,
        userAgent: this.userAgent,
        rootURL: rootURL,
        documentMode: this.documentMode,
        global: this.global
      });

      if (implementation === false) {
        _emberMetal.set(this, 'cancelRouterSetup', true);
        implementation = 'none';
      }

      var concrete = _emberUtils.getOwner(this).lookup('location:' + implementation);
      _emberMetal.set(concrete, 'rootURL', rootURL);

      _emberMetal.set(this, 'concreteImplementation', concrete);
    },

    initState: delegateToConcreteImplementation('initState'),
    getURL: delegateToConcreteImplementation('getURL'),
    setURL: delegateToConcreteImplementation('setURL'),
    replaceURL: delegateToConcreteImplementation('replaceURL'),
    onUpdateURL: delegateToConcreteImplementation('onUpdateURL'),
    formatURL: delegateToConcreteImplementation('formatURL'),

    willDestroy: function () {
      var concreteImplementation = _emberMetal.get(this, 'concreteImplementation');

      if (concreteImplementation) {
        concreteImplementation.destroy();
      }
    }
  });

  function delegateToConcreteImplementation(methodName) {
    return function () {
      var concreteImplementation = _emberMetal.get(this, 'concreteImplementation');

      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      return _emberUtils.tryInvoke(concreteImplementation, methodName, args);
    };
  }

  /*
    Given the browser's `location`, `history` and `userAgent`, and a configured
    root URL, this function detects whether the browser supports the [History
    API](https://developer.mozilla.org/en-US/docs/Web/API/History) and returns a
    string representing the Location object to use based on its determination.
  
    For example, if the page loads in an evergreen browser, this function would
    return the string "history", meaning the history API and thus HistoryLocation
    should be used. If the page is loaded in IE8, it will return the string
    "hash," indicating that the History API should be simulated by manipulating the
    hash portion of the location.
  
  */

  function detectImplementation(options) {
    var location = options.location;
    var userAgent = options.userAgent;
    var history = options.history;
    var documentMode = options.documentMode;
    var global = options.global;
    var rootURL = options.rootURL;

    var implementation = 'none';
    var cancelRouterSetup = false;
    var currentPath = _emberRoutingLocationUtil.getFullPath(location);

    if (_emberRoutingLocationUtil.supportsHistory(userAgent, history)) {
      var historyPath = getHistoryPath(rootURL, location);

      // If the browser supports history and we have a history path, we can use
      // the history location with no redirects.
      if (currentPath === historyPath) {
        return 'history';
      } else {
        if (currentPath.substr(0, 2) === '/#') {
          history.replaceState({ path: historyPath }, null, historyPath);
          implementation = 'history';
        } else {
          cancelRouterSetup = true;
          _emberRoutingLocationUtil.replacePath(location, historyPath);
        }
      }
    } else if (_emberRoutingLocationUtil.supportsHashChange(documentMode, global)) {
      var hashPath = getHashPath(rootURL, location);

      // Be sure we're using a hashed path, otherwise let's switch over it to so
      // we start off clean and consistent. We'll count an index path with no
      // hash as "good enough" as well.
      if (currentPath === hashPath || currentPath === '/' && hashPath === '/#/') {
        implementation = 'hash';
      } else {
        // Our URL isn't in the expected hash-supported format, so we want to
        // cancel the router setup and replace the URL to start off clean
        cancelRouterSetup = true;
        _emberRoutingLocationUtil.replacePath(location, hashPath);
      }
    }

    if (cancelRouterSetup) {
      return false;
    }

    return implementation;
  }

  /**
    @private
  
    Returns the current path as it should appear for HistoryLocation supported
    browsers. This may very well differ from the real current path (e.g. if it
    starts off as a hashed URL)
  */

  function getHistoryPath(rootURL, location) {
    var path = _emberRoutingLocationUtil.getPath(location);
    var hash = _emberRoutingLocationUtil.getHash(location);
    var query = _emberRoutingLocationUtil.getQuery(location);
    var rootURLIndex = path.indexOf(rootURL);
    var routeHash = undefined,
        hashParts = undefined;

    // By convention, Ember.js routes using HashLocation are required to start
    // with `#/`. Anything else should NOT be considered a route and should
    // be passed straight through, without transformation.
    if (hash.substr(0, 2) === '#/') {
      // There could be extra hash segments after the route
      hashParts = hash.substr(1).split('#');
      // The first one is always the route url
      routeHash = hashParts.shift();

      // If the path already has a trailing slash, remove the one
      // from the hashed route so we don't double up.
      if (path.slice(-1) === '/') {
        routeHash = routeHash.substr(1);
      }

      // This is the "expected" final order
      path = path + routeHash + query;

      if (hashParts.length) {
        path += '#' + hashParts.join('#');
      }
    } else {
      path = path + query + hash;
    }

    return path;
  }

  /**
    @private
  
    Returns the current path as it should appear for HashLocation supported
    browsers. This may very well differ from the real current path.
  
    @method _getHashPath
  */

  function getHashPath(rootURL, location) {
    var path = rootURL;
    var historyPath = getHistoryPath(rootURL, location);
    var routePath = historyPath.substr(rootURL.length);

    if (routePath !== '') {
      if (routePath.charAt(0) !== '/') {
        routePath = '/' + routePath;
      }

      path += '#' + routePath;
    }

    return path;
  }
});
enifed('ember-routing/location/hash_location', ['exports', 'ember-metal', 'ember-runtime', 'ember-routing/location/api'], function (exports, _emberMetal, _emberRuntime, _emberRoutingLocationApi) {
  'use strict';

  /**
  @module ember
  @submodule ember-routing
  */

  /**
    `Ember.HashLocation` implements the location API using the browser's
    hash. At present, it relies on a `hashchange` event existing in the
    browser.
  
    @class HashLocation
    @namespace Ember
    @extends Ember.Object
    @private
  */
  exports.default = _emberRuntime.Object.extend({
    implementation: 'hash',

    init: function () {
      _emberMetal.set(this, 'location', _emberMetal.get(this, '_location') || window.location);

      this._hashchangeHandler = undefined;
    },

    /**
      @private
       Returns normalized location.hash
       @since 1.5.1
      @method getHash
    */
    getHash: _emberRoutingLocationApi.default._getHash,

    /**
      Returns the normalized URL, constructed from `location.hash`.
       e.g. `#/foo` => `/foo` as well as `#/foo#bar` => `/foo#bar`.
       By convention, hashed paths must begin with a forward slash, otherwise they
      are not treated as a path so we can distinguish intent.
       @private
      @method getURL
    */
    getURL: function () {
      var originalPath = this.getHash().substr(1);
      var outPath = originalPath;

      if (outPath.charAt(0) !== '/') {
        outPath = '/';

        // Only add the # if the path isn't empty.
        // We do NOT want `/#` since the ampersand
        // is only included (conventionally) when
        // the location.hash has a value
        if (originalPath) {
          outPath += '#' + originalPath;
        }
      }

      return outPath;
    },

    /**
      Set the `location.hash` and remembers what was set. This prevents
      `onUpdateURL` callbacks from triggering when the hash was set by
      `HashLocation`.
       @private
      @method setURL
      @param path {String}
    */
    setURL: function (path) {
      _emberMetal.get(this, 'location').hash = path;
      _emberMetal.set(this, 'lastSetURL', path);
    },

    /**
      Uses location.replace to update the url without a page reload
      or history modification.
       @private
      @method replaceURL
      @param path {String}
    */
    replaceURL: function (path) {
      _emberMetal.get(this, 'location').replace('#' + path);
      _emberMetal.set(this, 'lastSetURL', path);
    },

    /**
      Register a callback to be invoked when the hash changes. These
      callbacks will execute when the user presses the back or forward
      button, but not after `setURL` is invoked.
       @private
      @method onUpdateURL
      @param callback {Function}
    */
    onUpdateURL: function (callback) {
      var _this = this;

      this._removeEventListener();

      this._hashchangeHandler = function () {
        _emberMetal.run(function () {
          var path = _this.getURL();
          if (_emberMetal.get(_this, 'lastSetURL') === path) {
            return;
          }

          _emberMetal.set(_this, 'lastSetURL', null);

          callback(path);
        });
      };

      window.addEventListener('hashchange', this._hashchangeHandler);
    },

    /**
      Given a URL, formats it to be placed into the page as part
      of an element's `href` attribute.
       This is used, for example, when using the {{action}} helper
      to generate a URL based on an event.
       @private
      @method formatURL
      @param url {String}
    */
    formatURL: function (url) {
      return '#' + url;
    },

    /**
      Cleans up the HashLocation event listener.
       @private
      @method willDestroy
    */
    willDestroy: function () {
      this._removeEventListener();
    },

    _removeEventListener: function () {
      if (this._hashchangeHandler) {
        window.removeEventListener('hashchange', this._hashchangeHandler);
      }
    }
  });
});
enifed('ember-routing/location/history_location', ['exports', 'ember-metal', 'ember-runtime', 'ember-routing/location/api'], function (exports, _emberMetal, _emberRuntime, _emberRoutingLocationApi) {
  'use strict';

  /**
  @module ember
  @submodule ember-routing
  */

  var popstateFired = false;

  /**
    Ember.HistoryLocation implements the location API using the browser's
    history.pushState API.
  
    @class HistoryLocation
    @namespace Ember
    @extends Ember.Object
    @private
  */
  exports.default = _emberRuntime.Object.extend({
    implementation: 'history',

    init: function () {
      this._super.apply(this, arguments);

      var base = document.querySelector('base');
      var baseURL = base ? base.getAttribute('href') : '';

      _emberMetal.set(this, 'baseURL', baseURL);
      _emberMetal.set(this, 'location', _emberMetal.get(this, 'location') || window.location);

      this._popstateHandler = undefined;
    },

    /**
      Used to set state on first call to setURL
       @private
      @method initState
    */
    initState: function () {
      var history = _emberMetal.get(this, 'history') || window.history;
      _emberMetal.set(this, 'history', history);

      if (history && 'state' in history) {
        this.supportsHistory = true;
      }

      this.replaceState(this.formatURL(this.getURL()));
    },

    /**
      Will be pre-pended to path upon state change
       @property rootURL
      @default '/'
      @private
    */
    rootURL: '/',

    /**
      Returns the current `location.pathname` without `rootURL` or `baseURL`
       @private
      @method getURL
      @return url {String}
    */
    getURL: function () {
      var location = _emberMetal.get(this, 'location');
      var path = location.pathname;

      var rootURL = _emberMetal.get(this, 'rootURL');
      var baseURL = _emberMetal.get(this, 'baseURL');

      // remove trailing slashes if they exists
      rootURL = rootURL.replace(/\/$/, '');
      baseURL = baseURL.replace(/\/$/, '');

      // remove baseURL and rootURL from start of path
      var url = path.replace(new RegExp('^' + baseURL + '(?=/|$)'), '').replace(new RegExp('^' + rootURL + '(?=/|$)'), '');

      var search = location.search || '';
      url += search;
      url += this.getHash();

      return url;
    },

    /**
      Uses `history.pushState` to update the url without a page reload.
       @private
      @method setURL
      @param path {String}
    */
    setURL: function (path) {
      var state = this.getState();
      path = this.formatURL(path);

      if (!state || state.path !== path) {
        this.pushState(path);
      }
    },

    /**
      Uses `history.replaceState` to update the url without a page reload
      or history modification.
       @private
      @method replaceURL
      @param path {String}
    */
    replaceURL: function (path) {
      var state = this.getState();
      path = this.formatURL(path);

      if (!state || state.path !== path) {
        this.replaceState(path);
      }
    },

    /**
      Get the current `history.state`. Checks for if a polyfill is
      required and if so fetches this._historyState. The state returned
      from getState may be null if an iframe has changed a window's
      history.
       @private
      @method getState
      @return state {Object}
    */
    getState: function () {
      if (this.supportsHistory) {
        return _emberMetal.get(this, 'history').state;
      }

      return this._historyState;
    },

    /**
     Pushes a new state.
      @private
     @method pushState
     @param path {String}
    */
    pushState: function (path) {
      var state = { path: path };

      _emberMetal.get(this, 'history').pushState(state, null, path);

      this._historyState = state;

      // used for webkit workaround
      this._previousURL = this.getURL();
    },

    /**
     Replaces the current state.
      @private
     @method replaceState
     @param path {String}
    */
    replaceState: function (path) {
      var state = { path: path };
      _emberMetal.get(this, 'history').replaceState(state, null, path);

      this._historyState = state;

      // used for webkit workaround
      this._previousURL = this.getURL();
    },

    /**
      Register a callback to be invoked whenever the browser
      history changes, including using forward and back buttons.
       @private
      @method onUpdateURL
      @param callback {Function}
    */
    onUpdateURL: function (callback) {
      var _this = this;

      this._removeEventListener();

      this._popstateHandler = function () {
        // Ignore initial page load popstate event in Chrome
        if (!popstateFired) {
          popstateFired = true;
          if (_this.getURL() === _this._previousURL) {
            return;
          }
        }
        callback(_this.getURL());
      };

      window.addEventListener('popstate', this._popstateHandler);
    },

    /**
      Used when using `{{action}}` helper.  The url is always appended to the rootURL.
       @private
      @method formatURL
      @param url {String}
      @return formatted url {String}
    */
    formatURL: function (url) {
      var rootURL = _emberMetal.get(this, 'rootURL');
      var baseURL = _emberMetal.get(this, 'baseURL');

      if (url !== '') {
        // remove trailing slashes if they exists
        rootURL = rootURL.replace(/\/$/, '');
        baseURL = baseURL.replace(/\/$/, '');
      } else if (baseURL.match(/^\//) && rootURL.match(/^\//)) {
        // if baseURL and rootURL both start with a slash
        // ... remove trailing slash from baseURL if it exists
        baseURL = baseURL.replace(/\/$/, '');
      }

      return baseURL + rootURL + url;
    },

    /**
      Cleans up the HistoryLocation event listener.
       @private
      @method willDestroy
    */
    willDestroy: function () {
      this._removeEventListener();
    },

    /**
      @private
       Returns normalized location.hash
       @method getHash
    */
    getHash: _emberRoutingLocationApi.default._getHash,

    _removeEventListener: function () {
      if (this._popstateHandler) {
        window.removeEventListener('popstate', this._popstateHandler);
      }
    }
  });
});
enifed('ember-routing/location/none_location', ['exports', 'ember-metal', 'ember-runtime'], function (exports, _emberMetal, _emberRuntime) {
  'use strict';

  /**
  @module ember
  @submodule ember-routing
  */

  /**
    Ember.NoneLocation does not interact with the browser. It is useful for
    testing, or when you need to manage state with your Router, but temporarily
    don't want it to muck with the URL (for example when you embed your
    application in a larger page).
  
    @class NoneLocation
    @namespace Ember
    @extends Ember.Object
    @private
  */
  exports.default = _emberRuntime.Object.extend({
    implementation: 'none',
    path: '',

    detect: function () {
      var rootURL = this.rootURL;
    },

    /**
      Will be pre-pended to path.
       @private
      @property rootURL
      @default '/'
    */
    rootURL: '/',

    /**
      Returns the current path without `rootURL`.
       @private
      @method getURL
      @return {String} path
    */
    getURL: function () {
      var path = _emberMetal.get(this, 'path');
      var rootURL = _emberMetal.get(this, 'rootURL');

      // remove trailing slashes if they exists
      rootURL = rootURL.replace(/\/$/, '');

      // remove rootURL from url
      return path.replace(new RegExp('^' + rootURL + '(?=/|$)'), '');
    },

    /**
      Set the path and remembers what was set. Using this method
      to change the path will not invoke the `updateURL` callback.
       @private
      @method setURL
      @param path {String}
    */
    setURL: function (path) {
      _emberMetal.set(this, 'path', path);
    },

    /**
      Register a callback to be invoked when the path changes. These
      callbacks will execute when the user presses the back or forward
      button, but not after `setURL` is invoked.
       @private
      @method onUpdateURL
      @param callback {Function}
    */
    onUpdateURL: function (callback) {
      this.updateCallback = callback;
    },

    /**
      Sets the path and calls the `updateURL` callback.
       @private
      @method handleURL
      @param callback {Function}
    */
    handleURL: function (url) {
      _emberMetal.set(this, 'path', url);
      this.updateCallback(url);
    },

    /**
      Given a URL, formats it to be placed into the page as part
      of an element's `href` attribute.
       This is used, for example, when using the {{action}} helper
      to generate a URL based on an event.
       @private
      @method formatURL
      @param url {String}
      @return {String} url
    */
    formatURL: function (url) {
      var rootURL = _emberMetal.get(this, 'rootURL');

      if (url !== '') {
        // remove trailing slashes if they exists
        rootURL = rootURL.replace(/\/$/, '');
      }

      return rootURL + url;
    }
  });
});
enifed('ember-routing/location/util', ['exports'], function (exports) {
  /**
    @private
  
    Returns the current `location.pathname`, normalized for IE inconsistencies.
  */
  'use strict';

  exports.getPath = getPath;
  exports.getQuery = getQuery;
  exports.getHash = getHash;
  exports.getFullPath = getFullPath;
  exports.getOrigin = getOrigin;
  exports.supportsHashChange = supportsHashChange;
  exports.supportsHistory = supportsHistory;
  exports.replacePath = replacePath;

  function getPath(location) {
    var pathname = location.pathname;
    // Various versions of IE/Opera don't always return a leading slash
    if (pathname.charAt(0) !== '/') {
      pathname = '/' + pathname;
    }

    return pathname;
  }

  /**
    @private
  
    Returns the current `location.search`.
  */

  function getQuery(location) {
    return location.search;
  }

  /**
    @private
  
    Returns the current `location.hash` by parsing location.href since browsers
    inconsistently URL-decode `location.hash`.
  
    Should be passed the browser's `location` object as the first argument.
  
    https://bugzilla.mozilla.org/show_bug.cgi?id=483304
  */

  function getHash(location) {
    var href = location.href;
    var hashIndex = href.indexOf('#');

    if (hashIndex === -1) {
      return '';
    } else {
      return href.substr(hashIndex);
    }
  }

  function getFullPath(location) {
    return getPath(location) + getQuery(location) + getHash(location);
  }

  function getOrigin(location) {
    var origin = location.origin;

    // Older browsers, especially IE, don't have origin
    if (!origin) {
      origin = location.protocol + '//' + location.hostname;

      if (location.port) {
        origin += ':' + location.port;
      }
    }

    return origin;
  }

  /*
    `documentMode` only exist in Internet Explorer, and it's tested because IE8 running in
    IE7 compatibility mode claims to support `onhashchange` but actually does not.
  
    `global` is an object that may have an `onhashchange` property.
  
    @private
    @function supportsHashChange
  */

  function supportsHashChange(documentMode, global) {
    return 'onhashchange' in global && (documentMode === undefined || documentMode > 7);
  }

  /*
    `userAgent` is a user agent string. We use user agent testing here, because
    the stock Android browser is known to have buggy versions of the History API,
    in some Android versions.
  
    @private
    @function supportsHistory
  */

  function supportsHistory(userAgent, history) {
    // Boosted from Modernizr: https://github.com/Modernizr/Modernizr/blob/master/feature-detects/history.js
    // The stock browser on Android 2.2 & 2.3, and 4.0.x returns positive on history support
    // Unfortunately support is really buggy and there is no clean way to detect
    // these bugs, so we fall back to a user agent sniff :(

    // We only want Android 2 and 4.0, stock browser, and not Chrome which identifies
    // itself as 'Mobile Safari' as well, nor Windows Phone.
    if ((userAgent.indexOf('Android 2.') !== -1 || userAgent.indexOf('Android 4.0') !== -1) && userAgent.indexOf('Mobile Safari') !== -1 && userAgent.indexOf('Chrome') === -1 && userAgent.indexOf('Windows Phone') === -1) {
      return false;
    }

    return !!(history && 'pushState' in history);
  }

  /**
    Replaces the current location, making sure we explicitly include the origin
    to prevent redirecting to a different origin.
  
    @private
  */

  function replacePath(location, path) {
    location.replace(getOrigin(location) + path);
  }
});
enifed('ember-routing/services/routing', ['exports', 'ember-utils', 'ember-runtime', 'ember-metal', 'ember-routing/utils'], function (exports, _emberUtils, _emberRuntime, _emberMetal, _emberRoutingUtils) {
  /**
  @module ember
  @submodule ember-routing
  */

  'use strict';

  /**
    The Routing service is used by LinkComponent, and provides facilities for
    the component/view layer to interact with the router.
  
    While still private, this service can eventually be opened up, and provides
    the set of API needed for components to control routing without interacting
    with router internals.
  
    @private
    @class RoutingService
  */
  exports.default = _emberRuntime.Service.extend({
    router: null,

    targetState: _emberRuntime.readOnly('router.targetState'),
    currentState: _emberRuntime.readOnly('router.currentState'),
    currentRouteName: _emberRuntime.readOnly('router.currentRouteName'),
    currentPath: _emberRuntime.readOnly('router.currentPath'),

    availableRoutes: function () {
      return Object.keys(_emberMetal.get(this, 'router').router.recognizer.names);
    },

    hasRoute: function (routeName) {
      return _emberMetal.get(this, 'router').hasRoute(routeName);
    },

    transitionTo: function (routeName, models, queryParams, shouldReplace) {
      var router = _emberMetal.get(this, 'router');

      var transition = router._doTransition(routeName, models, queryParams);

      if (shouldReplace) {
        transition.method('replace');
      }

      return transition;
    },

    normalizeQueryParams: function (routeName, models, queryParams) {
      var router = _emberMetal.get(this, 'router');
      router._prepareQueryParams(routeName, models, queryParams);
    },

    generateURL: function (routeName, models, queryParams) {
      var router = _emberMetal.get(this, 'router');
      if (!router.router) {
        return;
      }

      var visibleQueryParams = {};
      _emberUtils.assign(visibleQueryParams, queryParams);

      this.normalizeQueryParams(routeName, models, visibleQueryParams);

      var args = _emberRoutingUtils.routeArgs(routeName, models, visibleQueryParams);
      return router.generate.apply(router, args);
    },

    isActiveForRoute: function (contexts, queryParams, routeName, routerState, isCurrentWhenSpecified) {
      var router = _emberMetal.get(this, 'router');

      var handlers = router.router.recognizer.handlersFor(routeName);
      var leafName = handlers[handlers.length - 1].handler;
      var maximumContexts = numberOfContextsAcceptedByHandler(routeName, handlers);

      // NOTE: any ugliness in the calculation of activeness is largely
      // due to the fact that we support automatic normalizing of
      // `resource` -> `resource.index`, even though there might be
      // dynamic segments / query params defined on `resource.index`
      // which complicates (and makes somewhat ambiguous) the calculation
      // of activeness for links that link to `resource` instead of
      // directly to `resource.index`.

      // if we don't have enough contexts revert back to full route name
      // this is because the leaf route will use one of the contexts
      if (contexts.length > maximumContexts) {
        routeName = leafName;
      }

      return routerState.isActiveIntent(routeName, contexts, queryParams, !isCurrentWhenSpecified);
    }
  });

  function numberOfContextsAcceptedByHandler(handler, handlerInfos) {
    var req = 0;
    for (var i = 0; i < handlerInfos.length; i++) {
      req = req + handlerInfos[i].names.length;
      if (handlerInfos[i].handler === handler) {
        break;
      }
    }

    return req;
  }
});
enifed('ember-routing/system/cache', ['exports', 'ember-utils', 'ember-runtime'], function (exports, _emberUtils, _emberRuntime) {
  'use strict';

  /**
    A two-tiered cache with support for fallback values when doing lookups.
    Uses "buckets" and then "keys" to cache values.
  
    @private
    @class BucketCache
  */
  exports.default = _emberRuntime.Object.extend({
    init: function () {
      this.cache = new _emberUtils.EmptyObject();
    },

    has: function (bucketKey) {
      return !!this.cache[bucketKey];
    },

    stash: function (bucketKey, key, value) {
      var bucket = this.cache[bucketKey];

      if (!bucket) {
        bucket = this.cache[bucketKey] = new _emberUtils.EmptyObject();
      }

      bucket[key] = value;
    },

    lookup: function (bucketKey, prop, defaultValue) {
      var cache = this.cache;
      if (!this.has(bucketKey)) {
        return defaultValue;
      }

      var bucket = cache[bucketKey];
      if (prop in bucket && bucket[prop] !== undefined) {
        return bucket[prop];
      } else {
        return defaultValue;
      }
    }
  });
});
enifed("ember-routing/system/controller_for", ["exports"], function (exports) {
  /**
  @module ember
  @submodule ember-routing
  */

  /**
    Finds a controller instance.
  
    @for Ember
    @method controllerFor
    @private
  */
  "use strict";

  exports.default = controllerFor;

  function controllerFor(container, controllerName, lookupOptions) {
    return container.lookup("controller:" + controllerName, lookupOptions);
  }
});
enifed('ember-routing/system/dsl', ['exports', 'ember-utils', 'ember-metal'], function (exports, _emberUtils, _emberMetal) {
  'use strict';

  /**
  @module ember
  @submodule ember-routing
  */

  function DSL(name, options) {
    this.parent = name;
    this.enableLoadingSubstates = options && options.enableLoadingSubstates;
    this.matches = [];
    this.explicitIndex = undefined;
    this.options = options;
  }

  exports.default = DSL;

  DSL.prototype = {
    route: function (name, options, callback) {
      var dummyErrorRoute = '/_unused_dummy_error_path_route_' + name + '/:error';
      if (arguments.length === 2 && typeof options === 'function') {
        callback = options;
        options = {};
      }

      if (arguments.length === 1) {
        options = {};
      }

      if (this.enableLoadingSubstates) {
        createRoute(this, name + '_loading', { resetNamespace: options.resetNamespace });
        createRoute(this, name + '_error', { resetNamespace: options.resetNamespace, path: dummyErrorRoute });
      }

      if (callback) {
        var fullName = getFullName(this, name, options.resetNamespace);
        var dsl = new DSL(fullName, this.options);

        createRoute(dsl, 'loading');
        createRoute(dsl, 'error', { path: dummyErrorRoute });

        callback.call(dsl);

        createRoute(this, name, options, dsl.generate());
      } else {
        createRoute(this, name, options);
      }
    },

    push: function (url, name, callback, serialize) {
      var parts = name.split('.');

      if (this.options.engineInfo) {
        var localFullName = name.slice(this.options.engineInfo.fullName.length + 1);
        var routeInfo = _emberUtils.assign({ localFullName: localFullName }, this.options.engineInfo);

        if (serialize) {
          routeInfo.serializeMethod = serialize;
        }

        this.options.addRouteForEngine(name, routeInfo);
      } else if (serialize) {
        throw new Error('Defining a route serializer on route \'' + name + '\' outside an Engine is not allowed.');
      }

      if (url === '' || url === '/' || parts[parts.length - 1] === 'index') {
        this.explicitIndex = true;
      }

      this.matches.push([url, name, callback]);
    },

    resource: function (name, options, callback) {
      if (arguments.length === 2 && typeof options === 'function') {
        callback = options;
        options = {};
      }

      if (arguments.length === 1) {
        options = {};
      }

      options.resetNamespace = true;

      this.route(name, options, callback);
    },

    generate: function () {
      var dslMatches = this.matches;

      if (!this.explicitIndex) {
        this.route('index', { path: '/' });
      }

      return function (match) {
        for (var i = 0; i < dslMatches.length; i++) {
          var dslMatch = dslMatches[i];
          match(dslMatch[0]).to(dslMatch[1], dslMatch[2]);
        }
      };
    }
  };

  function canNest(dsl) {
    return dsl.parent && dsl.parent !== 'application';
  }

  function getFullName(dsl, name, resetNamespace) {
    if (canNest(dsl) && resetNamespace !== true) {
      return dsl.parent + '.' + name;
    } else {
      return name;
    }
  }

  function createRoute(dsl, name, options, callback) {
    options = options || {};

    var fullName = getFullName(dsl, name, options.resetNamespace);

    if (typeof options.path !== 'string') {
      options.path = '/' + name;
    }

    dsl.push(options.path, fullName, callback, options.serialize);
  }

  DSL.map = function (callback) {
    var dsl = new DSL();
    callback.call(dsl);
    return dsl;
  };

  var uuid = 0;

  DSL.prototype.mount = function (_name, _options) {
    var options = _options || {};
    var engineRouteMap = this.options.resolveRouteMap(_name);
    var name = _name;

    if (options.as) {
      name = options.as;
    }

    var fullName = getFullName(this, name, options.resetNamespace);

    var engineInfo = {
      name: _name,
      instanceId: uuid++,
      mountPoint: fullName,
      fullName: fullName
    };

    var path = options.path;

    if (typeof path !== 'string') {
      path = '/' + name;
    }

    var callback = undefined;
    var dummyErrorRoute = '/_unused_dummy_error_path_route_' + name + '/:error';
    if (engineRouteMap) {
      var shouldResetEngineInfo = false;
      var oldEngineInfo = this.options.engineInfo;
      if (oldEngineInfo) {
        shouldResetEngineInfo = true;
        this.options.engineInfo = engineInfo;
      }

      var optionsForChild = _emberUtils.assign({ engineInfo: engineInfo }, this.options);
      var childDSL = new DSL(fullName, optionsForChild);

      createRoute(childDSL, 'loading');
      createRoute(childDSL, 'error', { path: dummyErrorRoute });

      engineRouteMap.call(childDSL);

      callback = childDSL.generate();

      if (shouldResetEngineInfo) {
        this.options.engineInfo = oldEngineInfo;
      }
    }

    var localFullName = 'application';
    var routeInfo = _emberUtils.assign({ localFullName: localFullName }, engineInfo);

    if (this.enableLoadingSubstates) {
      // These values are important to register the loading routes under their
      // proper names for the Router and within the Engine's registry.
      var substateName = name + '_loading';
      var _localFullName = 'application_loading';
      var _routeInfo = _emberUtils.assign({ localFullName: _localFullName }, engineInfo);
      createRoute(this, substateName, { resetNamespace: options.resetNamespace });
      this.options.addRouteForEngine(substateName, _routeInfo);

      substateName = name + '_error';
      _localFullName = 'application_error';
      _routeInfo = _emberUtils.assign({ localFullName: _localFullName }, engineInfo);
      createRoute(this, substateName, { resetNamespace: options.resetNamespace, path: dummyErrorRoute });
      this.options.addRouteForEngine(substateName, _routeInfo);
    }

    this.options.addRouteForEngine(fullName, routeInfo);

    this.push(path, fullName, callback);
  };
});
enifed('ember-routing/system/generate_controller', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  exports.generateControllerFactory = generateControllerFactory;
  exports.default = generateController;

  /**
  @module ember
  @submodule ember-routing
  */

  /**
    Generates a controller factory
  
    @for Ember
    @method generateControllerFactory
    @private
  */

  function generateControllerFactory(owner, controllerName) {
    var Factory = owner._lookupFactory('controller:basic').extend({
      isGenerated: true,
      toString: function () {
        return '(generated ' + controllerName + ' controller)';
      }
    });

    var fullName = 'controller:' + controllerName;

    owner.register(fullName, Factory);

    return Factory;
  }

  /**
    Generates and instantiates a controller extending from `controller:basic`
    if present, or `Ember.Controller` if not.
  
    @for Ember
    @method generateController
    @private
    @since 1.3.0
  */

  function generateController(owner, controllerName) {
    generateControllerFactory(owner, controllerName);

    var fullName = 'controller:' + controllerName;
    var instance = owner.lookup(fullName);

    if (_emberMetal.get(instance, 'namespace.LOG_ACTIVE_GENERATION')) {}

    return instance;
  }
});
enifed('ember-routing/system/query_params', ['exports', 'ember-runtime'], function (exports, _emberRuntime) {
  'use strict';

  exports.default = _emberRuntime.Object.extend({
    isQueryParams: true,
    values: null
  });
});
enifed('ember-routing/system/route', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime', 'ember-routing/system/generate_controller', 'ember-routing/utils'], function (exports, _emberUtils, _emberMetal, _emberRuntime, _emberRoutingSystemGenerate_controller, _emberRoutingUtils) {
  'use strict';

  exports.defaultSerialize = defaultSerialize;
  exports.hasDefaultSerialize = hasDefaultSerialize;
  var slice = Array.prototype.slice;

  function K() {
    return this;
  }

  function defaultSerialize(model, params) {
    if (params.length < 1) {
      return;
    }
    if (!model) {
      return;
    }

    var name = params[0];
    var object = {};

    if (params.length === 1) {
      if (name in model) {
        object[name] = _emberMetal.get(model, name);
      } else if (/_id$/.test(name)) {
        object[name] = _emberMetal.get(model, 'id');
      }
    } else {
      object = _emberMetal.getProperties(model, params);
    }

    return object;
  }

  var DEFAULT_SERIALIZE = _emberUtils.symbol('DEFAULT_SERIALIZE');

  defaultSerialize[DEFAULT_SERIALIZE] = true;

  function hasDefaultSerialize(route) {
    return !!route.serialize[DEFAULT_SERIALIZE];
  }

  /**
  @module ember
  @submodule ember-routing
  */

  /**
    The `Ember.Route` class is used to define individual routes. Refer to
    the [routing guide](http://emberjs.com/guides/routing/) for documentation.
  
    @class Route
    @namespace Ember
    @extends Ember.Object
    @uses Ember.ActionHandler
    @uses Ember.Evented
    @since 1.0.0
    @public
  */
  var Route = _emberRuntime.Object.extend(_emberRuntime.ActionHandler, _emberRuntime.Evented, {
    /**
      Configuration hash for this route's queryParams. The possible
      configuration options and their defaults are as follows
      (assuming a query param whose controller property is `page`):
       ```javascript
      queryParams: {
        page: {
          // By default, controller query param properties don't
          // cause a full transition when they are changed, but
          // rather only cause the URL to update. Setting
          // `refreshModel` to true will cause an "in-place"
          // transition to occur, whereby the model hooks for
          // this route (and any child routes) will re-fire, allowing
          // you to reload models (e.g., from the server) using the
          // updated query param values.
          refreshModel: false,
           // By default, changes to controller query param properties
          // cause the URL to update via `pushState`, which means an
          // item will be added to the browser's history, allowing
          // you to use the back button to restore the app to the
          // previous state before the query param property was changed.
          // Setting `replace` to true will use `replaceState` (or its
          // hash location equivalent), which causes no browser history
          // item to be added. This options name and default value are
          // the same as the `link-to` helper's `replace` option.
          replace: false,
           // By default, the query param URL key is the same name as
          // the controller property name. Use `as` to specify a
          // different URL key.
          as: 'page'
        }
      }
      ```
       @property queryParams
      @for Ember.Route
      @type Object
      @since 1.6.0
      @public
    */
    queryParams: {},

    /**
      The name of the route, dot-delimited.
       For example, a route found at `app/routes/posts/post.js` will have
      a `routeName` of `posts.post`.
       @property routeName
      @for Ember.Route
      @type String
      @since 1.0.0
      @public
    */

    /**
      Sets the name for this route, including a fully resolved name for routes
      inside engines.
       @private
      @method _setRouteName
      @param {String} name
    */
    _setRouteName: function (name) {
      this.routeName = name;
      this.fullRouteName = getEngineRouteName(_emberUtils.getOwner(this), name);
    },

    /**
      Populates the QP meta information in the BucketCache.
       @private
      @method _populateQPMeta
    */
    _populateQPMeta: function () {
      this._bucketCache.stash('route-meta', this.fullRouteName, this.get('_qp'));
    },

    /**
      @private
       @property _qp
    */
    _qp: _emberMetal.computed(function () {
      var _this = this;

      var controllerProto = undefined,
          combinedQueryParameterConfiguration = undefined;

      var controllerName = this.controllerName || this.routeName;
      var definedControllerClass = _emberUtils.getOwner(this)._lookupFactory('controller:' + controllerName);
      var queryParameterConfiguraton = _emberMetal.get(this, 'queryParams');
      var hasRouterDefinedQueryParams = !!Object.keys(queryParameterConfiguraton).length;

      if (definedControllerClass) {
        // the developer has authored a controller class in their application for this route
        // access the prototype, find its query params and normalize their object shape
        // them merge in the query params for the route. As a mergedProperty, Route#queryParams is always
        // at least `{}`
        controllerProto = definedControllerClass.proto();

        var controllerDefinedQueryParameterConfiguration = _emberMetal.get(controllerProto, 'queryParams');
        var normalizedControllerQueryParameterConfiguration = _emberRoutingUtils.normalizeControllerQueryParams(controllerDefinedQueryParameterConfiguration);
        combinedQueryParameterConfiguration = mergeEachQueryParams(normalizedControllerQueryParameterConfiguration, queryParameterConfiguraton);
      } else if (hasRouterDefinedQueryParams) {
        // the developer has not defined a controller but *has* supplied route query params.
        // Generate a class for them so we can later insert default values
        var generatedControllerClass = _emberRoutingSystemGenerate_controller.generateControllerFactory(_emberUtils.getOwner(this), controllerName);
        controllerProto = generatedControllerClass.proto();
        combinedQueryParameterConfiguration = queryParameterConfiguraton;
      }

      var qps = [];
      var map = {};
      var propertyNames = [];

      for (var propName in combinedQueryParameterConfiguration) {
        if (!combinedQueryParameterConfiguration.hasOwnProperty(propName)) {
          continue;
        }

        // to support the dubious feature of using unknownProperty
        // on queryParams configuration
        if (propName === 'unknownProperty' || propName === '_super') {
          // possible todo: issue deprecation warning?
          continue;
        }

        var desc = combinedQueryParameterConfiguration[propName];
        var scope = desc.scope || 'model';
        var parts = undefined;

        if (scope === 'controller') {
          parts = [];
        }

        var urlKey = desc.as || this.serializeQueryParamKey(propName);
        var defaultValue = _emberMetal.get(controllerProto, propName);

        if (Array.isArray(defaultValue)) {
          defaultValue = _emberRuntime.A(defaultValue.slice());
        }

        var type = desc.type || _emberRuntime.typeOf(defaultValue);

        var defaultValueSerialized = this.serializeQueryParam(defaultValue, urlKey, type);
        var scopedPropertyName = controllerName + ':' + propName;
        var qp = {
          undecoratedDefaultValue: _emberMetal.get(controllerProto, propName),
          defaultValue: defaultValue,
          serializedDefaultValue: defaultValueSerialized,
          serializedValue: defaultValueSerialized,

          type: type,
          urlKey: urlKey,
          prop: propName,
          scopedPropertyName: scopedPropertyName,
          controllerName: controllerName,
          route: this,
          parts: parts, // provided later when stashNames is called if 'model' scope
          values: null, // provided later when setup is called. no idea why.
          scope: scope
        };

        map[propName] = map[urlKey] = map[scopedPropertyName] = qp;
        qps.push(qp);
        propertyNames.push(propName);
      }

      return {
        qps: qps,
        map: map,
        propertyNames: propertyNames,
        states: {
          /*
            Called when a query parameter changes in the URL, this route cares
            about that query parameter, but the route is not currently
            in the active route hierarchy.
          */
          inactive: function (prop, value) {
            var qp = map[prop];
            _this._qpChanged(prop, value, qp);
          },
          /*
            Called when a query parameter changes in the URL, this route cares
            about that query parameter, and the route is currently
            in the active route hierarchy.
          */
          active: function (prop, value) {
            var qp = map[prop];
            _this._qpChanged(prop, value, qp);
            return _this._activeQPChanged(map[prop], value);
          },
          /*
            Called when a value of a query parameter this route handles changes in a controller
            and the route is currently in the active route hierarchy.
          */
          allowOverrides: function (prop, value) {
            var qp = map[prop];
            _this._qpChanged(prop, value, qp);
            return _this._updatingQPChanged(map[prop]);
          }
        }
      };
    }),

    /**
      @private
       @property _names
    */
    _names: null,

    /**
      @private
       @method _stashNames
    */
    _stashNames: function (_handlerInfo, dynamicParent) {
      var handlerInfo = _handlerInfo;
      if (this._names) {
        return;
      }
      var names = this._names = handlerInfo._names;

      if (!names.length) {
        handlerInfo = dynamicParent;
        names = handlerInfo && handlerInfo._names || [];
      }

      var qps = _emberMetal.get(this, '_qp.qps');

      var namePaths = new Array(names.length);
      for (var a = 0; a < names.length; ++a) {
        namePaths[a] = handlerInfo.name + '.' + names[a];
      }

      for (var i = 0; i < qps.length; ++i) {
        var qp = qps[i];
        if (qp.scope === 'model') {
          qp.parts = namePaths;
        }
      }
    },

    /**
      @private
       @property _activeQPChanged
    */
    _activeQPChanged: function (qp, value) {
      var router = this.router;
      router._activeQPChanged(qp.scopedPropertyName, value);
    },

    /**
      @private
      @method _updatingQPChanged
    */
    _updatingQPChanged: function (qp) {
      var router = this.router;
      router._updatingQPChanged(qp.urlKey);
    },

    mergedProperties: ['queryParams'],

    /**
      Returns a hash containing the parameters of an ancestor route.
       Example
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('member', { path: ':name' }, function() {
          this.route('interest', { path: ':interest' });
        });
      });
      ```
       ```app/routes/member.js
      export default Ember.Route.extend({
        queryParams: {
          memberQp: { refreshModel: true }
        }
      });
      ```
       ```app/routes/member/interest.js
      export default Ember.Route.extend({
        queryParams: {
          interestQp: { refreshModel: true }
        },
         model() {
          return this.paramsFor('member');
        }
      });
      ```
       If we visit `/turing/maths?memberQp=member&interestQp=interest` the model for
      the `member.interest` route is hash with:
       * `name`: `turing`
      * `memberQp`: `member`
       @method paramsFor
      @param {String} name
      @return {Object} hash containing the parameters of the route `name`
      @since 1.4.0
      @public
    */
    paramsFor: function (name) {
      var route = _emberUtils.getOwner(this).lookup('route:' + name);

      if (!route) {
        return {};
      }

      var transition = this.router.router.activeTransition;
      var state = transition ? transition.state : this.router.router.state;

      var fullName = route.fullRouteName;
      var params = _emberUtils.assign({}, state.params[fullName]);
      var queryParams = getQueryParamsFor(route, state);

      return Object.keys(queryParams).reduce(function (params, key) {
        params[key] = queryParams[key];
        return params;
      }, params);
    },

    /**
      Serializes the query parameter key
       @method serializeQueryParamKey
      @param {String} controllerPropertyName
      @private
    */
    serializeQueryParamKey: function (controllerPropertyName) {
      return controllerPropertyName;
    },

    /**
      Serializes value of the query parameter based on defaultValueType
       @method serializeQueryParam
      @param {Object} value
      @param {String} urlKey
      @param {String} defaultValueType
      @private
    */
    serializeQueryParam: function (value, urlKey, defaultValueType) {
      // urlKey isn't used here, but anyone overriding
      // can use it to provide serialization specific
      // to a certain query param.
      return this.router._serializeQueryParam(value, defaultValueType);
    },

    /**
      Deserializes value of the query parameter based on defaultValueType
       @method deserializeQueryParam
      @param {Object} value
      @param {String} urlKey
      @param {String} defaultValueType
      @private
    */
    deserializeQueryParam: function (value, urlKey, defaultValueType) {
      // urlKey isn't used here, but anyone overriding
      // can use it to provide deserialization specific
      // to a certain query param.
      return this.router._deserializeQueryParam(value, defaultValueType);
    },

    /**
      @private
       @property _optionsForQueryParam
    */
    _optionsForQueryParam: function (qp) {
      return _emberMetal.get(this, 'queryParams.' + qp.urlKey) || _emberMetal.get(this, 'queryParams.' + qp.prop) || {};
    },

    /**
      A hook you can use to reset controller values either when the model
      changes or the route is exiting.
       ```app/routes/articles.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        resetController(controller, isExiting, transition) {
          if (isExiting) {
            controller.set('page', 1);
          }
        }
      });
      ```
       @method resetController
      @param {Controller} controller instance
      @param {Boolean} isExiting
      @param {Object} transition
      @since 1.7.0
      @public
    */
    resetController: K,

    /**
      @private
       @method exit
    */
    exit: function () {
      this.deactivate();
      this.trigger('deactivate');
      this.teardownViews();
    },

    /**
      @private
       @method _reset
      @since 1.7.0
    */
    _reset: function (isExiting, transition) {
      var controller = this.controller;
      controller._qpDelegate = _emberMetal.get(this, '_qp.states.inactive');

      this.resetController(controller, isExiting, transition);
    },

    /**
      @private
       @method enter
    */
    enter: function () {
      this.connections = [];
      this.activate();
      this.trigger('activate');
    },

    /**
      The name of the template to use by default when rendering this routes
      template.
       ```app/routes/posts/list.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        templateName: 'posts/list'
      });
      ```
       ```app/routes/posts/index.js
      import PostsList from '../posts/list';
       export default PostsList.extend();
      ```
       ```app/routes/posts/archived.js
      import PostsList from '../posts/list';
       export default PostsList.extend();
      ```
       @property templateName
      @type String
      @default null
      @since 1.4.0
      @public
    */
    templateName: null,

    /**
      The name of the controller to associate with this route.
       By default, Ember will lookup a route's controller that matches the name
      of the route (i.e. `App.PostController` for `App.PostRoute`). However,
      if you would like to define a specific controller to use, you can do so
      using this property.
       This is useful in many ways, as the controller specified will be:
       * passed to the `setupController` method.
      * used as the controller for the template being rendered by the route.
      * returned from a call to `controllerFor` for the route.
       @property controllerName
      @type String
      @default null
      @since 1.4.0
      @public
    */
    controllerName: null,

    /**
      The `willTransition` action is fired at the beginning of any
      attempted transition with a `Transition` object as the sole
      argument. This action can be used for aborting, redirecting,
      or decorating the transition from the currently active routes.
       A good example is preventing navigation when a form is
      half-filled out:
       ```app/routes/contact-form.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          willTransition(transition) {
            if (this.controller.get('userHasEnteredData')) {
              this.controller.displayNavigationConfirm();
              transition.abort();
            }
          }
        }
      });
      ```
       You can also redirect elsewhere by calling
      `this.transitionTo('elsewhere')` from within `willTransition`.
      Note that `willTransition` will not be fired for the
      redirecting `transitionTo`, since `willTransition` doesn't
      fire when there is already a transition underway. If you want
      subsequent `willTransition` actions to fire for the redirecting
      transition, you must first explicitly call
      `transition.abort()`.
       To allow the `willTransition` event to continue bubbling to the parent
      route, use `return true;`. When the `willTransition` method has a
      return value of `true` then the parent route's `willTransition` method
      will be fired, enabling "bubbling" behavior for the event.
       @event willTransition
      @param {Transition} transition
      @since 1.0.0
      @public
    */

    /**
      The `didTransition` action is fired after a transition has
      successfully been completed. This occurs after the normal model
      hooks (`beforeModel`, `model`, `afterModel`, `setupController`)
      have resolved. The `didTransition` action has no arguments,
      however, it can be useful for tracking page views or resetting
      state on the controller.
       ```app/routes/login.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          didTransition() {
            this.controller.get('errors.base').clear();
            return true; // Bubble the didTransition event
          }
        }
      });
      ```
       @event didTransition
      @since 1.2.0
      @public
    */

    /**
      The `loading` action is fired on the route when a route's `model`
      hook returns a promise that is not already resolved. The current
      `Transition` object is the first parameter and the route that
      triggered the loading event is the second parameter.
       ```app/routes/application.js
      export default Ember.Route.extend({
        actions: {
          loading(transition, route) {
            let controller = this.controllerFor('foo');
            controller.set('currentlyLoading', true);
             transition.finally(function() {
              controller.set('currentlyLoading', false);
            });
          }
        }
      });
      ```
       @event loading
      @param {Transition} transition
      @param {Ember.Route} route The route that triggered the loading event
      @since 1.2.0
      @public
    */

    /**
      When attempting to transition into a route, any of the hooks
      may return a promise that rejects, at which point an `error`
      action will be fired on the partially-entered routes, allowing
      for per-route error handling logic, or shared error handling
      logic defined on a parent route.
       Here is an example of an error handler that will be invoked
      for rejected promises from the various hooks on the route,
      as well as any unhandled errors from child routes:
       ```app/routes/admin.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        beforeModel() {
          return Ember.RSVP.reject('bad things!');
        },
         actions: {
          error(error, transition) {
            // Assuming we got here due to the error in `beforeModel`,
            // we can expect that error === "bad things!",
            // but a promise model rejecting would also
            // call this hook, as would any errors encountered
            // in `afterModel`.
             // The `error` hook is also provided the failed
            // `transition`, which can be stored and later
            // `.retry()`d if desired.
             this.transitionTo('login');
          }
        }
      });
      ```
       `error` actions that bubble up all the way to `ApplicationRoute`
      will fire a default error handler that logs the error. You can
      specify your own global default error handler by overriding the
      `error` handler on `ApplicationRoute`:
       ```app/routes/application.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          error(error, transition) {
            this.controllerFor('banner').displayError(error.message);
          }
        }
      });
      ```
      @event error
      @param {Error} error
      @param {Transition} transition
      @since 1.0.0
      @public
    */

    /**
      This event is triggered when the router enters the route. It is
      not executed when the model for the route changes.
       ```app/routes/application.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        collectAnalytics: Ember.on('activate', function(){
          collectAnalytics();
        })
      });
      ```
       @event activate
      @since 1.9.0
      @public
    */

    /**
      This event is triggered when the router completely exits this
      route. It is not executed when the model for the route changes.
       ```app/routes/index.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        trackPageLeaveAnalytics: Ember.on('deactivate', function(){
          trackPageLeaveAnalytics();
        })
      });
      ```
       @event deactivate
      @since 1.9.0
      @public
    */

    /**
      The controller associated with this route.
       Example
       ```app/routes/form.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          willTransition(transition) {
            if (this.controller.get('userHasEnteredData') &&
                !confirm('Are you sure you want to abandon progress?')) {
              transition.abort();
            } else {
              // Bubble the `willTransition` action so that
              // parent routes can decide whether or not to abort.
              return true;
            }
          }
        }
      });
      ```
       @property controller
      @type Ember.Controller
      @since 1.6.0
      @public
    */

    actions: {

      /**
        This action is called when one or more query params have changed. Bubbles.
         @method queryParamsDidChange
        @param changed {Object} Keys are names of query params that have changed.
        @param totalPresent {Object} Keys are names of query params that are currently set.
        @param removed {Object} Keys are names of query params that have been removed.
        @returns {boolean}
        @private
       */
      queryParamsDidChange: function (changed, totalPresent, removed) {
        var qpMap = _emberMetal.get(this, '_qp').map;

        var totalChanged = Object.keys(changed).concat(Object.keys(removed));
        for (var i = 0; i < totalChanged.length; ++i) {
          var qp = qpMap[totalChanged[i]];
          if (qp && _emberMetal.get(this._optionsForQueryParam(qp), 'refreshModel') && this.router.currentState) {
            this.refresh();
          }
        }

        return true;
      },

      finalizeQueryParamChange: function (params, finalParams, transition) {
        if (this.fullRouteName !== 'application') {
          return true;
        }

        // Transition object is absent for intermediate transitions.
        if (!transition) {
          return;
        }

        var handlerInfos = transition.state.handlerInfos;
        var router = this.router;
        var qpMeta = router._queryParamsFor(handlerInfos);
        var changes = router._qpUpdates;
        var replaceUrl = undefined;

        _emberRoutingUtils.stashParamNames(router, handlerInfos);

        for (var i = 0; i < qpMeta.qps.length; ++i) {
          var qp = qpMeta.qps[i];
          var route = qp.route;
          var controller = route.controller;
          var presentKey = qp.urlKey in params && qp.urlKey;

          // Do a reverse lookup to see if the changed query
          // param URL key corresponds to a QP property on
          // this controller.
          var value = undefined,
              svalue = undefined;
          if (changes && qp.urlKey in changes) {
            // Value updated in/before setupController
            value = _emberMetal.get(controller, qp.prop);
            svalue = route.serializeQueryParam(value, qp.urlKey, qp.type);
          } else {
            if (presentKey) {
              svalue = params[presentKey];
              value = route.deserializeQueryParam(svalue, qp.urlKey, qp.type);
            } else {
              // No QP provided; use default value.
              svalue = qp.serializedDefaultValue;
              value = copyDefaultValue(qp.defaultValue);
            }
          }

          controller._qpDelegate = _emberMetal.get(route, '_qp.states.inactive');

          var thisQueryParamChanged = svalue !== qp.serializedValue;
          if (thisQueryParamChanged) {
            if (transition.queryParamsOnly && replaceUrl !== false) {
              var options = route._optionsForQueryParam(qp);
              var replaceConfigValue = _emberMetal.get(options, 'replace');
              if (replaceConfigValue) {
                replaceUrl = true;
              } else if (replaceConfigValue === false) {
                // Explicit pushState wins over any other replaceStates.
                replaceUrl = false;
              }
            }

            _emberMetal.set(controller, qp.prop, value);
          }

          // Stash current serialized value of controller.
          qp.serializedValue = svalue;

          var thisQueryParamHasDefaultValue = qp.serializedDefaultValue === svalue;
          if (!thisQueryParamHasDefaultValue) {
            finalParams.push({
              value: svalue,
              visible: true,
              key: presentKey || qp.urlKey
            });
          }
        }

        if (replaceUrl) {
          transition.method('replace');
        }

        qpMeta.qps.forEach(function (qp) {
          var routeQpMeta = _emberMetal.get(qp.route, '_qp');
          var finalizedController = qp.route.controller;
          finalizedController._qpDelegate = _emberMetal.get(routeQpMeta, 'states.active');
        });

        router._qpUpdates = null;
      }
    },

    /**
      This hook is executed when the router completely exits this route. It is
      not executed when the model for the route changes.
       @method deactivate
      @since 1.0.0
      @public
    */
    deactivate: K,

    /**
      This hook is executed when the router enters the route. It is not executed
      when the model for the route changes.
       @method activate
      @since 1.0.0
      @public
    */
    activate: K,

    /**
      Transition the application into another route. The route may
      be either a single route or route path:
       ```javascript
      this.transitionTo('blogPosts');
      this.transitionTo('blogPosts.recentEntries');
      ```
       Optionally supply a model for the route in question. The model
      will be serialized into the URL using the `serialize` hook of
      the route:
       ```javascript
      this.transitionTo('blogPost', aPost);
      ```
       If a literal is passed (such as a number or a string), it will
      be treated as an identifier instead. In this case, the `model`
      hook of the route will be triggered:
       ```javascript
      this.transitionTo('blogPost', 1);
      ```
       Multiple models will be applied last to first recursively up the
      route tree.
       ```app/routes.js
      // ...
       Router.map(function() {
        this.route('blogPost', { path:':blogPostId' }, function() {
          this.route('blogComment', { path: ':blogCommentId' });
        });
      });
       export default Router;
      ```
       ```javascript
      this.transitionTo('blogComment', aPost, aComment);
      this.transitionTo('blogComment', 1, 13);
      ```
       It is also possible to pass a URL (a string that starts with a
      `/`). This is intended for testing and debugging purposes and
      should rarely be used in production code.
       ```javascript
      this.transitionTo('/');
      this.transitionTo('/blog/post/1/comment/13');
      this.transitionTo('/blog/posts?sort=title');
      ```
       An options hash with a `queryParams` property may be provided as
      the final argument to add query parameters to the destination URL.
       ```javascript
      this.transitionTo('blogPost', 1, {
        queryParams: { showComments: 'true' }
      });
       // if you just want to transition the query parameters without changing the route
      this.transitionTo({ queryParams: { sort: 'date' } });
      ```
       See also [replaceWith](#method_replaceWith).
       Simple Transition Example
       ```app/routes.js
      // ...
       Router.map(function() {
        this.route('index');
        this.route('secret');
        this.route('fourOhFour', { path: '*:' });
      });
       export default Router;
      ```
       ```app/routes/index.js
      import Ember from 'ember':
       export Ember.Route.extend({
        actions: {
          moveToSecret(context) {
            if (authorized()) {
              this.transitionTo('secret', context);
            } else {
              this.transitionTo('fourOhFour');
            }
          }
        }
      });
      ```
       Transition to a nested route
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('articles', { path: '/articles' }, function() {
          this.route('new');
        });
      });
       export default Router;
      ```
       ```app/routes/index.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          transitionToNewArticle() {
            this.transitionTo('articles.new');
          }
        }
      });
      ```
       Multiple Models Example
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('index');
         this.route('breakfast', { path: ':breakfastId' }, function() {
          this.route('cereal', { path: ':cerealId' });
        });
      });
       export default Router;
      ```
       ```app/routes/index.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          moveToChocolateCereal() {
            let cereal = { cerealId: 'ChocolateYumminess' };
            let breakfast = { breakfastId: 'CerealAndMilk' };
             this.transitionTo('breakfast.cereal', breakfast, cereal);
          }
        }
      });
      ```
       Nested Route with Query String Example
       ```app/routes.js
      // ...
       Router.map(function() {
        this.route('fruits', function() {
          this.route('apples');
        });
      });
       export default Router;
      ```
       ```app/routes/index.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          transitionToApples() {
            this.transitionTo('fruits.apples', { queryParams: { color: 'red' } });
          }
        }
      });
      ```
       @method transitionTo
      @param {String} name the name of the route or a URL
      @param {...Object} models the model(s) or identifier(s) to be used while
        transitioning to the route.
      @param {Object} [options] optional hash with a queryParams property
        containing a mapping of query parameters
      @return {Transition} the transition object associated with this
        attempted transition
      @since 1.0.0
      @public
    */
    transitionTo: function (name, context) {
      var router = this.router;
      return router.transitionTo.apply(router, _emberRoutingUtils.prefixRouteNameArg(this, arguments));
    },

    /**
      Perform a synchronous transition into another route without attempting
      to resolve promises, update the URL, or abort any currently active
      asynchronous transitions (i.e. regular transitions caused by
      `transitionTo` or URL changes).
       This method is handy for performing intermediate transitions on the
      way to a final destination route, and is called internally by the
      default implementations of the `error` and `loading` handlers.
       @method intermediateTransitionTo
      @param {String} name the name of the route
      @param {...Object} models the model(s) to be used while transitioning
      to the route.
      @since 1.2.0
      @public
     */
    intermediateTransitionTo: function () {
      var router = this.router;
      router.intermediateTransitionTo.apply(router, _emberRoutingUtils.prefixRouteNameArg(this, arguments));
    },

    /**
      Refresh the model on this route and any child routes, firing the
      `beforeModel`, `model`, and `afterModel` hooks in a similar fashion
      to how routes are entered when transitioning in from other route.
      The current route params (e.g. `article_id`) will be passed in
      to the respective model hooks, and if a different model is returned,
      `setupController` and associated route hooks will re-fire as well.
       An example usage of this method is re-querying the server for the
      latest information using the same parameters as when the route
      was first entered.
       Note that this will cause `model` hooks to fire even on routes
      that were provided a model object when the route was initially
      entered.
       @method refresh
      @return {Transition} the transition object associated with this
        attempted transition
      @since 1.4.0
      @public
     */
    refresh: function () {
      return this.router.router.refresh(this);
    },

    /**
      Transition into another route while replacing the current URL, if possible.
      This will replace the current history entry instead of adding a new one.
      Beside that, it is identical to `transitionTo` in all other respects. See
      'transitionTo' for additional information regarding multiple models.
       Example
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('index');
        this.route('secret');
      });
       export default Router;
      ```
       ```app/routes/secret.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        afterModel() {
          if (!authorized()){
            this.replaceWith('index');
          }
        }
      });
      ```
       @method replaceWith
      @param {String} name the name of the route or a URL
      @param {...Object} models the model(s) or identifier(s) to be used while
        transitioning to the route.
      @return {Transition} the transition object associated with this
        attempted transition
      @since 1.0.0
      @public
    */
    replaceWith: function () {
      var router = this.router;
      return router.replaceWith.apply(router, _emberRoutingUtils.prefixRouteNameArg(this, arguments));
    },

    /**
      Sends an action to the router, which will delegate it to the currently
      active route hierarchy per the bubbling rules explained under `actions`.
       Example
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('index');
      });
       export default Router;
      ```
       ```app/routes/application.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          track(arg) {
            console.log(arg, 'was clicked');
          }
        }
      });
      ```
       ```app/routes/index.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        actions: {
          trackIfDebug(arg) {
            if (debug) {
              this.send('track', arg);
            }
          }
        }
      });
      ```
       @method send
      @param {String} name the name of the action to trigger
      @param {...*} args
      @since 1.0.0
      @public
    */
    send: function () {
      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      if (this.router && this.router.router || !_emberMetal.isTesting()) {
        var _router;

        (_router = this.router).send.apply(_router, args);
      } else {
        var _name2 = args[0];
        args = slice.call(args, 1);
        var action = this.actions[_name2];
        if (action) {
          return this.actions[_name2].apply(this, args);
        }
      }
    },

    /**
      This hook is the entry point for router.js
       @private
      @method setup
    */
    setup: function (context, transition) {
      var _this2 = this;

      var controller = undefined;

      var controllerName = this.controllerName || this.routeName;
      var definedController = this.controllerFor(controllerName, true);

      if (!definedController) {
        controller = this.generateController(controllerName);
      } else {
        controller = definedController;
      }

      // Assign the route's controller so that it can more easily be
      // referenced in action handlers. Side effects. Side effects everywhere.
      if (!this.controller) {
        var propNames = _emberMetal.get(this, '_qp.propertyNames');
        addQueryParamsObservers(controller, propNames);
        this.controller = controller;
      }

      var queryParams = _emberMetal.get(this, '_qp');

      var states = queryParams.states;

      controller._qpDelegate = states.allowOverrides;

      if (transition) {
        (function () {
          // Update the model dep values used to calculate cache keys.
          _emberRoutingUtils.stashParamNames(_this2.router, transition.state.handlerInfos);

          var params = transition.params;
          var allParams = queryParams.propertyNames;
          var cache = _this2._bucketCache;

          allParams.forEach(function (prop) {
            var aQp = queryParams.map[prop];

            aQp.values = params;
            var cacheKey = _emberRoutingUtils.calculateCacheKey(aQp.controllerName, aQp.parts, aQp.values);

            if (cache) {
              var value = cache.lookup(cacheKey, prop, aQp.undecoratedDefaultValue);
              _emberMetal.set(controller, prop, value);
            }
          });
        })();
      }

      if (transition) {
        var qpValues = getQueryParamsFor(this, transition.state);
        controller.setProperties(qpValues);
      }

      this.setupController(controller, context, transition);

      if (this._environment.options.shouldRender) {
        this.renderTemplate(controller, context);
      }
    },

    /*
      Called when a query parameter for this route changes, regardless of whether the route
      is currently part of the active route hierarchy. This will update the query parameter's
      value in the cache so if this route becomes active, the cache value has been updated.
    */
    _qpChanged: function (prop, value, qp) {
      if (!qp) {
        return;
      }

      var cacheKey = _emberRoutingUtils.calculateCacheKey(qp.controllerName, qp.parts, qp.values);

      // Update model-dep cache
      var cache = this._bucketCache;
      if (cache) {
        cache.stash(cacheKey, prop, value);
      }
    },

    /**
      This hook is the first of the route entry validation hooks
      called when an attempt is made to transition into a route
      or one of its children. It is called before `model` and
      `afterModel`, and is appropriate for cases when:
       1) A decision can be made to redirect elsewhere without
         needing to resolve the model first.
      2) Any async operations need to occur first before the
         model is attempted to be resolved.
       This hook is provided the current `transition` attempt
      as a parameter, which can be used to `.abort()` the transition,
      save it for a later `.retry()`, or retrieve values set
      on it from a previous hook. You can also just call
      `this.transitionTo` to another route to implicitly
      abort the `transition`.
       You can return a promise from this hook to pause the
      transition until the promise resolves (or rejects). This could
      be useful, for instance, for retrieving async code from
      the server that is required to enter a route.
       @method beforeModel
      @param {Transition} transition
      @return {Promise} if the value returned from this hook is
        a promise, the transition will pause until the transition
        resolves. Otherwise, non-promise return values are not
        utilized in any way.
      @since 1.0.0
      @public
    */
    beforeModel: K,

    /**
      This hook is called after this route's model has resolved.
      It follows identical async/promise semantics to `beforeModel`
      but is provided the route's resolved model in addition to
      the `transition`, and is therefore suited to performing
      logic that can only take place after the model has already
      resolved.
       ```app/routes/posts.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        afterModel(posts, transition) {
          if (posts.get('length') === 1) {
            this.transitionTo('post.show', posts.get('firstObject'));
          }
        }
      });
      ```
       Refer to documentation for `beforeModel` for a description
      of transition-pausing semantics when a promise is returned
      from this hook.
       @method afterModel
      @param {Object} resolvedModel the value returned from `model`,
        or its resolved value if it was a promise
      @param {Transition} transition
      @return {Promise} if the value returned from this hook is
        a promise, the transition will pause until the transition
        resolves. Otherwise, non-promise return values are not
        utilized in any way.
      @since 1.0.0
      @public
     */
    afterModel: K,

    /**
      A hook you can implement to optionally redirect to another route.
       If you call `this.transitionTo` from inside of this hook, this route
      will not be entered in favor of the other hook.
       `redirect` and `afterModel` behave very similarly and are
      called almost at the same time, but they have an important
      distinction in the case that, from one of these hooks, a
      redirect into a child route of this route occurs: redirects
      from `afterModel` essentially invalidate the current attempt
      to enter this route, and will result in this route's `beforeModel`,
      `model`, and `afterModel` hooks being fired again within
      the new, redirecting transition. Redirects that occur within
      the `redirect` hook, on the other hand, will _not_ cause
      these hooks to be fired again the second time around; in
      other words, by the time the `redirect` hook has been called,
      both the resolved model and attempted entry into this route
      are considered to be fully validated.
       @method redirect
      @param {Object} model the model for this route
      @param {Transition} transition the transition object associated with the current transition
      @since 1.0.0
      @public
    */
    redirect: K,

    /**
      Called when the context is changed by router.js.
       @private
      @method contextDidChange
    */
    contextDidChange: function () {
      this.currentModel = this.context;
    },

    /**
      A hook you can implement to convert the URL into the model for
      this route.
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('post', { path: '/posts/:post_id' });
      });
       export default Router;
      ```
       The model for the `post` route is `store.findRecord('post', params.post_id)`.
       By default, if your route has a dynamic segment ending in `_id`:
       * The model class is determined from the segment (`post_id`'s
        class is `App.Post`)
      * The find method is called on the model class with the value of
        the dynamic segment.
       Note that for routes with dynamic segments, this hook is not always
      executed. If the route is entered through a transition (e.g. when
      using the `link-to` Handlebars helper or the `transitionTo` method
      of routes), and a model context is already provided this hook
      is not called.
       A model context does not include a primitive string or number,
      which does cause the model hook to be called.
       Routes without dynamic segments will always execute the model hook.
       ```javascript
      // no dynamic segment, model hook always called
      this.transitionTo('posts');
       // model passed in, so model hook not called
      thePost = store.findRecord('post', 1);
      this.transitionTo('post', thePost);
       // integer passed in, model hook is called
      this.transitionTo('post', 1);
       // model id passed in, model hook is called
      // useful for forcing the hook to execute
      thePost = store.findRecord('post', 1);
      this.transitionTo('post', thePost.id);
      ```
        This hook follows the asynchronous/promise semantics
      described in the documentation for `beforeModel`. In particular,
      if a promise returned from `model` fails, the error will be
      handled by the `error` hook on `Ember.Route`.
       Example
       ```app/routes/post.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        model(params) {
          return this.store.findRecord('post', params.post_id);
        }
      });
      ```
       @method model
      @param {Object} params the parameters extracted from the URL
      @param {Transition} transition
      @return {Object|Promise} the model for this route. If
        a promise is returned, the transition will pause until
        the promise resolves, and the resolved value of the promise
        will be used as the model for this route.
      @since 1.0.0
      @public
    */
    model: function (params, transition) {
      var name = undefined,
          sawParams = undefined,
          value = undefined;
      var queryParams = _emberMetal.get(this, '_qp.map');

      for (var prop in params) {
        if (prop === 'queryParams' || queryParams && prop in queryParams) {
          continue;
        }

        var match = prop.match(/^(.*)_id$/);
        if (match) {
          name = match[1];
          value = params[prop];
        }
        sawParams = true;
      }

      if (!name && sawParams) {
        return _emberRuntime.copy(params);
      } else if (!name) {
        if (transition.resolveIndex < 1) {
          return;
        }

        var parentModel = transition.state.handlerInfos[transition.resolveIndex - 1].context;

        return parentModel;
      }

      return this.findModel(name, value);
    },

    /**
      @private
      @method deserialize
      @param {Object} params the parameters extracted from the URL
      @param {Transition} transition
      @return {Object|Promise} the model for this route.
       Router.js hook.
     */
    deserialize: function (params, transition) {
      return this.model(this.paramsFor(this.routeName), transition);
    },

    /**
       @method findModel
      @param {String} type the model type
      @param {Object} value the value passed to find
      @private
    */
    findModel: function () {
      var store = _emberMetal.get(this, 'store');
      return store.find.apply(store, arguments);
    },

    /**
      Store property provides a hook for data persistence libraries to inject themselves.
       By default, this store property provides the exact same functionality previously
      in the model hook.
       Currently, the required interface is:
       `store.find(modelName, findArguments)`
       @method store
      @param {Object} store
      @private
    */
    store: _emberMetal.computed(function () {
      var owner = _emberUtils.getOwner(this);
      var routeName = this.routeName;
      var namespace = _emberMetal.get(this, 'router.namespace');

      return {
        find: function (name, value) {
          var modelClass = owner._lookupFactory('model:' + name);

          if (!modelClass) {
            return;
          }

          return modelClass.find(value);
        }
      };
    }),

    /**
      A hook you can implement to convert the route's model into parameters
      for the URL.
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('post', { path: '/posts/:post_id' });
      });
       ```
       ```app/routes/post.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        model(params) {
          // the server returns `{ id: 12 }`
          return Ember.$.getJSON('/posts/' + params.post_id);
        },
         serialize(model) {
          // this will make the URL `/posts/12`
          return { post_id: model.id };
        }
      });
      ```
       The default `serialize` method will insert the model's `id` into the
      route's dynamic segment (in this case, `:post_id`) if the segment contains '_id'.
      If the route has multiple dynamic segments or does not contain '_id', `serialize`
      will return `Ember.getProperties(model, params)`
       This method is called when `transitionTo` is called with a context
      in order to populate the URL.
       @method serialize
      @param {Object} model the routes model
      @param {Array} params an Array of parameter names for the current
        route (in the example, `['post_id']`.
      @return {Object} the serialized parameters
      @since 1.0.0
      @public
    */
    serialize: defaultSerialize,

    /**
      A hook you can use to setup the controller for the current route.
       This method is called with the controller for the current route and the
      model supplied by the `model` hook.
       By default, the `setupController` hook sets the `model` property of
      the controller to the `model`.
       If you implement the `setupController` hook in your Route, it will
      prevent this default behavior. If you want to preserve that behavior
      when implementing your `setupController` function, make sure to call
      `_super`:
       ```app/routes/photos.js
      import Ember from 'ebmer';
       export default Ember.Route.extend({
        model() {
          return this.store.findAll('photo');
        },
         setupController(controller, model) {
          // Call _super for default behavior
          this._super(controller, model);
          // Implement your custom setup after
          this.controllerFor('application').set('showingPhotos', true);
        }
      });
      ```
       The provided controller will be one resolved based on the name
      of this route.
       If no explicit controller is defined, Ember will automatically create one.
       As an example, consider the router:
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('post', { path: '/posts/:post_id' });
      });
       export default Router;
      ```
       For the `post` route, a controller named `App.PostController` would
      be used if it is defined. If it is not defined, a basic `Ember.Controller`
      instance would be used.
       Example
       ```app/routes/post.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        setupController(controller, model) {
          controller.set('model', model);
        }
      });
      ```
       @method setupController
      @param {Controller} controller instance
      @param {Object} model
      @since 1.0.0
      @public
    */
    setupController: function (controller, context, transition) {
      if (controller && context !== undefined) {
        _emberMetal.set(controller, 'model', context);
      }
    },

    /**
      Returns the resolved model of the current route, or a parent (or any ancestor)
      route in a route hierarchy.
       The controller instance must already have been created, either through entering the
      associated route or using `generateController`.
       ```app/routes/post.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        setupController(controller, post) {
          this._super(controller, post);
          this.controllerFor('posts').set('currentPost', post);
        }
      });
      ```
       @method controllerFor
      @param {String} name the name of the route or controller
      @return {Ember.Controller}
      @since 1.0.0
      @public
    */
    controllerFor: function (name, _skipAssert) {
      var owner = _emberUtils.getOwner(this);
      var route = owner.lookup('route:' + name);
      var controller = undefined;

      if (route && route.controllerName) {
        name = route.controllerName;
      }

      controller = owner.lookup('controller:' + name);

      // NOTE: We're specifically checking that skipAssert is true, because according
      //   to the old API the second parameter was model. We do not want people who
      //   passed a model to skip the assertion.

      return controller;
    },

    /**
      Generates a controller for a route.
       Example
       ```app/routes/post.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        setupController(controller, post) {
          this._super(controller, post);
          this.generateController('posts');
        }
      });
      ```
       @method generateController
      @param {String} name the name of the controller
      @private
    */
    generateController: function (name) {
      var owner = _emberUtils.getOwner(this);

      return _emberRoutingSystemGenerate_controller.default(owner, name);
    },

    /**
      Returns the resolved model of a parent (or any ancestor) route
      in a route hierarchy.  During a transition, all routes
      must resolve a model object, and if a route
      needs access to a parent route's model in order to
      resolve a model (or just reuse the model from a parent),
      it can call `this.modelFor(theNameOfParentRoute)` to
      retrieve it. If the ancestor route's model was a promise,
      its resolved result is returned.
       Example
       ```app/router.js
      // ...
       Router.map(function() {
          this.route('post', { path: '/post/:post_id' }, function() {
            this.route('comments', { resetNamespace: true });
          });
      });
       export default Router;
      ```
       ```app/routes/comments.js
      import Ember from 'ember';
       export default Ember.Route.extend({
          afterModel() {
            this.set('post', this.modelFor('post'));
          }
      });
      ```
       @method modelFor
      @param {String} name the name of the route
      @return {Object} the model object
      @since 1.0.0
      @public
    */
    modelFor: function (_name) {
      var name = undefined;
      var owner = _emberUtils.getOwner(this);

      // Only change the route name when there is an active transition.
      // Otherwise, use the passed in route name.
      if (owner.routable && this.router && this.router.router.activeTransition) {
        name = getEngineRouteName(owner, _name);
      } else {
        name = _name;
      }

      var route = _emberUtils.getOwner(this).lookup('route:' + name);
      var transition = this.router ? this.router.router.activeTransition : null;

      // If we are mid-transition, we want to try and look up
      // resolved parent contexts on the current transitionEvent.
      if (transition) {
        var modelLookupName = route && route.routeName || name;
        if (transition.resolvedModels.hasOwnProperty(modelLookupName)) {
          return transition.resolvedModels[modelLookupName];
        }
      }

      return route && route.currentModel;
    },

    /**
      A hook you can use to render the template for the current route.
       This method is called with the controller for the current route and the
      model supplied by the `model` hook. By default, it renders the route's
      template, configured with the controller for the route.
       This method can be overridden to set up and render additional or
      alternative templates.
       ```app/routes/posts.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        renderTemplate(controller, model) {
          let favController = this.controllerFor('favoritePost');
           // Render the `favoritePost` template into
          // the outlet `posts`, and display the `favoritePost`
          // controller.
          this.render('favoritePost', {
            outlet: 'posts',
            controller: favController
          });
        }
      });
      ```
       @method renderTemplate
      @param {Object} controller the route's controller
      @param {Object} model the route's model
      @since 1.0.0
      @public
    */
    renderTemplate: function (controller, model) {
      this.render();
    },

    /**
      `render` is used to render a template into a region of another template
      (indicated by an `{{outlet}}`). `render` is used both during the entry
      phase of routing (via the `renderTemplate` hook) and later in response to
      user interaction.
       For example, given the following minimal router and templates:
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('photos');
      });
       export default Router;
      ```
       ```handlebars
      <!-- application.hbs -->
      <div class='something-in-the-app-hbs'>
        {{outlet "anOutletName"}}
      </div>
      ```
       ```handlebars
      <!-- photos.hbs -->
      <h1>Photos</h1>
      ```
       You can render `photos.hbs` into the `"anOutletName"` outlet of
      `application.hbs` by calling `render`:
       ```app/routes/post.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        renderTemplate() {
          this.render('photos', {
            into: 'application',
            outlet: 'anOutletName'
          })
        }
      });
      ```
       `render` additionally allows you to supply which `controller` and
      `model` objects should be loaded and associated with the rendered template.
        ```app/routes/posts.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        renderTemplate(controller, model){
          this.render('posts', {    // the template to render, referenced by name
            into: 'application',    // the template to render into, referenced by name
            outlet: 'anOutletName', // the outlet inside `options.template` to render into.
            controller: 'someControllerName', // the controller to use for this template, referenced by name
            model: model            // the model to set on `options.controller`.
          })
        }
      });
      ```
       The string values provided for the template name, and controller
      will eventually pass through to the resolver for lookup. See
      Ember.Resolver for how these are mapped to JavaScript objects in your
      application. The template to render into needs to be related to  either the
      current route or one of its ancestors.
       Not all options need to be passed to `render`. Default values will be used
      based on the name of the route specified in the router or the Route's
      `controllerName` and `templateName` properties.
       For example:
       ```app/router.js
      // ...
       Router.map(function() {
        this.route('index');
        this.route('post', { path: '/posts/:post_id' });
      });
       export default Router;
      ```
       ```app/routes/post.js
      import Ember from 'ember';
       export default Ember.Route.extend({
        renderTemplate() {
          this.render(); // all defaults apply
        }
      });
      ```
       The name of the route, defined by the router, is `post`.
       The following equivalent default options will be applied when
      the Route calls `render`:
       ```javascript
      this.render('post', {  // the template name associated with 'post' Route
        into: 'application', // the parent route to 'post' Route
        outlet: 'main',      // {{outlet}} and {{outlet 'main'}} are synonymous,
        controller: 'post',  // the controller associated with the 'post' Route
      })
      ```
       By default the controller's `model` will be the route's model, so it does not
      need to be passed unless you wish to change which model is being used.
       @method render
      @param {String} name the name of the template to render
      @param {Object} [options] the options
      @param {String} [options.into] the template to render into,
                      referenced by name. Defaults to the parent template
      @param {String} [options.outlet] the outlet inside `options.template` to render into.
                      Defaults to 'main'
      @param {String|Object} [options.controller] the controller to use for this template,
                      referenced by name or as a controller instance. Defaults to the Route's paired controller
      @param {Object} [options.model] the model object to set on `options.controller`.
                      Defaults to the return value of the Route's model hook
      @since 1.0.0
      @public
    */
    render: function (_name, options) {

      var namePassed = typeof _name === 'string' && !!_name;
      var isDefaultRender = arguments.length === 0 || _emberMetal.isEmpty(arguments[0]);
      var name = undefined;

      if (typeof _name === 'object' && !options) {
        name = this.templateName || this.routeName;
        options = _name;
      } else {
        name = _name;
      }

      var renderOptions = buildRenderOptions(this, namePassed, isDefaultRender, name, options);
      this.connections.push(renderOptions);
      _emberMetal.run.once(this.router, '_setOutlets');
    },

    /**
      Disconnects a view that has been rendered into an outlet.
       You may pass any or all of the following options to `disconnectOutlet`:
       * `outlet`: the name of the outlet to clear (default: 'main')
      * `parentView`: the name of the view containing the outlet to clear
         (default: the view rendered by the parent route)
       Example:
       ```app/routes/application.js
      import Ember from 'ember';
       export default App.Route.extend({
        actions: {
          showModal(evt) {
            this.render(evt.modalName, {
              outlet: 'modal',
              into: 'application'
            });
          },
          hideModal(evt) {
            this.disconnectOutlet({
              outlet: 'modal',
              parentView: 'application'
            });
          }
        }
      });
      ```
       Alternatively, you can pass the `outlet` name directly as a string.
       Example:
       ```app/routes/application.js
      import Ember from 'ember';
       export default App.Route.extend({
        actions: {
          showModal(evt) {
            // ...
          },
          hideModal(evt) {
            this.disconnectOutlet('modal');
          }
        }
      });
       @method disconnectOutlet
      @param {Object|String} options the options hash or outlet name
      @since 1.0.0
      @public
    */
    disconnectOutlet: function (options) {
      var outletName = undefined;
      var parentView = undefined;
      if (!options || typeof options === 'string') {
        outletName = options;
      } else {
        outletName = options.outlet;
        parentView = options.parentView;

        if (options && Object.keys(options).indexOf('outlet') !== -1 && typeof options.outlet === 'undefined') {
          throw new _emberMetal.Error('You passed undefined as the outlet name.');
        }
      }
      parentView = parentView && parentView.replace(/\//g, '.');
      outletName = outletName || 'main';
      this._disconnectOutlet(outletName, parentView);
      for (var i = 0; i < this.router.router.currentHandlerInfos.length; i++) {
        // This non-local state munging is sadly necessary to maintain
        // backward compatibility with our existing semantics, which allow
        // any route to disconnectOutlet things originally rendered by any
        // other route. This should all get cut in 2.0.
        this.router.router.currentHandlerInfos[i].handler._disconnectOutlet(outletName, parentView);
      }
    },

    _disconnectOutlet: function (outletName, parentView) {
      var parent = parentRoute(this);
      if (parent && parentView === parent.routeName) {
        parentView = undefined;
      }
      for (var i = 0; i < this.connections.length; i++) {
        var connection = this.connections[i];
        if (connection.outlet === outletName && connection.into === parentView) {
          // This neuters the disconnected outlet such that it doesn't
          // render anything, but it leaves an entry in the outlet
          // hierarchy so that any existing other renders that target it
          // don't suddenly blow up. They will still stick themselves
          // into its outlets, which won't render anywhere. All of this
          // statefulness should get the machete in 2.0.
          this.connections[i] = {
            owner: connection.owner,
            into: connection.into,
            outlet: connection.outlet,
            name: connection.name,
            controller: undefined,
            template: undefined,
            ViewClass: undefined
          };
          _emberMetal.run.once(this.router, '_setOutlets');
        }
      }
    },

    willDestroy: function () {
      this.teardownViews();
    },

    /**
      @private
       @method teardownViews
    */
    teardownViews: function () {
      if (this.connections && this.connections.length > 0) {
        this.connections = [];
        _emberMetal.run.once(this.router, '_setOutlets');
      }
    }
  });

  _emberRuntime.deprecateUnderscoreActions(Route);

  Route.reopenClass({
    isRouteFactory: true
  });

  function parentRoute(route) {
    var handlerInfo = handlerInfoFor(route, route.router.router.state.handlerInfos, -1);
    return handlerInfo && handlerInfo.handler;
  }

  function handlerInfoFor(route, handlerInfos, _offset) {
    if (!handlerInfos) {
      return;
    }

    var offset = _offset || 0;
    var current = undefined;
    for (var i = 0; i < handlerInfos.length; i++) {
      current = handlerInfos[i].handler;
      if (current === route) {
        return handlerInfos[i + offset];
      }
    }
  }

  function buildRenderOptions(route, namePassed, isDefaultRender, _name, options) {
    var into = options && options.into && options.into.replace(/\//g, '.');
    var outlet = options && options.outlet || 'main';

    var name = undefined,
        templateName = undefined;
    if (_name) {
      name = _name.replace(/\//g, '.');
      templateName = name;
    } else {
      name = route.routeName;
      templateName = route.templateName || name;
    }

    var owner = _emberUtils.getOwner(route);
    var controller = options && options.controller;
    if (!controller) {
      if (namePassed) {
        controller = owner.lookup('controller:' + name) || route.controllerName || route.routeName;
      } else {
        controller = route.controllerName || owner.lookup('controller:' + name);
      }
    }

    if (typeof controller === 'string') {
      var controllerName = controller;
      controller = owner.lookup('controller:' + controllerName);
      if (!controller) {
        throw new _emberMetal.Error('You passed `controller: \'' + controllerName + '\'` into the `render` method, but no such controller could be found.');
      }
    }

    if (options && Object.keys(options).indexOf('outlet') !== -1 && typeof options.outlet === 'undefined') {
      throw new _emberMetal.Error('You passed undefined as the outlet name.');
    }

    if (options && options.model) {
      controller.set('model', options.model);
    }

    var template = owner.lookup('template:' + templateName);

    var parent = undefined;
    if (into && (parent = parentRoute(route)) && into === parent.routeName) {
      into = undefined;
    }

    var renderOptions = {
      owner: owner,
      into: into,
      outlet: outlet,
      name: name,
      controller: controller,
      template: template || route._topLevelViewTemplate,
      ViewClass: undefined
    };

    var LOG_VIEW_LOOKUPS = _emberMetal.get(route.router, 'namespace.LOG_VIEW_LOOKUPS');
    if (LOG_VIEW_LOOKUPS && !template) {}

    return renderOptions;
  }

  function getFullQueryParams(router, state) {
    if (state.fullQueryParams) {
      return state.fullQueryParams;
    }

    state.fullQueryParams = {};
    _emberUtils.assign(state.fullQueryParams, state.queryParams);

    router._deserializeQueryParams(state.handlerInfos, state.fullQueryParams);
    return state.fullQueryParams;
  }

  function getQueryParamsFor(route, state) {
    state.queryParamsFor = state.queryParamsFor || {};
    var name = route.fullRouteName;

    if (state.queryParamsFor[name]) {
      return state.queryParamsFor[name];
    }

    var fullQueryParams = getFullQueryParams(route.router, state);

    var params = state.queryParamsFor[name] = {};

    // Copy over all the query params for this route/controller into params hash.
    var qpMeta = _emberMetal.get(route, '_qp');
    var qps = qpMeta.qps;
    for (var i = 0; i < qps.length; ++i) {
      // Put deserialized qp on params hash.
      var qp = qps[i];

      var qpValueWasPassedIn = (qp.prop in fullQueryParams);
      params[qp.prop] = qpValueWasPassedIn ? fullQueryParams[qp.prop] : copyDefaultValue(qp.defaultValue);
    }

    return params;
  }

  function copyDefaultValue(value) {
    if (Array.isArray(value)) {
      return _emberRuntime.A(value.slice());
    }
    return value;
  }

  /*
    Merges all query parameters from a controller with those from
    a route, returning a new object and avoiding any mutations to
    the existing objects.
  */
  function mergeEachQueryParams(controllerQP, routeQP) {
    var keysAlreadyMergedOrSkippable = undefined;
    var qps = {};

    keysAlreadyMergedOrSkippable = {
      defaultValue: true,
      type: true,
      scope: true,
      as: true
    };

    // first loop over all controller qps, merging them with any matching route qps
    // into a new empty object to avoid mutating.
    for (var cqpName in controllerQP) {
      if (!controllerQP.hasOwnProperty(cqpName)) {
        continue;
      }

      var newControllerParameterConfiguration = {};
      _emberUtils.assign(newControllerParameterConfiguration, controllerQP[cqpName]);
      _emberUtils.assign(newControllerParameterConfiguration, routeQP[cqpName]);

      qps[cqpName] = newControllerParameterConfiguration;

      // allows us to skip this QP when we check route QPs.
      keysAlreadyMergedOrSkippable[cqpName] = true;
    }

    // loop over all route qps, skipping those that were merged in the first pass
    // because they also appear in controller qps
    for (var rqpName in routeQP) {
      if (!routeQP.hasOwnProperty(rqpName) || keysAlreadyMergedOrSkippable[rqpName]) {
        continue;
      }

      var newRouteParameterConfiguration = {};
      _emberUtils.assign(newRouteParameterConfiguration, routeQP[rqpName], controllerQP[rqpName]);
      qps[rqpName] = newRouteParameterConfiguration;
    }

    return qps;
  }

  function addQueryParamsObservers(controller, propNames) {
    propNames.forEach(function (prop) {
      controller.addObserver(prop + '.[]', controller, controller._qpChanged);
    });
  }

  function getEngineRouteName(engine, routeName) {
    if (engine.routable) {
      var prefix = engine.mountPoint;

      if (routeName === 'application') {
        return prefix;
      } else {
        return prefix + '.' + routeName;
      }
    }

    return routeName;
  }

  exports.default = Route;
});
enifed('ember-routing/system/router', ['exports', 'ember-utils', 'ember-console', 'ember-metal', 'ember-runtime', 'ember-routing/system/route', 'ember-routing/system/dsl', 'ember-routing/location/api', 'ember-routing/utils', 'ember-routing/system/router_state', 'router'], function (exports, _emberUtils, _emberConsole, _emberMetal, _emberRuntime, _emberRoutingSystemRoute, _emberRoutingSystemDsl, _emberRoutingLocationApi, _emberRoutingUtils, _emberRoutingSystemRouter_state, _router4) {
  'use strict';

  exports.triggerEvent = triggerEvent;

  function K() {
    return this;
  }

  var slice = Array.prototype.slice;

  /**
    The `Ember.Router` class manages the application state and URLs. Refer to
    the [routing guide](http://emberjs.com/guides/routing/) for documentation.
  
    @class Router
    @namespace Ember
    @extends Ember.Object
    @uses Ember.Evented
    @public
  */
  var EmberRouter = _emberRuntime.Object.extend(_emberRuntime.Evented, {
    /**
      The `location` property determines the type of URL's that your
      application will use.
       The following location types are currently available:
       * `history` - use the browser's history API to make the URLs look just like any standard URL
      * `hash` - use `#` to separate the server part of the URL from the Ember part: `/blog/#/posts/new`
      * `none` - do not store the Ember URL in the actual browser URL (mainly used for testing)
      * `auto` - use the best option based on browser capabilites: `history` if possible, then `hash` if possible, otherwise `none`
       Note: If using ember-cli, this value is defaulted to `auto` by the `locationType` setting of `/config/environment.js`
       @property location
      @default 'hash'
      @see {Ember.Location}
      @public
    */
    location: 'hash',

    /**
     Represents the URL of the root of the application, often '/'. This prefix is
     assumed on all routes defined on this router.
      @property rootURL
     @default '/'
     @public
    */
    rootURL: '/',

    _initRouterJs: function () {
      var router = this.router = new _router4.default();
      router.triggerEvent = triggerEvent;

      router._triggerWillChangeContext = K;
      router._triggerWillLeave = K;

      var dslCallbacks = this.constructor.dslCallbacks || [K];
      var dsl = this._buildDSL();

      dsl.route('application', { path: '/', resetNamespace: true, overrideNameAssertion: true }, function () {
        for (var i = 0; i < dslCallbacks.length; i++) {
          dslCallbacks[i].call(this);
        }
      });

      if (_emberMetal.get(this, 'namespace.LOG_TRANSITIONS_INTERNAL')) {
        router.log = _emberConsole.default.debug;
      }

      router.map(dsl.generate());
    },

    _buildDSL: function () {
      var moduleBasedResolver = this._hasModuleBasedResolver();
      var options = {
        enableLoadingSubstates: !!moduleBasedResolver
      };

      var owner = _emberUtils.getOwner(this);
      var router = this;

      options.resolveRouteMap = function (name) {
        return owner._lookupFactory('route-map:' + name);
      };

      options.addRouteForEngine = function (name, engineInfo) {
        if (!router._engineInfoByRoute[name]) {
          router._engineInfoByRoute[name] = engineInfo;
        }
      };

      return new _emberRoutingSystemDsl.default(null, options);
    },

    init: function () {
      this._super.apply(this, arguments);

      this._qpCache = new _emberUtils.EmptyObject();
      this._resetQueuedQueryParameterChanges();
      this._handledErrors = _emberUtils.dictionary(null);
      this._engineInstances = new _emberUtils.EmptyObject();
      this._engineInfoByRoute = new _emberUtils.EmptyObject();
    },

    /*
      Resets all pending query paramter changes.
      Called after transitioning to a new route
      based on query parameter changes.
    */
    _resetQueuedQueryParameterChanges: function () {
      this._queuedQPChanges = {};
    },

    /**
      Represents the current URL.
       @method url
      @return {String} The current URL.
      @private
    */
    url: _emberMetal.computed(function () {
      return _emberMetal.get(this, 'location').getURL();
    }),

    _hasModuleBasedResolver: function () {
      var owner = _emberUtils.getOwner(this);

      if (!owner) {
        return false;
      }

      var resolver = owner.application && owner.application.__registry__ && owner.application.__registry__.resolver;

      if (!resolver) {
        return false;
      }

      return !!resolver.moduleBasedResolver;
    },

    /**
      Initializes the current router instance and sets up the change handling
      event listeners used by the instances `location` implementation.
       A property named `initialURL` will be used to determine the initial URL.
      If no value is found `/` will be used.
       @method startRouting
      @private
    */
    startRouting: function () {
      var initialURL = _emberMetal.get(this, 'initialURL');

      if (this.setupRouter()) {
        if (typeof initialURL === 'undefined') {
          initialURL = _emberMetal.get(this, 'location').getURL();
        }
        var initialTransition = this.handleURL(initialURL);
        if (initialTransition && initialTransition.error) {
          throw initialTransition.error;
        }
      }
    },

    setupRouter: function () {
      var _this = this;

      this._initRouterJs();
      this._setupLocation();

      var router = this.router;
      var location = _emberMetal.get(this, 'location');

      // Allow the Location class to cancel the router setup while it refreshes
      // the page
      if (_emberMetal.get(location, 'cancelRouterSetup')) {
        return false;
      }

      this._setupRouter(router, location);

      location.onUpdateURL(function (url) {
        _this.handleURL(url);
      });

      return true;
    },

    /**
      Handles updating the paths and notifying any listeners of the URL
      change.
       Triggers the router level `didTransition` hook.
       For example, to notify google analytics when the route changes,
      you could use this hook.  (Note: requires also including GA scripts, etc.)
       ```javascript
      let Router = Ember.Router.extend({
        location: config.locationType,
         didTransition: function() {
          this._super(...arguments);
           return ga('send', 'pageview', {
            'page': this.get('url'),
            'title': this.get('url')
          });
        }
      });
      ```
       @method didTransition
      @public
      @since 1.2.0
    */
    didTransition: function (infos) {
      updatePaths(this);

      this._cancelSlowTransitionTimer();

      this.notifyPropertyChange('url');
      this.set('currentState', this.targetState);

      // Put this in the runloop so url will be accurate. Seems
      // less surprising than didTransition being out of sync.
      _emberMetal.run.once(this, this.trigger, 'didTransition');

      if (_emberMetal.get(this, 'namespace').LOG_TRANSITIONS) {
        _emberConsole.default.log('Transitioned into \'' + EmberRouter._routePath(infos) + '\'');
      }
    },

    _setOutlets: function () {
      // This is triggered async during Ember.Route#willDestroy.
      // If the router is also being destroyed we do not want to
      // to create another this._toplevelView (and leak the renderer)
      if (this.isDestroying || this.isDestroyed) {
        return;
      }

      var handlerInfos = this.router.currentHandlerInfos;
      var route = undefined;
      var defaultParentState = undefined;
      var liveRoutes = null;

      if (!handlerInfos) {
        return;
      }

      for (var i = 0; i < handlerInfos.length; i++) {
        route = handlerInfos[i].handler;
        var connections = route.connections;
        var ownState = undefined;
        for (var j = 0; j < connections.length; j++) {
          var appended = appendLiveRoute(liveRoutes, defaultParentState, connections[j]);
          liveRoutes = appended.liveRoutes;
          if (appended.ownState.render.name === route.routeName || appended.ownState.render.outlet === 'main') {
            ownState = appended.ownState;
          }
        }
        if (connections.length === 0) {
          ownState = representEmptyRoute(liveRoutes, defaultParentState, route);
        }
        defaultParentState = ownState;
      }

      // when a transitionTo happens after the validation phase
      // during the initial transition _setOutlets is called
      // when no routes are active. However, it will get called
      // again with the correct values during the next turn of
      // the runloop
      if (!liveRoutes) {
        return;
      }

      if (!this._toplevelView) {
        var owner = _emberUtils.getOwner(this);
        var OutletView = owner._lookupFactory('view:-outlet');
        this._toplevelView = OutletView.create();
        this._toplevelView.setOutletState(liveRoutes);
        var instance = owner.lookup('-application-instance:main');
        instance.didCreateRootView(this._toplevelView);
      } else {
        this._toplevelView.setOutletState(liveRoutes);
      }
    },

    /**
      Handles notifying any listeners of an impending URL
      change.
       Triggers the router level `willTransition` hook.
       @method willTransition
      @public
      @since 1.11.0
    */
    willTransition: function (oldInfos, newInfos, transition) {
      _emberMetal.run.once(this, this.trigger, 'willTransition', transition);

      if (_emberMetal.get(this, 'namespace').LOG_TRANSITIONS) {
        _emberConsole.default.log('Preparing to transition from \'' + EmberRouter._routePath(oldInfos) + '\' to \'' + EmberRouter._routePath(newInfos) + '\'');
      }
    },

    handleURL: function (url) {
      // Until we have an ember-idiomatic way of accessing #hashes, we need to
      // remove it because router.js doesn't know how to handle it.
      url = url.split(/#(.+)?/)[0];
      return this._doURLTransition('handleURL', url);
    },

    _doURLTransition: function (routerJsMethod, url) {
      var transition = this.router[routerJsMethod](url || '/');
      didBeginTransition(transition, this);
      return transition;
    },

    /**
      Transition the application into another route. The route may
      be either a single route or route path:
       See [Route.transitionTo](http://emberjs.com/api/classes/Ember.Route.html#method_transitionTo) for more info.
       @method transitionTo
      @param {String} name the name of the route or a URL
      @param {...Object} models the model(s) or identifier(s) to be used while
        transitioning to the route.
      @param {Object} [options] optional hash with a queryParams property
        containing a mapping of query parameters
      @return {Transition} the transition object associated with this
        attempted transition
      @public
    */
    transitionTo: function () {
      var queryParams = undefined;

      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      if (resemblesURL(args[0])) {
        return this._doURLTransition('transitionTo', args[0]);
      }

      var possibleQueryParams = args[args.length - 1];
      if (possibleQueryParams && possibleQueryParams.hasOwnProperty('queryParams')) {
        queryParams = args.pop().queryParams;
      } else {
        queryParams = {};
      }

      var targetRouteName = args.shift();
      return this._doTransition(targetRouteName, args, queryParams);
    },

    intermediateTransitionTo: function () {
      var _router;

      (_router = this.router).intermediateTransitionTo.apply(_router, arguments);

      updatePaths(this);

      var infos = this.router.currentHandlerInfos;
      if (_emberMetal.get(this, 'namespace').LOG_TRANSITIONS) {
        _emberConsole.default.log('Intermediate-transitioned into \'' + EmberRouter._routePath(infos) + '\'');
      }
    },

    replaceWith: function () {
      return this.transitionTo.apply(this, arguments).method('replace');
    },

    generate: function () {
      var _router2;

      var url = (_router2 = this.router).generate.apply(_router2, arguments);
      return this.location.formatURL(url);
    },

    /**
      Determines if the supplied route is currently active.
       @method isActive
      @param routeName
      @return {Boolean}
      @private
    */
    isActive: function (routeName) {
      var router = this.router;
      return router.isActive.apply(router, arguments);
    },

    /**
      An alternative form of `isActive` that doesn't require
      manual concatenation of the arguments into a single
      array.
       @method isActiveIntent
      @param routeName
      @param models
      @param queryParams
      @return {Boolean}
      @private
      @since 1.7.0
    */
    isActiveIntent: function (routeName, models, queryParams) {
      return this.currentState.isActiveIntent(routeName, models, queryParams);
    },

    send: function (name, context) {
      var _router3;

      (_router3 = this.router).trigger.apply(_router3, arguments);
    },

    /**
      Does this router instance have the given route.
       @method hasRoute
      @return {Boolean}
      @private
    */
    hasRoute: function (route) {
      return this.router.hasRoute(route);
    },

    /**
      Resets the state of the router by clearing the current route
      handlers and deactivating them.
       @private
      @method reset
     */
    reset: function () {
      if (this.router) {
        this.router.reset();
      }
    },

    willDestroy: function () {
      if (this._toplevelView) {
        this._toplevelView.destroy();
        this._toplevelView = null;
      }

      this._super.apply(this, arguments);

      this.reset();

      var instances = this._engineInstances;
      for (var _name in instances) {
        for (var id in instances[_name]) {
          _emberMetal.run(instances[_name][id], 'destroy');
        }
      }
    },

    /*
      Called when an active route's query parameter has changed.
      These changes are batched into a runloop run and trigger
      a single transition.
    */
    _activeQPChanged: function (queryParameterName, newValue) {
      this._queuedQPChanges[queryParameterName] = newValue;
      _emberMetal.run.once(this, this._fireQueryParamTransition);
    },

    _updatingQPChanged: function (queryParameterName) {
      if (!this._qpUpdates) {
        this._qpUpdates = {};
      }
      this._qpUpdates[queryParameterName] = true;
    },

    /*
      Triggers a transition to a route based on query parameter changes.
      This is called once per runloop, to batch changes.
       e.g.
       if these methods are called in succession:
      this._activeQPChanged('foo', '10');
        // results in _queuedQPChanges = { foo: '10' }
      this._activeQPChanged('bar', false);
        // results in _queuedQPChanges = { foo: '10', bar: false }
        _queuedQPChanges will represent both of these changes
      and the transition using `transitionTo` will be triggered
      once.
    */
    _fireQueryParamTransition: function () {
      this.transitionTo({ queryParams: this._queuedQPChanges });
      this._resetQueuedQueryParameterChanges();
    },

    _setupLocation: function () {
      var location = _emberMetal.get(this, 'location');
      var rootURL = _emberMetal.get(this, 'rootURL');
      var owner = _emberUtils.getOwner(this);

      if ('string' === typeof location && owner) {
        var resolvedLocation = owner.lookup('location:' + location);

        if ('undefined' !== typeof resolvedLocation) {
          location = _emberMetal.set(this, 'location', resolvedLocation);
        } else {
          // Allow for deprecated registration of custom location API's
          var options = {
            implementation: location
          };

          location = _emberMetal.set(this, 'location', _emberRoutingLocationApi.default.create(options));
        }
      }

      if (location !== null && typeof location === 'object') {
        if (rootURL) {
          _emberMetal.set(location, 'rootURL', rootURL);
        }

        // Allow the location to do any feature detection, such as AutoLocation
        // detecting history support. This gives it a chance to set its
        // `cancelRouterSetup` property which aborts routing.
        if (typeof location.detect === 'function') {
          location.detect();
        }

        // ensure that initState is called AFTER the rootURL is set on
        // the location instance
        if (typeof location.initState === 'function') {
          location.initState();
        }
      }
    },

    _getHandlerFunction: function () {
      var _this2 = this;

      var seen = new _emberUtils.EmptyObject();
      var owner = _emberUtils.getOwner(this);

      return function (name) {
        var routeName = name;
        var routeOwner = owner;
        var engineInfo = _this2._engineInfoByRoute[routeName];

        if (engineInfo) {
          var engineInstance = _this2._getEngineInstance(engineInfo);

          routeOwner = engineInstance;
          routeName = engineInfo.localFullName;
        }

        var fullRouteName = 'route:' + routeName;

        var handler = routeOwner.lookup(fullRouteName);

        if (seen[name]) {
          return handler;
        }

        seen[name] = true;

        if (!handler) {
          var DefaultRoute = routeOwner._lookupFactory('route:basic');

          routeOwner.register(fullRouteName, DefaultRoute.extend());
          handler = routeOwner.lookup(fullRouteName);

          if (_emberMetal.get(_this2, 'namespace.LOG_ACTIVE_GENERATION')) {}
        }

        handler._setRouteName(routeName);
        handler._populateQPMeta();

        if (engineInfo && !_emberRoutingSystemRoute.hasDefaultSerialize(handler)) {
          throw new Error('Defining a custom serialize method on an Engine route is not supported.');
        }

        return handler;
      };
    },

    _getSerializerFunction: function () {
      var _this3 = this;

      return function (name) {
        var engineInfo = _this3._engineInfoByRoute[name];

        // If this is not an Engine route, we fall back to the handler for serialization
        if (!engineInfo) {
          return;
        }

        return engineInfo.serializeMethod || _emberRoutingSystemRoute.defaultSerialize;
      };
    },

    _setupRouter: function (router, location) {
      var lastURL = undefined;
      var emberRouter = this;

      router.getHandler = this._getHandlerFunction();
      router.getSerializer = this._getSerializerFunction();

      var doUpdateURL = function () {
        location.setURL(lastURL);
      };

      router.updateURL = function (path) {
        lastURL = path;
        _emberMetal.run.once(doUpdateURL);
      };

      if (location.replaceURL) {
        (function () {
          var doReplaceURL = function () {
            location.replaceURL(lastURL);
          };

          router.replaceURL = function (path) {
            lastURL = path;
            _emberMetal.run.once(doReplaceURL);
          };
        })();
      }

      router.didTransition = function (infos) {
        emberRouter.didTransition(infos);
      };

      router.willTransition = function (oldInfos, newInfos, transition) {
        emberRouter.willTransition(oldInfos, newInfos, transition);
      };
    },

    /**
      Serializes the given query params according to their QP meta information.
       @private
      @method _serializeQueryParams
      @param {Arrray<HandlerInfo>} handlerInfos
      @param {Object} queryParams
      @return {Void}
    */
    _serializeQueryParams: function (handlerInfos, queryParams) {
      var _this4 = this;

      forEachQueryParam(this, handlerInfos, queryParams, function (key, value, qp) {
        if (qp) {
          delete queryParams[key];
          queryParams[qp.urlKey] = qp.route.serializeQueryParam(value, qp.urlKey, qp.type);
        } else if (value === undefined) {
          return; // We don't serialize undefined values
        } else {
            queryParams[key] = _this4._serializeQueryParam(value, _emberRuntime.typeOf(value));
          }
      });
    },

    /**
      Serializes the value of a query parameter based on a type
       @private
      @method _serializeQueryParam
      @param {Object} value
      @param {String} type
    */
    _serializeQueryParam: function (value, type) {
      if (type === 'array') {
        return JSON.stringify(value);
      }

      return '' + value;
    },

    /**
      Deserializes the given query params according to their QP meta information.
       @private
      @method _deserializeQueryParams
      @param {Array<HandlerInfo>} handlerInfos
      @param {Object} queryParams
      @return {Void}
    */
    _deserializeQueryParams: function (handlerInfos, queryParams) {
      forEachQueryParam(this, handlerInfos, queryParams, function (key, value, qp) {
        // If we don't have QP meta info for a given key, then we do nothing
        // because all values will be treated as strings
        if (qp) {
          delete queryParams[key];
          queryParams[qp.prop] = qp.route.deserializeQueryParam(value, qp.urlKey, qp.type);
        }
      });
    },

    /**
      Deserializes the value of a query parameter based on a default type
       @private
      @method _deserializeQueryParam
      @param {Object} value
      @param {String} defaultType
    */
    _deserializeQueryParam: function (value, defaultType) {
      if (defaultType === 'boolean') {
        return value === 'true' ? true : false;
      } else if (defaultType === 'number') {
        return Number(value).valueOf();
      } else if (defaultType === 'array') {
        return _emberRuntime.A(JSON.parse(value));
      }

      return value;
    },

    /**
      Removes (prunes) any query params with default values from the given QP
      object. Default values are determined from the QP meta information per key.
       @private
      @method _pruneDefaultQueryParamValues
      @param {Array<HandlerInfo>} handlerInfos
      @param {Object} queryParams
      @return {Void}
    */
    _pruneDefaultQueryParamValues: function (handlerInfos, queryParams) {
      var qps = this._queryParamsFor(handlerInfos);
      for (var key in queryParams) {
        var qp = qps.map[key];
        if (qp && qp.serializedDefaultValue === queryParams[key]) {
          delete queryParams[key];
        }
      }
    },

    _doTransition: function (_targetRouteName, models, _queryParams) {
      var targetRouteName = _targetRouteName || _emberRoutingUtils.getActiveTargetName(this.router);

      var queryParams = {};

      this._processActiveTransitionQueryParams(targetRouteName, models, queryParams, _queryParams);

      _emberUtils.assign(queryParams, _queryParams);
      this._prepareQueryParams(targetRouteName, models, queryParams);

      var transitionArgs = _emberRoutingUtils.routeArgs(targetRouteName, models, queryParams);
      var transition = this.router.transitionTo.apply(this.router, transitionArgs);

      didBeginTransition(transition, this);

      return transition;
    },

    _processActiveTransitionQueryParams: function (targetRouteName, models, queryParams, _queryParams) {
      // merge in any queryParams from the active transition which could include
      // queryParams from the url on initial load.
      if (!this.router.activeTransition) {
        return;
      }

      var unchangedQPs = {};
      var qpUpdates = this._qpUpdates || {};
      for (var key in this.router.activeTransition.queryParams) {
        if (!qpUpdates[key]) {
          unchangedQPs[key] = this.router.activeTransition.queryParams[key];
        }
      }

      // We need to fully scope queryParams so that we can create one object
      // that represents both pased in queryParams and ones that aren't changed
      // from the active transition.
      this._fullyScopeQueryParams(targetRouteName, models, _queryParams);
      this._fullyScopeQueryParams(targetRouteName, models, unchangedQPs);
      _emberUtils.assign(queryParams, unchangedQPs);
    },

    /**
      Prepares the query params for a URL or Transition. Restores any undefined QP
      keys/values, serializes all values, and then prunes any default values.
       @private
      @method _prepareQueryParams
      @param {String} targetRouteName
      @param {Array<Object>} models
      @param {Object} queryParams
      @return {Void}
    */
    _prepareQueryParams: function (targetRouteName, models, queryParams) {
      var state = calculatePostTransitionState(this, targetRouteName, models);
      this._hydrateUnsuppliedQueryParams(state, queryParams);
      this._serializeQueryParams(state.handlerInfos, queryParams);
      this._pruneDefaultQueryParamValues(state.handlerInfos, queryParams);
    },

    /**
      Returns the meta information for the query params of a given route. This
      will be overriden to allow support for lazy routes.
       @private
      @method _getQPMeta
      @param {HandlerInfo} handlerInfo
      @return {Object}
    */
    _getQPMeta: function (handlerInfo) {
      var route = handlerInfo.handler;
      return route && _emberMetal.get(route, '_qp');
    },

    /**
      Returns a merged query params meta object for a given set of handlerInfos.
      Useful for knowing what query params are available for a given route hierarchy.
       @private
      @method _queryParamsFor
      @param {Array<HandlerInfo>} handlerInfos
      @return {Object}
     */
    _queryParamsFor: function (handlerInfos) {
      var leafRouteName = handlerInfos[handlerInfos.length - 1].name;
      if (this._qpCache[leafRouteName]) {
        return this._qpCache[leafRouteName];
      }

      var shouldCache = true;
      var qpsByUrlKey = {};
      var map = {};
      var qps = [];

      for (var i = 0; i < handlerInfos.length; ++i) {
        var qpMeta = this._getQPMeta(handlerInfos[i]);

        if (!qpMeta) {
          shouldCache = false;
          continue;
        }

        // Loop over each QP to make sure we don't have any collisions by urlKey
        for (var _i = 0; _i < qpMeta.qps.length; _i++) {
          var qp = qpMeta.qps[_i];
          var urlKey = qp.urlKey;
          var qpOther = qpsByUrlKey[urlKey];

          if (qpOther && qpOther.controllerName !== qp.controllerName) {
            var otherQP = qpsByUrlKey[urlKey];
          }

          qpsByUrlKey[urlKey] = qp;
          qps.push(qp);
        }

        _emberUtils.assign(map, qpMeta.map);
      }

      var finalQPMeta = {
        qps: qps,
        map: map
      };

      if (shouldCache) {
        this._qpCache[leafRouteName] = finalQPMeta;
      }

      return finalQPMeta;
    },

    /**
      Maps all query param keys to their fully scoped property name of the form
      `controllerName:propName`.
       @private
      @method _fullyScopeQueryParams
      @param {String} leafRouteName
      @param {Array<Object>} contexts
      @param {Object} queryParams
      @return {Void}
    */
    _fullyScopeQueryParams: function (leafRouteName, contexts, queryParams) {
      var state = calculatePostTransitionState(this, leafRouteName, contexts);
      var handlerInfos = state.handlerInfos;

      for (var i = 0, len = handlerInfos.length; i < len; ++i) {
        var qpMeta = this._getQPMeta(handlerInfos[i]);

        if (!qpMeta) {
          continue;
        }

        for (var j = 0, qpLen = qpMeta.qps.length; j < qpLen; ++j) {
          var qp = qpMeta.qps[j];

          var presentProp = qp.prop in queryParams && qp.prop || qp.scopedPropertyName in queryParams && qp.scopedPropertyName || qp.urlKey in queryParams && qp.urlKey;

          if (presentProp) {
            if (presentProp !== qp.scopedPropertyName) {
              queryParams[qp.scopedPropertyName] = queryParams[presentProp];
              delete queryParams[presentProp];
            }
          }
        }
      }
    },

    /**
      Hydrates (adds/restores) any query params that have pre-existing values into
      the given queryParams hash. This is what allows query params to be "sticky"
      and restore their last known values for their scope.
       @private
      @method _hydrateUnsuppliedQueryParams
      @param {TransitionState} state
      @param {Object} queryParams
      @return {Void}
    */
    _hydrateUnsuppliedQueryParams: function (state, queryParams) {
      var handlerInfos = state.handlerInfos;
      var appCache = this._bucketCache;

      for (var i = 0; i < handlerInfos.length; ++i) {
        var qpMeta = this._getQPMeta(handlerInfos[i]);

        if (!qpMeta) {
          continue;
        }

        for (var j = 0, qpLen = qpMeta.qps.length; j < qpLen; ++j) {
          var qp = qpMeta.qps[j];

          var presentProp = qp.prop in queryParams && qp.prop || qp.scopedPropertyName in queryParams && qp.scopedPropertyName || qp.urlKey in queryParams && qp.urlKey;

          if (presentProp) {
            if (presentProp !== qp.scopedPropertyName) {
              queryParams[qp.scopedPropertyName] = queryParams[presentProp];
              delete queryParams[presentProp];
            }
          } else {
            var cacheKey = _emberRoutingUtils.calculateCacheKey(qp.controllerName, qp.parts, state.params);
            queryParams[qp.scopedPropertyName] = appCache.lookup(cacheKey, qp.prop, qp.defaultValue);
          }
        }
      }
    },

    _scheduleLoadingEvent: function (transition, originRoute) {
      this._cancelSlowTransitionTimer();
      this._slowTransitionTimer = _emberMetal.run.scheduleOnce('routerTransitions', this, '_handleSlowTransition', transition, originRoute);
    },

    currentState: null,
    targetState: null,

    _handleSlowTransition: function (transition, originRoute) {
      if (!this.router.activeTransition) {
        // Don't fire an event if we've since moved on from
        // the transition that put us in a loading state.
        return;
      }

      this.set('targetState', _emberRoutingSystemRouter_state.default.create({
        emberRouter: this,
        routerJs: this.router,
        routerJsState: this.router.activeTransition.state
      }));

      transition.trigger(true, 'loading', transition, originRoute);
    },

    _cancelSlowTransitionTimer: function () {
      if (this._slowTransitionTimer) {
        _emberMetal.run.cancel(this._slowTransitionTimer);
      }
      this._slowTransitionTimer = null;
    },

    // These three helper functions are used to ensure errors aren't
    // re-raised if they're handled in a route's error action.
    _markErrorAsHandled: function (errorGuid) {
      this._handledErrors[errorGuid] = true;
    },

    _isErrorHandled: function (errorGuid) {
      return this._handledErrors[errorGuid];
    },

    _clearHandledError: function (errorGuid) {
      delete this._handledErrors[errorGuid];
    },

    _getEngineInstance: function (_ref) {
      var name = _ref.name;
      var instanceId = _ref.instanceId;
      var mountPoint = _ref.mountPoint;

      var engineInstances = this._engineInstances;

      if (!engineInstances[name]) {
        engineInstances[name] = new _emberUtils.EmptyObject();
      }

      var engineInstance = engineInstances[name][instanceId];

      if (!engineInstance) {
        var owner = _emberUtils.getOwner(this);

        engineInstance = owner.buildChildEngineInstance(name, {
          routable: true,
          mountPoint: mountPoint
        });

        engineInstance.boot();

        engineInstances[name][instanceId] = engineInstance;
      }

      return engineInstance;
    }
  });

  /*
    Helper function for iterating over routes in a set of handlerInfos that are
    at or above the given origin route. Example: if `originRoute` === 'foo.bar'
    and the handlerInfos given were for 'foo.bar.baz', then the given callback
    will be invoked with the routes for 'foo.bar', 'foo', and 'application'
    individually.
  
    If the callback returns anything other than `true`, then iteration will stop.
  
    @private
    @param {Route} originRoute
    @param {Array<HandlerInfo>} handlerInfos
    @param {Function} callback
    @return {Void}
   */
  function forEachRouteAbove(originRoute, handlerInfos, callback) {
    var originRouteFound = false;

    for (var i = handlerInfos.length - 1; i >= 0; --i) {
      var handlerInfo = handlerInfos[i];
      var route = handlerInfo.handler;

      if (originRoute === route) {
        originRouteFound = true;
      }

      if (!originRouteFound) {
        continue;
      }

      if (callback(route) !== true) {
        return;
      }
    }
  }

  // These get invoked when an action bubbles above ApplicationRoute
  // and are not meant to be overridable.
  var defaultActionHandlers = {

    willResolveModel: function (transition, originRoute) {
      originRoute.router._scheduleLoadingEvent(transition, originRoute);
    },

    // Attempt to find an appropriate error route or substate to enter.
    error: function (error, transition, originRoute) {
      var handlerInfos = transition.state.handlerInfos;
      var router = originRoute.router;

      forEachRouteAbove(originRoute, handlerInfos, function (route) {
        // Check for the existence of an 'error' route.
        // We don't check for an 'error' route on the originRoute, since that would
        // technically be below where we're at in the route hierarchy.
        if (originRoute !== route) {
          var errorRouteName = findRouteStateName(route, 'error');
          if (errorRouteName) {
            router.intermediateTransitionTo(errorRouteName, error);
            return false;
          }
        }

        // Check for an 'error' substate route
        var errorSubstateName = findRouteSubstateName(route, 'error');
        if (errorSubstateName) {
          router.intermediateTransitionTo(errorSubstateName, error);
          return false;
        }

        return true;
      });

      logError(error, 'Error while processing route: ' + transition.targetName);
    },

    // Attempt to find an appropriate loading route or substate to enter.
    loading: function (transition, originRoute) {
      var handlerInfos = transition.state.handlerInfos;
      var router = originRoute.router;

      forEachRouteAbove(originRoute, handlerInfos, function (route) {
        // Check for the existence of a 'loading' route.
        // We don't check for a 'loading' route on the originRoute, since that would
        // technically be below where we're at in the route hierarchy.
        if (originRoute !== route) {
          var loadingRouteName = findRouteStateName(route, 'loading');
          if (loadingRouteName) {
            router.intermediateTransitionTo(loadingRouteName);
            return false;
          }
        }

        // Check for loading substate
        var loadingSubstateName = findRouteSubstateName(route, 'loading');
        if (loadingSubstateName) {
          router.intermediateTransitionTo(loadingSubstateName);
          return false;
        }

        // Don't bubble above pivot route.
        return transition.pivotHandler !== route;
      });
    }
  };

  function logError(_error, initialMessage) {
    var errorArgs = [];
    var error = undefined;
    if (_error && typeof _error === 'object' && typeof _error.errorThrown === 'object') {
      error = _error.errorThrown;
    } else {
      error = _error;
    }

    if (initialMessage) {
      errorArgs.push(initialMessage);
    }

    if (error) {
      if (error.message) {
        errorArgs.push(error.message);
      }
      if (error.stack) {
        errorArgs.push(error.stack);
      }

      if (typeof error === 'string') {
        errorArgs.push(error);
      }
    }

    _emberConsole.default.error.apply(this, errorArgs);
  }

  /**
    Finds the name of the substate route if it exists for the given route. A
    substate route is of the form `route_state`, such as `foo_loading`.
  
    @private
    @param {Route} route
    @param {String} state
    @return {String}
  */
  function findRouteSubstateName(route, state) {
    var router = route.router;
    var owner = _emberUtils.getOwner(route);

    var routeName = route.routeName;
    var substateName = routeName + '_' + state;

    var routeNameFull = route.fullRouteName;
    var substateNameFull = routeNameFull + '_' + state;

    return routeHasBeenDefined(owner, router, substateName, substateNameFull) ? substateNameFull : '';
  }

  /**
    Finds the name of the state route if it exists for the given route. A state
    route is of the form `route.state`, such as `foo.loading`. Properly Handles
    `application` named routes.
  
    @private
    @param {Route} route
    @param {String} state
    @return {String}
  */
  function findRouteStateName(route, state) {
    var router = route.router;
    var owner = _emberUtils.getOwner(route);

    var routeName = route.routeName;
    var stateName = routeName === 'application' ? state : routeName + '.' + state;

    var routeNameFull = route.fullRouteName;
    var stateNameFull = routeNameFull === 'application' ? state : routeNameFull + '.' + state;

    return routeHasBeenDefined(owner, router, stateName, stateNameFull) ? stateNameFull : '';
  }

  /**
    Determines whether or not a route has been defined by checking that the route
    is in the Router's map and the owner has a registration for that route.
  
    @private
    @param {Owner} owner
    @param {Ember.Router} router
    @param {String} localName
    @param {String} fullName
    @return {Boolean}
  */
  function routeHasBeenDefined(owner, router, localName, fullName) {
    var routerHasRoute = router.hasRoute(fullName);
    var ownerHasRoute = owner.hasRegistration('template:' + localName) || owner.hasRegistration('route:' + localName);
    return routerHasRoute && ownerHasRoute;
  }

  function triggerEvent(handlerInfos, ignoreFailure, args) {
    var name = args.shift();

    if (!handlerInfos) {
      if (ignoreFailure) {
        return;
      }
      throw new _emberMetal.Error('Can\'t trigger action \'' + name + '\' because your app hasn\'t finished transitioning into its first route. To trigger an action on destination routes during a transition, you can call `.send()` on the `Transition` object passed to the `model/beforeModel/afterModel` hooks.');
    }

    var eventWasHandled = false;
    var handlerInfo = undefined,
        handler = undefined;

    for (var i = handlerInfos.length - 1; i >= 0; i--) {
      handlerInfo = handlerInfos[i];
      handler = handlerInfo.handler;

      if (handler && handler.actions && handler.actions[name]) {
        if (handler.actions[name].apply(handler, args) === true) {
          eventWasHandled = true;
        } else {
          // Should only hit here if a non-bubbling error action is triggered on a route.
          if (name === 'error') {
            var errorId = _emberUtils.guidFor(args[0]);
            handler.router._markErrorAsHandled(errorId);
          }
          return;
        }
      }
    }

    if (defaultActionHandlers[name]) {
      defaultActionHandlers[name].apply(null, args);
      return;
    }

    if (!eventWasHandled && !ignoreFailure) {
      throw new _emberMetal.Error('Nothing handled the action \'' + name + '\'. If you did handle the action, this error can be caused by returning true from an action handler in a controller, causing the action to bubble.');
    }
  }

  function calculatePostTransitionState(emberRouter, leafRouteName, contexts) {
    var routerjs = emberRouter.router;
    var state = routerjs.applyIntent(leafRouteName, contexts);
    var handlerInfos = state.handlerInfos;
    var params = state.params;

    for (var i = 0; i < handlerInfos.length; ++i) {
      var handlerInfo = handlerInfos[i];

      // If the handlerInfo is not resolved, we serialize the context into params
      if (!handlerInfo.isResolved) {
        params[handlerInfo.name] = handlerInfo.serialize(handlerInfo.context);
      } else {
        params[handlerInfo.name] = handlerInfo.params;
      }
    }
    return state;
  }

  function updatePaths(router) {
    var infos = router.router.currentHandlerInfos;
    if (infos.length === 0) {
      return;
    }

    var path = EmberRouter._routePath(infos);
    var currentRouteName = infos[infos.length - 1].name;

    _emberMetal.set(router, 'currentPath', path);
    _emberMetal.set(router, 'currentRouteName', currentRouteName);

    var appController = _emberUtils.getOwner(router).lookup('controller:application');

    if (!appController) {
      // appController might not exist when top-level loading/error
      // substates have been entered since ApplicationRoute hasn't
      // actually been entered at that point.
      return;
    }

    if (!('currentPath' in appController)) {
      _emberMetal.defineProperty(appController, 'currentPath');
    }

    _emberMetal.set(appController, 'currentPath', path);

    if (!('currentRouteName' in appController)) {
      _emberMetal.defineProperty(appController, 'currentRouteName');
    }

    _emberMetal.set(appController, 'currentRouteName', currentRouteName);
  }

  EmberRouter.reopenClass({
    router: null,

    /**
      The `Router.map` function allows you to define mappings from URLs to routes
      in your application. These mappings are defined within the
      supplied callback function using `this.route`.
       The first parameter is the name of the route which is used by default as the
      path name as well.
       The second parameter is the optional options hash. Available options are:
         * `path`: allows you to provide your own path as well as mark dynamic
          segments.
        * `resetNamespace`: false by default; when nesting routes, ember will
          combine the route names to form the fully-qualified route name, which is
          used with `{{link-to}}` or manually transitioning to routes. Setting
          `resetNamespace: true` will cause the route not to inherit from its
          parent route's names. This is handy for preventing extremely long route names.
          Keep in mind that the actual URL path behavior is still retained.
       The third parameter is a function, which can be used to nest routes.
      Nested routes, by default, will have the parent route tree's route name and
      path prepended to it's own.
       ```javascript
      App.Router.map(function(){
        this.route('post', { path: '/post/:post_id' }, function() {
          this.route('edit');
          this.route('comments', { resetNamespace: true }, function() {
            this.route('new');
          });
        });
      });
      ```
       For more detailed documentation and examples please see
      [the guides](http://emberjs.com/guides/routing/defining-your-routes/).
       @method map
      @param callback
      @public
    */
    map: function (callback) {
      if (!this.dslCallbacks) {
        this.dslCallbacks = [];
        this.reopenClass({ dslCallbacks: this.dslCallbacks });
      }

      this.dslCallbacks.push(callback);

      return this;
    },

    _routePath: function (handlerInfos) {
      var path = [];

      // We have to handle coalescing resource names that
      // are prefixed with their parent's names, e.g.
      // ['foo', 'foo.bar.baz'] => 'foo.bar.baz', not 'foo.foo.bar.baz'

      function intersectionMatches(a1, a2) {
        for (var i = 0; i < a1.length; ++i) {
          if (a1[i] !== a2[i]) {
            return false;
          }
        }
        return true;
      }

      var name = undefined,
          nameParts = undefined,
          oldNameParts = undefined;
      for (var i = 1; i < handlerInfos.length; i++) {
        name = handlerInfos[i].name;
        nameParts = name.split('.');
        oldNameParts = slice.call(path);

        while (oldNameParts.length) {
          if (intersectionMatches(oldNameParts, nameParts)) {
            break;
          }
          oldNameParts.shift();
        }

        path.push.apply(path, nameParts.slice(oldNameParts.length));
      }

      return path.join('.');
    }
  });

  function didBeginTransition(transition, router) {
    var routerState = _emberRoutingSystemRouter_state.default.create({
      emberRouter: router,
      routerJs: router.router,
      routerJsState: transition.state
    });

    if (!router.currentState) {
      router.set('currentState', routerState);
    }
    router.set('targetState', routerState);

    transition.promise = transition.catch(function (error) {
      var errorId = _emberUtils.guidFor(error);

      if (router._isErrorHandled(errorId)) {
        router._clearHandledError(errorId);
      } else {
        throw error;
      }
    });
  }

  function resemblesURL(str) {
    return typeof str === 'string' && (str === '' || str.charAt(0) === '/');
  }

  function forEachQueryParam(router, handlerInfos, queryParams, callback) {
    var qpCache = router._queryParamsFor(handlerInfos);

    for (var key in queryParams) {
      if (!queryParams.hasOwnProperty(key)) {
        continue;
      }
      var value = queryParams[key];
      var qp = qpCache.map[key];

      callback(key, value, qp);
    }
  }

  function findLiveRoute(liveRoutes, name) {
    if (!liveRoutes) {
      return;
    }
    var stack = [liveRoutes];
    while (stack.length > 0) {
      var test = stack.shift();
      if (test.render.name === name) {
        return test;
      }
      var outlets = test.outlets;
      for (var outletName in outlets) {
        stack.push(outlets[outletName]);
      }
    }
  }

  function appendLiveRoute(liveRoutes, defaultParentState, renderOptions) {
    var target = undefined;
    var myState = {
      render: renderOptions,
      outlets: new _emberUtils.EmptyObject(),
      wasUsed: false
    };
    if (renderOptions.into) {
      target = findLiveRoute(liveRoutes, renderOptions.into);
    } else {
      target = defaultParentState;
    }
    if (target) {
      _emberMetal.set(target.outlets, renderOptions.outlet, myState);
    } else {
      if (renderOptions.into) {

        // Megahax time. Post-3.0-breaking-changes, we will just assert
        // right here that the user tried to target a nonexistent
        // thing. But for now we still need to support the `render`
        // helper, and people are allowed to target templates rendered
        // by the render helper. So instead we defer doing anyting with
        // these orphan renders until afterRender.
        appendOrphan(liveRoutes, renderOptions.into, myState);
      } else {
        liveRoutes = myState;
      }
    }
    return {
      liveRoutes: liveRoutes,
      ownState: myState
    };
  }

  function appendOrphan(liveRoutes, into, myState) {
    if (!liveRoutes.outlets.__ember_orphans__) {
      liveRoutes.outlets.__ember_orphans__ = {
        render: {
          name: '__ember_orphans__'
        },
        outlets: new _emberUtils.EmptyObject()
      };
    }
    liveRoutes.outlets.__ember_orphans__.outlets[into] = myState;
    _emberMetal.run.schedule('afterRender', function () {});
  }

  function representEmptyRoute(liveRoutes, defaultParentState, route) {
    // the route didn't render anything
    var alreadyAppended = findLiveRoute(liveRoutes, route.routeName);
    if (alreadyAppended) {
      // But some other route has already rendered our default
      // template, so that becomes the default target for any
      // children we may have.
      return alreadyAppended;
    } else {
      // Create an entry to represent our default template name,
      // just so other routes can target it and inherit its place
      // in the outlet hierarchy.
      defaultParentState.outlets.main = {
        render: {
          name: route.routeName,
          outlet: 'main'
        },
        outlets: {}
      };
      return defaultParentState;
    }
  }

  exports.default = EmberRouter;
});

/**
@module ember
@submodule ember-routing
*/

// `wasUsed` gets set by the render helper.
enifed('ember-routing/system/router_state', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime'], function (exports, _emberUtils, _emberMetal, _emberRuntime) {
  'use strict';

  exports.default = _emberRuntime.Object.extend({
    emberRouter: null,
    routerJs: null,
    routerJsState: null,

    isActiveIntent: function (routeName, models, queryParams, queryParamsMustMatch) {
      var state = this.routerJsState;
      if (!this.routerJs.isActiveIntent(routeName, models, null, state)) {
        return false;
      }

      var emptyQueryParams = _emberMetal.isEmpty(Object.keys(queryParams));

      if (queryParamsMustMatch && !emptyQueryParams) {
        var visibleQueryParams = {};
        _emberUtils.assign(visibleQueryParams, queryParams);

        this.emberRouter._prepareQueryParams(routeName, models, visibleQueryParams);
        return shallowEqual(visibleQueryParams, state.queryParams);
      }

      return true;
    }
  });

  function shallowEqual(a, b) {
    var k = undefined;
    for (k in a) {
      if (a.hasOwnProperty(k) && a[k] !== b[k]) {
        return false;
      }
    }
    for (k in b) {
      if (b.hasOwnProperty(k) && a[k] !== b[k]) {
        return false;
      }
    }
    return true;
  }
});
enifed('ember-routing/utils', ['exports', 'ember-utils', 'ember-metal'], function (exports, _emberUtils, _emberMetal) {
  'use strict';

  exports.routeArgs = routeArgs;
  exports.getActiveTargetName = getActiveTargetName;
  exports.stashParamNames = stashParamNames;
  exports.calculateCacheKey = calculateCacheKey;
  exports.normalizeControllerQueryParams = normalizeControllerQueryParams;
  exports.prefixRouteNameArg = prefixRouteNameArg;

  var ALL_PERIODS_REGEX = /\./g;

  function routeArgs(targetRouteName, models, queryParams) {
    var args = [];
    if (typeof targetRouteName === 'string') {
      args.push('' + targetRouteName);
    }
    args.push.apply(args, models);
    args.push({ queryParams: queryParams });
    return args;
  }

  function getActiveTargetName(router) {
    var handlerInfos = router.activeTransition ? router.activeTransition.state.handlerInfos : router.state.handlerInfos;
    return handlerInfos[handlerInfos.length - 1].name;
  }

  function stashParamNames(router, handlerInfos) {
    if (handlerInfos._namesStashed) {
      return;
    }

    // This helper exists because router.js/route-recognizer.js awkwardly
    // keeps separate a handlerInfo's list of parameter names depending
    // on whether a URL transition or named transition is happening.
    // Hopefully we can remove this in the future.
    var targetRouteName = handlerInfos[handlerInfos.length - 1].name;
    var recogHandlers = router.router.recognizer.handlersFor(targetRouteName);
    var dynamicParent = null;

    for (var i = 0; i < handlerInfos.length; ++i) {
      var handlerInfo = handlerInfos[i];
      var names = recogHandlers[i].names;

      if (names.length) {
        dynamicParent = handlerInfo;
      }

      handlerInfo._names = names;

      var route = handlerInfo.handler;
      route._stashNames(handlerInfo, dynamicParent);
    }

    handlerInfos._namesStashed = true;
  }

  function _calculateCacheValuePrefix(prefix, part) {
    // calculates the dot seperated sections from prefix that are also
    // at the start of part - which gives us the route name

    // given : prefix = site.article.comments, part = site.article.id
    //      - returns: site.article (use get(values[site.article], 'id') to get the dynamic part - used below)

    // given : prefix = site.article, part = site.article.id
    //      - returns: site.article. (use get(values[site.article], 'id') to get the dynamic part - used below)

    var prefixParts = prefix.split('.');
    var currPrefix = '';

    for (var i = 0; i < prefixParts.length; i++) {
      var currPart = prefixParts.slice(0, i + 1).join('.');
      if (part.indexOf(currPart) !== 0) {
        break;
      }
      currPrefix = currPart;
    }

    return currPrefix;
  }

  /*
    Stolen from Controller
  */

  function calculateCacheKey(prefix, _parts, values) {
    var parts = _parts || [];
    var suffixes = '';
    for (var i = 0; i < parts.length; ++i) {
      var part = parts[i];
      var cacheValuePrefix = _calculateCacheValuePrefix(prefix, part);
      var value = undefined;
      if (values) {
        if (cacheValuePrefix && cacheValuePrefix in values) {
          var partRemovedPrefix = part.indexOf(cacheValuePrefix) === 0 ? part.substr(cacheValuePrefix.length + 1) : part;
          value = _emberMetal.get(values[cacheValuePrefix], partRemovedPrefix);
        } else {
          value = _emberMetal.get(values, part);
        }
      }
      suffixes += '::' + part + ':' + value;
    }
    return prefix + suffixes.replace(ALL_PERIODS_REGEX, '-');
  }

  /*
    Controller-defined query parameters can come in three shapes:
  
    Array
      queryParams: ['foo', 'bar']
    Array of simple objects where value is an alias
      queryParams: [
        {
          'foo': 'rename_foo_to_this'
        },
        {
          'bar': 'call_bar_this_instead'
        }
      ]
    Array of fully defined objects
      queryParams: [
        {
          'foo': {
            as: 'rename_foo_to_this'
          },
        }
        {
          'bar': {
            as: 'call_bar_this_instead',
            scope: 'controller'
          }
        }
      ]
  
    This helper normalizes all three possible styles into the
    'Array of fully defined objects' style.
  */

  function normalizeControllerQueryParams(queryParams) {
    var qpMap = {};

    for (var i = 0; i < queryParams.length; ++i) {
      accumulateQueryParamDescriptors(queryParams[i], qpMap);
    }

    return qpMap;
  }

  function accumulateQueryParamDescriptors(_desc, accum) {
    var desc = _desc;
    var tmp = undefined;
    if (typeof desc === 'string') {
      tmp = {};
      tmp[desc] = { as: null };
      desc = tmp;
    }

    for (var key in desc) {
      if (!desc.hasOwnProperty(key)) {
        return;
      }

      var singleDesc = desc[key];
      if (typeof singleDesc === 'string') {
        singleDesc = { as: singleDesc };
      }

      tmp = accum[key] || { as: null, scope: 'model' };
      _emberUtils.assign(tmp, singleDesc);

      accum[key] = tmp;
    }
  }

  /*
    Check if a routeName resembles a url instead
  
    @private
  */
  function resemblesURL(str) {
    return typeof str === 'string' && (str === '' || str.charAt(0) === '/');
  }

  /*
    Returns an arguments array where the route name arg is prefixed based on the mount point
  
    @private
  */

  function prefixRouteNameArg(route, args) {
    var routeName = args[0];
    var owner = _emberUtils.getOwner(route);
    var prefix = owner.mountPoint;

    // only alter the routeName if it's actually referencing a route.
    if (owner.routable && typeof routeName === 'string') {
      if (resemblesURL(routeName)) {
        throw new _emberMetal.Error('Programmatic transitions by URL cannot be used within an Engine. Please use the route name instead.');
      } else {
        routeName = prefix + '.' + routeName;
        args[0] = routeName;
      }
    }

    return args;
  }
});
enifed('ember-runtime/compare', ['exports', 'ember-runtime/utils', 'ember-runtime/mixins/comparable'], function (exports, _emberRuntimeUtils, _emberRuntimeMixinsComparable) {
  'use strict';

  exports.default = compare;

  var TYPE_ORDER = {
    'undefined': 0,
    'null': 1,
    'boolean': 2,
    'number': 3,
    'string': 4,
    'array': 5,
    'object': 6,
    'instance': 7,
    'function': 8,
    'class': 9,
    'date': 10
  };

  //
  // the spaceship operator
  //
  //                      `. ___
  //                     __,' __`.                _..----....____
  //         __...--.'``;.   ,.   ;``--..__     .'    ,-._    _.-'
  //   _..-''-------'   `'   `'   `'     O ``-''._   (,;') _,'
  // ,'________________                          \`-._`-','
  //  `._              ```````````------...___   '-.._'-:
  //     ```--.._      ,.                     ````--...__\-.
  //             `.--. `-` "INFINITY IS LESS     ____    |  |`
  //               `. `.   THAN BEYOND"        ,'`````.  ;  ;`
  //                 `._`.        __________   `.      \'__/`
  //                    `-:._____/______/___/____`.     \  `
  //                                |       `._    `.    \
  //                                `._________`-.   `.   `.___
  //                                              SSt  `------'`
  function spaceship(a, b) {
    var diff = a - b;
    return (diff > 0) - (diff < 0);
  }

  /**
   Compares two javascript values and returns:
  
    - -1 if the first is smaller than the second,
    - 0 if both are equal,
    - 1 if the first is greater than the second.
  
    ```javascript
    Ember.compare('hello', 'hello');  // 0
    Ember.compare('abc', 'dfg');      // -1
    Ember.compare(2, 1);              // 1
    ```
  
   If the types of the two objects are different precedence occurs in the
   following order, with types earlier in the list considered `<` types
   later in the list:
  
    - undefined
    - null
    - boolean
    - number
    - string
    - array
    - object
    - instance
    - function
    - class
    - date
  
    ```javascript
    Ember.compare('hello', 50);       // 1
    Ember.compare(50, 'hello');       // -1
    ```
  
   @method compare
   @for Ember
   @param {Object} v First value to compare
   @param {Object} w Second value to compare
   @return {Number} -1 if v < w, 0 if v = w and 1 if v > w.
   @public
  */

  function compare(v, w) {
    if (v === w) {
      return 0;
    }

    var type1 = _emberRuntimeUtils.typeOf(v);
    var type2 = _emberRuntimeUtils.typeOf(w);

    if (_emberRuntimeMixinsComparable.default) {
      if (type1 === 'instance' && _emberRuntimeMixinsComparable.default.detect(v) && v.constructor.compare) {
        return v.constructor.compare(v, w);
      }

      if (type2 === 'instance' && _emberRuntimeMixinsComparable.default.detect(w) && w.constructor.compare) {
        return w.constructor.compare(w, v) * -1;
      }
    }

    var res = spaceship(TYPE_ORDER[type1], TYPE_ORDER[type2]);

    if (res !== 0) {
      return res;
    }

    // types are equal - so we have to check values now
    switch (type1) {
      case 'boolean':
      case 'number':
        return spaceship(v, w);

      case 'string':
        return spaceship(v.localeCompare(w), 0);

      case 'array':
        var vLen = v.length;
        var wLen = w.length;
        var len = Math.min(vLen, wLen);

        for (var i = 0; i < len; i++) {
          var r = compare(v[i], w[i]);
          if (r !== 0) {
            return r;
          }
        }

        // all elements are equal now
        // shorter array should be ordered first
        return spaceship(vLen, wLen);

      case 'instance':
        if (_emberRuntimeMixinsComparable.default && _emberRuntimeMixinsComparable.default.detect(v)) {
          return v.compare(v, w);
        }
        return 0;

      case 'date':
        return spaceship(v.getTime(), w.getTime());

      default:
        return 0;
    }
  }
});
enifed('ember-runtime/computed/computed_macros', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  exports.empty = empty;
  exports.notEmpty = notEmpty;
  exports.none = none;
  exports.not = not;
  exports.bool = bool;
  exports.match = match;
  exports.equal = equal;
  exports.gt = gt;
  exports.gte = gte;
  exports.lt = lt;
  exports.lte = lte;
  exports.oneWay = oneWay;
  exports.readOnly = readOnly;
  exports.deprecatingAlias = deprecatingAlias;

  /**
  @module ember
  @submodule ember-metal
  */

  function expandPropertiesToArray(predicateName, properties) {
    var expandedProperties = [];

    function extractProperty(entry) {
      expandedProperties.push(entry);
    }

    for (var i = 0; i < properties.length; i++) {
      var property = properties[i];

      _emberMetal.expandProperties(property, extractProperty);
    }

    return expandedProperties;
  }

  function generateComputedWithPredicate(name, predicate) {
    return function () {
      for (var _len = arguments.length, properties = Array(_len), _key = 0; _key < _len; _key++) {
        properties[_key] = arguments[_key];
      }

      var expandedProperties = expandPropertiesToArray(name, properties);

      var computedFunc = _emberMetal.computed(function () {
        var lastIdx = expandedProperties.length - 1;

        for (var i = 0; i < lastIdx; i++) {
          var value = _emberMetal.get(this, expandedProperties[i]);
          if (!predicate(value)) {
            return value;
          }
        }

        return _emberMetal.get(this, expandedProperties[lastIdx]);
      });

      return computedFunc.property.apply(computedFunc, expandedProperties);
    };
  }

  /**
    A computed property that returns true if the value of the dependent
    property is null, an empty string, empty array, or empty function.
  
    Example
  
    ```javascript
    let ToDoList = Ember.Object.extend({
      isDone: Ember.computed.empty('todos')
    });
  
    let todoList = ToDoList.create({
      todos: ['Unit Test', 'Documentation', 'Release']
    });
  
    todoList.get('isDone'); // false
    todoList.get('todos').clear();
    todoList.get('isDone'); // true
    ```
  
    @since 1.6.0
    @method empty
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which negate
    the original value for property
    @public
  */

  function empty(dependentKey) {
    return _emberMetal.computed(dependentKey + '.length', function () {
      return _emberMetal.isEmpty(_emberMetal.get(this, dependentKey));
    });
  }

  /**
    A computed property that returns true if the value of the dependent
    property is NOT null, an empty string, empty array, or empty function.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      hasStuff: Ember.computed.notEmpty('backpack')
    });
  
    let hamster = Hamster.create({ backpack: ['Food', 'Sleeping Bag', 'Tent'] });
  
    hamster.get('hasStuff');         // true
    hamster.get('backpack').clear(); // []
    hamster.get('hasStuff');         // false
    ```
  
    @method notEmpty
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which returns true if
    original value for property is not empty.
    @public
  */

  function notEmpty(dependentKey) {
    return _emberMetal.computed(dependentKey + '.length', function () {
      return !_emberMetal.isEmpty(_emberMetal.get(this, dependentKey));
    });
  }

  /**
    A computed property that returns true if the value of the dependent
    property is null or undefined. This avoids errors from JSLint complaining
    about use of ==, which can be technically confusing.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      isHungry: Ember.computed.none('food')
    });
  
    let hamster = Hamster.create();
  
    hamster.get('isHungry'); // true
    hamster.set('food', 'Banana');
    hamster.get('isHungry'); // false
    hamster.set('food', null);
    hamster.get('isHungry'); // true
    ```
  
    @method none
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which
    returns true if original value for property is null or undefined.
    @public
  */

  function none(dependentKey) {
    return _emberMetal.computed(dependentKey, function () {
      return _emberMetal.isNone(_emberMetal.get(this, dependentKey));
    });
  }

  /**
    A computed property that returns the inverse boolean value
    of the original value for the dependent property.
  
    Example
  
    ```javascript
    let User = Ember.Object.extend({
      isAnonymous: Ember.computed.not('loggedIn')
    });
  
    let user = User.create({loggedIn: false});
  
    user.get('isAnonymous'); // true
    user.set('loggedIn', true);
    user.get('isAnonymous'); // false
    ```
  
    @method not
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which returns
    inverse of the original value for property
    @public
  */

  function not(dependentKey) {
    return _emberMetal.computed(dependentKey, function () {
      return !_emberMetal.get(this, dependentKey);
    });
  }

  /**
    A computed property that converts the provided dependent property
    into a boolean value.
  
    ```javascript
    let Hamster = Ember.Object.extend({
      hasBananas: Ember.computed.bool('numBananas')
    });
  
    let hamster = Hamster.create();
  
    hamster.get('hasBananas'); // false
    hamster.set('numBananas', 0);
    hamster.get('hasBananas'); // false
    hamster.set('numBananas', 1);
    hamster.get('hasBananas'); // true
    hamster.set('numBananas', null);
    hamster.get('hasBananas'); // false
    ```
  
    @method bool
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which converts
    to boolean the original value for property
    @public
  */

  function bool(dependentKey) {
    return _emberMetal.computed(dependentKey, function () {
      return !!_emberMetal.get(this, dependentKey);
    });
  }

  /**
    A computed property which matches the original value for the
    dependent property against a given RegExp, returning `true`
    if the value matches the RegExp and `false` if it does not.
  
    Example
  
    ```javascript
    let User = Ember.Object.extend({
      hasValidEmail: Ember.computed.match('email', /^.+@.+\..+$/)
    });
  
    let user = User.create({loggedIn: false});
  
    user.get('hasValidEmail'); // false
    user.set('email', '');
    user.get('hasValidEmail'); // false
    user.set('email', 'ember_hamster@example.com');
    user.get('hasValidEmail'); // true
    ```
  
    @method match
    @for Ember.computed
    @param {String} dependentKey
    @param {RegExp} regexp
    @return {Ember.ComputedProperty} computed property which match
    the original value for property against a given RegExp
    @public
  */

  function match(dependentKey, regexp) {
    return _emberMetal.computed(dependentKey, function () {
      var value = _emberMetal.get(this, dependentKey);

      return typeof value === 'string' ? regexp.test(value) : false;
    });
  }

  /**
    A computed property that returns true if the provided dependent property
    is equal to the given value.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      napTime: Ember.computed.equal('state', 'sleepy')
    });
  
    let hamster = Hamster.create();
  
    hamster.get('napTime'); // false
    hamster.set('state', 'sleepy');
    hamster.get('napTime'); // true
    hamster.set('state', 'hungry');
    hamster.get('napTime'); // false
    ```
  
    @method equal
    @for Ember.computed
    @param {String} dependentKey
    @param {String|Number|Object} value
    @return {Ember.ComputedProperty} computed property which returns true if
    the original value for property is equal to the given value.
    @public
  */

  function equal(dependentKey, value) {
    return _emberMetal.computed(dependentKey, function () {
      return _emberMetal.get(this, dependentKey) === value;
    });
  }

  /**
    A computed property that returns true if the provided dependent property
    is greater than the provided value.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      hasTooManyBananas: Ember.computed.gt('numBananas', 10)
    });
  
    let hamster = Hamster.create();
  
    hamster.get('hasTooManyBananas'); // false
    hamster.set('numBananas', 3);
    hamster.get('hasTooManyBananas'); // false
    hamster.set('numBananas', 11);
    hamster.get('hasTooManyBananas'); // true
    ```
  
    @method gt
    @for Ember.computed
    @param {String} dependentKey
    @param {Number} value
    @return {Ember.ComputedProperty} computed property which returns true if
    the original value for property is greater than given value.
    @public
  */

  function gt(dependentKey, value) {
    return _emberMetal.computed(dependentKey, function () {
      return _emberMetal.get(this, dependentKey) > value;
    });
  }

  /**
    A computed property that returns true if the provided dependent property
    is greater than or equal to the provided value.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      hasTooManyBananas: Ember.computed.gte('numBananas', 10)
    });
  
    let hamster = Hamster.create();
  
    hamster.get('hasTooManyBananas'); // false
    hamster.set('numBananas', 3);
    hamster.get('hasTooManyBananas'); // false
    hamster.set('numBananas', 10);
    hamster.get('hasTooManyBananas'); // true
    ```
  
    @method gte
    @for Ember.computed
    @param {String} dependentKey
    @param {Number} value
    @return {Ember.ComputedProperty} computed property which returns true if
    the original value for property is greater or equal then given value.
    @public
  */

  function gte(dependentKey, value) {
    return _emberMetal.computed(dependentKey, function () {
      return _emberMetal.get(this, dependentKey) >= value;
    });
  }

  /**
    A computed property that returns true if the provided dependent property
    is less than the provided value.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      needsMoreBananas: Ember.computed.lt('numBananas', 3)
    });
  
    let hamster = Hamster.create();
  
    hamster.get('needsMoreBananas'); // true
    hamster.set('numBananas', 3);
    hamster.get('needsMoreBananas'); // false
    hamster.set('numBananas', 2);
    hamster.get('needsMoreBananas'); // true
    ```
  
    @method lt
    @for Ember.computed
    @param {String} dependentKey
    @param {Number} value
    @return {Ember.ComputedProperty} computed property which returns true if
    the original value for property is less then given value.
    @public
  */

  function lt(dependentKey, value) {
    return _emberMetal.computed(dependentKey, function () {
      return _emberMetal.get(this, dependentKey) < value;
    });
  }

  /**
    A computed property that returns true if the provided dependent property
    is less than or equal to the provided value.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      needsMoreBananas: Ember.computed.lte('numBananas', 3)
    });
  
    let hamster = Hamster.create();
  
    hamster.get('needsMoreBananas'); // true
    hamster.set('numBananas', 5);
    hamster.get('needsMoreBananas'); // false
    hamster.set('numBananas', 3);
    hamster.get('needsMoreBananas'); // true
    ```
  
    @method lte
    @for Ember.computed
    @param {String} dependentKey
    @param {Number} value
    @return {Ember.ComputedProperty} computed property which returns true if
    the original value for property is less or equal than given value.
    @public
  */

  function lte(dependentKey, value) {
    return _emberMetal.computed(dependentKey, function () {
      return _emberMetal.get(this, dependentKey) <= value;
    });
  }

  /**
    A computed property that performs a logical `and` on the
    original values for the provided dependent properties.
  
    You may pass in more than two properties and even use
    property brace expansion.  The computed property will
    return the first falsy value or last truthy value
    just like JavaScript's `&&` operator.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      readyForCamp: Ember.computed.and('hasTent', 'hasBackpack'),
      readyForHike: Ember.computed.and('hasWalkingStick', 'hasBackpack')
    });
  
    let tomster = Hamster.create();
  
    tomster.get('readyForCamp'); // false
    tomster.set('hasTent', true);
    tomster.get('readyForCamp'); // false
    tomster.set('hasBackpack', true);
    tomster.get('readyForCamp'); // true
    tomster.set('hasBackpack', 'Yes');
    tomster.get('readyForCamp'); // 'Yes'
    tomster.set('hasWalkingStick', null);
    tomster.get('readyForHike'); // null
    ```
  
    @method and
    @for Ember.computed
    @param {String} dependentKey*
    @return {Ember.ComputedProperty} computed property which performs
    a logical `and` on the values of all the original values for properties.
    @public
  */
  var and = generateComputedWithPredicate('and', function (value) {
    return value;
  });

  exports.and = and;
  /**
    A computed property which performs a logical `or` on the
    original values for the provided dependent properties.
  
    You may pass in more than two properties and even use
    property brace expansion.  The computed property will
    return the first truthy value or last falsy value just
    like JavaScript's `||` operator.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      readyForRain: Ember.computed.or('hasJacket', 'hasUmbrella'),
      readyForBeach: Ember.computed.or('{hasSunscreen,hasUmbrella}')
    });
  
    let tomster = Hamster.create();
  
    tomster.get('readyForRain'); // undefined
    tomster.set('hasUmbrella', true);
    tomster.get('readyForRain'); // true
    tomster.set('hasJacket', 'Yes');
    tomster.get('readyForRain'); // 'Yes'
    tomster.set('hasSunscreen', 'Check');
    tomster.get('readyForBeach'); // 'Check'
    ```
  
    @method or
    @for Ember.computed
    @param {String} dependentKey*
    @return {Ember.ComputedProperty} computed property which performs
    a logical `or` on the values of all the original values for properties.
    @public
  */
  var or = generateComputedWithPredicate('or', function (value) {
    return !value;
  });

  exports.or = or;
  /**
    Creates a new property that is an alias for another property
    on an object. Calls to `get` or `set` this property behave as
    though they were called on the original property.
  
    ```javascript
    let Person = Ember.Object.extend({
      name: 'Alex Matchneer',
      nomen: Ember.computed.alias('name')
    });
  
    let alex = Person.create();
  
    alex.get('nomen'); // 'Alex Matchneer'
    alex.get('name');  // 'Alex Matchneer'
  
    alex.set('nomen', '@machty');
    alex.get('name');  // '@machty'
    ```
  
    @method alias
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which creates an
    alias to the original value for property.
    @public
  */

  /**
    Where `computed.alias` aliases `get` and `set`, and allows for bidirectional
    data flow, `computed.oneWay` only provides an aliased `get`. The `set` will
    not mutate the upstream property, rather causes the current property to
    become the value set. This causes the downstream property to permanently
    diverge from the upstream property.
  
    Example
  
    ```javascript
    let User = Ember.Object.extend({
      firstName: null,
      lastName: null,
      nickName: Ember.computed.oneWay('firstName')
    });
  
    let teddy = User.create({
      firstName: 'Teddy',
      lastName:  'Zeenny'
    });
  
    teddy.get('nickName');              // 'Teddy'
    teddy.set('nickName', 'TeddyBear'); // 'TeddyBear'
    teddy.get('firstName');             // 'Teddy'
    ```
  
    @method oneWay
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which creates a
    one way computed property to the original value for property.
    @public
  */

  function oneWay(dependentKey) {
    return _emberMetal.alias(dependentKey).oneWay();
  }

  /**
    This is a more semantically meaningful alias of `computed.oneWay`,
    whose name is somewhat ambiguous as to which direction the data flows.
  
    @method reads
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which creates a
      one way computed property to the original value for property.
    @public
   */

  /**
    Where `computed.oneWay` provides oneWay bindings, `computed.readOnly` provides
    a readOnly one way binding. Very often when using `computed.oneWay` one does
    not also want changes to propagate back up, as they will replace the value.
  
    This prevents the reverse flow, and also throws an exception when it occurs.
  
    Example
  
    ```javascript
    let User = Ember.Object.extend({
      firstName: null,
      lastName: null,
      nickName: Ember.computed.readOnly('firstName')
    });
  
    let teddy = User.create({
      firstName: 'Teddy',
      lastName:  'Zeenny'
    });
  
    teddy.get('nickName');              // 'Teddy'
    teddy.set('nickName', 'TeddyBear'); // throws Exception
    // throw new Ember.Error('Cannot Set: nickName on: <User:ember27288>' );`
    teddy.get('firstName');             // 'Teddy'
    ```
  
    @method readOnly
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computed property which creates a
    one way computed property to the original value for property.
    @since 1.5.0
    @public
  */

  function readOnly(dependentKey) {
    return _emberMetal.alias(dependentKey).readOnly();
  }

  /**
    Creates a new property that is an alias for another property
    on an object. Calls to `get` or `set` this property behave as
    though they were called on the original property, but also
    print a deprecation warning.
  
    ```javascript
    let Hamster = Ember.Object.extend({
      bananaCount: Ember.computed.deprecatingAlias('cavendishCount', {
        id: 'hamster.deprecate-banana',
        until: '3.0.0'
      })
    });
  
    let hamster = Hamster.create();
  
    hamster.set('bananaCount', 5); // Prints a deprecation warning.
    hamster.get('cavendishCount'); // 5
    ```
  
    @method deprecatingAlias
    @for Ember.computed
    @param {String} dependentKey
    @param {Object} options Options for `Ember.deprecate`.
    @return {Ember.ComputedProperty} computed property which creates an
    alias with a deprecation to the original value for property.
    @since 1.7.0
    @public
  */

  function deprecatingAlias(dependentKey, options) {
    return _emberMetal.computed(dependentKey, {
      get: function (key) {
        return _emberMetal.get(this, dependentKey);
      },
      set: function (key, value) {
        _emberMetal.set(this, dependentKey, value);
        return value;
      }
    });
  }
});
enifed('ember-runtime/computed/reduce_computed_macros', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime/compare', 'ember-runtime/utils', 'ember-runtime/system/native_array'], function (exports, _emberUtils, _emberMetal, _emberRuntimeCompare, _emberRuntimeUtils, _emberRuntimeSystemNative_array) {
  /**
  @module ember
  @submodule ember-runtime
  */

  'use strict';

  exports.sum = sum;
  exports.max = max;
  exports.min = min;
  exports.map = map;
  exports.mapBy = mapBy;
  exports.filter = filter;
  exports.filterBy = filterBy;
  exports.uniq = uniq;
  exports.uniqBy = uniqBy;
  exports.intersect = intersect;
  exports.setDiff = setDiff;
  exports.collect = collect;
  exports.sort = sort;

  function reduceMacro(dependentKey, callback, initialValue) {
    return _emberMetal.computed(dependentKey + '.[]', function () {
      var _this = this;

      var arr = _emberMetal.get(this, dependentKey);

      if (arr === null || typeof arr !== 'object') {
        return initialValue;
      }

      return arr.reduce(function (previousValue, currentValue, index, array) {
        return callback.call(_this, previousValue, currentValue, index, array);
      }, initialValue);
    }).readOnly();
  }

  function arrayMacro(dependentKey, callback) {
    // This is a bit ugly
    var propertyName = undefined;
    if (/@each/.test(dependentKey)) {
      propertyName = dependentKey.replace(/\.@each.*$/, '');
    } else {
      propertyName = dependentKey;
      dependentKey += '.[]';
    }

    return _emberMetal.computed(dependentKey, function () {
      var value = _emberMetal.get(this, propertyName);
      if (_emberRuntimeUtils.isArray(value)) {
        return _emberRuntimeSystemNative_array.A(callback.call(this, value));
      } else {
        return _emberRuntimeSystemNative_array.A();
      }
    }).readOnly();
  }

  function multiArrayMacro(dependentKeys, callback) {
    var args = dependentKeys.map(function (key) {
      return key + '.[]';
    });

    args.push(function () {
      return _emberRuntimeSystemNative_array.A(callback.call(this, dependentKeys));
    });

    return _emberMetal.computed.apply(this, args).readOnly();
  }

  /**
    A computed property that returns the sum of the values
    in the dependent array.
  
    @method sum
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computes the sum of all values in the dependentKey's array
    @since 1.4.0
    @public
  */

  function sum(dependentKey) {
    return reduceMacro(dependentKey, function (sum, item) {
      return sum + item;
    }, 0);
  }

  /**
    A computed property that calculates the maximum value in the
    dependent array. This will return `-Infinity` when the dependent
    array is empty.
  
    ```javascript
    let Person = Ember.Object.extend({
      childAges: Ember.computed.mapBy('children', 'age'),
      maxChildAge: Ember.computed.max('childAges')
    });
  
    let lordByron = Person.create({ children: [] });
  
    lordByron.get('maxChildAge'); // -Infinity
    lordByron.get('children').pushObject({
      name: 'Augusta Ada Byron', age: 7
    });
    lordByron.get('maxChildAge'); // 7
    lordByron.get('children').pushObjects([{
      name: 'Allegra Byron',
      age: 5
    }, {
      name: 'Elizabeth Medora Leigh',
      age: 8
    }]);
    lordByron.get('maxChildAge'); // 8
    ```
  
    If the types of the arguments are not numbers,
    they will be converted to numbers and the type
    of the return value will always be `Number`.
    For example, the max of a list of Date objects will be
    the highest timestamp as a `Number`.
    This behavior is consistent with `Math.max`.
  
    @method max
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computes the largest value in the dependentKey's array
    @public
  */

  function max(dependentKey) {
    return reduceMacro(dependentKey, function (max, item) {
      return Math.max(max, item);
    }, -Infinity);
  }

  /**
    A computed property that calculates the minimum value in the
    dependent array. This will return `Infinity` when the dependent
    array is empty.
  
    ```javascript
    let Person = Ember.Object.extend({
      childAges: Ember.computed.mapBy('children', 'age'),
      minChildAge: Ember.computed.min('childAges')
    });
  
    let lordByron = Person.create({ children: [] });
  
    lordByron.get('minChildAge'); // Infinity
    lordByron.get('children').pushObject({
      name: 'Augusta Ada Byron', age: 7
    });
    lordByron.get('minChildAge'); // 7
    lordByron.get('children').pushObjects([{
      name: 'Allegra Byron',
      age: 5
    }, {
      name: 'Elizabeth Medora Leigh',
      age: 8
    }]);
    lordByron.get('minChildAge'); // 5
    ```
  
    If the types of the arguments are not numbers,
    they will be converted to numbers and the type
    of the return value will always be `Number`.
    For example, the min of a list of Date objects will be
    the lowest timestamp as a `Number`.
    This behavior is consistent with `Math.min`.
  
    @method min
    @for Ember.computed
    @param {String} dependentKey
    @return {Ember.ComputedProperty} computes the smallest value in the dependentKey's array
    @public
  */

  function min(dependentKey) {
    return reduceMacro(dependentKey, function (min, item) {
      return Math.min(min, item);
    }, Infinity);
  }

  /**
    Returns an array mapped via the callback
  
    The callback method you provide should have the following signature.
    `item` is the current item in the iteration.
    `index` is the integer index of the current item in the iteration.
  
    ```javascript
    function(item, index);
    ```
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      excitingChores: Ember.computed.map('chores', function(chore, index) {
        return chore.toUpperCase() + '!';
      })
    });
  
    let hamster = Hamster.create({
      chores: ['clean', 'write more unit tests']
    });
  
    hamster.get('excitingChores'); // ['CLEAN!', 'WRITE MORE UNIT TESTS!']
    ```
  
    @method map
    @for Ember.computed
    @param {String} dependentKey
    @param {Function} callback
    @return {Ember.ComputedProperty} an array mapped via the callback
    @public
  */

  function map(dependentKey, callback) {
    return arrayMacro(dependentKey, function (value) {
      return value.map(callback, this);
    });
  }

  /**
    Returns an array mapped to the specified key.
  
    ```javascript
    let Person = Ember.Object.extend({
      childAges: Ember.computed.mapBy('children', 'age')
    });
  
    let lordByron = Person.create({ children: [] });
  
    lordByron.get('childAges'); // []
    lordByron.get('children').pushObject({ name: 'Augusta Ada Byron', age: 7 });
    lordByron.get('childAges'); // [7]
    lordByron.get('children').pushObjects([{
      name: 'Allegra Byron',
      age: 5
    }, {
      name: 'Elizabeth Medora Leigh',
      age: 8
    }]);
    lordByron.get('childAges'); // [7, 5, 8]
    ```
  
    @method mapBy
    @for Ember.computed
    @param {String} dependentKey
    @param {String} propertyKey
    @return {Ember.ComputedProperty} an array mapped to the specified key
    @public
  */

  function mapBy(dependentKey, propertyKey) {

    return map(dependentKey + '.@each.' + propertyKey, function (item) {
      return _emberMetal.get(item, propertyKey);
    });
  }

  /**
    Filters the array by the callback.
  
    The callback method you provide should have the following signature.
    `item` is the current item in the iteration.
    `index` is the integer index of the current item in the iteration.
    `array` is the dependant array itself.
  
    ```javascript
    function(item, index, array);
    ```
  
    ```javascript
    let Hamster = Ember.Object.extend({
      remainingChores: Ember.computed.filter('chores', function(chore, index, array) {
        return !chore.done;
      })
    });
  
    let hamster = Hamster.create({
      chores: [
        { name: 'cook', done: true },
        { name: 'clean', done: true },
        { name: 'write more unit tests', done: false }
      ]
    });
  
    hamster.get('remainingChores'); // [{name: 'write more unit tests', done: false}]
    ```
  
    @method filter
    @for Ember.computed
    @param {String} dependentKey
    @param {Function} callback
    @return {Ember.ComputedProperty} the filtered array
    @public
  */

  function filter(dependentKey, callback) {
    return arrayMacro(dependentKey, function (value) {
      return value.filter(callback, this);
    });
  }

  /**
    Filters the array by the property and value
  
    ```javascript
    let Hamster = Ember.Object.extend({
      remainingChores: Ember.computed.filterBy('chores', 'done', false)
    });
  
    let hamster = Hamster.create({
      chores: [
        { name: 'cook', done: true },
        { name: 'clean', done: true },
        { name: 'write more unit tests', done: false }
      ]
    });
  
    hamster.get('remainingChores'); // [{ name: 'write more unit tests', done: false }]
    ```
  
    @method filterBy
    @for Ember.computed
    @param {String} dependentKey
    @param {String} propertyKey
    @param {*} value
    @return {Ember.ComputedProperty} the filtered array
    @public
  */

  function filterBy(dependentKey, propertyKey, value) {
    var callback = undefined;

    if (arguments.length === 2) {
      callback = function (item) {
        return _emberMetal.get(item, propertyKey);
      };
    } else {
      callback = function (item) {
        return _emberMetal.get(item, propertyKey) === value;
      };
    }

    return filter(dependentKey + '.@each.' + propertyKey, callback);
  }

  /**
    A computed property which returns a new array with all the unique
    elements from one or more dependent arrays.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      uniqueFruits: Ember.computed.uniq('fruits')
    });
  
    let hamster = Hamster.create({
      fruits: [
        'banana',
        'grape',
        'kale',
        'banana'
      ]
    });
  
    hamster.get('uniqueFruits'); // ['banana', 'grape', 'kale']
    ```
  
    @method uniq
    @for Ember.computed
    @param {String} propertyKey*
    @return {Ember.ComputedProperty} computes a new array with all the
    unique elements from the dependent array
    @public
  */

  function uniq() {
    for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    return multiArrayMacro(args, function (dependentKeys) {
      var _this2 = this;

      var uniq = _emberRuntimeSystemNative_array.A();

      dependentKeys.forEach(function (dependentKey) {
        var value = _emberMetal.get(_this2, dependentKey);
        if (_emberRuntimeUtils.isArray(value)) {
          value.forEach(function (item) {
            if (uniq.indexOf(item) === -1) {
              uniq.push(item);
            }
          });
        }
      });

      return uniq;
    });
  }

  /**
    A computed property which returns a new array with all the unique
    elements from an array, with uniqueness determined by specific key.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      uniqueFruits: Ember.computed.uniqBy('fruits', 'id')
    });
    let hamster = Hamster.create({
      fruits: [
        { id: 1, 'banana' },
        { id: 2, 'grape' },
        { id: 3, 'peach' },
        { id: 1, 'banana' }
      ]
    });
    hamster.get('uniqueFruits'); // [ { id: 1, 'banana' }, { id: 2, 'grape' }, { id: 3, 'peach' }]
    ```
  
    @method uniqBy
    @for Ember.computed
    @param {String} dependentKey
    @param {String} propertyKey
    @return {Ember.ComputedProperty} computes a new array with all the
    unique elements from the dependent array
    @public
  */

  function uniqBy(dependentKey, propertyKey) {
    return _emberMetal.computed(dependentKey + '.[]', function () {
      var uniq = _emberRuntimeSystemNative_array.A();
      var seen = new _emberUtils.EmptyObject();
      var list = _emberMetal.get(this, dependentKey);
      if (_emberRuntimeUtils.isArray(list)) {
        list.forEach(function (item) {
          var guid = _emberUtils.guidFor(_emberMetal.get(item, propertyKey));
          if (!(guid in seen)) {
            seen[guid] = true;
            uniq.push(item);
          }
        });
      }
      return uniq;
    }).readOnly();
  }

  /**
    Alias for [Ember.computed.uniq](/api/#method_computed_uniq).
  
    @method union
    @for Ember.computed
    @param {String} propertyKey*
    @return {Ember.ComputedProperty} computes a new array with all the
    unique elements from the dependent array
    @public
  */
  var union = uniq;

  exports.union = union;
  /**
    A computed property which returns a new array with all the duplicated
    elements from two or more dependent arrays.
  
    Example
  
    ```javascript
    let obj = Ember.Object.extend({
      friendsInCommon: Ember.computed.intersect('adaFriends', 'charlesFriends')
    }).create({
      adaFriends: ['Charles Babbage', 'John Hobhouse', 'William King', 'Mary Somerville'],
      charlesFriends: ['William King', 'Mary Somerville', 'Ada Lovelace', 'George Peacock']
    });
  
    obj.get('friendsInCommon'); // ['William King', 'Mary Somerville']
    ```
  
    @method intersect
    @for Ember.computed
    @param {String} propertyKey*
    @return {Ember.ComputedProperty} computes a new array with all the
    duplicated elements from the dependent arrays
    @public
  */

  function intersect() {
    for (var _len2 = arguments.length, args = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
      args[_key2] = arguments[_key2];
    }

    return multiArrayMacro(args, function (dependentKeys) {
      var _this3 = this;

      var arrays = dependentKeys.map(function (dependentKey) {
        var array = _emberMetal.get(_this3, dependentKey);

        return _emberRuntimeUtils.isArray(array) ? array : [];
      });

      var results = arrays.pop().filter(function (candidate) {
        for (var i = 0; i < arrays.length; i++) {
          var found = false;
          var array = arrays[i];
          for (var j = 0; j < array.length; j++) {
            if (array[j] === candidate) {
              found = true;
              break;
            }
          }

          if (found === false) {
            return false;
          }
        }

        return true;
      });

      return _emberRuntimeSystemNative_array.A(results);
    });
  }

  /**
    A computed property which returns a new array with all the
    properties from the first dependent array that are not in the second
    dependent array.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      likes: ['banana', 'grape', 'kale'],
      wants: Ember.computed.setDiff('likes', 'fruits')
    });
  
    let hamster = Hamster.create({
      fruits: [
        'grape',
        'kale',
      ]
    });
  
    hamster.get('wants'); // ['banana']
    ```
  
    @method setDiff
    @for Ember.computed
    @param {String} setAProperty
    @param {String} setBProperty
    @return {Ember.ComputedProperty} computes a new array with all the
    items from the first dependent array that are not in the second
    dependent array
    @public
  */

  function setDiff(setAProperty, setBProperty) {
    if (arguments.length !== 2) {
      throw new _emberMetal.Error('setDiff requires exactly two dependent arrays.');
    }

    return _emberMetal.computed(setAProperty + '.[]', setBProperty + '.[]', function () {
      var setA = this.get(setAProperty);
      var setB = this.get(setBProperty);

      if (!_emberRuntimeUtils.isArray(setA)) {
        return _emberRuntimeSystemNative_array.A();
      }
      if (!_emberRuntimeUtils.isArray(setB)) {
        return _emberRuntimeSystemNative_array.A(setA);
      }

      return setA.filter(function (x) {
        return setB.indexOf(x) === -1;
      });
    }).readOnly();
  }

  /**
    A computed property that returns the array of values
    for the provided dependent properties.
  
    Example
  
    ```javascript
    let Hamster = Ember.Object.extend({
      clothes: Ember.computed.collect('hat', 'shirt')
    });
  
    let hamster = Hamster.create();
  
    hamster.get('clothes'); // [null, null]
    hamster.set('hat', 'Camp Hat');
    hamster.set('shirt', 'Camp Shirt');
    hamster.get('clothes'); // ['Camp Hat', 'Camp Shirt']
    ```
  
    @method collect
    @for Ember.computed
    @param {String} dependentKey*
    @return {Ember.ComputedProperty} computed property which maps
    values of all passed in properties to an array.
    @public
  */

  function collect() {
    for (var _len3 = arguments.length, dependentKeys = Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
      dependentKeys[_key3] = arguments[_key3];
    }

    return multiArrayMacro(dependentKeys, function () {
      var properties = _emberMetal.getProperties(this, dependentKeys);
      var res = _emberRuntimeSystemNative_array.A();
      for (var key in properties) {
        if (properties.hasOwnProperty(key)) {
          if (_emberMetal.isNone(properties[key])) {
            res.push(null);
          } else {
            res.push(properties[key]);
          }
        }
      }
      return res;
    });
  }

  /**
    A computed property which returns a new array with all the
    properties from the first dependent array sorted based on a property
    or sort function.
  
    The callback method you provide should have the following signature:
  
    ```javascript
    function(itemA, itemB);
    ```
  
    - `itemA` the first item to compare.
    - `itemB` the second item to compare.
  
    This function should return negative number (e.g. `-1`) when `itemA` should come before
    `itemB`. It should return positive number (e.g. `1`) when `itemA` should come after
    `itemB`. If the `itemA` and `itemB` are equal this function should return `0`.
  
    Therefore, if this function is comparing some numeric values, simple `itemA - itemB` or
    `itemA.get( 'foo' ) - itemB.get( 'foo' )` can be used instead of series of `if`.
  
    Example
  
    ```javascript
    let ToDoList = Ember.Object.extend({
      // using standard ascending sort
      todosSorting: ['name'],
      sortedTodos: Ember.computed.sort('todos', 'todosSorting'),
  
      // using descending sort
      todosSortingDesc: ['name:desc'],
      sortedTodosDesc: Ember.computed.sort('todos', 'todosSortingDesc'),
  
      // using a custom sort function
      priorityTodos: Ember.computed.sort('todos', function(a, b){
        if (a.priority > b.priority) {
          return 1;
        } else if (a.priority < b.priority) {
          return -1;
        }
  
        return 0;
      })
    });
  
    let todoList = ToDoList.create({todos: [
      { name: 'Unit Test', priority: 2 },
      { name: 'Documentation', priority: 3 },
      { name: 'Release', priority: 1 }
    ]});
  
    todoList.get('sortedTodos');      // [{ name:'Documentation', priority:3 }, { name:'Release', priority:1 }, { name:'Unit Test', priority:2 }]
    todoList.get('sortedTodosDesc');  // [{ name:'Unit Test', priority:2 }, { name:'Release', priority:1 }, { name:'Documentation', priority:3 }]
    todoList.get('priorityTodos');    // [{ name:'Release', priority:1 }, { name:'Unit Test', priority:2 }, { name:'Documentation', priority:3 }]
    ```
  
    @method sort
    @for Ember.computed
    @param {String} itemsKey
    @param {String or Function} sortDefinition a dependent key to an
    array of sort properties (add `:desc` to the arrays sort properties to sort descending) or a function to use when sorting
    @return {Ember.ComputedProperty} computes a new sorted array based
    on the sort property array or callback function
    @public
  */

  function sort(itemsKey, sortDefinition) {

    if (typeof sortDefinition === 'function') {
      return customSort(itemsKey, sortDefinition);
    } else {
      return propertySort(itemsKey, sortDefinition);
    }
  }

  function customSort(itemsKey, comparator) {
    return arrayMacro(itemsKey, function (value) {
      var _this4 = this;

      return value.slice().sort(function (x, y) {
        return comparator.call(_this4, x, y);
      });
    });
  }

  // This one needs to dynamically set up and tear down observers on the itemsKey
  // depending on the sortProperties
  function propertySort(itemsKey, sortPropertiesKey) {
    var cp = new _emberMetal.ComputedProperty(function (key) {
      var _this5 = this;

      var itemsKeyIsAtThis = itemsKey === '@this';
      var sortProperties = _emberMetal.get(this, sortPropertiesKey);

      var normalizedSortProperties = normalizeSortProperties(sortProperties);

      // Add/remove property observers as required.
      var activeObserversMap = cp._activeObserverMap || (cp._activeObserverMap = new _emberMetal.WeakMap());
      var activeObservers = activeObserversMap.get(this);

      if (activeObservers) {
        activeObservers.forEach(function (args) {
          return _emberMetal.removeObserver.apply(null, args);
        });
      }

      function sortPropertyDidChange() {
        this.notifyPropertyChange(key);
      }

      activeObservers = normalizedSortProperties.map(function (_ref) {
        var prop = _ref[0];

        var path = itemsKeyIsAtThis ? '@each.' + prop : itemsKey + '.@each.' + prop;
        var args = [_this5, path, sortPropertyDidChange];
        _emberMetal.addObserver.apply(null, args);
        return args;
      });

      activeObserversMap.set(this, activeObservers);

      // Sort and return the array.
      var items = itemsKeyIsAtThis ? this : _emberMetal.get(this, itemsKey);

      if (_emberRuntimeUtils.isArray(items)) {
        return sortByNormalizedSortProperties(items, normalizedSortProperties);
      } else {
        return _emberRuntimeSystemNative_array.A();
      }
    });

    cp._activeObserverMap = undefined;

    return cp.property(sortPropertiesKey + '.[]').readOnly();
  }

  function normalizeSortProperties(sortProperties) {
    return sortProperties.map(function (p) {
      var _p$split = p.split(':');

      var prop = _p$split[0];
      var direction = _p$split[1];

      direction = direction || 'asc';

      return [prop, direction];
    });
  }

  function sortByNormalizedSortProperties(items, normalizedSortProperties) {
    return _emberRuntimeSystemNative_array.A(items.slice().sort(function (itemA, itemB) {
      for (var i = 0; i < normalizedSortProperties.length; i++) {
        var _normalizedSortProperties$i = normalizedSortProperties[i];
        var prop = _normalizedSortProperties$i[0];
        var direction = _normalizedSortProperties$i[1];

        var result = _emberRuntimeCompare.default(_emberMetal.get(itemA, prop), _emberMetal.get(itemB, prop));
        if (result !== 0) {
          return direction === 'desc' ? -1 * result : result;
        }
      }

      return 0;
    }));
  }
});
enifed('ember-runtime/controllers/controller', ['exports', 'ember-metal', 'ember-runtime/system/object', 'ember-runtime/mixins/controller', 'ember-runtime/inject', 'ember-runtime/mixins/action_handler'], function (exports, _emberMetal, _emberRuntimeSystemObject, _emberRuntimeMixinsController, _emberRuntimeInject, _emberRuntimeMixinsAction_handler) {
  'use strict';

  /**
  @module ember
  @submodule ember-runtime
  */

  /**
    @class Controller
    @namespace Ember
    @extends Ember.Object
    @uses Ember.ControllerMixin
    @public
  */
  var Controller = _emberRuntimeSystemObject.default.extend(_emberRuntimeMixinsController.default);

  _emberRuntimeMixinsAction_handler.deprecateUnderscoreActions(Controller);

  function controllerInjectionHelper(factory) {}

  /**
    Creates a property that lazily looks up another controller in the container.
    Can only be used when defining another controller.
  
    Example:
  
    ```javascript
    App.PostController = Ember.Controller.extend({
      posts: Ember.inject.controller()
    });
    ```
  
    This example will create a `posts` property on the `post` controller that
    looks up the `posts` controller in the container, making it easy to
    reference other controllers. This is functionally equivalent to:
  
    ```javascript
    App.PostController = Ember.Controller.extend({
      needs: 'posts',
      posts: Ember.computed.alias('controllers.posts')
    });
    ```
  
    @method controller
    @since 1.10.0
    @for Ember.inject
    @param {String} name (optional) name of the controller to inject, defaults
           to the property's name
    @return {Ember.InjectedProperty} injection descriptor instance
    @public
  */
  _emberRuntimeInject.createInjectionHelper('controller', controllerInjectionHelper);

  exports.default = Controller;
});
enifed('ember-runtime/copy', ['exports', 'ember-metal', 'ember-runtime/system/object', 'ember-runtime/mixins/copyable'], function (exports, _emberMetal, _emberRuntimeSystemObject, _emberRuntimeMixinsCopyable) {
  'use strict';

  exports.default = copy;

  function _copy(obj, deep, seen, copies) {
    var ret = undefined,
        loc = undefined,
        key = undefined;

    // primitive data types are immutable, just return them.
    if (typeof obj !== 'object' || obj === null) {
      return obj;
    }

    // avoid cyclical loops
    if (deep && (loc = seen.indexOf(obj)) >= 0) {
      return copies[loc];
    }

    // IMPORTANT: this specific test will detect a native array only. Any other
    // object will need to implement Copyable.
    if (Array.isArray(obj)) {
      ret = obj.slice();

      if (deep) {
        loc = ret.length;

        while (--loc >= 0) {
          ret[loc] = _copy(ret[loc], deep, seen, copies);
        }
      }
    } else if (_emberRuntimeMixinsCopyable.default && _emberRuntimeMixinsCopyable.default.detect(obj)) {
      ret = obj.copy(deep, seen, copies);
    } else if (obj instanceof Date) {
      ret = new Date(obj.getTime());
    } else {
      ret = {};

      for (key in obj) {
        // support Null prototype
        if (!Object.prototype.hasOwnProperty.call(obj, key)) {
          continue;
        }

        // Prevents browsers that don't respect non-enumerability from
        // copying internal Ember properties
        if (key.substring(0, 2) === '__') {
          continue;
        }

        ret[key] = deep ? _copy(obj[key], deep, seen, copies) : obj[key];
      }
    }

    if (deep) {
      seen.push(obj);
      copies.push(ret);
    }

    return ret;
  }

  /**
    Creates a shallow copy of the passed object. A deep copy of the object is
    returned if the optional `deep` argument is `true`.
  
    If the passed object implements the `Ember.Copyable` interface, then this
    function will delegate to the object's `copy()` method and return the
    result. See `Ember.Copyable` for further details.
  
    For primitive values (which are immutable in JavaScript), the passed object
    is simply returned.
  
    @method copy
    @for Ember
    @param {Object} obj The object to clone
    @param {Boolean} [deep=false] If true, a deep copy of the object is made.
    @return {Object} The copied object
    @public
  */

  function copy(obj, deep) {
    // fast paths
    if ('object' !== typeof obj || obj === null) {
      return obj; // can't copy primitives
    }

    if (_emberRuntimeMixinsCopyable.default && _emberRuntimeMixinsCopyable.default.detect(obj)) {
      return obj.copy(deep);
    }

    return _copy(obj, deep, deep ? [] : null, deep ? [] : null);
  }
});
enifed('ember-runtime/ext/function', ['exports', 'ember-environment', 'ember-metal'], function (exports, _emberEnvironment, _emberMetal) {
  /**
  @module ember
  @submodule ember-runtime
  */

  'use strict';

  var a_slice = Array.prototype.slice;
  var FunctionPrototype = Function.prototype;

  if (_emberEnvironment.ENV.EXTEND_PROTOTYPES.Function) {
    /**
      The `property` extension of Javascript's Function prototype is available
      when `EmberENV.EXTEND_PROTOTYPES` or `EmberENV.EXTEND_PROTOTYPES.Function` is
      `true`, which is the default.
       Computed properties allow you to treat a function like a property:
       ```javascript
      MyApp.President = Ember.Object.extend({
        firstName: '',
        lastName:  '',
         fullName: function() {
          return this.get('firstName') + ' ' + this.get('lastName');
        }.property() // Call this flag to mark the function as a property
      });
       let president = MyApp.President.create({
        firstName: 'Barack',
        lastName: 'Obama'
      });
       president.get('fullName'); // 'Barack Obama'
      ```
       Treating a function like a property is useful because they can work with
      bindings, just like any other property.
       Many computed properties have dependencies on other properties. For
      example, in the above example, the `fullName` property depends on
      `firstName` and `lastName` to determine its value. You can tell Ember
      about these dependencies like this:
       ```javascript
      MyApp.President = Ember.Object.extend({
        firstName: '',
        lastName:  '',
         fullName: function() {
          return this.get('firstName') + ' ' + this.get('lastName');
           // Tell Ember.js that this computed property depends on firstName
          // and lastName
        }.property('firstName', 'lastName')
      });
      ```
       Make sure you list these dependencies so Ember knows when to update
      bindings that connect to a computed property. Changing a dependency
      will not immediately trigger an update of the computed property, but
      will instead clear the cache so that it is updated when the next `get`
      is called on the property.
       See [Ember.ComputedProperty](/api/classes/Ember.ComputedProperty.html), [Ember.computed](/api/classes/Ember.computed.html).
       @method property
      @for Function
      @public
    */
    FunctionPrototype.property = function () {
      var ret = _emberMetal.computed(this);
      // ComputedProperty.prototype.property expands properties; no need for us to
      // do so here.
      return ret.property.apply(ret, arguments);
    };

    /**
      The `observes` extension of Javascript's Function prototype is available
      when `EmberENV.EXTEND_PROTOTYPES` or `EmberENV.EXTEND_PROTOTYPES.Function` is
      true, which is the default.
       You can observe property changes simply by adding the `observes`
      call to the end of your method declarations in classes that you write.
      For example:
       ```javascript
      Ember.Object.extend({
        valueObserver: function() {
          // Executes whenever the "value" property changes
        }.observes('value')
      });
      ```
       In the future this method may become asynchronous.
       See `Ember.observer`.
       @method observes
      @for Function
      @public
    */
    FunctionPrototype.observes = function () {
      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      args.push(this);
      return _emberMetal.observer.apply(this, args);
    };

    FunctionPrototype._observesImmediately = function () {

      // observes handles property expansion
      return this.observes.apply(this, arguments);
    };
    /**
      The `observesImmediately` extension of Javascript's Function prototype is
      available when `EmberENV.EXTEND_PROTOTYPES` or
      `EmberENV.EXTEND_PROTOTYPES.Function` is true, which is the default.
       You can observe property changes simply by adding the `observesImmediately`
      call to the end of your method declarations in classes that you write.
      For example:
       ```javascript
      Ember.Object.extend({
        valueObserver: function() {
          // Executes immediately after the "value" property changes
        }.observesImmediately('value')
      });
      ```
       In the future, `observes` may become asynchronous. In this event,
      `observesImmediately` will maintain the synchronous behavior.
       See `Ember.immediateObserver`.
       @method observesImmediately
      @for Function
      @deprecated
      @private
    */
    FunctionPrototype.observesImmediately = _emberMetal.deprecateFunc('Function#observesImmediately is deprecated. Use Function#observes instead', { id: 'ember-runtime.ext-function', until: '3.0.0' }, FunctionPrototype._observesImmediately);

    /**
      The `on` extension of Javascript's Function prototype is available
      when `EmberENV.EXTEND_PROTOTYPES` or `EmberENV.EXTEND_PROTOTYPES.Function` is
      true, which is the default.
       You can listen for events simply by adding the `on` call to the end of
      your method declarations in classes or mixins that you write. For example:
       ```javascript
      Ember.Mixin.create({
        doSomethingWithElement: function() {
          // Executes whenever the "didInsertElement" event fires
        }.on('didInsertElement')
      });
      ```
       See `Ember.on`.
       @method on
      @for Function
      @public
    */
    FunctionPrototype.on = function () {
      var events = a_slice.call(arguments);
      this.__ember_listens__ = events;

      return this;
    };
  }
});
enifed('ember-runtime/ext/rsvp', ['exports', 'rsvp', 'ember-metal'], function (exports, _rsvp, _emberMetal) {
  'use strict';

  exports.onerrorDefault = onerrorDefault;

  var backburner = _emberMetal.run.backburner;
  _emberMetal.run._addQueue('rsvpAfter', 'destroy');

  _rsvp.configure('async', function (callback, promise) {
    backburner.schedule('actions', null, callback, promise);
  });

  _rsvp.configure('after', function (cb) {
    backburner.schedule('rsvpAfter', null, cb);
  });

  _rsvp.on('error', onerrorDefault);

  function onerrorDefault(reason) {
    var error = errorFor(reason);
    if (error) {
      _emberMetal.dispatchError(error);
    }
  }

  function errorFor(reason) {
    if (!reason) return;

    if (reason.errorThrown) {
      return unwrapErrorThrown(reason);
    }

    if (reason.name === 'UnrecognizedURLError') {
      return;
    }

    if (reason.name === 'TransitionAborted') {
      return;
    }

    return reason;
  }

  function unwrapErrorThrown(reason) {
    var error = reason.errorThrown;
    if (typeof error === 'string') {
      error = new Error(error);
    }
    Object.defineProperty(error, '__reason_with_error_thrown__', {
      value: reason,
      enumerable: false
    });
    return error;
  }

  exports.default = _rsvp;
});
enifed('ember-runtime/ext/string', ['exports', 'ember-environment', 'ember-runtime/system/string'], function (exports, _emberEnvironment, _emberRuntimeSystemString) {
  /**
  @module ember
  @submodule ember-runtime
  */

  'use strict';

  var StringPrototype = String.prototype;

  if (_emberEnvironment.ENV.EXTEND_PROTOTYPES.String) {
    /**
      See [Ember.String.fmt](/api/classes/Ember.String.html#method_fmt).
       @method fmt
      @for String
      @private
      @deprecated
    */
    StringPrototype.fmt = function () {
      for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      return _emberRuntimeSystemString.fmt(this, args);
    };

    /**
      See [Ember.String.w](/api/classes/Ember.String.html#method_w).
       @method w
      @for String
      @private
    */
    StringPrototype.w = function () {
      return _emberRuntimeSystemString.w(this);
    };

    /**
      See [Ember.String.loc](/api/classes/Ember.String.html#method_loc).
       @method loc
      @for String
      @private
    */
    StringPrototype.loc = function () {
      for (var _len2 = arguments.length, args = Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        args[_key2] = arguments[_key2];
      }

      return _emberRuntimeSystemString.loc(this, args);
    };

    /**
      See [Ember.String.camelize](/api/classes/Ember.String.html#method_camelize).
       @method camelize
      @for String
      @private
    */
    StringPrototype.camelize = function () {
      return _emberRuntimeSystemString.camelize(this);
    };

    /**
      See [Ember.String.decamelize](/api/classes/Ember.String.html#method_decamelize).
       @method decamelize
      @for String
      @private
    */
    StringPrototype.decamelize = function () {
      return _emberRuntimeSystemString.decamelize(this);
    };

    /**
      See [Ember.String.dasherize](/api/classes/Ember.String.html#method_dasherize).
       @method dasherize
      @for String
      @private
    */
    StringPrototype.dasherize = function () {
      return _emberRuntimeSystemString.dasherize(this);
    };

    /**
      See [Ember.String.underscore](/api/classes/Ember.String.html#method_underscore).
       @method underscore
      @for String
      @private
    */
    StringPrototype.underscore = function () {
      return _emberRuntimeSystemString.underscore(this);
    };

    /**
      See [Ember.String.classify](/api/classes/Ember.String.html#method_classify).
       @method classify
      @for String
      @private
    */
    StringPrototype.classify = function () {
      return _emberRuntimeSystemString.classify(this);
    };

    /**
      See [Ember.String.capitalize](/api/classes/Ember.String.html#method_capitalize).
       @method capitalize
      @for String
      @private
    */
    StringPrototype.capitalize = function () {
      return _emberRuntimeSystemString.capitalize(this);
    };
  }
});
enifed('ember-runtime/index', ['exports', 'ember-runtime/ext/string', 'ember-runtime/ext/function', 'ember-runtime/system/object', 'ember-runtime/system/string', 'ember-runtime/mixins/registry_proxy', 'ember-runtime/mixins/container_proxy', 'ember-runtime/copy', 'ember-runtime/inject', 'ember-runtime/compare', 'ember-runtime/is-equal', 'ember-runtime/mixins/array', 'ember-runtime/mixins/comparable', 'ember-runtime/system/namespace', 'ember-runtime/system/array_proxy', 'ember-runtime/system/object_proxy', 'ember-runtime/system/core_object', 'ember-runtime/system/native_array', 'ember-runtime/mixins/action_handler', 'ember-runtime/mixins/copyable', 'ember-runtime/mixins/enumerable', 'ember-runtime/mixins/freezable', 'ember-runtime/mixins/-proxy', 'ember-runtime/system/lazy_load', 'ember-runtime/mixins/observable', 'ember-runtime/mixins/mutable_enumerable', 'ember-runtime/mixins/mutable_array', 'ember-runtime/mixins/target_action_support', 'ember-runtime/mixins/evented', 'ember-runtime/mixins/promise_proxy', 'ember-runtime/computed/computed_macros', 'ember-runtime/computed/reduce_computed_macros', 'ember-runtime/controllers/controller', 'ember-runtime/mixins/controller', 'ember-runtime/system/service', 'ember-runtime/ext/rsvp', 'ember-runtime/utils', 'ember-runtime/string_registry'], function (exports, _emberRuntimeExtString, _emberRuntimeExtFunction, _emberRuntimeSystemObject, _emberRuntimeSystemString, _emberRuntimeMixinsRegistry_proxy, _emberRuntimeMixinsContainer_proxy, _emberRuntimeCopy, _emberRuntimeInject, _emberRuntimeCompare, _emberRuntimeIsEqual, _emberRuntimeMixinsArray, _emberRuntimeMixinsComparable, _emberRuntimeSystemNamespace, _emberRuntimeSystemArray_proxy, _emberRuntimeSystemObject_proxy, _emberRuntimeSystemCore_object, _emberRuntimeSystemNative_array, _emberRuntimeMixinsAction_handler, _emberRuntimeMixinsCopyable, _emberRuntimeMixinsEnumerable, _emberRuntimeMixinsFreezable, _emberRuntimeMixinsProxy, _emberRuntimeSystemLazy_load, _emberRuntimeMixinsObservable, _emberRuntimeMixinsMutable_enumerable, _emberRuntimeMixinsMutable_array, _emberRuntimeMixinsTarget_action_support, _emberRuntimeMixinsEvented, _emberRuntimeMixinsPromise_proxy, _emberRuntimeComputedComputed_macros, _emberRuntimeComputedReduce_computed_macros, _emberRuntimeControllersController, _emberRuntimeMixinsController, _emberRuntimeSystemService, _emberRuntimeExtRsvp, _emberRuntimeUtils, _emberRuntimeString_registry) {
  /**
  @module ember
  @submodule ember-runtime
  */

  'use strict';

  exports.Object = _emberRuntimeSystemObject.default;
  exports.FrameworkObject = _emberRuntimeSystemObject.FrameworkObject;
  exports.String = _emberRuntimeSystemString.default;
  exports.RegistryProxyMixin = _emberRuntimeMixinsRegistry_proxy.default;
  exports.buildFakeRegistryWithDeprecations = _emberRuntimeMixinsRegistry_proxy.buildFakeRegistryWithDeprecations;
  exports.ContainerProxyMixin = _emberRuntimeMixinsContainer_proxy.default;
  exports.copy = _emberRuntimeCopy.default;
  exports.inject = _emberRuntimeInject.default;
  exports.compare = _emberRuntimeCompare.default;
  exports.isEqual = _emberRuntimeIsEqual.default;
  exports.Array = _emberRuntimeMixinsArray.default;
  exports.objectAt = _emberRuntimeMixinsArray.objectAt;
  exports.isEmberArray = _emberRuntimeMixinsArray.isEmberArray;
  exports.addArrayObserver = _emberRuntimeMixinsArray.addArrayObserver;
  exports.removeArrayObserver = _emberRuntimeMixinsArray.removeArrayObserver;
  exports.Comparable = _emberRuntimeMixinsComparable.default;
  exports.Namespace = _emberRuntimeSystemNamespace.default;
  exports.isNamespaceSearchDisabled = _emberRuntimeSystemNamespace.isSearchDisabled;
  exports.setNamespaceSearchDisabled = _emberRuntimeSystemNamespace.setSearchDisabled;
  exports.ArrayProxy = _emberRuntimeSystemArray_proxy.default;
  exports.ObjectProxy = _emberRuntimeSystemObject_proxy.default;
  exports.CoreObject = _emberRuntimeSystemCore_object.default;
  exports.NativeArray = _emberRuntimeSystemNative_array.default;
  exports.A = _emberRuntimeSystemNative_array.A;
  exports.ActionHandler = _emberRuntimeMixinsAction_handler.default;
  exports.deprecateUnderscoreActions = _emberRuntimeMixinsAction_handler.deprecateUnderscoreActions;
  exports.Copyable = _emberRuntimeMixinsCopyable.default;
  exports.Enumerable = _emberRuntimeMixinsEnumerable.default;
  exports.Freezable = _emberRuntimeMixinsFreezable.Freezable;
  exports.FROZEN_ERROR = _emberRuntimeMixinsFreezable.FROZEN_ERROR;
  exports._ProxyMixin = _emberRuntimeMixinsProxy.default;
  exports.onLoad = _emberRuntimeSystemLazy_load.onLoad;
  exports.runLoadHooks = _emberRuntimeSystemLazy_load.runLoadHooks;
  exports._loaded = _emberRuntimeSystemLazy_load._loaded;
  exports.Observable = _emberRuntimeMixinsObservable.default;
  exports.MutableEnumerable = _emberRuntimeMixinsMutable_enumerable.default;
  exports.MutableArray = _emberRuntimeMixinsMutable_array.default;
  exports.removeAt = _emberRuntimeMixinsMutable_array.removeAt;
  exports.TargetActionSupport = _emberRuntimeMixinsTarget_action_support.default;
  exports.Evented = _emberRuntimeMixinsEvented.default;
  exports.PromiseProxyMixin = _emberRuntimeMixinsPromise_proxy.default;
  exports.empty = _emberRuntimeComputedComputed_macros.empty;
  exports.notEmpty = _emberRuntimeComputedComputed_macros.notEmpty;
  exports.none = _emberRuntimeComputedComputed_macros.none;
  exports.not = _emberRuntimeComputedComputed_macros.not;
  exports.bool = _emberRuntimeComputedComputed_macros.bool;
  exports.match = _emberRuntimeComputedComputed_macros.match;
  exports.equal = _emberRuntimeComputedComputed_macros.equal;
  exports.gt = _emberRuntimeComputedComputed_macros.gt;
  exports.gte = _emberRuntimeComputedComputed_macros.gte;
  exports.lt = _emberRuntimeComputedComputed_macros.lt;
  exports.lte = _emberRuntimeComputedComputed_macros.lte;
  exports.oneWay = _emberRuntimeComputedComputed_macros.oneWay;
  exports.readOnly = _emberRuntimeComputedComputed_macros.readOnly;
  exports.deprecatingAlias = _emberRuntimeComputedComputed_macros.deprecatingAlias;
  exports.and = _emberRuntimeComputedComputed_macros.and;
  exports.or = _emberRuntimeComputedComputed_macros.or;
  exports.sum = _emberRuntimeComputedReduce_computed_macros.sum;
  exports.min = _emberRuntimeComputedReduce_computed_macros.min;
  exports.max = _emberRuntimeComputedReduce_computed_macros.max;
  exports.map = _emberRuntimeComputedReduce_computed_macros.map;
  exports.sort = _emberRuntimeComputedReduce_computed_macros.sort;
  exports.setDiff = _emberRuntimeComputedReduce_computed_macros.setDiff;
  exports.mapBy = _emberRuntimeComputedReduce_computed_macros.mapBy;
  exports.filter = _emberRuntimeComputedReduce_computed_macros.filter;
  exports.filterBy = _emberRuntimeComputedReduce_computed_macros.filterBy;
  exports.uniq = _emberRuntimeComputedReduce_computed_macros.uniq;
  exports.uniqBy = _emberRuntimeComputedReduce_computed_macros.uniqBy;
  exports.union = _emberRuntimeComputedReduce_computed_macros.union;
  exports.intersect = _emberRuntimeComputedReduce_computed_macros.intersect;
  exports.collect = _emberRuntimeComputedReduce_computed_macros.collect;
  exports.Controller = _emberRuntimeControllersController.default;
  exports.ControllerMixin = _emberRuntimeMixinsController.default;
  exports.Service = _emberRuntimeSystemService.default;
  exports.RSVP = _emberRuntimeExtRsvp.default;
  exports.onerrorDefault = _emberRuntimeExtRsvp.onerrorDefault;
  // just for side effect of extending Ember.RSVP
  exports.isArray = _emberRuntimeUtils.isArray;
  exports.typeOf = _emberRuntimeUtils.typeOf;
  exports.getStrings = _emberRuntimeString_registry.getStrings;
  exports.setStrings = _emberRuntimeString_registry.setStrings;
});
// just for side effect of extending String.prototype
// just for side effect of extending Function.prototype
enifed('ember-runtime/inject', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  exports.default = inject;
  exports.createInjectionHelper = createInjectionHelper;
  exports.validatePropertyInjections = validatePropertyInjections;

  /**
    Namespace for injection helper methods.
  
    @class inject
    @namespace Ember
    @static
    @public
  */

  function inject() {}

  // Dictionary of injection validations by type, added to by `createInjectionHelper`
  var typeValidators = {};

  /**
    This method allows other Ember modules to register injection helpers for a
    given container type. Helpers are exported to the `inject` namespace as the
    container type itself.
  
    @private
    @method createInjectionHelper
    @since 1.10.0
    @for Ember
    @param {String} type The container type the helper will inject
    @param {Function} validator A validation callback that is executed at mixin-time
  */

  function createInjectionHelper(type, validator) {
    typeValidators[type] = validator;

    inject[type] = function (name) {
      return new _emberMetal.InjectedProperty(type, name);
    };
  }

  /**
    Validation function that runs per-type validation functions once for each
    injected type encountered.
  
    @private
    @method validatePropertyInjections
    @since 1.10.0
    @for Ember
    @param {Object} factory The factory object
  */

  function validatePropertyInjections(factory) {
    var proto = factory.proto();
    var types = [];

    for (var key in proto) {
      var desc = proto[key];
      if (desc instanceof _emberMetal.InjectedProperty && types.indexOf(desc.type) === -1) {
        types.push(desc.type);
      }
    }

    if (types.length) {
      for (var i = 0; i < types.length; i++) {
        var validator = typeValidators[types[i]];

        if (typeof validator === 'function') {
          validator(factory);
        }
      }
    }

    return true;
  }
});
enifed('ember-runtime/is-equal', ['exports'], function (exports) {
  /**
    Compares two objects, returning true if they are equal.
  
    ```javascript
    Ember.isEqual('hello', 'hello');                   // true
    Ember.isEqual(1, 2);                               // false
    ```
  
    `isEqual` is a more specific comparison than a triple equal comparison.
    It will call the `isEqual` instance method on the objects being
    compared, allowing finer control over when objects should be considered
    equal to each other.
  
    ```javascript
    let Person = Ember.Object.extend({
      isEqual(other) { return this.ssn == other.ssn; }
    });
  
    let personA = Person.create({name: 'Muhammad Ali', ssn: '123-45-6789'});
    let personB = Person.create({name: 'Cassius Clay', ssn: '123-45-6789'});
  
    Ember.isEqual(personA, personB); // true
    ```
  
    Due to the expense of array comparisons, collections will never be equal to
    each other even if each of their items are equal to each other.
  
    ```javascript
    Ember.isEqual([4, 2], [4, 2]);                     // false
    ```
  
    @method isEqual
    @for Ember
    @param {Object} a first object to compare
    @param {Object} b second object to compare
    @return {Boolean}
    @public
  */
  'use strict';

  exports.default = isEqual;

  function isEqual(a, b) {
    if (a && typeof a.isEqual === 'function') {
      return a.isEqual(b);
    }

    if (a instanceof Date && b instanceof Date) {
      return a.getTime() === b.getTime();
    }

    return a === b;
  }
});
enifed('ember-runtime/mixins/-proxy', ['exports', 'glimmer-reference', 'ember-metal', 'ember-runtime/computed/computed_macros'], function (exports, _glimmerReference, _emberMetal, _emberRuntimeComputedComputed_macros) {
  /**
  @module ember
  @submodule ember-runtime
  */

  'use strict';

  function contentPropertyWillChange(content, contentKey) {
    var key = contentKey.slice(8); // remove "content."
    if (key in this) {
      return;
    } // if shadowed in proxy
    _emberMetal.propertyWillChange(this, key);
  }

  function contentPropertyDidChange(content, contentKey) {
    var key = contentKey.slice(8); // remove "content."
    if (key in this) {
      return;
    } // if shadowed in proxy
    _emberMetal.propertyDidChange(this, key);
  }

  var ProxyTag = (function (_CachedTag) {
    babelHelpers.inherits(ProxyTag, _CachedTag);

    function ProxyTag(proxy) {
      _CachedTag.call(this);

      var content = _emberMetal.get(proxy, 'content');

      this.proxy = proxy;
      this.proxyWrapperTag = new _glimmerReference.DirtyableTag();
      this.proxyContentTag = new _glimmerReference.UpdatableTag(_emberMetal.tagFor(content));
    }

    /**
      `Ember.ProxyMixin` forwards all properties not defined by the proxy itself
      to a proxied `content` object.  See Ember.ObjectProxy for more details.
    
      @class ProxyMixin
      @namespace Ember
      @private
    */

    ProxyTag.prototype.compute = function compute() {
      return Math.max(this.proxyWrapperTag.value(), this.proxyContentTag.value());
    };

    ProxyTag.prototype.dirty = function dirty() {
      this.proxyWrapperTag.dirty();
    };

    ProxyTag.prototype.contentDidChange = function contentDidChange() {
      var content = _emberMetal.get(this.proxy, 'content');
      this.proxyContentTag.update(_emberMetal.tagFor(content));
    };

    return ProxyTag;
  })(_glimmerReference.CachedTag);

  exports.default = _emberMetal.Mixin.create({
    /**
      The object whose properties will be forwarded.
       @property content
      @type Ember.Object
      @default null
      @private
    */
    content: null,

    init: function () {
      this._super.apply(this, arguments);
      _emberMetal.meta(this).setProxy();
    },

    _initializeTag: _emberMetal.on('init', function () {
      _emberMetal.meta(this)._tag = new ProxyTag(this);
    }),

    _contentDidChange: _emberMetal.observer('content', function () {
      _emberMetal.tagFor(this).contentDidChange();
    }),

    isTruthy: _emberRuntimeComputedComputed_macros.bool('content'),

    _debugContainerKey: null,

    willWatchProperty: function (key) {
      var contentKey = 'content.' + key;
      _emberMetal._addBeforeObserver(this, contentKey, null, contentPropertyWillChange);
      _emberMetal.addObserver(this, contentKey, null, contentPropertyDidChange);
    },

    didUnwatchProperty: function (key) {
      var contentKey = 'content.' + key;
      _emberMetal._removeBeforeObserver(this, contentKey, null, contentPropertyWillChange);
      _emberMetal.removeObserver(this, contentKey, null, contentPropertyDidChange);
    },

    unknownProperty: function (key) {
      var content = _emberMetal.get(this, 'content');
      if (content) {
        return _emberMetal.get(content, key);
      }
    },

    setUnknownProperty: function (key, value) {
      var m = _emberMetal.meta(this);
      if (m.proto === this) {
        // if marked as prototype then just defineProperty
        // rather than delegate
        _emberMetal.defineProperty(this, key, null, value);
        return value;
      }

      var content = _emberMetal.get(this, 'content');

      return _emberMetal.set(content, key, value);
    }
  });
});
enifed('ember-runtime/mixins/action_handler', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  /**
  @module ember
  @submodule ember-runtime
  */

  'use strict';

  exports.deprecateUnderscoreActions = deprecateUnderscoreActions;

  /**
    `Ember.ActionHandler` is available on some familiar classes including
    `Ember.Route`, `Ember.Component`, and `Ember.Controller`.
    (Internally the mixin is used by `Ember.CoreView`, `Ember.ControllerMixin`,
    and `Ember.Route` and available to the above classes through
    inheritance.)
  
    @class ActionHandler
    @namespace Ember
    @private
  */
  var ActionHandler = _emberMetal.Mixin.create({
    mergedProperties: ['actions'],

    /**
      The collection of functions, keyed by name, available on this
      `ActionHandler` as action targets.
       These functions will be invoked when a matching `{{action}}` is triggered
      from within a template and the application's current route is this route.
       Actions can also be invoked from other parts of your application
      via `ActionHandler#send`.
       The `actions` hash will inherit action handlers from
      the `actions` hash defined on extended parent classes
      or mixins rather than just replace the entire hash, e.g.:
       ```js
      App.CanDisplayBanner = Ember.Mixin.create({
        actions: {
          displayBanner(msg) {
            // ...
          }
        }
      });
       App.WelcomeRoute = Ember.Route.extend(App.CanDisplayBanner, {
        actions: {
          playMusic() {
            // ...
          }
        }
      });
       // `WelcomeRoute`, when active, will be able to respond
      // to both actions, since the actions hash is merged rather
      // then replaced when extending mixins / parent classes.
      this.send('displayBanner');
      this.send('playMusic');
      ```
       Within a Controller, Route or Component's action handler,
      the value of the `this` context is the Controller, Route or
      Component object:
       ```js
      App.SongRoute = Ember.Route.extend({
        actions: {
          myAction() {
            this.controllerFor("song");
            this.transitionTo("other.route");
            ...
          }
        }
      });
      ```
       It is also possible to call `this._super(...arguments)` from within an
      action handler if it overrides a handler defined on a parent
      class or mixin:
       Take for example the following routes:
       ```js
      App.DebugRoute = Ember.Mixin.create({
        actions: {
          debugRouteInformation() {
            console.debug("trololo");
          }
        }
      });
       App.AnnoyingDebugRoute = Ember.Route.extend(App.DebugRoute, {
        actions: {
          debugRouteInformation() {
            // also call the debugRouteInformation of mixed in App.DebugRoute
            this._super(...arguments);
             // show additional annoyance
            window.alert(...);
          }
        }
      });
      ```
       ## Bubbling
       By default, an action will stop bubbling once a handler defined
      on the `actions` hash handles it. To continue bubbling the action,
      you must return `true` from the handler:
       ```js
      App.Router.map(function() {
        this.route("album", function() {
          this.route("song");
        });
      });
       App.AlbumRoute = Ember.Route.extend({
        actions: {
          startPlaying: function() {
          }
        }
      });
       App.AlbumSongRoute = Ember.Route.extend({
        actions: {
          startPlaying() {
            // ...
             if (actionShouldAlsoBeTriggeredOnParentRoute) {
              return true;
            }
          }
        }
      });
      ```
       @property actions
      @type Object
      @default null
      @public
    */

    /**
      Triggers a named action on the `ActionHandler`. Any parameters
      supplied after the `actionName` string will be passed as arguments
      to the action target function.
       If the `ActionHandler` has its `target` property set, actions may
      bubble to the `target`. Bubbling happens when an `actionName` can
      not be found in the `ActionHandler`'s `actions` hash or if the
      action target function returns `true`.
       Example
       ```js
      App.WelcomeRoute = Ember.Route.extend({
        actions: {
          playTheme() {
             this.send('playMusic', 'theme.mp3');
          },
          playMusic(track) {
            // ...
          }
        }
      });
      ```
       @method send
      @param {String} actionName The action to trigger
      @param {*} context a context to send with the action
      @public
    */
    send: function (actionName) {
      for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
        args[_key - 1] = arguments[_key];
      }

      if (this.actions && this.actions[actionName]) {
        var shouldBubble = this.actions[actionName].apply(this, args) === true;
        if (!shouldBubble) {
          return;
        }
      }

      var target = _emberMetal.get(this, 'target');
      if (target) {
        target.send.apply(target, arguments);
      }
    },

    willMergeMixin: function (props) {

      if (props._actions) {

        props.actions = props._actions;
        delete props._actions;
      }
    }
  });

  exports.default = ActionHandler;

  function deprecateUnderscoreActions(factory) {
    Object.defineProperty(factory.prototype, '_actions', {
      configurable: true,
      enumerable: false,
      set: function (value) {},
      get: function () {
        return _emberMetal.get(this, 'actions');
      }
    });
  }
});
enifed('ember-runtime/mixins/array', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime/mixins/enumerable', 'ember-runtime/system/each_proxy'], function (exports, _emberUtils, _emberMetal, _emberRuntimeMixinsEnumerable, _emberRuntimeSystemEach_proxy) {
  /**
  @module ember
  @submodule ember-runtime
  */

  // ..........................................................
  // HELPERS
  //
  'use strict';

  var _Mixin$create;

  exports.addArrayObserver = addArrayObserver;
  exports.removeArrayObserver = removeArrayObserver;
  exports.objectAt = objectAt;
  exports.arrayContentWillChange = arrayContentWillChange;
  exports.arrayContentDidChange = arrayContentDidChange;
  exports.isEmberArray = isEmberArray;

  function arrayObserversHelper(obj, target, opts, operation, notify) {
    var willChange = opts && opts.willChange || 'arrayWillChange';
    var didChange = opts && opts.didChange || 'arrayDidChange';
    var hasObservers = _emberMetal.get(obj, 'hasArrayObservers');

    if (hasObservers === notify) {
      _emberMetal.propertyWillChange(obj, 'hasArrayObservers');
    }

    operation(obj, '@array:before', target, willChange);
    operation(obj, '@array:change', target, didChange);

    if (hasObservers === notify) {
      _emberMetal.propertyDidChange(obj, 'hasArrayObservers');
    }

    return obj;
  }

  function addArrayObserver(array, target, opts) {
    return arrayObserversHelper(array, target, opts, _emberMetal.addListener, false);
  }

  function removeArrayObserver(array, target, opts) {
    return arrayObserversHelper(array, target, opts, _emberMetal.removeListener, true);
  }

  function objectAt(content, idx) {
    if (content.objectAt) {
      return content.objectAt(idx);
    }

    return content[idx];
  }

  function arrayContentWillChange(array, startIdx, removeAmt, addAmt) {
    var removing = undefined,
        lim = undefined;

    // if no args are passed assume everything changes
    if (startIdx === undefined) {
      startIdx = 0;
      removeAmt = addAmt = -1;
    } else {
      if (removeAmt === undefined) {
        removeAmt = -1;
      }

      if (addAmt === undefined) {
        addAmt = -1;
      }
    }

    if (array.__each) {
      array.__each.arrayWillChange(array, startIdx, removeAmt, addAmt);
    }

    _emberMetal.sendEvent(array, '@array:before', [array, startIdx, removeAmt, addAmt]);

    if (startIdx >= 0 && removeAmt >= 0 && _emberMetal.get(array, 'hasEnumerableObservers')) {
      removing = [];
      lim = startIdx + removeAmt;

      for (var idx = startIdx; idx < lim; idx++) {
        removing.push(objectAt(array, idx));
      }
    } else {
      removing = removeAmt;
    }

    array.enumerableContentWillChange(removing, addAmt);

    return array;
  }

  function arrayContentDidChange(array, startIdx, removeAmt, addAmt) {
    // if no args are passed assume everything changes
    if (startIdx === undefined) {
      startIdx = 0;
      removeAmt = addAmt = -1;
    } else {
      if (removeAmt === undefined) {
        removeAmt = -1;
      }

      if (addAmt === undefined) {
        addAmt = -1;
      }
    }

    var adding = undefined;
    if (startIdx >= 0 && addAmt >= 0 && _emberMetal.get(array, 'hasEnumerableObservers')) {
      adding = [];
      var lim = startIdx + addAmt;

      for (var idx = startIdx; idx < lim; idx++) {
        adding.push(objectAt(array, idx));
      }
    } else {
      adding = addAmt;
    }

    array.enumerableContentDidChange(removeAmt, adding);

    if (array.__each) {
      array.__each.arrayDidChange(array, startIdx, removeAmt, addAmt);
    }

    _emberMetal.sendEvent(array, '@array:change', [array, startIdx, removeAmt, addAmt]);

    var meta = _emberMetal.peekMeta(array);
    var cache = meta && meta.readableCache();

    if (cache) {
      if (cache.firstObject !== undefined && objectAt(array, 0) !== _emberMetal.cacheFor.get(cache, 'firstObject')) {
        _emberMetal.propertyWillChange(array, 'firstObject');
        _emberMetal.propertyDidChange(array, 'firstObject');
      }
      if (cache.lastObject !== undefined && objectAt(array, _emberMetal.get(array, 'length') - 1) !== _emberMetal.cacheFor.get(cache, 'lastObject')) {
        _emberMetal.propertyWillChange(array, 'lastObject');
        _emberMetal.propertyDidChange(array, 'lastObject');
      }
    }
    return array;
  }

  var EMBER_ARRAY = _emberUtils.symbol('EMBER_ARRAY');

  function isEmberArray(obj) {
    return obj && !!obj[EMBER_ARRAY];
  }

  // ..........................................................
  // ARRAY
  //
  /**
    This mixin implements Observer-friendly Array-like behavior. It is not a
    concrete implementation, but it can be used up by other classes that want
    to appear like arrays.
  
    For example, ArrayProxy is a concrete classes that can
    be instantiated to implement array-like behavior. Both of these classes use
    the Array Mixin by way of the MutableArray mixin, which allows observable
    changes to be made to the underlying array.
  
    Unlike `Ember.Enumerable,` this mixin defines methods specifically for
    collections that provide index-ordered access to their contents. When you
    are designing code that needs to accept any kind of Array-like object, you
    should use these methods instead of Array primitives because these will
    properly notify observers of changes to the array.
  
    Although these methods are efficient, they do add a layer of indirection to
    your application so it is a good idea to use them only when you need the
    flexibility of using both true JavaScript arrays and "virtual" arrays such
    as controllers and collections.
  
    You can use the methods defined in this module to access and modify array
    contents in a KVO-friendly way. You can also be notified whenever the
    membership of an array changes by using `.observes('myArray.[]')`.
  
    To support `Ember.Array` in your own class, you must override two
    primitives to use it: `length()` and `objectAt()`.
  
    Note that the Ember.Array mixin also incorporates the `Ember.Enumerable`
    mixin. All `Ember.Array`-like objects are also enumerable.
  
    @class Array
    @namespace Ember
    @uses Ember.Enumerable
    @since Ember 0.9.0
    @public
  */
  var ArrayMixin = _emberMetal.Mixin.create(_emberRuntimeMixinsEnumerable.default, (_Mixin$create = {}, _Mixin$create[EMBER_ARRAY] = true, _Mixin$create.length = null, _Mixin$create.objectAt = function (idx) {
    if (idx < 0 || idx >= _emberMetal.get(this, 'length')) {
      return undefined;
    }

    return _emberMetal.get(this, idx);
  }, _Mixin$create.objectsAt = function (indexes) {
    var _this = this;

    return indexes.map(function (idx) {
      return objectAt(_this, idx);
    });
  }, _Mixin$create.nextObject = function (idx) {
    return objectAt(this, idx);
  }, _Mixin$create['[]'] = _emberMetal.computed({
    get: function (key) {
      return this;
    },
    set: function (key, value) {
      this.replace(0, _emberMetal.get(this, 'length'), value);
      return this;
    }
  }), _Mixin$create.firstObject = _emberMetal.computed(function () {
    return objectAt(this, 0);
  }).readOnly(), _Mixin$create.lastObject = _emberMetal.computed(function () {
    return objectAt(this, _emberMetal.get(this, 'length') - 1);
  }).readOnly(), _Mixin$create.contains = function (obj) {
    if (true) {}

    return this.indexOf(obj) >= 0;
  }, _Mixin$create.slice = function (beginIndex, endIndex) {
    var ret = _emberMetal.default.A();
    var length = _emberMetal.get(this, 'length');

    if (_emberMetal.isNone(beginIndex)) {
      beginIndex = 0;
    }

    if (_emberMetal.isNone(endIndex) || endIndex > length) {
      endIndex = length;
    }

    if (beginIndex < 0) {
      beginIndex = length + beginIndex;
    }

    if (endIndex < 0) {
      endIndex = length + endIndex;
    }

    while (beginIndex < endIndex) {
      ret[ret.length] = objectAt(this, beginIndex++);
    }

    return ret;
  }, _Mixin$create.indexOf = function (object, startAt) {
    var len = _emberMetal.get(this, 'length');

    if (startAt === undefined) {
      startAt = 0;
    }

    if (startAt < 0) {
      startAt += len;
    }

    for (var idx = startAt; idx < len; idx++) {
      if (objectAt(this, idx) === object) {
        return idx;
      }
    }

    return -1;
  }, _Mixin$create.lastIndexOf = function (object, startAt) {
    var len = _emberMetal.get(this, 'length');

    if (startAt === undefined || startAt >= len) {
      startAt = len - 1;
    }

    if (startAt < 0) {
      startAt += len;
    }

    for (var idx = startAt; idx >= 0; idx--) {
      if (objectAt(this, idx) === object) {
        return idx;
      }
    }

    return -1;
  }, _Mixin$create.addArrayObserver = function (target, opts) {
    return addArrayObserver(this, target, opts);
  }, _Mixin$create.removeArrayObserver = function (target, opts) {
    return removeArrayObserver(this, target, opts);
  }, _Mixin$create.hasArrayObservers = _emberMetal.computed(function () {
    return _emberMetal.hasListeners(this, '@array:change') || _emberMetal.hasListeners(this, '@array:before');
  }), _Mixin$create.arrayContentWillChange = function (startIdx, removeAmt, addAmt) {
    return arrayContentWillChange(this, startIdx, removeAmt, addAmt);
  }, _Mixin$create.arrayContentDidChange = function (startIdx, removeAmt, addAmt) {
    return arrayContentDidChange(this, startIdx, removeAmt, addAmt);
  }, _Mixin$create['@each'] = _emberMetal.computed(function () {
    // TODO use Symbol or add to meta
    if (!this.__each) {
      this.__each = new _emberRuntimeSystemEach_proxy.default(this);
    }

    return this.__each;
  }).volatile().readOnly(), _Mixin$create));

  if (true) {
    ArrayMixin.reopen({
      /**
        Returns `true` if the passed object can be found in the array.
        This method is a Polyfill for ES 2016 Array.includes.
        If no `startAt` argument is given, the starting location to
        search is 0. If it's negative, searches from the index of
        `this.length + startAt` by asc.
         ```javascript
        [1, 2, 3].includes(2);     // true
        [1, 2, 3].includes(4);     // false
        [1, 2, 3].includes(3, 2);  // true
        [1, 2, 3].includes(3, 3);  // false
        [1, 2, 3].includes(3, -1); // true
        [1, 2, 3].includes(1, -1); // false
        [1, 2, 3].includes(1, -4); // true
        [1, 2, NaN].includes(NaN); // true
        ```
         @method includes
        @param {Object} obj The object to search for.
        @param {Number} startAt optional starting location to search, default 0
        @return {Boolean} `true` if object is found in the array.
        @public
      */
      includes: function (obj, startAt) {
        var len = _emberMetal.get(this, 'length');

        if (startAt === undefined) {
          startAt = 0;
        }

        if (startAt < 0) {
          startAt += len;
        }

        for (var idx = startAt; idx < len; idx++) {
          var currentObj = objectAt(this, idx);

          // SameValueZero comparison (NaN !== NaN)
          if (obj === currentObj || obj !== obj && currentObj !== currentObj) {
            return true;
          }
        }

        return false;
      }
    });
  }

  exports.default = ArrayMixin;
});
// ES6TODO: Ember.A

/**
  __Required.__ You must implement this method to apply this mixin.
   Your array must support the `length` property. Your replace methods should
  set this property whenever it changes.
   @property {Number} length
  @public
*/

/**
  Returns the object at the given `index`. If the given `index` is negative
  or is greater or equal than the array length, returns `undefined`.
   This is one of the primitives you must implement to support `Ember.Array`.
  If your object supports retrieving the value of an array item using `get()`
  (i.e. `myArray.get(0)`), then you do not need to implement this method
  yourself.
   ```javascript
  let arr = ['a', 'b', 'c', 'd'];
   arr.objectAt(0);   // 'a'
  arr.objectAt(3);   // 'd'
  arr.objectAt(-1);  // undefined
  arr.objectAt(4);   // undefined
  arr.objectAt(5);   // undefined
  ```
   @method objectAt
  @param {Number} idx The index of the item to return.
  @return {*} item at index or undefined
  @public
*/

/**
  This returns the objects at the specified indexes, using `objectAt`.
   ```javascript
  let arr = ['a', 'b', 'c', 'd'];
   arr.objectsAt([0, 1, 2]);  // ['a', 'b', 'c']
  arr.objectsAt([2, 3, 4]);  // ['c', 'd', undefined]
  ```
   @method objectsAt
  @param {Array} indexes An array of indexes of items to return.
  @return {Array}
  @public
 */

// overrides Ember.Enumerable version

/**
  This is the handler for the special array content property. If you get
  this property, it will return this. If you set this property to a new
  array, it will replace the current content.
   This property overrides the default property defined in `Ember.Enumerable`.
   @property []
  @return this
  @public
*/

// optimized version from Enumerable

// Add any extra methods to Ember.Array that are native to the built-in Array.
/**
  Returns a new array that is a slice of the receiver. This implementation
  uses the observable array methods to retrieve the objects for the new
  slice.
   ```javascript
  let arr = ['red', 'green', 'blue'];
   arr.slice(0);       // ['red', 'green', 'blue']
  arr.slice(0, 2);    // ['red', 'green']
  arr.slice(1, 100);  // ['green', 'blue']
  ```
   @method slice
  @param {Number} beginIndex (Optional) index to begin slicing from.
  @param {Number} endIndex (Optional) index to end the slice at (but not included).
  @return {Array} New array with specified slice
  @public
*/

/**
  Returns the index of the given object's first occurrence.
  If no `startAt` argument is given, the starting location to
  search is 0. If it's negative, will count backward from
  the end of the array. Returns -1 if no match is found.
   ```javascript
  let arr = ['a', 'b', 'c', 'd', 'a'];
   arr.indexOf('a');       //  0
  arr.indexOf('z');       // -1
  arr.indexOf('a', 2);    //  4
  arr.indexOf('a', -1);   //  4
  arr.indexOf('b', 3);    // -1
  arr.indexOf('a', 100);  // -1
  ```
   @method indexOf
  @param {Object} object the item to search for
  @param {Number} startAt optional starting location to search, default 0
  @return {Number} index or -1 if not found
  @public
*/

/**
  Returns the index of the given object's last occurrence.
  If no `startAt` argument is given, the search starts from
  the last position. If it's negative, will count backward
  from the end of the array. Returns -1 if no match is found.
   ```javascript
  let arr = ['a', 'b', 'c', 'd', 'a'];
   arr.lastIndexOf('a');       //  4
  arr.lastIndexOf('z');       // -1
  arr.lastIndexOf('a', 2);    //  0
  arr.lastIndexOf('a', -1);   //  4
  arr.lastIndexOf('b', 3);    //  1
  arr.lastIndexOf('a', 100);  //  4
  ```
   @method lastIndexOf
  @param {Object} object the item to search for
  @param {Number} startAt optional starting location to search, default 0
  @return {Number} index or -1 if not found
  @public
*/

// ..........................................................
// ARRAY OBSERVERS
//

/**
  Adds an array observer to the receiving array. The array observer object
  normally must implement two methods:
   * `arrayWillChange(observedObj, start, removeCount, addCount)` - This method will be
    called just before the array is modified.
  * `arrayDidChange(observedObj, start, removeCount, addCount)` - This method will be
    called just after the array is modified.
   Both callbacks will be passed the observed object, starting index of the
  change as well as a count of the items to be removed and added. You can use
  these callbacks to optionally inspect the array during the change, clear
  caches, or do any other bookkeeping necessary.
   In addition to passing a target, you can also include an options hash
  which you can use to override the method names that will be invoked on the
  target.
   @method addArrayObserver
  @param {Object} target The observer object.
  @param {Object} opts Optional hash of configuration options including
    `willChange` and `didChange` option.
  @return {Ember.Array} receiver
  @public
*/

/**
  Removes an array observer from the object if the observer is current
  registered. Calling this method multiple times with the same object will
  have no effect.
   @method removeArrayObserver
  @param {Object} target The object observing the array.
  @param {Object} opts Optional hash of configuration options including
    `willChange` and `didChange` option.
  @return {Ember.Array} receiver
  @public
*/

/**
  Becomes true whenever the array currently has observers watching changes
  on the array.
   @property {Boolean} hasArrayObservers
  @public
*/

/**
  If you are implementing an object that supports `Ember.Array`, call this
  method just before the array content changes to notify any observers and
  invalidate any related properties. Pass the starting index of the change
  as well as a delta of the amounts to change.
   @method arrayContentWillChange
  @param {Number} startIdx The starting index in the array that will change.
  @param {Number} removeAmt The number of items that will be removed. If you
    pass `null` assumes 0
  @param {Number} addAmt The number of items that will be added. If you
    pass `null` assumes 0.
  @return {Ember.Array} receiver
  @public
*/

/**
  If you are implementing an object that supports `Ember.Array`, call this
  method just after the array content changes to notify any observers and
  invalidate any related properties. Pass the starting index of the change
  as well as a delta of the amounts to change.
   @method arrayContentDidChange
  @param {Number} startIdx The starting index in the array that did change.
  @param {Number} removeAmt The number of items that were removed. If you
    pass `null` assumes 0
  @param {Number} addAmt The number of items that were added. If you
    pass `null` assumes 0.
  @return {Ember.Array} receiver
  @public
*/

/**
  Returns a special object that can be used to observe individual properties
  on the array. Just get an equivalent property on this object and it will
  return an enumerable that maps automatically to the named key on the
  member objects.
   `@each` should only be used in a non-terminal context. Example:
   ```javascript
  myMethod: computed('posts.@each.author', function(){
    ...
  });
  ```
   If you merely want to watch for the array being changed, like an object being
  replaced, added or removed, use `[]` instead of `@each`.
   ```javascript
  myMethod: computed('posts.[]', function(){
    ...
  });
  ```
   @property @each
  @public
*/
enifed('ember-runtime/mixins/comparable', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  /**
  @module ember
  @submodule ember-runtime
  */

  /**
    Implements some standard methods for comparing objects. Add this mixin to
    any class you create that can compare its instances.
  
    You should implement the `compare()` method.
  
    @class Comparable
    @namespace Ember
    @since Ember 0.9
    @private
  */
  exports.default = _emberMetal.Mixin.create({

    /**
      __Required.__ You must implement this method to apply this mixin.
       Override to return the result of the comparison of the two parameters. The
      compare method should return:
       - `-1` if `a < b`
      - `0` if `a == b`
      - `1` if `a > b`
       Default implementation raises an exception.
       @method compare
      @param a {Object} the first object to compare
      @param b {Object} the second object to compare
      @return {Number} the result of the comparison
      @private
    */
    compare: null
  });
});
enifed('ember-runtime/mixins/container_proxy', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  /**
  @module ember
  @submodule ember-runtime
  */
  'use strict';

  /**
    ContainerProxyMixin is used to provide public access to specific
    container functionality.
  
    @class ContainerProxyMixin
    @private
  */
  exports.default = _emberMetal.Mixin.create({
    /**
     The container stores state.
      @private
     @property {Ember.Container} __container__
     */
    __container__: null,

    /**
     Returns an object that can be used to provide an owner to a
     manually created instance.
      Example:
      ```
     let owner = Ember.getOwner(this);
      User.create(
       owner.ownerInjection(),
       { username: 'rwjblue' }
     )
     ```
      @public
     @method ownerInjection
     @return {Object}
    */
    ownerInjection: function () {
      return this.__container__.ownerInjection();
    },

    /**
     Given a fullName return a corresponding instance.
      The default behaviour is for lookup to return a singleton instance.
     The singleton is scoped to the container, allowing multiple containers
     to all have their own locally scoped singletons.
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      registry.register('api:twitter', Twitter);
      let twitter = container.lookup('api:twitter');
      twitter instanceof Twitter; // => true
      // by default the container will return singletons
     let twitter2 = container.lookup('api:twitter');
     twitter2 instanceof Twitter; // => true
      twitter === twitter2; //=> true
     ```
      If singletons are not wanted an optional flag can be provided at lookup.
      ```javascript
     let registry = new Registry();
     let container = registry.container();
      registry.register('api:twitter', Twitter);
      let twitter = container.lookup('api:twitter', { singleton: false });
     let twitter2 = container.lookup('api:twitter', { singleton: false });
      twitter === twitter2; //=> false
     ```
      @public
     @method lookup
     @param {String} fullName
     @param {Object} options
     @return {any}
     */
    lookup: function (fullName, options) {
      return this.__container__.lookup(fullName, options);
    },

    /**
     Given a fullName return the corresponding factory.
      @private
     @method _lookupFactory
     @param {String} fullName
     @return {any}
     */
    _lookupFactory: function (fullName, options) {
      return this.__container__.lookupFactory(fullName, options);
    },

    /**
     Given a name and a source path, resolve the fullName
      @private
     @method _resolveLocalLookupName
     @param {String} fullName
     @param {String} source
     @return {String}
     */
    _resolveLocalLookupName: function (name, source) {
      return this.__container__.registry.expandLocalLookup('component:' + name, {
        source: source
      });
    },

    /**
     @private
     */
    willDestroy: function () {
      this._super.apply(this, arguments);

      if (this.__container__) {
        _emberMetal.run(this.__container__, 'destroy');
      }
    }
  });
});
enifed('ember-runtime/mixins/controller', ['exports', 'ember-metal', 'ember-runtime/mixins/action_handler', 'ember-runtime/mixins/controller_content_model_alias_deprecation'], function (exports, _emberMetal, _emberRuntimeMixinsAction_handler, _emberRuntimeMixinsController_content_model_alias_deprecation) {
  'use strict';

  /**
    @class ControllerMixin
    @namespace Ember
    @uses Ember.ActionHandler
    @private
  */
  exports.default = _emberMetal.Mixin.create(_emberRuntimeMixinsAction_handler.default, _emberRuntimeMixinsController_content_model_alias_deprecation.default, {
    /* ducktype as a controller */
    isController: true,

    /**
      The object to which actions from the view should be sent.
       For example, when a Handlebars template uses the `{{action}}` helper,
      it will attempt to send the action to the view's controller's `target`.
       By default, the value of the target property is set to the router, and
      is injected when a controller is instantiated. This injection is applied
      as part of the application's initialization process. In most cases the
      `target` property will automatically be set to the logical consumer of
      actions for the controller.
       @property target
      @default null
      @public
    */
    target: null,

    store: null,

    /**
      The controller's current model. When retrieving or modifying a controller's
      model, this property should be used instead of the `content` property.
       @property model
      @public
     */
    model: null,

    /**
      @private
    */
    content: _emberMetal.alias('model')

  });
});
enifed('ember-runtime/mixins/controller_content_model_alias_deprecation', ['exports', 'ember-metal'], function (exports, _emberMetal) {
  'use strict';

  /*
    The ControllerContentModelAliasDeprecation mixin is used to provide a useful
    deprecation warning when specifying `content` directly on a `Ember.Controller`
    (without also specifying `model`).
  
    Ember versions prior to 1.7 used `model` as an alias of `content`, but due to
    much confusion this alias was reversed (so `content` is now an alias of `model).
  
    This change reduces many caveats with model/content, and also sets a
    simple ground rule: Never set a controllers content, rather always set
    its model and ember will do the right thing.
  
    Used internally by Ember in `Ember.Controller`.
  */
  exports.default = _emberMetal.Mixin.create({
    /**
      @private
       Moves `content` to `model`  at extend time if a `model` is not also specified.
       Note that this currently modifies the mixin themselves, which is technically
      dubious but is practically of little consequence. This may change in the
      future.
       @method willMergeMixin
      @since 1.4.0
    */
    willMergeMixin: function (props) {
      // Calling super is only OK here since we KNOW that
      // there is another Mixin loaded first.
      this._super.apply(this, arguments);

      var modelSpecified = !!props.model;

      if (props.content && !modelSpecified) {
        props.model = props.content;
        delete props['content'];
      }
    }
  });
});
enifed('ember-runtime/mixins/copyable', ['exports', 'ember-metal', 'ember-runtime/mixins/freezable'], function (exports, _emberMetal, _emberRuntimeMixinsFreezable) {
  /**
  @module ember
  @submodule ember-runtime
  */

  'use strict';

  /**
    Implements some standard methods for copying an object. Add this mixin to
    any object you create that can create a copy of itself. This mixin is
    added automatically to the built-in array.
  
    You should generally implement the `copy()` method to return a copy of the
    receiver.
  
    Note that `frozenCopy()` will only work if you also implement
    `Ember.Freezable`.
  
    @class Copyable
    @namespace Ember
    @since Ember 0.9
    @private
  */
  exports.default = _emberMetal.Mixin.create({
    /**
      __Required.__ You must implement this method to apply this mixin.
       Override to return a copy of the receiver. Default implementation raises
      an exception.
       @method copy
      @param {Boolean} deep if `true`, a deep copy of the object should be made
      @return {Object} copy of receiver
      @private
    */
    copy: null,

    /**
      If the object implements `Ember.Freezable`, then this will return a new
      copy if the object is not frozen and the receiver if the object is frozen.
       Raises an exception if you try to call this method on a object that does
      not support freezing.
       You should use this method whenever you want a copy of a freezable object
      since a freezable object can simply return itself without actually
      consuming more memory.
       @method frozenCopy
      @return {Object} copy of receiver or receiver
      @deprecated Use `Object.freeze` instead.
      @private
    */
    frozenCopy: function () {
      if (_emberRuntimeMixinsFreezable.Freezable && _emberRuntimeMixinsFreezable.Freezable.detect(this)) {
        return _emberMetal.get(this, 'isFrozen') ? this : this.copy().freeze();
      } else {
        throw new _emberMetal.Error(this + ' does not support freezing');
      }
    }
  });
});
enifed('ember-runtime/mixins/enumerable', ['exports', 'ember-utils', 'ember-metal', 'ember-runtime/compare', 'require'], function (exports, _emberUtils, _emberMetal, _emberRuntimeCompare, _require) {
  /**
  @module ember
  @submodule ember-runtime
  */

  // ..........................................................
  // HELPERS
  //

  'use strict';

  var _emberA = undefined;

  function emberA() {
    return (_emberA || (_emberA = _require.default('ember-runtime/system/native_array').A))();
  }

  var contexts = [];

  function popCtx() {
    return contexts.length === 0 ? {} : contexts.pop();
  }

  function pushCtx(ctx) {
    contexts.push(ctx);
    return null;
  }

  function iter(key, value) {
    var valueProvided = arguments.length === 2;

    function i(item) {
      var cur = _emberMetal.get(item, key);
      return valueProvided ? value === cur : !!cur;
    }

    return i;
  }

  /**
    This mixin defines the common interface implemented by enumerable objects
    in Ember. Most of these methods follow the standard Array iteration
    API defined up to JavaScript 1.8 (excluding language-specific features that
    cannot be emulated in older versions of JavaScript).
  
    This mixin is applied automatically to the Array class on page load, so you
    can use any of these methods on simple arrays. If Array already implements
    one of these methods, the mixin will not override them.
  
    ## Writing Your Own Enumerable
  
    To make your own custom class enumerable, you need two items:
  
    1. You must have a length property. This property should change whenever
       the number of items in your enumerable object changes. If you use this
       with an `Ember.Object` subclass, you should be sure to change the length
       property using `set().`
  
    2. You must implement `nextObject().` See documentation.
  
    Once you have these two methods implemented, apply the `Ember.Enumerable` mixin
    to your class and you will be able to enumerate the contents of your object
    like any other collection.
  
    ## Using Ember Enumeration with Other Libraries
  
    Many other libraries provide some kind of iterator or enumeration like
    facility. This is often where the most common API conflicts occur.
    Ember's API is designed to be as friendly as possible with other
    libraries by implementing only methods that mostly correspond to the
    JavaScript 1.8 API.
  
    @class Enumerable
    @namespace Ember
    @since Ember 0.9
    @private
  */
  var Enumerable = _emberMetal.Mixin.create({

    /**
      __Required.__ You must implement this method to apply this mixin.
       Implement this method to make your class enumerable.
       This method will be called repeatedly during enumeration. The index value
      will always begin with 0 and increment monotonically. You don't have to
      rely on the index value to determine what object to return, but you should
      always check the value and start from the beginning when you see the
      requested index is 0.
       The `previousObject` is the object that was returned from the last call
      to `nextObject` for the current iteration. This is a useful way to
      manage iteration if you are tracing a linked list, for example.
       Finally the context parameter will always contain a hash you can use as
      a "scratchpad" to maintain any other state you need in order to iterate
      properly. The context object is reused and is not reset between
      iterations so make sure you setup the context with a fresh state whenever
      the index parameter is 0.
       Generally iterators will continue to call `nextObject` until the index
      reaches the current length-1. If you run out of data before this
      time for some reason, you should simply return undefined.
       The default implementation of this method simply looks up the index.
      This works great on any Array-like objects.
       @method nextObject
      @param {Number} index the current index of the iteration
      @param {Object} previousObject the value returned by the last call to
        `nextObject`.
      @param {Object} context a context object you can use to maintain state.
      @return {Object} the next object in the iteration or undefined
      @private
    */
    nextObject: null,

    /**
      Helper method returns the first object from a collection. This is usually
      used by bindings and other parts of the framework to extract a single
      object if the enumerable contains only one item.
       If you override this method, you should implement it so that it will
      always return the same value each time it is called. If your enumerable
      contains only one object, this method should always return that object.
      If your enumerable is empty, this method should return `undefined`.
       ```javascript
      let arr = ['a', 'b', 'c'];
      arr.get('firstObject');  // 'a'
       let arr = [];
      arr.get('firstObject');  // undefined
      ```
       @property firstObject
      @return {Object} the object or undefined
      @readOnly
      @public
    */
    firstObject: _emberMetal.computed('[]', function () {
      if (_emberMetal.get(this, 'length') === 0) {
        return undefined;
      }

      // handle generic enumerables
      var context = popCtx();
      var ret = this.nextObject(0, null, context);

      pushCtx(context);

      return ret;
    }).readOnly(),

    /**
      Helper method returns the last object from a collection. If your enumerable
      contains only one object, this method should always return that object.
      If your enumerable is empty, this method should return `undefined`.
       ```javascript
      let arr = ['a', 'b', 'c'];
      arr.get('lastObject');  // 'c'
       let arr = [];
      arr.get('lastObject');  // undefined
      ```
       @property lastObject
      @return {Object} the last object or undefined
      @readOnly
      @public
    */
    lastObject: _emberMetal.computed('[]', function () {
      var len = _emberMetal.get(this, 'length');

      if (len === 0) {
        return undefined;
      }

      var context = popCtx();
      var idx = 0;
      var last = null;
      var cur = undefined;

      do {
        last = cur;
        cur = this.nextObject(idx++, last, context);
      } while (cur !== undefined);

      pushCtx(context);

      return last;
    }).readOnly(),

    /**
      Returns `true` if the passed object can be found in the receiver. The
      default version will iterate through the enumerable until the object
      is found. You may want to override this with a more efficient version.
       ```javascript
      let arr = ['a', 'b', 'c'];
       arr.contains('a'); // true
      arr.contains('z'); // false
      ```
       @method contains
      @deprecated Use `Enumerable#includes` instead. See http://emberjs.com/deprecations/v2.x#toc_enumerable-contains
      @param {Object} obj The object to search for.
      @return {Boolean} `true` if object is found in enumerable.
      @public
    */
    contains: function (obj) {
      if (true) {}

      var found = this.find(function (item) {
        return item === obj;
      });

      return found !== undefined;
    },

    /**
      Iterates through the enumerable, calling the passed function on each
      item. This method corresponds to the `forEach()` method defined in
      JavaScript 1.6.
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(item, index, enumerable);
      ```
       - `item` is the current item in the iteration.
      - `index` is the current index in the iteration.
      - `enumerable` is the enumerable object itself.
       Note that in addition to a callback, you can also pass an optional target
      object that will be set as `this` on the context. This is a good way
      to give your iterator function access to the current object.
       @method forEach
      @param {Function} callback The callback to execute
      @param {Object} [target] The target object to use
      @return {Object} receiver
      @public
    */
    forEach: function (callback, target) {
      if (typeof callback !== 'function') {
        throw new TypeError();
      }

      var context = popCtx();
      var len = _emberMetal.get(this, 'length');
      var last = null;

      if (target === undefined) {
        target = null;
      }

      for (var idx = 0; idx < len; idx++) {
        var next = this.nextObject(idx, last, context);
        callback.call(target, next, idx, this);
        last = next;
      }

      last = null;
      context = pushCtx(context);

      return this;
    },

    /**
      Alias for `mapBy`
       @method getEach
      @param {String} key name of the property
      @return {Array} The mapped array.
      @public
    */
    getEach: _emberMetal.aliasMethod('mapBy'),

    /**
      Sets the value on the named property for each member. This is more
      ergonomic than using other methods defined on this helper. If the object
      implements Ember.Observable, the value will be changed to `set(),` otherwise
      it will be set directly. `null` objects are skipped.
       @method setEach
      @param {String} key The key to set
      @param {Object} value The object to set
      @return {Object} receiver
      @public
    */
    setEach: function (key, value) {
      return this.forEach(function (item) {
        return _emberMetal.set(item, key, value);
      });
    },

    /**
      Maps all of the items in the enumeration to another value, returning
      a new array. This method corresponds to `map()` defined in JavaScript 1.6.
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(item, index, enumerable);
      ```
       - `item` is the current item in the iteration.
      - `index` is the current index in the iteration.
      - `enumerable` is the enumerable object itself.
       It should return the mapped value.
       Note that in addition to a callback, you can also pass an optional target
      object that will be set as `this` on the context. This is a good way
      to give your iterator function access to the current object.
       @method map
      @param {Function} callback The callback to execute
      @param {Object} [target] The target object to use
      @return {Array} The mapped array.
      @public
    */
    map: function (callback, target) {
      var ret = emberA();

      this.forEach(function (x, idx, i) {
        return ret[idx] = callback.call(target, x, idx, i);
      });

      return ret;
    },

    /**
      Similar to map, this specialized function returns the value of the named
      property on all items in the enumeration.
       @method mapBy
      @param {String} key name of the property
      @return {Array} The mapped array.
      @public
    */
    mapBy: function (key) {
      return this.map(function (next) {
        return _emberMetal.get(next, key);
      });
    },

    /**
      Returns an array with all of the items in the enumeration that the passed
      function returns true for. This method corresponds to `filter()` defined in
      JavaScript 1.6.
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(item, index, enumerable);
      ```
       - `item` is the current item in the iteration.
      - `index` is the current index in the iteration.
      - `enumerable` is the enumerable object itself.
       It should return `true` to include the item in the results, `false`
      otherwise.
       Note that in addition to a callback, you can also pass an optional target
      object that will be set as `this` on the context. This is a good way
      to give your iterator function access to the current object.
       @method filter
      @param {Function} callback The callback to execute
      @param {Object} [target] The target object to use
      @return {Array} A filtered array.
      @public
    */
    filter: function (callback, target) {
      var ret = emberA();

      this.forEach(function (x, idx, i) {
        if (callback.call(target, x, idx, i)) {
          ret.push(x);
        }
      });

      return ret;
    },

    /**
      Returns an array with all of the items in the enumeration where the passed
      function returns false. This method is the inverse of filter().
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(item, index, enumerable);
      ```
       - *item* is the current item in the iteration.
      - *index* is the current index in the iteration
      - *enumerable* is the enumerable object itself.
       It should return a falsey value to include the item in the results.
       Note that in addition to a callback, you can also pass an optional target
      object that will be set as "this" on the context. This is a good way
      to give your iterator function access to the current object.
       @method reject
      @param {Function} callback The callback to execute
      @param {Object} [target] The target object to use
      @return {Array} A rejected array.
      @public
    */
    reject: function (callback, target) {
      return this.filter(function () {
        return !callback.apply(target, arguments);
      });
    },

    /**
      Returns an array with just the items with the matched property. You
      can pass an optional second argument with the target value. Otherwise
      this will match any property that evaluates to `true`.
       @method filterBy
      @param {String} key the property to test
      @param {*} [value] optional value to test against.
      @return {Array} filtered array
      @public
    */
    filterBy: function (key, value) {
      return this.filter(iter.apply(this, arguments));
    },

    /**
      Returns an array with the items that do not have truthy values for
      key.  You can pass an optional second argument with the target value.  Otherwise
      this will match any property that evaluates to false.
       @method rejectBy
      @param {String} key the property to test
      @param {String} [value] optional value to test against.
      @return {Array} rejected array
      @public
    */
    rejectBy: function (key, value) {
      var exactValue = function (item) {
        return _emberMetal.get(item, key) === value;
      };
      var hasValue = function (item) {
        return !!_emberMetal.get(item, key);
      };
      var use = arguments.length === 2 ? exactValue : hasValue;

      return this.reject(use);
    },

    /**
      Returns the first item in the array for which the callback returns true.
      This method works similar to the `filter()` method defined in JavaScript 1.6
      except that it will stop working on the array once a match is found.
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(item, index, enumerable);
      ```
       - `item` is the current item in the iteration.
      - `index` is the current index in the iteration.
      - `enumerable` is the enumerable object itself.
       It should return the `true` to include the item in the results, `false`
      otherwise.
       Note that in addition to a callback, you can also pass an optional target
      object that will be set as `this` on the context. This is a good way
      to give your iterator function access to the current object.
       @method find
      @param {Function} callback The callback to execute
      @param {Object} [target] The target object to use
      @return {Object} Found item or `undefined`.
      @public
    */
    find: function (callback, target) {
      var len = _emberMetal.get(this, 'length');

      if (target === undefined) {
        target = null;
      }

      var context = popCtx();
      var found = false;
      var last = null;
      var next = undefined,
          ret = undefined;

      for (var idx = 0; idx < len && !found; idx++) {
        next = this.nextObject(idx, last, context);

        found = callback.call(target, next, idx, this);
        if (found) {
          ret = next;
        }

        last = next;
      }

      next = last = null;
      context = pushCtx(context);

      return ret;
    },

    /**
      Returns the first item with a property matching the passed value. You
      can pass an optional second argument with the target value. Otherwise
      this will match any property that evaluates to `true`.
       This method works much like the more generic `find()` method.
       @method findBy
      @param {String} key the property to test
      @param {String} [value] optional value to test against.
      @return {Object} found item or `undefined`
      @public
    */
    findBy: function (key, value) {
      return this.find(iter.apply(this, arguments));
    },

    /**
      Returns `true` if the passed function returns true for every item in the
      enumeration. This corresponds with the `every()` method in JavaScript 1.6.
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(item, index, enumerable);
      ```
       - `item` is the current item in the iteration.
      - `index` is the current index in the iteration.
      - `enumerable` is the enumerable object itself.
       It should return the `true` or `false`.
       Note that in addition to a callback, you can also pass an optional target
      object that will be set as `this` on the context. This is a good way
      to give your iterator function access to the current object.
       Example Usage:
       ```javascript
      if (people.every(isEngineer)) {
        Paychecks.addBigBonus();
      }
      ```
       @method every
      @param {Function} callback The callback to execute
      @param {Object} [target] The target object to use
      @return {Boolean}
      @public
    */
    every: function (callback, target) {
      return !this.find(function (x, idx, i) {
        return !callback.call(target, x, idx, i);
      });
    },

    /**
      Returns `true` if the passed property resolves to the value of the second
      argument for all items in the enumerable. This method is often simpler/faster
      than using a callback.
       @method isEvery
      @param {String} key the property to test
      @param {String} [value] optional value to test against. Defaults to `true`
      @return {Boolean}
      @since 1.3.0
      @public
    */
    isEvery: function (key, value) {
      return this.every(iter.apply(this, arguments));
    },

    /**
      Returns `true` if the passed function returns true for any item in the
      enumeration. This corresponds with the `some()` method in JavaScript 1.6.
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(item, index, enumerable);
      ```
       - `item` is the current item in the iteration.
      - `index` is the current index in the iteration.
      - `enumerable` is the enumerable object itself.
       It should return the `true` to include the item in the results, `false`
      otherwise.
       Note that in addition to a callback, you can also pass an optional target
      object that will be set as `this` on the context. This is a good way
      to give your iterator function access to the current object.
       Usage Example:
       ```javascript
      if (people.any(isManager)) {
        Paychecks.addBiggerBonus();
      }
      ```
       @method any
      @param {Function} callback The callback to execute
      @param {Object} [target] The target object to use
      @return {Boolean} `true` if the passed function returns `true` for any item
      @public
    */
    any: function (callback, target) {
      var len = _emberMetal.get(this, 'length');
      var context = popCtx();
      var found = false;
      var last = null;
      var next = undefined;

      if (target === undefined) {
        target = null;
      }

      for (var idx = 0; idx < len && !found; idx++) {
        next = this.nextObject(idx, last, context);
        found = callback.call(target, next, idx, this);
        last = next;
      }

      next = last = null;
      context = pushCtx(context);
      return found;
    },

    /**
      Returns `true` if the passed property resolves to the value of the second
      argument for any item in the enumerable. This method is often simpler/faster
      than using a callback.
       @method isAny
      @param {String} key the property to test
      @param {String} [value] optional value to test against. Defaults to `true`
      @return {Boolean}
      @since 1.3.0
      @public
    */
    isAny: function (key, value) {
      return this.any(iter.apply(this, arguments));
    },

    /**
      This will combine the values of the enumerator into a single value. It
      is a useful way to collect a summary value from an enumeration. This
      corresponds to the `reduce()` method defined in JavaScript 1.8.
       The callback method you provide should have the following signature (all
      parameters are optional):
       ```javascript
      function(previousValue, item, index, enumerable);
      ```
       - `previousValue` is the value returned by the last call to the iterator.
      - `item` is the current item in the iteration.
      - `index` is the current index in the iteration.
      - `enumerable` is the enumerable objec