import { on } from '@ember/object/evented';
import Component from '@ember/component';
import Growable from "liquid-fire/growable";
import { measure } from "./liquid-measured";
import layout from "liquid-fire/templates/components/liquid-container";

export default Component.extend(Growable, {
  layout,
  classNames: ['liquid-container'],

  lockSize: function(elt, want) {
    elt.outerWidth(want.width);
    elt.outerHeight(want.height);
  },

  unlockSize: function() {
    let doUnlock = () => {
      this.updateAnimatingClass(false);
      let elt = this.$();
      if (elt) {
        elt.css({width: '', height: ''});
      }
    };
    if (this._scaling) {
      this._scaling.then(doUnlock);
    } else {
      doUnlock();
    }
  },

  // We're doing this manually instead of via classNameBindings
  // because it depends on upward-data-flow, which generates warnings
  // under Glimmer.
  updateAnimatingClass(on){
    if (this.isDestroyed) {
      return;
    }
    if (on) {
      this.$().addClass('liquid-animating');
    } else {
      this.$().removeClass('liquid-animating');
    }
  },

  startMonitoringSize: on('didInsertElement', function() {
    this._wasInserted = true;
  }),

  actions: {

    willTransition: function(versions) {
      if (!this._wasInserted) {
        return;
      }

      // Remember our own size before anything changes
      let elt = this.$();
      this._cachedSize = measure(elt);

      // And make any children absolutely positioned with fixed sizes.
      for (let i = 0; i < versions.length; i++) {
        goAbsolute(versions[i]);
      }

    },

    afterChildInsertion: function(versions) {
      let elt = this.$();
      let enableGrowth = this.get('enableGrowth') !== false;

      // Measure children
      let sizes = [];
      for (let i = 0; i < versions.length; i++) {
        if (versions[i].view) {
          sizes[i] = measure(versions[i].view.$());
        }
      }

      // Measure ourself again to see how big the new children make
      // us.
      let want = measure(elt);
      let have = this._cachedSize || want;

      // Make ourself absolute
      if (enableGrowth) {
        this.lockSize(elt, have);
      } else {
        this.lockSize(elt, {
          height: Math.max(want.height, have.height),
          width: Math.max(want.width, have.width)
        });
      }

      // Apply '.liquid-animating' to liquid-container allowing
      // any customizable CSS control while an animating is occuring
      this.updateAnimatingClass(true);

      // Make the children absolute and fixed size.
      for (let i = 0; i < versions.length; i++) {
        goAbsolute(versions[i], sizes[i]);
      }

      // Kick off our growth animation
      if (enableGrowth) {
        this._scaling = this.animateGrowth(elt, have, want);
      }
    },

    afterTransition: function(versions) {
      for (let i = 0; i < versions.length; i++) {
        goStatic(versions[i]);
      }
      this.unlockSize();
    }
  }
});

function goAbsolute(version, size) {
  if (!version.view) {
    return;
  }
  let elt = version.view.$();
  let pos = elt.position();
  if (!size) {
    size = measure(elt);
  }
  elt.outerWidth(size.width);
  elt.outerHeight(size.height);
  elt.css({
    position: 'absolute',
    top: pos.top,
    left: pos.left
  });
}

function goStatic(version) {
  if (version.view && !version.view.isDestroyed) {
    version.view.$().css({width: '', height: '', position: ''});
  }
}
