var Promise = require( 'bluebird' )
  , path    = require( 'path' )
  , async   = require( 'async' )
  , spawn   = require( 'win-spawn' )
  , fs      = Promise.promisifyAll( require( 'fs' ) )
  , utils   = require( path.join( __dirname, 'utils' ) );

Promise.longStackTraces( );

/**
 * Helper function for running clever-install if there are modules to install
 *
 * @param  {Object} projectFolder location returned by locations( )
 * @param  {String[]} modules
 * @return {Promise}
 * @api public
 */

exports.setupModules = function ( projectFolder, modules ) {
  var def   = Promise.defer( )
    , proc  = spawn( path.join( __dirname, '..', 'bin', 'clever-install' ), modules, { cwd: projectFolder, stdio: 'inherit' } );

  proc.on( 'close', function ( ) {
    def.resolve( );
  } );

  return def.promise;
}

/**
 * Finds a frontend and backend seed and determines
 * whether or not we can use Bower or NPM
 *
 * @return {Promise}
 * @api public
 */

exports.useUtils = function ( ) {
  var def = Promise.defer( );

  locations( )
  .then( function ( locations ) {
    var useNPM = false
      , useBower = false;

    if (locations.length > 1) {
      useNPM    = true;
      useBower  = true;
    }
    else if (locations.length < 1) {
      lib.utils.fail( 'Couldn\'t find any CleverStack seeds within ' + process.cwd( ) );
    }
    else {
      useNPM    = locations[ 0 ].type === "backend";
      useBower  = locations[ 0 ].type === "frontend";
    }

    locations = locations.filter( function ( location ) {
      return ( useNPM && useBower ) || ( useNPM && location.type === "backend" ) || ( useBower && location.type === "frontend" );
    } );

    def.resolve( [ locations, useNPM, useBower ] );
  } );

  return def.promise;
}

/**
 * Installs CleverStack modules listed within the bundleDependencies array
 *
 * @param  {String} project
 * @param  {Array} deps
 * @return {Promise}
 * @api public
 */

module.exports.installBundledModules = function ( project, deps ) {
  deps = Array.isArray( deps ) ? deps : [deps];

  if (deps.length < 1) {
    return new Promise( function ( res ) {
      res( );
    } );
  }

  utils.info( 'Installing modules: ' + deps.join( ' ' ) );

  var def   = Promise.defer( )
    , proc  = spawn( path.join( __dirname, '..', 'bin', 'clever-install' ), deps, { cwd: project, stdio: 'inherit' } );

  proc.on( 'close', function ( code ) {
    if (code !== 0) {
      return def.reject( );
    }

    def.resolve( );
  } );

  return def.promise;
}

/**
 * We're essentially creating a small grunt utility to bypass
 * grunt functions... this is better than requiring grunt
 * entirely and trying to regex parse...
 *
 * @param  {String} filePath
 * @return {Object}
 * @api private
 */

function gruntUtility ( filePath ) {
  var file  = require( filePath )( )
    , tasks = [ ];


  if (Array.isArray( file ) && typeof file[file.length-1] === "function") {
    file[ file.length-1 ].call( file, {
      registerTask: function ( cmd ) {
        tasks.push( cmd );
      },
      loadNpmTasks: function ( ) { }
    } );
  }

  return tasks;
}

/**
 * Reads/parses Grunt tasks within pathSrc
 * @param  {String} pathSrc - Source to the Gruntfile
 * @param  {Boolean} [silence=true] - Silences errors returned from reading the Gruntfile
 * @return {Promise}
 * @api public
 */

module.exports.readGruntTasks = function ( pathSrc, silence ) {
  var def = Promise.defer( );

  if (typeof silence === "undefined") {
    silence = true;
  }

  findGruntFile( pathSrc )
  .then( function ( filePath ) {
    var tasks = gruntUtility( filePath );
    def.resolve( [ tasks, pathSrc ] );
  } )
  .catch( function ( err ) {
    if (err.match( /^Gruntfile within/ ) !== null && silence === true) {
      return def.resolve( [ [], pathSrc ] );
    }

    def.reject( err );
  } );

  return def.promise;
}

/**
 * Finds the correct Gruntfile name
 *
 * @param  {String} pathSrc - Directory path where the Gruntfile resides.
 * @return {Promise}
 * @api public
 */

var findGruntFile = exports.findGruntFile = function ( pathSrc ) {
  var def = Promise.defer( );

  async.detect( [
    path.join( pathSrc, 'Gruntfile.js' ),
    path.join( pathSrc, 'gruntfile.js' ),
    path.join( pathSrc, 'Grunt.js' ),
    path.join( pathSrc, 'grunt.js' )
  ], fs.exists, function ( filePath ) {
    if (typeof filePath === "undefined") {
      return def.reject( 'Gruntfile within ' + pathSrc + ' could not be found.' );
    }

    def.resolve( filePath );
  } );

  return def.promise;
}

/**
 * Finds seed locations within CWD
 * @return {Promise}
 * @api public
 */

var find = exports.find = function( ) {
  var def = Promise.defer( );

  utils.info( 'Scanning for seed locations...' );

  // As of right now... our best bet is to look forward once
  // and then backwards first match. The limitation of this
  // is that we're presuming the folders are called backend and frontend
  // todo: Enable support for any folder name

  fs.readdirAsync( process.cwd( ) )
  .then( function ( list ) {
    list = list.filter( function ( f ) {
      var stats = fs.statSync( path.resolve( path.join( process.cwd( ), f ) ) );
      return stats.isDirectory( ) && ['frontend', 'backend'].indexOf( f ) > -1;
    } )
    .map( function ( f ) {
      return path.resolve( path.join( process.cwd( ), f ) );
    } );

    if (list.length > 0) {
      return def.resolve( list );
    }

    var behind = process.cwd( ).split( path.sep ).reverse( )
      , found  = false;

    async.until(
      function ( ) { return behind.length < 1 || found === true; },
      function ( next ) {
        if (['frontend', 'backend'].indexOf( behind[0] ) > -1) {
          found = true;
        } else {
          behind.shift( );
        }
        next( );
      },
      function ( err ) {
        if (!!err) {
          return fn( err );
        }

        if (!behind.length) {
          return utils.fail( 'Couldn\'t find a seed directory within ' + process.cwd( ) );
        }


        def.resolve( [behind.reverse( ).join( path.sep )] );
      }
    );
  } );

  return def.promise;
}

/**
 * Finds seed locations and returns an object with common properties
 * @param  {Boolean} [useCWD=false] - Whether or not we should use process.cwd()
 * @return {Promise}
 * @api public
 */

var locations = module.exports.location = module.exports.locations = function ( useCWD, fn ) {
  var def = Promise.defer( );

  if (typeof useCWD === "function") {
    fn      = useCWD;
    useCWD  = false;
  }

  find( ).then( function ( foundProjects ) {
    var projects         = []
    var projectTemplates = {
      backend: {
        name: 'backend',
        modulePath: 'modules'
      },
      frontend: {
        name: 'frontend',
        modulePath: path.join( 'app', 'modules' )
      }
    };

    foundProjects.forEach( function ( proj ) {
      var projectName = proj.split( path.sep ).slice( -1 )[ 0 ];

      if (projectTemplates.hasOwnProperty( projectName ) ) {
        projectTemplates[projectName].moduleDir = proj;
        projectTemplates[projectName].useCWD    = useCWD;
        projects.push( projectTemplates[projectName] );
      }
    } );

    def.resolve( projects );
  } );

  return def.promise;
}

/**
 * Looks into packahe.json within the seed's module directory
 * and finds each (dev)dependency and installs individually
 * due to use using the --prefix flag for NPM
 * @param  {Object} project
 * @param  {[type]} modulePath
 * @return {Promise}
 * @api public
 */

module.exports.installModule = function ( project, modulePath ) {
  var def = Promise.defer( );
  var projectFolder = project.moduleDir;
  var moduleDir = path.join( project.moduleDir, project.modulePath );

  var jsonPath = path.resolve( path.join( modulePath, 'package.json' ) )
    , jsonFile = require( jsonPath )
    , deps     = [];

  jsonFile.dependencies     = jsonFile.dependencies     || {};
  jsonFile.devDependencies  = jsonFile.devDependencies  || {};

  Object.keys( jsonFile.dependencies ).forEach( function ( k ) {
    deps.push( k + '@' + jsonFile.dependencies[k] );
  } );

  Object.keys( jsonFile.devDependencies ).forEach( function ( k ) {
    deps.push( k + '@' + jsonFile.devDependencies[k] );
  } );

  // we need series here in order to prevent NPM from overwriting packages
  async.eachSeries( deps, function ( dep, _next ) {
    var err  = ''
      , proc = spawn( 'npm', [ 'install', dep, '--prefix', projectFolder ], { cwd: moduleDir } );

    proc.stderr.on('data', function ( data ) {
      err += data + '';
    } );

    proc.on( 'close', function ( code ) {
      if (code !== 0) {
        return _next( err );
      }

      _next( );
    } );
  },
  function ( err ) {
    if (!!err) {
      return def.reject( err );
    }

    utils.success( 'Finished installing NPM packages for ' + modulePath );

    var bowerPath = path.resolve( path.join( projectFolder, 'bower.json' ) )
      , _err      = '';

    // backend folder?
    if (!fs.existsSync( bowerPath ) ) {
      utils.warn( 'Running database migrations...' );

      var args = [ '--base', projectFolder, '--gruntfile', path.resolve( path.join( projectFolder, 'Gruntfile.js' ) ), 'db:exec' ];

      var env       = process.env;
      env.NODE_ENV  = 'local';

      var proc = spawn( 'grunt', args, { env: env, cwd: projectFolder } );

      proc.stderr.on('data', function ( data ) {
        _err += data + '';
      } );

      proc.on( 'close', function ( code ) {
        if (code !== 0) {
          return def.reject( _err );
        }

        def.resolve( true );
      } );
    } else {
      utils.info( 'Installing bower packages for ' + modulePath);

      var npmProc = spawn( 'npm', ['run', 'setup'], { cwd: modulePath } );

      npmProc.stderr.on('data', function ( data ) {
        _err += data + '';
      } );

      npmProc.on( 'close', function ( code ) {
        if (code !== 0) {
          return def.reject( _err );
        }

        utils.success( 'Finished installing bower packages.' );
        def.resolve( true );
      } );
    }
  } );

  return def.promise;
}
