'use strict';

const http = require('http');
const https = require('https');
// const fs = require('fs');
// const uuid = require('node-uuid');
// const AWS = require('aws-sdk');
const _ = require('lodash');
const request = require('request');
// const rmdir = require('rimraf');
const jdbc = require('./jdbc');
const xml2js = require('xml2js');
const responseHandler = require('./responseHandler');
const util = require('./util');

/** A request handler that handles all JSON and JSONP content. */
const jsonRequestHandler = (context, output) => {

  if(util.notNull(output.replBody) && util.notNull(output.bodyTokens)) {
    if(typeof(output.replBody) === 'string') {
      _.forEach(output.bodyTokens, (value, key) => {
        const wrappedKey = `{${key}}`;
        if(output.replBody.includes(wrappedKey)) {
          const re = new RegExp(wrappedKey, 'g');
          output.replBody = output.replBody.replace(re, JSON.stringify(value));
        }
      });
      // Definitely NOT a security concern /s
      const replBody = eval("(" + output.replBody + ")"); // jshint ignore:line
      output.body = replBody;
    } else {
      output.body = util.tokenRepl(output.body, output.bodyTokens);
    }
    output.runPreRequestHooks(output);
  }

  if(util.notNull(output.body)) {
    var keys = Object.keys(output.body);
    if(keys !== null && keys.length > 0) {
      output.headers['Content-Length'] = Buffer.byteLength(JSON.stringify(output.body), 'utf-8');
    }
  }

  /* Wrap the context to output request timing values. */
  const outCtx = {
    succeed: o => {
      const end = new Date().getTime();
      console.log("vendor success in", (end - start));
      return context.succeed(o);
    },
    fail: e => {
      const end = new Date().getTime();
      console.log("vendor fail in", (end - start));
      context.fail(e);
    },
    done: (e, o) => {
      const end = new Date().getTime();
      console.log("vendor done in", (end - start));
      context.done(e, o);
    }
  };

  const start = new Date().getTime();

  let conn;
  if (output.protocol.startsWith('https')) {

    // console.log("making request", {
    //   headers: output.headers,
    //   protocol: output.protocol,
    //   body: output.body,
    //   host: output.host,
    //   hostname: output.hostname,
    //   method: output.method,
    //   path: output.path
    // });

    conn = https.request(output);
  } else {
    conn = http.request(output);
  }

  conn.on('error', err => { context.fail(util.failure(400, err)); });
  conn.on('response', jsonBackHandler(outCtx, output));

  if (util.notNull(output.body)) { conn.write(JSON.stringify(output.body)); }

  conn.setTimeout(10000, () => {
    context.fail(util.failure(504, 'Request timed out'));
  });

  conn.end();
};

/** The response lookup for JSON and JSONP requests. */
const jsonBackHandler = (context, output) => res => {
  let rawBody = "";
  res.on('readable', () => {
    const data = res.read();
    if (typeof(data) !== 'undefined' && data !== null) {
      rawBody += data;
    }
  });

  res.on('end', () => {
    if (res.statusCode >= 200 && res.statusCode <= 207) {
      const handler = responseHandler.getResponseHandler(res.headers);
      handler(res, rawBody, context, output);
    } else {
      context.done(util.failure(res.statusCode, rawBody, res.statusMessage));
    }
  });

  res.on('error', err => context.fail({error: err.toString()}));
};

/** A request handler that handles all SOAP content. */
const soapRequestHandler = (context, output) => {
  const soapEnvelope = 'soapenv:Envelope';
  const soapBody = 'soapenv:Body';
  const soapHeader = 'soapenv:Header';

  xml2js.parseString(output.body, (err, result) => {
    const envelope = result[soapEnvelope];

    if(util.notNull(envelope)) {
      if(util.notNull(envelope[soapBody]) && typeof envelope[soapBody] === 'object') {
        envelope[soapBody] = util.tokenRepl(envelope[soapBody], output.bodyTokens);
      }

      if(util.notNull(envelope[soapHeader])) {
        envelope[soapHeader] = util.tokenRepl(envelope[soapHeader], output.bodyTokens);
      }

      result[soapEnvelope] = envelope;
    }

    const builder = new xml2js.Builder();
    output.body = builder.buildObject(result);

    const method = output.method ? output.method.toLowerCase() : 'post';
    const reqOptions = {
      url: output.findConfig('base.url') + output.pathname,
      headers: output.headers,
      body: output.body,
    };

    request[method](reqOptions, (err, res, body) => {
      if (res.statusCode >= 200 && res.statusCode <= 207) {
        const handler = responseHandler.getResponseHandler(res.headers, 'soap');
        handler(res, body, context, output);
      } else {
        context.fail(util.failure(res.statusCode, err));
      }
    });
  });

};

/** A request handler that handles all database content. */
const jdbcRequestHandler = (context, output) => {
  jdbc.send(context, output);
};

/**
 * A request handler than handles all form-data/multipart content,
 * especially for binary upload.
 */
const multipartRequestHandler = (context, output) => {

  // fetchS3File(context, output)
  //   .then( tmpPath => {
  //     const formData = Object.assign({}, output.form, {
  //       file: fs.createReadStream(tmpPath + output.file.Key),
  //     });
  //
  //     const method = output.method.toLowerCase();
  //
  //     const reqOptions = {
  //       url: output.protocol + '//' + output.hostname + output.pathname,
  //       formData: formData,
  //       headers: output.headers,
  //     };
  //
  //     request[method](reqOptions, (err, res, body) => {
  //       if(!err) {
  //         rmdir(tmpPath, () => {
  //           if (res.statusCode >= 200 && res.statusCode <= 207) {
  //             const handler = responseHandler.getResponseHandler(res.headers);
  //             handler(res, body, context, output);
  //           }
  //         });
  //       } else {
  //         context.fail(util.failure(400, err));
  //       }
  //     });
  //   })
  //   .catch( error => {
  //     context.fail(error);
  //   });
};

/**
 * A request handler than handles all form-data/multipart content, where
 * the binary content is just encoded JSON.
 */
const encodedUploadHandler = (context, output) => {
  // fetchS3File(context, output)
  //   .then(tmpPath => {
  //     const filePayload = {
  //       name: output.file.Key,
  //       body: fs.readFileSync(tmpPath + output.file.Key, { encoding: 'base64' }),
  //       // type: mime.lookup(output.file.Key),
  //       // size: fs.statSync(tmpPath + output.file.Key).size,
  //     };
  //
  //     output.body = Object.assign({}, filePayload, output.body, output.form);
  //     output.headers['Content-Type'] = 'application/json';
  //     output.runPreRequestHooks(output);
  //    // This is a JSON request now, so hand it off to the JSON handler, yeah?
  //    jsonRequestHandler(context, output);
  //
  //   })
  //   .catch( error => {
  //     context.fail(error);
  //   });
};


/**
 * Prevents the execution handler from continuing to process; called
 * when we receive `shouldContinue == false`
 */
const stopExecutionHandler = (context, output) => {
  const convertedResponse = output.convertResponse({
    body: JSON.stringify(output.responseBody),
    status: 'Bad Request',
    statusCode: output.statusCode
  });
  const errors = convertedResponse.body.errors || convertedResponse.body;
  context.done(util.failure(convertedResponse.statusCode, JSON.stringify(errors), 'Bad Request'), convertedResponse);
};

/**
 * Determines how to make the request to the vendor. Usually based on
 * the content-type but some previous logic may define the handler for
 * special circumstances by attaching it to `output.requestHandler`
 */
const getRequestHandler = (context, output) => {
  output.headers['Content-Type'] = output.headers['Content-Type'] || 'application/json';
  const contentType = output.headers['Content-Type'].split(';')[0].toLowerCase();
  let handler;

  if(output.shouldContinue === false) {
    console.log('using stopExecutionHandler');
    return stopExecutionHandler;
  }


  if(output.element.protocolType === 'soap') {
    return soapRequestHandler;
  } else if(output.element.protocolType === 'jdbc') {
    return jdbcRequestHandler;
  }

  if(util.notNull(output.reqHandler)) {
    switch(output.reqHandler) {
      case 'encodedUpload':
        console.warn('encodedUploadHandler handler temporarily unavailable');
        handler = encodedUploadHandler;
        break;
      default:
        console.warn('An explicit handler was assigned to `output` but not defined in `requestHandler.js`');
    }
  } else {
    switch(contentType) {
      case 'application/json':
      case 'application/jsonp':
        handler =  jsonRequestHandler;
        break;
      case 'multipart/form-data':
        console.warn('multipartRequestHandler handler temporarily unavailable');
        handler = multipartRequestHandler;
        break;
      default:
        console.warn('Request handler not found');
    }
  }

  if(!util.notNull(handler)) {
    handler = jsonRequestHandler;
  }

  return handler;
};

module.exports = {getRequestHandler, stopExecutionHandler, encodedUploadHandler,
                  multipartRequestHandler, soapRequestHandler, jsonRequestHandler, jdbcRequestHandler};
