'use strict';

const _ = require('lodash');
const url = require('url');
const fs = require('fs');
const backend = require('./lib/backend');
const util = require('./lib/util');
const pager = require('./lib/pager');
const oauth1 = require('./lib/oauth1a');
const awsSign = require('aws-sign');

/**
 * The main element handler function, this is a curried function which
 * takes an element descriptor, event object (request), and context, and
 * then calls one of the context callback functions with a response or
 * error when the element API handling is done.
 *
 * Details of the request, response, and context callbacks, as well as
 * general usage guidelines are available in a separate document.
 */
const elementHandler = (element) => (event, context) => {
  console.log('Received event:', JSON.stringify(event, null, 2));

  const verror = verifyElement(element);
  if (verror != null) {
    return context.fail(util.failure(500, verror));
  }

  const start = new Date().getTime();

  /* Wrap the context to handle gateway-style requests/responses. */
  const apiCtx = util.apiGateway(context);

  /* Wrap the context to perform debugging. */
  const outCtx = {
    succeed: o => {
      const end = new Date().getTime();
      console.log("success in", (end - start));
      return apiCtx.succeed(o);
    },
    fail: e => {
      const end = new Date().getTime();
      console.log("fail in", (end - start));
      apiCtx.fail(e);
    },
    done: (e, o) => {
      const end = new Date().getTime();
      console.log("done in", (end - start));
      apiCtx.done(e, o);
    }
  };
  try {
    convert(event, element, (error, output) => {
      if (util.notNull(error)) { return outCtx.fail(util.failure(400, error)); }
      pager.page(backend.send)(outCtx, output);
    });
  } catch (e) {
    console.warn(e);
    outCtx.fail(util.failure(500, e));
  }
};

/**
 * Verifies that an element descriptor file is version-compatible with
 * this version of the handler.
 */
const verifyElement = (element) => {
  const versionTag = element.version;

  // Only for initial version only: versionTag is allowed to be null /
  // empty
  if (versionTag == null) { return null; }

  if (!_.isString(versionTag)) { return "version is not a string"; }

  const dot = versionTag.indexOf('.');
  const major = dot == -1 ? versionTag : versionTag.substr(0, dot);
  if (major == '1') { return null; }

  return `Incorrect version: ${major}`;
}

// This package should no longer provide the node handler directly:
// instead, it should be used as a dependency for the actual per-element
// handler packages. Each of those handler packages should contain
// nothing more than:
//
// 1. a dependency to this package,
// 2. the per-element element.json file (in data/element.json)
// 3. the following code in its main.js (also included here for
// backwards compatibility)
//
//    const celib = // .. require this package
//    const element = require('./data/element.json');
//    exports.handler = celib.handler(element);
//
let mainHandler;
if(fs.existsSync(`${__dirname}/data/element.json`)) {
  const element = require('./data/element.json');
  mainHandler = elementHandler(element);
}

/**
 * Converts an API Gateway-style input into an http options object that
 * can be used to make a backend request.
 */
const convert = (input, element, cb) => {

  // This algorithm is (very roughly) based on
  // RequestHelper::generateRequestValues()
  const resources = element.resources || [];
  const rmp = findBestResource(resources, input);
  if (_.isEmpty(rmp)) { return cb("No matching resource in builder JSON"); }

  const rsrc = rmp[0];
  const mpath = rmp[1];
  input.mappedPaths = mpath;

  let output = pseudoInstance(input, element, rsrc, mpath);
  return cb(null, output);
}

/**
 * Finds the resource that best serves the given request, according to
 * its path and method.
 */
const findBestResource = (resources, input) => {
  let inputPath = util.constructPath(input.operation.path, v => input.paths[v]);
  input.constructedPath = inputPath;

  // TODO: consistency: More closely match the Ant Path matching in
  // Voldemort
  let curly = /{[^}]*}/g;
  let rSort = _.orderBy(resources, [r =>(r.path || '').replace(curly, '{}').length], ['desc'])

  for (let r of rSort) {
    let mp = util.matchPath(r.path, inputPath);

    if (util.notNull(mp) &&
        (r.method || 'GET').toUpperCase() ==
        (input.operation.method || 'GET').toUpperCase()) { return [r, mp]; }
  }

  return null;
}

/**
 * Extracts values from the input, element and instance.
 */
const extractValues = (input, output, element, rsrc, mpath) => {
  const parameters = element.parameters || [];
  const rparams = rsrc.parameters || [];
  let values = {};
  parameters.concat(rparams).forEach(p => values[`${p.vendorName}:${p.vendorType}`] = {
    parameter: p,
    value: getInputValue(p, output, element, input, mpath)
  });
  return values;
}

/**
 * A Lambda doesn't actually use an element instance. However, certain
 * properties normally associated with an instance (for example, the
 * oauth user token) may come in on the request somewhere. We set up
 * these values here.
 */
const pseudoInstance = (input, element, rsrc, mpath) => {
  // Instance resources are REST APIs by default.
  let output = {
    headers: {
      "Content-Type": 'application/json',
      "Accept": 'application/json'
    },
    input: input,
    original_input: _.cloneDeep(input),
    element: element,
    resource: rsrc,
    findConfig: function(key) { return findConfigValue(key, this, element); },
    getConfig: function() { return getConfigMap(this, element); },
    finalize: function(in2) {
      if (!util.notNull(in2)) { in2 = {}; }
      return updateInput(in2, this, element, rsrc, mpath);
    },
    convertResponse: function(resp) { return convertResponse(this, resp); }
  };


  // Magical replacement: replace each 'x-vendor-Something' header with
  // a 'Something' header. Do this first, so that specific parameters or
  // hooks can override them later.
  const prefix = 'x-vendor-';
  for (let k of Object.keys(input.headers || {})) {
    if (k.toLowerCase().startsWith(prefix)) {
      let realk = k.substr(prefix.length);
      if (_.isEmpty(output.headers)) { output.headers = {}; }
      if(realk.startsWith('oauth_')) {
        realk = realk.replace(/_/g, '.');
      }
      output.headers[realk] = input.headers[k];
      input.headers[realk] = input.headers[k];
    }
  }

  // Also, if a header comes in whose key matches a configuration key,
  // set the output's configuration for that value. We later use the
  // "findConfig" function to pull these back up. In this sense,
  // "output.configuration" acts as a bank for configuration data that
  // would normally be on an instance.
  for (let k of Object.keys(input.headers || {})) {
    let kr;
    if(k.toLowerCase().includes('oauth')){
      kr = k.replace(/_/g, '.');
    }
    else {
      kr = k;
    }
    for (let c of (element.configuration || [])) {
      if (util.notEmpty(c.key) && kr.toLowerCase() === c.key.toLowerCase()) {
        if (_.isEmpty(output.configuration)) { output.configuration = {}; }
        output.configuration[c.key] = input.headers[k];
        delete input.headers[k];
      }
    }
  }

  return output;
}

/**
 * Creates/updates the output from the extracted values.
 */
const updateOutput = (output, input, values, element, rsrc) => {

  for (let k of Object.keys(values)) {
    let v = values[k];
    if(util.notNull(v.value)){
      if(typeof(v.value) === 'number' && isNaN(v.value)){ break;}
      setOutputValue(v.parameter, v.value, input, output);
    }
  }
  //if the output has bodyField, then merge the body and bodyField to body and remove bodyField
  if(!_.isNull(output.bodyField)) {
    if (_.isNull(output.body)) { output.body = {}; }
    output.body = _.extend(output.body, output.bodyField);
    delete output.bodyField;
  }

  output.pathname = util.constructPath(rsrc.vendorPath, v => output.paths ? output.paths[v] : `{${v}}`);
  output.constructedPath = output.pathname
}

/**
 * Sets up the proper security headers and signatures on the output.
 */
const secureOutput = (output, element, rsrc) => {
  if ((element.protocolType || 'http') !== 'http') return;

  const authType = (element.authentication || {}).type
  if (authType === 'oauth2') {
    if (!util.notNull(output.headers)) {
      output.headers = {};
    }
    if (!util.notNull(output.headers["authorization"])) {
      let userToken = output.findConfig('oauth.user.token');
      if (util.notNull(userToken)) {
        output.headers["authorization"] = `Bearer ${userToken}`;
      }
    }
  } else if (authType === 'oauth1') {
    let url;
    if (output.pathname.startsWith('http:') ||
      output.pathname.startsWith('https:')) {
      url = output.pathname;
    } else {
      let baseUrl = output.findConfig('base.url');
      if (util.notNull(baseUrl)) {
        url = baseUrl + output.pathname;
      } else {
        url = output.pathname;
      }
    }

    let method = output.resource.vendorMethod || output.resource.method;
    let oauthParams = oauth1.signReqParams(url, output.form || {},
      output.findConfig('oauth.authorization.url'),
      output.findConfig('oauth.token.url'), method,
      output.findConfig('oauth.api.key'), output.findConfig('oauth.api.secret'),
      output.findConfig('oauth.user.token'),
      output.findConfig('oauth.user.token.secret'), output.query || {});

    let headerOrQuery = output.findConfig('oauth.request.authorization.type');
    if ((headerOrQuery || 'query').toLowerCase() == 'query') {
      if (!util.notNull(output.query)) {
        output.query = {};
      }
      for (let k of Object.keys(oauthParams)) {
        output.query[k] = oauthParams[k]
      }
    } else {
      if (output.findConfig('oauth.signature.encoded')) {
        oauthParams['oauth_signature'] =
          encodeURIComponent(oauthParams['oauth_signature']);
      }

      let auth = `OAuth oauth_nonce="${oauthParams.oauth_nonce || ''}",` +
        `oauth_timestamp="${oauthParams.oauth_timestamp || ''}",` +
        `oauth_version="${oauthParams.oauth_version || ''}",` +
        `oauth_signature_method="${oauthParams.oauth_version || ''}",` +
        `oauth_consumer_key="${oauthParams.oauth_consumer_key || ''}",` +
        `oauth_token="${oauthParams.oauth_token || ''}",` +
        `oauth_signature="${oauthParams.oauth_signature || ''}"`;
      if (!util.notNull(output.headers)) {
        output.headers = {};
      }
      output.headers.Authorization = auth;
    }
  } else if (authType === 'aws') {
    let consumerKey;
    let consumerSecret;
    if (output.input.headers.elementConsumerKey) {
      consumerKey = output.input.headers.elementConsumerKey;
    } else if (output.input.headers['access-key']) {
      consumerKey = output.input.headers['access-key'];
    }
    if (output.input.headers.elementConsumerSecret) {
      consumerSecret = output.input.headers.elementConsumerSecret;
    } else if (output.input.headers['secret-key']) {
      consumerSecret = output.input.headers['secret-key'];
    }
    var signer = new awsSign({
      accessKeyId: consumerKey,
      secretAccessKey: consumerSecret
    });
    // Parse bucket name out of the subdomain
    let bucketName = output.input.headers['x-vendor-subdomain'].substring(0, output.input.headers['x-vendor-subdomain'].indexOf('.s3'));
    // The host we pass to aws-sign is intentionally using the incorrect value (without region accounted for)
    // That's because the package doesn't support different regions, and the signing logic doesn't need to know what region it is
    // So this will always work, for any region, even though it looks incorrect
    var options = {
      method: output.method,
      host: `${bucketName}.s3.amazonaws.com`,
      path: output.pathname,
      headers: output.headers
    };
    signer.sign(options);
    // Copy the two values the aws-sign package provides for us
    output.headers.date = options.headers.date;
    output.headers.authorization = options.headers.Authorization;
  } else if (authType === 'basic') {
    if (!util.notNull(output.headers)) {
      output.headers = {};
    }
    if (!util.notNull(output.headers["authorization"])) {
      let user = output.findConfig('username');
      let pass = output.findConfig('password');
      if (user && pass) {
        output.headers["authorization"] = 'Basic ' +
          util.encodeBase64(user + ':' + pass);
      }
    }
  }
}

/**
 * Finalizes the output object so that it's in the correct form to be
 * used by the nodejs http package.
 */
const finalizeOutput = (output, element, rsrc) => {
  if ((element.protocolType || 'http') !== 'http') return;

  let baseUrl = output.findConfig('base.url');
  if (util.notNull(baseUrl)) {
    let parsedObj = url.parse(baseUrl + output.pathname)
    output.protocol = parsedObj.protocol;
    output.hostname = parsedObj.hostname;
    if (util.notNull(parsedObj.port)) output.port = parsedObj.port;
    output.pathname = parsedObj.pathname;
    output.method = rsrc.vendorMethod || rsrc.method;
  } else {
    return "No base URL found in configuration";
  }
}

/**
 * Runs all the non-legacy request hooks in-process.
 */
const runPreRequestHooks = (input, mpath, output, element, rsrc) => {

  for (let hook of (rsrc.hooks || []).concat(element.hooks || [])) {
    if (hook.type !== 'preRequest') { continue; }
    if (!util.notNull(hook.isLegacy)) { hook.isLegacy = true; }
    if (hook.isLegacy) { continue; }
    if (hook.mimeType !== 'application/javascript') {
      console.warn("can't execute non-javascript hook");
      continue;
    }
    // TODO: `safeAddQueryAttributeList`, `safeAddConvertObjectToMap`,
    // configuration and full vendor_url construction (see
    // ScriptHook::buildHookContext())

    // When the hook succeeds, this is what we do.
    const done = data => {
      let o;
      if(util.notNull(data)) {

        if (util.notNull(o = data.continue)) { output.shouldContinue = data.continue; }
        if (util.notNull(o = data.response_status_code)) { output.statusCode = o; }
        if (util.notNull(o = data.response_body)) {output.responseBody = o;}
        if (util.notNull(o = data.request_vendor_body)) {output.body = o;}
        if (util.notNull(o = data.request_vendor_headers)) {output.headers = o;}
        if (util.notNull(o = data.request_vendor_path)) {
          //If the request_vendor_path is passed from pre-hook, regenerate the vendorPath
          output.constructPath = o;
          output.pathname = o;
          finalizeOutput(output, element, rsrc);
        }
        if (util.notNull(o = data.request_vendor_parameters)) {output.query = o;}
        if (util.notNull(o = data.request_vendor_method)) {
          output.method = o.toUpperCase();
        }
      }
    }

    output.continue = true;
    // When the hook fails, this is what we do.
    const fail = error => console.warn("preRequest hook failure", error);
    let cont = true;
    let test = null;
    let fn = new Function('request_body', 'request_body_map', 'request_headers', 'request_path',
        'request_parameters', 'request_method', 'request_vendor_parameters',
        'request_vendor_path', 'request_vendor_headers',
        'request_vendor_body', 'request_vendor_url',
        'request_vendor_method', 'request_expression',
        'configuration', 'done', 'fail','contin', 'request_vendor_body_map', 'require', hook.body);
    fn(input.body, input.body, input.headers, input.constructedPath, input.queryParameters,
      (input.operation || {}).method, output.query, output.constructedPath,
      output.headers, output.body, output.url, output.method,
      input.query, output.getConfig(), done, fail, output.continue, output.body, require);

  }
}

/**
 * Keeps all derived used-by-http values up to date.
 */
const deriveOutput = (output, element) => {
  if ((element.protocolType || 'http') !== 'http') return;

  output.path = output.pathname + url.format({query: output.query});
}

/**
 * Finds an input specified by a parameter.
 */
const getInputValue = (param, output, element, input, mpath) => {
  const type = param.type;

  switch (type) {
    case 'body': return input.body;
    case 'value': return param.name;
    case 'configuration':
       return output.findConfig(param.name);

    case 'path':
      const paths = mpath || {};
      return paths[param.name];

    case 'query':
      const qp = input.queryParameters || {};
      // Sometimes query params come in the body?
      // Looking at you Hubspot
      return qp[param.name] || (input.body || {})[param.name];

    case 'header':
      const hp = input.headers || {};
      return hp[param.name];

    case 'multipart':
      return {
        Key: input.body['x-s3-filename'],
        Bucket: input.body['x-s3-bucketname']
      };

    default: console.warn("Unrecognized input parameter type", type);
  }
}

/**
 * Sets an output value specified by a parameter.
 */
const setOutputValue = (param, value, input, output) => {
  const vname = param.vendorName;
  const vtype = param.vendorType;

  if (vtype === 'body' && param.type !== 'multipart') {
    if(param.type === 'value') {
      output.replBody = param.name;
    }
    output.body = value;

  } else if(param.type === 'multipart') {
    output.file = value;
    // If the vendor expects the upload as a body
    // Tell our reqHandler to perform this as an encoded
    // file upload
    if(vtype === 'body') {
      output.reqHandler = 'encodedUpload';
    }
  } else if (util.notEmpty(vname)) {
    if (vtype === 'path') {
      if (_.isEmpty(output.paths)) { output.paths = {}; }
      output.paths[vname] = value;
    } else if (vtype === 'query') {
      if (_.isEmpty(output.query)) { output.query = {}; }
      output.query[vname] = value;
    } else if (vtype === 'header') {
      if (_.isEmpty(output.headers)) { output.headers = {}; }
      output.headers[vname] = value;
    } else if (vtype === 'form') {
      if (param.type === 'query') {
        let formBody = {};
        formBody[vname] = value;
        output.body = formBody;
      }
      if (param.type === 'body') {
        output.body = value;
      }
      if (_.isEmpty(output.headers)) {
         output.headers = {
           'Content-Type': 'application/x-www-form-urlencoded'
         };
      } else {
        output.headers['Content-Type'] = 'application/x-www-form-urlencoded';
      }
    } else if (vtype === 'bodyField') {
      if (_.isEmpty(output.bodyField)) { output.bodyField = {}; }
      output.bodyField[vname] = value;
    } else if(vtype === 'bodyToken') {
      if (_.isEmpty(output.bodyTokens)) { output.bodyTokens = {}; }
      if(param.type !== 'body') {
        output.bodyTokens[vname] = util.getDeepPropValue(param.name, output);
      } else {
        output.bodyTokens[vname] = input.body;
      }
    } else if(vtype === 'no-op') {
      // Don't do anything?
    } else {
      console.warn("unrecognized output parameter type", vtype);
    }
  }
}

/**
 * Finds a configuration value from its key, either from the output
 * configuration (if it exists), or from the element's default value.
 */
const findConfigValue = (key, output, element) => {
  const eConfig = element.configuration || [];
  const oConfig = output.configuration || {};

  let oValue = oConfig[key];
  if (util.notNull(oValue)) { return oValue; }

  let configItem = _.find(eConfig, c => c.key === key);

  if (util.notNull(configItem)) {
    return util.constructPath(configItem.defaultValue, v => output.findConfig(v) || `{${v}}`);
  }

  return null;
}

/**
 * Finds all configuration keys and values, from the output
 * configuration (if it exists), or from the element's default value.
 */
const getConfigMap = (output, element) => {
  const oConfig = _.clone(output.configuration || {});
  for (let ec of element.configuration || []) {
    if (!util.notNull(oConfig[ec.key])) {
      oConfig[ec.key] = util.constructPath(ec.defaultValue,
          v => output.findConfig(v) || `{${v}}`)
    }
  }

  return oConfig;
}

/**
 * Updates the output for a given set of input values. During paging,
 * individual page inputs (especially the input query parameters for
 * 'page' and 'pageSize') will be set for each page.
 */
const updateInput = (input, output, element, rsrc, mpath) => {
  // only update values that are given in the input
  output.input = _.merge(_.cloneDeep(output.original_input), input);

  // Hydrate the the continuationToken
  // Similar to LambdRequest::hydrate in the java
  if( util.notNull(output.input.queryParameters)
      && output.input.queryParameters.hasOwnProperty('continuationToken')
      && util.notNull(_.get(output.input.queryParameters, 'continuationToken'))
    ) {
      if(!util.notNull(output.query)) { output.query = {} }
      output.input.queryParameters['nextPage'] = _.get(output.input.queryParameters, 'continuationToken');
      output.input.queryParameters['page'] = _.get(output.input.queryParameters, 'continuationToken');
    }

  //If a pageSize is not passed  or not present, set the default pageSize to 10
  //as the pageSize can be used in prehook
  if ((output.resource.parameters || []).filter(p =>
      (p.name === 'page' || p.name === 'pageSize') &&
      p.type === 'query').length > 0) {
      let qParams = output.input.queryParameters || {};
      if(qParams.pageSize === null || qParams.pageSize === undefined || isNaN(qParams.pageSize)){
        qParams.pageSize = 10; //default the pageSize to 10
      }
      if(qParams.page === null || qParams.page === undefined || isNaN(qParams.page)){
        qParams.page = 1; //default the pageSize to 1
      }
      output.input.queryParameters = qParams;
  }

  let values = extractValues(output.input, output, element, rsrc, mpath);
  updateOutput(output, output.input, values, element, rsrc);
  finalizeOutput(output, element, rsrc);
  runPreRequestHooks(output.input, mpath, output, element, rsrc);
  secureOutput(output, element, rsrc);
  convertRequest(output);
  deriveOutput(output, element);
  output.runPreRequestHooks = (out) => { runPreRequestHooks(output.input, mpath, out, element, rsrc); }
  return output;
}

/**
 * Converts a request body for a resource into its proper form via the
 * configured rootKey for that resource.
 */
const convertRequest = (output) => {
  const rootKey = (output.resource.rootKey || '').split('|')[0];
  if (util.notEmpty(rootKey) && util.notNull(output.body)) {
    const body = output.body;
    output.body = _.set({}, rootKey, body)
  }
}

/**
 * Tweaks a response to be more Cloud-Elements like by applying rootKey
 * transformations. During paging, this is applied to each individual
 * paged response, and not the final merged response.
 */
const convertResponse = (output, response) => {
  // We run the post request hooks first, since they might actually
  // set the response body to match the rootKey
  try {
    runPostRequestHooks(output, response);
  } catch (ex) {
    response.originalBody = response.body;
    response.body = {errors: "postRequestHook failure: " + ex.toString()}
    response.statusCode = 500;
    return response;
  }

  const rootKey = (output.resource.rootKey || '').split('|')[1];
  if (util.notEmpty(rootKey) && (response.statusCode >= 200 && response.statusCode <= 207)) {
    let rootKeyRes = _.get(response.body, rootKey);
    //Set the response body only when it is undefined
    if(rootKeyRes !== undefined) {
      response.originalBody = response.body;
      response.body = rootKeyRes;
    }
  }

  return response;
}

/**
 * Runs all the non-legacy request hooks in-process.
 */
const runPostRequestHooks = (output, response) => {
  const rsrc = output.resource || {};
  for (let hook of (rsrc.hooks || []).concat(output.element.hooks || [])) {
    if (hook.type !== 'postRequest') { continue; }
    if (!util.notNull(hook.isLegacy)) { hook.isLegacy = true; }
    if (hook.isLegacy) { continue; }
    if (hook.mimeType !== 'application/javascript') {
      console.warn("can't execute non-javascript hook");
      continue;
    }

    // When the hook succeeds, this is what we do.
    const done = data => {
      let o;
      if (util.notNull(data)) {
        if (util.notNull(data.response_body)) { response.body = data.response_body; }
        if (util.notNull(data.response_headers)) { response.headers = data.response_headers; }
        if (util.notNull(data.response_status_code)) {
          response.statusCode = data.response_status_code;
        }
      }
    }

    // When the hook fails, this is what we do.
    const fail = error => console.warn("postRequest hook failure", error);

    let fn = new Function('response_body', 'response_body_raw',
        'response_body_map', 'response_headers', 'response_status_code',
        'response_iserror', 'response_body_raw_map', 'response_error',
        'request_body', 'request_headers', 'request_path',
        'request_parameters', 'request_method', 'request_vendor_parameters',
        'request_vendor_path', 'request_vendor_headers',
        'request_vendor_body', 'request_vendor_url',
        'request_vendor_method', 'configuration', 'done', 'fail', 'require', hook.body);
    const body = response.body || {};
    const headers = response.headers || {};
    const raw_body_map = _.map(body, JSON.stringify)
    const code = response.statusCode || 0;
    const input = output.input || {};
    const mpath = input.mappedPaths || {};

    fn(body, JSON.stringify(body), body, headers, code,
      code >= 200 && code <= 207, raw_body_map, response.status,
      input.body, input.headers, input.constructedPath, input.queryParameters,
      (input.operation || {}).method, output.query, output.constructedPath,
      output.headers, output.body, output.url, output.method,
      output.getConfig(), done, fail, require);
  }
}

module.exports = {
  handler: elementHandler,
  mainHandler: mainHandler,
  convert: convert,
  verifyElement: verifyElement,
  convertResponse: convertResponse,
  convertRequest: convertRequest,
  getConfigMap: getConfigMap,
  secureOutput: secureOutput,
  findBestResource: findBestResource,
  extractValues: extractValues,
  pseudoInstance: pseudoInstance,
  updateOutput: updateOutput,
  finalizeOutput: finalizeOutput,
  runPreRequestHooks: runPreRequestHooks,
  deriveOutput: deriveOutput,
  getInputValue: getInputValue,
  setOutputValue: setOutputValue,
  findConfigValue: findConfigValue,
  updateInput: updateInput,
  runPostRequestHooks: runPostRequestHooks,
}
