'use strict';

/*
 * This inappropriately named 'jdbc' module is roughly analogous to the
 * AbstractJdbcHelper class (and friends) in the Java executor package.
 * It is responsible for delivering the appropriate mysql (or whatever)
 * queries, and calling context.succeed (or context.fail) with correct
 * data.
 */

const _ = require('lodash');
const util = require('./util');
const mysql = require('mysql');
const psql = require('pg');
const named = require('node-postgres-named');

/**
 * Sends output to the proper JDBC channel.
 */
exports.send = (context, output) => {
  if (isSchemaRequest(context, output)) {
    requestSchema(context, output);
  } else if (isValidateConnection(context, output)) {
    validateConnection(context, output);
  } else {
    requestJdbc(context, output);
  }
};

/**
 * Schema requests are not currently supported in the executor; always
 * returns `false`
 */
const isSchemaRequest = (context, output) => {
  // TODO
  return false;
};

/**
 * Schema requests are not currently supported in the executor; does
 * nothing.
 */
const requestSchema = (context, output) => {
  // TODO
};

/**
 * Validate connection requests are not currently supported in the
 * executor; always returns false.
 */
const isValidateConnection = (context, output) => {
  // TODO
  return false;
};

/**
 * Validate connection requests are not currently supported in the
 * executor; does nothing.
 */
const validateConnection = (context, output) => {
};

/**
 * Makes a normal database-style request from an output to the back-end
 * database, choosing the correct request style (query, script, insert,
 * delete, etc.) based on the output structure.
 */
const requestJdbc = (context, output) => {
  if (isNamedQuery(context, output)) {
    executeNamedQuery(context, output);
  } else if (isQuery(context, output)) {
    const script = getScript(context, output);
    if (_.isEmpty(script)) {
      return context.fail({error: `No script found for JDBC query`});
    }

    // TODO: Set MAX_ROWS
    execute(script, context, output);
  } else if (isGetAll(context, output)) {
    executeGetAll(context, output);
  } else if (isGetById(context, output)) {
    executeGetById(context, output);
  } else if (isDelete(context, output)) {
    executeDelete(context, output);
  } else if (isInsert(context, output)) {
    executeInsert(context, output);
  } else if (isUpdate(context, output)) {
    executeUpdate(context, output);
  } else{
    console.log('bingss');
  }
};
exports.requestJdbc = (context, output) => requestJdbc(context, output);

/**
 * Determines if the given output represents a SQL query.
 */
const isQuery = (context, output) => {
  return (output.resource.method || '').toUpperCase() === 'POST' &&
      (output.resource.path || '').endsWith('/query');
};
exports.isQuery = (context, output) => isQuery(context, output);

/**
 * Determines if the given output represents a SQL script.
 */
const getScript = (context, output) => {
  return (output.query || {}).script || (output.input.body || '').toString();
};
exports.getScript = (context, output) => getScript(context, output);

/**
 * Determines if the given output represents a SQL named query.
 */
const isNamedQuery = (context, output) => {
  const vpUpper = (output.input.sql || output.resource.vendorPath || '')
      .trim().toUpperCase();
  return vpUpper.startsWith("INSERT") || vpUpper.startsWith("UPDATE") ||
      vpUpper.startsWith("DELETE") || vpUpper.startsWith("SELECT") ||
      (vpUpper.indexOf('(') !== -1 && vpUpper.indexOf(')') !== -1);
};
exports.isNamedQuery = (context, output) => isNamedQuery(context, output);

/**
 * Determines if the given output represents a SQL query that returns
 * multiple rows.
 */
const isGetAll = (context, output) => {
  return (output.resource.method || 'GET').toUpperCase() === 'GET' &&
      !(output.resource.path || '').endsWith('/{objectId}') &&
      !(output.resource.path || '').endsWith('/{id}');
};
exports.isGetAll = (context, output) => isGetAll(context, output);

/**
 * Determines if the given output represents a SQL query that returns
 * a single row.
 */
const isGetById = (context, output) => {
  return (output.resource.method || 'GET').toUpperCase() === 'GET' &&
      ((output.resource.path || '').endsWith('/{objectId}') ||
      (output.resource.path || '').endsWith('/{id}'));
};
exports.isGetById = (context, output) => isGetById(context, output);

/**
 * Determines if the given output represents a SQL delete statement.
 */
const isDelete = (context, output) => {
  return (output.resource.method || '').toUpperCase() === 'DELETE';
};
exports.isDelete = (context, output) => isDelete(context, output);

/**
 * Determines if the given output represents a SQL insert statement.
 */
const isInsert = (context, output) => {
  return (output.resource.method || '').toUpperCase() === 'POST';
};
exports.isInsert = (context, output) => isInsert(context, output);

/**
 * Determines if the given output represents a SQL update statement.
 */
const isUpdate = (context, output) => {
  return ((output.resource.method || '').toUpperCase() === 'PATCH') || ((output.resource.method || '').toUpperCase() === 'PUT');
};
exports.isUpdate = (context, output) => isUpdate(context, output);

/**
 * Executes a named query on the database represented by the given
 * request object.
 */
const executeNamedQuery = (context, output) => {
  const scriptNamedValues = findNamedValues(context, output);
  const script = scriptNamedValues[0];
  const namedValues = scriptNamedValues[1];

  if (script.toUpperCase().indexOf('RETURNING') !== -1) {
    // we are hosed: mysql doesn't support "returning"
    output.fail("Returning unsupported in mysql");
  } else {
    withConnection(context, output, (conn, cb) => {
      console.log("in with conn", script, namedValues);
      conn.query(script, namedValues, cb);
    }, standardResponse(context, output));
  }
};
exports.executeNamedQuery = (context, output) => executeNamedQuery(context, output);

/**
 * Executes a statement on the database represented by the given request
 * object.
 */
const execute = (script, context, output) => {
  withConnection(context, output, (conn, cb) => { conn.query(script, cb); },
      standardResponse(context, output));
};
exports.execute = (script, context, output) => execute(script, context, output);

/**
 * Executes a query that returns multiple rows on the database
 * represented by the given request object.
 */
const executeGetAll = (context, output) => {
  const orderBy = (output.query || {}).orderBy || '';

  let whereClause = (output.query || {}).where || '';
  if (!_.isEmpty(whereClause)) { whereClause = " where " + whereClause; }

  let tname = (output.input.mappedPaths || {}).objectName;
  if (_.isEmpty(tname)) {
    return context.fail({error: "No `objectName` value found."});
  }

  const fields = findFields(context, output, tname);

  let page = parseInt((output.input.query || {}).page);
  if (isNaN(page)) { page = 1; }
  let pageSize = parseInt((output.input.query || {}).pageSize);
  if (isNaN(pageSize)) { pageSize = 50; }
  const offset = pageOffset(page, pageSize);

  let script;
  if (!_.isEmpty(orderBy)) {
    script = `select ${fields} from ${tname}${whereClause} ` +
        `order by ${orderBy} limit ${pageSize} offset ${offset}`;
  } else {
    script = `select ${fields} from ${tname}${whereClause} ` +
        `limit ${pageSize} offset ${offset}`;
  }

  withConnection(context, output, (conn, cb) => {
    conn.query(script, cb);
  }, standardResponse(context, output));
};
exports.executeGetAll = (context, output) => executeGetAll(context, output);

/**
 * Executes a query that returns a single row on the database
 * represented by the given request object.
 */
const executeGetById = (context, output) => {
  let tname = (output.input.mappedPaths || {}).objectName;
  if (_.isEmpty(tname)) {
    return context.fail({error: "No `objectName` value found."});
  }

  let objectId = (output.query || {}).objectId;
  if (_.isEmpty(objectId) && !_.isNumber(objectId)) {
    return context.fail({error: `No ID found in query for ${tname}.`});
  }

  const pk = findPk(context, output, tname);
  if (_.isEmpty(pk)) {
    return context.fail({error: `No PK found in query for ${tname}.`});
  }

  const fields = findFields(context, output, tname);
  const script = `select ${fields} from ${tname} where ${pk} = :val limit 1`;

  withConnection(context, output, (conn, cb) => {
    conn.query(script, {val: objectId}, cb);
  }, standardResponse(context, output));
};
exports.executeGetById = (context, output) => executeGetById(context, output);

/**
 * Executes a delete statement represented by the given request object.
 */
const executeDelete = (context, output) => {
  let tname = (output.input.mappedPaths || {}).objectName;
  if (_.isEmpty(tname)) {
    return context.fail({error: "No `objectName` value found."});
  }

  let objectId = (output.query || {}).objectId;
  if (_.isEmpty(objectId)) {
    return context.fail({error: `No ID found in query for ${tname}.`});
  }

  const pk = findPk(context, output, tname);
  if (_.isEmpty(pk)) {
    return context.fail({error: `No PK found in query for ${tname}.`});
  }

  const script = `delete from ${tname} where ${pk} = :val`;

  withConnection(context, output, (conn, cb) => {
    conn.query(script, {val: objectId}, cb);
  }, standardResponse(context, output));
};
exports.executeDelete = (context, output) => executeDelete(context, output);

/**
 * Executes an insert statement represented by the given request object.
 */
const executeInsert = (context, output) => {
  let tname = (output.input.mappedPaths || {}).objectName;
  if (_.isEmpty(tname)) {
    return context.fail({error: "No `objectName` value found."});
  }

  const body = output.body || {};

  const ks = Object.keys(body);
  const kstr = ks.join(',');
  const vstr = ks.map(k => `:${k}`).join(',');

  let script = `insert into ${tname} (${kstr}) values (${vstr})`;
  // const pk = findPk(context, output, tname);
  // TODO: append " returning ${pk}" for DBs that support it

  withConnection(context, output, (conn, cb) => {
    conn.query(script, body, cb);
  }, (err, result) => {
    if (_.isEmpty(result.id)) {
      if (!util.notNull(output.query)) { output.query = {}; }
      output.query.objectId = result.id;
      executeGetById(context, output);
    } else {
      standardResponse(context, output)(err, result);
    }
  });
};
exports.executeInsert = (context, output) => executeInsert(context, output);

/**
 * Executes an update statement represented by the given request object.
 */
const executeUpdate = (context, output) => {
  let tname = (output.input.mappedPaths || {}).objectName;
  if (_.isEmpty(tname)) {
    return context.fail({error: "No `objectName` value found."});
  }

  let objectId = (output.query || {}).objectId;
  if (_.isEmpty(objectId)) {
    return context.fail({error: `No ID found in query for ${tname}.`});
  }

  const pk = findPk(context, output, tname);
  if (_.isEmpty(pk)) {
    return context.fail({error: `No PK found in query for ${tname}.`});
  }

  const body = output.body || output.input.body ? output.input.body : {};

  const ks = Object.keys(body);
  const kvstr = ks.map(k => `${k} = :${k}`).join(', ');

  let script = `update ${tname} set ${kvstr} where ${pk} = :pkval00`;
  let args = _.clone(body); args.pkval00 = objectId;

  withConnection(context, output, (conn, cb) => {
    conn.query(script, args, cb);
  }, (err, result) => {
    console.log('err', err);
    if (util.notNull(err)) { return context.fail({error: err}); }
    if (util.notNull(result.count) && result.count > 0) {
      executeGetById(context, output);
    } else {
      context.fail({error:
          `Unable to update: no row in ${tname} where ${pk} = ${objectId}`});
    }
  });
};
exports.executeUpdate = (context, output) => executeUpdate(context, output);

/**
 * Finds request values that are referred to by a query or statement.
 */
const findNamedValues = (context, output) => {
  let namedValues = {};
  let script = (output.input.sql || output.resource.vendorPath || '');
  script = fillInQuery(script, output);

  ((output.query || {}).namedParameters || '').split(/,/).forEach(np => {
    namedValues[np.trim()] = null;
  });

  Object.keys(output.query || {}).forEach(qk => {
    if (qk === 'namedParameters') { return; }
    script = script.replace(new RegExp(`{${qk}}`, 'g'), `:${qk}`);
    namedValues[qk] = output.query[qk];
  });

  return [script, namedValues];
};
exports.findNamedValues = (context, output) => findNamedValues(context, output);

/**
 * Replace curly parameters in the SQL query with their path parameters,
 * handling the "orderBy" and "where" parameters specially.
 */
const fillInQuery = (script, output) => {
  const paths = output.paths || {};
  return util.constructPath(script, v => {
    if (v === 'orderBy' && !_.isEmpty(paths.orderBy)) {
      return `order by ${paths.orderBy}`;
    } else if (v === 'where' && !_.isEmpty(paths.where)) {
      return `where ${paths.where}`;
    } else {
      return paths[v];
    }
  });
};
exports.fillInQuery = (script, output) => fillInQuery(script, output);

/**
 * Find the fields in a query or statement that should be selected.
 */
const findFields = (context, output, tname) => {
  let fields = (output.query || {}).fields || '*';
  if (fields === '*' &&
      !/^true$/i.test(output.findConfig('db.selectall.fields') || 'false')) {
    let table = _.find(JSON.parse(output.findConfig('db.schema')),
        t => t.name === tname);
    if (util.notNull(table)) {
      fields = table.columns.filter(c => util.notNull(c.name) &&
          c.dataType !== 'object').map(c => {
        if (c.name.indexOf(' ') !== -1) { return `"${c.name}"`; }
        else { return c.name; }
      }).join(',');
    }
  }
  return fields;
};
exports.findFields = (context, output, tname) => findFields(context, output, tname);

/**
 * Finds the primary key in a statement or query that should be used or
 * returned.
 */
const findPk = (context, output, tname) => {
  let pks = null;
  let table = _.find(JSON.parse(output.findConfig('db.schema')),
      t => t.name === tname);
  if (util.notNull(table)) {
    pks = table.columns.filter(c => util.notNull(c.name) &&
        c.primaryKey).map(c => {
      if (c.name.indexOf(' ') !== -1) { return `"${c.name}"`; }
      else { return c.name; }
    });
    return pks[0];
  }
  return null;
};
exports.findPk = (context, output, tname) => findPk(context, output, tname);

/**
 * Performs a given function with a database connection.
 */
const withConnection = (context, output, fn, cb) => {
  createConnection(context, output, (err, conn) => {
    if (util.notNull(err)) { return cb(err); }
    fn(conn, (err, result) => {
      if (util.notNull(err)) { conn.end(); return cb(err); }
      conn.end(err => {
        if (util.notNull(err)) { return cb(err); }
        return cb(null, standardizeMySqlResult(context, output, result));
      });
    });
  });
};
exports.withConnection = (context, output, fn, cb) => withConnection(context, output, fn, cb);

/**
 * Standardizes a SQL result from a MySql database into a normal format.
 */
const standardizeMySqlResult = (context, output, result) => {
  const inheads = (output.input.headers || {});
  const driver = inheads['jdbc.driverClassName'] ||
      output.findConfig('jdbc.driver.classname');

  if (driver === 'org.postgresql.Driver') {
    if (util.notNull(result.rows)) {
      result = result.rows;
      if (result.length > 0) {
        const keys = _.keys(result[0]);
        if (keys.length === 1) {
          result.id = result[0][keys[0]];
        }
      }
    } else {
      result.count = result.rowCount;
    }
  }

  if (isGetById(context, output) && result.length >= 1) result = result[0];
  if (util.notNull(result.insertId)) result.id = result.insertId;
  if (util.notNull(result.affectedRows)) result.count = result.affectedRows;
  return result;
};
exports.standardizeMySqlResult = (context, output, result) => standardizeMySqlResult(context, output, result);

/**
 * Creates a new connection to a database.
 */
const createConnection = (context, output, cb) => {
  const inheads = (output.input.headers || {});
  const driver = inheads['jdbc.driverClassName'] ||
      output.findConfig('jdbc.driver.classname');

  if (driver === 'com.mysql.jdbc.Driver') {
    cb(null, createMysqlConnection(context, output));
  } else if (driver === 'org.postgresql.Driver') {
    createPsqlConnection(context, output, cb);
  } else {
    cb(`Unrecognized driver: ${driver}`);
  }
};
exports.createConnection = (context, output, cb) => createConnection(context, output, cb);

/**
 * Creates a new connection to a MySql database.
 */
const createMysqlConnection = (context, output) => {
  const inheads = (output.input.headers || {});
  const hostport = inheads['jdbc.host'] || output.findConfig('db.host');
  const colon = hostport.indexOf(':');
  let host = hostport;
  let port = 3306;
  if (colon !== -1) {
    host = hostport.substr(0, colon);
    port = parseInt(hostport.substr(colon + 1));
    if (isNaN(port)) { port = 3306; }
  }
  let conn = mysql.createConnection({
    host: host,
    port: port,
    user: inheads['jdbc.username'] || output.findConfig('username'),
    password: inheads['jdbc.password'] || output.findConfig('password'),
    database: inheads['jdbc.database'] || output.findConfig('db.name')
  });

  // Here's how mysql can handle named parameters:
  conn.config.queryFormat = (query, values) => {
    if (!values) { return query; }
    return query.replace(/\:(\w+)/g, function (txt, key) {
      if (values.hasOwnProperty(key) && util.notNull(values[key])) {
        return mysql.escape(values[key]);
      }
      return '';
    }.bind(this));
  };

  return conn;
};
exports.createMysqlConnection = (context, output) => createMysqlConnection(context, output);

/**
 * Creates a new connection to a PostgreSQL database.
 */
const createPsqlConnection = (context, output, cb) => {
  // psql coincidently (or not so coincidently) has the same basic
  // config structure as mysql
  const inheads = (output.input.headers || {});
  const hostport = inheads['jdbc.host'] || output.findConfig('db.host');
  const colon = hostport.indexOf(':');
  let host = hostport;
  let port = 5432;
  if (colon !== -1) {
    host = hostport.substr(0, colon);
    port = parseInt(hostport.substr(colon + 1));
    if (isNaN(port)) { port = 5432; }
  }

  let client = new psql.Client({
    host: host,
    port: port,
    user: inheads['jdbc.username'] || output.findConfig('username'),
    password: inheads['jdbc.password'] || output.findConfig('password'),
    database: inheads['jdbc.database'] || output.findConfig('db.name')
  });
  named.patch(client);
  client.connect(err => cb(err, client));
};
exports.createPsqlConnection = (context, output, cb) => createPsqlConnection(context, output, cb);

/**
 * Generate a resonse, and send it correctly to the context.
 */
const standardResponse = (context, output) => (err, result) => {
  if (util.notNull(err)) { return context.fail({error: err}); }
  else { context.succeed({headers: {}, body: result}); }
};
exports.standardResponse = (context, output) => standardResponse(context, output);

/**
 * Correctly re-calculates the offset based on page and page size.
 */
const pageOffset = (page, pageSize) => {
  if (page < 1 || pageSize < 1) { return 0; }
  return (page - 1) * pageSize;
};
exports.pageOffset = (page, pageSize) => pageOffset(page, pageSize);
