'use strict';

const _ = require('lodash');
const async = require('async');
const url = require('url');
const util = require('./util');

/**
 * Takes a function(context, finalizedOutput), and returns a new
 * function(context, unfinalizedOutput). This effectively wraps the
 * output handler into something that can transparently make multiple
 * requests on the backend.
 */
exports.page = (send) => {
  return (context, output) => {
    if ((output.resource.parameters || []).filter(p =>
        (p.name === 'page' || p.name === 'pageSize') &&
        p.type === 'query').length === 0) {
      return send(context, output.finalize());
    }

    const errStrategy = createStrategy(output);
    const err = errStrategy[0]; const strategy = errStrategy[1];
    if (util.notNull(err) || !util.notNull(strategy)) {
      return context.fail(util.failure(400,
          `Can't determine paging strategy: ${err}`));
    }

    /* Keep making backend requests as long as we need to. */
    let objs = [];
    async.during(
      cb => {
        let innerContext = _.assign(_.clone(context), {
          succeed: data => strategy.hasMore(objs, data, cb),
          fail: failure => cb(failure),
          done: (err, data) => {
            if (util.notNull(err)) { cb(err); }
            else { strategy.hasMore(objs, data, cb); }
          }
        });

        let innerOutput = _.cloneDeep(output).finalize(strategy.input());
        send(innerContext, innerOutput);
      },
      cb => cb(),   // noop: all the work is done in the test
      failure => {
        if (util.notNull(failure)) { return context.fail(failure); }
        else {
          return context.succeed(convertMerged(output, strategy, objs));
        }
      }
    );
  };
};

/**
 * Finds a pagination strategy that lets us iterate through all the
 * backend requests to get the data that we need to return.
 */
const createStrategy = (output) => {
  const inQuery = output.input.queryParameters || {};
  const nextDetails = util.decodeJson64(inQuery.nextPage) || {};
  let isCursor = false;
  let page = inQuery.page;
  if (!util.notNull(page)) {
    page = nextDetails.page;
    isCursor = true;
  } else { page -= 1; }
  if (!util.notNull(page)) { page = 0; }
  page = Math.max(0, page);

  let pageSize = inQuery.pageSize;
  if (!util.notNull(pageSize)) { pageSize = nextDetails.pageSize; }
  if (!util.notNull(pageSize)) { pageSize = 50; }
  pageSize = Math.max(0, pageSize);

  nextDetails.page = page;
  nextDetails.pageSize = pageSize;

  const pageType = output.findConfig('pagination.type') || 'page';
  const pageMax = output.findConfig('pagination.max') || 200;
  const start = page * pageSize;
  const end = (page + 1) * pageSize;

  if (pageType === 'offset') {
    return [null, offsetStrategy(page, pageSize, start, end, pageMax)];
  } else if (pageType === 'page') {
    let startInd = parseInt(output.findConfig('pagination.page.startindex'));
    if (isNaN(startInd)) { startInd = 1; }
    return [null, pageStrategy(page, pageSize, start, end, pageMax, startInd, isCursor)];
  } else if (pageType === 'cursor') {
    let offset = nextDetails.offset;
    let cursor = nextDetails.providerNextPage;
    if (!util.notNull(offset)) {
      offset = nextDetails.page * nextDetails.pageSize;
    }
    const nextPageKey = output.resource.nextPageKey;
    if (!util.notNull(nextPageKey)) {
      return ["unable to use cursor: no nextPageKey"];
    }
    return [null, cursorStrategy(page, pageSize, start, end, pageMax,
        cursor, offset, nextPageKey)];
  }
};
exports.createStrategy = (output) => createStrategy(output);

/**
 * An offset strategy asks the backend for a particular offset and
 * limit, as often needed until all results are in. Offsets are always
 * zero-based, so an offset of 0 is the same as asking for no offset at
 * all.
 */
const offsetStrategy = (page, pageSize, start, end, max) => {
  let offset = start;
  let limit = Math.min(max, end - offset);
  let lastHeaders = {};
  let hasMore = true;

  return {
    hasMore: (objs, data, cb) => {
      lastHeaders = data.headers;
      // TODO : check response header or something?
      hasMore = (data.body || []).length >= limit;
      objs.push.apply(objs, data.body);
      offset += (data.body || []).length;
      limit = Math.min(max, end - offset);
      return cb(null, hasMore && offset < end);
    },
    input: () => ({ queryParameters: { page: offset, pageSize: limit } }),
    headers: () => lastHeaders,
    cursor: () => {
      if (!hasMore) return null;
      return ({
        page: page + 1,
        pageSize: pageSize,
        offset: offset
      });
    }
  };
};
exports.offsetStrategy = (page, pageSize, start, end, max) => offsetStrategy(page, pageSize, start, end, max);

/**
 * A page strategy asks the backend for a particular page and pageSize,
 * possibly multiple times, finding the optimal page and pageSize every
 * request. The 'startInd' argument indicates which index the backend
 * thinks is the first record, and is usually 1 (for 1-based-index
 * backends) or 0 (for 0-based-index backends).
 */
const pageStrategy = (page, pageSize, start, end, max, startInd, isCursor) => {
  let offset = start;
  let optimal = optimalPaging(offset, end, max);
  let lastHeaders = {};
  let hasMore = true;
  // non-cursor has been decremented
  let incrementNumber = isCursor ? 1 : 2 ;

  return {
    hasMore: (objs, data, cb) => {
      lastHeaders = data.headers;
      // TODO : check response header or something?
      if(util.notNull(data.body) && Array.isArray(data.body)) {
        hasMore = data.body.length >= optimal.pageSize;
        objs.push.apply(objs, data.body.slice(optimal.slice));
        offset = optimal.page * optimal.pageSize + data.body.length;
        optimal = optimalPaging(offset, end, max);
        return cb(null, hasMore && offset < end);
      }
      return cb(null, false);
    },
    input: () => ({ queryParameters: shiftIndex(optimal, startInd) }),
    headers: () => lastHeaders,
    cursor: () => {
      if (!hasMore) return null;
      return ({
        page: page + incrementNumber,
        pageSize: pageSize,
        offset: offset
      });
    }
  };
};
exports.pageStrategy = (page, pageSize, start, end, max, startInd, isCursor) => pageStrategy(page, pageSize, start, end, max, startInd, isCursor);

/**
 * Finds a pageSize that maximizes:
 *
 * ```
 * min(end, (pageSize * (page + 1)))
 * where page = floor(offset / pageSize)
 *     , pageSize <= max
 * ```
 *
 * Where multiple pageSize values have the same score, we want to choose
 * the smallest one.
 *
 * For example, if offset=55, end=75, max=10, the optimal pageSize is
 * 9 which returns items 54-62: 8 of which are values we care about, and
 * not (as one might naively suspect) 10 which gets items 50-59, of
 * which only 5 we care about.
 */
const optimalPaging = (offset, end, max) => {
  // start with max size, and go downward until

  // TODO : use prime factorization combinatorics instead?
  let pageSize = Math.min(max, (end - offset) * 2);
  let bestPage;
  let bestPageSize;
  let maxCapture = 0;
  while (pageSize > 0) {
    let page = Math.floor(offset / pageSize);
    let capture = Math.min(end, pageSize * (page + 1));
    if (capture >= maxCapture) {
      maxCapture = capture;
      bestPage = page;
      bestPageSize = pageSize;
    }
    if (maxCapture >= pageSize + offset) break;
    --pageSize;
  }

  return {
    page: bestPage,
    pageSize: bestPageSize,
    slice: offset - bestPageSize * bestPage
  };
};
exports.optimalPaging = (offset, end, max) => optimalPaging(offset, end, max);

const shiftIndex = (optimal, startInd) => {
  return {page: optimal.page + startInd, pageSize: optimal.pageSize};
};
exports.shiftIndex = (optimal, startInd) => shiftIndex(optimal, startInd);

/**
 * A cursor strategy sends a cursor to the backend, which is a
 * continuation token that resumes record retrieval at some offset
 * indicated by 'cOffset'.
 */
const cursorStrategy = (page, pageSize, start, end, max, cursor, cOffset,
    nextPageKey) => {
  let offset = cOffset;
  if (offset > start) {
    // boo, restart from the top
    offset = 0;
    cursor = null;
  }
  let slice = start - offset;
  let size = Math.min(max, end - offset);
  let lastHeaders = {};

  return {
    hasMore: (objs, data, cb) => {
      lastHeaders = data.headers;
      extractCursor(data, nextPageKey, (err, gotCursor) => {
        if (util.notNull(err)) { return cb(err); }
        cursor = gotCursor;
        if(util.notNull(data.body)) {
          offset += data.body.length;
          //If the results is an array then add the results to the obj,
          //if its an object then just insert the object to objs
          if(data.body instanceof Array) {
            objs.push.apply(objs, data.body.slice(slice));
          } else {
            objs.push(data.body);
          }
        }
        if (start > offset) { slice = start - offset; }
        else { slice = 0; }
        size = Math.min(max, end - offset);
        return cb(null, util.notNull(cursor) && offset < end);
      });
    },
    input: () => ({ queryParameters: { pageSize: size, nextPage: cursor }}),
    headers: () => lastHeaders,
    cursor: () => {
      if (!util.notNull(cursor)) return null;
      return ({
        page: page + 1,
        pageSize: pageSize,
        offset: offset,
        providerNextPage: cursor
      });
    }
  };
};
exports.cursorStrategy = (page, pageSize, start, end, max, cursor, cOffset, nextPageKey) => cursorStrategy(page, pageSize, start, end, max, cursor, cOffset, nextPageKey);

/**
 * Extracts a cursor from a response. The 'nextPageKey' determines where
 * in the response to find the cursor, and can be
 *
 *    * `header.<header_name>` : found as a header
 *    * `body.<body_location>` : found as a body value
 *    * `body.<body_location>?queryParam : query param on a body value
 */
const extractCursor = (data, nextPageKey, cb) => {
  if (nextPageKey.startsWith("header.")) {
    nextPageKey = nextPageKey.substr(7);
    cb(null, (data.headers || {})[nextPageKey]);
  } else if (nextPageKey.startsWith("body.")) {
    nextPageKey = nextPageKey.substr(5);
    let q = nextPageKey.indexOf('?');
    let query = null;
    if (q !== -1) {
      query = nextPageKey.substr(q + 1);
      nextPageKey = nextPageKey.substr(0, q);
    }
    let val = _.get((data.originalBody || data.body), nextPageKey);
    if (util.notNull(query) && util.notNull(val)) {
      val = url.parse(val, true).query[query];
    }
    cb(null, val);
  } else {
    cb(util.failure(500, `Unknown nextPageKey format: ${nextPageKey}`));
  }
};
exports.extractCursor = (data, nextPageKey, cb) => extractCursor(data, nextPageKey, cb);

/**
 * Converts an output to its final form by adding appropriate
 * page-related headers.
 */
const convertMerged = (output, strategy, objs) => {
  let headers = strategy.headers() || {};
  headers["Elements-Returned-Count"] = objs.length;

  let cursor;
  if (util.notNull(cursor = strategy.cursor())) {
    headers["Elements-Next-Page-Token"] = util.encodeJson64(cursor);
  }

  return {
    body: objs,
    headers: headers
  };
};
exports.convertMerged = (output, strategy, objs) => convertMerged(output, strategy, objs);
