'use strict';

const util = require('./util');
const parseString = require('xml2js').parseString;
const parseNumbers = require('xml2js').processors.parseNumbers;
const parseBooleans = require('xml2js').processors.parseBooleans;
const httpStatus = require('http-status');

/** A response handler that handles all JSON and JSONP content. */
const jsonResponseHandler = (res, rawBody, context, output) => {
   let convertedResponse;
   if(res.statusCode === 204) {
    convertedResponse = output.convertResponse({
        headers: res.headers,
        body: {},
        status: res.statusMessage,
        statusCode: res.statusCode
      });
   } else {

     try {
        const parsed = JSON.parse(rawBody.trim());

        convertedResponse = output.convertResponse({
          headers: res.headers,
          body: parsed,
          status: res.statusMessage,
          statusCode: res.statusCode
        });

    } catch (err) {
      console.warn("couldn't parse JSON:", rawBody.trim(), err);
      context.fail(util.failure(500, err, 'Internal Service Error'));
    }
  }

  if(convertedResponse.statusCode >= 200 && convertedResponse.statusCode <= 207) {
    context.succeed(convertedResponse);
  } else {
    // Sometimes vendors tell us everything is okay, but it's really not
    const errors = convertedResponse.body.errors || convertedResponse.body;
    context.done(util.failure(convertedResponse.statusCode, JSON.stringify(errors), httpStatus[convertedResponse.statusCode]), convertedResponse);
  }
};

/** A handler that handles all XML and SOAP response content. */
const xmlResponseHandler = (res, rawBody, context, output) => {
  const options = {
    trim: true,
    ignoreAttrs: false,
    explicitArray: false,
    explicitRoot: true,
    mergeAttrs:true,
    valueProcessors: [ parseNumbers, parseBooleans]
  };

  parseString(rawBody, options, (err, result) => {
    let convertedResponse;
    if (util.notNull(err)) { context.fail(util.failure(400, err)); }
    else {
      try {
        convertedResponse = output.convertResponse({
          headers: res.headers,
          body: result,
          status: res.statusMessage,
          statusCode: res.statusCode
        });
      } catch (err) {
        console.warn("couldn't parse JSON:", rawBody.trim(), err);
        context.fail(util.failure(500, err, 'Internal Service Error'));
      }

      if(convertedResponse.statusCode >= 200 && convertedResponse.statusCode <= 207) {
        context.succeed(convertedResponse);
      } else {
        const errors = convertedResponse.body.errors || convertedResponse.body;

        context.done(util.failure(convertedResponse.statusCode, JSON.stringify(errors), httpStatus[convertedResponse.statusCode]), convertedResponse);
      }
    }
  });
};

/**
 * Determines and returns the correct handler, based on resource
 * configuration and request/response content-type
 */
const getResponseHandler = (headers, handlerType) => {

  const contentTypeHeader = headers['content-type'] ? headers['content-type'].split(';')[0].toLowerCase() : '';
  const contentType = handlerType ? handlerType : contentTypeHeader;
  let handler;
  switch(contentType) {
    case 'application/json':
    case 'application/jsonp':
      handler =  jsonResponseHandler;
      break;
    case 'application/atom+xml':
    case 'application/xml':
      handler =  xmlResponseHandler;
      break;
    case 'soap':
      handler = xmlResponseHandler;
      break;
  }

  // If the handler is null and the contentType contains xml anywhere in the string
  // use xmlResponseHandler
  if(!util.notNull(handler) && contentType.includes('xml')) {
    handler =  xmlResponseHandler;
  }

  if(!util.notNull(handler)) {
    console.warn('No response type was returned from the vendor. Assuming JSON');
    handler =  jsonResponseHandler;
  }

  return handler;
};

module.exports = {getResponseHandler, xmlResponseHandler, jsonResponseHandler};
