'use strict';

const _ = require('lodash');

/** Returns if something is not null. */
exports.notNull = v => v !== undefined && v !== null;

/** Returns if something is not null or empty. */
exports.notEmpty = v => exports.notNull(v) && v.length > 0;

/** Encodes a string via base64. */
exports.encodeBase64 = s => new Buffer(s, 'utf8').toString('base64');

/** Decodes a string via base64. */
exports.decodeBase64 = s => new Buffer(s, 'base64').toString('utf8');

/** Converts a base64 string into a JSON object. */
exports.decodeJson64 = (json64) => {
  if (!exports.notNull(json64)) { return null; }
  let np = exports.decodeBase64(json64);
  try { return JSON.parse(np); } catch (ex) { return null; }
};

/** Converts a JSON object into a base64 string. */
exports.encodeJson64 = (json) => {
  if (!exports.notNull(json)) { return null; }
  return exports.encodeBase64(JSON.stringify(json));
};

/** Constructs a failure object that can be sent to context.fail() */
exports.failure = (code, msg, type) => {
  if (!exports.notNull(type)) { type = 'Bad request'; }
  return {
    statusCode: code,
    providerMessage: JSON.stringify(msg),
    errorType: type
  };
};

/**
 * Constructs a path from a curly-bars template (like "/abc/{id}/def"),
 * and a lookup function that returns the variable values.
 */
exports.constructPath = (tmpl, varLookup) => {

  const re = /([^{]*)(?:{([^}]+)})?/g;
  let o = '';
  let e = null;
  while ((e = re.exec(tmpl)) !== null && (e[1] || e[2])) {
    o += (e[1] || '');
    if (e[2]) { o += (varLookup(e[2]) || ''); }
  }

  return o;
};

/**
 * Given a curly-bars path template (like "/abc/{id}/def") and a path
 * that might match it (like "/abc/123/def"), return null if it didn't
 * match, or the sub-matches (like {"id": "123"}) if it did.
 */
exports.matchPath = (tmpl, path) => {
  // TODO: improve: this is not particularly stable, or flexible

  const re = /([^{]*)(?:{([^}]+)})?/g;
  let o = '^';
  let e = null;
  let vnames = [];
  while ((e = re.exec(tmpl)) !== null && (e[1] || e[2])) {
    o += (e[1] || '');
    if (e[2]) { o += '([^/]*)'; vnames.push(e[2]); }
  }
  let rre = RegExp(o);
  let eResult = null;
  if ((eResult = rre.exec(path)) !== null) {
    let values = {};
    for (let i = 0; i < vnames.length; ++i) {
      values[vnames[i]] = eResult[i + 1];
    }
    return values;
  } else {
    return null;
  }
};

/** Constructs a context for use by an API Gateway. */
exports.apiGateway = (context) => {
  return _.assign(_.clone(context), {
    succeed: (data) => context.succeed(apiGatewayResponse(data)),
    fail: (err) => context.fail(apiGatewayError(err, context)),
    done: (err, data) =>
      context.done(apiGatewayError(err, context), apiGatewayResponse(data))
  });
};

/**
 * Morphs the response object into something the API Gateway can use.
 */
const apiGatewayResponse = (resp) => {
  if (!exports.notNull(resp)) return null;
  let headers = resp.headers || {};
  let resposeData;
  if (exports.notNull(headers["Elements-Next-Page-Token"]) ||
      exports.notNull(headers["Elements-Returned-Count"])) {
    resposeData = {
      data: resp.body,
      nextContinuationToken: headers["Elements-Next-Page-Token"],
      returnedCount: headers["Elements-Returned-Count"]
    };
  } else if(_.isArray(resp.body)) {
    resposeData = {
      data: resp.body,
      returnedCount: resp.body.length
    };
  } else {
    resposeData = resp.body;
  }

  return {
    resposeData: resposeData,
    resposeHeaders: resp.headers,
    status: resp.status
  };
};

/** Morphs the failure object into something that an API Gateway can use. */
const apiGatewayError = (err, context) => {
  if (!exports.notNull(err)) return null;
  if (err.providerMessage && err.providerMessage.length > 1000) {
    err.providerMessage = err.providerMessage.substr(0, 997) + '...';
  }
  return JSON.stringify({
    httpStatus: determineStatusCode(err.statusCode),
    message: 'Request failed',
    providerMessage: err.providerMessage,
    errorType: err.errorType,
    requestId: context.awsRequestId
  });
};

/** Finds and returns a deeply-nested property value. */
exports.getDeepPropValue = (needle, haystack) => {
  let value;

  Object.keys(haystack).forEach( (key) => {
    if(!exports.notNull(value)) {
      if(key === needle) {
        value = haystack[needle];
      } else if(typeof haystack[key] === 'object') {
        value = exports.getDeepPropValue(needle, haystack[key]);
      }
    }
  });

  return value;
};

/** Performs mass token replacment on a javascript object. */
exports.tokenRepl =  (o, tokens) => {

  const subst = (orig, toks) => {
    if (orig.startsWith('{') && orig.indexOf('}') === orig.length - 1) {
      let k = orig.substr(1, orig.length - 2);
      let val = _.get(toks, k);
      // If val is still undefined, try checking in the body
      if(!exports.notNull(val)) { val = _.get(toks, `body.${k}`); }
      if(k === 'pageSize' && (!exports.notNull(val) || isNaN(parseFloat(val)))) { val = 10; } // Add a default pageSize
      if (exports.notNull(val)) { return val; }
      else { return null; }
    }
    return orig.replace(/{([^}]*)}/g, (m, k) => toks[k]);
  };

  const repl = o => {
    if (_.isString(o)) { return subst(o, tokens); }
    else if (_.isObject(o)) {
      return _.each(o, (v, k) => {
        const repd = repl(v);
        if (exports.notNull(repd)) o[k] = repd;
         else {
            if (_.isArray(o)) {
              o.splice(k, 1);
            } else {
              delete o[k];
            }
          }
      });
    }
    else { return o; }
  };

  return repl(o);
};

/** Converts a free-form status code into an acceptable code. */
const determineStatusCode = (code) => {
  const acceptableErrorsCodes = ['200', '400', '401', '403', '404', '405', '409', '500', '502'];

  if(acceptableErrorsCodes.indexOf('' + code) !== -1) {
    return '' + code;
  } else {
    return ('' + code).charAt(0) + '00';
  }
};
