var d3 = require('d3'),
    _  = require('lodash');

var ZoomTransition = function(diameter, allCircles, allLabels) {
  this.zoomToNode = function(targetNode) {
    var k = diameter / (targetNode.r * 2);
    _.each([allCircles, allLabels], function(selection) {
      selection.attr('transform', function(d) {
        return 'translate(' + (d.x - targetNode.x) * k + ',' + (d.y - targetNode.y) * k + ')';
      });
    });
    allCircles.attr('r', function(node) { return k * node.r; });
  };

  this.start = function(sourceNode, targetNode) {
    var self = this;
    var transition = d3.transition()
      .duration(d3.event.altKey ? 5000 : 200)
      .tween('zoom', function() {
        var i = d3.interpolateZoom([sourceNode.x, sourceNode.y, sourceNode.r], [targetNode.x, targetNode.y, targetNode.r]);
        return function(t) {
          var targetView = i(t);
          self.zoomToNode({ x: targetView[0], y: targetView[1], r: targetView[2] });
        };
      });
    return transition;
  };
};

module.exports = function(options) {
  this.bindTo = function(charts, model) {
    var mainChart = _.find(charts, { 'name': 'main' });
    var allCircles = mainChart
      .getComponentByName('node-data')
      .getElement()
      .selectAll('circle');

    var allLabels = mainChart
      .getComponentByName('text-data')
      .getElement()
      .selectAll('text');

    var zoomTransition = new ZoomTransition(options.diameter, allCircles, allLabels);

    allCircles.on('click', function(node) {
      if (!node.isLeaf()) {
        var targetNode = model.currentFocus === node ? model.rootNode : node;
        var transition = zoomTransition.start(model.currentFocus, targetNode);
        model.currentFocus = targetNode;
        transition.on('end', mainChart.updateComponents.bind(mainChart));
      }
      d3.event.stopPropagation();
    });
    zoomTransition.zoomToNode(model.rootNode);
  };
};
