"use strict";
const Vue = require("vue");
const event_hub_1 = require("utils/event-hub");
const merge = require("lodash/merge");
exports.types = {
    SIDEBAR: 'codex/SIDEBAR',
    TOGGLE_SIDEBAR: 'codex/TOGGLE_SIDEBAR',
    CLOSE_SIDEBAR: 'codex/CLOSE_SIDEBAR',
    OPEN_SIDEBAR: 'codex/OPEN_SIDEBAR',
    HIDE_SIDEBAR: 'codex/HIDE_SIDEBAR',
    SHOW_SIDEBAR: 'codex/SHOW_SIDEBAR',
    UPDATE_HEIGHTS: 'codex/UPDATE_HEIGHTS',
    UPDATE_MIN_HEIGHTS: 'codex/UPDATE_MIN_HEIGHTS',
};
function parseBool(val) {
    return val === true || val === 1 || val === 'true' || val === '1';
}
function setCookie(k, v) {
    Vue.cookie.set(k, JSON.stringify(v));
}
function getCookie(k) {
    return JSON.parse(Vue.cookie.get(k));
}
function emit(...params) {
    params[0] = 'store.' + params[0];
    event_hub_1.default.$emit.apply(event_hub_1.default, params);
}
exports.module = {
    state: {
        sidebar: { closed: getCookie('codex.sidebar.closed') || false, hidden: getCookie('codex.sidebar.hidden') || false },
        heights: {
            viewPort: 0,
            header: 0,
            footer: 0,
            page: 0,
            sidebar: 0,
            sidebarInner: 0,
            content: 0,
            document: 0
        },
        minHeights: {
            page: 0,
            document: 0,
            sidebarInner: 0
        }
    },
    mutations: {
        [exports.types.TOGGLE_SIDEBAR](state) {
            state.sidebar.closed = state.sidebar.closed === false;
            setCookie('sidebar.closed', state.sidebar.closed);
            emit(exports.types.TOGGLE_SIDEBAR);
        },
        [exports.types.CLOSE_SIDEBAR](state) {
            state.sidebar.closed = true;
            setCookie('sidebar.closed', state.sidebar.closed);
            emit(exports.types.CLOSE_SIDEBAR);
        },
        [exports.types.OPEN_SIDEBAR](state) {
            state.sidebar.closed = false;
            setCookie('sidebar.closed', state.sidebar.closed);
            emit(exports.types.OPEN_SIDEBAR);
        },
        [exports.types.HIDE_SIDEBAR](state) {
            state.sidebar.hidden = true;
            setCookie('sidebar.hidden', state.sidebar.hidden);
            emit(exports.types.HIDE_SIDEBAR);
        },
        [exports.types.SHOW_SIDEBAR](state) {
            state.sidebar.hidden = false;
            setCookie('sidebar.hidden', state.sidebar.hidden);
            emit(exports.types.SHOW_SIDEBAR);
        },
        [exports.types.UPDATE_HEIGHTS](state, heights) {
            state.heights = merge(state.heights, heights);
            emit(exports.types.UPDATE_HEIGHTS);
        },
        [exports.types.UPDATE_MIN_HEIGHTS](state, heights) {
            state.minHeights = merge(state.minHeights, heights);
            emit(exports.types.UPDATE_MIN_HEIGHTS);
        },
    },
    actions: {
        toggleSidebar({ state, commit, dispatch }, payload = undefined) {
            commit(exports.types.TOGGLE_SIDEBAR);
        },
        closeSidebar({ state, commit, dispatch }, payload = undefined) {
            commit(exports.types.CLOSE_SIDEBAR);
        },
        openSidebar({ state, commit, dispatch }, payload = undefined) {
            commit(exports.types.OPEN_SIDEBAR);
        },
        hideSidebar({ state, commit, dispatch }, payload = undefined) {
            commit(exports.types.HIDE_SIDEBAR);
        },
        showSidebar({ state, commit, dispatch }, payload = undefined) {
            commit(exports.types.SHOW_SIDEBAR);
        },
        updateHeights({ state, commit, dispatch }, payload = undefined) {
            commit(exports.types.UPDATE_HEIGHTS, payload);
            commit(exports.types.UPDATE_MIN_HEIGHTS, {
                page: state.heights.viewPort - state.heights.header - state.heights.footer
            });
        },
    },
    getters: {
        sidebar: state => state.sidebar,
        heights: state => state.heights,
        minHeights: state => state.minHeights
    }
};
//# sourceMappingURL=index.js.map