const Q = require("q");
const axios_1 = require("axios");
const util_1 = require("@radic/util");
const md5_1 = require("./utils/md5");
class Api {
    constructor(options = {}) {
        this.errorMessages = {
            500: `The requested resource doesn't exist!`,
            502: `Server error, please try again.`,
            401: `You aren't authorized to access this resource.`
        };
        this.options = {
            apiUrl: '',
            debug: false,
            axios: {}
        };
        this.requests = {};
        _.merge(this.options, options);
        this.cache = util_1.Storage.getOrCreateBag('codex.api.cache', "local");
        if (parseInt(this.cache.getSize('')) >= 8) {
            this.cache.clear();
        }
        this.$http = axios_1.create(options.axios);
        this.$http.interceptors.request.use(config => {
            config.url = `${this.options.apiUrl}/${config.url}`;
            return config;
        }, this.catchInterceptorError);
        this.$http.interceptors.response.use(response => {
            this.cache.set(response.config.url + this.getParamsHash(response.config.params), response.data, {
                expires: 600000
            });
            return response;
        }, this.catchInterceptorError);
    }
    getParamsHash(params) {
        if (params === undefined)
            return '';
        let objectText = Object.keys(params).map(key => {
            let val = params[key];
            if (typeof val === 'undefined') {
                throw new Error('as this is on key ' + key);
            }
            return key + val.toString();
        }).join('');
        return md5_1.default(objectText);
    }
    resolveAndDelay(deferred, data) {
        setTimeout(() => deferred.resolve(data), this.options.debug ? 500 : 0);
    }
    apiCall(path, options = {}) {
        const deferred = Q.defer();
        const requestKey = `${this.options.apiUrl}/${path}` + this.getParamsHash(options.params);
        const cachedObject = this.cache.get(requestKey, {});
        if (!_.isEmpty(cachedObject)) {
            deferred.resolve(cachedObject);
        }
        else {
            if (this.requests[requestKey] === undefined) {
                this.requests[requestKey] = this.$http.get(path, options);
            }
            this.requests[requestKey]
                .then(({ data }) => {
                delete this.requests[requestKey];
                deferred.resolve(data);
            })
                .catch(this.catchError);
        }
        return deferred.promise;
    }
    spreadData(result, spreadMembers) {
        const data = {};
        spreadMembers.forEach((member, i) => {
            data[member] = result[i];
        });
        return data;
    }
    get(path, options = {}) {
        return this.apiCall(path, options);
    }
    post(path, options) {
        return this.$http.post(path, options);
    }
    all(path, spreadMembers) {
        const deferred = Q.defer();
        const apiCalls = path.map(p => this.apiCall(p));
        axios.all(apiCalls)
            .then(result => deferred.resolve(this.spreadData(result, spreadMembers)))
            .catch(this.catchError);
        return deferred.promise;
    }
    catchInterceptorError(error) {
        return Promise.reject(error);
    }
    catchError(response) {
        let error = {
            code: response.status,
            message: response.message
        };
        const errorMessage = this.errorMessages[response.status];
        if (errorMessage) {
            error = {
                code: 500,
                message: errorMessage
            };
        }
        Promise.reject(error);
    }
}
exports.Api = Api;
//# sourceMappingURL=api.js.map