export default class HashPathParser {
    static Class: string    = 'class';
    static Method: string   = 'method';
    static Property: string = 'property';
    static Source: string   = 'source';

    hashPath: string  = '';
    classPath: string = '';
    type: string      = 'class'
    method: string    = ''
    property: string  = ''
    line: number      = 0

    constructor(hashPath) {
        if ( ! this.isHashPath() ) return;
        let exp     = /^\#\!\/(.*?)(?:$|\:{2}(.*)|\!(.*?)(?:\@(\d*)|$))$/gm
        let matches = [];
        let myArr;
        while ( (myArr = exp.exec(hashPath)) !== null ) { matches.push(myArr); }

        if ( matches.length !== 1 || matches[ 0 ].length !== 5 ) throw new Error('not valid url string');

        let m         = matches[ 0 ];
        this.hashPath = hashPath;
        this.type     = 'class';

        if ( m[ 2 ] !== undefined ) {
            this.type = m[ 2 ].indexOf('()') !== - 1 ? 'method' : 'property';
        }

        if ( m[ 3 ] !== undefined ) {
            this.type = m[ 3 ]; // 'source'
        }

        this.classPath    = m[ 1 ];
        this[ this.type ] = this.type === 'method' ? m[ 2 ].replace('()', '') : m[ 2 ];
        this.line         = parseInt(m[ 4 ]) || 0;
    }

    static parse(hashPath): HashPathParser {
        return new HashPathParser(hashPath);
    }

    is(type: string): boolean { return this.type === type; }

    isClass(): boolean { return this.is(HashPathParser.Class); }

    isMethod(): boolean { return this.is(HashPathParser.Method); }

    isProperty(): boolean { return this.is(HashPathParser.Property); }

    isSource(): boolean { return this.is(HashPathParser.Source); }

    hasLine(): boolean { return this.isSource() && this.line > 0 }

    /** Checks if the current location is a "hashed path" **/
    isHashPath() {
        return location.hash.indexOf('#!/') !== - 1
    }

    /** Returns the current location's hash path **/
    getHashPath() {
        if ( ! this.isHashPath() ) throw new Error('Current location is not a hashed path')
        return location.hash;
    }

    toQuery(): string {
        if ( this.isClass() ) return this.classPath;
        if ( this.isMethod() ) return this.classPath + '::' + this.method + '()';
        if ( this.isProperty() ) return this.classPath + '::' + this.property;
        if ( this.isSource() ) {
            return this.classPath + '::'; // todo
        }
    }

    toString(): string { return this.hashPath; }
}
