/**
 * Imports
 */

import test from 'tape'
import * as hamt from '../src'
import times from '@f/times'

/**
 * Tests
 */

test('should work', t => {
  const p = hamt.set(hamt.empty, 'test', 1)

  t.equal(hamt.get(p, 'test'), 1)

  const p2 = hamt.set(p, 'test2', 2)

  t.equal(hamt.get(p2, 'test'), 1)
  t.equal(hamt.get(p2, 'test2'), 2)
  t.equal(hamt.get(p, 'test'), 1)
  t.equal(hamt.get(p, 'test2'), undefined)

  const p3 = hamt.del(p2, 'test')

  t.equal(hamt.get(p3, 'test'), undefined)
  t.equal(hamt.get(p3, 'test2'), 2)
  t.equal(hamt.get(p2, 'test'), 1)

  t.end()
})

test('branching', t => {
  // These two keys fall into the same first
  // bucket
  const key1 = '6.3.7'
  const key2 = '5.2.3.0.7.0.8.8.3'

  let p = hamt.set(hamt.empty, key1, 1)

  p = hamt.set(p, key2, 2)

  // This stuff caused issues by random trial and error
  p = hamt.set(p, 'test', 3)
  p = hamt.set(p, '2', 2)

  t.equal(hamt.get(p, key1), 1)
  t.equal(hamt.get(p, key2), 2)

  t.end()
})

test('special', t => {
  const key1 = '2.2.2.7.1.1'
  const key2 = '8.4.9'
  const p = hamt.set(hamt.empty, key1, 1)
  const p2 = hamt.set(p, key2, 1)

  t.equal(hamt.get(p, key1), 1)
  t.equal(hamt.get(p2, key1), 1)
  t.end()
})

test('random', t => {
  const strs = unique(times(10000, randomString))
  const p = strs.reduce((p, str, i) => hamt.set(p, str, i), hamt.empty)
  strs.forEach((str, i) => t.equal(hamt.get(p, str), i, str))
  t.end()
})

test('update existing', t => {
  let p = hamt.set(hamt.empty, 'test', 1)
  p = hamt.set(p, 'test', 2)
  t.equal(hamt.get(p, 'test'), 2)
  t.end()
})

test('collisions', t => {
  // Generated by collisions.js in the root directory
  const keys = [
    '8.5.9.7.1.7.7.8', '7.8.8.2.6.8.4.9',
    '0.5.4.2.9.2.3.2', '1.2.5.7.4.1.6.1',
    '3.3.5.2.9.2.6.9.0', '3.2.8.1.4.7.7.6.1',
    '5.4.2.7.2.3.6.9', '1.1.8.4.9.5.9.1',
    '6.6.5.3.9.0.5', '1.1.2.0.3.6.3.6.8',
    '4.8.0.2.1.4.5.7.1', '8.5.2.1.9.9.2',
    '3.3.5.1.5.9.2.7', '4.0.6.6.0.8.5.6',
    '8.7.2.3.1.0.8.2', '4.4.0.6.9.2.4.0.0',
    '2.0.6.9.1.8.7.3', '1.3.5.4.6.9.4.4',
    '3.3.1.5.1.1.5.3', '2.6.0.0.6.2.2.4',
    '5.6.2.9.1.0.6.2', '4.9.1.4.6.1.3.3',
    '1.2.6.8.3.5.2.6.1', '1.3.3.9.8.0.1.9.0',
    '6.3.1.5.1.1.5.8', '2.0.7.2.8.3.8.0',
    '6.3.7.6.2.3.8.2', '5.6.6.1.7.4.5.3'
  ]

  // Throw some other random stuff in there for good measure
  const strs = unique(times(100, randomString).concat(keys))
  const p = strs.reduce((p, str, i) => hamt.set(p, str, i), hamt.empty)

  strs.forEach((str, i) => t.equal(hamt.get(p, str), i, str))
  t.end()
})

test('delete all keys', t => {
  let p = hamt.set(hamt.empty, 'test', 1)
  t.equal(hamt.del(p, 'test'), hamt.empty)
  t.end()
})

/**
 * Helpers
 */

function randomString () {
  var octets = Math.floor(Math.random() * 10)
  var parts = []

  for (var i = 0; i < octets; i++) {
   parts.push(Math.floor(Math.random() * 10))
  }

  return parts.join('.')
}

function unique (items) {
  const seen = {}
  return items.reduce((acc, item) => {
    if (!seen[item]) {
      acc.push(item)
      seen[item] = true
    }

    return acc
  }, [])
}
