'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.empty = exports.del = exports.get = exports.set = undefined;

var _popcount = require('@f/popcount');

var _popcount2 = _interopRequireDefault(_popcount);

var _hashStr = require('@f/hash-str');

var _hashStr2 = _interopRequireDefault(_hashStr);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Constants
 */

/**
 * Imports
 */

var bits = 5;
var size = Math.pow(2, bits);
var mask = size - 1;

/**
 * Types
 */

var LEAF = 'LEAF';
var BRANCH = 'BRANCH';
var COLLISION = 'COLLISION';

/**
 * Mini HAMT
 */

var empty = createBranch();

function set(hamt, key, value) {
  var code = (0, _hashStr2.default)(key);
  return insert(hamt, code, key, value);
}

function insert(node, code, key, value) {
  var depth = arguments.length <= 4 || arguments[4] === undefined ? 0 : arguments[4];

  var frag = getFrag(code, depth);
  var mask = 1 << frag;

  switch (node.type) {
    case LEAF:
      {
        if (node.code === code) {
          if (node.key === key) {
            return createLeaf(code, key, value);
          }

          return createCollision(code, [node, createLeaf(code, key, value)]);
        } else {
          var prevFrag = getFrag(node.code, depth);

          if (prevFrag === frag) {
            // XXX Optimize this
            return createBranch(mask, [insert(insert(empty, code, key, value, depth + 1), node.code, node.key, node.value, depth + 1)]);
          }

          var prevMask = 1 << prevFrag;
          var children = prevFrag < frag ? [node, createLeaf(code, key, value)] : [createLeaf(code, key, value), node];

          return createBranch(mask | prevMask, children);
        }
      }
    case BRANCH:
      {
        var idx = (0, _popcount2.default)(node.mask, frag);
        var children = node.children;

        // If there is already a node for this bit, recurse
        if (node.mask & mask) {
          var child = children[idx];
          return createBranch(node.mask, arrayReplace(children, idx, insert(child, code, key, value, depth + 1)));
        } else {
          return createBranch(node.mask | mask, arrayInsert(children, idx, createLeaf(code, key, value)));
        }
      }
    case COLLISION:
      {
        for (var i = 0, len = node.children.length; i < len; ++i) {
          if (node.children[i].key === key) {
            return createCollision(node.code, arrayReplace(node.children, i, createLeaf(code, key, value)));
          }
        }

        return createCollision(node.code, node.children.concat(createLeaf(code, key, value)));
      }
  }
}

function get(hamt, key) {
  var code = (0, _hashStr2.default)(key);
  var node = hamt;
  var depth = -1;

  while (true) {
    ++depth;

    switch (node.type) {
      case BRANCH:
        {
          var frag = getFrag(code, depth);
          var _mask = 1 << frag;
          if (node.mask & _mask) {
            var idx = (0, _popcount2.default)(node.mask, frag);
            node = node.children[idx];
            continue;
          } else {
            return;
          }
        }
      case COLLISION:
        {
          for (var i = 0, len = node.children.length; i < len; ++i) {
            var child = node.children[i];
            if (child.key === key) {
              return child.value;
            }
          }

          return undefined;
        }
      case LEAF:
        {
          return node.key === key ? node.value : undefined;
        }
    }
  }
}

function del(hamt, key) {
  var code = (0, _hashStr2.default)(key);
  var res = remove(hamt, code, key, 0);
  if (res === undefined) return hamt;
  if (res === null) return empty;
  return res;
}

function remove(node, code, key, depth) {
  var frag = getFrag(code, depth);
  var mask = 1 << frag;

  switch (node.type) {
    case LEAF:
      {
        // null means remove, undefined
        // means do nothing
        return node.key === key ? null : undefined;
      }
    case BRANCH:
      {
        if (node.mask & mask) {
          var idx = (0, _popcount2.default)(node.mask, frag);
          var res = remove(node.children[idx], code, key, depth + 1);
          if (res === null) {
            var newMask = node.mask & ~mask;

            if (newMask === 0) {
              return null;
            } else {
              return createBranch(newMask, arrayRemove(node.children, idx));
            }
          } else if (res === undefined) {
            return undefined;
          } else {
            return createBranch(node.mask, node.children);
          }
        } else {
          return undefined;
        }
      }
    case COLLISION:
      {
        if (node.code === code) {
          for (var i = 0, len = node.children.length; i < len; ++i) {
            var child = node.children[i];

            if (child.key === key) {
              return createCollision(node.code, arrayRemove(node.children, i));
            }
          }
        }

        return undefined;
      }
  }
}

/**
 * Node creators
 */

function createBranch() {
  var mask = arguments.length <= 0 || arguments[0] === undefined ? 0 : arguments[0];
  var children = arguments.length <= 1 || arguments[1] === undefined ? [] : arguments[1];

  return {
    type: BRANCH,
    mask: mask,
    children: children
  };
}

function createCollision(code, children) {
  return {
    type: COLLISION,
    code: code,
    children: children
  };
}

function createLeaf(code, key, value) {
  return {
    type: LEAF,
    code: code,
    key: key,
    value: value
  };
}

/**
 * Helpers
 */

function arrayInsert(arr, idx, item) {
  arr = arr.slice();
  arr.splice(idx, 0, item);
  return arr;
}

function arrayRemove(arr, idx) {
  arr = arr.slice();
  arr.splice(idx, 1);
  return arr;
}

function arrayReplace(arr, idx, item) {
  arr = arr.slice();
  arr[idx] = item;
  return arr;
}

function getFrag(code, depth) {
  return code >>> 4 * depth & mask;
}

/**
 * Exports
 */

exports.set = set;
exports.get = get;
exports.del = del;
exports.empty = empty;