/**
 * Imports
 */

'use strict';

Object.defineProperty(exports, '__esModule', {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

var _objectOmit = require('object.omit');

var _objectOmit2 = _interopRequireDefault(_objectOmit);

/**
 * Action types
 */

var CREATE = 'CREATE_EPHEMERAL';
var UPDATE = 'UPDATE_EPHEMERAL';
var DESTROY = 'DESTROY_EPHEMERAL';

/**
 * Reducer map
 *
 * Note: We store this outside of state because, strictly speaking, it is not *state*.  It is inherent in the UI rendering tree,
 * and as such, is redundant information.  It is also unserializable, so it would be wrong to materialize it into state.
 */

var localReducers = {};

/**
 * Ephemeral state reducer
 */

function reducer(state, action) {
  switch (action.type) {
    case CREATE:
      {
        var locale = action.meta.locale;
        var path = locale.split('.');
        localReducers[locale] = action.payload.reducer;
        return update(path, action.payload.initialState, state);
      }
    case UPDATE:
      {
        var locale = action.meta.locale;
        var path = locale.split('.');
        var _reducer = localReducers[locale];

        return update(path, _reducer(get(path, state), action.payload), state);
      }
    case DESTROY:
      {
        return destroy(action.meta.locale.split('.'), state);
      }
  }

  return state;
}

/**
 * Utilities
 */

function get(path, state) {
  return path.reduce(function (acc, key) {
    return acc[key];
  }, state);
}

function update(path, value, state) {
  if (value === undefined) value = {};

  var key = path[0];

  if (path.length === 0) {
    return value;
  }

  return _extends({}, state, _defineProperty({}, key, update(path.slice(1), value, state[key])));
}

function destroy(path, state) {
  var key = path[0];

  if (path.length === 1) {
    return (0, _objectOmit2['default'])(state, key);
  }

  return _extends({}, state, _defineProperty({}, key, destroy(path.slice(1), state[key])));
}

/**
 * Action creators
 */

function createEphemeral(locale, reducer, initialState) {
  return {
    type: CREATE,
    payload: { initialState: initialState, reducer: reducer },
    meta: { locale: locale }
  };
}

function updateEphemeral(locale, action) {
  return {
    type: UPDATE,
    payload: action,
    meta: { locale: locale }
  };
}

function destroyEphemeral(locale) {
  return {
    type: DESTROY,
    meta: { locale: locale }
  };
}

function computePath(locale) {
  var path = [];

  while (locale) {
    path.push(locale);
    locale = parents[locale];
  }

  return path.reverse();
}

/**
 * Exports
 */

exports['default'] = reducer;
exports.createEphemeral = createEphemeral;
exports.updateEphemeral = updateEphemeral;
exports.destroyEphemeral = destroyEphemeral;