/**
 * Vars
 */

'use strict';

Object.defineProperty(exports, '__esModule', {
  value: true
});
var HANDLE_EVENT = 'HANDLE_EVENT';
var UNHANDLE_EVENT = 'UNHANDLE_EVENT';
var docEvents = ['DOMContentLoaded', 'click'];

/**
 * Events
 */

function eventMiddleware() {
  var globals = arguments.length <= 0 || arguments[0] === undefined ? {} : arguments[0];
  var _globals$wnd = globals.wnd;
  var wnd = _globals$wnd === undefined ? window : _globals$wnd;
  var _globals$doc = globals.doc;
  var doc = _globals$doc === undefined ? document : _globals$doc;

  var map = {};
  var idGen = idGenerator();

  return function (_ref) {
    var dispatch = _ref.dispatch;
    var getState = _ref.getState;
    return function (next) {
      return function (action) {
        return action.type === HANDLE_EVENT || action.type === UNHANDLE_EVENT ? Promise.resolve(handle(dispatch, action)) : next(action);
      };
    };
  };

  function handle(dispatch, action) {
    var _action$payload = action.payload;
    var id = _action$payload.id;
    var event = _action$payload.event;
    var cb = _action$payload.cb;
    var once = _action$payload.once;

    var el = isDocEvent(event) ? doc : wnd;

    switch (action.type) {
      case HANDLE_EVENT:
        id = idGen();
        var fn = compose(maybeDispatch, cb);

        if (once) {
          fn = function (e) {
            maybeDispatch({ type: UNHANDLE_EVENT, payload: { id: id, event: event } });
            maybeDispatch(cb(e));
          };
        }

        if (event === 'domready') {
          var hack = doc.documentElement.doScroll;
          if ((hack ? /^loaded|^c/ : /^loaded|^i|^c/).test(doc.readyState)) {
            fn();
            return;
          }

          event = 'DOMContentLoaded';
        }

        el.addEventListener(event, fn);
        map[id] = fn;
        return id;
      case UNHANDLE_EVENT:
        el.removeEventListener(event, map[id]);
        delete map[id];
        return;
    }

    function maybeDispatch(action) {
      action && dispatch(action);
    }
  }
}

function isDocEvent(evt) {
  return docEvents.indexOf(evt) !== -1;
}

function idGenerator() {
  var id = 0;
  return function () {
    return id++;
  };
}

function compose() {
  for (var _len = arguments.length, fns = Array(_len), _key = 0; _key < _len; _key++) {
    fns[_key] = arguments[_key];
  }

  return fns.reduce(function (memo, fn) {
    return function (arg) {
      return memo(fn(arg));
    };
  }, function (arg) {
    return arg;
  });
}

/**
 * Action creators
 */

function handle(event, cb) {
  return {
    type: HANDLE_EVENT,
    payload: { event: event, cb: cb }
  };
}

function handleOnce(event, cb) {
  return {
    type: HANDLE_EVENT,
    payload: { cb: cb, event: event, once: true }
  };
}

function unhandle(event, id) {
  return {
    type: UNHANDLE_EVENT,
    payload: { event: event, id: id }
  };
}

/**
 * Exports
 */

exports['default'] = eventMiddleware;
exports.handle = handle;
exports.handleOnce = handleOnce;
exports.unhandle = unhandle;