






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * @class apf
 * The Ajax.org Platform.
 *
 * @author    Ruben Daniels (ruben AT ajax DOT org)
 * @version   3.0
 * @default_private
 *
 */
/**
 * @event domready      Fires when the browsers' DOM is ready to be manipulated.
 */
/** 
 * @event movefocus         Fires when the focus moves from one element to another.
 * @param {apf.AmlElement} toElement The element that receives the focus.
 */
/** 
 * @event exit              Fires when the application wants to exit.
 * @cancelable  Prevents the application from exiting. The return value of the event object is displayed in a popup, asking the user for permission.
 */
/** 
 * @event keyup         Fires when the user stops pressing a key (by lifting up)
 * @cancelable Prevents the keypress.
 * @param {Object} e An object containing the following properties:
 *  - keyCode ([[Number]]): The character code of the pressed key.
 *  - ctrlKey ([[Boolean]]): Whether the [[keys: Ctrl]] key was pressed.
 *  - shiftKey ([[Boolean]]): Whether the [[keys: Shift]] key was pressed.
 *  - altKey ([[Boolean]]): Whether the [[keys: Alt]] key was pressed.
 *  - htmlEvent ([[Object]]): The html event object.
 */
/** 
 * @event mousescroll   Fires when the user scrolls the mouse
 * @cancelable Prevents the container from scrolling
 * @param {Object} e An object containing the following properties:
 *  - htmlEvent ([[Object]]): The HTML event object
 *  - amlElement ([[apf.AmlElement]]): The element which was clicked.
 *  - delta ([[Number]]): The scroll impulse.
 */
/** 
 * @event hotkey        Fires when the user presses a hotkey
 * @bubbles
 * @cancelable Prevents the default hotkey behavior.
 * @param {Object} e An object containing the following properties:
 *  - keyCode ([[Number]]): The character code of the pressed key.
 *  - ctrlKey ([[Boolean]]): Whether the [[keys: Ctrl]] key was pressed.
 *  - shiftKey ([[Boolean]]): Whether the [[keys: Shift]] key was pressed.
 *  - altKey ([[Boolean]]): Whether the [[keys: Alt]] key was pressed.
 *  - htmlEvent ([[Object]]): The html event object.
 */
/** 
 * @event keydown       Fires when the user presses down on a key
 * @bubbles
 * @cancelable Prevents the default hotkey behavior.
 * @param {Object} e An object containing the following properties:
 *  - keyCode ([[Number]]): The character code of the pressed key.
 *  - ctrlKey ([[Boolean]]): Whether the [[keys: Ctrl]] key was pressed.
 *  - shiftKey ([[Boolean]]): Whether the [[keys: Shift]] key was pressed.
 *  - altKey ([[Boolean]]): Whether the [[keys: Alt]] key was pressed.
 *  - htmlEvent ([[Object]]): The html event object.
 */
/** 
 * @event mousedown     Fires when the user presses a mouse button
 * @param {Object} e An object containing the following properties:
 *  - htmlEvent ([[Object]]): The HTML event object
 *  - amlElement ([[apf.AmlElement]]): The element which was clicked.
 */
/** 
 * @event onbeforeprint Fires before the application prints.
 */
/** 
 * @event onafterprint  Fires after the application prints.
 */
/** 
 * @event load          Fires after the application is loaded.
 */
/** 
 * @event error         Fires when a communication error has occured while making a request for this element.
 * @cancelable Prevents the error from being thrown.
 * @bubbles
 * @param {Object} e An object containing the following properties:
 *   - error ([[Error]]): The error object that is thrown when the event's callback doesn't return `false`.
 *   - state ([[Number]]): The state of the call. Possible values include:
 *     - `apf.SUCCESS` : The request was successful
 *     - `apf.TIMEOUT`:  The request timed out
 *     - `apf.ERROR`:    An error occurred while making the request
 *     - `apf.OFFLINE`:  The request was made while the application was offline.
 *   - userdata (`Mixed`): Data that the caller made available in the callback of the HTTP request.
 *   - http ([[XMLHttpRequest]]): The object that executed the actual http request
 *   - url ([[String]]): The url that was requested
 *   - tpModule ([[apf.http]]): The teleport module that is making the request
 *   - id ([[Number]]): The id of the request
 *   - message ([[String]]): The error message
 * 
 */
 apf = {
VERSION:'3.0beta',
    // Content Distribution Network URL:
    
    /**
     * The url to the content delivery network.
     * @type {String}
     */
    CDN            : "",
    

    /**
     * Specifies whether apf is ready for DOM operations.
     * @type {Boolean}
     */
    READY          : false,

    //AML nodeFunc constants
    /**
     * A constant for the hidden AML element.
     * @type {Number}
     */
    NODE_HIDDEN    : 101,
    /**
     * A constant for a visible AML element.
     * @type {Number}
     */
    NODE_VISIBLE   : 102,
    /**
     * A constant for an o3 widget.
     * @type {Number}
     */
    NODE_O3 : 103,

    /**
     * A constant for specifying that a widget is using only the keyboard to receive focus.
     * @type {Number}
     * @see apf.GuiElement@focus
     */
    KEYBOARD       : 2,
    /**
     * A constant for specifying that a widget is using the keyboard or the mouse to receive focus.
     * @type {Boolean}
     * @see apf.GuiElement@focus
     */
    KEYBOARD_MOUSE : true,
    /**
     * A constant for specifying that a widget is a menu
     * @type {Number}
     */
    MENU           : 3,

    /**
     * A constant for specifying success.
     * @type {Number}
     */
    SUCCESS : 1,
    /**
     * A constant for specifying a timeout.
     * @type {Number}
     */
    TIMEOUT : 2,
    /**
     * A constant for specifying an error.
     * @type {Number}
     */
    ERROR   : 3,
    /**
     * A constant for specifying the application is offline.
     * @type {Number}
     */
    OFFLINE : 4,

    
    debug         : false,
    

    includeStack  : [],
    initialized   : false,
    AppModules    : [],
    
    /**
     * Specifies whether APF tries to load a skin from skins.xml when no skin element is specified.
     * @type {Boolean}
     */
    autoLoadSkin  : false,
    /**
     * Specifies whether APF has started loading scripts and started the init process.
     * @type {Boolean}
     */
    started       : false,
    /**
     * The namespace for all crypto libraries included with Ajax.org Platform.
     * @type {Object}
     */
    crypto        : {}, //namespace
    config        : {},
    _GET          : {},
    $asyncObjects : {"apf.oHttp" : 1, "apf.ajax": 1},
    
    /**
     * A string specifying the basepath for loading APF from seperate files.
     * @type {String}
     */
    basePath      : "",

    
    /**
     * Contains several known and often used namespace URI's.
     * @type {Object}
     * @private
     */
    ns : {
        apf    : "http://ajax.org/2005/aml",
        aml    : "http://ajax.org/2005/aml",
        xsd    : "http://www.w3.org/2001/XMLSchema",
        xhtml  : "http://www.w3.org/1999/xhtml",
        xslt   : "http://www.w3.org/1999/XSL/Transform",
        xforms : "http://www.w3.org/2002/xforms",
        ev     : "http://www.w3.org/2001/xml-events"
    },
    
    
    xPathAxis  : {"self":1, "following-sibling":1, "ancestor":1}, //@todo finish list
    
    hasRequireJS : typeof requirejs !== "undefined",

    availHTTP  : [],
    /**
     * @private
     */
    releaseHTTP: function(http){
        if (apf.brokenHttpAbort) 
            return;
        if (self.XMLHttpRequestUnSafe && http.constructor == XMLHttpRequestUnSafe) 
            return;
        
        http.onreadystatechange = function(){};
        
        http.abort();
        this.availHTTP.push(http);
    },

    /**
     * @private
     */
    browserDetect : function(){
        if (this.$bdetect)
            return;
        
        /* Browser -  platform and feature detection, based on prototype's and mootools 1.3.
         *
         * Major browser/engines flags
         *
         * 'Browser.name' reports the name of the Browser as string, identical to the property names of the following Boolean values:
         *  - Browser.ie - (boolean) True if the current browser is Internet Explorer.
         *  - Browser.firefox - (boolean) True if the current browser is Firefox.
         *  - Browser.safari - (boolean) True if the current browser is Safari.
         *  - Browser.chrome - (boolean) True if the current browser is Chrome.
         *  - Browser.opera - (boolean) True if the current browser is Opera.
         *
         * In addition to one of the above properties a second property consisting of the name
         * and the major version is provided ('Browser.ie6', 'Browser.chrome15', ...).
         * If 'Browser.chrome' is True, all other possible properties, like 'Browser.firefox', 'Browser.ie', ... , will be undefined.
         *
         * 'Browser.version' reports the version of the Browser as number.
         *
         * 'Browser.Plaform' reports the platform name:
         *  - Browser.Platform.mac - (boolean) True if the platform is Mac.
         *  - Browser.Platform.win - (boolean) True if the platform is Windows.
         *  - Browser.Platform.linux - (boolean) True if the platform is Linux.
         *  - Browser.Platform.ios - (boolean) True if the platform is iOS.
         *  - Browser.Platform.android - (boolean) True if the platform is Android
         *  - Browser.Platform.webos - (boolean) True if the platform is WebOS
         *  - Browser.Platform.other - (boolean) True if the platform is neither Mac, Windows, Linux, Android, WebOS nor iOS.
         *  - Browser.Platform.name - (string) The name of the platform.
         */
        var Browser = this.$bdetect = (function() {
            
            var ua       = navigator.userAgent.toLowerCase(),
                platform = navigator.platform.toLowerCase(),
                UA       = ua.match(/(opera|ie|firefox|chrome|version)[\s\/:]([\w\d\.]+)?.*?(safari|version[\s\/:]([\w\d\.]+)|$)/) || [null, 'unknown', 0],
                mode     = UA[1] == 'ie' && document.documentMode;

            var b = {

                name: (UA[1] == 'version') ? UA[3] : UA[1],

                version: mode || parseFloat((UA[1] == 'opera' && UA[4]) ? UA[4] : UA[2]),

                Platform: {
                    name: ua.match(/ip(?:ad|od|hone)/) ? 'ios' : (ua.match(/(?:webos|android)/) || platform.match(/mac|win|linux/) || ['other'])[0]
                },

                Features: {
                    xpath: !!(document.evaluate),
                    air:   !!(window.runtime),
                    query: !!(document.querySelector),
                    json:  !!(window.JSON)
                },

                Plugins: {}
            };

            b[b.name] = true;
            b[b.name + parseInt(b.version, 10)] = true;
            b.Platform[b.Platform.name] = true;
            
            return b;
            
        })();

        var UA = navigator.userAgent.toLowerCase();
        
        this.isGecko       = !!Browser.firefox;
        this.isChrome      = !!Browser.chrome;
        this.isSafari      = !!Browser.safari;
        this.isSafariOld   = Browser.safari && Browser.version === 2.4;
        this.isWebkit      = this.isSafari || this.isChrome || UA.indexOf("konqueror") != -1;
        this.isOpera       = !!Browser.opera;
        this.isIE          = !!Browser.ie;
        
        this.isWin         = Browser.Platform.win;
        this.isMac         = Browser.Platform.mac;
        this.isLinux       = Browser.Platform.linux;
        this.isIphone      = Browser.Platform.ios || UA.indexOf("aspen simulator") != -1;
        this.isAIR         = Browser.Features.air;
        
        // @deprecated, cleanup in apf modules 
        this.versionWebkit = this.isWebkit ? Browser.version : null;
        this.versionGecko  = this.isGecko ? Browser.version : null;
        // @deprecated, cleanup in apf modules 
        this.isGecko3      = Browser.firefox3;
        this.isGecko35     = this.isGecko3 && Browser.version >= 3.5;
        // @deprecated, cleanup in apf modules 
        this.versionFF     = this.isGecko ? Browser.version : null;
        this.versionSafari = this.isSafari ? Browser.version : null;
        this.versionChrome = this.isChrome ? Browser.version : null;
        this.versionOpera  = this.isOpera ? Browser.version : null;
        // bad logic, needs review among apf modules 
        this.isIE6         = this.isIE && Browser.ie6;
        this.isIE7         = this.isIE && Browser.ie7;
        this.isIE8         = this.isIE && Browser.ie8;
        this.isIE7Emulate  = this.isIE && document.documentMode && Browser.ie7;
        this.isIE          = this.isIE ? Browser.version : null;

        
        
        
    },

    /**
     * @private
     */
    setCompatFlags : function(){
        //Set Compatibility
        this.TAGNAME                   = apf.isIE ? "baseName" : "localName";
        this.styleSheetRules           = apf.isIE ? "rules" : "cssRules";
        this.brokenHttpAbort           = apf.isIE6;
        this.canUseHtmlAsXml           = apf.isIE;
        this.supportNamespaces         = !apf.isIE;
        this.cannotSizeIframe          = apf.isIE;
        this.hasConditionCompilation   = apf.isIE;
        this.supportOverflowComponent  = apf.isIE;
        // http://robertnyman.com/2010/12/02/css3-flexible-box-layout-module-aka-flex-box-introduction-and-demostest-cases/
        this.hasFlexibleBox            = apf.versionGecko >= 3 || (apf.isWebkit && apf.versionWebkit >= 3.2);
        this.hasFileApi                = !!(window["File"] && window["FileReader"] && window["Blob"] && window["FileError"]);
        this.hasEventSrcElement        = apf.isIE;
        this.canHaveHtmlOverSelects    = !apf.isIE6 && !apf.isIE5;
        this.hasInnerText              = apf.isIE;
        this.hasMsRangeObject          = apf.isIE;
        this.descPropJs                = apf.isIE;
        this.hasClickFastBug           = apf.isIE;
        this.hasExecScript             = window.execScript ? true : false;
        this.canDisableKeyCodes        = apf.isIE;
        this.hasTextNodeWhiteSpaceBug  = apf.isIE || apf.isIE >= 8;
        this.hasCssUpdateScrollbarBug  = apf.isIE;
        this.canUseInnerHtmlWithTables = !apf.isIE;
        this.hasSingleResizeEvent      = !apf.isIE;
        this.hasStyleFilters           = apf.isIE;
        this.supportOpacity            = !apf.isIE || apf.isIE >= 9;
        this.supportPng24              = !apf.isIE6 && !apf.isIE5;
        this.cantParseXmlDefinition    = apf.isIE50;
        this.hasDynamicItemList        = !apf.isIE || apf.isIE >= 7;
        this.canImportNode             = apf.isIE;
        this.hasSingleRszEvent         = !apf.isIE;
        this.hasXPathHtmlSupport       = !apf.isIE;
        this.hasFocusBug               = apf.isIE;
        this.hasHeightAutoDrawBug      = apf.isIE && apf.isIE < 8;
        //this.hasIndexOfNodeList        = !apf.isIE;
        this.hasReadyStateBug          = apf.isIE50;
        this.dateSeparator             = apf.isIE ? "-" : "/";
        this.canCreateStyleNode        = !apf.isIE;
        this.supportFixedPosition      = !apf.isIE || apf.isIE >= 7;
        this.hasHtmlIdsInJs            = apf.isIE && apf.isIE < 8 || apf.isWebkit;
        this.needsCssPx                = !apf.isIE;
        this.hasCSSChildOfSelector     = !apf.isIE || apf.isIE >= 8;
        this.hasStyleAnchors           = !apf.isIE || apf.isIE >= 8;
        this.styleAttrIsObj            = apf.isIE < 8;
        this.hasAutocompleteXulBug     = apf.isGecko;
        this.loadsLocalFilesSync       = apf.isIE || apf.isGecko;
        this.mouseEventBuffer          = apf.isIE ? 20 : 6;
        this.hasComputedStyle          = typeof document.defaultView != "undefined"
                                           && typeof document.defaultView.getComputedStyle != "undefined";
        this.w3cRange                  = Boolean(window["getSelection"]);
        this.locale                    = (apf.isIE
                                            ? navigator.userLanguage
                                            : navigator.language).toLowerCase();
        this.characterSet              = document.characterSet || document.defaultCharset || "utf-8";
        var t = document.createElement("div");
        this.hasContentEditable        = (typeof t.contentEditable == "string"
                                       || typeof t.contentEditable == "boolean");
        apf.hasContentEditableContainerBug = apf.isWebkit;
        // Try transform first for forward compatibility
        var props   = ["transform", "OTransform", "KhtmlTransform", "MozTransform", "WebkitTransform"],
            props2  = ["transition", "OTransition", "KhtmlTransition", "MozTransition", "WebkitTransition"],
            prefixR = ["", "O", "Khtml", "Moz", "Webkit"],
            prefixC = ["", "o-", "khtml-", "moz-", "webkit-"],
            events  = ["transitionend", "transitionend", "transitionend", "transitionend", "webkitTransitionEnd"],
            i       = 0,
            l       = 5;
        this.supportCSSAnim            = false;
        this.supportCSSTransition      = false;
        for (; i < l && !this.supportCSSAnim; ++i) {
            if (typeof t.style[props[i]] == "undefined") continue;
            this.supportCSSAnim     = props[i];
            this.supportCSSTransition = props2[i];
            this.runtimeStylePrefix = prefixR[i];
            this.classNamePrefix    = prefixC[i];
            this.cssAnimEvent       = events[i];
        }
        t = null;
        delete t;

        this.supportVML                = apf.isIE;
        this.supportSVG                = !apf.isIE || apf.isIE > 8;
        this.supportCanvas             = !!document.createElement("canvas").getContext;
        this.supportCanvasText         = !!(this.supportCanvas
            && typeof document.createElement("canvas").getContext("2d").fillText == "function")

        this.hasVideo                  = !!document.createElement("video")["canPlayType"];
        this.hasAudio                  = !!document.createElement("audio")["canPlayType"];
        this.supportHashChange         = ("onhashchange" in self) && (!apf.isIE || apf.isIE >= 8);

        if (self.XMLHttpRequest) {
            var xhr = new XMLHttpRequest();
            this.hasXhrProgress = !!xhr.upload;
            if (this.hasXhrBinary = !!(xhr.sendAsBinary || xhr.upload)) {
                this.hasHtml5File      = !!(File && File.prototype.getAsDataURL);
                this.hasHtml5FileSlice = !!(File && File.prototype.slice);
            }
        }
        else {
            this.hasXhrProgress = this.hasXhrBinary = this.hasHtml5File 
                = this.hasHtml5FileSlice = false;
        }

        this.windowHorBorder           = 
        this.windowVerBorder           = apf.isIE8 && (!self.frameElement 
            || parseInt(self.frameElement.frameBorder)) ? 4 : 0;
        
        

        this.enableAnim   = !apf.isIE || apf.isIE > 8;
        this.animSteps    = apf.isIE ? 0.3 : 1;
        this.animInterval = apf.isIE ? 7 : 1;

        this.CSSFLOAT    = apf.isIE ? "styleFloat" : "cssFloat";
        this.CSSPREFIX   = apf.isGecko ? "Moz" : (apf.isWebkit ? "webkit" : "");
        this.CSSPREFIX2  = apf.isGecko ? "-moz" : (apf.isWebkit ? "-webkit" : "");
        this.INLINE      = apf.isIE && apf.isIE < 8 ? "inline" : "inline-block";
        this.needZoomForLayout = apf.isIE && apf.isIE < 8;

        //Other settings
        this.maxHttpRetries = apf.isOpera ? 0 : 3;

        
        this.percentageMatch = new RegExp();
        this.percentageMatch.compile("([\\-\\d\\.]+)\\%", "g");
        
        
        this.reMatchXpath = new RegExp();
        this.reMatchXpath.compile("(^|\\|)(?!\\@|[\\w-]+::)", "g");

        
    },

    hasGeoLocation: function() {
        
        return false;
        
    },

    

    /**
     * Extends an object with one or more other objects by copying all of its
     * properties.
     * @param {Object} dest The destination object
     * @param {Object} src The object that is copied from
     * @return {Object} The destination object
     */
    extend : function(dest, src){
        var prop, i, x = !dest.notNull;
        if (arguments.length == 2) {
            for (prop in src) {
                if (x || src[prop])
                    dest[prop] = src[prop];
            }
            return dest;
        }

        for (i = 1; i < arguments.length; i++) {
            src = arguments[i];
            for (prop in src) {
                if (x || src[prop])
                    dest[prop] = src[prop];
            }
        }
        return dest;
    },
    
    $extend : function(dest, src){
        for (var prop in src) {
            dest[prop] = src[prop];
        }
        return dest;
    },
    
    
    /**
     * Sends and retrieves data from remote locations over http.
     * 
     * #### Example
     * 
     * ```javascript
     *  var content = apf.ajax("http://www.ajax.org", {
     *      method   : "POST",
     *      data     : "<data />",
     *      async    : false,
     *      callback : function( data, state ) {
     *          if (state == apf.SUCCESS)
     *              alert("Success");
     *          else
     *              alert("Failure")
     *      }
     *  });
     *  alert(content);
     * ```
     *
     * @param {String}   url       The url that is accessed.
     * @param {Object}   options   The options for the HTTP request. It has the following properties:
     *   - async ([[Boolean]]): Whether the request is sent asynchronously. Defaults to true.
     *   - userdata (`Mixed`): Custom data that is available to the callback function.
     *   - method ([[String]]): The request method (`POST`|`GET`|`PUT`|`DELETE`). Defaults to `GET`.
     *   - nocache ([[Boolean]]): Specifies whether browser caching is prevented.
     *   - data ([[String]]): The data sent in the body of the message.
     *   - useXML ([[Boolean]]): Specifies whether the result should be interpreted as xml.
     *   - autoroute ([[Boolean]]): Specifies whether the request can fallback to a server proxy.
     *   - caching ([[Boolean]]): Specifies whether the request should use internal caching.
     *   - ignoreOffline ([[Boolean]]): Specifies whether to ignore offline catching.
     *   - contentType ([[String]]): The MIME type of the message
     *   - callback ([[Function]]): The handler that gets called whenever the
     *                            request completes succesfully, with an error, or times out.
     */
    ajax : (function(){
        var f = function(){
            return this.oHttp.get.apply(this.oHttp, arguments);
        };
        
        f.exec = function(method, args, callback, options){
            if (method == "ajax" && args[0]) {
                var opt = args[1] || {};
                return this.oHttp.exec(opt.method || "GET", [args[0]], 
                    opt.callback, apf.extend(options || {}, opt));
            }
        };

        return f;
    })(),
    

    /**
     * Starts the application.
     * @private
     */
    start : function(){
        this.started = true;
        var sHref = location.href.split("#")[0].split("?")[0];

        //Set Variables
        this.host     = location.hostname && sHref.replace(/(\/\/[^\/]*)\/.*$/, "$1");
        this.hostPath = sHref.replace(/\/[^\/]*$/, "") + "/";

        

        //mozilla root detection
        //try{ISROOT = !window.opener || !window.opener.apf}catch(e){ISROOT = true}

        //Browser Specific Stuff
        //this.browserDetect();
        this.setCompatFlags();

        if (apf.onstart && apf.onstart() === false)
            return false;

        

        //Load Browser Specific Code
        
        if (this.isIE) apf.runIE();
            //this.importClass(apf.runIE, true, self);
        
        
        if (apf.isWebkit) apf.runWebkit();
            //this.importClass(apf.runSafari, true, self);
        
        
        
        if (this.isGecko || !this.isIE && !apf.isWebkit && !this.isOpera)
            apf.runGecko();
            //this.importClass(apf.runGecko, true, self);
        
        
        

        
        // Start HTTP object
        this.oHttp = new this.http();
        
        
        
        // Load user defined includes
        this.Init.addConditional(this.parseAppMarkup, apf, ["body"]);
        //@todo, as an experiment I removed 'HTTP' and 'Teleport'
        

        //IE fix
        try {
            if (apf.isIE)
                document.execCommand("BackgroundImageCache", false, true);
        }
        catch(e) {}

        
        //apf.window.init();
        

        this.started = true;
        
        
        // DOMReady already fired, so plz continue the loading and parsing
        if (this.load_done)
            this.execDeferred();
        

        //try{apf.root = !window.opener || !window.opener.apf;}
        //catch(e){apf.root = false}
        this.root = true;
        
        
        for (var i = 0; i < apf.$required.length; i++) {
            apf.include(apf.$required[i]);
        }
        apf.require = apf.include;
        
        
        

    },

    nsqueue   : {},

    
    /**
     * @private
     */
    findPrefix : function(xmlNode, xmlns){
        var docEl;
        if (xmlNode.nodeType == 9) {
            if (!xmlNode.documentElement)
                return false;
            if (xmlNode.documentElement.namespaceURI == xmlns)
                return xmlNode.prefix || xmlNode.scopeName;
            docEl = xmlNode.documentElement;
        }
        else {
            if (xmlNode.namespaceURI == xmlns)
                return xmlNode.prefix || xmlNode.scopeName;
            docEl = xmlNode.ownerDocument.documentElement;
            if (docEl && docEl.namespaceURI == xmlns)
                return xmlNode.prefix || xmlNode.scopeName;

            while (xmlNode.parentNode) {
                xmlNode = xmlNode.parentNode;
                if (xmlNode.namespaceURI == xmlns)
                    return xmlNode.prefix || xmlNode.scopeName;
            }
        }

        if (docEl) {
            for (var i = 0; i < docEl.attributes.length; i++) {
                if (docEl.attributes[i].nodeValue == xmlns)
                    return docEl.attributes[i][apf.TAGNAME]
            }
        }

        return false;
    },
    

    /**
     * @private
     */
    importClass : function(ref, strip, win){
        if (!ref)
            throw new Error(apf.formatErrorString(1018, null,
                "importing class",
                "Could not load reference. Reference is null"));

        if (!strip)
            return apf.jsexec(ref.toString(), win);

        var q = ref.toString().replace(/^\s*function\s*\w*\s*\([^\)]*\)\s*\{/, "")
                              .replace(/\}\s*$/, "");

        return apf.jsexec(q, win);
    },

    /**
    * This method returns a string representation of the object
    * @return {String}    Returns a string representing the object.
    */
    toString : function(){
        return "[Ajax.org Platform (apf)]";
    },

    all : [],

    /**
    * This method implements all the properties and methods to this object from another class
    * @param {Function}    classRef    The class reference to implement
    * @private
    */
    implement : function(classRef) {
        // for speed, we check for the most common  case first
        if (arguments.length == 1) {
            
            classRef.call(this);//classRef
        }
        else {
            for (var a, i = 0, l = arguments.length; i < l; i++) {
                a = arguments[i];
                
                arguments[i].call(this);//classRef
            }
        }

        return this;
    },

    /**
     * @private
     */
    uniqueHtmlIds : 0,

    /**
     * Adds a unique id attribute to an HTML element.
     * @param {HTMLElement} oHtml the object getting the attribute.
     */
    setUniqueHtmlId : function(oHtml){
        var id;
        oHtml.setAttribute("id", id = "q" + this.uniqueHtmlIds++);
        return id;
    },

    /**
     * Retrieves a new unique id
     * @returns {Number} A number representing the new ID.
     */
    getUniqueId : function(){
        return this.uniqueHtmlIds++;
    },

    /**
     * Finds an AML element based on its unique id.
     * @param {Number} uniqueId The unique id to search on.
     * @returns {apf.AmlElement} The returned element.
     */
    lookup : function(uniqueId){
        return this.all[uniqueId];
    },

    /**
     * Searches in the HTML tree from a certain point to find the
     * AML element that is responsible for rendering a specific html
     * element.
     * @param {HTMLElement} oHtml The html context to start the search from.
     * @returns {apf.AmlElement} The parent HTML element
     */
    findHost : function(o){
        while (o && o.parentNode) { //!o.host && 
            try {
                if ((o.host || o.host === false) && typeof o.host != "string")
                    return o.host;
            }
            catch(e){}
            
            o = o.parentNode;
        }
        
        return null;
    },

    /**
     * Sets a reference to an object (by name) in the global JavaScript space.
     * @param {String} name The name of the reference.
     * @param {Mixed}  o    The reference to the object subject to the reference.
     */
    setReference : function(name, o){
        return self[name] && self[name].hasFeature
            ? 0
            : (self[name] = o);
    },

    /*
     * The console outputs to the debug screen and offers differents ways to do
     * this.
     */
    console : {
        

        /**
         * Writes a message to the console.
         * @param {String} msg      The message to display in the console.
         * @param {String} subtype  The category for this message. This is used for filtering the messages.
         * @param {String} data     Extra data that might help in debugging.
         */
        debug : function(msg, subtype, data){
            
        },

        /**
         * Writes a message to the console with the time icon next to it.
         * @param {String} msg      The message to display in the console.
         * @param {String} subtype  The category for this message. This is used for filtering the messages.
         * @param {String} data     Extra data that might help in debugging.
         */
        time : function(msg, subtype, data){
            
        },

        /**
         * Writes a message to the console.
         * @param {String} msg      The message to display in the console.
         * @param {String} subtype  The category for this message. This is used for filtering the messages.
         * @param {String} data     Extra data that might help in debugging.
         */
        log : function(msg, subtype, data){
            
        },

        /**
         * Writes a message to the console with the visual "info" icon and color
         * coding.
         * @param {String} msg      The message to display in the console.
         * @param {String} subtype  The category for this message. This is used for filtering the messages.
         * @param {String} data     Extra data that might help in debugging.
         */
        info : function(msg, subtype, data){
            
        },

        /**
         * Writes a message to the console with the visual "warning" icon and
         * color coding.
         * @param {String} msg      The message to display in the console.
         * @param {String} subtype  The category for this message. This is used for filtering the messages.
         * @param {String} data     Extra data that might help in debugging.
         */
        warn : function(msg, subtype, data){
            
        },

        /**
         * Writes a message to the console with the visual "error" icon and
         * color coding.
         * @param {String} msg      The message to display in the console.
         * @param {String} subtype  The category for this message. This is used for filtering the messages.
         * @param {String} data     Extra data that might help in debugging.
         */
        error : function(msg, subtype, data){
            
        },

        /**
         * Prints a listing of all properties of the object.
         * @param {Mixed} obj The object whose properties you want displayed.
         */
        dir : function(obj){
            var s = apf.$debugwin.$serializeObject(obj, "Inspected via apf.console.dir");
            if (typeof s == "string") {
                this.write(s, "custom", null, null, null, true);
            }
            else {
                this.write(obj
                    ? "Could not serialize object: " + s.message
                    : obj, "error", null, null, null, true);
            }
            
            //this.info(apf.vardump(obj, null, false).replace(/ /g, "&nbsp;").replace(/</g, "&lt;"));
        }
        
        
    },

    html_entity_decode : function(s){return s},
    htmlentities : function(s){return s},

    /**
     * Formats an Ajax.org Platform error message.
     * @param {Number}      number      The number of the error. This can be used to look up more information about the error.
     * @param {apf.AmlElement}  control     The aml element that will throw the error.
     * @param {String}      process     The action that was being executed.
     * @param {String}      message     The actual error message.
     * @param {XMLElement}  amlContext  The XML relevant to the error. For instance, this could be a piece of Ajax.org Markup Language XML.
     */
    formatErrorString : function(number, control, process, message, amlContext, outputname, output){
        
        apf.lastErrorMessage = message;
        return message;
        
    },

    /* Init */

    /**
     * Returns the directory portion of a URL.
     * @param {String} url The URL to retrieve from
     * @returns {String} The directory portion of a URL
     */
    getDirname : function(url){
        //(?:\w+\:\/\/)?
        return ((url || "").match(/^([^#]*\/)[^\/]*(?:$|\#)/) || {})[1]; //Mike will check out how to optimize this line
    },
    
    /**
     * Returns the file portion of a URL.
     * @param {String} url The URL to retrieve from.
     * @return {String} The file portion of a URL.
     */
    getFilename : function(url){
        return ((url || "").split("?")[0].match(/(?:\/|^)([^\/]+)$/) || {})[1];
    },
    
    /**
     * Returns an absolute url based on url.
     * @param {String} base The start of the URL where relative URLs work.
     * @param {String} url  The URL to transform.
     * @return {String} The absolute URL.
     */
    getAbsolutePath : function(base, url){
        return url && url.charAt(0) == "/"
            ? url
            : (!url || !base || url.match(/^\w+\:\/\//) ? url : base.replace(/\/$/, "") + "/" + url.replace(/^\//, ""));
    },
    
    getCtrlKey : function(event){
        return apf.isMac ? event.metaKey : event.ctrlKey;
    },

    /**
     * Loads Javascript from a specific URL.
     * 
     * @param {String}    sourceFile The URL where the JavaScript is located.
     * @param {Boolean}   [doBase]   Checks for a base path via [[apf.getAbsolutePath]]
     * @param {String}    [type]     Sets the type of a script tag, for later use
     * @param {String}    [text]     
     * @param {Function}  [callback] Calls this function after the script is loaded
     * @returns {String} The constructed script tag    
     */
    include : function(sourceFile, doBase, type, text, callback){
        
        
        var sSrc = doBase ? apf.getAbsolutePath(apf.basePath || "", sourceFile) : sourceFile;
        var head     = document.getElementsByTagName("head")[0],//$("head")[0]
            elScript = document.createElement("script");
        //elScript.defer = true;
        if (type)
            elScript.setAttribute("_apf_type", type);
        if (text) {
            if (apf.isIE)
                window.execScript(text);
            else
                elScript.text = text;
        }
        else 
            elScript.src   = sSrc;
        head.appendChild(elScript);

        if (callback)
            elScript[apf.isIE ? "onreadystatechange" : "onload"] = callback;
        
        return elScript;
    },
    
    $required : [],
    require : function(){
        var dir = apf.getDirname(location.href),
            i   = 0,
            l   = arguments.length;
        for (; i < l; i++)
            this.$required.push(apf.getAbsolutePath(dir, arguments[i]));
    },

    /**
     * @private
     */
    Init : {
        queue : [],
        cond  : {
            combined : []
        },
        done  : {},

        add   : function(func, o){
            if (this.inited)
                func.call(o);
            else if (func)
                this.queue.push([func, o]);
        },

        addConditional : function(func, o, strObj){
            if (typeof strObj != "string") {
                if (this.checkCombined(strObj))
                    return func.call(o);
                this.cond.combined.push([func, o, strObj]);
            }
            else if (self[strObj]) {
                func.call(o);
            }
            else {
                if (!this.cond[strObj])
                    this.cond[strObj] = [];
                this.cond[strObj].push([func, o]);

                this.checkAllCombined();
            }
        },

        checkAllCombined : function(){
            for (var i = 0; i < this.cond.combined.length; i++) {
                if (!this.cond.combined[i]) continue;

                if (this.checkCombined(this.cond.combined[i][2])) {
                    this.cond.combined[i][0].call(this.cond.combined[i][1])
                    this.cond.combined[i] = null;
                }
            }
        },
        
        checkCombined : function(arr){
            for (var i = 0; i < arr.length; i++) {
                if (!this.done[arr[i]])
                    return false;
            }

            return true;
        },

        run : function(strObj){
            this.inited = this.done[strObj] = true;

            this.checkAllCombined();

            var data = strObj ? this.cond[strObj] : this.queue;
            if (!data) return;
            for (var i = 0; i < data.length; i++)
                data[i][0].call(data[i][1]);
        }
    },

    
    
    

    /**
     * Determines the way APF tries to render this application. Set this value
     * before APF is starts parsing.
     *   
     * Possible values include:
     *   - 0    Auto (The default)
     *   - 1    Partial
     *   - 11   Partial from a comment
     *   - 2    Full from serialized document or file fallback
     *   - 21   Full from file
     * @type {Number}
     */
    parseStrategy : 0,

    

    /**
     * @private
     */
    parseAppMarkup : function(docElement){
        var isEmptyDocument = false;
        
        if (document.documentElement.getAttribute("skipParse") == "true") {
            return;
        }
        
        

        

        

        
    },
    
    namespaces : {},
    setNamespace : function(namespaceURI, oNamespace){
        this.namespaces[namespaceURI] = oNamespace;
        oNamespace.namespaceURI = namespaceURI;
    },

    /**
     * @private
     */
    initialize : function(xmlStr){
        
        
        

        apf.console.info("Initializing...");
        clearInterval(apf.Init.interval);

        // Run Init
        apf.Init.run(); //Process load dependencies
        
        
        
        var bodyMarginTop = parseFloat(apf.getStyle(document.body, "marginTop"));
        apf.doesNotIncludeMarginInBodyOffset = (document.body.offsetTop !== bodyMarginTop);

        
        {
            apf.window.init(xmlStr);
        }
    },

    
    /**
     * @private
     */
    execDeferred: function() {
        // execute each function in the stack in the order they were added
        var len = apf.load_events.length;
        while (len--)
            (apf.load_events.shift())();
    },

    load_events: [],
    load_timer : null,
    load_done  : false,
    load_init  : null,

    /**
     * @private
     */
    addDomLoadEvent: function(func) {
        if (!this.$bdetect)
            this.browserDetect();

        if (apf.load_done)
            return func();

        // create event function stack
        //apf.done = arguments.callee.done;
        if (!apf.load_init) {
            apf.load_init = function() {
                if (apf.load_done) return;
                // kill the timer
                clearInterval(apf.load_timer);
                apf.load_timer = null;
                apf.load_done  = true;
                if (apf.started)
                    apf.execDeferred();
            };
        }

        apf.load_events.push(func);

        if (func && apf.load_events.length == 1) {
            // Catch cases where addDomLoadEvent() is called after the browser
            // event has already occurred.
            var doc = document, UNDEF = "undefined";
            if ((typeof doc.readyState != UNDEF && doc.readyState == "complete")
              || (doc.getElementsByTagName("body")[0] || doc.body))
                return apf.load_init();

            // for Mozilla/Opera9.
            // Mozilla, Opera (see further below for it) and webkit nightlies
            // currently support this event
            if (doc.addEventListener && !apf.isOpera) {
                // We're using "window" and not "document" here, because it results
                // in a memory leak, especially in FF 1.5:
                // https://bugzilla.mozilla.org/show_bug.cgi?id=241518
                // See also:
                // http://bitstructures.com/2007/11/javascript-method-callbacks
                // http://www-128.ibm.com/developerworks/web/library/wa-memleak/
                window.addEventListener("DOMContentLoaded", apf.load_init, false);
            }
            // If IE is used and is not in a frame
            else if (apf.isIE && window == top) {
                apf.load_timer = setInterval(function() {
                    try {
                        // If IE is used, use the trick by Diego Perini
                        // http://javascript.nwbox.com/IEContentLoaded/
                        doc.documentElement.doScroll("left");
                    }
                    catch(ex) {
                        $setTimeout(arguments.callee, 0);
                        return;
                    }
                    // no exceptions anymore, so we can call the init!
                    apf.load_init();
                }, 10);
            }
            else if (apf.isOpera) {
                doc.addEventListener("DOMContentLoaded", function() {
                    apf.load_timer = setInterval(function() {
                        for (var i = 0, l = doc.styleSheets.length; i < l; i++) {
                            if (doc.styleSheets[i].disabled)
                                return;
                        }
                        // all is fine, so we can call the init!
                        apf.load_init();
                    }, 10);
                }, false);
            }
            else if (apf.isWebkit && !apf.isIphone) {
                var aSheets = doc.getElementsByTagName("link"),
                    i       = aSheets.length,
                    iSheets;
                for (; i >= 0; i++) {
                    if (!aSheets[i] || aSheets[i].getAttribute("rel") != "stylesheet")
                        aSheets.splice(i, 0);
                }
                iSheets = aSheets.length;
                apf.load_timer  = setInterval(function() {
                    if (/loaded|complete/.test(doc.readyState)
                      && doc.styleSheets.length == iSheets)
                        apf.load_init(); // call the onload handler
                }, 10);
            }
            // for other browsers set the window.onload, but also execute the
            // old window.onload
            else {
                var old_onload = window.onload;
                window.onload  = function () {
                    apf.load_init();
                    if (old_onload)
                        old_onload();
                };
            }
        }
    },
    
    
    fireEvent : function(el, type, e, capture){
        if (el.dispatchEvent)
            el.dispatchEvent(type, e, capture);
        else
            el.fireEvent("on" + type, e);
    },
    
    addListener : function(el, type, fn, capture){
        if (el.addEventListener)
            el.addEventListener(type, fn, capture || false);
        else if (el.attachEvent)
            el.attachEvent("on" + type, fn);
        return this;
    },
    
    removeListener : function(el, type, fn, capture){
        if (el.removeEventListener)
            el.removeEventListener(type, fn, capture || false);
        else if (el.detachEvent)
            el.detachEvent("on" + type, fn);
        return this;
    },

    stopEvent: function(e){
        this.stopPropagation(e).preventDefault(e);
        return false;
    },

    stopPropagation: function(e){
        if (e.stopPropagation)
            e.stopPropagation();
        else
            e.cancelBubble = true;
        return this;
    },

    preventDefault: function(e){
        if (e.preventDefault)
            e.preventDefault();
        else
            e.returnValue = false;
        return this;
    },

    /* Destroy */

    /**
     * Unloads the aml application.
     */
    unload : function(exclude){
        

        this.isDestroying = true;

        
        this.popup.destroy();
        

        var node,
            i = 0,
            l = this.all.length;
        for (; i < l; i++) {
            node = this.all[i];
            if (node && node != exclude && node.destroy && !node.apf)
                node.destroy(false);
        }

        //this.dispatchEvent("DOMNodeRemovedFromDocument", {});//@todo apf3.0
        
        for (i = 0, l = this.availHTTP.length; i < l; i++)
            this.availHTTP[i] = null;
        
        this.availHTTP.length = 0;

        
        if (apf.xmldb)
            apf.xmldb.unbind(apf.window);
        

        this.isDestroying = false;
    }
};

/*
 * Replacement for getElementsByTagNameNS because some browsers don't support
 * this call yet.
 */
var $xmlns = function(xmlNode, tag, xmlns, prefix){
    if (!apf.supportNamespaces) {
        if (!prefix)
            prefix = apf.findPrefix(xmlNode, xmlns);

        if (xmlNode.style || xmlNode == document)
            return xmlNode.getElementsByTagName(tag)
        else {
            if (prefix)
                (xmlNode.nodeType == 9 ? xmlNode : xmlNode.ownerDocument)
                    .setProperty("SelectionNamespaces",
                        "xmlns:" + prefix + "='" + xmlns + "'");

            return xmlNode.selectNodes(".//" + (prefix ? prefix + ":" : "") + tag);
        }
    }
    return xmlNode.getElementsByTagNameNS(xmlns, tag);
};

var $setTimeout  = setTimeout;
var $setInterval = setInterval;

apf.setTimeout = function(f, t){
    apf.$eventDepth++;
    return $setTimeout(function(){
        f();
        
        if (--apf.$eventDepth == 0)
            apf.queue.empty();
    }, t);
}

/*$setTimeout = function(f, ms){
    setTimeout(function(){
        console.log(f.toString());
        if (typeof f == "string") eval(f)
        else f();
    }, ms);
}*/

document.documentElement.className += " has_apf";
document.body && (document.body.style.display = "none");

apf.browserDetect();
apf.Init.run("apf");










/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * All elements that implemented this {@link term.baseclass baseclass} have
 * {@link term.propertybinding property binding},
 * event handling and constructor & destructor hooks. The event system is
 * implemented following the W3C specification, similar to the
 * {@link http://en.wikipedia.org/wiki/DOM_Events event system of the HTML DOM}.
 *
 * @class apf.Class
 *
 * @baseclass
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 */

/**
 * @event propertychange Fires when a property changes.
 * @param {Object} e An object containing the following properties:
 * - name ([[String]]): The name of the changed property
 * - originalvalue (`Mixed`): The value it had before the change
 * - value (`Mixed`): The value it has after the change
 *
 */
apf.Class = function(){};

apf.Class.prototype = new (function(){
    // privates
    var FUN   = "function",
        OBJ   = "object",
        UNDEF = "undefined",
        SEL   = "model", //selected|selection|properties|
        PROP  = "prop.",
        MODEL = "model",
        VALUE = "value";

    this.$regbase   = 0;
    /**
     * Tests whether this object has implemented a {@link term.baseclass baseclass}.
     * @param {Number} test The unique number of the {@link term.baseclass baseclass}.
     */
    this.hasFeature = function(test){
        return this.$regbase & test;
    };

    this.$initStack    = [];
    this.$bufferEvents = [];
    this.$init = function(callback, nodeFunc, struct){
        if (typeof callback == FUN || callback === true) {
            this.$bufferEvents = this.$bufferEvents.slice();

            if (callback === true)
                return this;

            this.$initStack = this.$initStack.slice(); //Our own private stack
            this.$initStack.push(callback);

            return this;
        }

        this.addEventListener = realAddEventListener;
        //this.$removalQueue = [];

        if (this.nodeType != 2) //small little hack
            this.$uniqueId = apf.all.push(this) - 1;

        this.$captureStack = {};
        this.$eventsStack  = {};
        this.$funcHandlers = {};

        var i = 0, l = this.$initStack.length;
        for (; i < l; i++)
            this.$initStack[i].apply(this, arguments);

        for (i = 0, l = this.$bufferEvents.length; i < l; i++)
            this.addEventListener.apply(this, this.$bufferEvents[i]);

        delete realAddEventListener;
        delete this.$initStack;
        delete this.$bufferEvents;

        if (struct && (struct.htmlNode || this.nodeFunc == apf.NODE_HIDDEN)) {
            this.$pHtmlNode = struct.htmlNode;

            
                if (this.ownerDocument && this.ownerDocument.$domParser)
                    this.ownerDocument.$domParser.$continueParsing(this);

                
                apf.queue.empty();
                
            
        }

        return this;
    };

    this.implement = apf.implement;

    // **** Property Binding **** //

    this.$handlePropSet = function(prop, value){
        this[prop] = value;
    };

    

    /**
     * Binds a property of another compontent to a property of this element.
     *
     * @param  {String} myProp           the name of the property of this element
     *                                   of which the value is communicated to
     *                                   `bObject`.
     * @param  {Class}  bObject          the object which will receive the property
     *                                   change message.
     * @param  {String} bProp            the property of `bObject` which
     *                                   will be set using the value of
     *                                   `myProp` optionally
     *                                   processed using `strDynamicProp`.
     * @param  {String} [strDynamicProp] a javascript statement which contains the
     *                                   value of `myProp`. The string
     *                                   is used to calculate a new value.
     * @private
     */
    this.$bindProperty = function(myProp, bObject, bProp, fParsed, bRecip){
        if (!fParsed)
            return bObject.$handlePropSet(bProp, this[myProp]);

        var eventName = PROP + myProp, eFunc, isBeingCalled, isLang;
        (this.$eventsStack[eventName] || (this.$eventsStack[eventName] = [])).push(eFunc = function(e){
            if (isBeingCalled) //Prevent circular refs
                return;

            
            isBeingCalled = true;

            try {
                if (fParsed.asyncs) { //if async
                    return fParsed.call(bObject, bObject.xmlRoot, function(value){
                        bObject.setProperty(bProp, value, true, false, 10);

                        

                        isBeingCalled = false;
                    });
                }
                else {
                    var value = fParsed.call(bObject, bObject.xmlRoot);
                }
            }
            catch(e) {
                apf.console.warn("[331] Could not execute binding for property "
                    + bProp + "\n\n" + e.message);

                isBeingCalled = false;

                return;
            }

            //Can't do this when using xml nodes, doesnt seem needed anyway
            //if (bObject[bProp] != value)
                bObject.setProperty(bProp, value, true, false, 10);//e.initial ? 0 :

            

            isBeingCalled = false;
        });

        //Bi-directional property binding
        if (bRecip) {
            eventName = PROP + bProp;
            var _self = this;
            // add bidirectional binding to funcHandlers for visualconnect
            

            (bObject.$eventsStack[eventName] || (bObject.$eventsStack[eventName] = [])).push(
                eFunc.recip = function(){
                    if (isBeingCalled) //Prevent circular refs
                        return;

                    isBeingCalled = true;
                    _self.setProperty(myProp, bObject[bProp], false, false, 10);//e.initial ? 0 :
                    isBeingCalled = false;
                });
        };

        //eFunc({initial: true});

        return eFunc;
    };

    /**
     * Sets a dynamic property from a string.
     *
     * The string used for this function is the same as used in AML to set a
     * dynamic property:
     * ```xml
     *  <a:button visible="{rbTest.value == 'up'}" />
     *  <a:textbox id="rbTest" value="" />
     * ```
     *
     * @param  {String}  prop   The name of the property of this element to set
     *                          using a dynamic rule.
     * @param  {String}  pValue The dynamic property binding rule.
     */
    this.$attrExcludePropBind = false;
    this.$setDynamicProperty = function(prop, pValue){
        var exclNr = this.$attrExcludePropBind[prop],
            options;

        //@todo apf3.0, please generalize this - cache objects, seems slow
        if (SEL.indexOf(prop) > -1 || exclNr == 3) {
            options = {
                xpathmode : 2
                //parsecode : true //@todo is this also good for exclNr 3 ?
            }
        }
        else if (exclNr == 2) {
            options = {nostring : true};
        }
        else if (exclNr === 0) {
            options = {
                parsecode : true
                
            };
        }

        if (this.liveedit)
            (options || (options = {})).liveedit = true;

        

        //Compile pValue through JSLT parser
        
        {
            var fParsed = apf.lm.compile(pValue, options);
        }

        //Special case for model due to needed extra signalling
        if (prop == MODEL)
            (this.$modelParsed = fParsed).instruction = pValue
        

        //if it's only text return setProperty()
        if (fParsed.type == 2) {
            this[prop] = !pValue; //@todo apf3.0 is this needed?
            return this.setProperty(prop, fParsed.str, null, null, 10); //@todo is 10 here right?
        }

        //if there's xpath: Add apf.DataBinding if not inherited.
        //Add compiled binding rule. Load databinding if not loaded.
        
        var check = 1;
        if (exclNr == 2 || fParsed.xpaths.length && exclNr != 1) {
            if (!this.hasFeature(apf.__DATABINDING__)) {
                this.implement(apf.StandardBinding);
                if (this.$attrExcludePropBind[prop] == 1)
                    check = 0;
            }

            if (check)
                this.$addAttrBind(prop, fParsed, pValue);
        }
        

        //if there's prop binding: Add generated function to each obj/prop in the list
        var matches = exclNr && exclNr != 3 && prop != MODEL ? {} : fParsed.props, //@todo apf3.0 sign of broken abstraction, please fix this with a bit mask
            found   = false,
            _self   = this,
            o, node, bProp, p;

        for (p in matches) {
            

            o = p.split(".");
            if (o.length > 2) { //apf.offline.syncing
                bProp = o.pop();
                try{
                    node  = eval(o.join("."));
                }
                catch(e){
                    if (arguments[2]) {
                        apf.console.warn("[287] Could not execute binding test : "
                            + pValue.replace(/</g, "&lt;") + "\n\n" + e.message);
                    }
                    else {
                        apf.queue.add(prop + ":" + this.$uniqueId, function(){
                            _self.$clearDynamicProperty(prop);
                            _self.$setDynamicProperty(prop, pValue, true);
                        });
                    }
                    continue;
                }

                if (!node || typeof node != OBJ || (!node.$regbase && node.$regbase !== 0)) {
                    bProp = o[1];
                    node  = self[o[0]] || apf.nameserver.get("all", o[0]);
                }
                else {
                    o.push(bProp);
                }
            }
            else {
                bProp = o[1];
                node  = self[o[0]] || apf.nameserver.get("all", o[0]) || o[0] == "this" && this;
            }

            if (!node) {
                if (arguments[2]) {
                    apf.console.warn("[287] Could not create property binding: "
                        + " '"  + o[0] + "' does not exist. \n"
                        + pValue.replace(/</g, "&lt;").substr(0, 400));

                    var _self = this;
                    apf.nameserver.waitFor(o[0], function(){
                        _self.$setDynamicProperty(prop, pValue);
                    })
                    return;
                }
                else {
                    //@todo this is sloppy and not efficient - shouldn't clear
                    //and reset and should check if was changed or removed when
                    //it's set
                    apf.queue.add(prop + ":" + this.$uniqueId, function(){
                        _self.$clearDynamicProperty(prop);
                        _self.$setDynamicProperty(prop, pValue, true);
                    });
                    return;
                }
            }

            if (!node.$bindProperty)
                continue;  //return

            if (!this.$funcHandlers[prop])
                this.$funcHandlers[prop] = [];

            var last;
            this.$funcHandlers[prop].push(last = {
                amlNode : node,
                prop    : bProp,
                handler : node.$bindProperty(bProp, this, prop, fParsed,
                    //@todo check if it breaks something. I removed
                    // "&& exclNr != 3" from the expression to enable two way
                    // binding of selections
                    fParsed.type == 4 && SEL.indexOf(prop) == -1) /*,
                bidir   :
                  && this.$bindProperty(prop, node, bProp, function(){
                    return _self[prop];
                  })*/
            });

            found = true;
        }

        if (found) {
            last.handler({initial: true});
        }
        else {
            //@todo optimize this
            if (exclNr)
                return this.setProperty(prop, pValue, null, null, 10); //@todo is 10 here right?

            

            try {
                if (fParsed.asyncs) { //if async
                    return fParsed.call(this, this.xmlRoot, function(value){
                        _self.setProperty(prop, value, true, null, 10); //@todo is 10 here right?

                        
                    });
                }
                else {
                    var value = fParsed.call(this, this.xmlRoot);
                }
            }
            catch(e){
                apf.console.warn("[331] Could not execute binding test or: "
                    + pValue.replace(/</g, "&lt;") + "\n\n" + e.message);
                return;
            }

            this[prop] = !value; //@todo isnt this slow and unneccesary?
            this.setProperty(prop, value, true, null, 10); //@todo is 10 here right?

            
        }
    };

    //@todo setAttribute should delete this from apf.language when not doing
    //$setDynamicProperty
    this.$clearDynamicProperty = function(prop){
        if (this.$removeAttrBind)
            this.$removeAttrBind(prop);

        

        if (this.$inheritProperties)
            delete this.$inheritProperties[prop];

        if (prop == MODEL)
            this.$modelParsed = null;

        //Remove any bounds if relevant
        var f, i, l, h = this.$funcHandlers[prop];
        if (h && typeof h != FUN) {
            for (i = 0, l = h.length; i < l; i++) {
                (f = h[i]).amlNode.removeEventListener(PROP + f.prop, f.handler);
                if (f.handler && f.handler.recip) //@todo handler shouldn't be unset - how does this happen?
                    this.removeEventListener(PROP + prop, f.handler.recip);
            }
            delete this.$funcHandlers[prop];
        }
    };

    

    

    /**
     * Gets an array of properties for this element which can be bound.
     * @returns {Array}
     */
    this.getAvailableProperties = function(){
        return this.$supportedProperties.slice();
    };

    /**
     * Sets the value of a property of this element.
     *
     * Note: The value is the only thing set. Dynamic properties remain bound and the
     * value will be overridden.
     *
     * @param  {String}  prop        The name of the property of this element to
     *                               set using a dynamic rule.
     * @param  {String}  value       The value of the property to set.
     * @param  {Boolean} [forceOnMe] Specifies whether the property should be set even when
     *                               it has the same value.
     */
    this.setProperty = function(prop, value, forceOnMe, setAttr, inherited){
        var s, r, arr, e, i, l,
            oldvalue = this[prop],
            eventName = PROP + prop;//@todo prop event should be called too;

        //Try catch here, because comparison of a string with xmlnode gives and error in IE
        try{
            var isChanged = (typeof value == OBJ)
                ? value != (typeof oldvalue == OBJ ? oldvalue : null)
                : (this.$booleanProperties && this.$booleanProperties[prop]
                    ? oldvalue != apf.isTrue(value)
                    : String(oldvalue) !== String(value));
        } catch(e){
            var isChanged = true;
        }

        //Check if property has changed
        if (isChanged) {
            if (!forceOnMe) { //Recursion protection
                //Check if this property is bound to data
                if (typeof value != OBJ //this.xmlRoot &&
                  && (!(s = this.$attrExcludePropBind[prop]))// || s == 2
                  && (r = (this.$attrBindings && this.$attrBindings[prop]
                  || prop != VALUE && this.xmlRoot && this.$bindings[prop]
                  && this.$bindings[prop][0]))) {

                    //Check if rule has single xpath
                    if (r.cvalue.type == 3) {
                        

                        //Set the xml value - this should probably use execProperty
                        return apf.setNodeValue(
                            this.$getDataNode(prop.toLowerCase(), this.xmlRoot, true),
                            value, true);
                    }
                }
                
            }

            if (setAttr && !this.$funcHandlers[prop])
                this.setAttribute(prop, value, true);

            if (this.$handlePropSet(prop, value, forceOnMe) === false)
                return;

            

            value = this[prop];
        }

        //Optimized event calling
        if ((arr = this.$eventsStack[eventName]) && isChanged) {
            /*for (i = 0, l = arr.length; i < l; i++) {
                if (arr[i].call(this, e || (e = new apf.AmlEvent(eventName, {
                    prop     : prop,
                    value    : value,
                    oldvalue : oldvalue
                }))) === false) {
                    e.returnValue = false;
                }
            }*/
            if (this.dispatchEvent(eventName, {
                prop     : prop,
                value    : value,
                oldvalue : oldvalue,
                changed  : isChanged
            }) === false) {
                e.returnValue = false;
            }
        }

        
        /*
            States:
                    -1 Set
             undefined Pass through
                     2 Inherited
                     3 Semi-inherited
                    10 Dynamic property
        */
        //@todo this whole section should be about attribute inheritance and moved
        //      to AmlElement
        if ((aci || (aci = apf.config.$inheritProperties))[prop]) {
            //@todo this is actually wrong. It should be about removing attributes.
            var resetting = value === "" || typeof value == "undefined";
            if (inherited != 10 && !value) {
                delete this.$inheritProperties[prop];
                if (this.$setInheritedAttribute && this.$setInheritedAttribute(prop))
                    return;
            }
            else if (inherited != 10) { //Keep the current setting (for dynamic properties)
                this.$inheritProperties[prop] = inherited || -1;
            }

            //cancelable, needed for transactions
            //@todo the check on $amlLoaded is not as optimized as can be because $loadAml is not called yet
            if (this.$amlLoaded && (!e || e.returnValue !== false) && this.childNodes) {
                var inheritType = aci[prop];

                (function recur(nodes) {
                    var i, l, node, n;
                    for (i = 0, l = nodes.length; i < l; i++) {
                        node = nodes[i];
                        if (node.nodeType != 1 && node.nodeType != 7)
                            continue;

                        //Pass through
                        n = node.$inheritProperties[prop];
                        if (inheritType == 1 && !n)
                            recur(node.childNodes);

                        //Set inherited property
                        //@todo why are dynamic properties overwritten??
                        else if(!(n < 0)) {//Will also pass through undefined - but why??? @todo seems inefficient
                            if (n == 3 || inherited == 3) { //Because when parent sets semi-inh. prop the value can be the same
                                var sameValue = node[prop];
                                node[prop] = null;
                            }
                            node.setProperty(prop, n != 3
                                ? value
                                : sameValue, false, false, n || 2); //This is recursive already
                        }
                    }
                })(this.childNodes);
            }
        }
        

        return value;
    };
    var aci;

    /**
     * Gets the value of a property of this element.
     *
     * @param  {String}  prop   The name of the property of this element for which to get the value.
     */
    this.getProperty = function(prop){
        return this[prop];
    };

    // *** Event Handling ****/

    apf.$eventDepth = 0;
    this.$eventDepth = 0;

    /**
     * Calls all functions that are registered as listeners for an event.
     *
     * @param  {String}  eventName  The name of the event to dispatch.
     * @param  {Object}  [options]  The properties of the event object that will be created and passed through. These can be:
     *  - bubbles ([[Boolean]]): Specifies whether the event should bubble up to it's parent
     *  - captureOnly ([[Boolean]]): Specifies whether only the captured event handlers should be executed
     * @return {Mixed} return value of the event
     */
    this.dispatchEvent = function(eventName, options, e){
    //var allowEvents = {"DOMNodeInsertedIntoDocument":1,"DOMNodeRemovedFromDocument":1};
        var arr, result, rValue, i, l;

        if (!apf.AmlEvent)
            return;

        apf.$eventDepth++;
        this.$eventDepth++;

        e = options && options.name ? options : e;

        /*if (this.disabled && !allowEvents[eventName]) {
            result = false;
        }
        else {*/
            

            //@todo rewrite this and all dependencies to match w3c
            if ((!e || !e.currentTarget) && (!options || !options.currentTarget)) {
                if (!(options || (options = {})).currentTarget)
                    options.currentTarget = this;

                //Capture support
                if (arr = this.$captureStack[eventName]) {
                    // Clone so that event handlers don't get called multiple times
                    // should a new handler be inserted by the called handler.
                    arr = arr.concat();
                    for (i = 0, l = arr.length; i < l; i++) {
                        rValue = arr[i].call(this, e || (e = new apf.AmlEvent(eventName, options)));
                        if (typeof rValue != UNDEF)
                            result = rValue;
                    }
                }
            }

            //@todo this should be the bubble point

            if (options && options.captureOnly) {
                return e && typeof e.returnValue != UNDEF ? e.returnValue : result;
            }
            else {
                if (this["on" + eventName]) {
                    result = this["on" + eventName].call(this, e
                        || (e = new apf.AmlEvent(eventName, options))); //Backwards compatibility
                }

                if (arr = this.$eventsStack[eventName]) {
                    // Clone so that event handlers don't get called multiple times
                    // should a new handler be inserted by the called handler.
                    arr = arr.concat();
                    for (i = 0, l = arr.length; i < l; i++) {
                        if (!arr[i]) continue;
                        rValue = arr[i].call(this, e
                            || (e = new apf.AmlEvent(eventName, options)));
                        if (typeof rValue != UNDEF)
                            result = rValue;
                    }
                }
            }
        //}

        /*var p;
        while (this.$removalQueue.length) {
            p = this.$removalQueue.shift();
            p[0].remove(p[1]);
        }*/

        
        if ((e && e.bubbles && !e.cancelBubble || !e && options && options.bubbles) && this != apf) {
            rValue = (this.parentNode || this.ownerElement || apf).dispatchEvent(eventName, options, e);
            // || (e = new apf.AmlEvent(eventName, options))

            if (typeof rValue != UNDEF)
                result = rValue;
        }
        

        if (--apf.$eventDepth == 0 && this.ownerDocument
          && !this.ownerDocument.$domParser.$parseContext
          && !apf.isDestroying && apf.loaded
          
          && apf.queue
        ) {
            apf.queue.empty();
        }

        this.$eventDepth--;

        

        if (options) {
            try {
                delete options.currentTarget;
            }
            catch(ex) {
                options.currentTarget = null;
            }
        }

        return e && typeof e.returnValue != UNDEF ? e.returnValue : result;
    };

    /**
     * Adds a function to be called when a event is called.
     *
     * @param  {String}   eventName The name of the event for which to register
     *                              a function.
     * @param  {Function} callback  The code to be called when an event is dispatched.
     */
    this.addEventListener = function(a, b, c){
        this.$bufferEvents.push([a,b,c]);
    };

    var realAddEventListener = function(eventName, callback, useCapture){
        

        if (eventName.substr(0, 2) == "on")
            eventName = eventName.substr(2);

        var s, stack = useCapture ? this.$captureStack : this.$eventsStack;
        if (!(s = stack[eventName]))
            s = stack[eventName] = [];

        if (s.indexOf(callback) > -1)
            return;

        s.unshift(callback);

        var f;
        if (f = this.$eventsStack["$event." + eventName])
            f[0].call(this, callback);
    };

    /**
     * Removes a function registered for an event.
     *
     * @param  {String}   eventName The name of the event for which to unregister
     *                              a function.
     * @param  {Function} callback  The function to be removed from the event stack.
     */
    this.removeEventListener = function(eventName, callback, useCapture){
        var stack = (useCapture ? this.$captureStack : this.$eventsStack)[eventName];

        //@todo is this the best way?
        if (stack) {
            if (this.$eventDepth)
                stack = (useCapture ? this.$captureStack : this.$eventsStack)[eventName] = stack.slice()

            stack.remove(callback);
            if (!stack.length)
                delete (useCapture ? this.$captureStack : this.$eventsStack)[eventName];
        }
    };

    /**
     * Checks if there is an event listener specified for the event.
     *
     * @param  {String}  eventName  The name of the event to check.
     * @return {Boolean} Specifies whether the event has listeners
     */
    this.hasEventListener = function(eventName){
        return (this.$eventsStack[eventName] && this.$eventsStack[eventName].length > 0);
    };

    /**
     * The destructor of a Class.
     * Calls all the destructor functions, and removes all memory leaking references.
     * This function is called when exiting the application or closing the window.
     * @param {Boolean} deep whether the children of this element should be destroyed.
     * @param {Boolean} [clean]
     */
    this.destroy = function(deep, clean){
        //Remove from apf.all
        if (typeof this.$uniqueId == UNDEF && this.nodeType != 2)
            return;

        this.$amlLoaded    = false;
        this.$amlDestroyed = true;

        if (this.$destroy)
            this.$destroy();

        this.dispatchEvent("DOMNodeRemoved", {
            relatedNode  : this.parentNode,
            bubbles      : !apf.isDestroying
        });
        this.dispatchEvent("DOMNodeRemovedFromDocument");

        apf.all[this.$uniqueId] = undefined;

        // != 2 && this.nodeType != 3
        if (!this.nodeFunc && !this.nodeType) { //If this is not a AmlNode, we're done.
            //Remove id from global js space
            try {
                if (this.id || this.name)
                    delete self[this.id || this.name];
            }
            catch (ex) {}
            return;
        }

        if (this.$ext && !this.$ext.isNative) { // && this.$ext.nodeType == 1
            if (this.nodeType == 1 && this.localName != "a")
                this.$ext.oncontextmenu = this.$ext.host = null;
            if (clean) {
                if (this.localName != "collection" && this.$ext.parentNode)
                    this.$ext.parentNode.removeChild(this.$ext);
            }
        }
        if (this.$int && !this.$int.isNative && this.$int.nodeType == 1 && this.localName != "a")
            this.$int.host = null;

        //if (this.$aml && this.$aml.parentNode)
            //this.$aml.parentNode.removeChild(this.$aml);
        this.$aml = null;

        //Clear all children too
        if (deep && this.childNodes) {
            var nodes = this.childNodes;
            for (i = nodes.length - 1; i >= 0; i--) {
                if (nodes[i].destroy)
                    nodes[i].destroy(true, clean && this.localName == "collection");
            }
            this.childNodes = null;
        }

        //Remove from DOM tree if we are still connected
        if (this.parentNode && this.removeNode)
            this.removeNode();
        else if (this.ownerElement && !this.ownerElement.$amlDestroyed)
            this.ownerElement.removeAttributeNode(this);

        //Remove from focus list - Should be in AmlNode
        
        if (this.$focussable && this.focussable)
            apf.window.$removeFocus(this);
        

        
        //Remove dynamic properties
        /*var f, i, l, h;
        for (prop in this.$funcHandlers) {
            h = this.$funcHandlers[prop];

            //Remove any bounds if relevant
            if (h && typeof h != FUN) {
                for (i = 0, l = h.length; i < l; i++) {
                    (f = h[i]).amlNode.removeEventListener(PROP + f.prop, f.handler);
                }
            }
        }*/
        

        if (this.attributes) {
            var attr = this.attributes;
            for (var i = attr.length - 1; i >= 0; i--) {
                
                this.$clearDynamicProperty(attr[i].nodeName);
                
                attr[i].destroy();
            }
        }

        

        //Remove id from global js space
        try {
            if (this.id || this.name)
                self[this.id || this.name] = null;
        }
        catch (ex) {}

        for (var prop in this.$captureStack) this.$captureStack[prop] = null;
        for (var prop in this.$eventsStack) this.$eventsStack[prop] = null;
        for (var prop in this.$funcHandlers) this.$funcHandlers[prop] = null;

        if (this.$bufferEvents) {
            for (var i = this.$bufferEvents.length - 1; i >= 0; i--)
                this.$bufferEvents = null;
        }

        
        apf.nameserver.remove(this.localName, this);
        
    };
})();

apf.extend(apf, new apf.Class().$init());
apf.Init.run("class");





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



apf.color = {
/*
    colors: {
        aliceblue:"#f0f8ff",antiquewhite:"#faebd7",aqua:"#00ffff",
        aquamarine:"#7fffd4",azure:"#f0ffff",beige:"#f5f5dc",bisque:"#ffe4c4",
        black:"#000000",blanchedalmond:"#ffebcd",blue:"#0000ff",
        blueviolet:"#8a2be2",brown:"#a52a2a",burlywood:"#deb887",
        cadetblue:"#5f9ea0",chartreuse:"#7fff00",chocolate:"#d2691e",
        coral:"#ff7f50",cornflowerblue:"#6495ed",cornsilk:"#fff8dc",
        crimson:"#dc143c",cyan:"#00ffff",darkblue:"#00008b",darkcyan:"#008b8b",
        darkgoldenrod:"#b8860b",darkgray:"#a9a9a9",darkgrey:"#a9a9a9",
        darkgreen:"#006400",darkkhaki:"#bdb76b",darkmagenta:"#8b008b",
        darkolivegreen:"#556b2f",darkorange:"#ff8c00",darkorchid:"#9932cc",
        darkred:"#8b0000",darksalmon:"#e9967a",darkseagreen:"#8fbc8f",
        darkslateblue:"#483d8b",darkslategray:"#2f4f4f",
        darkslategrey:"#2f4f4f",darkturquoise:"#00ced1",darkviolet:"#9400d3",
        deeppink:"#ff1493",deepskyblue:"#00bfff",dimgray:"#696969",
        dimgrey:"#696969",dodgerblue:"#1e90ff",firebrick:"#b22222",
        floralwhite:"#fffaf0",forestgreen:"#228b22",fuchsia:"#ff00ff",
        gainsboro:"#dcdcdc",ghostwhite:"#f8f8ff",gold:"#ffd700",
        goldenrod:"#daa520",gray:"#808080",grey:"#808080",green:"#008000",
        greenyellow:"#adff2f",honeydew:"#f0fff0",hotpink:"#ff69b4",
        indianred:"#cd5c5c",indigo:"#4b0082",ivory:"#fffff0",khaki:"#f0e68c",
        lavender:"#e6e6fa",lavenderblush:"#fff0f5",lawngreen:"#7cfc00",
        lemonchiffon:"#fffacd",lightblue:"#add8e6",lightcoral:"#f08080",
        lightcyan:"#e0ffff",lightgoldenrodyellow:"#fafad2",lightgray:"#d3d3d3",
        lightgrey:"#d3d3d3",lightgreen:"#90ee90",lightpink:"#ffb6c1",
        lightsalmon:"#ffa07a",lightseagreen:"#20b2aa",lightskyblue:"#87cefa",
        lightslategray:"#778899",lightslategrey:"#778899",
        lightsteelblue:"#b0c4de",lightyellow:"#ffffe0",lime:"#00ff00",
        limegreen:"#32cd32",linen:"#faf0e6",magenta:"#ff00ff",maroon:"#800000",
        mediumaquamarine:"#66cdaa",mediumblue:"#0000cd",
        mediumorchid:"#ba55d3",mediumpurple:"#9370d8",mediumseagreen:"#3cb371",
        mediumslateblue:"#7b68ee",mediumspringgreen:"#00fa9a",
        mediumturquoise:"#48d1cc",mediumvioletred:"#c71585",
        midnightblue:"#191970",mintcream:"#f5fffa",mistyrose:"#ffe4e1",
        moccasin:"#ffe4b5",navajowhite:"#ffdead",navy:"#000080",
        oldlace:"#fdf5e6",olive:"#808000",olivedrab:"#6b8e23",orange:"#ffa500",
        orangered:"#ff4500",orchid:"#da70d6",palegoldenrod:"#eee8aa",
        palegreen:"#98fb98",paleturquoise:"#afeeee",palevioletred:"#d87093",
        papayawhip:"#ffefd5",peachpuff:"#ffdab9",peru:"#cd853f",pink:"#ffc0cb",
        plum:"#dda0dd",powderblue:"#b0e0e6",purple:"#800080",red:"#ff0000",
        rosybrown:"#bc8f8f",royalblue:"#4169e1",saddlebrown:"#8b4513",
        salmon:"#fa8072",sandybrown:"#f4a460",seagreen:"#2e8b57",
        seashell:"#fff5ee",sienna:"#a0522d",silver:"#c0c0c0",skyblue:"#87ceeb",
        slateblue:"#6a5acd",slategray:"#708090",slategrey:"#708090",
        snow:"#fffafa",springgreen:"#00ff7f",steelblue:"#4682b4",tan:"#d2b48c",
        teal:"#008080",thistle:"#d8bfd8",tomato:"#ff6347",turquoise:"#40e0d0",
        violet:"#ee82ee",wheat:"#f5deb3",white:"#ffffff",whitesmoke:"#f5f5f5",
        yellow:"#ffff00",yellowgreen:"#9acd32"
    },*/
    colorshex: {
        aliceblue:0xf0f8ff,antiquewhite:0xfaebd7,aqua:0x00ffff,
        aquamarine:0x7fffd4,azure:0xf0ffff,beige:0xf5f5dc,bisque:0xffe4c4,
        black:0x000000,blanchedalmond:0xffebcd,blue:0x0000ff,
        blueviolet:0x8a2be2,brown:0xa52a2a,burlywood:0xdeb887,
        cadetblue:0x5f9ea0,chartreuse:0x7fff00,chocolate:0xd2691e,
        coral:0xff7f50,cornflowerblue:0x6495ed,cornsilk:0xfff8dc,
        crimson:0xdc143c,cyan:0x00ffff,darkblue:0x00008b,darkcyan:0x008b8b,
        darkgoldenrod:0xb8860b,darkgray:0xa9a9a9,darkgrey:0xa9a9a9,
        darkgreen:0x006400,darkkhaki:0xbdb76b,darkmagenta:0x8b008b,
        darkolivegreen:0x556b2f,darkorange:0xff8c00,darkorchid:0x9932cc,
        darkred:0x8b0000,darksalmon:0xe9967a,darkseagreen:0x8fbc8f,
        darkslateblue:0x483d8b,darkslategray:0x2f4f4f,
        darkslategrey:0x2f4f4f,darkturquoise:0x00ced1,darkviolet:0x9400d3,
        deeppink:0xff1493,deepskyblue:0x00bfff,dimgray:0x696969,
        dimgrey:0x696969,dodgerblue:0x1e90ff,firebrick:0xb22222,
        floralwhite:0xfffaf0,forestgreen:0x228b22,fuchsia:0xff00ff,
        gainsboro:0xdcdcdc,ghostwhite:0xf8f8ff,gold:0xffd700,
        goldenrod:0xdaa520,gray:0x808080,grey:0x808080,green:0x008000,
        greenyellow:0xadff2f,honeydew:0xf0fff0,hotpink:0xff69b4,
        indianred:0xcd5c5c,indigo:0x4b0082,ivory:0xfffff0,khaki:0xf0e68c,
        lavender:0xe6e6fa,lavenderblush:0xfff0f5,lawngreen:0x7cfc00,
        lemonchiffon:0xfffacd,lightblue:0xadd8e6,lightcoral:0xf08080,
        lightcyan:0xe0ffff,lightgoldenrodyellow:0xfafad2,lightgray:0xd3d3d3,
        lightgrey:0xd3d3d3,lightgreen:0x90ee90,lightpink:0xffb6c1,
        lightsalmon:0xffa07a,lightseagreen:0x20b2aa,lightskyblue:0x87cefa,
        lightslategray:0x778899,lightslategrey:0x778899,
        lightsteelblue:0xb0c4de,lightyellow:0xffffe0,lime:0x00ff00,
        limegreen:0x32cd32,linen:0xfaf0e6,magenta:0xff00ff,maroon:0x800000,
        mediumaquamarine:0x66cdaa,mediumblue:0x0000cd,
        mediumorchid:0xba55d3,mediumpurple:0x9370d8,mediumseagreen:0x3cb371,
        mediumslateblue:0x7b68ee,mediumspringgreen:0x00fa9a,
        mediumturquoise:0x48d1cc,mediumvioletred:0xc71585,
        midnightblue:0x191970,mintcream:0xf5fffa,mistyrose:0xffe4e1,
        moccasin:0xffe4b5,navajowhite:0xffdead,navy:0x000080,
        oldlace:0xfdf5e6,olive:0x808000,olivedrab:0x6b8e23,orange:0xffa500,
        orangered:0xff4500,orchid:0xda70d6,palegoldenrod:0xeee8aa,
        palegreen:0x98fb98,paleturquoise:0xafeeee,palevioletred:0xd87093,
        papayawhip:0xffefd5,peachpuff:0xffdab9,peru:0xcd853f,pink:0xffc0cb,
        plum:0xdda0dd,powderblue:0xb0e0e6,purple:0x800080,red:0xff0000,
        rosybrown:0xbc8f8f,royalblue:0x4169e1,saddlebrown:0x8b4513,
        salmon:0xfa8072,sandybrown:0xf4a460,seagreen:0x2e8b57,
        seashell:0xfff5ee,sienna:0xa0522d,silver:0xc0c0c0,skyblue:0x87ceeb,
        slateblue:0x6a5acd,slategray:0x708090,slategrey:0x708090,
        snow:0xfffafa,springgreen:0x00ff7f,steelblue:0x4682b4,tan:0xd2b48c,
        teal:0x008080,thistle:0xd8bfd8,tomato:0xff6347,turquoise:0x40e0d0,
        violet:0xee82ee,wheat:0xf5deb3,white:0xffffff,whitesmoke:0xf5f5f5,
        yellow:0xffff00,yellowgreen:0x9acd32
    },
    fixHSB: function (hsb) {
        return {
            h: Math.min(360, Math.max(0, hsb.h)),
            s: Math.min(100, Math.max(0, hsb.s)),
            b: Math.min(100, Math.max(0, hsb.b))
        };
    },

    fixRGB: function (rgb) {
        return {
            r: Math.min(255, Math.max(0, rgb.r)),
            g: Math.min(255, Math.max(0, rgb.g)),
            b: Math.min(255, Math.max(0, rgb.b))
        };
    },

    fixHex: function (hex, asBrowser) {
        hex = hex.toLowerCase().replace(/[^a-f0-9]/g, "");
        var len = 6 - hex.length;
        if (len > 0) {
            var ch = "0";
            var o = [];
            var i = 0;
            if (asBrowser) {
                ch = hex.charAt(hex.length - 1);
                o.push(hex);
            }
            for (; i < len; i++)
                o.push(ch);
            if (!asBrowser)
                o.push(hex);
            hex = o.join("");
        }
        return hex;
    },
    
    hexToRGB: function (hex) {
        hex = parseInt(((hex.indexOf("#") > -1) ? hex.substring(1) : hex), 16);
        return {r: hex >> 16, g: (hex & 0x00FF00) >> 8, b: (hex & 0x0000FF)};
    },

    hexToHSB: function (hex) {
        return this.RGBToHSB(this.hexToRGB(hex));
    },

    RGBToHSB: function (rgb) {
        var hsb = {
            h: 0,
            s: 0,
            b: 0
        };
        var min   = Math.min(rgb.r, rgb.g, rgb.b),
            max   = Math.max(rgb.r, rgb.g, rgb.b),
            delta = max - min;
        hsb.b = max;
        if (max != 0) { }
        hsb.s = max != 0 ? 255 * delta / max : 0;
        if (hsb.s != 0) {
            if (rgb.r == max)
                hsb.h = (rgb.g - rgb.b) / delta;
            else if (rgb.g == max)
                hsb.h = 2 + (rgb.b - rgb.r) / delta;
            else
                hsb.h = 4 + (rgb.r - rgb.g) / delta;
        }
        else
            hsb.h = -1;
        hsb.h *= 60;
        if (hsb.h < 0)
            hsb.h += 360;
        hsb.s *= 100/255;
        hsb.b *= 100/255;
        return hsb;
    },
    
    HSBToRGB: function(hsb) {
        var rgb = {},
            h   = Math.round(hsb.h),
            s   = Math.round(hsb.s * 255 / 100),
            v   = Math.round(hsb.b * 255 / 100);
        if (s == 0)
            rgb.r = rgb.g = rgb.b = v;
        else {
            var t1 = v,
                t2 = (255 - s) * v / 255,
                t3 = (t1 - t2) * (h % 60)/60;
            if (h == 360)
                h = 0;
            if (h < 60)
                rgb.r = t1, rgb.b = t2, rgb.g = t2 + t3;
            else if (h < 120)
                rgb.g = t1, rgb.b = t2, rgb.r = t1 - t3;
            else if (h < 180)
                rgb.g = t1, rgb.r = t2, rgb.b = t2 + t3;
            else if (h < 240)
                rgb.b = t1, rgb.r = t2, rgb.g = t1 - t3;
            else if (h < 300)
                rgb.b = t1, rgb.g = t2, rgb.r = t2 + t3;
            else if (h < 360)
                rgb.r = t1, rgb.g = t2, rgb.b = t1 - t3;
            else
                rgb.r = 0, rgb.g = 0, rgb.b = 0;
        }
        return {r: Math.round(rgb.r), g: Math.round(rgb.g), b: Math.round(rgb.b)};
    },

    RGBToHex: function(rgb) {
        return ('00000'+(rgb.r<<16 | rgb.g<<8 | rgb.b).toString(16)).slice(-6);
    },

    HSBToHex: function(hsb) {
        return this.RGBToHex(this.HSBToRGB(hsb));
    }
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
  





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/**
 * Performs an async function in serial on each of the list items.
 * 
 * @param {Array} list A list of elements to iterate over asynchronously
 * @param {Function} async An ssync function of the form `function(item, callback)`
 * @param {Function} callback A function of the form `function(error)`, which is
 *      called after all items have been processed
 */
apf.asyncForEach = function(list, async, callback) {
    var i = 0;
    var len = list.length;

    if (!len) return callback(null, []);

    async(list[i], function handler(err) {
        if (err) return callback(err);
        i++;

        if (i < len) {
            async(list[i], handler, i);
        } else {
            callback(null);
        }
    }, i);
};

/**
 * Performs an async function in serial, as long as the function 'condition' (first 
 * argument) evaluates to true.
 * 
 * @param {Function} condition A function that returns a [Boolean], which determines
 *                             if the loop should continue
 * @param {Function} async     async A function of the form `function(iteration_no, callback)`
 * @param {Function} callback  A function of the form `function(error)`, which is
 *                             called after all items have been processed
 */
apf.asyncWhile = function(condition, async, callback) {
    var i = 0;
    async(i, function handler(err) {
        if (err)
            return callback ? callback(err, i) : null;

        ++i;
        if (condition(i))
            async(i, handler);
        else
            callback && callback(null, i);
    });
};

/**
 * Maps each element from the list to the result returned by the async mapper
 * function. 
 *
 * The mapper takes an element from the list and a callback as arguments.
 * After completion, the mapper has to call the callback with an (optional) error
 * object as the first argument, and the result of the map as second argument. After all
 * list elements have been processed, the last callback is called with the mapped
 * array as second argument.
 * 
 * @param {Array} list A list of elements to iterate over asynchronously
 * @param {Function}  mapper A function of the form `function(item, next)`
 * @param {Function} callback A function of the form `function(error, result)`
 */
apf.asyncMap = function(list, mapper, callback) {
    var i = 0;
    var len = list.length;

    if (!len) return callback(null, []);
    var map = [];

    async(list[i], function handler(err, value) {
        if (err) return callback(err);
        
        map[i] = value;
        i++;

        if (i < len) {
            async(list[i], handler);
        } else {
            callback(null, map);
        }
    });
};


/**
 * Chains an array of functions. 
 *
 * Each of the functions (except the last one) must
 * have exactly one `callback` argument, which must be called after the functions has
 * finished. If the callback fails, it must pass a non-null error object as the
 * first argument to the callback.
 * 
 * @param {Array} funcs An array of functions to chain together.
 */
apf.asyncChain = function(funcs) {
    var i = 0;
    var len = funcs.length;
    
    function next() {
        var f = funcs[i++];
        if (i == len)
            f()
        else
            f(next)
    }
    
    next();
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
 


/**
 * Sets a name/value pair which is stored in the browser and sent to the server
 * with every request. This is also known as a cookie. 
 * Warning: Be careful setting cookies, because they can take up a lot of bandwidth, 
 * especially for Ajax applications.
 * 
 * @param {String}  name     The cookie name
 * @param {String}  value    The cookie value
 * @param {Date}    expire   The expiration date representing the number of milliseconds
 *                            since 1 January 1970 00:00:00 UTC.
 * @param {String}  path     The path name
 * @param {String}  domain   The domain name
 * @param {Boolean} secure   If true, the secure attribute of the cookie will be set and 
 *                            the cookie transmission will require a secure protocol (like HTTPS)
 * @return {String} Returns the cookie name.
 */
apf.setcookie = function(name, value, expire, path, domain, secure) {
    var ck = name + "=" + escape(value) + ";";
    if (expire) ck += "expires=" + new Date(expire
        + new Date().getTimezoneOffset() * 60).toGMTString() + ";";
    if (path)   ck += "path=" + path + ";";
    if (domain) ck += "domain=" + domain + ";";
    if (secure) ck += "secure";

    document.cookie = ck;
    return value
};

/**
 * Gets the value of a stored name/value cookie pair.
 * 
 * @param {String} name The name of the stored cookie.
 * @return {String} Returns a value of the cookie, or the empty string if it isn't found
 */
apf.getcookie = function(name) {
  var aCookie = document.cookie.split("; ");
  for (var i = 0; i < aCookie.length; i++) {
      var aCrumb = aCookie[i].split("=");
      if (name == aCrumb[0])
          return unescape(aCrumb[1]);
  }

  return "";
};

/**
 * Deletes a stored name/value pair called a cookie.
 * 
 * @param {String} name     The name of the stored cookie
 * @param {String} domain   The name of the domain of stored cookie
 */
apf.delcookie = function (name, domain){
    document.cookie = name + "=blah; expires=Fri, 31 Dec 1999 23:59:59 GMT;"
        + (domain ? 'domain='+domain : '');
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



// start closure:
//(function(){

if (typeof isFinite == "undefined") {
    function isFinite(val){
        return val + 1 != val;
    }
}

apf.NUMBER   = 1;
apf.BOOLEAN  = 2;
apf.STRING   = 3;
apf.ARRAY    = 4;
apf.DATE     = 5;
apf.REGEXP   = 6;
apf.FUNCTION = 7;

Array.prototype.dataType    = apf.ARRAY;
Number.prototype.dataType   = apf.NUMBER;
Date.prototype.dataType     = apf.DATE;
Boolean.prototype.dataType  = apf.BOOLEAN;
String.prototype.dataType   = apf.STRING;
RegExp.prototype.dataType   = apf.REGEXP;
Function.prototype.dataType = apf.FUNCTION;

/**
 * Converts a JavaScript object to a CGI string.
 *
 * @param {Object} args The object to convert
 * @param {Boolean} [multicall] Indicates if there will be a function call name
 * @param {Function} [mcallname] A name to append to the CGI string, prefixed with `func=`
 * @returns {Object} The converted object, joined with `"&"`
 *
 * @see apf.convertXml
 */
apf.getCgiString = function(args, multicall, mcallname){
    var prop, vars = [];

    function recur(o, stack) {
        if (apf.isArray(o)) {
            for (var j = 0; j < o.length; j++)
                recur(o[j], stack + "%5B%5D");//" + j + "
        }
        else if (typeof o == "object") {
            for (prop in o) {
                if (apf.isSafariOld && (!o[prop] || typeof o[prop] != "object"))
                    continue;

                if (typeof o[prop] == "function")
                    continue;
                recur(o[prop], stack + "%5B" + encodeURIComponent(prop) + "%5D");
            }
        }
        else
            vars.push(stack + "=" + encodeURIComponent(o));
    };

    if (multicall) {
        vars.push("func=" + mcallname);
        for (var i = 0; i < args[0].length; i++)
            recur(args[0][i], "f%5B" + i + "%5D");
    }
    else {
        for (prop in args) {
            if (apf.isSafariOld && (!args[prop] || typeof args[prop] == "function"))
                continue;

            recur(args[prop], prop);
        }
    }

    return vars.join("&");
};

/**
 * Converts a CGI string to a JavaScript object.
 *
 * @param {String} args The argument string to convert, separated by `"&"`
 * @returns {Object} The converted object
 *
 * @see apf.convertXml
 */
apf.fromCgiString = function(args) {
    if (!args)
        return false;

    var obj = {};
    args = args.split("&");
    for (var data, i = 0; i < args.length; i++) {
        data = args[i].split("=");
        data[0] = decodeURIComponent(data[0]);
        var path = data[0].replace(/\]/g, "").split("[");

        var spare = obj;
        for (var j = 0; j < path.length; j++) {
            if (spare[path[j]])
                spare = spare[path[j]];
            else if (path.length == j+1) {
                if (path[j])
                    spare[path[j]] = decodeURIComponent(data[1]);
                else
                    spare.push(decodeURIComponent(data[1]));
                break; //assuming last
            }
            else{
                spare[path[j]] = !path[j+1] ? [] : {};
                spare = spare[path[j]];
            }
        }
    }

    return obj;
}




/*
 * Extends a Function object with properties from other objects, specified as
 * arguments.
 *
 * @param {Mixed} obj1, obj2, obj3, etc.
 * @type Function
 * @see apf.extend
 */
Function.prototype.extend = function() {
    apf.extend.apply(this, [this].concat(Array.prototype.slice.call(arguments)));
    return this;
};

/*
 * Attach a Function object to an event as handler method. If apf.AbstractEvent
 * is available, the active event is extended with convinience accessors as
 * declared in apf.AbstractEvent
 *
 * @param {Object} The context the execute the Function within
 * @param {Boolean} Whether the passed event object should be extended with AbstractEvent
 * @param {Mixed}  param1, param2, param3, etc.
 * @type Function
 * @see apf.AbstractEvent
 */
Function.prototype.bindWithEvent = function() {
    var __method = this,
        args     = Array.prototype.slice.call(arguments),
        o        = args.shift(),
        ev       = args.shift();
    return function(event) {
        if (!event)
            event = window.event;
        
        return __method.apply(o, [event].concat(args)
            .concat(Array.prototype.slice.call(arguments)));
    }
};

/*
 * The bind function creates a new function (a bound function) that calls the
 * function that is its this value (the bound function's target function) with
 * a specified this parameter, which cannot be overridden. bind also accepts
 * leading default arguments to provide to the target function when the bound
 * function is called.  A bound function may also be constructed using the new
 * operator: doing so acts as though the target function had instead been
 * constructed.  The provided this value is ignored, while prepended arguments
 * are provided to the emulated function.
 *
 * @param {Object} context The 'this' context of the bound function
 * @type Function
 */
if (!Function.prototype.bind)
    Function.prototype.bind = function(context /*, arg1, arg2... */) {
        if (typeof this !== 'function') throw new TypeError();
        var _arguments = Array.prototype.slice.call(arguments, 1),
            _this = this,
            _concat = Array.prototype.concat,
            _function = function() {
                return _this.apply(this instanceof _dummy ? this : context,
                    _concat.apply(_arguments, arguments));
            },
            _dummy = function() {};
        _dummy.prototype = _this.prototype;
        _function.prototype = new _dummy();
        return _function;
};

/*
 * Copy an array, like this statement would: 'this.concat([])', but then do it
 * recursively.
 */
Array.prototype.copy = function(){
    var ar = [];
    for (var i = 0, j = this.length; i < j; i++)
        ar[i] = this[i] && this[i].copy ? this[i].copy() : this[i];

    return ar;
};

/*
 * Concatenate the current Array instance with one (or more) other Arrays, like
 * Array.concat(), but return the current Array instead of a new one that
 * results from the merge.
 *
 * @param {Array} array1, array2, array3, etc.
 * @type  {Array}
 */
Array.prototype.merge = function(){
    for (var i = 0, k = arguments.length; i < k; i++) {
        for (var j = 0, l = arguments[i].length; j < l; j++) {
            this.push(arguments[i][j]);
        }
    }
};

/*
 * Add the values of one or more arrays to the current instance by using the
 * '+=' operand on each value.
 *
 * @param {Array} array1, array2, array3, etc.
 * @type  {Array}
 * @see Array.copy
 */
Array.prototype.arrayAdd = function(){
    var s = this.copy();
    for (var i = 0, k = arguments.length; i < k; i++) {
        for (var j = 0, l = s.length; j < l; j++) {
            s[j] += arguments[i][j];
        }
    }

    return s;
};

/*
 * Check if an object is contained within the current Array instance.
 *
 * @param {Mixed}   obj The value to check for inside the Array
 * @type  {Boolean}
 */
Array.prototype.equals = function(obj){
    for (var i = 0, j = this.length; i < j; i++)
        if (this[i] != obj[i])
            return false;
    return true;
};

/*
 * Make sure that an array instance contains only unique values (NO duplicates).
 * Elaborate implementation to allow for O(n) time complexity compared to O(n^2)
 * time complexity when using Array.prototype.indexOf.
 * @see http://bbenvie.com/articles/2012-06-10/Array-prototype-unique-in-O-n-time-complexity
 * @see http://jsperf.com/array-unique2/9
 *
 * @type {Array}
 */
var uniqueBenvie = function(){
    var hasOwn = {}.hasOwnProperty,
        uids = {};

    // use hash for primitives and tagging for objects
    function uid(){
        var chars = [], i = 20, num;
        while (i--) {
            num = Math.random() * 52 | 0;
            chars[i] = String.fromCharCode(num + (num >= 26 ? 71 : 65));
        }
        chars = chars.join("");

        if (chars in uids)
            return uid();

        uids[chars] = true;
        return chars;
    }

    function unique(array){
        var strings = {}, numbers = {}, others = {},
            tagged = [], failed = [],
            count = 0, i = array.length,
            item, type;

        var id = uid();

        while (i--) {
            item = array[i];
            type = typeof item;
            if (item === null || type !== "object" && type !== "function") {
                // primitive
                switch (type) {
                    case "string":
                        strings[item] = true;
                        break;
                    case "number":
                        numbers[item] = true;
                        break;
                    default:
                        others[item] = item;
                        break;
                }
            }
            else {
                // object
                if (!hasOwn.call(item, id)) {
                    try {
                        item[id] = true;
                        tagged[count++] = item;
                    }
                    catch (e){
                        if (failed.indexOf(item) === -1)
                            failed[failed.length] = item;
                    }
                }
            }
        }

        // remove the tags
        while (count--)
            delete tagged[count][id];

        tagged = tagged.concat(failed);
        count = tagged.length;

        // append primitives to results
        for (i in strings)
            if (hasOwn.call(strings, i))
                tagged[count++] = i;

        for (i in numbers)
            if (hasOwn.call(numbers, i))
                tagged[count++] = +i;

        for (i in others)
            if (hasOwn.call(others, i))
                tagged[count++] = others[i];

        return tagged;
    }

    return unique;
}();

if (typeof Set !== "undefined") {
    Array.prototype.makeUnique = function(){
        var out = [],
            seen = new Set,
            i = this.length;

        while (i--) {
            if (!seen.has(this[i])) {
                out[out.length] = this[i];
                seen.add(this[i]);
            }
        }

        return out;
    }
}
else {
    Array.prototype.makeUnique = function(){
        return uniqueBenvie(this);
    };
}

/*
 * Check if this array instance contains a value 'obj'.
 *
 * @param {Mixed}  obj    The value to check for inside the array
 * @param {Number} [from] Left offset index to start the search from
 * @type  {Boolean}
 * @see Array.indexOf
 */
Array.prototype.contains = function(obj, from){
    return this.indexOf(obj, from) != -1;
};

/*
 * Search for the index of the first occurence of a value 'obj' inside an array
 * instance.
 * July 29, 2008: added 'from' argument support to indexOf()
 *
 * @param {Mixed}  obj    The value to search for inside the array
 * @param {Number} [from] Left offset index to start the search from
 * @type  {Number}
 */
Array.prototype.indexOf = Array.prototype.indexOf || function(obj, from){
    var len = this.length;
    for (var i = (from < 0) ? Math.max(0, len + from) : from || 0; i < len; i++) {
        if (this[i] === obj)
            return i;
    }
    return -1;
};

/*
 * Search for the index of the last occurence of a value 'obj' inside an array
 * instance.
 *
 * @param {Mixed}  obj    The value to search for inside the array
 * @param {Number} [from] Left offset index to start the search from
 * @type  {Number}
 */
Array.prototype.lastIndexOf = Array.prototype.lastIndexOf || function(obj, from) {
    //same as indexOf(), but in reverse loop, JS spec 1.6
    var len = this.length;
    for (var i = (from >= len) ? len - 1 : (from < 0) ? from + len : len - 1; i >= 0; i--) {
        if (this[i] === obj)
            return i;
    }
    return -1;
};

/*
 * Like Array.push, but only invoked when the value 'item' is already present
 * inside the array instance.
 *
 * @param {Mixed} item, item, ...
 * @type  {Array}
 */
Array.prototype.pushUnique = function(){
    var item,
        i = 0,
        l = arguments.length;
    for (; i < l; ++i) {
        item = arguments[i];
    if (this.indexOf(item) == -1)
        this.push(item);
    }
    return this;
};

/*
 * @todo: Ruben: could you please comment on this function? Seems to serve a very
 * specific purpose...
 *
 * I also could not find an occurrence in our codebase.
 */
Array.prototype.search = function(){
    for (var i = 0, length = arguments.length; i < length; i++) {
        if (typeof this[i] != "array")
            continue;
        for (var j = 0; j < length; j++) {
            if (this[i][j] != arguments[j])
                break;
            else if (j == (length - 1))
                return this[i];
        }
    }
};

/*
 * Iterate through each value of an array instance from left to right (front to
 * back) and execute a callback Function for each value.
 *
 * @param {Function} fn
 * @type  {Array}
 */
Array.prototype.each =
Array.prototype.forEach = Array.prototype.forEach || function(fn) {
    for (var i = 0, l = this.length; i < l; i++)
        if (fn.call(this, this[i], i, this) === false)
            break;
    return this;
}

/*
 * Search for a value 'obj' inside an array instance and remove it when found.
 *
 * @type {Mixed} obj
 * @type {Array}
 */
Array.prototype.remove = function(obj){
    for (var i = this.length - 1; i >= 0; i--) {
        if (this[i] != obj)
            continue;

        this.splice(i, 1);
    }

    return this;
};

/*
 * Remove an item from an array instance which can be identified with key 'i'
 *
 * @param  {Number} i
 * @return {Mixed}  The removed item
 */
Array.prototype.removeIndex = function(i){
    if (!this.length) return null;
    return this.splice(i, 1);
};

/*
 * Insert a new value at a specific object; alias for Array.splice.
 *
 * @param {Mixed}  obj Value to insert
 * @param {Number} i   Index to insert 'obj' at
 * @type  {Number}
 */
Array.prototype.insertIndex = function(obj, i){
    this.splice(i, 0, obj);
};

/*
 * Reverses the order of the elements of an array; the first becomes the last,
 * and the last becomes the first.
 *
 * @type {Array}
 */
Array.prototype.invert =
Array.prototype.reverse = Array.prototype.reverse || function(){
    var l = this.length - 1;
    for (var temp, i = 0; i < Math.ceil(0.5 * l); i++) {
        temp        = this[i];
        this[i]     = this[l - i]
        this[l - i] = temp;
    }

    return this;
};



/*
 * Attempt to fully comply (in terms of functionality) with the JS specification,
 * up 'till version 1.7:
 * @link http://developer.mozilla.org/en/docs/Core_JavaScript_1.5_Reference:Global_Objects:Array
 */

/*
 * Creates a new array with all of the elements of this array for which the
 * provided filtering function returns true.
 *
 * @param {Function} fn   Function to test each element of the array.
 * @param {Object}   bind Object to use as this when executing callback.
 * @type  {Array}
 */
Array.prototype.filter = Array.prototype.filter || function(fn, bind){
    var results = [];
    for (var i = 0, l = this.length; i < l; i++) {
        if (fn.call(bind, this[i], i, this))
            results.push(this[i]);
    }
    return results;
};

/*
 * Returns true if every element in this array satisfies the provided testing
 * function.
 *
 * @param {Function} fn   Function to test for each element.
 * @param {Object}   bind Object to use as this when executing callback.
 * @type  {Boolean}
 */
Array.prototype.every = Array.prototype.every || function(fn, bind){
    for (var i = 0, l = this.length; i < l; i++) {
        if (!fn.call(bind, this[i], i, this))
            return false;
    }
    return true;
};

/*
 * Creates a new array with the results of calling a provided function on every
 * element in this array.
 *
 * @param {Function} fn   Function that produces an element of the new Array from an element of the current one.
 * @param {Object}   bind Object to use as this when executing callback.
 * @type  {Array}
 */
Array.prototype.map = Array.prototype.map || function(fn, bind){
    var results = [];
    for (var i = 0, l = this.length; i < l; i++)
        results[i] = fn.call(bind, this[i], i, this);
    return results;
};

/*
 * Tests whether some element in the array passes the test implemented by the
 * provided function.
 *
 * @param {Function} fn   Function to test for each element.
 * @param {Object}   bind Object to use as this when executing callback.
 * @type  {Boolean}
 */
Array.prototype.some = Array.prototype.some || function(fn, bind){
    for (var i = 0, l = this.length; i < l; i++) {
        if (fn.call(bind, this[i], i, this))
            return true;
    }
    return false;
};



/*
 * Transform a number to a string and pad it with a zero digit its length is one.
 *
 * @type {String}
 */
Number.prototype.toPrettyDigit = Number.prototype.toPrettyDigit || function() {
    var n = this.toString();
    return (n.length == 1) ? "0" + n : n;
};

RegExp.prototype.getNativeFlags = function() {
    return (this.global     ? "g" : "") +
           (this.ignoreCase ? "i" : "") +
           (this.multiline  ? "m" : "") +
           (this.extended   ? "x" : "") +
           (this.sticky     ? "y" : "");
};

/*
 * Accepts flags; returns a new XRegExp object generated by recompiling
 * the regex with the additional flags (may include non-native flags).
 * the original regex object is not altered.
 */
RegExp.prototype.addFlags = function(flags){
    return new RegExp(this.source, (flags || "") + this.getNativeFlags());
};

/*
 * Casts the first character in a string to uppercase.
 *
 * @type {String}
 */
String.prototype.uCaseFirst = function(){
    return this.substr(0, 1).toUpperCase() + this.substr(1)
};

/*
 * Removes spaces and other space-like characters from the left and right ends
 * of a string
 *
 * @type {String}
 */
String.prototype.trim = function(){
    return this.replace(/[\s\n\r]*$/, "").replace(/^[\s\n\r]*/, "");
};

/*
 * Concatenate a string with itself n-times.
 *
 * @param {Number} times Number of times to repeat the String concatenation
 * @type  {String}
 */
String.prototype.repeat = function(times){
    return Array(times + 1).join(this);
};

/*
 * Count the number of occurences of substring 'str' inside a string
 *
 * @param {String} str
 * @type  {Number}
 */
String.prototype.count = function(str){
    return this.split(str).length - 1;
};

/*
 * Remove HTML or any XML-like tags from a string
 *
 * @type {String}
 */
String.prototype.stripTags = function() {
    return this.replace(/<\/?[^>]+>/gi, "");
};

/*
 * Wrapper for the global 'escape' function for strings
 *
 * @type {String}
 */
String.prototype.escape = function() {
    return escape(this);
};

/*
 * Returns an xml document
 * @type {XMLElement}
 */
String.prototype.toXml = function(){
    var node = apf.getXml("<root>" + this + "</root>");
    if (node.childNodes.length == 1) {
        return node.childNodes[0];
    }
    else {
        var docFrag = node.ownerDocument.createDocumentFragment(),
            nodes   = node.childNodes;
        while (nodes.length)
            docFrag.appendChild(nodes[0]);
        return docFrag;
    }
};


if (typeof window != "undefined" && typeof window.document != "undefined"
  && typeof window.document.createElement == "function") {
    /*
     * Encode HTML entities to its HTML equivalents, like '&amp;' to '&amp;amp;'
     * and '&lt;' to '&amp;lt;'.
     *
     * @type {String}
     * @todo is this fast?
     */
    String.prototype.escapeHTML = function() {
        this.escapeHTML.text.data = this;
        return this.escapeHTML.div.innerHTML;
    };

    /*
     * Decode HTML equivalent entities to characters, like '&amp;amp;' to '&amp;'
     * and '&amp;lt;' to '&lt;'.
     *
     * @type {String}
     */
    String.prototype.unescapeHTML = function() {
        var div = document.createElement("div");
        div.innerHTML = this.stripTags();
        if (div.childNodes[0]) {
            if (div.childNodes.length > 1) {
                var out = [];
                for (var i = 0; i < div.childNodes.length; i++)
                    out.push(div.childNodes[i].nodeValue);
                return out.join("");
            }
            else
                return div.childNodes[0].nodeValue;
        }
        return "";
    };

    String.prototype.escapeHTML.div  = document.createElement("div");
    String.prototype.escapeHTML.text = document.createTextNode("");
    String.prototype.escapeHTML.div.appendChild(String.prototype.escapeHTML.text);

    if ("<\n>".escapeHTML() !== "&lt;\n&gt;")
        String.prototype.escapeHTML = null;

    if ("&lt;\n&gt;".unescapeHTML() !== "<\n>")
        String.prototype.unescapeHTML = null;
}

if (!String.prototype.escapeHTML) {
    String.prototype.escapeHTML = function() {
        return this.replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;");
    };
}

if (!String.prototype.unescapeHTML) {
    String.prototype.unescapeHTML = function() {
        return this.stripTags().replace(/&lt;/g,"<").replace(/&gt;/g,">").replace(/&amp;/g,"&");
    };
}

/*
 * Trim a string down to a specific number of characters. Optionally, append an
 * ellipsis ('...') as a suffix.
 *
 * @param {Number}  nr
 * @param {Boolean} [ellipsis] Append an ellipsis
 * @type  {String}
 */
String.prototype.truncate = function(nr, ellipsis){
    return this.length >= nr
        ? this.substring(0, nr - (ellipsis ? 4 : 1)) + (ellipsis ? "..." : "")
        : this;
};

/*
 * Pad a string at the right or left end with a string 'pad' to a specific
 * number of characters. Highly optimized version for speed, not readability.
 *
 * @param {Number}  len   Specifies the amount of characters required to pad to.
 * @param {String}  pad   Specifies the character(s) to pad the string with
 * @param {Boolean} [dir] Specifies at which end to append the 'pad' character (left or right).
 * @type  {String}
 */
String.prototype.pad = function(len, pad, dir) {
    return dir ? (this + Array(len).join(pad)).slice(0, len)
        : (Array(len).join(pad) + this).slice(-len);
};

apf.PAD_LEFT  = false;
apf.PAD_RIGHT = true;

/*
 * Special String.split; optionally lowercase a string and trim all results from
 * the left and right.
 *
 * @param {String}  separator
 * @param {Number}  limit      Maximum number of items to return
 * @param {Boolean} bLowerCase Flag to lowercase the string prior to split
 * @type  {String}
 */
String.prototype.splitSafe = function(separator, limit, bLowerCase) {
    return (bLowerCase && this.toLowerCase() || this)
        .replace(/(?:^\s+|\n|\s+$)/g, "")
        .split(new RegExp("[\\s ]*" + separator + "[\\s ]*", "g"), limit || 999);
};

/*
 * Appends a random number with a specified length to this String instance.
 *
 * @see randomGenerator
 * @param {Number} length
 * @type  {String}
 */
String.prototype.appendRandomNumber = function(length) {
    for (var arr = [], i = 1; i <= length; i++)
        arr.push(apf.randomGenerator.generate(1, 9));
    // Create a new string from the old one, don't just create a copy
    return this.toString() + arr.join("");
};

/*
 * Prepends a random number with a specified length to this String instance.
 *
 * @see randomGenerator
 * @param {Number} length
 * @type  {String}
 */
String.prototype.prependRandomNumber = function(length) {
    for (var arr = [], i = 1; i <= length; i++)
        arr.push(apf.randomGenerator.generate(1, 9));
    // Create a new string from the old one, don't just create a copy
    return arr.join("") + this.toString();
};

/*
 * Returns a string produced according to the formatting string. It replaces
 * all <i>%s</i> occurrences with the arguments provided.
 *
 * @link http://www.php.net/sprintf
 * @type {String}
 */
String.prototype.sprintf = function() {
    // Create a new string from the old one, don't just create a copy
    var str = this.toString(),
        i   = 0,
        inx = str.indexOf("%s");
    while (inx >= 0) {
        var replacement = arguments[i++] || " ";
        str = str.substr(0, inx) + replacement + str.substr(inx + 2);
        inx = str.indexOf("%s");
    }
    return str;
};

/*
 * The now method returns the milliseconds elapsed since
 * 1 January 1970 00:00:00 UTC up until now as a number.
 *
 * @type {Number}
 */
if (!Date.now) {
    Date.now = function now() {
        return +new Date();
    };
}

//})(); //end closure






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */







//@todo maybe generalize this to pub/sub event system??
/**
 * @private
 */
apf.hotkeys = {};
(function() {
    /**
     * @private
     */
    var keyMods = {"ctrl": 1, "alt": 2, "option" : 2, "shift": 4, "meta": 8, "command": 8};

    /**
     * @private
     */
    this.keyNames = {
        "8"  : "Backspace",
        "9"  : "Tab",
        "13" : "Enter",
        "27" : "Esc",
        "32" : "Space",
        "33" : "PageUp",
        "34" : "PageDown",
        "35" : "End",
        "36" : "Home",
        "37" : "Left",
        "38" : "Up",
        "39" : "Right",
        "40" : "Down",
        "45" : "Insert",
        "46" : "Del",
        "107": "+",
        "112": "F1",
        "113": "F2",
        "114": "F3",
        "115": "F4",
        "116": "F5",
        "117": "F6",
        "118": "F7",
        "119": "F8",
        "120": "F9",
        "121": "F10",
        "122": "F11",
        "123": "F12",
        "188": ",",
        "219": "[",
        "221": "]"
    };

    var macUnicode = {
        "meta"     : "\u2318", // ⌘
        "command"  : "\u2318",
        "alt"      : "\u2325", // ⌥
        "option"   : "\u2325",
        "shift"    : "\u21E7", // ⇧
        "esc"      : "\u238B", // ⎋
        "ctrl"     : "\u2303", // ⌃
        "backspace": "\u232B", // ⌫
        "del"      : "\u2326", // ⌦
        "enter"    : "\u21A9"  // ↩
    };
    
    var macUnicodeHtml = {
        "meta"     : "&#8984;", // ⌘
        "command"  : "&#8984;",
        "alt"      : "&#8997;", // ⌥
        "option"   : "&#8997;",
        "shift"    : "&#8679;", // ⇧
        "esc"      : "&#9099;", // ⎋
        "ctrl"     : "&#2303;", // ⌃ TODO
        "backspace": "&#232B;", // ⌫ TODO
        "del"      : "&#2326;", // ⌦ TODO
        "enter"    : "&#21A9;"  // ↩ TODO
    };

    // hash to store the hotkeys in
    this.$keys = {};

    var _self = this, trace = 0;
    
    function register(hotkey, handler, remove) {
        var key,
            hashId = 0,
            keys   = hotkey.splitSafe("\\-", null, true),
            i      = 0,
            l      = keys.length;

        for (; i < l; ++i) {
            if (keyMods[keys[i]])
                hashId = hashId | keyMods[keys[i]];
            else
                key = keys[i] || "-"; //when empty, the splitSafe removed a '-'
        }

        
        if (!key) return;
        

        if (!_self.$keys[hashId])
            _self.$keys[hashId] = {};

        if (remove) {
            if (handler == _self.$keys[hashId][key])
                _self.$keys[hashId][key] = null;
        }
        else
            _self.$keys[hashId][key] = handler;
    }

    /**
     * Registers a hotkey handler to a key combination.
     * 
     * #### Example:
     * ```javascript
     *   apf.registerHotkey('Ctrl-Z', undoHandler);
     * ```
     * @param {String}   hotkey  The key combination to user. This is a
     * combination of [[keys: Ctrl]], [[keys: Alt]], [[keys: Shift]] and a normal key to press. Use `+` to
     * seperate the keys.
     * @param {Function} handler The code to be executed when the key
     * combination is pressed.
     */
    apf.registerHotkey = this.register = function(hotkey, handler){
        var parts = hotkey.split("|"),
            i     = 0,
            l     = parts.length;
        for (; i < l; ++i)
            register(parts[i], handler);
    };

    this.$exec = function(eInfo) {
        var handler
        var hashId = 0 | (eInfo.ctrlKey ? 1 : 0) | (eInfo.altKey ? 2 : 0)
            | (eInfo.shiftKey ? 4 : 0) | (eInfo.metaKey ? 8 : 0);
        var code   = eInfo.keyCode;

        var key = _self.keyNames[code] 
            || (code && code > 46 && code != 91 ? String.fromCharCode(code) : null);
        if (!hashId && (!key || !key.match(/^F\d{1,2}$/)) || !key) //Hotkeys should always have one of the modifiers
            return;

        if (_self.$keys[hashId] && (handler = _self.$keys[hashId][key.toLowerCase()])) {
            handler(eInfo.htmlEvent);
            eInfo.returnValue = false;
            
            apf.queue.empty();
            
        }

        return eInfo.returnValue;
    };

    /**
     * Removes a registered hotkey.
     * @param {String} hotkey The hotkey combination to remove
     * @param {Function} handler The code to be executed when the key
     * combination is pressed.
     */
    apf.removeHotkey = this.remove = this.unregister = function(hotkey, handler) {
        var parts = hotkey.split("|"),
            i     = 0,
            l     = parts.length;
        for (; i < l; ++i)
            register(parts[i], handler, true);
    };
    
    function toMacNotation(hotkey, bHtml) {
        var t,
            keys = hotkey.splitSafe("\\-"),
            i    = 0,
            l    = keys.length;

        for (; i < l; ++i) {
            if (!keys[i]) continue;
            if (t = (bHtml ? macUnicodeHtml : macUnicode)[keys[i].toLowerCase()])
                keys[i] = t;
        }
        return keys.join(" ");
    }

    this.toMacNotation = function(hotkey, bHtml) {
        var parts = hotkey.split("|"),
            i     = 0,
            l     = parts.length,
            res   = [];
        for (; i < l; ++i)
            res.push(toMacNotation(parts[i], bHtml));
        return res.join(" | ");
    };

    apf.addEventListener("keydown", function(eInfo) {
        var e = eInfo.htmlEvent;
        //Hotkey
        if (/*!eInfo.isTextInput && */_self.$exec(eInfo) === false
          || eInfo.returnValue === false) {
            apf.stopEvent(e);
            if (apf.canDisableKeyCodes) {
                try {
                    e.keyCode = 0;
                }
                catch(e) {}
            }
            return false;
        }

        
    });
}).call(apf.hotkeys);






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */








/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * @private
 */
apf.nameserver = {
    lookup : {},
    
    add : function(type, item){
        if (!this.lookup[type])
            this.lookup[type] = [];
        
        
        
        return this.lookup[type].push(item) - 1;
    },
    
    register : function(type, id, item){
        if (!this.lookup[type])
            this.lookup[type] = {};

        
        
        if (this.waiting[id]) {
            var list = this.waiting[id];
            for (var i = 0; i < list.length; i++) {
                list[i]();
            }
            delete this.waiting[id];
        }

        return (this.lookup[type][id] = item);
    },
    
    waiting : {},
    waitFor : function(name, callback){
        (this.waiting[name] || (this.waiting[name] = [])).push(callback);
    },
    
    remove : function(type, item){
        var list = this.lookup[type];
        if (list) {
            for (var prop in list) {
                if (list[prop] == item) {
                    delete list[prop];
                }
            }
        }
    },
    
    get : function(type, id){
        return this.lookup[type] ? this.lookup[type][id] : null;
    },
    
    getAll : function(type){
        var name, arr = [], l = this.lookup[type];
        if (!l) return arr;
        
        if (l.dataType == apf.ARRAY) {
            for (var i = 0; i < l.length; i++) {
                arr.push(l[i]);
            }
        }
        else {
            for (name in l) {
                
                
                
                arr.push(l[name]);
            }
        }
        
        return arr;
    }, 
    
    getAllNames : function(type){
        var name, arr = [];
        for (name in this.lookup[type]){
            if (parseInt(name) == name) continue;
            arr.push(name);
        }
        return arr;
    }
};








/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * @todo needs refactor
 * @private
 */
apf.plane = {
    $set    : [],
    $lookup : {},
    $find : function(id){
        if (this.$lookup[id])
            return this.$lookup[id];
        
        var item = this.$set.pop();
        if (!item)
            item = this.$factory();
        
        //item.id = id;
        this.$lookup[id] = item;
        
        return item;
    },
    
    get : function(options){
        return this.$find(options && options.protect || "default");
    },
    
    show : function(o, reAppend, copyCursor, useRealSize, options){
        this.options = options || {};
        var item = this.$find(options && options.protect || "default");
        item.show(o, reAppend, copyCursor, useRealSize, options);
    },
    
    hide : function(protect, noAnim){
        var item = this.$lookup[protect || "default"];
        if (item) {
            item.hide(noAnim);
            delete this.$lookup[protect || "default"];
            this.$set.push(item);
        }
    },

    $factory : function(){
        var _self = this,
            spacerPath = "url(" + (apf.skins.skins["default"] 
            ? apf.skins.skins["default"].mediaPath + "spacer.gif" : "images/spacer.gif") + ")";
        
        function getCover(){
            var obj = document.createElement("DIV");
            
            if(!_self.options || !_self.options.customCover)
                return obj;
            
            obj.innerHTML = apf.getXmlString(_self.options.customCover);
            return obj.firstChild;
        }
        
        function createCover(){
            var cover = document.body.appendChild(getCover());
            if(!_self.options.customCover)
                cover.style.background = spacerPath;

            cover.style.position = "fixed";
            cover.style.left     = 0;
            cover.style.top      = 0;
            cover.host           = false;
            
            return cover;
        }
        
        var plane = createCover();
        
        return {
            host          : this,
            plane         : plane,
            lastCursor    : null,
            lastCoverType :"default",
            
            show : function(o, reAppend, copyCursor, useRealSize, options){
                var coverType = options && options.customCover ? "custom" : "default",
                    plane;
                
                if (coverType == "custom" || this.lastCoverType != coverType)
                    this.plane = createCover();
                
                plane = this.plane;
            
                if(!options || !options.customCover)
                    this.plane.style.background = options && options.color || spacerPath;
                
                this.animate = options && options.animate;
                this.protect = options && options.protect;
                
                if (this.protect)
                    apf.setProperty("planes", (apf.planes || 0) + 1);
                
                if (o) { //@experimental
                    this.current = o;
                    if (reAppend) { 
                        this.$originalPlace = [o.parentNode, o.nextSibling];
                        this.plane.appendChild(o);
                    }
                }
                apf.window.zManager.set("plane", this.plane, !reAppend && o);
                
                useRealSize = apf.isIE;
                var pWidth = (plane.parentNode == document.body
                    ? useRealSize ? document.documentElement.offsetWidth : apf.getWindowWidth()
                    : plane.parentNode.offsetWidth);
         
                var pHeight = (plane.parentNode == document.body
                    ? useRealSize ? document.documentElement.offsetHeight : apf.getWindowHeight()
                    : plane.parentNode.offsetHeight);
                
                if (copyCursor) {
                    if (this.lastCursor === null)
                        this.lastCursor = document.body.style.cursor;
                    document.body.style.cursor = apf.getStyle(o, "cursor");
                }
                
                this.plane.style.display = "block";
                //this.plane.style.left    = p.scrollLeft;
                //this.plane.style.top     = p.scrollTop;
                
                var toOpacity = parseFloat(options && options.opacity) || 1;
                if (this.animate) {
                    var _self = this;
                    apf.setOpacity(this.plane, 0);
                    setTimeout(function(){
                        apf.tween.single(_self.plane, {
                            steps    : 5,
                            interval : 10,
                            type     : "fade",
                            from     : 0,
                            to       : toOpacity
                        });
                    }, 100);
                }
                else
                    apf.setOpacity(this.plane, toOpacity);
                
                var diff = apf.getDiff(plane);
                this.plane.style.width  = "100%";//(pWidth - diff[0]) + "px";
                this.plane.style.height = "100%";//(pHeight - diff[1]) + "px";
        
                this.lastCoverType = options && options.customCover ? "custom" : "default";
        
                return plane;
            },
        
            hide : function(noAnim){
                if (this.protect)
                    apf.setProperty("planes", apf.planes - 1);
                
                var isChild; // try...catch block is needed to work around a FF3 Win issue with HTML elements
                try {
                    isChild = apf.isChildOf(this.plane, document.activeElement);
                }
                catch (ex) {
                    isChild = false;
                }
                if (this.current && this.current.parentNode == this.plane)
                    this.$originalPlace[0].insertBefore(this.current, this.$originalPlace[1]);
                
                if (this.animate && !noAnim) {
                    var _self = this;
                    setTimeout(function(){
                        apf.tween.single(_self.plane, {
                            steps    : 5,
                            interval : 10,
                            type     : "fade",
                            from     : apf.getOpacity(_self.plane),
                            to       : 0,
                            onfinish : function(){
                                _self.plane.style.display  = "none";
                                
                                if (_self.current)
                                    apf.window.zManager.clear(_self.current);
                            }
                        });
                    }, 100);
                }
                else {
                    apf.setOpacity(this.plane, 0);
                    if (this.current)
                        apf.window.zManager.clear(this.plane, this.current);
                    this.plane.style.display  = "none";
                }
                
                if (isChild && apf.document.activeElement) {
                    if (!apf.isIE)
                        document.activeElement.focus();
                    apf.document.activeElement.$focus();
                }
                
                this.current = null;
                
                if (this.lastCursor !== null) {
                    document.body.style.cursor = this.lastCursor;
                    this.lastCursor = null;
                }
                
                return this.plane;
            }
        };
    }
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * @private
 */
apf.popup = {
    cache      : {},
    focusFix   : {"INPUT":1,"TEXTAREA":1,"SELECT":1},
    
    setContent : function(cacheId, content, style, width, height){
        if (!this.popup) this.init();

        this.cache[cacheId] = {
            content : content,
            style   : style,
            width   : width,
            height  : height
        };
        content.style.position = "absolute";
        //if(content.parentNode) content.parentNode.removeChild(content);
        //if(style) apf.importCssString(style, this.popup.document);
        
        content.onmousedown  = function(e) {
            if (!e) e = event;

            
            
            //@todo can this cancelBubble just go?
            //apf.cancelBubble(e, null, true);
            //e.cancelBubble = true;
        };
        
        return content.ownerDocument;
    },
    
    removeContent : function(cacheId){
        this.cache[cacheId] = null;
        delete this.cache[cacheId];
    },
    
    init : function(){
        //consider using iframe
        this.popup = {};
        
        apf.addEventListener("hotkey", function(e){
            if (e.keyCode == "27" || e.altKey) 
                apf.popup.forceHide();
        });
    },
    
    show : function(cacheId, options){
        if (!this.popup) this.init();
        
        options = apf.extend({
            x            : 0,
            y            : 0,
            animate      : false,
            ref          : null,
            width        : null,
            height       : null,
            callback     : null,
            draggable    : false,
            resizable    : false,
            allowTogether: false,
            autoCorrect  : true,
            noleft       : false,
            setZindex    : true
        }, options);
        
        if ((!options.allowTogether 
          || options.allowTogether !== true && options.allowTogether != this.last) 
          && this.last != cacheId
          && this.cache[this.last]
          && (!this.cache[this.last].options || this.cache[this.last].options.autohide !== false))
            this.hide();

        var o = this.cache[cacheId];
        o.options = options;
        //if(this.last != cacheId) 
        //this.popup.document.body.innerHTML = o.content.outerHTML;

        var dp,
            popup  = o.content,
            moveUp = false,
            moveLeft = false,
            fixed  = false;

        if (options.setZindex)
            apf.window.zManager.set("popup", o.content);
        
        if ((dp = o.content.style.display) && dp.indexOf("none") > -1)
            o.content.style.display = "";

        var x = options.x;
        var y = options.y;

        var refNode = options.ref;
        while (refNode && refNode.nodeType == 1) {
            if (fixed = apf.getStyle(refNode, "position") == "fixed")
                break;
            refNode = refNode.parentNode || refNode.$parentNode;
        }

        if (!fixed) {
            if (refNode) {
                var pos = apf.getAbsolutePosition(options.ref, 
                    o.content.offsetParent || o.content.parentNode);
                x = (x || 0) + pos[0];
                y = (y || 0) + pos[1];
            }
            
            if (options.width || o.width)
                popup.style.width = ((options.width || o.width) - 3) + "px";

            popup.style.position = "absolute";
            
            var parentMenu = this.cache[options.allowTogether];
            var pOverflow  = apf.getOverflowParent(o.content);
            var edgeY      = (pOverflow == document.documentElement
                ? (apf.isIE 
                    ? pOverflow.offsetHeight 
                    : (window.innerHeight + window.pageYOffset)) + pOverflow.scrollTop
                : pOverflow.offsetHeight + pOverflow.scrollTop);
            moveUp = options.up || options.autoCorrect && (y
                + (options.height || o.height || o.content.offsetHeight))
                > edgeY;

            if (moveUp) {
                var value;
                if (options.ref)
                    value = (pos[1] - (options.height || o.height || o.content.offsetHeight)) + 3;
                else
                    value = Math.max(0, edgeY - (options.height || o.height || o.content.offsetHeight));
                
                popup.style.top = (!options.up && value < 0 ? y : value) + "px";
            }
            else {
                popup.style.top = y + "px";
            }
            
            if (!options.noleft) {
                var edgeX     = (pOverflow == document.documentElement
                    ? (apf.isIE 
                        ? pOverflow.offsetWidth
                        : (window.innerWidth + window.pageXOffset)) + pOverflow.scrollLeft
                    : pOverflow.offsetWidth + pOverflow.scrollLeft);
                moveLeft = options.autoCorrect && (x
                    + (options.width || o.width || o.content.offsetWidth))
                    > edgeX;

                if (moveLeft) {
                    var value;
                    if (options.ref) {
                        value = (pos[0] - (options.width || o.width || o.content.offsetWidth))
                                + (options.ref.offsetWidth);
                    }
                    else {
                        value = (edgeX - (options.width || o.width || o.content.offsetWidth) 
                                - (parentMenu ? (parentMenu.width || parentMenu.content.offsetWidth) : 0));
                    }
                    popup.style.left = value < 0 ? x : (value - 1) + "px";
                }
                else {
                    popup.style.left = x + "px";
                }
            }
        }
        else {
            pos = apf.getAbsolutePosition(options.ref, refNode);
            y = (y || 0) + pos[1] + refNode.offsetTop;
            pos[0] += refNode.offsetLeft;
            popup.style.position = "fixed";
            popup.style.top      = y + "px";
            
            if (!options.noleft)
                popup.style.left = x + "px";
        }

        
        // set a className that specifies the direction, to help skins with
        // specific styling options.
        apf.setStyleClass(popup, moveUp ? "upward" : "downward", [moveUp ? "downward" : "upward"]);
        apf.setStyleClass(popup, moveLeft ? "moveleft" : "moveright", [moveLeft ? "moveright" : "moveleft"]);
        

        if (options.animate) {
            if (options.animate == "fade") {
                apf.tween.single(popup, {
                    type  : 'fade',
                    from  : 0,
                    to    : 1,
                    anim  : apf.tween.NORMAL,
                    steps : options.steps || 15 * apf.animSteps
                });
            }
            else {
                var iVal, steps = apf.isIE8 ? 5 : 7, i = 0;
                iVal = setInterval(function(){
                    var value = ++i * ((options.height || o.height) / steps);

                    popup.style.height = value + "px";
                    if (moveUp)
                        popup.style.top = (y - value - (options.y || 0)) + "px";
                    else
                        (options.container || popup).scrollTop = -1 * (i - steps) * ((options.height || o.height) / steps);
                    popup.style.display = "block";

                    if (i >= steps) {
                        clearInterval(iVal)
                        
                        if (options.callback)
                            options.callback(popup);
                    }
                }, 10);
            }
        }
        else {
            if (!refNode) {
                if (options.height || o.height)
                    popup.style.height = (options.height || o.height) + "px";
                value = (edgeY - (options.height || o.height || o.content.offsetHeight));
                popup.style.top = y + (options.height || o.height || o.content.offsetHeight) < edgeY 
                                    ? y 
                                    : value 
                                  + "px";
            }
            popup.style.display = "block";
            
            if (options.callback)
               options.callback(popup);
        }

        $setTimeout(function(){
            apf.popup.last = cacheId;
        });

        if (options.draggable) {
            options.id = cacheId;
            this.makeDraggable(options);
        }
    },
    
    hide : function(){
        if (this.isDragging) return;

        var o = this.cache[this.last];
        if (o) {
            if (o.content)
                o.content.style.display = "none";

            if (o.options && o.options.onclose) {
                o.options.onclose(apf.extend(o.options, {htmlNode: o.content}));
                o.options.onclose = false;
            }
        }
    },
    
    isShowing : function(cacheId){
        return this.last && this.last == cacheId 
            && this.cache[this.last]
            && this.cache[this.last].content.style.display != "none";
    },

    isDragging   : false,

    makeDraggable: function(options) {
        if (!apf.Interactive || this.cache[options.id].draggable) 
            return;

        var oHtml = this.cache[options.id].content;
        this.cache[options.id].draggable = true;
        var o = {
            $propHandlers : {},
            minwidth      : 10,
            minheight     : 10,
            maxwidth      : 10000,
            maxheight     : 10000,
            dragOutline   : false,
            resizeOutline : false,
            draggable     : true,
            resizable     : options.resizable,
            $ext          : oHtml,
            oDrag         : oHtml.firstChild
        };

        oHtml.onmousedown =
        oHtml.firstChild.onmousedown = function(e){
            if (!e) e = event;
            
            
            
            (e || event).cancelBubble = true;
        }

        apf.implement.call(o, apf.Interactive);

        o.$propHandlers["draggable"].call(o, true);
        o.$propHandlers["resizable"].call(o, true);
    },
    
    getCurrentElement : function(){
        return typeof this.last == "number" && apf.lookup(this.last);
    },
    
    $mousedownHandler : function(amlNode, e){
        if (!this.last || (amlNode && this.last == amlNode.$uniqueId) || !this.cache[this.last])
          return;

        var htmlNode = e.srcElement || e.target;
        
        var uId = this.last;
        
        while (this.cache[uId]) {
            if (apf.isChildOf(this.cache[uId].content, htmlNode, true))
                return;
            
            if (!this.cache[uId].options)
                return;
            
            uId = this.cache[uId].options.allowTogether;
        }
        
        this.forceHide();
    },
    
    forceHide : function(){
        if (this.last 
          
          && !apf.plane.current
          
          && this.isShowing(this.last)
          && this.cache[this.last]
          && this.cache[this.last].options
          && this.cache[this.last].options.autohide !== false) {
            var o = apf.lookup(this.last);
            if (!o)
                this.last = null;
            else if (o.dispatchEvent("popuphide") !== false)
                this.hide();
        }
    },

    destroy : function(){
        for (var cacheId in this.cache) {
            if (this.cache[cacheId]) {
                this.cache[cacheId].content.onmousedown = null;
                apf.destroyHtmlNode(this.cache[cacheId].content);
                this.cache[cacheId].content = null;
                this.cache[cacheId] = null;
            }
        }
        
        if (!this.popup) return;
        //this.popup.document.body.c = null;
        //this.popup.document.body.onmouseover = null;
    }
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * This method sets a single CSS rule.
 * @param {String} name         The CSS name of the rule (i.e. `.cls` or `#id`).
 * @param {String} type         The CSS property to change.
 * @param {String} value        The CSS value of the property.
 * @param {String} [stylesheet] The name of the stylesheet to change.
 * @param {Object} [win]        A reference to a window
 */
apf.setStyleRule = function(name, type, value, stylesheet, win){
    name = name.toLowerCase();
    
    if (!stylesheet) {
        var sheets = (win || self).document.styleSheets;
        for (var j = sheets.length - 1; j >= 0; j--) {
            try {
                var rules = sheets[j][apf.styleSheetRules];
                if (!rules) return false;
                
                for (var i = 0; i < rules.length; i++) {
                    if (rules.item(i).selectorText && rules.item(i).selectorText.toLowerCase() == name) {
                        rules.item(i).style[type] = value;
                        return true;
                    }
                }
            }
            catch(e){}
        }
    }
    else {
        if (!stylesheet)
            stylesheet = (win || self).document.styleSheets[0];
        
        var rules = stylesheet[apf.styleSheetRules];
        if (!rules) return false;
        
        for (var i = 0; i < rules.length; i++) {
            if (rules.item(i).selectorText && rules.item(i).selectorText.toLowerCase() == name) {
                rules.item(i).style[type] = value;
                return true;
            }
        }
    }
    
    return false;
};

/**
 * This method gets a single CSS rule.
 * @param {String} name         The CSS name of the rule (i.e. `.cls` or `#id`).
 * @param {String} type         The CSS property to change.
 * @param {String} [stylesheet] The name of the stylesheet to change.
 * @param {Object} [win]        A reference to a window
 */
apf.getStyleRule = function(name, type, stylesheet, win){
    name = name.toLowerCase();
    
    if (!stylesheet) {
        var sheets = (win || self).document.styleSheets;
        for (var j = sheets.length - 1; j >= 0; j--) {
            try {
                var rules = sheets[j][apf.styleSheetRules];
                for (var i = 0; i < rules.length; i++) {
                    if (rules.item(i).selectorText && rules.item(i).selectorText.toLowerCase() == name) {
                        return rules.item(i).style[type];
                    }
                }
            }
            catch(e){}
        }
    }
    else {
        var rules = (win || self).document.styleSheets[stylesheet || 0][apf.styleSheetRules];
        for (var i = 0; i < rules.length; i++) {
            if (rules.item(i).selectorText && rules.item(i).selectorText.toLowerCase() == name) {
                return rules.item(i).style[type];
            }
        }
    }
    
    return false;
};

/**
 * This method adds one class name to an HTMLElement. It can also remove classes.
 * @param {HTMLElement} oHtml        The HTMLElement to apply the CSS class to.
 * @param {String}      className    The name of the CSS class to apply.
 * @param {Array}       [exclusion]  A list of strings specifying names of CSS classes to remove.
 * @returns {HTMLElement} The modified `oHtml` element.
 */
apf.setStyleClass = function(oHtml, className, exclusion, userAction){
    if (!oHtml || userAction && this.disabled)
        return;

    

    if (className) {
        if (exclusion)
            exclusion[exclusion.length] = className;
        else
            exclusion = [className];
    }

    //Create regexp to remove classes
    //var re = new RegExp("(?:(^| +)" + (exclusion ? exclusion.join("|") : "") + "($| +))", "gi");
    var re = new RegExp("(^| +)(?:" + (exclusion ? exclusion.join("|") : "") + ")", "gi");

    //Set new class
    oHtml.className != null
        ? (oHtml.className = oHtml.className.replace(re, " ") + (className ? " " + className : ""))
        : oHtml.setAttribute("class", (oHtml.getAttribute("class") || "")
            .replace(re, " ") + (className ? " " + className : ""));

    return oHtml;
};

/**
 * This method imports a CSS stylesheet from a string.
 * @param {String} cssString  The CSS definition
 * @param {Object} [doc]      The reference to the document where the CSS is applied on
 * @param {String} [media]    The media to which this CSS applies (_i.e._ `print` or `screen`)
 */
apf.importCssString = function(cssString, doc, media){
    doc = doc || document;
    var htmlNode = doc.getElementsByTagName("head")[0];//doc.documentElement.getElementsByTagName("head")[0];

    

    if (apf.canCreateStyleNode) {
        //var head  = document.getElementsByTagName("head")[0];
        var style = doc.createElement("style");
        style.appendChild(doc.createTextNode(cssString));
        if (media)
            style.setAttribute('media', media);
        htmlNode.appendChild(style);
    }
    else {
        htmlNode.insertAdjacentHTML("beforeend", ".<style media='"
         + (media || "all") + "'>" + cssString + "</style>");

        /*if(document.body){
            document.body.style.height = "100%";
            $setTimeout('document.body.style.height = "auto"');
        }*/
    }
};

/**
 * This method retrieves the current value of a property on a HTML element
 * recursively. If the style isn't found on the element itself, its parent is
 * checked.
 * @param {HTMLElement} el    The element to read the property from
 * @param {String}      prop  The property to read
 * @returns {String} The retrieved value
 */
apf.getStyleRecur = function(el, prop) {
    var value = apf.hasComputedStyle
        ? document.defaultView.getComputedStyle(el,'').getPropertyValue(
            prop.replace(/([A-Z])/g, function(m, m1){
                return "-" + m1.toLowerCase();
            }))
        : el.currentStyle[prop]

    return ((!value || value == "transparent" || value == "inherit")
      && el.parentNode && el.parentNode.nodeType == 1)
        ? this.getStyleRecur(el.parentNode, prop)
        : value;
};

/**
 * This method imports a stylesheet defined by a multidimensional array. 
 * @param {Array}    def  A multidimensional array specifying stylesheets to import
 * @param {Object}   [win] A reference to a window
 * @method
 * @deprecated
 */    
apf.importStylesheet = function (def, win, stylesheet) {
    if (!def.length)
        return;
    
    if (!stylesheet) {
        var re = new RegExp("^" + document.domain, 'g');
        var doc = (win || window).document;
        for (var index=document.styleSheets.length - 1; index >= 0; index--) {
            if (!doc.styleSheets[index].href || doc.styleSheets[index].href.match(re)) {
                break;
            }
        }
        stylesheet = doc.styleSheets[index];
    }
    
    if (!stylesheet)
        stylesheet = apf.createStylesheet(win);
    
    for (var i = 0; i < def.length; i++) {
        if (!def[i][1])
            continue;

        if (apf.isIE)
            stylesheet.addRule(def[i][0], def[i][1]);
        else {
            var rule = def[i][0] + " {" + def[i][1] + "}";
            try {
                stylesheet.insertRule(rule, 0);
            }
            catch (e) {
                stylesheet = newStyleSheet();
                stylesheet.insertRule(rule, 0);
            }
        }
    }
};

/**
 * This method constructs a stylesheet.
 * @param {Object}  [win] A reference to a window
 * @returns {String} The created CSS stylesheet
 */ 
apf.createStylesheet = function(win){
    var doc = (win || window).document;
    
    if (doc.createStyleSheet)
        return doc.createStyleSheet();
    else {
        var elem = doc.createElement("style");
        elem.type = "text/css";
        doc.getElementsByTagName("head")[0].appendChild(elem);
        return elem.sheet;
    }
};

/**
 * This method determines if specified coordinates are within the HTMLElement.
 * @param {HTMLElement} el  The element to check
 * @param {Number}      x   The x-coordinate in pixels
 * @param {Number}      y   The y-coordinate in pixels
 * @returns {Boolean} `true` if the coordinates are within the element.
 */
apf.isInRect = function(oHtml, x, y){
    var pos = this.getAbsolutePosition(oHtml);
    if (x < pos[0] || y < pos[1] || x > oHtml.offsetWidth + pos[0] - 10
      || y > oHtml.offsetHeight + pos[1] - 10)
        return false;
    return true;
};

/**
 * Retrieves the parent providing the rectangle to which the HTMLElement is
 * bound and cannot escape. In CSS, this is accomplished by having the overflow
 * property set to `"hidden"` or `"auto"`.
 * @param {HTMLElement} o  The element to check
 * @returns {HTMLElement} The parent element
 */
apf.getOverflowParent = function(o){
    //not sure if this is the correct way. should be tested

    o = o.offsetParent;
    while (o && (this.getStyle(o, "overflow") != "hidden"
      || "absolute|relative".indexOf(this.getStyle(o, "position")) == -1)) {
        o = o.offsetParent;
    }
    return o || document.documentElement;
};

/**
 * Retrieves the first parent element which has a position `absolute` or
 * `relative` set.
 * @param {HTMLElement} o  The element to check
 * @returns {HTMLElement} The parent element
 */
apf.getPositionedParent = function(o){
    o = o.offsetParent;
    while (o && o.tagName.toLowerCase() != "body"
      && "absolute|relative".indexOf(this.getStyle(o, "position")) == -1) {
        o = o.offsetParent;
    }
    return o || document.documentElement;
};

/**
 * Retrieves the absolute x- and y-coordinates, relative to the browser's
 * drawing area or the specified `refParent`.
 * @param {HTMLElement} o           The element to check
 * @param {HTMLElement} [refParent] The reference parent
 * @param {Boolean}     [inclSelf]  Whether to include the position of the element to check in the return value.
 * @returns {Array} The x- and y-coordinates of `oHtml`.
 */
apf.getAbsolutePosition = function(o, refParent, inclSelf){
    if ("getBoundingClientRect" in document.documentElement) { 
        if (apf.doesNotIncludeMarginInBodyOffset && o == document.body) {
            return [
                o.offsetLeft + (parseFloat(apf.getStyle(o, "marginLeft")) || 0),
                  + (o.scrollLeft || 0),
                o.offsetTop  + (parseFloat(apf.getStyle(o, "marginTop")) || 0)
                  + (o.scrollTop || 0)
            ];
        }
        
        var box  = o.getBoundingClientRect(), 
            top  = box.top,
            left = box.left,
            corr = (apf.isIE && apf.isIE < 8);

        if (refParent && refParent != document.body) {
            var pos = apf.getAbsolutePosition(refParent, null, true);
            top -= pos[1];
            left -= pos[0];
        }
        
        if (!(apf.isIE && o == document.documentElement)) {
            left += (refParent || document.body).scrollLeft || document.documentElement.scrollLeft || 0;
            top  += (refParent || document.body).scrollTop  || document.documentElement.scrollTop  || 0;
        }
        
        if (inclSelf && !refParent) {
            left += parseInt(apf.getStyle(o, "borderLeftWidth")) || 0
            top  += parseInt(apf.getStyle(o, "borderTopWidth")) || 0;
        }

        return [left - (corr ? 2 : 0), top - (corr ? 2 : 0)];
    }
    
    //@todo code below might be deprecated one day
    var wt = inclSelf ? 0 : o.offsetLeft, ht = inclSelf ? 0 : o.offsetTop;
    o = inclSelf ? o : o.offsetParent || o.parentNode ;

    if (apf.isIE8 && refParent) {
        bw = this.getStyle(o, "borderLeftWidth");
        wt -= (apf.isIE && o.currentStyle.borderLeftStyle != "none" 
          && bw == "medium" ? 2 : parseInt(bw) || 0);
        bh = this.getStyle(o, "borderTopWidth");
        ht -= (apf.isIE && o.currentStyle.borderTopStyle != "none" 
          && bh == "medium" ? 2 : parseInt(bh) || 0);
    }

    var bw, bh, fl;
    while (o && o != refParent) {//&& o.tagName.toLowerCase() != "html"
        //Border - Left
        bw = apf.isOpera || apf.isIE8 ? 0 : this.getStyle(o, "borderLeftWidth");

        wt += (apf.isIE && o.currentStyle.borderLeftStyle != "none" && bw == "medium"
            ? 2
            : parseInt(bw) || 0) + o.offsetLeft;

        if (apf.isIE && !apf.isIE8 && apf.getStyle(o, "styleFloat") == "none" 
          && apf.getStyle(o, "position") == "relative") {
            var q = o.previousSibling;
            while (q) {
                if (q.nodeType == 1) {
                    fl = apf.getStyle(q, "styleFloat");
                    if (fl == "left") {
                        wt -= parseInt(apf.getStyle(o, "marginLeft")) 
                            || 0;//-1 * (o.parentNode.offsetWidth - o.offsetWidth)/2; //assuming auto
                        break;
                    }
                    else if (fl == "right")
                        break;
                }
                q = q.previousSibling;
            }
        }

        //Border - Top
        bh = apf.isOpera || apf.isIE8 ? 0 : this.getStyle(o, "borderTopWidth");
        ht += (apf.isIE && o.currentStyle.borderTopStyle != "none" && bh == "medium"
            ? 2
            : parseInt(bh) || 0) + o.offsetTop;

        //Scrolling
        if (!apf.isGecko && o != refParent && (o.tagName != "HTML" || o.ownerDocument != document)) {
            wt -= o.scrollLeft;
            ht -= o.scrollTop;
        }
        
        //Table support
        if (o.tagName.toLowerCase() == "table") {
            ht -= parseInt(o.border || 0) + parseInt(o.cellSpacing || 0);
            wt -= parseInt(o.border || 0) + parseInt(o.cellSpacing || 0) * 2;
        }
        else if (o.tagName.toLowerCase() == "tr") {
            var cp;
            ht -= (cp = parseInt(o.parentNode.parentNode.cellSpacing));
            while (o.previousSibling)
                ht -= (o = o.previousSibling).offsetHeight + cp;
        }

        if (apf.isIE && !o.offsetParent && o.parentNode.nodeType == 1) {
            wt -= o.parentNode.scrollLeft;
            ht -= o.parentNode.scrollTop;
        }

        o = o.offsetParent;
    }

    return [wt, ht];
};

//@todo its much faster to move these to browser specific files and eliminate apf.getStyle()
/**
 * Returns the distance between the border left and border right values of an element.
 * @param {HTMLElement} oHtml The element to check
 * @returns {Number} The final calculation, or 0, if there's no difference
 * @see apf.getWidthDiff
 */
apf.getHorBorders = function(oHtml){
    return Math.max(0,
          (parseInt(apf.getStyle(oHtml, "borderLeftWidth")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderRightWidth")) || 0));
};

/**
 * Returns the distance between the border top and border bottom values of an element.
 * @param {HTMLElement} oHtml The element to check
 * @returns {Number} The final calculation, or 0, if there's no difference
 */
apf.getVerBorders = function(oHtml){
    return Math.max(0,
          (parseInt(apf.getStyle(oHtml, "borderTopWidth")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderBottomWidth")) || 0));
};

/**
 * Returns the distance between the border left and border right values of an element, taking padding into consideration.
 * @param {HTMLElement} oHtml The element to check
 * @returns {Number} The final calculation, or 0, if there's no difference
 * @see apf.getHorBorders
 */
apf.getWidthDiff = function(oHtml){
    if (apf.hasFlexibleBox 
      && apf.getStyle(oHtml, apf.CSSPREFIX + "BoxSizing") != "content-box")
        return 0;
    
    return Math.max(0, (parseInt(apf.getStyle(oHtml, "paddingLeft")) || 0)
        + (parseInt(apf.getStyle(oHtml, "paddingRight")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderLeftWidth")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderRightWidth")) || 0));
};

/**
 * Returns the distance between the border top and border bottom values of an element, taking padding into consideration.
 * @param {HTMLElement} oHtml The element to check
 * @returns {Number} The final calculation, or 0, if there's no difference
 */
apf.getHeightDiff = function(oHtml){
    if (apf.hasFlexibleBox 
      && apf.getStyle(oHtml, apf.CSSPREFIX + "BoxSizing") != "content-box")
        return 0;
    
    return Math.max(0, (parseInt(apf.getStyle(oHtml, "paddingTop")) || 0)
        + (parseInt(apf.getStyle(oHtml, "paddingBottom")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderTopWidth")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderBottomWidth")) || 0));
};

/**
 * Returns an array with two elements. The first is the distance between the border top and border bottom values of an element, taking padding into consideration; 
 * the second is the distance between the border top and border bottom values of an element, taking padding into consideration.
 * @param {HTMLElement} oHtml The element to check
 * @returns {[Number]} An array containing the differences
 */
apf.getDiff = function(oHtml){
    if (apf.hasFlexibleBox 
      && apf.getStyle(oHtml, apf.CSSPREFIX + "BoxSizing") != "content-box")
        return [0,0];
    
    return [Math.max(0, (parseInt(apf.getStyle(oHtml, "paddingLeft")) || 0)
        + (parseInt(apf.getStyle(oHtml, "paddingRight")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderLeftWidth")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderRightWidth")) || 0)),
        Math.max(0, (parseInt(apf.getStyle(oHtml, "paddingTop")) || 0)
        + (parseInt(apf.getStyle(oHtml, "paddingBottom")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderTopWidth")) || 0)
        + (parseInt(apf.getStyle(oHtml, "borderBottomWidth")) || 0))];
};

/**
 * Returns an array with two elements. The first is the distance between the margin left and margin right values of an element; 
 * the second is the distance between the margin top top and margin bottom values of an element.
 * @param {HTMLElement} oHtml The element to check
 * @returns {[Number]} An array containing the differences
 */
apf.getMargin = function(oHtml) {
    return [(parseInt(apf.getStyle(oHtml, "marginLeft")) || 0)
        + (parseInt(apf.getStyle(oHtml, "marginRight")) || 0),
      (parseInt(apf.getStyle(oHtml, "marginTop")) || 0)
        + (parseInt(apf.getStyle(oHtml, "marginBottom")) || 0)]
};

/**
 * Returns the difference between an element's `offsetWidth`, with its border left and border right widths removed. 
 * @param {HTMLElement} oHtml The element to check
 * @returns {Number} The final calculation
 */
apf.getHtmlInnerWidth = function(oHtml){
    return (oHtml.offsetWidth
        - (parseInt(apf.getStyle(oHtml, "borderLeftWidth")) || 0)
        - (parseInt(apf.getStyle(oHtml, "borderRightWidth")) || 0));
};

/**
 * Returns the difference between an element's `offsetWidth`, with its border top and border bottom widths removed. 
 * @param {HTMLElement} oHtml The element to check
 * @returns {Number} The final calculation
 */
apf.getHtmlInnerHeight = function(oHtml){
    return (oHtml.offsetHeight
        - (parseInt(apf.getStyle(oHtml, "borderTopWidth")) || 0)
        - (parseInt(apf.getStyle(oHtml, "borderBottomWidth")) || 0));
};

/**
 * Returns the viewport of a window.
 *
 * @param  {WindowImplementation} [win] The window to take the measurements of.
 * @returns {Object}                    Viewport object with  x, y, w, and h properties.
 */
apf.getViewPort = function(win) {
    win = win || window;
    var doc = (!win.document.compatMode
      || win.document.compatMode == "CSS1Compat")
        //documentElement for an iframe
        ? win.document.html || win.document.documentElement
        : win.document.body;

    // Returns viewport size excluding scrollbars
    return {
        x     : win.pageXOffset || doc.scrollLeft,
        y     : win.pageYOffset || doc.scrollTop,
        width : win.innerWidth  || doc.clientWidth,
        height: win.innerHeight || doc.clientHeight
    };
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * Opens a window with the string in it.
 * @param {String} str The HTML string to display in the new window.
 */
apf.pasteWindow = function(str){
    var win = window.open("about:blank");
    win.document.write(str);
};



/**
 * Escapes HTML from a string.
 * @param {String} str The html to be escaped.
 * @return {String} The escaped string.
 */
apf.htmlentities = function(str){
    return str.escapeHTML();
};

/**
 * Escape an XML string, making it ascii compatible.
 * @param {String} str The xml string to escape.
 * @return {String} The escaped string.
 * @method xmlentities
 *
 */
 /* @todo This function does something completely different from htmlentities, 
 *       the name is confusing and misleading.
 */
apf.xmlentities = apf.escapeXML;

/**
 * Unescape an HTML string.
 * @param {String} str The string to unescape.
 * @return {String} The unescaped string.
 */
apf.html_entity_decode = function(str){
    return (str || "").replace(/\&\#38;/g, "&").replace(/&lt;/g, "<")
        .replace(/&gt;/g, ">").replace(/&amp;/g, "&").replace(/&nbsp;/g, " ");
};



/**
 * Determines whether the keyboard input was a character that can influence
 * the value of an element (like a textbox).
 * @param {Number} charCode The ascii character code
 * @returns {Boolean} `true` if it was a character
 */
apf.isCharacter = function(charCode){
    return (charCode < 112 || charCode > 122)
      && (charCode == 32 || charCode > 42 || charCode == 8);
};

/**
 * This random number generator has been added to provide a more robust and
 * reliable random number spitter than the native Ecmascript Math.random()
 * function.
 * 
 * It is an implementation of the Park-Miller algorithm. (See 'Random Number
 * Generators: Good Ones Are Hard to Find', by Stephen K. Park and Keith W.
 * Miller, Communications of the ACM, 31(10):1192-1201, 1988.)
 * @class apf.randomGenerator
 * @author David N. Smith of IBM's T. J. Watson Research Center.
 * @author Mike de Boer (mike AT c9 DOT io)
 * 
 */
apf.randomGenerator = {
    d: new Date(),
    seed: null,
    A: 48271,
    M: 2147483647,
    Q: null,
    R: null,
    oneOverM: null,

    /**
     * Generates a random [[Number]] between a lower and upper boundary.
     * The algorithm uses the system time, in minutes and seconds, to 'seed'
     * itself, that is, to create the initial values from which it will generate
     * a sequence of numbers. If you are familiar with random number generators,
     * you might have reason to use some other value for the seed. Otherwise,
     * you should probably not change it.
     * @param {Number} lnr Lower boundary
     * @param {Number} unr Upper boundary
     * @returns A random number between `lnr` and`unr`
     * @type Number
     */
    generate: function(lnr, unr) {
        if (this.seed == null)
            this.seed = 2345678901 + (this.d.getSeconds() * 0xFFFFFF) + (this.d.getMinutes() * 0xFFFF);
        this.Q = this.M / this.A;
        this.R = this.M % this.A;
        this.oneOverM = 1.0 / this.M;
        return Math.floor((unr - lnr + 1) * this.next() + lnr);
    },

    /**
     * Returns a new random number, based on the 'seed', generated by the
     * [[apf.randomGenerator.generate]] method.
     * @type Number
     */
    next: function() {
        var hi = this.seed / this.Q;
        var lo = this.seed % this.Q;
        var test = this.A * lo - this.R * hi;
        if (test > 0)
            this.seed = test;
        else
            this.seed = test + this.M;
        return (this.seed * this.oneOverM);
    }
};

/**
 * Adds a time stamp to the URL to prevent the browser from caching it.
 * @param {String} url The URL to add the timestamp to.
 * @return {String} The url... with a timestamp!
 */
apf.getNoCacheUrl = function(url){
    return url
        + (url.indexOf("?") == -1 ? "?" : "&")
        + "nocache=" + new Date().getTime();
};

/**
 * Checks if the string contains curly braces at the start and end. If so, it's
 * processed as Javascript via `eval()`. Otherwise, the original string is returned.
 * @param {String} str The string to parse.
 * @return {String} The result of the parsing.
 */
apf.parseExpression = function(str){
    if (!apf.parseExpression.regexp.test(str))
        return str;

    
        return eval(RegExp.$1);
    
};
apf.parseExpression.regexp = /^\{([\s\S]*)\}$/;

/**
 * @private
 */
apf.formatNumber = function(num, prefix){
    var nr = parseFloat(num);
    if (!nr) return num;

    var str = new String(Math.round(nr * 100) / 100).replace(/(\.\d?\d?)$/, function(m1){
        return m1.pad(3, "0", apf.PAD_RIGHT);
    });
    if (str.indexOf(".") == -1)
        str += ".00";

    return prefix + str;
};

/**
 * Execute a script in the global scope.
 *
 * @param {String} str  The JavaScript code to execute
 * @param {Object} [win] A reference to a window
 * @return {String} The executed JavaScript code
 */
apf.jsexec = function(str, win){
    if (!str)
        return str;
    if (!win)
        win = self;

    if (apf.isO3)
        eval(str, self);
    else if (apf.hasExecScript) {
        win.execScript(str);
    }
    else {
        var head = win.document.getElementsByTagName("head")[0];
        if (head) {
            var script = win.document.createElement('script');
            script.setAttribute('type', 'text/javascript');
            script.text = str;
            head.appendChild(script);
            head.removeChild(script);
        } else
            eval(str, win);
    }

    return str;
};

/*
 * Shorthand for an empty function.
 */
apf.K = function(){};



/**
 * Reliably determines whether a variable is a Number.
 *
 * @param {Mixed}   value The variable to check
 * @type  {Boolean} `true` if the argument is a number
 */
apf.isNumber = function(value){
    return parseFloat(value) == value;
};

/**
 * Reliably determines whether a variable is an array. For more information, see 
 * <http://perfectionkills.com/instanceof-considered-harmful-or-how-to-write-a-robust-isarray/>
 *
 * @param {Mixed}   value The variable to check
 * @type  {Boolean} `true` if the argument is an array
 */
apf.isArray = function(value) {
    return Object.prototype.toString.call(value) === "[object Array]";
};

/**
 * Determines whether a string is true (in the HTML attribute sense).
 * @param {Mixed} value The variable to check. Possible truth values include:
 *  - true  
 *  - 'true'
 *  - 'on'  
 *  - 1     
 *  - '1'   
 * @return {Boolean} Whether the string is considered to imply truth.
 */
apf.isTrue = function(c){
    return (c === true || c === "true" || c === "on" || typeof c == "number" && c > 0 || c === "1");
};

/**
 * Determines whether a string is false (in the HTML attribute sense).
 * @param {Mixed} value The variable to check. Possible false values include:
 *   - false   
 *   - 'false' 
 *   - 'off'   
 *   - 0       
 *   - '0'     
 * @return {Boolean} whether the string is considered to imply untruth.
 */
apf.isFalse = function(c){
    return (c === false || c === "false" || c === "off" || c === 0 || c === "0");
};

/**
 * Determines whether a value should be considered false. This excludes, amongst
 * others, the number 0.
 * @param {Mixed} value The variable to check
 * @return {Boolean} Whether the variable is considered false.
 */
apf.isNot = function(c){
    // a var that is null, false, undefined, Infinity, NaN and c isn't a string
    return (!c && typeof c != "string" && c !== 0 || (typeof c == "number" && !isFinite(c)));
};

/**
 * Creates a relative URL based on an absolute URL.
 * @param {String} base The start of the URL to which relative URL's work.
 * @param {String} url  The URL to transform.
 * @return {String} The relative URL.
 */
apf.removePathContext = function(base, url){
    if (!url)  return "";

    if (url.indexOf(base) > -1)
        return url.substr(base.length);

    return url;
};

/*
 * @private
 * @todo why is this done like this?
 */
apf.cancelBubble = function(e, o, noPropagate){
    if (e.stopPropagation)
        e.stopPropagation()
    else 
        e.cancelBubble = true;
    
    //if (o.$focussable && !o.disabled)
        //apf.window.$focus(o);
    
    
    /*if (apf.isIE)
        o.$ext.fireEvent("on" + e.type, e);
    else 
        o.$ext.dispatchEvent(e.name, e);*/
    
    if (!noPropagate) {
        if (o && o.$ext && o.$ext["on" + (e.type || e.name)])
            o.$ext["on" + (e.type || e.name)](e);
        apf.window.$mousedown(e);
    }
    //if (apf.isGecko)
        //apf.window.$mousedown(e);
    
    
};



/*
 * Attempt to fix memory leaks
 * @private
 */
apf.destroyHtmlNode = function (element) {
    if (!element) return;

    if (!apf.isIE || element.ownerDocument != document) {
        if (element.parentNode)
            element.parentNode.removeChild(element);
        return;
    }

    var garbageBin = document.getElementById('IELeakGarbageBin');
    if (!garbageBin) {
        garbageBin    = document.createElement('DIV');
        garbageBin.id = 'IELeakGarbageBin';
        garbageBin.style.display = 'none';
        document.body.appendChild(garbageBin);
    }

    // move the element to the garbage bin
    garbageBin.appendChild(element);
    garbageBin.innerHTML = '';
};


/**
 * @private
 */
apf.getRules = function(node){
    var rules = {};

    for (var w = node.firstChild; w; w = w.nextSibling){
        if (w.nodeType != 1)
            continue;
        else {
            if (!rules[w[apf.TAGNAME]])
                rules[w[apf.TAGNAME]] = [];
            rules[w[apf.TAGNAME]].push(w);
        }
    }

    return rules;
};


apf.isCoord = function (n){
    return n || n === 0;
};

apf.getCoord = function (n, other){
    return n || n === 0 ? n : other;
};

/**
 * @private
 */
apf.getBox = function(value, base){
    if (!base) base = 0;

    if (value == null || (!parseInt(value) && parseInt(value) != 0))
        return [0, 0, 0, 0];

    var x = String(value).splitSafe(" ");
    for (var i = 0; i < x.length; i++)
        x[i] = parseInt(x[i]) || 0;
    switch (x.length) {
        case 1:
            x[1] = x[0];
            x[2] = x[0];
            x[3] = x[0];
            break;
        case 2:
            x[2] = x[0];
            x[3] = x[1];
            break;
        case 3:
            x[3] = x[1];
            break;
    }

    return x;
};

/**
 * @private
 */
apf.getNode = function(data, tree){
    var nc = 0;//nodeCount
    //node = 1
    if (data != null) {
        for (var i = 0; i < data.childNodes.length; i++) {
            if (data.childNodes[i].nodeType == 1) {
                if (nc == tree[0]) {
                    data = data.childNodes[i];
                    if (tree.length > 1) {
                        tree.shift();
                        data = this.getNode(data, tree);
                    }
                    return data;
                }
                nc++
            }
        }
    }

    return null;
};

/**
 * Retrieves the first XML node with a `nodeType` of 1 from the children of an XML element.
 * @param {XMLElement} xmlNode The XML element that is the parent of the element to select.
 * @return {XMLElement} The first child element of the XML parent.
 * @throws An error when no child element is found.
 */
apf.getFirstElement = function(xmlNode){
    

    return xmlNode.firstChild.nodeType == 1
        ? xmlNode.firstChild
        : xmlNode.firstChild.nextSibling;
};

/**
 * Retrieves the last XML node with `nodeType` of 1 from the children of an XML element.
 * @param {XMLElement} xmlNode The XML element that is the parent of the element to select.
 * @return {XMLElement} The last child element of the XML parent.
 * @throw An error when no child element is found.
 */
apf.getLastElement = function(xmlNode){
    

    return xmlNode.lastChild.nodeType == 1
        ? xmlNode.lastChild
        : xmlNode.lastChild.previousSibling;
};

/**
 * Selects the content of an HTML element. This currently only works in
 * Internet Explorer.
 * @param {HTMLElement} oHtml The container in which the content receives the selection.
 */
apf.selectTextHtml = function(oHtml){
    if (!apf.hasMsRangeObject) return;// oHtml.focus();

    var r = document.selection.createRange();
    try {r.moveToElementText(oHtml);} catch(e){}
    r.select();
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * Manages visibility hooks for elements that need to be visible to set their
 * layout.
 *
 * @private
 */
apf.visibilitymanager = function(){
    var tree  = {};
    var _self  = this;
    var inited = false;
    
    this.check = function(amlNode, type, callback) {
        if (amlNode.$ext.offsetHeight || amlNode.$ext.offsetWidth)
            return true;

        if (amlNode.$visibleCheck) {
            if (amlNode.$visibleCheck[type])
                return;
        }
        else
            amlNode.$visibleCheck = {};

        function cleanup(setInsertion){
            var p = amlNode;
            while (p) {
                p.removeEventListener("prop.visible", check);
                p.removeEventListener("DOMNodeRemoved", remove); 
                p.removeEventListener("DOMNodeRemovedFromDocument", remove); 
                if (setInsertion)
                    p.addEventListener("DOMNodeInserted", add);
                p = p.parentNode || p.$parentNode;
            }
            
            delete amlNode.$visibleCheck[type];
        }

        function check(e){
            //apf.isTrue(e.value)
            if (!amlNode.$ext.offsetHeight && !amlNode.$ext.offsetWidth)
                return;
                
            callback.call(amlNode);
            cleanup();
        }
        
        function remove(e){
            if (e.currentTarget != this)
                return;
            
            cleanup(e.name == "DOMNodeRemoved");
        }

        function add(){
            //Set events on the parent tree
            var p = amlNode;
            while (p) {
                p.addEventListener("prop.visible", check);
                p.addEventListener("DOMNodeRemoved", remove); 
                p.addEventListener("DOMNodeRemovedFromDocument", remove); 
                p.removeEventListener("DOMNodeInserted", add);
                p = p.parentNode || p.$parentNode;
            }
            
            amlNode.$visibleCheck[type] = true;
        }
        
        add();
        
        return false;
    }
    
    this.permanent = function(amlNode, show, hide){
        var state = amlNode.$ext && (amlNode.$ext.offsetHeight || amlNode.$ext.offsetWidth);
        function check(e){
            var newState = amlNode.$ext && (amlNode.$ext.offsetHeight || amlNode.$ext.offsetWidth);
            if (newState == state)
                return;
            
            if (newState) show();
            else hide();
            
            state = newState;
        }

        //Set events on the parent tree
        /*var p = amlNode;
        while (p) {
            p.addEventListener("prop.visible", check);
            p = p.parentNode || p.$parentNode;
        }*/
        
        function cleanup(setInsertion){
            var p = amlNode;
            while (p) {
                p.removeEventListener("prop.visible", check);
                p.removeEventListener("DOMNodeRemoved", remove); 
                p.removeEventListener("DOMNodeRemovedFromDocument", remove); 
                if (setInsertion)
                    p.addEventListener("DOMNodeInserted", add);
                p = p.parentNode || p.$parentNode;
            }
            
            check();
        }

        function remove(e){
            if (e.currentTarget != this)
                return;
            
            cleanup(e.name == "DOMNodeRemoved");
        }

        function add(){
            //Set events on the parent tree
            var p = amlNode;
            while (p) {
                p.addEventListener("prop.visible", check);
                p.addEventListener("DOMNodeRemoved", remove); 
                p.addEventListener("DOMNodeRemovedFromDocument", remove); 
                p.removeEventListener("DOMNodeInserted", add);
                p = p.parentNode || p.$parentNode;
            }
            
            check();
        }
        
        add();

        return state;
    }
    
    this.removePermanent = function(amlNode){
        
    }
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * Determines whether a node is a child of another node.
 *
 * @param {DOMNode} pNode      The potential parent element.
 * @param {DOMNode} childnode  The potential child node.
 * @param {Boolean} [orItself] Whether the method also returns `true` when `pNode` is the `childnode`.
 * @return {Boolean} `false` if the second argument is not a child of the first.
 */
apf.isChildOf = function(pNode, childnode, orItself){
    if (!pNode || !childnode)
        return false;

    if (childnode.nodeType == 2)
        childnode = childnode.ownerElement || childnode.selectSingleNode("..");

    if (orItself && pNode == childnode)
        return true;

    var loopnode = childnode.parentNode;
    while(loopnode){
        if(loopnode == pNode)
            return true;
        loopnode = loopnode.parentNode;
    }

    return false;
};

apf.xmlEntityMap = {
    "quot": "34", "amp": "38", "apos": "39", "lt": "60", "gt": "62",
    "nbsp": "160", "iexcl": "161", "cent": "162", "pound": "163", "curren": "164",
    "yen": "165", "brvbar": "166", "sect": "167", "uml": "168", "copy": "169",
    "ordf": "170", "laquo": "171", "not": "172", "shy": "173", "reg": "174",
    "macr": "175", "deg": "176", "plusmn": "177", "sup2": "178", "sup3": "179",
    "acute": "180", "micro": "181", "para": "182", "middot": "183", "cedil": "184",
    "sup1": "185", "ordm": "186", "raquo": "187", "frac14": "188", "frac12": "189",
    "frac34": "190", "iquest": "191", "agrave": ["192", "224"], "aacute": ["193", "225"],
    "acirc": ["194", "226"], "atilde": ["195", "227"], "auml": ["196", "228"],
    "aring": ["197", "229"], "aelig": ["198", "230"], "ccedil": ["199", "231"],
    "egrave": ["200", "232"], "eacute": ["201", "233"], "ecirc": ["202", "234"],
    "euml": ["203", "235"], "igrave": ["204", "236"], "iacute": ["205", "237"],
    "icirc": ["206", "238"], "iuml": ["207", "239"], "eth": ["208", "240"],
    "ntilde": ["209", "241"], "ograve": ["210", "242"], "oacute": ["211", "243"],
    "ocirc": ["212", "244"], "otilde": ["213", "245"], "ouml": ["214", "246"],
    "times": "215", "oslash": ["216", "248"], "ugrave": ["217", "249"],
    "uacute": ["218", "250"], "ucirc": ["219", "251"], "uuml": ["220", "252"],
    "yacute": ["221", "253"], "thorn": ["222", "254"], "szlig": "223", "divide": "247",
    "yuml": ["255", "376"], "oelig": ["338", "339"], "scaron": ["352", "353"],
    "fnof": "402", "circ": "710", "tilde": "732", "alpha": ["913", "945"],
    "beta": ["914", "946"], "gamma": ["915", "947"], "delta": ["916", "948"],
    "epsilon": ["917", "949"], "zeta": ["918", "950"], "eta": ["919", "951"],
    "theta": ["920", "952"], "iota": ["921", "953"], "kappa": ["922", "954"],
    "lambda": ["923", "955"], "mu": ["924", "956"], "nu": ["925", "957"],
    "xi": ["926", "958"], "omicron": ["927", "959"], "pi": ["928", "960"],
    "rho": ["929", "961"], "sigma": ["931", "963"], "tau": ["932", "964"],
    "upsilon": ["933", "965"], "phi": ["934", "966"], "chi": ["935", "967"],
    "psi": ["936", "968"], "omega": ["937", "969"], "sigmaf": "962", "thetasym": "977",
    "upsih": "978", "piv": "982", "ensp": "8194", "emsp": "8195", "thinsp": "8201",
    "zwnj": "8204", "zwj": "8205", "lrm": "8206", "rlm": "8207", "ndash": "8211",
    "mdash": "8212", "lsquo": "8216", "rsquo": "8217", "sbquo": "8218", "ldquo": "8220",
    "rdquo": "8221", "bdquo": "8222", "dagger": ["8224", "8225"], "bull": "8226",
    "hellip": "8230", "permil": "8240", "prime": ["8242", "8243"], "lsaquo": "8249",
    "rsaquo": "8250", "oline": "8254", "frasl": "8260", "euro": "8364",
    "image": "8465", "weierp": "8472", "real": "8476", "trade": "8482",
    "alefsym": "8501", "larr": ["8592", "8656"], "uarr": ["8593", "8657"],
    "rarr": ["8594", "8658"], "darr": ["8595", "8659"], "harr": ["8596", "8660"],
    "crarr": "8629", "forall": "8704", "part": "8706", "exist": "8707", "empty": "8709",
    "nabla": "8711", "isin": "8712", "notin": "8713", "ni": "8715", "prod": "8719",
    "sum": "8721", "minus": "8722", "lowast": "8727", "radic": "8730", "prop": "8733",
    "infin": "8734", "ang": "8736", "and": "8743", "or": "8744", "cap": "8745",
    "cup": "8746", "int": "8747", "there4": "8756", "sim": "8764", "cong": "8773",
    "asymp": "8776", "ne": "8800", "equiv": "8801", "le": "8804", "ge": "8805",
    "sub": "8834", "sup": "8835", "nsub": "8836", "sube": "8838", "supe": "8839",
    "oplus": "8853", "otimes": "8855", "perp": "8869", "sdot": "8901", "lceil": "8968",
    "rceil": "8969", "lfloor": "8970", "rfloor": "8971", "lang": "9001", "rang": "9002",
    "loz": "9674", "spades": "9824", "clubs": "9827", "hearts": "9829", "diams": "9830"
};

/**
 * Escapes "&amp;", greater than, less than signs, quotation marks, and others into
 * the proper XML entities.
 * 
 * @param {String} str The XML string to escape.
 * @param {Boolean} strictMode By default, this function attempts to NOT double-escape XML entities. This flag turns that behavior off when set to `true`.
 * @return {String} The escaped string
 */
apf.escapeXML = function(str, strictMode) {
    if (typeof str != "string")
        return str;
    if (strictMode)
        str = (str || "").replace(/&/g, "&#38;");
    else
        str = (str || "").replace(/&(?!#[0-9]{2,5};|[a-zA-Z]{2,};)/g, "&#38;");
    var map = apf.xmlEntityMap;
    var isArray = apf.isArray;
    return str
        .replace(/"/g, "&#34;")
        .replace(/</g, "&#60;")
        .replace(/>/g, "&#62;")
        .replace(/'/g, "&#39;")
        .replace(/&([a-zA-Z]+);/gi, function(a, m) {
            var x = map[m.toLowerCase()];
            if (x)
                return "&#" + (isArray(x) ? x[0] : x) + ";";
            return a;
        });
};

/**
 * Unescapes `"&#38;"` and other similar XML entities into HTML entities, and then replaces
 * 'special' ones (`&apos;`, `&gt;`, `&lt;`, `&quot;`, `&amp;`) into characters
 * (`'`, `>`, `<`, `"`, `&`).
 *
 * @param {String} str The XML string to unescape
 * @return {String} The unescaped string
 */
apf.unescapeXML = function(str) {
    if (typeof str != "string")
        return str;
    var map = apf.xmlEntityMapReverse;
    var isArray = apf.isArray;
    if (!map) {
        map = apf.xmlEntityMapReverse = {};
        var origMap = apf.xmlEntityMap;
        var keys = Object.keys(origMap);
        for (var val, j, l2, i = 0, l = keys.length; i < l; ++i) {
            val = origMap[keys[i]];
            if (isArray(val)) {
                for (j = 0, l2 = val.length; j < l2; ++j)
                    map[val[j]] = keys[i];
            }
            else
                map[val] = keys[i];
        }
    }
    return str
        .replace(/&#([0-9]{2,5});/g, function(a, m) {
            var x = map[m];
            if (x)
                return "&" + x + ";";
            return a;
        })
        .replace(/&apos;/gi, "'")
        .replace(/&gt;/gi, ">")
        .replace(/&lt;/gi, "<")
        .replace(/&quot;/gi, "\"")
        .replace(/&amp;/gi, "&");
};

/**
 * Determines whether a node is its parent's only child.
 * @param {DOMNode} node     The potential only child.
 * @param {Array}   nodeType List of the node types that this child can be.
 * @returns {Boolean} Whether the node is the only child and of one of the specified node types.
 */
apf.isOnlyChild = function(node, nodeType){
    if (!node || !node.parentNode || nodeType && nodeType.indexOf(node.nodeType) == -1)
        return false;

    var i, l, cnode, nodes = node.parentNode.childNodes;
    for (i = 0, l = nodes.length; i < l; i++) {
        cnode = nodes[i];
        if (cnode.nodeType == 1 && cnode != node)
            return false;
        if (cnode.nodeType == 3 && !cnode.nodeValue.trim())
            return false;
    }

    return true;
};

/**
 * Gets the position of a DOM node within the list of child nodes of its
 * parent.
 *
 * @param {DOMNode} node The node for which the child position is being determined.
 * @return {Number} The child position of the node.
 */
apf.getChildNumber = function(node, fromList){
    if (!node) return -1;

    var p = node.parentNode, j = 0;
    if (!p) return -1;
    if (!fromList)
        fromList = p.childNodes;

    if (fromList.indexOf) {
        var idx = fromList.indexOf(node);
        return idx == -1 ? fromList.length : idx;
    }

    for (var i = 0, l = fromList.length; i < l; i++) {
        if (fromList[i] == node)
            return j;
        j++;
    }
    return -1;
};


 // @todo More information will follow....when?
/**
 * Integrates nodes as children of a parent. Optionally, attributes are
 * copied as well.
 *
 * @param {XMLNode} xmlNode The data to merge.
 * @param {XMLNode} parent  The node to merge on.
 * @param {Object}  options An object with the following optional properties:
 *   - [copyAttributes] ([[Boolean]]): Whether the attributes of `xmlNode` are copied as well.
 *   - [clearContents] ([[Boolean]]): Whether the contents of parent is cleared.
 *   - [start] ([[Number]]): This feature is used for the virtual viewport. More information will follow.
 *   - [length] ([[Number]]): This feature is used for the virtual viewport. More information will follow.
 *   - [documentId] ([[Number]]): This feature is used for the virtual viewport. More information will follow.
 *   - [marker] ([[XMLElement]]): This feature is used for the virtual viewport. More information will follow.
 * @return  {XMLNode}  The created xml node
 */
apf.mergeXml = function(XMLRoot, parentNode, options){
    if (typeof parentNode != "object")
        parentNode = apf.xmldb.getElementById(parentNode);

    if (options && options.clearContents) {
        //Signal listening elements
        var node, j, i,
            nodes = parentNode.selectNodes("descendant::node()[@" + apf.xmldb.xmlListenTag + "]");
        for (i = nodes.length - 1; i >= 0; i--) {
            var s = nodes[i].getAttribute(apf.xmldb.xmlListenTag).split(";");
            for (j = s.length - 1; j >= 0; j--) {
                node = apf.all[s[j]];
                if (!node) continue;
                if (node.dataParent && node.dataParent.xpath)
                    node.dataParent.parent.signalXmlUpdate[node.$uniqueId] = true;
                else if (node.$model)
                    node.$model.$waitForXml(node);
            }
        }

        //clean parent
        nodes = parentNode.childNodes;
        for (i = nodes.length - 1; i >= 0; i--)
            parentNode.removeChild(nodes[i]);
    }

    
    if (options && options.start) { //Assuming each node is in count
        var reserved, beforeNode, nodes, doc, i, l, marker = options.marker;
        if (!marker){
            //optionally find marker
        }

        //This code assumes that the dataset fits inside this marker

        //Start of marker
        if (marker.getAttribute("start") - options.start == 0) {
            marker.setAttribute("start", options.start + options.length);
            reserved = parseInt(marker.getAttribute("reserved"), 10);
            marker.setAttribute("reserved", reserved + options.length);
            beforeNode = marker;
        }
        //End of marker
        else if (options.start + options.length == marker.getAttribute("end")) {
            marker.setAttribute("end", options.start + options.length);
            beforeNode = marker.nextSibling;
            reserved = parseInt(marker.getAttribute("reserved"), 10) +
                parseInt(marker.getAttribute("end"), 10) - options.length;
        }
        //Middle of marker
        else {
            var m2 = marker.parentNode.insertBefore(marker.cloneNode(true), marker);
            m2.setAttribute("end", options.start - 1);
            marker.setAttribute("start", options.start + options.length);
            reserved = parseInt(marker.getAttribute("reserved"), 10);
            marker.setAttribute("reserved", reserved + options.length);
            beforeNode = marker;
        }

        nodes = XMLRoot.childNodes;

        if (parentNode.ownerDocument.importNode) {
            doc = parentNode.ownerDocument;
            for (i = 0, l = nodes.length; i < l; i++) {
                parentNode.insertBefore(doc.importNode(nodes[i], true), beforeNode)
                  .setAttribute(apf.xmldb.xmlIdTag, options.documentId + "|" + (reserved + i));
            }
        }
        else {
            for (i = nodes.length - 1; i >= 0; i--) {
                parentNode.insertBefore(nodes[0], beforeNode)
                  .setAttribute(apf.xmldb.xmlIdTag, options.documentId + "|" + (reserved + i));
            }
        }
    }
    else
    
    {
        beforeNode = options && options.beforeNode ? options.beforeNode : apf.getNode(parentNode, [0]);
        nodes      = XMLRoot.childNodes;
        
        if (options.filter)
            nodes = options.filter(parentNode, nodes);

        if (parentNode.ownerDocument.importNode) {
            doc = parentNode.ownerDocument;
            for (i = 0, l = nodes.length; i < l; i++)
                parentNode.insertBefore(doc.importNode(nodes[i], true), beforeNode);
        }
        else
            for (i = nodes.length - 1; i >= 0; i--)
                parentNode.insertBefore(nodes[0], beforeNode);
    }

    if (options && options.copyAttributes) {
        var attr = XMLRoot.attributes;
        for (i = 0; i < attr.length; i++)
            if (attr[i].nodeName != apf.xmldb.xmlIdTag)
                parentNode.setAttribute(attr[i].nodeName, attr[i].nodeValue);
    }

    return parentNode;
};

/**
 * Sets the node value of a DOM node.
 *
 * @param {XMLElement} xmlNode       The XML node that should receive the node value.
 *                                   When an element node is passed the first text node is set.
 * @param {String}     nodeValue     The value to set.
 * @param {Boolean}    applyChanges  Whether the changes are propagated to the databound elements.
 * @param {apf.UndoData}    undoObj       The undo object that is responsible for archiving the changes.
 */
apf.setNodeValue = function(xmlNode, nodeValue, applyChanges, options){
    if (!xmlNode)
        return;

    var undoObj, xpath, newNodes;
    if (options) {
        undoObj  = options.undoObj;
        xpath    = options.xpath;
        newNodes = options.newNodes;

        undoObj.extra.oldValue = options.forceNew
            ? ""
            : apf.queryValue(xmlNode, xpath);

        undoObj.xmlNode        = xmlNode;
        if (xpath) {
            xmlNode = apf.createNodeFromXpath(xmlNode, xpath, newNodes, options.forceNew);
        }

        undoObj.extra.appliedNode = xmlNode;
    }

    if (xmlNode.nodeType == 1) {
        if (!xmlNode.firstChild)
            xmlNode.appendChild(xmlNode.ownerDocument.createTextNode(""));

        xmlNode.firstChild.nodeValue = apf.isNot(nodeValue) ? "" : nodeValue;

        if (applyChanges)
            apf.xmldb.applyChanges("text", xmlNode, undoObj);
    }
    else {
        // @todo: this should be fixed in libxml
        if (apf.isO3 && xmlNode.nodeType == 2)
            nodeValue = nodeValue.replace(/&/g, "&amp;");

        var oldValue      = xmlNode.nodeValue;
        xmlNode.nodeValue = nodeValue === undefined || nodeValue === null ||
            nodeValue == NaN ? "" : String(nodeValue);

        if (undoObj) {
            undoObj.name = xmlNode.nodeName;
        }

        //AML support - getters/setters would be awesome
        if (xmlNode.$triggerUpdate)
            xmlNode.$triggerUpdate(null, oldValue);

        if (applyChanges) {
            apf.xmldb.applyChanges(xmlNode.nodeType == 2 ? "attribute" : "text", xmlNode.parentNode ||
                xmlNode.ownerElement || xmlNode.selectSingleNode(".."), undoObj);
        }
    }

    
    if (applyChanges) {
        var node;
        if (xpath) {
            var node = undoObj.xmlNode;//.selectSingleNode(newNodes.foundpath);
            if (node.nodeType == 9) {
                node = node.documentElement;
                xpath = xpath.replace(/^[^\/]*\//, "");//xpath.substr(newNodes.foundpath.length);
            }
        }
        else
            node = xmlNode;

        apf.xmldb.applyRDB(["setValueByXpath", node, nodeValue, xpath,
            options && options.forceNew],
            undoObj || {xmlNode: xmlNode}
        );
    }
    
};

/**
 * Sets a value of an XML node based on an xpath statement executed on a referenced XMLNode.
 *
 * @param  {XMLNode}  xmlNode  The reference XML node.
 * @param  {String}  xpath  The xpath used to select a XML node.
 * @param  {String}  value  The value to set.
 * @param  {Boolean}  local  Whether the call updates databound UI.
 * @return  {XMLNode}  The changed XML node
 */
apf.setQueryValue = function(xmlNode, xpath, value, local){
    var node = apf.createNodeFromXpath(xmlNode, xpath);
    if (!node)
        return null;

    apf.setNodeValue(node, value, !local);
    //apf.xmldb.setTextNode(node, value);
    return node;
};

/**
 * Removes an XML node based on an xpath statement executed on a referenced XML node.
 *
 * @param  {XMLNode}  xmlNode  The reference XML node.
 * @param  {String}  xpath  The xpath used to select a XML node.
 * @return  {XMLNode}  The changed XML node
 */
apf.removeQueryNode = function(xmlNode, xpath, local){
    var node = apf.queryNode(xmlNode, xpath);
    if (!node)
        return false;

    if (local)
        node.parentNode.removeChild(node);
    else
        apf.xmldb.removeNode(node);

    return node;
};

/**
 * Queries an XML node using xpath for a single string value.
 * @param {XMLElement} xmlNode The XML element to query
 * @param {String}     xpath   The xpath query
 * @return {String} The value of the query result or empty string
 */
apf.queryValue = function (xmlNode, xpath){
    if (!xmlNode)
        return "";
    if (xmlNode.nodeType == 2)
        return xmlNode.nodeValue;

    if (xpath) {
        xmlNode = xmlNode.selectSingleNode(xpath);
        if (!xmlNode)
            return "";
    }
   return xmlNode.nodeType == 1
        ? (!xmlNode.firstChild ? "" : xmlNode.firstChild.nodeValue)
        : xmlNode.nodeValue;
};

/**
 * Queries an xml node using xpath for multiple string value.
 * @param {XMLElement} xmlNode The xml element to query
 * @param {String}     xpath   The xpath query
 * @return {Array} A list of values resulting from the query
 */
apf.queryValues = function(xmlNode, xpath){
    var out = [];
    if (!xmlNode) return out;

    var nodes = xmlNode.selectNodes(xpath);
    if (!nodes.length) return out;

    for (var i = 0; i < nodes.length; i++) {
        var n = nodes[i];
        if (n.nodeType == 1)
            n = n.firstChild;
        out.push(n.nodeValue || "");
    }
    return out;
};

/**
 * Executes an xpath expression on any DOM node. This is especially useful
 * for DOM nodes that don't have a good native xpath processor, such as HTML
 * in some versions of Internet Explorer and XML in Webkit.
 *
 * @param {DOMNode} contextNode  The XML node that is subject to the query
 * @param {String}  sExpr        The xpath expression
 * @returns {Array} A list of found XML nodes. The list can be empty
 */
apf.queryNodes = function(contextNode, sExpr){
    if (contextNode && (apf.hasXPathHtmlSupport && contextNode.selectSingleNode || !contextNode.style))
        return contextNode.selectNodes(sExpr); //IE55
    //if (contextNode.ownerDocument != document)
    //    return contextNode.selectNodes(sExpr);

    return apf.XPath.selectNodes(sExpr, contextNode);
};

/**
 * Executes an xpath expression on any DOM node. 
 * This is especially useful for DOM nodes that don't have a good native 
 * xpath processor such as html in some versions of internet explorer and XML in
 * webkit. This function only returns the first node found.
 *
 * @param {DOMNode} contextNode  The DOM node that is subject to the query.
 * @param {String}  sExpr        The xpath expression.
 * @returns {XMLNode} The DOM node, or `null` if none was found.
 */
apf.queryNode = function(contextNode, sExpr){
    if (contextNode && (apf.hasXPathHtmlSupport && contextNode.selectSingleNode || !contextNode.style))
        return contextNode.selectSingleNode(sExpr); //IE55
    //if (contextNode.ownerDocument != document)
    //    return contextNode.selectSingleNode(sExpr);

    var nodeList = apf.queryNodes(contextNode ? contextNode : null,
        sExpr + (apf.isIE ? "" : "[1]"));
    return nodeList.length > 0 ? nodeList[0] : null;
};

/**
 * Retrieves the attribute of an XML node, or the first parent node that has
 * that attribute set. If no attribute is set, the value is searched for on
 * the appsettings element.
 *
 * @param {XMLElement} xml    The XML node that is the starting point of the search.
 * @param {String}     attr   The name of the attribute.
 * @param {Function}   [func] A callback that is run for every node that is searched.
 * @return {String} The found value, or empty string if none was found.
 */
apf.getInheritedAttribute = function(xml, attr, func){
    var result, avalue;

    //@todo optimize this and below
    if (xml.nodeType != 1)
        xml = xml.parentNode;

    while (xml && (xml.nodeType != 1 || !(result = attr
      && ((avalue = xml.getAttribute(attr)) || typeof avalue == "string")
      || func && func(xml)))) {
        xml = xml.parentNode;
    }
    if (avalue === "")
        return "";

    return !result && attr && apf.config
        ? apf.config[attr]
        : result;
};


/**
 * Creates an XML node based on an xpath statement.
 *
 * @param {DOMNode} contextNode  The DOM node that is subject to the query
 * @param {String}  xPath        The xpath query
 * @param {Array}   [addedNodes] An array that is filled with the added nodes
 * @param {Boolean} [forceNew]   Defines whether a new node is always created
 * @return {DOMNode} The last element found
 */
apf.createNodeFromXpath = function(contextNode, xPath, addedNodes, forceNew){
    // @todo generalize this to include attributes in if format []
    var xmlNode, foundpath = "", paths = xPath.replace(/('.*?')|(".*?")|\|/g, function(m, m1, m2){
        if (m1 || m2) return m1 || m2;
        return "-%-|-%-";
    }).split("-%-|-%-")[0].split(/\/(?!\/)/);//.split("/");
    if (!forceNew && (xmlNode = contextNode.selectSingleNode(xPath)))
        return xmlNode;

    var len = paths.length - 1;
    if (forceNew) {
        if (paths[len].trim().match(/^\@(.*)$|^text\(\)$/))
            len--;
    }

    //Directly forwarding to the document element because of a bug in the o3 xml lib
    if (!paths[0]) {
        contextNode = contextNode.ownerDocument.documentElement;
        paths.shift();paths.shift();
        len--;len--;
    }

    for (var addedNode, isAdding = false, i = 0; i < len; i++) {
        if (!isAdding && contextNode.selectSingleNode(foundpath
          + (i != 0 ? "/" : "") + paths[i])) {
            foundpath += (i != 0 ? "/" : "") + paths[i];// + "/";
            continue;
        }

        //Temp hack
        var isAddId = paths[i].match(/(\w+)\[@([\w-]+)=(\w+)\]/);
        

        if (isAddId)
            paths[i] = isAddId[1];

        isAdding = true;
        addedNode = contextNode.selectSingleNode(foundpath || ".")
            .appendChild(contextNode.ownerDocument.createElement(paths[i]));

        if (isAddId) {
            addedNode.setAttribute(isAddId[2], isAddId[3]);
            foundpath += (foundpath ? "/" : "") + isAddId[0];// + "/";
        }
        else
            foundpath += (foundpath ? "/" : "") + paths[i];// + "/";

        if (addedNodes)
            addedNodes.push(addedNode);
    }

    if (!foundpath)
        foundpath = ".";
    if (addedNodes)
        addedNodes.foundpath = foundpath;

    var newNode, lastpath = paths[len],
        doc = contextNode.nodeType == 9 ? contextNode : contextNode.ownerDocument;
    do {
        if (lastpath.match(/^\@(.*)$/)) {
            (newNode || contextNode.selectSingleNode(foundpath))
                .setAttributeNode(newNode = doc.createAttribute(RegExp.$1));
        }
        else if (lastpath.trim() == "text()") {
            newNode = (newNode || contextNode.selectSingleNode(foundpath))
                .appendChild(doc.createTextNode(""));
        }
        else {
            var hasId = lastpath.match(/(\w+)\[@([\w-]+)=(\w+)\]/);
            if (hasId) lastpath = hasId[1];
            newNode = (newNode || contextNode.selectSingleNode(foundpath))
                .appendChild(doc.createElement(lastpath));
            if (hasId)
                newNode.setAttribute(hasId[2], hasId[3]);
        }

        if (addedNodes)
            addedNodes.push(newNode);

        foundpath += (foundpath ? "/" : "") + paths[len];
    } while((lastpath = paths[++len]));

    return newNode;
};

/*
 * @private
 * @class apf.convertMethods
 */
apf.convertMethods = {
    /**
     * Gets a JSON object containing all the name/value pairs of the elements
     * using this element as it's validation group.
     *
     * @return  {String}  the string representation of a the json object
     */
    "json": function(xml){
        return JSON.stringify(apf.xml2json(xml));
        /*
        var result = {}, filled = false, nodes = xml.childNodes;
        for (var i = 0; i < nodes.length; i++) {
            if (nodes[i].nodeType != 1)
                continue;
            var name = nodes[i].tagName;
            filled = true;

            //array
            var sameNodes = xml.selectNodes(x);
            if (sameNodes.length > 1) {
                var z = [];
                for (var j = 0; j < sameNodes.length; j++) {
                    z.push(this.json(sameNodes[j], result));
                }
                result[name] = z;
            }
            else //single value
                result[name] = this.json(sameNodes[j], result);
        }

        return filled ? result : apf.queryValue(xml, "text()");*/
    },

    "cgivars": function(xml, basename){
        if (!basename)
            basename = "";

        var value, name, sameNodes, j, l2,
            str   = [],
            nodes = xml.childNodes,
            done  = {},
            i     = 0,
            l     = nodes.length;
        for (; i < l; ++i) {
            if (nodes[i].nodeType != 1)
                continue;
            name = nodes[i].tagName;
            if (done[name])
                continue;

            //array
            sameNodes = xml.selectNodes(name);
            if (sameNodes.length > 1) {
                done[name] = true;
                for (j = 0, l2 = sameNodes.length; j < l2; j++) {
                    value = this.cgivars(sameNodes[j], basename + name + "[" + j + "]");
                    if (value)
                        str.push(value);
                }
            }
            else { //single value
                value = this.cgivars(nodes[i], basename + name);
                if (value)
                    str.push(value);
            }
        }

        var attr = xml.attributes;
        for (i = 0, l = attr.length; i < l; i++) {
            if (attr[i].nodeValue) {
                if (basename)
                    str.push(basename + "[" + attr[i].nodeName + "]="
                        + escape(attr[i].nodeValue));
                else
                    str.push(attr[i].nodeName + "=" + escape(attr[i].nodeValue));
            }
        }

        if (str.length)
            return str.join("&");

        value = apf.queryValue(xml, "text()");
        if (basename && value)
            return basename + "=" + escape(value);

        return "";
    },

    "cgiobjects": function(xml, basename, isSub, includeEmpty){
        if (!basename)
            basename = "";

        var node, name, value, a, i, j, attr, attr_len, isOnly,
            nodes    = xml.childNodes,
            output   = [],
            tagNames = {},
            nm       = "";

        for (i = 0; i < nodes.length; i++) {
            node = nodes[i];

            if (node.nodeType == 1) {
                name = node.tagName;

                isOnly = node.parentNode.selectNodes(name).length == 1
                    ? true
                    : false;

                if (typeof tagNames[name] == "undefined") {
                    tagNames[name] = 0;
                }

                nm = basename
                   + (isSub ? "[" : "") + name + (isSub ? "]" : "")
                   + (isOnly ? "" : "[" + tagNames[name] + "]");

                attr     = node.attributes;
                attr_len = node.attributes.length;

                if (attr_len > 0) {
                    for (j = 0; j < attr_len; j++) {
                        if (!(a = attr[j]).nodeValue)
                            continue;

                        output.push(nm + "[_" + a.nodeName + "]="
                            + escape(a.nodeValue.trim()));
                    }
                }

                value = this.cgiobjects(node, nm, true);

                if (value.dataType !== 1) {
                    output.push(value);
                }
                else {
                    if (node.firstChild && node.firstChild.nodeValue.trim()) {
                        output.push(nm + (attr_len > 0 ? "[_]=" : "=")
                            + escape(node.firstChild.nodeValue.trim()));
                    }
                    else {
                        if (attr_len == 0) {
                            if (includeEmpty) {
                                output.push(nm);
                            }
                        }
                    }
                }

                tagNames[name]++;
            }
            //@todo, that's that ??
            //handle node values (for form submission)
            else if (node.nodeType == 3 && isSub) {
                var nval = node.nodeValue;

                if (nval && nval.trim() !== "") {
                    output.push(basename + "=" + escape(nval));
                }

                //was: output = node.nodeType;
            }
        }

        if (!isSub && xml.getAttribute("id"))
            output.push("id=" + escape(xml.getAttribute("id")));

        if (output.length)
            return output.join("&");

        return output;
    }
};

/**
 * Converts XML to another format.
 *
 * @param {XMLElement} xml  The [[term.datanode data node]] to convert.
 * @param {String}     to   The format to convert the XML to. It can be one of the following values:
 *   - `"json"`:       converts to a json string
 *   - `"cgivars"`:    converts to cgi string
 *   - `"cgiobjects"`: converts to cgi object
 * @return {String} The result of the conversion.
 */
apf.convertXml = function(xml, to){
    return apf.convertMethods[to](xml);
};

/**
 * Returns the first text or cdata child of a [[term.datanode data node]].
 *
 * @param {XMLElement} x The XML node to search.
 * @return {XMLNode} The found XML node, or `null`.
 */
apf.getTextNode = function(x){
    for (var i = 0, l = x.childNodes.length; i < l; ++i) {
        if (x.childNodes[i].nodeType == 3 || x.childNodes[i].nodeType == 4)
            return x.childNodes[i];
    }
    return false;
};

/**
 * @private
 */
apf.getBoundValue = function(amlNode, xmlRoot, applyChanges){
    if (!xmlRoot && !amlNode.xmlRoot)
        return "";

    var xmlNode = amlNode.$getDataNode("value", amlNode.xmlRoot);

    return xmlNode ? apf.queryValue(xmlNode) : "";
};

/**
 * @private
 */
apf.getArrayFromNodelist = function(nodelist){
    for (var nodes = [], j = 0, l = nodelist.length; j < l; ++j)
        nodes.push(nodelist[j]);
    return nodes;
};

/**
 * Serializes the children of a node into a string.
 *
 * @param {XMLElement} xmlNode The XML node to serialize.
 * @return {String} The children as a string
 */
apf.serializeChildren = function(xmlNode){
    var node,
        s     = [],
        nodes = xmlNode.childNodes,
        i     = 0,
        l     = nodes.length;
    for (; i < l; ++i) {
        s[i] = (node = nodes[i]).nodeType == 1
            ? node.xml || node.serialize()
            : (node.nodeType == 8 ? "" : node.nodeValue);
    }
    return s.join("");
};

/**
 * Returns a string version of the {@link term.datanode data node}.
 *
 * @param {XMLElement} xmlNode The {@link term.datanode data node} to serialize.
 * @return {String} The serialized version of the {@link term.datanode data node}.
 */
apf.getXmlString = function(xmlNode){
    var xml = apf.xmldb.cleanNode(xmlNode.cloneNode(true));
    return xml.xml || xml.serialize();
};

/**
 * Creates XML nodes from an XML string recursively.
 *
 * @param {String}  strXml     The XML definition
 * @param {Boolean} [noError]  Whether an exception should be thrown by the parser
 *                             when the XML is not valid
 * @param {Boolean} [preserveWhiteSpace]  Whether whitespace that is present between
 *                                        XML elements should be preserved
 * @return {XMLNode} The created XML node
 */
apf.getXml = function(strXml, noError, preserveWhiteSpace){
    return apf.getXmlDom(strXml, noError, preserveWhiteSpace).documentElement;
};

/**
 * Formats an XML string with proper indentation. Also known as pretty printing.
 * @param {String} strXml The XML string to format.
 * @return {String} The formatted string.
 */
apf.formatXml = function(strXml){
    strXml = strXml.trim();

    var lines = strXml.split("\n"),
        depth = 0,
        i     = 0,
        l     = lines.length;
    for (; i < l; ++i)
        lines[i] = lines[i].trim();
    lines = lines.join("\n").replace(/\>\n/g, ">").replace(/\>/g, ">\n")
        .replace(/\n\</g, "<").replace(/\</g, "\n<").split("\n");
    lines.removeIndex(0);//test if this is actually always fine
    lines.removeIndex(lines.length);

    for (i = 0, l = lines.length; i < l; i++)
        lines[i] = "    ".repeat((lines[i].match(/^\s*\<\//)
            ? (depth==0)?0:--depth
            : (lines[i].match(/^\s*\<[^\?][^>]+[^\/]\>/) ? depth++ : depth))) + lines[i];
    if (!strXml)
        return "";

    return lines.join("\n");
};

//@todo this function needs to be 100% proof, it's the core of the system
//for RDB: xmlNode --> Xpath statement
apf.xmlToXpath = function(xmlNode, xmlContext, useAID){
    if (!xmlNode) //@todo apf3.0
        return "";

    if (useAID === true && xmlNode.nodeType == 1 && xmlNode.getAttribute(apf.xmldb.xmlIdTag)) {
        return "//node()[@" + apf.xmldb.xmlIdTag + "='"
            + xmlNode.getAttribute(apf.xmldb.xmlIdTag) + "']";
    }

    if (apf != this && this.lookup && this.select) {
        var unique, def = this.lookup[xmlNode.tagName];
        if (def) {
            //unique should not have ' in it... -- can be fixed...
            unique = xmlNode.selectSingleNode(def).nodeValue;
            return "//" + xmlNode.tagName + "[" + def + "='" + unique + "']";
        }

        for (var i = 0, l = this.select.length; i < l; i++) {
            if (xmlNode.selectSingleNode(this.select[i][0])) {
                unique = xmlNode.selectSingleNode(this.select[i][1]).nodeValue;
                return "//" + this.select[i][0] + "[" + this.select[i][1]
                    + "='" + unique + "']";
            }
        }
    }

    if (xmlNode == xmlContext)
        return ".";

    if (xmlNode.nodeType != 2 && !xmlNode.parentNode && !xmlNode.ownerElement) {
        

        return false;
    }

    var str = [], lNode = xmlNode;
    if (lNode.nodeType == 2) {
        str.push("@" + lNode.nodeName);
        lNode = lNode.ownerElement || xmlNode.selectSingleNode("..");
    }

    var id;//, pfx = "";
    while(lNode && lNode.nodeType == 1) {
        if (lNode == xmlContext) {
            //str.unshift("/");//pfx = "//";
            break;
        }
        str.unshift((lNode.nodeType == 1 ? lNode.tagName : "text()") +
            "[" + (useAID && (id = lNode.nodeType == 1 && lNode.getAttribute(apf.xmldb.xmlIdTag))
                ? "@" + apf.xmldb.xmlIdTag + "='" + id + "'"
                : (apf.getChildNumber(lNode, lNode.parentNode.selectNodes(lNode.nodeType == 1 ? lNode.tagName : "text()")) + 1)) +
            "]");
        lNode = lNode.parentNode;
    };

    return (str[0] == "/" || xmlContext && xmlContext.nodeType == 1 ? "" : "/") + str.join("/"); //pfx +
};

//for RDB: Xpath statement --> xmlNode
apf.xpathToXml = function(xpath, xmlNode){
    if (!xmlNode) {
        

        return false;
    }

    return xmlNode.selectSingleNode(xpath);
};


apf.n = function(xml, xpath){
    return new apf.xmlset(xml, xpath, true);
};

apf.b = function(xml, xpath){
    return new apf.xmlset(xml, xpath);
};

apf.b.$queue = [];
apf.b.$state = 0;

/*
 * Naive jQuery like set implementation
 * @todo add dirty flags
 * @todo add query queue
 * @todo rewrite to use arrays
 */
apf.xmlset = function(xml, xpath, local, previous){
    if (typeof xml == "string")
        xml = apf.getXml(xml);

    this.$xml = xml;
    if (xml)
        this.$nodes = xml.dataType == apf.ARRAY ? xml : (xpath ? xml.selectNodes(xpath) : [xml]);
    this.$xpath = xpath || ".";
    this.$local = local;
    this.$previous = previous;
};

(function(){
    this.add = function(){}; //@todo not implemented

    this.begin = function(){
        apf.b.$state = 1;
        return this;
    };

    this.commit = function(at, rmt, uri){
        if (apf.b.$queue.length) {
            if (rmt) {
                var _self = this;
                rmt.addEventListener("rdbsend", function(e){
                    if (!uri || e.uri == uri) {
                        _self.rdbstack = e.message;
                        rmt.removeEventListener("rdbsend", arguments.callee);
                    }
                });
            }

            at.execute({
                action : 'multicall',
                args   : apf.b.$queue
            });

            if (rmt)
                rmt.$processQueue(rmt);
        }

        apf.b.$queue = [];
        apf.b.$state = 0;
        return this;
    };

    this.rollback = function(){
        apf.b.$queue = [];
        apf.b.$state = 0;
        return this;
    };

    this.getRDBMessage = function(){
        return this.rdbstack || [];
    };

    this.before = function(el){
        el = typeof el == "function" ? el(i) : el;

        for (var node, i = 0, l = this.$nodes.length; i < l; i++) {
            node = this.$nodes[i];
            if (this.$local)
                node.parentNode.insertBefore(el, node);
            else
                apf.xmldb.appendChild(node.parentNode, el, node);
        }
        return this;
    };

    this.after = function(el){
        el = typeof el == "function" ? el(i) : el;

        for (var node, i = 0, l = this.$nodes.length; i < l; i++) {
            node = this.$nodes[i];
            if (this.$local)
                node.parentNode.insertBefore(el, node.nextSibling);
            else
                apf.xmldb.appendChild(node.parentNode, el, node.nextSibling);
        }

        return this;
    };

    this.andSelf = function(){};

    this.append = function(el){
        for (var node, child, i = 0, l = this.$nodes.length; i < l; i++) {
            node = this.$nodes[i];
            child = typeof el == "function" ? el(i, node) : el;

            if (apf.b.$state)
                apf.b.$queue.push({
                    action : 'appendChild',
                    args   : [node, child]
                });
            else if (this.$local)
                node.appendChild(child);
            else
                apf.xmldb.appendChild(node, child);

        }

        return this;
    };

    this.appendTo = function(target){
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            target.appendChild(this.$nodes[i]);
        }
        return this;
    };

    this.prepend = function(el){
        for (var node, i = 0, l = this.$nodes.length; i < l; i++) {
            node = this.$nodes[i];
            node.insertBefore(typeof el == "function" ? el(i, node) : el, node.firstChild);
        }

        return this;
    };

    this.prependTo = function(target){
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            target.insertBefore(this.$nodes[i], target.firstChild);
        }
        return this;
    };

    this.attr = function(attrName, value){
        if (arguments.length === 1) {
            return this.$nodes && this.$nodes[0] && this.$nodes[0].getAttribute(attrName) || "";
        }
        else {
            for (var i = 0, l = this.$nodes.length; i < l; i++) {
                if (apf.b.$state)
                    apf.b.$queue.push({
                        action : 'setAttribute',
                        args   : [this.$nodes[i], attrName, value]
                    });
                else if (this.$local)
                    this.$nodes[i].setAttribute(attrName, value);
                else
                    apf.xmldb.setAttribute(this.$nodes[i], attrName, value);
            }
        }

        return this;
    };

    this.removeAttr = function(attrName){
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            if (apf.b.$state)
                apf.b.$queue.push({
                    action : 'removeAttribute',
                    args   : [this.$nodes[i], attrName]
                });
            else if (this.$local)
                this.$nodes[i].removeAttribute(attrName);
            else
                apf.xmldb.removeAttribute(this.$nodes[i], attrName);
        }

        return this;
    };

    this.xml = function(){
        var str = [];
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            str.push(this.$nodes[i].xml);
        }
        return str.join("\n");
    };

    this.get   =
    this.index = function(idx){
        idx = idx || 0;
        return apf.getChildNumber(this.$nodes[idx], this.$nodes[idx].parentNode.getElementsByTagName("*"))
    };

    this.eq    = function(index){
        return index < 0 ? this.$nodes[this.$nodes.length - index] : this.$nodes[index];
    };

    this.size   =
    this.length = function(){
        return this.$nodes.length;
    };

    this.load = function(url){

    };

    this.next = function(selector){
        if (!selector)
            selector = "node()[local-name()]";
        return new apf.xmlset(this.$xml, "((following-sibling::" + (this.$xpath == "." ? "node()" : this.$xpath) +
            ")[1])[self::" + selector.split("|").join("|self::") + "]", this.$local, this);
    };

    this.nextAll = function(selector){
        if (!selector)
            selector = "node()[local-name()]";
        return new apf.xmlset(this.$xml, "(following-sibling::" + (this.$xpath == "." ? "node()" : this.$xpath) +
            ")[self::" + selector.split("|").join("|self::") + "]", this.$local, this);
    };

    this.nextUntil = function(){};

    this.prev = function(selector){
        if (!selector)
            selector = "node()[local-name()]";
        return new apf.xmlset(this.$xml, "((preceding-sibling::" + (this.$xpath == "." ? "node()" : this.$xpath) +
            ")[1])[self::" + selector.split("|").join("|self::") + "]", this.$local, this);
    };

    this.prevAll = function(selector){
        if (!selector)
            selector = "node()[local-name()]";
        return new apf.xmlset(this.$xml, "(preceding-sibling::" + (this.$xpath == "." ? "node()" : this.$xpath) +
            ")[self::" + selector.split("|").join("|self::") + "]", this.$local, this);
    };

    this.not = function(){};

    this.parent = function(selector){
        return new apf.xmlset(this.$xml.parentNode, this.$local, this);
    };

    this.parents = function(selector){};
    this.pushStack = function(){};
    this.replaceAll = function(){};
    this.replaceWith = function(el){
        for (var node, child, i = 0, l = this.$nodes.length; i < l; i++) {
            node = this.$nodes[i];
            child = typeof el == "function" ? el(i, node) : el;

            if (apf.b.$state)
                apf.b.$queue.push({
                    action : 'replaceNode',
                    args   : [child, node]
                });
            else if (this.$local)
                node.parentNode.replaceChild(child, node);
            else
                apf.xmldb.replaceNode(child, node);

        }

        return this;
    };

    this.siblings = function(selector){
        //preceding-sibling::
        //return new apf.xmlset(this.$xml, "(" + this.$xpath + ")/node()[self::" + selector.split("|").join("|self::") + "]");
    };

    this.text = function(){

    };

    this.toArray = function(){
        var arr = [];
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            arr.push(this.$nodes[i]);
        }
        return arr;
    };

    this.detach = function(selector){
        var items = [];

        for (var node, i = 0, l = this.$nodes.length; i < l; i++) {
            node = this.$nodes[i];
            if (!node.selectSingleNode("self::node()[" + selector + "]"))
                continue;

            if (apf.b.$state)
                apf.b.$queue.push({
                    action : 'removeNode',
                    args   : [node]
                });
            else if (this.$local)
                node.parentNode.removeChild(node);
            else
                apf.xmldb.removeNode(node);

            items.push(node);
        }

        return new apf.xmlset(items, "", this.$local, this);
    };

    this.remove = function(selector){
        for (var node, n = this.$nodes, i = n.length - 1; i >= 0; i--) {
            node = n[i];
            if (selector && !node.selectSingleNode("self::node()[" + selector + "]"))
                continue;

            if (apf.b.$state)
                apf.b.$queue.push({
                    action : 'removeNode',
                    args   : [node]
                });
            else if (this.$local)
                node.parentNode.removeChild(node);
            else
                apf.xmldb.removeNode(node);
        }

        return this;
    };

    this.children = function(selector){
        var nodes = [];
        for (var node, i = 0, l = this.$nodes.length; i < l; i++) {
            var list = (node = this.$nodes[i]).selectNodes(selector);
            for (var j = 0, jl = list.length; j < jl; j++) {
                nodes.push(list[j]);
            }
        }
        return new apf.xmlset(nodes, null, this.$local, this);
    };

    this.children2 = function(selector){
        return new apf.xmlset(this.$xml, "(" + this.$xpath + ")/node()[self::" +
            selector.split("|").join("|self::") + "]", this.$local, this);
    };

    this.has  =
    this.find = function(path){
        return new apf.xmlset(this.$xml, "(" + this.$xpath + ")//" +
            path.split("|").join("|self::"), this.$local, this);
    };

    this.query = function(path){
        return new apf.xmlset(this.$xml, "(" + this.$xpath + ")/" +
            path.split("|").join("|(" + this.$xpath + ")/"), this.$local, this);
    };

    this.filter = function(filter){
        var newList = [];
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            if (this.$nodes[i].selectSingleNode("self::node()[" + filter + "]"))
                newList.push(this.$nodes[i]);
        }
        return new apf.xmlset(newList, null, this.$local, this);
    };

    this.end = function(){
        return this.$previous || this;
    };

    this.is = function(selector) {
        return this.filter(selector) ? true : false;
    };

    this.contents = function(){
        return this.children("node()");
    };

    this.has = function(){
        //return this.children(
    };

    this.val = function(value){
        if (arguments.length) {
            apf.setQueryValue(this.$xml, this.$xpath, value);
            return this;
        }
        else
            return apf.queryValue(this.$xml, this.$xpath);
    };

    this.vals = function(){
        return apf.queryValues(this.$xml, this.$xpath);
    };

    this.node = function(){
        return apf.queryNode(this.$xml, this.$xpath);
    };

    this.nodes = function(){
        return apf.queryNodes(this.$xml, this.$xpath);
    };

    this.clone = function(deep){
        if (this.$nodes.length == 1)
            return new apf.xmlset(this.$nodes[0].cloneNode(true), "", this.$local, this);

        var nodes = [];
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            nodes.push(this.$nodes[i].cloneNode(deep == undefined ? true : deep));
        }

        return new apf.xmlset(nodes, "", this.$local, this);
    };

    this.context = function(){
        return this.$xml;
    };

    this.data = function(data){
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            apf.setQueryValue(this.$nodes[i], ".", data);
        }
        return this;
    };

    this.each = function(func){
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            func.call(this.$nodes[i], i);
        }
        return this;
    };

    this.eachrev = function(func){
        for (var i = this.$nodes.length - 1; i >= 0; i--) {
            func.call(this.$nodes[i], i);
        }
        return this;
    };

    this.map = function(callback){
        var values = [];
        for (var i = 0, l = this.$nodes.length; i < l; i++) {
            values.push(callback(this.$nodes[i]));
        }
        return new apf.xmlset(values, "", this.$local, this); //blrghhh
    };

    this.empty  = function(){
        this.children().detach();
        return this;
    };

    this.first = function(){
        return new apf.xmlset(this.$xml, "(" + this.$xpath + ")[1]", this.$local, this);
    };

    this.last = function(){
        return new apf.xmlset(this.$xml, "(" + this.$xpath + ")[last()]", this.$local, this);
    };
}).call(apf.xmlset.prototype);








/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * Manages the z-index of all elements in the UI. It takes care of different
 * regions in the z-dimension preserved for certain common UI scenarios.
 *
 * #### Remarks
 * 
 *  The following regions are defined:
 *  From:         To:           For:
 *           10        10000  Common Elements (each element a unique z index)
 *       100000       110000  Plane (Modal Windows / Maximized Panels) (latest shown highest)
 *       200000       210000  Popup (Menus / Dropdown Containers) (latest shown highest)
 *       300000       310000  Notifiers
 *       400000       410000  Drag Indicators
 *      1000000      1100000  Print
 *
 * @private
 */
apf.zmanager = function(){
    var count = {
        "default" : {
            level  : 10
        },
        "plane" : {
            level  : 100000
        },
        "popup" : {
            level  : 200000
        },
        "notifier" : {
            level  : 300000
        },
        "drag" : {
            level  : 400000
        },
        "print" : {
            level  : 1000000
        }
    };
    
    this.set = function(type, main, companion){
        main.style.zIndex = count[type].level++;
        if (companion) {
            //if (companion.$storedZ == undefined)
                companion.$storedZ = companion.style.zIndex;
            companion.style.zIndex = count[type].level++
        }
    }
    
    this.clear = function(main, companion){
        if (companion.style.zIndex == parseInt(main.style.zIndex) + 1)
            companion.style.zIndex = companion.$storedZ;
        companion.$storedZ = undefined;
    }
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * @class apf.history
 *
 * Implements a hash change listener. The 'hash' is the part of the
 * location string in your browser that takes care of pointing to a section
 * within the current application.
 * 
 * #### Example
 * ```
 *  www.example.com/index.php#products
 * ```
 * 
 * #### Remarks
 * 
 * In modern browsers (after 2009) the location hash can be set by script and
 * {@link apf.history@hashchange} is called when it's changed by using the back or forward
 * button of the browsers. In most of the current (2009) browsers this is not the case.
 * This object handles that logic for those browsers in such a way that the user
 * of the application can use the back and forward buttons in an intuitive manner.
 *
 * Note: For Internet Explorer 8, when switching between the IE7 compatibility mode
 * and IE8 mode the history navigation breaks. A browser restart is then 
 * required to fix it. Individually history navigation works fine in each mode.
 *
 */
/**
 * @event hashchange Fires when the hash changes. This can be either by setting
 * a new hash value or when a user uses the back or forward button. Typing a
 * new hash value in the location bar will also trigger this function.
 *
 * #### Example
 *
 * ```javascript
 *  apf.addEventListener("hashchange", function(e){
 *      var info = e.page.split(":");
 *
 *      switch(info[0]) {
 *          case "product": //hash is for instance 'product:23849'
 *              //Sets the application state to display product info
 *              //For more information see apf.state
 *              stProduct.activate(); 
 *              //Loads a product by id
 *              loadProduct(info[1]); 
 *              break;
 *          case "news":
 *              stNews.activate();
 *              break;
 *      }
 *  });
 * ```
 *
 * @default_private
 */
apf.history = {
    inited: false,
    page  : null,
    past  : [],
    future: [],
    delay : 1,

    init  : function(defName, getVar, delay){
        if (this.inited || window.history.pushState)
            return;

        if (delay)
            this.delay = delay;

        this.inited = true;

        var name, _self = this;
        function preInit() {
            name = apf.dispatchEvent("hashinit")
              || location.href.match(/#(.*)$/) && decodeURI(RegExp.$1)
              || apf._GET[getVar || -1] || defName;


            location.hash = name;
            _self.hasChanged(name || null);
        }

        if (apf.supportHashChange) {
            $setTimeout(function() {
                preInit();

                window.onhashchange = function(){
                    var page = location.hash.replace("#", "");
                    apf.history.hasChanged(decodeURI(page));
                };
            });
        }
        else if (apf.isIE) {
            preInit();
            var str =
                "<style>\
                    BODY, HTML{margin:0}\
                    h1{height:100px; margin:0; padding:0; overflow:hidden}\
                </style>\
                <body>\
                    <h1 id='" + name + "'>0</h1>\
                </body>\
                <script>\
                    var lastURL = -1;\
                    if (document.all)\
                        document.body.onscroll = checkUrl;\
                    else\
                        setInterval('checkUrl()', 200);\
                    \
                    function checkUrl(){\
                        var iScr = (document.all ? document.body : document.documentElement).scrollTop;\
                        if (lastURL == iScr) return;\
                        top.apf.history.hasChanged(document.getElementsByTagName('h1')[Math.round(iScr / 100)].id, true);\
                        lastURL = iScr;\
                        }\
                    checkUrl();\
                </script>";

            if (top == self) {
                document.body.insertAdjacentHTML("beforeend",
                    "<iframe name='nav' style2='position:absolute;left:10px;top:10px;height:100px;width:100px;z-index:1000'\
                       style='width:1px;height:1px;' src='about:blank'></iframe>");
                document.frames["nav"].document.open();
                document.frames["nav"].document.write(str);
                document.frames["nav"].document.close();
            }

            this.iframe = document.frames["nav"];// : document.getElementById("nav").contentWindow;
            //Check to see if url has been manually changed
            this.timer2 = setInterval(function(){
                if (!apf.history.changingHash && location.hash != "#" + apf.history.page) {
                    var name = location.hash.replace(/^#/, "");
                    var page = apf.history.page;
                    apf.history.setHash(name, true, true);
                    apf.history.page = page;
                    apf.history.hasChanged(name);
                }
            }, apf.history.delay || 1);
        }
        else {
            preInit();
            apf.history.lastUrl = location.href.toString();
            this.timer2 = setInterval(function(){
                if (apf.history.lastUrl == location.href.toString())
                    return;

                apf.history.lastUrl = location.href.toString();
                //var page            = location.href.replace(/^.*#(.*)$/, "$1")
                var page = location.hash.replace("#", "");//.replace(/^.*#(.*)$/,"$1");
                apf.history.hasChanged(decodeURI(page));
            }, 20);
        }
    },
    to_name : null,
    
    /**
     * Sets the hash value of the location bar in the browser. This is used
     * to represent the state of the application for use by the back and forward
     * buttons, as well as for use when bookmarking or sharing URLs.
     * @param {String}  name    The new hash value
     * @param {Boolean} timed   Whether to add a delay to setting the value
     * @param {Boolean} force   Forces the change to overwrite any existing value
     */
    setHash : function(name, timed, force){
        if (this.changing || this.page == name || !force
          && decodeURIComponent(location.hash) == "#" + decodeURIComponent(name)) {
            this.to_name = name;
            return;
        }

        if (!apf.supportHashChange && apf.isIE  && !timed) {
            this.to_name = name;
            return $setTimeout(function(){
                apf.history.setHash(apf.history.to_name, true, force);
            }, 200);
        }

        this.changePage(name);
        if (!this.inited)
            return this.init(name);

        if (!apf.supportHashChange && apf.isIE) {
            var h       = this.iframe.document.body
                .appendChild(this.iframe.document.createElement('h1'));
            h.id        = name;
            h.innerHTML = "1";
        }

        (!apf.supportHashChange && apf.isIE ? this.iframe : window).location.href = "#" + name;
        
        if (!apf.isIE && !apf.isGecko && !apf.isIphone)
            apf.history.lastUrl = location.href.toString();
        //else if (apf.isIE8)
    },

    timer : null,
    changePage: function(page, force){
        if (!apf.supportHashChange && apf.isIE) {
            this.page = page;
            this.changingHash = true;
            clearTimeout(this.timer);
            this.timer = $setTimeout(function(){
                location.hash = page;
                apf.history.changingHash = false;
            }, 1);
        }
    },

    update: function(page) {
        var i, l, idx = 0;

        // check past:
        for (i = 0, l = this.past.length; i < l && idx === 0; i++) {
            if (this.past[i] == page)
                idx = i + 1;
        }
        if (idx > 0) {
            // part of past up till page (Array.slice), EXCLUDING page
            this.future = this.past.slice(idx, this.past.length - 1)
                                   .concat(this.future).makeUnique();
            this.past.splice(idx, this.past.length - (idx));
            idx = -idx;
        }
        else {
            // check future:
            for (i = 0, l = this.future.length; i < l && idx === 0; i++) {
                if (this.future[i] == page) {
                    idx = i + 1;
                    // current past + part of the future up till page
                    // (Array.splice), INCLUDING page
                    this.past = this.past.concat(this.future
                        .splice(0, this.future.length - idx)).makeUnique();
                }
            }
            if (idx === 0) {
                this.past.push(page);
                idx = 1;
            }
        }

        return idx;
    },

    hasChanged: function(page, force){
        if (page == this.page && !force) 
            return;
        this.changePage(page, force);

        this.changing = true;
        if (apf.dispatchEvent("hashchange", {
            oldURL : this.page,
            newURL : page,
            page   : page, 
            index  : this.update(page)
        }) === false) {
            page = location.hash = this.page;
        };
        this.changing = false;

        this.page = page;
    }
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



apf.config = new apf.Class().$init();
apf.extend(apf.config, {
    //Defaults
    disableRightClick  : false,
    allowSelect        : false,
    allowBlur          : true,
    autoDisableActions : true,
    autoDisable        : false, /** @todo fix this to only autodisable when createmodel is not true */
    disableF5          : true,
    autoHideLoading    : true,
    disableSpace       : true,
    defaultPage        : "home",
    disableBackspace   : true,
    undokeys           : false,
    outline            : false,
    dragOutline        : false,
    resizeOutline      : false,
    autoDisableNavKeys : true,
    disableTabbing     : false,
    resourcePath       : null,
    initdelay          : true,
    liveText           : false,
    
    
    
    skinset            : "default",
    name               : apf.isO3 ? "o3App" : self.window && window.location.href.replace(/[^0-9A-Za-z_]/g, "_"),

    tags               : {},
    defaults           : {},
    baseurl            : "",
    
    "empty-message"    : "No items",
    "loading-message"  : "Loading...",
    "offline-message"  : "You are currently offline.",
    
    setDefaults : function(){
        
    },

    getDefault : function(type, prop){
        var d = this.defaults[type];
        if (!d)
            return;

        for (var i = d.length - 1; i >= 0; i--) {
            if (d[i][0] == prop)
                return d[i][1];
        }
    },

    $handlePropSet : function(name, value){
        //this[name] = value;
        //@todo I dont want to go through all the code again, maybe later
        this[name.replace(/-(\w)/g, function(m, m1){
            return m1.toUpperCase()
        })] = this[name] = value;
        
        (this.$propHandlers && this.$propHandlers[name]
          || apf.GuiElement.propHandlers[name] || apf.K).call(this, value);
    },
    
    $inheritProperties : {},
    
    $propHandlers : {
        "baseurl" : function(value){
            this.baseurl = apf.parseExpression(value);
        },
        "language" : function(value){
            
        },
        "resource-path" : function(value){
            this.resourcePath = apf.parseExpression(value || "")
              .replace(/resources\/?|\/$/g, '');
        },
        
        
        "skinset" : function(value) {
            if (this.$amlLoaded)
                apf.skins.changeSkinset(value);
        },
        
        
        "outline" : function(value) {
            this.dragOutline    =
            this.resizeOutline  =
            this.outline        = apf.isTrue(value);
        },
        "drag-outline" : function(value){
            this.dragOutline    = value
              ? apf.isTrue(value)
              : false;
        },
        "resize-outline" : function(value){
            this.resizeOutline  = value
              ? !apf.isFalse(value)
              : false;
        },
        
        
        "login" : function(value, x) {
            apf.auth.init(x);
        },
        
        
        
        
        
        "debug" : function(value) {
            
            apf.debug = value;
        }
    }
});


if (apf.history)
    apf.addEventListener("load", function(){
        apf.history.init(apf.config.defaultPage, "page");
    });







/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.offline = {
    onLine : true
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */




/**
 * Stores data using a {@link term.datainstruction data instruction}.
 *
 * 
 * @param {String}      instruction  The {@link term.datainstruction data instruction} to be used to store the data.
 * @param {Object}      [options]    The options for this instruction. Available properties include:
 *   - multicall ([[Boolean]]): Whether this call should not be executed immediately, but saved for later sending using the `purge()` command
 *   - userdata (`Mixed`): Any data that is useful to access in the callback function
 *   - args ([[Array]]): The arguments of the call, overriding any specified in the data instruction
 *   - [xmlContext] ([[XMLElement]]): The subject of the xpath queries
 *   - [callback] ([[Function]]): The code that is executed when the call returns, either successfully or not
 *   {: #saveDataOptions}
 */
apf.saveData = 

/**
 * Retrieves data using a {@link term.datainstruction data instruction}.
 * 
 * #### Example
 * 
 * Here are several uses for data instructions:
 * 
 * ```xml
 *  <!-- loading aml from an xml file -->
 *  <a:bar aml="moreaml.xml" />
 *
 *  <a:bindings>
 *    <!-- loads data using an remote procedure protocol -->
 *    <a:load   get = "{comm.getData()}" />
 *
 *    <!-- inserts data using an remote procedure protocol -->
 *    <a:insert get = "{comm.getSubData([@id])}" />
 *  </a:bindings>
 *
 *  <a:actions>
 *    <!-- notifies the server that a file is renamed -->
 *    <a:rename set = "update_file.jsp?id=[@id]&name=[@name]" />
 *
 *    <!-- adds a node by retrieving it's xml from the server. -->
 *    <a:add    get = "new_user.xml" />
 *  </a:actions>
 *
 *  <!-- creates a model which is loaded into a list -->
 *  <a:list model="{webdav.getRoot()}" />
 *
 *  <!-- loads data into a model and when submitted sends the altered data back -->
 *  <a:model load="load_contact.jsp" submission="save_contact.jsp" />
 * ```
 *
 * @method getData
 * @param {String}      instruction  The {@link term.datainstruction data instruction} to be used to retrieve the data.
 * @param {XMLElement}  [xmlContext] The subject of the xpath queries
 * @param {Object}      [options]    The options for this instruction. Available properties include:
 *   - multicall ([[Boolean]]): Whether this call should not be executed immediately, but saved for later sending using the `purge()` command
 *   - userdata (`Mixed`): Any data that is useful to access in the callback function
 *   - args ([[Array]]): The arguments of the call, overriding any specified in the data instruction
 *   - [xmlContext] ([[XMLElement]]): The subject of the xpath queries
 *   - [callback] ([[Function]]): The code that is executed when the call returns, either successfully or not
 * @param {Function}    [callback]   The code that is executed when the call returns, either successfully or not
 */
apf.getData = function(instruction, options){
    if (!instruction) return false;

    //Instruction type detection
    var result, chr = instruction.charAt(0), callback = options.callback;

    
    var gCallback = 
    

    function(data, state, extra){
        var callback = options.callback
        
        if (state != apf.SUCCESS)
            return callback(data, state, extra || {});

        //Change this to warning?
        /*if (!data) {
            throw new Error(apf.formatErrorString(0, null,
                "Loading new data", "Could not load data. \n\
                Data instruction: '" + instruction + "'"));
        }*/

        return callback(data, state, extra || {});
    }
    
    if (!options) options = {}; //@todo optimize?
    var fParsed = options.fParsed || (instruction.indexOf("{") > -1 || instruction.indexOf("[") > -1
        ? apf.lm.compile(instruction, {
            withopt     : true, 
            precall     : options.precall,
            alwayscb    : true, 
            simplexpath : true
          })
        : {str: instruction, type: 2}); 

    if (options.precall && !options._pc) {
        options.asyncs = fParsed.asyncs;
        options._pc    = 1;
    }

    //@todo hack because we're not using compileNode.. an imperfection..
    if (fParsed.type == 3){
        if (fParsed.xpaths[0]) {
            var model = fParsed.xpaths[0], xpath = fParsed.xpaths[1];
            
            //@todo can this be async?
            if (model == "#" || xpath == "#") { //When there is a set model and not a generated xpath
                var m = (apf.lm.compile(instruction, {
                    xpathmode: 5
                }))();
                
                //@todo apf3 this needs to be fixed in live markup
                if (typeof m != "string") {
                    model = m.model && m.model.$isModel && m.model;
                    if (model)
                        xpath = m.xpath;
                    else if (m.model) {
                        model = apf.xmldb.findModel(m.model);
                        xpath = apf.xmlToXpath(m.model, model.data) + (m.xpath ? "/" + m.xpath : ""); //@todo make this better
                    }
                    else {
                        //Model is not yet available. When it comes available we will be recalled (at least for prop binds)
                        return;
                    }
                }
                else model = null;
            }
            else {
                
                model = apf.nameserver.get("model", model)
                
            }
            
            
        
            return gCallback(model.data.selectSingleNode(xpath), apf.SUCCESS, {});
        }
        else {
            
            
            return gCallback(options.xmlNode.data.selectSingleNode(fParsed.xpaths[1]), apf.SUCCESS, {});
        }
    }
    
    //xml generation
    if (chr == "<") {
        //string only
        if (fParsed.type == 2)
            result = fParsed.str;
        else
            return fParsed(options.xmlNode, gCallback, options);
    }
    //jslt fetching data
    else if ((chr == "[" || chr == "{")) { //(fParsed.asyncs || fParsed.models) && 
        return fParsed(options.xmlNode, gCallback, options);
    }
    //url
    else {
        if (fParsed.type == 1 || fParsed.type == 3) {
            var callback2 = callback;
            callback = options.callback = function(data, state, extra){
                if (options._pc === true)
                    return;
                
                if (state != apf.SUCCESS)
                    return callback2.apply(this, arguments);

                var url = data.split(" "), method = "get";
                if (url.length > 1 && url[0].length < 10) {
                    method = url.shift();
                    url    = url.join(" ");
                }
                else url = data;
                
                callback = options.callback = callback2;
                apf.oHttp.exec(method, [url], gCallback, options);
            }
            fParsed(options.xmlNode, gCallback, options);
        }
        else {
            if (options._pc === true)
                return;
            
            var url = instruction.split(" "), method = "get";
            if (url.length > 1 && url[0].length < 10) {
                method = url.shift();
                url    = url.join(" ");
            }
            else {
                url = instruction;
            }
            
            apf.oHttp.exec(method, [url.replace(/\\/g, "")], gCallback, options);
        }
    }
    
    if (result) {
        if (callback)
            gCallback(result, apf.SUCCESS, {});
        else {
            //apf.console.warn("Returning data directly in apf.getData(). \
                //This means that all callback communication ends in void!");
            return result;
        }
    }
};


/**
 * Creates a model object based on a {@link term.datainstruction data instruction}.
 *
 * @param {String} instruction  The {@link term.datainstruction data instruction} to be used to retrieve the data for the model
 * @param {apf.AmlNode} amlNode     The element the model is added to
 */
apf.setModel = function(instruction, amlNode){
    if (!instruction) return;

    //Find existing model
    var fParsed = instruction.indexOf("{") > -1 || instruction.indexOf("[") > -1
        ? apf.lm.compile(instruction, {
            //precall  : false, 
            alwayscb : true
        })
        : {
            type: 2,
            str : instruction
        };

    if (instruction == "@default" || fParsed.type == 2) {
        
        var model = apf.nameserver.get("model", instruction);
        if (model)
            return model.register(amlNode);
        else
        
            if (instruction == "@default")
            return;
        
        //@todo apf3.0 check here if string is valid url (relative or absolute)
        if (instruction.indexOf(".") == -1 && instruction.indexOf("/") == -1) {
            
            return;
        }
    }

    //Just an xpath doesnt work. We don't have context
    //var l, x;
    if (fParsed.type == 3) {//This won't work for complex xpaths
        if (fParsed.models) { //check for # in xpaths[i] to determine if its calculated
            if (fParsed.xpaths.length == 2 && fParsed.xpaths[0] != '#' && fParsed.xpaths [1] != '#') {
                
                
                
                apf.nameserver.get("model", fParsed.xpaths[0]).register(amlNode, fParsed.xpaths[1]);
                
                return;
            }
        }
        
    }

    if (amlNode.clear)
        amlNode.clear("loading");

    //Complex data fetch (possibly async) - data is loaded only once. 
    //Potential property binding has to take of the rest
    apf.getData(instruction, {
      parsed   : fParsed, 
      xmlNode  : amlNode && amlNode.xmlRoot,
      callback : function(data, state, extra){
        //@todo apf3.0 call onerror on amlNode
        if (state != apf.SUCCESS) {
            throw new Error(apf.formatErrorString(0, null,
                "Loading new data", "Could not load data into model. \
                \nMessage: " + extra.message + "\
                \nInstruction: '" + instruction + "'"));
        }
        
        if (!data)
            return amlNode.clear && amlNode.clear();

        if (typeof data == "string") {
            if (data.charAt(0) == "<")
                data = apf.getXml(data);
            else {
                //Assuming web service returned url
                if (data.indexOf("http://") == 0)
                    return apf.setModel(data, amlNode);
                else {
                    throw new Error("Invalid data from server");//@todo apf3.0 make proper apf error handling. apf.onerror
                }
            }
        }
        
        if (data.nodeFunc) { //Assuming a model was passed -- data.localName == "model" && 
            data.register(amlNode);
            return;
        }
        
        var model = apf.xmldb.findModel(data); //See if data is already loaded into a model
        if (model)
            model.register(amlNode, apf.xmlToXpath(data, model.data)); //@todo move function to xml library
        else
            new apf.model().register(amlNode).load(data);
    }});
};







/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

/*
 * @version: 1.0 Alpha-1
 * @author: Coolite Inc. http://www.coolite.com/
 * @date: 2008-04-13
 * @copyright: Copyright (c) 2006-2008, Coolite Inc. (http://www.coolite.com/). All rights reserved.
 * @license: Licensed under The MIT License. See license.txt and http://www.datejs.com/license/. 
 * @website: http://www.datejs.com/
 */
 
(function () {
    var $C = {
        /* Culture Name */
        name: "en-US",
        englishName: "English (United States)",
        nativeName: "English (United States)",
        
        /* Day Name Strings */
        dayNames: ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"],
        abbreviatedDayNames: ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"],
        shortestDayNames: ["Su", "Mo", "Tu", "We", "Th", "Fr", "Sa"],
        firstLetterDayNames: ["S", "M", "T", "W", "T", "F", "S"],
        
        /* Month Name Strings */
        monthNames: ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"],
        abbreviatedMonthNames: ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"],
    
        /* AM/PM Designators */
        amDesignator: "AM",
        pmDesignator: "PM",
    
        firstDayOfWeek: 0,
        twoDigitYearMax: 2029,
        
        /**
         * The dateElementOrder is based on the order of the 
         * format specifiers in the formatPatterns.DatePattern. 
         *
         * Example:
         <pre>
         shortDatePattern    dateElementOrder
         ------------------  ---------------- 
         "M/d/yyyy"          "mdy"
         "dd/MM/yyyy"        "dmy"
         "yyyy-MM-dd"        "ymd"
         </pre>
         *
         * The correct dateElementOrder is required by the parser to
         * determine the expected order of the date elements in the
         * string being parsed.
         */
        dateElementOrder: "mdy",
        
        /* Standard date and time format patterns */
        formatPatterns: {
            shortDate: "M/d/yyyy",
            longDate: "dddd, MMMM dd, yyyy",
            shortTime: "h:mm tt",
            longTime: "h:mm:ss tt",
            fullDateTime: "dddd, MMMM dd, yyyy h:mm:ss tt",
            sortableDateTime: "yyyy-MM-ddTHH:mm:ss",
            universalSortableDateTime: "yyyy-MM-dd HH:mm:ssZ",
            rfc1123: "ddd, dd MMM yyyy HH:mm:ss GMT",
            monthDay: "MMMM dd",
            yearMonth: "MMMM, yyyy"
        },
    
        /**
         * NOTE: If a string format is not parsing correctly, but
         * you would expect it parse, the problem likely lies below. 
         * 
         * The following regex patterns control most of the string matching
         * within the parser.
         * 
         * The Month name and Day name patterns were automatically generated
         * and in general should be (mostly) correct. 
         *
         * Beyond the month and day name patterns are natural language strings.
         * Example: "next", "today", "months"
         *
         * These natural language string may NOT be correct for this culture. 
         * If they are not correct, please translate and edit this file
         * providing the correct regular expression pattern. 
         *
         * If you modify this file, please post your revised CultureInfo file
         * to the Datejs Forum located at http://www.datejs.com/forums/.
         *
         * Please mark the subject of the post with [CultureInfo]. Example:
         *    Subject: [CultureInfo] Translated "da-DK" Danish(Denmark)
         * 
         * We will add the modified patterns to the master source files.
         *
         * As well, please review the list of "Future Strings" section below. 
         */    
        regexPatterns: {
            jan: /^jan(uary)?/i,
            feb: /^feb(ruary)?/i,
            mar: /^mar(ch)?/i,
            apr: /^apr(il)?/i,
            may: /^may/i,
            jun: /^jun(e)?/i,
            jul: /^jul(y)?/i,
            aug: /^aug(ust)?/i,
            sep: /^sep(t(ember)?)?/i,
            oct: /^oct(ober)?/i,
            nov: /^nov(ember)?/i,
            dec: /^dec(ember)?/i,
    
            sun: /^su(n(day)?)?/i,
            mon: /^mo(n(day)?)?/i,
            tue: /^tu(e(s(day)?)?)?/i,
            wed: /^we(d(nesday)?)?/i,
            thu: /^th(u(r(s(day)?)?)?)?/i,
            fri: /^fr(i(day)?)?/i,
            sat: /^sa(t(urday)?)?/i,
    
            future: /^next/i,
            past: /^last|past|prev(ious)?/i,
            add: /^(\+|aft(er)?|from|hence)/i,
            subtract: /^(\-|bef(ore)?|ago)/i,
            
            yesterday: /^yes(terday)?/i,
            today: /^t(od(ay)?)?/i,
            tomorrow: /^tom(orrow)?/i,
            now: /^n(ow)?/i,
            
            millisecond: /^ms|milli(second)?s?/i,
            second: /^sec(ond)?s?/i,
            minute: /^mn|min(ute)?s?/i,
            hour: /^h(our)?s?/i,
            week: /^w(eek)?s?/i,
            month: /^m(onth)?s?/i,
            day: /^d(ay)?s?/i,
            year: /^y(ear)?s?/i,
    
            shortMeridian: /^(a|p)/i,
            longMeridian: /^(a\.?m?\.?|p\.?m?\.?)/i,
            timezone: /^((e(s|d)t|c(s|d)t|m(s|d)t|p(s|d)t)|((gmt)?\s*(\+|\-)\s*\d\d\d\d?)|gmt|utc)/i,
            ordinalSuffix: /^\s*(st|nd|rd|th)/i,
            timeContext: /^\s*(\:|a(?!u|p)|p)/i
        },
    
        timezones: [
            {name:"UTC", offset:"-000"},
            {name:"GMT", offset:"-000"},
            {name:"EST", offset:"-0500"},
            {name:"EDT", offset:"-0400"},
            {name:"CST", offset:"-0600"},
            {name:"CDT", offset:"-0500"},
            {name:"MST", offset:"-0700"},
            {name:"MDT", offset:"-0600"},
            {name:"PST", offset:"-0800"},
            {name:"PDT", offset:"-0700"}
        ]
    };
    
    var $D = Date, 
        $P = $D.prototype,
        p = function(s, l) {
            if (!l)
                l = 2;
            return ("000" + s).slice(l * -1);
        };

    /**
     * Resets the time of this Date object to 12:00 AM (00:00), which is the 
     * start of the day.
     * @return {Date}    this
     */
    $P.clearTime = function() {
        this.setHours(0);
        this.setMinutes(0);
        this.setSeconds(0);
        this.setMilliseconds(0);
        return this;
    };

    /**
     * Resets the time of this Date object to the current time ('now').
     * @return {Date}    this
     */
    $P.setTimeToNow = function() {
        var n = new Date();
        this.setHours(n.getHours());
        this.setMinutes(n.getMinutes());
        this.setSeconds(n.getSeconds());
        this.setMilliseconds(n.getMilliseconds());
        return this;
    };

    /** 
     * Gets a date that is set to the current date. The time is set to the start 
     * of the day (00:00 or 12:00 AM).
     * @return {Date}    The current date.
     */
    $D.today = function() {
        return new Date().clearTime();
    };

    /**
     * Compares the first date to the second date and returns an number indication 
     * of their relative values.  
     * @param {Date}     First Date object to compare [Required].
     * @param {Date}     Second Date object to compare to [Required].
     * @return {Number}  -1 = date1 is lessthan date2. 0 = values are equal. 
     *                    1 = date1 is greaterthan date2.
     */
    $D.compare = function(date1, date2) {
        if (isNaN(date1) || isNaN(date2))
            throw new Error(date1 + " - " + date2); 
        else if (date1 instanceof Date && date2 instanceof Date)
            return (date1 < date2) ? -1 : (date1 > date2) ? 1 : 0;
        else
            throw new TypeError(date1 + " - " + date2); 
    };
    
    /**
     * Compares the first Date object to the second Date object and returns true 
     * if they are equal.  
     * @param {Date}     First Date object to compare [Required]
     * @param {Date}     Second Date object to compare to [Required]
     * @return {Boolean} true if dates are equal. false if they are not equal.
     */
    $D.equals = function(date1, date2) { 
        return (date1.compareTo(date2) === 0); 
    };

    /**
     * Gets the day number (0-6) if given a CultureInfo specific string which is 
     * a valid dayName, abbreviatedDayName or shortestDayName (two char).
     * @param {String}   The name of the day (eg. "Monday, "Mon", "tuesday", "tue", "We", "we").
     * @return {Number}  The day number
     */
    $D.getDayNumberFromName = function(name) {
        var n = $C.dayNames,
            m = $C.abbreviatedDayNames,
            o = $C.shortestDayNames,
            s = name.toLowerCase();
        for (var i = 0; i < n.length; i++) { 
            if (n[i].toLowerCase() == s || m[i].toLowerCase() == s || o[i].toLowerCase() == s)
                return i; 
        }
        return -1;  
    };
    
    /**
     * Gets the month number (0-11) if given a Culture Info specific string which 
     * is a valid monthName or abbreviatedMonthName.
     * @param {String}   The name of the month (eg. "February, "Feb", "october", "oct").
     * @return {Number}  The day number
     */
    $D.getMonthNumberFromName = function(name) {
        var n = $C.monthNames,
            m = $C.abbreviatedMonthNames,
            s = name.toLowerCase();
        for (var i = 0; i < n.length; i++) {
            if (n[i].toLowerCase() == s || m[i].toLowerCase() == s)
                return i; 
        }
        return -1;
    };

    /**
     * Determines if the current date instance is within a LeapYear.
     * @param {Number}   The year.
     * @return {Boolean} true if date is within a LeapYear, otherwise false.
     */
    $D.isLeapYear = function(year) { 
        return ((year % 4 === 0 && year % 100 !== 0) || year % 400 === 0); 
    };

    /**
     * Gets the number of days in the month, given a year and month value. 
     * Automatically corrects for LeapYear.
     * @param {Number}   The year.
     * @param {Number}   The month (0-11).
     * @return {Number}  The number of days in the month.
     */
    $D.getDaysInMonth = function(year, month) {
        return [31, ($D.isLeapYear(year) ? 29 : 28), 31, 30, 31, 30, 31, 31, 30, 31, 30, 31][month];
    };
 
    $D.getTimezoneAbbreviation = function(offset) {
        var z = $C.timezones, p;
        for (var i = 0; i < z.length; i++) {
            if (z[i].offset === offset)
                return z[i].name;
        }
        return null;
    };
    
    $D.getTimezoneOffset = function(name) {
        var z = $C.timezones, p;
        for (var i = 0; i < z.length; i++) {
            if (z[i].name === name.toUpperCase())
                return z[i].offset;
        }
        return null;
    };

    /**
     * Returns a new Date object that is an exact date and time copy of the 
     * original instance.
     * @return {Date}    A new Date instance
     */
    $P.clone = function() {
        return new Date(this.getTime()); 
    };

    /**
     * Compares this instance to a Date object and returns an number indication 
     * of their relative values.  
     * @param {Date}     Date object to compare [Required]
     * @return {Number}  -1 = this is lessthan date. 0 = values are equal.
     *                    1 = this is greaterthan date.
     */
    $P.compareTo = function(date) {
        return Date.compare(this, date);
    };

    /**
     * Compares this instance to another Date object and returns true if they are equal.  
     * @param {Date}     Date object to compare. If no date to compare, new Date() 
     *                   [now] is used.
     * @return {Boolean} true if dates are equal. false if they are not equal.
     */
    $P.equals = function(date) {
        return Date.equals(this, date || new Date());
    };

    /**
     * Determines if this instance is between a range of two dates or equal to 
     * either the start or end dates.
     * @param {Date}     Start of range [Required]
     * @param {Date}     End of range [Required]
     * @return {Boolean} true is this is between or equal to the start and end 
     *                   dates, else false
     */
    $P.between = function(start, end) {
        return this.getTime() >= start.getTime() && this.getTime() <= end.getTime();
    };

    /**
     * Determines if this date occurs after the date to compare to.
     * @param {Date}     Date object to compare. If no date to compare, new Date() 
     *                   ("now") is used.
     * @return {Boolean} true if this date instance is greater than the date to 
     *                   compare to (or "now"), otherwise false.
     */
    $P.isAfter = function(date) {
        return this.compareTo(date || new Date()) === 1;
    };

    /**
     * Determines if this date occurs before the date to compare to.
     * @param {Date}     Date object to compare. If no date to compare, new Date()
     *                   ("now") is used.
     * @return {Boolean} true if this date instance is less than the date to 
     *                   compare to (or "now").
     */
    $P.isBefore = function(date) {
        return (this.compareTo(date || new Date()) === -1);
    };

    /**
     * Determines if the current Date instance occurs on the same Date as the supplied 'date'. 
     * If no 'date' to compare to is provided, the current Date instance is compared to 'today'. 
     * @param {Date}     Date object to compare. If no date to compare, the current Date ("now") is used.
     * @return {Boolean} true if this Date instance occurs on the same Day as the supplied 'date'.
     */
    $P.isToday = $P.isSameDay = function(date) {
        return this.clone().clearTime().equals((date || new Date()).clone().clearTime());
    };
    
    /**
     * Adds the specified number of milliseconds to this instance. 
     * @param {Number}   The number of milliseconds to add. The number can be 
     *                   positive or negative [Required]
     * @return {Date}    this
     */
    $P.addMilliseconds = function(value) {
        this.setMilliseconds(this.getMilliseconds() + value * 1);
        return this;
    };

    /**
     * Adds the specified number of seconds to this instance. 
     * @param {Number}   The number of seconds to add. The number can be positive 
     *                   or negative [Required]
     * @return {Date}    this
     */
    $P.addSeconds = function(value) { 
        return this.addMilliseconds(value * 1000); 
    };

    /**
     * Adds the specified number of seconds to this instance. 
     * @param {Number}   The number of seconds to add. The number can be positive 
     *                   or negative [Required]
     * @return {Date}    this
     */
    $P.addMinutes = function(value) { 
        return this.addMilliseconds(value * 60000); /* 60*1000 */
    };

    /**
     * Adds the specified number of hours to this instance. 
     * @param {Number}   The number of hours to add. The number can be positive 
     *                   or negative [Required]
     * @return {Date}    this
     */
    $P.addHours = function(value) { 
        return this.addMilliseconds(value * 3600000); /* 60*60*1000 */
    };

    /**
     * Adds the specified number of days to this instance. 
     * @param {Number}   The number of days to add. The number can be positive 
     *                   or negative [Required]
     * @return {Date}    this
     */
    $P.addDays = function(value) {
        this.setDate(this.getDate() + value * 1);
        return this;
    };

    /**
     * Adds the specified number of weeks to this instance. 
     * @param {Number}   The number of weeks to add. The number can be positive 
     *                   or negative [Required]
     * @return {Date}    this
     */
    $P.addWeeks = function(value) { 
        return this.addDays(value * 7);
    };

    /**
     * Adds the specified number of months to this instance. 
     * @param {Number}   The number of months to add. The number can be positive 
     *                   or negative [Required]
     * @return {Date}    this
     */
    $P.addMonths = function(value) {
        var n = this.getDate();
        this.setDate(1);
        this.setMonth(this.getMonth() + value * 1);
        this.setDate(Math.min(n, $D.getDaysInMonth(this.getFullYear(), this.getMonth())));
        return this;
    };

    /**
     * Adds the specified number of years to this instance. 
     * @param {Number}   The number of years to add. The number can be positive 
     *                   or negative [Required]
     * @return {Date}    this
     */
    $P.addYears = function(value) {
        return this.addMonths(value * 12);
    };

    /**
     * Adds (or subtracts) to the value of the years, months, weeks, days, hours, 
     * minutes, seconds, milliseconds of the date instance using given configuration 
     * object. Positive and Negative values allowed.
     * Example
    <pre><code>
    Date.today().add( { days: 1, months: 1 } )
     
    new Date().add( { years: -1 } )
    </code></pre> 
     * @param {Object}   Configuration object containing attributes (months, days, etc.)
     * @return {Date}    this
     */
    $P.add = function(config) {
        if (typeof config == "number") {
            this._orient = config;
            return this;    
        }
        
        var x = config;
        
        if (x.milliseconds)
            this.addMilliseconds(x.milliseconds); 
        if (x.seconds)
            this.addSeconds(x.seconds); 
        if (x.minutes)
            this.addMinutes(x.minutes); 
        if (x.hours)
            this.addHours(x.hours); 
        if (x.weeks)
            this.addWeeks(x.weeks); 
        if (x.months)
            this.addMonths(x.months); 
        if (x.years)
            this.addYears(x.years); 
        if (x.days)
            this.addDays(x.days); 
        return this;
    };
    
    var $y, $m, $d;
    
    /**
     * Get the week number. Week one (1) is the week which contains the first 
     * Thursday of the year. Monday is considered the first day of the week.
     * This algorithm is a JavaScript port of the work presented by Claus 
     * Tøndering at http://www.tondering.dk/claus/cal/node8.html#SECTION00880000000000000000
     * .getWeek() Algorithm Copyright (c) 2008 Claus Tondering.
     * The .getWeek() function does NOT convert the date to UTC. The local datetime 
     * is used. Please use .getISOWeek() to get the week of the UTC converted date.
     * @return {Number}  1 to 53
     */
    $P.getWeek = function() {
        var a, b, c, d, e, f, g, n, s, w;
        
        $y = (!$y) ? this.getFullYear() : $y;
        $m = (!$m) ? this.getMonth() + 1 : $m;
        $d = (!$d) ? this.getDate() : $d;

        if ($m <= 2) {
            a = $y - 1;
            b = (a / 4 | 0) - (a / 100 | 0) + (a / 400 | 0);
            c = ((a - 1) / 4 | 0) - ((a - 1) / 100 | 0) + ((a - 1) / 400 | 0);
            s = b - c;
            e = 0;
            f = $d - 1 + (31 * ($m - 1));
        } else {
            a = $y;
            b = (a / 4 | 0) - (a / 100 | 0) + (a / 400 | 0);
            c = ((a - 1) / 4 | 0) - ((a - 1) / 100 | 0) + ((a - 1) / 400 | 0);
            s = b - c;
            e = s + 1;
            f = $d + ((153 * ($m - 3) + 2) / 5) + 58 + s;
        }
        
        g = (a + b) % 7;
        d = (f + g - e) % 7;
        n = (f + 3 - d) | 0;

        if (n < 0) {
            w = 53 - ((g - s) / 5 | 0);
        } else if (n > 364 + s) {
            w = 1;
        } else {
            w = (n / 7 | 0) + 1;
        }
        
        $y = $m = $d = null;
        
        return w;
    };
    
    /**
     * Get the ISO 8601 week number. Week one ("01") is the week which contains the 
     * first Thursday of the year. Monday is considered the first day of the week.
     * The .getISOWeek() function does convert the date to it's UTC value. 
     * Please use .getWeek() to get the week of the local date.
     * @return {String}  "01" to "53"
     */
    $P.getISOWeek = function() {
        $y = this.getUTCFullYear();
        $m = this.getUTCMonth() + 1;
        $d = this.getUTCDate();
        return p(this.getWeek());
    };

    /**
     * Moves the date to Monday of the week set. Week one (1) is the week which 
     * contains the first Thursday of the year.
     * @param {Number}   A Number (1 to 53) that represents the week of the year.
     * @return {Date}    this
     */    
    $P.setWeek = function(n) {
        return this.moveToDayOfWeek(1).addWeeks(n - this.getWeek());
    };

    // private
    var validate = function(n, min, max, name) {
        if (typeof n == "undefined")
            return false;
        else if (typeof n != "number")
            throw new TypeError(n + " is not a Number.");
        else if (n < min || n > max)
            throw new RangeError(n + " is not a valid value for " + name + "."); 
        return true;
    };

    /**
     * Validates the number is within an acceptable range for milliseconds [0-999].
     * @param {Number}   The number to check if within range.
     * @return {Boolean} true if within range, otherwise false.
     */
    $D.validateMillisecond = function(value) {
        return validate(value, 0, 999, "millisecond");
    };

    /**
     * Validates the number is within an acceptable range for seconds [0-59].
     * @param {Number}   The number to check if within range.
     * @return {Boolean} true if within range, otherwise false.
     */
    $D.validateSecond = function(value) {
        return validate(value, 0, 59, "second");
    };

    /**
     * Validates the number is within an acceptable range for minutes [0-59].
     * @param {Number}   The number to check if within range.
     * @return {Boolean} true if within range, otherwise false.
     */
    $D.validateMinute = function(value) {
        return validate(value, 0, 59, "minute");
    };

    /**
     * Validates the number is within an acceptable range for hours [0-23].
     * @param {Number}   The number to check if within range.
     * @return {Boolean} true if within range, otherwise false.
     */
    $D.validateHour = function(value) {
        return validate(value, 0, 23, "hour");
    };

    /**
     * Validates the number is within an acceptable range for the days in a month 
     * [0 - MaxDaysInMonth].
     * @param {Number}   The number to check if within range.
     * @return {Boolean} true if within range, otherwise false.
     */
    $D.validateDay = function(value, year, month) {
        return validate(value, 1, $D.getDaysInMonth(year, month), "day");
    };

    /**
     * Validates the number is within an acceptable range for months [0-11].
     * @param {Number}   The number to check if within range.
     * @return {Boolean} true if within range, otherwise false.
     */
    $D.validateMonth = function(value) {
        return validate(value, 0, 11, "month");
    };

    /**
     * Validates the number is within an acceptable range for years.
     * @param {Number}   The number to check if within range.
     * @return {Boolean} true if within range, otherwise false.
     */
    $D.validateYear = function(value) {
        return validate(value, 0, 9999, "year");
    };

    /**
     * Set the value of year, month, day, hour, minute, second, millisecond of 
     * date instance using given configuration object.
     * Example
    <pre><code>
    Date.today().set( { day: 20, month: 1 } )

    new Date().set( { millisecond: 0 } )
    </code></pre>
     * 
     * @param {Object}   Configuration object containing attributes (month, day, etc.)
     * @return {Date}    this
     */
    $P.set = function(config) {
        if ($D.validateMillisecond(config.millisecond))
            this.addMilliseconds(config.millisecond - this.getMilliseconds()); 
        
        if ($D.validateSecond(config.second))
            this.addSeconds(config.second - this.getSeconds()); 
        
        if ($D.validateMinute(config.minute))
            this.addMinutes(config.minute - this.getMinutes()); 
        
        if ($D.validateHour(config.hour))
            this.addHours(config.hour - this.getHours()); 
        
        if ($D.validateMonth(config.month))
            this.addMonths(config.month - this.getMonth()); 

        if ($D.validateYear(config.year))
            this.addYears(config.year - this.getFullYear()); 
        
        /* day has to go last because you can't validate the day without first knowing the month */
        if ($D.validateDay(config.day, this.getFullYear(), this.getMonth()))
            this.addDays(config.day - this.getDate()); 
        
        if (config.timezone)
            this.setTimezone(config.timezone); 
        
        if (config.timezoneOffset)
            this.setTimezoneOffset(config.timezoneOffset); 

        if (config.week && validate(config.week, 0, 53, "week"))
            this.setWeek(config.week);
        
        return this;
    };

    /**
     * Moves the date to the first day of the month.
     * @return {Date}    this
     */
    $P.moveToFirstDayOfMonth = function() {
        return this.set({ day: 1 });
    };

    /**
     * Moves the date to the last day of the month.
     * @return {Date}    this
     */
    $P.moveToLastDayOfMonth = function() { 
        return this.set({ day: $D.getDaysInMonth(this.getFullYear(), this.getMonth())});
    };

    /**
     * Moves the date to the next n'th occurrence of the dayOfWeek starting from 
     * the beginning of the month. The number (-1) is a magic number and will return 
     * the last occurrence of the dayOfWeek in the month.
     * @param {Number}   The dayOfWeek to move to
     * @param {Number}   The n'th occurrence to move to. Use (-1) to return the 
     *                   last occurrence in the month
     * @return {Date}    this
     */
    $P.moveToNthOccurrence = function(dayOfWeek, occurrence) {
        var shift = 0;
        if (occurrence > 0) {
            shift = occurrence - 1;
        }
        else if (occurrence === -1) {
            this.moveToLastDayOfMonth();
            if (this.getDay() !== dayOfWeek)
                this.moveToDayOfWeek(dayOfWeek, -1);
            return this;
        }
        return this.moveToFirstDayOfMonth().addDays(-1)
                   .moveToDayOfWeek(dayOfWeek, +1).addWeeks(shift);
    };

    /**
     * Move to the next or last dayOfWeek based on the orient value.
     * @param {Number}   The dayOfWeek to move to
     * @param {Number}   Forward (+1) or Back (-1). Defaults to +1. [Optional]
     * @return {Date}    this
     */
    $P.moveToDayOfWeek = function(dayOfWeek, orient) {
        var diff = (dayOfWeek - this.getDay() + 7 * (orient || +1)) % 7;
        return this.addDays((diff === 0) ? diff += 7 * (orient || +1) : diff);
    };

    /**
     * Move to the next or last month based on the orient value.
     * @param {Number}   The month to move to. 0 = January, 11 = December
     * @param {Number}   Forward (+1) or Back (-1). Defaults to +1. [Optional]
     * @return {Date}    this
     */
    $P.moveToMonth = function(month, orient) {
        var diff = (month - this.getMonth() + 12 * (orient || +1)) % 12;
        return this.addMonths((diff === 0) ? diff += 12 * (orient || +1) : diff);
    };

    /**
     * Get the Ordinal day (numeric day number) of the year, adjusted for leap year.
     * @return {Number} 1 through 365 (366 in leap years)
     */
    $P.getOrdinalNumber = function() {
        return Math.ceil((this.clone().clearTime() 
            - new Date(this.getFullYear(), 0, 1)) / 86400000) + 1;
    };

    /**
     * Get the time zone abbreviation of the current date.
     * @return {String} The abbreviated time zone name (e.g. "EST")
     */
    $P.getTimezone = function() {
        return $D.getTimezoneAbbreviation(this.getUTCOffset());
    };

    $P.setTimezoneOffset = function(offset) {
        var here = this.getTimezoneOffset(), there = Number(offset) * -6 / 10;
        return this.addMinutes(there - here); 
    };

    $P.setTimezone = function(offset) { 
        return this.setTimezoneOffset($D.getTimezoneOffset(offset)); 
    };

    /**
     * Indicates whether Daylight Saving Time is observed in the current time zone.
     * @return {Boolean} true|false
     */
    $P.hasDaylightSavingTime = function() { 
        return (Date.today().set({month: 0, day: 1}).getTimezoneOffset() 
            !== Date.today().set({month: 6, day: 1}).getTimezoneOffset());
    };
    
    /**
     * Indicates whether this Date instance is within the Daylight Saving Time 
     * range for the current time zone.
     * @return {Boolean} true|false
     */
    $P.isDaylightSavingTime = function() {
        return Date.today().set({month: 0, day: 1}).getTimezoneOffset() != this.getTimezoneOffset();
    };

    /**
     * Get the offset from UTC of the current date.
     * @return {String} The 4-character offset string prefixed with + or - (e.g. "-0500")
     */
    $P.getUTCOffset = function() {
        var n = this.getTimezoneOffset() * -10 / 6, r;
        if (n < 0) { 
            r = (n - 10000).toString(); 
            return r.charAt(0) + r.substr(2); 
        }
        else { 
            r = (n + 10000).toString();  
            return "+" + r.substr(1); 
        }
    };
    
    $P.getUTCTime = function() {
        //Date.UTC(year, month[, date[, hrs[, min[, sec[, ms]]]]])
        return Date.UTC(this.getUTCFullYear(), this.getUTCMonth(), this.getUTCDate(),
            this.getUTCHours(), this.getUTCMinutes(), this.getUTCSeconds(),
            this.getUTCMilliseconds());
    };

    /**
     * Returns the number of milliseconds between this date and date.
     * @param {Date} Defaults to now
     * @return {Number} The diff in milliseconds
     */
    $P.getElapsed = function(date) {
        return (date || new Date()) - this;
    };

    if (!$P.toISOString) {
        /**
         * Converts the current date instance into a string with an ISO 8601 format. 
         * The date is converted to it's UTC value.
         * @return {String}  ISO 8601 string of date
         */
        $P.toISOString = function() {
            // From http://www.json.org/json.js. Public Domain. 
            function f(n) {
                return n < 10 ? '0' + n : n;
            }

            return '"' + this.getUTCFullYear()   + '-' +
                f(this.getUTCMonth() + 1) + '-' +
                f(this.getUTCDate())      + 'T' +
                f(this.getUTCHours())     + ':' +
                f(this.getUTCMinutes())   + ':' +
                f(this.getUTCSeconds())   + 'Z"';
        };
    }
    
    // private
    $P._toString = $P.toString;

    /**
     * Converts the value of the current Date object to its equivalent string representation.
     * Format Specifiers
    <pre>
    CUSTOM DATE AND TIME FORMAT STRINGS
    Format  Description                                                                  Example
    ------  ---------------------------------------------------------------------------  -----------------------
     s      The seconds of the minute between 0-59.                                      "0" to "59"
     ss     The seconds of the minute with leading zero if required.                     "00" to "59"
     
     m      The minute of the hour between 0-59.                                         "0"  or "59"
     mm     The minute of the hour with leading zero if required.                        "00" or "59"
     
     h      The hour of the day between 1-12.                                            "1"  to "12"
     hh     The hour of the day with leading zero if required.                           "01" to "12"
     
     H      The hour of the day between 0-23.                                            "0"  to "23"
     HH     The hour of the day with leading zero if required.                           "00" to "23"
     
     d      The day of the month between 1 and 31.                                       "1"  to "31"
     dd     The day of the month with leading zero if required.                          "01" to "31"
     ddd    Abbreviated day name. $C.abbreviatedDayNames.                                "Mon" to "Sun" 
     dddd   The full day name. $C.dayNames.                                              "Monday" to "Sunday"
     
     M      The month of the year between 1-12.                                          "1" to "12"
     MM     The month of the year with leading zero if required.                         "01" to "12"
     MMM    Abbreviated month name. $C.abbreviatedMonthNames.                            "Jan" to "Dec"
     MMMM   The full month name. $C.monthNames.                                          "January" to "December"

     yy     The year as a two-digit number.                                              "99" or "08"
     yyyy   The full four digit year.                                                    "1999" or "2008"
     
     t      Displays the first character of the A.M./P.M. designator.                    "A" or "P"
            $C.amDesignator or $C.pmDesignator
     tt     Displays the A.M./P.M. designator.                                           "AM" or "PM"
            $C.amDesignator or $C.pmDesignator
     
     S      The ordinal suffix ("st, "nd", "rd" or "th") of the current day.            "st, "nd", "rd" or "th"

|| *Format* || *Description* || *Example* ||
|| d      || The CultureInfo shortDate Format Pattern                                     || "M/d/yyyy" ||
|| D      || The CultureInfo longDate Format Pattern                                      || "dddd, MMMM dd, yyyy" ||
|| F      || The CultureInfo fullDateTime Format Pattern                                  || "dddd, MMMM dd, yyyy h:mm:ss tt" ||
|| m      || The CultureInfo monthDay Format Pattern                                      || "MMMM dd" ||
|| r      || The CultureInfo rfc1123 Format Pattern                                       || "ddd, dd MMM yyyy HH:mm:ss GMT" ||
|| s      || The CultureInfo sortableDateTime Format Pattern                              || "yyyy-MM-ddTHH:mm:ss" ||
|| t      || The CultureInfo shortTime Format Pattern                                     || "h:mm tt" ||
|| T      || The CultureInfo longTime Format Pattern                                      || "h:mm:ss tt" ||
|| u      || The CultureInfo universalSortableDateTime Format Pattern                     || "yyyy-MM-dd HH:mm:ssZ" ||
|| y      || The CultureInfo yearMonth Format Pattern                                     || "MMMM, yyyy" ||
     

    STANDARD DATE AND TIME FORMAT STRINGS
    Format  Description                                                                  Example ("en-US")
    ------  ---------------------------------------------------------------------------  -----------------------
     d      The CultureInfo shortDate Format Pattern                                     "M/d/yyyy"
     D      The CultureInfo longDate Format Pattern                                      "dddd, MMMM dd, yyyy"
     F      The CultureInfo fullDateTime Format Pattern                                  "dddd, MMMM dd, yyyy h:mm:ss tt"
     m      The CultureInfo monthDay Format Pattern                                      "MMMM dd"
     r      The CultureInfo rfc1123 Format Pattern                                       "ddd, dd MMM yyyy HH:mm:ss GMT"
     s      The CultureInfo sortableDateTime Format Pattern                              "yyyy-MM-ddTHH:mm:ss"
     t      The CultureInfo shortTime Format Pattern                                     "h:mm tt"
     T      The CultureInfo longTime Format Pattern                                      "h:mm:ss tt"
     u      The CultureInfo universalSortableDateTime Format Pattern                     "yyyy-MM-dd HH:mm:ssZ"
     y      The CultureInfo yearMonth Format Pattern                                     "MMMM, yyyy"
    </pre>
     * @param {String}   A format string consisting of one or more format spcifiers [Optional].
     * @return {String}  A string representation of the current Date object.
     */
    $P.toString = function(format) {
        var x = this;
        
        // Standard Date and Time Format Strings. Formats pulled from CultureInfo file and
        // may vary by culture. 
        if (format && format.length == 1) {
            var c = $C.formatPatterns;
            x.t = x.toString;
            switch (format) {
            case "d": 
                return x.t(c.shortDate);
            case "D":
                return x.t(c.longDate);
            case "F":
                return x.t(c.fullDateTime);
            case "m":
                return x.t(c.monthDay);
            case "r":
                return x.t(c.rfc1123);
            case "s":
                return x.t(c.sortableDateTime);
            case "t":
                return x.t(c.shortTime);
            case "T":
                return x.t(c.longTime);
            case "u":
                return x.t(c.universalSortableDateTime);
            case "y":
                return x.t(c.yearMonth);
            }    
        }
        
        var ord = function (n) {
                switch (n * 1) {
                case 1: 
                case 21: 
                case 31: 
                    return "st";
                case 2: 
                case 22: 
                    return "nd";
                case 3: 
                case 23: 
                    return "rd";
                default: 
                    return "th";
                }
            };
        
        return format ? format.replace(/(\\)?(dd?d?d?|MM?M?M?|yy?y?y?|hh?|HH?|mm?|ss?|tt?|S)/g, 
        function (m) {
            if (m.charAt(0) === "\\") {
                return m.replace("\\", "");
            }
            x.h = x.getHours;
            switch (m) {
            case "hh":
                return p(x.h() < 13 ? (x.h() === 0 ? 12 : x.h()) : (x.h() - 12));
            case "h":
                return x.h() < 13 ? (x.h() === 0 ? 12 : x.h()) : (x.h() - 12);
            case "HH":
                return p(x.h());
            case "H":
                return x.h();
            case "mm":
                return p(x.getMinutes());
            case "m":
                return x.getMinutes();
            case "ss":
                return p(x.getSeconds());
            case "s":
                return x.getSeconds();
            case "yyyy":
                return p(x.getFullYear(), 4);
            case "yy":
                return p(x.getFullYear());
            case "dddd":
                return $C.dayNames[x.getDay()];
            case "ddd":
                return $C.abbreviatedDayNames[x.getDay()];
            case "dd":
                return p(x.getDate());
            case "d":
                return x.getDate();
            case "MMMM":
                return $C.monthNames[x.getMonth()];
            case "MMM":
                return $C.abbreviatedMonthNames[x.getMonth()];
            case "MM":
                return p((x.getMonth() + 1));
            case "M":
                return x.getMonth() + 1;
            case "t":
                return x.h() < 12 ? $C.amDesignator.substring(0, 1) : $C.pmDesignator.substring(0, 1);
            case "tt":
                return x.h() < 12 ? $C.amDesignator : $C.pmDesignator;
            case "S":
                return ord(x.getDate());
            default: 
                return m;
            }
        }
        ) : this._toString();
    };
}());





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: socket://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * @class apf.layout
 *
 * Takes care of the spatial order of elements within the display area
 * of the browser. Layouts can be saved to XML and loaded again. Window
 * elements are dockable, which means the user can change the layout as s/he
 * wishes. The state of the layout can be saved as XML at any time.
 *
 * #### Example
 * 
 * This example shows five windows which have a layout defined in layout.xml.
 * 
 * ```xml
 *  <a:appsettings layout="[mdlLayouts::layout[1]]" />
 *  <a:model id="mdlLayouts" src="layout.xml" />
 *  
 *  <a:window title="Main Window" id="b1" />
 *  <a:window title="Tree Window" id="b2" />
 *  <a:window title="Window of Oppertunity" id="b3" />
 *  <a:window title="Small window" id="b4" />
 *  <a:window title="Some Window" id="b5" />
 * ```
 *
 * This is the layout file containing two layouts (_layout.xml_):
 * 
 * ```xml
 *  <layouts>
 *      <layout name="Layout 1" margin="2,2,2,2">
 *          <vbox edge="splitter">
 *              <node name="b1" edge="2"/>
 *              <hbox edge="2">
 *                  <vbox weight="1">
 *                      <node name="b2"/>
 *                      <node name="b3"/>
 *                  </vbox>
 *                  <node name="b4" weight="1" />
 *              </hbox>
 *              <node name="b5" height="20" />
 *          </vbox>
 *      </layout>
 *
 *      <layout name="Layout 2">
 *          <vbox edge="splitter">
 *              <node name="b1" edge="2" />
 *              <node name="b2" height="100" />
 *              <hbox edge="2">
 *                  <node name="b3" width="20%" />
 *                  <node name="b4" width="100" />
 *              </hbox>
 *              <node name="b5" height="20" />
 *          </vbox>
 *      </layout>
 *  </layouts>
 * ```
 *
 * By binding on the _layout.xml_ you can easily create a layout manager.
 * 
 * ```xml
 *  <a:list id="lstLayouts"
 *    model          = "mdlLayouts"
 *    allowdeselect  = "false"
 *    onafterselect  = "
 *      if(!this.selected || apf.layout.isLoadedXml(this.selected))
 *          return;
 *     
 *      apf.layout.saveXml();
 *      apf.layout.loadXml(this.selected);
 *    "
 *    onbeforeremove = "return confirm('Do you want to delete this layout?')">
 *      <a:bindings>
 *          <a:caption match="[@name]" />
 *          <a:icon value="layout.png" />
 *          <a:each match="[layout]" />
 *      </a:bindings>
 *      <a:actions>
 *          <a:rename match="[.]" />
 *          <a:remove match="[.]" />
 *      </a:actions>
 *  </a:list>
 *  <a:button
 *    onclick = "
 *      if (!lstLayouts.selected)
 *          return;
 *     
 *      var newLayout = apf.layout.getXml(document.body);
 *      newLayout.setAttribute('name', 'New');
 *      apf.xmldb.appendChild(lstLayouts.selected.parentNode, newLayout);
 *      lstLayouts.select(newLayout, null, null, null, null, true);
 *      apf.layout.loadXml(newLayout);
 *      lstLayouts.startRename();
 *    ">
 *    Add Layout
 *  </a:button>
 * ```
 *
 * @default_private
 */
 // @todo a __WITH_DOM_REPARENTING should be added which can remove many of the functions of this element.

apf.layout = {
    compile : function(oHtml){
        var l = this.layouts[oHtml.getAttribute("id")];
        if (!l) return false;

        var root = l.root.copy();//is there a point to copying?
        
        l.layout.compile(root);
        l.layout.reset();
    },

    removeAll : function(aData) {
        aData.children.length = null

        var htmlId = this.getHtmlId(aData.pHtml);
        if (!this.rules[htmlId])
            delete this.qlist[htmlId];
    },
    
    timer : null,
    qlist : {},
    dlist : [],
    $hasQueue : false,
    
    queue : function(oHtml, obj, compile, q){
        if (!q) {
            this.$hasQueue = true;
            q = this.qlist;
        }
        
        var id;
        if (!(id = this.getHtmlId(oHtml)))
            id = apf.setUniqueHtmlId(oHtml);
            
        if (q[id]) {
            if (obj)
                q[id][2].push(obj);
            if (compile)
                q[id][1] = compile;
            return;
        }

        q[id] = [oHtml, compile, [obj]];

        if (!this.timer)
            this.timer = apf.setZeroTimeout(function(){
                apf.layout.processQueue();
            });
    },

    processQueue : function(){
        var i, id, l, qItem, list;

        for (i = 0; i < this.dlist.length; i++) {
            if (this.dlist[i].hidden)
                this.dlist[i].hide();
            else
                this.dlist[i].show();
        }

        do {
            var newq = {};
            var qlist = this.qlist;
            this.qlist = {};
            
            this.$hasQueue = false;
            
            for (id in qlist) {
                qItem = qlist[id];
    
                if (qItem[1])
                    apf.layout.compileAlignment(qItem[1]);
    
                list = qItem[2];
                for (i = 0, l = list.length; i < l; i++) {
                    if (list[i]) {
                        if (list[i].$amlDestroyed)
                            continue;
                        //if (list[i].$amlLoaded)
                            list[i].$updateLayout();
                        /*else
                            this.queue(qItem[0], list[i], null, newq);*/
                    }
                }
    
                apf.layout.activateRules(qItem[0]);
            }
        } while (this.$hasQueue);
        
        if (apf.hasSingleRszEvent)
            apf.layout.forceResize();

        this.dlist = [];
        
        apf.setZeroTimeout.clearTimeout(this.timer);
        this.timer = null;
    },
    
    rules     : {},
    onresize  : {},

    getHtmlId : function(oHtml){
        return oHtml.getAttribute ? oHtml.getAttribute("id") : 1;
    },

    /**
     * Adds layout rules to the resize event of the browser. Use this instead
     * of `"onresize"` events to add rules that specify determine the layout.
     * @param {HTMLElement} oHtml       The element that triggers the execution of the rules.
     * @param {String}      id          The identifier for the rules within the resize function of this element. Use this to easily update or remove the rules added.
     * @param {String}      rules       The JavaScript code that is executed when the html element resizes.
     * @param {Boolean}     [overwrite] Whether the rules are added to the resize function or overwrite the previous set rules with the specified id.
     */
    setRules : function(oHtml, id, rules, overwrite){
        if (!this.getHtmlId(oHtml))
            apf.setUniqueHtmlId(oHtml);
        if (!this.rules[this.getHtmlId(oHtml)])
            this.rules[this.getHtmlId(oHtml)] = {};

        var ruleset = this.rules[this.getHtmlId(oHtml)][id];
        if (!overwrite && ruleset) {
            this.rules[this.getHtmlId(oHtml)][id] = rules + "\n" + ruleset;
        }
        else
            this.rules[this.getHtmlId(oHtml)][id] = rules;
    },

    /**
     * Retrieves the rules set for the `"resize"` event of an HTML element specified by an identifier
     * @param {HTMLElement} oHtml       The element that triggers the execution of the rules.
     * @param {String}      id          The identifier for the rules within the resize function of this element.
     */
    getRules : function(oHtml, id){
        return id
            ? this.rules[this.getHtmlId(oHtml)][id]
            : this.rules[this.getHtmlId(oHtml)];
    },

    /**
     * Removes the rules set for the `"resize"` event of an html element specified by an identifier
     * @param {HTMLElement} oHtml       The element that triggers the execution of the rules.
     * @param {String}      id          The identifier for the rules within the resize function of this element.
     */
    removeRule : function(oHtml, id){
        var htmlId = this.getHtmlId(oHtml);
        if (!this.rules[htmlId])
            return;

        var ret = this.rules[htmlId][id] ||  false;
        delete this.rules[htmlId][id];

        var prop;
        for (prop in this.rules[htmlId]) {

        }
        if (!prop)
            delete this.rules[htmlId]

        if (apf.hasSingleRszEvent) {
            if (this.onresize[htmlId])
                this.onresize[htmlId] = null;
            else {
                var p = oHtml.parentNode;
                while (p && p.nodeType == 1 && !this.onresize[p.getAttribute("id")]) {
                    p = p.parentNode;
                }
    
                if (p && p.nodeType == 1) {
                    var x = this.onresize[p.getAttribute("id")];
                    if (x.children)
                        delete x.children[htmlId]
                }
            }
        }
        
        return ret;
    },

    /**
     * Activates the rules set for an HTML element
     * @param {HTMLElement} oHtml       The element that triggers the execution of the rules.
     * @param {Boolean} [no_exec]       
     */
    activateRules : function(oHtml, no_exec){
        if (!oHtml) { //!apf.hasSingleRszEvent &&
            var prop, obj;
            for(prop in this.rules) {
                obj = document.getElementById(prop);
                if (!obj || obj.onresize) // || this.onresize[prop]
                    continue;
                this.activateRules(obj);
            }

             if (apf.hasSingleRszEvent && apf.layout.$onresize)
                apf.layout.$onresize();
            return;
        }

        var rsz, id, rule, rules, strRules = [];
        if (!apf.hasSingleRszEvent) {
            rules = this.rules[this.getHtmlId(oHtml)];
            if (!rules){
                oHtml.onresize = null;
                return false;
            }

            for (id in rules) { //might need optimization using join()
                if (typeof rules[id] != "string")
                    continue;
                strRules.push(rules[id]);
            }

            //apf.console.info(strRules.join("\n"));
            rsz = apf.needsCssPx
                ? new Function(strRules.join("\n"))
                : new Function(strRules.join("\n").replace(/ \+ 'px'|try\{\}catch\(e\)\{\}\n/g,""))

            oHtml.onresize = rsz;
            if (!no_exec) 
                rsz();
        }
        else {
            var htmlId = this.getHtmlId(oHtml);
            rules = this.rules[htmlId];
            if (!rules){
                //@todo keep .children
                //delete this.onresize[htmlId];
                return false;
            }

            for (id in rules) { //might need optimization using join()
                if (typeof rules[id] != "string")
                    continue;
                strRules.push(rules[id]);
            }
            
            var p = oHtml.parentNode;
            while (p && p.nodeType == 1 && !this.onresize[p.getAttribute("id")]) {
                p = p.parentNode;
            }

            var f = new Function(strRules.join("\n"));//.replace(/try\{/g, "").replace(/}catch\(e\)\{\s*\}/g, "\n")
            if (this.onresize[htmlId])
                f.children = this.onresize[htmlId].children;
            
            if (p && p.nodeType == 1) {
                var x = this.onresize[p.getAttribute("id")];
                this.onresize[htmlId] = (x.children || (x.children = {}))[htmlId] = f;
            }
            else {
                this.onresize[htmlId] = f;
            }
            if (!no_exec)
                f();

            if (!apf.layout.$onresize) {
                var rsz = function(f){
                    //@todo fix this
                    try{
                        var c = [];
                        for (var name in f)
                            if (f[name])
                                c.unshift(f[name]);
                        for (var i = 0; i < c.length; i++){
                            c[i]();
                            if (c[i].children) {
                                rsz(c[i].children);
                            }
                        }
                    }
                    catch(e){
                        
                    }
                }
                
                apf.addListener(window, "resize", apf.layout.$onresize = function(){
                    if (apf.config.resize !== false) {
                        rsz(apf.layout.onresize);
                    }
                });
            }
        }
    },

    /**
     * Forces calling the resize rules for an HTML element
     * @param {HTMLElement} oHtml  The element for which the rules are executed.
     */
    forceResize : function(oHtml){
        if (apf.hasSingleRszEvent)
            return apf.layout.$onresize && apf.layout.$onresize();

        /* @todo this should be done recursive, old way for now
        apf.hasSingleRszEvent
            ? this.onresize[this.getHtmlId(oHtml)]
            :
        */

        var rsz = oHtml.onresize;
        if (rsz)
            rsz();

        var els = oHtml.getElementsByTagName("*");
        for (var i = 0, l = els.length; i < l; i++) {
            if (els[i] && els[i].onresize)
                els[i].onresize();
        }
    },

    paused : {},

    /**
     * Temporarily disables the resize rules for the HTML element.
     * @param {HTMLElement} oHtml  The element for which the rules are paused.
     * @param {Function}    func   The resize code that is used temporarily for resize of the HTML element.
     */
    pause  : function(oHtml, replaceFunc){
        if (apf.hasSingleRszEvent) {
            var htmlId = this.getHtmlId(oHtml);
            this.paused[htmlId] = this.onresize[htmlId] || true;

            if (replaceFunc) {
                this.onresize[htmlId] = replaceFunc;
                this.onresize[htmlId].children = this.paused[htmlId].children;
                replaceFunc();
            }
            else
                delete this.onresize[htmlId];
        }
        else {
            this.paused[this.getHtmlId(oHtml)] = oHtml.onresize || true;

            if (replaceFunc) {
                oHtml.onresize = replaceFunc;
                replaceFunc();
            }
            else
                oHtml.onresize = null;
        }
    },

    /**
     * Enables paused resize rules for the HTML element
     * @param {HTMLElement} oHtml  The element for which the rules were paused.
     */
    play : function(oHtml){
        if (!this.paused[this.getHtmlId(oHtml)])
            return;

        if (apf.hasSingleRszEvent) {
            var htmlId = this.getHtmlId(oHtml);
            var oldFunc = this.paused[htmlId];
            if (typeof oldFunc == "function") {
                this.onresize[htmlId] = oldFunc;
                //oldFunc();
            }
            else
                delete this.onresize[htmlId];

            if (apf.layout.$onresize)
                apf.layout.$onresize();

            this.paused[this.getHtmlId(oHtml)] = null;
        }
        else {
            var oldFunc = this.paused[this.getHtmlId(oHtml)];
            if (typeof oldFunc == "function") {
                oHtml.onresize = oldFunc;
                oldFunc();
            }
            else
                oHtml.onresize = null;

            this.paused[this.getHtmlId(oHtml)] = null;
        }
    }
};


/**
 * @private
 */
apf.getWindowWidth = function(){
    return apf.isIE ? document.documentElement.offsetWidth - apf.windowHorBorder : window.innerWidth;
};
/**
 * @private
 */
apf.getWindowHeight = function(){
    return apf.isIE ? document.documentElement.offsetHeight - apf.windowVerBorder : window.innerHeight;
};



/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



// Only add setZeroTimeout to the window object, and hide everything
// else in a closure.
apf.setZeroTimeout = !window.postMessage
  ? (function() {
        function setZeroTimeout() {
            return $setTimeout.apply(null, arguments);
        }
        setZeroTimeout.clearTimeout = function() {
             return clearTimeout.apply(null, arguments);
        };
        return setZeroTimeout;
    })()
  : (function() {
        var timeouts = [];
        var messageName = "zero-timeout-message";

        // Like setTimeout, but only takes a function argument.  There's
        // no time argument (always zero) and no arguments (you have to
        // use a closure).
        function setZeroTimeout(fn) {
            var id = timeouts.push(fn);
            window.postMessage(messageName, "*");
            return id;
        }
        
        setZeroTimeout.clearTimeout = function(id){
            timeouts[id] = null;
        }

        function handleMessage(e) {
            if (!e) e = event;
            if (e.source == window && e.data == messageName) {
                apf.stopPropagation(e);
                if (timeouts.length > 0 && (t = timeouts.shift()))
                    t();
            }
        }

        apf.addListener(window, "message", handleMessage, true);

        // Add the one thing we want added to the window object.
        return setZeroTimeout;
    })();




/*
 *
 */
apf.queue = {
    //@todo apf3.0
    q : {},

    timer : null,
    add : function(id, f){
        this.q[id] = f;
        if (!this.timer)
            
            this.timer = apf.setZeroTimeout(function(){
                apf.queue.empty();
            });
            
    },

    remove : function(id){
        delete this.q[id];
    },

    empty : function(prop){
        
        apf.setZeroTimeout.clearTimeout(this.timer);
        
        this.timer = null;

        
        if (apf.layout && apf.layout.$hasQueue)
            apf.layout.processQueue();
        
        
        if (apf.xmldb && apf.xmldb.$hasQueue)
            apf.xmldb.notifyQueued();
        

        var q  = this.q;
        this.q = {};
        for (var prop in q){
            var f = q[prop];
            if (f) {
                delete q[prop];
                f();
            }
        }
    }
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * This abstraction is using for resizing block elements. Resizing is allowed
 * with square elements in vertical, horizontal or both planes. Symmetric
 * resizing is possible with SHIFT button.
 * 
 * @private
 * @default_private
 * @constructor
 * 
 * @author      Lukasz Lipinski
 * @version     %I%, %G%
 * @since       1.0
 * 
 */

apf.resize = function() {
    /** 
     *     {Boolean} scalex       resizing in horizontal plane, default is true
     *         Possible values:
     *         true   resizing in horizontal plane is allowed
     *         false  resizing in horizontal plane is not allowed
     *     {Boolean} scaley       resizing in vertical plane, default is true
     *         Possible values:
     *         true   resizing in vertical plane is allowed
     *         false  resizing in vertical plane is not allowed
     *     {Boolean} scaleratio   resizing in horizontal or vertical plane only is not allowed. Resizing in two dimensions plane at the same time is allowed.
     *         Possible values:
     *         true   resizing in two dimensions plane at the same time is allowed
     *         false  Resizing in two dimensions plane at the same time is not allowed
     *     {Number}  dwidth       the minimal horizontal size of Block element, default is 56 pixels
     *     {Number}  dheight      the minimal vertical size of Block element, default is 56 pixels
     */
    this.scales = {
        scalex    : false,
        scaley    : false,
        scaleratio: false,
        dwidth    : 0,
        dheight   : 0,
        snap      : false,
        gridW     : 48,
        gridH     : 48
    };

    /*
     * html representation of resized block element
     */
    this.htmlElement;

    /**
     * store object representations of inputs elements
     */
    var squares = [];

    this.init = function() {
        squares = [
            new apf.resize.square("top",    "left",   this),
            new apf.resize.square("top",    "middle", this),
            new apf.resize.square("top",    "right",  this),
            new apf.resize.square("middle", "left",   this),
            new apf.resize.square("middle", "right",  this),
            new apf.resize.square("bottom", "left",   this),
            new apf.resize.square("bottom", "middle", this),
            new apf.resize.square("bottom", "right",  this)];
    };
    
    /**
     * Links block element with resize feature
     * 
     * @param {HTMLElement}   oHtml    html representation of block element
     * @param {Object}        scales   blocks scale settings
     */
    this.grab = function(oHtml, scales) {
        this.htmlElement = oHtml;
        this.scales = scales;

        if (!squares.length)
            this.init();
        this.show();
    };

    /**
     * Hides all block squares
     */
    this.hide = function() {
        for (var i = 0, l = squares.length; i < l; i++) {
            squares[i].visible = false;
            squares[i].repaint();
        }
    };

    /**
     * Shows all block squares
     */
    this.show = function() {
        var sx   = this.scales.scalex;
        var sy   = this.scales.scaley;
        var sr   = this.scales.scaleratio;

        for (var i = 0, l = squares.length, s; i < l; i++) {
            s = squares[i];
            s.visible = sx && sy
                ? true
                : (sy && !sx
                    ? (s.posX == "middle"
                        ? true
                        : false)
                    : (sx && !sy
                        ? (s.posY == "middle"
                            ? true
                            : false)
                        : (sr
                            ? ((s.posY == "top" || s.posY == "bottom")
                              && s.posX !== "middle"
                                ? true
                                : false)
                            : false)));
            
            s.repaint();
        }
    };

    /**
     * Destroys all block squares
     */
    this.destroy = function(){
        for (var i = 0; i < squares.length; i++) {
            squares[i].destroy();
        }
    };
};

/*
 * Creates html and object representation for square element. Square is used for
 * resizing block elements.
 * 
 * @param {String}   posY        square vertical align relative to resized block element
 *     Possible values:
 *     top      square is on top of resized block element
 *     middle   square is in the middle of the resized block element
 *     bottom   square is on the bottom of resized block element
 * @param {String}   posX        square vertical align relative to resized block element
 *     Possible values:
 *     left     square is on the left of resized block element
 *     middle   square is in the middle of the resized block element
 *     right    square is on the right of resized block element
 * @param {Object}   objResize   object of resize class
 * @constructor
 */
apf.resize.square = function(posY, posX, objResize) {
    /*
     * Square visibility
     */
    this.visible  = true;
    /*
     * square vertical align relative to resized block element
     */
    this.posX     = posX;
    /*
     * square vertical align relative to resized block element
     */
    this.posY     = posY;

    var margin = 0;
    var _self  = this;

    /*
     * html represenation of square element
     */
    this.htmlElement = objResize.htmlElement.parentNode.appendChild(document.createElement('div'));
    apf.setStyleClass(this.htmlElement, "square");

    /*
     * Repaints square
     */
    this.repaint = function() {
        if (this.visible) {
            var block = objResize.htmlElement;
            this.htmlElement.style.display = "block";

            var bw = parseInt(block.style.width) + apf.getDiff(block)[0];
            var bh = parseInt(block.style.height) + apf.getDiff(block)[1];
            var bt = parseInt(block.style.top);
            var bl = parseInt(block.style.left);

            var sw = this.htmlElement.offsetWidth;
            var sh = this.htmlElement.offsetHeight;

            var t = posY == "top"
                ? bt - margin - sh
                : posY == "middle"
                    ? bt + bh/2 - sh/2
                    : bt + bh + margin;
            var l = posX == "left"
                ? bl - margin - sw
                : posX == "middle"
                    ? bl + bw/2 - sw/2
                    : bl + bw + margin;

            var c = (posY == "middle" 
                ? "w-resize"
                : (posX == "middle"
                     ? "n-resize"
                     : (posY + posX == "topleft"
                       || posY + posX == "bottomright") 
                         ? "nw-resize" 
                         : "ne-resize"));

            this.htmlElement.style.top    = (t - 1) + "px";
            this.htmlElement.style.left   = (l - 1) + "px";
            this.htmlElement.style.cursor = c;
        }
        else {
            //IE bug
            var sw = this.htmlElement.offsetWidth;
            this.htmlElement.style.display = 'none';
        }
    };

    this.destroy = function(){
        apf.destroyHtmlNode(this.htmlElement);
    };

    /* Events */
    this.htmlElement.onmouseover = function(e) {
        apf.setStyleClass(_self.htmlElement, "squareHover");
    };

    this.htmlElement.onmouseout = function(e) {
        apf.setStyleClass(_self.htmlElement, "", ["squareHover"]);
    };

    this.htmlElement.onmousedown = function(e) {
        e = (e || event);

        var block = objResize.htmlElement,

            sx = e.clientX,
            sy = e.clientY,

            pt = block.parentNode.offsetTop,
            pl = block.parentNode.offsetLeft,

            dw = objResize.scales.dwidth,
            dh = objResize.scales.dheight,
            
            snap = objResize.scales.snap,
            gridH = objResize.scales.gridH,
            gridW = objResize.scales.gridW,

            objBlock = apf.flow.isBlock(block),
            r = objBlock.other.ratio,

            posX = _self.posX,
            posY = _self.posY,

            width, height, top, left, dx, dy,
            prev_w, prev_h,

            l = parseInt(block.style.left),
            t = parseInt(block.style.top),
            w = parseInt(block.style.width),
            h = parseInt(block.style.height),
            resized = false;
            
        objResize.onresizedone(w, h, t, l);

        if (e.preventDefault) {
            e.preventDefault();
        }

        document.onmousemove = function(e) {
            e = (e || event);

            dx = e.clientX - sx;
            dy = e.clientY - sy;
            var shiftKey = e.shiftKey,
                proportion = r;

            if (shiftKey) {
                if (posX == "right" && posY == "bottom") {
                    width  = w + dx;
                    height = width/proportion;
                    left   = l;
                    top    = t;
                }
                else if (posX == "right" && posY == "top") {
                    width  = w + dx;
                    height = width/proportion;
                    left   = l;
                    top    = t - dx/proportion;
                }
                else if (posX == "left" && posY == "bottom") {
                    width  = w - dx;
                    height = width/proportion;
                    left   = l + dx;
                    top    = t;
                }
                else if (posX == "left" && posY == "top") {
                    width  = w - dx;
                    height = width/proportion;
                    left   = l + dx;
                    top    = t + dx/proportion;
                }

                /* Keep minimal size */
                if(width >= dw && height >= dh) {
                    width  = prev_w = Math.max(dw, width);
                    height = prev_h = Math.max(dh, height);
                }
                else {
                    width  = prev_w;
                    height = prev_h;
                    return false;
                }
            }
            else {
                width = posX == "right"
                    ? w + dx
                    : (posX == "left"
                        ? w - dx
                        : w);
                height = posY == "bottom"
                    ? h + dy
                    : (posY == "top"
                        ? h - dy
                        : h);
                left = posX == "right"
                    ? l
                    : (posX == "left"
                        ? Math.min(l + w - dw, l + dx)
                        : l);
                top = posY == "bottom"
                    ? t
                    : (posY == "top"
                        ? Math.min(t + h - dh, t + dy)
                        : t);

                /* Keep minimal size */
                width = Math.max(dw, width);
                height = Math.max(dh, height);
            }

            if (snap) {
                left   = Math.floor(left / gridW) * gridW;
                top    = Math.floor(top / gridH) * gridH;
                width  = Math.ceil(width / gridW) * gridW;
                height = Math.ceil(height / gridH) * gridH;
            }

            if (objResize.onresize) {
                objResize.onresize(block, top, left, width, height);
            }

            objResize.show();
            
            resized = true;
        };

        document.onmouseup = function(e) {
            document.onmousemove = null;
            if (objResize.onresizedone && resized) {
                objResize.onresizedone(width, height, top, left);
                objBlock.other.ratio = width / height;
                resized = false;
            }
        };
    };
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * 
 * Controls the skinning modifications for AML.
 *
 * @private
 */
apf.skins = {
    skins  : {},
    css    : [],
    // @TODO Doc these ?
    events : ["onmousemove", "onmousedown", "onmouseup", "onmouseout",
        "onclick", "ondragcopy", "ondragstart", "ondblclick"],

    /* ***********
     Init
     ************/
    Init: function(xmlNode, refNode, path){
        /*
         get data from refNode || xmlNode
         - name
         - icon-path
         - media-path

         all paths of the xmlNode are relative to the src attribute of refNode
         all paths of the refNode are relative to the index.html
         images/ is replaced if there is a refNode to the relative path from index to the skin + /images/
         */
        var name      = (refNode ? refNode.getAttribute("id") : null)
            || xmlNode.getAttribute("id");
        var base      = (refNode ? (refNode.getAttribute("src") || "").match(/\//) || path : "")
            ? (path || refNode.getAttribute("src")).replace(/\/[^\/]*$/, "") + "/"
            : ""; //@todo make this absolute?

        var mediaPath = null, iconPath = null;
        mediaPath = xmlNode.getAttribute("media-path");
        if (mediaPath !== null)
            mediaPath = apf.getAbsolutePath(base || apf.hostPath, mediaPath);
        else if (refNode) {
            mediaPath = refNode.getAttribute("media-path");
            if (mediaPath !== null)
                mediaPath = apf.getAbsolutePath(apf.hostPath, mediaPath);
            else
                mediaPath = apf.getAbsolutePath(base || apf.hostPath, "images/");
        }
        
        iconPath = xmlNode.getAttribute("icon-path");
        if (iconPath !== null)
            iconPath = apf.getAbsolutePath(base || apf.hostPath, iconPath);
        else if (refNode) {
            iconPath = refNode.getAttribute("icon-path");
            if (iconPath !== null)
                iconPath = apf.getAbsolutePath(apf.hostPath, iconPath);
            else
                iconPath = apf.getAbsolutePath(base || apf.hostPath, "icons/");
        }
        
        if (!name)
            name = "default";

        if (xmlNode.getAttribute("id"))
            document.body.className += " " + xmlNode.getAttribute("id");

        var names = name.split("|");
        name = names[0];

        if (!this.skins[name] || name == "default") {
            this.skins[name] = {
                base     : base,
                name     : name,
                iconPath : iconPath,
                mediaPath: mediaPath,
                templates: {},
                originals: {},
                xml      : xmlNode
            }
            
            if (names.length > 1) {
                for (var i = 0; i < names.length; i++)
                    this.skins[names[i]] = this.skins[name];
            }
        }
        
        if (!this.skins["default"] && this.$first == refNode)
            this.skins["default"] = this.skins[name];

        var nodes = xmlNode.childNodes;
        for (var i = nodes.length - 1; i >= 0; i--) {
            if (nodes[i].nodeType != 1)
                continue;

            //this.templates[nodes[i].tagName] = nodes[i];
            this.skins[name].templates[nodes[i].getAttribute("name")] = nodes[i];
            if (nodes[i].ownerDocument)
                this.importSkinDef(nodes[i], base, name);
        }

        this.purgeCss(mediaPath, iconPath);
        
        if (this.queue[name]) {
            for (var prop in this.queue[name]) {
                this.queue[name][prop]();
            }
        }
    },

    /**
     * Loads a stylesheet from a URL.
     * @param {String}    filename  The url to load the stylesheet from
     * @param {String}    [title]  Title of the stylesheet to load
     * @method loadStylesheet
     */
    loadStylesheet: function(filename, title){
        var o;
        with (o = document.getElementsByTagName("head")[0].appendChild(document.createElement("LINK"))) {
            rel   = "stylesheet";
            type  = "text/css";
            href  = filename;
            title = title;
        }

        return o;
    },

    /* ***********
     Import
     ************/
    importSkinDef: function(xmlNode, basepath, name){
        var i, l, nodes = $xmlns(xmlNode, "style", apf.ns.aml), tnode, node;
        for (i = 0, l = nodes.length; i < l; i++) {
            node = nodes[i];

            if (node.getAttribute("src"))
                this.loadStylesheet(apf.getAbsolutePath(basepath, node.getAttribute("src")));
            else {
                var test = true;
                if (node.getAttribute("condition")) {
                    try {
                        test = eval(node.getAttribute("condition"));
                    }
                    catch (e) {
                        test = false;
                    }
                }

                if (test) {
                    //#-ifndef __PROCESSED
                    tnode = node.firstChild;
                    while (tnode) {
                        this.css.push(tnode.nodeValue);
                        tnode = tnode.nextSibling;
                    }
                    /*#-else
                    this.css.push(nodes[i].firstChild.nodeValue);
                    #-endif*/
                }
            }
        }

        nodes = $xmlns(xmlNode, "alias", apf.ns.apf);
        var t = this.skins[name].templates;
        for (i = 0; i < nodes.length; i++) {
            if (!nodes[i].firstChild)
                continue;
            t[nodes[i].firstChild.nodeValue.toLowerCase()] = xmlNode;
        }
    },

    loadedCss : "",
    purgeCss: function(imagepath, iconpath){
        if (!this.css.length)
            return;

        var cssString = this.css.join("\n").replace(/images\//g, imagepath).replace(/icons\//g, iconpath);
        apf.importCssString(cssString);

        

        this.css = [];
    },

    loadCssInWindow : function(skinName, win, imagepath, iconpath){
        this.css = [];
        var name = skinName.split(":");
        var skin = this.skins[name[0]];
        var template = skin.templates[name[1]];
        this.importSkinDef(template, skin.base, skin.name);
        var cssString = this.css.join("\n").replace(/images\//g, imagepath).replace(/icons\//g, iconpath);
        apf.importCssString(cssString);

        this.css = [];
    },

    /* ***********
     Retrieve
     ************/
    setSkinPaths: function(skinName, amlNode){
        skinName = skinName.split(":");
        var name = skinName[0];
        var type = skinName[1];

        

        amlNode.iconPath  = this.skins[name].iconPath;
        amlNode.mediaPath = this.skins[name].mediaPath;
    },

    getTemplate: function(skinName, noError){
        skinName = skinName.split(":");
        var name = skinName[0];
        var type = skinName[1];

        if (!this.skins[name]) {
            if (noError)
                return false;
            
            
            
            return false;
        }

        if (!this.skins[name].templates[type])
            return false;

        var skin      = this.skins[name].templates[type];
        var originals = this.skins[name].originals[type];
        if (!originals) {
            originals = this.skins[name].originals[type] = {};

            

            var nodes = $xmlns(skin, "presentation", apf.ns.aml)[0].childNodes;
            for (var i = 0; i < nodes.length; i++) {
                if (nodes[i].nodeType != 1) continue;
                originals[nodes[i].baseName || nodes[i][apf.TAGNAME]] = nodes[i];
            }
        }

        /*for (var item in originals) {
            pNodes[item] = originals[item];
        }*/

        return originals;
    },

    getCssString : function(skinName){
        return apf.queryValue($xmlns(this.skins[skinName.split(":")[0]].xml,
            "style", apf.ns.aml)[0], "text()");
    },

    
    changeSkinset : function(value){
        var node = apf.document.documentElement;
        while (node) {
            if (node && node.nodeFunc == apf.NODE_VISIBLE
              && node.hasFeature(apf.__PRESENTATION__) && !node.skinset) {
                node.$propHandlers["skinset"].call(node, value);//$forceSkinChange
                node.skinset = null;
            }

            //Walk tree
            if (node.firstChild || node.nextSibling) {
                node = node.firstChild || node.nextSibling;
            }
            else {
                do {
                    node = node.parentNode;
                } while (node && !node.nextSibling)

                if (node)
                    node = node.nextSibling;
            }
        }
    },
    
    
    queue : {},
    waitForSkin : function(skinset, id, callback){
        if (this.skins[skinset])
            return;
        
        (this.queue[skinset] || (this.queue[skinset] = {}))[id] = callback;
        return true;
    },

    

    setIcon : function(oHtml, strQuery, iconPath){
        if (!strQuery) {
            oHtml.style.backgroundImage = "";
            return;
        }

        if (oHtml.tagName.toLowerCase() == "img") {
            oHtml.setAttribute("src", strQuery
                ? (iconPath || "") + strQuery
                : "");
            return;
        }

        

        //Assuming image url
        {
            

            oHtml.style.backgroundImage = "url(" + (iconPath || "")
                + strQuery + ")";
        }
    }
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * Object handling sorting in a similar way as xslt.
 *
 * @constructor
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 *
 * @private
 */
apf.Sort = function(xmlNode){
    var settings = {};
    
    //use this function to parse the each node
    this.parseXml = function(xmlNode, clear){
        if (clear) settings = {};

        settings.order     = xmlNode.order;
        settings.getValue  = xmlNode.csort || xmlNode.$compile("sort");
        settings.getNodes  = self[xmlNode["nodes-method"]];

        settings.ascending = (settings.order || "").indexOf("desc") == -1;
        settings.order     = null;

        if (xmlNode["data-type"])
            settings.method = sort_methods[xmlNode["data-type"]];
        else if (xmlNode["sort-method"]) {
            settings.method = self[xmlNode["sort-method"]];
            
            
        }
        else
            settings.method = sort_methods["alpha"];
        
        var str = xmlNode["date-format"];
        if (str) {
            settings.sort_dateFmtStr = str;
            settings.method = sort_methods["date"];
            var result = str.match(/(D+|Y+|M+|h+|m+|s+)/g);
            if (result) {
                for (var pos = {}, i = 0; i < result.length; i++) 
                    pos[result[i].substr(0, 1)] = i + 1;
                settings.dateFormat = new RegExp(str.replace(/([^\sDYMhms])/g, '\\$1')
                    .replace(/YYYY/, "(\\d\\d\\d\\d)")
                    .replace(/(DD|YY|MM|hh|mm|ss)/g, "(\\d\\d)"));
                settings.dateReplace = "$" + pos["M"] + "/$" + pos["D"] + "/$" + pos["Y"];
                if (pos["h"]) 
                    settings.dateReplace += " $" + pos["h"] + ":$" + pos["m"] + ":$" + pos["s"];
            }
        }
    };
    
    this.set = function(struct, clear){
        if (clear) settings = {};
        
        apf.extend(settings, struct);

        if (settings.ascending == undefined)
            settings.ascending = struct.order 
                ? struct.order.indexOf("desc") == -1
                : true;
        
        settings.order = null;
        
        if (struct["type"]) 
            settings.method = sort_methods[struct["type"]];
        else if (struct["method"])
            settings.method = self[struct["method"]];
        else if (!settings.method) 
            settings.method = sort_methods["alpha"];
        
        if (struct.format) {
            settings.sort_dateFmtStr = struct.format;
            //settings.method = sort_methods["date"];
            var result = str.match(/(D+|Y+|M+|h+|m+|s+)/g);
            if (result) {
                for (var pos = {}, i = 0; i < result.length; i++) 
                    pos[result[i].substr(0, 1)] = i + 1;
                settings.dateFormat = new RegExp(str.replace(/([^\sDYMhms])/g, '\\$1')
                    .replace(/YYYY/, "(\\d\\d\\d\\d)")
                    .replace(/(DD|YY|MM|hh|mm|ss)/g, "(\\d\\d)"));
                settings.dateReplace = "$" + pos["M"] + "/$" + pos["D"] + "/$" + pos["Y"];
                if (pos["h"]) 
                    settings.dateReplace += " $" + pos["h"] + ":$" + pos["m"] + ":$" + pos["s"];
            }
        }
        
        if (!settings.getValue) {
            settings.getValue = function(item){
                return apf.queryValue(item, settings.xpath);
            }
        }
    };
    
    this.get = function(){
        return apf.extend({}, settings);
    };
    
    //use this function in __xmlUpdate [this function isnt done yet]
    this.findSortSibling = function(pNode, xmlNode){
        var nodes = getNodes ? getNodes(pNode, xmlNode) : this.getTraverseNodes(pNode);
        
        for (var i = 0; i < nodes.length; i++) 
            if (!compare(xmlNode, nodes[i], true, sortSettings)) 
                return nodes[i];
        
        return null;
    };
    
    // Sorting methods for sort()
    var sort_intmask = ["", "0", "00", "000", "0000", "00000", "000000",
        "0000000", "00000000", "000000000", "0000000000", "00000000000",
        "000000000000", "0000000000000", "00000000000000"];
    var sort_methods = {
        "alpha" : function (n){
            return n.toString().toLowerCase()
        },

        "number" : function (t){
            if (!t) t = 0;
            return (t.length < sort_intmask.length
                ? sort_intmask[sort_intmask.length - t.length]
                : "") + t;
        },

        "date" : function (t, args){
            var sort_dateFormat = settings.dateFormat;
            var sort_dateReplace = settings.dateReplace;
            var sort_dateFmtStr = settings.sort_dateFmtStr;
            
            var d;//|| (args && sort_dateFmtStr != args[0])
            if (!sort_dateFormat) {
                d = new Date(t);
            }
            else if (sort_dateFmtStr == '*') 
                d = apf.date.getDateTime(t);
            else 
                d = (new Date(t.replace(sort_dateFormat, sort_dateReplace))).getTime();
            t = "" + d.getTime();//parseInt(d);
            if (t == "NaN") 
                t = "0";
            return (t.length < sort_intmask.length ? sort_intmask[sort_intmask.length
                - t.length] : "") + t;
        }
    };

    /*
     sort(xpath, sort_xpath, sort_alpha, boolDesc, from, len)
     jsort(n,f,p,ps,sm,desc,sp,ep)
     */
    //var order, xpath, type, method, getNodes, dateFormat, dateReplace, sort_dateFmtStr, getValue;
    this.apply = function(n, args, func, start, len){
        var sa = [], i = n.length;
        
        // build string-sortable list with sort method
        while (i--) {
            var v = settings.getValue(n[i]);
            if (n) 
                sa[sa.length] = {
                    toString: function(){
                        return this.v;
                    },
                    xmlNode : n[i],
                    v       : (settings.method || sort_methods.alpha)(v || "", args, n[i])
                };
        }
        
        // sort it
        sa.sort();
        
        //iterate like foreach
        var end = len ? Math.min(sa.length, start + len) : sa.length;
        if (!start) 
            start = 0;
        
        if (func) {
            if (settings.ascending) 
                for (i = start; i < end; i++) 
                    f(i, end, sa[i].xmlNode, sa[i].v);
            else 
                for (i = end - 1; i >= start; i--) 
                    f(end - i - 1, end, sa[i].xmlNode, sa[i].v);
        }
        else {
            //this could be optimized by reusing n... time it later
            var res = [];
            if (settings.ascending) 
                for (i = start; i < end; i++) 
                    res[res.length] = sa[i].xmlNode;
            else 
                for (i = end - 1; i >= start; i--) 
                    res[res.length] = sa[i].xmlNode;
            return res;
        }
    };
    
    if (xmlNode) 
        this.parseXml(xmlNode);
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * The library that is used for the animations inside elements.
 *
 * @class apf.tween
 *
 * @default_private
 */
apf.tween = (function(apf) {

var modules = {
        //Animation Modules
    left: function(oHtml, value){
        oHtml.style.left = value + PX;
    },
    right: function(oHtml, value){
        oHtml.style.left  = "";
        oHtml.style.right = value + PX;
    },
    top: function(oHtml, value){
        oHtml.style.top = value + PX;
    },
    bottom: function(oHtml, value){
        oHtml.style.top    = "";
        oHtml.style.bottom = value + PX;
    },
    width: function(oHtml, value, center){
        oHtml.style.width = value + PX;
    },
    height: function(oHtml, value, center){
        oHtml.style.height = value + PX;
    },
    scrollTop: function(oHtml, value, center){
        oHtml.scrollTop = value;
    },
    scrollLeft: function(oHtml, value, center){
        oHtml.scrollLeft = value;
    },
    paddingTop: function(oHtml, value, center){
        oHtml.style.paddingTop = value + "px";
    },
    "height-rsz": function(oHtml, value, center){
        oHtml.style.height = value + PX;
        if (apf.hasSingleResizeEvent && apf.layout.$onresize)
            apf.layout.$onresize();
    },
    mwidth: function(oHtml, value, info) {
        var diff = apf.getDiff(oHtml);
        oHtml.style.width = value + PX;
        oHtml.style.marginLeft = -1 * (value / 2 + (parseInt(apf.getStyle(oHtml,
            "borderLeftWidth")) || diff[0]/2) + (info.margin || 0)) + PX;
    },
    mheight: function(oHtml, value, info) {
        var diff = apf.getDiff(oHtml);
        oHtml.style.height = value + PX;
        oHtml.style.marginTop = (-1 * value / 2 - (parseInt(apf.getStyle(oHtml,
            "borderTopWidth")) || diff[1]/2) + (info.margin || 0)) + PX;
    },
    scrollwidth: function(oHtml, value){
        oHtml.style.width = value + PX;
        oHtml.scrollLeft  = oHtml.scrollWidth;
    },
    scrollheight_old: function(oHtml, value){
        try {
            oHtml.style.height = value + PX;
            oHtml.scrollTop    = oHtml.scrollHeight;
        }
        catch (e) {
            alert(value)
        }
    },
    scrollheight: function(oHtml, value, info){
        var diff = apf.getHeightDiff(oHtml),
            oInt = info.$int || oHtml;

        oHtml.style.height = Math.max((value + (info.diff || 0)), 0) + PX;
        oInt.scrollTop     = oInt.scrollHeight - oInt.offsetHeight - diff 
            + (info.diff || 0) - (apf.isGecko ? 16 : 0); //@todo where does this 16 come from??
    },
    scrolltop: function(oHtml, value){
        oHtml.style.height = value + PX;
        oHtml.style.top    = (-1 * value - 2) + PX;
        oHtml.scrollTop    = 0;//oHtml.scrollHeight - oHtml.offsetHeight;
    },
    clipright: function(oHtml, value, center){
        oHtml.style.clip       = "rect(auto, auto, auto, " + value + "px)";
        oHtml.style.marginLeft = (-1 * value) + PX;
    },
    fade: function(oHtml, value){
        if (!apf.supportOpacity && apf.hasStyleFilters)
            oHtml.style.filter  = value == 1 ? "" : "alpha(opacity=" + parseInt(value * 100) + ")";
        else
            oHtml.style.opacity = value;
    },
    bgcolor: function(oHtml, value){
        oHtml.style.backgroundColor = value;
    },
    textcolor: function(oHtml, value){
        oHtml.style.color = value;
    },
    htmlcss : function(oHtml, value, obj){
        if (apf.hasStyleFilters && obj.type == "filter")
            oHtml.style.filter = value == 1 ? "" : "progid:DXImageTransform.Microsoft.Alpha(opacity=" + value + ")";
        else
            oHtml.style[obj.type] = value + (obj.needsPx ? PX : "");
    },
    transformscale: function(oHtml, value, obj) {
        oHtml.style[obj.type] = SCALEA + parseFloat(value) + SCALEB;
    },
    transformrotate: function(oHtml, value, obj) {
        oHtml.style[obj.type] = ROTATEA + parseFloat(value) + ROTATEB;
    },
    transformvalscale: function(value) {
        return SCALEA + parseFloat(value) + SCALEB;
    },
    transformvalrotate: function(value) {
        return ROTATEA + parseFloat(value) + ROTATEB;
    }
};

var ID        = "id",
    PX        = "px",
    NUM       = "number",
    TRANSVAL  = "transformval",
    TRANSFORM = "transform",
    SCALE     = "scale",
    SCALEA    = "scale(",
    ROTATEA   = "rotate(",
    SCALEB    = ")",
    ROTATEB   = "deg)",
    CSSTIMING = ["linear", "ease-in", "ease-out", "ease", "ease-in-out", "cubic-bezier"],
    CSSPROPS  = {
        "left"        : "left",
        "right"       : "right",
        "top"         : "top",
        "bottom"      : "bottom",
        "width"       : "width",
        "height"      : "height",
        "scrollTop"   : false,
        "scrollLeft"  : false,
        "mwidth"      : false,
        "mheight"     : false,
        "scrollwidth" : false,
        "scrollheight": false,
        "fade"        : "opacity",
        "opacity"     : "opacity",
        "bgcolor"     : "background-color",
        "textcolor"   : "color",
        "transform"   : "transform"
    },
    __pow   = Math.pow,
    __round = Math.round,

    queue = {},

    current= null,

    setQueue = function(oHtml, stepFunction){
        var id = oHtml.getAttribute(ID);
        if (!id) {
            apf.setUniqueHtmlId(oHtml);
            id = oHtml.getAttribute(ID);
        }

        if (!queue[id])
            queue[id] = [];

        queue[id].push(stepFunction);
        if (queue[id].length == 1)
            stepFunction(0);
    },

    nextQueue = function(oHtml){
        var q = queue[oHtml.getAttribute(ID)];
        if (!q) return;

        q.shift(); //Remove current step function

        if (q.length)
            q[0](0);
    },

    clearQueue = function(oHtml, bStop){
        var q = queue[oHtml.getAttribute(ID)];
        if (!q) return;

        if (bStop && current && current.control)
            current.control.stop = true;
        q.length = 0;
    },

    purgeQueue = function(oHtml) {
        var id = oHtml.getAttribute(ID);
        if (!id) {
            apf.setUniqueHtmlId(oHtml);
            id = oHtml.getAttribute(ID);
        }

        for (var i in queue) {
            if (i == id)
                queue[i] = [];
        }
    },

     // @TODO Doc
    /**
     * Calculates all the steps of an animation between a
     * begin and end value based on three tween strategies
     *
     * @method calcSteps
     * @param func {Function}
     * @param fromValue {String}
     * @param fromValue {String}
     * @param nrOfSteps {Number}
     */
    calcSteps = function(func, fromValue, toValue, nrOfSteps){
        var i     = 0,
            l     = nrOfSteps - 1,
            steps = [fromValue];

        // backward compatibility...
        if (typeof func == NUM) {
            if (!func)
                func = apf.tween.linear;
            else if (func == 1)
                func = apf.tween.easeInCubic;
            else if (func == 2)
                func = apf.tween.easeOutCubic;
        }

        /*
        func should have the following signature:
        func(t, x_min, dx)
        where 0 <= t <= 1, dx = x_max - x_min

        easeInCubic: function(t, x_min, dx) {
            return dx * pow(t, 3) + x_min;
        }
        */
        for (i = 0; i < l; ++i)
            steps.push(func(i / nrOfSteps, fromValue, toValue - fromValue));
        steps.push(toValue);
        
        return steps;
    },

     // @TODO Doc
    /**
     * Calculates all the steps of an animation between a
     * begin and end value for colors
     *  
     * @method calcColorSteps   
     * @param animtype {Function}
     * @param fromValue {String}
     * @param fromValue {String}
     * @param nrOfSteps {Number}
     */
    calcColorSteps = function(animtype, fromValue, toValue, nrOfSteps){
        var d2, d1,
            c   = apf.color.colorshex,
            a   = parseInt((c[fromValue] || fromValue).slice(1), 16),
            b   = parseInt((c[toValue] || toValue).slice(1), 16),
            i   = 0,
            out = [];

        for (; i < nrOfSteps; i++){
            d1 = i / (nrOfSteps - 1), d2 = 1 - d1;
            out[out.length] = "#" + ("000000" +
                ((__round((a & 0xff) * d2 + (b & 0xff) * d1) & 0xff) |
                (__round((a & 0xff00) * d2 + (b & 0xff00) * d1) & 0xff00) |
                (__round((a & 0xff0000) * d2 + (b & 0xff0000) * d1) & 0xff0000)).toString(16)).slice(-6);
        }

        return out;
    },

     // @TODO Doc wtf is stop ?
    /**
     * Tweens a single property of a single element or HTML element from a
     * start to an end value.
     * 
     * #### Example
     * 
     * ```javascript
     *  apf.tween.single(myDiv, {
     *      type : "left",
     *      from : 10,
     *      to   : 100,
     *      anim : apf.tween.EASEIN
     *  });
     * ```
     *
     * #### Example
     * 
     * Multiple animations can be run after each other
     * by calling this function multiple times.
     * 
     * ```javascript
     *  apf.tween.single(myDiv, options).single(myDiv2, options2);
     * ```
     *
     * @method single
     * @param {DOMNode}  oHtml The object to animate.
     * @param {Object}   info  The animation settings. The following properties are available:
     *   - type ([[String]]): The property to be animated. These are predefined
     *                          property handlers and can be added by adding a
     *                          method to `apf.tween` with the name of the property
     *                          modifier. There are several handlers available.
     *      - `"left"`:            Sets the left position
     *      - `"right"`:           Sets the right position
     *      - `"top"`:            Sets the top position
     *      - `"bottom"`:          Sets the bottom position
     *      - `"width"` :          Sets the horizontal size
     *      - `"height"`:          Sets the vertical size
     *      - `"scrollTop"`:       Sets the scoll position
     *      - `"mwidth"` :         Sets the width and the margin-left to width/2
     *      - `"mheight"` :        Sets the height ant the margin-top to height/2
     *      - `"scrollwidth"`:     Sets the width an sets the scroll to the maximum size
     *      - `"scrollheight"`:    Sets the height an sets the scroll to the maximum size
     *      - `"scrolltop"` :      Sets the height and the top as the negative height value
     *      - `"fade"` :           Sets the opacity property
     *      - `"bgcolor"`:         Sets the background color
     *      - `"textcolor"`:       Sets the text color
     *   - from ([[Number]] or [[String]]): The start value of the animation
     *   - to ([[Number]] or [[String]]): The end value of the animation
     *   - [steps] ([[Number]]): The number of steps to divide the tween in
     *   - [interval] ([[Number]]): The time between each step
     *   - [anim] ([[Number]]): The distribution of change between the step over the entire animation.             
     *   - [color] ([[Boolean]]): Specifies whether the specified values are colors
     *   - [userdata] (`Mixed`): Any data you would like to have available in your callback methods
     *   - [onfinish] ([[Function]]): A function that is called at the end of the animation
     *   - [oneach] ([[Function]]): A function that is called at each step of the animation
     *   - [control] ([[Object]]): An object that can stop the animation at any point
     *     Methods:
     *     stop                 set on the object passed .
     */
    single = function(oHtml, info){
        info = apf.extend({steps: 10, interval: 5, anim: apf.tween.linear, control: {}}, info);
        info.steps    = Math.ceil(info.steps * apf.animSteps);
        info.interval = Math.ceil(info.interval * apf.animInterval);

        if (oHtml.nodeFunc > 100) {
            info.$int = oHtml.$int;
            oHtml     = oHtml.$ext;
        }
        try { //@TODO hack where currentStyle is still undefined
            if ("fixed|absolute|relative".indexOf(apf.getStyle(oHtml, "position")) == -1)
                oHtml.style.position = "relative";
        } catch(e){}
        
        var useCSSAnim  = (false && apf.supportCSSAnim && apf.supportCSSTransition && CSSPROPS[info.type]),
            isTransform = (info.type == TRANSFORM);

        info.method = useCSSAnim ? info.type : isTransform
            ? modules[TRANSFORM + (info.subType || SCALE)]
            : modules[info.type]
                ? modules[info.type]
                : (info.needsPx = needsPix[info.type] || false)
                    ? modules.htmlcss
                    : modules.htmlcss;

        

        if (useCSSAnim) {
            var type = CSSPROPS[info.type];
            if (type === false)
                return apf.tween;
            info.type = type || info.type;
            if (isTransform) {
                if (!info.subType)
                    info.subType = SCALE;
                info.type = apf.supportCSSAnim;
            }

            var transform = (isTransform)
                ? modules[TRANSVAL + (info.subType || SCALE)]
                : null;

            oHtml.style[info.type] = isTransform
                ? transform(info.from)
                : info.from + (needsPix[info.type] ? PX : "");
            $setTimeout(function() {
                oHtml.style[info.type] = isTransform
                    ? transform(info.to)
                    : info.to + (needsPix[info.type] ? PX : "");
                oHtml.offsetTop; //force style recalc
                oHtml.style[apf.cssPrefix + "Transition"] = info.type + " " + ((info.steps
                    * info.interval) / 1000) + "s "
                    + CSSTIMING[info.anim || 0];
                var f = function() {
                    if (info.onfinish)
                        info.onfinish(oHtml, info.userdata);
                    oHtml.style[apf.cssPrefix + "Transition"] = "";
                    oHtml.removeEventListener(apf.cssAnimEvent, f);
                };
                oHtml.addEventListener(apf.cssAnimEvent, f);
            });
            return apf.tween;
        }

        if (info.control) {
            info.control.state = apf.tween.RUNNING;
            info.control.stop = function(){
                info.control.state = apf.tween.STOPPING;
                clearQueue(oHtml);
                if (info.onstop)
                    info.onstop(oHtml, info.userdata);
            }
        }

        var steps = info.color
                ? calcColorSteps(info.anim, info.from, info.to, info.steps)
                : calcSteps(info.anim, parseFloat(info.from), parseFloat(info.to), info.steps),
            stepFunction = function(step){
                if (info.control && info.control.state) {
                    info.control.state = apf.tween.STOPPED;
                    return;
                }
                
                current = info;

                if (info.onbeforeeach
                  && info.onbeforeeach(oHtml, info.userdata) === false)
                    return;

                try {
                   info.method(oHtml, steps[step], info);
                }
                catch (e) {}

                if (info.oneach)
                    info.oneach(oHtml, info.userdata);

                if (step < info.steps)
                    return $setTimeout(function(){stepFunction(step + 1)}, info.interval);

                current = null;
                if (info.control)
                    info.control.state = apf.tween.STOPPED;
                if (info.onfinish)
                    info.onfinish(oHtml, info.userdata);

                nextQueue(oHtml);
            };

        if (info.type.indexOf("scroll") > -1)
            purgeQueue(oHtml);
        setQueue(oHtml, stepFunction);

        return apf.tween;
    },

     // @TODO Doc wtf is stop
    /**
     * Tweens multiple properties of a single element or html element from a
     * start to an end value.
     * 
     * #### Example
     *
     * Here we are, animating both the left and width at the same time:
     *
     * ```javascript
     *  apf.tween.multi(myDiv, {
     *      anim   : apf.tween.EASEIN
     *      tweens : [{
     *          type : "left",
     *          from : 10,
     *          to   : 100,
     *      },
     *      {
     *          type : "width",
     *          from : 100,
     *          to   : 400,
     *      }]
     *  });
     * ````
     *
     * #### Example
     *
     * Multiple animations can be run after each other
     * by calling this function multiple times.
     *
     * ```javascript
     *  apf.tween.multi(myDiv, options).multi(myDiv2, options2);
     * ```
     *
     * @method multi
     * @param {DOMNode}  oHtml The object to animate.
     * @param {Object} info The settings of the animation. It contains the following properties:
     *   - [steps] ([[Number]]): The number of steps to divide the tween in
     *   - [interval] ([[Number]]): The time between each step
     *   - [anim] ([[Number]]): The distribution of change between the step over
     *                          the entire animation
     *   - [onfinish] ([[Function]]): A function that is called at the end of the animation
     *   - [oneach] ([[Function]]): A function that is called at each step of the animation
     *   - [oHtml] ([[HTMLElement]]): Another HTML element to animate.
     *   - [control] ([[Object]]): An object that can stop the animation at any point. It contains the following properties:
     *     - stop ([[Boolean]]): Specifies whether the animation should stop.
     *   - [tweens] ([[Array]]): A collection of simple objects specifying the single
     *                          value animations that are to be executed simultaneously.
     *                          (for the properties of these single tweens see the
     *                          [[apf.tween.single]] method).
     */
    multi = function(oHtml, info){ 
        info = apf.extend({steps: 10, interval: 5, anim: apf.tween.linear, control: {}}, info);
        info.steps    = Math.ceil(info.steps * apf.animSteps);
        info.interval = Math.ceil(info.interval * apf.animInterval);

        if (oHtml.nodeFunc > 100) {
            info.$int = oHtml.$int;
            oHtml = oHtml.$ext;
        }

        var animCSS, isTransform,
            useCSSAnim  = false && apf.supportCSSAnim && apf.supportCSSTransition,
            hasCSSAnims = false,
            cssDuration = ((info.steps * info.interval) / 1000),
            cssAnim     = CSSTIMING[info.anim || 0],
            steps       = [],
            stepsTo     = [],
            i           = 0,
            l           = info.tweens.length;

        for (; i < l; i++) {
            var data = info.tweens[i];

            if (data.oHtml && data.oHtml.nodeFunc > 100) {
                data.$int  = data.oHtml.$int;
                data.oHtml = data.oHtml.$ext;
            }

            animCSS     = (useCSSAnim && CSSPROPS[data.type]);
            isTransform = (data.type == TRANSFORM);
            if (isTransform) {
                if (!data.subType)
                    data.subType = SCALE;
                data.type = apf.supportCSSAnim;
            }

            data.method = animCSS
                ? data.type
                : isTransform
                    ? modules[TRANSFORM + (data.subType)]
                    : modules[data.type]
                        ? modules[data.type]
                        : (data.needsPx = needsPix[data.type] || false)
                            ? modules.htmlcss
                            : modules.htmlcss;


            

            if (animCSS) {
                var type = isTransform ? data.type : CSSPROPS[data.type];
                data.type = type || data.type;
                var transform = modules[TRANSVAL + (data.subType)]

                oHtml.style[data.type] = isTransform
                    ? transform(data.from)
                    : data.from + (needsPix[data.type] ? PX : "");
                stepsTo.push([data.type, isTransform
                    ? transform(data.to)
                    : data.to + (needsPix[data.type] ? PX : "")]);
                steps.push(data.type + " " + cssDuration + "s " + cssAnim + " 0");

                hasCSSAnims = true;
            }
            else {
                steps.push(data.color
                    ? calcColorSteps(info.anim, data.from, data.to, info.steps)
                    : calcSteps(info.anim, parseFloat(data.from), parseFloat(data.to), info.steps));
            }
        }

        if (hasCSSAnims) {
            oHtml.style[apf.cssPrefix + "Transition"] = steps.join(",");
            oHtml.offsetTop; //force style recalc
            var count = 0,
                func  = function() {
                    count++;
                    if (count == stepsTo.length) {
                        if (info.onfinish)
                            info.onfinish(oHtml, info.userdata);
                        oHtml.style[apf.cssPrefix + "Transition"] = "";
                        oHtml.removeEventListener(apf.cssAnimEvent, func);
                    }
                };
            oHtml.addEventListener(apf.cssAnimEvent, func, false);
            for (var k = 0, j = stepsTo.length; k < j; k++)
                oHtml.style[stepsTo[k][0]] = stepsTo[k][1];
            return apf.tween;
        }
        
        if (info.control) {
            info.control.state = apf.tween.RUNNING;
            info.control.stop = function(){
                if (info.control.state == apf.tween.STOPPED)
                    return;
                
                info.control.state = apf.tween.STOPPING;
                clearQueue(oHtml);
                if (info.onstop)
                    info.onstop(oHtml, info.userdata);
            }
        }

        var tweens       = info.tweens,
            stepFunction = function(step){
                if (info.control && info.control.state) {
                    info.control.state = apf.tween.STOPPED;
                    return;
                }
                
                current = info;

                try {
                    for (var i = 0; i < steps.length; i++) {
                        tweens[i].method(tweens[i].oHtml || oHtml,
                          steps[i][step], tweens[i]);
                    }
                } catch (e) {}

                if (info.oneach)
                    info.oneach(oHtml, info.userdata);

                if (step < info.steps)
                    return $setTimeout(function(){stepFunction(step + 1)}, info.interval);

                current = null;
                if (info.control)
                    info.control.state = apf.tween.STOPPED;
                if (info.onfinish)
                    info.onfinish(oHtml, info.userdata);

                nextQueue(oHtml);
            };

        setQueue(oHtml, stepFunction);

        return apf.tween;
    },

    /**
     * Tweens an element or HTML element from its current state to a CSS class.
     *
     * #### Example
     *
     * Multiple animations can be run after each other by calling this function
     * multiple times.
     * 
     * ```javascript
     *  apf.tween.css(myDiv, 'class1').multi(myDiv2, 'class2');
     * ```
     *
     * @method apf.tween.css
     * @param {DOMNode}  oHtml The object to animate.
     * @param {String} className The class name that defines the CSS properties to be set or removed.
     * @param {Object} info The settings of the animation. The following properties are available:
     *   Properties:
     *   - [steps] ([[Number]]): The number of steps to divide the tween in
     *   - [interval] ([[Number]]): The time between each step
     *   - [anim] ([[Number]]): The distribution of change between the step over the entire animation
     *   - [onfinish] ([[Function]]): A function that is called at the end of the animation
     *   - [oneach] ([[Function]]): A function that is called at each step of the animation
     *   - [control] ([[Object]]): An object that can stop the animation at any point. It contains the following property:
     *     - stop ([[Boolean]]): Specifies whether the animation should stop.
     * @param {Boolean} remove Specifies whether the class is set or removed from the element
     */
    css = function(oHtml, className, info, remove){
        (info = info || {}).tweens = [];

        if (oHtml.nodeFunc > 100)
            oHtml = oHtml.$ext;

        if (remove)
            apf.setStyleClass(oHtml, "", [className]);

        var resetAnim = function(remove, callback){
            if (remove)
                apf.setStyleClass(oHtml, "", [className]);
            else
                apf.setStyleClass(oHtml, className);

            //Reset CSS values
            for (var i = 0; i < info.tweens.length; i++){
                if (info.tweens[i].type == "filter")
                    continue;

                oHtml.style[info.tweens[i].type] = "";
            }

            if (callback)
                callback.apply(this, arguments);
        }

        var onfinish  = info.onfinish,
            onstop    = info.onstop;
        info.onfinish = function(){resetAnim(remove, onfinish);}
        info.onstop   = function(){resetAnim(!remove, onstop);}

        var result, newvalue, curvalue, j, isColor, style, rules, i,
            tweens = {};
        for (i = 0; i < document.styleSheets.length; i++) {
            rules = document.styleSheets[i][apf.styleSheetRules];
            if (!rules || !rules.length)
                continue;
            for (j = rules.length - 1; j >= 0; j--) {
                var rule = rules[j];

                if (!rule.style || !rule.selectorText.match("\." + className + "$"))
                    continue;

                for (style in rule.style) {
                    if (!rule.style[style] || cssProps.indexOf("|" + style + "|") == -1)
                        continue;

                    if (style == "filter") {
                        if (!rule.style[style].match(/opacity\=([\d\.]+)/))
                            continue;
                        newvalue = RegExp.$1;

                        result   = (apf.getStyleRecur(oHtml, style) || "")
                            .match(/opacity\=([\d\.]+)/);
                        curvalue = result ? RegExp.$1 : 100;
                        isColor  = false;

                        if (newvalue == curvalue) {
                            if (remove) curvalue = 100;
                            else newvalue = 100;
                        }
                    }
                    else {
                        newvalue = remove && oHtml.style[style] || rule.style[style];
                        if (remove) oHtml.style[style] = "";
                        curvalue = apf.getStyleRecur(oHtml, style);
                        isColor = style.match(/color/i) ? true : false;
                    }

                    tweens[style] = {
                        type    : style,
                        from    : (isColor ? String : parseFloat)(remove
                                    ? newvalue
                                    : curvalue),
                        to      : (isColor ? String : parseFloat)(remove
                                    ? curvalue
                                    : newvalue),
                        color   : isColor,
                        needsPx : needsPix[style.toLowerCase()] || false
                    };
                }
            }
        }

        for (var prop in tweens)
            info.tweens.push(tweens[prop]);

        if (remove)
            apf.setStyleClass(oHtml, className);

        return multi(oHtml, info);
    },

    cssRemove = function(oHtml, className, info){
        css(oHtml, className, info, true);
    },

    needsPix = {
        "left"        : true,
        "top"         : true,
        "bottom"      : true,
        "right"       : true,
        "width"       : true,
        "height"      : true,
        "fontSize"    : true,
        "lineHeight"  : true,
        "textIndent"  : true,
        "marginLeft"  : true,
        "marginTop"   : true,
        "marginRight" : true,
        "marginBottom": true
    },

    cssProps = "|backgroundColor|backgroundPosition|color|width|filter"
             + "|height|left|top|bottom|right|fontSize"
             + "|letterSpacing|lineHeight|textIndent|opacity"
             + "|paddingLeft|paddingTop|paddingRight|paddingBottom"
             + "|borderLeftWidth|borderTopWidth|borderRightWidth|borderBottomWidth"
             + "|borderLeftColor|borderTopColor|borderRightColor|borderBottomColor"
             + "|marginLeft|marginTop|marginRight|marginBottom"
             + "|transform|", // transforms are special and get special treatment
    cssTransforms = "|scale|rotate|";

return {
    single: single,
    multi: multi,
    css: css,
    cssRemove: cssRemove,
    clearQueue: clearQueue,
    addModule: function(name, func, force) {
        if (typeof name != "string" || typeof func != "function" || (modules[name] && !force))
            return this;
        modules[name] = func;
        return this;
    },
    /** Linear tweening method */
    NORMAL: 0,
    /** Ease-in tweening method */
    EASEIN: 1,
    /** Ease-out tweening method */
    EASEOUT: 2,
    
    RUNNING: 0,
    STOPPING: 1,
    STOPPED: 2,
    
    calcColorSteps: calcColorSteps,

    linear: function(t, x_min, dx) {
        return dx * t + x_min;
    },
    easeInQuad: function(t, x_min, dx) {
        return dx * __pow(t, 2) + x_min;
    },
    easeOutQuad: function(t, x_min, dx) {
        return -dx * t * (t - 2) + x_min;
    },
    easeInOutQuad: function(t, x_min, dx) {
        if ((t /= .5) < 1)
            return dx / 2 * t * t + x_min;
        return -dx / 2 * ((--t) * (t - 2) - 1) + x_min;
    },
    easeInCubic: function(t, x_min, dx) {
        return dx * __pow(t, 3) + x_min;
    },
    easeOutCubic: function(t, x_min, dx) {
        return dx * (__pow(t - 1, 3) + 1) + x_min;
    },
    easeInOutCubic: function(t, x_min, dx) {
        if ((t /= .5) < 1)
            return dx / 2 * __pow(t, 3) + x_min;
        return dx / 2 * (__pow(t - 2, 3) + 2) + x_min;
    },
    easeInQuart: function(t, x_min, dx) {
        return dx * __pow(t, 4) + x_min;
    },
    easeOutQuart: function(t, x_min, dx) {
        return -dx * (__pow(t - 1, 4) - 1) + x_min;
    },
    easeInOutQuart: function(t, x_min, dx) {
        if ((t /= .5) < 1)
            return dx / 2 * __pow(t, 4) + x_min;
        return -dx / 2 * (__pow(t - 2, 4) - 2) + x_min;
    },
    easeInQuint: function(t, x_min, dx) {
        return dx * __pow(t, 5) + x_min;
    },
    easeOutQuint: function(t, x_min, dx) {
        return dx * (__pow(t - 1, 5) + 1) + x_min;
    },
    easeInOutQuint: function(t, x_min, dx) {
        if ((t /= .5) < 1)
            return dx / 2 * __pow(t, 5) + x_min;
        return dx / 2 * (__pow(t - 2, 5) + 2) + x_min;
    },
    easeInSine: function(t, x_min, dx) {
        return -dx * Math.cos(t * (Math.PI / 2)) + dx + x_min;
    },
    easeOutSine: function(t, x_min, dx) {
        return dx * Math.sin(t * (Math.PI / 2)) + x_min;
    },
    easeInOutSine: function(t, x_min, dx) {
        return -dx / 2 * (Math.cos(Math.PI * t) - 1) + x_min;
    },
    easeInExpo: function(t, x_min, dx) {
        return (t == 0) ? x_min : dx * __pow(2, 10 * (t - 1)) + x_min;
    },
    easeOutExpo: function(t, x_min, dx) {
        return (t == 1) ? x_min + dx : dx * (-__pow(2, -10 * t) + 1) + x_min;
    },
    easeInOutExpo: function(t, x_min, dx) {
        if (t == 0)
            return x_min;
        if (t == 1)
            return x_min + dx;
        if ((t /= .5) < 1)
            return dx / 2 * __pow(2, 10 * (t - 1)) + x_min;
        return dx / 2 * (-__pow(2, -10 * --t) + 2) + x_min;
    },
    easeInCirc: function(t, x_min, dx) {
        return -dx * (Math.sqrt(1 - t * t) - 1) + x_min;
    },
    easeOutCirc: function(t, x_min, dx) {
        return dx * Math.sqrt(1 - (t -= 1) * t) + x_min;
    },
    easeInOutCirc: function(t, x_min, dx) {
        if ((t /= .5) < 1)
            return -dx / 2 * (Math.sqrt(1 - t * t) - 1) + x_min;
        return dx / 2 * (Math.sqrt(1 - (t -= 2) * t) + 1) + x_min;
    },
    easeInElastic: function(t, x_min, dx) {
        var s = 1.70158,
            p = .3,
            a = dx;
        if (t == 0)
            return x_min;
        if (t == 1)
            return x_min + dx;
        if (!a || a < Math.abs(dx)) {
            a = dx;
            s = p / 4;
        }
        else
            s = p / (2 * Math.PI) * Math.asin (dx / a);
        return -(a * __pow(2, 10 * (t -= 1)) * Math.sin((t * 1 - s) * (2 * Math.PI) / p)) + x_min;
    },
    easeOutElastic: function(t, x_min, dx) {
        var s = 1.70158,
            p = .3,
            a = dx;
        if (t == 0)
            return x_min;
        if (t == 1)
            return x_min + dx;
        if (a < Math.abs(dx)) {
            a = dx;
            s = p / 4;
        }
        else {
            s = p / (2 * Math.PI) * Math.asin(dx / a);
        }
        return a * __pow(2, -10 * t) * Math.sin((t - s) * (2 * Math.PI) / p) + dx + x_min;
    },
    easeInOutElastic: function(t, x_min, dx) {
        var s = 1.70158,
            p = 0,
            a = dx;
        if (t == 0)
            return x_min;
        if ((t / 2) == 2)
            return x_min + dx;
        if (!p)
            p = .3 * 1.5;
        if (a < Math.abs(dx)) {
            a = dx;
            s = p / 4;
        }
        else {
            s = p / (2 * Math.PI) * Math.asin(dx / a);
        }
        if (t < 1)
            return -.5 * (a * __pow(2, 10 * (t -= 1)) * Math.sin((t - s) * (2 * Math.PI) / p)) + x_min;
        return a * __pow(2, -10 * (t -= 1)) * Math.sin((t - s) * (2 * Math.PI) / p) * .5 + dx + x_min;
    },
    easeInBack: function(t, x_min, dx) {
        var s = 1.70158;
        return dx * __pow(t, 2) * ((s + 1) * t - s) + x_min;
    },
    easeOutBack: function(t, x_min, dx) {
        var s = 1.70158;
        return dx * ((t -= 1) * t * ((s + 1) * t + s) + 1) + x_min;
    },
    easeInOutBack: function(t, x_min, dx) {
        var s = 1.70158;
        if ((t / 2) < 1)
            return dx / 2 * (t * t * (((s *= (1.525)) + 1) * t - s)) + x_min;
        return dx / 2 * ((t -= 2) * t * (((s *= (1.525)) + 1) * t + s) + 2) + x_min;
    },
    easeInBounce: function(t, x_min, dx) {
        return dx - apf.tween.easeOutBounce(1 - t, 0, dx) + x_min;
    },
    easeOutBounce: function(t, x_min, dx) {
        if (t < (1 / 2.75))
            return dx * (7.5625 * t * t) + x_min;
        else if (t < (2 / 2.75))
            return dx * (7.5625 * (t -= (1.5 / 2.75)) * t + .75) + x_min;
        else if (t < (2.5 / 2.75))
            return dx * (7.5625 * (t -= (2.25 / 2.75)) * t + .9375) + x_min;
        else
            return dx * (7.5625 * (t -= (2.625 / 2.75)) * t + .984375) + x_min;
    },
    easeInOutBounce: function(t, x_min, dx) {
        if (t < 1 / 2)
            return apf.tween.easeInBounce(t * 2, 0, dx) * .5 + x_min;
        return apf.tween.easeOutBounce(t * 2 - 1, 0, dx) * .5 + dx * .5 + x_min;
    }
};

})(apf);












/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
 





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * The XML database object provides local storage for XML data. This object
 * routes all changes to the XML data to the data bound objects. It also
 * provides utility functions for XML handling.
 *
 * @class apf.xmldb
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 * @additional
 *
 * @default_private
 */
apf.xmldb = new (function(){
    var _self = this;

    this.xmlDocTag    = "a_doc";
    this.xmlIdTag     = "a_id";
    this.xmlListenTag = "a_listen";
    this.htmlIdTag    = "id";
    this.disableRDB   = false;

    this.$xmlDocLut   = [];
    this.$nodeCount   = {};

    var cleanRE       = /(?:a_doc|a_id|a_listen|a_loaded)=(?:"|')[^'"]+(?:"|')/g,
        whiteRE       = />[\s\n\r\t]+</g;

    /**
     * Clear XML document cache periodically when no model is referencing it
     */
    this.garbageCollect = function(){
        var xmlNode, cache = apf.xmldb.$xmlDocLut, docId, model;
        for (var i = 0, l = cache.length; i < l; i++) {
            xmlNode = cache[i];

            if (!xmlNode || xmlNode.nodeFunc)
                continue;

            docId = i;//xmlNode.getAttribute(apf.xmldb.xmlDocTag);
            model = apf.nameserver.get("model", docId);

            if (!model || model.data != xmlNode) {
                cache[i] = null;
            }
        }
    };

    this.$gcInterval = window.setInterval
        ? setInterval(function(){
              _self.garbageCollect();
          }, 60000)
        : null;

    /*
     * @private
     */
    this.getElementById = function(id, doc){
        if (!doc)
            doc = this.$xmlDocLut[id.split("\|")[0]];
        if (!doc)
            return false;

        return doc.selectSingleNode("descendant-or-self::node()[@"
            + this.xmlIdTag + "='" + id + "']");
    };

    /*
     * @private
     */
    this.getNode = function(htmlNode){
        if (!htmlNode || !htmlNode.getAttribute(this.htmlIdTag))
            return false;

        return this.getElementById(htmlNode.getAttribute(this.htmlIdTag)
            .split("\|", 2).join("|"));
    };

    /*
     * @private
     */
    this.getNodeById = function(id, doc){
        var q = id.split("\|");
        q.pop();
        return this.getElementById(q.join("|"), doc);//id.split("\|", 2).join("|")
    };

    /*
     * @private
     */
    this.getDocumentById = function(id){
        return this.$xmlDocLut[id];
    };

    /*
     * @private
     */
    this.getDocument = function(node){
        return this.$xmlDocLut[node.getAttribute(this.xmlIdTag).split("\|")[0]];
    };

    /**
     * @private
     */
    this.getID = function(xmlNode, o){
        return xmlNode.getAttribute(this.xmlIdTag) + "|" + o.$uniqueId;
    };

    /*
     * @private
     */
    this.getElement = function(parent, nr){
        var nodes = parent.childNodes;
        for (var j = 0, i = 0; i < nodes.length; i++) {
            if (nodes[i].nodeType != 1)
                continue;
            if (j++ == nr)
                return nodes[i];
        }
    };

    /**
     * Sets the model of an element.
     *
     * @param {apf.model} The model to be set
     *
     */
    this.setModel = function(model){
        
        apf.nameserver.register("model", model.data.ownerDocument
            .documentElement.getAttribute(this.xmlDocTag), model);
        
    };

    /**
     * Finds the model of an element.
     *
     * @param {XMLNode} xmlNode The {@link term.datanode data node} to find its model.
     *
     */
    this.findModel = function(xmlNode){
        
        return apf.nameserver.get("model", xmlNode.ownerDocument
            .documentElement.getAttribute(this.xmlDocTag));
        
    };

    /*
     * @private
     */
    this.getXmlId = function(xmlNode){
        return xmlNode.getAttribute(this.xmlIdTag) ||
          this.nodeConnect(apf.xmldb.getXmlDocId(xmlNode), xmlNode);
    };

    /**
     * Gets the HTML representation of an XML node for a certain element.
     *
     * @param {XMLNode} xmlNode  The {@link term.datanode data node} which is represented by the HTML element.
     * @param {apf.AmlNode} oComp    The element that has created the representation.
     * @return {DOMNode} The HTML node representing the XML node.
     */
    this.getHtmlNode = function(xmlNode, oComp){
        if (xmlNode && xmlNode.nodeType == 1 && xmlNode.getAttribute(this.xmlIdTag)) {
            return oComp.$findHtmlNode(xmlNode.getAttribute(this.xmlIdTag)
                + "|" + oComp.$uniqueId);
        }
        return null;
    };

    /**
     * Finds the HTML representation of an XML node for a certain element.
     *
     * @param {XMLNode} xmlNode  The {@link term.datanode data node} which is represented by the HTML element.
     * @param {apf.AmlNode} oComp    The element that has created the representation.
     * @return {DOMNode} The HTML node representing the XML node.
     */
    this.findHtmlNode = function(xmlNode, oComp){
        do {
            if (xmlNode.nodeType == 1 && xmlNode.getAttribute(this.xmlIdTag)) {
                return oComp.$findHtmlNode(xmlNode.getAttribute(this.xmlIdTag)
                    + "|" + oComp.$uniqueId);
            }
            if (xmlNode == oComp.xmlRoot)
                return null;

            xmlNode = xmlNode.parentNode;
        }
        while (xmlNode && xmlNode.nodeType != 9)

        return null;
    };

    /**
     * Finds the {@link term.datanode data node} that is represented by the HTML node.
     *
     * @param {DOMNode} htmlNode  The HTML node representing the an XML node.
     * @return {XMLNode} The {@link term.datanode data node} for which the HTML node is its representation.
     */
    this.findXmlNode = function(htmlNode){
        if (!htmlNode)
            return false;

        var id;
        while (htmlNode && htmlNode.nodeType == 1 && (
          htmlNode.tagName.toLowerCase() != "body" && !(id = htmlNode.getAttribute(this.htmlIdTag))
          || (id || (id = htmlNode.getAttribute(this.htmlIdTag))) && id.match(/^q/)
        )) {
            if (htmlNode.host && htmlNode.host.$ext == htmlNode)
                return htmlNode.host.xmlRoot;

            htmlNode = htmlNode.parentNode;
        }
        if (!htmlNode || htmlNode.nodeType != 1)
            return false;

        if (htmlNode.tagName.toLowerCase() == "body")
            return false;

        return this.getNode(htmlNode);
    };

    this.getXml = apf.getXml;

    /*
     * @private
     */
    this.nodeConnect = function(documentId, xmlNode, htmlNode, o){
        if (!this.$nodeCount[documentId])
            this.$nodeCount[documentId] = 0;

        var xmlId;
        xmlId = xmlNode.getAttribute(this.xmlIdTag)
          || xmlNode.setAttribute(this.xmlIdTag, (xmlId = documentId
               + "|" + ++this.$nodeCount[documentId])) || xmlId;

        if (!o)
            return xmlId;

        var htmlId = xmlId + "|" + o.$uniqueId;
        if (htmlNode)
            htmlNode.setAttribute(this.htmlIdTag, htmlId);

        return htmlId;
    };



    this.$listeners = [null];    
    // make sure that "0" is never a listener index    
    // @todo this is cleanup hell! Listeners should be completely rearchitected
    /*
     * @private
     * 
     */
    this.addNodeListener = function(xmlNode, o, uId){
        

        var id, listen = String(xmlNode.getAttribute(this.xmlListenTag) || "");
        //id || (id = String(o.$uniqueId));

        if (!uId) uId = String(o.$uniqueId);
        if (uId.charAt(0) == "p") {
            var sUId = uId.split("|");
            id = this.$listeners.push(function(args){
                //@todo apf3.0 should this be exactly like in class.js?
                //@todo optimize this to check the async flag: parsed[3] & 4

                var amlNode = apf.all[sUId[1]]; //It's possible the aml node dissapeared in this loop.
                if (amlNode) {
                    var model = apf.all[sUId[3]];
                    if (!model)
                        return;

                    if (model.$propBinds[sUId[1]][sUId[2]]) {
                        if (!apf.isChildOf(model.data, xmlNode, true))
                            return false;

                        var xpath = model.$propBinds[sUId[1]][sUId[2]].listen; //root
                        var node  = xpath
                            ? apf.queryNode(model.data, xpath)
                            : xmlNode;
                    }
                    if (node)
                        amlNode.$execProperty(sUId[2], node, args[3]);
                }
            }) - 1;
            this.$listeners[uId] = id;
        }
        else {
            //@todo apf3 potential cleanup problem
            id = "e" + uId;
            if (!this.$listeners[id]) {
                this.$listeners[id] = function(args){
                    var amlNode = apf.all[uId];
                    if (amlNode)
                        amlNode.$xmlUpdate.apply(amlNode, args);
                };
            }

            
        }

        if (!listen || listen.indexOf(";" + id + ";") == -1)
            xmlNode.setAttribute(this.xmlListenTag, (listen ? listen + id : ";" + id) + ";");

        return xmlNode;
    };

    /*
     * @todo  Use this function when an element really unbinds from a
     *        piece of data and does not uses it for caching
     * @private
     */
    this.removeNodeListener = function(xmlNode, o, id){
        var listen = xmlNode.getAttribute(this.xmlListenTag);
        var nodes = (listen ? listen.split(";") : []);
        if (id && id.charAt(0) == "p") {
            id = this.$listeners[id];
            delete this.$listeners[id];
        }
        else {
            id = "e" + o.$uniqueId;

            
        }

        for (var newnodes = [], i = 0; i < nodes.length; i++) {
            if (nodes[i] != id)
                newnodes.push(nodes[i]);
        }

        xmlNode.setAttribute(this.xmlListenTag, newnodes.join(";"));// + ";"

        return xmlNode;
    };

    /**
     * Sets the value of a text node. If the node doesn't exist, it is created.
     *
     * Changes are propagated to the databound elements listening for changes
     * on the data changed.
     *
     * @param {XMLElement} pNode     The parent of the text node.
     * @param {String}     value     The value of the text node.
     * @param {String}     [xpath]   The xpath statement which selects the text node.
     * @param {apf.UndoData}    [undoObj] The undo object that is responsible for archiving the changes.
     */
    this.setTextNode =
    apf.setTextNode  = function(pNode, value, xpath, undoObj, range){
        var tNode;

        if (xpath) {
            tNode = pNode.selectSingleNode(xpath);
            if (!tNode)
                return;
            pNode = tNode.nodeType == 1 ? tNode : null;
        }
        if (pNode.nodeType != 1)
            tNode = pNode;
        else if (pNode || !tNode) {
            tNode = pNode.selectSingleNode("text()");

            if (!tNode)
                tNode = pNode.appendChild(pNode.ownerDocument.createTextNode(""));//createCDATASection
        }

        //Action Tracker Support
        if (undoObj && !undoObj.$filled) {
            undoObj.extra.oldValue = tNode.nodeValue;
            undoObj.$filled = true;
        }

        //Apply Changes
        if (range) { //@todo apf3.0 range
            undoObj.extra.range = range;

        }
        else {
            tNode.nodeValue = value;

            if (tNode.$regbase)
                tNode.$setValue(value);
        }

        this.applyChanges("text", tNode.parentNode, undoObj);

        
        this.applyRDB(["setTextNode", pNode, value, xpath], undoObj || {xmlNode: pNode}); //@todo apf3.0 for range support
        
    };

    /**
     * Sets an attribute on a node. Changes are propagated to the databound
     * elements listening for changes on the data changed.
     *
     * @param {XMLElement} xmlNode   The XML node to set the attribute on.
     * @param {String}     name      The name of the attribute.
     * @param {String}     value     The value of the attribute.
     * @param {String}     [xpath]   The xpath statement to select the attribute.
     * @param {apf.UndoData}    [undoObj] The undo object that is responsible for archiving the changes.
     */
    this.setAttribute =
    apf.setAttribute  = function(xmlNode, name, value, xpath, undoObj, range){
        //Action Tracker Support
        if (undoObj && !undoObj.$filled) {
            undoObj.name = name;
            undoObj.$filled = true;
        }

        //Apply Changes
        if (range) { //@todo apf3.0 range
            undoObj.extra.range = range;
        }
        else
            (xpath ? xmlNode.selectSingleNode(xpath) : xmlNode).setAttribute(name, value);

        this.applyChanges("attribute", xmlNode, undoObj);
        
        this.applyRDB(["setAttribute", xmlNode, name, value, xpath], undoObj || {xmlNode: xmlNode});  //@todo apf3.0 for range support
        
    };

    /**
     * Removes an attribute of an XML node. Changes are propagated to the
     * databound elements listening for changes on the data changed.
     *
     * @param {XMLElement} xmlNode   The XML node to delete the attribute from
     * @param {String}     name      The name of the attribute.
     * @param {String}     [xpath]   The xpath statement to select the attribute.
     * @param {apf.UndoData}    [undoObj] The undo object that is responsible for archiving the changes.
     */
    this.removeAttribute =
    apf.removeAttribute  = function(xmlNode, name, xpath, undoObj){
        //if(xmlNode.nodeType != 1) xmlNode.nodeValue = value;

        //Action Tracker Support
        if (undoObj && !undoObj.$filled) {
            undoObj.name = name;
            undoObj.$filled = true;
        }

        //Apply Changes
        (xpath ? xmlNode.selectSingleNode(xpath) : xmlNode).removeAttribute(name);
        this.applyChanges("attribute", xmlNode, undoObj);

        
        this.applyRDB(["removeAttribute", xmlNode, name, xpath], undoObj || {xmlNode: xmlNode});
        
    };

    /**
     * Replace one node with another. Changes are propagated to the
     * databound elements listening for changes on the data changed.
     *
     * @param {XMLElement} oldNode   The XML node to remove.
     * @param {XMLElement} newNode   The XML node to set.
     * @param {String}     [xpath]   The xpath statement to select the attribute.
     * @param {apf.UndoData}    [undoObj] The undo object that is responsible for archiving the changes.
     */
    this.replaceNode =
    apf.replaceNode  = function(newNode, oldNode, xpath, undoObj){
        //if(xmlNode.nodeType != 1) xmlNode.nodeValue = value;

        //Apply Changes
        if (xpath)
            oldNode = oldNode.selectSingleNode(xpath);

        // @todo: only do this once! - should store on the undo object
        if (oldNode.ownerDocument.importNode && newNode.ownerDocument != oldNode.ownerDocument) {
            var oldNodeS = newNode;
            newNode = oldNode.ownerDocument.importNode(newNode, true); //Safari issue not auto importing nodes
            if (oldNodeS.parentNode)
                oldNodeS.parentNode.removeChild(oldNodeS);
        }

        
        this.applyRDB(["replaceNode", oldNode, this.cleanXml(newNode.xml), xpath], undoObj || {xmlNode: oldNode});
        

        //Action Tracker Support
        if (undoObj && !undoObj.$filled) {
            undoObj.$filled = true;
            undoObj.oldNode = oldNode;
            undoObj.xmlNode = newNode;
        }

        this.cleanNode(newNode);

        var parentNode = oldNode.parentNode;
        if (!parentNode)
            return;

        parentNode.replaceChild(newNode, oldNode);
        this.copyConnections(oldNode, newNode);

        this.applyChanges("replacenode", newNode, undoObj);

        return newNode;
    };

    /**
     * Creates a new element under a parent XML node. Changes are propagated
     * to the databound elements listening for changes on the data changed.
     *
     * @param {XMLElement} pNode       The parent XML node to add the new element to.
     * @param {String}     tagName     The tagName of the {@link term.datanode data node} to add.
     * @param {Array}      attr        list of the attributes to set. Each item is another array with the name and value.
     * @param {XMLElement} beforeNode  The XML node which indicates the insertion point.
     * @param {String}     [xpath]     The xpath statement to select the attribute.
     * @param {apf.UndoData}    [undoObj]   The undo object that is responsible for archiving the changes.
     */
    this.addChildNode =
    apf.addChildNode  = function(pNode, tagName, attr, beforeNode, undoObj){
        //Create New Node
        var xmlNode = pNode.insertBefore(pNode.ownerDocument
            .createElement(tagName), beforeNode);

        //Set Attributes
        for (var i = 0; i < attr.length; i++)
            xmlNode.setAttribute(attr[i][0], attr[i][1]);

        //Action Tracker Support
        if (undoObj && !undoObj.$filled) {
            undoObj.extra.addedNode = xmlNode;
            undoObj.$filled = true;
        }

        this.applyChanges("add", xmlNode, undoObj);

        
        this.applyRDB(["addChildNode", pNode, tagName, attr, beforeNode], undoObj || {xmlNode: pNode});
        

        return xmlNode;
    };

    /**
     * Appends an XML node to a parent. Changes are propagated
     * to the databound elements listening for changes on the data changed.
     *
     * @param {XMLElement} pNode       The parent XML node to add the element to.
     * @param {XMLElement} xmlNode     The XML node to insert.
     * @param {XMLElement} beforeNode  The XML node which indicates the insertion point.
     * @param {Boolean}    unique      Specifies whether the parent can only contain one element with a certain tag name.
     * @param {String}     [xpath]     The xpath statement to select the parent node.
     * @param {apf.UndoData}    [undoObj]   The undo object that is responsible for archiving the changes.
     */
    this.appendChild =
    apf.appendChild  = function(pNode, xmlNode, beforeNode, unique, xpath, undoObj){
        if (pNode == xmlNode.parentNode) //Shouldn't this be the same document?
            return apf.xmldb.moveNode(pNode, xmlNode, beforeNode, null, xpath, undoObj);

        if (unique && pNode.selectSingleNode(xmlNode.tagName))
            return false;

        // @todo: only do this once! - should store on the undo object
        if (pNode.ownerDocument.importNode && pNode.ownerDocument != xmlNode.ownerDocument) {
            var oldNode = xmlNode;
            xmlNode = pNode.ownerDocument.importNode(xmlNode, true); //Safari issue not auto importing nodes
            if (oldNode.parentNode)
                oldNode.parentNode.removeChild(oldNode);
        }

        
        this.applyRDB(["appendChild", pNode, this.cleanXml(xmlNode.xml), beforeNode, unique, xpath], undoObj || {xmlNode: pNode});
        

        //Add xmlNode to parent pNode or one selected by xpath statement
        if (xpath) {
            var addedNodes = [];
            pNode = apf.createNodeFromXpath(pNode, xpath, addedNodes);
            if (addedNodes.length) {
                pNode.appendChild(xmlNode);
                while(addedNodes.length) {
                    if (pNode == addedNodes.pop() && addedNodes.length)
                        pNode = pNode.parentNode;
                }
            }
        }
        else if (xmlNode.parentNode)
            this.removeNode(xmlNode);

        if (undoObj && !undoObj.$filled) {
            undoObj.$filled = true;
            this.cleanNode(xmlNode);
        }
        else
            this.cleanNode(xmlNode);

        pNode.insertBefore(xmlNode, beforeNode);

        //detect if xmlNode should be removed somewhere else
        //- [17-2-2004] changed pNode (2nd arg applychange) into xmlNode

        this.applyChanges("add", xmlNode, undoObj);

        return xmlNode;
    };

    /**
     * Moves an XML node to a parent node. Changes are propagated
     * to the databound elements listening for changes on the data changed.
     *
     * @param {XMLElement} pNode       The new parent XML node of the node.
     * @param {XMLElement} xmlNode     The XML node to move.
     * @param {XMLElement} beforeNode  The XML node which indicates the insertion point.
     * @param {String}     [xpath]     The xpath statement to select the parent node.
     * @param {apf.UndoData}    [undoObj]   The undo object that is responsible for archiving the changes.
     */
    this.moveNode =
    apf.moveNode  = function(pNode, xmlNode, beforeNode, xpath, undoObj){
        //Action Tracker Support
        if (!undoObj)
            undoObj = {extra:{}};

        undoObj.extra.oldParent  = xmlNode.parentNode;
        undoObj.extra.beforeNode = xmlNode.nextSibling;
        undoObj.extra.parent     = (xpath ? pNode.selectSingleNode(xpath) : pNode);

        this.applyChanges("move-away", xmlNode, undoObj);

        //Set new id if the node change document (for safari this should be fixed)
        //@todo I don't get this if...
        /*if (!apf.isWebkit
          && xmlNode.getAttribute(this.xmlIdTag)
          && apf.xmldb.getXmlDocId(xmlNode) != apf.xmldb.getXmlDocId(pNode)) {
            xmlNode.removeAttribute(this.xmlIdTag));
            this.nodeConnect(apf.xmldb.getXmlDocId(pNode), xmlNode);
        }*/

        // @todo: only do this once! - should store on the undo object
        if (pNode.ownerDocument.importNode && pNode.ownerDocument != xmlNode.ownerDocument) {
            var oldNode = xmlNode;
            xmlNode = pNode.ownerDocument.importNode(xmlNode, true); //Safari issue not auto importing nodes
            if (oldNode.parentNode)
                oldNode.parentNode.removeChild(oldNode);
        }

        
        this.applyRDB(["moveNode", pNode, xmlNode, beforeNode, xpath], undoObj || {xmlNode: pNode}); //note: important that transport of rdb is async
        

        undoObj.extra.parent.insertBefore(xmlNode, beforeNode);
        this.applyChanges("move", xmlNode, undoObj);
    };

    /**
     * Removes an XML node from its parent. Changes are propagated
     * to the databound elements listening for changes on the data changed.
     *
     * @param {XMLElement} xmlNode     The XML node to remove from the dom tree.
     * @param {String}     [xpath]     The xpath statement to select the parent node.
     * @param {apf.UndoData}    [undoObj]   The undo object that is responsible for archiving the changes.
     */
    this.removeNode =
    apf.removeNode  = function(xmlNode, xpath, undoObj){
        if (xpath)
            xmlNode = xmlNode.selectSingleNode(xpath);

        //ActionTracker Support
        if (undoObj && !undoObj.$filled) {
            undoObj.$filled           = true;
            undoObj.extra.parent      = xmlNode.parentNode;
            undoObj.extra.removedNode = xmlNode;
            undoObj.extra.beforeNode  = xmlNode.nextSibling;
        }

        
        this.applyRDB(["removeNode", xmlNode, xpath], undoObj || {xmlNode: xmlNode}); //note: important that transport of rdb is async
        

        //Apply Changes
        this.applyChanges("remove", xmlNode, undoObj);
        var p = xmlNode.parentNode;
        if (!p)
            return;

        p.removeChild(xmlNode);
        this.applyChanges("redo-remove", xmlNode, null, p);//undoObj

        //@todo clean xmlNode after removal??
    };

    /**
     * Removes a list of XML nodes from their parent. Changes are propagated
     * to the databound elements listening for changes on the data changed.
     *
     * @param {Array}   xmlNodeList A list of XML nodes to remove.
     * @param {apf.UndoData} [undoObj]   The undo object that is responsible for archiving the changes.
     */
    this.removeNodeList =
    apf.removeNodeList  = function(xmlNodeList, undoObj){
        
        this.applyRDB(["removeNodeList", xmlNodeList, null], undoObj || {xmlNode: p});
        

        //if(xpath) xmlNode = xmlNode.selectSingleNode(xpath);
        for (var rData = [], i = 0; i < xmlNodeList.length; i++) { //This can be optimized by looping nearer to xmlUpdate
            //ActionTracker Support
            if (undoObj) {
                rData.push({
                    pNode      : xmlNodeList[i].parentNode,
                    removedNode: xmlNodeList[i],
                    beforeNode : xmlNodeList[i].nextSibling
                });
            }

            //Apply Changes
            this.applyChanges("remove", xmlNodeList[i], undoObj);
            var p = xmlNodeList[i].parentNode;
            p.removeChild(xmlNodeList[i]);
            this.applyChanges("redo-remove", xmlNodeList[i], null, p);//undoObj
        }

        if (undoObj && !undoObj.$filled) {
            undoObj.$filled          = true;
            undoObj.extra.removeList = rData;
        }
    };

    /*
     * Looks for this.$listeners and executes their $xmlUpdate methods.
     * @private
     */
    var notifyQueue = {}, notifyTimer;
    this.$hasQueue = false;
    this.applyChanges = function(action, xmlNode, undoObj, nextloop){
        if (undoObj && undoObj.$dontapply) return;
        

        if (undoObj && !undoObj.xmlNode) //@todo are we sure about this?
            undoObj.xmlNode = xmlNode;

        //Set Variables
        var oParent  = nextloop,
            loopNode = (xmlNode.nodeType == 1 ? xmlNode : xmlNode.parentNode);

        //var xmlId = xmlNode.getAttribute(this.xmlIdTag);

        if (!this.delayUpdate && "|remove|move-away|".indexOf("|" + action + "|") > -1)
            this.notifyQueued(); //empty queue

        var listen, uId, uIds, i, j, hash, info, amlNode, runTimer, found, done = {};
        while (loopNode && loopNode.nodeType == 1) {
            //Get List of Node this.$listeners ID's
            listen = loopNode.getAttribute(this.xmlListenTag);

            if (listen) {
                uIds = listen.split(";");

                for (i = 0; i < uIds.length; i++) {
                    uId = uIds[i];
                    if (!uId || done[uId]) continue;
                    done[uId] = true;

                    //Property support
                    /*if (uId.charAt(0) == "p") {
                        uId = uId.split("|");

                        //@todo apf3.0 should this be exactly like in class.js?
                        //@todo optimize this to check the async flag: parsed[3] & 4

                        amlNode = apf.all[uId[1]]; //It's possible the aml node dissapeared in this loop.
                        if (amlNode) {
                            var model = apf.all[uId[3]];
                            var xpath = model.$propBinds[uId[1]][uId[2]].root;

                            amlNode.$execProperty(uId[2], xpath
                                ? model.data.selectSingleNode(xpath)
                                : model.data);
                        }
                        continue;
                    }*/

                    hash = notifyQueue[uId];
                    if (!hash)
                        notifyQueue[uId] = hash = [];

                    // Filtering
                    if (!apf.isO3 && "|update|attribute|text|".indexOf("|" + action + "|") > -1) {
                        found = false;
                        for (j = 0; j < hash.length; j++) {
                            if (hash[j] && xmlNode == hash[j][1]
                              && "|update|attribute|text|"
                              .indexOf("|" + hash[j][0] + "|") > -1) {
                                hash[j] = null;
                                found = true;
                                continue;
                            }
                        }

                        hash.push([action, xmlNode, loopNode, undoObj, oParent]);
                        runTimer = true;
                        continue;
                    }

                    //!this.delayUpdate && <- that doesnt work because of information that is destroyed
                    if (apf.isO3 || "|remove|move-away|move|add|".indexOf("|" + action + "|") > -1) {
                        if (this.$listeners[uId]) {
                            this.$listeners[uId]([action, xmlNode,
                                loopNode, undoObj, oParent]);
                        }
                        /*amlNode = apf.all[uId];
                        if (amlNode)
                            amlNode.$xmlUpdate(action, xmlNode,
                                loopNode, undoObj, oParent);*/
                    }
                    else {
                        hash.push([action, xmlNode, loopNode, undoObj, oParent]);
                        runTimer = true;
                    }
                }
            }

            //Go one level up
            loopNode = loopNode.parentNode || nextloop;
            if (loopNode == nextloop)
                nextloop = null;
        }

        if (true || undoObj && !this.delayUpdate) {
            //Ok this was an action let's not delay execution
            apf.xmldb.notifyQueued();
        }
        else if (runTimer) {
            apf.setZeroTimeout.clearTimeout(notifyTimer);
            //@todo find a better solution for this (at the end of a event stack unroll)
            this.$hasQueue = true;
            notifyTimer = apf.setZeroTimeout(function(){
                //this.$hasQueue = true;
                apf.xmldb.notifyQueued();
            });
        }
    };

    /*
     *  @todo in actiontracker - add stack auto purging
     *        - when undo item is purged which was a removed, remove cache item
     *  @todo shouldn't the removeNode method remove all this.$listeners?
     *  @todo rename to processQueue
     *  @private
     */
    this.notifyQueued = function(){
        this.$hasQueue = false;

        var myQueue = notifyQueue;
        notifyQueue = {};

        apf.setZeroTimeout.clearTimeout(notifyTimer);
        for (var uId in myQueue) {
            if (!uId) continue;

            var q       = myQueue[uId];
            var func    = this.$listeners[uId];
            //!amlNode ||
            if (!q || !func)
                continue;

            //Run queue items
            for (var i = 0; i < q.length; i++) {
                if (!q[i])
                    continue;

                //Update xml data
                //amlNode.$xmlUpdate.apply(amlNode, q[i]);
                func(q[i]);
            }
        }


    };

    /**
     * @private
     */
    this.notifyListeners = function(xmlNode){
        //This should be done recursive
        var listen = xmlNode.getAttribute(apf.xmldb.xmlListenTag);
        if (listen) {
            listen = listen.split(";");
            for (var j = 0; j < listen.length; j++) {
                apf.all[listen[j]].$xmlUpdate("synchronize", xmlNode, xmlNode);
                //load(xmlNode);
            }
        }
    };

    
    /*
     * Sends Message through transport to tell remote databound this.$listeners
     * that data has been changed
     * @private
     */
    this.applyRDB = function(args, undoObj){
        if (apf.xmldb.disableRDB)
            return;

        var xmlNode = undoObj.localName || !undoObj.xmlNode
            ? args[1] && args[1].length && args[1][0] || args[1]
            : undoObj.xmlNode;

        if (xmlNode.nodeType == 2)
            xmlNode = xmlNode.ownerElement || xmlNode.selectSingleNode("..");
        
        var mdlId   = apf.xmldb.getXmlDocId(xmlNode),
            model   = apf.nameserver.get("model", mdlId);
        if (!model && apf.isO3)
            model = self[mdlId];
        if (!model) {
            if (!apf.nameserver.getAll("remote").length)
                return;
            
            return;
        }

        if (!model.rdb) return;
        var rdb = model.rdb;

        // Add the messages to the undo object
        if (undoObj.action)
            rdb.$queueMessage(args, model, undoObj);
        // Or send message now
        else {
            clearTimeout(rdb.queueTimer);

            rdb.$queueMessage(args, model, rdb);
            // use a timeout to batch consecutive calls into one RDB call
            rdb.queueTimer = $setTimeout(function() {
                rdb.$processQueue(rdb);
            });
        }
        
    };
    

    /**
     * @private
     */
    this.copyConnections = function(fromNode, toNode){
        //This should copy recursive
        try {
            toNode.setAttribute(this.xmlListenTag, fromNode.getAttribute(this.xmlListenTag));
        }
        catch (e) {}
        try {
            toNode.setAttribute(this.xmlIdTag, fromNode.getAttribute(this.xmlIdTag));
        }
        catch (e) {}
    };

    /**
     * @private
     */
    this.cleanXml = function(xml) {
        if (typeof xml != "string")
            return xml;
        return xml.replace(cleanRE, "").replace(whiteRE, "><");
    };

    /**
     * @private
     */
    this.cleanNode = function(xmlNode){
        try {
            var i, nodes = xmlNode.selectNodes("descendant-or-self::node()[@" + this.xmlListenTag + "]");
            for (i = nodes.length - 1; i >= 0; i--)
                nodes[i].removeAttribute(this.xmlListenTag);
            nodes = xmlNode.selectNodes("descendant-or-self::node()[@" + this.xmlIdTag + "]");
            for (i = nodes.length - 1; i >= 0; i--)
                nodes[i].removeAttribute(this.xmlIdTag);
            nodes = xmlNode.selectNodes("descendant-or-self::node()[@" + this.xmlDocTag + "]");
            for (i = nodes.length - 1; i >= 0; i--)
                nodes[i].removeAttribute(this.xmlDocTag);
            nodes = xmlNode.selectNodes("descendant-or-self::node()[@a_loaded]");
            for (i = nodes.length - 1; i >= 0; i--)
                nodes[i].removeAttribute("a_loaded");
        }
        catch (e) {}

        return xmlNode;
    };

    /**
     * Returns a copy of the passed {@link term.datanode data node}. Bound
     * data nodes contain special attributes to track them. These attributes
     * are removed from the copied node when using this method.
     *
     * @param {XMLElement} xmlNode The {@link term.datanode data node} to copy.
     * @return {XMLElement} The copy of the {@link term.datanode data node}.
     */
    this.copy         =
    this.getCleanCopy =
    apf.getCleanCopy  = function(xmlNode){
        return apf.xmldb.cleanNode(xmlNode.cloneNode(true));
    };

    /**
     * Unbind all APF Elements from a certain Form
     * @private
     */
    this.unbind = function(frm){
        //Loop through objects of all apf
        for (var lookup = {}, i = 0; i < frm.apf.all.length; i++)
            if (frm.apf.all[i] && frm.apf.all[i].unloadBindings)
                lookup[frm.apf.all[i].unloadBindings()] = true;

        //Remove Listen Nodes
        for (var k = 0; k < this.$xmlDocLut.length; k++) {
            
            if (!this.$xmlDocLut[k]) continue;
            

            var Nodes = this.$xmlDocLut[k].selectNodes("//self::node()[@"
                + this.xmlListenTag + "]");
            if (!Nodes) continue;

            //Loop through Nodes and rebuild listen array
            for (var i = 0; i < Nodes.length; i++) {
                var listen = Nodes[i].getAttribute(this.xmlListenTag).split(";");
                for (var nListen = [], j = 0; j < listen.length; j++)
                    if (!lookup[listen[j]])
                        nListen.push(listen[j]);

                //Optimization??
                if (nListen.length != listen.length)
                    Nodes[i].setAttribute(this.xmlListenTag, nListen.join(";"));
            }
        }

        if (window.clearInterval)
            window.clearInterval(this.$gcInterval);
    };

    /*
     * @private
     * @todo xml doc leakage
     */
    this.getXmlDocId = function(xmlNode, model){
        var docEl = xmlNode.ownerDocument.documentElement;
        if (!apf.isChildOf(docEl, xmlNode))
            docEl = xmlNode;

        var docId = (docEl || xmlNode).getAttribute(this.xmlDocTag)
            || this.$xmlDocLut.indexOf(docEl || xmlNode.ownerDocument || xmlNode);

        if (model && apf.nameserver.get("model", docId) != model) {
            docId = null;
            docEl = xmlNode;
        }

        if (!docId || docId == -1) {
            docId = this.$xmlDocLut.push(docEl || xmlNode.ownerDocument || xmlNode) - 1;
            if (docEl)
                docEl.setAttribute(this.xmlDocTag, String(docId));
        }
        
        if (model)
            apf.nameserver.register("model", docId, model);
        

        return docId;
    };
});






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
 




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * @class apf.http
 *
 * This object does what is commonly known as Ajax; it **A**synchronously 
 * communicates using **J**avaScript, **A**nd in most 
 * cases it sends or receives **X**ml. It allows for easy HTTP 
 * communication from within the browser. 
 *
 * This object provides caching on top of
 * the browser's cache. This enables you to optimize your application, because
 * this can be set on a per call basis. 
 *
 * #### Example:
 *
 * Retrieving content over HTTP synchronously:
 *
 * ```javascript
 *  var http = new apf.http();
 *  var data = http.get("http://www.example.com/mydata.jsp", {async: false});
 *  alert(data);
 * ```
 *
 * #### Example:
 *
 * Retrieving content over HTTP asynchronously:
 *
 * ```javascript
 *  var http = new apf.http();
 *  http.get("http://www.example.com/mydata.jsp", {
 *      callback: function(data, state, extra){
 *         if (state != apf.SUCCESS)
 *             return alert('an error has occurred');
 *
 *         alert(data);
 *      }
 *  });
 * ```
 *
 * #### Example:
 *
 * An asynchronous HTTP request, with retry:
 *
 * ```javascript
 *  var http = new apf.http();
 *  http.get("http://www.example.com/mydata.jsp", {
 *      callback: function(data, state, extra){
 *          if (state != apf.SUCCESS) {
 *              var oError = new Error(apf.formatErrorString(0, null,
 *                  "While loading data", "Could not load data\n" + extra.message));
 *
 *              if (extra.tpModule.retryTimeout(extra, state, null, oError) === true)
 *                  return true;
 *
 *              throw oError;
 *          }
 *
 *          alert(data);
 *      }
 *  });
 * ```
 */
/**
 * @event error Fires when a communication error occurs.
 * @bubbles
 * @cancelable  Prevents a communication error to be thrown.
 * @param {Object} e An object returned by the callback. It contains the following properties:
 *     - error ([[Error]]): The error object that is thrown when the event
 *                                callback doesn't return false.
 *     - state ([[Number]]): The state of the call. Possible values include:
 *       - `apf.SUCCESS`:  The request was successfull
 *       - `apf.TIMEOUT`:  The request has timed out.
 *       - `apf.ERROR`:    An error has occurred while making the request.
 *       - `apf.OFFLINE`:  The request was made while the application was offline.
 *     - userdata (`Mixed`): Data that the caller wanted to be available in
 *                                the callback of the HTTP request.
 *     - http ([[XMLHttpRequest]]): The object that executed the actual HTTP request.
 *     - url ([[String]]): The URL that was requested.
 *     - tpModule ([[apf.http]]): The teleport module that is making the request.
 *     - id ([[Number]]): The id of the request.
 *     - message ([[String]]): The error message.
 *
 * @define http
 * @default_private
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.4
 */
apf.http = function(){
    this.queue     = [null];
    this.callbacks = {};
    this.cache     = {};

    /**
     * Sets the timeout of http requests in milliseconds. Default is 10000ms (10s).
     * @type {Number}
     */
    this.timeout   = this.timeout || 10000; //default 10 seconds
    
    /**
     * Sets whether this element routes traffic through a server proxy.
     *
     * #### Remarks
     *
     * This can also be set on a per call basis. See {@link apf.http.get}.
     *
     * 
     * @type {Boolean}
     */
    this.autoroute = this.autoroute || false;
    
    /**
     * String specifying the URL to the route script. 
     * 
     * #### Remarks
     *
     * The route script will receive the route information in three extra headers:
     *   - `X-Route-Request`     : Contains the destination URL
     *   - `X-Proxy-Request`     : Contains the proxy URL
     *   - `X-Compress-Response` : Set to 'gzip'
     *
     * @type {String}
     */
    this["route-server"] = this["route-server"] || null;

    if (!this.$uniqueId)
        this.$uniqueId = apf.all.push(this) - 1;

    this.toString = this.toString || function(){
        return "[Ajax.org Teleport Component : (HTTP)]";
    };

    

    /**
     * Makes an HTTP request that receives XML.
     * @param {String}   url       The url that is accessed.
     * @param {Object}   options   The options for the HTTP request. It contains the following properties:
     *   - async ([[Boolean]]): Specifies whether the request is sent asynchronously. Defaults to true.
     *   - userdata (`Mixed`): custom data that is available to the callback function.
     *   - method ([[String]]): The request method (`POST`|`GET`|`PUT`|`DELETE`). Defaults to `GET`.
     *   - nocache ([[Boolean]]): Specifies whether browser caching is prevented.
     *   - data ([[String]]): the data sent in the body of the message.
     *   - autoroute ([[Boolean]]): Specifies whether the request can fallback to a server proxy.
     *   - caching ([[Boolean]]): Specifies whether the request should use internal caching.
     *   - ignoreOffline ([[Boolean]]): Specifies whether to ignore offline catching.
     *   - callback ([[Function]]): The handler that gets called whenever the
     *                            request completes succesfully or with an error,
     *                            or when the request times out.
     */
    this.getXml = function(url, callback, options){
        if (!options) options = {};
        options.useXML = true;
        options.callback = callback;
        return this.get(url, options);
    };
    
    this.getJSON = function(url, callback, options){
        if (!options) options = {};
        options.callback = callback;
        options.useJSON = true;
        return this.get(url, options);
    };

    /**
     * Makes an HTTP request.
     * @param {String}   url       The URL that is accessed.
     * @param {Object}   options   The options for the HTTP request. It contains the following properties:
     *   - async ([[Boolean]]): Specifies whether the request is sent asynchronously. Defaults to true.
     *   - userdata (`Mixed`): Custom data that is available to the callback function.
     *   - method ([[String]]): The request method (POST|GET|PUT|DELETE). Defaults to GET.
     *   - nocache ([[Boolean]]): Specifies whether browser caching is prevented.
     *   - data ([[String]]): The data sent in the body of the message.
     *   - useXML ([[Boolean]]): Specifies whether the result should be interpreted as xml.
     *   - autoroute ([[Boolean]]): Specifies whether the request can fallback to a server proxy.
     *   - caching ([[Boolean]]): Specifies whether the request should use internal caching.
     *   - ignoreOffline ([[Boolean]]): Specifies whether to ignore offline catching.
     *   - contentType ([[String]]): The mime type of the message
     *   - withCredentials ([[Boolean]]): Value of the withCredentials field for CORS requests
     *   - callback ([[Function]]): The handler that gets called whenever the
     *                            request completes succesfully or with an error,
     *                            or when the request times out.
     */
    this.get = this.$get = function(url, options){
        if (!options)
            options = {};

        var _self = this;
        var id    = options.id;
        
        
        

        var binary = apf.hasXhrBinary && options.binary;
        var async = options.async = (options.async || binary 
            || typeof options.async == "undefined" || apf.isOpera || false);

        
        if (apf.isWebkit)
            url = apf.html_entity_decode(url);
        

        var data = options.data || "";

        if (apf.isNot(id)) {
            
                var http = apf.getHttpReq();

            id = this.queue.push({
                http     : http,
                url      : url,
                callback : options.callback,
                retries  : 0,
                options  : options
            }) - 1;

            
        }
        else {
            var http = this.queue[id].http;

            
                http.abort();
        }

        if (async) {
            
            {
                http.onreadystatechange = function(){
                    if (!_self.queue[id] || http.readyState != 4)
                        return;
                    if (async && arguments.callee.caller)
                        $setTimeout(function(){_self.receive(id)});
                    else
                        _self.receive(id);
                }
            }
        }

        var autoroute = this.autoroute && apf.isOpera
            ? true //Bug in opera
            : (options.autoroute || this.shouldAutoroute),
            httpUrl = autoroute ? this["route-server"] : url;

        
        var headers = [];
        
        function setRequestHeader(name, value){
            
            http.setRequestHeader(name, value);
        }

        var errorFound = false;
        try {
            if (options.nocache)
                httpUrl = apf.getNoCacheUrl(httpUrl);

            
            
            var requestedWithParam = apf.config ? apf.config["requested-with-getparam"] : null;
            if (requestedWithParam) {
                httpUrl += (httpUrl.indexOf("?") == -1 ? "?" : "&") +
                    encodeURIComponent(requestedWithParam) + "=1";
            }

            var withCredentials = false;
            if ("withCredentials" in options) {
                withCredentials = options.withCredentials;
            }
            else {
                withCredentials = (apf.config && apf.config["cors-with-credentials"]) || false;
            }
            http.withCredentials = withCredentials;

            // global support for protection against Cross Site Request Forgery
            // attacks by supplying a token to the global APF config object. This
            // token will be appended to the URL and sent for each XHR.
            // Warning: if you are doing CORS, be sure to use a different method!
            var method = this.method || options.method || "GET";
            var CSRFHeader = apf.config ? apf.config["csrf-header"] : null;
            var CSRFToken = apf.config ? apf.config["csrf-token"] : null;
            if (method !== "GET" && CSRFToken) {
                CSRFToken = CSRFToken.split("=").map(function(s) { return encodeURIComponent(s); }).join("=");
                httpUrl += (httpUrl.indexOf("?") == -1 ? "?" : "&") + CSRFToken;
            }

            http.open(method, httpUrl, async);

            if (method !== "GET" && CSRFHeader) {
                setRequestHeader("X-CSRF-Token", CSRFHeader);
            }

            if (options.username) {
                setRequestHeader("Authorization", "Basic "
                    + apf.crypto.Base64.encode(options.username + ":" + options.password))
            }

            //@todo OPERA ERROR's here... on retry [is this still applicable?]
            if (!requestedWithParam)
                setRequestHeader("X-Requested-With", "XMLHttpRequest");

            if (!options.headers || !options.headers["Content-type"])
                setRequestHeader("Content-type", options.contentType || this.contentType
                    || (this.useXML || options.useXML ? "text/xml" : "text/plain"));

            if (autoroute) {
                setRequestHeader("X-Route-Request", url);
                setRequestHeader("X-Proxy-Request", url);
                setRequestHeader("X-Compress-Response", "gzip");
            }
            
            if (binary) {
                setRequestHeader("Cache-Control", "no-cache");
                setRequestHeader("X-File-Name", binary.filename);
                setRequestHeader("X-File-Size", binary.filesize);
                setRequestHeader("Content-Type", "application/octet-stream");
            }
        }
        catch (e) {
            errorFound = e.message;
        }

        if (errorFound) {
            var useOtherXH = false;

            // Retry request by routing it
            if (!useOtherXH && this.autoroute && !autoroute) {
                

                this.shouldAutoroute = true;

                options.autoroute = true;
                return this.get(url, options);
            }

            if (!useOtherXH) {
                //Routing didn't work either... Throwing error
                var noClear = options.callback ? options.callback(null, apf.ERROR, {
                    userdata: options.userdata,
                    http    : http,
                    url     : url,
                    tpModule: this,
                    id      : id,
                    message : "Permission denied accessing remote resource: "
                              + url + "\nMessage: " + errorFound
                }) : false;
                if (!noClear)
                    this.clearQueueItem(id);

                return;
            }
        }

        if (this.$headerHook)
            this.$headerHook(http);

        //Set request headers
        if (options.headers) {
            for (var name in options.headers)
                setRequestHeader(name, options.headers[name]);
        }
        
        

        function handleError(){
            var msg = self.navigator && self.navigator.onLine
                ? "File or Resource not available " + url
                : "Browser is currently working offline";

            

            var state = self.navigator && navigator.onLine
                ? apf.ERROR
                : apf.TIMEOUT;

            // File not found
            var noClear = options.callback ? options.callback(null, state, {
                userdata : options.userdata,
                http     : http,
                url      : url,
                tpModule : _self,
                id       : id,
                message  : msg
            }) : false;
            if (!noClear)
                _self.clearQueueItem(id);
        }

        function send(isLocal){
            var hasError;

            if (apf.isIE && isLocal) { //When local IE calls onreadystate immediately
                var oldWinOnerror = window.onerror;
                window.onerror = function(){
                    if (arguments.caller && arguments.caller.callee == send) {
                        window.onerror = oldWinOnerror;
                        //_self.receive(id);
                        //setTimeout(function(){handleError();});
                        return true;
                    }
                    else {
                        window.onerror = oldWinOnerror;
                        
                        if (oldWinOnerror)
                            return oldWinOnerror.apply(window, arguments);
                    }
                }
                http.send(data);
                window.onerror = oldWinOnerror;
            }
            else {
                try {
                    if (binary && http.sendAsBinary) {
                        binary.blob = getBinaryBlob(data, http, binary);
                        http.sendAsBinary(binary.blob.data);
                    }
                    else
                        http.send(data);
                }
                catch(e){
                    hasError = true;
                }
            }

            if (hasError) {
                handleError();
                return;
            }
            else if (binary && http.upload) {
                http.upload.onprogress = function(e) {
                    apf.dispatchEvent("http.uploadprogress", {
                        loaded  : e.loaded - binary.blob.size,
                        extra   : e,
                        bubbles : true
                    });
                };
            }
        }

        if (!async) {
            send.call(this);
            return this.receive(id);
        }
        else {
            if (apf.loadsLocalFilesSync && location.protocol == "file:"
              && url.indexOf("http://") == -1) {
                $setTimeout(function(){
                    send.call(_self, true);
                });
            }
            else
                send.call(_self);

            return id;
        }
    };
    
    
    if (!this.exec) {
    /**
             * A method that all async objects should implement.
             *
     * @private
     */
        this.exec = function(method, args, callback, options){
            if (!options)
                options = {};
            
            var url = args[0], query = "";
            if (!options.method)
                options.method = method.toUpperCase();
            if (!options.callback)
                options.callback = callback;
            
            this.contentType = "application/x-www-form-urlencoded";
            this.$get(
                apf.getAbsolutePath(apf.config.baseurl, url), 
                options.method == "GET" 
                    ? options 
                    : apf.extend({data : query}, options)
            );
        }
    }
    
    
    /**
     * Sends the binary blob to the server, and multipart encodes it if needed. This code 
     * will only be executed on Gecko since it's currently the only browser that 
     * supports direct file access.
     * @private
     */
    function getBinaryBlob(data, http, binary) {
        var boundary      = "----apfbound".appendRandomNumber(5),
            dashdash      = "--",
            crlf          = "\r\n",
            multipartBlob = "",
            multipartSize = 0;

        // Build multipart request
        if (binary.multipart) {
            http.setRequestHeader("Content-Type", "multipart/form-data; boundary=" + boundary);
            // Build RFC2388 blob
            multipartBlob += dashdash + boundary + crlf +
                'Content-Disposition: form-data; name="' + (binary.filedataname || binary.filename)
                    + '"; filename="' + binary.filename + '"' + crlf +
                'Content-Type: application/octet-stream' + crlf + crlf +
                data + crlf +
                dashdash + boundary + dashdash + crlf;

            multipartSize = multipartBlob.length - data.length;
            data = multipartBlob;
        }
        // Send blob or multipart blob depending on config
        return {size: multipartSize, data: data};
    }

    /**
     * @private
     */
    this.receive = function(id){
        if (!this.queue[id])
            return false;

        var qItem    = this.queue[id],
            http     = qItem.http,
            callback = qItem.callback;
        //if (apf.isGecko)
        //    var apf = self.apf || apf;     // needed here to fix a rare ReferenceError in FF

        

        /*if (self.navigator && navigator.onLine === false &&
          (location.protocol != "file:"
          || qItem.url.indexOf("http://") > -1))
            return false;*/

        // Test if HTTP object is ready
        if (qItem.async) {
            try {
                if (http.status) {}
            }
            catch (e) {
                var _self = this;
                return $setTimeout(function(){
                    _self.receive(id)
                }, 10);
            }
        }
        
        

        //Gonna check for validity of the http response
        var errorMessage = [],
            extra = {
                
                tpModule : this,
                http     : http,
                status   : http.status,
                url      : qItem.url,
                callback : callback,
                id       : id,
                retries  : qItem.retries || 0,
                userdata : qItem.options.userdata
            };

        // Check HTTP Status
        // The message didn't receive the server. We consider this a timeout (i.e. 12027)
        if (http.status > 600)
            return this.$timeout(id);

        extra.data = qItem.options.useJSON 
            ? eval("(" + http.responseText + ")") 
            : http.responseText; //Can this error?

        if (http.status >= 400 && http.status < 600 || http.status < 10 
          && (http.status != 0 || !apf.isIE && !http.responseText)) { //qItem.url.substr(0, 6) == "file:/"
            
            //@todo This should probably have an RPC specific handler
            if (http.status == 401) {
                var auth = apf.document.getElementsByTagNameNS(apf.ns.apf, "auth")[0];
                if (auth) {
                    var wasDelayed = qItem.isAuthDelayed;
                    qItem.isAuthDelayed = true;
                    if (auth.authRequired(extra, wasDelayed) === true)
                        return;
                }
            }
            

            errorMessage.push("HTTP error [" + id + "]:" + http.status + "\n"
                + http.responseText);
        }

        // Check for XML Errors
        if (qItem.options.useXML || this.useXML) {
            /* Note (Mike, Oct 14th 2008): for WebDAV, I had to copy the lines below,
                                           it required custom responseXML handling/
                                           parsing.
                                           If you alter this code, please correct
                                           webdav.js appropriately.
            */
            if ((http.responseText || "").replace(/^[\s\n\r]+|[\s\n\r]+$/g, "") == "")
                errorMessage.push("Received an empty XML document (0 bytes)");
            else {
                try {
                    var xmlDoc = (http.responseXML && http.responseXML.documentElement)
                        ? apf.xmlParseError(http.responseXML)
                        : apf.getXmlDom(http.responseText);

                    if (!apf.supportNamespaces)
                        xmlDoc.setProperty("SelectionLanguage", "XPath");

                    extra.data = xmlDoc.documentElement;
                }
                catch(e){
                    errorMessage.push("Received invalid XML\n\n" + e.message);
                }
            }
        }

        //Process errors if there are any
        if (errorMessage.length) {
            extra.message = errorMessage.join("\n");

            

            // Send callback error state
            if (!callback || !callback(extra.data, apf.ERROR, extra))
                this.clearQueueItem(id);

            return;
        }

        

        

        //Http call was successfull Success
        if (!callback || !callback(extra.data, apf.SUCCESS, extra))
            this.clearQueueItem(id);

        return extra.data;
    };

    this.$timeout = function(id){
        if (!this.queue[id])
            return false;

        var qItem = this.queue[id],
            http  = qItem.http;

        

        // Test if HTTP object is ready
        try {
            if (http.status) {}
        }
        catch (e) {
            var _self = this;
            return $setTimeout(function(){
                _self.$timeout(id)
            }, 10);
        }

        var callback = qItem.callback;

        http.abort();

        

        var extra;
        var noClear = callback ? callback(null, apf.TIMEOUT, extra = {
            
            userdata: qItem.options.userdata,
            http    : http,
            url     : qItem.url,
            tpModule: this,
            id      : id,
            message : "HTTP Call timed out",
            retries : qItem.retries || 0
        }) : false;
        
        
        
        if (!noClear)
            this.clearQueueItem(id);
    };

    /**
     * Checks if the request has timed out. If so, it is retried
     * three times before an exception is thrown. Request retrying is a very
     * good way to create a robust Ajax application. In many cases, even with
     * good connections, requests still time out.
     * @param {Object}  extra      The information object given as a third
     *                             argument of the HTTP request callback.
     * @param {Number}  state      The return code of the HTTP request. It contains the following properties:
     *   - `apf.SUCCESS`:  the request was successfull
     *   - `apf.TIMEOUT`:  the request has timed out.
     *   - `apf.ERROR`:    an error occurred while making the request.
     *   - `apf.OFFLINE`:  the request was made while the application was offline.
     * @param {apf.AmlNode} [amlNode]    The element receiving the error event.
     * @param {Error}   [oError]     The error to be thrown when the request is
     *                               not retried.
     * @param {Number}  [maxRetries] The number of retries that are done before
     *                               the request times out. Default is 3.
     */
    this.retryTimeout = function(extra, state, amlNode, oError, maxRetries){
        if (state == apf.TIMEOUT
          && extra.retries < (maxRetries || apf.maxHttpRetries))
            return extra.tpModule.retry(extra.id);

        oError = oError || new Error(apf.formatErrorString(0,
            this, "Communication " + (state == apf.TIMEOUT
                ? "timeout"
                : "error"), "Url: " + extra.url + "\nInfo: " + extra.message));

        if ((amlNode || apf).dispatchEvent("error", apf.extend({
            error   : oError,
            state   : state,
            extra   : extra,
            bubbles : true
        }, extra)) === false)
            return 2;
    };

    /**
     * Removes the item from the queue. This is usually done automatically.
     * However, when the callback returns `true` the queue isn't cleared; for instance,
     * when a request is retried. The id of the call
     * is found on the `'extra'` object, the third argument of the callback.
     * 
     * #### Example
     *
     * ```javascript
     *  http.clearQueueItem(extra.id);
     * ```
     *
     * @param {Number} id The id of the call that should be removed from the queue.
     */
    this.clearQueueItem = function(id){
        if (!this.queue[id])
            return false;

        

        if (apf.releaseHTTP && !apf.isGecko)
            apf.releaseHTTP(this.queue[id].http);

        this.queue[id] = null;
        delete this.queue[id];

        return true;
    };

    /**
     * Retries a call based on its id. The id of the call is found on the
     * `'extra'` object, the third argument of the callback.
     * 
     * #### Example
     *
     * ```javascript
     *  function callback(data, state, extra){
     *      if (state == apf.TIMEOUT && extra.retries < apf.maxHttpRetries)
     *          return extra.tpModule.retry(extra.id);
     *
     *      //Do stuff here
     *  }
     * ```
     *
     * @param {Number} id The id of the call that should be retried.
     */
    this.retry = function(id){
        if (!this.queue[id])
            return false;

        var qItem = this.queue[id];

        

        

        qItem.retries++;
        qItem.options.id = id;
        this.get(qItem.url, qItem.options);

        return true;
    };

    /**
     * Cancels a call based on its id. The id of the call is found on the
     * `'extra'` object, the third argument of the callback.
     *
     * @see apf.http.clearQueueItem
     *
     * @param {Number} id The id of the call that should be canceled.
     */
    this.cancel = function(id){
        if (id === null)
            id = this.queue.length - 1;

        if (!this.queue[id])
            return false;

        if (apf.isGecko)
            this.queue[id].http.abort();

        this.clearQueueItem(id);
    };

    if (!this.$loadAml && !this.instantiate && !this.call) {
        /**
         * @private
         */
        this.$loadAml = function(x){
            var receive = this["receive"];

            for (var i = 0, l = this.childNodes.length; i < l; i++) {
                if (this.childNodes[i].nodeType != 1)
                    continue;

                var url      = this.childNodes[i].getAttribute("url"),
                    callback = self[this.childNodes[i].getAttribute("receive") || receive],
                    options  = {
                        useXML  : this.childNodes[i].getAttribute("type") == "XML",
                        async   : !apf.isFalse(this.childNodes[i].getAttribute("async"))
                    };

                this[this.childNodes[i].getAttribute("name")] = function(data, userdata){
                    options.userdata = userdata;
                    options.data     = data;
                    return this.get(url, options);
                }
            }
        };

        /**
         * @private
         */
        this.instantiate = function(x){
            var url     = x.getAttribute("src"),
                options = {
                    async   : x.getAttribute("async") != "false",
                    nocache : true
                };

            this.getURL = function(data, userdata){
                options.data     = data;
                options.userdata = userdata;
                options.callback = this.callbacks.getURL;
                return this.get(url, options);
            }

            var name = "http" + Math.round(Math.random() * 100000);
            apf.setReference(name, this);

            return name + ".getURL()";
        };

        /**
         * @private
         */
        this.call = function(method, args){
            this[method].call(this, args);
        };
    }
};



apf.Init.run("http");




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: socket://www.fsf.org.
 *
 */























/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * The parser of the Ajax.org Markup Language. Besides aml this parser takes care
 * of distributing parsing tasks to other parsers like the native html parser and
 * the xsd parser.
 * @parser
 * @private
 *
 * @define include element that loads another aml files.
 * Example:
 * <code>
 *   <a:include src="bindings.aml" />
 * </code>
 * @attribute {String} src the location of the aml file to include in this application.
 *
 */
apf.DOMParser = function(){};

apf.DOMParser.prototype = new (function(){
    this.caseInsensitive    = true;
    this.preserveWhiteSpace = false; //@todo apf3.0 whitespace issue

    this.$waitQueue  = {}
    this.$callCount  = 0;

    // privates
    var RE     = [
            /\<\!(DOCTYPE|doctype)[^>]*>/,
            /&nbsp;/g,
            /<\s*\/?\s*(?:\w+:\s*)[\w-]*[\s>\/]/g
        ],
        XPATH  = "//@*[not(contains(local-name(), '.')) and not(translate(local-name(), 'ABCDEFGHIJKLMNOPQRSTUVWXYZ', 'abcdefghijklmnopqrstuvwxyz') = local-name())]";

    this.parseFromString = function(xmlStr, mimeType, options){
        var xmlNode;
        if (this.caseInsensitive) {
            //replace(/&\w+;/, ""). replace this by something else
            //.replace(RE[1], " ")
            var str = xmlStr.replace(RE[0], "")
              .replace(RE[2], //.replace(/^[\r\n\s]*/, "")
                function(m){ return m.toLowerCase(); });

            /* @todo apf3.0 integrate this
            x.ownerDocument.setProperty("SelectionNamespaces",
                                    "xmlns:a='" + apf.ns.aml + "'");
            */

            if (!this.supportNamespaces)
                str = str.replace(/xmlns\=\"[^"]*\"/g, "");

            

            var xmlNode = apf.getXmlDom(str);
            if (apf.xmlParseError) apf.xmlParseError(xmlNode);
            xmlNode = xmlNode.documentElement;
            
        }
        else {
            xmlNode = apf.getXmlDom(xmlStr, null, this.preserveWhiteSpace || apf.debug).documentElement;
        }

        return this.parseFromXml(xmlNode, options);
    };

    //@todo prevent leakage by not recording .$aml
    this.parseFromXml = function(xmlNode, options){
        var doc, docFrag, amlNode, beforeNode;
        if (!options)
            options = {};

        if (!options.delayedRender && !options.include) {
            //Create a new document
            if (options.doc) {
                doc     = options.doc;
                docFrag = options.docFrag || doc.createDocumentFragment();
            }
            else {
                doc            = new apf.AmlDocument();
                doc.$aml       = xmlNode;
                doc.$domParser = this;
            }
            if (options.host)
                doc.$parentNode = options.host; //This is for sub docs that need to access the outside tree

            

            //Let's start building our tree
            amlNode = this.$createNode(doc, xmlNode.nodeType, xmlNode); //Root node
            (docFrag || doc).appendChild(amlNode);
            if (options.htmlNode)
                amlNode.$int = options.htmlNode;
        }
        else {
            amlNode    = options.amlNode;
            doc        = options.doc;

            if (options.include) {
                var n = amlNode.childNodes;
                var p = n.indexOf(options.beforeNode);
                var rest = p ? n.splice(p, n.length - p) : [];
            }
        }

        //Set parse context
        this.$parseContext = [amlNode, options];

        this.$addParseState(amlNode, options || {});

        //First pass - Node creation
        var nodes, nodelist = {}, prios = [], _self = this;
        var recur;
        (recur = function(amlNode, nodes){
            var cL, newNode, node, nNodes,
                cNodes = amlNode.childNodes,
                i      = 0,
                l      = nodes.length;
            for (; i < l; i++) {
                //Create child
                newNode = _self.$createNode(doc, (node = nodes[i]).nodeType, node);
                if (!newNode) continue; //for preserveWhiteSpace support

                cNodes[cL = cNodes.length] = newNode; //Add to children

                //Set tree refs
                newNode.parentNode = amlNode;
                if (cL > 0)
                    (newNode.previousSibling = cNodes[cL - 1]).nextSibling = newNode;

                //Create children
                if (!newNode.render && newNode.canHaveChildren && (nNodes = node.childNodes).length)
                    recur(newNode, nNodes);

                //newNode.$aml = node; //@todo should be deprecated...

                //Store high prio nodes for prio insertion
                if (newNode.$parsePrio) {
                    if (newNode.$parsePrio == "001") {
                        newNode.dispatchEvent("DOMNodeInsertedIntoDocument"); //{relatedParent : nodes[j].parentNode}
                        continue;
                    }

                    (nodelist[newNode.$parsePrio] || (prios.push(newNode.$parsePrio)
                      && (nodelist[newNode.$parsePrio] = []))).push(newNode); //for second pass
                }
            }

            amlNode.firstChild = cNodes[0];
            amlNode.lastChild  = cNodes[cL];
        })(amlNode, xmlNode.childNodes);

        if (options.include && rest.length) {
            var index = n.length - 1;
            n.push.apply(n, rest);
            var last = n[index];
            var next = n[index + 1];
            (next.previousSibling = last).nextSibling = next;
            amlNode.lastChild = n[n.length - 1];
        }

        if (options.delay) {
            amlNode.$parseOptions = {
                prios: prios,
                nodelist: nodelist
            };
            return (docFrag || doc);
        }

        //Second pass - Document Insert signalling
        prios.sort();
        var i, j, l, l2;
        for (i = 0, l = prios.length; i < l; i++) {
            nodes = nodelist[prios[i]];
            for (j = 0, l2 = nodes.length; j < l2; j++) {
                nodes[j].dispatchEvent("DOMNodeInsertedIntoDocument"); //{relatedParent : nodes[j].parentNode}
            }
        }

        if (this.$waitQueue[amlNode.$uniqueId]
          && this.$waitQueue[amlNode.$uniqueId].$shouldWait)
            return (docFrag || doc);

        if (options.timeout) {
            $setTimeout(function(){
                _self.$continueParsing(amlNode, options);
            });
        }
        else {
            this.$continueParsing(amlNode, options);
        }

        return (docFrag || doc);
    };

    this.$isPaused = function(amlNode){
        return this.$waitQueue[amlNode.$uniqueId] &&
          this.$waitQueue[amlNode.$uniqueId].$shouldWait > 0;
    }

    this.$addParseState = function(amlNode, options){
        var waitQueue = this.$waitQueue[amlNode.$uniqueId]
            || (this.$waitQueue[amlNode.$uniqueId] = [])
        waitQueue.pushUnique(options);

        return waitQueue;
    }

    this.$pauseParsing = function(amlNode, options){
        var waitQueue = this.$waitQueue[amlNode.$uniqueId];
        if (!waitQueue.$shouldWait) waitQueue.$shouldWait = 0;
        waitQueue.$shouldWait++;
    }

    this.$continueParsing = function(amlNode, options){
        if (!amlNode)
            amlNode = apf.document.documentElement;

        var uId  = amlNode.$uniqueId;
        if (uId in this.$waitQueue) {
            var item = this.$waitQueue[uId];

            if (item.$shouldWait && --item.$shouldWait)
                return false;

            var node = amlNode.parentNode;
            while (node && node.nodeType == 1) {
                if (this.$waitQueue[node.$uniqueId]
                  && this.$waitQueue[node.$uniqueId].$shouldWait)
                    return false;
                node = node.parentNode;
            }

            var parseAmlNode = apf.all[uId];
            delete this.$waitQueue[uId];
            if (parseAmlNode) {
                for (var i = 0; i < item.length; i++)
                    this.$parseState(parseAmlNode, item[i]);
            }

            //@todo Check for shouldWait here?
        }
        else
            this.$parseState(amlNode, options || {});

        delete this.$parseContext;
    }

    this.$parseState = function(amlNode, options) {
        this.$callCount++;

        if (amlNode.$parseOptions) {
            var prios    = amlNode.$parseOptions.prios,
                nodelist = amlNode.$parseOptions.nodelist,
                i, j, l, l2, node;
            delete amlNode.$parseOptions;

            //Second pass - Document Insert signalling
            prios.sort();
            for (i = 0, l = prios.length; i < l; i++) {
                var nodes = nodelist[prios[i]];
                for (j = 0, l2 = nodes.length; j < l2; j++) {
                    if (!(node = nodes[j]).parentNode || node.$amlLoaded) //@todo generalize this using compareDocumentPosition
                        continue;
                    nodes[j].dispatchEvent("DOMNodeInsertedIntoDocument"); //{relatedParent : nodes[j].parentNode}
                }
            }
        }

        //instead of $amlLoaded use something more generic see compareDocumentPosition
        if (!options.ignoreSelf && !amlNode.$amlLoaded)
            amlNode.dispatchEvent("DOMNodeInsertedIntoDocument"); //{relatedParent : nodes[j].parentNode}

        //Recursively signal non prio nodes
        (function _recur(nodes){
            var node, nNodes;
            for (var i = 0, l = nodes.length; i < l; i++) {
                if (!(node = nodes[i]).$amlLoaded) {
                    node.dispatchEvent("DOMNodeInsertedIntoDocument"); //{relatedParent : nodes[j].parentNode}
                }

                //Create children
                if (!node.render && (nNodes = node.childNodes).length)
                    _recur(nNodes);
            }
        })(amlNode.childNodes);

        if (!--this.$callCount && !options.delay)
            apf.queue.empty();

        if (options.callback)
            options.callback.call(amlNode.ownerDocument);
    };

    this.$createNode = function(doc, nodeType, xmlNode, namespaceURI, nodeName, nodeValue){
        var o;

        switch (nodeType) {
            case 1:
                var id, prefix;
                if (xmlNode) {
                    if ((namespaceURI = xmlNode.namespaceURI || apf.ns.xhtml)
                      && !(prefix = doc.$prefixes[namespaceURI])) {
                        doc.$prefixes[prefix = xmlNode.prefix || xmlNode.scopeName || ""] = namespaceURI;
                        doc.$namespaceURIs[namespaceURI] = prefix;

                        if (!doc.namespaceURI && !prefix) {
                            doc.namespaceURI = namespaceURI;
                            doc.prefix       = prefix;
                        }
                    }
                    nodeName = xmlNode.baseName || xmlNode.localName || xmlNode.tagName.split(":").pop();
                }
                else {
                    prefix = doc.$prefixes[namespaceURI] || "";
                }

                

                var els = apf.namespaces[namespaceURI].elements;

                

                o = new (els[nodeName] || els["@default"])(null, nodeName);

                o.prefix       = prefix || "";
                o.namespaceURI = namespaceURI;
                o.tagName      = prefix ? prefix + ":" + nodeName : nodeName;

                if (xmlNode) {
                    if ((id = xmlNode.getAttribute("id")) && !self[id])
                        o.$propHandlers["id"].call(o, o.id = id);

                    //attributes
                    var attr = xmlNode.attributes, n;
                    for (var a, na, i = 0, l = attr.length; i < l; i++) {
                        o.attributes.push(na = new apf.AmlAttr(o,
                            (n = (a = attr[i]).nodeName), a.nodeValue));
                        
                        if (n == "render")
                            o.render = true;
                        else
                        
                        if (n.substr(0, 2) == "on")
                            na.$triggerUpdate();
                    }
                }

                break;
            case 2:
                o = new apf.AmlAttr();
                o.name  = o.nodeName = nodeName;
                if (nodeValue || (nodeValue = xmlNode && xmlNode.nodeValue))
                    o.value = o.nodeValue = nodeValue;

                if (xmlNode) {
                    if (xmlNode.namespaceURI && !(o.prefix = doc.$namespaceURIs[o.namespaceURI = xmlNode.namespaceURI]))
                        doc.$prefixes[o.prefix = xmlNode.prefix || xmlNode.scopeName] = o.namespaceURI;
                }
                else {
                    o.prefix = doc.$prefixes[namespaceURI];
                }

                break;
            case 3:
                if (xmlNode)
                    nodeValue = xmlNode && xmlNode.nodeValue;
                if (!this.preserveWhiteSpace && !(nodeValue || "").trim())
                    return;

                o = new apf.AmlText();
                o.nodeValue = nodeValue || xmlNode && xmlNode.nodeValue;
                break;
            case 7:
                var target = nodeName || xmlNode && xmlNode.nodeName;
                
                o = new apf.aml.processingInstructions[target]();

                o.target = o.nodeName  = target;
                o.data   = o.nodeValue = nodeValue || xmlNode && xmlNode.nodeValue;
                break;
            case 4:
                o = new apf.AmlCDATASection();
                o.nodeValue = nodeValue || xmlNode && xmlNode.nodeValue;
                break;
            case 5: //unsupported
                o = new apf.AmlNode();
                o.nodeType = nodeType;
                break;
            case 6: //unsupported
                o = new apf.AmlNode();
                o.nodeType = nodeType;
                break;
            case 8:
                o = new apf.AmlComment();
                o.nodeValue = nodeValue || xmlNode && xmlNode.nodeValue;
                break;
            case 9:
                o = new apf.AmlDocument();
                o.$domParser = this;
                break;
            case 10: //unsupported
                o = new apf.AmlNode();
                o.nodeType = nodeType;
                break;
            case 11:
                o = new apf.AmlDocumentFragment();
                break;
        }

        o.ownerDocument = doc;
        o.$aml          = xmlNode;

        return o;
    };
})();

/**
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 */
apf.AmlNamespace = function(){
    this.elements = {};
    this.processingInstructions = {};
};

apf.AmlNamespace.prototype = {
    setElement : function(tagName, fConstr){
        return this.elements[tagName] = fConstr;
    },

    setProcessingInstruction : function(target, fConstr){
        this.processingInstructions[target] = fConstr;
    }
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * The parser of the Ajax.org Markup Language. Besides aml this parser takes care
 * of distributing parsing tasks to other parsers like the native html parser and
 * the xsd parser.
 * @parser
 * @private
 *
 * @define include element that loads another aml files.
 * Example:
 * <code>
 *   <a:include src="bindings.aml" />
 * </code>
 * @attribute {String} src the location of the aml file to include in this application.
 *
 */
apf.aml = new apf.AmlNamespace();
apf.setNamespace("http://ajax.org/2005/aml", apf.aml);




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__AMLNODE__ = 1 << 14;



/**
 * All elements inheriting from this {@link term.baseclass baseclass} have Document Object Model (DOM) support. The DOM
 * is the primary method for accessing and manipulating an XML document. This
 * includes HTML documents and AML documents. Every element in the ajax.org
 * markup language can be manipulated using the W3C DOM. This means
 * that every element and attribute you can set in the XML format, can be
 * changed, set, removed, reparented, _e.t.c._ at runtime. This offers a great deal of
 * flexibility. 
 *
 * Well known methods
 * from this specification are: `appendChild`, `removeChild`, `setAttribute`, and
 * `insertBefore`--to name a few. The Ajax.org Platform aims to implement DOM1
 * completely and parts of DOM2. For more information see {@link http://www.w3.org/DOM/} 
 * or {@link http://www.w3schools.com/dom/default.asp}.
 * 
 * #### Example:
 *
 * Here's a basic window using the Ajax.org Markup Language (AML): 
 *
 * ```xml
 *  <a:window id="winExample" title="Example" visible="true">
 *      <a:button id="tstButton" />
 *  </a:window>
 * ```
 *
 * 
 * Using the Document Object Model in JavaScript:
 *
 * ```javascript
 *  //The following line is only there for completeness sake. In fact apf
 *  //automatically adds a reference in javascript called winExample based
 *  //on the id it has.
 *  var winExample = apf.document.getElementById("winExample");
 *  winExample.setAttribute("title", "Example");
 *  winExample.setAttribute("icon", "icoFolder.gif");
 *  winExample.setAttribute("left", "100");
 *
 *  var lblNew = apf.document.createElement("label");
 *  winExample.appendChild(lblNew);
 *  lblNew.setAttribute("caption", "Example");
 *
 *  tstButton.setAttribute("caption", "Click me");
 * ```
 *
 * That would be the same as having the following AML:
 * 
 * ```xml
 *  <a:window id="winExample"
 *    title   = "Example"
 *    icon    = "icoFolder.gif"
 *    left    = "100"
 *    visible = "true">
 *      <a:button id="tstButton" caption="Click me"/>
 *      <a:label caption="Example" />
 *  </a:window>
 * ```
 *
 * #### Remarks
 * Because the W3C DOM is native to all modern browsers the internet is full
 * of tutorials and documentation for this API. If you need more information,
 * it's a good idea to search for tutorials online.
 *
 * @class apf.AmlNode
 * @baseclass
 * @inherits apf.Class
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.5
 */
/**
 * @event DOMNodeInserted Fires when a DOM node is inserted.
 */
/** 
 * @event DOMNodeInsertedIntoDocument Fires when a DOM node is inserted into the document.
 */
/** 
 * @event DOMNodeRemoved Fires when a DOM node is removed.
 */
/** 
 * @event DOMNodeRemovedFromDocument Fires when a DOM node is removed from a document.
 */

apf.AmlNode = function(){
    this.$init(function(){
        /**
         * Nodelist containing all the child nodes of this element.
         */
        this.childNodes = []; //@todo AmlNodeList
    });
};

(function() {

    
    /**
     * Returns a string representation of this object.
     * @returns A string defining the object.
     */
    this.toString = function(){
        if (this.nodeName)
            return "[" + this.nodeName.uCaseFirst() + " Node]";
        
        return "[" + this.localName.uCaseFirst() + " Element Node, <" 
            + (this.prefix ? this.prefix + ":" : "") + this.localName + " "
            + this.attributes.join(" ")
            + " /> : " + (this.name || this.$uniqueId || "") + "]";
    };
    
    
    /**
     * Number specifying the type of node within the document.
     * @type {Number}
     */
    this.$regbase = this.$regbase | apf.__AMLNODE__;
    
    /**
     * The constant for a DOM element node.
     * @type {Number}
     */
    this.NODE_ELEMENT                = 1;
    /**
     * The constant for a DOM attribute node.
     * @type {Number}
     */
    this.NODE_ATTRIBUTE              = 2;
    /**
     * The constant for a DOM text node.
     * @type {Number}
     */
    this.NODE_TEXT                   = 3;
    /**
     * The constant for a DOM cdata section node.
     * @type {Number}
     */
    this.NODE_CDATA_SECTION          = 4;
    /**
     * The constant for a DOM entity reference node.
     * @type {Number}
     */
    this.NODE_ENTITY_REFERENCE       = 5;
    /**
     * The constant for a DOM entity node.
     * @type {Number}
     */
    this.NODE_ENTITY                 = 6;
    /**
     * The constant for a DOM processing instruction node.
     * @type {Number}
     */
    this.NODE_PROCESSING_INSTRUCTION = 7;
    /**
     * The constant for a DOM comment node.
     * @type {Number}
     */
    this.NODE_COMMENT                = 8;
    /**
     * The constant for a DOM document node.
     * @type {Number}
     */
    this.NODE_DOCUMENT               = 9;
    /**
     * The constant for a DOM document type node.
     * @type {Number}
     */
    this.NODE_DOCUMENT_TYPE          = 10;
    /**
     * The constant for a DOM document fragment node.
     * @type {Number}
     */
    this.NODE_DOCUMENT_FRAGMENT      = 11;
    /**
     * The constant for a DOM notation node.
     * @type {Number}
     */
    this.NODE_NOTATION               = 12;
    
    

    /**
     * The document node of this application
     * @type {apf.AmlDocument}
     */
    this.ownerDocument = null;

    /**
     * Returns the value of the current node. 
     * @type {apf.AmlNode}
     */
    this.nodeValue = "";
    
    /**
     * The namespace URI of the node, or `null` if it is unspecified (read-only). 
     *
     * When the node is a document, it returns the XML namespace for the current 
     * document.
     * @type {String}
     */
    this.namespaceURI = "";
    
    /*
     * @todo
     */
    //this.baseURI = alsdjlasdj
    
    /*
     * @todo
     */
    //this.prefix = asdkljahqsdkh
        
    /**
     * 
     * @inheritdoc apf.AmlNode.insertBefore
     * 
     */
    this.appendChild =

    /**
     * Inserts an element before another element in the list of children of this
     * element. If the element was already a child of another element it is
     * removed from that parent before adding it this element.
     *
     * @method insertBefore
     * @param  {apf.AmlNode}  amlNode     The element to insert as child of this element.
     * @param  {apf.AmlNode}  beforeNode  The element which determines the insertion position of the element.
     * @return  {apf.AmlNode}  The inserted node
     */
    this.insertBefore = function(amlNode, beforeNode, noHtmlDomEdit){
        

        if (this.nodeType == this.NODE_DOCUMENT) {
            if (this.childNodes.length) {
                throw new Error(apf.formatErrorString(0, this,
                    "Insertbefore DOM operation",
                    "Only one top level element is allowed in an AML document."));
            }
            else this.documentElement = amlNode; //@todo apf3.0 removal
        }
        
        if (amlNode == beforeNode)
            return amlNode;
        
        if (this == amlNode) {
            throw new Error(apf.formatErrorString(0, this,
                "Insertbefore DOM operation",
                "Cannot append node as a child of itself."));
        }

        if (amlNode.nodeType == this.NODE_DOCUMENT_FRAGMENT) {
            var nodes = amlNode.childNodes.slice(0);
            for (var i = 0, l = nodes.length; i < l; i++) {
                this.insertBefore(nodes[i], beforeNode);
            }
            return amlNode;
        }
        
        var isMoveWithinParent = amlNode.parentNode == this,
            oldParentHtmlNode  = amlNode.$pHtmlNode,
            oldParent          = amlNode.parentNode,
            index              = -1,
            _self              = this;
        
        if (beforeNode) {
            index = this.childNodes.indexOf(beforeNode);
            if (index < 0) {
                

                return false;
            }
        }

        if (!amlNode.ownerDocument)
            amlNode.ownerDocument = this.ownerDocument || apf.ownerDocument;

        if (amlNode.parentNode)
            amlNode.removeNode(isMoveWithinParent, true);//noHtmlDomEdit);
        amlNode.parentNode = this;

        if (beforeNode)
            index = this.childNodes.indexOf(beforeNode);

        if (beforeNode) {
            amlNode.nextSibling = beforeNode;
            amlNode.previousSibling = beforeNode.previousSibling;
            beforeNode.previousSibling = amlNode;
            if (amlNode.previousSibling)
                amlNode.previousSibling.nextSibling = amlNode;
        }

        if (index >= 0) {
            this.childNodes = this.childNodes.slice(0, index).concat(amlNode,
                this.childNodes.slice(index));
        }
        else {
            index = this.childNodes.push(amlNode) - 1;

            amlNode.nextSibling = null;
            if (index > 0) {
                amlNode.previousSibling = this.childNodes[index - 1];
                amlNode.previousSibling.nextSibling = amlNode;
            }
            else {
                amlNode.previousSibling = null;
            }
        }

        this.firstChild = this.childNodes[0];
        this.lastChild  = this.childNodes[this.childNodes.length - 1];

        //@todo fix event struture, fix tree events
        var initialAppend = !amlNode.$amlLoaded;
        function triggerUpdate(){
            amlNode.$pHtmlNode = _self.canHaveChildren ? _self.$int : document.body;

            //@todo this is a hack, a good solution should be found
            if (document.adoptNode && amlNode.$ext && amlNode.$ext.nodeType == 1) {
                var reappendlist = [];
                var iframelist   = apf.getArrayFromNodelist(
                    amlNode.$ext.getElementsByTagName("iframe"));
                if (amlNode.$ext.tagName == "IFRAME")
                    document.adoptNode(amlNode.$ext);
                    
                for (var i = 0; i < iframelist.length; i++) {
                    reappendlist[i] = [
                        iframelist[i].parentNode,
                        iframelist[i].nextSibling,
                        document.adoptNode(iframelist[i]),
                    ]
                }
            }

            var nextNode = beforeNode;
            if (!initialAppend && !noHtmlDomEdit && amlNode.$ext && !amlNode.$coreHtml) {
                nextNode = beforeNode;
                while (nextNode && !(nextNode.$altExt || nextNode.$ext)) {
                    nextNode = nextNode.nextSibling;
                }
                
                amlNode.$pHtmlNode.insertBefore(amlNode.$altExt || amlNode.$ext,
                    nextNode && (nextNode.$altExt || nextNode.$ext) || null);
                    
                for (var i = reappendlist.length - 1; i >= 0; i--) {
                    reappendlist[i][0].insertBefore(
                        reappendlist[i][2],
                        reappendlist[i][1]);
                }
                reappendlist = [];
            }
            
            //Signal node and all it's ancestors
            amlNode.dispatchEvent("DOMNodeInserted", {
                $beforeNode         : beforeNode,
                relatedNode         : _self,
                $isMoveWithinParent : isMoveWithinParent,
                $oldParentHtmlNode  : oldParentHtmlNode,
                $oldParent          : oldParent,
                bubbles             : true
            });
            
            if (initialAppend && !noHtmlDomEdit && beforeNode && amlNode.$ext && !amlNode.$coreHtml) {
                nextNode = beforeNode;
                while (nextNode && !(nextNode.$altExt || nextNode.$ext)) {
                    nextNode = nextNode.nextSibling;
                }
                
                amlNode.$pHtmlNode.insertBefore(amlNode.$altExt || amlNode.$ext,
                    nextNode && (nextNode.$altExt || nextNode.$ext) || null);
                
                for (var i = reappendlist.length - 1; i >= 0; i--) {
                    reappendlist[i][0].insertBefore(
                        reappendlist[i][2],
                        reappendlist[i][1]);
                }
            }
        }

        var doc = this.nodeType == this.NODE_DOCUMENT ? this : this.ownerDocument;
        if (!doc || doc.$domParser.$isPaused(this))
            return amlNode;

        // Don't update the tree if this is a doc fragment or if this element is not inited yet
        if (this.nodeType == this.NODE_DOCUMENT_FRAGMENT || !this.$amlLoaded)
            return amlNode; 

        //@todo review this...
        if (initialAppend && !amlNode.render) { // && (nNodes = node.childNodes).length ??
            (this.ownerDocument || this).$domParser.$continueParsing(amlNode, {delay: true});
        }

        triggerUpdate();
        return amlNode;
    };

    /**
     * Removes this element from the document hierarchy. Call-chaining is
     * supported.
     *
     */
    this.removeNode = function(doOnlyAdmin, noHtmlDomEdit){
        

        if (!this.parentNode || !this.parentNode.childNodes)
            return this;

        

        this.parentNode.childNodes.remove(this);

        //If we're not loaded yet, just remove us from the aml to be parsed
        if (this.$amlLoaded && !apf.isDestroying) {
            //this.parentNode.$aml.removeChild(this.$aml);

            this.dispatchEvent("DOMNodeRemoved", {
                relatedNode  : this.parentNode,
                bubbles      : true,
                $doOnlyAdmin : doOnlyAdmin
            });

            if (!noHtmlDomEdit && !doOnlyAdmin && this.$ext && this.$ext.parentNode) {
                this.$ext.parentNode.removeChild(this.$ext);
                //delete this.$ext; //WTF???
            }
        }

        if (this.parentNode.firstChild == this)
            this.parentNode.firstChild = this.nextSibling;
        if (this.parentNode.lastChild == this)
            this.parentNode.lastChild = this.previousSibling;

        if (this.nextSibling)
            this.nextSibling.previousSibling = this.previousSibling;
        if (this.previousSibling)
            this.previousSibling.nextSibling = this.nextSibling;

        this.$pHtmlNode      =
        this.parentNode      =
        this.previousSibling =
        this.nextSibling     = null;

        return this;
    };

    /**
     * Removes a child from the node list of this element. Call-chaining is
     * supported.
     * @param {apf.AmlNode} childNode The child node to remove
     */
    this.removeChild = function(childNode) {
        

        childNode.removeNode();
        return this;
    };
    
    //@todo
    this.replaceChild = function(){};

    /**
     * Clones this element, creating an exact copy of it--but does not insert
     * it in the document hierarchy.
     *
     * @param {Boolean} deep Specifies whether the elements are cloned recursively.
     * @return {apf.AmlNode} The cloned element.
     */
    this.cloneNode = function(deep){
        if (deep && this.nodeType == 1) {
            return this.ownerDocument.$domParser.parseFromXml(this, {
                doc   : this.ownerDocument,
                delay : true
            }).childNodes[0];
        }
        else {
            return this.ownerDocument.$domParser.$createNode(
                this.ownerDocument, this.nodeType, this);
        }
    };
    
    //@todo
    this.canDispatch = function(namespaceURI, type){};
    
    //@todo
    this.compareDocumentPosition = function(otherNode){
        /*
            DOCUMENT_POSITION_DISCONNECTED = 0x01;
            DOCUMENT_POSITION_PRECEDING = 0x02;
            DOCUMENT_POSITION_FOLLOWING = 0x04;
            DOCUMENT_POSITION_CONTAINS = 0x08;
            DOCUMENT_POSITION_CONTAINED_BY = 0x10;
        */
    };
    
    this.hasAttributes = function(){
        return this.attributes && this.attributes.length;
    };
    
    this.hasChildNodes = function(){
        return this.childNodes && this.childNodes.length;
    };
    
    this.isDefaultNamespace = function(namespaceURI){
        if (node.nodeType == 1) {
            if (!this.prefix)
                return this.namespaceURI == namespaceURI;
            
            //@todo Loop through attributes here
        }
        
        var node = this.parentNode || this.ownerElement;
        return node && node.isDefaultNamespace(namespaceURI);
    };
    
    this.lookupNamespaceURI = function(prefix){
        if (node.nodeType == 1) {
            if (this.namespaceURI && prefix == this.prefix)
                return this.namespaceURI ;
                
            //@todo Loop through attributes here
        }
        
        var node = this.parentNode || this.ownerElement;
        return node && node.lookupNamespaceURI(prefix);
    };
    
    this.lookupPrefix = function(namespaceURI){
        if (this.nodeType == 1) {
            if (namespaceURI == this.namespaceURI && this.prefix)
                return this.prefix;
            
            //@todo Loop through attributes here
        }
        
        var node = this.parentNode || this.ownerElement;
        return node && node.lookupPrefix(namespaceURI);    
    };
    
    this.normalize = function(){};
    
    // *** Xpath support *** //

    /**
     * Queries the AML DOM using the W3C xPath query language and returns a node
     * list. This is not an official API call, but can be useful in certain cases.
     *
     * @param {String}  sExpr          The xpath expression to query the AML DOM tree with.
     * @param {apf.AmlNode} [contextNode]  The element that serves as the starting point of the search. Defaults to this element.
     * @returns {NodeList} List of found nodes.
     */
    this.selectNodes = function(sExpr, contextNode){
        if (!apf) return;
        
        if (!apf.XPath)
            apf.runXpath();
        return apf.XPath.selectNodes(sExpr,
            contextNode || (this.nodeType == 9 ? this.documentElement : this));
    };

    /**
     * Queries the AML dom using the W3C xPath query language and returns a single
     * node. This is not an official API call, but can be useful in certain cases.
     * 
     * @param {String}  sExpr          The xpath expression to query the AML DOM tree with.
     * @param {apf.AmlNode} [contextNode]  The element that serves as the starting point of the search. Defaults to this element.
     * @returns {apf.AmlNode} The first node that matches the query.
     */
    this.selectSingleNode  = function(sExpr, contextNode){
        if (!apf) return;
        
        if (!apf.XPath)
            apf.runXpath();
        return apf.XPath.selectNodes(sExpr,
            contextNode || (this.nodeType == 9 ? this.documentElement : this))[0];
    };
    
    /*this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        
    }, true);*/
}).call(apf.AmlNode.prototype = new apf.Class());





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

/**
 * Represents a single element within an AML node.
 *
 * @class apf.AmlElement
 * @baseclass
 * @inherits apf.AmlNode
 */


apf.AmlElement = function(struct, tagName){
    var $init = this.$init;
    this.$init = function(tagName, nodeFunc, struct){
        this.$supportedProperties = this.$supportedProperties.slice();
        
        var prop, p, q;
        p = this.$propHandlers;
        q = this.$propHandlers = {};
        for (prop in p)
            q[prop] = p[prop];
        
        p = this.$booleanProperties;
        q = this.$booleanProperties = {};
        for (prop in p)
            q[prop] = p[prop];
        
        return $init.call(this, tagName, nodeFunc, struct);
    };
    
    this.$init(function(tagName, nodeFunc, struct){
        this.$events            = {};
        this.$inheritProperties = {};
        
        /*
         * A node list containing all the attributes. This is implemented according to the
         * W3C specification.
         * 
         * For more information, see [[apf.AmlElement.getAttribute]] and [[apf.AmlElement.setAttribute]].
         *
         * #### Example
         * 
         * ```javascript
         *  for (var i = 0; i < obj.attributes.length; i++) {
         *      alert(obj.attributes.item(i));
         *  }
         * ```
         * @type {apf.AmlNamedNodeMap}
         */
        this.attributes = new apf.AmlNamedNodeMap(this); //@todo apf3.0 move to init?
        
        /**
         * Defines the purpose of this element. Possible values include:
         * - `apf.NODE_VISIBLE`:  This element has a GUI representation
         * - `apf.NODE_HIDDEN`:   This element does not display a GUI
         * @type {Number}
         */
        this.nodeFunc = nodeFunc;
        
        /**
         * The local name of this element
         * @type {String}
         */
        this.localName = tagName; //@todo
        
        //Parse struct to create attributes and child nodes
        if (struct) {
            var nodes, prop, i, l, attr;
            if (struct.childNodes) {
                nodes = struct.childNodes;
                delete struct.childNodes; //why delete?
            }
            
            //Attributes
            for (prop in struct){ 
                if (prop == "htmlNode") continue;
                
                attr = new apf.AmlAttr(this, prop, struct[prop]);
                
                //These exceptions should be generalized
                if (prop == "id")
                    this.$propHandlers["id"].call(this, this.id = struct.id);
                else if (prop == "hotkey")
                    this.$propHandlers["hotkey"].call(this, this.hotkey = struct.hotkey);
                else if (prop.substr(0, 2) == "on")
                    attr.$triggerUpdate();

                this.attributes.push(attr);
            }
            
            if (!this.ownerDocument) {
                this.ownerDocument = apf.document;
                this.prefix        = "a";
                this.namespaceURI  = apf.ns.aml;
                this.tagName       = tagName;
            }
            
            if (nodes) {
                this.childNodes = nodes;

                for (i = 0, l = nodes.length; i < l; i++) {
                    nodes[i].nextSibling = nodes[i + 1] || null;
                    nodes[i].previousSibling = nodes[i - 1] || null;
                    nodes[i].parentNode = this;
                }
                this.firstChild = nodes[0] || null;
                this.lastChild  = nodes[nodes.length - 1] || null;
            }

            //Temp hack
            this.$aml = apf.$emptyNode || (apf.$emptyNode = apf.getXml("<empty />"));
        }
    });
    
    if (tagName) //of typeof is not function and not true
        $init.call(this, tagName, apf.NODE_HIDDEN, struct);
};

(function(){
    /**
     * A number specifying the type of node within the document.
     * @type {Number}
     */
    this.nodeType = this.NODE_ELEMENT;
    this.canHaveChildren = true;
    
    this.$propHandlers = {
        /**
         * @attribute {String} id The identifier of this element. When set, this
         * identifier is the name of the variable in JavaScript to access this
         * element directly. This identifier is also the way to get a reference to
         * this element using `apf.document.getElementById()`.
         * 
         * #### Example
         *
         * ```xml
         *  <a:bar id="barExample" />
         *  <a:script>
         *      alert(barExample);
         *  </a:script>
         * ```
         */
        "id": function(value){
            
            
            if (this.name == value || !value)
                return;
    
            if (self[this.name] == this) {
                self[this.name] = null;
                
                apf.nameserver.remove(this.localName, this);
                apf.nameserver.remove("all", this);
                
            }
    
            
    
            if (!self[value] || !self[value].hasFeature) {
                try {
                    self[value] = this;
                }
                catch(ex) {
                    
                }
            }
            
            
            //@todo dispatch event for new name creation.
            //@todo old name disposal
            
            apf.nameserver.register(this.localName, value, this)
            apf.nameserver.register("all", value, this)
            
            
            this.name = value;
        }
    };
    
    this.$booleanProperties   = {};
    this.$inheritProperties   = {};
    this.$supportedProperties = [];
    
    /**
     * Returns a list of elements with the given tag name.
     *
     * The subtree below the specified element is searched, excluding the
     * element itself.
     *
     * @param  {String}  tagName  The tag name to look for. The special string "*" represents any tag name.
     * @param {Boolean} [norecur] If specified, defines whether or not to check recursively
     * @return  {NodeList}  Contains any nodes matching the search string
     */
    this.getElementsByTagName = function(tagName, norecur){
        tagName = tagName.toLowerCase();
        var node, i, l,
            nodes  = this.childNodes,
            result = [];
        for (i = 0, l = nodes.length; i < l; i++) {
            if ((node = nodes[i]).nodeType != 1)
                continue;
            
            if (node.tagName == tagName || tagName == "*")
                result.push(node);

            if (!norecur && node.nodeType == 1)
                result = result.concat(node.getElementsByTagName(tagName));
        }
        
        return result;
    };

    /**
     * Returns a list of elements with the given tag name and the specified namespace URI.
     *
     * The subtree below the specified element is searched, excluding the
     * element itself.
     *
     * @param  {String}  namespaceURI  The namespace URI name to look for.
     * @param  {String}  localName  The tag name to look for. The special string "*" represents any tag name.
     * @param {Boolean} [norecur] If specified, defines whether or not to check recursively
     * @return  {NodeList}  Contains any nodes matching the search string
     */    
    this.getElementsByTagNameNS = function(namespaceURI, localName, norecur){
        localName = localName.toLowerCase();
        var node, i, l,
            nodes  = this.childNodes,
            result = [];
        for (i = 0, l = nodes.length; i < l; i++) {
            if ((node = nodes[i]).nodeType != 1)
                continue;

            if (node.namespaceURI == namespaceURI && (node.localName == localName || localName == "*"))
                result.push(node);

            if (!norecur && node.nodeType == 1)
                result = result.concat(node.getElementsByTagNameNS(namespaceURI, localName));
        }
        
        return result;
    };

    /**
     * Sets an attribute on this element.
     * @chainable
     * @param {String} name The name of the attribute to which the value is set
     * @param {String} value The new value of the attribute.
     * @param {Boolean} [noTrigger] If specified, does not emit events 
     * [[apf.AmlNode@DOMNodeInsertedIntoDocument]] and [[apf.AmlNode@DOMNodeInserted]].
     */
    this.setAttribute = function(name, value, noTrigger) {
        name = name.toLowerCase();
        
        var a = this.attributes.getNamedItem(name);
        if (!a) {
            this.attributes.push(a = new apf.AmlAttr(this, name, value));
        
            if (!this.$amlLoaded && name != "id" && name != "hotkey")
                return;
            
            if (noTrigger)
                a.$setValue(value);
            else {
                //@todo apf3.0 domattr
                a.dispatchEvent("DOMNodeInsertedIntoDocument", {
                    relatedNode : this
                });
                
                //@todo apf3.0 domattr
                a.dispatchEvent("DOMNodeInserted", {
                    relatedNode : this,
                    bubbles     : true
                });
            }

            return;
        }

        var oldValue = a.nodeValue;
        a.$setValue(value);
        
        if (noTrigger || !this.$amlLoaded)
            return;
        
        //@todo apf3.0 domattr
        a.$triggerUpdate(null, oldValue);
    };
    
    //@todo apf3.0 domattr
    this.setAttributeNode = function(attrNode){
        this.attributes.setNamedItem(attrNode);
    };
    
    this.setAttributeNS = function(namespaceURI, name, value){
        return this.setAttribute(name, value);
    };
    
    //@todo apf3.0 domattr
    this.hasAttribute = function(name){
        return this.getAttributeNode(name) ? true : false;
    };
    
    //@todo
    this.hasAttributeNS = function(namespaceURI, name){
        return this.hasAttribute(name);
    };
    
    /**
     * Removes an attribute from this element. 
     * @chainable
     * @param {String} name The name of the attribute to remove.
     * @returns {apf.AmlElement} The modified element.
     */
    this.removeAttribute = function(name){ //@todo apf3.0 domattr
        this.attributes.removeNamedItem(name);
        return this;
    };
    
    //@todo apf3.0 domattr
    this.removeAttributeNS = function(namespaceURI, name){
        return this.removeAttribute(name);
    };
    
    //@todo apf3.0 domattr
    this.removeAttributeNode = function(attrNode){
        this.attributes.removeNamedItem(attrNode.name); //@todo this should probably be slightly different.
    };

    /**
     * Retrieves the value of an attribute of this element.
     *
     * @param  {String}  name       The name of the attribute for which to return the value.
     * @param  {Boolean} [inherited] if specified, takes into consideration that the attribute is inherited
     * @return {String} The value of the attribute, or `null` if none was found with the name specified.
     */
    this.getAttribute = function(name, inherited){
        var item = this.attributes.getNamedItem(name);
        return item ? (inherited 
            ? item.inheritedValue || item.nodeValue 
            : item.nodeValue) : null;
    };
    
    /**
     * Retrieves the attribute node for a given name
     *
     * @param {String} name The name of the attribute to find.
     * @return {apf.AmlNode} The attribute node, or `null` if none was found with the name specified.
     */
    this.getAttributeNode = function(name){
        return this.attributes.getNamedItem(name);
    };

    this.getBoundingClientRect = function(){
        return new apf.AmlTextRectangle(this);
    };
    
    //@todo
    this.querySelector = function(){
        // here we should use: http://code.google.com/p/css2xpath/source/browse/trunk/src/css2xpath.js
    };
    
    //@todo
    this.querySelectorAll = function(){
        // here we should use: http://code.google.com/p/css2xpath/source/browse/trunk/src/css2xpath.js
    };
    
    //@todo
    this.scrollIntoView = function(){
        
    };
    
    /**
     * Replaces the child AML elements with new AML.
     * @param {Mixed}       amlDefNode  The AML to be loaded. This can be a string or a parsed piece of XML.
     * @param {HTMLElement} oInt        The HTML parent of the created AML elements.
     */
    this.replaceMarkup = function(amlDefNode, options) {
        

        if (!options)
            options = {};

        if (!options.$intAML)
            options.$intAML = this.$aml;
        if (!options.$int)
            options.$int = this.$int;
        options.clear = true;
        
        //Remove All the childNodes
        for (var i = this.childNodes.length - 1; i >= 0; i--) {
            var oItem = this.childNodes[i];
            /*var nodes = oItem.childNodes;
            for (var k = 0; k < nodes.length; k++)
                if (nodes[k].destroy)
                    nodes[k].destroy(true);

            if (oItem.$aml && oItem.$aml.parentNode)
                oItem.$aml.parentNode.removeChild(oItem.$aml);*/

            if (oItem.destroy)
                oItem.destroy(true);

            if (oItem.$ext != this.$int)
                apf.destroyHtmlNode(oItem.$ext);
        }
        
        this.childNodes.length = 0;
        
        if(options.noLoadingMsg !== false)
            this.$int.innerHTML = "<div class='loading'>loading...</div>";

        //Do an insertMarkup
        this.insertMarkup(amlDefNode, options);
    };

    /**
     * Inserts new AML into this element.
     * @param {Mixed}       amlDefNode  The AML to be loaded. This can be a string or a parsed piece of XML.
     * @param {Object}      options     Additional options to pass. It can include the following properties:
     *                                  - callback ([[Function]]): A function to call once the insertion completes.
     *                                  - clear ([[Boolean]]): If set, the AML has the attribute "clear" attached to it
     */
    this.insertMarkup = function(amlDefNode, options){
        

        

        var _self   = this;
        var include = new apf.XiInclude();
        
        if (amlDefNode.trim().charAt(0) == "<")
            amlDefNode = apf.getXml(amlDefNode);
        
        include.setAttribute("href", amlDefNode);
        if (options && options.clear)
            include.setAttribute("clear", true);
        include.options  = options;
        include.callback = function(e){
            _self.dispatchEvent("afteramlinserted", {src: amlDefNode});
            options && options.callback && options.callback(e);
            setTimeout(function(){
                include.destroy(true, true);
            });
        };
        this.appendChild(include);
    };
    
    //@todo prefix only needs on top element
    this.serialize = function(shallow){
        if (shallow || !this.firstChild) {
            return "<" 
                + (this.prefix 
                  ? this.prefix + ":" + this.localName + " xmlns:" 
                    + this.prefix + "=\"" + this.namespaceURI + "\""
                  : this.localName) + (this.attributes && this.attributes.length ? " " : "")
                + (this.attributes && this.attributes.join(" ") || "")
                + "/>";
        }
        else {
            var str = ["<" 
                + (this.prefix 
                  ? this.prefix + ":" + this.localName + " xmlns:" 
                    + this.prefix + "=\"" + this.namespaceURI + "\""
                  : this.localName) + (this.attributes && this.attributes.length ? " " : "")
                + (this.attributes && this.attributes.join(" ") || "")
                + ">"];
            
            for (var i = this.firstChild; i; i = i.nextSibling)
                str.push(i.serialize());
            
            return str.join("") + "</" + (this.prefix ? this.prefix 
                + ":" + this.localName : this.localName) + ">";
        }
    };
    
    this.$setInheritedAttribute = function(prop){
        var value, node = this, isInherit = false;
        
        value = node.getAttribute(prop);
        if (!value) {
            node = node.parentNode;
            
            //Second argument fetches special inheritance value, if any
            while (node && node.nodeType == 1 && !(value = node.getAttribute(prop, true))) {
                node = node.parentNode;
            }
            
            isInherit = true;
        }
        
        if (!value && apf.config && prop)
            value = apf.config[prop];
    
        if (value) {
            
            //Remove any bounds if relevant
            this.$clearDynamicProperty(prop);
            
        }
        
        if (isInherit)
            this.$inheritProperties[prop] = 2;
        
        if (value) {
            
            if (typeof value == "string" 
              && (value.indexOf("{") > -1 || value.indexOf("[") > -1)) {
                this.$setDynamicProperty(prop, value);
            }
            else 
            
                this.setProperty(prop, value, false, false, 2);
        }
        
        return value;
    };
    
    //@todo in proper W3C implementation this needs to change
    //@todo this won't work with a combo of remove/append
    this.addEventListener("DOMNodeInserted", function(e){
        if (e.currentTarget != this || e.$isMoveWithinParent || !e.$oldParent)
            return;

        //Check inherited attributes for reparenting
        /*
            States:
                    -1 Set
             undefined Pass through
                     2 Inherited
                     3 Semi-inherited
                    10 Dynamic property
        */
        var vOld, vNew;
        var aci = apf.config.$inheritProperties;
        for (var prop in aci) {
            vOld = apf.getInheritedAttribute(e.$oldParent, prop);
            vNew = apf.getInheritedAttribute(this.parentNode, prop);
            
            //Property has changed, lets recursively set it on inherited nodes
            if (vOld != vNew) {
                //@todo code duplication from class.js
                (function recur(nodes) {
                    var i, l, node, n;
                    for (i = 0, l = nodes.length; i < l; i++) {
                        node = nodes[i];
                        if (node.nodeType != 1 && node.nodeType != 7)
                            continue;

                        //Pass through
                        n = node.$inheritProperties[prop];
                        if (aci[prop] == 1 && !n)
                            recur(node.childNodes);
                        
                        //Set inherited property
                        //@todo why are dynamic properties overwritten??
                        else if(!(n < 0)) {//Will also pass through undefined - but why??? @todo seems inefficient
                            if (n == 3) {
                                var sameValue = node[prop];
                                node[prop] = null;
                            }
                            node.setProperty(prop, n != 3
                                ? vNew
                                : sameValue, false, false, n); //This is recursive already
                        }
                    }
                })([this]);
            }
        }
    });
    
    this.$handlePropSet = function(prop, value, force){
        if (this.$booleanProperties[prop])
            value = apf.isTrue(value);

        

        this[prop] = value;

        var handler;
        return (handler = this.$propHandlers && this.$propHandlers[prop]
          || this.nodeFunc == apf.NODE_VISIBLE && apf.GuiElement && apf.GuiElement.propHandlers[prop] || null)
          && handler.call(this, value, prop, force);
    };
    
    //var aci = apf.config.$inheritProperties; << UNUSED
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var a, i, l, attr = this.attributes;

        

        
        
        

        //Set all attributes
        for (i = 0, l = attr.length; i < l; i++) {
            attr[i].dispatchEvent("DOMNodeInsertedIntoDocument");
        }
    }, true);
    
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        this.$amlLoaded = true;
    });
}).call(apf.AmlElement.prototype = new apf.AmlNode());





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


//@todo apf3.0 The functions seem to not set nodeValue...
apf.AmlCharacterData = function(){
    this.data = "";
    this.length = 0;
    
    this.$init(true);
    
    this.appendData = function(sValue){
        this.dispatchEvent("DOMCharacterDataModified", {
            value : sValue
        });
    };
    
    this.deleteData = function(nOffset, nCount){
        this.dispatchEvent("DOMCharacterDataModified", {
            offset: nOffset,
            count : nCount
        });
    };
    
    this.insertData = function(nOffset, nCount){
        this.dispatchEvent("DOMCharacterDataModified", {
            offset: nOffset,
            count : nCount
        });
    };
    
    this.replaceData = function(nOffset, nCount, sValue){
        this.dispatchEvent("DOMCharacterDataModified", {
            offset: nOffset,
            count : nCount,
            value : sValue
        });
    };
    
    this.substringData = function(nOffset, nCount){};
}
apf.AmlCharacterData.prototype = new apf.AmlNode();




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlText = function(isPrototype){
    this.$init(isPrototype);
};

(function(){
    this.nodeType = this.NODE_TEXT;
    this.nodeName = "#text";
    
    this.serialize = function(){
        return apf.escapeXML(this.nodeValue);
    };
    
    
    
    //@todo think about using this.replaceData();
    this.$setValue = function(value){
        //if (!this.$amlLoaded)
            //return;
        
        this.dispatchEvent("DOMCharacterDataModified", {
            bubbles   : true,
            prevValue : this.nodeValue,
            newValue  : this.nodeValue = value
        });
        
        if (this.$amlLoaded && this.$ext)
            this.$ext.nodeValue = value;
    }

    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var pHtmlNode;
        if (!(pHtmlNode = this.parentNode.$int) || this.parentNode.hasFeature(apf.__CHILDVALUE__)) 
            return;

        this.$amlLoaded = true;
        
        var nodeValue = this.nodeValue;

        //@todo optimize for inside elements?
        if (apf.config.liveText && !this.parentNode.hasFeature(apf.__CHILDVALUE__) 
          && (nodeValue.indexOf("{") > -1 || nodeValue.indexOf("[") > -1)) {
            
            //Convert to live markup pi
            this.$supportedProperties = [];
            this.$propHandlers        = {};
            this.$booleanProperties   = {};
            this.$inheritProperties   = {};
            
            this.$propHandlers["calcdata"] = apf.LiveMarkupPi.prototype.$propHandlers["calcdata"];
            
            this.$setInheritedAttribute = apf.AmlElement.prototype.$setInheritedAttribute;
            
            this.implement(apf.StandardBinding);
            
            
            pHtmlNode.appendChild(this.$ext = document.createElement("span"));
            this.$setDynamicProperty("calcdata", this.nodeValue);
            
            return;
        }

        if (apf.hasTextNodeWhiteSpaceBug) {
            var nodeValue = nodeValue.replace(/[\t\n\r ]+/g, " ");

            if (nodeValue && nodeValue != " ")
                this.$ext = pHtmlNode.appendChild(
                  pHtmlNode.ownerDocument.createTextNode(nodeValue));
        }
        else
            this.$ext = pHtmlNode.appendChild(
              pHtmlNode.ownerDocument.createTextNode(nodeValue));
    }, true);
}).call(apf.AmlText.prototype = new apf.AmlCharacterData());





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlAttr = function(ownerElement, name, value){
    this.$init();
    
    if (ownerElement) {
        this.ownerElement  = ownerElement;
        this.ownerDocument = ownerElement.ownerDocument;
    }
    
    this.nodeName  = this.name  = name;
    this.nodeValue = this.value = value;
};

(function(){
    this.nodeType  = this.NODE_ATTRIBUTE;
    
    this.MODIFICATION = 1;
    this.ADDITION     = 2;
    this.REMOVAL      = 3;
    
    this.serialize = 
    this.toString  = function(){
        return this.name + "=\"" + apf.escapeXML(String(this.value)) + "\"";
    };
    
    
    
    this.$setValue = function(value){
        this.nodeValue = this.value = value;
        this.specified = true;

        //@todo apf3.0 domattr
        this.ownerElement.dispatchEvent("DOMAttrModified", {
            relatedNode : this,
            attrChange  : this.MODIFICATION,
            attrName    : this.name,
            newValue    : value,
            prevValue   : this.$lastValue || "",
            bubbles     : true
        });
        
        this.$lastValue = value;
    };
    
    this.$triggerUpdate = function(e, oldValue){
        var name  = this.name,
            value = this.value || this.nodeValue,
            host  = this.ownerElement,
            isEvent = name.substr(0, 2) == "on";

        if (!this.specified) {
            //@todo This should be generalized
            if (isEvent && this.$lastValue == value
              || name == "id" && host.id) {
                this.specified = true;
                return;
            }
        }

        if (isEvent) {
            if (host.$events[name])
                host.removeEventListener(name.substr(2), host.$events[name]);
            if (value)
                host.addEventListener(name, (host.$events[name] = 
                  (typeof value == "string"
                    ? 
                      new Function('event', value)
                      
                    : value)));
            return;
        }
        
        if (this.specified)
            host.$clearDynamicProperty(name);
        
        if (typeof value == "string" && (host.$attrExcludePropBind[name] || 
          (value.indexOf("{") > -1 || value.indexOf("[") > -1)))
            host.$setDynamicProperty(name, value);
        else
        
        {
            host.setProperty(name, value); //@todo apf3.0 is this a lot slower?
        }
        //host.$handlePropSet(name, value);

        if (this.specified) {
            //@todo apf3.0 domattr - slow?
            host.dispatchEvent("DOMAttrModified", { //@todo this is not good, node might not be specified at init
                relatedNode : this,
                attrChange  : this.MODIFICATION,
                attrName    : name,
                newValue    : value,
                prevValue   : this.$lastValue || "",
                bubbles     : true
            });
        }
        else this.specified = true;
            
        this.$lastValue = value;
    };
    
    //@todo apf3.0 domattr
    this.addEventListener("DOMNodeInsertedIntoDocument", this.$triggerUpdate);
}).call(apf.AmlAttr.prototype = new apf.AmlNode());




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlCDATASection = function(isPrototype){
    this.nodeType = this.NODE_CDATA_SECTION;
    this.nodeName = "#cdata-section";
    
    this.$init(isPrototype);
};

apf.AmlCDATASection.prototype = new apf.AmlText(true);
apf.AmlCDATASection.prototype.serialize = function(){
    return "<![CDATA[" + this.nodeValue + "]]>";
};




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlComment = function(isPrototype){
    this.nodeType = this.NODE_COMMENT;
    this.nodeName = "#comment";
    
    this.$init(isPrototype);
};

(function(){
    this.serialize = function(){
        return "<!--" + this.nodeValue + "-->";
    };
    
    this.$setValue = function(value){
        this.dispatchEvent("DOMCharacterDataModified", {
            bubbles   : true,
            newValue  : value,
            prevValue : this.nodeValue
        });
    }
}).call(apf.AmlComment.prototype = new apf.AmlCharacterData());




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlConfiguration = function(isPrototype){
    this.parameterNames = [];

    this.$init(isPrototype);
};

(function(){
    this.setParameter = this.setProperty;
    
    this.getParameter = this.getProperty;
    
    this.canSetParameter = function(name, value){ //@todo for value
        return this.parameterNames.indexOf(name) > -1;
    };
}).call(apf.AmlConfiguration.prototype = new apf.Class());




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * The AML document. This is the root of the DOM tree and has a nodeType with 
 * value 9 (`apf.NODE_DOCUMENT`). 
 *
 * @class apf.AmlDocument
 * @inherits apf.AmlNode
 * @inherits apf.Class
 * @default_private 
 * @see apf.AmlDom
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 */
apf.AmlDocument = function(){
    this.$prefixes      = {};
    this.$namespaceURIs = {};
    
    this.domConfig      = new apf.AmlConfiguration();
    
    
    this.$init();
};

(function() {
    /**
     * The type of node within the document.
     * @type {Number}
     */
    this.nodeType   = this.NODE_DOCUMENT;
    this.nodeFunc   = apf.NODE_HIDDEN;
    this.nodeName   = "#document";
    
    this.$amlLoaded = true;
    
    this.activeElement   = null; //@todo alias of window.foccussed;
    this.doctype         = null;
    this.domConfig       = null;
    this.implementation  = null;
    this.characterSet    = apf.characterSet;
    
    /**
     * The root element node of the AML application. This is an element with
     * the tagName `'application'`. This is similar to the `'html'` element for regular HTML.
     * @type {apf.AmlNode}
     */
    this.documentElement = null;
    
    /**
     * Gets a AML element based on its id.
     * @param {String} id The id of the AML element to return.
     * @return {apf.AmlElement} The AML element with the id specified.
     */
    this.getElementById = function(id){
        return self[id];
    };

    /**
     * Returns a list of elements with the given tag name.
     *
     * The subtree below the [[apf.AmlDocument.documentElement]] is searched, excluding the
     * element itself.
     *
     * @param  {String}  tagName  The tag name to look for. The special string "*" represents any tag name.
     * @return  {NodeList}  Contains any nodes matching the search string
     */ 
    this.getElementsByTagName = function(tagName){
        var docEl, res = (docEl = this.documentElement)
            .getElementsByTagName(tagName);

        if (tagName == "*" || docEl.tagName == tagName)
            res.unshift(docEl);
        return res;
    };

    /**
     * Returns a list of elements with the given tag name and the specified namespace URI.
     *
     * The subtree below the [[apf.AmlDocument.documentElement]] is searched, excluding the
     * element itself.
     *
     * @param  {String}  namespaceURI  The namespace URI name to look for.
     * @param  {String}  tagName  The tag name to look for. The special string "*" represents any tag name.
     * @return  {NodeList}  Contains any nodes matching the search string
     */ 
    this.getElementsByTagNameNS = function(nameSpaceURI, tagName){
        var docEl,
            res = (docEl = this.documentElement)
                .getElementsByTagNameNS(nameSpaceURI, tagName);

        if (tagName == "*" || docEl.tagName == tagName && docEl.namespaceURI == nameSpaceURI)
            res.unshift(docEl);
        return res;
    };

    /**
     * Creates a new AML element.
     *
     * @param {Mixed} qualifiedName Information about the new node to create. Possible values include:
     *                              - [[String]]:     The tag name of the new element to create
     *                              - [[String]]:    The AML definition for a single or multiple elemnts
     *                              - [[XMLElement]]: The AML definition for a single or multiple elements
     * @return {apf.AmlElement} The created AML element
     */
    this.createElement = function(qualifiedName){
        return this.$domParser.$createNode(this, this.NODE_ELEMENT, null,
            this.namespaceURI, qualifiedName);
    };

    /**
     * Creates a new AML element within the given namespace.
     *
     * @param  {String}  namespaceURI  The namespace URI name to use
     * @param {Mixed} qualifiedName Information about the new node to create. Possible values include:
     *                              - [[String]]:     The tag name of the new element to create
     *                              - [[String]]:     The AML definition for a single or multiple elemnts
     *                              - [[XMLElement]]: The AML definition for a single or multiple elements
     * @return {apf.AmlElement} The created AML element
     */        
    this.createElementNS = function(namespaceURI, qualifiedName){
        return this.$domParser.$createNode(this, this.NODE_ELEMENT, null,
            namespaceURI, qualifiedName);
    };

    /**
     * Creates a copy of a node from an external document that can be inserted into the current document.
     *
     * @param  {apf.AmlNode}  node  The node to import and copy
     * @param {Boolean} [deep]      Indicates whether the descendants of the imported node should also be imported
     * @return {apf.AmlNode} The imported node
     */     
    this.importNode = function(node, deep){
        if (deep && node.nodeType == 1) {
            return this.$domParser.parseFromXml(node, {
                doc   : this,
                delay : true
            }).childNodes[0];
        }
        else {
            return this.$domParser.$createNode(this, node.nodeType, node);
        }
    };
    
    /**
     * Creates and returns a new attribute node.
     *
     * @param  {String}  nodeName  The name of the attribute
     * @return {apf.AmlNode} The attribute node
     */ 
    this.createAttribute = function(nodeName){
        return this.$domParser.$createNode(this, this.NODE_ATTRIBUTE, null,
            this.nameSpaceURI, nodeName);
    };
    
    /**
     * Creates and returns a new attribute node, within a specified URI.
     *
     * @param  {String} nameSpaceURI  The name of the URI
     * @param  {String}  nodeName  The name of the attribute
     * @return {apf.AmlNode} The attribute node
     */ 
    this.createAttributeNS = function(nameSpaceURI, nodeName){
        return this.$domParser.$createNode(this, this.NODE_ATTRIBUTE, null,
            nameSpaceURI, nodeName);
    };
    /**
     * Creates and returns a new [[apf.AmlEvent]] .
     */     
    this.createEvent = function(){
        return new apf.AmlEvent();
    };

    /**
     * Creates and returns a new comment node.
     * @param {String} nodeValue The data to be added to the comment
     * @return {apf.AmlNode} The comment node
     */    
    this.createComment = function(nodeValue){
        return this.$domParser.$createNode(this, this.NODE_COMMENT, null, null,
            null, nodeValue);
    };

    /**
     * Creates and returns a new processing instruction node.
     * @param {String} target The target part of the processing instruction node, like `<?_target_ ...?>`
     * @param {String} data The data to be added to the PI
     * @return {apf.AmlNode} The processing instruction node
     */     
    this.createProcessingInstruction = function(target, data){
        return this.$domParser.$createNode(this, this.NODE_PROCESSING_INSTRUCTION,
            null, null, target, data);
    };
 
    /**
     * Creates and returns a new CDATA section node.
     * @param {String} nodeValue The data to be added to the CDATA node
     * @return {apf.AmlNode} The CDATA section node
     */     
    this.createCDATASection = function(nodeValue){
        return this.$domParser.$createNode(this, this.NODE_CDATA_SECTION, null,
            null, null, nodeValue);
    };
  
    /**
     * Creates and returns a new Text node.
     * @param {String} nodeValue The data to be added to the text node
     * @return {apf.AmlNode} The Text node
     */      
    this.createTextNode = function(nodeValue){
        return this.$domParser.$createNode(this, this.NODE_TEXT, null, null,
            null, nodeValue);
    };

    /**
     * Creates and returns a new document fragment.
     */ 

    this.createDocumentFragment = function(){
        return this.$domParser.$createNode(this, this.NODE_DOCUMENT_FRAGMENT);
    };

    // @todo
    this.querySelector = function(){};
 
     // @todo   
    this.querySelectorAll = function(){};

    

    // @todo
    this.hasFocus = function(){
        
    }

    
}).call(apf.AmlDocument.prototype = new apf.AmlNode());





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlDocumentFragment = function(isPrototype){
    this.$init(isPrototype);
};

apf.AmlDocumentFragment.prototype = new apf.AmlNode();
apf.AmlDocumentFragment.prototype.nodeName = "#document-fragment";
apf.AmlDocumentFragment.prototype.nodeType = 
    apf.AmlDocumentFragment.prototype.NODE_DOCUMENT_FRAGMENT;




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * Implementation of the W3C event object. An instance of this class is passed as
 * the first argument of any event handler. As per event, it contains different
 * properties giving context based information about the event.
 * @class apf.AmlEvent
 * @default_private
 */
apf.AmlEvent = function(name, data){
    this.name = name;
    
    var prop;
    for (prop in data)
        this[prop] = data[prop];
};

apf.AmlEvent.prototype = {
    
    bubbles : false,
    cancelBubble : false,
    

    /**
     * Cancels the event (if it is cancelable), without stopping further 
     * propagation of the event. 
     */
    preventDefault : function(){
        this.returnValue = false;
    },

    
    /**
     * Prevents further propagation of the current event. 
     */
    stopPropagation : function(){
        this.cancelBubble = true;
    },
    

    stop : function() {
        this.returnValue = false;
        
        this.cancelBubble = true;
        
    }
};




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


//@todo apf3.0
apf.AmlNamedNodeMap = function(host){
    this.$host = host;
};

(function(){
    this.getNamedItem    = function(name){
        for (var i = 0; i < this.length; i++) {
            if (this[i].name == name)
                return this[i];
        }
        return false;
    };
    
    this.setNamedItem    = function(node){
        var name = node.name;
        for (var item, i = this.length - 1; i >= 0; i--) {
            if (this[i].name == name) {
                this[i].ownerElement = null;
                this.splice(i, 1);
                break;
            }
        }
        
        this.push(node);
        
        node.ownerElement = this.$host;
        node.ownerDocument = this.$host.ownerDocument;
        node.$triggerUpdate();
    };
    
    //@todo apf3.0 domattr
    this.removeNamedItem = function(name){
        //Should deconstruct dynamic properties
        
        for (var item, i = this.length - 1; i >= 0; i--) {
            if (this[i].name == name) {
                item = this[i];
                this.splice(i, 1);
                break;
            }
        }
        if (!item) return false;

        //@todo hack!
        //this should be done properly
        var oldValue = item.nodeValue;
        item.nodeValue = item.value = "";
        item.$triggerUpdate(null, oldValue);
        item.ownerElement = null;
        item.nodeValue    = item.value = oldValue;
        
        return item;
    };
    
    this.item            = function(i){
        return this[i];
    };

    //if (apf.isIE < 8) { //Only supported by IE8 and above
        this.length = 0;
        
        this.splice = function(pos, length){
            for (var i = pos, l = this.length - length; i < l; i++) {
                this[i] = this[i + 1];
            }
            delete this[i];
            this.length -= length;
        }
        
        this.push = function(o) {
            this[this.length++] = o;
            return this.length;
        }
    //}
    
    this.join = function(glue){
        var x = [];
        for (var e, a, i = 0, l = this.length; i < l; i++) {
            if ((e = (a = this[i]).ownerElement) && e.$inheritProperties[a.nodeName] != 2)
                x.push(this[i]);
        }
        return x.join(glue);
    }
}).call(apf.AmlNamedNodeMap.prototype = {}); //apf.isIE < 8 ? {} : []




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlProcessingInstruction = function(isPrototype){
    this.$init(isPrototype);
};

(function(){
    this.nodeType = this.NODE_PROCESSING_INSTRUCTION;
    
    /*
     * @todo docs
     */
    this.data   = null;
    
    /*
     * @todo docs
     */
    this.target = null;
    
    this.serialize = function(){
        return "<?" + this.target + "\n" + apf.escapeXML(this.nodeValue) + "\n?>";
    };
    
    this.reload = function(){
        this.$handlePropSet("data", this.data);
    };
    
    //1 = force no bind rule, 2 = force bind rule
    this.$attrExcludePropBind = apf.extend({
        calcdata : 0 //Start in code mode
    }, this.$attrExcludePropBind);
    
    this.getAttribute = function(){};
    this.$setInheritedAttribute = apf.AmlElement.prototype.$setInheritedAttribute;
    this.$supportedProperties = [];
    this.$propHandlers        = {};
    this.$booleanProperties   = {};
    this.$inheritProperties   = {};
    
    
    
    this.$setValue = function(value){
        this.setProperty("data", value);
    };
    
    this.$handlePropSet = function(prop, value, force){
        this[prop] = value;
        if (prop == "data") {
            this.$clearDynamicProperty("calcdata");
            this.$setDynamicProperty("calcdata", value);
        }
        
        else if (prop == "target") {
            //not implemented
        }
        else if (this.$propHandlers[prop]) {
            this.$propHandlers[prop].call(this, value, prop);
        }
    };

    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var pHtmlNode = e.pHtmlNode;
        if (!pHtmlNode && (this.parentNode.$bindingRule 
          || !(pHtmlNode = this.parentNode.$int))) 
            return;

        pHtmlNode.appendChild(this.$ext = document.createElement("span"));
        this.$ext.host = this;

        

        this.$setDynamicProperty("calcdata", this.data);
        
        
    }, true);
    
    /*this.addEventListener("DOMNodeRemovedFromDocument", function(e){
        this.$clearDynamicProperty("calcdata");
    });*/
    
    this.$destroy = function(){
        this.$clearDynamicProperty("calcdata");
        this.$propHandlers["calcdata"].call(this, "");
    };
}).call(apf.AmlProcessingInstruction.prototype = new apf.AmlNode());




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.AmlTextRectangle = function(host){
    var _self = this;
    function handler(){
        var pos = _self.getAbsolutePosition(_self.$ext);
        _self.setProperty("left", pos[0]);
        _self.setProperty("top", pos[1]);
        _self.setProperty("right", document.documentElement.offsetWidth - pos[0]);
        _self.setProperty("bottom", document.documentElement.offsetWidth - pos[1]);
    }
    
    host.addEventListener("prop.width", handler);
    host.addEventListener("prop.height", handler);
    host.addEventListener("prop.left", handler);
    host.addEventListener("prop.top", handler);

    handler.call(host);
};
apf.AmlTextRectangle.prototype = new apf.Class();




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/*
 * An object creating the XHTML namespace for the aml parser.
 *
 * @constructor
 * @parser
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 */
apf.xhtml = new apf.AmlNamespace();
apf.setNamespace("http://www.w3.org/1999/xhtml", apf.xhtml);


/*
if (apf.getTextNode(x)) {
    var data = {
        amlNode  : x,
        htmlNode : o
    }

    
}

*/



/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



apf.XhtmlElement = function(struct, tagName){
    this.$init(tagName || true, apf.NODE_VISIBLE, struct);
    
    this.$xoe                = this.addEventListener;
    this.addEventListener    = this.$xae;
    this.removeEventListener = this.$xre;
    
    var _self = this;
    this.$de = function(e){
        _self.dispatchEvent(e.type, null, e);
    }
};

(function(){
    var excludedEvents = {
        "contextmenu": 1,
        "keydown": 1,
        "keypress": 1,
        "keyup": 1,
        "DOMNodeInserted": 2,
        "DOMNodeInsertedIntoDocument": 2,
        "DOMNodeRemoved": 2,
        "DOMNodeRemovedFromDocument": 2
    };
    
    this.$xae = function(type, fn){
        this.$xoe.apply(this, arguments);
        
        if (excludedEvents[type] > (this.editable ? 0 : 1)
          || type.substr(0, 5) == "prop.")
            return;
        
        if (this.$ext) {
            if (type.substr(0,2) == "on")
                type = type.substr(2);
            apf.addListener(this.$ext, type, this.$de);
        }
    };
    
    this.$xre = function(type, fn) {
        apf.AmlElement.prototype.removeEventListener.apply(this, arguments);
        
        
        
        if (this.$ext)
            apf.removeListener(this.$ext, type, this.$de);
    }
    
    this.$handlePropSet = function(name, value, force, inherit){
        if (this.$booleanProperties[name])
            value = apf.isTrue(value);

        this[name] = value;
        var handler = this.$propHandlers && this.$propHandlers[name]
          || apf.GuiElement.propHandlers[name];

        if (handler)
            handler.call(this, value, null, name);
        else if (this.$int && (force || this.$amlLoaded)) {
            this.$int.setAttribute(apf.isIE && apf.isIE < 8 && name == "class" 
                ? "className" : name, value);
        }
    };
    
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var pHtmlNode;
        if (!(pHtmlNode = this.$pHtmlNode = this.parentNode.$int)) 
            return;

        var str, aml = this.$aml;
        if (aml) {
            if (aml.serialize)
                str = aml.serialize();
            else {
                aml = aml.cloneNode(false);
                str = aml.xml || aml.nodeValue;
            }

            str = str.replace(/ on\w+="[^"]*"| on\w+='[^']*'/g, "");
            
            this.$ext = 
            this.$int = apf.insertHtmlNode(null, pHtmlNode, null, apf.html_entity_decode(str));
        }
        else {
            this.$ext = this.$int = 
              pHtmlNode.appendChild(document.createElement(this.localName));
        }
        
        if (this.localName != "a")
            this.$ext.host = this;

        this.style = this.$ext.style;
    }, true);
    
    
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        this.$amlLoaded = true;
        
        if (this.$setLayout)
            this.$setLayout();
    });
    
}).call(apf.XhtmlElement.prototype = new apf.AmlElement());

apf.Init.addConditional(function(){
    if (apf.isO3) return;
    var prot = apf.XhtmlElement.prototype;

    //prot.implement(apf.Interactive);
    prot.implement(
        
        apf.Anchoring
        
    );

    
    prot.$drawn = true;
    prot.$setLayout = apf.GuiElement.prototype.$setLayout;
    
    prot.addEventListener("DOMNodeInserted", function(e){
        if (e.currentTarget == this 
          && "vbox|hbox|table".indexOf(this.parentNode.localName) == -1) {
            this.$setLayout();
        }
    }); 
    
}, null, ["interactive"]);

apf.xhtml.setElement("@default", apf.XhtmlElement);





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.XhtmlBodyElement = function(struct, tagName){
    this.$init(tagName || "body", apf.NODE_VISIBLE, struct);
};

(function(){
    
    
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        if (!this.ownerDocument.body)
            this.ownerDocument.body = this;
        
        this.$ext = 
        this.$int = document.body;
    }, true);
}).call(apf.XhtmlBodyElement.prototype = new apf.AmlElement());

apf.Init.addConditional(function(){
    if (apf.isO3) return;
    var prot = apf.XhtmlBodyElement.prototype;

    
}, null, ["interactive"]);

apf.xhtml.setElement("body", apf.XhtmlBodyElement);





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * @todo description
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.4
 */
apf.XhtmlHtmlElement = function(struct, tagName){
    this.$init(tagName || "html", apf.NODE_VISIBLE, struct);
    
    
    
    this.$ext        = document.documentElement;
    this.$ext.host   = this;
    
    this.$int        = document.body;
    this.$tabList    = []; //Prevents documentElement from being focussed
    this.$focussable = apf.KEYBOARD;
    this.focussable  = true;
    this.visible     = true;
    this.$isWindowContainer = true;
    //this.focus = function(){ this.dispatchEvent("focus"); };
    //this.blur  = function(){ this.dispatchEvent("blur"); };
    
    this.implement(apf.Focussable);
    
    
    apf.window.$addFocus(this);
    
    
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var i, l, n, a, c,
            attr = this.attributes, doc = this.ownerDocument;
        for (i = 0, l = attr.length; i < l; i++) {
            n = (a = attr[i]).nodeName.split(":");
            if (n[0] == "xmlns") {
                if (c = n[1]) {
                    doc.$prefixes[c] = a.nodeValue;
                    doc.$namespaceURIs[a.nodeValue] = c;
                }
                else {
                    doc.namespaceURI = a.nodeValue;
                }
            }
        }
        
        if (!doc.namespaceURI)
            doc.namespaceURI = apf.ns.xhtml;
    });
};
apf.XhtmlHtmlElement.prototype = new apf.XhtmlElement();

apf.xhtml.setElement("html", apf.XhtmlHtmlElement);





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.XhtmlIgnoreElement = function(struct, tagName){
    this.$init(tagName, apf.NODE_VISIBLE, struct);
};

apf.XhtmlIgnoreElement.prototype = new apf.AmlElement();

apf.xhtml.setElement("script",   apf.XhtmlIgnoreElement);
apf.xhtml.setElement("noscript", apf.XhtmlIgnoreElement);
apf.xhtml.setElement("head",     apf.XhtmlIgnoreElement);
apf.xhtml.setElement("meta",     apf.XhtmlIgnoreElement);




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.XhtmlInputElement = function(struct, tagName){
    this.$init(tagName || "input", apf.NODE_VISIBLE, struct);
};

(function(){
    this.$xae = apf.XhtmlElement.prototype.$xae;
    this.$xre = apf.XhtmlElement.prototype.$xre;
    this.$handlePropSet = function(name, value, force){
        if (name == "type")
            return;

        return apf.XhtmlElement.prototype.$handlePropSet.call(this, name, value, force);
    };

    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var pHtmlNode;
        if (!(pHtmlNode = this.parentNode.$int))
            return;

        if (this.$aml) {
            this.$ext =
            this.$int = apf.insertHtmlNode(this.$aml.serialize
                ? this.$aml
                : this.$aml.cloneNode(false), pHtmlNode);
        }
        else {
            this.$ext = this.$int = document.createElement(this.localName);
            if (this.getAttribute("type"))
                this.$int.setAttribute("type", this.getAttribute("type"));
            pHtmlNode.appendChild(this.$int);
        }
    }, true);
}).call(apf.XhtmlInputElement.prototype = new apf.AmlElement());

apf.xhtml.setElement("input", apf.XhtmlInputElement);




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



apf.XhtmlOptionElement = function(struct, tagName){
    this.$init(tagName || "option", apf.NODE_VISIBLE, struct);
};

(function(){
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        this.$ext = 
        this.$int = this.parentNode.$int.appendChild(
          this.parentNode.$int.ownerDocument.createElement("option"));

        if (this.value)
            this.$int.setAttribute("value", this.value);
    }, true);
}).call(apf.XhtmlOptionElement.prototype = new apf.AmlElement());

apf.xhtml.setElement("option", apf.XhtmlOptionElement);




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


apf.XhtmlSkipChildrenElement = function(struct, tagName){
    this.$init(tagName, apf.NODE_VISIBLE, struct);
};

(function(){
    this.canHaveChildren = false;
    
    this.$redraw = function(){
        var _self = this;
        apf.queue.add("redraw" + this.$uniqueId, function(){
            var pHtmlNode  = _self.$ext.parentNode;
            var beforeNode = _self.$ext.nextSibling;
            pHtmlNode.removeChild(_self.$ext);
            
            _self.$ext = apf.insertHtmlNode(null, pHtmlNode, beforeNode, _self.$aml 
                ? (_self.$aml.serialize ? _self.$aml.serialize() : _self.$aml.xml)
                : _self.serialize());
        });
    }
    
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var pHtmlNode;
        if (!(pHtmlNode = this.parentNode.$int)) 
            return;

        this.$ext = apf.insertHtmlNode(null, pHtmlNode, null, this.$aml 
            ? (this.$aml.serialize ? this.$aml.serialize() : this.$aml.xml)
            : this.serialize());
    }, true);
}).call(apf.XhtmlSkipChildrenElement.prototype = new apf.AmlElement());

apf.xhtml.setElement("object", apf.XhtmlSkipChildrenElement);
apf.xhtml.setElement("embed", apf.XhtmlSkipChildrenElement);
apf.xhtml.setElement("table", apf.XhtmlSkipChildrenElement);

apf.xhtml.setElement("pre", apf.XhtmlSkipChildrenElement);




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/* #-ifdef __WITH_HTML5
if (tagName == "input") {
    objName = apf.HTML5INPUT[objName = x.getAttribute("type")]
        || objName || "textbox";
}
//#-endif*/




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





//XForms







/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * Object creating the XML Include namespace for the aml parser.
 *
 * @constructor
 * @parser
 *
 * @allownode simpleType, complexType
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 */
apf.xinclude = new apf.AmlNamespace();
apf.setNamespace("http://www.w3.org/2001/XInclude", apf.xinclude);





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * Defines a list of acceptable values
 */
apf.XiInclude = function(struct, tagName){
    this.$init(tagName || "include", apf.NODE_HIDDEN, struct);
};

apf.xinclude.setElement("include", apf.XiInclude);
apf.aml.setElement("include", apf.XiInclude);

//@todo test defer="true" situation
(function(){
    this.$parsePrio = "002";

    //1 = force no bind rule, 2 = force bind rule 
    /*this.$attrExcludePropBind = apf.extend({
        href : 1,
        src  : 1
    }, this.$attrExcludePropBind);*/

    this.$propHandlers["href"] = 
    this.$propHandlers["src"]  = function(value){
        if (typeof value != "string")
            return finish.call(this, value);
        
        if (value.trim().charAt(0) == "<") {
            loadIncludeFile.call(this, value.trim());
            return;
        }

        this.$path = value.charAt(0) == "{" //@todo this shouldn't happen anymore
          ? value
          : apf.getAbsolutePath(apf.hostPath, value);
        
        var domParser = this.ownerDocument.$domParser;
        if (!this.defer) {
            domParser.$pauseParsing.apply(domParser, 
              this.$parseContext = domParser.$parseContext || [this.parentNode]);
        }

        //var basePath = apf.hostPath;//only for recursion: apf.getDirname(xmlNode.getAttribute("filename")) || 
        loadIncludeFile.call(this, this.$path);
    };
    
    function done(xmlNode) {
        var addedNode = this.previousSibling || this.nextSibling;
        
        if (this.callback) {
            this.callback({
                xmlNode : xmlNode,
                amlNode : this.parentNode,
                addedNode: addedNode
            })
        }
        
        addedNode.dispatchEvent("DOMNodeInserted", {
            $beforeNode         : addedNode.nextSibling,
            relatedNode         : this.parentNode,
            $isMoveWithinParent : false,
            bubbles             : true
        });
        
        //@todo hack!! this should never happen. Find out why it happens
        if (this.parentNode)
            this.parentNode.removeChild(this);
    }
    
    function finish(xmlNode){
        var domParser = this.ownerDocument.$domParser;

        if (this.clear)
            this.parentNode.$int.innerHTML = "";

        if (xmlNode) {
            domParser.parseFromXml(xmlNode, {
                doc        : this.ownerDocument,
                amlNode    : this.parentNode,
                beforeNode : this,
                include    : true
            });

            if (!this.defer && this.$parseContext) {
                var o     = (this.$parseContext[1] || (this.$parseContext[1] = {})),
                    cb    = o.callback,
                    _self = this;

                o.callback = function(){
                    done.call(_self, xmlNode);
                    if (cb)
                        cb.call(_self.ownerDocument);
                };

                //@todo this is wrong... probably based on load order of last include element. Please rearchitect parse continuation.
                if (domParser.$continueParsing(this.$parseContext[0]) === false) {
                    var o2  = (domParser.$parseContext[1] || (domParser.$parseContext[1] = {})),
                        cb2 = o.callback;
                    o2.callback = function(){
                        if (cb)
                            cb.call(_self.ownerDocument);
                        domParser.$continueParsing(_self.$parseContext[0]);
                    };
                }
            }
            else
                done.call(this, xmlNode);
        }
        else {
            if (!this.defer)
                domParser.$continueParsing(this.$parseContext[0]);
            
            done.call(this, xmlNode);
        }
    }
    
    function loadIncludeFile(path){
        

        var _self = this;
        apf.getData(path, apf.extend(this.options || {}, {
            
            callback : function(xmlString, state, extra){
                if (state != apf.SUCCESS) {
                    var oError = new Error(apf.formatErrorString(1007,
                        _self, "Loading Includes", "Could not load Include file '"
                        + (path || _self.src)
                        + "'\nReason: " + extra.message));

                    if (extra.tpModule.retryTimeout(extra, state, null, oError) === true)
                        return true;

                    apf.console.error(oError.message);

                    finish.call(_self, null);

                    //throw oError;
                    return;
                }

                //@todo apf3.0 please make one way of doing this
                xmlString = xmlString.replace(/\<\!DOCTYPE[^>]*>/, "")
                    .replace(/^[\r\n\s]*/, ""); //.replace(/&nbsp;/g, " ")
                if (!apf.supportNamespaces)
                    xmlString = xmlString.replace(/xmlns\=\"[^"]*\"/g, "");
                
                if (xmlString.indexOf("<a:application") == -1)
                    xmlString = "<a:application xmlns:a='" + apf.ns.aml +"'>"
                      + xmlString + "</a:application>";

                var xmlNode = apf.getXml(xmlString, null, true);//apf.getAmlDocFromString(xmlString);
            
                if (!xmlNode) {
                    throw new Error(apf.formatErrorString(0, null,
                        "Loading include",
                        "Could not parse include file. Maybe the file does not exist?", xmlNode));
                }
                xmlNode.setAttribute("filename", extra.url);

                

                finish.call(_self, xmlNode); //@todo add recursive includes support here
            },
            async         : true,
            ignoreOffline : true
        }));
    }
}).call(apf.XiInclude.prototype = new apf.AmlElement());




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
apf.__LIVEEDIT__  = 1 << 23;






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */







/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__ANCHORING__ = 1 << 13;



/**
 * All elements inheriting from this {@link term.baseclass baseclass} have anchoring features. Each side of the
 * element can be attached at a certain distance to its parent's rectangle.
 *
 * When the parent is resized, the anchored side of the element stays
 * at the specified distance at all times. If both sides are anchored, the
 * element size is changed to make sure the specified distance is maintained.
 *
 * #### Example
 *
 * This example shows a bar that has a 10% margin around it, and contains a
 * frame that is displayed using different calculations and settings.
 *
 * ```xml
 *  <a:bar width="80%" height="80%" top="10%" left="10%">
 *      <a:frame 
 *        caption = "Example" 
 *        left    = "50%+10"
 *        top     = "100"
 *        right   = "10%"
 *        bottom  = "Math.round(0.232*100)" />
 *  </a:bar>
 * ```
 *
 * ### Remarks
 *
 * This is one of three positioning methods. The other two are Alignment and Grid.
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.3
 * @baseclass
 * @layout
 */
apf.Anchoring = function(){
    this.$regbase = this.$regbase | apf.__ANCHORING__;
    this.$anchors = [];

    var VERTICAL   = 1;
    var HORIZONTAL = 2;

    this.$updateQueue = 0;
    this.$inited      =
    this.$parsed      =
    this.$anchoringEnabled = false;
    this.$hordiff     = 
    this.$verdiff     = 0;
    this.$rule_v      =
    this.$rule_h      =
    this.$rule_header = "";

    var l = apf.layout;
    
    this.$supportedProperties.push("anchors");
    
    var propHandlers = {
        "right" : function(value, prop){
            if (!this.$anchoringEnabled && !this.$setLayout("anchoring"))
                return;
            
            if (!value && value !== 0)
                this.$ext.style[prop] = "";

            //@note Removed apf.isParsing here to activate general queuing
            if (!this.$updateQueue)
                l.queue(this.$pHtmlNode, this);
            this.$updateQueue = this.$updateQueue | HORIZONTAL;
        },

        "bottom" : function(value, prop){
            if (!this.$anchoringEnabled && !this.$setLayout("anchoring"))
                return;

            if (!value && value !== 0)
                this.$ext.style[prop] = "";

            //@note Removed apf.isParsing here to activate general queuing            
            if (!this.$updateQueue)
                l.queue(this.$pHtmlNode, this);
            this.$updateQueue = this.$updateQueue | VERTICAL;
        }
    };
    propHandlers.left = propHandlers.width = propHandlers.right;
    propHandlers.top = propHandlers.height = propHandlers.bottom;
    
    this.$propHandlers["anchors"] = function(value){
        this.$anchors = value ? value.splitSafe("(?:, *| )") : [];

        if (!this.$anchoringEnabled && !this.$setLayout("anchoring"))
            return;

        if (!this.$updateQueue && apf.loaded)
            l.queue(this.$pHtmlNode, this);
        this.$updateQueue = this.$updateQueue | HORIZONTAL | VERTICAL;
    };

    /**
     * Turns anchoring off.
     *
     */
    this.$disableAnchoring = function(activate){
        //!this.$parsed || 
        if (!this.$inited || !this.$anchoringEnabled || !this.$pHtmlNode)
            return;

        l.removeRule(this.$pHtmlNode, this.$uniqueId + "_anchors");
        if (l.queue)
            l.queue(this.$pHtmlNode);

        for (var prop in propHandlers) {
            delete this.$propHandlers[prop];
        }

        this.removeEventListener("DOMNodeRemoved", remove); 
        this.removeEventListener("DOMNodeInserted", reparent); 

        if (this.$ext) {
            this.$ext.style.left   = 
            this.$ext.style.right  = 
            this.$ext.style.top    = 
            this.$ext.style.bottom = 
            this.$ext.style.width  = 
            this.$ext.style.height = 
            this.$ext.style.position = "";
        }
        
        /*if (this.right)
            this.$ext.style.left = apf.getHtmlLeft(this.$ext) + "px";

        if (this.bottom)
            this.$ext.style.top = apf.getHtmlTop(this.$ext) + "px";*/

        this.removeEventListener("prop.visible", visibleHandler);

        this.$inited   = false;
        this.$anchoringEnabled = false; //isn't this redundant?
    };


    /**
     * @attribute {Number | String} [left]   Sets or gets a way to determine the amount of pixels from the left border of this element to the left edge of it's parent's border. This attribute can also contain percentages, arithmetic and even full expressions.
     * 
     * #### Example
     *
     * ```xml
     * <a:bar left="(20% + 10) * SOME_JS_VAR" />
     * ```
     */
    /**
     * @attribute {Number | String} [right]  Sets or gets a way to determine the amount of pixels from the right border of this element to the right edge of its parent's border.
     *                                      This attribute can also contain percentages, arithmetic and even full expressions.
     * 
     * #### Example
     *
     * ```xml
     * <a:bar right="(20% + 10) * SOME_JS_VAR" />
     * ```
     */
    /** 
     * @attribute {Number | String} [width]  Sets or gets a way to determine the amount of pixels from the left border to the right border of this element.
     *                                      This attribute can also contain percentages, arithmetic and even full expressions.
     * 
     * #### Example
     *
     * ```xml
     * <a:bar width="(20% + 10) * SOME_JS_VAR" />
     * ```
     */
    /** 
     * @attribute {Number | String} [top]    Sets or gets a way to determine the amount of pixels from the top border of this element to the top edge of its parent's border.
     *                                      This attribute can also contain percentages, arithmetic and even full expressions.
     * 
     * #### Example
     *
     * ```xml
     * <a:bar top="(20% + 10) * SOME_JS_VAR" />
     * ```
     */
    /** 
     * @attribute {Number | String} [bottom] Sets or gets a way to determine the amount of pixels from the bottom border of this element to the bottom edge of its parent's border.
     *                                      This attribute can also contain percentages, arithmetic and even full expressions.
     * 
     * #### Example
     *
     * ```xml
     * <a:bar bottom="(20% + 10) * SOME_JS_VAR" />
     * ```
     */
    /** 
     * @attribute {Number | String} [height] Sets or gets a way to determine the amount of pixels from the top border to the bottom border of this element.
     *                                      This attribute can also contain percentages, arithmetic and even full expressions.
     * 
     * #### Example
     *
     * ```xml
     * <a:bar height="(20% + 10) * SOME_JS_VAR" />
     * ```
     */
    /*
     * Enables anchoring based on attributes set in the AML of this element
     */
    this.$enableAnchoring = function(){
        if (this.$inited) //@todo add code to reenable anchoring rules (when showing)
            return;

        // *** Properties and Attributes *** //
        apf.extend(this.$propHandlers, propHandlers);

        // *** Event handlers *** //
        this.addEventListener("DOMNodeRemoved", remove); 
        this.addEventListener("DOMNodeInserted", reparent); 
        this.addEventListener("prop.visible", visibleHandler);

        this.$updateQueue = 0 
            | ((this.left || this.width || this.right || this.anchors) && HORIZONTAL) 
            | ((this.top || this.height || this.bottom || this.anchors) && VERTICAL) ;

        if (this.$updateQueue)
            l.queue(this.$pHtmlNode, this);

        this.$inited   = true;
        this.$anchoringEnabled = true;
    };
    
    function visibleHandler(e){
        if (!(this.$rule_header || this.$rule_v || this.$rule_h) || !this.parentNode)
            return;

        if (e.value) {
            if (this.$rule_v || this.$rule_h) {
                var rules = this.$rule_header + "\n" + this.$rule_v + "\n" + this.$rule_h;
                l.setRules(this.$pHtmlNode, this.$uniqueId + "_anchors", rules);
                //this.$ext.style.display = "none";
                l.queue(this.$pHtmlNode, this);
            }
            l.processQueue();
        }
        else {
            l.removeRule(this.$pHtmlNode, this.$uniqueId + "_anchors");
            l.queue(this.$pHtmlNode)
        }
    }
    
    function remove(e){
        if (e && (e.$doOnlyAdmin || e.currentTarget != this))
            return;

        if (l.queue && this.$pHtmlNode) {
            l.removeRule(this.$pHtmlNode, this.$uniqueId + "_anchors");
            l.queue(this.$pHtmlNode)
        }
    }

    function reparent(e){
        if (!this.$amlLoaded || e.currentTarget != this)
            return;

        if (!e.$isMoveWithinParent && this.$parsed) //@todo hmm weird state check
            this.$moveAnchoringRules(e.$oldParentHtmlNode);
        //else if (e.relatedNode == this) //@todo test this
            //e.currentTarget.$setLayout("anchoring");
    }

    this.$moveAnchoringRules = function(oldParent, updateNow){
        var rules = oldParent && l.removeRule(oldParent, this.$uniqueId + "_anchors");
        if (rules)
            l.queue(oldParent);

        if (!this.$rule_v && !this.$rule_h && !this.$rule_header)
            return;

        this.$rule_header = getRuleHeader.call(this);
        rules = this.$rule_header + "\n" + this.$rule_v + "\n" + this.$rule_h;

        //@todo sometimes the content is not displayed anymore (when reparenting by xinclude)
        //this.$ext.style.display = "none";

        l.setRules(this.$pHtmlNode, this.$uniqueId + "_anchors", rules);
        l.queue(this.$pHtmlNode, this);
    };

    this.$hasAnchorRules = function(){
        return this.$rule_v || this.$rule_h ? true : false;
    };

    function getRuleHeader(){
        if (!this.$pHtmlDoc) return "";
        return "try{\
            var oHtml = " + (apf.hasHtmlIdsInJs
                ? this.$ext.getAttribute("id")
                : "document.getElementById('"
                    + this.$ext.getAttribute("id") + "')") + ";\
            \
            var pWidth = " + (this.$pHtmlNode == this.$pHtmlDoc.body
                ? "apf.getWindowWidth()" //@todo only needed for debug?
                : "apf.getHtmlInnerWidth(oHtml.parentNode)") + ";\
            \
            var pHeight = " + (this.$pHtmlNode == this.$pHtmlDoc.body
                ? "apf.getWindowHeight()" //@todo only needed for debug?
                : "apf.getHtmlInnerHeight(oHtml.parentNode)") + ";\
            }catch(e){\
            }";
    }

    /**
     * Sets the anchoring percentage.
     * @param {String} expr An expression that's converted to a string
     * @param {Number} An integer value that's used to convert to a percentage; for example, 50 becomes .5
     * @returns {String} The anchor percentage
     */
    function setPercentage(expr, value){
        return String(expr).replace(apf.percentageMatch, "((" + value + " * $1)/100)");
    }

     
    this.$recalcAnchoring = function(queueDelay){
        this.$updateQueue = this.$updateQueue | HORIZONTAL | VERTICAL;
        this.$updateLayout();
        l.queue(this.$pHtmlNode, this);
        
        if (!queueDelay)
            l.processQueue();
    };
    

    function visCheck(){
        if (this.$updateQueue) {
            this.$updateLayout();
            apf.layout.activateRules(this.$ext.parentNode);
        }
    }

    this.$updateLayout = function(){
        if (!this.$anchoringEnabled)
            return;

        if (!apf.window.vManager.check(this, "anchoring", visCheck))
            return;

        if (!this.$parsed) {
            if (!this.$ext.getAttribute("id"))
                apf.setUniqueHtmlId(this.$ext);

            this.$rule_header = getRuleHeader.call(this);
            this.$parsed      = true;
        }

        if (!this.$updateQueue) {
            if (this.visible && this.$ext.style.display == "none")
                this.$ext.style.display = "";
            return;
        }

        if (this.draggable == "relative") {
            if ("absolute|fixed|relative".indexOf(apf.getStyle(this.$ext, "position")) == -1) //@todo apf3.1 the IDE doesn't like this
                this.$ext.style.position = "absolute";
        }
        else if (this.left || this.left ===  0 || this.top || this.top === 0 
          || this.right || this.right === 0 || this.bottom || this.bottom === 0 
          || this.$anchors.length) {
            if ("absolute|fixed".indexOf(apf.getStyle(this.$ext, "position")) == -1)
                this.$ext.style.position = "absolute";
        }
        else if (!this.center) {
            if ("absolute|fixed|relative".indexOf(apf.getStyle(this.$ext, "position")) == -1)
                this.$ext.style.position = "relative";
            if (!this.width)
                this.$ext.style.width    = "";
            if (!this.height)
                this.$ext.style.height   = "";
        }

        var rules;
        if (this.$updateQueue & HORIZONTAL) {
            rules = [];
            
            this.$hordiff = apf.getWidthDiff(this.$ext);

            var left  = this.$anchors[3] || this.left,
                right = this.$anchors[1] || this.right,
                width = this.width, hasLeft = left || left === 0,
                hasRight = right || right === 0, 
                hasWidth = width || width === 0;

            if (right && typeof right == "string")
                right = setPercentage(right, "pWidth");

            if (hasLeft) {
                if (parseInt(left) != left) {
                    left = setPercentage(left,  "pWidth");
                    rules.push("oHtml.style.left = (" + left + ") + 'px'");
                }
                else
                    this.$ext.style.left = left + "px";
            }
            if ((apf.hasStyleAnchors || !hasLeft) && hasRight) {
                if (parseInt(right) != right) {
                    right = setPercentage(right, "pWidth");
                    rules.push("oHtml.style.right = (" + right + ") + 'px'");
                }
                else
                    this.$ext.style.right = right + "px";
            }

            if (hasLeft && hasRight) { //right != null && left != null) {
                if (!apf.hasStyleAnchors)
                    rules.push("oHtml.style.width = (pWidth - (" + right
                        + ") - (" + left + ") - " + this.$hordiff + ") + 'px'");
                else
                    this.$ext.style.width = "";
            }
            else if (hasWidth && typeof this.maxwidth == "number" && typeof this.minwidth == "number") {
                if (parseInt(width) != width) {
                    this.width = width = (this.width || "").replace(/--(\d+)/, "-(-$1)");
                    width = setPercentage(width, "pWidth");
                    rules.push("oHtml.style.width = Math.max(" 
                        + (this.minwidth - this.$hordiff)
                        + ", Math.min(" + (this.maxwidth - this.$hordiff) + ", "
                        + width + " - " + this.$hordiff + ")) + 'px'");
                }
                else {
                    this.$ext.style.width = ((width > this.minwidth
                        ? (width < this.maxwidth
                            ? width
                            : this.maxwidth)
                        : this.minwidth) - this.$hordiff) + "px";
                }
            }

            this.$rule_h = (rules.length
                ? "try{" + rules.join(";}catch(e){};try{") + ";}catch(e){};"
                : "");
        }

        if (this.$updateQueue & VERTICAL) {
            rules = [];

            this.$verdiff = apf.getHeightDiff(this.$ext);

            var top    = this.$anchors[0] || this.top,
                bottom = this.$anchors[2] || this.bottom,
                height = this.height, hasTop = top || top === 0,
                hasBottom = bottom || bottom === 0, 
                hasHeight = height || height === 0;

            if (bottom && typeof bottom == "string")
                bottom = setPercentage(bottom, "pHeight");

            if (hasTop) {
                if (parseInt(top) != top) {
                    top = setPercentage(top, "pHeight");
                    rules.push("oHtml.style.top = (" + top + ") + 'px'");
                }
                else
                    this.$ext.style.top = top + "px";
            }
            if ((apf.hasStyleAnchors || !hasTop) && hasBottom) {
                if (parseInt(bottom) != bottom) {
                    rules.push("oHtml.style.bottom = (" + bottom + ") + 'px'");
                }
                else
                    this.$ext.style.bottom = bottom + "px";
            }
            if (hasTop && hasBottom) { //bottom != null && top != null) {
                if (!apf.hasStyleAnchors)
                    rules.push("oHtml.style.height = (pHeight - (" + bottom +
                        ") - (" + top + ") - " + this.$verdiff + ") + 'px'");
                else
                    this.$ext.style.height = "";
            }
            else if (hasHeight && typeof this.minheight == "number") {
                if (parseInt(height) != height) {
                    height = setPercentage(height, "pHeight");
                    rules.push("oHtml.style.height = Math.max(" 
                        + (this.minheight - this.$verdiff)
                        + ", Math.min(" + (this.maxheight - this.$verdiff) + ", "
                        + height + " - " + this.$verdiff + ")) + 'px'");
                }
                else {
                    this.$ext.style.height = Math.max(0, (height > this.minheight
                        ? (height < this.maxheight
                            ? height
                            : this.maxheight)
                        : this.minheight) - this.$verdiff) + "px";
                }
            }

            this.$rule_v = (rules.length
                ? "try{" + rules.join(";}catch(e){};try{") + ";}catch(e){};"
                : "");
        }

        if (this.$rule_v || this.$rule_h) {
            l.setRules(this.$pHtmlNode, this.$uniqueId + "_anchors",
                this.$rule_header + "\n" + this.$rule_v + "\n" + this.$rule_h, true);
        }
        else {
            l.removeRule(this.$pHtmlNode, this.$uniqueId + "_anchors");
        }

        this.$updateQueue = 0;
        
        if (this.$box && !apf.hasFlexibleBox) //temporary fix
            apf.layout.forceResize(this.$ext);
    };

    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        //if (this.$updateQueue)
            //this.$updateLayout();
    });

    this.addEventListener("DOMNodeRemovedFromDocument", function(e){
        this.$disableAnchoring();
    });
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
apf.__CONTENTEDITABLE__  = 1 << 24;





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__GUIELEMENT__ = 1 << 15;
apf.__VALIDATION__ = 1 << 6;



/**
 * All elements inheriting from this {@link term.baseclass baseclass} are an AML component.
 *
 * @class apf.GuiElement
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.4
 *
 * @baseclass
 * @inherits apf.AmlElement
 * @inherits apf.Anchoring
 * @inherits apf.DelayedRender
 * @inherits apf.DragDrop
 * @inherits apf.Focussable
 * @inherits apf.Interactive
 * @inherits apf.Validation
 *
 */
/**
 * @attribute {String} span     Sets or gets the number of columns that this element spans. Only used inside a table element.
 */
/**
 * @attribute {String | Number} margin  Sets or gets margin values. 
 *
 * Set these sizes as a quarter of strings, in the usual top, right, bottom, left sequence, or pass an empty string to turn off margins.
 */
/**
 * @attribute {String} align Sets or gets the edge of the parent to which this
 *                                  element aligns. 
 *
 * The possible values are a combination of "left", "middle", "right", "top", "bottom" and "slider" ,and optionally a size. 
 * Combinations are combined with the pipe (`"|"`) character.
 * 
 */
/**
 * @attribute {Mixed} left Sets or gets the left position of this element. Depending
 * on the choosen layout method the unit can be pixels, a percentage or an
 * expression.
 */
/**
 * @attribute {Mixed} top Sets or gets the top position of this element. Depending
 * on the choosen layout method the unit can be pixels, a percentage or an
 * expression.
 */
/**
 * @attribute {Mixed} right Sets or gets the right position of this element. Depending
 * on the choosen layout method the unit can be pixels, a percentage or an
 * expression.
 */
/**
 * @attribute {Mixed} bottom Sets or gets the bottom position of this element. Depending
 * on the choosen layout method the unit can be pixels, a percentage or an
 * expression.
 */
/**
 * @attribute {Mixed} width Sets or gets the different between the left edge and the
 * right edge of this element. Depending on the choosen layout method the
 * unit can be pixels, a percentage or an expression.
 * 
 * #### Remarks
 *
 * When used as a child of a grid element the width can also be set as '*'. 
 * This will fill the rest space.
 */
/**
 * @attribute {Mixed} height Sets or gets the different between the top edge and the
 * bottom edge of this element. Depending on the choosen layout method the
 * unit can be pixels, a percentage or an expression.
 * 
 * #### Remarks
 *
 * When used as a child of a grid element the height can also be set as '*'. 
 * This will fill the rest space.
 */
/**
 * @event resize Fires when the element changes width or height. 
 */
/** 
 * @event contextmenu Fires when the user requests a context menu, either
 * using the keyboard or mouse.
 * @bubbles
 * @cancelable Prevents the default context menu from appearing.
 * @param {Object} e The standard event object. Contains the following properties:
 *                   - x ([[Number]]): The x coordinate where the contextmenu is requested on
 *                   - y ([[Number]]): The y coordinate where the contextmenu is requested on
 *                   - htmlEvent ([[Event]]): The HTML event object that triggered this event from being called
 */
/**  
 * @event focus       Fires when this element receives focus.
 */
/** 
 * @event blur        Fires when this element loses focus.
 */
/**  
 * @event keydown     Fires when this element has focus and the user presses a key on the keyboard.
 * @bubbles
 * @cancelable Prevents the default key action.
 * @param {Object} e The standard event object. Contains the following properties:
 *                   - ctrlKey ([[Boolean]]): Specifies whether the [[keys: Ctrl]] key was pressed
 *                   - shiftKey ([[Boolean]]): Specifies whether the [[keys: Shift]] key was pressed
 *                   - altKey ([[Boolean]]): Specifies whether the [[keys: Alt ]] key was pressed
 *                   - keyCode ([[Number]]): Indicates which key was pressed. This is an ascii number
 *                   - htmlEvent ([[Event]]): the HTML event object that triggered this event from being called
 * 
 */
apf.GuiElement = function(){
    this.$init(true);
};

(function(){
    this.$regbase    = this.$regbase | apf.__GUIELEMENT__;
    
    this.$focussable = apf.KEYBOARD_MOUSE; // Each GUINODE can get the focus by default
    this.visible     = 2; //default value;
    
    this.minwidth    = 1;
    this.minheight   = 1;
    
    /*this.minwidth   = 5;
    this.minheight  = 5;
    this.maxwidth   = 10000;
    this.maxheight  = 10000;*/
    
    
    this.$booleanProperties["disable-keyboard"] = true;
    
    this.$booleanProperties["visible"]          = true;
    
    /**
     * @attribute {Boolean} draggable If true, the element can be dragged around the screen.
     */    
    /**
     * @attribute {Boolean} resizable If true, the element can by resized by the user.
     * 
     */
    
    this.$supportedProperties.push("draggable", "resizable");
    
    this.$supportedProperties.push(
        "focussable", "zindex", "disabled", "tabindex",
        "disable-keyboard", "contextmenu", "visible", "autosize", 
        "loadaml", "actiontracker", "alias",
        "width", "left", "top", "height", "tooltip"
    );

    this.$setLayout = function(type, insert){
        if (!this.$drawn || !this.$pHtmlNode)
            return false;

        if (this.parentNode) {
            
            if (this.parentNode.localName == "table") {
                if (this.$disableCurrentLayout)
                    this.$disableCurrentLayout();
                this.parentNode.register(this, insert);
                this.$disableCurrentLayout = null;
                return type == "table";
            }else
            

            
            if (this.parentNode.$box) {
                if (this.$disableCurrentLayout)
                    this.$disableCurrentLayout();
                this.parentNode.register(this, insert);
                this.$disableCurrentLayout = null;
                return type == this.parentNode.localName;
            } //else
            
        }
        
        
        if (!this.$anchoringEnabled) {
            if (this.$disableCurrentLayout)
                this.$disableCurrentLayout();
            this.$enableAnchoring();
            this.$disableCurrentLayout = this.$disableAnchoring;
        }
        return type == "anchoring";
        
    }

    this.addEventListener("DOMNodeInserted", function(e){
//        if (e.currentTarget == this 
//          && (this.parentNode.$box || "table" == this.parentNode.localName)) {
            this.$setLayout();
//        }
    }); 

    this.implement(
        
        apf.Anchoring
        
        
        
    );
    
    // **** Convenience functions for gui nodes **** //

    

    // *** Geometry *** //

    /**
     * Sets the difference between the left edge and the right edge of this
     * element. 
     *
     * Depending on the choosen layout method, the unit can be
     * pixels, a percentage, or an expression. 
     *
     * @chainable
     * @param {Number | String} value The new width of this element.
     */
    this.setWidth = function(value){
        this.setProperty("width", value, false, true);
        return this;
    };

    /**
     * Sets the different between the top edge and the bottom edge of this
     * element. 
     *
     * Depending on the choosen layout method the unit can be
     * pixels, a percentage or an expression.
     *
     * @chainable
     * @param {Number | String} value the new height of this element.
     */
    this.setHeight = function(value){
        this.setProperty("height", value, false, true);
        return this;
    };

    /**
     * Sets the left position of this element. 
     *
     * Depending on the choosen layout method the unit can be pixels, 
     * a percentage or an expression.
     *
     * @chainable
     * @param {Number | String} value The new left position of this element.
     */
    this.setLeft   = function(value){
        this.setProperty("left", value, false, true);
        return this;
    };

    /**
     * Sets the top position of this element. 
     *
     * Depending on the choosen layout method the unit can be pixels, 
     * a percentage or an expression.
     *
     * @chainable
     * @param {Number | String} value The new top position of this element.
     */
    this.setTop    = function(value){
        this.setProperty("top", value, false, true);
        return this;
    };

    if (!this.show) {
        /**
         * Makes the elements visible. 
         * @chainable
         */
        this.show = function(){
            this.setProperty("visible", true, false, true);
            return this;
        };
    }

    if (!this.hide) {
        /**
         * Makes the elements invisible. 
         * @chainable
         */
        this.hide = function(){
            this.setProperty("visible", false, false, true);
            return this;
        };
    }

    /**
     * Retrieves the calculated width in pixels for this element.
     */
    this.getWidth  = function(){
        return (this.$ext || {}).offsetWidth;
    };

    /**
     * Retrieves the calculated height in pixels for this element.
     */
    this.getHeight = function(){
        return (this.$ext || {}).offsetHeight;
    };

    /**
     * Retrieves the calculated left position in pixels for this element,
     * relative to the offsetParent.
     */
    this.getLeft   = function(){
        return (this.$ext || {}).offsetLeft;
    };

    /**
     * Retrieves the calculated top position in pixels for this element,
     * relative to the offsetParent.
     */
    this.getTop    = function(){
        return (this.$ext || {}).offsetTop;
    };

    // *** Disabling *** //

    /**
     * Activates the functions of this element. 
     * @chainable
     */
    this.enable  = function(){
        this.setProperty("disabled", false, false, true);
        return this;
    };

    /**
     * Deactivates the functions of this element.
     * @chainable
     */
    this.disable = function(){
        this.setProperty("disabled", true, false, true);
        return this;
    };

    // *** z-Index *** //

    /**
     * Moves this element to the lowest z ordered level.
     * @chainable
     */
    this.sendToBack = function(){
        this.setProperty("zindex", 0, false, true);
        return this;
    };

    /**
     * Moves this element to the highest z ordered level.
     * @chainable
     */
    this.bringToFront  = function(){
        this.setProperty("zindex", apf.all.length + 1, false, true);
        return this;
    };

    /**
     * Moves this element one z order level deeper.
     * @chainable
     */
    this.sendBackwards = function(){
        this.setProperty("zindex", this.zindex - 1, false, true);
        return this;
    };

    /**
     * Moves this element one z order level higher.
     * @chainable
     */
    this.bringForward  = function(){
        this.setProperty("zindex", this.zindex + 1, false, true);
        return this;
    };

    
    
    this.hasFocus = function(){}

    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var x = this.$aml;

        // will $pHtmlNode be deprecated soon?
        // check used to be:
        //if (!this.$pHtmlNode && this.parentNode)
        if (this.parentNode) {
            if (this.localName == "item" 
              && this.parentNode.hasFeature(apf.__MULTISELECT__)) //special case for item nodes, using multiselect rendering
                this.$pHtmlNode = this.parentNode.$container;
            else
                this.$pHtmlNode = this.parentNode.$int; //@todo apf3.0 change this in the mutation events
        }

        if (!this.$pHtmlNode) //@todo apf3.0 retry on DOMNodeInserted
            return;
        
        this.$pHtmlDoc  = this.$pHtmlNode.ownerDocument || document;

        if (this.$initSkin)
            this.$initSkin(x);

        if (this.$draw)
            this.$draw();

        if (e.id)
            this.$ext.setAttribute("id", e.id);

        if (typeof this.visible == "undefined")
            this.visible = true;

        

        this.$drawn = true;
    }, true);
    
    var f = function(e){
        if (!this.$pHtmlNode) //@todo apf3.0 retry on DOMInsert or whatever its called
            return;
        
        this.$setLayout(); //@todo apf3.0 moving an element minwidth/height should be recalced
        
        //@todo apf3.0 set this also for skin change
        if (this.$ext) {
            var hasPres = (this.hasFeature(apf.__PRESENTATION__)) || false;
            var type        = this.$isLeechingSkin ? this.localName : "main";
            this.minwidth   = Math.max(this.minwidth || 0, apf.getCoord(hasPres && parseInt(this.$getOption(type, "minwidth")), 0));
            this.minheight  = Math.max(this.minheight || 0, apf.getCoord(hasPres && parseInt(this.$getOption(type, "minheight")), 0));
            if (this.maxwidth == undefined)
                this.maxwidth   = apf.getCoord(hasPres && parseInt(this.$getOption(type, "maxwidth")), 10000);
            if (this.maxheight == undefined)
                this.maxheight  = apf.getCoord(hasPres && parseInt(this.$getOption(type, "maxheight")), 10000);

            //--#ifdef __WITH_CONTENTEDITABLE
            //@todo slow??
            var diff = apf.getDiff(this.$ext);
            this.$ext.style.minWidth = Math.max(0, this.minwidth - diff[0]) + "px";
            this.$ext.style.minHeight = Math.max(0, this.minheight - diff[1]) + "px";
            this.$ext.style.maxWidth = Math.max(0, this.maxwidth - diff[0]) + "px";
            this.$ext.style.maxHeight = Math.max(0, this.maxheight - diff[1]) + "px";
            
            if (this.$altExt && apf.isGecko) {
                this.$altExt.style.minHeight = this.$ext.style.minHeight;
                this.$altExt.style.maxHeight = this.$ext.style.maxHeight;
                this.$altExt.style.minWidth = this.$ext.style.minWidth;
                this.$altExt.style.maxWidth = this.$ext.style.maxWidth;
            }
            //--#endif
        }
        
        if (this.$loadAml)
            this.$loadAml(this.$aml); //@todo replace by event
        
        
        if (this.$focussable && typeof this.focussable == "undefined")
            apf.GuiElement.propHandlers.focussable.call(this, true);
        
        
        
        if (setResizeEvent)
            f2();
        
    };
    
    this.addEventListener("DOMNodeInsertedIntoDocument", f);
    this.addEventListener("$skinchange", f);
    
    
    var f2, setResizeEvent;
    this.addEventListener("$event.resize", f2 = function(c){
        if (!this.$ext) {
            setResizeEvent = true;
            return;
        }
        
        apf.layout.setRules(this.$ext, "resize", "var o = apf.all[" + this.$uniqueId + "];\
            if (o) o.dispatchEvent('resize');", true);

        apf.layout.queue(this.$ext);
        //apf.layout.activateRules(this.$ext);
        this.removeEventListener("$event.resize", f2);
    });
    

    
    this.addEventListener("contextmenu", function(e){
        
        
        if (!this.contextmenus) return;
        
        if (this.hasFeature(apf.__DATABINDING__)) {
            var contextmenu;
            var xmlNode = this.hasFeature(apf.__MULTISELECT__)
                ? this.selected
                : this.xmlRoot;

            var i, l, m, isRef, sel, menuId, cm, result;
            for (i = 0, l = this.contextmenus.length; i < l; i++) {
                isRef  = (typeof (cm = this.contextmenus[i]) == "string");
                result = null;
                if (!isRef && cm.match && xmlNode) {//@todo apf3.0 cache this statement
                    result = (cm.cmatch || (cm.cmatch = apf.lm.compile(cm.match, {
                        xpathmode  : 3,
                        injectself : true
                    })))(xmlNode)
                }

                if (isRef || xmlNode && result || !cm.match) { //!xmlNode && 
                    menuId = isRef
                        ? cm
                        : cm.menu

                    if (!self[menuId]) {
                        
                        
                        return;
                    }

                    self[menuId].display(e.x, e.y, null, this, xmlNode);

                    e.returnValue  = false;//htmlEvent.
                    e.cancelBubble = true;
                    break;
                }
            }

            //IE6 compatiblity
            /*
            @todo please test that disabling this is OK
            if (!apf.config.disableRightClick) {
                document.oncontextmenu = function(){
                    document.oncontextmenu = null;
                    e.cancelBubble = true;
                    return false;
                }
            }*/
        }
        else {
            menuId = typeof this.contextmenus[0] == "string"
                ? this.contextmenus[0]
                : this.contextmenus[0].getAttribute("menu")

            if (!self[menuId]) {
                
                
                return;
            }

            self[menuId].display(e.x, e.y, null, this);

            e.returnValue = false;//htmlEvent.
            e.cancelBubble = true;
        }
    });
    
}).call(apf.GuiElement.prototype = new apf.AmlElement());

/*
 * @for apf.amlNode
 * @private
 */
apf.GuiElement.propHandlers = {
    /**
     * @attribute {Number} minwidth Sets or gets the minimum width for this element.
     */
    /**
     * @attribute {Number} maxwidth Sets or gets the maximum width for this element.
     */
    /**
     * @attribute {Number} minheight Sets or gets the minimum height for this element.
     */
    /**
     * @attribute {Number} maxheight Sets or gets the maximum height for this element.
     */
    "minwidth": function(value){ this.$ext.style.minWidth = Math.max(0, value - apf.getWidthDiff(this.$ext)) + "px"; },
    "minheight": function(value){ this.$ext.style.minHeight = Math.max(0, value - apf.getHeightDiff(this.$ext)) + "px"; },
    "maxwidth": function(value){ this.$ext.style.maxWidth = Math.max(0, value - apf.getWidthDiff(this.$ext)) + "px"; },
    "maxheight": function(value){ this.$ext.style.maxHeight = Math.max(0, value - apf.getHeightDiff(this.$ext)) + "px"; },
    
    
    /**
     * @attribute {Boolean} focussable Sets or gets whether this element can receive the focus.
     * The focused element receives keyboard event.
     */
    "focussable": function(value){
        this.focussable = typeof value == "undefined" || value;

        if (value == "container") {
            this.$isWindowContainer = true;
            this.focussable = true;
        }
        else
            this.focussable = apf.isTrue(value);

        if (!this.hasFeature(apf.__FOCUSSABLE__)) //@todo should this be on the prototype
            this.implement(apf.Focussable);

        if (this.focussable) {
            apf.window.$addFocus(this, this.tabindex);
            
            if (value == "container")
                this.$tabList.remove(this);
        }
        else {
            apf.window.$removeFocus(this);
        }
    },

    /**
     * @attribute {Number} tabindex Sets or gets the tab index for this element.
     */    
    "tabindex": function(value){
        if (!this.hasFeature(apf.__FOCUSSABLE__)) 
            return;
        
        this.setTabIndex(parseInt(value) || null);
    },
    

    /**
     * @attribute {Number} zindex Sets or gets the z ordered layer in which this element is
     * drawn.
     */
    "zindex": function(value){
        this.$ext.style.zIndex = value;
    },

    /**
     * @attribute {Boolean} visible Sets or gets whether this element is shown.
     */
    "visible": function(value){
        if (apf.isFalse(value) || typeof value == "undefined") {
            if (this.$ext)
                this.$ext.style.display = "none";
            
            if (apf.document.activeElement == this || this.canHaveChildren == 2
              && apf.isChildOf(this, apf.document.activeElement, false)) {
                if (apf.config.allowBlur && this.hasFeature(apf.__FOCUSSABLE__))
                    this.blur();
                else
                    apf.window.moveNext();
            }
            
            this.visible = false;
        }
        else { //if (apf.isTrue(value)) default
            if (this.$ext) {
                this.$ext.style.display = ""; //Some form of inheritance detection
                if (!this.$ext.offsetHeight)
                    this.$ext.style.display = this.$display || "block";
            }
            
            
            if (apf.layout && this.$int) //apf.hasSingleRszEvent)
                apf.layout.forceResize(this.$int);//this.$int
            
            
            this.visible = true;
        }
    },

    /**
     * @attribute {Boolean} disabled Sets or gets whether this element's functions are active.
     * For elements that can contain other `apf.NODE_VISIBLE` elements, this
     * attribute applies to all its children.
     */
    "disabled": function(value){
        if (!this.$drawn) {
            var _self     = this;
            //this.disabled = false;

            this.addEventListener("DOMNodeInsertedIntoDocument", 
                this.$updateDisabled || (this.$updateDisabled = function(e){
                    apf.GuiElement.propHandlers.disabled.call(_self, _self.disabled);
                }));
            return;
        }
        else
            apf.queue.remove("disable" + this.$uniqueId);

        //For child containers we only disable its children
        if (this.canHaveChildren) {
            //@todo Fix focus here first.. else it will jump whilst looping
            if (value != -1)
                value = this.disabled = apf.isTrue(value);

            var nodes = this.childNodes;
            for (var node, i = 0, l = nodes.length; i < l; i++) {
                node = nodes[i];
                if (node.nodeFunc == apf.NODE_VISIBLE) {
                    if (value && node.disabled != -1)
                        node.$disabled = node.disabled || false;
                    node.setProperty("disabled", value ? -1 : false);
                }
            }

            //this.disabled = undefined;
            if (this.$isWindowContainer)
                return;
        }

        if (value == -1 || value == false) {
            //value = true;
        }
        else if (typeof this.$disabled == "boolean") {
            if (value === null) {
                value = this.$disabled;
                this.$disabled = null;
            }
            else {
                this.$disabled = value || false;
                return;
            }
        }

        if (apf.isTrue(value) || value == -1) {
            this.disabled = false;
            if (apf.document.activeElement == this) {
                apf.window.moveNext(true); //@todo should not include window
                if (apf.document.activeElement == this)
                    this.$blur();
            }

            if (this.hasFeature(apf.__PRESENTATION__))
                this.$setStyleClass(this.$ext, this.$baseCSSname + "Disabled");

            if (this.$disable)
                this.$disable();

            

            this.disabled = value;
        }
        else {
            if (this.hasFeature(apf.__DATABINDING__) && apf.config.autoDisable
              && !(!this.$bindings || this.xmlRoot))
                return false;

            this.disabled = false;

            if (apf.document.activeElement == this)
                this.$focus();

            if (this.hasFeature(apf.__PRESENTATION__))
                this.$setStyleClass(this.$ext, null, [this.$baseCSSname + "Disabled"]);

            if (this.$enable)
                this.$enable();

            
        }
    },

    /**
     * @attribute {Boolean} enables Sets or gets whether this element's functions are active.
     * For elements that can contain other `apf.NODE_VISIBLE` elements, this
     * attribute applies to all its children.
     */
    "enabled" : function(value){
       this.setProperty("disabled", !value);
    },

    /**
     * @attribute {Boolean} disable-keyboard Sets or gets whether this element receives
     * keyboard input. This allows you to disable keyboard independently from
     * focus handling.
     */
    "disable-keyboard": function(value){
        this.disableKeyboard = apf.isTrue(value);
    },
    
    /**
     * @attribute {String}  tooltip  Sets or gets the text displayed when a user hovers with 
     * the mouse over the element.
     */
    "tooltip" : function(value){
        this.$ext.setAttribute("title", (value || "") + (this.hotkey ? " ("
            + (apf.isMac ? apf.hotkeys.toMacNotation(this.hotkey) : this.hotkey) + ")" : ""));
    },
    
    
    /**
     * @attribute {String} contextmenu Sets or gets the name of the menu element that will
     * be shown when the user right clicks or uses the context menu keyboard
     * shortcut.
     *
     * #### Example
     * 
     * ```xml
     *  <a:menu id="mnuExample">
     *      <a:item>test</a:item>
     *      <a:item>test2</a:item>
     *  </a:menu>
     *   
     *  <a:list 
     *    contextmenu = "mnuExample" 
     *    width       = "200" 
     *    height      = "150" />
     *  <a:bar 
     *    contextmenu = "mnuExample" 
     *    width       = "200" 
     *    height      = "150" />
     * ```
     */
    "contextmenu": function(value){
        this.contextmenus = [value];
    },
    

    
    /**
     * @attribute {String} actiontracker Sets or gets the name of the [[apf.actiontracker action tracker]] that
     * is used for this element and its children. If the actiontracker doesn't
     * exist yet, it is created.
     *
     * #### Example
     *
     * In this example, the list uses a different action tracker than the two
     * textboxes which determine their actiontracker based on the one that
     * is defined on the bar.
     * 
     * ```xml
     *  <a:list actiontracker="newAT" />
     *
     *  <a:bar actiontracker="someAT">
     *      <a:textbox />
     *      <a:textbox />
     *  </a:bar>
     * ```
     */
    "actiontracker": function(value){
        if (!value) {
            this.$at = null;
        }
        else if (value.localName == "actiontracker") {
            this.$at = value;
        }
        else {
            
            this.$at = typeof value == "string" && self[value]
              ? apf.nameserver.get("actiontracker", value) || self[value].getActionTracker()
              : apf.setReference(value,
                  apf.nameserver.register("actiontracker",
                      value, new apf.actiontracker()));

            if (!this.$at.name)
                this.$at.name = value;
            
        }
    },
    
    
    //Load subAML
    /**
     * @attribute {String} aml Sets or gets  the {@link term.datainstruction data instruction} 
     * that loads new AML as children of this element.
     */
    "aml": function(value){
        this.replaceMarkup(value);
    }

    /*
     * @attribute {String} sets this aml element to be editable
     * that loads new aml as children of this element.
     */
    // @todo Doc WTF?
    
   
    
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__PRESENTATION__ = 1 << 9;


/**
 * All elements inheriting from this {@link term.baseclass baseclass} have skinning features. A skin is a description
 * of how the element is rendered. In the web browser, this is done using HTML
 * elements and CSS.
 *
 * #### Remarks
 *
 * The skin is set using the `skin` attribute. The skin of each element can be
 * changed at run time. Other than just changing the look of an element, a skin
 * change can help the user to perceive information in a different way. For 
 * example, a list element has a default skin, but can also use the thumbnail 
 * skin to display thumbnails of the {@link term.datanode data nodes}.
 *
 * #### Example
 *
 * A skin for an element is always built up out of a standard set of parts:
 *
 * ```xml
 *   <a:textbox name="textbox">
 *      <a:alias>
 *          ...
 *      </a:alias>
 *      <a:style><![CDATA[
 *          ...
 *      ]]></a:style>
 *  
 *      <a:presentation>
 *          <a:main>
 *              ...
 *          </a:main>
 *          ...
 *      </a:presentation>
 *   </a:textbox>
 * ```
 *
 * The alias contains a name that contains alternative names for the skin. The
 * style tags contain the CSS. The main tag contains the HTML elements that are
 * created when the component is created. Any other skin items are used to render
 * other elements of the widget. In this reference guide you will find these
 * skin items described on the pages of each widget.
 *
 * @class apf.Presentation
 * @define presentation
 * @inherits apf.GuiElement
 * @baseclass
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.5
 */
apf.Presentation = function(){
    this.$init(true);
};

(function(){
    this.$regbase = this.$regbase | apf.__PRESENTATION__;
    
    // *** Properties and Attributes *** //

    this.$supportedProperties.push("skin");
    
    /**
     * @attribute {String} skinset Sets or gets the skinset for
     * this element. If none are specified ,the `skinset` attribute
     * of the app settings is used. When that's not defined, the default skinset
     * is used.
     * 
     * #### Example
     *
     * ```xml
     *  <a:list skinset="perspex" />
     * ```
     */
    this.$propHandlers["skinset"] =

    
    /**
     * @attribute {String} skin Sets or gets the name of the skin in the skinset that defines 
     * how this element is rendered. When a skin is changed, the full state of the
     * element is kept, including its selection, all the
     * AML attributes, loaded data, and focus and disabled states.
     *
     * #### Example
     *
     * In XML:
     *
     * ```xml
     *  <a:list id="lstExample" skin="thumbnails" />
     * ```
     * 
     * Or, in JavaScript:
     *
     * ```javascript
     *  lstExample.setAttribute("skin", "list");
     * ```
     */
    this.$propHandlers["skin"] = function(value){
        if (!this.$amlLoaded) //If we didn't load a skin yet, this will be done when we attach to a parent
            return;

        if (!this.$skinTimer) {
            var _self = this;
            clearTimeout(this.$skinTimer);
            this.$skinTimer = $setTimeout(function(){
                changeSkin.call(_self, _self.skin);
                delete _self.$skinTimer;
            });
        }
    }
    

    /**
     * @attribute {String} style Sets or gets the CSS style applied to the this element. This can be a string containing one or more CSS rules.
     */
    this.$propHandlers["style"] = function(value){
        if (!this.styleAttrIsObj && this.$amlLoaded)
            this.$ext.setAttribute("style", value);
    }
    
    /**
     * @attribute {String} border Sets or gets border values. Set these sizes as a quarter of strings, in the usual top, right, bottom, left sequence, or pass an empty string to turn off borders.
     */
    this.$propHandlers["border"] = function(value){
        if (!value)
            this.$ext.style.borderWidth = "";
        else
            this.$ext.style.borderWidth = apf.getBox(value).join("px ") + "px";
    }
    
    /**
     * @attribute {String | Number} margin Sets or gets margin values. Set these sizes as a quarter of strings, in the usual top, right, bottom, left sequence, or pass an empty string to turn off margins.
     */
    this.$propHandlers["margin"] = function(value){
        if (!value)
            this.$ext.style.margin = "";
        else
            this.$ext.style.margin = apf.getBox(value).join("px ") + "px";
    }

    /**
     * @attribute {String} class Sets or gets the name of the CSS style class applied to this element.
     */
    this.$propHandlers["class"] = function(value){
        this.$setStyleClass(this.$ext, value, this.$lastClassValue ? [this.$lastClassValue] : null);
        this.$lastClassValue = value;
    }

    
    this.$forceSkinChange = function(skin, skinset){
        changeSkin.call(this, skin, skinset);
    }

    //@todo objects don't always have an $int anymore.. test this
    function changeSkin(skin, skinset){
        clearTimeout(this.$skinTimer);

        //var skinName = (skinset || this.skinset || apf.config.skinset)
        //    + ":" + (skin || this.skin || this.localName);

        
        //Store selection
        if (this.selectable)
            var valueList = this.getSelection();//valueList;
        

        //Store needed state information
        var oExt       = this.$ext,
            oInt       = this.$int,
            pNode      = this.$ext.parentNode,
            beforeNode = oExt.nextSibling,
            idExt      = this.$ext.getAttribute("id"),
            idInt      = this.$int && this.$int.getAttribute("id"),
            oldBase    = this.$baseCSSname;

        if (oExt.parentNode)
            oExt.parentNode.removeChild(oExt);

        //@todo changing skin will leak A LOT, should call $destroy here, with some extra magic
        if (this.$destroy)
            this.$destroy(true);

        //Load the new skin
        this.skin = skin;
        this.$loadSkin(skinset ? skinset + ":" + skin : null);

        //Draw
        if (this.$draw)
            this.$draw(true);

        if (idExt)
            this.$ext.setAttribute("id", idExt);

        if (beforeNode || pNode != this.$ext.parentNode)
            pNode.insertBefore(this.$ext, beforeNode);

        //Style
        
        //Border
        
        //Margin

        //Classes
        var i, l, newclasses = [],
               classes    = (oExt.className || "").splitSafe("\\s+");
        for (i = 0; i < classes.length; i++) {
            if (classes[i] && classes[i].indexOf(oldBase) != 0)
                newclasses.push(classes[i].replace(oldBase, this.$baseCSSname));
        }
        apf.setStyleClass(this.$ext, newclasses.join(" "));

        //Copy events
        var en, ev = apf.skins.events;
        for (i = 0, l = ev.length; i < l; i++) {
            en = ev[i];
            if (typeof oExt[en] == "function" && !this.$ext[en])
                this.$ext[en] = oExt[en];
        }

        //Copy css state (dunno if this is best)
        this.$ext.style.left     = oExt.style.left;
        this.$ext.style.top      = oExt.style.top;
        this.$ext.style.width    = oExt.style.width;
        this.$ext.style.height   = oExt.style.height;
        this.$ext.style.right    = oExt.style.right;
        this.$ext.style.bottom   = oExt.style.bottom;
        this.$ext.style.zIndex   = oExt.style.zIndex;
        this.$ext.style.position = oExt.style.position;
        this.$ext.style.display  = oExt.style.display;

        //Widget specific
        //if (this.$loadAml)
            //this.$loadAml(this.$aml);
        
        if (idInt)
            this.$int.setAttribute("id", idInt);
        
        if (this.$int && this.$int != oInt) {
            var node, newNode = this.$int, nodes = oInt.childNodes;
            for (var i = nodes.length - 1; i >= 0; i--) {
                if ((node = nodes[i]).host) {
                    node.host.$pHtmlNode = newNode;
                    if (node.host.$isLeechingSkin)
                        setLeechedSkin.call(node.host);
                }
                newNode.insertBefore(node, newNode.firstChild);
            }
            //this.$int.onresize = oInt.onresize;
        }
        
        
        //DragDrop
        if (this.hasFeature(apf.__DRAGDROP__)) {
            if (document.elementFromPointAdd) {
                document.elementFromPointRemove(oExt);
                document.elementFromPointAdd(this.$ext);
            }
        }
        

        //Check disabled state
        if (this.disabled)
            this.$disable(); //@todo apf3.0 test

        //Check focussed state
        if (this.$focussable && apf.document.activeElement == this)
            this.$focus(); //@todo apf3.0 test

        //Dispatch event
        this.dispatchEvent("$skinchange", {
            ext  : oExt,
            "int": oInt
        });

        
        //Reload data
        if (this.hasFeature(apf.__DATABINDING__) && this.xmlRoot)
            this.reload();
        else
        
        if (this.value)
            this.$propHandlers["value"].call(this, this.value);

        
        //Set Selection
        if (this.hasFeature(apf.__MULTISELECT__)) {
            if (this.selectable)
                this.selectList(valueList, true);
        }
        

        //Move layout rules
        if (!apf.hasSingleRszEvent) {
            apf.layout.activateRules(this.$ext);
            if (this.$int)
                apf.layout.activateRules(this.$int);
        }

/*        
        if (this.hasFeature(apf.__ANCHORING__))
            this.$recalcAnchoring();
        

        
        if (this.hasFeature(apf.__ALIGNMENT__)) {
            if (this.aData)
                this.aData.oHtml = this.$ext;

            if (this.pData) {
                this.pData.oHtml = this.$ext;
                this.pData.pHtml = this.$int;

                var nodes = this.pData.childNodes;
                for (i = 0; i < nodes.length; i++)
                    nodes[i].pHtml = this.$int; //Should this be recursive??
            }
        }
        
*/
        
        if (this.draggable && this.$propHandlers["draggable"]) //@todo move these to the event below apf3.0)
            this.$propHandlers["draggable"].call(this, this.draggable);
        if (this.resizable && this.$propHandlers["resizable"])
            this.$propHandlers["resizable"].call(this, this.resizable);
        

        
        apf.layout.forceResize(this.$ext);
        
    };
    

    // *** Private methods *** //

    this.$setStyleClass = apf.setStyleClass;

    function setLeechedSkin(e){
        if (!this.$amlLoaded || e && (e.$isMoveWithinParent 
          || e.currentTarget != this || !e.$oldParent))
            return;

        this.$setInheritedAttribute(this, "skinset");

        if (this.attributes.getNamedItem("skin"))
            return;

        //e.relatedNode
        var skinName, pNode = this.parentNode, skinNode;
        if ((skinName = this.$canLeechSkin.dataType 
          == apf.STRING ? this.$canLeechSkin : this.localName)
          && pNode.$originalNodes 
          && (skinNode = pNode.$originalNodes[skinName])
          && skinNode.getAttribute("inherit")) {
            var link = skinNode.getAttribute("link");
            this.$isLeechingSkin = true;
            if (link) {
                this.$forceSkinChange(link);
            }
            else {
                var skin = pNode.skinName.split(":");
                this.$forceSkinChange(skin[1], skin[0]);
            }
        }
        else if (this.$isLeechingSkin) {
            delete this.skin;
            this.$isLeechingSkin = false;
            this.$forceSkinChange();
        }
    }

    //Skin Inheritance
    //@todo Probably requires some cleanup
    this.$initSkin = function(x){
        if (this.$canLeechSkin) {
            this.addEventListener("DOMNodeInserted", setLeechedSkin);
        }
        
        if (!this.skin)
            this.skin = this.getAttribute("skin");
        
        var skinName, pNode = this.parentNode, skinNode;
        if (this.$canLeechSkin && !this.skin 
          && (skinName = this.$canLeechSkin.dataType == apf.STRING 
            ? this.$canLeechSkin 
            : this.localName)
          && pNode && pNode.$originalNodes 
          && (skinNode = pNode.$originalNodes[skinName])
          && skinNode.getAttribute("inherit")) {
            var link = skinNode.getAttribute("link");
            this.$isLeechingSkin = true;
            if (link) {
                this.skin = link;
                this.$loadSkin();
            }
            else {
                this.$loadSkin(pNode.skinName);
            }
        }
        else {
            if (!this.skinset)
                this.skinset = this.getAttribute("skinset");
            
            this.$loadSkin(null, this.$canLeechSkin);
        }
    }

    /*
     * Initializes the skin for this element when none has been set up.
     *
     * @param  {String}  skinName   Identifier for the new skin (for example: 'default:List' or 'win').
     * @param  {Boolean} [noError]
     */
    this.$loadSkin = function(skinName, noError){
        //this.skinName || //where should this go and why?
        this.baseSkin = skinName || (this.skinset 
            || this.$setInheritedAttribute("skinset")) 
            + ":" + (this.skin || this.localName);

        clearTimeout(this.$skinTimer);

        if (this.skinName) {
            this.$blur();
            this.$baseCSSname = null;
        }

        this.skinName = this.baseSkin; //Why??
        //this.skinset  = this.skinName.split(":")[0];

        this.$pNodes = {}; //reset the this.$pNodes collection
        this.$originalNodes = apf.skins.getTemplate(this.skinName, true);

        if (!this.$originalNodes) {
            var skin = this.skin;
            if (skin) {
                var skinset = this.skinName.split(":")[0];
                this.baseName = this.skinName = "default:" + skin;
                this.$originalNodes = apf.skins.getTemplate(this.skinName);
                
                if (!this.$originalNodes && skinset != "default") {
                    this.baseName = this.skinName = skinset + ":" + this.localName;
                    this.$originalNodes = apf.skins.getTemplate(this.skinName, true);
                }
            }
            
            if (!this.$originalNodes) {
                this.baseName = this.skinName = "default:" + this.localName;
                this.$originalNodes = apf.skins.getTemplate(this.skinName);
            }

            if (!this.$originalNodes) {
                if (noError) {
                    return (this.baseName = this.skinName = 
                        this.originalNode = null);
                }
                
                throw new Error(apf.formatErrorString(1077, this,
                    "Presentation",
                    "Could not load skin: " + this.baseSkin));
            }
            
            //this.skinset = this.skinName.split(":")[0];
        }

        if (this.$originalNodes)
            apf.skins.setSkinPaths(this.skinName, this);
    };

    this.$getNewContext = function(type, amlNode){
        

        this.$pNodes[type] = this.$originalNodes[type].cloneNode(true);
    };

    this.$hasLayoutNode = function(type){
        

        return this.$originalNodes[type] ? true : false;
    };

    this.$getLayoutNode = function(type, section, htmlNode){
        

        var node = this.$pNodes[type] || this.$originalNodes[type];
        if (!node) {
            
            return false;
        }

        if (!section)
            return htmlNode || apf.getFirstElement(node);

        var textNode = node.getAttribute(section);
        if (!textNode)
            return null;

        return (htmlNode
            ? apf.queryNode(htmlNode, textNode)
            : apf.getFirstElement(node).selectSingleNode(textNode));
    };

    this.$getOption = function(type, section){
        type = type.toLowerCase(); //HACK: lowercasing should be solved in the comps.

        //var node = this.$pNodes[type];
        var node = this.$pNodes[type] || this.$originalNodes[type];
        if (!section || !node)
            return node;//apf.getFirstElement(node);
        var option = node.selectSingleNode("@" + section);

        return option ? option.nodeValue : "";
    };

    this.$getExternal = function(tag, pNode, func, aml){
        if (!pNode)
            pNode = this.$pHtmlNode;
        if (!tag)
            tag = "main";
        //if (!aml)
            //aml = this.$aml;

        tag = tag.toLowerCase(); //HACK: make components case-insensitive

        this.$getNewContext(tag);
        var oExt = this.$getLayoutNode(tag);
        
        var node;
        if (node = (aml || this).getAttributeNode("style"))
            oExt.setAttribute("style", node.nodeValue);

        //if (node = (aml || this).getAttributeNode("class"))
            //this.$setStyleClass(oExt, (oldClass = node.nodeValue));

        if (func)
            func.call(this, oExt);

        oExt = apf.insertHtmlNode(oExt, pNode);
        oExt.host = this;
        if (node = (aml || this).getAttributeNode("bgimage"))
            oExt.style.backgroundImage = "url(" + apf.getAbsolutePath(
                this.mediaPath, node.nodeValue) + ")";

        if (!this.$baseCSSname)
            this.$baseCSSname = oExt.className.trim().split(" ")[0];

        return oExt;
    };

    // *** Focus *** //
    this.$focus = function(){
        if (!this.$ext)
            return;

        this.$setStyleClass(this.oFocus || this.$ext, this.$baseCSSname + "Focus");
    };

    this.$blur = function(){
        
        if (this.renaming)
            this.stopRename(null, true);
        

        if (!this.$ext)
            return;

        this.$setStyleClass(this.oFocus || this.$ext, "", [this.$baseCSSname + "Focus"]);
    };

    this.$fixScrollBug = function(){
        if (this.$int != this.$ext)
            this.oFocus = this.$int;
        else {
            this.oFocus =
            this.$int   =
                this.$ext.appendChild(document.createElement("div"));

            this.$int.style.height = "100%";
            this.$int.className = "focusbug"
        }
    };

    // *** Caching *** //
    /*
    this.$setClearMessage    = function(msg){};
    this.$updateClearMessage = function(){}
    this.$removeClearMessage = function(){};*/
}).call(apf.Presentation.prototype = new apf.GuiElement());

apf.config.$inheritProperties["skinset"] = 1;






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__VALIDATION__ = 1 << 6;



//if checkequal then notnull = true
apf.validator = {
    macro : {
        

        //var temp
        "pattern"     : "value.match(",
        "pattern_"    : ")",
        "custom"      : "(",
        "custom_"     : ")",
        "min"         : "parseInt(value) >= ",
        "max"         : "parseInt(value) <= ",
        "maxlength"   : "value.toString().length <= ",
        "minlength"   : "value.toString().length >= ",
        "notnull"     : "value.toString().length > 0",
        "checkequal"  : "!(temp = ",
        "checkequal_" : ").isValid() || temp.getValue() == value"
    },
    
    compile : function(options){
        var m = this.macro, s = ["var temp, valid = true; \
            if (!validityState) \
                validityState = new apf.validator.validityState(); "];

        if (options.required) {
            s.push("if (checkRequired && (!value || value.toString().trim().length == 0)) {\
                validityState.$reset();\
                validityState.valueMissing = true;\
                valid = false;\
            }")
        }
        
        s.push("validityState.$reset();\
            if (value) {");
        
        for (prop in options) {
            if (!m[prop]) continue;
            s.push("if (!(", m[prop], options[prop], m[prop + "_"] || "", ")){\
                validityState.$set('", prop, "');\
                valid = false;\
            }");
        }

        s.push("};validityState.valid = valid; return validityState;");
        return new Function('value', 'checkRequired', 'validityState', s.join(""));
    }
};

/**
 * Object containing information about the validation state. It contains
 * properties that specify whether a certain validation was passed.
 * Remarks:
 * This is part of {@link http://www.whatwg.org/specs/web-apps/current-work/multipage/forms.html#validitystatethe HTML 5 specification}.
 */
apf.validator.validityState = function(){
    this.valueMissing    = false,
    this.typeMismatch    = false,
    this.patternMismatch = false,
    this.tooLong         = false,
    this.rangeUnderflow  = false,
    this.rangeOverflow   = false,
    this.stepMismatch    = false,
    this.customError     = false,
    this.valid           = true,

    this.$reset = function(){
        for (var prop in this) {
            if (prop.substr(0,1) == "$") 
                continue;
            this[prop] = false;
        }
        this.valid = true;
    },

    this.$set = function(type) {
        switch (type) {
            case "min"         : this.rangeUnderflow  = true; break;
            case "max"         : this.rangeOverflow   = true; break;
            case "minlength"   : this.tooShort        = true; break;
            case "maxlength"   : this.tooLong         = true; break;
            case "pattern"     : this.patternMismatch = true; break;
            case "datatype"    : this.typeMismatch    = true; break;
            case "notnull"     : this.typeMismatch    = true; break;
            case "checkequal"  : this.typeMismatch    = true; break;
        }
    }
};

/**
 * All elements inheriting from this {@link term.baseclass baseclass} have validation support.
 *
 * #### Example
 * 
 * ```xml, demo
 * <a:application xmlns:a="http://ajax.org/2005/aml">
 *   <!-- startcontent -->
 *   <a:bar validgroup="vgExample">
 *        <a:label>Number</a:label>
 *        <a:textbox required="true" min="3" max="10" 
 *          invalidmsg="Invalid Entry;Please enter a number between 3 and 10" />
 *        <a:label>Name</a:label>
 *        <a:textbox required="true" minlength="3" 
 *          invalidmsg="Invalid Entry;Please enter your name" />
 *        <a:label>Message</a:label>
 *        <a:textarea required="true" 
 *          invalidmsg="Invalid Message;Please enter a message!" />
 *   
 *        <a:button onclick="if(vgExample.isValid()) alert('valid!')">
 *            Validate
 *        </a:button>
 *    </a:bar>
 *   <!-- endcontent -->
 * </a:application>
 * ```
 * 
 * @class apf.Validation
 * @inherits apf.AmlElement
 * @baseclass
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.5
 */
 /**
  * @event invalid    Fires when this component goes into an invalid state.
  *
  */
apf.Validation = function(){
    this.$regbase = this.$regbase | apf.__VALIDATION__;

    /**
     * Checks if this element's value is valid.
     *
     * @param  {Boolean} [checkRequired] Specifies whether this check also adheres to the `'required'` rule.
     * @returns  {Boolean} Specifies whether the value is valid
     * @see  apf.ValidationGroup
     * @see  element.submitform
     */
    this.isValid = function(checkRequired){
        if (!this.$vOptions)
            return true;
        
        (this.$vOptions.isValid || (this.$vOptions.isValid
          = apf.validator.compile(this.$vOptions))).call(this,
            typeof this.getValue == "function" ? this.getValue(null, true) : null, 
            checkRequired, this.validityState || 
            (this.validityState = new apf.validator.validityState()));
        
        var valid = this.validityState.valid;
        
        
        
        this.dispatchEvent(!valid ? "invalid" : "valid", this.validityState);
            
        return valid;
    };

    /*
     * @private
     */
    this.setCustomValidity = function(message){
        //do stuff
    }

    /*
     * @private
     * @todo This method should also scroll the element into view
     */
    this.showMe = function(){
        var p = this.parentNode;
        while (p) {
            if (p.show)
                p.show();
            p = p.parentNode;
        }
    };

    
    
    /**
     * Puts this element in the error state, optionally showing the
     * error box if this element is invalid.
     *
     * @method validate
     * @param  {Boolean} [ignoreReq]  Specifies whether this element required check is turned on.
     * @param  {Boolean} [nosetError] Specifies whether the error box is displayed if this component does not validate.
     * @param  {Boolean} [force]      Specifies whether this element is in the error state, and doesn't check if the element's value is invalid.
     * @return  {Boolean}  Indicates whether the value is valid
     * @see  apf.ValidationGroup
     */
    this.validate = function(ignoreReq, nosetError, force){
        //if (!this.$validgroup) return this.isValid();

        if (force || !this.isValid(!ignoreReq) && !nosetError) {
            this.setError();
            return false;
        }
        else {
            this.clearError();
            return true;
        }
    };

    /*
     *    @private
     */
    this.setError = function(value){
        if (!this.$validgroup)
            this.$propHandlers["validgroup"].call(this, "vg" + this.parentNode.$uniqueId);

        var errBox = this.$validgroup.getErrorBox(this);

        if (!this.$validgroup.allowMultipleErrors)
            this.$validgroup.hideAllErrors();

        errBox.setMessage(this.invalidmsg || value);
        
        apf.setStyleClass(this.$ext, this.$baseCSSname + "Error");
        this.showMe(); //@todo scroll refHtml into view

        if (this.invalidmsg || value)
            errBox.display(this);
        
        
        
        if (apf.document.activeElement && apf.document.activeElement != this)
            this.focus(null, {mouse:true}); //arguable...
    };

    /*
     *    @private
     */
    this.clearError = function(value){
        if (this.$setStyleClass)
            this.$setStyleClass(this.$ext, "", [this.$baseCSSname + "Error"]);

        if (this.$validgroup) {
            var errBox = this.$validgroup.getErrorBox(null, true);
            if (!errBox || errBox.host != this)
                return;

            errBox.hide();
        }
    };

    this.addEventListener("DOMNodeRemovedFromDocument", function(e){
        if (this.$validgroup)
            this.$validgroup.unregister(this);
    });

    /**
     *
     * @attribute  {Boolean}  required     Sets or gets whether a valid value for this element is required.
     */
    /** 
     * @attribute  {RegExp}   pattern      Sets or gets the pattern tested against the value of this element to determine it's validity.
     */
    /**
     * @attribute  {String}   datatype     Sets or gets the datatype that the value of this element should adhere to. This can be any 
     * of a set of predefined types, or a simple type created by an XML Schema definition. 
     * 
     * Some possible values (all of which are [[String]]s) include:
     *   - `xsd:dateTime`
     *   - `xsd:time`
     *   - `xsd:date`
     *   - `xsd:gYearMonth`
     *   - `xsd:gYear`
     *   - `xsd:gMonthDay`
     *   - `xsd:gDay`
     *   - `xsd:gMonth`
     *   - `xsd:string`
     *   - `xsd:boolean`
     *   - `xsd:base64Binary`
     *   - `xsd:hexBinary`
     *   - `xsd:float`
     *   - `xsd:decimal`
     *   - `xsd:double`
     *   - `xsd:anyURI`
     *   - `xsd:integer`
     *   - `xsd:nonPositiveInteger`
     *   - `xsd:negativeInteger`
     *   - `xsd:long`
     *   - `xsd:int`
     *   - `xsd:short`
     *   - `xsd:byte`
     *   - `xsd:nonNegativeInteger`
     *   - `xsd:unsignedLong`
     *   - `xsd:unsignedInt`
     *   - `xsd:unsignedShort`
     *   - `xsd:unsignedByte`
     *   - `xsd:positiveInteger`
     *   - `apf:url`
     *   - `apf:website`
     *   - `apf:email`
     *   - `apf:creditcard`
     *   - `apf:expdate`
     *   - `apf:wechars`
     *   - `apf:phonenumber`
     *   - `apf:faxnumber`
     *   - `apf:mobile`
     */
    /**
     * @attribute  {Number}  min          Sets or gets the minimal value for which the value of this element is valid.
     */
    /**
     * @attribute  {Number}  max          Sets or gets the maximum value for which the value of this element is valid.
     */
    /**
     * @attribute  {Number}  minlength    Sets or gets the minimal length allowed for the value of this element.
     */
    /**
     * @attribute  {Number}  maxlength    Sets or gets the maximum length allowed for the value of this element.
     */
    /**
     * @attribute  {Boolean}  notnull      Sets or gets whether the value is filled. This rule is checked realtime when the element loses the focus.
     */
    /**
     * @attribute  {String}   checkequal   Sets or gets the id of the element to check if it has the same value as this element.
     */
    /**
     * @attribute  {String}   invalidmsg   Sets or gets the message displayed when this element has an invalid value. Use a `;` character to seperate the title from the message.
     */
    /**
     * @attribute  {String}   validgroup   Sets or gets the identifier for a group of items to be validated at the same time. This identifier can be new. It is inherited from a AML node upwards.
     */
    /**
     * @attribute  {String}   validtest    Sets or gets the instruction on how to test for success. This attribute is generally used to check the value on the server.
     * 
     * #### Example
     *
     * This example shows how to check the username on the server. In this case,
     * `comm.loginCheck` is an async RPC function that checks the availability of the
     * username. If it exists, it will return `0`--otherwise, it's `1`. The value variable
     * contains the current value of the element (in this case the textbox). It
     * can be used as a convenience variable.
     *
     * ```xml
     *  <a:label>Username</a:label>
     *  <a:textbox
     *    validtest  = "{comm.loginCheck(value) == 1}"
     *    pattern    = "/^[a-zA-Z0-9_\-. ]{3,20}$/"
     *    invalidmsg = "Invalid username;Please enter a valid username." />
     * ```
     */
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        //this.addEventListener(this.hasFeature(apf.__MULTISELECT__) ? "onafterselect" : "onafterchange", onafterchange);
        /* Temp disabled, because I don't understand it (RLD)
        this.addEventListener("beforechange", function(){
            if (this.xmlRoot && apf.getBoundValue(this) === this.getValue())
                return false;
        });*/
        
        // validgroup
        if (!this.validgroup)
            this.$setInheritedAttribute("validgroup");
    });
    
    //1 = force no bind rule, 2 = force bind rule
    this.$attrExcludePropBind = apf.extend({
        pattern   : 1,
        validtest : 3
    }, this.$attrExcludePropBind);

    this.$booleanProperties["required"] = true;
    this.$supportedProperties.push("validgroup", "required", "datatype",
        "pattern", "min", "max", "maxlength", "minlength", "validtest",
        "notnull", "checkequal", "invalidmsg", "requiredmsg");

    this.$fValidate = function(){
        if (this.liveedit)
            return;
        
        if (!this.$validgroup)
            this.validate(true);
        else {
             var errBox = this.$validgroup.getErrorBox(this);
             if (!errBox.visible || errBox.host != this)
                this.validate(true);
        }
    };
    this.addEventListener("blur", this.$fValidate);
    
    this.$propHandlers["validgroup"] = function(value){
        if (value) {
            var vgroup;
            if (typeof value != "string") {
                this.$validgroup = value.name;
                vgroup = value;
            }
            else {
                
                vgroup = apf.nameserver.get("validgroup", value);
                
            }

            this.$validgroup = vgroup || new apf.ValidationGroup(value);
            this.$validgroup.register(this);
            /*
                @todo What about children, when created after start
                See button login action
            */
        }
        else {
            this.$validgroup.unregister(this);
            this.$validgroup = null;
        }
    };
    
    this.$propHandlers["pattern"]    = function(value, prop){
        if (value.substr(0, 1) != "/")
            value = "/" + value + "/";

        (this.$vOptions || (this.$vOptions = {}))[prop] = value;
        delete this.$vOptions.isValid;
    };
    
    
    this.$propHandlers["required"]   = 
    this.$propHandlers["custom"]     = 
    this.$propHandlers["min"]        = 
    this.$propHandlers["max"]        = 
    this.$propHandlers["maxlength"]  = 
    this.$propHandlers["minlength"]  = 
    this.$propHandlers["notnull"]    = 
    this.$propHandlers["checkequal"] = function(value, prop){
        (this.$vOptions || (this.$vOptions = {}))[prop] = value;
        delete this.$vOptions.isValid;
    };
    
    //@todo rewrite this for apf3.0 - it should just execute a live markup
    this.$propHandlers["validtest"] = function(value){
        var _self = this, rvCache = {};
        /**
         * Removes the validation cache created by the validtest rule.
         */
        this.removeValidationCache = function(){
            rvCache = {};
        }
        
        this.$checkRemoteValidation = function(){
            var value = this.getValue();
            if(typeof rvCache[value] == "boolean") return rvCache[value];
            if(rvCache[value] == -1) return true;
            rvCache[value] = -1;

            apf.getData(this.validtest.toString(), {
               xmlNode : this.xmlRoot,
               value   : this.getValue(),
               callback : function(data, state, extra){
                  if (state != apf.SUCCESS) {
                      if (state == apf.TIMEOUT && extra.retries < apf.maxHttpRetries)
                          return extra.tpModule.retry(extra.id);
                      else {
                          var commError = new Error(apf.formatErrorString(0, _self, 
                            "Validating entry at remote source", 
                            "Communication error: \n\n" + extra.message));

                          if (_self.dispatchEvent("error", apf.extend({
                            error : commError, 
                            state : status
                          }, extra)) !== false)
                              throw commError;
                          return;
                      }
                  }

                  rvCache[value] = apf.isTrue(data);//instr[1] ? data == instr[1] : apf.isTrue(data);
                  
                  if(!rvCache[value]){
                    if (!_self.hasFocus())
                        _self.setError();
                  }
                  else _self.clearError();
              }
            });
            
            return true;
        };
        
        (this.$vOptions || (this.$vOptions = {})).custom = "apf.lookup(" + this.$uniqueId + ").$checkRemoteValidation()";
        delete this.$vOptions.isValid;
    };
};


apf.GuiElement.propHandlers["required"]   = 
apf.GuiElement.propHandlers["pattern"]    = 
apf.GuiElement.propHandlers["min"]        = 
apf.GuiElement.propHandlers["max"]        = 
apf.GuiElement.propHandlers["maxlength"]  = 
apf.GuiElement.propHandlers["minlength"]  = 
apf.GuiElement.propHandlers["notnull"]    = 
apf.GuiElement.propHandlers["checkequal"] = 
apf.GuiElement.propHandlers["validtest"]  = function(value, prop){
    this.implement(apf.Validation);
    this.$propHandlers[prop].call(this, value, prop);
}

/**
 * This object allows for a set of AML elements to be validated. An element that 
 * is not valid shows an errorbox.
 *
 * #### Example
 *
 * ```xml
 *  <a:bar validgroup="vgForm">
 *      <a:label>Phone number</a:label>
 *      <a:textbox id="txtPhone"
 *        required   = "true"
 *        pattern    = "(\d{3}) \d{4} \d{4}"
 *        invalidmsg = "Incorrect phone number entered" />
 *
 *      <a:label>Password</a:label>
 *      <a:textbox
 *        required   = "true"
 *        mask       = "password"
 *        minlength  = "4"
 *        invalidmsg = "Please enter a password of at least four characters" />
 *  </a:bar>
 * ```
 *
 * To check if the element has valid information entered, leaving the textbox
 * (focussing another element) will trigger a check. Programmatically, a check
 * can be done using the following code:
 *
 * 
 * ```javascript
 *  txtPhone.validate();
 *
 *  // Or, use the html5 syntax
 *  txtPhone.checkValidity();
 * ```
 *
 * To check for the entire group of elements, use the validation group. For only
 * the first non-valid element the errorbox is shown. That element also receives
 * focus.
 * 
 * ```javascript
 *  vgForm.validate();
 * ```
 *
 * @class apf.ValidationGroup
 * @inherits apf.Class
 * @default_private
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.9
 */
/**
 * @event validation Fires when the validation group isn't validated.
 */
apf.ValidationGroup = function(name){
    this.$init();
    
    this.childNodes = [];
    
    if (name)
        apf.setReference(name, this);
    
    this.name = name || "validgroup" + this.$uniqueId;
    
    apf.nameserver.register("validgroup", this.name, this);
    
};

(function(){
    /**
     * When set to true, only visible elements are validated.
     * @type Boolean
     */
    this.validateVisibleOnly = false;
    
    /**
     * When set to true, validation doesn't stop at the first invalid element.
     * @type Boolean
     */
    this.allowMultipleErrors = false;

    /**
     * Adds an AML element to this validation group.
     * @param o {apf.AmlElement} The AML element to add
     */
    this.register   = function(o){ 
        if (o.hasFeature(apf.__VALIDATION__)) 
            this.childNodes.push(o);
    };
    
    /**
     * Removes a AML element from this validation group.
     * @param o {apf.AmlElement} The AML element to remove
     */
    this.unregister = function(o){
        this.childNodes.remove(o); 
    };

    /**
     * Returns a string representation of this object.
     */
    this.toString = function(){
        return "[APF Validation Group]";
    };

    //Shared among all validationgroups
    var errbox;
    /**
     * Retrieves the {@link apf.errorbox} used for a specified element.
     *
     * @param  {apf.AmlNode}  o An AMLNode specifying the element for which the Errorbox should be found. If none is found, an Errorbox is created. Use the {@link apf.ValidationGroup.allowMultipleErrors} to influence when Errorboxes are created.
     * @param  {Boolean}  no_create    Boolean that specifies whether new Errorbox may be created when it doesn't exist already
     * @return  {apf.errorbox}  The found (or created) Errorbox
     */
    this.getErrorBox = function(o, no_create){
        if (this.allowMultipleErrors || !errbox && !no_create) {
            errbox            = new apf.errorbox();
            errbox.$pHtmlNode = o.$ext.parentNode;
            errbox.skinset    = apf.getInheritedAttribute(o.parentNode, "skinset"); //@todo use skinset here. Has to be set in presentation
            errbox.dispatchEvent("DOMNodeInsertedIntoDocument");
        }
        return errbox;
    };

    /**
     * Hide all Errorboxes for the elements using this element as their validation group.
     *
     */
    this.hideAllErrors = function(){
        if (errbox && errbox.host)
            errbox.host.clearError();
    };

    function checkValidChildren(oParent, ignoreReq, nosetError){
        var found;
        //Per Element
        for (var v, i = 0; i < oParent.childNodes.length; i++) {
            var oEl = oParent.childNodes[i];

            if (!oEl)
                continue;
            if (!oEl.disabled
              && (!this.validateVisibleOnly && oEl.visible || !oEl.$ext || oEl.$ext.offsetHeight)
              && (oEl.hasFeature(apf.__VALIDATION__) && oEl.isValid && !oEl.isValid(!ignoreReq))) {
                //|| !ignoreReq && oEl.required && (!(v = oEl.getValue()) || new String(v).trim().length == 0)
                
                if (!nosetError) {
                    if (!found) {
                        oEl.validate(true, null, true);
                        found = true;
                        if (!this.allowMultipleErrors)
                            return true; //Added (again)
                    }
                    else if (oEl.errBox && oEl.errBox.host == oEl)
                        oEl.errBox.hide();
                }
                else if (!this.allowMultipleErrors)
                    return true;
            }
            if (oEl.canHaveChildren && oEl.childNodes.length) {
                found = checkValidChildren.call(this, oEl, ignoreReq, nosetError) || found;
                if (found && !this.allowMultipleErrors)
                    return true; //Added (again)
            }
        }
        return found;
    }

    
    
    /**
     * 
     * @inheritDoc apf.ValidationGroup.isValid
     * @method
     */
    this.validate =
    
    /**
     * Checks if (part of) the set of element's registered to this element are
     * valid. When an element is found with an invalid value, the error state can
     * be set for that element.
     *
     * @method isValid
     * @param  {Boolean}    [ignoreReq]  Specifies whether to adhere to the 'required' check.
     * @param  {Boolean}    [nosetError  Specifies whether to not set the error state of the element with an invalid value
     * @param  {apf.AmlElement} [page]   The page for which the children will be checked. When not specified all elements of this validation group are checked.
     * @return  {Boolean}  Specifies whether the checked elements are valid.
     */
    this.isValid = function(ignoreReq, nosetError, page){
        var found = checkValidChildren.call(this, page || this, ignoreReq,
            nosetError);

        if (page) {
            
                if (page.validation && !eval(page.validation)) {
                    alert(page.invalidmsg);
                    found = true;
                }
            
        }

        //Global Rules
        //
        //if (!found)
            //found = this.dispatchEvent("validation");

        return !found;
    };
}).call(apf.ValidationGroup.prototype = new apf.Class());

apf.config.$inheritProperties["validgroup"] = 1;






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__DATABINDING__ = 1 << 1;



/**
 * This is a baseclass that adds data binding features to this element. 
 * Databinding takes care of automatically going from data to representation and establishing a
 * permanent link between the two. In this way data that is changed will
 * change the representation as well. Furthermore, actions that are executed on
 * the representation will change the underlying data.
 * 
 * #### Example
 *
 * ```xml
 *  <a:list>
 *      <a:model>
 *          <data>
 *              <item icon="ajax_org.gif">Item 1</item>
 *              <item icon="ajax_org.gif">Item 2</item>
 *          </data>
 *      </a:model>
 *      <a:bindings>
 *          <a:icon match="[@icon]" />
 *          <a:caption match="[text()]" />
 *          <a:each match="[item]" />
 *      </a:bindings>
 *  </a:list>
 * ```
 *
 * @class apf.DataBinding
 * @inherits apf.Presentation
 * @baseclass
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.4
 * @default_private
 */
/**
 * @event error             Fires when a communication error has occured while
 *                          making a request for this element.
 * @cancelable Prevents the error from being thrown.
 * @bubbles
 * @param {Object} e The standard event object. It contains the following properties:
 *                   - error ([[Error]]): the error object that is thrown when the event callback doesn't return false.
 *                   - state ([[Number]]): the state of the call
 *                                                - `apf.SUCCESS`:  The request was successfull
 *                                                - `apf.TIMEOUT`:  The request has timed out.
 *                                                - `apf.ERROR  `:  An error has occurred while making the request.
 *                                                - `apf.OFFLINE`:  The request was made while the application was offline.
 *                   - userdata (`Mixed`): Data that the caller wanted to be available in the callback of the http request.
 *                   - http ([[XMLHttpRequest]]): The object that executed the actual http request.
 *                   - url ([[String]]): The url that was requested.
 *                   - tpModule ([[apf.http]]): The teleport module that is making the request.
 *                   - id ([[Number]]): The ID of the request.
 *                   - message ([[String]]): The error message.
 */
/** 
 * @event beforeretrieve    Fires before a request is made to retrieve data.
 * @cancelable Prevents the data from being retrieved.
 */
/**
 * @event afterretrieve     Fires when the request to retrieve data returns both
 *                          on success and failure.
 */
/**
 * @event receive           Fires when data is successfully retrieved
 * @param {Object} e The standard event object. It contains the following properties:
 *                   - data ([[String]]): the retrieved data
 *
 */
apf.DataBinding = function(){
    this.$init(true);
    
    this.$loadqueue = 
    this.$dbTimer   = null;
    this.$regbase   = this.$regbase | apf.__DATABINDING__;
    this.$mainBind  = "value";
    
    this.$bindings     = 
    this.$cbindings    = 
    this.$attrBindings = false;

    //1 = force no bind rule, 2 = force bind rule
    this.$attrExcludePropBind = apf.extend({
        model     : 1,
        each      : 1
        //eachvalue : 1 //disabled because of line 1743 valueRule = in multiselect.js
    }, this.$attrExcludePropBind);

    // *** Public Methods *** //

    /**
     * Sets a value of an XMLNode based on an xpath statement executed on the data of this model.
     *
     * @param  {String}  xpath  The xpath used to select a XMLNode
     * @param  {String}  value  The value to set
     * @return  {XMLNode}  The changed XMLNode
     */
    this.setQueryValue = function(xpath, value, type){
        var node = apf.createNodeFromXpath(this[type || 'xmlRoot'], xpath);
        if (!node)
            return null;

        apf.setNodeValue(node, value, true);
        //apf.xmldb.setTextNode(node, value);
        return node;
    };

    /**
     * Queries the bound data for a string value
     *
     * @param {String} xpath  The XPath statement which queries on the data this element is bound on.
     * @param {String} type   The node that is used as the context node for the query. It can be one of the following possible values:
     *                         - `"selected"`:    The selected data anode of this element.
     *                         - `"xmlRoot"`:   The root data node that this element is bound on.
     *                         - `"indicator"`:   The data node that is highlighted for keyboard navigation.
     * @return {String}       The value of the selected XML Node
     * 
     */
    this.queryValue = function(xpath, type){
     /*   @todo
     *  lstRev.query('revision/text()', 'selected');
     *  lstRev.query('revision/text()', 'xmlRoot');
     *  lstRev.query('revision/text()', 'indicator');
     */
        return apf.queryValue(this[type || 'xmlRoot'], xpath );
    };
	/**
     * Queries the bound data for an array of string values
     *
     * @param {String} xpath The XPath statement which queries on the data this element is bound on.
     * @param {String} type   The node that is used as the context node for the query. It can be one of the following possible values:
     *                         - `"selected"`:    The selected data anode of this element.
     *                         - `"xmlRoot"`:   The root data node that this element is bound on.
     *                         - `"indicator"`:   The data node that is highlighted for keyboard navigation.
     * @return {String}       The value of the selected XML Node
     */
    this.queryValues = function(xpath, type){
        return apf.queryValues(this[type || 'xmlRoot'], xpath );
    };
	
    /**
     * Executes an XPath statement on the data of this model
     *
     * @param {String} xpath The XPath statement which queries on the data this element is bound on.
     * @param {String} type   The node that is used as the context node for the query. It can be one of the following possible values:
     *                         - `"selected"`:    The selected data anode of this element.
     *                         - `"xmlRoot"`:   The root data node that this element is bound on.
     *                         - `"indicator"`:   The data node that is highlighted for keyboard navigation.
     * @return  {Mixed}  An [[XMLNode]] or [[NodeList]] with the result of the selection
     */
    this.queryNode = function(xpath, type){
        var n = this[type||'xmlRoot'];
		return n ? n.selectSingleNode(xpath) : null;
    };

    /**
     * Executes an XPath statement on the data of this model
     *
     * @param  {String}   xpath    The XPath used to select the XMLNode(s)
     * @param {String} type   The node that is used as the context node for the query. It can be one of the following possible values:
     *                         - `"selected"`:    The selected data anode of this element.
     *                         - `"xmlRoot"`:   The root data node that this element is bound on.
     *                         - `"indicator"`:   The data node that is highlighted for keyboard navigation.
     * @return  {Mixed}  An [[XMLNode]] or [[NodeList]] with the result of the selection
     */
    this.queryNodes = function(xpath, type){
        var n = this[type||'xmlRoot'];
		return n ? n.selectNodes(xpath) : [];
    };
	
    this.$checkLoadQueue = function(){
        // Load from queued load request
        if (this.$loadqueue) {
            if (!this.caching)
                this.xmlRoot = null;
            var q = this.load(this.$loadqueue[0], {cacheId: this.$loadqueue[1]});
            if (!q || q.dataType != apf.ARRAY || q != this.$loadqueue)
                this.$loadqueue = null;
        }
        else return false;
    };
    
    //setProp
    this.$execProperty = function(prop, xmlNode, undoObj){
        var attr = this.$attrBindings[prop];
        
        //@todo this is a hacky solution for replaceNode support - Have to rethink this.
        if (this.nodeType == 7) {
            if (xmlNode != this.xmlRoot)
                this.xmlRoot = xmlNode;
        }
        
        
        
        

        
        try {
        
            if (attr.cvalue.asyncs) { //if async
                var _self = this;
                return attr.cvalue.call(this, xmlNode, function(value){
                    _self.setProperty(prop, value, true);
                    
                    
                
                }); 
            }
            else {
                var value = attr.cvalue.call(this, xmlNode);
            }
        
        }
        catch(e){
            apf.console.warn("[400] Could not execute binding for property "
                + prop + "\n\n" + e.message);
            return;
        }
        
        
        this.setProperty(prop, undoObj && undoObj.extra.range || value, true); //@todo apf3.0 range
        
        
    };
    
    //@todo apf3.0 contentEditable support
    this.$applyBindRule = function(name, xmlNode, defaultValue, callback, oHtml){
        var handler = this.$attrBindings[name] 
          && this.$attrBindings[name].cvalue || this.$cbindings[name];

        return handler ? handler.call(this, xmlNode, callback) : defaultValue || "";
    };

    
    
    this.$hasBindRule = function(name){
        return this.$attrBindings[name] || this.$bindings 
          && this.$bindings[name];
    };
    
    this.$getBindRule = function(name, xmlNode){
        return this.$attrBindings[name] || this.$bindings 
          && this.$bindings.getRule(name, xmlNode);
    };
    
    var ruleIsMatch = {"drag":1,"drop":1,"dragcopy":1}
    this.$getDataNode = function(name, xmlNode, createNode, ruleList, multiple){
        var node, rule = this.$attrBindings[name];
        if (rule) { //@todo apf3.0 find out why drag and drop rules are already compiled here
            if (rule.cvalue.type != 3) //@todo warn here?
                return false;
            
            var func = rule.cvalue2 || rule.compile("value", {
                xpathmode  : multiple ? 4 : 3,
                parsecode  : 1,
                injectself : ruleIsMatch[name]
            });
            if (func && (node = func(xmlNode, createNode))) {
                if (ruleList)
                    ruleList.push(rule);

                return node;
            }
            
            return false;
        }
        
        return this.$bindings 
           && this.$bindings.getDataNode(name, xmlNode, createNode, ruleList, multiple);
    };
    
    
    /**
     * Sets the model of the specified element. 
     * The model acts as a datasource for this element.
     *
     * @param  {apf.model}  The model this element is going to connect to.
     * 
     */
    this.setModel = function(model){
        this.setAttribute("model", model, false, true);
    };
    
    
    /**
     * Gets the model which this element is connected to.
     * The model acts as a datasource for this element.
     *
     * @param {Boolean} doRecur Specifies whether the model should be searched recursively up the data tree.
     * @returns  {apf.model}  The model this element is connected to.
     * @see apf.smartbinding
     */
    this.getModel = function(doRecur){
        if (doRecur && !this.$model)
            return this.dataParent ? this.dataParent.parent.getModel(true) : null;

        return this.$model;
    };
    
    /**
     * Reloads the data in this element.
     * @method 
     */
    this.reload = this.reload || function(){
        this.load(this.xmlRoot, {cacheId: this.cacheId, force: true});
    };

    /**
     * @event beforeload  Fires before loading data in this element.
     * @cancelable Prevents the data from being loaded.
     * @param {XMLElement} xmlNode The node that is loaded as the root {@link term.datanode data node}.
     *   
     */
    /** 
     * @event afterload   Fires after loading data in this element.
     * @param {XMLElement} xmlNode The node that is loaded as the root {@link term.datanode data node}.
     */
    /**
     * Loads data into this element using binding rules to transform the
     * data into a presentation.
     * 
     * #### Example
     * 
     * ```xml 
     *  <a:list id="lstExample">
     *      <a:bindings>
     *          <a:caption match="[text()]" />
     *          <a:icon match="[@icon]" />
     *          <a:each match="[image]" />
     *      </a:bindings>
     *  </a:list>
     *  
     *  <a:script><!--
     *      apf.onload = function() {
     *      lstExample.load('<images>\
     *          <image icon="icoTest.gif">image 1</image>\
     *          <image icon="icoTest.gif">image 2</image>\
     *          <image icon="icoTest.gif">image 3</image>\
     *          </images>');
     *      }
     *  --></a:script>
     * ```
     *
     * @param {XMLElement | String}  [xmlNode] The content to load into this element. It can be one of the following values:
     *                                                - {XMLElement}: An XML element that's loaded into this element
     *                                                - {String}: Either an XML string, or, an instruction to load the data from a remote source
     *                                                - `null`: Clears this element from its data
     * @param {Object} [options] Set of additional options to pass. Properties include:
     *                           - [xmlNode] ([[XMLElement]]):   The {@link term.datanode data node} that provides
     *                                                       context to the data instruction.
     *                           - [callback] ([[Function]]): The code executed when the data request returns
     *                           - [properties] (`Mixed`): Custom properties available in the data instruction
     *                           - [cacheId] ([[String]]): The xml element to which the binding rules are applied
     *                           - [force] ([[Boolean]]): Specifies whether cache is checked before loading the data
     *                           - [noClearMsg] ([[Boolean]]): Specifies whether a message is set when clear is called
     */
    this.load = function(xmlNode, options){
        if (options) {
            var cacheId      = options.cacheId,
                forceNoCache = options.force,
                noClearMsg   = options.noClearMsg;
        }
        if (cacheId && cacheId == this.cacheId && !forceNoCache)
            return;

        
        if (apf.popup.isShowing(this.$uniqueId))
            apf.popup.forceHide(); //This should be put in a more general position
        

        // Convert first argument to an xmlNode we can use;
        if (xmlNode) {
            if (typeof xmlNode == "string") {
                if (xmlNode.charAt(0) == "<")
                    xmlNode = apf.getXmlDom(xmlNode).documentElement;
                else {
                    return apf.model.prototype.$loadFrom.call(this, xmlNode, options);
                }
            }
            else if (xmlNode.nodeType == 9) {
                xmlNode = xmlNode.documentElement;
            }
            else if (xmlNode.nodeType == 3 || xmlNode.nodeType == 4) {
                xmlNode = xmlNode.parentNode;
            }
            else if (xmlNode.nodeType == 2) {
                xmlNode = xmlNode.ownerElement 
                    || xmlNode.parentNode 
                    || xmlNode.selectSingleNode("..");
            }
        }

        // If control hasn't loaded databinding yet, queue the call
        if (this.$preventDataLoad || !this.$canLoadData 
          && ((!this.$bindings && (!this.$canLoadDataAttr || !this.each)) || !this.$amlLoaded) 
          && (!this.hasFeature(apf.__MULTISELECT__) || !this.each) 
          || this.$canLoadData && !this.$canLoadData()) {
            
            if (!this.caching || !this.hasFeature(apf.__CACHE__)) {
                
                //@todo this is wrong. It is never updated when there are only
                //Property binds and then it just leaks xml nodes
                this.xmlRoot = xmlNode;
                
                
                this.setProperty("root", this.xmlRoot);
                
            }
            
            
            
            return this.$loadqueue = [xmlNode, cacheId];
        }
        this.$loadqueue = null;

        // If no xmlNode is given we clear the control, disable it and return
        if (this.dataParent && this.dataParent.xpath)
            this.dataParent.parent.signalXmlUpdate[this.$uniqueId] = !xmlNode;

        if (!xmlNode && (!cacheId || !this.$isCached || !this.$isCached(cacheId))) {
            

            this.clear(noClearMsg);

            
            if (apf.config.autoDisable && this.$createModel === false)
                this.setProperty("disabled", true);

            //@todo apf3.0 remove , true in clear above
            //this.setProperty("selected", null);
            
            return;
        }
        
        // If reloading current document, and caching is disabled, exit
        if (!this.caching && !forceNoCache && xmlNode 
          && !this.$loadqueue && xmlNode == this.xmlRoot)
            return;

        var disabled = this.disabled;
        this.disabled = false;

        //Run onload event
        if (this.dispatchEvent("beforeload", {xmlNode : xmlNode}) === false)
            return false;

        

        this.clear(true, true);

        this.cacheId = cacheId;

        if (this.dispatchEvent("$load", {
          forceNoCache : forceNoCache, 
          xmlNode  : xmlNode
        }) === false) {
            //delete this.cacheId;
            return;
        }
        
        //Set usefull vars
        this.documentId = apf.xmldb.getXmlDocId(xmlNode);
        this.xmlRoot    = xmlNode;
        
        
        this.setProperty("root", this.xmlRoot);
        

        

        // Draw Content
        this.$load(xmlNode);
        
        

        // Check if subtree should be loaded
        this.$loadSubData(xmlNode);

        if (this.$createModel === false) {
            this.disabled = true;
            this.setProperty("disabled", false);
        }
        else
            this.disabled = disabled;

        // Run onafteronload event
        this.dispatchEvent('afterload', {xmlNode : xmlNode});
    };
    
    // @todo Doc
    /*
     * @binding load Determines how new data is loaded data is loaded into this
     * element. Usually this is only the root node containing no children.
     * 
     * #### Example
     * 
     * This example shows a load rule in a text element. It gets its data from
     * a list. When a selection is made on the list the data is loaded into the
     * text element.
     * 
     * ```xml
     *  <a:list id="lstExample" width="200" height="200">
     *      <a:bindings>
     *          <a:caption match="[text()]" />
     *          <a:value match="[text()]" />
     *          <a:each match="[message]" />
     *      </a:bindings>
     *      <a:model>
     *          <messages>
     *              <message id="1">message 1</message>
     *              <message id="2">message 2</message>
     *          </messages>
     *      </a:model>
     *  </a:list>
     * 
     *  <a:text model="{lstExample.selected}" width="200" height="150">
     *      <a:bindings>
     *          <a:load get="http://localhost/getMessage.php?id=[@id]" />
     *          <a:contents match="[message/text()]" />
     *      </a:bindings>
     *  </a:text>
     * ```
     *
     */
     /**
      * @attribute {String} get Sets or gets the {@link term.datainstruction data instruction}
      *                     that is used to load data into the XML root of this component.
      */
    this.$loadSubData = function(xmlRootNode){
        if (this.$hasLoadStatus(xmlRootNode) && !this.$hasLoadStatus(xmlRootNode, "potential")) 
            return;

        //var loadNode = this.$applyBindRule("load", xmlRootNode);
        var rule = this.$getBindRule("load", xmlRootNode);
        if (rule && (!rule[1] || rule[1](xmlRootNode))) {
            
            
            this.$setLoadStatus(xmlRootNode, "loading");

            if (this.$setClearMessage)
                this.$setClearMessage(this["loading-message"], "loading");

            //||apf.xmldb.findModel(xmlRootNode)
            var mdl = this.getModel(true);
            

            var amlNode = this;
            if (mdl.$insertFrom(rule.getAttribute("get"), {
              xmlNode     : xmlRootNode,  //@todo apf3.0
              insertPoint : xmlRootNode, //this.xmlRoot,
              amlNode     : this,
              callback    : function(){
                    
                    amlNode.setProperty(amlNode.hasFeature(apf.__MULTISELECT__) 
                        ? "selected" 
                        : "root", xmlRootNode);
                    
                }
              }) === false
            ) {
                this.clear(true);
                
                if (apf.config.autoDisable)
                    this.setProperty("disabled", true);

                //amlNode.setProperty("selected", null); //@todo is this not already done in clear?
                
            }
        }
    };
    
    //@todo this function is called way too much for a single load of a tree
    //@todo should clear listener
    /*
     * Unloads data from this element and resets state displaying an empty message.
     * The empty message is set on the {@link apf.GuiElement.msg}.
     *
     * @param {Boolean} [nomsg]   Specifies whether to display the empty message.
     * @param {Boolean} [doEvent] Specifies whether to send select events.
     * @see baseclass.databinding.method.load
     * @private
     */
    this.clear = function(nomsg, doEvent, fakeClear){
        if (!this.$container)
            return;//@todo apf3.0

        if (this.clearSelection)
            this.clearSelection(true);//!doEvent);//@todo move this to the $clear event in multiselect.js

        var lastHeight = this.$container.offsetHeight;

        if (this.dispatchEvent("$clear") !== false)
            this.$container.innerHTML = ""; //@todo apf3.0

        if (typeof nomsg == "string") {
            var msgType = nomsg;
            nomsg = false;
            
            //@todo apf3.0 please use attr. inheritance
            if (!this[msgType + "-message"]) {
                this.$setInheritedAttribute(msgType + "-message");
            }
        }
        this.$lastClearType = msgType || null;

        if (!nomsg && this.$setClearMessage) {
            this.$setClearMessage(msgType 
              ? this[msgType + "-message"] 
              : this["empty-message"], msgType || "empty", lastHeight);

            //this.setProperty("selected", null); //@todo apf3.0 get the children to show loading... as well (and for each selected, null
            //c[i].o.clear(msgType, doEvent);
        }
        else if(this.$removeClearMessage)
            this.$removeClearMessage();

        if (!fakeClear)
            this.documentId = this.xmlRoot = this.cacheId = null;

        
        if (!nomsg) {
            if (this.hasFeature(apf.__MULTISELECT__)) //@todo this is all wrong
                this.setProperty("length", 0);
            //else 
                //this.setProperty("value", ""); //@todo redo apf3.0
        }
        
    };
    
    this.clearMessage = function(msg){
        this.customMsg = msg;
        this.clear("custom");
    };

    //@todo optimize this
    /**
     * @private
     */
    this.$setLoadStatus = function(xmlNode, state, remove){
        var group  = this.loadgroup || "default";
        var re     = new RegExp("\\|(\\w+)\\:" + group + ":(\\d+)\\|");
        var loaded = xmlNode.getAttribute("a_loaded") || "";

        var m;        
        if (!remove && (m = loaded.match(re)) && m[1] != "potential" && m[2] != this.$uniqueId)
            return;
        
        //remove old status if any
        var ostatus = loaded.replace(re, "")
        if (!remove)
            ostatus += "|" + state + ":" + group + ":" + this.$uniqueId + "|";

        xmlNode.setAttribute("a_loaded", ostatus);
    };

    /**
     * @private
     */
    this.$removeLoadStatus = function(xmlNode){
        this.$setLoadStatus(xmlNode, null, true);
    };

    /**
     * @private
     */
    this.$hasLoadStatus = function(xmlNode, state, unique){
        if (!xmlNode)
            return false;
        var ostatus = xmlNode.getAttribute("a_loaded");
        if (!ostatus)
            return false;
    
        var group  = this.loadgroup || "default";
        var re     = new RegExp("\\|" + (state || "\\w+") + ":" + group + ":" + (unique ? this.$uniqueId : "\\d+") + "\\|");
        return ostatus.match(re) ? true : false;
    };

    /*
     * @event beforeinsert Fires before data is inserted.
     * @cancelable Prevents the data from being inserted.
     * @param {XMLElement} xmlParentNode The parent in which the new data is inserted
     */
     /**
      * @event afterinsert Fires after data is inserted.
     */

    /**
     * @private
     */
    this.insert = function(xmlNode, options){
        if (typeof xmlNode == "string") {
            if (xmlNode.charAt(0) == "<") {
                
                if (options.whitespace === false)
                    xmlNode = xmlNode.replace(/>[\s\n\r]*</g, "><");
                
                xmlNode = apf.getXmlDom(xmlNode).documentElement;
            }
            else {
                if (!options.insertPoint)
                    options.insertPoint = this.xmlRoot;
                return apf.model.prototype.$insertFrom.call(this, xmlNode, options);
            }
        }
        
        var insertPoint = options.insertPoint || this.xmlRoot;

        if (this.dispatchEvent("beforeinsert", {
          xmlParentNode : insertPoint
        }) === false)
            return false;

        //Integrate XMLTree with parentNode
        if (typeof options.copyAttributes == "undefined")
            options.copyAttributes = true;
        
        if (this.filterUnique)
            options.filter = this.filterUnique;
        
        var newNode = apf.mergeXml(xmlNode, insertPoint, options);
        
        this.$isLoading = true; //Optimization for simpledata

        //Call __XMLUpdate on all listeners
        apf.xmldb.applyChanges("insert", insertPoint);
        
        this.$isLoading = false;

        //Select or propagate new data
        if (this.selectable && this.autoselect) {
            if (this.xmlNode == newNode)
                this.$selectDefault(this.xmlNode);
        }
        
        else if (this.xmlNode == newNode) {
            this.setProperty("root", this.xmlNode);
        }
        

        if (this.$hasLoadStatus(insertPoint, "loading"))
            this.$setLoadStatus(insertPoint, "loaded");

        this.dispatchEvent("afterinsert");

        //Check Connections
        //this one shouldn't be called because they are listeners anyway...(else they will load twice)
        //if(this.selected) this.setConnections(this.selected, "select");
    };
    
    /**
     * @attribute {Boolean} render-root Sets or gets whether the XML element loaded into this
     * element is rendered as well. The default is false.
     *
     * #### Example
     *
     * This example shows a tree which also renders the root element.
     * 
     * ```xml
     *  <a:tree render-root="true">
     *      <a:model>
     *          <root name="My Computer">
     *              <drive name="C">
     *                  <folder name="/Program Files" />
     *                  <folder name="/Desktop" />
     *              </drive>
     *          </root>
     *      </a:model>
     *      <a:bindings>
     *          <a:caption match="[@name]"></a:caption>
     *          <a:each match="[root|drive|folder]"></a:each>
     *      </a:bindings>
     *  </a:tree>
     * ```
     */
    this.$booleanProperties["render-root"] = true;
    this.$supportedProperties.push("empty-message", "loading-message",
        "offline-message", "render-root", "smartbinding",
        "bindings", "actions");

    /**
     * @attribute {Boolean} render-root Sets or gets whether the root node of the data loaded
     * into this element is rendered as well. 
     * @see apf.tree
     */
    this.$propHandlers["render-root"] = function(value){
        this.renderRoot = value;
    };
    
    /**
     * @attribute {String} empty-message Sets or gets the message displayed by this element
     * when it contains no data. This property is inherited from parent nodes.
     * When none is found, it is looked for on the appsettings element. Otherwise
     * it defaults to the string "No items".
     */
    this.$propHandlers["empty-message"] = function(value){
        this["empty-message"] = value;

        if (this.$updateClearMessage) 
            this.$updateClearMessage(this["empty-message"], "empty");
    };

    /**
     * @attribute {String} loading-message Sets or gets the message displayed by this
     * element when it's loading. This property is inherited from parent nodes.
     * When none is found, it is looked for on the appsettings element. Otherwise
     * it defaults to the string "Loading...".
     *
     * #### Example
     *
     * This example uses property bindings to update the loading message. The
     * position of the progressbar should be updated by the script taking care
     * of loading the data.
     *
     * ```xml
     *  <a:list loading-message="{'Loading ' + Math.round(progress1.value*100) + '%'}" />
     *  <a:progressbar id="progress1" />
     * ```
     *
     * #### Remarks
     *
     * Usually, a static loading message is displayed for only 100 milliseconds
     * or so, whilst loading the data from the server. For instance, this is done
     * when the load binding rule is used. In the code example below, a list
     * binds on the selection of a tree displaying folders. When the selection
     * changes, the list loads new data by extending the model. During the load
     * of this new data, the loading message is displayed.
     * 
     * ```xml
     *  <a:list model="[trFolders::element]">
     *      <a:bindings>
     *          ...
     *          <a:load get="{comm.getFiles([@path])}" />
     *      </bindings>
     *  </a:list>
     * ```
     */
    this.$propHandlers["loading-message"] = function(value){
        this["loading-message"] = value;

        if (this.$updateClearMessage)
            this.$updateClearMessage(this["loading-message"], "loading");
    };

    /**
     * @attribute {String} offline-message Sets or gets the message displayed by this
     * element when it can't load data because the application is offline.
     * This property is inherited from parent nodes. When none is found it is
     * looked for on the appsettings element. Otherwise it defaults to the
     * string "You are currently offline...".
     */
    this.$propHandlers["offline-message"] = function(value){
        this["offline-message"] = value;

        if (this.$updateClearMessage)
            this.$updateClearMessage(this["offline-message"], "offline");
    };

    /**
     * @attribute {String} smartbinding Sets or gets the name of the SmartBinding for this
     * element. 
     * 
     * A smartbinding is a collection of rules which define how data
     * is transformed into representation, how actions on the representation are
     * propagated to the data and it's original source, how drag&drop actions
     * change the data and where the data is loaded from. Each of these are
     * optionally defined in the smartbinding set and can exist independently
     * of the smartbinding object.
     * 
     * #### Example
     *
     * This example shows a fully specified smartbinding. Usually, only parts
     * are used. This example shows a tree with files and folders.
     * 
     * ```xml
     *  <a:tree smartbinding="sbExample" />
     * 
     *  <a:smartbinding id="sbExample">
     *      <a:bindings>
     *          <a:caption  match  = "[@caption|@filename]"/>
     *          <a:icon     match  = "[file]"
     *                      value  = "icoFile.gif" />
     *          <a:icon     value  = "icoFolder.gif" />
     *          <a:each     match  = "[file|folder|drive]" />
     *          <a:drag     match  = "[folder|file]" />
     *          <a:drop     match  = "[folder]" 
     *                      target = "[root]"
     *                      action = "tree-append" />
     *          <a:drop     match  = "[folder]" 
     *                      target = "[folder]"
     *                      action = "insert-before" />
     *          <a:drop     match  = "[file]"   
     *                      target = "[folder|root]" 
     *                      action = "tree-append" />
     *          <a:drop     match  = "[file]"   
     *                      target = "[file]"
     *                      action = "insert-before" />
     *      </a:bindings>
     *      <a:actions>
     *          <a:remove set = "remove.php?path=[@path]" />
     *          <a:rename set = "move.php?from=oldValue&amp;to=[@path]" />
     *      </a:actions>
     *      <a:model src="xml/filesystem.xml" />
     *  </a:smartbinding>
     * ```
     * 
     * #### Remarks
     *
     * The smartbinding parts can also be assigned to an element by adding them
     * directly as a child in aml.
     * 
     * ```xml
     *  <a:tree>
     *      <a:bindings>
     *          ...
     *      </bindings>
     *      <a:model />
     *  </a:tree>
     * </code>
     *
     * ### See Also
     *
     * There are several ways to be less verbose in assigning certain rules. For more information, see:
     *
     * * [[apf.bindings]]
     * * [[apf.actions]]
     * * [[apf.DragDrop]]
     * 
     */
    this.$propHandlers["smartbinding"] = 
    
    /**
     * @attribute {String} actions Sets or gets the id of the actions element which
     * provides the action rules for this element. Action rules are used to
     * send changes on the bound data to a server.
     *
     * #### Example
     *
     * ```xml
     *  <a:tree 
     *    id             = "tree" 
     *    height         = "200" 
     *    width          = "250" 
     *    actions       = "actExample"
     *    model          = "xml/filesystem.xml"
     *    actiontracker  = "atExample"
     *    startcollapsed = "false" 
     *    onerror        = "alert('Sorry this action is not permitted');return false">
     *      <a:each match="[folder|drive]">
     *          <a:caption match="[@caption|@filename]" />
     *          <a:icon value="Famfolder.gif" />
     *      </a:each>
     *  </a:tree>
     *  
     *  <a:actions id="actExample">
     *      <a:rename match = "[file]"   
     *               set    = "rename_folder.php?id=[@fid]" />
     *      <a:rename match = "[folder]" 
     *               set    = "rename_file.php?id=[@fid]" />
     *  </a:actions>
     *  
     *  <a:button 
     *    caption = "Rename"
     *    right   = "10" 
     *    top     = "10"
     *    onclick = "tree.startRename()" />
     *  <a:button onclick="tree.getActionTracker().undo();">Undo</a:button>
     * ```
     */
    this.$propHandlers["actions"] = 

    /**
     * @attribute {String} bindings Sets or gets the id of the bindings element which
     * provides the binding rules for this element.
     * 
     * #### Example
     *
     * This example shows a set of binding rules that transform data into the
     * representation of a list. In this case it displays the names of
     * several email accounts, with after each account name the number of unread
     * mails in that account. It uses JSLT to transform the caption.
     * 
     * ```xml
     *  <a:model id="mdlExample">
     *      <data>
     *          <account icon="application.png">Account 1
     *              <mail read="false" />
     *              <mail read="false" />
     *              <mail read="true" />
     *          </account>
     *          <account icon="application.png">Account 2</account>
     *      </data>
     *  </a:model>
     *  <a:list bindings="bndExample" model="mdlExample" />
     * 
     *   <a:bindings id="bndExample">
     *      <a:caption>[text()] (#[mail[@read != 'true']])</a:caption>
     *      <a:icon match="[@icon]" />
     *      <a:each match="[account]" sort="[text()]" />
     *  </a:bindings>
     * ```
     * 
     * #### Remarks
     *
     * Bindings can also be assigned directly by putting the bindings tag as a
     * child of this element.
     *
     * If the rule only contains a select attribute, it can be written in a
     * short way by adding an attribute with the name of the rule to the
     * element itself:
     * 
     * ```xml
     *  <a:list 
     *    caption = "[text()] (#[mail[@read != 'true']])"
     *    icon    = "[@icon]"
     *    each    = "[account]"
     *    sort    = "[text()]" />
     * ```
     */
    this.$propHandlers["bindings"] = function(value, prop){
        var local = "$" + prop + "Element";
        if (this[local])
            this[local].unregister(this);
        
        if (!value)
            return;
        
        

        apf.nameserver.get(prop, value).register(this);
        
        
        if (prop != "actions" && 
          this.$checkLoadQueue() === false && this.$amlLoaded)
            1+1; //@todo add reload queue.
            //this.reload();
    };

    
    var eachBinds = {"caption":1, "icon":1, "select":1, "css":1, "sort":1,
                     "drop":2, "drag":2, "dragcopy":2, "eachvalue":1}; //Similar to apf.Class
    
    this.$addAttrBind = function(prop, fParsed, expression) {
        //Detect if it uses an external model
        if (fParsed.models) {
            
            if (this.hasFeature(apf.__MULTISELECT__)) {
                
            }
            
        }

        //Set listener for all models
        var i, xpath, modelId, model,
            paths = fParsed.xpaths,
            list  = {};
        //@todo when there is no model in xpath modelId == null...
        for (i = 0; i < paths.length; i+=2) {
            if (!list[(modelId = paths[i])])
                list[modelId] = 1;
            else list[modelId]++
        }
        
        if (!this.$propsUsingMainModel)
            this.$propsUsingMainModel = {};

        var rule = (this.$attrBindings || (this.$attrBindings = {}))[prop] = {
            cvalue  : fParsed,
            value   : expression,
            compile : apf.BindingRule.prototype.$compile,
            models  : []
        };

        delete this.$propsUsingMainModel[prop];
        for (xpath, i = 0; i < paths.length; i+=2) {
            modelId = paths[i];
            if (list[modelId] == -1)
                continue;

            xpath = paths[i + 1];

            if (modelId == "#" || xpath == "#") {
                var m = (rule.cvalue3 || (rule.cvalue3 = apf.lm.compile(rule.value, {
                    xpathmode: 5
                }))).call(this, this.xmlRoot);
                
                //@todo apf3 this needs to be fixed in live markup
                if (typeof m != "string") {
                    model = m.model && m.model.$isModel && m.model;
                    if (model)
                        xpath = m.xpath;
                    else if (m.model) {
                        model = typeof m.model == "string" ? apf.xmldb.findModel(m.model) : m.model;
                        xpath = apf.xmlToXpath(m.model, model.data) + (m.xpath ? "/" + m.xpath : ""); //@todo make this better
                    }
                    else {
                        //wait until model becomes available
                        this.addEventListener("prop." + prop, function(e){
                            var m = (rule.cvalue3 || (rule.cvalue3 = apf.lm.compile(rule.value, {
                                xpathmode: 5
                            }))).call(this, this.xmlRoot);
                            
                            if (m.model) {
                                this.removeEventListener("prop." + prop, arguments.callee);
                                var _self = this;
                                $setTimeout(function(){
                                    _self.$clearDynamicProperty(prop);
                                    _self.$setDynamicProperty(prop, expression);
                                }, 10);
                            }
                        });
                        continue;
                    }
                }
                else model = null;
            }
            else model = null;

            if (!model) {
                if (modelId) {
                    
                    //@todo apf3.0 how is this cleaned up???
                    //Add change listener to the data of the model
                    model = apf.nameserver.get("model", modelId) //is model creation useful here?
                        || apf.setReference(modelId, apf.nameserver.register("model", modelId, new apf.model()));
                    
                }
                else {
                    if (!this.$model && !this.$initingModel)
                        initModel.call(this);
    
                    model = this.$model;

                    if (!this.hasFeature(apf.__MULTISELECT__) 
                      && eachBinds[prop] != 2 || !eachBinds[prop]) //@experimental - should not set this because model will load these attributes
                        this.$propsUsingMainModel[prop] = {
                            xpath    : xpath,
                            optimize : list[modelId] == 1
                        };
                }
            }
            
            //@todo warn here if no model??
            if (model && (!this.hasFeature(apf.__MULTISELECT__) 
              && eachBinds[prop] != 2 || !eachBinds[prop])) {
                //Create the attribute binding
                //@todo: remove listenRoot = expression.indexOf("*[") > -1 -> because it doesnt make sense in certain context. recheck selection handling
                model.$bindXmlProperty(this, prop, xpath, list[modelId] == 1); 
                rule.models.push(model);
            }
            
            list[modelId] = -1;
        }
        
        rule.xpath = xpath;

        this.$canLoadDataAttr = eachBinds[prop] == 1; //@todo apf3.0 remove
        this.$checkLoadQueue();
    }
    
    this.$removeAttrBind = function(prop){
        //@todo apf3.0
        //$model.$unbindXmlProperty
        var rule = this.$attrBindings[prop]
        if (!rule)
            return;
        
        delete this.$attrBindings[prop];
        delete this.$propsUsingMainModel[prop]
        
        var models = rule.models;
        if (models.length)
            for (var i = 0; i < models.length; i++) {
                models[i].$unbindXmlProperty(this, prop);
            }
        else if (this.$model)
            this.$model.$unbindXmlProperty(this, prop);
    };
    
    this.$initingModel;
    function initModel(){
        this.$initingModel = true;
        this.$setInheritedAttribute("model");
    }
    
    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        //Set empty message if there is no data
        if (!this.model && this.$setClearMessage && !this.value)
            this.$setClearMessage(this["empty-message"], "empty");
        
        this.$amlLoaded = true; //@todo this can probably be removed
        this.$checkLoadQueue();
    });
    
    

    /**
     * @attribute {String} model Sets or gets the name of the model to load data from, or a
     * datainstruction to load data.
     *
     * #### Example
     *
     * ```xml
     *  <a:model id="mdlExample" src="filesystem.xml" />
     *   <a:tree 
     *     height         = "200" 
     *     width          = "250" 
     *     model          = "mdlExample">
     *       <a:each match="[folder|drive]">
     *           <a:caption match="[@caption]" />
     *           <a:icon value="Famfolder.gif" />
     *       </a:each>
     *   </a:tree>
     * ```
     * 
     * #### Example
     *
     * Here's an example loading from an XML source:
     *
     * ```xml
     *  <a:tree 
     *    height         = "200" 
     *    width          = "250" 
     *    model          = "filesystem.xml">
     *      <a:each match="[folder|drive]">
     *          <a:caption match="[@caption]" />
     *          <a:icon value="Famfolder.gif" />
     *      </a:each>
     *  </a:tree>
     * ```
     * 
     * #### Example
     *
     * ```xml
     *  <a:tree 
     *     id             = "tree"
     *     height         = "200" 
     *     width          = "250" 
     *     model          = "filesystem.xml">
     *       <a:each match="[folder|drive]">
     *           <a:caption match="[@caption]" />
     *           <a:icon value="Famfolder.gif" />
     *       </a:each>
     *   </a:tree>
     *   <a:text 
     *     model  = "{tree.selected}" 
     *     value  = "[@caption]" 
     *     width  = "250" 
     *     height = "100" />
     * ```
     * 
     * #### Example
     *
     * This example shows a dropdown from which the user can select a country.
     * The list of countries is loaded from a model. Usually this would be loaded
     * from a separate url, but for clarity it's inlined. When the user selects
     * a country in the dropdown the value of the item is stored in the second
     * model (mdlForm) at the position specified by the ref attribute. In this
     * case this is the country element.
     * 
     * ```xml
     *  <a:label>Name</a:label>
     *  <a:textbox value="[name]" model="mdlForm" />
     * 
     *  <a:label>Country</a:label>
     *  <a:dropdown
     *    value   = "[mdlForm::country]"
     *    each    = "[mdlCountries::country]"
     *    caption = "[text()]">
     *  </a:dropdown>
     * 
     *  <a:model id="mdlCountries">
     *      <countries>
     *          <country value="USA">USA</country>
     *          <country value="GB">Great Britain</country>
     *          <country value="NL">The Netherlands</country>
     *      </countries>
     *  </a:model>
     * 
     *  <a:model id="mdlForm">
     *      <data>
     *          <name />
     *          <country />
     *      </data>
     *  </a:model>
     * ```
     *
     * #### Remarks
     *
     * This attribute is inherited from a parent when not set. You can use this
     * to tell sets of elements to use the same model.
     * 
     * ```xml
     *  <a:bar model="mdlForm">
     *      <a:label>Name</a:label>
     *      <a:textbox value="[name]" />
     * 
     *      <a:label>Happiness</a:label>
     *      <a:slider value="[happiness]" min="0" max="10" />
     *  </a:bar>
     * 
     *  <a:model id="mdlForm">
     *      <data />
     *  </a:model>
     * ```
     *
     * When no model is specified the default model is chosen. The default
     * model is the first model that is found without a name, or if all models
     * have a name, the first model found.
     * 
     * @see apf.DataBinding.model
     */
    this.$propHandlers["model"] = function(value){
        //Unset model
        if (!value && !this.$modelParsed) {
            if (this.$model) {
                this.clear();
                this.$model.unregister(this);
                this.$model = null;
                this.lastModelId = "";
            }
            else if (this.dataParent)
                this.dataParent.parent = null; //Should be autodisconnected by property binding

            return;
        }
        this.$initingModel = true;

        var fParsed;
        //Special case for property binding
        if ((fParsed = this.$modelParsed) && fParsed.type != 2) {
            var found, pb = fParsed.props;
            
            if (this.dataParent)
                this.dataParent = null; //Should be autodisconnected by property binding

            //Try to figure out who is the dataParent
            for (var prop in pb){
                

                this.dataParent = {
                    parent : self[prop.split(".")[0]],
                    xpath  : null,
                    model  : this.$modelParsed.instruction
                };
        
                found = true;
                break; // We currently only support one data parent
            }
            
            if (found) {
                //@todo this statement doesnt make sense
                /*//Maybe a compound model is found
                if (!this.dataParent && (pb = fParsed.xpaths && fParsed.xpaths[0])) {
                    this.dataParent = {
                        parent : self[pb.split(".")[0]],
                        xpath  : fParsed.xpaths[1],
                        model  : this.$modelParsed.instruction
                    };
                }*/
                
                if (this.dataParent && !this.dataParent.signalXmlUpdate)
                    this.dataParent.signalXmlUpdate = {};
            }
            
            this.$modelParsed = null;
        }

        //Analyze the data
        var model;
        if (typeof value == "object") {
            if (value.dataType == apf.ARRAY) { //Optimization used for templating
                
                model = apf.nameserver.get("model", value[0]);
                model.register(this, value[1]);
                return;
                
            }
            else if (value.$isModel) { // A model node is passed
                //Convert model object to value;
                model = value;
                value = this.model = model.name;
                if (!value)
                    model.setProperty("id", value = this.model = "model" + model.$uniqueId);
                
                //@todo why not set directly here?
            }
            else { //if (this.dataParent) { //Data came through data parent
                if (this.dataParent)
                    this.model = this.dataParent.model; //reset this property

                model = apf.xmldb.findModel(value);
                if (!model) //@todo very strange, this should never happen, but it does
                    return;
                var xpath = apf.xmlToXpath(value, null, true) || ".";
                
                
                
                model.register(this, xpath);
                return;
            }
            /*else {
                //@todo Error ??
            }*/
        }
        else if (value.indexOf("[::") > -1) { //@experimental
            var model, pNode = this;
            do {
                pNode = pNode.parentNode
                model = pNode.getAttribute("model");
            }
            while (pNode.parentNode && pNode.parentNode.nodeType == 1 && (!model || model == value));

            if (model && typeof model == "object")
                model = model.id;

            this.$inheritProperties.model = 3;
            if (model) {
                value = value.replace(/\[\:\:/g, "[" + model + "::");
            }
            else {
                apf.console.warn("No found model on any of the parents for this element while trying to overload model: " + value);
                return;
            }
        }

        //Optimize xmlroot position and set model async (unset the old one)
        //@todo apf3.0 this could be optimized by using apf.queue and only when not all info is there...
        clearTimeout(this.$dbTimer);
        if (!this.$amlLoaded && this.nodeType == 1) {
            var _self = this;
            this.$dbTimer = $setTimeout(function(){
                if (!_self.$amlDestroyed)
                    apf.setModel(value, _self);
            });
        }
        else
            apf.setModel(value, this);
    };

    
    /**
     * @attribute {String} viewport Sets or gets the way this element renders its data.
     * 
     * The possible values include:
     *   - `"virtual"`:  this element only renders data that it needs to display.
     *   - `"normal`"`:  this element renders all data at startup.
     * @experimental
     */
    this.$propHandlers["viewport"] = function(value){
        if (value != "virtual")
            return;

        this.implement(apf.VirtualViewport);
    };
    
};

    apf.DataBinding.prototype = new apf[apf.Presentation ? "Presentation" : "AmlElement"]();


apf.config.$inheritProperties["model"]           = 1;
apf.config.$inheritProperties["empty-message"]   = 1;
apf.config.$inheritProperties["loading-message"] = 1;
apf.config.$inheritProperties["offline-message"] = 1;
apf.config.$inheritProperties["noloading"]       = 1;

apf.Init.run("databinding");






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */


/**
 * All elements inheriting from this {@link term.baseclass baseclass} can bind to data
 * which contains multiple nodes.
 *
 *
 *
 * @class apf.MultiselectBinding
 * @inherits apf.DataBinding
 * @baseclass
 * @default_private
 * @allowchild  item, choices
 */

/*
 * @define  choices     Container for item nodes which receive presentation.
 * This element is part of the XForms specification. It is not necesary for
 * the Ajax.org Markup Language.
 * 
 * #### Example
 *
 * ```xml
 *  <a:list>
 *      <a:choices>
 *          <a:item>red</a:item>
 *          <a:item>blue</a:item>
 *          <a:item>green</a:item>
 *      </a:choices>
 *  </a:list>
 * ```
 * @allowchild  item
 */
apf.MultiselectBinding = function(){
    if (!this.setQueryValue)
        this.implement(apf.DataBinding);

    this.$regbase    = this.$regbase|apf.__MULTISELECT__; //We're pretending to have multiselect even though we might not.

    this.$init(function(){
        this.$selectTimer = {};
    });
};

(function(){
    this.length = 0;

    //1 = force no bind rule, 2 = force bind rule
    this.$attrExcludePropBind = apf.extend({
        caption   : 2,
        icon      : 2,
        eachvalue : 2,
        select    : 2,
        css       : 2,
        sort      : 2,
        drag      : 2,
        drop      : 2,
        dragcopy  : 2,
        selected  : 3,
        //caret     : 2,
        each      : 1,
        "selection"             : 3, //only databound when has an xpath
        "selection-unique"      : 3, //only databound when has an xpath
        "selection-constructor" : 3 //only databound when has an xpath
    }, this.$attrExcludePropBind);

     
    /**
     * Change the sorting order of this element.
     *
     * @param {Object}  options  The new sort options. These are applied incrementally.
     *                           Any property that is not set is maintained unless the clear
     *                           parameter is set to `true`. The following properties are available:
     *                  - order ([[String]])
     *                  - [xpath] ([[String]])
     *                  - [type] ([[String]])
     *                  - [method] ([[String]])
     *                  - [getNodes] ([[Function]]): A function that retrieves a list of nodes.
     *                  - [dateFormat] ([[String]])
     *                  - [getValue] ([[Function]]): A function that determines the string content based
     *                                            on an XML node as it's first argument.
     * @param {Boolean} clear    Removes the current sort options.
     * @param {Boolean} noReload Specifies whether to reload the data of this component.
     */
    this.resort = function(options, clear, noReload){
        if (!this.$sort)
            this.$sort = new apf.Sort();

        this.$sort.set(options, clear);

        if (this.clearAllCache)
            this.clearAllCache();

        if (noReload)
            return;

        
        /*if(this.hasFeature(apf.__VIRTUALVIEWPORT__)){
            this.$clearVirtualDataset(this.xmlRoot);
            this.reload();

            return;
        }*/
        

        var _self = this;
        (function sortNodes(xmlNode, htmlParent) {
            if(!xmlNode)
                return;
            var sNodes = _self.$sort.apply(
                apf.getArrayFromNodelist(xmlNode.selectNodes(_self.each)));

            for (var i = 0; i < sNodes.length; i++) {
                if (_self.$isTreeArch || _self.$withContainer){
                    var htmlNode = apf.xmldb.findHtmlNode(sNodes[i], _self);

                    

                    var container = _self.$findContainer(htmlNode);

                    htmlParent.appendChild(htmlNode);
                    if (!apf.isChildOf(htmlNode, container, true))
                        htmlParent.appendChild(container);

                    sortNodes(sNodes[i], container);
                }
                else
                    htmlParent.appendChild(apf.xmldb.findHtmlNode(sNodes[i], _self));
            }
        })(this.xmlRoot, this.$container);

        return options;
    };

    /**
     * Change sorting from ascending to descending, and vice versa!
     */
    this.toggleSortOrder = function(){
        return this.resort({"ascending" : !this.$sort.get().ascending}).ascending;
    };

    /**
     * Retrieves the current sort options.
     *
     * @returns {Object}  The current sort options. The following properties are available:
     *                     - order ([[String]])
     *                     - xpath ([[String]])
     *                     - type ([[String]])
     *                     - method ([[String]])
     *                     - getNodes ([[Function]]): A function that retrieves a list of nodes.
     *                     - dateFormat ([[String]])
     *                     - getValue ([[Function]]): A function that determines the string content based on
     *                                               an XML node as it's first argument.
     * 
     */
    this.getSortSettings = function(){
        return this.$sort.get();
    };
    

    /*
     * Optimizes load time when the xml format is very simple.
     */
    // @todo Doc
    this.$propHandlers["simpledata"] = function(value){
        if (value) {
            this.getTraverseNodes = function(xmlNode){
                
                if (this.$sort && !this.$isLoading) {
                    var nodes = apf.getArrayFromNodelist((xmlNode || this.xmlRoot).childNodes);
                    return this.$sort.apply(nodes);
                }
                

                return (xmlNode || this.xmlRoot).childNodes;
            };

            this.getFirstTraverseNode = function(xmlNode){
                return this.getTraverseNodes(xmlNode)[0];//(xmlNode || this.xmlRoot).childNodes[0];
            };

            this.getLastTraverseNode = function(xmlNode){
                var nodes = this.getTraverseNodes(xmlNode);//(xmlNode || this.xmlRoot).childNodes;
                return nodes[nodes.length - 1];
            };

            this.getTraverseParent = function(xmlNode){
                if (!xmlNode.parentNode || xmlNode == this.xmlRoot)
                    return false;

                return xmlNode.parentNode;
            };
        }
        else {
            delete this.getTraverseNodes;
            delete this.getFirstTraverseNode;
            delete this.getLastTraverseNode;
            delete this.getTraverseParent;
        }
    };

    /**
     * Retrieves a node list containing the {@link term.datanode data nodes} which
     * are rendered by this element.
     *
     * @param {XMLElement} [xmlNode] The parent element on which each query is applied.
     * @return {NodeList} The node list containing the data nodes
     */
    this.getTraverseNodes = function(xmlNode){
        

        
        if (this.$sort) {
            var nodes = apf.getArrayFromNodelist((xmlNode || this.xmlRoot).selectNodes(this.each));
            return this.$sort.apply(nodes);
        }
        

        return (xmlNode || this.xmlRoot).selectNodes(this.each);
    };

    /**
     * Retrieves the first {@link term.datanode data node} which gets representation
     * in this element. 
     *
     * @param {XMLElement} [xmlNode] The parent element on which the each query is executed.
     * @return {apf.AmlNode} The first represented {@link term.datanode data node}
     */
    this.getFirstTraverseNode = function(xmlNode){
        
        if (this.$sort) {
            var nodes = (xmlNode || this.xmlRoot).selectNodes(this.each);
            return this.$sort.apply(nodes)[0];
        }
        

        return (xmlNode || this.xmlRoot).selectSingleNode(this.each);
    };

    /**
     * Retrieves the last {@link term.datanode data node} which gets representation
     * in this element. 
     *
     * @param {XMLElement} [xmlNode] the parent element on which the each query is executed.
     * @return {XMLElement} The last represented {@link term.datanode data node}
     * 
     */
    this.getLastTraverseNode = function(xmlNode){
        var nodes = this.getTraverseNodes(xmlNode || this.xmlRoot);
        return nodes[nodes.length-1];
    };

    /**
     * Determines whether a {@link term.datanode data node} is an each node. 
     *
     * @param {XMLElement} [xmlNode] The parent element on which the each query is executed.
     * @return  {Boolean}  Identifies whether the XML element is a each node.
     * 
     */
    this.isTraverseNode = function(xmlNode){
        /*
            Added optimization, only when an object has a tree architecture is it
            important to go up to the each parent of the xmlNode, else the node
            should always be based on the xmlroot of this component
        */
        //this.$isTreeArch
        var nodes = this.getTraverseNodes(
          this.getTraverseParent(xmlNode) || this.xmlRoot);
        for (var i = 0; i < nodes.length; i++)
            if (nodes[i] == xmlNode)
                return true;
        return false;
    };

    /**
     * Retrieves the next `each` node to be selected from a given `each` node. 
     *
     * The method can do this in either direction and also return the Nth node for this algorithm.
     *
     * @param {XMLElement}  xmlNode  The starting point for determining the next selection.
     * @param {Boolean}     [up=false]     The direction of the selection.
     * @param {Number}     [count=1]  The distance in number of nodes.
     * @return  {XMLElement} The {@link term.datanode data node} to be selected next.
     */
    this.getNextTraverseSelected = function(xmlNode, up, count){
        if (!xmlNode)
            xmlNode = this.selected;
        if (!count)
            count = 1;

        var i = 0;
        var nodes = this.getTraverseNodes(this.getTraverseParent(xmlNode) || this.xmlRoot);
        while (nodes[i] && nodes[i] != xmlNode)
            i++;

        var node = (up == null)
            ? nodes[i + count] || nodes[i - count]
            : (up ? nodes[i + count] : nodes[i - count]);

        //arguments[2]
        return node || count && (i < count || (i + 1) > Math.floor(nodes.length / count) * count)
            ? node
            : (up ? nodes[nodes.length-1] : nodes[0]);
    };

    /**
     * Retrieves the next `each` node. 
     * 
     * The method can do this in either direction and also return the Nth next node.
     *
     * @param {XMLElement}  xmlNode  The starting point for determining the next selection.
     * @param {Boolean}     [up=false]     The direction of the selection.
     * @param {Number}     [count=1]  The distance in number of nodes.
     * @return  {XMLElement} The {@link term.datanode data node} to be selected next.
     */
    this.getNextTraverse = function(xmlNode, up, count){
        if (!count)
            count = 1;
        if (!xmlNode)
            xmlNode = this.selected;

        var i = 0;
        var nodes = this.getTraverseNodes(this.getTraverseParent(xmlNode) || this.xmlRoot);
        while (nodes[i] && nodes[i] != xmlNode)
            i++;

        var ind = i + (up ? -1 * count : count);
        return nodes[ind < 0 ? 0 : ind];
    };

    /**
     * Retrieves the parent each node. 
     *
     * In some cases the each rules has a complex form like 'children/item'. In
     * those cases, the generated tree has a different structure from that of the XML
     * data. For these situations, the `xmlNode.parentNode` property won't return
     * the each parent; instead, this method will give you the right parent.
     *
     * @param {XMLElement} xmlNode The node for which the parent element will be determined.
     * @return  {XMLElement} The parent node or `null` if none was found.
     */
    this.getTraverseParent = function(xmlNode){
        if (!xmlNode.parentNode || xmlNode == this.xmlRoot)
            return false;

        //@todo this can be removed when we have a new xpath implementation
        if (xmlNode.$regbase)
            return xmlNode.parentNode;

        var x, id = xmlNode.getAttribute(apf.xmldb.xmlIdTag);
        if (!id) {
            //return false;
            xmlNode.setAttribute(apf.xmldb.xmlIdTag, "temp");
            id = "temp";
        }

        /*
        do {
            xmlNode = xmlNode.parentNode;
            if (xmlNode == this.xmlRoot)
                return false;
            if (this.isTraverseNode(xmlNode))
                return xmlNode;
        } while (xmlNode.parentNode);
        */

        //This is not 100% correct, but good enough for now

        x = xmlNode.selectSingleNode("ancestor::node()[(("
            + this.each + ")/@" + apf.xmldb.xmlIdTag + ")='"
            + id + "']");

        if (id == "temp")
            xmlNode.removeAttribute(apf.xmldb.xmlIdTag);
        return x;
    };

    if (!this.$findHtmlNode) { //overwritten by apf.Cache
        /**
         * Finds HTML presentation node in cache by ID.
         *
         * @param  {String} id  The id of the HTMLElement which is looked up.
         * @return {HTMLElement} The HTMLElement found. When no element is found, `null` is returned.
         * @private
         */
        this.$findHtmlNode = function(id){
            return this.$pHtmlDoc.getElementById(id);
        };
    }

    this.$setClearMessage = function(msg, className, lastHeight){
        if (this.more && this.$addMoreItem) this.$addMoreItem();
        if (!this.$empty) {
            if (!this.$hasLayoutNode("empty"))
                return;

            this.$getNewContext("empty");

            var xmlEmpty = this.$getLayoutNode("empty");
            if (!xmlEmpty) return;

            this.$empty = apf.insertHtmlNode(xmlEmpty, this.$container);
        }
        else {
            this.$container.appendChild(this.$empty);
        }

        var empty = this.$getLayoutNode("empty", "caption", this.$empty);

        if (empty)
            apf.setNodeValue(empty, msg || "");

        this.$empty.setAttribute("id", "empty" + this.$uniqueId);
        apf.setStyleClass(this.$empty, className, ["loading", "empty", "offline"]);

        //@todo apf3.0 cleanup?
        var extH = apf.getStyle(this.$ext, "height");
        this.$empty.style.height = (lastHeight && (!extH || extH == "auto") && className != "empty")
            ? (Math.max(10, (lastHeight
               - apf.getHeightDiff(this.$empty)
               - apf.getHeightDiff(this.$ext))) + "px")
            : "";
    };

    this.$updateClearMessage = function(msg, className) {
        if (!this.$empty || this.$empty.parentNode != this.$container
          || this.$empty.className.indexOf(className) == -1)
            return;

        var empty = this.$getLayoutNode("empty", "caption", this.$empty);
        if (empty)
            apf.setNodeValue(empty, msg || "");
    }

    this.$removeClearMessage = function(){
        if (!this.$empty)
            this.$empty = document.getElementById("empty" + this.$uniqueId);
        if (this.$empty && this.$empty.parentNode)
            this.$empty.parentNode.removeChild(this.$empty);
    };

    /*
     * Set listeners, calls HTML creation methods and
     * initializes select and focus states of object.
     */
    this.$load = function(XMLRoot){
        //Add listener to XMLRoot Node
        apf.xmldb.addNodeListener(XMLRoot, this);

        this.$isLoading = true;

        var length = this.getTraverseNodes(XMLRoot).length;
        if (!this.renderRoot && !length)
            return this.clear(null, null, true); //@todo apf3.0 this should clear and set a listener


        //Traverse through XMLTree
        var nodes = this.$addNodes(XMLRoot, null, null, this.renderRoot, null, 0, "load");

        //Build HTML
        this.$fill(nodes);

        this.$isLoading = false;

        //Select First Child
        if (this.selectable) {
            
            //@todo apf3.0 optimize to not set selection when .selection or .selected is set on initial load
            if (this["default"])
                this.select(this["default"]);
            else if (this.autoselect) {
                if (!this.selected) {
                    if (this.renderRoot)
                        this.select(XMLRoot, null, null, null, true);
                    else if (nodes.length)
                        this.$selectDefault(XMLRoot);
                    //else @todo apf3.0 this one doesnt seem needed
                        //this.clearSelection();
                }
            }
            else {
                this.clearSelection(true);
                var xmlNode = this.renderRoot
                    ? this.xmlRoot
                    : this.getFirstTraverseNode(); //should this be moved to the clearSelection function?
                if (xmlNode)
                    this.setCaret(xmlNode);
                
                if (this.selected)
                    this.setProperty("selected", null);
                if (this.choosen)
                    this.setProperty("choosen", null);
                
            }
        }

        if (this.focussable)
            apf.document.activeElement == this ? this.$focus() : this.$blur();

        
        if (length != this.length)
            this.setProperty("length", length);
        
    };

    var actionFeature = {
        "insert"      : 127,//11111110
        "replacenode" : 127,//11111110
        "attribute"   : 255,//11111111
        "add"         : 251,//11110111
        "remove"      : 110, //01011110
        "redo-remove" : 79, //10011110
        "synchronize" : 127,//11111110
        "move-away"   : 297,//11010111
        "move"        : 141  //10011111
    };

    /**
     * @event xmlupdate Fires when XML of this element is updated.
     * @param {Object} e The standard event object. The following properties are available:
     *                      - action ([[String]]): The action that was executed on the XML. The following values are possible:
     *                            - `text`   :     A text node is set
     *                            - `attribute` :  An attribute is set
     *                            - `update`:      An XML node is updated
     *                            - `insert`  :    xml nodes are inserted
     *                            - `add`   :      An XML node is added
     *                            - `remove` :     An XML node is removed (parent still set)
     *                            - `redo`-remove`: An XML node is removed (parent not set)
     *                            - `synchronize`:  An unknown update
     *                            - `move-away` :  An XML node is moved (parent not set)
     *                            - `move`        An XML node is moved (parent still set)
     *                      - xmlNode ([[XMLElement]]): The node that is subject to the update
     *                      - result (`Mixed`): The result
     *                      - UndoObj ([[apf.UndoData]]): The undo information
     */
    /*
     * Loops through parents of a changed node to find the first
     * connected node. Based on the action, it will change, remove,
     * or update the representation of the data.
     */
    this.$xmlUpdate = function(action, xmlNode, listenNode, UndoObj, lastParent){
        if (!this.xmlRoot)
            return; //@todo think about purging cache when xmlroot is removed

        var result, length, pNode, htmlNode,
            startNode = xmlNode;
        if (!listenNode)
            listenNode = this.xmlRoot;

        if (action == "redo-remove") {
            var loc = [xmlNode.parentNode, xmlNode.nextSibling];
            lastParent.appendChild(xmlNode); //ahum, i'm not proud of this one
            var eachNode = this.isTraverseNode(xmlNode);
            if (loc[0])
                loc[0].insertBefore(xmlNode, loc[1]);
            else
                lastParent.removeChild(xmlNode);

            if (!eachNode)
                xmlNode = lastParent;
        }

        //Get First ParentNode connected
        do {
            if (action == "add" && this.isTraverseNode(xmlNode)
              && startNode == xmlNode)
                break; //@todo Might want to comment this out for adding nodes under a eachd node

            if (xmlNode.getAttribute(apf.xmldb.xmlIdTag)) {
                htmlNode = this.$findHtmlNode(
                    xmlNode.getAttribute(apf.xmldb.xmlIdTag)
                    + "|" + this.$uniqueId);

                if (xmlNode == listenNode && !this.renderRoot) {
                    if (xmlNode == this.xmlRoot && action != "insert" && action != "replacenode") {
                        //@todo apf3.0 - fix this for binding on properties
                        this.dispatchEvent("xmlupdate", {
                            action : action,
                            xmlNode: xmlNode,
                            UndoObj: UndoObj
                        });
                        return;
                    }
                    break;
                }

                if (htmlNode && actionFeature[action] & 2
                  && !this.isTraverseNode(xmlNode))
                    action = "remove"; //@todo why not break here?

                else if (!htmlNode && actionFeature[action] & 4
                  && this.isTraverseNode(xmlNode)){
                    action = "add";
                    break;
                }

                else if (htmlNode
                  && (startNode != xmlNode || xmlNode == this.xmlRoot)) {
                    if (actionFeature[action] & 1)
                        action = "update";
                    else if (action == "remove")
                        return;
                }

                if (htmlNode  || action == "move")
                    break;
            }
            else if (actionFeature[action] & 8 && this.isTraverseNode(xmlNode)){
                action = "add";
                break;
            }

            if (xmlNode == listenNode) {
                if (actionFeature[action] & 128) //The change is not for us.
                    return;

                break;
            }
            xmlNode = xmlNode.parentNode;
        }
        while (xmlNode && xmlNode.nodeType != 9);

        

        
        
        // @todo Think about not having this code here
        if (this.hasFeature(apf.__VIRTUALVIEWPORT__)) {
            if(!this.$isInViewport(xmlNode)) //xmlNode is a eachd node
                return;
        }
        

        //if(xmlNode == listenNode && !action.match(/add|synchronize|insert/))
        //    return; //deleting nodes in parentData of object

        var foundNode = xmlNode;
        if (xmlNode && xmlNode.nodeType == 9)
            xmlNode = startNode;

        if (action == "replacenode") {
            //var tmpNode;
            //Case for replacing the xmlroot or its direct parent
            if (UndoObj ? UndoObj.args[1] == this.xmlRoot : !this.xmlRoot.parentNode)
                return this.load(UndoObj ? UndoObj.xmlNode : listenNode, {force: true});

            //Case for replacing a node between the xmlroot and the traverse nodes
            var nodes = this.getTraverseNodes();
            for (var i = 0, l = nodes.length; i < l; i++) {
                if (apf.isChildOf(startNode, nodes[i]))
                    return this.load(this.xmlRoot, {force: true}); //This can be more optimized by using addNodes
            }
            //if ((tmpNode = this.getFirstTraverseNode()) && apf.isChildOf(startNode, tmpNode))
        }

        //Action Tracker Support - && xmlNode correct here??? - UndoObj.xmlNode works but fishy....
        if (UndoObj && xmlNode && !UndoObj.xmlNode)
            UndoObj.xmlNode = xmlNode;

        //Check Move -- if value node isn't the node that was moved then only perform a normal update
        if (action == "move" && foundNode == startNode) {
            //if(!htmlNode) alert(xmlNode.getAttribute("id")+"|"+this.$uniqueId);
            var isInThis  = apf.isChildOf(this.xmlRoot, xmlNode.parentNode, true); //@todo this.getTraverseParent(xmlNode)
            var wasInThis = apf.isChildOf(this.xmlRoot, UndoObj.extra.parent, true);

            //Move if both previous and current position is within this object
            if (isInThis && wasInThis)
                this.$moveNode(xmlNode, htmlNode, UndoObj.extra.oldParent);
            else if (isInThis) //Add if only current position is within this object
                action = "add";
            else if (wasInThis) //Remove if only previous position is within this object
                action = "remove";
        }
        else if (action == "move-away") {
            var goesToThis = apf.isChildOf(this.xmlRoot, UndoObj.extra.parent, true);
            if (!goesToThis)
                action = "remove";
        }

        //Remove loading message
        if (this.$removeClearMessage && this.$setClearMessage) {
            if (this.getFirstTraverseNode())
                this.$removeClearMessage();
            else
                this.$setClearMessage(this["empty-message"], "empty")
        }

        //Check Insert
        if (action == "insert" && (this.$isTreeArch || xmlNode == this.xmlRoot)) {
            if (!xmlNode)
                return;

            if (this.$hasLoadStatus(xmlNode) && this.$removeLoading)
                this.$removeLoading(xmlNode);

            if (this.$container.firstChild && !apf.xmldb.getNode(this.$container.firstChild)) {
                //Appearantly the content was cleared
                this.$container.innerHTML = "";

                if (!this.renderRoot) {
                    length = this.getTraverseNodes().length;
                    if (!length)
                        this.clear();
                }
            }

            result = this.$addNodes(xmlNode, null, true, false, null, null, "insert");//this.$isTreeArch??

            this.$fillParentHtml = (this.$getParentNode
                ? this.$getParentNode(htmlNode)
                : htmlNode);
            this.$fillParent = xmlNode;
            this.$fill(result);

            

            if (this.selectable && (length === 0 || !this.xmlRoot.selectSingleNode(this.each)))
                return;
        }
        else if (action == "add") {// || !htmlNode (Check Add)
            var parentHTMLNode;
            pNode = this.getTraverseParent(xmlNode) || this.xmlRoot;

            if (pNode == this.xmlRoot)
                parentHTMLNode = this.$container;

            if (!parentHTMLNode && this.$isTreeArch) {
                parentHTMLNode = this.$findHtmlNode(
                    pNode.getAttribute(apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);
            }

            //This should be moved into a function (used in setCache as well)
            
            if (!parentHTMLNode && this.getCacheItem)
                parentHTMLNode = this.getCacheItem(pNode.getAttribute(apf.xmldb.xmlIdTag)
                    || (pNode.getAttribute(apf.xmldb.xmlDocTag)
                         ? "doc" + pNode.getAttribute(apf.xmldb.xmlDocTag)
                         : false));
            

            //Only update if node is in current representation or in cache
            if (parentHTMLNode || this.$isTreeArch
              && pNode == this.xmlRoot) { //apf.isChildOf(this.xmlRoot, xmlNode)
                parentHTMLNode = (this.$findContainer && parentHTMLNode && parentHTMLNode.nodeType == 1
                    ? this.$findContainer(parentHTMLNode)
                    : parentHTMLNode) || this.$container;

                result = this.$addNodes(xmlNode, parentHTMLNode, true, true,
                    apf.xmldb.getHtmlNode(this.getNextTraverse(xmlNode), this));

                if (parentHTMLNode)
                    this.$fill(result);
            }
        }
        else if (action == "remove") { //Check Remove
            //&& (!xmlNode || foundNode == xmlNode && xmlNode.parentNode
            //if (!xmlNode || startNode != xmlNode) //@todo unsure if I can remove above commented out statement
                //return;
            //I've commented above code out, because it disabled removing a
            //subnode of a node that through an each rule makes the traverse
            //node no longer a traverse node.

            //Remove HTML Node
            if (htmlNode)
                this.$deInitNode(xmlNode, htmlNode);
            else if (startNode == this.xmlRoot) {
                return this.load(null, {
                    noClearMsg: !this.dataParent || !this.dataParent.autoselect
                });
            }
        }
        else if (htmlNode) {
            
            if (this.$sort)
                this.$moveNode(xmlNode, htmlNode);
            

            this.$updateNode(xmlNode, htmlNode);

            //Transaction 'niceties'
            if (action == "replacenode" && this.hasFeature(apf.__MULTISELECT__)
              && this.selected && xmlNode.getAttribute(apf.xmldb.xmlIdTag)
              == this.selected.getAttribute(apf.xmldb.xmlIdTag)) {
                this.selected = xmlNode;
            }

            //if(action == "synchronize" && this.autoselect) this.reselect();
        }
        else if (action == "redo-remove") { //Check Remove of the data (some ancestor) that this component is bound on
            var testNode = this.xmlRoot;
            while (testNode && testNode.nodeType != 9)
                testNode = testNode.parentNode;

            if (!testNode) {
                //Set Component in listening state until data becomes available again.
                var model = this.getModel(true);

                

                return model.$waitForXml(this);
            }
        }

        

        //For tree based nodes, update all the nodes up
        pNode = xmlNode ? xmlNode.parentNode : lastParent;
        if (this.$isTreeArch && !this.$preventRecursiveUpdate
          && pNode && pNode.nodeType == 1) {
            do {
                htmlNode = this.$findHtmlNode(pNode.getAttribute(
                    apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);

                if (htmlNode)
                    this.$updateNode(pNode, htmlNode);
            }
            while ((pNode = this.getTraverseParent(pNode)) && pNode.nodeType == 1);
        }

        //Make sure the selection doesn't become corrupted
        if (actionFeature[action] & 32 && this.selectable
          && startNode == xmlNode
          && (action != "insert" || xmlNode == this.xmlRoot)) {

            clearTimeout(this.$selectTimer.timer);
            // Determine next selection
            if (action == "remove" && apf.isChildOf(xmlNode, this.selected, true)
              || xmlNode == this.$selectTimer.nextNode) {
                this.$selectTimer.nextNode = this.getDefaultNext(xmlNode, this.$isTreeArch);
                if (this.$selectTimer.nextNode == this.xmlRoot && !this.renderRoot)
                    this.$selectTimer.nextNode = null;
            }

            //@todo Fix this by putting it after xmlUpdate when its using a timer
            var _self = this;
            this.$selectTimer.timer = $setTimeout(function(){
                _self.$checkSelection(_self.$selectTimer.nextNode);
                _self.$selectTimer.nextNode = null;
            });
        }

        
        //Set dynamic properties that relate to the changed content
        if (actionFeature[action] & 64) {
            if (!length)
                length = this.xmlRoot.selectNodes(this.each).length;
            if (action == "remove")
                length--;
            if (length != this.length)
                this.setProperty("length", length);
        }
        

        //Let's signal components that are waiting for xml to appear (@todo what about clearing the signalXmlUpdate)
        if (this.signalXmlUpdate && actionFeature[action] & 16) {
            var uniqueId;
            for (uniqueId in this.signalXmlUpdate) {
                if (parseInt(uniqueId, 10) != uniqueId) continue; //safari_old stuff

                var o = apf.lookup(uniqueId);
                if (!this.selected) continue;

                xmlNode = this.selected.selectSingleNode(o.dataParent.xpath);
                if (!xmlNode) continue;

                o.load(xmlNode);
            }
        }

        this.dispatchEvent("xmlupdate", {
            action : action,
            xmlNode: startNode,
            traverseNode : xmlNode,
            result : result,
            UndoObj: UndoObj
        });
    };

    /*
     * Loop through NodeList of selected Traverse Nodes
     * and check if it has representation. If it doesn't
     * representation is created via $add().
     */
    this.$addNodes = function(xmlNode, parent, checkChildren, isChild, insertBefore, depth, action){
        

        var htmlNode, lastNode, loopNode;
        isChild          = (isChild && (this.renderRoot && xmlNode == this.xmlRoot
            || this.isTraverseNode(xmlNode)));
        var nodes        = isChild ? [xmlNode] : this.getTraverseNodes(xmlNode);
        /*var loadChildren = nodes.length && this.$bindings["insert"]
            ? this.$applyBindRule("insert", xmlNode)
            : false; << UNUSED */

        
        var cId, cItem;
        if (this.$isTreeArch && this.caching
          && (!this.$bindings || !this.$bindings.each || !this.$bindings.each.filter)
          && (cItem = this.cache[(cId = xmlNode.getAttribute(apf.xmldb.xmlIdTag))])) {
            if (this.$subTreeCacheContext || this.$needsDepth) {
                //@todo
                //We destroy the current items, because currently we
                //don't support multiple treecachecontexts
                //and because datagrid needs to redraw depth
                this.clearCacheItem(cId);
            }
            else {
                this.$subTreeCacheContext = {
                    oHtml      : cItem,
                    container  : parent,
                    parentNode : null,
                    beforeNode : null
                };

                var htmlNode;
                while (cItem.childNodes.length)
                    (parent || this.$container).appendChild(htmlNode = cItem.childNodes[0]);

                return nodes;
            }
        }
        

        if (this.$isTreeArch && depth === null && action == "insert") {
            depth = 0, loopNode = xmlNode;
            while(loopNode && loopNode != this.xmlRoot) {
                depth++;
                loopNode = this.getTraverseParent(loopNode);
            }
        }

        for (var i = 0; i < nodes.length; i++) {
            if (nodes[i].nodeType != 1) {
                
                continue;
            }

            if (checkChildren) {
                htmlNode = this.$findHtmlNode(nodes[i]
                    .getAttribute(apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);
            }

            if (!htmlNode) {
                //Retrieve DataBind ID
                var Lid = apf.xmldb.nodeConnect(this.documentId, nodes[i], null, this);

                //Add Children
                var beforeNode = isChild
                        ? insertBefore
                        : (lastNode ? lastNode.nextSibling : null),//(parent || this.$container).firstChild);
                    parentNode = this.$add(nodes[i], Lid, isChild ? xmlNode.parentNode : xmlNode,
                        beforeNode ? parent || this.$container : parent, beforeNode,
                        (!beforeNode && i == nodes.length - 1), depth, nodes[i + 1], action);//Should use getTraverParent

                //Exit if component tells us its done with rendering
                if (parentNode === false) {
                    //Tag all needed xmlNodes for future reference
                    // @todo apf3.0 code below looks harmful... hence commented out (Mike)
                    /*for (var j = i; j < nodes.length; j++)
                        apf.xmldb.nodeConnect(this.documentId, nodes[j],
                            null, this);*/
                    break;
                }

                //Parse Children Recursively -> optimize: don't check children that can't exist
                //if(this.$isTreeArch) this.$addNodes(nodes[i], parentNode, checkChildren);
            }

            if (checkChildren)
                lastNode = htmlNode;// ? htmlNode.parentNode.parentNode : null;
        }

        return nodes;
    };

    this.$handleBindingRule = function(value, prop){
        if (!value)
            this[prop] = null;

        //@todo apf3.0 fix parsing
        if (prop == "each") {
            value = value.charAt(0) == "[" && value.charAt(value.length - 1) == "]"
                ? value.replace(/^\[|\]$/g, "")
                : value;

            if (value.match(/^\w+::/)) {
                var model = value.split("::"); //@todo this is all very bad
                if (!apf.xPathAxis[model[0]]) {
                    this.setProperty("model", model[0]);
                    this.each = model[1];
                }
                else
                    this.each = value;
            }
            else
                this.each = value;

            if (this.each == this.$lastEach)
                return;

            this.$lastEach = value;

            if (!this.$model && !this.$initingModel) {
                this.$initingModel = true;
                this.$setInheritedAttribute("model");

                return; //@experimental
            }

            if (this.$checkLoadQueue() !== false) //@experimental
                return;
        }

        //@todo apf3.0 find a better heuristic (portal demo)
        if (this.xmlRoot && !this.$bindRuleTimer && this.$amlLoaded) {
            var _self = this;
            apf.queue.add("reload" + this.$uniqueId, function(){
                
                _self.reload();
            });
        }
    };

    this.$select = function(o){
        
        if (this.renaming)
            this.stopRename(null, true);
        

        if (!o || !o.style)
            return;
        return this.$setStyleClass(o, "selected");
    };

    this.$deselect = function(o){
        
        if (this.renaming) {
            this.stopRename(null, true);

            if (this.ctrlselect)
                return false;
        }
        

        if (!o)
            return;
        return this.$setStyleClass(o, "", ["selected", "indicate"]);
    };

    this.$indicate = function(o){
        
        if (this.renaming)
            this.stopRename(null, true);
        

        if (!o)
            return;
        return this.$setStyleClass(o, "indicate");
    };

    this.$deindicate = function(o){
        
        if (this.renaming)
            this.stopRename(null, true);
        

        if (!o)
            return;
        return this.$setStyleClass(o, "", ["indicate"]);
    };

    
    /**
     * @attribute {String} each Sets or gets the XPath statement that determines which
     * {@link term.datanode data nodes} are rendered by this element (also known
     * as {@link term.eachnode each nodes}. 
     * 
     *
     * #### Example
     *
     * ```xml
     *  <a:label>Country</a:label>
     *  <a:dropdown
     *      model     = "mdlCountries"
     *      each      = "[country]"
     *      eachvalue = "[@value]"
     *      caption   = "[text()]">
     *  </a:dropdown>
     *
     *  <a:model id="mdlCountries">
     *      <countries>
     *          <country value="USA">USA</country>
     *          <country value="GB">Great Brittain</country>
     *          <country value="NL">The Netherlands</country>
     *          ...
     *      </countries>
     *  </a:model>
     * ```
     *
     * 
     */
    this.$propHandlers["each"] =

    /**
     * @attribute {String} caption Sets or gets the text displayed on the item.
     *
     * #### Example
     *
     * ```xml
     *  <a:list caption="[text()]" each="[item]" />
     * ```
     */
    this.$propHandlers["caption"]  =

    /**
     * @attribute {String} eachvalue Sets or gets the {@link term.expression}
     * that determines the value for each data nodes in the dataset of the element.
     *
     * #### Example
     *
     * ```xml
     *  <a:list value="[@value]" each="[item]" />
     * ```
     * 
     */
    this.$propHandlers["eachvalue"]  =

    /**
     * @attribute {String} icon Sets or gets the XPath statement that determines from
     * which XML node the icon URL is retrieved.
     *
     * #### Example
     *
     * ```xml
     *  <a:list icon="[@icon]" each="[item]" />
     * ```
     */
    this.$propHandlers["icon"]     =

    /**
     * @attribute {String} tooltip Sets or gets the XPath statement that determines from
     * which XML node the tooltip text is retrieved.
     *
     * #### Example
     *
     * ```xml
     *  <a:list tooltip="[text()]" each="[item]" />
     * ```
     */
    this.$propHandlers["tooltip"]  = this.$handleBindingRule;

    
    /**
     * @attribute {String} sort Sets or gets the XPath statement that selects the sortable value.
     *
     * #### Example
     *
     * ```xml
     *  <a:list sort="[@name]" each="[person]" />
     * ```
     * 
     */
    this.$propHandlers["sort"] = function(value){
        if (value) {
            this.$sort = new apf.Sort()
            this.$sort.set({
                getValue : apf.lm.compile(value)
            });
        }
        else {
            this.$sort = null;
        }
    }
    

    /**
     * @attribute {String} match Sets or gets the XPath statement that determines whether
     * this node is selectable.
     *
     * #### Example
     *
     * ```xml
     *  <a:list match="{[@disabled] != 1}" each="[item]" />
     * ```
     * 
     */
    //this.$propHandlers["select"]   =
    
}).call(apf.MultiselectBinding.prototype = new apf.DataBinding());




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * The baseclass for all standard data binding rules.
 *
 * @class apf.StandardBinding
 * @private
 * @baseclass
 * @inherits apf.DataBinding
 */
apf.StandardBinding = function(){
    this.$init(true);
    
    
    if (apf.Validation)
        this.implement(apf.Validation);
    
    
    if (!this.setQueryValue)
        this.implement(apf.DataBinding);

    if (!this.defaultValue) //@todo please use this in a sentence
        this.defaultValue = "";

    /**
     * Load XML into this element
     * @private
     */
    this.$load = function(xmlNode){
        //Add listener to XMLRoot Node
        apf.xmldb.addNodeListener(xmlNode, this);
        //Set Properties

        
        var b, lrule, rule, bRules, bRule, value;
        if (b = this.$bindings) {
	        for (rule in b) {
	            lrule = rule.toLowerCase();
	            if (this.$supportedProperties.indexOf(lrule) > -1) {
	                bRule = (bRules = b[lrule]).length == 1 
                      ? bRules[0] 
                      : this.$getBindRule(lrule, xmlNode);

                    value = bRule.value || bRule.match;

	                
                    //Remove any bounds if relevant
                    this.$clearDynamicProperty(lrule);
            
                    if (value.indexOf("{") > -1 || value.indexOf("[") > -1)
                        this.$setDynamicProperty(lrule, value);
                    else 
                    
                    if (this.setProperty)
                        this.setProperty(lrule, value, true);
	            }
	        }
	    }
        

        //Think should be set in the event by the Validation Class
        if (this.errBox && this.isValid && this.isValid())
            this.clearError();
    };

    /**
     * Set xml based properties of this element
     * @private
     */
    this.$xmlUpdate = function(action, xmlNode, listenNode, UndoObj){
        //Clear this component if some ancestor has been detached
        if (action == "redo-remove") {
            var retreatToListenMode = false, model = this.getModel(true);
            if (model) {
                var xpath = model.getXpathByAmlNode(this);
                if (xpath) {
                    xmlNode = model.data.selectSingleNode(xpath);
                    if (xmlNode != this.xmlRoot)
                        retreatToListenMode = true;
                }
            }
            
            if (retreatToListenMode || this.xmlRoot == xmlNode) {
                

                //Set Component in listening state untill data becomes available again.
                return model.$waitForXml(this);
            }
        }

        //Action Tracker Support
        if (UndoObj && !UndoObj.xmlNode)
            UndoObj.xmlNode = this.xmlRoot;

        //Set Properties

        
        var b, lrule, rule, bRules, bRule, value;
        if (b = this.$bindings) {
	        for (rule in b) {
	            lrule = rule.toLowerCase();
	            if (this.$supportedProperties.indexOf(lrule) > -1) {
                    bRule = (bRules = b[lrule]).length == 1 
                      ? bRules[0] 
                      : this.$getBindRule(lrule, xmlNode);

                    value = bRule.value || bRule.match;

	                
                    //Remove any bounds if relevant
                    this.$clearDynamicProperty(lrule);
            
                    if (value.indexOf("{") > -1 || value.indexOf("[") > -1)
                        this.$setDynamicProperty(lrule, value);
                    else 
                    
                    if (this.setProperty)
                        this.setProperty(lrule, value);
	            }
	        }
	    }
        

        //@todo Think should be set in the event by the Validation Class
        if (this.errBox && this.isValid && this.isValid())
            this.clearError();
        
        this.dispatchEvent("xmlupdate", {
            action : action,
            xmlNode: xmlNode,
            UndoObj: UndoObj
        });
    };

    //@todo apf3.0 this is wrong
    /**
     * @event $clear Clears the data loaded into this element resetting it's value.
     */
    this.addEventListener("$clear", function(nomsg, do_event){
        if (this.$propHandlers && this.$propHandlers["value"]) {
            this.value = -99999; //force resetting
            this.$propHandlers["value"].call(this, "");
        }
    });
};
apf.StandardBinding.prototype = new apf.DataBinding();

apf.Init.run("standardbinding");




/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__MULTISELECT__ = 1 << 8;



/**
 * All elements inheriting from this {@link term.baseclass baseclass} have selection features. This includes handling
 * for multiselect and several keyboard based selection interaction. It also
 * takes care of {@link term.caret caret} handling when multiselect is enabled. Furthermore features 
 * for dealing with multinode component are included like adding and removing 
 * {@link term.datanode data nodes}.
 *
 * #### Example
 *
 * In this example the tree contains nodes that have a disabled flag set. These nodes cannot be selected.
 *
 * ```xml
 *  <a:list width="200">
 *      <a:bindings>
 *          <a:selectable match="[self::node()[not(@disabled) or @disabled != 'true']]" />
 *          <a:each match="[person]"></a:each>
 *          <a:caption match="[@name]"></a:caption>
 *      </a:bindings>
 *      <a:model>
 *          <data>
 *              <person disabled="false" name="test 5"/>
 *              <person disabled="true" name="test 3"/>
 *              <person name="test 4"/>
 *              <person disabled="true" name="test 2"/>
 *              <person disabled="true" name="test 1"/>
 *          </data>
 *      </a:model>
 *  </a:list>
 * ```
 *
 * #### Example
 *
 * ```xml
 *  <a:dropdown onafterchange="alert(this.value)">
 *      <a:bindings>
 *          <a:caption match="[text()]" />
 *          <a:value match="[@value]" />
 *          <a:each match="[item]" />
 *      </a:bindings>
 *      <a:model>
 *          <items>
 *              <item value="#FF0000">red</item>
 *              <item value="#00FF00">green</item>
 *              <item value="#0000FF">blue</item>
 *          </items>
 *      </a:model>
 *  </a:dropdown>
 * ```
 *
 * @class apf.MultiSelect
 * @baseclass
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.5
 *
 * @inherits apf.MultiselectBinding
 *
 */
/**
 *
 * @binding select Determines whether the {@link term.eachnode each node} can be selected.
 *
 */
 /**
 *
 * * @binding value  Determines the way the value for the element is retrieved
 * from the selected node. The `apf.MultiSelect.value` property contains this value.
 *
 */
apf.MultiSelect = function(){
    this.$init(function(){
        this.$valueList       = [];
        this.$selectedList    = [];
    });
};

//@todo investigate if selectedList can be deprecated
(function() {
    this.$regbase    = this.$regbase|apf.__MULTISELECT__;

    // *** Properties *** //

    // @todo Doc is that right? 
    /**
     * The last selected item of this element.
     * @type {XMLElement} 
     */
    this.sellength    = 0;
    this.selected     = null;
    this.$selected    = null;
    
    /**
     * The XML element that has the {@link term.caret caret}.
     * @type {XMLElement} 
     */
    this.caret    = null;
    this.$caret   = null;
    
    /**
     * Specifies whether to use a {@link term.caret caret} in the interaction of this element.
     * @type {Boolean} 
     */
    this.useindicator = true;

    

    /**
     * Removes a {@link term.datanode data node} from the data of this element.
     *
     * #### Example
     *
     * A simple list showing products. This list is used in all the following examples.
     * 
     * ```xml
     *  <a:list id="myList">
     *      <a:bindings>
     *          <a:caption match="[@name]" />
     *          <a:value match="[@id]" />
     *          <a:icon>[@type].png</a:icon>
     *          <a:each match="[product]" />
     *      </a:bindings>
     *      <a:model>
     *          <products>
     *              <product name="Soundblaster" type="audio"    id="product10" length="12" />
     *              <product name="Teapot"       type="3d"       id="product13" />
     *              <product name="Coprocessor"  type="chips"    id="product15" />
     *              <product name="Keyboard"     type="input"    id="product17" />
     *              <product name="Diskdrive"    type="storage"  id="product20" />
     *          </products>
     *      </a:model>
     *  </a:list>
     * ```
     *
     * #### Example
     *
     * This example selects a product by its value and then removes the selection.
     * 
     * ```xml
     *  <a:script><!--
     *      apf.onload = function() {
     *          myList.setValue("product20");
     *          myList.remove();
     *      }
     *  --></a:script>
     * ```
     *
     * #### Example
     *
     * This example gets a product by its value and then removes it.
     * 
     * ```xml
     *  <a:script>
     *      var xmlNode = myList.findXmlNodeByValue("product20");
     *      myList.remove(xmlNode);
     *  </a:script>
     * ```
     *
     * #### Example
     *
     * This example retrieves all nodes from the list. All items with a length
     * greater than 10 are singled out and removed.
     * 
     * ```xml
     *  <a:script><![CDATA[
     *      apf.onload = function() {
     *          var list = myList.getTraverseNodes();
     * 
     *          var removeList = [];
     *          for (var i = 0; i < list.length; i++) {
     *              if (list[i].getAttribute("length") > 10)
     *                  removeList.push(list[i]);
     *          }
     *          myList.remove(removeList);
     *      }
     *   ]]></a:script>
     * ```
     * 
     * #### Remarks
     *
     * Another way to trigger this method is by using the action attribute on a
     * button.
     *
     * ```xml
     *  <a:button action="remove" target="myList">Remove item</a:button>
     * ```
     *
     * Using the action methodology, you can let the original data source
     * (usually the server) know that the user removed an item:
     * 
     * ```xml
     *     <a:list>
     *         <a:bindings />
     *         <a:remove set="remove_product.php?id=[@id]" />
     *     </a:list>
     * ```
     *
     * For undo, this action should be extended and the server should maintain a
     * copy of the deleted item.
     * 
     * ```xml
     *  <a:list actiontracker="atList">
     *      <a:bindings />
     *      <a:remove set  = "remove_product.php?id=[@id]"
     *                undo = "undo_remove_product.php?id=[@id]" />
     *  </a:list>
     *  <a:button 
     *    action = "remove" 
     *    target = "myList">Remove item</a:button>
     *   <a:button 
     *     caption  = "Undo"
     *     disabled = "{!atList.undolength}" 
     *     onclick  = "atList.undo()" />
     * ```
     *
     * @action
     * @param  {NodeList | XMLElement} [nodeList]  The {@link term.datanode data node}(s) to be removed. If none are specified, the current selection is removed.
     *
     * @return  {Boolean}  Indicates if the removal succeeded
     */
    this.remove = function(nodeList){
        //Use the current selection if no xmlNode is defined
        if (!nodeList)
            nodeList = this.$valueList;

        //If we're an xml node let's convert
        if (nodeList.nodeType)
            nodeList = [nodeList];

        //If there is no selection we'll exit, nothing to do
        if (!nodeList || !nodeList.length)
            return;

        

        var changes = [];
        for (var i = 0; i < nodeList.length; i++) {
            changes.push({
                action : "removeNode",
                args   : [nodeList[i]]
            });
        }

        if (this.$actions["removegroup"])
            return this.$executeAction("multicall", changes, "removegroup", nodeList[0]);
        else {
            return this.$executeAction("multicall", changes, "remove", 
              nodeList[0], null, null, nodeList.length > 1 ? nodeList : null);
        }
    };
    
    /**
     * Adds a {@link term.datanode data node} to the data of this element.
     *
     * #### Example
     *
     * A simple list showing products. This list is used in all following examples.
     * 
     * ```xml
     *  <a:list id="myList">
     *      <a:bindings>
     *          <a:caption match="[@name]" />
     *          <a:value match="[@id]" />
     *          <a:icon>[@type].png</a:icon>
     *          <a:each match="[product]" />
     *      </a:bindings>
     *      <a:model>
     *          <products>
     *              <product name="Soundblaster" type="audio"    id="product10" />
     *              <product name="Teapot"       type="3d"       id="product13" />
     *              <product name="Coprocessor"  type="chips"    id="product15" />
     *              <product name="Keyboard"     type="input"    id="product17" />
     *              <product name="Diskdrive"    type="storage"  id="product20" />
     *          </products>
     *      </a:model>
     *  </a:list>
     * ```
     *
     * #### Example
     *
     * This example adds a product to this element selection.
     *
     * ```xml
     *  <a:script><![CDATA[
     *      apf.onload = function() {
     *          myList.add('<product name="USB drive" type="storage" />');
     *      }
     *  ]]></a:script>
     * ```
     *
     * #### Example
     *
     * This example copys the selected product, changes its name, and then
     * adds it. After selecting the new node, the user is offered a rename input
     * box.
     * 
     * ```xml
     *  <a:script><![CDATA[
     *      apf.onload = function() {
     *          var xmlNode = apf.xmldb.copy(myList.selected);
     *          xmlNode.setAttribute("name", "New product");
     *          myList.add(xmlNode);
     *          myList.select(xmlNode);
     *          myList.startRename();
     *      }
     *  ]]></a:script>
     * ```
     * 
     * #### Remarks
     * Another way to trigger this method is by using the action attribute on a
     * button.
     *
     * ```xml
     *  <a:list>
     *      <a:bindings />
     *      <a:model />
     *      <a:actions>
     *          <a:add>
     *              <product name="New item" />
     *          </a:add>
     *      </a:actions>
     *  </a:list>
     *  <a:button action="add" target="myList">Add new product</a:button>
     * ```
     *
     * Using the action methodology you can let the original data source (usually the server) know that the user added an item.
     * 
     * ```xml
     *  <a:add get="{comm.addProduct()}" />
     * ```
     *
     * For undo, this action should be extended as follows.
     * 
     * ```xml
     *  <a:list id="myList" actiontracker="atList">
     *      <a:bindings />
     *      <a:model />
     *      <a:actions>
     *          <a:add set  = "add_product.php?xml=%[.]"
     *              undo = "remove_product.php?id=[@id]">
     *              <product name="New product" id="productId" />
     *          </a:add>
     *      </a:actions>
     *  </a:list>
     *  <a:button 
     *    action = "add" 
     *    target = "myList">Add new product</a:button>
     *  <a:button
     *     caption  = "Undo"
     *     disabled = "{!atList.undolength}" 
     *     onclick  = "atList.undo()" />
     * ```
     *
     * In some cases the server needs to create the new product before its
     * added. This is done as follows.
     * 
     * ```xml
     *  <a:add get="{comm.createNewProduct()}" />
     * ```
     * Alternatively the template for the addition can be provided as a child of
     * the action rule.
     * ```
     *  <a:add set="add_product.php?xml=%[.]">
     *      <product name="USB drive" type="storage" />
     *  </a:add>
     * ```
     *
     * @action
     * @param  {XMLElement} [xmlNode]    The {@link term.datanode data node} which is added. If none is specified the action will use the action rule to try to retrieve a new node to add
     * @param  {XMLElement} [pNode]      The parent node of the added {@link term.datanode data node}
     * @param  {XMLElement} [beforeNode] The position where the XML element should be inserted
     * @return  {XMLElement} The added {@link term.datanode data node} or false on failure
     */
    this.add = function(xmlNode, pNode, beforeNode, userCallback){
        var rule;

        if (this.$actions) {
            if (xmlNode && xmlNode.nodeType)
                rule = this.$actions.getRule("add", xmlNode);
            else if (typeof xmlNode == "string") {
                if (xmlNode.trim().charAt(0) == "<") {
                    xmlNode = apf.getXml(xmlNode);
                    rule = this.$actions.getRule("add", xmlNode);
                }
                else {
                    var rules = this.$actions["add"];
                    for (var i = 0, l = rules.length; i < l; i++) {
                        if (rules[i].getAttribute("type") == xmlNode) {
                            xmlNode = null;
                            rule = rules[i];
                            break;
                        }
                    }
                }
            }

            if (!rule) 
                rule = (this.$actions["add"] || {})[0];
        }
        else
            rule = null;
            
        
        
        var refNode  = this.$isTreeArch ? this.selected || this.xmlRoot : this.xmlRoot,
            amlNode  = this,
        callback = function(addXmlNode, state, extra){
            if (state != apf.SUCCESS) {
                var oError;

                oError = new Error(apf.formatErrorString(1032, amlNode,
                    "Loading xml data",
                    "Could not add data for control " + amlNode.name
                    + "[" + amlNode.tagName + "] \nUrl: " + extra.url
                    + "\nInfo: " + extra.message + "\n\n" + xmlNode));

                if (extra.tpModule.retryTimeout(extra, state, amlNode, oError) === true)
                    return true;

                throw oError;
            }

            /*if (apf.supportNamespaces && node.namespaceURI == apf.ns.xhtml) {
                node = apf.getXml(node.xml.replace(/xmlns\=\"[^"]*\"/g, ""));
                //@todo import here for webkit?
            }*/

            if (typeof addXmlNode != "object")
                addXmlNode = apf.getXmlDom(addXmlNode).documentElement;
            if (addXmlNode.getAttribute(apf.xmldb.xmlIdTag))
                addXmlNode.setAttribute(apf.xmldb.xmlIdTag, "");

            var actionNode = amlNode.$actions &&
              amlNode.$actions.getRule("add", amlNode.$isTreeArch
                ? amlNode.selected
                : amlNode.xmlRoot);
            if (!pNode) {
                if (actionNode && actionNode.parent) {
                    pNode = (actionNode.cparent 
                      || actionNode.compile("parent", {
                        xpathmode  : 2, 
                        injectself : true
                      }))(amlNode.$isTreeArch
                          ? amlNode.selected || amlNode.xmlRoot
                          : amlNode.xmlRoot);
                }
                else {
                    pNode = amlNode.$isTreeArch
                      ? amlNode.selected || amlNode.xmlRoot
                      : amlNode.xmlRoot
                }
            }

            if (!pNode)
                pNode = amlNode.xmlRoot;

            //Safari issue not auto importing nodes:
            if (apf.isWebkit && pNode.ownerDocument != addXmlNode.ownerDocument)
                addXmlNode = pNode.ownerDocument.importNode(addXmlNode, true); 

            

            if (amlNode.$executeAction("appendChild",
              [pNode, addXmlNode, beforeNode], "add", addXmlNode) !== false
              && amlNode.autoselect)
                amlNode.select(addXmlNode);

            if (userCallback)
                userCallback.call(amlNode, addXmlNode);

            return addXmlNode;
        };

        if (xmlNode)
            return callback(xmlNode, apf.SUCCESS);
        else {
            if (rule.get)
                return apf.getData(rule.get, {xmlNode: refNode, callback: callback})
            else {
                
            }
        }

        return addXmlNode;
    };

    if (!this.setValue) {
        /**
         * Sets the value of this element. The value
         * corresponds to an item in the list of loaded {@link term.datanode data nodes}. This
         * element will receive the selection. If no {@link term.datanode data node} is found, the
         * selection is cleared.
         *
         * @param  {String}  value  The new value for this element.
         * @see apf.MultiSelect.getValue
         */
        this.setValue = function(value, disable_event){
            // @todo apf3.0 what does noEvent do? in this scope it's useless and
            // doesn't improve codeflow with a global lookup and assignment
            noEvent = disable_event;
            this.setProperty("value", value, false, true);
            noEvent = false;
        };
    }

    /**
     * Retrieves an {@link term.datanode data node} that has a value that corresponds to the
     * string that is searched on.
     * @param {String} value The value to match.
     * @returns {XMLNode} The found node, or `false`
     */
    this.findXmlNodeByValue = function(value){
        var nodes   = this.getTraverseNodes(),
            bindSet = this.$attrBindings["eachvalue"]
                && "eachvalue" || this.$bindings["value"]
                && "value" || this.$hasBindRule("caption") && "caption";
        
        if (!bindSet)
            return false;
            
        for (var i = 0; i < nodes.length; i++) {
            if (this.$applyBindRule(bindSet, nodes[i]) == value)
                return nodes[i];
        }
    };

    if (!this.getValue) {
        /**
         * Retrieves the value of this element. This is the value of the
         * first selected {@link term.datanode data node}.
         * 
         */
        this.getValue = function(xmlNode, noError){
            return this.value;
            /*
            if (!this.bindingRules && !this.caption) 
                return false;

            

            return this.$applyBindRule(this.$mainBind, xmlNode || this.selected, null, true)
                || this.$applyBindRule("caption", xmlNode || this.selected, null, true);
            */
        };
    }

    /**
     * Select the current selection...again.
     *
     */
    this.reselect = function(){ // @todo Add support for multiselect
        if (this.selected) this.select(this.selected, null, this.ctrlselect,
            null, true);//no support for multiselect currently.
    };

    /**
     * @event  beforeselect  Fires before a {@link apf.MultiSelect.select selection} is made
     * @param {Object} e The standard event object. It contains the following properties:
     *                     - `selected` ([[XMLElement]]): The {@link term.datanode data node} that will be selected
     *                     - `selection` ([[Array]]): An array of {@link term.datanode data nodes} that will be selected
     *                     - `htmlNode` ([[HTMLElement]]): The HTML element that visually represents the {@link term.datanode data node} 
     */
    /**
     * @event  afterselect  Fires after a {@link apf.MultiSelect.select selection} is made
     * @param {Object} e The standard event object. It contains the following properties:
     *                     - `selected` ([[XMLElement]]):    the {@link term.datanode data node} that was selected
     *                     - `selection` ([[Array]]():       an array of {@link term.datanode data node} that are selected
     *                     - `htmlNode` ([[HTMLElement]](): the HTML element that visually represents the {@link term.datanode data node}
     */
    /**
     * Selects a single, or set, of {@link term.eachnode each nodes}.
     * The selection can be visually represented in this element.
     *
     * @param {Mixed}   xmlNode      The identifier to determine the selection. It can be one of the following values:
     *                                 - ([[XMLElement]]):  The {@link term.datanode data node} to be used in the selection as a start/end point or to toggle the selection on the node.
     *                                 - ([[HTMLElement]]): The HTML element node used as visual representation of {@link term.datanode data node}. 
     *                                                 Used to determine the {@link term.datanode data node} for selection.
     *                                 - ([[String]]):      The value of the {@link term.datanode data node} to be selected.
     * @param {Boolean} [ctrlKey]    Indicates whether the [[keys: Ctrl]] key was pressed
     * @param {Boolean} [shiftKey]   Indicates whether the [[keys: Shift]]  key was pressed
     * @param {Boolean} [fakeselect] Indicates whether only visually a selection is made
     * @param {Boolean} [force]      Indicates whether reselect is forced
     * @param {Boolean} [noEvent]    Indicates whether to not call any event
     * @return  {Boolean}  Indicates whether the selection could be made
     *
     */
    this.select  = function(xmlNode, ctrlKey, shiftKey, fakeselect, force, noEvent, userAction){
        if (!this.selectable || userAction && this.disabled) 
            return;

        if (parseInt(fakeselect) == fakeselect) {
            //Don't select on context menu
            if (fakeselect == 2) {
                fakeselect = true;
    	      	userAction = true;
            }
            else {
    	      	fakeselect = false;
    	      	userAction = true;
    	    }
        }

        if (this.$skipSelect) {
            this.$skipSelect = false;
            return;
        }

        if (this.ctrlselect && !shiftKey)
            ctrlKey = true;

        if (!this.multiselect)
            ctrlKey = shiftKey = false;
        
        // Selection buffering (for async compatibility)
        if (!this.xmlRoot) {
            if (!this.$buffered) {
                var f;
                this.addEventListener("afterload", f = function(){
                    this.select.apply(this, this.$buffered);
                    this.removeEventListener("afterload", f);
                    delete this.$buffered;
                });
            }

            this.$buffered = Array.prototype.slice.call(arguments);
            return;
        }

        var htmlNode;

        // *** Type Detection *** //
        if (!xmlNode) {
            

            return false;
        }

        if (typeof xmlNode != "object") {
            var str = xmlNode; xmlNode = null;
            if (typeof xmlNode == "string")
                xmlNode = apf.xmldb.getNodeById(str);

            //Select based on the value of the xml node
            if (!xmlNode) {
                xmlNode = this.findXmlNodeByValue(str);
                if (!xmlNode) {
                    this.clearSelection(noEvent);
                    return;
                }
            }
        }
        
        if (!(typeof (xmlNode.style || "") == "object")) {
            htmlNode = this.$findHtmlNode(xmlNode.getAttribute(
                    apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);
        }
        else {
            var id = (htmlNode = xmlNode).getAttribute(apf.xmldb.htmlIdTag);
            while (!id && htmlNode.parentNode)
                id = (htmlNode = htmlNode.parentNode).getAttribute(
                    apf.xmldb.htmlIdTag);

            xmlNode = apf.xmldb.getNodeById(id);//, this.xmlRoot);
        }

        if (!shiftKey && !ctrlKey && !force && !this.reselectable 
          && this.$valueList.length <= 1 && this.$valueList.indexOf(xmlNode) > -1)
            return;

        if (this.dispatchEvent('beforeselect', {
            selected    : xmlNode,
            htmlNode    : htmlNode,
            ctrlKey     : ctrlKey,
            shiftKey    : shiftKey,
            force       : force,
            captureOnly : noEvent
        }) === false)
              return false;

        // *** Selection *** //

        var lastIndicator = this.caret;
        this.caret        = xmlNode;

        //Multiselect with SHIFT Key.
        if (shiftKey) {
            var range = this.$calcSelectRange(
              this.$valueList[0] || lastIndicator, xmlNode);

            if (this.$caret)
                this.$deindicate(this.$caret);

            this.selectList(range);

            this.$selected  =
            this.$caret     = this.$indicate(htmlNode);
        }
        else if (ctrlKey) { //Multiselect with CTRL Key.
            //Node will be unselected
            if (this.$valueList.contains(xmlNode)) {
                if (this.selected == xmlNode) {
                    this.$deselect(this.$findHtmlNode(this.selected.getAttribute(
                        apf.xmldb.xmlIdTag) + "|" + this.$uniqueId));
                    
                    this.$deindicate(this.$caret);

                    if (this.$valueList.length && !fakeselect) {
                        //this.$selected = this.$selectedList[0];
                        this.selected = this.$valueList[0];
                    }
                }
                else
                    this.$deselect(htmlNode, xmlNode);

                if (!fakeselect) {
                    this.$selectedList.remove(htmlNode);
                    this.$valueList.remove(xmlNode);
                }

                if (htmlNode != this.$caret)
                    this.$deindicate(this.$caret);

                this.$selected  =
                this.$caret     = this.$indicate(htmlNode);
            }
            // Node will be selected
            else {
                if (this.$caret)
                    this.$deindicate(this.$caret);
                this.$caret = this.$indicate(htmlNode, xmlNode);

                this.$selected   = this.$select(htmlNode, xmlNode);
                this.selected    = xmlNode;

                if (!fakeselect) {
                    this.$selectedList.push(htmlNode);
                    this.$valueList.push(xmlNode);
                }
            }
        }
        else if (fakeselect && htmlNode && this.$selectedList.contains(htmlNode)) {//Return if selected Node is htmlNode during a fake select
            return;
        }
        else { //Normal Selection
            //htmlNode && this.$selected == htmlNode && this.$valueList.length <= 1 && this.$selectedList.contains(htmlNode)
            if (this.$selected)
                this.$deselect(this.$selected);
            if (this.$caret)
                this.$deindicate(this.$caret);
            if (this.selected)
                this.clearSelection(true);

            this.$caret = this.$indicate(htmlNode, xmlNode);
            this.$selected  = this.$select(htmlNode, xmlNode);
            this.selected   = xmlNode;

            this.$selectedList.push(htmlNode);
            this.$valueList.push(xmlNode);
        }

        if (this.delayedselect && (typeof ctrlKey == "boolean")){
            var _self = this;
            $setTimeout(function(){
                if (_self.selected == xmlNode)
                    _self.dispatchEvent("afterselect", {
                        selection   : _self.$valueList,
                        selected    : xmlNode,
                        caret       : _self.caret,
                        captureOnly : noEvent
                    });
            }, 10);
        }
        else {
            this.dispatchEvent("afterselect", {
                selection   : this.$valueList,
                selected    : xmlNode,
                caret       : this.caret,
                captureOnly : noEvent
            });
        }

        return true;
    };

    /**
     * @event  beforechoose  Fires before a choice is made.
     * @param {Object} e The standard event object. It contains the following properties:
     *                   - `xmlNode` ([[XMLElement]]):   The {@link term.datanode data node} that was choosen
     *
     */
    /**
     * @event  afterchoose   Fires after a choice is made.
     * @param {Object} e The standard event object. It contains the following properties:
     *                   - `xmlNode` ([[XMLElement]]):   The {@link term.datanode data node} that was choosen
     */
    /**
     * Chooses a selected item. This is done by double clicking on the item or
     * pressing the Enter key.
     *
     * @param {Mixed}   xmlNode      The identifier to determine the selection. It can be one of the following values: 
     *                                - [[XMLElement]]:  The {@link term.datanode data node} to be choosen
     *                                - [[HTMLElement]]: The HTML element node used as visual representation of {@link term.datanode data node}
     *                                                 Used to determine the {@link term.datanode data node}
     *                                - [[String]] :     The value of the {@link term.datanode data node} to be choosen
     *
     */
    this.choose = function(xmlNode, userAction){
        if (!this.selectable || userAction && this.disabled) return;

        if (this.dispatchEvent("beforechoose", {xmlNode : xmlNode}) === false)
            return false;

        if (xmlNode && !(typeof (xmlNode.style || "") == "object"))
            this.select(xmlNode);

        
        if (this.hasFeature(apf.__DATABINDING__)
          && this.dispatchEvent("afterchoose", {xmlNode : this.selected}) !== false)
            this.setProperty("chosen", this.selected);
        
    };

    /*
     * Removes the selection of one or more selected nodes.
     *
     * @param {Boolean} [noEvent]    Indicates whether or not to call any events
     */
    // @todo Doc
    this.clearSelection = function(noEvent, userAction){
        if (!this.selectable || userAction && this.disabled || !this.$valueList.length)
            return;
        
        if (!noEvent) {
            if (this.dispatchEvent("beforeselect", {
                selection : [],
                selected  : null,
                caret     : this.caret
            }) === false)
                return false;
        }

        //Deselect html nodes
        var htmlNode;
        for (var i = this.$valueList.length - 1; i >= 0; i--) {
            htmlNode = this.$findHtmlNode(this.$valueList[i].getAttribute(
                    apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);
            this.$deselect(htmlNode);
        }
        
        //Reset internal variables
        this.$selectedList.length = 0;
        this.$valueList.length    = 0;
        this.$selected            =
        this.selected             = null;

        //Redraw indicator
        if (this.caret) {
            htmlNode = this.$findHtmlNode(this.caret.getAttribute(
                    apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);

            this.$caret = this.$indicate(htmlNode);
        }

        if (!noEvent) {
            this.dispatchEvent("afterselect", {
                selection : this.$valueList,
                selected  : null,
                caret     : this.caret
            });
        }
    };

    /*
     * Selects a set of items
     *
     * @param {Array} xmlNodeList the {@link term.datanode data nodes} that will be selected.
     */
    //@todo Doc I think there are missing events here?
    this.selectList = function(xmlNodeList, noEvent, selected, userAction){
        if (!this.selectable || userAction && this.disabled) return;

        if (this.dispatchEvent("beforeselect", {
            selection   : xmlNodeList,
            selected    : selected || xmlNodeList[0],
            caret       : this.caret,
            captureOnly : noEvent
          }) === false)
            return false;

        this.clearSelection(true);

        for (var sel, i = 0; i < xmlNodeList.length; i++) {
            //@todo fix select state in unserialize after removing
            if (!xmlNodeList[i] || xmlNodeList[i].nodeType != 1) continue;
            var htmlNode,
                xmlNode = xmlNodeList[i];

            //Type Detection
            if (typeof xmlNode != "object")
                xmlNode = apf.xmldb.getNodeById(xmlNode);
            if (!(typeof (xmlNode.style || "") == "object"))
                htmlNode = this.$pHtmlDoc.getElementById(xmlNode.getAttribute(
                    apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);
            else {
                htmlNode = xmlNode;
                xmlNode  = apf.xmldb.getNodeById(htmlNode.getAttribute(
                    apf.xmldb.htmlIdTag));
            }

            if (!xmlNode) {
                
                continue;
            }

            //Select Node
            if (htmlNode) {
                if (!sel && selected == htmlNode)
                    sel = htmlNode;

                this.$select(htmlNode, xmlNode);
                this.$selectedList.push(htmlNode);
            }
            this.$valueList.push(xmlNode);
        }

        this.$selected = sel || this.$selectedList[0];
        this.selected  = selected || this.$valueList[0];

        this.dispatchEvent("afterselect", {
            selection   : this.$valueList,
            selected    : this.selected,
            caret       : this.caret,
            captureOnly : noEvent
        });
    };

    /**
     * @event indicate Fires when an item becomes the indicator.
     */

    /**
     * Sets the {@link term.caret caret} on an item to indicate to the user that the keyboard
     * actions are done relevant to that item. Using the keyboard,
     * a user can change the position of the indicator using the [[keys: Ctrl]] and arrow
     * keys while not making a selection. When making a selection with the mouse
     * or keyboard, the indicator is always set to the selected node. Unlike a
     * selection there can be only one indicator item.
     *
     * @param {Mixed}   xmlNode      The identifier to determine the indicator. Its possible values include:
     *                                - {XMLElement}  The {@link term.datanode data node} to be set as indicator.
     *                                - {HTMLElement} The HTML element node used as visual representation of
     *                                                   {@link term.datanode data node}. Used to determine the {@link term.datanode data node}.
     *                                - {String}      The value of the {@link term.datanode data node} to be set as an indicator.
     */
    this.setCaret = function(xmlNode){
        if (!xmlNode) {
            if (this.$caret)
                this.$deindicate(this.$caret);
            this.caret  =
            this.$caret = null;
            return;
        }

        // *** Type Detection *** //
        var htmlNode;
        if (typeof xmlNode != "object")
            xmlNode = apf.xmldb.getNodeById(xmlNode);
        if (!(typeof (xmlNode.style || "") == "object")) {
            htmlNode = this.$findHtmlNode(xmlNode.getAttribute(
                    apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);
        }
        else {
            var id = (htmlNode = xmlNode).getAttribute(apf.xmldb.htmlIdTag);
            while (!id && htmlNode.parentNode && htmlNode.parentNode.nodeType == 1)
                id = (htmlNode = htmlNode.parentNode).getAttribute(
                    apf.xmldb.htmlIdTag);
            if (!id) alert(this.$int.outerHTML);

            xmlNode = apf.xmldb.getNodeById(id);
        }

        if (this.$caret) {
            //this.$deindicate(this.$findHtmlNode(this.caret.getAttribute(
                //apf.xmldb.xmlIdTag) + "|" + this.$uniqueId));
            this.$deindicate(this.$caret);
        }
        
        this.$caret = this.$indicate(htmlNode);
        this.setProperty("caret", this.caret = xmlNode);
    };

    /*
     * @private
     */
    this.$setTempSelected = function(xmlNode, ctrlKey, shiftKey, down){
        clearTimeout(this.timer);

        if (this.$bindings.selectable) {
            while (xmlNode && !this.$getDataNode("selectable", xmlNode)) {
                xmlNode = this.getNextTraverseSelected(xmlNode, !down);
            }
            if (!xmlNode) return;
        }

        if (!this.multiselect)
            ctrlKey = shiftKey = false;

        if (ctrlKey || this.ctrlselect) {
            if (this.$tempsel) {
                this.select(this.$tempsel);
                this.$tempsel = null;
            }

            this.setCaret(xmlNode);
        }
        else if (shiftKey){
            if (this.$tempsel) {
                this.$selectTemp();
                this.$deselect(this.$tempsel);
                this.$tempsel = null;
            }

            this.select(xmlNode, null, shiftKey);
        }
        else if (!this.bufferselect || this.$valueList.length > 1) {
            this.select(xmlNode);
        }
        else {
            var id = apf.xmldb.getID(xmlNode, this);

            this.$deselect(this.$tempsel || this.$selected);
            this.$deindicate(this.$tempsel || this.$caret);
            this.$tempsel = this.$indicate(document.getElementById(id));
            this.$select(this.$tempsel);

            var _self = this;
            this.timer = $setTimeout(function(){
                _self.$selectTemp();
            }, 400);
        }
    };

    /*
     * @private
     */
    this.$selectTemp = function(){
        if (!this.$tempsel)
            return;

        clearTimeout(this.timer);
        this.select(this.$tempsel);

        this.$tempsel = null;
        this.timer    = null;
    };

    /**
     * Selects all the {@link term.eachnode each nodes} of this element
     * 
     */
    this.selectAll = function(userAction){
        if (!this.multiselect || !this.selectable
          || userAction && this.disabled || !this.xmlRoot)
            return;

        var nodes = this.$isTreeArch
            ? this.xmlRoot.selectNodes(".//" 
              + this.each.split("|").join("|.//"))
            : this.xmlRoot.selectNodes(this.each);
        
        this.selectList(nodes);
    };

    /**
     * Retrieves an Array or a document fragment containing all the selected
     * {@link term.datanode data nodes} from this element.
     *
     * @param {Boolean} [xmldoc] Specifies whether the method should return a document fragment.
     * @return {Mixed} The selection of this element.
     */
    this.getSelection = function(xmldoc){
        var i, r;
        if (xmldoc) {
            r = this.xmlRoot
                ? this.xmlRoot.ownerDocument.createDocumentFragment()
                : apf.getXmlDom().createDocumentFragment();
            for (i = 0; i < this.$valueList.length; i++)
                apf.xmldb.cleanNode(r.appendChild(
                    this.$valueList[i].cloneNode(true)));
        }
        else {
            for (r = [], i = 0; i < this.$valueList.length; i++)
                r.push(this.$valueList[i]);
        }

        return r;
    };
    
    this.$getSelection = function(htmlNodes){
        return htmlNodes ? this.$selectedList : this.$valueList;
    };

    /**
     * Selects the next {@link term.datanode data node} to be selected.
     *
     * @param  {XMLElement}  xmlNode  The context {@link term.datanode data node}.
     * @param  {Boolean}     [isTree] If `true`, indicates that this node is a tree, and should select children
     */
    this.defaultSelectNext = function(xmlNode, isTree){
        var next = this.getNextTraverseSelected(xmlNode);
        //if(!next && xmlNode == this.xmlRoot) return;

        //@todo Why not use this.$isTreeArch ??
        if (next || !isTree)
            this.select(next ? next : this.getTraverseParent(xmlNode));
        else
            this.clearSelection(true);
    };

    /**
     * Selects the next {@link term.datanode data node} when available.
     */
    this.selectNext = function(){
        var xmlNode = this.getNextTraverse();
        if (xmlNode)
            this.select(xmlNode);
    };

    /**
     * Selects the previous {@link term.datanode data node} when available.
     */
    this.selectPrevious = function(){
        var xmlNode = this.getNextTraverse(null, -1);
        if (xmlNode)
            this.select(xmlNode);
    };

    /*
     * @private
     */
    this.getDefaultNext = function(xmlNode, isTree){  //@todo why is isTree an argument
        var next = this.getNextTraverseSelected(xmlNode);
        //if(!next && xmlNode == this.xmlRoot) return;

        return (next && next != xmlNode)
            ? next
            : (isTree
                ? this.getTraverseParent(xmlNode)
                : null); //this.getFirstTraverseNode()
    };

    /**
     * Determines whether a node is selected.
     *
     * @param  {XMLElement} xmlNode  The {@link term.datanode data node} to be checked
     * @return  {Boolean} Identifies if the element is selected
     */
    this.isSelected = function(xmlNode){
        if (!xmlNode) return false;

        for (var i = 0; i < this.$valueList.length; i++) {
            if (this.$valueList[i] == xmlNode)
                return this.$valueList.length;
        }

        return false;
    };

    /*
     * This function checks whether the current selection is still correct.
     * Selection can become invalid when updates to the underlying data
     * happen. For instance when a selected node is removed.
     */
    this.$checkSelection = function(nextNode){
        if (this.$valueList.length > 1) {
            //Fix selection if needed
            for (var lst = [], i = 0, l = this.$valueList.length; i < l; i++) {
                if (apf.isChildOf(this.xmlRoot, this.$valueList[i]))
                    lst.push(this.$valueList[i]);
            }

            if (lst.length > 1) {
                this.selectList(lst);
                if(this.caret
                  && !apf.isChildOf(this.xmlRoot, this.caret)) {
                    this.setCaret(nextNode || this.selected);
                }
                return;
            }
            else if (lst.length) {
                //this.clearSelection(true); //@todo noEvents here??
                nextNode = lst[0];
            }
        }

        if (!nextNode) {
            if (this.selected
              && !apf.isChildOf(this.xmlRoot, this.selected)) {
                nextNode = this.getFirstTraverseNode();
            }
            else if(this.selected && this.caret
              && !apf.isChildOf(this.xmlRoot, this.caret)) {
                this.setCaret(this.selected);
            }
            else if (!this.selected){
                nextNode = this.xmlRoot
                    ? this.getFirstTraverseNode()
                    : null;
            }
            else {
                return; //Nothing to do
            }
        }

        if (nextNode) {
            if (this.autoselect) {
                this.select(nextNode);
            }
            else {
                this.clearSelection();
                this.setCaret(nextNode);
            }
        }
        else
            this.clearSelection();

        //if(action == "synchronize" && this.autoselect) this.reselect();
    };

    /**
     * @attribute {Boolean} [multiselect]   Sets or gets whether the user may select multiple items. Default is `true, but `false` for dropdown. 
     */
    /**
     * @attribute {Boolean} [autoselect]    Sets or gets whether a selection is made after data is loaded. Default is `true`, but `false` for dropdown. When the string 'all' is set, all {@link term.datanode data nodes} are selected.
     */
    /**
     *  @attribute {Boolean} [selectable]    Sets or gets whether the {@link term.datanode data nodes} of this element can be selected. Default is `true`.
     */
    /**
     *  @attribute {Boolean} [ctrlselect]    Sets or gets whether a selection is made as if the user is holding the [[keys: Ctrl]] key. When set to `true` each mouse selection will add to the current selection. Selecting an already selected element will deselect it.
     */
    /**
     *  @attribute {Boolean} [allowdeselect] Sets or gets whether the user can remove the selection of this element. When set to `true` it is possible for this element to have no selected {@link term.datanode data node}.
     */
    /**
     *  @attribute {Boolean} [reselectable]  Sets or gets whether selected nodes can be selected again, and the selection events are called again. Default is `false`. When set to `false` a selected {@link term.datanode data node} cannot be selected again.
     */
    /**
     *  @attribute {String}  [default]      Sets or gets the value that this component has when no selection is made.
     */
    /**
     *  @attribute {String}  [eachvalue]     Sets or gets the {@link term.expression expression} that determines the value for each {@link term.datanode data nodes} in the dataset of the element.
     * 
     */
    this.selectable = true;
    if (typeof this.ctrlselect == "undefined")
        this.ctrlselect = false;
    if (typeof this.multiselect == "undefined")
        this.multiselect = true;
    if (typeof this.autoselect == "undefined")
        this.autoselect = true;
    if (typeof this.delayedselect == "undefined")
        this.delayedselect = true;
    if (typeof this.allowdeselect == "undefined")
        this.allowdeselect = true;
    this.reselectable = false;

    this.$booleanProperties["selectable"]    = true;
    //this.$booleanProperties["ctrlselect"]    = true;
    this.$booleanProperties["multiselect"]   = true;
    this.$booleanProperties["autoselect"]    = true;
    this.$booleanProperties["delayedselect"] = true;
    this.$booleanProperties["allowdeselect"] = true;
    this.$booleanProperties["reselectable"]  = true;

    this.$supportedProperties.push("selectable", "ctrlselect", "multiselect",
        "autoselect", "delayedselect", "allowdeselect", "reselectable", 
        "selection", "selected", "default", "value", "caret");

    /**
     * @attribute {String} [value]  Sets or gets the value of the element that is selected.
     *
     */
    //@todo add check here
    this.$propHandlers["value"] = function(value){
        if (this.$lastValue == value) {
            delete this.$lastValue;
            return;
        }

        if (!this.$attrBindings["eachvalue"] && !this.$amlLoaded
          && this.getAttribute("eachvalue")) {
            var _self = this;
            return apf.queue.add("value" + this.$uniqueId, function(){
                _self.$propHandlers["value"].call(_self, value);
            });
        }
        
        

        if (value || value === 0 || this["default"])
            this.select(String(value) || this["default"]);
        else
            this.clearSelection();
    }
    
    this.$propHandlers["default"] = function(value, prop){
        if (!this.value || !this.$amlLoaded && !(this.getAttribute("value") 
          || this.getAttribute("selected") || this.getAttribute("selection"))) {
            this.$propHandlers["value"].call(this, "");
        }
    }
    
    /**
     * @attribute {String} [value]   Sets or gets the caret value of the element.
     */
    //@todo fill this in
    this.$propHandlers["caret"] = function(value, prop){
        if (value)
            this.setCaret(value);
    }
    
    
    
    //@todo optimize this thing. Also implement virtual dataset support.
    /**
     * @attribute {String} [selection]  Sets or gets the {@link term.expression expression} that determines the selection for this element. A reference to an XML nodelist can be passed as well.
     *
     */
    this.$propHandlers["selection"] = 
    
    /**
     * @attribute {String} [selected]   Sets or gets the {@link term.expression expression} that determines the selected node for this element. A reference to an XML element can be passed as well.
     * 
     */
    this.$propHandlers["selected"] = function(value, prop) {
        if (!value) value = this[prop] = null;

        if (prop == "selected" && typeof value != "string") { // && value == this.selected
            if (value && value.nodeType != 1)
                value = value.nodeValue;
            else
            //this.selected = null; //I don't remember why this is here. It removes the selected property without setting it again. (dropdown test)
                return;
        }
        
        

        if (this.$isSelecting) {
            this.selection = this.$valueList;
            return false;
        }

        var nodes, bindSet, getValue, i, j, c, d;
        //Update the selection
        if (prop == "selection") {
            if (typeof value == "object" && value == this.$valueList) {
                var pNode;
                //We're using an external model. Need to update bound nodeset
                if ((c = this.$attrBindings[prop]) && c.cvalue.models) { //added check, @todo whats up with above assumption?
                    this.$isSelecting = true; //Prevent reentrance (optimization)
    
                    bindSet = this.$attrBindings["eachvalue"] 
                        && "eachvalue" || this.$bindings["value"]
                        && "value" || this.$hasBindRule("caption") && "caption";
                    
                    if (!bindSet)
                        throw new Error("Missing bind rule set: eachvalue, value or caption");//@todo apf3.0 make this into a proper error
                    
                    //@todo this may be optimized by keeping a copy of the selection
                    var selNodes = this.$getDataNode(prop, this.xmlRoot);
                    nodes        = value;
                    getValue     = (d = this.$attrBindings["selection-unique"]) && d.cvalue;
                    
                    if (selNodes.length) {
                        pNode = selNodes[0].parentNode;
                    }
                    else {
                        var model, path;
                        if (c.cvalue.xpaths[0] == "#" || c.cvalue.xpaths[1] == "#") {
                            var m = (c.cvalue3 || (c.cvalue3 = apf.lm.compile(c.value, {
                                xpathmode: 5
                            })))(this.xmlRoot);
                            
                            model = m.model && m.model.$isModel && m.model;
                            if (model)
                                path = m.xpath;
                            else if (m.model) {
                                model = apf.xmldb.findModel(m.model);
                                path = apf.xmlToXpath(m.model, model.data) + (m.xpath ? "/" + m.xpath : ""); //@todo make this better
                            }
                            else {
                                //No selection - nothing to do
                            }
                        }
                        else {
                            
                            model = apf.nameserver.get("model", c.cvalue.xpaths[0]);
                            
                            path  = c.cvalue.xpaths[1];
                        }

                        if (!model || !model.data) {
                            this.$isSelecting = false;
                            return false;
                        }
                        
                        pNode = model.queryNode(path.replace(/\/[^\/]+$|^[^\/]*$/, "") || ".");

                        if (!pNode)
                            throw new Error("Missing parent node"); //@todo apf3.0 make this into a proper error
                    }
                    
                    //Nodes removed
                    remove_loop:
                    for (i = 0; i < selNodes.length; i++) {
                        //Value is either determined by special property or in the 
                        //same way as the value for the bound node.
                        value = getValue 
                          ? getValue(selNodes[i]) 
                          : this.$applyBindRule(bindSet, selNodes[i]);
    
                        //Compare the value with the traverse nodes
                        for (j = 0; j < nodes.length; j++) {
                            if (this.$applyBindRule(bindSet, nodes[j]) == value) //@todo this could be cached
                                continue remove_loop;
                        }
                        
                        //remove node
                        apf.xmldb.removeNode(selNodes[i]);
                    }
                    
                    //Nodes added
                    add_loop:
                    for (i = 0; i < nodes.length; i++) {
                        //Value is either determined by special property or in the 
                        //same way as the value for the bound node.
                        value = this.$applyBindRule(bindSet, nodes[i]);
    
                        //Compare the value with the traverse nodes
                        for (j = 0; j < selNodes.length; j++) {
                            if (getValue 
                              ? getValue(selNodes[j]) 
                              : this.$applyBindRule(bindSet, selNodes[j]) == value) //@todo this could be cached
                                continue add_loop;
                        }
                        
                        //add node
                        var node = this.$attrBindings["selection-constructor"] 
                          && this.$getDataNode("selection-constructor", nodes[i])
                          || apf.getCleanCopy(nodes[i]);
                        apf.xmldb.appendChild(pNode, node);
                    }
                    
                    //@todo above changes should be via the actiontracker
                    this.$isSelecting = false;
                }
                
                return;
            }
            this.selection = this.$valueList;
        }
        else {
            this.selected = null;
        }
        
        if (!this.xmlRoot) {
            if (!this.$buffered) {
                var f;
                this.addEventListener("afterload", f = function(){
                    this.removeEventListener("afterload", f);
                    this.$propHandlers["selected"].call(this, value, prop);
                    delete this.$buffered;
                });
                this.$buffered = true;
            }
            this[prop] = null;
            return false;
        }

        if (!value || typeof value != "object") {
            //this[prop] = null;

            if (this.$attrBindings[prop]) {
                //Execute the selection query
                nodes = this.$getDataNode(prop, this.xmlRoot);
                if (nodes && (nodes.length || nodes.nodeType == 1)) {
                    this.setProperty("selection", nodes);
                    return;
                }
                
                if (!nodes || nodes.length === 0)
                    return;
                
                //Current model, it's an init selection, we'll clear the bind
                /*if (typeof value == "string" 
                  && !this.$attrBindings[prop].cvalue.xpaths[0]) {
                    this.$removeAttrBind(prop);
                }*/
            }
            
            if (!value) {
                this.clearSelection();
            }
            else {
                this.select(value);
            }

            return false; //Disable signalling the listeners to this property
        }
        else if (typeof value.length == "number") {
            nodes = value;
            if (!nodes.length) {
                this.selected = null;
                if (this.$valueList.length) { //dont clear selection when no selection exists (at prop init)
                    this.clearSelection();
                    return false; //Disable signalling the listeners to this property
                }
                else return;
            }
            
            //For when nodes are traverse nodes of this element
            if (this.isTraverseNode(nodes[0]) 
              && apf.isChildOf(this.xmlRoot, nodes[0])) {
                if (!this.multiselect) {
                    this.select(nodes[0]);
                }
                else {
                    //this[prop] = null; //??
                    this.selectList(nodes);
                }
                return false; //Disable signalling the listeners to this property
            }
            
            //if external model defined, loop through items and find mate by value
            if (this.$attrBindings[prop]) { //Can assume an external model is in place
                bindSet = this.$attrBindings["eachvalue"] 
                    && "eachvalue" || this.$bindings["value"]
                    && "value" || this.$hasBindRule("caption") && "caption";
                
                if (!bindSet)
                    throw new Error("Missing bind rule set: eachvalue, value or caption");//@todo apf3.0 make this into a proper error
                
                var tNodes = !this.each 
                    ? this.getTraverseNodes()
                    : this.xmlRoot.selectNodes("//" + this.each.split("|").join("|//"));
                
                getValue = (c = this.$attrBindings["selection-unique"]) && c.cvalue;
                var selList  = [];
                for (i = 0; i < nodes.length; i++) {
                    //Value is either determined by special property or in the 
                    //same way as the value for the bound node.
                    value = getValue 
                        ? getValue(nodes[i]) 
                        : this.$applyBindRule(bindSet, nodes[i]);

                    //Compare the value with the traverse nodes
                    for (j = 0; j < tNodes.length; j++) {
                        if (this.$applyBindRule(bindSet, tNodes[j]) == value) //@todo this could be cached
                            selList.push(tNodes[j]);
                    }
                }
                
                //this[prop] = null; //???
                this.selectList(selList, true); //@todo noEvent to distinguish between user actions and not user actions... need to rethink this
                return false;
            }
            
            throw new Error("Show me which case this is");
        }
        else if (this.$valueList.indexOf(value) == -1) {
            //this.selected = null;
            this.select(value);
        }
    };
    
    
    
    this.$propHandlers["allowdeselect"] = function(value){
        if (value) {
            var _self = this;
            this.$container.onmousedown = function(e){
                if (!e)
                    e = event;
                if (e.ctrlKey || e.shiftKey)
                    return;

                var srcElement = e.srcElement || e.target;
                if (_self.allowdeselect && (srcElement == this
                  || srcElement.getAttribute(apf.xmldb.htmlIdTag)))
                    _self.clearSelection(); //hacky
            }
        }
        else {
            this.$container.onmousedown = null;
        }
    };

    this.$propHandlers["ctrlselect"] = function(value){
        if (value != "enter")
            this.ctrlselect = apf.isTrue(value);
    }

    function fAutoselect(){
        this.selectAll();
    }
    
    this.$propHandlers["autoselect"] = function(value){
        if (value == "all" && this.multiselect)
            this.addEventListener("afterload", fAutoselect);
        else
            this.removeEventListener("afterload", fAutoselect);
    };

    this.$propHandlers["multiselect"] = function(value){
        if (!value && this.$valueList.length > 1)
            this.select(this.selected);

        //if (value)
            //this.bufferselect = false; //@todo doesn't return to original value
    };

    // Select Bind class
    
    this.addEventListener("beforeselect", function(e){
        if (this.$bindings.selectable && !this.$getDataNode("selectable", e.selected))
            return false;
    }, true);
    

    
    this.addEventListener("afterselect", function (e){
        
        var combinedvalue = null;

        
        //@todo refactor below
        /*if (this.caret == this.selected || e.list && e.list.length > 1 && hasConnections) {
            //Multiselect databinding handling... [experimental]
            if (e.list && e.list.length > 1 && this.$getConnections().length) { //@todo this no work no more apf3.0
                var oEl  = this.xmlRoot.ownerDocument.createElement(this.selected.tagName);
                var attr = {};

                //Fill basic nodes
                var nodes = e.list[0].attributes;
                for (var j = 0; j < nodes.length; j++)
                    attr[nodes[j].nodeName] = nodes[j].nodeValue;

                //Remove nodes
                for (var prop, i = 1; i < e.list.length; i++) {
                    for (prop in attr) {
                        if (typeof attr[prop] != "string") continue;

                        if (!e.list[i].getAttributeNode(prop))
                            attr[prop] = undefined;
                        else if(e.list[i].getAttribute(prop) != attr[prop])
                            attr[prop] = "";
                    }
                }

                //Set attributes
                for (prop in attr) {
                    if (typeof attr[prop] != "string") continue;
                    oEl.setAttribute(prop, attr[prop]);
                }

                //missing is childnodes... will implement later when needed...

                oEl.setAttribute(apf.xmldb.xmlIdTag, this.$uniqueId);
                apf.MultiSelectServer.register(oEl.getAttribute(apf.xmldb.xmlIdTag),
                    oEl, e.list, this);
                apf.xmldb.addNodeListener(oEl, apf.MultiSelectServer);

                combinedvalue = oEl;
            }
        }*/
        
        
        //Set caret property
        this.setProperty("caret", e.caret);

        //Set selection length
        if (this.sellength != e.selection.length)
            this.setProperty("sellength", e.selection.length);
        
        //Set selection property
        delete this.selection;
        this.setProperty("selection", e.selection);
        if (!e.selection.length) {
            //Set selected property
            this.setProperty("selected", e.selected);
            
            //Set value property
            if (this.value)
                this.setProperty("value", "");
        }
        else {
            //Set selected property
            this.$chained = true;
            if (!e.force && (!this.dataParent || !this.dataParent.parent.$chained)) {
                var _self = this;
                $setTimeout(function(){
                    
                    if (_self.selected == e.selected) {
                        delete _self.selected;
                        _self.setProperty("selected", combinedvalue || e.selected);
                    }
                    
                    delete _self.$chained;
                }, 10);
            }
            else {
                
                this.setProperty("selected", combinedvalue || e.selected);
                
                delete this.$chained;
            }
            
            //Set value property
            var valueRule = this.$attrBindings["eachvalue"] && "eachvalue" 
                || this.$bindings["value"] && "value" 
                || this.$hasBindRule("caption") && "caption";

            if (valueRule) {
                //@todo this will call the handler again - should be optimized

                this.$lastValue = this.$applyBindRule(valueRule, e.selected)
                //this.$attrBindings["value"] && 
                if (this.$lastValue != 
                  (valueRule != "value" && (this.xmlRoot
                  && this.$applyBindRule("value", this.xmlRoot, null, true)) 
                  || this.value)) {
                    if (valueRule == "eachvalue" || this.xmlRoot != this)
                        this.change(this.$lastValue);
                    else
                        this.setProperty("value", this.$lastValue);
                }
                /*else {
                    this.setProperty("value", this.$lastValue);
                }*/
                delete this.$lastValue;
            }
        }
        
        

        
    }, true);
    
    
    

}).call(apf.MultiSelect.prototype = new apf.MultiselectBinding());



//@todo refactor below
/*
 * @private
 */
/*
apf.MultiSelectServer = {
    objects : {},

    register : function(xmlId, xmlNode, selList, jNode){
        if (!this.$uniqueId)
            this.$uniqueId = apf.all.push(this) - 1;

        this.objects[xmlId] = {
            xml   : xmlNode,
            list  : selList,
            jNode : jNode
        };
    },

    $xmlUpdate : function(action, xmlNode, listenNode, UndoObj){
        if (action != "attribute") return;

        var data = this.objects[xmlNode.getAttribute(apf.xmldb.xmlIdTag)];
        if (!data) return;

        var nodes = xmlNode.attributes;

        for (var j = 0; j < data.list.length; j++) {
            //data[j].setAttribute(UndoObj.name, xmlNode.getAttribute(UndoObj.name));
            apf.xmldb.setAttribute(data.list[j], UndoObj.name,
                xmlNode.getAttribute(UndoObj.name));
        }

        //apf.xmldb.synchronize();
    }
};
*/






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__CHILDVALUE__ = 1 << 27;


apf.ChildValue = function(){
    if (!this.$childProperty)
        this.$childProperty = "value";
    
    this.$regbase = this.$regbase | apf.__CHILDVALUE__;
    
    var f, re = /^[\s\S]*?>(<\?lm)?([\s\S]*?)(?:\?>)?<[^>]*?>$/;
    this.addEventListener("DOMCharacterDataModified", f = function(e){
        if (e && (e.currentTarget == this 
          || e.currentTarget.nodeType == 2 && e.relatedNode == this)
          || this.$amlDestroyed)
            return;

        if (this.getAttribute(this.$childProperty))
            return;
        
        //Get value from xml (could also serialize children, but that is slower
        var m = this.serialize().match(re),
            v = m && m[2] || "";
        if (m && m[1])
            v = "{" + v + "}";

        this.$norecur = true;

        
        if (v.indexOf("{") > -1 || v.indexOf("[") > -1)
            this.$setDynamicProperty(this.$childProperty, v);
        else
        
        if (this[this.$childProperty] != v)
            this.setProperty(this.$childProperty, v);
       
        this.$norecur = false;
    });
    
    //@todo Should be buffered
    this.addEventListener("DOMAttrModified", f);
    this.addEventListener("DOMNodeInserted", f);
    this.addEventListener("DOMNodeRemoved", f);
    
    this.addEventListener("$skinchange", function(e){
       this.$propHandlers[this.$childProperty].call(this, this.caption || "");
    });
    
    this.$init(function(){
       this.addEventListener("prop." + this.$childProperty, function(e){
           if (!this.$norecur && !e.value && !this.getAttributeNode(this.$childProperty))
               f.call(this);
       });
    });

    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        var hasNoProp = typeof this[this.$childProperty] == "undefined";
        
        //this.firstChild.nodeType != 7 && 
        if (hasNoProp
          && !this.getElementsByTagNameNS(this.namespaceURI, "*", true).length 
          && (this.childNodes.length > 1 || this.firstChild 
          && (this.firstChild.nodeType == 1 
          || this.firstChild.nodeValue.trim().length))) {
            //Get value from xml (could also serialize children, but that is slower
            var m = (this.$aml && this.$aml.xml || this.serialize()).match(re),
                v = m && m[2] || "";
            if (m && m[1])
                v = "{" + v + "}";

            
            if (v.indexOf("{") > -1 || v.indexOf("[") > -1)
                this.$setDynamicProperty(this.$childProperty, v);
            else
            
                this.setProperty(this.$childProperty, apf.html_entity_decode(v)); //@todo should be xml entity decode
        }
        else if (hasNoProp)
            this.$propHandlers[this.$childProperty].call(this, "");
    });
};





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__DATAACTION__ = 1 << 25;


/**
 * A [[term.baseclass baseclass]] that adds data action features to this element.
 * @class apf.DataAction
 */
apf.DataAction = function(){
    this.$regbase = this.$regbase | apf.__DATAACTION__;

    // *** Public Methods *** //

    /**
     * Gets the ActionTracker this element communicates with.
     *
     * @return {apf.actiontracker}
     * @see apf.smartbinding
     */
    this.getActionTracker = function(ignoreMe){
        if (!apf.AmlNode)
            return apf.window.$at;

        var pNode = this, tracker = ignoreMe ? null : this.$at;
        if (!tracker && this.dataParent)
            tracker = this.dataParent.parent.$at; //@todo apf3.0 change this to be recursive??

        while (!tracker) {
            if (!pNode.parentNode && !pNode.$parentNode) {
                var model;
                return (model = this.getModel && this.getModel(true)) && model.$at || apf.window.$at;
            }

            tracker = (pNode = pNode.parentNode || pNode.$parentNode).$at;
        }
        return tracker;
    };

    

    this.$actionsLog = {};
    this.$actions    = false;

    /**
     * @event locksuccess   Fires when a lock request succeeds
     * @bubbles 
     * @param {Object} e The standard event object, with the following properties:
     *                   - state ([[Number]]): The return code of the lock request
     *
     */
    /**
     * @event lockfailed    Fires when a lock request failes
     * @bubbles 
     * @param {Object} e The standard event object, with the following properties:
     *                   - state ([[Number]]): The return code of the lock request
     *
     */
    /**
     * @event unlocksuccess Fires when an unlock request succeeds
     * @bubbles 
     * @param {Object} e The standard event object, with the following properties:
     *                   - state ([[Number]]): The return code of the unlock request
     *
     */
    /**
     * @event unlockfailed  Fires when an unlock request fails
     * @bubbles 
     * @param {Object} e The standard event object, with the following properties:
     *                   - state ([[Number]]): The return code of the unlock request
     *
     */
    /*
     *  Starts the specified action, does optional locking and can be offline aware
     *  - or for optimistic locking it will record the timestamp (a setting
     *    <a:appsettings locking="optimistic"/>)
     *  - During offline work, optimistic locks will be handled by taking the
     *    timestamp of going offline
     *  - This method is always optional! The server should not expect locking to exist.
     *
     */
    this.$startAction = function(name, xmlContext, fRollback){
        if (this.disabled || this.liveedit && name != "edit")
            return false;

        var actionRule = this.$actions && this.$actions.getRule(name, xmlContext);
        if (!actionRule && apf.config.autoDisableActions && this.$actions) {
            

            return false;
        }

        var bHasOffline = typeof apf.offline != "undefined";
        

        if (this.dispatchEvent(name + "start", {
            xmlContext: xmlContext
        }) === false)
            return false;

        

        this.$actionsLog[name] = xmlContext;

        return true;
    };

    
    // @todo think about if this is only for rdb
    this.addEventListener("xmlupdate", function(e){
        if (apf.xmldb.disableRDB != 2)
            return;

        for (var name in this.$actionsLog) {
            if (apf.isChildOf(this.$actionsLog[name], e.xmlNode, true)) {
                //this.$stopAction(name, true);
                this.$actionsLog[name].rollback.call(this, this.$actionsLog[name].xmlContext);
            }
        }
    });
    

    this.$stopAction = function(name, isCancelled, curLock){
        delete this.$actionsLog[name];

        
    };

    /*
     * Executes an action using action rules set in the {@link apf.actions actions element}.
     *
     * @param {String}      atAction      The name of the action to be performed by the [[ActionTracker]]. Possible values include:
     *                                 - `"setTextNode"`:   Sets the first text node of an XML element. For more information, see {@link core.xmldb.method.setTextNode}
     *                                 - `"setAttribute"`:  Sets the attribute of an XML element. For more information, see {@link core.xmldb.method.setAttribute}
     *                                 - `"removeAttribute"`:   Removes an attribute from an XML element. For more information, see {@link core.xmldb.method.removeAttribute}
     *                                 - `"setAttributes"`:   Sets multiple attribute on an XML element. The arguments are in the form of `xmlNode, Array`
     *                                 - `"replaceNode"`:   Replaces an XML child with another one. For more information, see {@link core.xmldb.method.replaceNode}
     *                                 - `"addChildNode"`:   Adds a new XML node to a parent node. For more information, see {@link core.xmldb.method.addChildNode}
     *                                 - `"appendChild"`:   Appends an XML node to a parent node. For more information, see {@link core.xmldb.method.appendChild}
     *                                 - `"moveNode"` :  Moves an XML node from one parent to another. For more information, see {@link core.xmldb.method.moveNode}
     *                                 - `"removeNode"`:   Removes a node from it's parent. For more information, see {@link core.xmldb.method.removeNode}
     *                                 - `" removeNodeList"`:    Removes multiple nodes from their parent. For more information, see {@link core.xmldb.method.removeNodeList}
     *                                 - `"setValueByXpath"`:   Sets the nodeValue of an XML node which is selected
     *                                                           by an xpath statement. The arguments are in the form of `xmlNode, xpath, value`
     *                                 - `"multicall"`:          Calls multiple of the above actions. The argument`s are an array
     *                                                           of argument arrays for these actions each with a func`
     *                                                           property, which is the name of the action.
     * @param {Array}       args          the arguments to the function specified
     *                                    in <code>atAction</code>.
     * @param {String}      action        the name of the action rule defined in
     *                                    actions for this element.
     * @param {XMLElement}  xmlNode       the context for the action rules.
     * @param {Boolean}     [noevent]     whether or not to call events.
     * @param {XMLElement}  [contextNode] the context node for action processing
     *                                    (such as RPC calls). Usually the same
     *                                    as <code>xmlNode</code>
     * @return {Boolean} specifies success or failure
     * @see apf.smartbinding
     */
    this.$executeAction = function(atAction, args, action, xmlNode, noevent, contextNode, multiple){
        

        

        //Get Rules from Array
        var rule = this.$actions && this.$actions.getRule(action, xmlNode);
        if (!rule && this.$actions && apf.config.autoDisableActions
          && "action|change".indexOf(action) == -1) {
            apf.console.warn("Could not execute action '" + action + "'. \
              No valid action rule was found and auto-disable-actions is enabled");

            return false;
        }

        

        var newMultiple;
        if (multiple) {
            newMultiple = [];
            for (var k = multiple.length - 1; k >= 0; k--) {
                newMultiple.unshift({
                    xmlActionNode : rule, // && rule[4],
                    amlNode       : this,
                    selNode       : multiple[k],
                    xmlNode       : multiple[k]
                })
            }
        }

        //@todo apf3.0 Shouldn't the contextNode be made by the match
        var ev = new apf.AmlEvent("before" + action.toLowerCase(), {
            action        : atAction,
            args          : args,
            xmlActionNode : rule,
            amlNode       : this,
            selNode       : contextNode,
            multiple      : newMultiple || false
            
        });

        //Call Event and cancel if it returns false
        if (!noevent) {
            //Allow the action and arguments to be changed by the event
            if (this.dispatchEvent(ev.name, null, ev) === false)
                return false;

            delete ev.currentTarget;
        }

        //Call ActionTracker and return ID of Action in Tracker
        var at      = this.getActionTracker();
        if (!at)// This only happens at destruction of apf
            return UndoObj;

        var UndoObj = at.execute(ev);
        ev.xmlNode = UndoObj.xmlNode;
        ev.undoObj = UndoObj;

        //Call After Event
        if (!noevent) { //@todo noevent is not implemented for before.. ???
            ev.name         = "after" + action.toLowerCase();
            ev.cancelBubble = false;
            delete ev.returnValue;
            delete ev.currentTarget;
            this.dispatchEvent(ev.name, null, ev);
        }

        return UndoObj;
    };

    /*
     * Executes an action based on the set name and the new value
     * @param {String}      atName   the name of the action rule defined in actions for this element.
     * @param {String}      setName  the name of the binding rule defined in bindings for this element.
     * @param {XMLElement}  xmlNode  the xml element to which the rules are applied
     * @param {String}      value    the new value of the node
     */
    this.$executeSingleValue = function(atName, setName, xmlNode, value, getArgList){
        var xpath, args, rule = this.$getBindRule(setName, xmlNode);

        //recompile bindrule to create nodes
        if (!rule) {
            
                return false;
        }

        var compiled;
        ["valuematch", "match", "value"].each(function(type){
            if (!rule[type] || compiled)
                return;

            compiled = rule["c" + type]; //cvaluematch || (rule.value ? rule.cvalue : rule.cmatch);
            if (!compiled)
                compiled = rule.compile(type);

            if (compiled.type != 3)
                compiled = null;
        });

        

        var atAction, model, node,
            sel        = compiled.xpaths, //get first xpath
            shouldLoad = false;

        if (sel[0] == "#" || sel[1] == "#") {
            var m = (rule.cvalue3 || (rule.cvalue3 = apf.lm.compile(rule.value, {
                xpathmode: 5
            })))(xmlNode);

            model = m.model && m.model.$isModel && m.model;
            if (model) {
                node  = model.queryNode(m.xpath);
                xmlNode = model.data;
            }
            else if (m.model){
                model = apf.xmldb.findModel(m.model);
                node  = m.model.selectSingleNode(m.xpath);
                xmlNode = m.model;
            }
            else {

            }

            sel[1] = m.xpath;
        }
        else {
            
            model = sel[0] && apf.nameserver.get("model", sel[0]) || this.$model,
            node  = model
                ? model.queryNode(sel[1])
                : (xmlNode || this.xmlRoot).selectSingleNode(sel[1]);
            if (model && !xmlNode)
                xmlNode = model.data; //@experimental, after changing this, please run test/test_rename_edge.html
            
        }

        if (node) {
            if (apf.queryValue(node) == value) return; // Do nothing if value is unchanged

            atAction = (node.nodeType == 1 || node.nodeType == 3
                || node.nodeType == 4) ? "setTextNode" : "setAttribute";
            args = (node.nodeType == 1)
                ? [node, value]
                : (node.nodeType == 3 || node.nodeType == 4
                    ? [node.parentNode, value]
                    : [node.ownerElement || node.selectSingleNode(".."), node.nodeName, value]);
        }
        else {
            atAction = "setValueByXpath";
            xpath    = sel[1];
            
            if (!this.$createModel || this.getModel() && !this.getModel().$createModel) {
                throw new Error("Model data does not exist, and I am not "
                    + "allowed to create the element for xpath '" 
                    + xpath + "' and element " + this.serialize(true));
            }

            if (!xmlNode) {
                //Assuming this component is connnected to a model
                if (!model)
                    model = this.getModel();
                if (model) {
                    if (!model.data)
                        model.load("<data />");

                    xpath   = (model.getXpathByAmlNode(this) || ".")
                        + (xpath && xpath != "." ? "/" + xpath : "");
                    xmlNode = model.data;
                }
                else {
                    if (!this.dataParent)
                        return false;

                    xmlNode = this.dataParent.parent.selected || this.dataParent.parent.xmlRoot;
                    if (!xmlNode)
                        return false;

                    xpath = (this.dataParent.xpath || ".")
                        + (xpath && xpath != "." ? "/" + xpath : "");
                    shouldLoad = true;
                }
            }

            args = [xmlNode, value, xpath];
        }

        if (getArgList) {
            return {
                action : atAction,
                args   : args
            };
        }

        //Use Action Tracker
        var result = this.$executeAction(atAction, args, atName, xmlNode);

        if (shouldLoad)
            this.load(xmlNode.selectSingleNode(xpath));

        return result;
    };

    /**
     * Changes the value of this element.
     * @action
     * @param  {String} [string] The new value of this element.
     * 
     */
    this.change = function(value, force){ // @todo apf3.0 maybe not for multiselect?? - why is clearError handling not in setProperty for value
        
        if (this.errBox && this.errBox.visible && this.isValid && this.isValid())
            this.clearError();
        

        
        //Not databound
        if (!this.xmlRoot && !this.$createModel || !(this.$mainBind == "value"
          && this.hasFeature(apf.__MULTISELECT__)
            ? this.$attrBindings["value"]
            : this.$hasBindRule(this.$mainBind))) {
        
            if (!force && value === this.value
              || this.dispatchEvent("beforechange", {value : value}) === false)
                return false;

            //@todo in theory one could support actions
            //@todo disabled below, because it gives unexpected behaviour when
            //form elements are used for layout and other UI alterations
            /*this.getActionTracker().execute({
                action        : "setProperty",
                args          : [this, "value", value, false, true],
                amlNode       : this
            });*/
            this.setProperty("value", value);

            return this.dispatchEvent("afterchange", {value : value});
        
        }

        var valueRule = this.$attrBindings["eachvalue"] && "eachvalue"
            || this.$bindings["value"] && "value"
            || this.$hasBindRule("caption") && "caption";

        if (value === (valueRule != "value" && (this.xmlRoot
          && this.$applyBindRule("value", this.xmlRoot, null, true))
          || this.value))
            return false;

        this.$executeSingleValue("change", this.$mainBind, this.xmlRoot, value);
        
    };

    this.$booleanProperties["render-root"] = true;
    this.$supportedProperties.push("create-model", "actions");

    /**
     * @attribute {Boolean} create-model Sets or gets whether the model this element connects
     * to is extended when the data pointed to does not exist. Defaults to true.
     * 
     * #### Example
     *
     * In this example, a model is extended when the user enters information in
     * the form elements. Because no model is specified for the form elements,
     * the first available model is chosen. At the start, it doesn't have any
     * data; this changes when (for instance) the name is filled in. A root node
     * is created, and under that a 'name' element with a textnode containing
     * the entered text.
     * 
     * ```xml
     *  <a:bar>
     *      <a:label>Name</a:label>
     *      <a:textbox value="[name]" required="true" />
     *
     *      <a:label>Address</a:label>
     *      <a:textarea value="[address]" />
     *
     *      <a:label>Country</a:label>
     *      <a:dropdown
     *        value   = "[mdlForm::country]"
     *        model   = "countries.xml"
     *        each    = "[country]"
     *        caption = "[@name]" />
     *      <a:button action="submit">Submit</a:button>
     *  </a:bar>
     * ```
     */
    this.$propHandlers["create-model"] = function(value){
        this.$createModel = value;
    };

    this.addEventListener("DOMNodeInsertedIntoDocument", function(e){
        if (typeof this["create-model"] == "undefined"
          && !this.$setInheritedAttribute("create-model")) {
            this.$createModel = true;
        }
    });
};

apf.config.$inheritProperties["create-model"] = 1;





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__CACHE__ = 1 << 2;



/**
 * All elements inheriting from this {@link term.baseclass baseclass} have caching features. It takes care of
 * storing, retrieving, and updating rendered data (in HTML form)
 * to overcome the waiting time while rendering the contents every time the
 * data is loaded.
 *
 * @class apf.Cache
 * @baseclass
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.4
 */
apf.Cache = function(){
    /* ********************************************************************
                                        PROPERTIES
    *********************************************************************/
    this.cache               = {};
    this.$subTreeCacheContext = null;

    this.caching  = true;
    this.$regbase = this.$regbase | apf.__CACHE__;

    /* ********************************************************************
                                        PUBLIC METHODS
    *********************************************************************/

    this.addEventListener("$load", function(e){
        if (!this.caching || e.forceNoCache)
            return;

        // retrieve the cacheId
        if (!this.cacheId) {
            this.cacheId = this.$generateCacheId && this.$generateCacheId(e.xmlNode) 
                || e.xmlNode.getAttribute(apf.xmldb.xmlIdTag) 
                || apf.xmldb.nodeConnect(apf.xmldb.getXmlDocId(e.xmlNode), e.xmlNode);//e.xmlNode
        }

        // Retrieve cached version of document if available
        var fromCache = getCache.call(this, this.cacheId, e.xmlNode);
        if (fromCache) {
            if (fromCache == -1 || !this.getTraverseNodes)
                return (e.returnValue = false);

            var nodes = this.getTraverseNodes();

            //Information needs to be passed to the followers... even when cached...
            if (nodes.length) {
                if (this["default"])
                    this.select(this["default"]);
                else if (this.autoselect)
                    this.select(nodes[0], null, null, null, true);
            }
            else if (this.clearSelection)
                this.clearSelection(); //@todo apf3.0 was setProperty("selected", null

            if (!nodes.length) {
                // Remove message notifying user the control is without data
                this.$removeClearMessage();
                this.$setClearMessage(this["empty-message"], "empty");
            }
                
            
            //@todo move this to getCache??
            if (nodes.length != this.length)
                this.setProperty("length", nodes.length);
            

            return false;
        }
    });
    
    this.addEventListener("$clear", function(){
        if (!this.caching)
            return;

        /*
            Check if we borrowed an HTMLElement
            We should return it where it came from

            note: There is a potential that we can't find the exact location
            to put it back. We should then look at it's position in the xml.
            (but since I'm lazy it's not doing this right now)
            There might also be problems when removing the xmlroot
        */
        if (this.hasFeature(apf.__MULTISELECT__)
          && this.$subTreeCacheContext && this.$subTreeCacheContext.oHtml) {
            if (this.renderRoot) {
                this.$subTreeCacheContext.parentNode.insertBefore(
                    this.$subTreeCacheContext.oHtml, this.$subTreeCacheContext.beforeNode);
            }
            else {
                var container = this.$subTreeCacheContext.container || this.$container;
                while (container.childNodes.length)
                    this.$subTreeCacheContext.oHtml.appendChild(container.childNodes[0]);
            }

            this.documentId = this.xmlRoot = this.cacheId = this.$subTreeCacheContext = null;
        }
        else {
            /* If the current item was loaded whilst offline, we won't cache
             * anything
             */
            if (this.$loadedWhenOffline) {
                this.$loadedWhenOffline = false;
            }
            else {
                // Here we cache the current part
                var fragment = this.$getCurrentFragment();
                if (!fragment) return;//this.$setClearMessage(this["empty-message"]);

                fragment.documentId = this.documentId;
                fragment.xmlRoot    = this.xmlRoot;
                
                if (this.cacheId || this.xmlRoot)
                    setCache.call(this, this.cacheId ||
                        this.xmlRoot.getAttribute(apf.xmldb.xmlIdTag) || "doc"
                        + this.xmlRoot.getAttribute(apf.xmldb.xmlDocTag), fragment);
            }
        }
    });

    /*
     * Checks the cache for a cached item by ID. If the ID is found, the
     * representation is loaded from cache and set active.
     *
     * @param  {String} id  The id of the cache element which is looked up.
     * @param  {Object} xmlNode
     * @return {Boolean} If `true`, the cache element was found and set active
     * @see    baseclass.databinding.method.load
     * @private
     */
    function getCache(id, xmlNode){
        /*
            Let's check if the requested source is actually
            a sub tree of an already rendered part
        */
        
        if (xmlNode && this.hasFeature(apf.__MULTISELECT__) && this.$isTreeArch) {
            var cacheItem,
                htmlId = xmlNode.getAttribute(apf.xmldb.xmlIdTag) + "|" + this.$uniqueId,
                node   = this.$pHtmlDoc.getElementById(htmlId);
            if (node) 
                cacheItem = id ? false : this.$container; //@todo what is the purpose of this statement?
            else {
                for (var prop in this.cache) {
                    if (this.cache[prop] && this.cache[prop].nodeType) {
                        node = this.cache[prop].getElementById(htmlId);
                        if (node) {
                            cacheItem = id ? prop : this.cache[prop]; //@todo what is the purpose of this statement?
                            break;
                        }
                    }
                }
            }
            
            if (cacheItem && !this.cache[id]) {
                /*
                    Ok so it is, let's borrow it for a while
                    We can't clone it, because the updates will
                    get ambiguous, so we have to put it back later
                */
                var oHtml = this.$findHtmlNode(
                    xmlNode.getAttribute(apf.xmldb.xmlIdTag) + "|" + this.$uniqueId);
                this.$subTreeCacheContext = {
                    oHtml      : oHtml,
                    parentNode : oHtml.parentNode,
                    beforeNode : oHtml.nextSibling,
                    cacheItem  : cacheItem
                };

                this.documentId = apf.xmldb.getXmlDocId(xmlNode);
                this.cacheId    = id;
                this.xmlRoot    = xmlNode;

                //Load html
                if (this.renderRoot)
                    this.$container.appendChild(oHtml);
                else {
                    while (oHtml.childNodes.length)
                        this.$container.appendChild(oHtml.childNodes[0]);
                }

                return true;
            }
        }
        

        //Checking Cache...
        if (!this.cache[id]) return false;

        //Get Fragment and clear Cache Item
        var fragment    = this.cache[id];

        this.documentId = fragment.documentId;
        this.cacheId    = id;
        this.xmlRoot    = xmlNode;//fragment.xmlRoot;
        
        
        this.setProperty("root", this.xmlRoot);
        

        this.clearCacheItem(id);

        this.$setCurrentFragment(fragment);

        return true;
    };

    /*
     * Sets cache element and its ID.
     *
     * @param {String}           id        The id of the cache element to be stored.
     * @param {DocumentFragment} fragment  The data to be stored.
     * @private
     */
    function setCache(id, fragment){
        if (!this.caching) return;

        this.cache[id] = fragment;
    };

    /*
     * Finds HTML presentation node in cache by ID.
     *
     * @param  {String} id  The id of the HTMLElement which is looked up.
     * @return {HTMLElement} The HTMLElement found. When no element is found, `null` is returned.
     */
    this.$findHtmlNode = function(id){
        var node = this.$pHtmlDoc.getElementById(id);
        if (node) return node;

        for (var prop in this.cache) {
            if (this.cache[prop] && this.cache[prop].nodeType) {
                node = this.cache[prop].getElementById(id);
                if (node) return node;
            }
        }

        return null;
    };

    /**
     * Removes an item from the cache.
     *
     * @param {String}  id       The id of the HTMLElement which is looked up.
     * @param {Boolean} [remove] Specifies whether to destroy the Fragment.
     * @see baseclass.databinding.method.clear
     * @private
     */
    this.clearCacheItem = function(id, remove){
        this.cache[id].documentId = 
        this.cache[id].cacheId    =
        this.cache[id].xmlRoot    = null;

        if (remove)
            apf.destroyHtmlNode(this.cache[id]);

        this.cache[id] = null;
    };

    /*
     * Removes all items from the cache
     *
     * @see baseclass.databinding.method.clearCacheItem
     * @private
     */
    this.clearAllCache = function(){
        for (var prop in this.cache) {
            if (this.cache[prop])
                this.clearCacheItem(prop, true);
        }
    };

    /**
     * Gets the cache item by its id
     *
     * @param {String} id  The id of the HTMLElement which is looked up.
     * @see baseclass.databinding.method.clearCacheItem
     * @private
     */
    this.getCacheItem = function(id){
        return this.cache[id];
    };

    /*
     * Checks whether a cache item exists by the specified id
     *
     * @param {String} id  the id of the cache item to check.
     * @see baseclass.databinding.method.clearCacheItem
     * @private
     */
    this.$isCached = function(id){
        return this.cache[id] || this.cacheId == id ? true : false;
    };
    
    if (!this.$getCurrentFragment) {
        this.$getCurrentFragment = function(){
            var fragment = this.$container.ownerDocument.createDocumentFragment();
    
            while (this.$container.childNodes.length) {
                fragment.appendChild(this.$container.childNodes[0]);
            }
    
            return fragment;
        };
    
        this.$setCurrentFragment = function(fragment){
            this.$container.appendChild(fragment);
    
            if (!apf.window.hasFocus(this) && this.blur)
                this.blur();
        };
    }
    
    /**
     * @attribute {Boolean} caching Sets or gets whether caching is enabled for this element.
     */
    this.$booleanProperties["caching"] = true;
    this.$supportedProperties.push("caching");

    this.addEventListener("DOMNodeRemovedFromDocument", function(e){
        //Remove all cached Items
        this.clearAllCache();
    });
};

apf.GuiElement.propHandlers["caching"] = function(value) {
    if (!apf.isTrue(value)) return;
    
    if (!this.hasFeature(apf.__CACHE__))
        this.implement(apf.Cache);
};






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__RENAME__ = 1 << 10;



/**
 * All elements inheriting from this {@link term.baseclass baseclass} have the rename features. Rename is triggered by
 * pressing F2 on an item or by clicking once on an already selected item. This
 * will show an input element in place where the user can change the name of the
 * item to a new one. When the caption is changed the {@link term.datanode data node} is
 * changed accordingly.
 * 
 * #### Example
 *
 * This example shows a list containing products. Only products that have the
 * editable attribute set to 1 can be renamed by the user.
 * 
 * ```xml
 *  <a:model id="mdlTest">
 *      <data>
 *          <product name="TV" />
 *          <product name="LCD" editable="1" />
 *      </data>
 *  </a:model>
 *  <a:list id="list" model="mdlTest" width="200">
 *      <a:each match="[product]">
 *          <a:caption match="[@name]" />
 *      </a:each>
 *      <a:actions>
 *          <a:rename
 *            match = "[product[@editable='1']]"
 *            set   = "rename.php" />
 *      </a:actions>
 *  </a:list>
 *       
 *  <a:button
 *    caption = "Rename"
 *    onclick = "list.startRename()" />
 * ```
 *
 * @class apf.Rename
 * @baseclass
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.5
 */
/**
 * @event stoprename Fires when a rename action is cancelled.
 *
 */
apf.Rename = function(){
    this.$regbase       = this.$regbase|apf.__RENAME__;

    this.canrename      = false;
    this.$renameSubject =
    this.renameTimer    =
    this.lastCursor     = null;
    
    /**
     * @attribute  {Boolean}  rename  Sets or gets whether the user can start renaming rendered nodes in this element.
     */
    this.$booleanProperties["canrename"]  = true;
    this.$booleanProperties["autorename"] = true;
    this.$supportedProperties.push("canrename", "autorename");

    

    /**
     * Changes the data presented as the caption of a specified {@link term.datanode data node}.
     * 
     * If none are specified, the indicated node is used.
     *
     * @action
     * @param  {XMLElement} xmlNode The element to change the caption of.
     * @param  {String}     value   The value to set as the caption of the {@link term.datanode data node}.
     */
    this.rename = function(xmlNode, value){
        if (!xmlNode)
            xmlNode = this.caret || this.selected;

        if (!xmlNode) return;

        return this.$executeSingleValue("rename", "caption", xmlNode, value);
    };

    // @todo Doc params
    /**
     * Starts the rename process with a delay, allowing for cancellation when
     * necesary. Cancellation is necesary for instance, when double click was
     * intended or a dragdrop operation.
     *
     */
    this.startDelayedRename = function(e, time, userAction){
        clearTimeout(this.renameTimer);
        
        if (e && (e.button == 2 || e.ctrlKey || e.shiftKey) 
          || userAction && this.disabled)
            return;

        this.renameTimer = $setTimeout('apf.lookup('
            + this.$uniqueId + ').startRename()', time || 400);
    };
    
    // @todo Doc params
    /**
     * Starts the rename process by displaying an input box at the position
     * of the item that can be renamed by the user.
     *
     */
    this.startRename = function(force, startEmpty, userAction){
        if (!force && (this.renaming || !this.canrename 
          || !this.$startAction("rename", this.caret 
          || this.selected, this.stopRename))
          || userAction && this.disabled)
            return false;

        if (!this.hasFocus())
            this.focus(null, null, true);

        clearTimeout(this.renameTimer);

        var elCaption = this.$getCaptionElement
            ? this.$getCaptionElement()
            : this.$caret || this.$selected;

        if (!elCaption) 
            return this.stopRename();
        
        this.renaming       = true;
        this.$renameSubject = this.caret || this.selected;

        var wdt = elCaption.offsetWidth;
        this.lastCursor = elCaption.style.cursor;
        elCaption.style.cursor = "text";
        elCaption.parentNode.replaceChild(this.$txt, elCaption);
        elCaption.host = this;

        if (apf.isTrue(this.$getOption("main", "scalerename"))) {
            var diff = apf.getWidthDiff(this.$txt);
            this.$txt.style.width = (wdt - diff - 3) + "px";
        }

        this.$replacedNode = elCaption;
        var xmlNode       = this.$getCaptionXml
            ? this.$getCaptionXml(this.$renameSubject)
            : this.$getDataNode("caption", this.$renameSubject);

        //xmlNode.nodeType >= 2 && xmlNode.nodeType <= 4
        var value =  startEmpty || !xmlNode
            ? ""
            : (xmlNode.nodeType != 1
                ? unescape(xmlNode.nodeValue) //decodeURI( - throws an error when using % in a non expected way
                : (apf.isOnlyChild(xmlNode.firstChild, [3,4])
                    ? apf.queryValue(xmlNode)
                    : this.$applyBindRule("caption", this.$renameSubject))) || "";

        if (apf.hasContentEditable) {
            if (this.$multiLineRename)
                this.$txt.innerHTML = apf.htmlCleaner.prepare(value.trim()
                    .replace(/</g, "&lt;")
                    .replace(/>/g, "&gt;")
                    .replace(/\n/g, "<br />"));
            else
                this.$txt.innerHTML = value.replace(/</g, "&lt;")
                  || apf.hasContentEditableContainerBug && "<br>" || "";
        }
        else 
            this.$txt.value = value;

        this.$txt.unselectable = "Off";
        this.$txt.host         = this;

        //this.$txt.focus();
        var txt = this.$txt;
        var f = function(){
            try {
                txt.focus();
                txt.select();
            }
            catch(e) {}
        };
        if (apf.isIE) f() 
        else $setTimeout(f);
    };

    // @todo Doc params
    /**
     * Stops the renaming process and changes the data according to the set value.
     * Cancel the renaming process without changing data.
     *
     */
    this.stopRename = function(contextXml, success){
        clearTimeout(this.renameTimer);

        if (!this.renaming || contextXml && contextXml != this.$renameSubject
          || !this.$replacedNode)
            return false;
        
        this.renaming = false;

        if (this.$txt.parentNode && this.$txt.parentNode.nodeType == 1) {
            if (apf.isIE8 || apf.isIE7Emulate)
                this.$txt.blur();
            
            this.$txt.parentNode.replaceChild(this.$replacedNode, this.$txt);
        }

        if (this.$replacedNode) {
            this.$replacedNode.style.cursor = this.lastCursor || "";
            this.$replacedNode.host = null;
        }
        
        //apf.hasContentEditable ??
        if (this.$multiLineRename) {
            var value = apf.html_entity_decode(
                apf.htmlCleaner.parse(this.$txt.innerHTML, true)
                            .replace(/<br \/>/g, "")
                            .replace(/<\/?p>/g, ""));
        }
        else {
            var value = this.$txt[apf.hasContentEditable ? "innerText" : "value"]
                            .replace(/<.*?nobr>/gi, "").replace(/\n$/, ""); //last replace is for chrome
        }

        if (!success || (this.$validateRename && !this.$validateRename(value))) {
            this.dispatchEvent("stoprename");
            this.$stopAction("rename");
        }
        else {
            //this.$selected.innerHTML = this.$txt.innerHTML;
            if (this.rename(this.$renameSubject, value) !== false) {
                if (this.$replacedNode)
                    this.$replacedNode.innerHTML = value.replace(/</g, "&lt;").replace(/\r?\n/g, "<br />");
            }
        }

        if (!this.renaming) {
            this.$renameSubject    = null;
            this.$replacedNode     = null;
            this.$txt.style.width = "";
        }

        return true;
    };

    
    this.addEventListener("keydown", function(e){
        var key = e.keyCode;

        if (this.renaming) {
            if (key == 27 || this.$multiLineRename && e.ctrlKey && key == 13 
              || !this.$multiLineRename && key == 13) {
                this.stopRename(null, key == 13 && !this.$autocomplete);
                e.cancelBubble = true;
                return false;
            }
            else if (apf.hasContentEditableContainerBug && key == 8
              && this.$txt.innerHTML == "<br>") {
                e.preventDefault();
            }

            return;
        }

        //F2
        if (key == 113) {
            if (this.$tempsel)
                this.$selectTemp();

            if (this.caret != this.selected) {
                if (this.multiselect || this.isSelected(this.caret)) {
                    this.selected  = this.caret;
                    this.$selected = this.$caret;
                }
                else
                    this.select(this.caret, true);
            }

            this.startRename();

            return false;
        }
    }, true);
    

    this.addEventListener("DOMNodeRemovedFromDocument", function(e){
        this.$txt.refCount--;

        if (!this.$txt.refCount) {
            this.$txt.host        =
            this.$txt.onmouseover =
            this.$txt.onmousedown =
            this.$txt.select      =
            this.$txt.onfocus     =
            this.$txt.onblur      = null;
        }
        this.$txt = null;
    });
    
    this.$init(apf.Rename.initEditableArea);
};

apf.Rename.initEditableArea = function(){
    if (!(this.$txt = document.getElementById("txt_rename"))) {
        if (apf.hasContentEditable) {
            this.$txt = document.createElement("DIV");
            this.$txt.contentEditable = true;
            if (apf.isIE6)
                this.$txt.style.width = "1px";
            //this.$txt.canHaveHTML = false;
        }
        else {
            this.$txt              = document.createElement("input");
            this.$txt.id           = "txt_rename";
            this.$txt.autocomplete = false;
        }
    
        
    
        this.$txt.refCount         = 0;
        this.$txt.id               = "txt_rename";
        //this.$txt.style.whiteSpace = "nowrap";
        apf.importCssString("#txt_rename{white-space:nowrap}");
        this.$txt.onselectstart    = function(e){
            (e || event).cancelBubble = true;
        };
    
        this.$txt.onmouseover = 
        this.$txt.onmouseout  = 
        this.$txt.oncontextmenu =
        //this.$txt.onkeydown   = 
        this.$txt.onmouseup   = 
        this.$txt.ondblclick  =
        this.$txt.onmousedown = function(e){ 
            apf.stopPropagation(e || event)
        };
    
        this.$txt.onkeyup = function(e){
            //(e || event).cancelBubble = true;
            
            if (!this.host.$autocomplete)
                return;
    
            this.host.$lookup(this[apf.hasContentEditable ? "innerHTML" : "value"]);
        }
    
        var sel;
        this.$txt.select = function(){
            if (!apf.hasMsRangeObject) {
                if (window.getSelection && document.createRange) {
                    var sel = window.getSelection();
                    sel.removeAllRanges();
                    var r = document.createRange();
                    r.setStart(this.firstChild, 0);
                    var lastIndex = this.firstChild.nodeValue.lastIndexOf(".");
                    r.setEnd(this.firstChild, lastIndex > -1 ? lastIndex : this.firstChild.nodeValue.length);
                    sel.addRange(r)
                }
                else {
                    (sel || (sel = new apf.selection())).selectNode(this);
                }
                return;
            }
    
            var r = document.selection.createRange();
            //r.moveEnd("character", this.$ext.innerText.length);
            try {
                r.moveToElementText(this);
    
                if (apf.isFalse(this.host.$getOption("main", "selectrename"))
                  || typeof this.host.$renameStartCollapse != "undefined") //@todo please deprecate renameStartCollapse
                    r.collapse(this.host.$renameStartCollapse);
            } catch(e) {} //BUG!!!!
    
            r.select();
        };
    
        
    
        this.$txt.onblur = function(){
            //if (apf.isGecko)
                //return; //bug in firefox calling onblur too much
            //if (apf.isChrome && !arguments.callee.caller)
                //return;

            
    
            if (this.host.$autocomplete)
                return;
    
            this.host.stopRename(null, true);
        };
    }
    
    this.$txt.refCount++;
}







/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

apf.__ALIGNMENT__ = 1 << 29;





/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * Baseclass of an element that has one or two states and can be clicked on to
 * trigger an action (_i.e._ {@link apf.button} or {@link apf.checkbox}).
 *
 * @class apf.BaseButton
 * @baseclass
 * @author      Abe Ginner
 * @version     %I%, %G%
 * @since       0.8
 * @inherits apf.StandardBinding
 */
/**
 * @event click Fires when the user presses a mouse button while over this element...and then lets the mousebutton go. 
 */
apf.BaseButton = function(){
    this.$init(true);
};

(function() {
    
    this.implement(apf.ChildValue);
    
    
    this.$refKeyDown   =        // Number of keys pressed.
    this.$refMouseDown = 0;     // Mouse button down?
    this.$mouseOver    =        // Mouse hovering over the button?
    this.$mouseLeft    = false; // Has the mouse left the control since pressing the button.

    // *** Properties and Attributes *** //

    /**
     * @attribute {String} background Sets or gets a multistate background. The arguments
     * are seperated by pipes (`'|'`) and are in the order of:'imagefilename|mapdirection|nrofstates|imagesize'
     *
     * - The `mapdirection` argument may have the value of `'vertical'` or `'horizontal'`.
     * - The `nrofstates` argument specifies the number of states the iconfile contains:
     *     - 1: normal
     *     - 2: normal, hover
     *     - 3: normal, hover, down
     *     - 4: normal, hover, down, disabled
     * - The `imagesize` argument specifies how high or wide each icon is inside the
     * map, depending on the `mapdirection` argument.
     * {: #multiStateDoc}
     * 
     * #### Example
     * 
     * Here's a three state picture where each state is 16px high, vertically spaced:
     * 
     * ```xml
     * background="threestates.gif|vertical|3|16"
     * ```
     */
    this.$propHandlers["background"] = function(value){
        var oNode = this.$getLayoutNode("main", "background", this.$ext);
        
        if (!oNode) return;
        

        if (value) {
            var b = value.split("|");
            this.$background = b.concat(["vertical", 2, 16].slice(b.length - 1));

            oNode.style.backgroundImage  = "url(" + this.mediaPath + b[0] + ")";
            oNode.style.backgroundRepeat = "no-repeat";
        }
        else {
            oNode.style.backgroundImage  = "";
            oNode.style.backgroundRepeat = "";
            this.$background = null;
        }
    };

    // *** Keyboard Support *** //

    
    this.addEventListener("keydown", function(e){
        var key      = e.keyCode;
        //var ctrlKey  = e.ctrlKey;  << UNUSED
        //var shiftKey = e.shiftKey; << UNUSED

        switch (key) {
            case 13:
                if (this.localName != "checkbox")
                    this.$ext.onmouseup(e.htmlEvent, true);
                break;
            case 32:
                if (!e.htmlEvent.repeat) { // Only when first pressed, not on autorepeat.
                    this.$refKeyDown++;
                    this.$updateState(e.htmlEvent);
                }
                return false;
        }
    }, true);

    this.addEventListener("keyup", function(e){
        var key = e.keyCode;

        switch (key) {
            case 32:
                this.$refKeyDown--;

                if (this.$refKeyDown < 0) {
                    this.$refKeyDown = 0;
                    return false;
                }

                if (this.$refKeyDown + this.$refMouseDown == 0 && !this.disabled)
                    this.$ext.onmouseup(e, true);

                this.$updateState(e);
                return false;
        }
    }, true);
    

    // *** Private state handling methods *** //

    this.states = {
        "Out"   : 1,
        "Over"  : 2,
        "Down"  : 3
    };

    this.$updateState = function(e, strEvent) {
        if (e.reset) { //this.disabled || 
            this.$refKeyDown   = 0;
            this.$refMouseDown = 0;
            this.$mouseOver    = false;
            return false;
        }

        if (this.$refKeyDown > 0
          || (this.$refMouseDown > 0 && (this.$mouseOver || (!apf.isIE && this.$ext === e.currentTarget)))
          || (this.isBoolean && this.value)) {
            this.$setState("Down", e, strEvent);
        }
        else if (this.$mouseOver) {
            this.$setState("Over", e, strEvent);
        }
        else
            this.$setState("Out", e, strEvent);
    };

    this.$setupEvents = function() {
        if (this.editable)
            return;
        
        var _self = this;

        this.$ext.onmousedown = function(e) {
            e = e || window.event;

            if (_self.$notfromext && (e.srcElement || e.target) == this)
                return;

            _self.$refMouseDown = 1;
            _self.$mouseLeft    = false;
            
            if (_self.disabled)
                return;

            if (!apf.isIE) { // && (apf.isGecko || !_self.submenu) Causes a focus problem for menus
                if (_self.value)
                    apf.stopEvent(e);
                else
                    apf.cancelBubble(e);
            }
            
            _self.$updateState(e, "mousedown");
        };
        
        this.$ext.onmouseup = function(e, force) {
            e = e || window.event;
            //if (e)  e.cancelBubble = true;
            if (_self.disabled || !force && ((!_self.$mouseOver && (!apf.isIE && this !== e.currentTarget)) || !_self.$refMouseDown))
                return;

            _self.$refMouseDown = 0;
            _self.$updateState(e, "mouseup");

            // If this is coming from a mouse click, we shouldn't have left the button.
            if (_self.disabled || (e && e.type == "click" && _self.$mouseLeft == true))
                return false;

            // If there are still buttons down, this is not a real click.
            if (_self.$refMouseDown + _self.$refKeyDown)
                return false;

            if (_self.$clickHandler && _self.$clickHandler())
                _self.$updateState (e || event, "click");
            else
                _self.dispatchEvent("click", {htmlEvent : e});

            return false;
        };

        this.$ext.onmousemove = function(e) {
            if ((!_self.$mouseOver || _self.$mouseOver == 2)) {
                e = e || window.event;

                if (_self.$notfromext && (e.srcElement || e.target) == this)
                    return;

                _self.$mouseOver = true;
                
                if (!_self.disabled)
                    _self.$updateState(e, "mouseover");
            }
        };

        this.$ext.onmouseout = function(e) {
            e = e || window.event;

            //Check if the mouse out is meant for us
            var tEl = e.explicitOriginalTarget || e.toElement;
            if (apf.isChildOf(this, tEl)) //this == tEl ||
                return;

            _self.$mouseOver    = false;
            _self.$refMouseDown = 0;
            _self.$mouseLeft    = true;
            
            if (!_self.disabled)
                _self.$updateState(e, "mouseout");
        };

        

        if (apf.hasClickFastBug)
            this.$ext.ondblclick = this.$ext.onmouseup;
    };

    this.$doBgSwitch = function(nr){
        if (this.background && (this.$background[2] >= nr || nr == 4)) {
            if (nr == 4)
                nr = this.$background[2] + 1;

            var strBG = this.$background[1] == "vertical"
                ? "0 -" + (parseInt(this.$background[3]) * (nr - 1)) + "px"
                : "-"   + (parseInt(this.$background[3]) * (nr - 1)) + "px 0";

            this.$getLayoutNode("main", "background",
                this.$ext).style.backgroundPosition = strBG;
        }
    };

    // *** Focus Handling *** //

    this.$focus = function(){
        if (!this.$ext)
            return;

        this.$setStyleClass(this.$ext, this.$baseCSSname + "Focus");
    };

    this.$blur = function(e){
        if (!this.$ext)
            return; //FIREFOX BUG!

        this.$setStyleClass(this.$ext, "", [this.$baseCSSname + "Focus"]);
        /*this.$refKeyDown   = 0;
        this.$refMouseDown = 0;
        this.$mouseLeft    = true;*/

        
        /*if (this.submenu) {
            if (this.value) {
                this.$setState("Down", {}, "mousedown");
                this.$hideMenu();
            }
        }*/
        

        if (e)
            this.$updateState({});//, "onblur"
    };
    
    this.addEventListener("prop.disabled", function(e){
        this.$refKeyDown   =   
        this.$refMouseDown = 0;
        //this.$mouseOver    =   
        //this.$mouseLeft    = false;
    });

    /*** Clearing potential memory leaks ****/

    this.$destroy = function(skinChange){
        if (!skinChange && this.$ext) {
            this.$ext.onmousedown = this.$ext.onmouseup = this.$ext.onmouseover =
            this.$ext.onmouseout = this.$ext.onclick = this.$ext.ondblclick = null;
        }
    };

}).call(apf.BaseButton.prototype = new apf.StandardBinding());






/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */



/**
 * The baseclass of elements that allows the user to select one or more items
 * out of a list.
 *
 * @class apf.BaseList
 * @baseclass
 *
 * @inherits apf.MultiSelect
 * @inherits apf.Cache
 * @inherits apf.DataAction
 * @inheritsElsewhere apf.XForms
 *
 * @author      Ruben Daniels (ruben AT ajax DOT org)
 * @version     %I%, %G%
 * @since       0.8
 * @default_private
 *
 */
/**
 * @binding caption  Determines the caption of a node.
 */
/**
 * @binding icon     Determines the icon of a node. 
*
 * This binding rule is used
 * to determine the icon displayed when using a list skin. The {@link baseclass.baselist.binding.image image binding}
 * is used to determine the image in the thumbnail skin.
 */
/**
 * @binding image    Determines the image of a node. 
 * 
 * This binding rule is used
 * to determine the image displayed when using a thumbnail skin. The {@link baseclass.baselist.binding.icon icon binding}
 * is used to determine the icon in the list skin.
 *
 * #### Example
 *
 * In this example, the image URL is read from the thumbnail attribute of the data node.
 *
 * ```xml
 *  <a:thumbnail>
 *      <a:model>
 *          <data>
 *              <image caption="Thumb 1" thumbnail="img1" />
 *              <image caption="Thumb 2" thumbnail="img2" />
 *              <image caption="Thumb 3" />
 *          </data>
 *      </a:model>
 *      <a:bindings>
 *          <a:caption match="[@caption]" />
 *          <a:image match="[@thumbnail]" value="images/slideshow_img/[@thumbnail]_small.jpg" />
 *          <a:image value="images/slideshow_img/img29_small.jpg" />
 *          <a:each match="[image]" />
 *      </a:bindings>
 *  </a:thumbnail>
 * ```
 *
 */
/**
 * @binding css      Determines a CSS class for a node.
 *
 * #### Example
 *
 * In this example a node is bold when the folder contains unread messages:
 *
 * ```xml
 *  <a:tree>
 *      <a:model>
 *          <data>
 *              <folder caption="Folder 1">
 *                  <message unread="true" caption="message 1" />
 *              </folder>
 *              <folder caption="Folder 2" icon="email.png">
 *                  <message caption="message 2" />
 *              </folder>
 *              <folder caption="Folder 3">
 *                  <message caption="message 3" />
 *                  <message caption="message 4" />
 *              </folder>
 *          </data>
 *      </a:model>
 *      <a:bindings>
 *          <a:caption match="[@caption]" />
 *          <a:css match="[message[@unread]]" value="highlighUnread" />
 *          <a:icon match="[@icon]" />
 *          <a:icon match="[folder]" value="Famfolder.gif" />
 *          <a:each match="[folder|message]" />
 *      </a:bindings>
 *  </a:tree>
 * ```
 *
 */
/**
 * @binding tooltip  Determines the tooltip of a node. 
 */
/**
 * @event notunique Fires when the `more` attribute is set and an item is added that has a caption that already exists in the list.
 * @param {Object} e The standard event object, with the following properties:
 *                    - value ([[String]]): The value that was entered
 */
apf.BaseList = function(){
    this.$init(true);
    
    
    this.$dynCssClasses = [];
    
    
    this.listNodes   = [];
};

(function() {
    
    this.implement(
        
        apf.Cache,
        
        
        apf.DataAction,
        
        
        apf.K
    );
    

    // *** Properties and Attributes *** //

    this.$focussable = true; // This object can get the focus
    this.$isWindowContainer = -1;
    
    this.multiselect = true; // Initially Disable MultiSelect

    /**
     * @attribute {String} fill Sets or gets the set of items that should be loaded into this
     * element. Items are seperated by a comma (`,`). Ranges are specified by a start and end value seperated by a dash (`-`).
     *
     * #### Example
     *
     * This example loads a list with items starting at 1980 and ending at 2050. It also loads several other items and ranges.
     *
     * ```xml
     *  <a:dropdown fill="1980-2050" />
     *  <a:dropdown fill="red,green,blue,white" />
     *  <a:dropdown fill="None,100-110,1000-1100" /> <!-- 101, 102...110, 1000,1001, e.t.c. -->
     *  <a:dropdown fill="01-10" /> <!-- 01, 02, 03, 04, e.t.c. -->
     *  <a:dropdown fill="1-10" /> <!-- // 1 2 3 4 e.t.c. -->

     * ```
     */
    this.$propHandlers["fill"] = function(value){
        if (value)
            this.loadFillData(this.getAttribute("fill"));
        else
            this.clear();
    };
    
    
    
    /**
     * @attribute {String} mode Sets or gets the way this element interacts with the user.
     *  
     * The following values are possible:
     *
     *   - `check`: the user can select a single item from this element. The selected item is indicated.
     *   - `radio`: the user can select multiple items from this element. Each selected item is indicated.
     */
    this.$mode = 0;
    this.$propHandlers["mode"] = function(value){
        if ("check|radio".indexOf(value) > -1) {
            if (!this.hasFeature(apf.__MULTICHECK__))
                this.implement(apf.MultiCheck);
            
            this.addEventListener("afterrename", $afterRenameMode); //what does this do?
            
            this.multicheck = value == "check"; //radio is single
            this.$mode = this.multicheck ? 1 : 2;
        }
        else {
            //@todo undo actionRules setting
            this.removeEventListener("afterrename", $afterRenameMode);
            //@todo unimplement??
            this.$mode = 0;
        }
    };
    
    //@todo apf3.0 retest this completely
    function $afterRenameMode(){
    }
    
    

    // *** Keyboard support *** //

    

    //Handler for a plane list
    this.$keyHandler = function(e){
        var key      = e.keyCode,
            ctrlKey  = e.ctrlKey,
            shiftKey = e.shiftKey,
            selHtml  = this.$caret || this.$selected;

        if (e.returnValue == -1 || !selHtml || this.renaming) //@todo how about allowdeselect?
            return;

        var selXml = this.caret || this.selected,
            oExt   = this.$ext,
            // variables used in the switch statement below:
            node, margin, items, lines, hasScroll, hasScrollX, hasScrollY;

        switch (key) {
            case 13:
                if (this.$tempsel)
                    this.$selectTemp();

                if (this.ctrlselect == "enter")
                    this.select(this.caret, true);

                this.choose(this.selected);
                break;
            case 32:
                if (ctrlKey || !this.isSelected(this.caret))
                    this.select(this.caret, ctrlKey);
                break;
            case 109:
            case 46:
                //DELETE
                if (this.disableremove)
                    return;

                if (this.$tempsel)
                    this.$selectTemp();

                this.remove();
                break;
            case 36:
                //HOME
                var node = this.getFirstTraverseNode();
                
                
                if (this.hasFeature(apf.__VIRTUALVIEWPORT__))
                    return this.$viewport.scrollIntoView(node);
                
                    
                this.select(node, false, shiftKey);
                this.$container.scrollTop = 0;
                break;
            case 35:
                //END
                var node = this.getLastTraverseNode();
                
                
                if (this.hasFeature(apf.__VIRTUALVIEWPORT__))
                    return this.$viewport.scrollIntoView(node, true);
                
                
                this.select(node, false, shiftKey);
                this.$container.scrollTop = this.$container.scrollHeight;
                break;
            case 107:
                //+
                if (this.more)
   