"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util_1 = require("@glimmer/util");
const util_2 = require("@glimmer/util");
const wire_format_1 = require("@glimmer/wire-format");
class Block {
    constructor() {
        this.type = "block";
        this.statements = [];
        this.positionals = [];
    }
    toJSON() {
        return {
            statements: this.statements,
            locals: this.positionals
        };
    }
    push(statement) {
        this.statements.push(statement);
    }
}
exports.Block = Block;
class TemplateBlock extends Block {
    constructor() {
        super(...arguments);
        this.type = "template";
        this.yields = new util_2.DictSet();
        this.named = new util_2.DictSet();
        this.blocks = [];
        this.hasPartials = false;
    }
    toJSON() {
        return {
            statements: this.statements,
            locals: this.positionals,
            named: this.named.toArray(),
            yields: this.yields.toArray(),
            hasPartials: this.hasPartials
        };
    }
}
exports.TemplateBlock = TemplateBlock;
class ComponentBlock extends Block {
    constructor() {
        super(...arguments);
        this.type = "component";
        this.attributes = [];
        this.arguments = [];
        this.inParams = true;
    }
    push(statement) {
        if (this.inParams) {
            if (wire_format_1.Statements.isFlushElement(statement)) {
                this.inParams = false;
            }
            else if (wire_format_1.Statements.isArgument(statement)) {
                this.arguments.push(statement);
            }
            else if (wire_format_1.Statements.isAttribute(statement)) {
                this.attributes.push(statement);
            }
            else if (wire_format_1.Statements.isModifier(statement)) {
                throw new Error('Compile Error: Element modifiers are not allowed in components');
            }
            else {
                throw new Error('Compile Error: only parameters allowed before flush-element');
            }
        }
        else {
            this.statements.push(statement);
        }
    }
    toJSON() {
        let args = this.arguments;
        let keys = args.map(arg => arg[1]);
        let values = args.map(arg => arg[2]);
        return {
            attrs: this.attributes,
            args: [keys, values],
            locals: this.positionals,
            statements: this.statements
        };
    }
}
exports.ComponentBlock = ComponentBlock;
class Template {
    constructor(meta) {
        this.meta = meta;
        this.block = new TemplateBlock();
    }
    toJSON() {
        return {
            block: this.block.toJSON(),
            meta: this.meta
        };
    }
}
exports.Template = Template;
class JavaScriptCompiler {
    constructor(opcodes, meta) {
        this.blocks = new util_2.Stack();
        this.values = [];
        this.opcodes = opcodes;
        this.template = new Template(meta);
    }
    static process(opcodes, meta) {
        let compiler = new JavaScriptCompiler(opcodes, meta);
        return compiler.process();
    }
    process() {
        this.opcodes.forEach(([opcode, ...args]) => {
            if (!this[opcode]) {
                throw new Error(`unimplemented ${opcode} on JavaScriptCompiler`);
            }
            this[opcode](...args);
        });
        return this.template;
    }
    /// Nesting
    startBlock([program]) {
        let block = new Block();
        block.positionals = program.blockParams;
        this.blocks.push(block);
    }
    endBlock() {
        let { template, blocks } = this;
        template.block.blocks.push(blocks.pop().toJSON());
    }
    startProgram() {
        this.blocks.push(this.template.block);
    }
    endProgram() {
    }
    /// Statements
    text(content) {
        this.push([wire_format_1.Ops.Text, content]);
    }
    append(trusted) {
        this.push([wire_format_1.Ops.Append, this.popValue(), trusted]);
    }
    comment(value) {
        this.push([wire_format_1.Ops.Comment, value]);
    }
    modifier(path) {
        let params = this.popValue();
        let hash = this.popValue();
        this.push([wire_format_1.Ops.Modifier, path, params, hash]);
    }
    block(path, template, inverse) {
        let params = this.popValue();
        let hash = this.popValue();
        let blocks = this.template.block.blocks;
        util_1.assert(typeof template !== 'number' || blocks[template] !== null, 'missing block in the compiler');
        util_1.assert(typeof inverse !== 'number' || blocks[inverse] !== null, 'missing block in the compiler');
        this.push([wire_format_1.Ops.Block, path, params, hash, blocks[template], blocks[inverse]]);
    }
    openElement(tag, blockParams) {
        if (tag.indexOf('-') !== -1) {
            this.startComponent(blockParams);
        }
        else {
            this.push([wire_format_1.Ops.OpenElement, tag, blockParams]);
        }
    }
    flushElement() {
        this.push([wire_format_1.Ops.FlushElement]);
    }
    closeElement(tag) {
        if (tag.indexOf('-') !== -1) {
            let component = this.endComponent();
            this.push([wire_format_1.Ops.Component, tag, component]);
        }
        else {
            this.push([wire_format_1.Ops.CloseElement]);
        }
    }
    staticAttr(name, namespace) {
        let value = this.popValue();
        this.push([wire_format_1.Ops.StaticAttr, name, value, namespace]);
    }
    dynamicAttr(name, namespace) {
        let value = this.popValue();
        this.push([wire_format_1.Ops.DynamicAttr, name, value, namespace]);
    }
    trustingAttr(name, namespace) {
        let value = this.popValue();
        this.push([wire_format_1.Ops.TrustingAttr, name, value, namespace]);
    }
    staticArg(name) {
        let value = this.popValue();
        this.push([wire_format_1.Ops.StaticArg, name.slice(1), value]);
    }
    dynamicArg(name) {
        let value = this.popValue();
        this.push([wire_format_1.Ops.DynamicArg, name.slice(1), value]);
    }
    yield(to) {
        let params = this.popValue();
        this.push([wire_format_1.Ops.Yield, to, params]);
        this.template.block.yields.add(to);
    }
    debugger() {
        this.push([wire_format_1.Ops.Debugger, null, null]);
    }
    hasBlock(name) {
        this.pushValue([wire_format_1.Ops.HasBlock, name]);
        this.template.block.yields.add(name);
    }
    hasBlockParams(name) {
        this.pushValue([wire_format_1.Ops.HasBlockParams, name]);
        this.template.block.yields.add(name);
    }
    partial() {
        let params = this.popValue();
        this.push([wire_format_1.Ops.Partial, params[0]]);
        this.template.block.hasPartials = true;
    }
    /// Expressions
    literal(value) {
        if (value === undefined) {
            this.pushValue([wire_format_1.Ops.Undefined]);
        }
        else {
            this.pushValue(value);
        }
    }
    unknown(path) {
        this.pushValue([wire_format_1.Ops.Unknown, path]);
    }
    arg(path) {
        this.template.block.named.add(path[0]);
        this.pushValue([wire_format_1.Ops.Arg, path]);
    }
    get(path) {
        this.pushValue([wire_format_1.Ops.Get, path]);
    }
    concat() {
        this.pushValue([wire_format_1.Ops.Concat, this.popValue()]);
    }
    helper(path) {
        let params = this.popValue();
        let hash = this.popValue();
        this.pushValue([wire_format_1.Ops.Helper, path, params, hash]);
    }
    /// Stack Management Opcodes
    startComponent(blockParams) {
        let component = new ComponentBlock();
        component.positionals = blockParams;
        this.blocks.push(component);
    }
    endComponent() {
        let component = this.blocks.pop();
        util_1.assert(component.type === 'component', "Compiler bug: endComponent() should end a component");
        return component.toJSON();
    }
    prepareArray(size) {
        let values = [];
        for (let i = 0; i < size; i++) {
            values.push(this.popValue());
        }
        this.pushValue(values);
    }
    prepareObject(size) {
        util_1.assert(this.values.length >= size, `Expected ${size} values on the stack, found ${this.values.length}`);
        let keys = new Array(size);
        let values = new Array(size);
        for (let i = 0; i < size; i++) {
            keys[i] = this.popValue();
            values[i] = this.popValue();
        }
        this.pushValue([keys, values]);
    }
    /// Utilities
    push(args) {
        while (args[args.length - 1] === null) {
            args.pop();
        }
        this.blocks.current.push(args);
    }
    pushValue(val) {
        this.values.push(val);
    }
    popValue() {
        util_1.assert(this.values.length, "No expression found on stack");
        return this.values.pop();
    }
}
exports.default = JavaScriptCompiler;
//# sourceMappingURL=data:application/json;base64,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