import { assert } from "@glimmer/util";
import { Stack, DictSet } from "@glimmer/util";
import { Statements, Ops } from '@glimmer/wire-format';
export class Block {
    constructor() {
        this.type = "block";
        this.statements = [];
        this.positionals = [];
    }
    toJSON() {
        return {
            statements: this.statements,
            locals: this.positionals
        };
    }
    push(statement) {
        this.statements.push(statement);
    }
}
export class TemplateBlock extends Block {
    constructor() {
        super(...arguments);
        this.type = "template";
        this.yields = new DictSet();
        this.named = new DictSet();
        this.blocks = [];
        this.hasPartials = false;
    }
    toJSON() {
        return {
            statements: this.statements,
            locals: this.positionals,
            named: this.named.toArray(),
            yields: this.yields.toArray(),
            hasPartials: this.hasPartials
        };
    }
}
export class ComponentBlock extends Block {
    constructor() {
        super(...arguments);
        this.type = "component";
        this.attributes = [];
        this.arguments = [];
        this.inParams = true;
    }
    push(statement) {
        if (this.inParams) {
            if (Statements.isFlushElement(statement)) {
                this.inParams = false;
            }
            else if (Statements.isArgument(statement)) {
                this.arguments.push(statement);
            }
            else if (Statements.isAttribute(statement)) {
                this.attributes.push(statement);
            }
            else if (Statements.isModifier(statement)) {
                throw new Error('Compile Error: Element modifiers are not allowed in components');
            }
            else {
                throw new Error('Compile Error: only parameters allowed before flush-element');
            }
        }
        else {
            this.statements.push(statement);
        }
    }
    toJSON() {
        let args = this.arguments;
        let keys = args.map(arg => arg[1]);
        let values = args.map(arg => arg[2]);
        return {
            attrs: this.attributes,
            args: [keys, values],
            locals: this.positionals,
            statements: this.statements
        };
    }
}
export class Template {
    constructor(meta) {
        this.meta = meta;
        this.block = new TemplateBlock();
    }
    toJSON() {
        return {
            block: this.block.toJSON(),
            meta: this.meta
        };
    }
}
export default class JavaScriptCompiler {
    constructor(opcodes, meta) {
        this.blocks = new Stack();
        this.values = [];
        this.opcodes = opcodes;
        this.template = new Template(meta);
    }
    static process(opcodes, meta) {
        let compiler = new JavaScriptCompiler(opcodes, meta);
        return compiler.process();
    }
    process() {
        this.opcodes.forEach(([opcode, ...args]) => {
            if (!this[opcode]) {
                throw new Error(`unimplemented ${opcode} on JavaScriptCompiler`);
            }
            this[opcode](...args);
        });
        return this.template;
    }
    /// Nesting
    startBlock([program]) {
        let block = new Block();
        block.positionals = program.blockParams;
        this.blocks.push(block);
    }
    endBlock() {
        let { template, blocks } = this;
        template.block.blocks.push(blocks.pop().toJSON());
    }
    startProgram() {
        this.blocks.push(this.template.block);
    }
    endProgram() {
    }
    /// Statements
    text(content) {
        this.push([Ops.Text, content]);
    }
    append(trusted) {
        this.push([Ops.Append, this.popValue(), trusted]);
    }
    comment(value) {
        this.push([Ops.Comment, value]);
    }
    modifier(path) {
        let params = this.popValue();
        let hash = this.popValue();
        this.push([Ops.Modifier, path, params, hash]);
    }
    block(path, template, inverse) {
        let params = this.popValue();
        let hash = this.popValue();
        let blocks = this.template.block.blocks;
        assert(typeof template !== 'number' || blocks[template] !== null, 'missing block in the compiler');
        assert(typeof inverse !== 'number' || blocks[inverse] !== null, 'missing block in the compiler');
        this.push([Ops.Block, path, params, hash, blocks[template], blocks[inverse]]);
    }
    openElement(tag, blockParams) {
        if (tag.indexOf('-') !== -1) {
            this.startComponent(blockParams);
        }
        else {
            this.push([Ops.OpenElement, tag, blockParams]);
        }
    }
    flushElement() {
        this.push([Ops.FlushElement]);
    }
    closeElement(tag) {
        if (tag.indexOf('-') !== -1) {
            let component = this.endComponent();
            this.push([Ops.Component, tag, component]);
        }
        else {
            this.push([Ops.CloseElement]);
        }
    }
    staticAttr(name, namespace) {
        let value = this.popValue();
        this.push([Ops.StaticAttr, name, value, namespace]);
    }
    dynamicAttr(name, namespace) {
        let value = this.popValue();
        this.push([Ops.DynamicAttr, name, value, namespace]);
    }
    trustingAttr(name, namespace) {
        let value = this.popValue();
        this.push([Ops.TrustingAttr, name, value, namespace]);
    }
    staticArg(name) {
        let value = this.popValue();
        this.push([Ops.StaticArg, name.slice(1), value]);
    }
    dynamicArg(name) {
        let value = this.popValue();
        this.push([Ops.DynamicArg, name.slice(1), value]);
    }
    yield(to) {
        let params = this.popValue();
        this.push([Ops.Yield, to, params]);
        this.template.block.yields.add(to);
    }
    debugger() {
        this.push([Ops.Debugger, null, null]);
    }
    hasBlock(name) {
        this.pushValue([Ops.HasBlock, name]);
        this.template.block.yields.add(name);
    }
    hasBlockParams(name) {
        this.pushValue([Ops.HasBlockParams, name]);
        this.template.block.yields.add(name);
    }
    partial() {
        let params = this.popValue();
        this.push([Ops.Partial, params[0]]);
        this.template.block.hasPartials = true;
    }
    /// Expressions
    literal(value) {
        if (value === undefined) {
            this.pushValue([Ops.Undefined]);
        }
        else {
            this.pushValue(value);
        }
    }
    unknown(path) {
        this.pushValue([Ops.Unknown, path]);
    }
    arg(path) {
        this.template.block.named.add(path[0]);
        this.pushValue([Ops.Arg, path]);
    }
    get(path) {
        this.pushValue([Ops.Get, path]);
    }
    concat() {
        this.pushValue([Ops.Concat, this.popValue()]);
    }
    helper(path) {
        let params = this.popValue();
        let hash = this.popValue();
        this.pushValue([Ops.Helper, path, params, hash]);
    }
    /// Stack Management Opcodes
    startComponent(blockParams) {
        let component = new ComponentBlock();
        component.positionals = blockParams;
        this.blocks.push(component);
    }
    endComponent() {
        let component = this.blocks.pop();
        assert(component.type === 'component', "Compiler bug: endComponent() should end a component");
        return component.toJSON();
    }
    prepareArray(size) {
        let values = [];
        for (let i = 0; i < size; i++) {
            values.push(this.popValue());
        }
        this.pushValue(values);
    }
    prepareObject(size) {
        assert(this.values.length >= size, `Expected ${size} values on the stack, found ${this.values.length}`);
        let keys = new Array(size);
        let values = new Array(size);
        for (let i = 0; i < size; i++) {
            keys[i] = this.popValue();
            values[i] = this.popValue();
        }
        this.pushValue([keys, values]);
    }
    /// Utilities
    push(args) {
        while (args[args.length - 1] === null) {
            args.pop();
        }
        this.blocks.current.push(args);
    }
    pushValue(val) {
        this.values.push(val);
    }
    popValue() {
        assert(this.values.length, "No expression found on stack");
        return this.values.pop();
    }
}
//# sourceMappingURL=data:application/json;base64,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