'use strict';

Object.defineProperty(exports, "__esModule", {
    value: true
});
class Container {
    constructor(registry, resolver = null) {
        this._registry = registry;
        this._resolver = resolver;
        this._lookups = {};
        this._factoryDefinitionLookups = {};
    }
    factoryFor(specifier) {
        let factoryDefinition = this._factoryDefinitionLookups[specifier];
        if (!factoryDefinition) {
            if (this._resolver) {
                factoryDefinition = this._resolver.retrieve(specifier);
            }
            if (!factoryDefinition) {
                factoryDefinition = this._registry.registration(specifier);
            }
            if (factoryDefinition) {
                this._factoryDefinitionLookups[specifier] = factoryDefinition;
            }
        }
        if (!factoryDefinition) {
            return;
        }
        return this.buildFactory(specifier, factoryDefinition);
    }
    lookup(specifier) {
        let singleton = this._registry.registeredOption(specifier, 'singleton') !== false;
        if (singleton) {
            let lookup = this._lookups[specifier];
            if (lookup) {
                return lookup.instance;
            }
        }
        let factory = this.factoryFor(specifier);
        if (!factory) {
            return;
        }
        if (this._registry.registeredOption(specifier, 'instantiate') === false) {
            return factory.class;
        }
        let instance = factory.create();
        if (singleton && instance) {
            this._lookups[specifier] = { factory, instance };
        }
        return instance;
    }
    defaultInjections(specifier) {
        return {};
    }
    teardown() {
        let specifiers = Object.keys(this._lookups);
        for (let i = 0; i < specifiers.length; i++) {
            let specifier = specifiers[i];
            let { factory, instance } = this._lookups[specifier];
            factory.teardown(instance);
        }
    }
    defaultTeardown(instance) {}
    buildInjections(specifier) {
        let hash = this.defaultInjections(specifier);
        let injections = this._registry.registeredInjections(specifier);
        let injection;
        for (let i = 0; i < injections.length; i++) {
            injection = injections[i];
            hash[injection.property] = this.lookup(injection.source);
        }
        return hash;
    }
    buildFactory(specifier, factoryDefinition) {
        let injections = this.buildInjections(specifier);
        return {
            class: factoryDefinition,
            teardown: instance => {
                if (factoryDefinition.teardown) {
                    factoryDefinition.teardown(instance);
                } else {
                    this.defaultTeardown(instance);
                }
            },
            create(options) {
                let mergedOptions = Object.assign({}, injections, options);
                return factoryDefinition.create(mergedOptions);
            }
        };
    }
}
exports.default = Container;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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