"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CONSTANT = 0;
exports.INITIAL = 1;
exports.VOLATILE = NaN;
class RevisionTag {
    validate(snapshot) {
        return this.value() === snapshot;
    }
}
exports.RevisionTag = RevisionTag;
let $REVISION = exports.INITIAL;
class DirtyableTag extends RevisionTag {
    constructor(revision = $REVISION) {
        super();
        this.revision = revision;
    }
    value() {
        return this.revision;
    }
    dirty() {
        this.revision = ++$REVISION;
    }
}
exports.DirtyableTag = DirtyableTag;
function combineTagged(tagged) {
    let optimized = [];
    for (let i = 0, l = tagged.length; i < l; i++) {
        let tag = tagged[i].tag;
        if (tag === exports.VOLATILE_TAG)
            return exports.VOLATILE_TAG;
        if (tag === exports.CONSTANT_TAG)
            continue;
        optimized.push(tag);
    }
    return _combine(optimized);
}
exports.combineTagged = combineTagged;
function combineSlice(slice) {
    let optimized = [];
    let node = slice.head();
    while (node !== null) {
        let tag = node.tag;
        if (tag === exports.VOLATILE_TAG)
            return exports.VOLATILE_TAG;
        if (tag !== exports.CONSTANT_TAG)
            optimized.push(tag);
        node = slice.nextNode(node);
    }
    return _combine(optimized);
}
exports.combineSlice = combineSlice;
function combine(tags) {
    let optimized = [];
    for (let i = 0, l = tags.length; i < l; i++) {
        let tag = tags[i];
        if (tag === exports.VOLATILE_TAG)
            return exports.VOLATILE_TAG;
        if (tag === exports.CONSTANT_TAG)
            continue;
        optimized.push(tag);
    }
    return _combine(optimized);
}
exports.combine = combine;
function _combine(tags) {
    switch (tags.length) {
        case 0:
            return exports.CONSTANT_TAG;
        case 1:
            return tags[0];
        case 2:
            return new TagsPair(tags[0], tags[1]);
        default:
            return new TagsCombinator(tags);
    }
    ;
}
class CachedTag extends RevisionTag {
    constructor() {
        super(...arguments);
        this.lastChecked = null;
        this.lastValue = null;
    }
    value() {
        let { lastChecked, lastValue } = this;
        if (lastChecked !== $REVISION) {
            this.lastChecked = $REVISION;
            this.lastValue = lastValue = this.compute();
        }
        return this.lastValue;
    }
    invalidate() {
        this.lastChecked = null;
    }
}
exports.CachedTag = CachedTag;
class TagsPair extends CachedTag {
    constructor(first, second) {
        super();
        this.first = first;
        this.second = second;
    }
    compute() {
        return Math.max(this.first.value(), this.second.value());
    }
}
class TagsCombinator extends CachedTag {
    constructor(tags) {
        super();
        this.tags = tags;
    }
    compute() {
        let { tags } = this;
        let max = -1;
        for (let i = 0; i < tags.length; i++) {
            let value = tags[i].value();
            max = Math.max(value, max);
        }
        return max;
    }
}
class UpdatableTag extends CachedTag {
    constructor(tag) {
        super();
        this.tag = tag;
        this.lastUpdated = exports.INITIAL;
    }
    compute() {
        return Math.max(this.lastUpdated, this.tag.value());
    }
    update(tag) {
        if (tag !== this.tag) {
            this.tag = tag;
            this.lastUpdated = $REVISION;
            this.invalidate();
        }
    }
}
exports.UpdatableTag = UpdatableTag;
//////////
exports.CONSTANT_TAG = new (class ConstantTag extends RevisionTag {
    value() {
        return exports.CONSTANT;
    }
});
exports.VOLATILE_TAG = new (class VolatileTag extends RevisionTag {
    value() {
        return exports.VOLATILE;
    }
});
exports.CURRENT_TAG = new (class CurrentTag extends DirtyableTag {
    value() {
        return $REVISION;
    }
});
class CachedReference {
    constructor() {
        this.lastRevision = null;
        this.lastValue = null;
    }
    value() {
        let { tag, lastRevision, lastValue } = this;
        if (!lastRevision || !tag.validate(lastRevision)) {
            lastValue = this.lastValue = this.compute();
            this.lastRevision = tag.value();
        }
        return lastValue;
    }
    invalidate() {
        this.lastRevision = null;
    }
}
exports.CachedReference = CachedReference;
class MapperReference extends CachedReference {
    constructor(reference, mapper) {
        super();
        this.tag = reference.tag;
        this.reference = reference;
        this.mapper = mapper;
    }
    compute() {
        let { reference, mapper } = this;
        return mapper(reference.value());
    }
}
function map(reference, mapper) {
    return new MapperReference(reference, mapper);
}
exports.map = map;
//////////
class ReferenceCache {
    constructor(reference) {
        this.lastValue = null;
        this.lastRevision = null;
        this.initialized = false;
        this.tag = reference.tag;
        this.reference = reference;
    }
    peek() {
        if (!this.initialized) {
            return this.initialize();
        }
        return this.lastValue;
    }
    revalidate() {
        if (!this.initialized) {
            return this.initialize();
        }
        let { reference, lastRevision } = this;
        let tag = reference.tag;
        if (tag.validate(lastRevision))
            return NOT_MODIFIED;
        this.lastRevision = tag.value();
        let { lastValue } = this;
        let value = reference.value();
        if (value === lastValue)
            return NOT_MODIFIED;
        this.lastValue = value;
        return value;
    }
    initialize() {
        let { reference } = this;
        let value = this.lastValue = reference.value();
        this.lastRevision = reference.tag.value();
        this.initialized = true;
        return value;
    }
}
exports.ReferenceCache = ReferenceCache;
const NOT_MODIFIED = "adb3b78e-3d22-4e4b-877a-6317c2c5c145";
function isModified(value) {
    return value !== NOT_MODIFIED;
}
exports.isModified = isModified;
//# sourceMappingURL=data:application/json;base64,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