export const CONSTANT = 0;
export const INITIAL = 1;
export const VOLATILE = NaN;
export class RevisionTag {
    validate(snapshot) {
        return this.value() === snapshot;
    }
}
let $REVISION = INITIAL;
export class DirtyableTag extends RevisionTag {
    constructor(revision = $REVISION) {
        super();
        this.revision = revision;
    }
    value() {
        return this.revision;
    }
    dirty() {
        this.revision = ++$REVISION;
    }
}
export function combineTagged(tagged) {
    let optimized = [];
    for (let i = 0, l = tagged.length; i < l; i++) {
        let tag = tagged[i].tag;
        if (tag === VOLATILE_TAG)
            return VOLATILE_TAG;
        if (tag === CONSTANT_TAG)
            continue;
        optimized.push(tag);
    }
    return _combine(optimized);
}
export function combineSlice(slice) {
    let optimized = [];
    let node = slice.head();
    while (node !== null) {
        let tag = node.tag;
        if (tag === VOLATILE_TAG)
            return VOLATILE_TAG;
        if (tag !== CONSTANT_TAG)
            optimized.push(tag);
        node = slice.nextNode(node);
    }
    return _combine(optimized);
}
export function combine(tags) {
    let optimized = [];
    for (let i = 0, l = tags.length; i < l; i++) {
        let tag = tags[i];
        if (tag === VOLATILE_TAG)
            return VOLATILE_TAG;
        if (tag === CONSTANT_TAG)
            continue;
        optimized.push(tag);
    }
    return _combine(optimized);
}
function _combine(tags) {
    switch (tags.length) {
        case 0:
            return CONSTANT_TAG;
        case 1:
            return tags[0];
        case 2:
            return new TagsPair(tags[0], tags[1]);
        default:
            return new TagsCombinator(tags);
    }
    ;
}
export class CachedTag extends RevisionTag {
    constructor() {
        super(...arguments);
        this.lastChecked = null;
        this.lastValue = null;
    }
    value() {
        let { lastChecked, lastValue } = this;
        if (lastChecked !== $REVISION) {
            this.lastChecked = $REVISION;
            this.lastValue = lastValue = this.compute();
        }
        return this.lastValue;
    }
    invalidate() {
        this.lastChecked = null;
    }
}
class TagsPair extends CachedTag {
    constructor(first, second) {
        super();
        this.first = first;
        this.second = second;
    }
    compute() {
        return Math.max(this.first.value(), this.second.value());
    }
}
class TagsCombinator extends CachedTag {
    constructor(tags) {
        super();
        this.tags = tags;
    }
    compute() {
        let { tags } = this;
        let max = -1;
        for (let i = 0; i < tags.length; i++) {
            let value = tags[i].value();
            max = Math.max(value, max);
        }
        return max;
    }
}
export class UpdatableTag extends CachedTag {
    constructor(tag) {
        super();
        this.tag = tag;
        this.lastUpdated = INITIAL;
    }
    compute() {
        return Math.max(this.lastUpdated, this.tag.value());
    }
    update(tag) {
        if (tag !== this.tag) {
            this.tag = tag;
            this.lastUpdated = $REVISION;
            this.invalidate();
        }
    }
}
//////////
export const CONSTANT_TAG = new (class ConstantTag extends RevisionTag {
    value() {
        return CONSTANT;
    }
});
export const VOLATILE_TAG = new (class VolatileTag extends RevisionTag {
    value() {
        return VOLATILE;
    }
});
export const CURRENT_TAG = new (class CurrentTag extends DirtyableTag {
    value() {
        return $REVISION;
    }
});
export class CachedReference {
    constructor() {
        this.lastRevision = null;
        this.lastValue = null;
    }
    value() {
        let { tag, lastRevision, lastValue } = this;
        if (!lastRevision || !tag.validate(lastRevision)) {
            lastValue = this.lastValue = this.compute();
            this.lastRevision = tag.value();
        }
        return lastValue;
    }
    invalidate() {
        this.lastRevision = null;
    }
}
class MapperReference extends CachedReference {
    constructor(reference, mapper) {
        super();
        this.tag = reference.tag;
        this.reference = reference;
        this.mapper = mapper;
    }
    compute() {
        let { reference, mapper } = this;
        return mapper(reference.value());
    }
}
export function map(reference, mapper) {
    return new MapperReference(reference, mapper);
}
//////////
export class ReferenceCache {
    constructor(reference) {
        this.lastValue = null;
        this.lastRevision = null;
        this.initialized = false;
        this.tag = reference.tag;
        this.reference = reference;
    }
    peek() {
        if (!this.initialized) {
            return this.initialize();
        }
        return this.lastValue;
    }
    revalidate() {
        if (!this.initialized) {
            return this.initialize();
        }
        let { reference, lastRevision } = this;
        let tag = reference.tag;
        if (tag.validate(lastRevision))
            return NOT_MODIFIED;
        this.lastRevision = tag.value();
        let { lastValue } = this;
        let value = reference.value();
        if (value === lastValue)
            return NOT_MODIFIED;
        this.lastValue = value;
        return value;
    }
    initialize() {
        let { reference } = this;
        let value = this.lastValue = reference.value();
        this.lastRevision = reference.tag.value();
        this.initialized = true;
        return value;
    }
}
const NOT_MODIFIED = "adb3b78e-3d22-4e4b-877a-6317c2c5c145";
export function isModified(value) {
    return value !== NOT_MODIFIED;
}
//# sourceMappingURL=data:application/json;base64,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